(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    101920,       3321]
NotebookOptionsPosition[     87696,       2819]
NotebookOutlinePosition[     89265,       2861]
CellTagsIndexPosition[     89179,       2856]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"TransformationMatrix" :> 
          Documentation`HelpLookup["paclet:ref/TransformationMatrix"], 
          "GeometricTransformation" :> 
          Documentation`HelpLookup["paclet:ref/GeometricTransformation"], 
          "AffineTransform" :> 
          Documentation`HelpLookup["paclet:ref/AffineTransform"], 
          "RotationTransform" :> 
          Documentation`HelpLookup["paclet:ref/RotationTransform"], 
          "TranslationTransform" :> 
          Documentation`HelpLookup["paclet:ref/TranslationTransform"], 
          "ScalingTransform" :> 
          Documentation`HelpLookup["paclet:ref/ScalingTransform"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"TransformationMatrix\"\>", 
       2->"\<\"GeometricTransformation\"\>", 3->"\<\"AffineTransform\"\>", 
       4->"\<\"RotationTransform\"\>", 5->"\<\"TranslationTransform\"\>", 
       6->"\<\"ScalingTransform\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Geometric Transforms" :> 
          Documentation`HelpLookup["paclet:guide/GeometricTransforms"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Geometric Transforms\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["TransformationFunction", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TransformationFunction",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TransformationFunction"], "[", 
       StyleBox["data", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a transformation function that applies \
geometric and other transformations."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " works like ",
 Cell[BoxData[
  ButtonBox["Function",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Function"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{"TransformationFunction", "[", 
    StyleBox["\[Ellipsis]", "TR"], "]"}], "[", 
   StyleBox["vec", "TI"], "]"}]], "InlineFormula"],
 " applies the transformation function to a vector, returning a transformed \
vector."
}], "Notes",
 CellID->288523798],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " works with both numerical and symbolic vectors."
}], "Notes",
 CellID->207185040],

Cell[TextData[{
 "For purposes of display, a ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " is typically shown with a ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"d", "+", "1"}], ")"}], "\[Times]", " ", 
    RowBox[{"(", 
     RowBox[{"d", "+", "1"}], ")"}]}], TraditionalForm]], "InlineMath"],
 " homogenous matrix."
}], "Notes",
 CellID->209660300],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Composition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Composition"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["InverseFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InverseFunction"]], "InlineFormula"],
 " can be used with ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->713987337],

Cell[TextData[{
 "When possible, matrix forms for transformations can be obtained from ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " objects using ",
 Cell[BoxData[
  ButtonBox["TransformationMatrix",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationMatrix"]], "InlineFormula"],
 "."
}], "Notes",
 CellID->303436182],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["GeometricTransformation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricTransformation"]], "InlineFormula"],
 " can be used to represent the effect of applying a ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " object to geometrical or graphics objects."
}], "Notes",
 CellID->71279388]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->149624750],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->226357251],

Cell["Create a rotation transform:", "ExampleText",
 CellID->980191149],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "=", 
  RowBox[{"RotationTransform", "[", "\[Theta]", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->247847466],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->426315870]
}, Open  ]],

Cell[TextData[{
 "This rotates the vector ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["y", "TI"]}], "}"}]], "InlineFormula"],
 " by angle ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->279787753],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->509669506],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "-", 
    RowBox[{"y", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"y", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}]}], "}"}]], "Output",
 ImageSize->{263, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->535224651]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(13)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Constructing TransformationFunction",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(10)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->557287772],

Cell[TextData[{
 "A translation by the vector ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["qx", "TI"], ",", 
    StyleBox["qy", "TI"], ",", 
    StyleBox["qz", "TI"]}], "}"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->200115988],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"TranslationTransform", "[", 
   RowBox[{"{", 
    RowBox[{"qx", ",", "qy", ",", "qz"}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->42025119],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "qx"},
     {"0", "1", "0", "qy"},
     {"0", "0", "1", "qz"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{252, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->230337727]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49073389],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"qx", "+", "x"}], ",", 
   RowBox[{"qy", "+", "y"}], ",", 
   RowBox[{"qz", "+", "z"}]}], "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->193636383]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->84662931],

Cell[TextData[{
 "A rotation around the ",
 Cell[BoxData[
  StyleBox["z", "TI"]], "InlineFormula"],
 " axis:"
}], "ExampleText",
 CellID->371074523],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "=", 
  RowBox[{"RotationTransform", "[", 
   RowBox[{"\[Theta]", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28155084],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0", "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0", "0"},
     {"0", "0", "1", "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{323, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->367507988]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"r", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->655728535],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "-", 
    RowBox[{"y", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", 
   RowBox[{
    RowBox[{"y", " ", 
     RowBox[{"Cos", "[", "\[Theta]", "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], ",", "z"}], "}"}]], "Output",
 ImageSize->{281, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->70253597]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->491954993],

Cell["Scaling along the coordinate axes:", "ExampleText",
 CellID->309971784],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ScalingTransform", "[", 
  RowBox[{"{", 
   RowBox[{"sx", ",", "sy"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32236626],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"sx", "0", "0"},
     {"0", "sy", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{244, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15022460]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->127343740],

Cell[TextData[{
 "Shearing in the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " direction by an angle ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->21553636],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ShearingTransform", "[", 
  RowBox[{"\[Theta]", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->121015059],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", 
      RowBox[{"Tan", "[", "\[Theta]", "]"}], "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{265, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->554285227]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->388551663],

Cell[TextData[{
 "Reflecting in the ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], ",", 
   StyleBox["y", "TI"]}]], "InlineFormula"],
 " plane:"
}], "ExampleText",
 CellID->445521037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ReflectionTransform", "[", 
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "1"}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32545861],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0", "0"},
     {"0", "1", "0", "0"},
     {"0", "0", 
      RowBox[{"-", "1"}], "0"},
     {"0", "0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{253, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->495409537]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->155134094],

Cell[TextData[{
 "Rescaling the box ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"[", 
    RowBox[{
     StyleBox["xmin", "TI"], ",", 
     StyleBox["xmax", "TI"]}], "]"}], "\[Cross]", 
   RowBox[{"[", 
    RowBox[{
     StyleBox["ymin", "TI"], ",", 
     StyleBox["ymax", "TI"]}], "]"}]}]], "InlineFormula"],
 " to the unit square:"
}], "ExampleText",
 CellID->225612157],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RescalingTransform", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"xmin", ",", "xmax"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"ymin", ",", "ymax"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->640460778],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      FractionBox["1", 
       RowBox[{"xmax", "-", "xmin"}]], "0", 
      RowBox[{"-", 
       FractionBox["xmin", 
        RowBox[{"xmax", "-", "xmin"}]]}]},
     {"0", 
      FractionBox["1", 
       RowBox[{"ymax", "-", "ymin"}]], 
      RowBox[{"-", 
       FractionBox["ymin", 
        RowBox[{"ymax", "-", "ymin"}]]}]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{366, 65},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->201042413]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->375521382],

Cell[TextData[{
 "A general ",
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->68715309],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->461959536],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "2", "3"},
     {"4", "5", "6"},
     {"7", "8", "9"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->458328840]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->200816312],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"3", "+", "x", "+", 
     RowBox[{"2", " ", "y"}]}], 
    RowBox[{"9", "+", 
     RowBox[{"7", " ", "x"}], "+", 
     RowBox[{"8", " ", "y"}]}]], ",", 
   FractionBox[
    RowBox[{"6", "+", 
     RowBox[{"4", " ", "x"}], "+", 
     RowBox[{"5", " ", "y"}]}], 
    RowBox[{"9", "+", 
     RowBox[{"7", " ", "x"}], "+", 
     RowBox[{"8", " ", "y"}]}]]}], "}"}]], "Output",
 ImageSize->{170, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->648750301]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11624954],

Cell["A linear transformation:", "ExampleText",
 CellID->173676943],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"tl", "=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"{", " ", 
    RowBox[{"Array", "[", 
     RowBox[{
      RowBox[{
       SubscriptBox["a", 
        RowBox[{"#1", ",", "#2"}]], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "2"}], "}"}]}], "]"}], " ", "}"}], "]"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->139179960],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      SubscriptBox["a", 
       RowBox[{"1", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"1", ",", "2"}]], "0"},
     {
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "2"}]], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{262, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->393458163]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"tl", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->97109209],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"x", " ", 
     SubscriptBox["a", 
      RowBox[{"1", ",", "1"}]]}], "+", 
    RowBox[{"y", " ", 
     SubscriptBox["a", 
      RowBox[{"1", ",", "2"}]]}]}], ",", 
   RowBox[{
    RowBox[{"x", " ", 
     SubscriptBox["a", 
      RowBox[{"2", ",", "1"}]]}], "+", 
    RowBox[{"y", " ", 
     SubscriptBox["a", 
      RowBox[{"2", ",", "2"}]]}]}]}], "}"}]], "Output",
 ImageSize->{185, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->385381910]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->184015676],

Cell["An affine transformation:", "ExampleText",
 CellID->726939751],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ta", "=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"{", " ", 
    RowBox[{
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["a", 
         RowBox[{"#1", ",", "#2"}]], "&"}], ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", "2"}], "}"}]}], "]"}], ",", " ", 
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["b", "#"], "&"}], ",", 
       RowBox[{"{", "2", "}"}]}], "]"}]}], " ", "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->457458540],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      SubscriptBox["a", 
       RowBox[{"1", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"1", ",", "2"}]], 
      SubscriptBox["b", "1"]},
     {
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "2"}]], 
      SubscriptBox["b", "2"]},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{268, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->361809556]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ta", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->73542158],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["b", "1"], "+", 
    RowBox[{"x", " ", 
     SubscriptBox["a", 
      RowBox[{"1", ",", "1"}]]}], "+", 
    RowBox[{"y", " ", 
     SubscriptBox["a", 
      RowBox[{"1", ",", "2"}]]}]}], ",", 
   RowBox[{
    SubscriptBox["b", "2"], "+", 
    RowBox[{"x", " ", 
     SubscriptBox["a", 
      RowBox[{"2", ",", "1"}]]}], "+", 
    RowBox[{"y", " ", 
     SubscriptBox["a", 
      RowBox[{"2", ",", "2"}]]}]}]}], "}"}]], "Output",
 ImageSize->{235, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->123841534]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->697472489],

Cell["A linear fractional transformation:", "ExampleText",
 CellID->77138904],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"tlft", "=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["a", 
         RowBox[{"#1", ",", "#2"}]], "&"}], ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", "2"}], "}"}]}], "]"}], ",", " ", 
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["b", "#"], "&"}], ",", 
       RowBox[{"{", "2", "}"}]}], "]"}], ",", 
     RowBox[{"Array", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["c", "#"], "&"}], ",", 
       RowBox[{"{", "2", "}"}]}], "]"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->89629318],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      SubscriptBox["a", 
       RowBox[{"1", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"1", ",", "2"}]], 
      SubscriptBox["b", "1"]},
     {
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "2"}]], 
      SubscriptBox["b", "2"]},
     {
      SubscriptBox["c", "1"], 
      SubscriptBox["c", "2"], "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{268, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->650851166]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"tlft", "[", 
  RowBox[{"{", 
   RowBox[{"x", ",", "y"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->619904342],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{
     SubscriptBox["b", "1"], "+", 
     RowBox[{"x", " ", 
      SubscriptBox["a", 
       RowBox[{"1", ",", "1"}]]}], "+", 
     RowBox[{"y", " ", 
      SubscriptBox["a", 
       RowBox[{"1", ",", "2"}]]}]}], 
    RowBox[{"1", "+", 
     RowBox[{"x", " ", 
      SubscriptBox["c", "1"]}], "+", 
     RowBox[{"y", " ", 
      SubscriptBox["c", "2"]}]}]], ",", 
   FractionBox[
    RowBox[{
     SubscriptBox["b", "2"], "+", 
     RowBox[{"x", " ", 
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]]}], "+", 
     RowBox[{"y", " ", 
      SubscriptBox["a", 
       RowBox[{"2", ",", "2"}]]}]}], 
    RowBox[{"1", "+", 
     RowBox[{"x", " ", 
      SubscriptBox["c", "1"]}], "+", 
     RowBox[{"y", " ", 
      SubscriptBox["c", "2"]}]}]]}], "}"}]], "Output",
 ImageSize->{243, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->57595511]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Working with TransformationFunction as a Function",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->112582735],

Cell[TextData[{
 "Here ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " is a rotation of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Pi]", "/", "3"}], TraditionalForm]], "InlineMath"],
 " around the ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " axis: "
}], "ExampleText",
 CellID->223549394],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"RotationTransform", "[", 
   RowBox[{
    RowBox[{"\[Pi]", "/", "3."}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->109016475],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"0.5000000000000001`", 
      RowBox[{"-", "0.8660254037844386`"}], "0.`", "0.`"},
     {"0.8660254037844386`", "0.5000000000000001`", "0.`", "0.`"},
     {"0.`", "0.`", "1.`", "0.`"},
     {"0.`", "0.`", "0.`", "1.`"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{365, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18942656]
}, Open  ]],

Cell[TextData[{
 "This transforms the ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " axis:"
}], "ExampleText",
 CellID->1829689],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", 
  RowBox[{"{", 
   RowBox[{"1", ",", "0", ",", "0"}], "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->254950545],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5000000000000001`", ",", "0.8660254037844386`", ",", "0.`"}], 
  "}"}]], "Output",
 ImageSize->{132, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->221024316]
}, Open  ]],

Cell["This transforms a list of vectors: ", "ExampleText",
 CellID->49040006],

Cell[BoxData[
 RowBox[{
  RowBox[{"vl", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", "3"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->25698720],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "[", "vl", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->17490611],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.6236275766219147`"}], ",", "0.39969635249276353`", ",", 
     "0.6987257063517784`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.7010605959649987`"}], ",", "0.5892900783934985`", ",", 
     "0.43058001116462385`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.31934729748180984`"}], ",", "1.0023334147267333`", ",", 
     "0.9070346830698062`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.15416797474657384`", ",", "0.4454863032806336`", ",", 
     "0.18937342819463865`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.42623368156867814`"}], ",", "0.4895782768196556`", ",", 
     "0.4787669484078412`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.48475013237206094`"}], ",", "1.0823926252673828`", ",", 
     "0.771201409704819`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.17872198290524954`", ",", "0.49551519654850557`", ",", 
     "0.7149859181868385`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.1224638803560916`"}], ",", "0.4648730652096874`", ",", 
     "0.2587980957820184`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.36553783933977435`"}], ",", "0.6174234721657872`", ",", 
     "0.6342937449050241`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.025256683139218472`"}], ",", "1.1480453186183193`", ",", 
     "0.6244002222435034`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{446, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->619971542]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->268855721],

Cell["Composing two transformations:", "ExampleText",
 CellID->821959473],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t1", "=", 
  RowBox[{"RotationTransform", "[", "\[Theta]", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->524410875],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->213297444]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t2", "=", 
  RowBox[{"RotationTransform", "[", "\[Phi]", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->715708731],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Phi]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Phi]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Phi]", "]"}], 
      RowBox[{"Cos", "[", "\[Phi]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->580710421]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{"t1", ",", "t2"}], "]"}], "//", "Simplify"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->79553851],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", 
       RowBox[{"\[Theta]", "+", "\[Phi]"}], "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", 
        RowBox[{"\[Theta]", "+", "\[Phi]"}], "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", 
       RowBox[{"\[Theta]", "+", "\[Phi]"}], "]"}], 
      RowBox[{"Cos", "[", 
       RowBox[{"\[Theta]", "+", "\[Phi]"}], "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{348, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->38355828]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->167718222],

Cell["Computing the inverse:", "ExampleText",
 CellID->716032382],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"RotationTransform", "[", "\[Theta]", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->471611810],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], "0"},
     {
      RowBox[{"Sin", "[", "\[Theta]", "]"}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8183075]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"it", "=", 
  RowBox[{
   RowBox[{"InverseFunction", "[", "t", "]"}], "//", "Simplify"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->28804467],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"Cos", "[", "\[Theta]", "]"}], 
      RowBox[{"Sin", "[", "\[Theta]", "]"}], "0"},
     {
      RowBox[{"-", 
       RowBox[{"Sin", "[", "\[Theta]", "]"}]}], 
      RowBox[{"Cos", "[", "\[Theta]", "]"}], "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{308, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->13923555]
}, Open  ]],

Cell["This shows they are inverses:", "ExampleText",
 CellID->382240035],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Composition", "[", 
   RowBox[{"it", ",", "t"}], "]"}], "//", "Simplify"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->455174583],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "0", "0"},
     {"0", "1", "0"},
     {"0", "0", "1"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->73980967]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->317141207],

Cell[TextData[{
 "Computing the partial derivative ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[PartialD]", "x"], 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["\[PartialD]", "y"], "2"], 
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "y"}], "]"}]}]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->8630988],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  RowBox[{"TransformationFunction", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->428657653],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "2", "3"},
     {"4", "5", "6"},
     {"7", "8", "9"}
    },
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridDefaultElement->""], ")"}], "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->247091727]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Derivative", "[", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}], "]"}], "[", "f", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->374734954],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWs1S1EAQzm72B38oVxRFAVkpEATBu5ZFFQc9UPyoT7C1WsKB0gLu3H0D
KT2Jll7Usxdu+Dg8QsxMZnZnv3Q2M2RCwi5btUlmMtPT/XVPZrqnVxt7m2+3
G3tbzUb9xU7jw+ZWc7f+/P2OX+UWHKew4zjO8ojjP3v+E/+z367/Dx6Dp4fs
6sq3nrPoX0p+sSruc1AuQfv5mP7zMf3HORezUPtYKftX5xKU52Lel4EeShlH
f96Q/hivn4Haon9xoTd7rEKZUyu1y47j+rcyoMXGLYr6ItQPAMWKGLmSmHIF
KGcnI46QnoyjpDZR59gLJWajVwiLR67wfZxuOR/lVlmhojNakextqvWssEAb
yAIL2jqwlbRB1B7nocNaEUtTTCpiJBqL8LyLG88lJDGxjeyQqGaMBG0ZrJdc
AdX69PHQ+3KY46Jjkd0sJDtE9L4f6SNCWwr9PcrebqJX2riR+PsqSBSBiStG
uZ9TVPT3HzZRYVwURR92r3n6lnODQNguJjVyXPP5w6WuqD2CK6KB/NRzgsP1
jHHA8iDvNktyhavaINnK1o6NthC7OzREVa4vkzmQn7aMdOVHqvg+sA7uE5dU
2q7QV2sP1WG7jGpR2GJJcFEUeJaEpPT3kOrpijsVE0AUpixzOmSN07h5GSD9
S1D9CXJh76D1b6j9oaVN9tgk3xxD7ScoVwV3wcQImCuQPQ8oTkKet+NcE5oJ
ZtrR0ZET/A75KPv7+6K99Awq0O4jb1er1U5OTiiJOcrlFo6eMitieaZ2HXoY
hXtRmA94bVvrppWnUGsaN/tOUn0CtbbjXN/IUUcUfbuCV/WOezM3on5A6Xde
YkUrEbJLukuRiEV81xPVop3qtUK0WK9eidasaiFiUrsEFAON3oFaN/E4erpk
mMpVsIViKlEV1Kb5Lto0nqKnuwpZu0xq6a4WxfRn3FnFHVBnyXVoGukw0GEE
Vfb4zFO939FOrr32Ctlt1ZF7OCq6QWDQA7EDCvuS044lYPs1TmcsxBM9wwzm
g1JWffF0bH5IT79dRtD1wknbhv1MN++A94/YC5DfSk0dUJavllWEJKI21ntE
vqxJ93Re7hqF0CkRW+d3toa7YiS5h1O/KJcJ+V0ht4zRmXuiJe2erhPtia4p
7SneNzhZ1hQzANpP40DTNBfgQUz/OJ9m0Wt9f5Ra27kAU4b0TX2kBV4/GmEr
vZQLMJmZjGfn3z0itdnPuQCo9X7OBaCto39zAdA2+jcXgLaMi1wAtJCLXIDA
UuR+b5jqnYPT72xzAnBPmBdUss0J4Oe7oS+TngX1fm4AtQPvx9yAaY+yklsk
V/2TGzCeA/nzkBvAffaQ3Le9Duv08psbMGGZ0/RyA+h5+FlQPQS5ZsjWX6D8
NUa7EyQVdlkR49bJFpiDgOfh93gvjCv+IXoVpY468gvkqRVmCBzwSWieITCl
xc9fLVnpVuyybfhmEcqMS5N42D+S6gKUp5UygyVp/OqYHPWKZ+eM337sRz0B
tRv72fDa8WBKtjeRSOXnbL+Xoi3rkWPG49IkdXUVyvZP7aMiIPb9ShnxwBHN
fRHTWMdrLRSRaoPXy1V1UEuLNjWBKKR/6hsfB7cdW0iimZud3Hl6p+xSn/Hf
fdvoJvf2TXxwuTbA7BIoBSvDcIiDJCfpZLaoVUQxcmDuC+v6qBtQroZGesdv
tIeU+Gw8liJbF86rh/dSCx2UOLBY6f10PQEnva2svKxXSruywmmw3rMmwdm2
U/gPMmOFBg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{370, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->188814395]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Working with TransformationFunction as a Formula",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->93723694],

Cell["This defines a general transform:", "ExampleText",
 CellID->232691274],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"{", " ", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"3", ",", "4"}], "}"}]}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}]}], " ", "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->215996710],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", "2", "1"},
     {"3", "4", "2"},
     {"3", "1", "1"}
    },
    ColumnsEqual->False,
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
    RowsEqual->False], ")"}], "]"}]], "Output",
 ImageSize->{230, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->551516433]
}, Open  ]],

Cell["This is the corresponding formulae:", "ExampleText",
 CellID->75030959],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  RowBox[{"t", "[", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->136503702],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"1", "+", "x", "+", 
     RowBox[{"2", " ", "y"}]}], 
    RowBox[{"1", "+", 
     RowBox[{"3", " ", "x"}], "+", "y"}]], ",", 
   FractionBox[
    RowBox[{"2", "+", 
     RowBox[{"3", " ", "x"}], "+", 
     RowBox[{"4", " ", "y"}]}], 
    RowBox[{"1", "+", 
     RowBox[{"3", " ", "x"}], "+", "y"}]]}], "}"}]], "Output",
 ImageSize->{160, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->84702603]
}, Open  ]],

Cell["A derivative:", "ExampleText",
 CellID->202702190],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->170276240],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     FractionBox["6", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "+", 
         RowBox[{"3", " ", "x"}], "+", "y"}], ")"}], "2"]]}], "+", 
    FractionBox[
     RowBox[{"18", " ", 
      RowBox[{"(", 
       RowBox[{"1", "+", "x", "+", 
        RowBox[{"2", " ", "y"}]}], ")"}]}], 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"3", " ", "x"}], "+", "y"}], ")"}], "3"]]}], ",", 
   RowBox[{
    RowBox[{"-", 
     FractionBox["18", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "+", 
         RowBox[{"3", " ", "x"}], "+", "y"}], ")"}], "2"]]}], "+", 
    FractionBox[
     RowBox[{"18", " ", 
      RowBox[{"(", 
       RowBox[{"2", "+", 
        RowBox[{"3", " ", "x"}], "+", 
        RowBox[{"4", " ", "y"}]}], ")"}]}], 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "+", 
        RowBox[{"3", " ", "x"}], "+", "y"}], ")"}], "3"]]}]}], 
  "}"}]], "Output",
 ImageSize->{423, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->587440101]
}, Open  ]],

Cell["A limit:", "ExampleText",
 CellID->181871226],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{"f", ",", 
   RowBox[{"x", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->76736110],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"2", " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", "y"}], ")"}]}], 
    RowBox[{"4", "+", "y"}]], ",", 
   FractionBox[
    RowBox[{"5", "+", 
     RowBox[{"4", " ", "y"}]}], 
    RowBox[{"4", "+", "y"}]]}], "}"}]], "Output",
 ImageSize->{125, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->3887371]
}, Open  ]],

Cell["An integral:", "ExampleText",
 CellID->792324328],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{"f", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->473906299],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     FractionBox["1", "9"], " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"3", " ", "x"}], "+", "y"}], ")"}]}], "+", 
    RowBox[{
     FractionBox["1", "9"], " ", 
     RowBox[{"(", 
      RowBox[{"2", "+", 
       RowBox[{"5", " ", "y"}]}], ")"}], " ", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "+", 
       RowBox[{"3", " ", "x"}], "+", "y"}], "]"}]}]}], ",", 
   RowBox[{"x", "+", 
    RowBox[{
     FractionBox["1", "3"], " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"3", " ", "y"}]}], ")"}], " ", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "+", 
       RowBox[{"3", " ", "x"}], "+", "y"}], "]"}]}]}]}], "}"}]], "Output",
 ImageSize->{464, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->221234707]
}, Open  ]],

Cell["A plot:", "ExampleText",
 CellID->355523943],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Red", ",", "Green"}], "}"}]}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"1", "+", 
       RowBox[{"3", "x"}], "+", "y"}], "\[Equal]", "0"}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->71032048],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->2476398]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TransformationFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TransformationFunction"]], "InlineFormula"],
 " can be used as an argument to ",
 Cell[BoxData[
  ButtonBox["GeometricTransformation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GeometricTransformation"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->97803277],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"AffineTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"DiagonalMatrix", "[", 
      RowBox[{"{", 
       RowBox[{"2", ",", "2", ",", "1"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26146840],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"2", "0", "0", "1"},
     {"0", "2", "0", "1"},
     {"0", "0", "1", "1"},
     {"0", "0", "0", "1"}
    },
    ColumnsEqual->False,
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
    RowsEqual->False], ")"}], "]"}]], "Output",
 ImageSize->{245, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->101063720]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"GeometricTransformation", "[", 
    RowBox[{
     RowBox[{"Cylinder", "[", "]"}], ",", "t"}], "]"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->637544168],

Cell[BoxData[
 Graphics3DBox[
  InterpretationBox[GraphicsComplex3DBox[CompressedData["
1:eJztlU8oRFEUxt8gxgYLYmEjJZE/pdBs5mOFhCZJUkpkR9mIlWbDSliQRFKE
skVZvJ2UbCghJiVCahCF/O/dc+vd6dtZvlvT1NedM993zv11cjr7Qt1xlmUt
/H7iLTl+ON8PQVGKnzLrJveTcJ5+dxL2vWh9uGAxeaQiCSvP07MDl29aL5+a
TxmvSoTVt5iROfKl9eb3vf3txwSUrAfCEz0+iH49FyxrbIqHfIsu9+R32qKq
q/9HHfEhvkSPKN+SQ3SfK6df69VHhfXvH1cx97uKFiZy+o9j6vvTTpbGxjZj
/NwcrOVmDe7Ypv/S6MfMYPjUNvM23JZURHcu7VKjP8etQwEk3to+o59fq6N5
2ZF72+z/xm5Hfm151I4Y83LOg+3ObVlb6n6Nym3Wb1G5dT+VnxqVW/SQ8i+5
Ra9UeSW36K/qnuTWfUt16kpu0duVD/Flzktyi/75/XeiQckt+pnqy5Zxf1n6
uOau/93r9P3U8FOs5hQy/MtcK428+h0cuvsj7+Yi1d1PeWdtRv/lXVYZ83JO
DL8g/ILwC8IvCL8g/ILwC8IvCL8g/ILwC8IvCL8g/ILwC8IvCL8g/ILwC8Kv
zNXkF4RfEH5B+AXhF4RfEH5B+AXhF4RfEH5B+AXhF4RfEH5B+AXhF4RfEH5B
+FXz9favt3/d9b396+1fy9u/Wvf2r6N7+9fR/2v//gAU2sSb
    "], {
     {EdgeForm[None], 
      Polygon3DBox[{{1, 2, 42, 41}, {2, 3, 43, 42}, {3, 4, 44, 43}, {4, 5, 45,
        44}, {5, 6, 46, 45}, {6, 7, 47, 46}, {7, 8, 48, 47}, {8, 9, 49, 48}, {
       9, 10, 50, 49}, {10, 11, 51, 50}, {11, 12, 52, 51}, {12, 13, 53, 52}, {
       13, 14, 54, 53}, {14, 15, 55, 54}, {15, 16, 56, 55}, {16, 17, 57, 
       56}, {17, 18, 58, 57}, {18, 19, 59, 58}, {19, 20, 60, 59}, {20, 21, 61,
        60}, {21, 22, 62, 61}, {22, 23, 63, 62}, {23, 24, 64, 63}, {24, 25, 
       65, 64}, {25, 26, 66, 65}, {26, 27, 67, 66}, {27, 28, 68, 67}, {28, 29,
        69, 68}, {29, 30, 70, 69}, {30, 31, 71, 70}, {31, 32, 72, 71}, {32, 
       33, 73, 72}, {33, 34, 74, 73}, {34, 35, 75, 74}, {35, 36, 76, 75}, {36,
        37, 77, 76}, {37, 38, 78, 77}, {38, 39, 79, 78}, {39, 40, 80, 79}, {
       40, 1, 41, 80}}]}, 
     Polygon3DBox[{{120, 119, 118, 117, 116, 115, 114, 113, 112, 111, 110, 
      109, 108, 107, 106, 105, 104, 103, 102, 101, 100, 99, 98, 97, 96, 95, 
      94, 93, 92, 91, 90, 89, 88, 87, 86, 85, 84, 83, 82, 81}, {121, 122, 123,
       124, 125, 126, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 
      138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 
      152, 153, 154, 155, 156, 157, 158, 159, 160}}]},
    VertexColors->Automatic,
    VertexNormals->CompressedData["
1:eJztkztIA0EQhtckKCgEQVtBUWwUU1iIqGyKlMFH4jUaHwhq43k2IQRBi3TW
gkUgxEaT0laETWWjiY2taWJjIYKIIkhOcec/uAH7CDsQjnzs7f3fzO7AhpPY
DAghij+/oPAqKlilHiPHQ3lXppOl7a72ewnuLsZ6wk5Tzj1FJl6uGx63R2f6
+nNfUtRjI2PhZ4+X3k7ymcanXF1KhwbvXj1enTrcKhTf5S09wbEO74G30b47
+A7VPOVALvAM5YYHeEeqd3z47HSae+/r9WqZrZ/V+yth+fd3H37zKJvlWdH5
Fc9PnqrKfM/1OrXG+kOeKsD6SZ5qgfWfcqssmxdVRXtfwBvzVXvkzfe34E3l
UJ4P8gYvU354g9fIF97g5KngDR7Ungre4AnKgVx/zMvjnUeF+GT28gre4Lv+
8+Ctj6OPln//pvaUNsuTojmVWf4bmmuN+ZKnXGf9wTkO1f39xDlOsv7jXB6w
eVFF2X9zf6nM/dVl7i/1rTXvL6vuiuGGG/5vedRwww1vff4Nn2uBJQ==
     "]],
   GeometricTransformation[
    Cylinder[{{0, 0, -1}, {0, 0, 1}}], {{{2, 0, 0}, {0, 2, 0}, {0, 0, 1}}, {1,
      1, 1}}]],
  Axes->True]], "Output",
 ImageSize->{184, 155},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->316657407,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->255250272],

Cell["Integrate a function over a rhombic region:", "ExampleText",
 CellID->7367196],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "3"}], "+", 
      RowBox[{"y", "^", "3"}]}], ")"}], 
    RowBox[{"Boole", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "/", "2"}], "<", "y", "<", 
       RowBox[{
        RowBox[{"x", "/", "2"}], "+", 
        RowBox[{"3", "/", "4"}]}]}], "&&", 
      RowBox[{
       RowBox[{"y", "/", "2"}], "<", "x", "<", 
       RowBox[{
        RowBox[{"y", "/", "2"}], "+", 
        RowBox[{"3", "/", "4"}]}]}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"3", "/", "2"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", 
     RowBox[{"3", "/", "2"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->218345391],

Cell[BoxData[
 FractionBox["63", "64"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15000325]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " defines a change of variables that maps the unit square to the integration \
region:"
}], "ExampleText",
 CellID->35615684],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"AffineTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"1", "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", "2"}], ",", "1"}], "}"}]}], "}"}], "]"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->995049490],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {"1", 
      FractionBox["1", "2"], "0"},
     {
      FractionBox["1", "2"], "1", "0"},
     {"0", "0", "1"}
    },
    ColumnsEqual->False,
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
    RowsEqual->False], ")"}], "]"}]], "Output",
 ImageSize->{238, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->55428464]
}, Open  ]],

Cell["The integrand in the new coordinates:", "ExampleText",
 CellID->112927875],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fuv", "=", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "3"}], "+", 
    RowBox[{"y", "^", "3"}]}], "/.", 
   RowBox[{"Thread", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "y"}], "}"}], "\[Rule]", 
     RowBox[{"t", "[", 
      RowBox[{"{", 
       RowBox[{"u", ",", "v"}], "}"}], "]"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->373886528],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"u", "+", 
     FractionBox["v", "2"]}], ")"}], "3"], "+", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     FractionBox["u", "2"], "+", "v"}], ")"}], "3"]}]], "Output",
 ImageSize->{118, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->332968263]
}, Open  ]],

Cell["The jacobian:", "ExampleText",
 CellID->13069923],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"jac", "=", 
  RowBox[{"Det", "[", 
   RowBox[{"TransformationMatrix", "[", "t", "]"}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->100305527],

Cell[BoxData[
 FractionBox["3", "4"]], "Output",
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->215846629]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"fuv", " ", 
    RowBox[{"Abs", "[", "jac", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"u", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"v", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1479640],

Cell[BoxData[
 FractionBox["63", "64"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->49578928]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "Find the ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " power of a transformation:"
}], "ExampleText",
 CellID->113940733],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "3"}], ",", "2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", 
       RowBox[{"-", "2"}], ",", 
       RowBox[{"-", "4"}]}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->80011183],

Cell[BoxData[
 RowBox[{"TransformationFunction", "[", 
  RowBox[{"(", GridBox[{
     {
      RowBox[{"-", "3"}], "2", "1"},
     {"0", "1", "1"},
     {"4", 
      RowBox[{"-", "2"}], 
      RowBox[{"-", "4"}]}
    },
    ColumnsEqual->False,
    GridBoxDividers->{
     "ColumnsIndexed" -> {-2 -> True}, "RowsIndexed" -> {-2 -> True}},
    GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
    RowsEqual->False], ")"}], "]"}]], "Output",
 ImageSize->{254, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->236226190]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"tt", "[", "n_", "]"}], ":=", 
  RowBox[{"LinearFractionalTransform", "[", 
   RowBox[{"MatrixPower", "[", 
    RowBox[{
     RowBox[{"TransformationMatrix", "[", "t", "]"}], ",", "n"}], "]"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->892656215],

Cell[TextData[{
 "Apply ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " five times:"
}], "ExampleText",
 CellID->261361020],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Nest", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Together", "@", 
     RowBox[{"t", "@", "#"}]}], "&"}], ",", 
   RowBox[{"{", 
    RowBox[{"x0", ",", "y0"}], "}"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->29496898],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"691", "-", 
     RowBox[{"1403", " ", "x0"}], "+", 
     RowBox[{"702", " ", "y0"}]}], 
    RowBox[{"2", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1047"}], "+", 
       RowBox[{"2062", " ", "x0"}], "-", 
       RowBox[{"1031", " ", "y0"}]}], ")"}]}]], ",", 
   FractionBox[
    RowBox[{"351", "-", 
     RowBox[{"680", " ", "x0"}], "+", 
     RowBox[{"341", " ", "y0"}]}], 
    RowBox[{"2", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1047"}], "+", 
       RowBox[{"2062", " ", "x0"}], "-", 
       RowBox[{"1031", " ", "y0"}]}], ")"}]}]]}], "}"}]], "Output",
 ImageSize->{388, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->434650940]
}, Open  ]],

Cell[TextData[{
 "Apply ",
 Cell[BoxData[
  RowBox[{"tt", "[", "5", "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->146483248],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"tt", "[", "5", "]"}], "[", 
  RowBox[{"{", 
   RowBox[{"x0", ",", "y0"}], "}"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->424754096],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"691", "-", 
     RowBox[{"1403", " ", "x0"}], "+", 
     RowBox[{"702", " ", "y0"}]}], 
    RowBox[{
     RowBox[{"-", "2094"}], "+", 
     RowBox[{"4124", " ", "x0"}], "-", 
     RowBox[{"2062", " ", "y0"}]}]], ",", 
   FractionBox[
    RowBox[{"351", "-", 
     RowBox[{"680", " ", "x0"}], "+", 
     RowBox[{"341", " ", "y0"}]}], 
    RowBox[{
     RowBox[{"-", "2094"}], "+", 
     RowBox[{"4124", " ", "x0"}], "-", 
     RowBox[{"2062", " ", "y0"}]}]]}], "}"}]], "Output",
 ImageSize->{340, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->34758618]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"%", "\[Equal]", "%%"}], "//", "Simplify"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1006127556],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->4310994]
}, Open  ]],

Cell[TextData[{
 "Find the ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " iteration using ",
 Cell[BoxData[
  ButtonBox["RSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RSolve"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->695192784],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RSolve", "[", 
  RowBox[{
   RowBox[{"Thread", "/@", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"x", "[", 
          RowBox[{"n", "+", "1"}], "]"}], ",", 
         RowBox[{"y", "[", 
          RowBox[{"n", "+", "1"}], "]"}]}], "}"}], "\[Equal]", 
       RowBox[{"t", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"x", "[", "n", "]"}], ",", 
          RowBox[{"y", "[", "n", "]"}]}], "}"}], "]"}]}], ",", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"x", "[", "0", "]"}], ",", 
         RowBox[{"y", "[", "0", "]"}]}], "}"}], "\[Equal]", 
       RowBox[{"{", 
        RowBox[{"x0", ",", "y0"}], "}"}]}]}], "}"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", "n"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->69307435],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{577, 132},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->108595767]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "[", "5", "]"}], ",", 
     RowBox[{"y", "[", "5", "]"}]}], "}"}], "/.", 
   RowBox[{"%", "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->324286835],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{"691", "-", 
     RowBox[{"1403", " ", "x0"}], "+", 
     RowBox[{"702", " ", "y0"}]}], 
    RowBox[{
     RowBox[{"-", "2094"}], "+", 
     RowBox[{"4124", " ", "x0"}], "-", 
     RowBox[{"2062", " ", "y0"}]}]], ",", 
   FractionBox[
    RowBox[{"351", "-", 
     RowBox[{"680", " ", "x0"}], "+", 
     RowBox[{"341", " ", "y0"}]}], 
    RowBox[{
     RowBox[{"-", "2094"}], "+", 
     RowBox[{"4124", " ", "x0"}], "-", 
     RowBox[{"2062", " ", "y0"}]}]]}], "}"}]], "Output",
 ImageSize->{340, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->15874796]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["TransformationMatrix",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TransformationMatrix"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricTransformation",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricTransformation"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AffineTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AffineTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RotationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RotationTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TranslationTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TranslationTransform"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ScalingTransform",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ScalingTransform"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->30587104],

Cell[TextData[ButtonBox["Geometric Transforms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/GeometricTransforms"]], "MoreAbout",
 CellID->512216906]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"TransformationFunction - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 27, 35.2869288}", "context" -> "System`", 
    "keywords" -> {
     "affine transformation", "linear transformation", 
      "linear fractional transformation", "perspective transformation", 
      "rotation transformation", "scaling transformation", 
      "translation transformation", "shearing transformation"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "TransformationFunction[data] represents a transformation function that \
applies geometric and other transformations.", 
    "synonyms" -> {"transformation function"}, "title" -> 
    "TransformationFunction", "type" -> "Symbol", "uri" -> 
    "ref/TransformationFunction"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8082, 253, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->149624750]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 89036, 2849}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1905, 43, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2528, 70, 1587, 37, 70, "ObjectNameGrid"],
Cell[4118, 109, 566, 16, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[4721, 130, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5199, 147, 337, 12, 70, "Notes",
 CellID->1067943069],
Cell[5539, 161, 304, 9, 70, "Notes",
 CellID->288523798],
Cell[5846, 172, 247, 7, 70, "Notes",
 CellID->207185040],
Cell[6096, 181, 566, 19, 70, "Notes",
 CellID->209660300],
Cell[6665, 202, 491, 17, 70, "Notes",
 CellID->713987337],
Cell[7159, 221, 437, 13, 70, "Notes",
 CellID->303436182],
Cell[7599, 236, 446, 12, 70, "Notes",
 CellID->71279388]
}, Closed]],
Cell[CellGroupData[{
Cell[8082, 253, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->149624750],
Cell[CellGroupData[{
Cell[8467, 267, 148, 5, 70, "ExampleSection",
 CellID->226357251],
Cell[8618, 274, 71, 1, 70, "ExampleText",
 CellID->980191149],
Cell[CellGroupData[{
Cell[8714, 279, 143, 4, 28, "Input",
 CellID->247847466],
Cell[8860, 285, 617, 19, 64, "Output",
 CellID->426315870]
}, Open  ]],
Cell[9492, 307, 294, 12, 70, "ExampleText",
 CellID->279787753],
Cell[CellGroupData[{
Cell[9811, 323, 144, 5, 28, "Input",
 CellID->509669506],
Cell[9958, 330, 512, 17, 36, "Output",
 CellID->535224651]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10519, 353, 222, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[CellGroupData[{
Cell[10766, 364, 257, 7, 70, "ExampleSubsection",
 CellID->557287772],
Cell[11026, 373, 253, 10, 70, "ExampleText",
 CellID->200115988],
Cell[CellGroupData[{
Cell[11304, 387, 200, 6, 70, "Input",
 CellID->42025119],
Cell[11507, 395, 475, 15, 80, "Output",
 CellID->230337727]
}, Open  ]],
Cell[CellGroupData[{
Cell[12019, 415, 153, 5, 70, "Input",
 CellID->49073389],
Cell[12175, 422, 289, 10, 36, "Output",
 CellID->193636383]
}, Open  ]],
Cell[12479, 435, 124, 3, 70, "ExampleDelimiter",
 CellID->84662931],
Cell[12606, 440, 148, 6, 70, "ExampleText",
 CellID->371074523],
Cell[CellGroupData[{
Cell[12779, 450, 227, 7, 70, "Input",
 CellID->28155084],
Cell[13009, 459, 659, 20, 80, "Output",
 CellID->367507988]
}, Open  ]],
Cell[CellGroupData[{
Cell[13705, 484, 154, 5, 70, "Input",
 CellID->655728535],
Cell[13862, 491, 521, 17, 36, "Output",
 CellID->70253597]
}, Open  ]],
Cell[14398, 511, 125, 3, 70, "ExampleDelimiter",
 CellID->491954993],
Cell[14526, 516, 77, 1, 70, "ExampleText",
 CellID->309971784],
Cell[CellGroupData[{
Cell[14628, 521, 160, 5, 70, "Input",
 CellID->32236626],
Cell[14791, 528, 431, 14, 64, "Output",
 CellID->15022460]
}, Open  ]],
Cell[15237, 545, 125, 3, 70, "ExampleDelimiter",
 CellID->127343740],
Cell[15365, 550, 228, 9, 70, "ExampleText",
 CellID->21553636],
Cell[CellGroupData[{
Cell[15618, 563, 256, 8, 70, "Input",
 CellID->121015059],
Cell[15877, 573, 471, 15, 64, "Output",
 CellID->554285227]
}, Open  ]],
Cell[16363, 591, 125, 3, 70, "ExampleDelimiter",
 CellID->388551663],
Cell[16491, 596, 189, 8, 70, "ExampleText",
 CellID->445521037],
Cell[CellGroupData[{
Cell[16705, 608, 171, 5, 70, "Input",
 CellID->32545861],
Cell[16879, 615, 494, 16, 80, "Output",
 CellID->495409537]
}, Open  ]],
Cell[17388, 634, 125, 3, 70, "ExampleDelimiter",
 CellID->155134094],
Cell[17516, 639, 367, 14, 70, "ExampleText",
 CellID->225612157],
Cell[CellGroupData[{
Cell[17908, 657, 280, 9, 70, "Input",
 CellID->640460778],
Cell[18191, 668, 720, 24, 86, "Output",
 CellID->201042413]
}, Open  ]],
Cell[18926, 695, 125, 3, 70, "ExampleDelimiter",
 CellID->375521382],
Cell[19054, 700, 220, 8, 70, "ExampleText",
 CellID->68715309],
Cell[CellGroupData[{
Cell[19299, 712, 397, 12, 70, "Input",
 CellID->461959536],
Cell[19699, 726, 430, 14, 64, "Output",
 CellID->458328840]
}, Open  ]],
Cell[CellGroupData[{
Cell[20166, 745, 144, 5, 70, "Input",
 CellID->200816312],
Cell[20313, 752, 570, 20, 53, "Output",
 CellID->648750301]
}, Open  ]],
Cell[20898, 775, 124, 3, 70, "ExampleDelimiter",
 CellID->11624954],
Cell[21025, 780, 67, 1, 70, "ExampleText",
 CellID->173676943],
Cell[CellGroupData[{
Cell[21117, 785, 376, 13, 70, "Input",
 CellID->139179960],
Cell[21496, 800, 646, 22, 66, "Output",
 CellID->393458163]
}, Open  ]],
Cell[CellGroupData[{
Cell[22179, 827, 144, 5, 70, "Input",
 CellID->97109209],
Cell[22326, 834, 560, 21, 36, "Output",
 CellID->385381910]
}, Open  ]],
Cell[22901, 858, 125, 3, 70, "ExampleDelimiter",
 CellID->184015676],
Cell[23029, 863, 68, 1, 70, "ExampleText",
 CellID->726939751],
Cell[CellGroupData[{
Cell[23122, 868, 552, 18, 70, "Input",
 CellID->457458540],
Cell[23677, 888, 698, 24, 66, "Output",
 CellID->361809556]
}, Open  ]],
Cell[CellGroupData[{
Cell[24412, 917, 144, 5, 70, "Input",
 CellID->73542158],
Cell[24559, 924, 628, 23, 36, "Output",
 CellID->123841534]
}, Open  ]],
Cell[25202, 950, 125, 3, 70, "ExampleDelimiter",
 CellID->697472489],
Cell[25330, 955, 77, 1, 70, "ExampleText",
 CellID->77138904],
Cell[CellGroupData[{
Cell[25432, 960, 694, 23, 70, "Input",
 CellID->89629318],
Cell[26129, 985, 750, 26, 66, "Output",
 CellID->650851166]
}, Open  ]],
Cell[CellGroupData[{
Cell[26916, 1016, 147, 5, 70, "Input",
 CellID->619904342],
Cell[27066, 1023, 965, 35, 53, "Output",
 CellID->57595511]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[28080, 1064, 270, 7, 70, "ExampleSubsection",
 CellID->112582735],
Cell[28353, 1073, 335, 13, 70, "ExampleText",
 CellID->223549394],
Cell[CellGroupData[{
Cell[28713, 1090, 251, 8, 70, "Input",
 CellID->109016475],
Cell[28967, 1100, 589, 16, 80, "Output",
 CellID->18942656]
}, Open  ]],
Cell[29571, 1119, 151, 6, 70, "ExampleText",
 CellID->1829689],
Cell[CellGroupData[{
Cell[29747, 1129, 154, 5, 70, "Input",
 CellID->254950545],
Cell[29904, 1136, 255, 8, 36, "Output",
 CellID->221024316]
}, Open  ]],
Cell[30174, 1147, 77, 1, 70, "ExampleText",
 CellID->49040006],
Cell[30254, 1150, 227, 8, 70, "Input",
 CellID->25698720],
Cell[CellGroupData[{
Cell[30506, 1162, 97, 3, 70, "Input",
 CellID->17490611],
Cell[30606, 1167, 1663, 47, 105, "Output",
 CellID->619971542]
}, Open  ]],
Cell[32284, 1217, 125, 3, 70, "ExampleDelimiter",
 CellID->268855721],
Cell[32412, 1222, 73, 1, 70, "ExampleText",
 CellID->821959473],
Cell[CellGroupData[{
Cell[32510, 1227, 144, 4, 70, "Input",
 CellID->524410875],
Cell[32657, 1233, 617, 19, 64, "Output",
 CellID->213297444]
}, Open  ]],
Cell[CellGroupData[{
Cell[33311, 1257, 142, 4, 70, "Input",
 CellID->715708731],
Cell[33456, 1263, 609, 19, 64, "Output",
 CellID->580710421]
}, Open  ]],
Cell[CellGroupData[{
Cell[34102, 1287, 163, 5, 70, "Input",
 CellID->79553851],
Cell[34268, 1294, 749, 23, 64, "Output",
 CellID->38355828]
}, Open  ]],
Cell[35032, 1320, 125, 3, 70, "ExampleDelimiter",
 CellID->167718222],
Cell[35160, 1325, 65, 1, 70, "ExampleText",
 CellID->716032382],
Cell[CellGroupData[{
Cell[35250, 1330, 143, 4, 70, "Input",
 CellID->471611810],
Cell[35396, 1336, 615, 19, 64, "Output",
 CellID->8183075]
}, Open  ]],
Cell[CellGroupData[{
Cell[36048, 1360, 168, 6, 70, "Input",
 CellID->28804467],
Cell[36219, 1368, 616, 19, 64, "Output",
 CellID->13923555]
}, Open  ]],
Cell[36850, 1390, 72, 1, 70, "ExampleText",
 CellID->382240035],
Cell[CellGroupData[{
Cell[36947, 1395, 163, 5, 70, "Input",
 CellID->455174583],
Cell[37113, 1402, 429, 14, 64, "Output",
 CellID->73980967]
}, Open  ]],
Cell[37557, 1419, 125, 3, 70, "ExampleDelimiter",
 CellID->317141207],
Cell[37685, 1424, 353, 13, 70, "ExampleText",
 CellID->8630988],
Cell[CellGroupData[{
Cell[38063, 1441, 399, 12, 70, "Input",
 CellID->428657653],
Cell[38465, 1455, 430, 14, 64, "Output",
 CellID->247091727]
}, Open  ]],
Cell[CellGroupData[{
Cell[38932, 1474, 183, 6, 70, "Input",
 CellID->374734954],
Cell[39118, 1482, 1993, 37, 70, "Output",
 Evaluatable->False,
 CellID->188814395]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[41160, 1525, 268, 7, 70, "ExampleSubsection",
 CellID->93723694],
Cell[41431, 1534, 76, 1, 70, "ExampleText",
 CellID->232691274],
Cell[CellGroupData[{
Cell[41532, 1539, 496, 16, 70, "Input",
 CellID->215996710],
Cell[42031, 1557, 525, 16, 64, "Output",
 CellID->551516433]
}, Open  ]],
Cell[42571, 1576, 77, 1, 70, "ExampleText",
 CellID->75030959],
Cell[CellGroupData[{
Cell[42673, 1581, 169, 6, 70, "Input",
 CellID->136503702],
Cell[42845, 1589, 517, 18, 53, "Output",
 CellID->84702603]
}, Open  ]],
Cell[43377, 1610, 56, 1, 70, "ExampleText",
 CellID->202702190],
Cell[CellGroupData[{
Cell[43458, 1615, 169, 6, 70, "Input",
 CellID->170276240],
Cell[43630, 1623, 1153, 41, 54, "Output",
 CellID->587440101]
}, Open  ]],
Cell[44798, 1667, 51, 1, 70, "ExampleText",
 CellID->181871226],
Cell[CellGroupData[{
Cell[44874, 1672, 153, 5, 70, "Input",
 CellID->76736110],
Cell[45030, 1679, 425, 16, 53, "Output",
 CellID->3887371]
}, Open  ]],
Cell[45470, 1698, 55, 1, 70, "ExampleText",
 CellID->792324328],
Cell[CellGroupData[{
Cell[45550, 1703, 128, 4, 70, "Input",
 CellID->473906299],
Cell[45681, 1709, 868, 30, 51, "Output",
 CellID->221234707]
}, Open  ]],
Cell[46564, 1742, 50, 1, 70, "ExampleText",
 CellID->355523943],
Cell[CellGroupData[{
Cell[46639, 1747, 569, 19, 70, "Input",
 CellID->71032048],
Cell[47211, 1768, 11243, 188, 70, "Output",
 Evaluatable->False,
 CellID->2476398]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[58515, 1963, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[58745, 1972, 390, 12, 70, "ExampleText",
 CellID->97803277],
Cell[CellGroupData[{
Cell[59160, 1988, 354, 11, 70, "Input",
 CellID->26146840],
Cell[59517, 2001, 567, 17, 80, "Output",
 CellID->101063720]
}, Open  ]],
Cell[CellGroupData[{
Cell[60121, 2023, 274, 8, 70, "Input",
 CellID->637544168],
Cell[60398, 2033, 9534, 163, 176, 2861, 49, "CachedBoxData", "BoxData", \
"Output",
 CellID->316657407]
}, Open  ]],
Cell[69947, 2199, 125, 3, 70, "ExampleDelimiter",
 CellID->255250272],
Cell[70075, 2204, 84, 1, 70, "ExampleText",
 CellID->7367196],
Cell[CellGroupData[{
Cell[70184, 2209, 795, 27, 70, "Input",
 CellID->218345391],
Cell[70982, 2238, 179, 6, 51, "Output",
 CellID->15000325]
}, Open  ]],
Cell[71176, 2247, 206, 6, 70, "ExampleText",
 CellID->35615684],
Cell[CellGroupData[{
Cell[71407, 2257, 347, 13, 70, "Input",
 CellID->995049490],
Cell[71757, 2272, 574, 18, 82, "Output",
 CellID->55428464]
}, Open  ]],
Cell[72346, 2293, 80, 1, 70, "ExampleText",
 CellID->112927875],
Cell[CellGroupData[{
Cell[72451, 2298, 396, 14, 70, "Input",
 CellID->373886528],
Cell[72850, 2314, 375, 14, 51, "Output",
 CellID->332968263]
}, Open  ]],
Cell[73240, 2331, 55, 1, 70, "ExampleText",
 CellID->13069923],
Cell[CellGroupData[{
Cell[73320, 2336, 172, 5, 70, "Input",
 CellID->100305527],
Cell[73495, 2343, 178, 6, 51, "Output",
 CellID->215846629]
}, Open  ]],
Cell[CellGroupData[{
Cell[73710, 2354, 314, 10, 70, "Input",
 CellID->1479640],
Cell[74027, 2366, 179, 6, 51, "Output",
 CellID->49578928]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[74255, 2378, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[74496, 2387, 245, 9, 70, "ExampleText",
 CellID->113940733],
Cell[CellGroupData[{
Cell[74766, 2400, 460, 15, 70, "Input",
 CellID->80011183],
Cell[75229, 2417, 591, 19, 64, "Output",
 CellID->236226190]
}, Open  ]],
Cell[75835, 2439, 293, 9, 70, "Input",
 CellID->892656215],
Cell[76131, 2450, 145, 6, 70, "ExampleText",
 CellID->261361020],
Cell[CellGroupData[{
Cell[76301, 2460, 260, 9, 70, "Input",
 CellID->29496898],
Cell[76564, 2471, 795, 27, 53, "Output",
 CellID->434650940]
}, Open  ]],
Cell[77374, 2501, 137, 6, 70, "ExampleText",
 CellID->146483248],
Cell[CellGroupData[{
Cell[77536, 2511, 175, 6, 70, "Input",
 CellID->424754096],
Cell[77714, 2519, 676, 23, 53, "Output",
 CellID->34758618]
}, Open  ]],
Cell[CellGroupData[{
Cell[78427, 2547, 132, 4, 70, "Input",
 CellID->1006127556],
Cell[78562, 2553, 159, 5, 36, "Output",
 CellID->4310994]
}, Open  ]],
Cell[78736, 2561, 356, 14, 70, "ExampleText",
 CellID->695192784],
Cell[CellGroupData[{
Cell[79117, 2579, 871, 28, 70, "Input",
 CellID->69307435],
Cell[79991, 2609, 4178, 73, 70, "Output",
 Evaluatable->False,
 CellID->108595767]
}, Open  ]],
Cell[CellGroupData[{
Cell[84206, 2687, 289, 10, 70, "Input",
 CellID->324286835],
Cell[84498, 2699, 676, 23, 53, "Output",
 CellID->15874796]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[85235, 2729, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[85554, 2740, 1561, 53, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[87152, 2798, 318, 9, 70, "MoreAboutSection",
 CellID->30587104],
Cell[87473, 2809, 151, 3, 70, "MoreAbout",
 CellID->512216906]
}, Open  ]],
Cell[87639, 2815, 27, 0, 70, "History"],
Cell[87669, 2817, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

