(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     34317,       1239]
NotebookOptionsPosition[     27271,        997]
NotebookOutlinePosition[     28914,       1039]
CellTagsIndexPosition[     28828,       1034]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DescriptiveStatistics"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Descriptive Statistics\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Mean" :> Documentation`HelpLookup["paclet:ref/Mean"], "Median" :> 
          Documentation`HelpLookup["paclet:ref/Median"], "Quantile" :> 
          Documentation`HelpLookup["paclet:ref/Quantile"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Mean\"\>", 2->"\<\"Median\"\>", 
       3->"\<\"Quantile\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Descriptive Statistics" :> 
          Documentation`HelpLookup["paclet:guide/DescriptiveStatistics"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Descriptive Statistics\"\>", 
       2->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["TrimmedMean", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TrimmedMean",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TrimmedMean"], "[", 
       RowBox[{
        StyleBox["list", "TI"], ",", 
        StyleBox["f", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the mean of the elements in ",
     Cell[BoxData[
      StyleBox["list", "TI"]], "InlineFormula"],
     " after dropping a fraction ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " of the smallest and largest elements."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TrimmedMean",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TrimmedMean"], "[", 
       RowBox[{
        StyleBox["list", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["f", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["f", "TI"], 
           StyleBox["2", "TR"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the mean when a fraction ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["f", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " of the smallest elements and a fraction ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["f", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"],
     " of the largest elements are removed."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TrimmedMean",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TrimmedMean"], "[", 
       StyleBox["list", "TI"], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the 5% trimmed mean ",
     Cell[BoxData[
      RowBox[{
       ButtonBox["TrimmedMean",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TrimmedMean"], "[", 
       RowBox[{
        StyleBox["list", "TI"], ",", "0.05"}], "]"}]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->13531]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " gives a robust estimate of the mean by excluding extreme values."
}], "Notes",
 CellID->28780327],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["TrimmedMean",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TrimmedMean"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["y", "TI"], 
         StyleBox["1", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["2", "TR"]], ",", 
        SubscriptBox[
         StyleBox["y", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
    StyleBox["f", "TI"]}], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"TrimmedMean", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["f", "TI"]}], "]"}], ",", 
    RowBox[{"TrimmedMean", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["y", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["y", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["f", "TI"]}], "]"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->26555],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " works with ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->22482]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->188011125],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->219655764],

Cell["Give the trimmed mean after removing extreme values:", "ExampleText",
 CellID->330715362],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "10"}], ",", "1", ",", "1", ",", "1", ",", "1", ",", "20"}],
     "}"}], ",", "0.2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->121042669],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->438279981]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->734654086],

Cell["\<\
Give the trimmed mean after removing the smallest extreme values:\
\>", "ExampleText",
 CellID->2887753],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "10"}], ",", "1", ",", "1", ",", "1", ",", "1", ",", "20"}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.2", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->139458515],

Cell[BoxData[
 FractionBox["24", "5"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->131098013]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->32297],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " works with any real numeric quantities:"
}], "ExampleText",
 CellID->13924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], ",", "E", ",", "Pi", ",", 
     RowBox[{"Pi", "^", "2"}], ",", "1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"1", "/", "4"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19854],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "5"], " ", 
  RowBox[{"(", 
   RowBox[{"5", "+", 
    SqrtBox["2"], "+", "\[ExponentialE]", "+", "\[Pi]"}], ")"}]}]], "Output",
 ImageSize->{116, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->861386543]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24209],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " finds trimmed means of elements in each column:"
}], "ExampleText",
 CellID->15492],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"50", ",", "2"}], "}"}]}], "]"}], ",", ".1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16499],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.5172072505212677`", ",", "0.3984791893246749`"}], 
  "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->395275083]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158391064],

Cell["Obtain results at any precision: ", "ExampleText",
 CellID->160553898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"5", ",", "10", ",", "4", ",", "25", ",", "2", ",", "1"}], 
      "}"}], ",", "30"}], "]"}], ",", ".2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->190699768],

Cell[BoxData["5.25`29.999999999999996"], "Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->840474102]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->650909219],

Cell["Compute results for a large vector or matrix:", "ExampleText",
 CellID->285474028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"RandomReal", "[", 
    RowBox[{"1", ",", 
     RowBox[{"10", "^", "6"}]}], "]"}], ",", 
   RowBox[{"1", "/", "100"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->315224713],

Cell[BoxData["0.4993430749273984`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->258341714]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{"RandomReal", "[", 
   RowBox[{"1", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"10", "^", "5"}], ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->594479966],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.4991910760411483`", ",", "0.5006669677639342`", ",", 
   "0.5013745561273002`", ",", "0.5017626141048859`", ",", 
   "0.49960909276455195`"}], "}"}]], "Output",
 ImageSize->{343, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->87915023]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->26009],

Cell[TextData[{
 "Compute results for a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->625271644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sp", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], ":>", "i"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", "j_"}], "}"}], "/;", 
        RowBox[{"j", "\[Equal]", 
         RowBox[{"i", "+", "1"}]}]}], "\[RuleDelayed]", 
       RowBox[{"i", "-", "1"}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}], ",", "1"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19878761],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "17", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 17, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"100", ",", "10"}], "}"}], ",", "1"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->67140956]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", "sp", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->235263230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", 
   ",", "1", ",", "1"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->206495198]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4898],

Cell["\<\
Obtain a robust estimate of location when outliers are present:\
\>", "ExampleText",
 CellID->273453718],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"TrimmedMean", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "5", ",", "2", ",", "6", ",", "10", ",", 
      RowBox[{"10", "^", "5"}], ",", "5", ",", "4", ",", 
      RowBox[{"-", "200"}], ",", "5"}], "}"}], ",", ".1"}], "]"}], "//", 
  "N"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->128754409],

Cell[BoxData["4.75`"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->470054931]
}, Open  ]],

Cell[TextData[{
 "Extreme values have a large influence on the ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->9445300],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"{", 
    RowBox[{"1", ",", "5", ",", "2", ",", "6", ",", "10", ",", 
     RowBox[{"10", "^", "5"}], ",", "5", ",", "4", ",", 
     RowBox[{"-", "200"}], ",", "5"}], "}"}], "]"}], "//", "N"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->79767602],

Cell[BoxData["9983.8`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->151094698]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10974],

Cell[TextData[{
 "A 0% ",
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->651062907],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"Range", "[", "10", "]"}], ",", "0"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->653700463],

Cell[BoxData[
 FractionBox["11", "2"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->40527252]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"Range", "[", "10", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->307365152],

Cell[BoxData[
 FractionBox["11", "2"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->311989928]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->48899463],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " approaches ",
 Cell[BoxData[
  ButtonBox["Median",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Median"]], "InlineFormula"],
 " as ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " approaches 1/2:"
}], "ExampleText",
 CellID->40424003],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"RandomReal", "[", 
    RowBox[{
     RowBox[{"CauchyDistribution", "[", 
      RowBox[{"0", ",", "1"}], "]"}], ",", "1000"}], "]"}]}], ";"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->223020603],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"TrimmedMean", "[", 
    RowBox[{"data", ",", "f"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"f", ",", "0", ",", ".499"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->248398283],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->657518124]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Median", "[", "data", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->482252834],

Cell[BoxData["0.07228725170267342`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->78233818]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->24387],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TrimmedMean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TrimmedMean"]], "InlineFormula"],
 " requires numeric values:"
}], "ExampleText",
 CellID->9090],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", ".1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9034],

Cell[BoxData[
 RowBox[{
  RowBox[{"TrimmedMean", "::", "\<\"arg1\"\>"}], ":", 
  " ", "\<\"The first argument \\!\\({a, b, c}\\) is expected to be a numeric \
vector or matrix.\"\>"}]], "Message", "MSG",
 CellID->23019924],

Cell[BoxData[
 RowBox[{"TrimmedMean", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "0.1`"}], "]"}]], "Output",\

 ImageSize->{185, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->505430811]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->19149],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Mean",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Mean"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Median",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Median"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Quantile",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Quantile"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->1951]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->10204],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DescriptiveStatistics"]], "Tutorials",
 CellID->4063544]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Descriptive Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DescriptiveStatistics"]], "MoreAbout",
 CellID->26965052],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->158414685]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"TrimmedMean - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 27, 55.5061964}", "context" -> "System`", 
    "keywords" -> {
     "location statistic", "modified arithmetic mean", "modified mean", 
      "robust estimate", "robust location estimate", "truncated mean"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "TrimmedMean[list, f] gives the mean of the elements in list after \
dropping a fraction f of the smallest and largest elements. TrimmedMean[list, \
{f_1, f_2}] gives the mean when a fraction f_1 of the smallest elements and a \
fraction f_2 of the largest elements are removed. TrimmedMean[list] gives the \
5% trimmed mean TrimmedMean[list, 0.05].", "synonyms" -> {"trimmed mean"}, 
    "title" -> "TrimmedMean", "type" -> "Symbol", "uri" -> 
    "ref/TrimmedMean"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9277, 294, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->188011125]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 28685, 1027}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2071, 50, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2694, 77, 1576, 37, 70, "ObjectNameGrid"],
Cell[4273, 116, 2179, 68, 70, "Usage",
 CellID->13531]
}, Open  ]],
Cell[CellGroupData[{
Cell[6489, 189, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6967, 206, 241, 7, 70, "Notes",
 CellID->28780327],
Cell[7211, 215, 1701, 60, 70, "Notes",
 CellID->26555],
Cell[8915, 277, 325, 12, 70, "Notes",
 CellID->22482]
}, Closed]],
Cell[CellGroupData[{
Cell[9277, 294, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->188011125],
Cell[CellGroupData[{
Cell[9662, 308, 148, 5, 70, "ExampleSection",
 CellID->219655764],
Cell[9813, 315, 95, 1, 70, "ExampleText",
 CellID->330715362],
Cell[CellGroupData[{
Cell[9933, 320, 249, 8, 28, "Input",
 CellID->121042669],
Cell[10185, 330, 158, 5, 36, "Output",
 CellID->438279981]
}, Open  ]],
Cell[10358, 338, 125, 3, 70, "ExampleDelimiter",
 CellID->734654086],
Cell[10486, 343, 114, 3, 70, "ExampleText",
 CellID->2887753],
Cell[CellGroupData[{
Cell[10625, 350, 298, 10, 70, "Input",
 CellID->139458515],
Cell[10926, 362, 179, 6, 51, "Output",
 CellID->131098013]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11154, 374, 216, 7, 70, "ExampleSection",
 CellID->32297],
Cell[11373, 383, 219, 7, 70, "ExampleText",
 CellID->13924],
Cell[CellGroupData[{
Cell[11617, 394, 311, 9, 70, "Input",
 CellID->19854],
Cell[11931, 405, 301, 10, 51, "Output",
 CellID->861386543]
}, Open  ]],
Cell[12247, 418, 121, 3, 70, "ExampleDelimiter",
 CellID->24209],
Cell[12371, 423, 227, 7, 70, "ExampleText",
 CellID->15492],
Cell[CellGroupData[{
Cell[12623, 434, 244, 8, 70, "Input",
 CellID->16499],
Cell[12870, 444, 243, 8, 36, "Output",
 CellID->395275083]
}, Open  ]],
Cell[13128, 455, 125, 3, 70, "ExampleDelimiter",
 CellID->158391064],
Cell[13256, 460, 76, 1, 70, "ExampleText",
 CellID->160553898],
Cell[CellGroupData[{
Cell[13357, 465, 288, 9, 70, "Input",
 CellID->190699768],
Cell[13648, 476, 181, 5, 36, "Output",
 CellID->840474102]
}, Open  ]],
Cell[13844, 484, 125, 3, 70, "ExampleDelimiter",
 CellID->650909219],
Cell[13972, 489, 88, 1, 70, "ExampleText",
 CellID->285474028],
Cell[CellGroupData[{
Cell[14085, 494, 246, 8, 70, "Input",
 CellID->315224713],
Cell[14334, 504, 176, 5, 36, "Output",
 CellID->258341714]
}, Open  ]],
Cell[CellGroupData[{
Cell[14547, 514, 247, 8, 70, "Input",
 CellID->594479966],
Cell[14797, 524, 335, 10, 36, "Output",
 CellID->87915023]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15181, 540, 239, 7, 70, "ExampleSection",
 CellID->26009],
Cell[15423, 549, 211, 8, 70, "ExampleText",
 CellID->625271644],
Cell[CellGroupData[{
Cell[15659, 561, 580, 19, 70, "Input",
 CellID->19878761],
Cell[16242, 582, 493, 17, 36, "Output",
 CellID->67140956]
}, Open  ]],
Cell[CellGroupData[{
Cell[16772, 604, 108, 3, 70, "Input",
 CellID->235263230],
Cell[16883, 609, 291, 9, 36, "Output",
 CellID->206495198]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17223, 624, 222, 7, 70, "ExampleSection",
 CellID->4898],
Cell[17448, 633, 114, 3, 70, "ExampleText",
 CellID->273453718],
Cell[CellGroupData[{
Cell[17587, 640, 343, 10, 70, "Input",
 CellID->128754409],
Cell[17933, 652, 162, 5, 36, "Output",
 CellID->470054931]
}, Open  ]],
Cell[18110, 660, 218, 8, 70, "ExampleText",
 CellID->9445300],
Cell[CellGroupData[{
Cell[18353, 672, 303, 8, 70, "Input",
 CellID->79767602],
Cell[18659, 682, 164, 5, 36, "Output",
 CellID->151094698]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18872, 693, 233, 7, 70, "ExampleSection",
 CellID->10974],
Cell[19108, 702, 328, 13, 70, "ExampleText",
 CellID->651062907],
Cell[CellGroupData[{
Cell[19461, 719, 164, 5, 70, "Input",
 CellID->653700463],
Cell[19628, 726, 178, 6, 51, "Output",
 CellID->40527252]
}, Open  ]],
Cell[CellGroupData[{
Cell[19843, 737, 133, 4, 70, "Input",
 CellID->307365152],
Cell[19979, 743, 179, 6, 51, "Output",
 CellID->311989928]
}, Open  ]],
Cell[20173, 752, 124, 3, 70, "ExampleDelimiter",
 CellID->48899463],
Cell[20300, 757, 396, 15, 70, "ExampleText",
 CellID->40424003],
Cell[20699, 774, 256, 9, 70, "Input",
 CellID->223020603],
Cell[CellGroupData[{
Cell[20980, 787, 252, 8, 70, "Input",
 CellID->248398283],
Cell[21235, 797, 2226, 41, 70, "Output",
 Evaluatable->False,
 CellID->657518124]
}, Open  ]],
Cell[CellGroupData[{
Cell[23498, 843, 105, 3, 70, "Input",
 CellID->482252834],
Cell[23606, 848, 176, 5, 36, "Output",
 CellID->78233818]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[23831, 859, 226, 7, 70, "ExampleSection",
 CellID->24387],
Cell[24060, 868, 203, 7, 70, "ExampleText",
 CellID->9090],
Cell[CellGroupData[{
Cell[24288, 879, 185, 6, 70, "Input",
 CellID->9034],
Cell[24476, 887, 222, 5, 70, "Message",
 CellID->23019924],
Cell[24701, 894, 283, 10, 36, "Output",
 CellID->505430811]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[25045, 911, 311, 9, 70, "SeeAlsoSection",
 CellID->19149],
Cell[25359, 922, 683, 26, 70, "SeeAlso",
 CellID->1951]
}, Open  ]],
Cell[CellGroupData[{
Cell[26079, 953, 314, 9, 70, "TutorialsSection",
 CellID->10204],
Cell[26396, 964, 156, 3, 70, "Tutorials",
 CellID->4063544]
}, Open  ]],
Cell[CellGroupData[{
Cell[26589, 972, 299, 8, 70, "MoreAboutSection"],
Cell[26891, 982, 154, 3, 70, "MoreAbout",
 CellID->26965052],
Cell[27048, 987, 151, 3, 70, "MoreAbout",
 CellID->158414685]
}, Open  ]],
Cell[27214, 993, 27, 0, 70, "History"],
Cell[27244, 995, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

