(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    177262,       4800]
NotebookOptionsPosition[    165403,       4406]
NotebookOutlinePosition[    167284,       4451]
CellTagsIndexPosition[    167197,       4446]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"CellularAutomaton" :> 
          Documentation`HelpLookup["paclet:ref/CellularAutomaton"], 
          "ArrayPlot" :> Documentation`HelpLookup["paclet:ref/ArrayPlot"]}, #,
          2]& , Evaluator -> Automatic], {1->"\<\"CellularAutomaton\"\>", 
       2->"\<\"ArrayPlot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Computational Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/ComputationalSystemsAndDiscovery"], 
          "Discrete Mathematics" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteMathematics"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Computational Systems\"\>", 
       2->"\<\"Discrete Mathematics\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["TuringMachine", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TuringMachine",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TuringMachine"], "[", 
       RowBox[{
        StyleBox["rule", "TI"], ",", 
        StyleBox["init", "TI"], ",", 
        StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]generates a list representing the evolution of the \
Turing machine with the specified rule from initial condition ",
     Cell[BoxData[
      StyleBox["init", "TI"]], "InlineFormula"],
     " for ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     " steps. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["TuringMachine",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/TuringMachine"], "[", 
       RowBox[{
        StyleBox["rule", "TI"], ",", 
        StyleBox["init", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the result of evolving ",
     Cell[BoxData[
      StyleBox["init", "TI"]], "InlineFormula"],
     " for one step. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->31571]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "For a 1D Turing machine, each step in the evolution generated by ",
 Cell[BoxData[
  ButtonBox["TuringMachine",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TuringMachine"]], "InlineFormula"],
 " is given in the form ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      StyleBox["s", "TI"], ",", 
      StyleBox["x", "TI"], ",", 
      StyleBox["dx", "TI"]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}]], "InlineFormula"],
 ", where the head is in state ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 ", the cells on the tape have values ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["a", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 ",",
 " ",
 "the head is at position ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " relative to the ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["a", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 ", and has moved ",
 Cell[BoxData[
  StyleBox["dx", "TI"]], "InlineFormula"],
 " relative to its starting position."
}], "Notes",
 CellID->15060],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  StyleBox["dx", "TI"]], "InlineFormula"],
 " is omitted in the initial condition for a Turing machine, it is taken to \
be ",
 Cell[BoxData["0"], "InlineFormula"],
 "."
}], "Notes",
 CellID->537794123],

Cell[TextData[{
 "For a ",
 Cell[BoxData[
  StyleBox["d", "TI"]], "InlineFormula"],
 "-dimensional Turing machine, the tape is specified as a ",
 Cell[BoxData[
  StyleBox["d", "TI"]], "InlineFormula"],
 "-dimensional array, and the position ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and relative position ",
 Cell[BoxData[
  StyleBox["dx", "TI"]], "InlineFormula"],
 " are length-",
 Cell[BoxData[
  StyleBox["d", "TI"]], "InlineFormula"],
 " lists."
}], "Notes",
 CellID->206445663],

Cell[TextData[{
 "The rule for a Turing machine can be given as a list of replacements of the \
form ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SubscriptBox[
      StyleBox["s", "TI"], 
      StyleBox["i", "TI"]], ",", 
     SubscriptBox[
      StyleBox["a", "TI"], 
      StyleBox["i", "TI"]]}], "}"}], "->", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox[
      StyleBox["sp", "TI"], 
      StyleBox["i", "TI"]], ",", 
     SubscriptBox[
      StyleBox["ap", "TI"], 
      StyleBox["i", "TI"]], ",", 
     SubscriptBox[
      StyleBox["off", "TI"], 
      StyleBox["i", "TI"]]}], "}"}]}]], "InlineFormula"],
 ", with elements as follows:"
}], "Notes",
 CellID->1716],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    SubscriptBox[
     StyleBox["s", "TI"], 
     StyleBox["i", "TI"]], Cell["state of the head", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    SubscriptBox[
     StyleBox["a", "TI"], 
     StyleBox["i", "TI"]], Cell[
    "value of cell under the head", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    SubscriptBox[
     StyleBox["sp", "TI"], 
     StyleBox["i", "TI"]], Cell["new state of the head", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    SubscriptBox[
     StyleBox["ap", "TI"], 
     StyleBox["i", "TI"]], Cell[
    "new value of cell under the head", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    SubscriptBox[
     StyleBox["off", "TI"], 
     StyleBox["i", "TI"]], Cell[
    "offset by which the head moves", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->14597],

Cell["\<\
The states and cell values can be integers, patterns, or any other \
expressions. Individual cell values cannot be lists.\
\>", "Notes",
 CellID->13188],

Cell[TextData[{
 "In one dimension, each offset ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["off", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " is a single integer; in higher dimensions a list of integers. "
}], "Notes",
 CellID->7716],

Cell[TextData[{
 "When the states and cell values are taken to be integers in the range 1 to \
",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 " and 0 to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "-", "1"}], TraditionalForm]], "InlineMath"],
 " respectively, the following alternative forms can be given for ",
 Cell[BoxData[
  StyleBox["rule", "TI"]], "InlineFormula"],
 ": "
}], "Notes",
 CellID->25756],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    StyleBox["n", "TI"], Cell[TextData[{
     "2\[Hyphen]state, 2\[Hyphen]color machine with number ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["s", "TI"]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     "\[Hyphen]state, 2\[Hyphen]color machine with number ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["s", "TI"], ",", 
      StyleBox["k", "TI"]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     "\[Hyphen]state, ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     "\[Hyphen]color machine with number ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["s", "TI"], ",", 
      StyleBox["k", "TI"], ",", 
      StyleBox["r", "TI"]}], "}"}], Cell[TextData[{
     "allow ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["off", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " in the range ",
     Cell[BoxData[
      FormBox[
       RowBox[{"-", "r"}], TraditionalForm]], "InlineMath"],
     " to ",
     Cell[BoxData[
      FormBox[
       RowBox[{"+", "r"}], TraditionalForm]], "InlineMath"],
     " (excluding 0) "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["s", "TI"], ",", 
      StyleBox["k", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["r", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["r", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"], ",", 
        SubscriptBox[
         StyleBox["r", "TI"], 
         StyleBox["d", "TI"]]}], "}"}]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      FormBox["d", TraditionalForm]], "InlineMath"],
     "\[Hyphen]dimensional machine with ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[PlusMinus]", 
        SubscriptBox["r", "1"]}], TraditionalForm]], "InlineMath"],
     ", ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[PlusMinus]", 
        SubscriptBox["r", "2"]}], TraditionalForm]], "InlineMath"],
     ", \[Ellipsis] offsets"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["s", "TI"], ",", 
      StyleBox["k", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         SubscriptBox[
          StyleBox["off", "TI"], 
          StyleBox["1", "TR"]], "}"}], ",", 
        RowBox[{"{", 
         SubscriptBox[
          StyleBox["off", "TI"], 
          StyleBox["2", "TR"]], "}"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}], Cell[
    "machine allowing the specified explicit offsets", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    StyleBox["rule", "TI"], Cell[
    "machine with explicit rule given", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {Left, {Left}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->28],

Cell["The number of possible Turing machine rules is as follows: ", "Notes",
 CellID->6929],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], Cell[
    "2-state 2-color machines", "TableText"], Cell["4096", "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[TextData[{
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     "-state ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     "-color machines"
    }], "TableText"], Cell[TextData[{
     "(2 ",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     ")^(",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     ")"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[TextData[{
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     "-state ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     "-color range ",
     Cell[BoxData[
      StyleBox["r", "TI"]], "InlineFormula"],
     " machines"
    }], "TableText"], Cell[TextData[{
     "(2 ",
     Cell[BoxData[
      StyleBox["r", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     ")^(",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     ")"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[TextData[{
     "2D ",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     "-state ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     "-color machines"
    }], "TableText"], Cell[TextData[{
     "(4 ",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     ")^(",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     ")"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->23818108],

Cell["\<\
If the machine has no rule for the configuration it is in, its configuration \
will not be changed.\
\>", "Notes",
 CellID->878982814],

Cell[TextData[{
 "Typical forms for the initial conditions ",
 Cell[BoxData[
  StyleBox["init", "TI"]], "InlineFormula"],
 " are as follows: "
}], "Notes",
 CellID->3787],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["s", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], Cell[
    "head in state s, on a 1D tape filled with 0s", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["s", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["a", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["a", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", "0"}], "}"}]}], "}"}], 
    Cell[TextData[{
     "bounded region of values ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " on an infinite tape"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        StyleBox["s", "TI"], ",", 
        StyleBox["x", "TI"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["a", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["a", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", "0"}], "}"}]}], "}"}], 
    Cell[TextData[{
     "bounded region with the head initially at position ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        StyleBox["s", "TI"], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["a", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["b", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}]}], "}"}], Cell[
    TextData[{
     "repetitive background of value ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["b", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        StyleBox["s", "TI"], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}], Cell[
    "finite tape, assumed cyclic", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->468774850],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["TuringMachine",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TuringMachine"], "[", 
   RowBox[{
    StyleBox["rule", "TI"], ",", 
    StyleBox["init", "TI"], ",", 
    StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
 " generates an evolution list of length ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "+", "1"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->2278]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->621873397],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(4)", "ExampleCount"]
}], "ExampleSection",
 CellID->203795331],

Cell["\<\
2-state 2-color machine 2506 starting with a tape consisting of four 0s:\
\>", "ExampleText",
 CellID->1767263348],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TuringMachine", "[", 
  RowBox[{"2506", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", "2"}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->563351562],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0", ",", "0"}], "}"}]}], "}"}]}], 
  "}"}]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{515, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->146637191]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->548768179],

Cell["2-state 2-color machine 2506 with an infinite blank tape:", \
"ExampleText",
 CellID->1852004235],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TuringMachine", "[", 
  RowBox[{"2506", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "6"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1745972620],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWktu2zAQlaU4dhrUjgukTXc5Qw/RrnoIIyiQLIIWaW7WdtHfou3BVFGf
RBgPNfNISqLsGLDsocmZNzMkTT7y/fb++sPt9v7manv57m776frm6vPl2493
RVE2S5J0nSTJm4uk+J4X38p389oU70exkk7NM3uolRwR+YTIayK/Eto/A/VJ
MtVP5Sd/MHu+/pyB/mTmsWOVQ5UW4ln9+ZrRYsqpHtqu+dwoUJl6UrSngovq
RXG9ALOaqrJqas0LkZYfW8pXlnIqLxlvjThX2lmA9SWZjqGx4oDaiSUONhy2
ctc4oHYWjnHQjQ5ONmORi6IpL2fi+UN5aYPHxmlJra27bC7gVhoPNfOG3Wo3
JmmGRbDHFxlbu2409l99IuNq09XP3REmIaXjma5lnuf2+ZLW21nT7PjGaZdr
SXPbCowM1d/8l1A7VKY+amLXv3e0PR3dpywK1DfeF76Urn8lD6k8K/Wg/R/d
tdD6/IzqllMJK7pD8PVN+n0cr3Q9FfXNp6cOs4qhHlwQORPqx9RTpTnwJRiL
OHuq0W/+k1es9f3voXRPnwr1Y+qhdKVC9Z+DsYizhz7NpZV87uR3TP2Vx8Lb
3o91ANLL4u/BEi9parV5RfPvnzGe0RGd1vXWrfZtWVplS/xlW39W9y1OL4eL
8qWhcVG92nhx7Rqc6K6kYunz+sWdG4WR9u10Q5rRxz59Qv35ah7waU3fpw82
fRIu39MaLa5vqqgNxX7bTkNQtt+V/aZ4tHEY6lRoqFOAL7mmV4zB6Prw6zY+
PORpRyisrmcBPjbD896mXTUvo3x3GLaUZ7t1q2mULaW20R2SK9uN+BiGF/6e
u8wOYXbCyC7Md0WB7q/6Ybr73Cn9yF0y6ce6xZTB0CzpVDK4f2MxNJs4lUyG
5GhiymefbNtYuUUYqCzBGJV2/WZ+3uQ6RkVaa/owUBRXluiZnp9l1Jq498fs
oLujQ2N6fP2heH3vGf8yj53TWE0UpsD8oLg2Qnsbrt9OUYyVCbKVL5X2QzFB
J7U9GlftvVRtHGx+ofdlJTzVHIz2kqkxQ92/LtW4NDgxloiuGVKlVrco2T11
u0+qQVjN5vTmgtR6fL6IL+X3xKG5I2PluLHWERWdfzxm3Rm05NmfPMT8Mfx+
FcluaB7J7bRf561PriXcf/MQuR6OXRoyx+zIPHqUDye/+z2Wu+7fH9IoHouX
GjLX8u20KWUczXBItipr6UHnDxtb1eyLVznGVtF7Ulpc/8ooNnFHOahk9h//
N2J4\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{527, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->876700516]
}, Open  ]],

Cell["Plot the successive configurations of the tape:", "ExampleText",
 CellID->2548169],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Last", "/@", 
   RowBox[{"TuringMachine", "[", 
    RowBox[{"2506", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "50"}], "]"}]}],
   "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1032968966],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmFFrwjAQx1Nqt0ZBxhhMEJkfxAf3sj3tQxQZ6INsOL+wnyRepM6YtUk7
h0v/OaG9S3O//0MbL3d5K7bL93WxXS2K6eum+FyuFl/Tl48NPUoTIcQzXbuR
IF+Rd7jmdEuU0n6qx4efmNEt+Z65Ia9HXo9sVlo9zmqey5JkhUsVsnIcs8Iw
AjIn22eyhpSlRSalMY9K5tabYtLeLey1hEZK9XMtoZG5IzcwKdX5O0YkzeyI
Sto5jkmT7FdEIJJmz5jWEKx0rjRghT9SuFW/yV6YCu13K1yFdtUJK7gVzG8S
u4JZL8Wu4O90WaG5gl2Jx6xg/ldZ4VKF6nPXOBW0HUau8Mikg9TZ6x6cPO5W
yKRZ3zDpq4QRyarOB42s65WZ9J2qoJC+UzQU8iFa4k61zQ7hEu2yedhE890X
kWhWmXWD8FfS3SHcnQ8q8dSBSP0tx4FHHrMgRqS5P4UaadcDoUZOOhMxUr7V
fr0Id3a4TsTsX2bmp5hkDy6Wl1U=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{65, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->193053865]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->116126494],

Cell["\<\
\"Inject\" the state information into a representation of the tape:\
\>", "ExampleText",
 CellID->801102216],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Function", "[", 
   RowBox[{"u", ",", 
    RowBox[{"MapAt", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"[", 
           RowBox[{"1", ",", "1"}], "]"}], "]"}], ",", "#"}], "}"}], "&"}], 
      ",", 
      RowBox[{"u", "[", 
       RowBox[{"[", "2", "]"}], "]"}], ",", 
      RowBox[{"u", "[", 
       RowBox[{"[", 
        RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}]}], "]"}], "/@", 
  RowBox[{"TuringMachine", "[", 
   RowBox[{"2506", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "6"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->52041894],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "0"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "0"}], "}"}], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", "1", ",", "0", ",", "1"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "1"}], "}"}], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "0"}], "}"}], ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{483, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->507181634]
}, Open  ]],

Cell["Show the position of the head as a red square:", "ExampleText",
 CellID->226621018],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"Function", "[", 
    RowBox[{"u", ",", 
     RowBox[{"MapAt", "[", 
      RowBox[{
       RowBox[{"Red", "&"}], ",", 
       RowBox[{"u", "[", 
        RowBox[{"[", "2", "]"}], "]"}], ",", 
       RowBox[{"u", "[", 
        RowBox[{"[", 
         RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}]}], "]"}], "/@", 
   RowBox[{"TuringMachine", "[", 
    RowBox[{"2506", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "50"}], "]"}]}],
   "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->79478509],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyN08ENwyAMhWGeeu8OnaGbdIQskP1vJRcwGNs/EiiHjyfAzue6f9ertfbt
893n892HpHxVa1A+K86EaZLgCWtpt4C0QvqNIC2U0WFA2kHmzwXSFkkKCtKm
ZA1gMZTk7vAFNpnXCFbqKKNeOkuXnEjf85k0yaUcySRT6yBp5d0LeewiJ6OX
3GXSmUbmFZ+y7PZBoVT4N0UYSnL3KV1yIvMa7dIklzLqpWNmxGCalfkJpyyr
MyiUEs1ckwks724P4McfnCcHWw==
   "], {{0, 0}, {14, 51}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  ImageSize->{71.9544608223429, Automatic},
  PlotRangeClipping->True]], "Output",
 ImageSize->{76, 251},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->604171094]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1007198027],

Cell["A machine given by a set of transition rules:", "ExampleText",
 CellID->1357644820],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TuringMachine", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"state_", ",", 
        RowBox[{"color_", "?", "EvenQ"}]}], "}"}], "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"state", ",", 
        RowBox[{"color", "/", "2"}], ",", 
        RowBox[{"-", "1"}]}], "}"}]}], ",", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"state_", ",", 
        RowBox[{"color_", "?", "OddQ"}]}], "}"}], "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"state", ",", 
        RowBox[{"3", "color"}], ",", "1"}], "}"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], 
   ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17552443],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "2", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "1", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "3", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "3", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "3", ",", "9", ",", "4", ",", "5"}], "}"}]}], "}"}]}],
   "}"}]], "Output",
 ImageSize->{570, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->74312237]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "One-Dimensional Rules",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->454127739],

Cell["3-state 2-color machine 2139050:", "ExampleText",
 CellID->829560400],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Last", "/@", 
   RowBox[{"TuringMachine", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2139050", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "100"}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->473424328],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyN1EEOgkAQRNH6XsETeCWPQOLa++/EKA70b+JA2LzQlRoGuC3P+3JJ8liv
63qFcZLPMSgiRBEhighRRIgiQhQRDXkQx+MSuCpe0PYITDSUhmgoDdGQn01E
iCJCFBGiiOr+aK8P2zHIgzgel8BVR9lCaYiG4kERoogQRURDHsTxuASuur9l
F2/yII4/ISaJSWKSmCREiCJCFBGimj8+sEHnLcB4kIZqCVz1V1bpDXUd3DUi
RDV/+2doQX+LeRDH4xK46vcVrvEmD+L4E5K8jxdAoQYr
   "], {{0, 0}, {19, 101}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{47, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->231174044]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1446029959],

Cell["3-state 2-color machine 2139050, with jump offsets -1 and 2:", \
"ExampleText",
 CellID->692006509],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Last", "/@", 
   RowBox[{"TuringMachine", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2139050", ",", "3", ",", "2", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"-", "1"}], "}"}], ",", 
         RowBox[{"{", "2", "}"}]}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "100"}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->957806476],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyN0sENglAURNE7tmAFtmQJJK7tf6eYoDLz0CEhIWdxM3y4LPfrcgJuz/vM
eml3i6/HjnCipGgdkEoiWx21u+K8KKnKa6BoSbFpJUXLCSc54SQnnOSEkwbS
QNE6ILLVUTNi+7MaIlsdtbtUEtnq6H8eJznhFP+XjdXbOiJbHTUjtpduiGx1
1O4iWx1VeQ0UrdcnSFK0nHCSE04aSANF64B8mCipymugaO2O70O/rgdIfQXi

   "], {{0, 0}, {19, 101}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{47, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->38287460]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->353087814],

Cell["2-state 2-color machine 81756, with range 2: ", "ExampleText",
 CellID->1674065987],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Last", "/@", 
   RowBox[{"TuringMachine", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"16220", ",", "2", ",", "2", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "50"}], "]"}]}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1843784162],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyV0LENwzAQQ1F+T5HWM2QTj6AFsn9nwe7kDzg8gM3DQaK0j98xtiTfmc9M
CKyZaJ7eUUc96ukdddTTO+qoRz29o456ekcd9aind9RRT++oo76u357eUX9c
e9Ha/Q9/TOzw2239qt64vfXFtY9+wos3cwKTiQWT
   "], {{0, 0}, {24, 51}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  ImageSize->{75.99710820895523, Automatic},
  PlotRangeClipping->True]], "Output",
 ImageSize->{80, 157},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->464466309]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Initial Conditions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->368027935],

Cell["Place the head at position 5 on the initial tape:", "ExampleText",
 CellID->1407313371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TuringMachine", "[", 
  RowBox[{"2506", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
      "0", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
       "0", ",", "0", ",", "0"}], "}"}]}], "}"}], ",", "5"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->670809029],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW9tu00AQ3dpJCJQmAamU20PgKxCvSPDER0QVUvuAQKX/b7yOnVqns9mZ
vdlOFwmns/bezhyvd2Z2fu7ub3793t3fXu+2P+52f29ur/9tv/+5q4vKM6WK
L0qpb29V/XdV/9X87/59rf8/iHvpXF/Lw1NqBvJzkNcWGeu/8GxvI3zetz0c
v20+ofFCfUjbC42HdPzkfKyo6ftFLW7a31dwv2zLsR2s169PlZuew/5wfKb+
XeeBWjbNwzZuqlzMZuusivoyr0Usf1ZftDhnlmP9JYGOpL2FYVxLYTuhyqX9
2rSD+PjqxYSXbQ0LpZdQz8fWb5i3hpL1O6p7VfNDedOWYtyl2kJNFY5tU211
5UuPNuPddR0VR0MUEjFRpsaA7Iqh2Ri1hmCL/N21aQFXBPyunpNjwVoXIJdk
rQuyFPdZK0Fd/1JJb7bVE1GwaYjuG/HAXlcBsfZ5KpaepN+ohuHibxSP51gL
kaHXqqnz3Iad1DLi4YG10F7zwXqMPHdbz6U816MvjbPI/KYwszETV6iX9WWm
1CPGZkbHYrSdZ5nZFHY2Zi9ARu+NzeuSGZ4ZPm6G06gg4nkld+G5LcKiUep8
uLN29CXRG64yRfvcuvfb+YCxnPrdgCy1Dah59NtdtePBfcFr5jw2BhnHjb9S
Le0jFJ2eqRhZGMk30nZlqe8baYsdeZPaR/ryoUFOGnV5B/el0SOXKNKx8XX9
a+Eja0Ymz77WqRZLKF8Znk8VcRlbJMbVs68v79kaMvsq8V3v1nhOfC6VX39a
/n4/j74W9msJx/+Ob9aiFrU1OycYFdPnO+z+080PqS/bivMG8fxhtq9eyP39
GPHmrFifWHiTTJ09yBnrcFijx6C/KmRWh0SaRqY/54y3BG+OtarFbm/8pqKt
PK612t9DH7PqpFaevnxuZtThkC27Ts6W3fgsO5PFksqym8oZu/iWncmGOG6f
rI13Y9t3p3SqK5WVxznfY/JzhLL2xhhzSG3zIe6Xlud99mnTwTvc7pjCz/Qd
zTiHwxl5jTu8IuMdFG+6v0vWKDL2VSW1APvxuKvKPV7ZtwSnbAFKY445q+74
fIa2oFPjI8Wr+V4/+mpzUD7FLDtXr4B0/FKvwMxJS2PNtpNmY0mzuqaabTd0
TH4q2Y6uemn2Y+K3yM2LM1z2nU9WlVum1mlm343p5MU4fWxxRsXBdr9rQdvX
Vnv4XDzJKVPeCdVY54Bj5OKlOv0y3tPVblg375n3dyt9Rt502M7xdkqwTn8C
aWqcZ/k1vDmfLjvv6XB92NNfmeeUnD6j6enwfcgTeJntlJzZHo/tw5+CPE3O
SzkeMpevVOYcuFS5fL6nPG25erbnWHGjg2KlUTt19h8jA+Wj\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{568, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->483853477]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->330938339],

Cell["The tape can contain arbitrary expressions:", "ExampleText",
 CellID->2082457289],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Last", "/@", 
   RowBox[{"TuringMachine", "[", 
    RowBox[{"2506", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
          "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "x", ",", "0"}], 
          "}"}], ",", "0"}], "}"}]}], "}"}], ",", "40"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->299931293],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzt2sEKgkAQBuBdRCGNCCLw6At071DXOvUQIoEeoijvvWZvs81qxiakm0GY
8wc6MzXfXqp1pHZxnu4PcZ4lcbQ9x6c0Sy7R5nimpxwphFjTcQsF5Yqy4ljR
SSqlc0fXxUMs6SSfr3iUeZS5FF0jemZ9LXugoKCgoKCgoKCgoKCgoKC+VQFb
0f/35hfK1zU7EVQ1RKMIzJqN8OsrsBD+I/ISvvEp4SNevg9sxNjohGgW5m83
jqpNGFih4woTyBY5Up/sZVyl/XWOs7SbgSC7Sh2bJ2vItrsuyDK+v4eHFGLO
UuidacpOVNc7CPvpiY+oT9Z8RDi4zpmy3Rv+qdNunx9e56LXHca/cuUd/9Xp
tA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{61, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->299534920]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->849573237],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TuringMachine", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"s_", ",", "c_"}], "}"}], ":>", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"c", " ", "s"}], ",", 
       RowBox[{"c", "/", "s"}], ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], ",", "4"}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->98017843],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "2", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3", ",", "4", ",", "5"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"6", ",", "3", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       FractionBox["3", "2"], ",", "4", ",", "5"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"24", ",", "4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       FractionBox["3", "2"], ",", 
       FractionBox["2", "3"], ",", "5"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"120", ",", "1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       FractionBox["3", "2"], ",", 
       FractionBox["2", "3"], ",", 
       FractionBox["5", "24"]}], "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{519, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2035256602]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Multidimensional Rules",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->864249953],

Cell["A 2-d Turing machine:", "ExampleText",
 CellID->526230909],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TuringMachine", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"977401", ",", " ", "2", ",", " ", "2", ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"1", ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", "}"}], "}"}], ",", "0"}], "}"}]}], "}"}], ",", " ", 
   "4"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->864144494],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztnM1u2zAMx5U4X8uStOtpxz7NdtptLxAUA9rD0KHrQ21PmVlytK0MZZKS
HMs2C7Q1LYmi6L9l+WfDX46vj9++H1+fHo73n1+OPx6fHn7ef3p+qXdVM2Nm
v4wxXz+aevtUb7lf+/O7/m02m6339m/lS09mAex3wN4S5TeEP9j+g7A95Y+K
n/IP85E6vmu3T42fsiv7h8y6LZ+by3p+PzzqVaD+XaA+FhXWPhRXqH+YPWm8
uetz45EexTnrKNpay9qE+zfMeqvAfsrfvv5jzSrSH+UfU481l2D/OrCfmmNC
/mLzUZo/ON7Q8efmL59azfKv7do0MdkzBsaM9WnrrYNesNrQ6wGJids7x79s
zLKxwPzmySemlVK9UgqT51WucqlS4PUcnmtQkStg74Ft54blee5d1f9he+pc
hv7Q9cNF7vboXmosuC8qI9iI+/JCZZeXF6nSZs4PR+2SlaO0fMvKWGpU+MyA
5xW2vWX5ko67Ty9wROjZ29HMRumNOjeoscDVIsyFW0sv/tmqtelqjboKwvrw
fmtOZGz3n62z2pSVFjciXsSquqmqjqK2tpY1sXnL3sHcnsupeW53rn84/6dW
nhSX8nHdBPq/a6mP9Y+NL2d9LJ62/EnthqbaTcjl5VtK8t/aSvI52lOSPyyS
7/0tWEevL4JvR1sh8+nQCT7XnxL8S/+5CX5O9o1eG5N4MqcX2fi7pPlp2VKm
D70119ZUlk+Vc1n+mqhXPtPn+Zoe2XfXhd5JPhwrpZfyuEMcbxk/fXAzWDK5
h+VwbBS59xSCGrPqaly6oq5vsBzNgM5bqq9ALd4IcmZ+LM+8VXe43eguhc5X
JkzHYf3tqZ1eU+1DccXQeU68pdN5d3Y6Q2m80vi88XO4rtJ4pfHea4XMd7G0
eBtopzS+W39K4/PwZaXxkmyVT/qHzOQhS98Bm8vkqfuTaTH5a/L9IZF56Xuq
bW8EYuXTIg3XpBYlcS7Om/Wh1XJbZqAaqTf2VV1jVJdSelVZWyx5VKasXtXX
nfqU2OP95Sb8SuzD7ZXY522vxJ4TlxJ7an7lEHs7z1kz9os1Suzb4y3N3xiI
Pbwv89fiNEIMiZOEDce0U0rfre8hs3r/DRuKSIQYPfXeqjL6bn0NidTnebdN
eX2/vkoiXqwz/s33bKRfxFG19eurJLVJnjTaKFK+iIOVq9amozX5tTI+Z6q7
fn0NgeL7u47DiU/xPbUeO8XnxlMKxY9SDoylpbxrak35o+Knzre+x9f3UwXK
dvFdrJw5R6lLCo6RAwnlh2dsadQ+NR+h/OyijmZf7+mnUmnKP5YlzO+G2d/Q
KH1p8fGui1L15qHE3Xy1JZ4Spz8L2LD7HhOvH0LcnGPfXJchZaBaUwxN+mWc
nHQbvRMiZutLrwdWX0Mh/KXERenKrbKTZ+bcb/un3Kfn4R64GofKKkqJi7Ld
6ji7Gv3qhrqjVBWqCk+dqrAtO6pGVSNmqxpVjaWpUdrKc3WMty9M+Kv8Up6P
RkvE5VcGmL8+nj/k9E/Zrn9ncJ5ImNkf8HZFWg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{424, 84},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->198100154]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{"Cuboid", " ", "/@", " ", 
   RowBox[{"Position", "[", 
    RowBox[{
     RowBox[{"Last", " ", "/@", " ", 
      RowBox[{"TuringMachine", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"977401", ",", " ", "2", ",", " ", "2", ",", " ", 
          RowBox[{"{", 
           RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", " ", 
        RowBox[{"{", 
         RowBox[{"1", ",", " ", 
          RowBox[{"Table", "[", 
           RowBox[{"0", ",", " ", 
            RowBox[{"{", "10", "}"}], ",", " ", 
            RowBox[{"{", "10", "}"}]}], "]"}]}], "}"}], ",", " ", "30"}], 
       "]"}]}], ",", " ", "1"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1918526668],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->823141781]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1766150238],

Cell["A 2-d Turing machine specified by explicit transitions:", "ExampleText",
 CellID->1025229217],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"TuringMachine", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"2", ",", "1", ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"2", ",", "0", ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", 
            RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "0"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"4", ",", "1", ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", 
            RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "1"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"3", ",", "0", ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"3", ",", "0"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"3", ",", "1"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"2", ",", "1", ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"4", ",", "0"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"3", ",", "1", ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", 
            RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"4", ",", "1"}], "}"}], "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"2", ",", "0", ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", "0"}], "}"}]}], "}"}]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", "}"}], "}"}], ",", "0"}], "}"}]}], "}"}], ",", 
     "8000"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1370416316],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyNk90RgkAMhDd2oo+WYwk0YP9vgpefTS4HnAPj6LL5sgnP7fvZHgBe+/Xe
L2nO4udOdnzuyHDH99Cp56Wv606E5jbMsNIOrL9Y60KkQ1WJ+QD+0NSHocVN
OlaQGmIAw3cG0By1ZCSQbaFBiskSCttadfufs59sU565pX5YXpdZm8YUVuVA
9FkJohWaw9iu2TT4rDE1zQmAAc2VuiQlh1XaS6DIOhprXSubnuP5Hsy906wi
c13cGlLQpviZE5mfVwZWz+cYdTxz+kZdp33LcdZ80itCDHBVyRxMz9vhGcJT
lGzSR1mV7RYT8dVpi62UZ7IfNewF8Q==
   "], {{0, 0}, {42, 34}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  ImageSize->{125., Automatic},
  PlotRangeClipping->True]], "Output",
 ImageSize->{129, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->415828586]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(11)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["\<\
Show the evolutions of a sequence of 2-state, 2-color machines:\
\>", "ExampleText",
 CellID->32738829],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"Last", "/@", 
      RowBox[{"TuringMachine", "[", 
       RowBox[{"n", ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "40"}], 
       "]"}]}], ",", 
     RowBox[{"ImageSize", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "80"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "1950", ",", "1960"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->443064210],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{447, 169},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->130443471]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->159636520],

Cell["\<\
Trajectory of machine head from successive initial conditions:\
\>", "ExampleText",
 CellID->14145093],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"#", "[", 
       RowBox[{"[", 
        RowBox[{"1", ",", "3"}], "]"}], "]"}], "&"}], "/@", 
     RowBox[{"TuringMachine", "[", 
      RowBox[{"1507", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", 
           RowBox[{"-", "2"}], ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"IntegerDigits", "[", 
            RowBox[{"i", ",", "2", ",", "20"}], "]"}], ",", "0"}], "}"}]}], 
        "}"}], ",", "100"}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "1", ",", "100"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->528923899],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->412074242]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1350390581],

Cell["Path traced by a 2D machine:", "ExampleText",
 CellID->1654677938],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"Arrow", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"#", "[", 
      RowBox[{"[", 
       RowBox[{"1", ",", "2"}], "]"}], "]"}], "&"}], "/@", 
    RowBox[{"TuringMachine", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "0"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"2", ",", "1", ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"1", ",", "1"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"2", ",", "0", ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", 
             RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2", ",", "0"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"4", ",", "1", ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", 
             RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2", ",", "1"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"3", ",", "0", ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"3", ",", "0"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"1", ",", "1", ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"3", ",", "1"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"2", ",", "1", ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"4", ",", "0"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"3", ",", "1", ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", 
             RowBox[{"-", "1"}]}], "}"}]}], "}"}]}], ",", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"4", ",", "1"}], "}"}], "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"2", ",", "0", ",", 
           RowBox[{"{", 
            RowBox[{"1", ",", "0"}], "}"}]}], "}"}]}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"{", "}"}], "}"}], ",", "0"}], "}"}]}], "}"}], ",", 
      "200"}], "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1017714750],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmz1u2zAUx6nIiuSmRaSkATxyKeJu9mY0i6fWSwEdwULQIBmCCmkGbx09
+gg+gocOHn0Ejxp9BB9BffyAQlGWY7UW4SSPgGh+vPf/kdSX+Qx/jx5vf9xH
j3fXEf32EMW3d9e/6NefD9BkW4SQGzj+tAiUUyjx4yZN4VPkFyy34zgmIgUs
s8MwTHkiXxSr4XA4nU5Z0ef1xWLR7/dZ8Qqt0OrQrXzfB0MSsIz1yyv8crtX
p9OBnAQss9nds5OXHFHA6+iFXuhlzuuIt85ms988EZ9lNtzIugqllHUFvH+z
CrTuR0W2+v8xI1RBFVRBlRenIvcWn+D4/HJGjSqogiqoUpdKu91mW1G+KWXf
K3P7U7Dv8yTrSZJI//PtqqPRiG1Vfb5fnUwm+v51wZPsHwwGci4f9zJj3R5V
URVVURVVjajqkYowDJ/q4KWrltizSIa1XC4tsLPKCQ3w8DwPWkH6tCJDH5NV
vj7HYr6QKKVQXq1WHyquk76uW2he5gVJZZ7Ux1S8bJY1VG7TCNcpcF0jXLfA
dYxwmwWubYTLryJX5UKDECFntRD5veJkRCjUxzpVVhUSzK6eVeR1R8Res4UU
9HpYbualnjqxtvUQFa+jwrV6YoTbKHCbRrjHBa5rhOsVuI4R7rs0d4cCd71e
V1b5N/Z7vuCC3e12x+Mxl5HPh33TxLMIJtjr9cAL9qqkEisCawtaN3wz2XEE
NT6PkIIUpCAFKUg5bIqMLIvIc6vV0r1KItGZvR653kwpiTRnlOci03ok+/Wf
F6QgBSlIQcpbo1R9V+qUHd6VcRzn3uNAEdHH8vd+jjKfz6U/67dghE+78PYz
q3GJVmh1UFZJkojfruQ/XiilOSv1PzLE+gv1aAS4\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{102, 179},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->602367172]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->786069035],

Cell["Averaging tape of a 2D machine over many steps:", "ExampleText",
 CellID->142874901],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Total", "[", 
   RowBox[{"Last", " ", "/@", " ", 
    RowBox[{"TuringMachine", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"27726", ",", "2", ",", " ", "2", ",", " ", 
        RowBox[{"{", 
         RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"1", ",", " ", 
        RowBox[{"Table", "[", 
         RowBox[{"0", ",", " ", 
          RowBox[{"{", "50", "}"}], ",", " ", 
          RowBox[{"{", "50", "}"}]}], "]"}]}], "}"}], ",", " ", "5000"}], 
     "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1297650679],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->420313702]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1168081232],

Cell["\<\
Successive states sequences from successive initial conditions:\
\>", "ExampleText",
 CellID->1942557265],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"#", "[", 
        RowBox[{"[", 
         RowBox[{"1", ",", "1"}], "]"}], "]"}], "&"}], "/@", 
      RowBox[{"TuringMachine", "[", 
       RowBox[{"1507", ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"IntegerDigits", "[", 
             RowBox[{"i", ",", "2", ",", "20"}], "]"}], ",", "0"}], "}"}]}], 
         "}"}], ",", "100"}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "100"}], "}"}]}], "]"}], "-", "1"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->366067514],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzt1dEKwyAMBdC4P9kv7RPK+rz/f1uhL9Nck4iT6XoTCkJjT7lSet9ej+0m
Is/j2o8rZZ1Xfi+VdU5EVkSWRkRVOQBGTooIESKfjcp8DJHhiB5Sc6sgfmDf
QNx2ngFeMfbjIkLkQoiA0kN6bF1EpzoCMdvdj0rcw58RQYG1RtYbd6XtOPwk
QisiRIgQIUKECBEiRIgQaUekLO81QhYRIldD1AQcSub9tZA80VGI0YH9qMQ8
/FkRFFhbZL1xw/bDsJOIrYgQ+TMk9jGjMpSJETBVGcTds7fSsSj6D/8nSGfc
oN+Erxb3
   "], {{0, 0}, {101, 100}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 178},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1093125282]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->100505616],

Cell["\<\
Sequence of left or right movements for successive initial conditions:\
\>", "ExampleText",
 CellID->1275741570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Differences", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"#", "[", 
         RowBox[{"[", 
          RowBox[{"1", ",", "3"}], "]"}], "]"}], "&"}], "/@", 
       RowBox[{"TuringMachine", "[", 
        RowBox[{"1507", ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"IntegerDigits", "[", 
              RowBox[{"i", ",", "2", ",", "20"}], "]"}], ",", "0"}], "}"}]}], 
          "}"}], ",", "100"}], "]"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "100"}], "}"}]}], "]"}], "/.", 
   RowBox[{
    RowBox[{"-", "1"}], "\[Rule]", "0"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->911835342],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzt10EOgyAQhWHsTXqlHsF03/vv2jTd4DyZQTBK/SdpQsLDrxkV4n1+PeZb
Sun5+2U1ZVWc/Ex/E5ERxqiGQUxARLyrYGBcyFDIVEpg7Gv4yCCG+/j2MLxy
LiH+YeicwsC4ghE5glRsVGOZ28UolbtcIKH7fjZDInX9amz1FqTiM6Y4wsDA
wMDAwMDAwMDAsMYSMfMmEKMwMK5hCESE8tjIRlqf72esV2C5QJo23qMMidT0
q7HV25Ae5xQGxr8Y4i02kdr95LyGROIbT8PSeFW1ITo6wNA3tKXemFQWng==

   "], {{0, 0}, {100, 100}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->434014489]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->120726057],

Cell["Halting on a one-sided tape:", "ExampleText",
 CellID->412745834],

Cell[BoxData[
 RowBox[{
  RowBox[{"TuringPlot", "[", 
   RowBox[{"evolution_", ",", "opts___"}], "]"}], ":=", 
  RowBox[{"ArrayPlot", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"MapAt", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Blend", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{"#", "===", "Orange"}], ",", "#", ",", 
              RowBox[{"GrayLevel", "[", "#", "]"}]}], "]"}], ",", "Red"}], 
           "}"}], "]"}], "&"}], ",", 
        RowBox[{"#", "[", 
         RowBox[{"[", "2", "]"}], "]"}], ",", 
        RowBox[{"#", "[", 
         RowBox[{"[", 
          RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}], "&"}], "/@", 
     RowBox[{"(", 
      RowBox[{"evolution", "/.", 
       RowBox[{"Null", "\[Rule]", "Orange"}]}], ")"}]}], ",", "opts"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2138810395],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"TuringPlot", "[", 
      RowBox[{
       RowBox[{"TuringMachine", "[", 
        RowBox[{"1507", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"1", ",", 
             RowBox[{"-", "2"}], ",", "0"}], "}"}], ",", 
           RowBox[{"Append", "[", 
            RowBox[{
             RowBox[{"IntegerDigits", "[", 
              RowBox[{"i", ",", "2", ",", "10"}], "]"}], ",", "Null"}], 
            "]"}]}], "}"}], ",", "15"}], "]"}], ",", 
       RowBox[{"Mesh", "\[Rule]", "True"}], ",", 
       RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "10"}], "}"}]}], "]"}], ",", 
   RowBox[{"Spacer", "[", "10", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1872946362],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{434, 210},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->814828207]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1793248997],

Cell["Computed function on a one-sided tape:", "ExampleText",
 CellID->816969448],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Last", "@", 
       RowBox[{"TuringMachine", "[", 
        RowBox[{"1507", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"1", ",", 
             RowBox[{"-", "2"}], ",", "0"}], "}"}], ",", 
           RowBox[{"Append", "[", 
            RowBox[{
             RowBox[{"IntegerDigits", "[", 
              RowBox[{"i", ",", "2", ",", "20"}], "]"}], ",", "Null"}], 
            "]"}]}], "}"}], ",", "50"}], "]"}]}], "/.", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"_", ",", "z_", ",", "_"}], "}"}], ",", "y_"}], "}"}], "/;",
           
          RowBox[{
           RowBox[{"y", "[", 
            RowBox[{"[", "z", "]"}], "]"}], "===", "Null"}]}], 
         "\[RuleDelayed]", 
         RowBox[{"FromDigits", "[", 
          RowBox[{
           RowBox[{"Most", "[", "y", "]"}], ",", "2"}], "]"}]}], ",", 
        RowBox[{"_", "\[Rule]", "Null"}]}], "}"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "1000"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1530295598],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXXtQVNcZv+7dXQSDhhATO51GJ51Op0lsOp12Ou00LO8oaOIDUHnIK44a
lTjWpplObHyxPOQNGnaXNYCAwgIaX8lolTci2FjfaTt2puYPtcZJrUMfyXS2
3znfvWfv3r3ALq9d8DLLvff7znde3++e73znfGdhafr2Deu2pG/fmJm+4PVt
6Vs3bMz81YLod7YBi5/BcdxJ+M2ax8GzHZ7o70m7He7S6ytw1UKSBu4auPPC
/RmSShLs4o88VUqL96Ah+HPgzivUIr1z3EJy1WGd7ErzcLwyTR7nOLfUjRyz
yUUvzeFevsCx5HO7hSJNIJxB7pT3Q497GSggy/mxtg6p16mqlVfdyOFSpzx9
hLaPRTckn8aDeryhE0/bSOR5T/vkpffjlUnCjdogHaPd1Lw3NbPQrmyJh7f7
wRK9uGv3h7L3DrvvOis9L+uxmXKfo/qqr7sMCYbXTPAxZrc9evQfzcOH/+K/
+urfnP+DB4Nwh8/jx/+FBODDM1zhmdPCneMhAYTgw5FcnA6l4AMsIkW5s1he
LBCvUianIfXp4QkKxPpQSEMKeQpJVirmYTISST1jYiGET1rgL5UHQSwHSY6n
3WEskKN5eFYNp4E76x9rCU0h3ABZibJqMCcrDVuGObVIoyjl6Fn/nPM7JOEh
gOL3LFxNnDghUTT5i313EE1ORy5RSCyJrSnI78wxtlvNl6osAx/uvwj80tKe
kqLuivIL5aUXcnPa84wdwLSY+kFyx/tnSgq6gVledgGYmFpR3lde2puX01Fa
3AvM4uLuQigzuy3H2FZc2F35YT+WAMxcYztKHqjos5gGGLPKPGC1DFjMA7nZ
7cAE+bLSXpSEZuTndhQVdCHfWnUJMpoqL+YZ25AJGSugMTwUw0WaKvtRDlik
srJelIMKgI9MKBFqBcniQpIfOl+Q14kl/u79s9B5YJYU90AhUD2kQn3FRd2M
CT2HjkFSXm5HeUkPVodM6C1I5tA+gELNpn5kYqNRElVWeaAfJKGEslKix7KS
XuAXF3RbzP2QcX9FHzADHCNUywai4TWzP8HPsC6zFdAiSndBC5huogWkIlrI
lKEFbXNFC8t0RWtffocrWpjRCS2Uc0UL5WRoIR4ytEDdrmgxpjtomUz9rmhB
vYpo5RnbR0JLz9AKFTELIJj9Mi3V5g5UrF5fgMpbSJWV9EwEUvW1lx1IkZmR
4xPXHNmz6/er4hs4DUGJXxR9MCzETB7pSPt5ZIR1qoDltXE1GWiJft+XDwZv
3rwfv7JewCsyvGrFskPkEWkYbjj6uJnk8tO3MlpUzLyF2dNSzNrO305PtSUn
NX587Eaz7VpKUmPcijogPzn9ObgnCXH1MO4AQzPF7cdpKTYfhGtao+U0wqiX
Tn3HwcGvb938+927/+S0VIrR8Fmb3MjR2e3VlKQm34Fpeo+qUeC0YlmdYB9D
mX2k89vL6anNKlYT7GOMgFSgFKmjrddTU5pSkhsb6i7bmq6CxVwcbaWgmXFG
+0GYwaJiNYHjamvWCdAzTEyIlrb6o0s4YgoLuuhOSgDjZGa0wBVaD+wv7vzj
xo37h2o/w3G2e+e5d7efJo/UPn4vdnG1itR4ISXsWYUbLNSqIU4JcQ2ISlRE
FeLEOGj0gM9wgiTmPy5/k/qPWpn/qCeXF6WjTfUrxjKuamv+EBtTnZ/bLowr
cPoy0mxg6cABpHg9zzj78to3vf0x8BlevT1/g6Q1qw6DWWy2XQUTGb+yDtdu
IAmG8je/Ph0msZLzAVd1dI16zqI77XNhOoJJCzeHEYZvvvkfPn9x5xFOaVJH
BB/u3XsMs9n2badcVmR0RH07I61FRcNjW2cX4iVjQ2XZG7Vs3WyJef0j5gfO
w/GiWjKPsHhhLFiAPcT1sLWq/9SJW5s3Hd+z69ybS2vQ2aNWbG5UhFUdGZMy
Mi5evMNpQJzjIRtkAB8PR4o2Pa1Z3H03sf2JIPDnVDQmDQ22P6sXN9UFRw8S
ZhH+bLBfqoUap7mbaT8Itc/8MmP2+cTVh8HzQq8N+O9knQDmusyWzRuPSf2v
WUtiap7Q0aCz48kNeRReN+GjhBSIOwy3bz+8eeM+dbOpB87owcGv95f3unhm
fvTyVkbrE4TTTAke7pwjUTyt5AX8Vi537OUR8MJCzLjHoIUxN+2t3LigpmWo
1dddRl3uy+8E8sD+C6hbHW7qIFoHrQMCd++e8yVF3fAI4odqPxNwKCzowqIE
KaTxh3H9gIsbR0tja5LoyjUhXoxiLYmpBoeceeOLSNTR2I4fFUONYFFl2MHc
j9rDjR02LnRIyyKFOhYphOSEODHei7LMvxDy4g/j+tEs9ZgXKoexGL/CzQhk
ZGZGyzTDT2/37Nye4hicz3AENwIWQik0EggkrIXS1jYli6Ri8HDD+qPof0gz
Skk88zZXyoLnrC0n4AEBPHvmL1jiqZO3bE1XkxMb41fUMcm0VFt0RBXH/P0w
HJFTGjzOgZ7npygVMeQZhoJ3A5MYzFAO+aEjH89SGqH4kpz0ozRMhjgV4ryn
ZzRQH+w4GyqcfwqB1dcUBQEx8LMrn/BU9BEnVcsgFWawBBIt/wJWvlNOs06r
oInWXKBMc5UH+mJjqnfuOAMGJWvLcVgZvRFTDcsinNV/BrO876uRG2ZNP9H6
1Er1qSFihC2LrGXvbXvv3U9YZO0nNLLmo8r0UV1Ko18S30UTKpxV/BHMg6oe
h9JjkN05KpWS3AjjnUWl2N7I2xuPbc06HhlmCRUtwEJ4V31Bnb46xiFJXgeu
fxkNC+HfvvepcmTppZSkJp94Rydt/hmzLmHwO+JBi6Md8aDvR4Zb1fdyJF3O
luqyteU6WWskk3gOrFJg/icHi+MaFkcfFP3W74JH9URqsr7uMo7UffmdmMdx
yobmCWB0eVkPygEbagHFijO/dvfOcznZbbJIzAKYw56cuUe7Kt5xygXzhIda
BCuImmT0yuV17NQLapLtmDhOvSCNToBBnKe+Q3Q6sW+lN22k9OQJ5nGcTaF5
5jEaBzLuL6AOMfOqhAY8oLc6viE6yspcfmSCFxBuMIuj/ltLYmqm5bh+mnD1
oBPc8mYqorkD//T5A9mGOD7jw727j69duwcvoXIs4rmoCOs0GrfDairYHU0d
OXwFZpfENUdgdjnz6Z/xnADdn6L6CoZ3bLq/UXPd0dOVK3chidUrTCuJjcVF
XRyNFM9hJ9Wnl3aeGYV2wg0W3PUW7b9wWBJX1k8tXVI7JbzeYfXywij0Qsz/
4uq8nDbFjeFFUVbJGR3/yAjrCG8PNlv8ke/5eXXvzzNdBo1Cl27sOVYe6AsL
wZ0HXWqKTUGDeGJUGmlw57v4Xt+/9gnt4o6uEDEOMQt+3kz0AyOsoGSSxV/V
s+d6nq24/7vrg7PNtmubNxFncP26VrQVdJbm4e0W9OI3jvomJP7NFVxn5Rjb
BT0jDRYc42H8cN9tCKYS27YSiaREUQIWXwanbz/oGWfj+qNwzd7bRrWtZdpm
MXWMg5cWdwuxVVzUse1cMaa++3xRYRfWwCLtQl6HlDSmzrgkpm4x9WNeeFvg
nWHn9nhjNgnUKcVlCT1R7zvigKs0WHoJODhoxMGxZnP97kKwooR03Yc4MA77
doMMB1l8nO316GRfhBjn+DjiAKQkPi5Za5ISYAYlTQ8xkRXXxCKBZ9RIRFgn
oxGJ4b6VEDyiBCLBOPsrerFwisR8hoQ0C5CtzdelpOJeMn7/QZZRSjoi3IzF
+oYQdHb8FZNaW643N11dm9wo3Z2GCo4c/qPoofB25qEM9Vdk5P7JGkqzPgrp
YFud8GDG1cXSo2XHpoK9F9IdFlcr0iBiZzOU6Gkmeq1239CB2gpfa4X6Vqqt
UN8HtRW+3QrvvZVYjfgjKsQ3vQy5L8lL6NH4omovvNsLQnqx9S5/6ZFyR3Fy
1ot9cDkxyk+pPkhXOFoJCr5ne1TfQW2Fb7ZCfSvVVqjvg9oK326F995Kpf/w
4Fu+BX75R/TU8Dqc/+zjq4JxjRd4px+erGZ8b3UzHq2X0qspjVHLffmdTlHM
woIuAfXhTpMGO0vk5XaghCOuiT05aB0QyhRHxvArmdH3ZuhYIPZG4USnQYz9
BQ8l4Xzmk2fBN4XVjrvxHGzt0PEybO1wZyeDR5TQyspUnjvw/xBI/1MBN+P/
bqua8g==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->201523900]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1057808415],

Cell["Show only steps on which the head reaches a new cell:", "ExampleText",
 CellID->2012325058],

Cell[BoxData[
 RowBox[{
  RowBox[{"TuringPlot", "[", 
   RowBox[{"evolution_", ",", "opts___"}], "]"}], ":=", 
  RowBox[{"ArrayPlot", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"MapAt", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Blend", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{"#", "===", "Orange"}], ",", "#", ",", 
              RowBox[{"GrayLevel", "[", "#", "]"}]}], "]"}], ",", "Red"}], 
           "}"}], "]"}], "&"}], ",", 
        RowBox[{"#", "[", 
         RowBox[{"[", "2", "]"}], "]"}], ",", 
        RowBox[{"#", "[", 
         RowBox[{"[", 
          RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}], "&"}], "/@", 
     RowBox[{"(", 
      RowBox[{"evolution", "/.", 
       RowBox[{"Null", "\[Rule]", "Orange"}]}], ")"}]}], ",", "opts"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1363213689],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Module", "[", 
  RowBox[{
   RowBox[{"{", "new", "}"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"new", "[", "x_", "]"}], ":=", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"new", "[", "x", "]"}], "=", "False"}], ";", "True"}], 
      ")"}]}], ";", 
    RowBox[{"TuringPlot", "[", 
     RowBox[{
      RowBox[{"Cases", "[", 
       RowBox[{
        RowBox[{"TuringMachine", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"2791284850", ",", "4"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"{", "}"}], ",", "0"}], "}"}]}], "}"}], ",", "300"}], 
         "]"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"_", ",", "_", ",", 
            RowBox[{"_", "?", "new"}]}], "}"}], ",", "_"}], "}"}]}], "]"}], 
      ",", 
      RowBox[{"Mesh", "\[Rule]", "True"}]}], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->508293099],

Cell[BoxData[
 GraphicsBox[{RasterBox[CompressedData["
1:eJztm9uNwkAUQyNtJVRCD5Sw0n5vy1sCJaBI+5uIIde59uRYQocvzyM2aEK4
ff8+fr6WZbn9v9b35+h5r2HVuOly2590n7+mXNCLWrntT4rPVv67cnK1XqSv
N91nNP/k5DOfqnFT1pviU5V/ctI7n1Gl5FPto86/2/WlF/tKz/OoT1f+3a5v
ei/c1pXi45Z/euHJUaX4pOR/dP5bcutXulJyXuWTzpT7velKyXOVz6x0+30w
XSl5rvK5GtWiF54+6ecIeuGplF6Q/8+oltu46v3h838OquU2bnovyP85VGvW
+at9yH8v1Zp1/lU+5N+TanWvT7UPoz7kP4tqda/v7F6Q/zmY8jyJWuq+wDno
9jyJWvQCHmHX8yRq0QuoYLpG18U5Ar7DdG2ti/zDI1RLPU/yDxVUq2qe5B+e
yZT7vd37BOEqt/u93fsB4R7d/j8IoTOrxDkCzsRRkX94BVad3yG8AvlegPt8
Ad7kxCk=
    "], {{0, 0}, {33, 33}}, {0, 1}], {
    {GrayLevel[
      NCache[-1 + GoldenRatio, 0.6180339887498949]], 
     LineBox[{{{0, 33}, {33, 33}}, {{0, 32}, {33, 32}}, {{0, 31}, {33, 
      31}}, {{0, 30}, {33, 30}}, {{0, 29}, {33, 29}}, {{0, 28}, {33, 28}}, {{
      0, 27}, {33, 27}}, {{0, 26}, {33, 26}}, {{0, 25}, {33, 25}}, {{0, 24}, {
      33, 24}}, {{0, 23}, {33, 23}}, {{0, 22}, {33, 22}}, {{0, 21}, {33, 
      21}}, {{0, 20}, {33, 20}}, {{0, 19}, {33, 19}}, {{0, 18}, {33, 18}}, {{
      0, 17}, {33, 17}}, {{0, 16}, {33, 16}}, {{0, 15}, {33, 15}}, {{0, 14}, {
      33, 14}}, {{0, 13}, {33, 13}}, {{0, 12}, {33, 12}}, {{0, 11}, {33, 
      11}}, {{0, 10}, {33, 10}}, {{0, 9}, {33, 9}}, {{0, 8}, {33, 8}}, {{0, 
      7}, {33, 7}}, {{0, 6}, {33, 6}}, {{0, 5}, {33, 5}}, {{0, 4}, {33, 
      4}}, {{0, 3}, {33, 3}}, {{0, 2}, {33, 2}}, {{0, 1}, {33, 1}}, {{0, 0}, {
      33, 0}}}]}, 
    {GrayLevel[
      NCache[-1 + GoldenRatio, 0.6180339887498949]], 
     LineBox[{{{0, 0}, {0, 33}}, {{1, 0}, {1, 33}}, {{2, 0}, {2, 33}}, {{3, 
      0}, {3, 33}}, {{4, 0}, {4, 33}}, {{5, 0}, {5, 33}}, {{6, 0}, {6, 
      33}}, {{7, 0}, {7, 33}}, {{8, 0}, {8, 33}}, {{9, 0}, {9, 33}}, {{10, 
      0}, {10, 33}}, {{11, 0}, {11, 33}}, {{12, 0}, {12, 33}}, {{13, 0}, {13, 
      33}}, {{14, 0}, {14, 33}}, {{15, 0}, {15, 33}}, {{16, 0}, {16, 33}}, {{
      17, 0}, {17, 33}}, {{18, 0}, {18, 33}}, {{19, 0}, {19, 33}}, {{20, 0}, {
      20, 33}}, {{21, 0}, {21, 33}}, {{22, 0}, {22, 33}}, {{23, 0}, {23, 
      33}}, {{24, 0}, {24, 33}}, {{25, 0}, {25, 33}}, {{26, 0}, {26, 33}}, {{
      27, 0}, {27, 33}}, {{28, 0}, {28, 33}}, {{29, 0}, {29, 33}}, {{30, 0}, {
      30, 33}}, {{31, 0}, {31, 33}}, {{32, 0}, {32, 33}}, {{33, 0}, {33, 
      33}}}]}}},
  Frame->False,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  ImageSize->Small,
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1714030618]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->647052228],

Cell["\<\
Show only steps on which the head returns to its initial location:\
\>", "ExampleText",
 CellID->149607588],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Last", "/@", 
   RowBox[{"Select", "[", 
    RowBox[{
     RowBox[{"TuringMachine", "[", 
      RowBox[{"1413", ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", 
         RowBox[{"RandomInteger", "[", 
          RowBox[{"1", ",", "100"}], "]"}]}], "}"}], ",", "6000"}], "]"}], 
     ",", 
     RowBox[{
      RowBox[{
       RowBox[{"#", "[", 
        RowBox[{"[", 
         RowBox[{"1", ",", "2"}], "]"}], "]"}], "===", "1"}], "&"}]}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2089392171],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzd2F1uwyAMAGB7N9k5dosdodr77v+2boH4BzAm2NFUVVXTxuEDowTT98f3
5+MNAD6e76/nGxHWX7/XOYKwxHmid5FL41hEro1jDbk4jiXk6jiWJ/4FELwB
2ZmRf5audARvQOCGkeAN6dp9bnkQ5FASQqtIHoIyZSkIsM8sBCkoDQFxlIMg
D0pCQB1nICiDUhBovsUjqIMSEP1bBtLUVwlI/5dYpFMnhiOjOYpEuvVuMDK+
a+KQQd0eiljPsShkuP8IROyVJQYx9lFhyGytj0DM/WAQMq++9hFrHCVmG7HK
IkeMCxmPw9cRD2IVRW7E3LhO6i4ncoauITVTHoSFriCUqTlSs2V1oltIUPsz
pA672855eWeN55myEcpsp51yeacHbL8wRYjotFMvh+akKOUnCCNQZ4TaaIap
qmwTEYRCxHRIBNnJGaIIgbDpoGVEEC6kIRjCp0Pl8iCOiZohmijTW1tld9jf
c7GerMSJkEYdOQ4rUeaU/SkChajdofBM4wdwPQ0E
   "], {{0, 0}, {100, 59}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1748663012]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1180076596],

Cell["Causal network from random initial tape:", "ExampleText",
 CellID->1535868927],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GraphPlot", "[", 
  RowBox[{"With", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"rules", "=", 
      RowBox[{"MapIndexed", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"First", "[", "#2", "]"}], ",", 
           RowBox[{"#", "[", 
            RowBox[{"[", 
             RowBox[{"1", ",", "2"}], "]"}], "]"}], ",", 
           RowBox[{"#", "[", 
            RowBox[{"[", 
             RowBox[{"2", ",", 
              RowBox[{"#", "[", 
               RowBox[{"[", 
                RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}], "]"}]}], 
          "}"}], "&"}], ",", 
        RowBox[{"TuringMachine", "[", 
         RowBox[{"1507", ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{"RandomInteger", "[", 
             RowBox[{"1", ",", "30"}], "]"}]}], "}"}], ",", "200"}], "]"}]}], 
       "]"}]}], "}"}], ",", 
    RowBox[{"Join", "[", 
     RowBox[{
      RowBox[{"Rule", "@@@", 
       RowBox[{"Partition", "[", 
        RowBox[{"rules", ",", "2", ",", "1"}], "]"}]}], ",", 
      RowBox[{"Flatten", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"Rule", "@@@", 
           RowBox[{"Partition", "[", 
            RowBox[{"#", ",", "2", ",", "1"}], "]"}]}], "&"}], "/@", 
         RowBox[{"Last", "@", 
          RowBox[{"Reap", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"Sow", "[", 
              RowBox[{"#", ",", 
               RowBox[{"#", "[", 
                RowBox[{"[", "2", "]"}], "]"}]}], "]"}], "&"}], "/@", 
            "rules"}], "]"}]}]}], ",", "1"}], "]"}]}], "]"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->595619149],

Cell[BoxData[
 GraphicsBox[
  TagBox[GraphicsComplexBox[CompressedData["
1:eJw1Vnk01WsXNlWG6JhlPI7jzIdzSCj6PbckuSIldStSKZJMIZWuKZIhFOIa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    "], {
     {RGBColor[0.5, 0., 0.], LineBox[{1, 2}], LineBox[{1, 3}], 
      LineBox[{2, 3}], LineBox[{2, 6}], LineBox[{3, 4}], LineBox[{3, 5}], 
      LineBox[{4, 5}], LineBox[{4, 10}], LineBox[{5, 6}], LineBox[{5, 9}], 
      LineBox[{6, 7}], LineBox[{6, 8}], LineBox[{7, 8}], LineBox[{7, 113}], 
      LineBox[{8, 9}], LineBox[{8, 114}], LineBox[{9, 10}], LineBox[{9, 115}],
       LineBox[{10, 11}], LineBox[{10, 116}], LineBox[{11, 12}], 
      LineBox[{11, 117}], LineBox[{12, 13}], LineBox[{12, 118}], 
      LineBox[{13, 14}], LineBox[{13, 119}], LineBox[{14, 15}], 
      LineBox[{14, 120}], LineBox[{15, 16}], LineBox[{15, 121}], 
      LineBox[{16, 17}], LineBox[{16, 122}], LineBox[{17, 18}], 
      LineBox[{17, 29}], LineBox[{18, 19}], LineBox[{18, 20}], 
      LineBox[{19, 20}], LineBox[{19, 21}], LineBox[{20, 21}], 
      LineBox[{20, 28}], LineBox[{21, 22}], LineBox[{21, 27}], 
      LineBox[{22, 23}], LineBox[{22, 26}], LineBox[{23, 24}], 
      LineBox[{23, 25}], LineBox[{24, 25}], LineBox[{24, 34}], 
      LineBox[{25, 26}], LineBox[{25, 33}], LineBox[{26, 27}], 
      LineBox[{26, 32}], LineBox[{27, 28}], LineBox[{27, 31}], 
      LineBox[{28, 29}], LineBox[{28, 30}], LineBox[{29, 30}], 
      LineBox[{29, 123}], LineBox[{30, 31}], LineBox[{30, 124}], 
      LineBox[{31, 32}], LineBox[{31, 125}], LineBox[{32, 33}], 
      LineBox[{32, 126}], LineBox[{33, 34}], LineBox[{33, 89}], 
      LineBox[{34, 35}], LineBox[{34, 64}], LineBox[{35, 36}], 
      LineBox[{35, 49}], LineBox[{36, 37}], LineBox[{36, 38}], 
      LineBox[{37, 38}], LineBox[{37, 39}], LineBox[{38, 39}], 
      LineBox[{38, 48}], LineBox[{39, 40}], LineBox[{39, 47}], 
      LineBox[{40, 41}], LineBox[{40, 46}], LineBox[{41, 42}], 
      LineBox[{41, 45}], LineBox[{42, 43}], LineBox[{42, 44}], 
      LineBox[{43, 44}], LineBox[{43, 55}], LineBox[{44, 45}], 
      LineBox[{44, 54}], LineBox[{45, 46}], LineBox[{45, 53}], 
      LineBox[{46, 47}], LineBox[{46, 52}], LineBox[{47, 48}], 
      LineBox[{47, 51}], LineBox[{48, 49}], LineBox[{48, 50}], 
      LineBox[{49, 50}], LineBox[{49, 63}], LineBox[{50, 51}], 
      LineBox[{50, 62}], LineBox[{51, 52}], LineBox[{51, 61}], 
      LineBox[{52, 53}], LineBox[{52, 60}], LineBox[{53, 54}], 
      LineBox[{53, 59}], LineBox[{54, 55}], LineBox[{54, 58}], 
      LineBox[{55, 56}], LineBox[{55, 57}], LineBox[{56, 57}], 
      LineBox[{56, 72}], LineBox[{57, 58}], LineBox[{57, 71}], 
      LineBox[{58, 59}], LineBox[{58, 70}], LineBox[{59, 60}], 
      LineBox[{59, 69}], LineBox[{60, 61}], LineBox[{60, 68}], 
      LineBox[{61, 62}], LineBox[{61, 67}], LineBox[{62, 63}], 
      LineBox[{62, 66}], LineBox[{63, 64}], LineBox[{63, 65}], 
      LineBox[{64, 65}], LineBox[{64, 88}], LineBox[{65, 66}], 
      LineBox[{65, 87}], LineBox[{66, 67}], LineBox[{66, 86}], 
      LineBox[{67, 68}], LineBox[{67, 85}], LineBox[{68, 69}], 
      LineBox[{68, 84}], LineBox[{69, 70}], LineBox[{69, 83}], 
      LineBox[{70, 71}], LineBox[{70, 82}], LineBox[{71, 72}], 
      LineBox[{71, 81}], LineBox[{72, 73}], LineBox[{72, 80}], 
      LineBox[{73, 74}], LineBox[{73, 79}], LineBox[{74, 75}], 
      LineBox[{74, 78}], LineBox[{75, 76}], LineBox[{75, 77}], 
      LineBox[{76, 77}], LineBox[{76, 102}], LineBox[{77, 78}], 
      LineBox[{77, 101}], LineBox[{78, 79}], LineBox[{78, 100}], 
      LineBox[{79, 80}], LineBox[{79, 99}], LineBox[{80, 81}], 
      LineBox[{80, 98}], LineBox[{81, 82}], LineBox[{81, 97}], 
      LineBox[{82, 83}], LineBox[{82, 96}], LineBox[{83, 84}], 
      LineBox[{83, 95}], LineBox[{84, 85}], LineBox[{84, 94}], 
      LineBox[{85, 86}], LineBox[{85, 93}], LineBox[{86, 87}], 
      LineBox[{86, 92}], LineBox[{87, 88}], LineBox[{87, 91}], 
      LineBox[{88, 89}], LineBox[{88, 90}], LineBox[{89, 90}], 
      LineBox[{89, 127}], LineBox[{90, 91}], LineBox[{90, 128}], 
      LineBox[{91, 92}], LineBox[{91, 129}], LineBox[{92, 93}], 
      LineBox[{92, 130}], LineBox[{93, 94}], LineBox[{93, 131}], 
      LineBox[{94, 95}], LineBox[{94, 132}], LineBox[{95, 96}], 
      LineBox[{95, 133}], LineBox[{96, 97}], LineBox[{96, 134}], 
      LineBox[{97, 98}], LineBox[{97, 135}], LineBox[{98, 99}], 
      LineBox[{98, 136}], LineBox[{99, 100}], LineBox[{99, 137}], 
      LineBox[{100, 101}], LineBox[{100, 138}], LineBox[{101, 102}], 
      LineBox[{101, 109}], LineBox[{102, 103}], LineBox[{102, 104}], 
      LineBox[{103, 104}], LineBox[{103, 105}], LineBox[{104, 105}], 
      LineBox[{104, 108}], LineBox[{105, 106}], LineBox[{105, 107}], 
      LineBox[{106, 107}], LineBox[{106, 112}], LineBox[{107, 108}], 
      LineBox[{107, 111}], LineBox[{108, 109}], LineBox[{108, 110}], 
      LineBox[{109, 110}], LineBox[{109, 139}], LineBox[{110, 111}], 
      LineBox[{110, 140}], LineBox[{111, 112}], LineBox[{111, 141}], 
      LineBox[{112, 113}], LineBox[{112, 142}], LineBox[{113, 114}], 
      LineBox[{113, 143}], LineBox[{114, 115}], LineBox[{114, 144}], 
      LineBox[{115, 116}], LineBox[{115, 145}], LineBox[{116, 117}], 
      LineBox[{116, 146}], LineBox[{117, 118}], LineBox[{117, 147}], 
      LineBox[{118, 119}], LineBox[{118, 148}], LineBox[{119, 120}], 
      LineBox[{119, 149}], LineBox[{120, 121}], LineBox[{120, 150}], 
      LineBox[{121, 122}], LineBox[{121, 151}], LineBox[{122, 123}], 
      LineBox[{122, 152}], LineBox[{123, 124}], LineBox[{123, 153}], 
      LineBox[{124, 125}], LineBox[{124, 154}], LineBox[{125, 126}], 
      LineBox[{125, 155}], LineBox[{126, 127}], LineBox[{126, 156}], 
      LineBox[{127, 128}], LineBox[{127, 157}], LineBox[{128, 129}], 
      LineBox[{128, 158}], LineBox[{129, 130}], LineBox[{129, 159}], 
      LineBox[{130, 131}], LineBox[{130, 160}], LineBox[{131, 132}], 
      LineBox[{131, 161}], LineBox[{132, 133}], LineBox[{132, 162}], 
      LineBox[{133, 134}], LineBox[{133, 163}], LineBox[{134, 135}], 
      LineBox[{134, 164}], LineBox[{135, 136}], LineBox[{135, 165}], 
      LineBox[{136, 137}], LineBox[{136, 166}], LineBox[{137, 138}], 
      LineBox[{137, 167}], LineBox[{138, 139}], LineBox[{138, 168}], 
      LineBox[{139, 140}], LineBox[{139, 169}], LineBox[{140, 141}], 
      LineBox[{140, 170}], LineBox[{141, 142}], LineBox[{141, 171}], 
      LineBox[{142, 143}], LineBox[{142, 172}], LineBox[{143, 144}], 
      LineBox[{143, 173}], LineBox[{144, 145}], LineBox[{144, 174}], 
      LineBox[{145, 146}], LineBox[{145, 175}], LineBox[{146, 147}], 
      LineBox[{146, 176}], LineBox[{147, 148}], LineBox[{147, 177}], 
      LineBox[{148, 149}], LineBox[{148, 178}], LineBox[{149, 150}], 
      LineBox[{149, 179}], LineBox[{150, 151}], LineBox[{150, 180}], 
      LineBox[{151, 152}], LineBox[{151, 181}], LineBox[{152, 153}], 
      LineBox[{152, 182}], LineBox[{153, 154}], LineBox[{153, 183}], 
      LineBox[{154, 155}], LineBox[{154, 184}], LineBox[{155, 156}], 
      LineBox[{155, 185}], LineBox[{156, 157}], LineBox[{156, 186}], 
      LineBox[{157, 158}], LineBox[{157, 187}], LineBox[{158, 159}], 
      LineBox[{158, 188}], LineBox[{159, 160}], LineBox[{159, 189}], 
      LineBox[{160, 161}], LineBox[{160, 190}], LineBox[{161, 162}], 
      LineBox[{161, 191}], LineBox[{162, 163}], LineBox[{162, 192}], 
      LineBox[{163, 164}], LineBox[{163, 193}], LineBox[{164, 165}], 
      LineBox[{164, 194}], LineBox[{165, 166}], LineBox[{165, 195}], 
      LineBox[{166, 167}], LineBox[{166, 196}], LineBox[{167, 168}], 
      LineBox[{167, 197}], LineBox[{168, 169}], LineBox[{168, 198}], 
      LineBox[{169, 170}], LineBox[{169, 199}], LineBox[{170, 171}], 
      LineBox[{170, 200}], LineBox[{171, 172}], LineBox[{171, 201}], 
      LineBox[{172, 173}], LineBox[{173, 174}], LineBox[{174, 175}], 
      LineBox[{175, 176}], LineBox[{176, 177}], LineBox[{177, 178}], 
      LineBox[{178, 179}], LineBox[{179, 180}], LineBox[{180, 181}], 
      LineBox[{181, 182}], LineBox[{182, 183}], LineBox[{183, 184}], 
      LineBox[{184, 185}], LineBox[{185, 186}], LineBox[{186, 187}], 
      LineBox[{187, 188}], LineBox[{188, 189}], LineBox[{189, 190}], 
      LineBox[{190, 191}], LineBox[{191, 192}], LineBox[{192, 193}], 
      LineBox[{193, 194}], LineBox[{194, 195}], LineBox[{195, 196}], 
      LineBox[{196, 197}], LineBox[{197, 198}], LineBox[{198, 199}], 
      LineBox[{199, 200}], LineBox[{200, 201}]}, 
     {RGBColor[0., 0., 0.7], 
      TagBox[
       TooltipBox[PointBox[1],
        RowBox[{"{", 
          RowBox[{"1", ",", "1", ",", "0"}], "}"}]],
       Annotation[#, {1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2],
        RowBox[{"{", 
          RowBox[{"2", ",", "2", ",", "1"}], "}"}]],
       Annotation[#, {2, 2, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3],
        RowBox[{"{", 
          RowBox[{"3", ",", "1", ",", "1"}], "}"}]],
       Annotation[#, {3, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[4],
        RowBox[{"{", 
          RowBox[{"4", ",", "30", ",", "0"}], "}"}]],
       Annotation[#, {4, 30, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[5],
        RowBox[{"{", 
          RowBox[{"5", ",", "1", ",", "0"}], "}"}]],
       Annotation[#, {5, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[6],
        RowBox[{"{", 
          RowBox[{"6", ",", "2", ",", "0"}], "}"}]],
       Annotation[#, {6, 2, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[7],
        RowBox[{"{", 
          RowBox[{"7", ",", "3", ",", "1"}], "}"}]],
       Annotation[#, {7, 3, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[8],
        RowBox[{"{", 
          RowBox[{"8", ",", "2", ",", "1"}], "}"}]],
       Annotation[#, {8, 2, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[9],
        RowBox[{"{", 
          RowBox[{"9", ",", "1", ",", "1"}], "}"}]],
       Annotation[#, {9, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[10],
        RowBox[{"{", 
          RowBox[{"10", ",", "30", ",", "1"}], "}"}]],
       Annotation[#, {10, 30, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[11],
        RowBox[{"{", 
          RowBox[{"11", ",", "29", ",", "1"}], "}"}]],
       Annotation[#, {11, 29, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[12],
        RowBox[{"{", 
          RowBox[{"12", ",", "28", ",", "1"}], "}"}]],
       Annotation[#, {12, 28, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[13],
        RowBox[{"{", 
          RowBox[{"13", ",", "27", ",", "1"}], "}"}]],
       Annotation[#, {13, 27, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[14],
        RowBox[{"{", 
          RowBox[{"14", ",", "26", ",", "1"}], "}"}]],
       Annotation[#, {14, 26, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[15],
        RowBox[{"{", 
          RowBox[{"15", ",", "25", ",", "1"}], "}"}]],
       Annotation[#, {15, 25, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[16],
        RowBox[{"{", 
          RowBox[{"16", ",", "24", ",", "1"}], "}"}]],
       Annotation[#, {16, 24, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[17],
        RowBox[{"{", 
          RowBox[{"17", ",", "23", ",", "1"}], "}"}]],
       Annotation[#, {17, 23, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[18],
        RowBox[{"{", 
          RowBox[{"18", ",", "22", ",", "1"}], "}"}]],
       Annotation[#, {18, 22, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[19],
        RowBox[{"{", 
          RowBox[{"19", ",", "21", ",", "0"}], "}"}]],
       Annotation[#, {19, 21, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[20],
        RowBox[{"{", 
          RowBox[{"20", ",", "22", ",", "1"}], "}"}]],
       Annotation[#, {20, 22, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[21],
        RowBox[{"{", 
          RowBox[{"21", ",", "21", ",", "1"}], "}"}]],
       Annotation[#, {21, 21, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[22],
        RowBox[{"{", 
          RowBox[{"22", ",", "20", ",", "1"}], "}"}]],
       Annotation[#, {22, 20, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[23],
        RowBox[{"{", 
          RowBox[{"23", ",", "19", ",", "1"}], "}"}]],
       Annotation[#, {23, 19, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[24],
        RowBox[{"{", 
          RowBox[{"24", ",", "18", ",", "0"}], "}"}]],
       Annotation[#, {24, 18, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[25],
        RowBox[{"{", 
          RowBox[{"25", ",", "19", ",", "0"}], "}"}]],
       Annotation[#, {25, 19, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[26],
        RowBox[{"{", 
          RowBox[{"26", ",", "20", ",", "0"}], "}"}]],
       Annotation[#, {26, 20, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[27],
        RowBox[{"{", 
          RowBox[{"27", ",", "21", ",", "0"}], "}"}]],
       Annotation[#, {27, 21, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[28],
        RowBox[{"{", 
          RowBox[{"28", ",", "22", ",", "0"}], "}"}]],
       Annotation[#, {28, 22, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[29],
        RowBox[{"{", 
          RowBox[{"29", ",", "23", ",", "1"}], "}"}]],
       Annotation[#, {29, 23, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[30],
        RowBox[{"{", 
          RowBox[{"30", ",", "22", ",", "1"}], "}"}]],
       Annotation[#, {30, 22, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[31],
        RowBox[{"{", 
          RowBox[{"31", ",", "21", ",", "1"}], "}"}]],
       Annotation[#, {31, 21, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[32],
        RowBox[{"{", 
          RowBox[{"32", ",", "20", ",", "1"}], "}"}]],
       Annotation[#, {32, 20, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[33],
        RowBox[{"{", 
          RowBox[{"33", ",", "19", ",", "1"}], "}"}]],
       Annotation[#, {33, 19, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[34],
        RowBox[{"{", 
          RowBox[{"34", ",", "18", ",", "1"}], "}"}]],
       Annotation[#, {34, 18, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[35],
        RowBox[{"{", 
          RowBox[{"35", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {35, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[36],
        RowBox[{"{", 
          RowBox[{"36", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {36, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[37],
        RowBox[{"{", 
          RowBox[{"37", ",", "15", ",", "0"}], "}"}]],
       Annotation[#, {37, 15, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[38],
        RowBox[{"{", 
          RowBox[{"38", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {38, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[39],
        RowBox[{"{", 
          RowBox[{"39", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {39, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[40],
        RowBox[{"{", 
          RowBox[{"40", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {40, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[41],
        RowBox[{"{", 
          RowBox[{"41", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {41, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[42],
        RowBox[{"{", 
          RowBox[{"42", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {42, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[43],
        RowBox[{"{", 
          RowBox[{"43", ",", "11", ",", "0"}], "}"}]],
       Annotation[#, {43, 11, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[44],
        RowBox[{"{", 
          RowBox[{"44", ",", "12", ",", "0"}], "}"}]],
       Annotation[#, {44, 12, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[45],
        RowBox[{"{", 
          RowBox[{"45", ",", "13", ",", "0"}], "}"}]],
       Annotation[#, {45, 13, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[46],
        RowBox[{"{", 
          RowBox[{"46", ",", "14", ",", "0"}], "}"}]],
       Annotation[#, {46, 14, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[47],
        RowBox[{"{", 
          RowBox[{"47", ",", "15", ",", "0"}], "}"}]],
       Annotation[#, {47, 15, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[48],
        RowBox[{"{", 
          RowBox[{"48", ",", "16", ",", "0"}], "}"}]],
       Annotation[#, {48, 16, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[49],
        RowBox[{"{", 
          RowBox[{"49", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {49, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[50],
        RowBox[{"{", 
          RowBox[{"50", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {50, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[51],
        RowBox[{"{", 
          RowBox[{"51", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {51, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[52],
        RowBox[{"{", 
          RowBox[{"52", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {52, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[53],
        RowBox[{"{", 
          RowBox[{"53", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {53, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[54],
        RowBox[{"{", 
          RowBox[{"54", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {54, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[55],
        RowBox[{"{", 
          RowBox[{"55", ",", "11", ",", "1"}], "}"}]],
       Annotation[#, {55, 11, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[56],
        RowBox[{"{", 
          RowBox[{"56", ",", "10", ",", "0"}], "}"}]],
       Annotation[#, {56, 10, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[57],
        RowBox[{"{", 
          RowBox[{"57", ",", "11", ",", "0"}], "}"}]],
       Annotation[#, {57, 11, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[58],
        RowBox[{"{", 
          RowBox[{"58", ",", "12", ",", "0"}], "}"}]],
       Annotation[#, {58, 12, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[59],
        RowBox[{"{", 
          RowBox[{"59", ",", "13", ",", "0"}], "}"}]],
       Annotation[#, {59, 13, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[60],
        RowBox[{"{", 
          RowBox[{"60", ",", "14", ",", "0"}], "}"}]],
       Annotation[#, {60, 14, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[61],
        RowBox[{"{", 
          RowBox[{"61", ",", "15", ",", "0"}], "}"}]],
       Annotation[#, {61, 15, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[62],
        RowBox[{"{", 
          RowBox[{"62", ",", "16", ",", "0"}], "}"}]],
       Annotation[#, {62, 16, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[63],
        RowBox[{"{", 
          RowBox[{"63", ",", "17", ",", "0"}], "}"}]],
       Annotation[#, {63, 17, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[64],
        RowBox[{"{", 
          RowBox[{"64", ",", "18", ",", "1"}], "}"}]],
       Annotation[#, {64, 18, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[65],
        RowBox[{"{", 
          RowBox[{"65", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {65, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[66],
        RowBox[{"{", 
          RowBox[{"66", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {66, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[67],
        RowBox[{"{", 
          RowBox[{"67", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {67, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[68],
        RowBox[{"{", 
          RowBox[{"68", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {68, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[69],
        RowBox[{"{", 
          RowBox[{"69", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {69, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[70],
        RowBox[{"{", 
          RowBox[{"70", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {70, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[71],
        RowBox[{"{", 
          RowBox[{"71", ",", "11", ",", "1"}], "}"}]],
       Annotation[#, {71, 11, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[72],
        RowBox[{"{", 
          RowBox[{"72", ",", "10", ",", "1"}], "}"}]],
       Annotation[#, {72, 10, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[73],
        RowBox[{"{", 
          RowBox[{"73", ",", "9", ",", "1"}], "}"}]],
       Annotation[#, {73, 9, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[74],
        RowBox[{"{", 
          RowBox[{"74", ",", "8", ",", "1"}], "}"}]],
       Annotation[#, {74, 8, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[75],
        RowBox[{"{", 
          RowBox[{"75", ",", "7", ",", "1"}], "}"}]],
       Annotation[#, {75, 7, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[76],
        RowBox[{"{", 
          RowBox[{"76", ",", "6", ",", "0"}], "}"}]],
       Annotation[#, {76, 6, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[77],
        RowBox[{"{", 
          RowBox[{"77", ",", "7", ",", "0"}], "}"}]],
       Annotation[#, {77, 7, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[78],
        RowBox[{"{", 
          RowBox[{"78", ",", "8", ",", "0"}], "}"}]],
       Annotation[#, {78, 8, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[79],
        RowBox[{"{", 
          RowBox[{"79", ",", "9", ",", "0"}], "}"}]],
       Annotation[#, {79, 9, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[80],
        RowBox[{"{", 
          RowBox[{"80", ",", "10", ",", "0"}], "}"}]],
       Annotation[#, {80, 10, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[81],
        RowBox[{"{", 
          RowBox[{"81", ",", "11", ",", "0"}], "}"}]],
       Annotation[#, {81, 11, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[82],
        RowBox[{"{", 
          RowBox[{"82", ",", "12", ",", "0"}], "}"}]],
       Annotation[#, {82, 12, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[83],
        RowBox[{"{", 
          RowBox[{"83", ",", "13", ",", "0"}], "}"}]],
       Annotation[#, {83, 13, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[84],
        RowBox[{"{", 
          RowBox[{"84", ",", "14", ",", "0"}], "}"}]],
       Annotation[#, {84, 14, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[85],
        RowBox[{"{", 
          RowBox[{"85", ",", "15", ",", "0"}], "}"}]],
       Annotation[#, {85, 15, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[86],
        RowBox[{"{", 
          RowBox[{"86", ",", "16", ",", "0"}], "}"}]],
       Annotation[#, {86, 16, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[87],
        RowBox[{"{", 
          RowBox[{"87", ",", "17", ",", "0"}], "}"}]],
       Annotation[#, {87, 17, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[88],
        RowBox[{"{", 
          RowBox[{"88", ",", "18", ",", "0"}], "}"}]],
       Annotation[#, {88, 18, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[89],
        RowBox[{"{", 
          RowBox[{"89", ",", "19", ",", "1"}], "}"}]],
       Annotation[#, {89, 19, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[90],
        RowBox[{"{", 
          RowBox[{"90", ",", "18", ",", "1"}], "}"}]],
       Annotation[#, {90, 18, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[91],
        RowBox[{"{", 
          RowBox[{"91", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {91, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[92],
        RowBox[{"{", 
          RowBox[{"92", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {92, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[93],
        RowBox[{"{", 
          RowBox[{"93", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {93, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[94],
        RowBox[{"{", 
          RowBox[{"94", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {94, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[95],
        RowBox[{"{", 
          RowBox[{"95", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {95, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[96],
        RowBox[{"{", 
          RowBox[{"96", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {96, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[97],
        RowBox[{"{", 
          RowBox[{"97", ",", "11", ",", "1"}], "}"}]],
       Annotation[#, {97, 11, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[98],
        RowBox[{"{", 
          RowBox[{"98", ",", "10", ",", "1"}], "}"}]],
       Annotation[#, {98, 10, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[99],
        RowBox[{"{", 
          RowBox[{"99", ",", "9", ",", "1"}], "}"}]],
       Annotation[#, {99, 9, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[100],
        RowBox[{"{", 
          RowBox[{"100", ",", "8", ",", "1"}], "}"}]],
       Annotation[#, {100, 8, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[101],
        RowBox[{"{", 
          RowBox[{"101", ",", "7", ",", "1"}], "}"}]],
       Annotation[#, {101, 7, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[102],
        RowBox[{"{", 
          RowBox[{"102", ",", "6", ",", "1"}], "}"}]],
       Annotation[#, {102, 6, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[103],
        RowBox[{"{", 
          RowBox[{"103", ",", "5", ",", "0"}], "}"}]],
       Annotation[#, {103, 5, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[104],
        RowBox[{"{", 
          RowBox[{"104", ",", "6", ",", "1"}], "}"}]],
       Annotation[#, {104, 6, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[105],
        RowBox[{"{", 
          RowBox[{"105", ",", "5", ",", "1"}], "}"}]],
       Annotation[#, {105, 5, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[106],
        RowBox[{"{", 
          RowBox[{"106", ",", "4", ",", "0"}], "}"}]],
       Annotation[#, {106, 4, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[107],
        RowBox[{"{", 
          RowBox[{"107", ",", "5", ",", "0"}], "}"}]],
       Annotation[#, {107, 5, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[108],
        RowBox[{"{", 
          RowBox[{"108", ",", "6", ",", "0"}], "}"}]],
       Annotation[#, {108, 6, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[109],
        RowBox[{"{", 
          RowBox[{"109", ",", "7", ",", "1"}], "}"}]],
       Annotation[#, {109, 7, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[110],
        RowBox[{"{", 
          RowBox[{"110", ",", "6", ",", "1"}], "}"}]],
       Annotation[#, {110, 6, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[111],
        RowBox[{"{", 
          RowBox[{"111", ",", "5", ",", "1"}], "}"}]],
       Annotation[#, {111, 5, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[112],
        RowBox[{"{", 
          RowBox[{"112", ",", "4", ",", "1"}], "}"}]],
       Annotation[#, {112, 4, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[113],
        RowBox[{"{", 
          RowBox[{"113", ",", "3", ",", "1"}], "}"}]],
       Annotation[#, {113, 3, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[114],
        RowBox[{"{", 
          RowBox[{"114", ",", "2", ",", "1"}], "}"}]],
       Annotation[#, {114, 2, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[115],
        RowBox[{"{", 
          RowBox[{"115", ",", "1", ",", "1"}], "}"}]],
       Annotation[#, {115, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[116],
        RowBox[{"{", 
          RowBox[{"116", ",", "30", ",", "1"}], "}"}]],
       Annotation[#, {116, 30, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[117],
        RowBox[{"{", 
          RowBox[{"117", ",", "29", ",", "1"}], "}"}]],
       Annotation[#, {117, 29, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[118],
        RowBox[{"{", 
          RowBox[{"118", ",", "28", ",", "1"}], "}"}]],
       Annotation[#, {118, 28, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[119],
        RowBox[{"{", 
          RowBox[{"119", ",", "27", ",", "1"}], "}"}]],
       Annotation[#, {119, 27, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[120],
        RowBox[{"{", 
          RowBox[{"120", ",", "26", ",", "1"}], "}"}]],
       Annotation[#, {120, 26, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[121],
        RowBox[{"{", 
          RowBox[{"121", ",", "25", ",", "1"}], "}"}]],
       Annotation[#, {121, 25, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[122],
        RowBox[{"{", 
          RowBox[{"122", ",", "24", ",", "1"}], "}"}]],
       Annotation[#, {122, 24, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[123],
        RowBox[{"{", 
          RowBox[{"123", ",", "23", ",", "1"}], "}"}]],
       Annotation[#, {123, 23, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[124],
        RowBox[{"{", 
          RowBox[{"124", ",", "22", ",", "1"}], "}"}]],
       Annotation[#, {124, 22, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[125],
        RowBox[{"{", 
          RowBox[{"125", ",", "21", ",", "1"}], "}"}]],
       Annotation[#, {125, 21, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[126],
        RowBox[{"{", 
          RowBox[{"126", ",", "20", ",", "1"}], "}"}]],
       Annotation[#, {126, 20, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[127],
        RowBox[{"{", 
          RowBox[{"127", ",", "19", ",", "1"}], "}"}]],
       Annotation[#, {127, 19, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[128],
        RowBox[{"{", 
          RowBox[{"128", ",", "18", ",", "1"}], "}"}]],
       Annotation[#, {128, 18, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[129],
        RowBox[{"{", 
          RowBox[{"129", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {129, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[130],
        RowBox[{"{", 
          RowBox[{"130", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {130, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[131],
        RowBox[{"{", 
          RowBox[{"131", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {131, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[132],
        RowBox[{"{", 
          RowBox[{"132", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {132, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[133],
        RowBox[{"{", 
          RowBox[{"133", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {133, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[134],
        RowBox[{"{", 
          RowBox[{"134", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {134, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[135],
        RowBox[{"{", 
          RowBox[{"135", ",", "11", ",", "1"}], "}"}]],
       Annotation[#, {135, 11, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[136],
        RowBox[{"{", 
          RowBox[{"136", ",", "10", ",", "1"}], "}"}]],
       Annotation[#, {136, 10, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[137],
        RowBox[{"{", 
          RowBox[{"137", ",", "9", ",", "1"}], "}"}]],
       Annotation[#, {137, 9, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[138],
        RowBox[{"{", 
          RowBox[{"138", ",", "8", ",", "1"}], "}"}]],
       Annotation[#, {138, 8, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[139],
        RowBox[{"{", 
          RowBox[{"139", ",", "7", ",", "1"}], "}"}]],
       Annotation[#, {139, 7, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[140],
        RowBox[{"{", 
          RowBox[{"140", ",", "6", ",", "1"}], "}"}]],
       Annotation[#, {140, 6, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[141],
        RowBox[{"{", 
          RowBox[{"141", ",", "5", ",", "1"}], "}"}]],
       Annotation[#, {141, 5, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[142],
        RowBox[{"{", 
          RowBox[{"142", ",", "4", ",", "1"}], "}"}]],
       Annotation[#, {142, 4, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[143],
        RowBox[{"{", 
          RowBox[{"143", ",", "3", ",", "1"}], "}"}]],
       Annotation[#, {143, 3, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[144],
        RowBox[{"{", 
          RowBox[{"144", ",", "2", ",", "1"}], "}"}]],
       Annotation[#, {144, 2, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[145],
        RowBox[{"{", 
          RowBox[{"145", ",", "1", ",", "1"}], "}"}]],
       Annotation[#, {145, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[146],
        RowBox[{"{", 
          RowBox[{"146", ",", "30", ",", "1"}], "}"}]],
       Annotation[#, {146, 30, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[147],
        RowBox[{"{", 
          RowBox[{"147", ",", "29", ",", "1"}], "}"}]],
       Annotation[#, {147, 29, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[148],
        RowBox[{"{", 
          RowBox[{"148", ",", "28", ",", "1"}], "}"}]],
       Annotation[#, {148, 28, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[149],
        RowBox[{"{", 
          RowBox[{"149", ",", "27", ",", "1"}], "}"}]],
       Annotation[#, {149, 27, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[150],
        RowBox[{"{", 
          RowBox[{"150", ",", "26", ",", "1"}], "}"}]],
       Annotation[#, {150, 26, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[151],
        RowBox[{"{", 
          RowBox[{"151", ",", "25", ",", "1"}], "}"}]],
       Annotation[#, {151, 25, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[152],
        RowBox[{"{", 
          RowBox[{"152", ",", "24", ",", "1"}], "}"}]],
       Annotation[#, {152, 24, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[153],
        RowBox[{"{", 
          RowBox[{"153", ",", "23", ",", "1"}], "}"}]],
       Annotation[#, {153, 23, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[154],
        RowBox[{"{", 
          RowBox[{"154", ",", "22", ",", "1"}], "}"}]],
       Annotation[#, {154, 22, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[155],
        RowBox[{"{", 
          RowBox[{"155", ",", "21", ",", "1"}], "}"}]],
       Annotation[#, {155, 21, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[156],
        RowBox[{"{", 
          RowBox[{"156", ",", "20", ",", "1"}], "}"}]],
       Annotation[#, {156, 20, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[157],
        RowBox[{"{", 
          RowBox[{"157", ",", "19", ",", "1"}], "}"}]],
       Annotation[#, {157, 19, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[158],
        RowBox[{"{", 
          RowBox[{"158", ",", "18", ",", "1"}], "}"}]],
       Annotation[#, {158, 18, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[159],
        RowBox[{"{", 
          RowBox[{"159", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {159, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[160],
        RowBox[{"{", 
          RowBox[{"160", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {160, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[161],
        RowBox[{"{", 
          RowBox[{"161", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {161, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[162],
        RowBox[{"{", 
          RowBox[{"162", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {162, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[163],
        RowBox[{"{", 
          RowBox[{"163", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {163, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[164],
        RowBox[{"{", 
          RowBox[{"164", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {164, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[165],
        RowBox[{"{", 
          RowBox[{"165", ",", "11", ",", "1"}], "}"}]],
       Annotation[#, {165, 11, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[166],
        RowBox[{"{", 
          RowBox[{"166", ",", "10", ",", "1"}], "}"}]],
       Annotation[#, {166, 10, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[167],
        RowBox[{"{", 
          RowBox[{"167", ",", "9", ",", "1"}], "}"}]],
       Annotation[#, {167, 9, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[168],
        RowBox[{"{", 
          RowBox[{"168", ",", "8", ",", "1"}], "}"}]],
       Annotation[#, {168, 8, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[169],
        RowBox[{"{", 
          RowBox[{"169", ",", "7", ",", "1"}], "}"}]],
       Annotation[#, {169, 7, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[170],
        RowBox[{"{", 
          RowBox[{"170", ",", "6", ",", "1"}], "}"}]],
       Annotation[#, {170, 6, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[171],
        RowBox[{"{", 
          RowBox[{"171", ",", "5", ",", "1"}], "}"}]],
       Annotation[#, {171, 5, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[172],
        RowBox[{"{", 
          RowBox[{"172", ",", "4", ",", "1"}], "}"}]],
       Annotation[#, {172, 4, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[173],
        RowBox[{"{", 
          RowBox[{"173", ",", "3", ",", "1"}], "}"}]],
       Annotation[#, {173, 3, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[174],
        RowBox[{"{", 
          RowBox[{"174", ",", "2", ",", "1"}], "}"}]],
       Annotation[#, {174, 2, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[175],
        RowBox[{"{", 
          RowBox[{"175", ",", "1", ",", "1"}], "}"}]],
       Annotation[#, {175, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[176],
        RowBox[{"{", 
          RowBox[{"176", ",", "30", ",", "1"}], "}"}]],
       Annotation[#, {176, 30, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[177],
        RowBox[{"{", 
          RowBox[{"177", ",", "29", ",", "1"}], "}"}]],
       Annotation[#, {177, 29, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[178],
        RowBox[{"{", 
          RowBox[{"178", ",", "28", ",", "1"}], "}"}]],
       Annotation[#, {178, 28, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[179],
        RowBox[{"{", 
          RowBox[{"179", ",", "27", ",", "1"}], "}"}]],
       Annotation[#, {179, 27, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[180],
        RowBox[{"{", 
          RowBox[{"180", ",", "26", ",", "1"}], "}"}]],
       Annotation[#, {180, 26, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[181],
        RowBox[{"{", 
          RowBox[{"181", ",", "25", ",", "1"}], "}"}]],
       Annotation[#, {181, 25, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[182],
        RowBox[{"{", 
          RowBox[{"182", ",", "24", ",", "1"}], "}"}]],
       Annotation[#, {182, 24, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[183],
        RowBox[{"{", 
          RowBox[{"183", ",", "23", ",", "1"}], "}"}]],
       Annotation[#, {183, 23, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[184],
        RowBox[{"{", 
          RowBox[{"184", ",", "22", ",", "1"}], "}"}]],
       Annotation[#, {184, 22, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[185],
        RowBox[{"{", 
          RowBox[{"185", ",", "21", ",", "1"}], "}"}]],
       Annotation[#, {185, 21, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[186],
        RowBox[{"{", 
          RowBox[{"186", ",", "20", ",", "1"}], "}"}]],
       Annotation[#, {186, 20, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[187],
        RowBox[{"{", 
          RowBox[{"187", ",", "19", ",", "1"}], "}"}]],
       Annotation[#, {187, 19, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[188],
        RowBox[{"{", 
          RowBox[{"188", ",", "18", ",", "1"}], "}"}]],
       Annotation[#, {188, 18, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[189],
        RowBox[{"{", 
          RowBox[{"189", ",", "17", ",", "1"}], "}"}]],
       Annotation[#, {189, 17, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[190],
        RowBox[{"{", 
          RowBox[{"190", ",", "16", ",", "1"}], "}"}]],
       Annotation[#, {190, 16, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[191],
        RowBox[{"{", 
          RowBox[{"191", ",", "15", ",", "1"}], "}"}]],
       Annotation[#, {191, 15, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[192],
        RowBox[{"{", 
          RowBox[{"192", ",", "14", ",", "1"}], "}"}]],
       Annotation[#, {192, 14, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[193],
        RowBox[{"{", 
          RowBox[{"193", ",", "13", ",", "1"}], "}"}]],
       Annotation[#, {193, 13, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[194],
        RowBox[{"{", 
          RowBox[{"194", ",", "12", ",", "1"}], "}"}]],
       Annotation[#, {194, 12, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[195],
        RowBox[{"{", 
          RowBox[{"195", ",", "11", ",", "1"}], "}"}]],
       Annotation[#, {195, 11, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[196],
        RowBox[{"{", 
          RowBox[{"196", ",", "10", ",", "1"}], "}"}]],
       Annotation[#, {196, 10, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[197],
        RowBox[{"{", 
          RowBox[{"197", ",", "9", ",", "1"}], "}"}]],
       Annotation[#, {197, 9, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[198],
        RowBox[{"{", 
          RowBox[{"198", ",", "8", ",", "1"}], "}"}]],
       Annotation[#, {198, 8, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[199],
        RowBox[{"{", 
          RowBox[{"199", ",", "7", ",", "1"}], "}"}]],
       Annotation[#, {199, 7, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[200],
        RowBox[{"{", 
          RowBox[{"200", ",", "6", ",", "1"}], "}"}]],
       Annotation[#, {200, 6, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[201],
        RowBox[{"{", 
          RowBox[{"201", ",", "5", ",", "1"}], "}"}]],
       Annotation[#, {201, 5, 1}, "Tooltip"]& ]}}],
   Annotation[#, VertexCoordinateRules -> CompressedData["
1:eJw1Vnk01WsXNlWG6JhlPI7jzIdzSCj6PbckuSIldStSKZJMIZWuKZIhFOIa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     "]]& ],
  AspectRatio->Automatic,
  FrameTicks->None,
  PlotRange->All,
  PlotRangePadding->Scaled[0.1]]], "Output",
 ImageSize->{184, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->985080493]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->27357],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["CellularAutomaton",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CellularAutomaton"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ArrayPlot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ArrayPlot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->32532]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->3616],

Cell[TextData[{
 ButtonBox["Demonstrations with TuringMachine",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=TuringMachine"],
     None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[{
 ButtonBox["NKS|Online (",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/section-3.4"], None}],
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/section-3.4"], None}],
  FontSlant->"Italic"],
 ButtonBox[")",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/section-3.4"], None}]
}], "RelatedLinks",
 CellID->28900]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->283121036],

Cell[TextData[ButtonBox["Computational Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ComputationalSystemsAndDiscovery"]], "MoreAbout",
 CellID->400464714],

Cell[TextData[ButtonBox["Discrete Mathematics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteMathematics"]], "MoreAbout",
 CellID->94766179],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->421165893]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"TuringMachine - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 28, 1.4750992}", "context" -> "System`", 
    "keywords" -> {
     "A New Kind of Science", "Alan Turing", "Chomsky hierarchy", 
      "computation universality", "discrete models", "finite automata", 
      "finite state machines", "formal languages", "Langton's ant", 
      "Mealy machines", "mobile automata", "NKS", "state machines", 
      "theoretical computer science", "theory of computation", "TM", 
      "transducers", "Turing machines", "Turmites", "Turning machines", 
      "universal computer", "universal machine", "universality"}, "index" -> 
    True, "label" -> "Built-in Mathematica Symbol", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "TuringMachine[rule, init, t] generates a list representing the evolution \
of the Turing machine with the specified rule from initial condition init for \
t steps. TuringMachine[rule, init] gives the result of evolving init for one \
step. ", "synonyms" -> {"turing machine"}, "title" -> "TuringMachine", "type" -> 
    "Symbol", "uri" -> "ref/TuringMachine"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[20751, 690, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->621873397]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 167053, 4439}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1670, 40, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2293, 67, 1578, 37, 70, "ObjectNameGrid"],
Cell[3874, 106, 1275, 39, 70, "Usage",
 CellID->31571]
}, Open  ]],
Cell[CellGroupData[{
Cell[5186, 150, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5664, 167, 1298, 47, 70, "Notes",
 CellID->15060],
Cell[6965, 216, 241, 9, 70, "Notes",
 CellID->537794123],
Cell[7209, 227, 504, 18, 70, "Notes",
 CellID->206445663],
Cell[7716, 247, 691, 26, 70, "Notes",
 CellID->1716],
Cell[8410, 275, 1106, 30, 70, "2ColumnTableMod",
 CellID->14597],
Cell[9519, 307, 162, 4, 70, "Notes",
 CellID->13188],
Cell[9684, 313, 247, 8, 70, "Notes",
 CellID->7716],
Cell[9934, 323, 428, 14, 70, "Notes",
 CellID->25756],
Cell[10365, 339, 3847, 119, 70, "2ColumnTableMod",
 CellID->28],
Cell[14215, 460, 91, 1, 70, "Notes",
 CellID->6929],
Cell[14309, 463, 2354, 82, 70, "2ColumnTableMod",
 CellID->23818108],
Cell[16666, 547, 144, 4, 70, "Notes",
 CellID->878982814],
Cell[16813, 553, 170, 6, 70, "Notes",
 CellID->3787],
Cell[16986, 561, 3281, 106, 70, "2ColumnTableMod",
 CellID->468774850],
Cell[20270, 669, 444, 16, 70, "Notes",
 CellID->2278]
}, Closed]],
Cell[CellGroupData[{
Cell[20751, 690, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->621873397],
Cell[CellGroupData[{
Cell[21136, 704, 148, 5, 70, "ExampleSection",
 CellID->203795331],
Cell[21287, 711, 124, 3, 70, "ExampleText",
 CellID->1767263348],
Cell[CellGroupData[{
Cell[21436, 718, 270, 9, 28, "Input",
 CellID->563351562],
Cell[21709, 729, 822, 28, 36, "Output",
 CellID->146637191]
}, Open  ]],
Cell[22546, 760, 125, 3, 70, "ExampleDelimiter",
 CellID->548768179],
Cell[22674, 765, 103, 2, 70, "ExampleText",
 CellID->1852004235],
Cell[CellGroupData[{
Cell[22802, 771, 274, 10, 28, "Input",
 CellID->1745972620],
Cell[23079, 783, 1497, 29, 71, "Output",
 Evaluatable->False,
 CellID->876700516]
}, Open  ]],
Cell[24591, 815, 88, 1, 70, "ExampleText",
 CellID->2548169],
Cell[CellGroupData[{
Cell[24704, 820, 351, 12, 28, "Input",
 CellID->1032968966],
Cell[25058, 834, 728, 16, 246, "Output",
 Evaluatable->False,
 CellID->193053865]
}, Open  ]],
Cell[25801, 853, 125, 3, 70, "ExampleDelimiter",
 CellID->116126494],
Cell[25929, 858, 118, 3, 70, "ExampleText",
 CellID->801102216],
Cell[CellGroupData[{
Cell[26072, 865, 765, 27, 70, "Input",
 CellID->52041894],
Cell[26840, 894, 1112, 36, 54, "Output",
 CellID->507181634]
}, Open  ]],
Cell[27967, 933, 89, 1, 70, "ExampleText",
 CellID->226621018],
Cell[CellGroupData[{
Cell[28081, 938, 658, 22, 70, "Input",
 CellID->79478509],
Cell[28742, 962, 654, 17, 272, "Output",
 CellID->604171094]
}, Open  ]],
Cell[29411, 982, 126, 3, 70, "ExampleDelimiter",
 CellID->1007198027],
Cell[29540, 987, 89, 1, 70, "ExampleText",
 CellID->1357644820],
Cell[CellGroupData[{
Cell[29654, 992, 808, 26, 70, "Input",
 CellID->17552443],
Cell[30465, 1020, 1436, 49, 54, "Output",
 CellID->74312237]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31950, 1075, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[CellGroupData[{
Cell[32196, 1086, 242, 7, 70, "ExampleSubsection",
 CellID->454127739],
Cell[32441, 1095, 75, 1, 70, "ExampleText",
 CellID->829560400],
Cell[CellGroupData[{
Cell[32541, 1100, 409, 14, 70, "Input",
 CellID->473424328],
Cell[32953, 1116, 627, 16, 237, "Output",
 CellID->231174044]
}, Open  ]],
Cell[33595, 1135, 126, 3, 70, "ExampleDelimiter",
 CellID->1446029959],
Cell[33724, 1140, 105, 2, 70, "ExampleText",
 CellID->692006509],
Cell[CellGroupData[{
Cell[33854, 1146, 572, 19, 70, "Input",
 CellID->957806476],
Cell[34429, 1167, 582, 16, 237, "Output",
 CellID->38287460]
}, Open  ]],
Cell[35026, 1186, 125, 3, 70, "ExampleDelimiter",
 CellID->353087814],
Cell[35154, 1191, 89, 1, 70, "ExampleText",
 CellID->1674065987],
Cell[CellGroupData[{
Cell[35268, 1196, 425, 14, 70, "Input",
 CellID->1843784162],
Cell[35696, 1212, 545, 15, 178, "Output",
 CellID->464466309]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[36290, 1233, 239, 7, 70, "ExampleSubsection",
 CellID->368027935],
Cell[36532, 1242, 93, 1, 70, "ExampleText",
 CellID->1407313371],
Cell[CellGroupData[{
Cell[36650, 1247, 397, 13, 70, "Input",
 CellID->670809029],
Cell[37050, 1262, 1602, 30, 70, "Output",
 Evaluatable->False,
 CellID->483853477]
}, Open  ]],
Cell[38667, 1295, 125, 3, 70, "ExampleDelimiter",
 CellID->330938339],
Cell[38795, 1300, 87, 1, 70, "ExampleText",
 CellID->2082457289],
Cell[CellGroupData[{
Cell[38907, 1305, 458, 15, 70, "Input",
 CellID->299931293],
Cell[39368, 1322, 582, 14, 70, "Output",
 Evaluatable->False,
 CellID->299534920]
}, Open  ]],
Cell[39965, 1339, 125, 3, 70, "ExampleDelimiter",
 CellID->849573237],
Cell[CellGroupData[{
Cell[40115, 1346, 490, 17, 70, "Input",
 CellID->98017843],
Cell[40608, 1365, 1320, 43, 81, "Output",
 CellID->2035256602]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[41977, 1414, 243, 7, 70, "ExampleSubsection",
 CellID->864249953],
Cell[42223, 1423, 64, 1, 70, "ExampleText",
 CellID->526230909],
Cell[CellGroupData[{
Cell[42312, 1428, 462, 15, 70, "Input",
 CellID->864144494],
Cell[42777, 1445, 1777, 33, 70, "Output",
 Evaluatable->False,
 CellID->198100154]
}, Open  ]],
Cell[CellGroupData[{
Cell[44591, 1483, 744, 20, 70, "Input",
 CellID->1918526668],
Cell[45338, 1505, 4291, 74, 70, "Output",
 Evaluatable->False,
 CellID->823141781]
}, Open  ]],
Cell[49644, 1582, 126, 3, 70, "ExampleDelimiter",
 CellID->1766150238],
Cell[49773, 1587, 99, 1, 70, "ExampleText",
 CellID->1025229217],
Cell[CellGroupData[{
Cell[49897, 1592, 2501, 79, 70, "Input",
 CellID->1370416316],
Cell[52401, 1673, 708, 18, 123, "Output",
 CellID->415828586]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[53170, 1698, 228, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[53401, 1707, 113, 3, 70, "ExampleText",
 CellID->32738829],
Cell[CellGroupData[{
Cell[53539, 1714, 626, 20, 70, "Input",
 CellID->443064210],
Cell[54168, 1736, 3071, 54, 70, "Output",
 Evaluatable->False,
 CellID->130443471]
}, Open  ]],
Cell[57254, 1793, 125, 3, 70, "ExampleDelimiter",
 CellID->159636520],
Cell[57382, 1798, 112, 3, 70, "ExampleText",
 CellID->14145093],
Cell[CellGroupData[{
Cell[57519, 1805, 766, 24, 70, "Input",
 CellID->528923899],
Cell[58288, 1831, 15650, 261, 70, "Output",
 Evaluatable->False,
 CellID->412074242]
}, Open  ]],
Cell[73953, 2095, 126, 3, 70, "ExampleDelimiter",
 CellID->1350390581],
Cell[74082, 2100, 72, 1, 70, "ExampleText",
 CellID->1654677938],
Cell[CellGroupData[{
Cell[74179, 2105, 2614, 80, 70, "Input",
 CellID->1017714750],
Cell[76796, 2187, 1107, 22, 70, "Output",
 Evaluatable->False,
 CellID->602367172]
}, Open  ]],
Cell[77918, 2212, 125, 3, 70, "ExampleDelimiter",
 CellID->786069035],
Cell[78046, 2217, 90, 1, 70, "ExampleText",
 CellID->142874901],
Cell[CellGroupData[{
Cell[78161, 2222, 638, 18, 70, "Input",
 CellID->1297650679],
Cell[78802, 2242, 6252, 107, 70, "Output",
 Evaluatable->False,
 CellID->420313702]
}, Open  ]],
Cell[85069, 2352, 126, 3, 70, "ExampleDelimiter",
 CellID->1168081232],
Cell[85198, 2357, 115, 3, 70, "ExampleText",
 CellID->1942557265],
Cell[CellGroupData[{
Cell[85338, 2364, 709, 23, 70, "Input",
 CellID->366067514],
Cell[86050, 2389, 655, 17, 199, "Output",
 CellID->1093125282]
}, Open  ]],
Cell[86720, 2409, 125, 3, 70, "ExampleDelimiter",
 CellID->100505616],
Cell[86848, 2414, 122, 3, 70, "ExampleText",
 CellID->1275741570],
Cell[CellGroupData[{
Cell[86995, 2421, 812, 25, 70, "Input",
 CellID->911835342],
Cell[87810, 2448, 646, 17, 201, "Output",
 CellID->434014489]
}, Open  ]],
Cell[88471, 2468, 125, 3, 70, "ExampleDelimiter",
 CellID->120726057],
Cell[88599, 2473, 71, 1, 70, "ExampleText",
 CellID->412745834],
Cell[88673, 2476, 934, 29, 70, "Input",
 CellID->2138810395],
Cell[CellGroupData[{
Cell[89632, 2509, 873, 25, 70, "Input",
 CellID->1872946362],
Cell[90508, 2536, 5284, 91, 70, "Output",
 Evaluatable->False,
 CellID->814828207]
}, Open  ]],
Cell[95807, 2630, 126, 3, 70, "ExampleDelimiter",
 CellID->1793248997],
Cell[95936, 2635, 81, 1, 70, "ExampleText",
 CellID->816969448],
Cell[CellGroupData[{
Cell[96042, 2640, 1339, 40, 70, "Input",
 CellID->1530295598],
Cell[97384, 2682, 4430, 77, 70, "Output",
 Evaluatable->False,
 CellID->201523900]
}, Open  ]],
Cell[101829, 2762, 126, 3, 70, "ExampleDelimiter",
 CellID->1057808415],
Cell[101958, 2767, 97, 1, 70, "ExampleText",
 CellID->2012325058],
Cell[102058, 2770, 934, 29, 70, "Input",
 CellID->1363213689],
Cell[CellGroupData[{
Cell[103017, 2803, 1038, 34, 70, "Input",
 CellID->508293099],
Cell[104058, 2839, 2561, 47, 201, "Output",
 CellID->1714030618]
}, Open  ]],
Cell[106634, 2889, 125, 3, 70, "ExampleDelimiter",
 CellID->647052228],
Cell[106762, 2894, 117, 3, 70, "ExampleText",
 CellID->149607588],
Cell[CellGroupData[{
Cell[106904, 2901, 568, 19, 70, "Input",
 CellID->2089392171],
Cell[107475, 2922, 869, 20, 130, "Output",
 CellID->1748663012]
}, Open  ]],
Cell[108359, 2945, 126, 3, 70, "ExampleDelimiter",
 CellID->1180076596],
Cell[108488, 2950, 84, 1, 70, "ExampleText",
 CellID->1535868927],
Cell[CellGroupData[{
Cell[108597, 2955, 1727, 52, 70, "Input",
 CellID->595619149],
Cell[110327, 3009, 52061, 1281, 70, "Output",
 CellID->985080493]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[162449, 4297, 311, 9, 70, "SeeAlsoSection",
 CellID->27357],
Cell[162763, 4308, 471, 17, 70, "SeeAlso",
 CellID->32532]
}, Open  ]],
Cell[CellGroupData[{
Cell[163271, 4330, 320, 9, 70, "RelatedLinksSection",
 CellID->3616],
Cell[163594, 4341, 371, 12, 70, "RelatedLinks"],
Cell[163968, 4355, 504, 15, 70, "RelatedLinks",
 CellID->28900]
}, Open  ]],
Cell[CellGroupData[{
Cell[164509, 4375, 319, 9, 70, "MoreAboutSection",
 CellID->283121036],
Cell[164831, 4386, 165, 3, 70, "MoreAbout",
 CellID->400464714],
Cell[164999, 4391, 150, 3, 70, "MoreAbout",
 CellID->94766179],
Cell[165152, 4396, 179, 3, 70, "MoreAbout",
 CellID->421165893]
}, Open  ]],
Cell[165346, 4402, 27, 0, 70, "History"],
Cell[165376, 4404, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

