(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     42021,       1417]
NotebookOptionsPosition[     35186,       1177]
NotebookOutlinePosition[     36394,       1213]
CellTagsIndexPosition[     36308,       1208]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"WhittakerM" :> 
          Documentation`HelpLookup["paclet:ref/WhittakerM"], 
          "ParabolicCylinderD" :> 
          Documentation`HelpLookup["paclet:ref/ParabolicCylinderD"], 
          "HypergeometricU" :> 
          Documentation`HelpLookup["paclet:ref/HypergeometricU"], 
          "Hypergeometric1F1" :> 
          Documentation`HelpLookup["paclet:ref/Hypergeometric1F1"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"WhittakerM\"\>", 
       2->"\<\"ParabolicCylinderD\"\>", 3->"\<\"HypergeometricU\"\>", 
       4->"\<\"Hypergeometric1F1\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Hypergeometric Functions" :> 
          Documentation`HelpLookup["paclet:guide/HypergeometricFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Hypergeometric Functions\"\>", 
       2->"\<\"New in 6.0: Mathematical Functions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["WhittakerW", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["WhittakerW",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/WhittakerW"], "[", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the Whittaker function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["W", 
         RowBox[{"k", ",", "m"}]], "(", "z", ")"}], TraditionalForm]], 
      "InlineMath"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->21913]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerW",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerW"]], "InlineFormula"],
 " is related to the confluent hypergeometric function by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["W", 
      RowBox[{"k", ",", "m"}]], "(", "z", ")"}], "=", 
    RowBox[{
     SuperscriptBox["e", 
      RowBox[{"-", "z", "/", "2"}]], 
     SuperscriptBox["z", 
      RowBox[{"m", "+", "1", "/", "2"}]], " ", 
     RowBox[{"U", "(", 
      RowBox[{
       RowBox[{"m", "-", "k", "+", 
        FractionBox["1", "2"]}], ";", 
       RowBox[{"1", "+", "2", "m"}], ";", "z"}], ")"}]}]}], TraditionalForm]],
   "InlineMath"],
 ". "
}], "Notes",
 CellID->22713],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["W", 
     RowBox[{"k", ",", "m"}]], "(", "z", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is infinite at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", "=", "0"}], TraditionalForm]], "InlineMath"],
 " for integer ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", ">", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->743],

Cell[TextData[{
 "For certain special arguments, ",
 Cell[BoxData[
  ButtonBox["WhittakerW",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerW"]], "InlineFormula"],
 " automatically evaluates to exact values."
}], "Notes",
 CellID->377576220],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerW",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerW"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerW",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerW"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["WhittakerW",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WhittakerW"], "[", 
   RowBox[{
    StyleBox["k", "TI"], ",", 
    StyleBox["m", "TI"], ",", 
    StyleBox["z", "TI"]}], "]"}]], "InlineFormula"],
 " has a branch cut discontinuity in the complex ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " plane running from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->29416]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->10498291],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->150497096],

Cell["Evaluate numerically:", "ExampleText",
 CellID->231069639],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerW", "[", 
  RowBox[{"2", ",", "3", ",", "4.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22333397],

Cell[BoxData["11.306294801893106`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17490563]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1249957],

Cell[TextData[{
 "Plot ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox[
     TagBox["W",
      WhittakerW], 
     RowBox[{"2", ",", 
      RowBox[{"1", "/", "2"}]}]], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->469428808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"WhittakerW", "[", 
    RowBox[{"2", ",", 
     RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "15"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->319759576],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->448193302]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19586552],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand in terms of hypergeometric functions:"
}], "ExampleText",
 CellID->308640546],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{" ", 
  RowBox[{"FunctionExpand", "[", 
   RowBox[{"WhittakerW", "[", 
    RowBox[{"k", ",", "m", ",", "x"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->52307413],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", "x"}], "/", "2"}]], " ", 
  SuperscriptBox["x", 
   RowBox[{
    FractionBox["1", "2"], "+", "m"}]], " ", 
  RowBox[{"HypergeometricU", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", "2"], "-", "k", "+", "m"}], ",", 
    RowBox[{"1", "+", 
     RowBox[{"2", " ", "m"}]}], ",", "x"}], "]"}]}]], "Output",
 ImageSize->{300, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->530061813]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14479],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->192155330],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerW", "[", 
  RowBox[{
   RowBox[{"1", "+", "I"}], ",", "1.5", ",", 
   RowBox[{"2", "-", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->130939040],

Cell[BoxData[
 RowBox[{"1.2788861577646713`", "\[InvisibleSpace]", "+", 
  RowBox[{"3.5686601912205798`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->373904972]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->181032742],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->29664063],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"WhittakerW", "[", 
    RowBox[{"1", ",", "2", ",", "3"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->477389994],

Cell[BoxData["2.\
1176517466292269823865367440784107047102673914003758799562936354093`50."], \
"Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->493812680]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->505722001],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->121593466],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerW", "[", 
  RowBox[{"1", ",", "2", ",", "3.00000000000000000000000"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->121305533],

Cell[BoxData["2.11765174662922698238653706462908157198`22.240817401570958"], \
"Output",
 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->37922464]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->374030955],

Cell["Evaluate symbolically at the origin:", "ExampleText",
 CellID->11121899],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerW", "[", 
  RowBox[{"k", ",", 
   RowBox[{"1", "/", "2"}], ",", "0"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->253284775],

Cell[BoxData[
 FractionBox["1", 
  RowBox[{"Gamma", "[", 
   RowBox[{"1", "-", "k"}], "]"}]]], "Output",
 ImageSize->{84, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->570984785]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9036255],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["WhittakerW",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerW"]], "InlineFormula"],
 " threads element\[Hyphen]wise over lists:"
}], "ExampleText",
 CellID->221234742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"WhittakerW", "[", 
  RowBox[{"2", ",", "3", ",", 
   RowBox[{"{", 
    RowBox[{"1.5", ",", "2.5", ",", "3.5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->772982833],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "86.38286431253367`", ",", "32.876723534790415`", ",", 
   "17.992042770417793`"}], "}"}]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->134719228]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->203792484],

Cell["Series expansion:", "ExampleText",
 CellID->114119141],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"WhittakerW", "[", 
    RowBox[{"k", ",", "1", ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11716619],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox["4", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"4", " ", 
        SuperscriptBox["k", "2"]}]}], ")"}], " ", 
     RowBox[{"Gamma", "[", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "-", "k"}], "]"}], " ", 
     SqrtBox["x"]}]], "+", 
   FractionBox[
    RowBox[{"4", " ", "k", " ", 
     SqrtBox["x"]}], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"2", " ", "k"}]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"2", " ", "k"}]}], ")"}], " ", 
     RowBox[{"Gamma", "[", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "-", "k"}], "]"}]}]], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], 
     RowBox[{"3", "/", "2"}]],
    SeriesData[$CellContext`x, 0, {}, -1, 3, 2],
    Editable->False]}],
  SeriesData[$CellContext`x, 0, {
   4 (-1 + 4 $CellContext`k^2)^(-1)/Gamma[Rational[-1, 2] - $CellContext`k], 
    0, 4 $CellContext`k (-1 + 2 $CellContext`k)^(-1) (1 + 
      2 $CellContext`k)^(-1)/Gamma[Rational[-1, 2] - $CellContext`k]}, -1, 3, 
   2],
  Editable->False]], "Output",
 ImageSize->{463, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->271742523]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->79869998],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->871763648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"WhittakerW", "[", 
   RowBox[{"m", ",", "k", ",", "x"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7891564],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox[
    TagBox["W",
     WhittakerW], 
    RowBox[{"m", ",", "k"}]], "(", "x", ")"}], TraditionalForm]], "Output",
 ImageSize->{42, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->723101791]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->19700],

Cell["Green's function of the 3D Coulomb potential:", "ExampleText",
 CellID->143641663],

Cell[BoxData[
 RowBox[{
  RowBox[{"gf", "[", 
   RowBox[{"r1_", ",", "r2_", ",", "e_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"\[Nu]", ",", "a1", ",", "a2", ",", "\[Delta]", ",", 
      RowBox[{"\[ScriptV]\[ScriptL]", "=", 
       RowBox[{
        RowBox[{"Sqrt", "[", 
         RowBox[{"Total", "[", 
          RowBox[{"#1", "^", "2"}], "]"}], "]"}], "&"}]}]}], "}"}], ",", 
    RowBox[{
     RowBox[{"\[Nu]", "=", 
      RowBox[{"e", "^", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "1"}], "/", "2"}], ")"}]}]}], ";", 
     RowBox[{"\[Delta]", "=", 
      RowBox[{"\[ScriptV]\[ScriptL]", "[", 
       RowBox[{"r2", "-", "r1"}], "]"}]}], ";", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"a1", ",", "a2"}], "}"}], "=", " ", 
      RowBox[{
       RowBox[{"\[ScriptV]\[ScriptL]", "[", "r2", "]"}], "+", 
       RowBox[{"\[ScriptV]\[ScriptL]", "[", "r1", "]"}], "+", 
       RowBox[{"{", 
        RowBox[{"\[Delta]", ",", 
         RowBox[{"-", "\[Delta]"}]}], "}"}]}]}], ";", 
     RowBox[{
      FractionBox[
       RowBox[{"-", 
        RowBox[{"Gamma", "[", 
         RowBox[{"1", "-", 
          RowBox[{"I", " ", "\[Nu]"}]}], "]"}]}], 
       RowBox[{"4", " ", "\[Pi]", " ", "\[Delta]"}]], 
      RowBox[{"Det", "[", 
       RowBox[{"(", GridBox[{
          {
           RowBox[{"WhittakerW", "[", 
            RowBox[{
             RowBox[{"\[ImaginaryI]", " ", "\[Nu]"}], ",", 
             FractionBox["1", "2"], ",", 
             RowBox[{"-", 
              FractionBox[
               RowBox[{"\[ImaginaryI]", " ", "a2"}], "\[Nu]"]}]}], "]"}], 
           RowBox[{"WhittakerM", "[", 
            RowBox[{
             RowBox[{"\[ImaginaryI]", " ", "\[Nu]"}], ",", 
             FractionBox["1", "2"], ",", 
             RowBox[{"-", 
              FractionBox[
               RowBox[{"\[ImaginaryI]", " ", "a1"}], "\[Nu]"]}]}], "]"}]},
          {
           RowBox[{
            SuperscriptBox["WhittakerW", 
             TagBox[
              RowBox[{"(", 
               RowBox[{"0", ",", "0", ",", "1"}], ")"}],
              Derivative],
             MultilineFunction->None], "[", 
            RowBox[{
             RowBox[{"\[ImaginaryI]", " ", "\[Nu]"}], ",", 
             FractionBox["1", "2"], ",", 
             RowBox[{"-", 
              FractionBox[
               RowBox[{"\[ImaginaryI]", " ", "a2"}], "\[Nu]"]}]}], "]"}], 
           RowBox[{
            SuperscriptBox["WhittakerM", 
             TagBox[
              RowBox[{"(", 
               RowBox[{"0", ",", "0", ",", "1"}], ")"}],
              Derivative],
             MultilineFunction->None], "[", 
            RowBox[{
             RowBox[{"\[ImaginaryI]", " ", "\[Nu]"}], ",", 
             FractionBox["1", "2"], ",", 
             RowBox[{"-", 
              FractionBox[
               RowBox[{"\[ImaginaryI]", " ", "a1"}], "\[Nu]"]}]}], "]"}]}
         }], ")"}], "]"}]}]}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->247595368],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"gf", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", "e"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"e", ",", 
     RowBox[{"-", "2"}], ",", ".01"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->345537891],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WnmMVdUZv7513iyICxZjFdSIu0VtrZharZQRhmVgQEAQESZNwC4qYqI2
bdpopRpcqnEhIg6bw6KJVqI14r6hrcUQNabWmCFp4A+JTUlNSExef9/3ne2e
e+59C2/6Am/uucvv9+3nO+e+6UtWLvvZL5esXN6/ZOyVK5bcuGx5/81ju29Y
gVP5I6Io2oH/14+OcFzFEf9/ulqtRnk64k9UpK9oIZ89ePCQe9YZX4CvHIYV
GhQOHfr2wIFv5BKdjQr79x9Ud0ZR2UG8ZeULn3y8nx/i8e6//+vXt79EhxfS
swvmDwri6nveWHztNoWIh6L8VXM2rx/4kC7mgFlyMG/41fOAsZhDQ18vWrjV
YM7p2ySYq+56bd7cpxQmHooKs2ZuXPPorqSUFlH0vuzSNa7eQMzTXzyV55tw
YMBxHDWGDmEhsoueY3mtTxRPwSVpjAMmfH/XXjocjy+CTuHIgyOSD+kXFYHk
kmSw5J5c97cLeTDvqqesyTRXm+HCR1gcTfKWhIKxmKHLW29+qWLmgnS2kscG
u+UatpqNpOFj4kjOT53ypCRXdD6PDRNzEF6h6QgoeWOfoa3p7BDkhx58d/u2
PTHknsnrqAykIJOvkeDVzFwueb422MiPffv+kyU19Hpl5z9rSA35YPRU5HYX
GdmzfNmzgvziC5+lIvNZJ5/H10SGcv1Ln6kX2Vq61cjWzq1Ghp1n9W7EDBFG
rmQhp0cHP+tUte9pbPgJtxyG1IL83LOf3Lv6reFAtvZoHXLFs8Z5Gvm2W1+k
TDhcZBsdBvm3v3mZnhMfIi6nThkII6dXc8GGJWAPqhZB7PYs7DSp271sObfF
yAcO/Bdg5EODfPcfXweZqdYyl+IAJDRT5Hku9ShSODo86c+hL5p7HnrwHUyx
/Iyae0a4c4/oY6ZVQ/b73+2swWX1Oasq3QHzu0wdSSa5bCwX5cCV1klaxs6q
dApGqy2DH62663WXq5Tkyml7OlzpkdWpIwulmCrN2Zrr+T9/qgK5Qa7aenXx
GAmINKTDszTnKzs/x12pnDCFPt8w5wgew3qIJzo8U0fjnj378B3vHcpgIlT+
NOq1EdqiqFWf/+OrBFObw+BlVJ0MR/IY3upf+jRVgzM0A04RQJLhzjteXXD1
ltrRMJLHWHbgFMX46TQuCDJKD4K/rLGvuPxx3FHVfVz90h+lvQ+1SfpxDgdM
BnLDgQjDySY4jtYcOEUcpzkcUvsb56DZteDGn7pKUTMGB3nvml4XnKSuafl0
bNM18SSiBJ6kWOmqejXRwelydJCKxzoUPR2iHHXVQQ3KHqr0ye66ZoynW8kb
f9d5QrRJro1MdMKZVOc7EneUPdR2RzOkJ5I0WzPb64ih7Bos7yHnE9zfUV4s
1NB0lLrPRUjTmLOGVvZwiwR1vpq2cszSHaEPmxndoa2Ec7ZXK348DotXJaPQ
3UBUdHxqTVGvZrDNpIlrjWaoRdJ+tVoz8qrOx1oaHcNn4Cx0r0g/3FNK3FNU
eJx7RdEFhypb+X6qIajeMq0ACgUECmXUkDhD7Rz0I/WkOnQbxWcgI25AvcNB
kqfo4UonYHSMqPMwGv7i589t3rS7quf9dA3LymIlxxO5yJ4/Uf3lxFIFUe8Q
jWYZERmwL/II6ZD3nu6M+0ESxvoBeNL11ZLSny/EpohuwOEsFgIsS8kwGacX
NRf6Y3SLGK19/K+YCzVX+jpsLNNjgMeAC3NiOhVPxXhQMyG+4ZHorHL3A/z5
8wbr1OkUvgTYNY+9j7kbrLCsRH3Z5UNuYkiymN4HT8iCH8fI24ULtqRwco6X
wInuEQS4HRyAmurJwvNvEWZFMEoBoVl5pB9xPMZdYliJC2r6lMdxOGPa+pgF
1E6QSCMzzdDQ14DAveDKQfGogAMM4V1gdHuSna6tRKnbuxFzD55XPZyVTOYV
iAVkLSlLVpGOEoeLrtk6fy7tE1126WMQjJYtJZMR2/VcoFZFnh27zBj6QDDA
If6pWYwiiWmYlSzmz29+DvuV3b/e9Di219c6fpldZWcbZ+EiRoh1/eIFtl7R
nJcgEZuWZV2AsIMDpQrM7ttkY0QtBJ2IdSTgyl4xZkeoIPIRzeJXioWR3hOn
eTYZF9KRai90kn62zYsmsQIuQ9+qqrR8ErwP3P82Ksv0aetT5B9D322QSxol
PAGlAYWqAabvV+PRLftIUiaR1GqO9iWxe7dGEhgXYmB6ntozkCLJqSwJrA4t
ULDAgFIB+6GwQfnxniS8K5zHNchKK4uiJwn7vSI2AY0wQSnAwiTdk55IkYLz
tw2IyBBoiFIJ/0miwqnneFLwnrvaoyEpCiEpOiS4jBRARsPyyMPvTbxi7aze
DSSGrfK8Xi6DSh4COdggN62lz/TYxTvx+pZg75LdVsMOW6KlkOKv2NkEip93
xkoggAuABP2FfJzH/YNqsoK53BwZnbxz8q4Qw7OTu9cliYWX/VmCcMILNyEg
IfqpQV67axLkpVwQi6uMBlmqxoxYREwj/KViUy92ssd7Ectn9DV1RHIRkbLi
ph0+H/x73bXbfL4fMh9kh4IwLygQP6pL8/kwkA0Mjw9xjw7K8sGwiKdJE59Y
vCjBd4lTE8EHshM9LpHIvAcRIrVnCnWvW7zdMOGuAXjIMtnI/ZFTp8ADg54Q
5DF1KsaDW9DyWI18HlejH2smRAnQEAzHB5mk86nFNGf25nvufiPMdLnOb9Qz
8tLoII/xUowHVRDdheUBKepfmOcnrBHMD3UQgAiO44JM5s1JjAlHMFd9TBP5
CYQebEcIozyei1kS2ZtxIw4ioSu1cSAcqOYL5g/G4+CnfL/dgzg2yCAXhYHX
ZR2SMyrWPXRXg0mMB0sAkvCPCeJLpjaD3+3hHx3ER3uHrlPwY2+kaUZRe328
S1NKEgnPlZoHnqC+4KggDzIdxcXpWZLvjZVmkd6HSdVsso4Wtc83MsgojUA2
o9QbOZNlyynMYHdmjwwyyvyZyViRemBOxhjt/mNPVVVjWVyqvtvnk3zN1jCL
z9WwR2eH2u8OM+IK6GoyuifvvOPVMKOshaQHo8Oumow6PguOZhF/+pc+A8sL
z+23/iXGM41xbVfZGeSRLtnNMxQ9dFiih4u/6g+vBfAR+YhGmsl9/Ak8hvaq
1jF6RaItjG7jYDo/a9/jdASx0V/OnWOxO7KwXckFXSpcOrp929oY+gx+2r5R
bB8GdPtWsRXotOci6L38tMwe7K4gOszWN3NTfT6180qvl9thbPrNTO9Gg92e
LXnZQbb+bB3yTH7S/iqhLYgMX8+csaEZZBvhYWQUcqRjM8imp2kh8iyddfKi
R/XqNZArglYrOgTbVqu6sNuD2OvWfrB96x5PatNL1odcDiJHOQJO7HoJg63o
tRhi/UabKeOMZUt6mKuPsexvnvxfKE3QWTath96UpP1WrRykTNGvz+MIc5q3
M6m/jwOPTFvy+7iCkIIS/2zt7/Myo0m2srA1oqF0TofJKTtXdXPaqvX/4Jzt
RU+xBmes82goZmYrj9MqIZ0JFkcr2lzWCYPdwThcBluLBNlWjOGS3fbVw8Vg
d/KGiwFTAdZRlgHOUL3DBC3Bn+5/m/pDqdgbBj7c9d6QAA1u3o1qf9/qN/fu
/fcXX3xlKjalg60HOyTgjvgf9QVeKA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->335665987]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->31086],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["FunctionExpand",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FunctionExpand"]], "InlineFormula"],
 " to expand ",
 Cell[BoxData[
  ButtonBox["WhittakerW",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WhittakerW"]], "InlineFormula"],
 " into other functions:"
}], "ExampleText",
 CellID->156298838],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"WhittakerW", "[", 
   RowBox[{"5", ",", 
    RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->42482196],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", "x"}], "/", "2"}]], " ", "x", " ", 
  RowBox[{"(", 
   RowBox[{"120", "-", 
    RowBox[{"240", " ", "x"}], "+", 
    RowBox[{"120", " ", 
     SuperscriptBox["x", "2"]}], "-", 
    RowBox[{"20", " ", 
     SuperscriptBox["x", "3"]}], "+", 
    SuperscriptBox["x", "4"]}], ")"}]}]], "Output",
 ImageSize->{240, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->172822021]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"WhittakerW", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "/", "4"}], ",", 
    RowBox[{"1", "/", "4"}], ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->417632044],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"x", "/", "2"}]], " ", 
  SqrtBox["\[Pi]"], " ", 
  SuperscriptBox["x", 
   RowBox[{"1", "/", "4"}]], " ", 
  RowBox[{"(", 
   RowBox[{"1", "-", 
    RowBox[{"Erf", "[", 
     SqrtBox["x"], "]"}]}], ")"}]}]], "Output",
 ImageSize->{176, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->56783456]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FunctionExpand", "[", 
  RowBox[{"WhittakerW", "[", 
   RowBox[{"0", ",", "m", ",", "x"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->285678544],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox["x"], " ", 
   RowBox[{"BesselK", "[", 
    RowBox[{"m", ",", 
     FractionBox["x", "2"]}], "]"}]}], 
  SqrtBox["\[Pi]"]]], "Output",
 ImageSize->{127, 44},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->126336153]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30204456],

Cell["Integrate expressions involving Whittaker functions:", "ExampleText",
 CellID->696244865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    SuperscriptBox["x", "a"], 
    RowBox[{"WhittakerM", "[", 
     RowBox[{"k", ",", "m", ",", "x"}], "]"}], 
    RowBox[{"WhittakerW", "[", 
     RowBox[{
      RowBox[{"-", "k"}], ",", "m", ",", "x"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{
     RowBox[{"k", ">", 
      RowBox[{"1", "/", "2"}]}], "&&", 
     RowBox[{"m", ">", "0"}], "&&", 
     RowBox[{
      RowBox[{
       RowBox[{"2", "k"}], "-", "1"}], ">", "a", ">=", "0"}]}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->41947832],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"2", "+", "a"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", "2"]}], "-", 
     FractionBox["a", "2"], "+", "k"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", 
     FractionBox["a", "2"], "+", "m"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", 
     RowBox[{"2", " ", "m"}]}], "]"}]}], 
  RowBox[{"2", " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{
     FractionBox["3", "2"], "+", 
     FractionBox["a", "2"], "+", "k"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["a", "2"]}], "+", "m"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{
     FractionBox["1", "2"], "+", "k", "+", "m"}], "]"}]}]]], "Output",
 ImageSize->{386, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9852860]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12195],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["WhittakerM",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/WhittakerM"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ParabolicCylinderD",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ParabolicCylinderD"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HypergeometricU",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HypergeometricU"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Hypergeometric1F1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Hypergeometric1F1"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->5423]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->29240],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->30082]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->86801182],

Cell[TextData[ButtonBox["Hypergeometric Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/HypergeometricFunctions"]], "MoreAbout",
 CellID->393564458],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->2442811]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"WhittakerW - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 29, 36.9619186}", "context" -> "System`", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "WhittakerW[k, m, z] gives the Whittaker function W k, m (z). ", 
    "synonyms" -> {"Whittaker W"}, "title" -> "WhittakerW", "type" -> 
    "Symbol", "uri" -> "ref/WhittakerW"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8414, 277, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->10498291]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 36166, 1201}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2356, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2979, 84, 1575, 37, 70, "ObjectNameGrid"],
Cell[4557, 123, 745, 25, 70, "Usage",
 CellID->21913]
}, Open  ]],
Cell[CellGroupData[{
Cell[5339, 153, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5817, 170, 120, 4, 70, "Notes",
 CellID->20640],
Cell[5940, 176, 726, 25, 70, "Notes",
 CellID->22713],
Cell[6669, 203, 409, 17, 70, "Notes",
 CellID->743],
Cell[7081, 222, 252, 8, 70, "Notes",
 CellID->377576220],
Cell[7336, 232, 222, 7, 70, "Notes",
 CellID->31226],
Cell[7561, 241, 205, 7, 70, "Notes",
 CellID->28162],
Cell[7769, 250, 608, 22, 70, "Notes",
 CellID->29416]
}, Closed]],
Cell[CellGroupData[{
Cell[8414, 277, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->10498291],
Cell[CellGroupData[{
Cell[8798, 291, 148, 5, 70, "ExampleSection",
 CellID->150497096],
Cell[8949, 298, 64, 1, 70, "ExampleText",
 CellID->231069639],
Cell[CellGroupData[{
Cell[9038, 303, 140, 4, 28, "Input",
 CellID->22333397],
Cell[9181, 309, 175, 5, 36, "Output",
 CellID->17490563]
}, Open  ]],
Cell[9371, 317, 123, 3, 70, "ExampleDelimiter",
 CellID->1249957],
Cell[9497, 322, 274, 13, 70, "ExampleText",
 CellID->469428808],
Cell[CellGroupData[{
Cell[9796, 339, 282, 9, 28, "Input",
 CellID->319759576],
Cell[10081, 350, 2673, 48, 121, "Output",
 Evaluatable->False,
 CellID->448193302]
}, Open  ]],
Cell[12769, 401, 124, 3, 70, "ExampleDelimiter",
 CellID->19586552],
Cell[12896, 406, 246, 8, 70, "ExampleText",
 CellID->308640546],
Cell[CellGroupData[{
Cell[13167, 418, 200, 6, 70, "Input",
 CellID->52307413],
Cell[13370, 426, 530, 18, 52, "Output",
 CellID->530061813]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13949, 450, 216, 7, 70, "ExampleSection",
 CellID->14479],
Cell[14168, 459, 89, 1, 70, "ExampleText",
 CellID->192155330],
Cell[CellGroupData[{
Cell[14282, 464, 189, 6, 70, "Input",
 CellID->130939040],
Cell[14474, 472, 273, 7, 36, "Output",
 CellID->373904972]
}, Open  ]],
Cell[14762, 482, 125, 3, 70, "ExampleDelimiter",
 CellID->181032742],
Cell[14890, 487, 69, 1, 70, "ExampleText",
 CellID->29664063],
Cell[CellGroupData[{
Cell[14984, 492, 194, 6, 70, "Input",
 CellID->477389994],
Cell[15181, 500, 235, 7, 36, "Output",
 CellID->493812680]
}, Open  ]],
Cell[15431, 510, 125, 3, 70, "ExampleDelimiter",
 CellID->505722001],
Cell[15559, 515, 113, 3, 70, "ExampleText",
 CellID->121593466],
Cell[CellGroupData[{
Cell[15697, 522, 163, 4, 70, "Input",
 CellID->121305533],
Cell[15863, 528, 218, 6, 36, "Output",
 CellID->37922464]
}, Open  ]],
Cell[16096, 537, 125, 3, 70, "ExampleDelimiter",
 CellID->374030955],
Cell[16224, 542, 78, 1, 70, "ExampleText",
 CellID->11121899],
Cell[CellGroupData[{
Cell[16327, 547, 163, 5, 70, "Input",
 CellID->253284775],
Cell[16493, 554, 234, 8, 53, "Output",
 CellID->570984785]
}, Open  ]],
Cell[16742, 565, 123, 3, 70, "ExampleDelimiter",
 CellID->9036255],
Cell[16868, 570, 222, 7, 70, "ExampleText",
 CellID->221234742],
Cell[CellGroupData[{
Cell[17115, 581, 204, 6, 70, "Input",
 CellID->772982833],
Cell[17322, 589, 274, 9, 36, "Output",
 CellID->134719228]
}, Open  ]],
Cell[17611, 601, 125, 3, 70, "ExampleDelimiter",
 CellID->203792484],
Cell[17739, 606, 60, 1, 70, "ExampleText",
 CellID->114119141],
Cell[CellGroupData[{
Cell[17824, 611, 256, 8, 70, "Input",
 CellID->11716619],
Cell[18083, 621, 1387, 46, 67, "Output",
 CellID->271742523]
}, Open  ]],
Cell[19485, 670, 124, 3, 70, "ExampleDelimiter",
 CellID->79869998],
Cell[19612, 675, 203, 7, 70, "ExampleText",
 CellID->871763648],
Cell[CellGroupData[{
Cell[19840, 686, 179, 6, 70, "Input",
 CellID->7891564],
Cell[20022, 694, 309, 11, 52, "Output",
 CellID->723101791]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20380, 711, 223, 7, 70, "ExampleSection",
 CellID->19700],
Cell[20606, 720, 88, 1, 70, "ExampleText",
 CellID->143641663],
Cell[20697, 723, 3004, 84, 70, "Input",
 CellID->247595368],
Cell[CellGroupData[{
Cell[23726, 811, 396, 13, 70, "Input",
 CellID->345537891],
Cell[24125, 826, 3905, 68, 70, "Output",
 Evaluatable->False,
 CellID->335665987]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[28079, 900, 233, 7, 70, "ExampleSection",
 CellID->31086],
Cell[28315, 909, 359, 13, 70, "ExampleText",
 CellID->156298838],
Cell[CellGroupData[{
Cell[28699, 926, 205, 6, 70, "Input",
 CellID->42482196],
Cell[28907, 934, 512, 17, 39, "Output",
 CellID->172822021]
}, Open  ]],
Cell[CellGroupData[{
Cell[29456, 956, 252, 8, 70, "Input",
 CellID->417632044],
Cell[29711, 966, 420, 15, 43, "Output",
 CellID->56783456]
}, Open  ]],
Cell[CellGroupData[{
Cell[30168, 986, 181, 5, 70, "Input",
 CellID->285678544],
Cell[30352, 993, 315, 12, 65, "Output",
 CellID->126336153]
}, Open  ]],
Cell[30682, 1008, 124, 3, 70, "ExampleDelimiter",
 CellID->30204456],
Cell[30809, 1013, 95, 1, 70, "ExampleText",
 CellID->696244865],
Cell[CellGroupData[{
Cell[30929, 1018, 674, 22, 70, "Input",
 CellID->41947832],
Cell[31606, 1042, 947, 32, 68, "Output",
 CellID->9852860]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32614, 1081, 311, 9, 70, "SeeAlsoSection",
 CellID->12195],
Cell[32928, 1092, 996, 35, 70, "SeeAlso",
 CellID->5423]
}, Open  ]],
Cell[CellGroupData[{
Cell[33961, 1132, 314, 9, 70, "TutorialsSection",
 CellID->29240],
Cell[34278, 1143, 144, 3, 70, "Tutorials",
 CellID->30082]
}, Open  ]],
Cell[CellGroupData[{
Cell[34459, 1151, 318, 9, 70, "MoreAboutSection",
 CellID->86801182],
Cell[34780, 1162, 159, 3, 70, "MoreAbout",
 CellID->393564458],
Cell[34942, 1167, 172, 3, 70, "MoreAbout",
 CellID->2442811]
}, Open  ]],
Cell[35129, 1173, 27, 0, 70, "History"],
Cell[35159, 1175, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

