(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    100015,       2196]
NotebookOptionsPosition[     93923,       1987]
NotebookOutlinePosition[     95214,       2024]
CellTagsIndexPosition[     95128,       2019]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Orthogonal Polynomials" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/OrthogonalPolynomials"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Orthogonal Polynomials\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"], 
          "JacobiP" :> Documentation`HelpLookup["paclet:ref/JacobiP"], 
          "SphericalHarmonicY" :> 
          Documentation`HelpLookup["paclet:ref/SphericalHarmonicY"], 
          "Hypergeometric2F1" :> 
          Documentation`HelpLookup["paclet:ref/Hypergeometric2F1"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BesselJ\"\>", 2->"\<\"JacobiP\"\>", 
       3->"\<\"SphericalHarmonicY\"\>", 
       4->"\<\"Hypergeometric2F1\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Functions Used in Optics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInOptics"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Functions Used in Optics\"\>", 
       2->"\<\"Special Functions\"\>", 
       3->"\<\"New in 6.0: Mathematical Functions\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ZernikeR", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ZernikeR",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ZernikeR"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["m", "TI"], ",", 
        StyleBox["r", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the radial Zernike polynomial ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["R", "n", "m"], "(", "r", ")"}], TraditionalForm]], 
      "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell["Explicit polynomials are given when possible. ", "Notes",
 CellID->26319],

Cell[TextData[{
 "The Zernike polynomials are orthogonal with weight ",
 Cell[BoxData[
  FormBox["r", TraditionalForm]], "InlineMath"],
 " over the unit interval."
}], "Notes",
 CellID->457162106],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ZernikeR",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ZernikeR"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ZernikeR",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ZernikeR"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->199128792],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->313299963],

Cell["Evaluate numerically:", "ExampleText",
 CellID->75293636],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ZernikeR", "[", 
  RowBox[{"3", ",", "1", ",", "0.5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->514286172],

Cell[BoxData[
 RowBox[{"-", "0.625`"}]], "Output",
 ImageSize->{47, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->393298414]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->99617739],

Cell["Evaluate symbolically:", "ExampleText",
 CellID->916987318],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ZernikeR", "[", 
  RowBox[{"5", ",", "3", ",", "r"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->177321292],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["r", "3"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "4"}], "+", 
    RowBox[{"5", " ", 
     SuperscriptBox["r", "2"]}]}], ")"}]}]], "Output",
 ImageSize->{84, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->484390320]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->270817465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"ZernikeR", "[", 
    RowBox[{"7", ",", "1", ",", "r"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->253820562],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->607555774]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Evaluate for complex arguments and parameters:", "ExampleText",
 CellID->62944296],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ZernikeR", "[", 
  RowBox[{
   RowBox[{"1.3", "-", "I"}], ",", "0.7", ",", 
   RowBox[{"2.5", "+", "I"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->318679557],

Cell[BoxData[
 RowBox[{"1.8430411284527144`", "\[InvisibleSpace]", "-", 
  RowBox[{"2.704387078399177`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->260864957]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->206642428],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->152036528],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"ZernikeR", "[", 
    RowBox[{
     RowBox[{"5", "/", "2"}], ",", 
     RowBox[{"1", "/", "2"}], ",", 
     RowBox[{"1", "/", "3"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->69905385],

Cell[BoxData[
 RowBox[{"-", 
  "0.705650329009542601066737398391281334680402140441266181880844713205`50."}]\
], "Output",
 ImageSize->{376, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->100474483]
}, Open  ]],

Cell["\<\
The precision of the output tracks the precision of the input:\
\>", "ExampleText",
 CellID->150856807],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ZernikeR", "[", 
  RowBox[{
   RowBox[{"5", "/", "2"}], ",", 
   RowBox[{"1", "/", "2"}], ",", "0.33333333333333333333333333333"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7664213],

Cell[BoxData[
 RowBox[{"-", 
  "0.70565032900954260106673739839096058453`28.21485512857035"}]], "Output",
 ImageSize->{222, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->129572938]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->181029827],

Cell["Simple exact values are generated automatically:", "ExampleText",
 CellID->481339490],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ZernikeR", "[", 
  RowBox[{
   RowBox[{"7", "/", "3"}], ",", 
   RowBox[{"1", "/", "3"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->47943293],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14903485]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->646689022],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["TraditionalForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TraditionalForm"]], "InlineFormula"],
 " formatting:"
}], "ExampleText",
 CellID->114818066],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ZernikeR", "[", 
   RowBox[{"n", ",", "m", ",", "r"}], "]"}], "//", 
  "TraditionalForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->127365055],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox["R", "n", "m"], "(", "r", ")"}], 
  TraditionalForm]], "Output",
 ImageSize->{32, 17},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//TraditionalForm=",
 CellID->65886904]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["\<\
A function to convert radial representation to Cartesian one:\
\>", "ExampleText",
 CellID->573583986],

Cell[BoxData[
 RowBox[{
  RowBox[{"r2c", "[", "e_", "]"}], ":=", 
  RowBox[{"Simplify", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"TrigExpand", "[", "e", "]"}], "/.", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"Sin", "[", "\[Theta]", "]"}], "\[RuleDelayed]", 
        RowBox[{"y", "/", "r"}]}], ",", 
       RowBox[{
        RowBox[{"Cos", "[", "\[Theta]", "]"}], "\[RuleDelayed]", 
        RowBox[{"x", "/", "r"}]}]}], "}"}]}], "/.", 
    RowBox[{
     RowBox[{"r", "^", "n_"}], "\[RuleDelayed]", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"x", "^", "2"}], "+", 
        RowBox[{"y", "^", "2"}]}], ")"}], "^", 
      RowBox[{"(", 
       RowBox[{"n", "/", "2"}], ")"}]}]}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->480972180],

Cell[BoxData[
 RowBox[{
  RowBox[{"zPlot", "[", "e_", "]"}], ":=", 
  RowBox[{"With", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"f", "=", " ", 
      RowBox[{"r2c", "[", "e", "]"}]}], "}"}], ",", 
    RowBox[{"DensityPlot", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Cos", "[", 
        RowBox[{"2", " ", "Pi", " ", "f"}], "]"}], "^", "2"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
      RowBox[{"RegionFunction", "\[RuleDelayed]", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"#1", "^", "2"}], "+", 
           RowBox[{"#2", "^", "2"}]}], "\[LessEqual]", "1"}], "&"}], ")"}]}], 
      ",", 
      RowBox[{"PlotPoints", "\[Rule]", "25"}], ",", 
      RowBox[{"BoundaryStyle", "\[Rule]", "Black"}], ",", 
      RowBox[{"Frame", "\[Rule]", "False"}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->41094925],

Cell["\<\
Visualize combined effect of x-astigmatism and y-coma aberrations:\
\>", "ExampleText",
 CellID->185057665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"zPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"ZernikeR", "[", 
     RowBox[{"2", ",", "2", ",", "r"}], "]"}], 
    RowBox[{"Cos", "[", 
     RowBox[{"2", "\[Theta]"}], "]"}]}], "+", 
   RowBox[{"2", 
    RowBox[{"ZernikeR", "[", 
     RowBox[{"3", ",", "1", ",", "r"}], "]"}], 
    RowBox[{"Sin", "[", "\[Theta]", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->183187299],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->458802571]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell["Obtain from generating function:", "ExampleText",
 CellID->100040833],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", "3"}], ";", 
  RowBox[{"Series", "[", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "+", "t", "-", 
        SqrtBox[
         RowBox[{"1", "+", 
          RowBox[{"2", "t", 
           RowBox[{"(", 
            RowBox[{"1", "-", 
             RowBox[{"2", 
              RowBox[{"r", "^", "2"}]}]}], ")"}]}], "+", 
          RowBox[{"t", "^", "2"}]}]]}], ")"}], "^", "m"}], 
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"2", " ", "r", " ", "t"}], ")"}], "m"], 
      SqrtBox[
       RowBox[{"1", "+", 
        RowBox[{"2", "t", 
         RowBox[{"(", 
          RowBox[{"1", "-", 
           RowBox[{"2", 
            RowBox[{"r", "^", "2"}]}]}], ")"}]}], "+", 
        RowBox[{"t", "^", "2"}]}]]}]], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "5"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->127294118],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{492, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->442119683]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "-", 
  RowBox[{"Sum", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"t", "^", "n"}], " ", 
     RowBox[{"ZernikeR", "[", 
      RowBox[{
       RowBox[{"m", "+", 
        RowBox[{"2", "n"}]}], ",", "m", ",", "r"}], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "0", ",", "5"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->174100625],

Cell[BoxData[
 InterpretationBox[
  SuperscriptBox[
   RowBox[{"O", "[", "t", "]"}], "6"],
  SeriesData[$CellContext`t, 0, {}, 6, 6, 1],
  Editable->False]], "Output",
 ImageSize->{38, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->869765361]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["JacobiP",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/JacobiP"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["SphericalHarmonicY",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/SphericalHarmonicY"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Hypergeometric2F1",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Hypergeometric2F1"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Orthogonal Polynomials",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/OrthogonalPolynomials"]], "Tutorials",
 CellID->3346]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with ZernikeR",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=ZernikeR"], 
    None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Functions Used in Optics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInOptics"]], "MoreAbout",
 CellID->106085251],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->473396234],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->201805260],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->254174313]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ZernikeR - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 30, 6.7908072}", "context" -> "System`", 
    "keywords" -> {
     "circle polynomials", "optical aberrations", "radial polynomials"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "ZernikeR[n, m, r] gives the radial Zernike polynomial \
SubsuperscriptBox[R, n, m](r).", "synonyms" -> {"Zernike R"}, "title" -> 
    "ZernikeR", "type" -> "Symbol", "uri" -> "ref/ZernikeR"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6981, 213, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->199128792]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 94985, 2012}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2672, 63, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3295, 90, 1573, 37, 70, "ObjectNameGrid"],
Cell[4871, 129, 732, 24, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5640, 158, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6118, 175, 120, 4, 70, "Notes",
 CellID->20640],
Cell[6241, 181, 79, 1, 70, "Notes",
 CellID->26319],
Cell[6323, 184, 196, 6, 70, "Notes",
 CellID->457162106],
Cell[6522, 192, 218, 7, 70, "Notes",
 CellID->31226],
Cell[6743, 201, 201, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[6981, 213, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->199128792],
Cell[CellGroupData[{
Cell[7366, 227, 148, 5, 70, "ExampleSection",
 CellID->313299963],
Cell[7517, 234, 63, 1, 70, "ExampleText",
 CellID->75293636],
Cell[CellGroupData[{
Cell[7605, 239, 139, 4, 28, "Input",
 CellID->514286172],
Cell[7747, 245, 180, 6, 36, "Output",
 CellID->393298414]
}, Open  ]],
Cell[7942, 254, 124, 3, 70, "ExampleDelimiter",
 CellID->99617739],
Cell[8069, 259, 65, 1, 70, "ExampleText",
 CellID->916987318],
Cell[CellGroupData[{
Cell[8159, 264, 137, 4, 70, "Input",
 CellID->177321292],
Cell[8299, 270, 323, 12, 39, "Output",
 CellID->484390320]
}, Open  ]],
Cell[8637, 285, 125, 3, 70, "ExampleDelimiter",
 CellID->270817465],
Cell[CellGroupData[{
Cell[8787, 292, 253, 8, 28, "Input",
 CellID->253820562],
Cell[9043, 302, 2665, 48, 122, "Output",
 Evaluatable->False,
 CellID->607555774]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11757, 356, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[11981, 365, 88, 1, 70, "ExampleText",
 CellID->62944296],
Cell[CellGroupData[{
Cell[12094, 370, 191, 6, 70, "Input",
 CellID->318679557],
Cell[12288, 378, 272, 7, 36, "Output",
 CellID->260864957]
}, Open  ]],
Cell[12575, 388, 125, 3, 70, "ExampleDelimiter",
 CellID->206642428],
Cell[12703, 393, 70, 1, 70, "ExampleText",
 CellID->152036528],
Cell[CellGroupData[{
Cell[12798, 398, 269, 9, 70, "Input",
 CellID->69905385],
Cell[13070, 409, 252, 8, 36, "Output",
 CellID->100474483]
}, Open  ]],
Cell[13337, 420, 113, 3, 70, "ExampleText",
 CellID->150856807],
Cell[CellGroupData[{
Cell[13475, 427, 216, 7, 70, "Input",
 CellID->7664213],
Cell[13694, 436, 236, 7, 36, "Output",
 CellID->129572938]
}, Open  ]],
Cell[13945, 446, 125, 3, 70, "ExampleDelimiter",
 CellID->181029827],
Cell[14073, 451, 91, 1, 70, "ExampleText",
 CellID->481339490],
Cell[CellGroupData[{
Cell[14189, 456, 184, 6, 70, "Input",
 CellID->47943293],
Cell[14376, 464, 157, 5, 36, "Output",
 CellID->14903485]
}, Open  ]],
Cell[14548, 472, 125, 3, 70, "ExampleDelimiter",
 CellID->646689022],
Cell[14676, 477, 203, 7, 70, "ExampleText",
 CellID->114818066],
Cell[CellGroupData[{
Cell[14904, 488, 179, 6, 70, "Input",
 CellID->127365055],
Cell[15086, 496, 266, 9, 52, "Output",
 CellID->65886904]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[15401, 511, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[15631, 520, 112, 3, 70, "ExampleText",
 CellID->573583986],
Cell[15746, 525, 788, 25, 70, "Input",
 CellID->480972180],
Cell[16537, 552, 1049, 31, 70, "Input",
 CellID->41094925],
Cell[17589, 585, 117, 3, 70, "ExampleText",
 CellID->185057665],
Cell[CellGroupData[{
Cell[17731, 592, 412, 13, 70, "Input",
 CellID->183187299],
Cell[18146, 607, 67402, 1109, 70, "Output",
 Evaluatable->False,
 CellID->458802571]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[85597, 1722, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[85838, 1731, 75, 1, 70, "ExampleText",
 CellID->100040833],
Cell[CellGroupData[{
Cell[85938, 1736, 945, 32, 70, "Input",
 CellID->127294118],
Cell[86886, 1770, 2664, 48, 70, "Output",
 Evaluatable->False,
 CellID->442119683]
}, Open  ]],
Cell[CellGroupData[{
Cell[89587, 1823, 385, 13, 70, "Input",
 CellID->174100625],
Cell[89975, 1838, 297, 10, 36, "Output",
 CellID->869765361]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[90333, 1855, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[90652, 1866, 979, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[91668, 1906, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[91989, 1917, 153, 3, 70, "Tutorials",
 CellID->3346]
}, Open  ]],
Cell[CellGroupData[{
Cell[92179, 1925, 305, 8, 70, "RelatedLinksSection"],
Cell[92487, 1935, 361, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[92885, 1952, 299, 8, 70, "MoreAboutSection"],
Cell[93187, 1962, 157, 3, 70, "MoreAbout",
 CellID->106085251],
Cell[93347, 1967, 145, 3, 70, "MoreAbout",
 CellID->473396234],
Cell[93495, 1972, 174, 3, 70, "MoreAbout",
 CellID->201805260],
Cell[93672, 1977, 179, 3, 70, "MoreAbout",
 CellID->254174313]
}, Open  ]],
Cell[93866, 1983, 27, 0, 70, "History"],
Cell[93896, 1985, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

