(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    189012,       4632]
NotebookOptionsPosition[    178383,       4259]
NotebookOutlinePosition[    179992,       4300]
CellTagsIndexPosition[    179949,       4297]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Constrained Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationOverview"], 
         "Linear Programming" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationLinearProgramming"], 
         "Numerical Nonlinear Local Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationLocalNumerical"], 
         "Numerical Nonlinear Global Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationGlobalNumerical"]}, #, 2]& ,
        Evaluator -> Automatic], {1->"\<\"Constrained Optimization\"\>", 
      2->"\<\"Linear Programming\"\>", 
      3->"\<\"Numerical Nonlinear Local Optimization\"\>", 
      4->"\<\"Numerical Nonlinear Global Optimization\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Exact Global Optimization", "Title",
 CellID->509267359],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellID->412364824],

Cell[TextData[{
 "Exact global optimization problems can be solved exactly using ",
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Maximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Maximize"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->1534169418],

Cell[TextData[{
 "This computes the radius of the circle, centered at the origin, \
circumscribed about the set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "4"], "+", 
     RowBox[{"3", " ", 
      SuperscriptBox["y", "4"]}]}], "\[LessEqual]", "7"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->617807409],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Maximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}]], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "4"], "+", 
       RowBox[{"3", 
        SuperscriptBox["y", "4"]}]}], "\[LessEqual]", "7"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->435640193],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SqrtBox["2"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["7", "3"], ")"}], 
     RowBox[{"1", "/", "4"}]]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "21"}], "+", 
          RowBox[{"4", " ", 
           SuperscriptBox["#1", "4"]}]}], "&"}], ",", "1"}], "]"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "7"}], "+", 
          RowBox[{"12", " ", 
           SuperscriptBox["#1", "4"]}]}], "&"}], ",", "1"}], "]"}]}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{437, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->74604153]
}, Open  ]],

Cell[TextData[{
 "This computes the radius of the circle, centered at the origin, \
circumscribed about the set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"a", " ", 
      SuperscriptBox["x", "2"]}], "+", 
     RowBox[{"b", " ", 
      SuperscriptBox["y", "2"]}]}], "\[LessEqual]", "1"}], TraditionalForm]], 
  "InlineMath"],
 " as a function of the parameters ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->1002525455],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Maximize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}]], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"a", " ", 
        SuperscriptBox["x", "2"]}], "+", 
       RowBox[{"b", " ", 
        SuperscriptBox["y", "2"]}]}], "\[LessEqual]", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->529133],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{520, 422},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->181748558]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Algorithms", "Section",
 CellID->248466559],

Cell["\<\
Depending on the type of problem, several different algorithms can be used. \
\>", "Text",
 CellID->59244523],

Cell["\<\
The most general method is based on the cylindrical algebraic decomposition \
(CAD) algorithm. It applies when the objective function and the constraints \
are real algebraic functions. The method can always compute global extrema \
(or extremal values, if the extrema are not attained). If parameters are \
present, the extrema can be computed as piecewise-algebraic functions of the \
parameters. A downside of the method is its high, doubly exponential \
complexity in the number of variables. In certain special cases not involving \
parameters, faster methods can be used.\
\>", "Text",
 CellID->867022447],

Cell["\<\
When the objective function and all constraints are linear with rational \
number coefficients, global extrema can be computed exactly using the simplex \
algorithm.\
\>", "Text",
 CellID->576808257],

Cell["\<\
For univariate problems, equation and inequality solving methods are used to \
find the constraint solution set and discontinuity points and zeros of the \
derivative of the objective function within the set.\
\>", "Text",
 CellID->78529832],

Cell[TextData[{
 "Another approach to finding global extrema is to find all the local \
extrema, using the Lagrange multipliers or the Karush\[Dash]Kuhn\[Dash]Tucker \
(KKT) conditions, and pick the smallest (or the greatest). However, for a \
fully automatic method, there are additional complications. In addition to \
solving the necessary conditions for local extrema, it needs to check \
smoothness of the objective function and smoothness and nondegeneracy of the \
constraints. It also needs to check for extrema at the boundary of the set \
defined by the constraints and at infinity, if the set is unbounded. This in \
general requires exact solving of systems of equations and inequalities over \
the reals, which may lead to CAD computations that are harder than in the \
optimization by CAD algorithm. Currently ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses Lagrange multipliers only for equational constraints within a bounded \
box. The method also requires that the number of stationary points and the \
number of singular points of the constraints be finite. An advantage of this \
method over the CAD-based algorithm is that it can solve some transcendental \
problems, as long as they lead to systems of equations that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can solve. "
}], "Text",
 CellID->324748596]
}, Open  ]],

Cell[CellGroupData[{

Cell["Optimization by Cylindrical Algebraic Decomposition", "Section",
 CellID->828680012],

Cell[CellGroupData[{

Cell["Examples", "Subsection",
 CellID->370765765],

Cell[TextData[{
 "This finds the point on the cubic curve ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "-", "x", "+", 
     SuperscriptBox["y", "2"]}], "\[LongEqual]", 
    FractionBox["1", "4"]}], TraditionalForm]], "InlineMath"],
 " which is closest to the origin. "
}], "MathCaption",
 CellID->836781195],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "3"], "+", 
       SuperscriptBox["y", "2"], "-", "x"}], "\[Equal]", 
      FractionBox["1", "4"]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2058623809],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"16", " ", "#1"}], "-", 
       RowBox[{"32", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       RowBox[{"16", " ", 
        SuperscriptBox["#1", "3"]}]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "1"}], "-", 
          RowBox[{"4", " ", "#1"}], "+", 
          RowBox[{"4", " ", 
           SuperscriptBox["#1", "3"]}]}], "&"}], ",", "2"}], "]"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{499, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->157084549]
}, Open  ]],

Cell[TextData[{
 "This finds the point on the cubic curve ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "-", "x", "+", 
     SuperscriptBox["y", "2"]}], "\[LongEqual]", "a"}], TraditionalForm]], 
  "InlineMath"],
 " which is closest to the origin, as a function of the parameter ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->270435084],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"min", "=", 
  RowBox[{"Minimize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "3"], "+", 
        SuperscriptBox["y", "2"], "-", "x"}], "\[Equal]", "a"}]}], "}"}], ",",
     
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->182671036],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{612, 339},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->350235341]
}, Open  ]],

Cell[TextData[{
 "This visualization shows the point on the cubic curve ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "-", "x", "+", 
     SuperscriptBox["y", "2"]}], "\[LongEqual]", "a"}], TraditionalForm]], 
  "InlineMath"],
 " which is closest to the origin, and the distance ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " of the point from the origin. The value of parameter ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " can be modified using the slider. The visualization uses the minimum \
computed by ",
 Cell[BoxData[
  ButtonBox["Minimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Minimize"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->88579835],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"plot", "[", "a_", "]"}], ":=", 
   RowBox[{"ContourPlot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "3"], "+", 
       SuperscriptBox["y", "2"], "-", "x"}], "\[Equal]", "a"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "3"}], ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "3"}], ",", "3"}], "}"}], ",", 
     RowBox[{"PlotRange", "->", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "3"}], ",", "3"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "3"}], ",", "3"}], "}"}]}], "}"}]}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"minval", "[", "a_", "]"}], ":=", 
  RowBox[{"Evaluate", "[", 
   RowBox[{"min", "[", 
    RowBox[{"[", "1", "]"}], "]"}], "]"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"minpt", "[", "a_", "]"}], ":=", 
  RowBox[{"Evaluate", "[", 
   RowBox[{"min", "[", 
    RowBox[{"[", "2", "]"}], "]"}], "]"}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"mmark", "=", 
   RowBox[{"Graphics", "[", 
    RowBox[{"Text", "[", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{"\"\<m=\>\"", ",", "10"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.25", ",", "2.5"}], "}"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"mvalue", "[", "a_", "]"}], ":=", 
   RowBox[{"Graphics", "[", 
    RowBox[{"Text", "[", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{
        RowBox[{"PaddedForm", "[", 
         RowBox[{
          RowBox[{"minval", "[", "a", "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"5", ",", "3"}], "}"}]}], "]"}], ",", "10"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "2.5"}], "}"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"amark", "=", 
   RowBox[{"Graphics", "[", 
    RowBox[{"Text", "[", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{"\"\<a=\>\"", ",", "10"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.25", ",", "2.8"}], "}"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"avalue", "[", "a_", "]"}], ":=", 
   RowBox[{"Graphics", "[", 
    RowBox[{"Text", "[", 
     RowBox[{
      RowBox[{"Style", "[", 
       RowBox[{
        RowBox[{"PaddedForm", "[", 
         RowBox[{"a", ",", 
          RowBox[{"{", 
           RowBox[{"5", ",", "3"}], "}"}]}], "]"}], ",", "10"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "2.8"}], "}"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"mpoint", "[", "a_", "]"}], ":=", 
   RowBox[{"Graphics", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "0.03", "]"}], ",", "Red", ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Re", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "/.", 
         RowBox[{"minpt", "[", "a", "]"}]}], "]"}], "]"}]}], "}"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Manipulate", "[", 
  RowBox[{
   RowBox[{"Show", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"plot", "[", "a", "]"}], ",", "amark", ",", 
      RowBox[{"avalue", "[", "a", "]"}], ",", "mmark", ",", 
      RowBox[{"mvalue", "[", "a", "]"}], ",", 
      RowBox[{"mpoint", "[", "a", "]"}]}], "}"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"a", ",", "4.5"}], "}"}], ",", 
     RowBox[{"-", "5"}], ",", "5"}], "}"}], ",", 
   RowBox[{"SaveDefinitions", "\[Rule]", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->282229366],

Cell[BoxData[
 TagBox[
  StyleBox[
   DynamicModuleBox[{$CellContext`a$$ = 4.5, Typeset`show$$ = True, 
    Typeset`bookmarkList$$ = {}, Typeset`bookmarkMode$$ = "Menu", 
    Typeset`animator$$, Typeset`animvar$$ = 1, Typeset`name$$ = 
    "\"untitled\"", Typeset`specs$$ = {{{
       Hold[$CellContext`a$$], 4.5}, -5, 5}}, Typeset`size$$ = {
    360., {180., 184.}}, Typeset`update$$ = 0, Typeset`initDone$$, 
    Typeset`skipInitDone$$ = False, $CellContext`a$89$$ = 0}, 
    DynamicBox[Manipulate`ManipulateBoxes[
     1, StandardForm, "Variables" :> {$CellContext`a$$ = 4.5}, 
      "ControllerVariables" :> {
        Hold[$CellContext`a$$, $CellContext`a$89$$, 0]}, 
      "OtherVariables" :> {
       Typeset`show$$, Typeset`bookmarkList$$, Typeset`bookmarkMode$$, 
        Typeset`animator$$, Typeset`animvar$$, Typeset`name$$, 
        Typeset`specs$$, Typeset`size$$, Typeset`update$$, Typeset`initDone$$,
         Typeset`skipInitDone$$}, "Body" :> Show[{
         $CellContext`plot[$CellContext`a$$], $CellContext`amark, 
         $CellContext`avalue[$CellContext`a$$], $CellContext`mmark, 
         $CellContext`mvalue[$CellContext`a$$], 
         $CellContext`mpoint[$CellContext`a$$]}], 
      "Specifications" :> {{{$CellContext`a$$, 4.5}, -5, 5}}, "Options" :> {},
       "DefaultOptions" :> {}],
     ImageSizeCache->{406., {224., 229.}},
     SingleEvaluation->True],
    Deinitialization:>None,
    DynamicModuleValues:>{},
    Initialization:>({$CellContext`plot[
         Pattern[$CellContext`a, 
          Blank[]]] := 
       ContourPlot[$CellContext`x^3 + $CellContext`y^2 - $CellContext`x == \
$CellContext`a, {$CellContext`x, -3, 3}, {$CellContext`y, -3, 3}, 
         PlotRange -> {{-3, 3}, {-3, 3}}], $CellContext`amark = Graphics[
         Text[
          Style["a=", 10], {1.25, 2.8}]], $CellContext`avalue[
         Pattern[$CellContext`a, 
          Blank[]]] := Graphics[
         Text[
          Style[
           PaddedForm[$CellContext`a, {5, 3}], 10], {
          2, 2.8}]], $CellContext`mmark = Graphics[
         Text[
          Style["m=", 10], {1.25, 2.5}]], $CellContext`mvalue[
         Pattern[$CellContext`a, 
          Blank[]]] := Graphics[
         Text[
          Style[
           PaddedForm[
            $CellContext`minval[$CellContext`a], {5, 3}], 10], {
          2, 2.5}]], $CellContext`minval[
         Pattern[$CellContext`a, 
          Blank[]]] := 
       Piecewise[{{
          1/9, $CellContext`a == 8/27}, {(-5 + 27 $CellContext`a)/27, 
           Inequality[8/27, Less, $CellContext`a, LessEqual, 80/27]}}, 
         Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 1]], $CellContext`mpoint[
        
         Pattern[$CellContext`a, 
          Blank[]]] := Graphics[{
          PointSize[0.03], Red, 
          Point[
           Re[
            ReplaceAll[{$CellContext`x, $CellContext`y}, 
             $CellContext`minpt[$CellContext`a]]]]}], $CellContext`minpt[
         Pattern[$CellContext`a, 
          Blank[]]] := {$CellContext`x -> Piecewise[{{(-1)/3, 
             Or[$CellContext`a == 8/27, 
              Inequality[8/27, Less, $CellContext`a, LessEqual, 80/27]]}, {
             
             Root[-$CellContext`a + 
              Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 
                1] - # - #^2 + #^3& , 1], 
             Or[$CellContext`a > 80/27, $CellContext`a < (-2)/(3 Sqrt[3])]}}, 
           
           
           Root[-$CellContext`a + 
            Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 1] - # - #^2 + #^3& , 
            2]], $CellContext`y -> 
         Piecewise[{{
            0, $CellContext`a == 8/27}, {-Sqrt[(-8)/27 + $CellContext`a], 
             Inequality[8/27, Less, $CellContext`a, LessEqual, 80/27]}, {-
             Sqrt[$CellContext`a + 
               Root[-$CellContext`a + 
                 Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 
                   1] - # - #^2 + #^3& , 1] - 
               Root[-$CellContext`a + 
                 Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 
                   1] - # - #^2 + #^3& , 1]^3], 
             
             Or[$CellContext`a > 
              80/27, $CellContext`a < (-2)/(3 Sqrt[3])]}}, -
           Sqrt[$CellContext`a + 
             Root[-$CellContext`a + 
               Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 
                 1] - # - #^2 + #^3& , 2] - 
             Root[-$CellContext`a + 
               Root[-$CellContext`a^2 + # - 2 #^2 + #^3& , 
                 1] - # - #^2 + #^3& , 2]^3]]}}; Typeset`initDone$$ = True),
    SynchronousInitialization->True,
    UnsavedVariables:>{Typeset`initDone$$},
    UntrackedVariables:>{Typeset`size$$}], "Manipulate",
   Deployed->True,
   StripOnInput->False],
  Manipulate`InterpretManipulate[1]]], "Output",
 ImageSize->{410, 455},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->305289938]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Customized CAD Algorithm for Optimization", "Subsection",
 CellID->4979907],

Cell[TextData[{
 "The cylindrical algebraic decomposition (CAD) algorithm is available in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " directly as ",
 Cell[BoxData[
  ButtonBox["CylindricalDecomposition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CylindricalDecomposition"]], "InlineFormula"],
 ". The algorithm is described in more detail in ",
 ButtonBox["Real Polynomial Systems",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/RealReduce"],
 ". The following describes how to customize the CAD algorithm to solve the \
global optimization problem."
}], "Text",
 CellID->905009369]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reduction to Minimizing a Coordinate Function", "Subsection",
 CellID->211764667],

Cell[TextData[{
 "Suppose it is required to minimize an algebraic function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", 
    RowBox[{"x", ",", "t"}], ")"}], TraditionalForm]], "InlineMath"],
 " over the solution sets of algebraic constraints ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalPhi]", "(", 
    RowBox[{"x", ",", "t"}], ")"}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " is a vector of variables and ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " is a vector of parameters. Let ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " be a new variable. The minimization of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " over the constraints ",
 Cell[BoxData[
  FormBox["\[CapitalPhi]", TraditionalForm]], "InlineMath"],
 " is equivalent to the minimization of the coordinate function ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " over the semialgebraic set given by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"y", "\[LongEqual]", 
     RowBox[{"f", "(", 
      RowBox[{"x", ",", "t"}], ")"}]}], "\[And]", 
    RowBox[{"\[CapitalPhi]", "(", 
     RowBox[{"x", ",", "t"}], ")"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->437653647],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " happens to be a monotonic function of one variable ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "1"], TraditionalForm]], "InlineMath"],
 ", a new variable is not needed, as ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "1"], TraditionalForm]], "InlineMath"],
 " can be minimized instead."
}], "Text",
 CellID->5471425]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Projection Phase of CAD", "Subsection",
 CellID->215663124],

Cell[TextData[{
 "The variables are projected, with ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " first, then the new variable ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ", and then the parameters ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->296849349],

Cell["\<\
If algebraic functions are present, they are replaced with new variables; \
equations and inequalities satisfied by the new variables are added. The \
variables replacing algebraic functions are projected first. They also \
require special handling in the lifting phase of the algorithm.\
\>", "Text",
 CellID->202589417],

Cell[TextData[{
 "Projection operator improvements by Hong, McCallum, and Brown can be used \
here, including the use of equational constraints. Note that if a new \
variable needs to be introduced, there is at least one equational constraint, \
namely ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Equal]", "f"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->194733701]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Lifting Phase of CAD", "Subsection",
 CellID->431999683],

Cell[TextData[{
 "The parameters ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " are lifted first, constructing the algebraic function equation and \
inequality description of the cells. If there are constraints that depend \
only on parameters and you can determine that ",
 Cell[BoxData[
  FormBox["\[CapitalPhi]", TraditionalForm]], "InlineMath"],
 " is identically false over a parameter cell, you do not need to lift this \
cell further."
}], "Text",
 CellID->480512515],

Cell[TextData[{
 "When lifting the minimization variable ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ", you start with the smallest values of ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ", and proceed (lifting the remaining variables in the depth-first manner) \
until you find the first cell for which the constraints are satisfied. If \
this cell corresponds to a root of a projection polynomial in ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ", the root is the minimum value of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ", and the coordinates corresponding to ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " of any point in the cell give a point at which the minimum is attained. If \
parameters are present, you get a parametric description of a point in the \
cell in terms of roots of polynomials bounding the cell. If there are no \
parameters, you can simply give the sample point used by the CAD algorithm. \
If the first cell satisfying the constraints corresponds to an interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"r", ",", "s"}], ")"}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["r", TraditionalForm]], "InlineMath"],
 " is a root of a projection polynomial in ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ", then ",
 Cell[BoxData[
  FormBox["r", TraditionalForm]], "InlineMath"],
 " is the infimum of values of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ", and the infimum value is not attained. Finally, if the first cell \
satisfying the constraints corresponds to an interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "\[Infinity]"}], ",", "s"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is unbounded from below."
}], "Text",
 CellID->382255947]
}, Open  ]],

Cell[CellGroupData[{

Cell["Strict Inequality Constraints", "Subsection",
 CellID->289075302],

Cell[TextData[{
 "If there are no parameters, all constraints are strict inequalities, and \
you only need the extremum value, a significantly simpler version of the \
algorithm can be used. (You can safely make inequality constraints strict if \
you know that ",
 Cell[BoxData[
  FormBox[
   RowBox[{"C", "\[SubsetEqual]", 
    OverscriptBox[
     RowBox[{"int", "(", "C", ")"}], "_"]}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["C", TraditionalForm]], "InlineMath"],
 " is the solution set of the constraints.) In this case many \
lower-dimensional cells can be disregarded; hence, the projection may only \
consist of the leading coefficients, the resultants, and the discriminants. \
In the lifting phase, only full-dimensional cells need be constructed; hence, \
there is no need for algebraic number computations."
}], "Text",
 CellID->161034174],

Cell[BoxData[GridBox[{
   {
    RowBox[{"Experimental`Infimum", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        StyleBox["cons", "TI"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["y", "TI"], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "find the infimum of values of ",
     Cell[BoxData[
      FormBox["f", TraditionalForm]], "InlineMath"],
     " on the set of points satisfying the constraints ",
     Cell[BoxData[
      StyleBox["cons", "TI"]], "InlineFormula"],
     "."
    }], "TableText"]},
   {
    RowBox[{"Experimental`Supremum", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        StyleBox["f", "TI"], ",", 
        StyleBox["cons", "TI"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        StyleBox["y", "TI"], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "find the supremum of values of ",
     Cell[BoxData[
      FormBox["f", TraditionalForm]], "InlineMath"],
     " on the set of points satisfying the constraints ",
     Cell[BoxData[
      StyleBox["cons", "TI"]], "InlineFormula"],
     "."
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->966076272],

Cell["Finding extremum values.", "Caption",
 CellID->520248874],

Cell[TextData[{
 "This finds the smallest ball centered at the origin which contains the set \
bounded by the surface ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "4"], "-", 
     RowBox[{"y", " ", "z", " ", "x"}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox["y", "4"]}], "+", 
     RowBox[{"3", " ", 
      SuperscriptBox["z", "4"]}]}], "\[LongEqual]", "1"}], TraditionalForm]], 
  "InlineMath"],
 ". A full ",
 Cell[BoxData[
  ButtonBox["Maximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Maximize"]], "InlineFormula"],
 " call with the same input did not finish in 10 minutes."
}], "MathCaption",
 CellID->133061386],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Experimental`Supremum", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", 
       SuperscriptBox["z", "2"]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "4"], "+", 
        RowBox[{"2", " ", 
         SuperscriptBox["y", "4"]}], "+", 
        RowBox[{"3", " ", 
         SuperscriptBox["z", "4"]}], "-", 
        RowBox[{"x", " ", "y", " ", "z"}]}], "<", "1"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->25818272],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.813`", ",", 
   RowBox[{"-", 
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"-", "1341154819099"}], "-", 
        RowBox[{"114665074208", " ", "#1"}], "+", 
        RowBox[{"4968163024164", " ", 
         SuperscriptBox["#1", "2"]}], "+", 
        RowBox[{"288926451967", " ", 
         SuperscriptBox["#1", "3"]}], "-", 
        RowBox[{"7172215018940", " ", 
         SuperscriptBox["#1", "4"]}], "-", 
        RowBox[{"240349978752", " ", 
         SuperscriptBox["#1", "5"]}], "+", 
        RowBox[{"5066800071680", " ", 
         SuperscriptBox["#1", "6"]}], "+", 
        RowBox[{"69844008960", " ", 
         SuperscriptBox["#1", "7"]}], "-", 
        RowBox[{"1756156133376", " ", 
         SuperscriptBox["#1", "8"]}], "-", 
        RowBox[{"2717908992", " ", 
         SuperscriptBox["#1", "9"]}], "+", 
        RowBox[{"239175991296", " ", 
         SuperscriptBox["#1", "10"]}]}], "&"}], ",", "1"}], "]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{551, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->170074348]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Linear Optimization", "Section",
 CellID->765665707],

Cell["\<\
When the objective function and all constraints are linear, global extrema \
can be computed exactly using the simplex algorithm.\
\>", "Text",
 CellID->24449235],

Cell["\<\
This solves a random linear minimization problem with ten variables.\
\>", "MathCaption",
 CellID->448246],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"SeedRandom", "[", "1", "]"}], ";", 
  RowBox[{"n", "=", "10"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"B", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Alpha]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Beta]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Gamma]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"X", "=", 
   RowBox[{"x", "/@", 
    RowBox[{"Range", "[", "n", "]"}]}]}], ";"}], "\n", 
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"\[Gamma]", ".", "X"}], ",", " ", 
     RowBox[{
      RowBox[{"And", "@@", 
       RowBox[{"Thread", "[", 
        RowBox[{
         RowBox[{"A", ".", "X"}], "\[Equal]", "\[Alpha]"}], "]"}]}], "&&", 
      " ", 
      RowBox[{"And", "@@", 
       RowBox[{"Thread", "[", 
        RowBox[{"\[Beta]", "\[LessEqual]", " ", 
         RowBox[{"B", ".", "X"}], "\[LessEqual]", " ", 
         RowBox[{"\[Beta]", "+", "100"}]}], "]"}]}]}]}], "}"}], ",", "X"}], 
  "]"}]}], "Input",
 CellLabel->"In[15]:=",
 CellID->214212387],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["6053416204117714679590329859484149", 
    "1194791208768786909167074679920"], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "1", "]"}], "\[Rule]", 
      FractionBox["1231164669474551725622041404999", 
       "1194791208768786909167074679920"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "2", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["1324409686130055761704674699781", 
        "597395604384393454583537339960"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "3", "]"}], "\[Rule]", 
      FractionBox["33103498981835356980792655092", 
       "74674450548049181822942167495"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "4", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["859057104531672755759277109213", 
        "597395604384393454583537339960"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "5", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["1101359025510393235751743044237", 
        "1194791208768786909167074679920"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "6", "]"}], "\[Rule]", 
      FractionBox["681114758987787242569015281099", 
       "597395604384393454583537339960"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "7", "]"}], "\[Rule]", 
      FractionBox["3008784898283435639647867743", 
       "14934890109609836364588433499"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "8", "]"}], "\[Rule]", 
      FractionBox["656889989559037679422691779229", 
       "597395604384393454583537339960"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "9", "]"}], "\[Rule]", 
      FractionBox["1769243029064640615513519505823", 
       "597395604384393454583537339960"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "10", "]"}], "\[Rule]", 
      FractionBox["699425860731183550585590812579", 
       "1194791208768786909167074679920"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{584, 300},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->331407315]
}, Open  ]],

Cell["\<\
Optimization problems where the objective is a fraction of linear functions \
and the constraints are linear (linear fractional programs) reduce to linear \
optimization problems. This solves a random linear fractional minimization \
problem with ten variables.\
\>", "MathCaption",
 CellID->710925819],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"SeedRandom", "[", "2", "]"}], ";", 
  RowBox[{"n", "=", "10"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"B", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Alpha]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Beta]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"n", "/", "2"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Gamma]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Delta]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"X", "=", 
   RowBox[{"x", "/@", 
    RowBox[{"Range", "[", "n", "]"}]}]}], ";"}], "\n", 
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"\[Gamma]", ".", "X"}], "/", 
      RowBox[{"\[Delta]", ".", "X"}]}], ",", " ", 
     RowBox[{
      RowBox[{"And", "@@", 
       RowBox[{"Thread", "[", 
        RowBox[{
         RowBox[{"A", ".", "X"}], "\[Equal]", "\[Alpha]"}], "]"}]}], "&&", 
      " ", 
      RowBox[{"And", "@@", 
       RowBox[{"Thread", "[", 
        RowBox[{"\[Beta]", "\[LessEqual]", " ", 
         RowBox[{"B", ".", "X"}], "\[LessEqual]", " ", 
         RowBox[{"\[Beta]", "+", "100"}]}], "]"}]}]}]}], "}"}], ",", "X"}], 
  "]"}]}], "Input",
 CellLabel->"In[23]:=",
 CellID->110045428],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", 
    FractionBox["1286274653702415809313525025452519", 
     "437743412320661916541674600912158"]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "1", "]"}], "\[Rule]", 
      FractionBox["611767491996227433062183883923", 
       "599276957533098032663796688622"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "2", "]"}], "\[Rule]", 
      FractionBox["2665078586976600235350409286849", 
       "1198553915066196065327593377244"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "3", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["215391679158483849611061030533", 
        "299638478766549016331898344311"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "4", "]"}], "\[Rule]", 
      FractionBox["1477394491589036027204142993013", 
       "599276957533098032663796688622"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "5", "]"}], "\[Rule]", 
      FractionBox["473657331854113835444689628600", 
       "299638478766549016331898344311"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "6", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["955420726065204315229251112109", 
        "599276957533098032663796688622"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "7", "]"}], "\[Rule]", 
      FractionBox["265603080958760324085018021123", 
       "1198553915066196065327593377244"]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "8", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["447840634450080124431365644067", 
        "599276957533098032663796688622"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "9", "]"}], "\[Rule]", 
      RowBox[{"-", 
       FractionBox["2155930215697442604517040669063", 
        "1198553915066196065327593377244"]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "10", "]"}], "\[Rule]", 
      FractionBox["18201652848869287002844477177", 
       "299638478766549016331898344311"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{320, 330},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->137126462]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Univariate Optimization", "Section",
 CellID->132591628],

Cell["\<\
For univariate problems, equation and inequality solving methods are used to \
find the constraint solution set and discontinuity points and zeros of the \
derivative of the objective function within the set.\
\>", "Text",
 CellID->341542595],

Cell["\<\
This solves a univariate optimization problem with a transcendental objective \
function.\
\>", "MathCaption",
 CellID->299751836],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Minimize", "[", 
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "2"], "+", 
     SuperscriptBox["2", "x"]}], ",", "x"}], "]"}]}]], "Input",
 CellLabel->"In[32]:=",
 CellID->132767492],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SuperscriptBox["2", 
     RowBox[{"-", 
      FractionBox[
       RowBox[{"ProductLog", "[", 
        FractionBox[
         SuperscriptBox[
          RowBox[{"Log", "[", "2", "]"}], "2"], "2"], "]"}], 
       RowBox[{"Log", "[", "2", "]"}]]}]], "+", 
    FractionBox[
     SuperscriptBox[
      RowBox[{"ProductLog", "[", 
       FractionBox[
        SuperscriptBox[
         RowBox[{"Log", "[", "2", "]"}], "2"], "2"], "]"}], "2"], 
     SuperscriptBox[
      RowBox[{"Log", "[", "2", "]"}], "2"]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", 
      FractionBox[
       RowBox[{"ProductLog", "[", 
        FractionBox[
         SuperscriptBox[
          RowBox[{"Log", "[", "2", "]"}], "2"], "2"], "]"}], 
       RowBox[{"Log", "[", "2", "]"}]]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{462, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->672414823]
}, Open  ]],

Cell["Here is a visualization of the minimum found.", "MathCaption",
 CellID->10734012],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["2", "x"]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.02", "]"}], ",", "Red", ",", 
       RowBox[{"Point", "[", 
        RowBox[{"N", "[", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"x", "/.", 
            RowBox[{"m", "[", 
             RowBox[{"[", "2", "]"}], "]"}]}], ",", 
           RowBox[{"m", "[", 
            RowBox[{"[", "1", "]"}], "]"}]}], "}"}], "]"}], "]"}]}], "}"}], 
     "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->52910432],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->414767002]
}, Open  ]],

Cell[TextData[{
 "Here ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " recognizes that the objective functions and the constraints are periodic."
}], "MathCaption",
 CellID->368440594],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"Tan", "[", 
       RowBox[{
        RowBox[{"2", " ", "x"}], "-", 
        FractionBox["\[Pi]", "2"]}], "]"}], "2"], ",", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "2"]}], "\[LessEqual]", 
      RowBox[{"Sin", "[", "x", "]"}], "\[LessEqual]", 
      FractionBox["1", "2"]}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[34]:=",
 CellID->446387923],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", "3"], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     FractionBox["\[Pi]", "6"]}], "}"}]}], "}"}]], "Output",
 ImageSize->{90, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->94446437]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Optimization by Finding Stationary and Singular Points", "Section",
 CellID->60567692],

Cell["\<\
Here is an example where the minimum is attained at a singular point of the \
constraints.\
\>", "MathCaption",
 CellID->270474848],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Minimize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["y", "3"], "\[Equal]", 
        SuperscriptBox["x", "2"]}], "&&", 
       RowBox[{
        RowBox[{"-", "2"}], "\[LessEqual]", "x", "\[LessEqual]", "2"}], "&&", 
       
       RowBox[{
        RowBox[{"-", "2"}], "\[LessEqual]", "y", "\[LessEqual]", "2"}]}]}], 
     "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[35]:=",
 CellID->449087732],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0"}], ",", 
     RowBox[{"y", "\[Rule]", "0"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{122, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]=",
 CellID->677228962]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"ContourPlot", "[", 
     RowBox[{
      RowBox[{
       SuperscriptBox["y", "3"], "\[Equal]", 
       SuperscriptBox["x", "2"]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "0.5"}], ",", "2"}], "}"}]}], "]"}], ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.02", "]"}], ",", "Red", ",", 
       RowBox[{"Point", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "/.", 
         RowBox[{"m", "[", 
          RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[36]:=",
 CellID->93237792],

Cell[BoxData[
 GraphicsBox[{GraphicsComplexBox[CompressedData["
1:eJw1lwtYjekWxzNlznDG0KgzDVOIiHLJnMgzOX8Mgzmiq8oIuUa2MEkOpY2T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    "], {{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      TagBox[
       TooltipBox[
        LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
         18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 
         35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 
         52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 
         69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 
         86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 
         102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115,
          116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 
         129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
          143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 
         156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169,
          170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 
         183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196,
          197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 
         210}],
        RowBox[{
          SuperscriptBox["y", "3"], "\[Equal]", 
          SuperscriptBox["x", "2"]}]],
       Annotation[#, $CellContext`y^3 == $CellContext`x^2, "Tooltip"]& ]}}], 
   {RGBColor[1, 0, 0], PointSize[0.02], PointBox[{0, 0}]}},
  AspectRatio->1,
  Frame->True,
  PlotRange->{{-2, 2}, {-0.5, 2}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 175},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]=",
 CellID->104532701]
}, Open  ]],

Cell["\<\
The maximum of the same objective function is attained on the boundary of the \
set defined by the constraints.\
\>", "MathCaption",
 CellID->585339400],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Maximize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox["y", "3"], "\[Equal]", 
        SuperscriptBox["x", "2"]}], "&&", 
       RowBox[{
        RowBox[{"-", "2"}], "\[LessEqual]", "x", "\[LessEqual]", "2"}], "&&", 
       
       RowBox[{
        RowBox[{"-", "2"}], "\[LessEqual]", "y", "\[LessEqual]", "2"}]}]}], 
     "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[37]:=",
 CellID->71252671],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "4"}], "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "2"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"-", "4"}], "+", 
          SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{351, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[37]=",
 CellID->556570459]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"ContourPlot", "[", 
     RowBox[{
      RowBox[{
       SuperscriptBox["y", "3"], "\[Equal]", 
       SuperscriptBox["x", "2"]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "0.5"}], ",", "2"}], "}"}]}], "]"}], ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.02", "]"}], ",", "Red", ",", 
       RowBox[{"Point", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "/.", 
         RowBox[{"m", "[", 
          RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[38]:=",
 CellID->95861976],

Cell[BoxData[
 GraphicsBox[{GraphicsComplexBox[CompressedData["
1:eJw1lwtYjekWxzNlznDG0KgzDVOIiHLJnMgzOX8Mgzmiq8oIuUa2MEkOpY2T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    "], {{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      TagBox[
       TooltipBox[
        LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
         18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 
         35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 
         52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 
         69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 
         86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 
         102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115,
          116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 
         129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
          143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 
         156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169,
          170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181, 182, 
         183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196,
          197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 
         210}],
        RowBox[{
          SuperscriptBox["y", "3"], "\[Equal]", 
          SuperscriptBox["x", "2"]}]],
       Annotation[#, $CellContext`y^3 == $CellContext`x^2, "Tooltip"]& ]}}], 
   {RGBColor[1, 0, 0], PointSize[0.02], 
    PointBox[NCache[{-2, Root[-4 + #^3& , 1, 0]}, {-2, 
       1.5874010519681996`}]]}},
  AspectRatio->1,
  Frame->True,
  PlotRange->{{-2, 2}, {-0.5, 2}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 175},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->95960192]
}, Open  ]],

Cell["There are no stationary points in this example.", "MathCaption",
 CellID->45506597],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "3"], "\[Equal]", 
     SuperscriptBox["x", "2"]}], "&&", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", "x", " ", "\[Lambda]"}], "\[Equal]", "0"}], "&&",
     
    RowBox[{
     RowBox[{"1", "+", 
      RowBox[{"3", " ", 
       SuperscriptBox["y", "2"], " ", "\[Lambda]"}]}], "\[Equal]", "0"}]}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "\[Lambda]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[39]:=",
 CellID->55164905],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->95238586]
}, Open  ]],

Cell["\<\
Here is a set of 3-dimensional examples with the same constraints. Depending \
on the objective function, the maximum is attained at a stationary point of \
the objective function on the solution set of the constraints, at a \
stationary point of the restriction of the objective function to the boundary \
of the solution set of the constraints, and at the boundary of the boundary \
of the solution set of the constraints.\
\>", "Text",
 CellID->55696895],

Cell["\<\
Here the maximum is attained at a stationary point of the objective function \
on the solution set of the constraints.\
\>", "MathCaption",
 CellID->3617633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Maximize", "[", 
   RowBox[{
    RowBox[{"x", "+", "y", "+", "z"}], ",", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", 
       SuperscriptBox["z", "2"]}], "\[Equal]", "9"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "x", "\[LessEqual]", "2"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "y", "\[LessEqual]", "2"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "z", "\[LessEqual]", "2"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[40]:=",
 CellID->789468889],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"3", " ", 
    SqrtBox["3"]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      SqrtBox["3"]}], ",", 
     RowBox[{"y", "\[Rule]", 
      SqrtBox["3"]}], ",", 
     RowBox[{"z", "\[Rule]", 
      SqrtBox["3"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{241, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]=",
 CellID->250129973]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"ContourPlot3D", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"], "+", 
         SuperscriptBox["z", "2"], "-", "9"}], "\[Equal]", "0"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"z", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"Graphics3D", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"PointSize", "[", "0.03", "]"}], ",", "Red", ",", 
        RowBox[{"Point", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"x", ",", "y", ",", "z"}], "}"}], "/.", 
          RowBox[{"m", "[", 
           RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "]"}]}], "}"}], 
   ",", 
   RowBox[{"ViewPoint", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"3", ",", "3", ",", "3"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[41]:=",
 CellID->470626743],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 411},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->62606439]
}, Open  ]],

Cell["\<\
Here the maximum is attained at a stationary point of the restriction of the \
objective function to the boundary of the solution set of the constraints.\
\>", "MathCaption",
 CellID->10984675],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Maximize", "[", 
   RowBox[{
    RowBox[{"x", "+", "y", "+", 
     RowBox[{"2", "z"}]}], ",", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", 
       SuperscriptBox["z", "2"]}], "\[Equal]", "9"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "x", "\[LessEqual]", "2"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "y", "\[LessEqual]", "2"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "z", "\[LessEqual]", "2"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[42]:=",
 CellID->266245322],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"4", "+", 
    SqrtBox["10"]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      SqrtBox[
       FractionBox["5", "2"]]}], ",", 
     RowBox[{"y", "\[Rule]", 
      SqrtBox[
       FractionBox["5", "2"]]}], ",", 
     RowBox[{"z", "\[Rule]", "2"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{255, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->140461290]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"ContourPlot3D", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"], "+", 
         SuperscriptBox["z", "2"], "-", "9"}], "\[Equal]", "0"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"z", ",", 
         RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], ",", 
     RowBox[{"Graphics3D", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"PointSize", "[", "0.03", "]"}], ",", "Red", ",", 
        RowBox[{"Point", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"x", ",", "y", ",", "z"}], "}"}], "/.", 
          RowBox[{"m", "[", 
           RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "]"}]}], "}"}], 
   ",", 
   RowBox[{"ViewPoint", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"3", ",", "7", ",", "7"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[43]:=",
 CellID->202213462],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 411},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[43]=",
 CellID->281311077]
}, Open  ]],

Cell["\<\
Here the maximum is attained at the boundary of the boundary of the solution \
set of the constraints.\
\>", "MathCaption",
 CellID->100704496],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Maximize", "[", 
   RowBox[{
    RowBox[{"x", "+", 
     RowBox[{"2", "y"}], "+", 
     RowBox[{"2", "z"}]}], ",", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", 
       SuperscriptBox["z", "2"]}], "\[Equal]", "9"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "x", "\[LessEqual]", "2"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "y", "\[LessEqual]", "2"}], "&&", 
     RowBox[{
      RowBox[{"-", "2"}], "\[LessEqual]", "z", "\[LessEqual]", "2"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y", ",", "z"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[44]:=",
 CellID->17535316],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"9", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1"}], ",", 
     RowBox[{"y", "\[Rule]", "2"}], ",", 
     RowBox[{"z", "\[Rule]", "2"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{163, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[44]=",
 CellID->516987253]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"ContourPlot3D", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"], "+", 
        SuperscriptBox["z", "2"], "-", "9"}], "\[Equal]", "0"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", 
        RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], ",", 
    RowBox[{"Graphics3D", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.03", "]"}], ",", "Red", ",", 
       RowBox[{"Point", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", "y", ",", "z"}], "}"}], "/.", 
         RowBox[{"m", "[", 
          RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[45]:=",
 CellID->336764021],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 411},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[45]=",
 CellID->27593519]
}, Open  ]],

Cell["\<\
The Lagrange multiplier method works for some optimization problems involving \
transcendental functions.\
\>", "MathCaption",
 CellID->704153443],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"y", "+", 
      RowBox[{"Sin", "[", 
       RowBox[{"10", " ", "x"}], "]"}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["y", "3"], "\[Equal]", 
       RowBox[{"Cos", "[", 
        RowBox[{"5", " ", "x"}], "]"}]}], "&&", 
      RowBox[{
       RowBox[{"-", "5"}], "\[LessEqual]", "x", "\[LessEqual]", "5"}], "&&", 
      RowBox[{
       RowBox[{"-", "5"}], "\[LessEqual]", "y", "\[LessEqual]", "5"}]}]}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[46]:=",
 CellID->874919934],

Cell[BoxData[
 RowBox[{
  RowBox[{"Minimize", "::", "\<\"ztest\"\>"}], ":", 
  " ", "\<\"Unable to decide whether numeric quantities \
\\!\\({\\(\\(\\(\\(Sin[\\(\\(4\\\\ \\(\\((\[Pi] - \
\\(\\(ArcTan[\\(\\(AlgebraicNumber[\\(\\(\[LeftSkeleton] 2 \
\[RightSkeleton]\\)\\)]\\)\\)]\\)\\))\\)\\)\\)\\)]\\)\\) - \
\\(\\(Sin[\\(\\(4\\\\ \\(\\((\\(\\(2\\\\ \[Pi]\\)\\) - \\(\\(ArcTan[\\(\\(\
\[LeftSkeleton] 1 \[RightSkeleton]\\)\\)]\\)\\))\\)\\)\\)\\)]\\)\\)\\)\\), \
\\(\\(\[LeftSkeleton] 5 \[RightSkeleton]\\)\\), \\(\\(\\(\\(Sin[\\(\\(4\\\\ \
\\(\\((\[Pi] - \\(\\(ArcTan[\\(\\(AlgebraicNumber[\\(\\(\[LeftSkeleton] 2 \
\[RightSkeleton]\\)\\)]\\)\\)]\\)\\))\\)\\)\\)\\)]\\)\\) + \\(\\(Sin[\\(\\(\
\[LeftSkeleton] 1 \[RightSkeleton]\\)\\)]\\)\\)\\)\\)}\\) are equal to zero. \
Assuming they are.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[46]:=",
 CellID->946665195],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"AlgebraicNumber", "[", 
     RowBox[{
      RowBox[{"Root", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"43046721", "-", 
          RowBox[{"95659380", " ", 
           SuperscriptBox["#1", "2"]}], "-", 
          RowBox[{"59049", " ", 
           SuperscriptBox["#1", "3"]}], "+", 
          RowBox[{"78653268", " ", 
           SuperscriptBox["#1", "4"]}], "-", 
          RowBox[{"32805", " ", 
           SuperscriptBox["#1", "5"]}], "-", 
          RowBox[{"29052108", " ", 
           SuperscriptBox["#1", "6"]}], "-", 
          RowBox[{"7290", " ", 
           SuperscriptBox["#1", "7"]}], "+", 
          RowBox[{"4763286", " ", 
           SuperscriptBox["#1", "8"]}], "-", 
          RowBox[{"810", " ", 
           SuperscriptBox["#1", "9"]}], "-", 
          RowBox[{"358668", " ", 
           SuperscriptBox["#1", "10"]}], "-", 
          RowBox[{"45", " ", 
           SuperscriptBox["#1", "11"]}], "+", 
          RowBox[{"11988", " ", 
           SuperscriptBox["#1", "12"]}], "-", 
          SuperscriptBox["#1", "13"], "-", 
          RowBox[{"180", " ", 
           SuperscriptBox["#1", "14"]}], "+", 
          SuperscriptBox["#1", "16"]}], "&"}], ",", "6"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", 
        FractionBox["2825", "256"], ",", 
        FractionBox["1", "81"], ",", 
        RowBox[{"-", 
         FractionBox["10645", "768"]}], ",", 
        FractionBox["1271", "186624"], ",", 
        FractionBox["117277", "20736"], ",", 
        FractionBox["421", "279936"], ",", 
        RowBox[{"-", 
         FractionBox["177851", "186624"]}], ",", 
        FractionBox["157", "944784"], ",", 
        FractionBox["13523", "186624"], ",", 
        FractionBox["625", "68024448"], ",", 
        RowBox[{"-", 
         FractionBox["36749", "15116544"]}], ",", 
        FractionBox["83", "408146688"], ",", 
        FractionBox["4975", "136048896"], ",", "0", ",", 
        RowBox[{"-", 
         FractionBox["83", "408146688"]}]}], "}"}]}], "]"}], "-", 
    RowBox[{"Sin", "[", 
     RowBox[{"4", " ", 
      RowBox[{"(", 
       RowBox[{"\[Pi]", "-", 
        RowBox[{"ArcTan", "[", 
         RowBox[{"AlgebraicNumber", "[", 
          RowBox[{
           RowBox[{"Root", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"43046721", "-", 
               RowBox[{"95659380", " ", 
                SuperscriptBox["#1", "2"]}], "-", 
               RowBox[{"59049", " ", 
                SuperscriptBox["#1", "3"]}], "+", 
               RowBox[{"78653268", " ", 
                SuperscriptBox["#1", "4"]}], "-", 
               RowBox[{"32805", " ", 
                SuperscriptBox["#1", "5"]}], "-", 
               RowBox[{"29052108", " ", 
                SuperscriptBox["#1", "6"]}], "-", 
               RowBox[{"7290", " ", 
                SuperscriptBox["#1", "7"]}], "+", 
               RowBox[{"4763286", " ", 
                SuperscriptBox["#1", "8"]}], "-", 
               RowBox[{"810", " ", 
                SuperscriptBox["#1", "9"]}], "-", 
               RowBox[{"358668", " ", 
                SuperscriptBox["#1", "10"]}], "-", 
               RowBox[{"45", " ", 
                SuperscriptBox["#1", "11"]}], "+", 
               RowBox[{"11988", " ", 
                SuperscriptBox["#1", "12"]}], "-", 
               SuperscriptBox["#1", "13"], "-", 
               RowBox[{"180", " ", 
                SuperscriptBox["#1", "14"]}], "+", 
               SuperscriptBox["#1", "16"]}], "&"}], ",", "6"}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", 
             FractionBox["1", "3"], ",", "0", ",", "0", ",", "0", ",", "0", 
             ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
             "0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "]"}], "]"}]}], 
       ")"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["2", "5"]}], " ", 
       RowBox[{"(", 
        RowBox[{"\[Pi]", "-", 
         RowBox[{"ArcTan", "[", 
          RowBox[{"AlgebraicNumber", "[", 
           RowBox[{
            RowBox[{"Root", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"43046721", "-", 
                RowBox[{"95659380", " ", 
                 SuperscriptBox["#1", "2"]}], "-", 
                RowBox[{"59049", " ", 
                 SuperscriptBox["#1", "3"]}], "+", 
                RowBox[{"78653268", " ", 
                 SuperscriptBox["#1", "4"]}], "-", 
                RowBox[{"32805", " ", 
                 SuperscriptBox["#1", "5"]}], "-", 
                RowBox[{"29052108", " ", 
                 SuperscriptBox["#1", "6"]}], "-", 
                RowBox[{"7290", " ", 
                 SuperscriptBox["#1", "7"]}], "+", 
                RowBox[{"4763286", " ", 
                 SuperscriptBox["#1", "8"]}], "-", 
                RowBox[{"810", " ", 
                 SuperscriptBox["#1", "9"]}], "-", 
                RowBox[{"358668", " ", 
                 SuperscriptBox["#1", "10"]}], "-", 
                RowBox[{"45", " ", 
                 SuperscriptBox["#1", "11"]}], "+", 
                RowBox[{"11988", " ", 
                 SuperscriptBox["#1", "12"]}], "-", 
                SuperscriptBox["#1", "13"], "-", 
                RowBox[{"180", " ", 
                 SuperscriptBox["#1", "14"]}], "+", 
                SuperscriptBox["#1", "16"]}], "&"}], ",", "6"}], "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"0", ",", 
              FractionBox["1", "3"], ",", "0", ",", "0", ",", "0", ",", "0", 
              ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",",
               "0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "]"}], "]"}]}], 
        ")"}]}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"AlgebraicNumber", "[", 
       RowBox[{
        RowBox[{"Root", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"43046721", "-", 
            RowBox[{"95659380", " ", 
             SuperscriptBox["#1", "2"]}], "-", 
            RowBox[{"59049", " ", 
             SuperscriptBox["#1", "3"]}], "+", 
            RowBox[{"78653268", " ", 
             SuperscriptBox["#1", "4"]}], "-", 
            RowBox[{"32805", " ", 
             SuperscriptBox["#1", "5"]}], "-", 
            RowBox[{"29052108", " ", 
             SuperscriptBox["#1", "6"]}], "-", 
            RowBox[{"7290", " ", 
             SuperscriptBox["#1", "7"]}], "+", 
            RowBox[{"4763286", " ", 
             SuperscriptBox["#1", "8"]}], "-", 
            RowBox[{"810", " ", 
             SuperscriptBox["#1", "9"]}], "-", 
            RowBox[{"358668", " ", 
             SuperscriptBox["#1", "10"]}], "-", 
            RowBox[{"45", " ", 
             SuperscriptBox["#1", "11"]}], "+", 
            RowBox[{"11988", " ", 
             SuperscriptBox["#1", "12"]}], "-", 
            SuperscriptBox["#1", "13"], "-", 
            RowBox[{"180", " ", 
             SuperscriptBox["#1", "14"]}], "+", 
            SuperscriptBox["#1", "16"]}], "&"}], ",", "6"}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          FractionBox["2825", "256"], ",", 
          FractionBox["1", "81"], ",", 
          RowBox[{"-", 
           FractionBox["10645", "768"]}], ",", 
          FractionBox["1271", "186624"], ",", 
          FractionBox["117277", "20736"], ",", 
          FractionBox["421", "279936"], ",", 
          RowBox[{"-", 
           FractionBox["177851", "186624"]}], ",", 
          FractionBox["157", "944784"], ",", 
          FractionBox["13523", "186624"], ",", 
          FractionBox["625", "68024448"], ",", 
          RowBox[{"-", 
           FractionBox["36749", "15116544"]}], ",", 
          FractionBox["83", "408146688"], ",", 
          FractionBox["4975", "136048896"], ",", "0", ",", 
          RowBox[{"-", 
           FractionBox["83", "408146688"]}]}], "}"}]}], "]"}]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{537, 455},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]=",
 CellID->442654596]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"%", ",", "20"}], "]"}]], "Input",
 CellLabel->"In[47]:=",
 CellID->444105983],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.90075003466751512295585390925876475682`20."}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.77209298024514961134235872631693620354`20."}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "0.90958837944086038551935651620396532374`20."}]}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{579, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[47]=",
 CellID->25384973]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Optimization over the Integers", "Section",
 CellID->7488435],

Cell[CellGroupData[{

Cell["Integer Linear Programming", "Subsection",
 CellID->683534434],

Cell["\<\
An integer linear programming problem is an optimization problem in which the \
objective function is linear, the constraints are linear and bounded, and the \
variables range over the integers.\
\>", "Text",
 CellID->329354925],

Cell[TextData[{
 "To solve an integer linear programming problem ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " first solves the equational constraints, reducing the problem to one \
containing inequality constraints only. Then it uses lattice reduction \
techniques to put the inequality system in a simpler form. Finally, it solves \
the simplified optimization problem using a branch-and-bound method."
}], "Text",
 CellID->467235243],

Cell["\<\
This solves a randomly generated integer linear programming problem with 7 \
variables.\
\>", "MathCaption",
 CellID->183730461],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"SeedRandom", "[", "1", "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"A", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "3", "}"}], ",", 
     RowBox[{"{", "7", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Alpha]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "3", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"B", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "3", "}"}], ",", 
     RowBox[{"{", "7", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Beta]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "3", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[Gamma]", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"RandomInteger", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1000"}], ",", "1000"}], "}"}], "]"}], ",", 
     RowBox[{"{", "7", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"X", "=", 
   RowBox[{"x", "/@", 
    RowBox[{"Range", "[", "7", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqns", "=", 
   RowBox[{"And", "@@", 
    RowBox[{"Thread", "[", 
     RowBox[{
      RowBox[{"A", ".", "X"}], "\[Equal]", "\[Alpha]"}], "]"}]}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"ineqs", "=", 
   RowBox[{"And", "@@", 
    RowBox[{"Thread", "[", 
     RowBox[{
      RowBox[{"B", ".", "X"}], "\[LessEqual]", "\[Beta]"}], "]"}]}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"bds", "=", 
   RowBox[{
    RowBox[{"And", "@@", 
     RowBox[{"Thread", "[", 
      RowBox[{"X", "\[GreaterEqual]", "0"}], "]"}]}], "&&", 
    RowBox[{
     RowBox[{"Total", "[", "X", "]"}], "\[LessEqual]", 
     SuperscriptBox["10", "100"]}]}]}], ";"}], "\n", 
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"\[Gamma]", ".", "X"}], ",", 
     RowBox[{"eqns", "&&", "ineqs", "&&", "bds", "&&", 
      RowBox[{"X", "\[Element]", "Integers"}]}]}], "}"}], ",", "X"}], 
  "]"}]}], "Input",
 CellLabel->"In[48]:=",
 CellID->93380149],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"448932", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "1", "]"}], "\[Rule]", "990"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "2", "]"}], "\[Rule]", "1205"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "3", "]"}], "\[Rule]", "219"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "4", "]"}], "\[Rule]", "60"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "5", "]"}], "\[Rule]", "823"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "6", "]"}], "\[Rule]", "137"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "7", "]"}], "\[Rule]", "34"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{548, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[58]=",
 CellID->109297721]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Optimization over the Reals Combined with Integer Solution Finding\
\>", "Subsection",
 CellID->544345886],

Cell[TextData[{
 "Suppose a function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "\[Element]", 
    RowBox[{"\[DoubleStruckCapitalZ]", "[", 
     StyleBox["x",
      FontSlant->"Italic"], "]"}]}], TraditionalForm]], "InlineMath"],
 " needs to be minimized over the integer solution set of constraints ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalPhi]", "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ". Let ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " be the minimum of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " over the real solution set of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalPhi]", "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ". If there exists an integer point satisfying ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], "\[LongEqual]", 
     RowBox[{"\[LeftCeiling]", "m", "\[RightCeiling]"}]}], "\[And]", 
    RowBox[{"\[CapitalPhi]", "(", "x", ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ", then ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LeftCeiling]", "m", "\[RightCeiling]"}], TraditionalForm]], 
  "InlineMath"],
 " is the minimum of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " over the integer solution set of ",
 Cell[BoxData[
  FormBox["\[CapitalPhi]", TraditionalForm]], "InlineMath"],
 ". Otherwise you try to find an integer solution of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], "\[LongEqual]", 
     RowBox[{
      RowBox[{"\[LeftCeiling]", "m", "\[RightCeiling]"}], "+", "1"}]}], 
    "\[And]", 
    RowBox[{"\[CapitalPhi]", "(", "x", ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ", and so on. This heuristic works if you can solve the real optimization \
problem and all the integer solution finding problems, and you can find an \
integer solution within a predefined number of attempts. (By default ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " tries 10 candidate optimum values. This can be changed using the ",
 Cell[BoxData["IntegerOptimumCandidates"], "InlineFormula"],
 " system option.)"
}], "Text",
 CellID->49004951],

Cell[TextData[{
 "This finds a point with integer coordinates maximizing ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "+", "y"}], TraditionalForm]], "InlineMath"],
 " among the points lying below the cubic ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "3"], "+", 
     SuperscriptBox["y", "3"]}], "\[LongEqual]", "1000"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->117632681],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"Maximize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"x", "+", "y"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "3"], "+", 
         SuperscriptBox["y", "3"]}], "\[LessEqual]", "1000"}], "&&", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "|", "y"}], ")"}], "\[Element]", "Integers"}]}]}], 
     "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[59]:=",
 CellID->25444687],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"15", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "6"}], ",", 
     RowBox[{"y", "\[Rule]", "9"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{129, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[59]=",
 CellID->613231999]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"ContourPlot", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox["x", "3"], "+", 
        SuperscriptBox["y", "3"]}], "\[Equal]", "1000"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "20"}], ",", "20"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "20"}], ",", "20"}], "}"}]}], "]"}], ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.02", "]"}], ",", "Red", ",", 
       RowBox[{"Point", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "/.", 
         RowBox[{"m", "[", 
          RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "]"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[60]:=",
 CellID->662448907],

Cell[BoxData[
 GraphicsBox[{GraphicsComplexBox[CompressedData["
1:eJx9Vns0VHkcHx0hYSl5zHqMmXvHeMWYnqr1PVusUKRtykqF6IFNWlsbrVqp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    "], {{}, {}, 
     {Hue[0.67, 0.6, 0.6], 
      TagBox[
       TooltipBox[
        LineBox[{1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
         18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 
         35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 
         52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 
         69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 
         86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 
         102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115,
          116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 
         129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142,
          143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 
         156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166}],
        RowBox[{
          RowBox[{
            SuperscriptBox["x", "3"], "+", 
            SuperscriptBox["y", "3"]}], "\[Equal]", "1000"}]],
       Annotation[#, $CellContext`x^3 + $CellContext`y^3 == 1000, 
        "Tooltip"]& ]}}], 
   {RGBColor[1, 0, 0], PointSize[0.02], PointBox[{6, 9}]}},
  AspectRatio->1,
  Frame->True,
  PlotRange->{{-20, 20}, {-20, 20}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 176},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[60]=",
 CellID->101091225]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->203495676],

Cell[TextData[ButtonBox["Constrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationOverview"]], "RelatedTutorials",
 CellID->1633042],

Cell[TextData[ButtonBox["Linear Programming",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationLinearProgramming"]], \
"RelatedTutorials",
 CellID->282186215],

Cell[TextData[ButtonBox["Numerical Nonlinear Local Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationLocalNumerical"]], \
"RelatedTutorials",
 CellID->511088424],

Cell[TextData[ButtonBox["Numerical Nonlinear Global Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationGlobalNumerical"]], \
"RelatedTutorials",
 CellID->125693634]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Exact Global Optimization - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 33, 25.8271524}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "Exact global optimization problems can be solved exactly using Minimize \
and Maximize. This computes the radius of the circle, centered at the origin, \
circumscribed about the set x^4+3 y^4<=7. This computes the radius of the \
circle, centered at the origin, circumscribed about the set a x^2+b y^2<=1 as \
a function of the parameters a and b.", "synonyms" -> {}, "title" -> 
    "Exact Global Optimization", "type" -> "Tutorial", "uri" -> 
    "tutorial/ConstrainedOptimizationExact"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1262, 28, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1887, 55, 62, 1, 70, "Title",
 CellID->509267359],
Cell[CellGroupData[{
Cell[1974, 60, 51, 1, 70, "Section",
 CellID->412364824],
Cell[2028, 63, 370, 13, 70, "Text",
 CellID->1534169418],
Cell[2401, 78, 366, 13, 70, "MathCaption",
 CellID->617807409],
Cell[CellGroupData[{
Cell[2792, 95, 471, 17, 40, "Input",
 CellID->435640193],
Cell[3266, 114, 880, 32, 52, "Output",
 CellID->74604153]
}, Open  ]],
Cell[4161, 149, 570, 20, 70, "MathCaption",
 CellID->1002525455],
Cell[CellGroupData[{
Cell[4756, 173, 502, 18, 40, "Input",
 CellID->529133],
Cell[5261, 193, 5507, 94, 443, "Output",
 Evaluatable->False,
 CellID->181748558]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10817, 293, 49, 1, 70, "Section",
 CellID->248466559],
Cell[10869, 296, 119, 3, 70, "Text",
 CellID->59244523],
Cell[10991, 301, 621, 10, 70, "Text",
 CellID->867022447],
Cell[11615, 313, 209, 5, 70, "Text",
 CellID->576808257],
Cell[11827, 320, 251, 5, 70, "Text",
 CellID->78529832],
Cell[12081, 327, 1350, 23, 70, "Text",
 CellID->324748596]
}, Open  ]],
Cell[CellGroupData[{
Cell[13468, 355, 90, 1, 70, "Section",
 CellID->828680012],
Cell[CellGroupData[{
Cell[13583, 360, 50, 1, 70, "Subsection",
 CellID->370765765],
Cell[13636, 363, 347, 11, 70, "MathCaption",
 CellID->836781195],
Cell[CellGroupData[{
Cell[14008, 378, 461, 16, 70, "Input",
 CellID->2058623809],
Cell[14472, 396, 865, 29, 39, "Output",
 CellID->157084549]
}, Open  ]],
Cell[15352, 428, 429, 14, 70, "MathCaption",
 CellID->270435084],
Cell[CellGroupData[{
Cell[15806, 446, 477, 17, 30, "Input",
 CellID->182671036],
Cell[16286, 465, 5918, 101, 360, "Output",
 Evaluatable->False,
 CellID->350235341]
}, Open  ]],
Cell[22219, 569, 759, 23, 70, "MathCaption",
 CellID->88579835],
Cell[CellGroupData[{
Cell[23003, 596, 3621, 113, 70, "Input",
 CellID->282229366],
Cell[26627, 711, 4872, 110, 70, "Output",
 CellID->305289938]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31548, 827, 81, 1, 70, "Subsection",
 CellID->4979907],
Cell[31632, 830, 606, 16, 70, "Text",
 CellID->905009369]
}, Open  ]],
Cell[CellGroupData[{
Cell[32275, 851, 87, 1, 70, "Subsection",
 CellID->211764667],
Cell[32365, 854, 1333, 40, 70, "Text",
 CellID->437653647],
Cell[33701, 896, 418, 14, 70, "Text",
 CellID->5471425]
}, Open  ]],
Cell[CellGroupData[{
Cell[34156, 915, 69, 1, 70, "Subsection",
 CellID->215663124],
Cell[34228, 918, 349, 12, 70, "Text",
 CellID->296849349],
Cell[34580, 932, 331, 6, 70, "Text",
 CellID->202589417],
Cell[34914, 940, 387, 10, 70, "Text",
 CellID->194733701]
}, Open  ]],
Cell[CellGroupData[{
Cell[35338, 955, 66, 1, 70, "Subsection",
 CellID->431999683],
Cell[35407, 958, 497, 12, 70, "Text",
 CellID->480512515],
Cell[35907, 972, 1994, 52, 70, "Text",
 CellID->382255947]
}, Open  ]],
Cell[CellGroupData[{
Cell[37938, 1029, 71, 1, 70, "Subsection",
 CellID->289075302],
Cell[38012, 1032, 886, 20, 70, "Text",
 CellID->161034174],
Cell[38901, 1054, 1377, 44, 70, "DefinitionBox",
 CellID->966076272],
Cell[40281, 1100, 63, 1, 70, "Caption",
 CellID->520248874],
Cell[40347, 1103, 669, 21, 70, "MathCaption",
 CellID->133061386],
Cell[CellGroupData[{
Cell[41041, 1128, 673, 22, 70, "Input",
 CellID->25818272],
Cell[41717, 1152, 1158, 33, 94, "Output",
 CellID->170074348]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[42936, 1192, 58, 1, 70, "Section",
 CellID->765665707],
Cell[42997, 1195, 172, 4, 70, "Text",
 CellID->24449235],
Cell[43172, 1201, 116, 3, 70, "MathCaption",
 CellID->448246],
Cell[CellGroupData[{
Cell[43313, 1208, 2337, 77, 70, "Input",
 CellID->214212387],
Cell[45653, 1287, 2050, 54, 291, "Output",
 CellID->331407315]
}, Open  ]],
Cell[47718, 1344, 312, 6, 70, "MathCaption",
 CellID->710925819],
Cell[CellGroupData[{
Cell[48055, 1354, 2668, 88, 70, "Input",
 CellID->110045428],
Cell[50726, 1444, 2100, 56, 321, "Output",
 CellID->137126462]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[52875, 1506, 62, 1, 70, "Section",
 CellID->132591628],
Cell[52940, 1509, 252, 5, 70, "Text",
 CellID->341542595],
Cell[53195, 1516, 140, 4, 70, "MathCaption",
 CellID->299751836],
Cell[CellGroupData[{
Cell[53360, 1524, 231, 8, 70, "Input",
 CellID->132767492],
Cell[53594, 1534, 993, 33, 70, "Output",
 CellID->672414823]
}, Open  ]],
Cell[54602, 1570, 87, 1, 70, "MathCaption",
 CellID->10734012],
Cell[CellGroupData[{
Cell[54714, 1575, 839, 27, 49, "Input",
 CellID->52910432],
Cell[55556, 1604, 2552, 46, 139, "Output",
 Evaluatable->False,
 CellID->414767002]
}, Open  ]],
Cell[58123, 1653, 191, 6, 70, "MathCaption",
 CellID->368440594],
Cell[CellGroupData[{
Cell[58339, 1663, 491, 16, 70, "Input",
 CellID->446387923],
Cell[58833, 1681, 308, 11, 51, "Output",
 CellID->94446437]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[59190, 1698, 92, 1, 70, "Section",
 CellID->60567692],
Cell[59285, 1701, 141, 4, 70, "MathCaption",
 CellID->270474848],
Cell[CellGroupData[{
Cell[59451, 1709, 570, 19, 70, "Input",
 CellID->449087732],
Cell[60024, 1730, 318, 11, 36, "Output",
 CellID->677228962]
}, Open  ]],
Cell[CellGroupData[{
Cell[60379, 1746, 838, 27, 70, "Input",
 CellID->93237792],
Cell[61220, 1775, 4826, 87, 70, "Output",
 CellID->104532701]
}, Open  ]],
Cell[66061, 1865, 162, 4, 70, "MathCaption",
 CellID->585339400],
Cell[CellGroupData[{
Cell[66248, 1873, 569, 19, 70, "Input",
 CellID->71252671],
Cell[66820, 1894, 677, 25, 39, "Output",
 CellID->556570459]
}, Open  ]],
Cell[CellGroupData[{
Cell[67534, 1924, 838, 27, 70, "Input",
 CellID->95861976],
Cell[68375, 1953, 4895, 89, 70, "Output",
 CellID->95960192]
}, Open  ]],
Cell[73285, 2045, 89, 1, 70, "MathCaption",
 CellID->45506597],
Cell[CellGroupData[{
Cell[73399, 2050, 553, 19, 70, "Input",
 CellID->55164905],
Cell[73955, 2071, 162, 5, 36, "Output",
 CellID->95238586]
}, Open  ]],
Cell[74132, 2079, 467, 8, 70, "Text",
 CellID->55696895],
Cell[74602, 2089, 167, 4, 70, "MathCaption",
 CellID->3617633],
Cell[CellGroupData[{
Cell[74794, 2097, 703, 20, 70, "Input",
 CellID->789468889],
Cell[75500, 2119, 446, 17, 43, "Output",
 CellID->250129973]
}, Open  ]],
Cell[CellGroupData[{
Cell[75983, 2141, 1181, 36, 49, "Input",
 CellID->470626743],
Cell[77167, 2179, 22372, 371, 432, "Output",
 Evaluatable->False,
 CellID->62606439]
}, Open  ]],
Cell[99554, 2553, 203, 4, 70, "MathCaption",
 CellID->10984675],
Cell[CellGroupData[{
Cell[99782, 2561, 724, 21, 70, "Input",
 CellID->266245322],
Cell[100509, 2584, 483, 18, 60, "Output",
 CellID->140461290]
}, Open  ]],
Cell[CellGroupData[{
Cell[101029, 2607, 1181, 36, 49, "Input",
 CellID->202213462],
Cell[102213, 2645, 24504, 406, 432, "Output",
 Evaluatable->False,
 CellID->281311077]
}, Open  ]],
Cell[126732, 3054, 153, 4, 70, "MathCaption",
 CellID->100704496],
Cell[CellGroupData[{
Cell[126910, 3062, 744, 22, 70, "Input",
 CellID->17535316],
Cell[127657, 3086, 360, 12, 36, "Output",
 CellID->516987253]
}, Open  ]],
Cell[CellGroupData[{
Cell[128054, 3103, 1030, 32, 49, "Input",
 CellID->336764021],
Cell[129087, 3137, 24934, 413, 432, "Output",
 Evaluatable->False,
 CellID->27593519]
}, Open  ]],
Cell[154036, 3553, 156, 4, 70, "MathCaption",
 CellID->704153443],
Cell[CellGroupData[{
Cell[154217, 3561, 649, 21, 70, "Input",
 CellID->874919934],
Cell[154869, 3584, 882, 15, 70, "Message",
 CellID->946665195],
Cell[155754, 3601, 8168, 204, 446, "Output",
 CellID->442654596]
}, Open  ]],
Cell[CellGroupData[{
Cell[163959, 3810, 122, 4, 70, "Input",
 CellID->444105983],
Cell[164084, 3816, 514, 15, 36, "Output",
 CellID->25384973]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[164647, 3837, 67, 1, 70, "Section",
 CellID->7488435],
Cell[CellGroupData[{
Cell[164739, 3842, 68, 1, 70, "Subsection",
 CellID->683534434],
Cell[164810, 3845, 238, 5, 70, "Text",
 CellID->329354925],
Cell[165051, 3852, 445, 9, 70, "Text",
 CellID->467235243],
Cell[165499, 3863, 138, 4, 70, "MathCaption",
 CellID->183730461],
Cell[CellGroupData[{
Cell[165662, 3871, 2557, 86, 70, "Input",
 CellID->93380149],
Cell[168222, 3959, 773, 24, 54, "Output",
 CellID->109297721]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[169044, 3989, 116, 3, 70, "Subsection",
 CellID->544345886],
Cell[169163, 3994, 2165, 64, 70, "Text",
 CellID->49004951],
Cell[171331, 4060, 436, 15, 70, "MathCaption",
 CellID->117632681],
Cell[CellGroupData[{
Cell[171792, 4079, 550, 19, 70, "Input",
 CellID->25444687],
Cell[172345, 4100, 319, 11, 36, "Output",
 CellID->613231999]
}, Open  ]],
Cell[CellGroupData[{
Cell[172701, 4116, 874, 28, 70, "Input",
 CellID->662448907],
Cell[173578, 4146, 3837, 72, 70, "Output",
 CellID->101091225]
}, Open  ]],
Cell[CellGroupData[{
Cell[177452, 4223, 72, 1, 70, "RelatedTutorialsSection",
 CellID->203495676],
Cell[177527, 4226, 178, 4, 70, "RelatedTutorials",
 CellID->1633042],
Cell[177708, 4232, 185, 5, 70, "RelatedTutorials",
 CellID->282186215],
Cell[177896, 4239, 202, 5, 70, "RelatedTutorials",
 CellID->511088424],
Cell[178101, 4246, 204, 5, 70, "RelatedTutorials",
 CellID->125693634]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[178356, 4257, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

