(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    473419,      10011]
NotebookOptionsPosition[    425326,       8688]
NotebookOutlinePosition[    455002,       9439]
CellTagsIndexPosition[    454889,       9433]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Constrained Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationOverview"], 
         "Linear Programming" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationLinearProgramming"], 
         "Numerical Nonlinear Local Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationLocalNumerical"], 
         "Exact Global Optimization" :> 
         Documentation`HelpLookup[
          "paclet:tutorial/ConstrainedOptimizationExact"]}, #, 2]& , 
       Evaluator -> Automatic], {1->"\<\"Constrained Optimization\"\>", 
      2->"\<\"Linear Programming\"\>", 
      3->"\<\"Numerical Nonlinear Local Optimization\"\>", 
      4->"\<\"Exact Global Optimization\"\>"}, "\<\"related tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Numerical Nonlinear Global Optimization", "Title",
 CellID->85183321],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellID->469184158],

Cell[TextData[{
 "Numerical algorithms for constrained nonlinear optimization can be broadly \
categorized into ",
 StyleBox["gradient based methods",
  FontSlant->"Italic"],
 " and ",
 StyleBox["direct search methods. ",
  FontSlant->"Italic"],
 "Gradient-based methods use first derivatives (gradients) or second \
derivatives (Hessians). Examples are the sequential quadratic programming \
(SQP) method, the augmented Lagrangian method, and the (nonlinear) interior \
point method. ",
 StyleBox["Direct search methods",
  FontColor->GrayLevel[0]],
 " do not use derivative information. Examples are Nelder\[Dash]Mead, genetic \
algorithm and differential evolution, and simulated annealing. Direct search \
methods tend to converge more slowly, but can be more tolerant to the \
presence of noise in the function and constraints."
}], "Text",
 CellID->289642637],

Cell[TextData[{
 "Typically, algorithms only build up a local model of the problems. \
Furthermore, many such algorithms insist on certain decrease of the objective \
function, or decrease of a merit function which is a combination of the \
objective and constraints, to ensure convergence of the iterative process. \
Such algorithms will, if convergent, only find local optima, and are called \
",
 StyleBox["local optimization algorithms",
  FontSlant->"Italic"],
 ". In ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " local optimization problems can be solved using",
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"],
   FontFamily->"Courier"]], "InlineFormula"],
 "."
}], "Text",
 CellID->65223348],

Cell[TextData[{
 StyleBox["Global optimization",
  FontSlant->"Italic"],
 " ",
 StyleBox["algorithms",
  FontSlant->"Italic"],
 ", on the other hand, attempt to find the global optimum, typically by \
allowing decrease as well as increase of the objective/merit function. Such \
algorithms are usually computationally more expensive. Global optimization \
problems can be solved exactly using ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Minimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Minimize"],
   FontFamily->"Courier"]], "InlineFormula"],
 " or numerically using ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Courier"]], "InlineFormula"],
 "."
}], "Text",
 CellID->166790006],

Cell["This solves a nonlinear programming problem,", "Text",
 CellID->303464556],

Cell[BoxData[{
 FormBox[
  RowBox[{
   RowBox[{"Min", "     ", "x"}], "-", "y"}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"s", ".", "t", "."}], "   ", "-", 
    RowBox[{"3", " ", 
     SuperscriptBox["x", "2"]}], "+", 
    RowBox[{"2", " ", "x", " ", "y"}], "-", 
    SuperscriptBox["y", "2"]}], "\[GreaterEqual]", 
   RowBox[{"-", "1"}]}], TraditionalForm]}], "DisplayMath",
 CellID->401358986],

Cell[TextData[{
 "using ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Minimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Minimize"],
   FontFamily->"Courier"]], "InlineFormula"],
 StyleBox[",",
  FontFamily->"Courier"],
 "which gives an exact solution"
}], "Text",
 CellID->73215332],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Minimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"-", "3"}], " ", 
        SuperscriptBox["x", "2"]}], "+", 
       RowBox[{"2", " ", "x", " ", "y"}], "-", 
       SuperscriptBox["y", "2"]}], "\[GreaterEqual]", 
      RowBox[{"-", "1"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->213704],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0"}], ",", 
     RowBox[{"y", "\[Rule]", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{130, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->423974823]
}, Open  ]],

Cell[TextData[{
 "This solves the same problem numerically. ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NMinimize",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NMinimize"],
   FontFamily->"Courier"]], "InlineFormula"],
 " returns a machine-number solution."
}], "MathCaption",
 CellID->109220874],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"-", "3"}], " ", 
        SuperscriptBox["x", "2"]}], "+", 
       RowBox[{"2", " ", "x", " ", "y"}], "-", 
       SuperscriptBox["y", "2"]}], "\[GreaterEqual]", 
      RowBox[{"-", "1"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->38568035],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.0000000009276877`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.907012579220415`*^-6"}], ",", 
     RowBox[{"y", "\[Rule]", "1.0000019079402669`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{221, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1586759045]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindMinimum",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindMinimum"],
   FontFamily->"Courier"]], "InlineFormula"],
 " numerically finds a local minimum. In this example the local minimum found \
is also a global minimum."
}], "MathCaption",
 CellID->583307620],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "-", "y"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"-", "3"}], " ", 
        SuperscriptBox["x", "2"]}], "+", 
       RowBox[{"2", " ", "x", " ", "y"}], "-", 
       SuperscriptBox["y", "2"]}], "\[GreaterEqual]", 
      RowBox[{"-", "1"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->7703726],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.000000810766966`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "2.7830123428585406`*^-17"}], ",", 
     RowBox[{"y", "\[Rule]", "1.000000810766966`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{226, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->2132335936]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The NMinimize Function", "Section",
 CellID->217856200],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " implement several algorithms for finding constrained global optima. The \
methods are flexible enough to cope with functions that are not \
differentiable or continuous and are not easily trapped by local optima. "
}], "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->16616413],

Cell["\<\
Finding a global optimum can be arbitrarily difficult, even without \
constraints, and so the methods used may fail. It may frequently be useful to \
optimize the function several times with different starting conditions and \
take the best of the results. \
\>", "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->441818026],

Cell[TextData[{
 "This finds the maximum of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"sin", "(", 
     RowBox[{"x", "+", "y"}], ")"}], "-", 
    SuperscriptBox["x", "2"], "-", 
    SuperscriptBox["y", "2"]}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->13823121],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMaximize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"x", "+", "y"}], "]"}], "-", 
    SuperscriptBox["x", "2"], "-", 
    SuperscriptBox["y", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[46]:=",
 CellID->158556129],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.40048861211337905`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.36954256616664977`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.3695425670003307`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{269, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[46]=",
 CellID->147730805]
}, Open  ]],

Cell[TextData[{
 "This finds the minimum of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"y", "-", 
       FractionBox["1", "2"]}], ")"}], "2"], "+", 
    SuperscriptBox["x", "2"]}], TraditionalForm]], "InlineMath"],
 " subject to the constraints ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[GreaterEqual]", "0"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[GreaterEqual]", 
    RowBox[{"x", "+", "1"}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->8449423],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"y", "-", ".5"}], ")"}], "2"]}], ",", 
     RowBox[{
      RowBox[{"y", "\[GreaterEqual]", "0"}], "&&", 
      RowBox[{"y", "\[GreaterEqual]", 
       RowBox[{"x", "+", "1"}]}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[47]:=",
 CellID->479319689],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.12499999906867743`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.24999999906876735`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0.7499999990685875`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{200, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[47]=",
 CellID->420543703]
}, Open  ]],

Cell[TextData[{
 "The constraints to ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " may be either a list or logical combination of equalities, inequalities, \
and domain specifications. Equalities and inequalities may be nonlinear. Any \
strong inequalities will be converted to weak inequalities due to limits of \
working with approximate numbers. Specify a domain for a variable using ",
 Cell[BoxData[
  ButtonBox["Element",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Element"]], "InlineFormula"],
 ", for example, ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Element",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Element"], "[", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    ButtonBox["Integers",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Integers"]}], "]"}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], 
   StyleBox["\[Element]", "TR"], 
   ButtonBox["Integers",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Integers"]}]], "InlineFormula"],
 ". Variables must be either integers or real numbers, and will be assumed to \
be real numbers unless specified otherwise. Constraints are generally \
enforced by adding penalties when points leave the feasible region. "
}], "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->55770639],

Cell[TextData[{
 "Constraints can contain logical operators like ",
 Cell[BoxData[
  ButtonBox["And",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/And"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Or",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Or"]], "InlineFormula"],
 ", and so on. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->278597936],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], ",", 
     RowBox[{
      RowBox[{"x", "\[GreaterEqual]", "1"}], "||", 
      RowBox[{"y", "\[GreaterEqual]", "2"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[3]:=",
 CellID->105413790],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{143, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[3]=",
 CellID->218869012]
}, Open  ]],

Cell[TextData[{
 "Here ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " is restricted to being an integer. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->869892494],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"x", "-", 
         RowBox[{"1", "/", "3"}]}], ")"}], "2"], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"y", "-", 
         RowBox[{"1", "/", "3"}]}], ")"}], "2"]}], ",", 
     RowBox[{"x", "\[Element]", "Integers"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[4]:=",
 CellID->434989722],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.1111111111111111`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0"}], ",", 
     RowBox[{"y", "\[Rule]", "0.3333333333333333`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{220, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[4]=",
 CellID->386809913]
}, Open  ]],

Cell[TextData[{
 "In order for ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " to work, it needs a rectangular initial region in which to start. This is \
similar to giving other numerical methods a starting point or starting \
points. The initial region is specified by giving each variable a finite \
upper and lower bound. This is done by including ",
 Cell[BoxData[
  FormBox[
   RowBox[{"a", "\[LessEqual]", "x", "\[LessEqual]", "b"}], TraditionalForm]],
   "InlineMath"],
 " in the constraints, or ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["x", "TI"], ",", 
    StyleBox["a", "TI"], ",", 
    StyleBox["b", "TI"]}], "}"}]], "InlineFormula"],
 " in the variables. If both are given, the bounds in the variables are used \
for the initial region, and the constraints are just used as constraints. If \
no initial region is specified for a variable ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 ", the default initial region of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"-", "1"}], "\[LessEqual]", "x", "\[LessEqual]", "1"}], 
   TraditionalForm]], "InlineMath"],
 " is used. Different variables can have initial regions defined in different \
ways. "
}], "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->408968279],

Cell["\<\
Here the initial region is taken from the variables. The problem is \
unconstrained. \
\>", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->92233752],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   SuperscriptBox["x", "2"], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"x", ",", "3", ",", "4"}], "}"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[5]:=",
 CellID->328088122],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "0.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[5]=",
 CellID->71577405]
}, Open  ]],

Cell["Here the initial region is taken from the constraints. ", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->588472900],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     SuperscriptBox["x", "2"], ",", 
     RowBox[{"3", "\[LessEqual]", "x", "\[LessEqual]", "4"}]}], "}"}], ",", 
   RowBox[{"{", "x", "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[6]:=",
 CellID->160567602],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"9.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "3.`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[6]=",
 CellID->285978531]
}, Open  ]],

Cell[TextData[{
 "Here the initial region for ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " is taken from the constraints, the initial region for ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " is taken from the variables, and the initial region for ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " is taken to be the default. The problem is unconstrained in ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 ", but not ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->542356352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"], "+", 
      SuperscriptBox["z", "2"]}], ",", 
     RowBox[{"3", "\[LessEqual]", "x", "\[LessEqual]", "4"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "2", ",", "5"}], "}"}], ",", "z"}], "}"}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[7]:=",
 CellID->164914745],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"9.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "3.`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.`"}], ",", 
     RowBox[{"z", "\[Rule]", "0.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{191, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[7]=",
 CellID->393800755]
}, Open  ]],

Cell[TextData[{
 "The polynomial ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"4", " ", 
     SuperscriptBox["x", "4"]}], "-", 
    RowBox[{"4", " ", 
     SuperscriptBox["x", "2"]}], "+", "1"}], TraditionalForm]], 
  "InlineMath"],
 " has global minima at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[Rule]", 
    RowBox[{"\[PlusMinus]", 
     FractionBox[
      SqrtBox["2"], "2"]}]}], TraditionalForm]], "InlineMath"],
 ". ",
 StyleBox["NelderMead", "MR"],
 " finds one of the minima."
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->110392765],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"4", 
     SuperscriptBox["x", "4"]}], "-", 
    RowBox[{"4", 
     SuperscriptBox["x", "2"]}], "+", "1"}], ",", " ", "x", ",", " ", 
   RowBox[{"Method", "->", "\"\<NelderMead\>\""}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[48]:=",
 CellID->841906160],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "0.7071067801802013`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[48]=",
 CellID->709334044]
}, Open  ]],

Cell[TextData[{
 "The other minimum can be found by using a different ",
 Cell[BoxData["RandomSeed"], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->262757712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"4", 
     SuperscriptBox["x", "4"]}], "-", 
    RowBox[{"4", 
     SuperscriptBox["x", "2"]}], "+", "1"}], ",", " ", "x", ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<NelderMead\>\"", ",", " ", 
      RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "111"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[50]:=",
 CellID->891110009],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.`", ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "0.7071067811865476`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[50]=",
 CellID->434666499]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["NMaximize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMaximize"]], "InlineFormula"],
 " have several optimization methods available: ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<DifferentialEvolution\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<NelderMead\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<RandomSearch\>\""], "InlineFormula"],
 ", and ",
 Cell[BoxData["\"\<SimulatedAnnealing\>\""], "InlineFormula"],
 ". The optimization method is controlled by the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option, which either takes the method as a string, or takes a list whose \
first element is the method as a string and whose remaining elements are \
method\[Hyphen]specific options. All method\[Hyphen]specific option, left\
\[Hyphen]hand sides should also be given as strings."
}], "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->21544639],

Cell["The following function has a large number of local minima.", \
"MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->646694894],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"Clear", "[", "f", "]"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"Sin", "[", 
      RowBox[{"50", " ", "x"}], "]"}]], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{"60", " ", 
      SuperscriptBox["\[ExponentialE]", "y"]}], "]"}], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{"70", " ", 
      RowBox[{"Sin", "[", "x", "]"}]}], "]"}], "+", 
    RowBox[{"Sin", "[", 
     RowBox[{"Sin", "[", 
      RowBox[{"80", " ", "y"}], "]"}], "]"}], "-", 
    RowBox[{"Sin", "[", 
     RowBox[{"10", " ", 
      RowBox[{"(", 
       RowBox[{"x", "+", "y"}], ")"}]}], "]"}], "+", 
    RowBox[{
     FractionBox["1", "4"], " ", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], ")"}]}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot3D", "[", 
  RowBox[{"f", ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", 
     RowBox[{"-", "1"}], ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"y", ",", " ", 
     RowBox[{"-", "1"}], ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"PlotPoints", "\[Rule]", "50"}], ",", " ", 
   RowBox[{"Mesh", "\[Rule]", "False"}]}], "]"}]}], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[51]:=",
 CellID->167316588],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[53]=",
 CellID->26605553]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData["RandomSearch"], "InlineFormula"],
 " to find a minimum."
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->583164225],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<RandomSearch\>\""}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[54]:=",
 CellID->310508122],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "2.8514873980368893`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.44909445947393306`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.2914429522539311`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[54]=",
 CellID->1843386608]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData["RandomSearch"], "InlineFormula"],
 " with more starting points to find the global minimum. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->178377548],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<RandomSearch\>\"", ",", 
      RowBox[{"\"\<SearchPoints\>\"", "\[Rule]", "250"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[55]:=",
 CellID->205282400],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "3.3068686474752376`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.024403079694333304`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0.2106124271555866`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{284, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[55]=",
 CellID->52145387]
}, Open  ]],

Cell[TextData[{
 "With the default method, ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " picks which method to use based on the type of problem. If the objective \
function and constraints are linear, ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 " is used. If there are integer variables, or if the head of the objective \
function is not a numeric function, differential ",
 "evolution",
 " is used. For everything else, it uses Nelder-Mead, but if Nelder-Mead does \
poorly, it switches to differential evolution. "
}], "Text",
 CellID->413147826],

Cell[TextData[{
 "Because the methods used by ",
 Cell[BoxData[
  ButtonBox["NMinimize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NMinimize"]], "InlineFormula"],
 " may not improve every iteration, convergence is only checked after several \
iterations have occurred."
}], "Text",
 CellID->199644619]
}, Open  ]],

Cell[CellGroupData[{

Cell["Numerical Algorithms for Constrained Global Optimization", "Section",
 CellID->252245038],

Cell[CellGroupData[{

Cell["Nelder\[Dash]Mead", "Subsection",
 CellID->323477034],

Cell[TextData[{
 "The Nelder\[Dash]Mead method is a direct search method. For a function of \
",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " variables, the algorithm maintains a set of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "+", "1"}], TraditionalForm]], "InlineMath"],
 " points forming the vertices of a polytope in ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "-dimensional space. This method",
 " ",
 "is often termed the \"simplex\" method, which should not be confused with \
the well-known simplex method for linear programming."
}], "Text",
 CellID->202391529],

Cell[TextData[{
 "At each iteration, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "+", "1"}], TraditionalForm]], "InlineMath"],
 " points ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], ",", 
    SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
    SubscriptBox["x", 
     RowBox[{"n", "+", "1"}]]}], TraditionalForm]], "InlineMath"],
 " form a polytope. The points are ordered so that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "1"], ")"}], "\[LessEqual]", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "2"], ")"}], "\[LessEqual]", "\[Ellipsis]", 
    "\[LessEqual]", 
    RowBox[{
     RowBox[{"f", "(", 
      SubscriptBox["x", 
       RowBox[{"n", "+", "1"}]], ")"}], "."}]}], TraditionalForm]], 
  "InlineMath"],
 " A new point is then generated to replace the worst point ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", 
     RowBox[{"n", "+", "1"}]], "."}], TraditionalForm]], "InlineMath"]
}], "Text",
 CellID->829209197],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " be the centroid of the polytope consisting of the best ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " points, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"c", "=", 
    RowBox[{
     FractionBox["1", "n"], 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "n"], 
      SubscriptBox["x", "i"]}]}]}], TraditionalForm]], "InlineMath"],
 ". A trial point ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "t"], " "}], TraditionalForm]], "InlineMath"],
 "is generated by reflecting the worst point through the centroid, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "t"], "=", 
    RowBox[{"c", "+", 
     RowBox[{"\[Alpha]", " ", 
      RowBox[{"(", 
       RowBox[{"c", "-", 
        SubscriptBox["x", 
         RowBox[{"n", "+", "1"}]]}], ")"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Alpha]", ">", "0"}], TraditionalForm]], "InlineMath"],
 " is a parameter."
}], "Text",
 CellID->500686794],

Cell[TextData[{
 "If the new point ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "t"], TraditionalForm]], "InlineMath"],
 " is neither a new worst point nor a new best point, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "1"], ")"}], "\[LessEqual]", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "t"], ")"}], "\[LessEqual]", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "n"], ")"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "t"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"n", "+", "1"}]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->324409526],

Cell[TextData[{
 "If the new point ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "t"], TraditionalForm]], "InlineMath"],
 " is better than the best point, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "t"], ")"}], "<", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "1"], ")"}]}], TraditionalForm]], "InlineMath"],
 ", the reflection is very successful and can be carried out further to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "e"], "=", 
    RowBox[{"c", "+", 
     RowBox[{"\[Beta]", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["x", "t"], "-", "r"}], ")"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Beta]", ">", "1"}], TraditionalForm]], "InlineMath"],
 " is a parameter to expand the polytope. If the expansion is successful, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "e"], ")"}], "<", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "t"], ")"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "e"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"n", "+", "1"}]], TraditionalForm]], "InlineMath"],
 "; otherwise the expansion failed, and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "t"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"n", "+", "1"}]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->55919927],

Cell[TextData[{
 "If the new point ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "t"], TraditionalForm]], "InlineMath"],
 " is worse than the second worst point, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "t"], ")"}], "\[GreaterEqual]", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "n"], ")"}]}], TraditionalForm]], "InlineMath"],
 ", the polytope is assumed to be too large and needs to be contracted. A new \
trial point is defined as "
}], "Text",
 CellID->519062731],

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["x", "c"], "=", 
   RowBox[{"{", GridBox[{
      {
       RowBox[{
        RowBox[{"c", "+", 
         RowBox[{"\[Gamma]", 
          RowBox[{"(", 
           RowBox[{
            SubscriptBox["x", 
             RowBox[{"n", "+", "1"}]], "-", "c"}], ")"}]}]}], ",", 
        RowBox[{
         RowBox[{"if", 
          RowBox[{"f", "(", 
           SubscriptBox["x", "t"], ")"}]}], "\[GreaterEqual]", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"n", "+", "1"}]], ")"}]}], ","}]},
      {
       RowBox[{
        RowBox[{"c", "+", 
         RowBox[{"\[Gamma]", 
          RowBox[{"(", 
           RowBox[{
            SubscriptBox["x", "t"], "-", "c"}], ")"}]}]}], ",", " ", 
        RowBox[{
         RowBox[{"if", " ", 
          RowBox[{"f", "(", 
           SubscriptBox["x", "t"], ")"}]}], "<", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"n", "+", "1"}]], ")"}]}], ","}]}
     }]}]}], TraditionalForm]], "InlineMath"]], "Text",
 CellID->31377205],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"0", "<", "\[Gamma]", "<", "1"}], TraditionalForm]], 
  "InlineMath"],
 " is a parameter. If ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "c"], ")"}], "<", 
    RowBox[{"Min", "(", 
     RowBox[{
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"n", "+", "1"}]], ")"}], ",", 
      RowBox[{"f", "(", 
       SubscriptBox["x", "t"], ")"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ", the contraction is successful, and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "c"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"n", "+", "1"}]], TraditionalForm]], "InlineMath"],
 ". Otherwise a further contraction is carried out."
}], "Text",
 CellID->281027891],

Cell[TextData[{
 "The process is assumed to have converged if the difference between the best \
function values in the new and old polytope, as well as the distance between \
the new best point and the old best point, are less than the tolerances \
provided by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 "."
}], "Text",
 CellID->455205358],

Cell["\<\
Strictly speaking, Nelder\[Dash]Mead is not a true global optimization \
algorithm; however, in practice it tends to work reasonably well for problems \
that do not have a lot of local minima.\
\>", "Text",
 CellID->682966497],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<ContractRatio\>\"", "0.5", Cell[
    "ratio used for contraction ", "TableText"]},
   {"\"\<ExpandRatio\>\"", "2.0", Cell[
    "ratio used for expansion ", "TableText"]},
   {"\"\<InitialPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "set of initial points ", "TableText"]},
   {"\"\<PenaltyFunction\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "function applied to constraints to penalize invalid points", 
     "TableText"]},
   {"\"\<PostProcess\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "whether to post-process using local search methods ", "TableText"]},
   {"\"\<RandomSeed\>\"", "0", Cell[
    "starting value for the random number generator", "TableText"]},
   {"\"\<ReflectRatio\>\"", "1.0", Cell[
    "ratio used for reflection ", "TableText"]},
   {"\"\<ShrinkRatio\>\"", "0.5", Cell[
    "ratio used for shrinking ", "TableText"]},
   {"\"\<Tolerance\>\"", "0.001", Cell[
    "tolerance for accepting constraint violations ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->555161960],

Cell[TextData[{
 Cell[BoxData["NelderMead"], "InlineFormula"],
 " specific options."
}], "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->125418266],

Cell[TextData[{
 "Here the function inside the unit disk is minimized using ",
 Cell[BoxData["NelderMead"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->78037682],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"100", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "-", 
          SuperscriptBox["x", "2"]}], ")"}], "2"]}], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "x"}], ")"}], "2"]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], "\[LessEqual]", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<NelderMead\>\""}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[82]:=",
 CellID->103075800],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.04567480849953805`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.7864151546826513`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.6176983133341885`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{276, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[82]=",
 CellID->593065773]
}, Open  ]],

Cell["\<\
Here is a function with several local minima that are all different depths. \
\>", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->453777393],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Clear", "[", 
   RowBox[{"a", ",", "f"}], "]"}], ";", 
  RowBox[{"a", "=", 
   RowBox[{"Reverse", "/@", 
    RowBox[{"Distribute", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "32"}], ",", 
          RowBox[{"-", "16"}], ",", "0", ",", "16", ",", "32"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "32"}], ",", 
          RowBox[{"-", "16"}], ",", "0", ",", "16", ",", "32"}], "}"}]}], 
       "}"}], ",", "List"}], "]"}]}]}], ";", 
  RowBox[{"f", "=", 
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"0.002", "+", 
      RowBox[{"Plus", "@@", 
       RowBox[{"MapIndexed", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"1", "/", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
             "#2", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
             "+", 
             RowBox[{"Plus", "@@", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"{", 
                   RowBox[{"x", ",", "y"}], "}"}], "-", "#1"}], ")"}], "^", 
                "6"}], ")"}]}]}], ")"}]}], "&"}], ",", "a"}], "]"}]}]}], 
     ")"}]}]}], ";", 
  RowBox[{"Plot3D", "[", 
   RowBox[{"f", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", " ", 
    RowBox[{"Mesh", "\[Rule]", "None"}], ",", " ", 
    RowBox[{"PlotPoints", "\[Rule]", "25"}]}], "]"}]}]], "Input",
 CellLabel->"In[83]:=",
 CellID->75686477],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 288},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[83]=",
 CellID->174171999]
}, Open  ]],

Cell[TextData[{
 "With the default parameters, ",
 Cell[BoxData["NelderMead"], "InlineFormula"],
 " is too easily trapped in a local minimum. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->11406102],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Do", "[", 
  RowBox[{
   RowBox[{"Print", "[", 
    RowBox[{"NMinimize", "[", 
     RowBox[{"f", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}]}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<NelderMead\>\"", ",", 
         RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "i"}]}], "}"}]}]}], "]"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[116]:=",
 CellID->45005793],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.968250105568163`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "15.981573438746885`"}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.960834569533965`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[116]:=",
 CellID->2029791019],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"12.670505812010909`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.02779001080493091`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.5739352038003517`*^-6"}]}], "}"}]}], 
  "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[116]:=",
 CellID->1919923738],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"10.763180666779332`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.941210169130315`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "0.025346537833999896`"}]}], "}"}]}], 
  "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[116]:=",
 CellID->68984744],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.992030900220718`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.986402444197545`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.97033526825762`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[116]:=",
 CellID->1932906675],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"16.440907314171774`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.963353148511601`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "15.963357890238806`"}]}], "}"}]}], 
  "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[116]:=",
 CellID->1377358556]
}, Open  ]]
}, Open  ]],

Cell["\<\
By using settings that are more aggressive and less likely to make the \
simplex smaller, the results are better. \
\>", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->100162883],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Do", "[", 
  RowBox[{
   RowBox[{"Print", "[", 
    RowBox[{"NMinimize", "[", 
     RowBox[{"f", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}]}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<NelderMead\>\"", ",", " ", 
         RowBox[{"\"\<ShrinkRatio\>\"", "\[Rule]", "0.95"}], ",", 
         RowBox[{"\"\<ContractRatio\>\"", "\[Rule]", "0.95"}], ",", 
         RowBox[{"\"\<ReflectRatio\>\"", "\[Rule]", "2"}], ",", 
         RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "i"}]}], "}"}]}]}], "]"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[117]:=",
 CellID->64152227],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.968250105568163`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "15.981574267496091`"}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.96083459696116`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[117]:=",
 CellID->1539019679],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.9821051568166754`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "0.013236235205762331`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.965090174239275`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[117]:=",
 CellID->2010700485],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.9920309002207184`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.986400106179346`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.97033574742561`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[117]:=",
 CellID->1040382203],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"16.440907314171778`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.963352697143403`"}]}], ",", 
     RowBox[{"y", "\[Rule]", "15.963352026770222`"}]}], "}"}]}], 
  "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[117]:=",
 CellID->891831947],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.99800383779445`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.97833480591634`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.97833476498007`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[117]:=",
 CellID->1731160816]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Differential Evolution", "Subsection",
 CellID->24713453],

Cell["\<\
Differential evolution is a simple stochastic function minimizer.\
\>", "Text",
 CellID->15517963],

Cell[TextData[{
 "The algorithm maintains a population of ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " points, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["x", "j"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["x", "m"]}], "}"}], TraditionalForm]], "InlineMath"],
 ", where typically ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", ">>", 
    StyleBox["n", "TI"]}], TraditionalForm]], "InlineMath"],
 ", with ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " being the number of variables. "
}], "Text",
 CellID->400968379],

Cell[TextData[{
 "During each iteration of the algorithm, a new population of ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " points is generated. The ",
 Cell[BoxData[
  FormBox["j", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " new point is generated by picking three random points, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "u"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "v"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "w"], TraditionalForm]], "InlineMath"],
 ", from the old population, and forming ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "s"], "=", 
    RowBox[{
     SubscriptBox["x", "w"], "+", 
     RowBox[{"s", "(", 
      RowBox[{
       SubscriptBox["x", "u"], "-", 
       SubscriptBox["x", "v"]}], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 " is a real scaling factor. Then a new point ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "new"], TraditionalForm]], "InlineMath"],
 " is constructed from ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "j"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "s"], TraditionalForm]], "InlineMath"],
 " by taking the ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " coordinate from ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "s"], TraditionalForm]], "InlineMath"],
 " with probability ",
 Cell[BoxData[
  FormBox["\[Rho]", TraditionalForm]], "InlineMath"],
 " and otherwise taking the coordinate from ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "j"], TraditionalForm]], "InlineMath"],
 ". If ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "new"], ")"}], "<", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "j"], ")"}]}], TraditionalForm]], "InlineMath"],
 ", then ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "new"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "j"], TraditionalForm]], "InlineMath"],
 " in the population. The probability \[Rho] is controlled by the ",
 Cell[BoxData["\"\<CrossProbability\>\""], "InlineFormula"],
 " option."
}], "Text",
 CellID->134958075],

Cell[TextData[{
 "The process is assumed to have converged if the difference between the best \
function values in the new and old populations, as well as the distance \
between the new best point and the old best point, are less than the \
tolerances provided by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 "."
}], "Text",
 CellID->666318363],

Cell["\<\
The differential evolution method is computationally expensive, but is \
relatively robust and tends to work well for problems that have more local \
minimums.\
\>", "Text",
 CellID->373195045],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<CrossProbability\>\"", "0.5", Cell[TextData[{
     "probability that a gene is taken from ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"]
    }], "TableText"]},
   {"\"\<InitialPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "set of initial points ", "TableText"]},
   {"\"\<PenaltyFunction\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "function applied to constraints to penalize invalid points", 
     "TableText"]},
   {"\"\<PostProcess\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "whether to post-process using local search methods ", "TableText"]},
   {"\"\<RandomSeed\>\"", "0", Cell[
    "starting value for the random number generator", "TableText"]},
   {"\"\<ScalingFactor\>\"", "0.6", Cell[
    "scale applied to the difference vector in creating a mate ", 
     "TableText"]},
   {"\"\<SearchPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "size of the population used for evolution ", "TableText"]},
   {"\"\<Tolerance\>\"", "0.001", Cell[
    "tolerance for accepting constraint violations ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->18427541],

Cell[TextData[{
 Cell[BoxData["DifferentialEvolution"], "InlineFormula"],
 " specific options."
}], "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->142614438],

Cell[TextData[{
 "Here the function inside the unit disk is minimized using ",
 Cell[BoxData["DifferentialEvolution"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->91972744],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"100", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "-", 
          SuperscriptBox["x", "2"]}], ")"}], "2"]}], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "x"}], ")"}], "2"]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], "\[LessEqual]", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<DifferentialEvolution\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[125]:=",
 CellID->676994573],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.04567480849953805`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.7864151546826513`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.6176983133341885`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{276, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[125]=",
 CellID->1832991483]
}, Open  ]],

Cell[TextData[{
 "The following constrained optimization problem has a global minimum of ",
 Cell[BoxData[
  FormBox["7.667180068813135`", TraditionalForm]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->79261794],

Cell[BoxData[
 RowBox[{"Clear", "[", 
  RowBox[{
  "f", ",", "c", ",", "v", ",", "x1", ",", "x2", ",", "y1", ",", "y2", ",", 
   "y3"}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[126]:=",
 CellID->103880104],

Cell[BoxData[{
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{
    RowBox[{"2", " ", "x1"}], "+", 
    RowBox[{"3", " ", "x2"}], "+", 
    RowBox[{"3", " ", 
     RowBox[{"y1", "/", "2"}]}], "+", 
    RowBox[{"2", " ", "y2"}], "-", 
    RowBox[{"y3", "/", "2"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"c", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x1", "^", "2"}], "+", "y1"}], "\[Equal]", 
      RowBox[{"5", "/", "4"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x2", "^", 
        RowBox[{"(", 
         RowBox[{"3", "/", "2"}], ")"}]}], "+", 
       RowBox[{"3", " ", 
        RowBox[{"y2", "/", "2"}]}]}], "==", "3"}], ",", 
     RowBox[{
      RowBox[{"x1", "+", "y1"}], "\[LessEqual]", 
      RowBox[{"8", "/", "5"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"4", " ", 
        RowBox[{"x2", "/", "3"}]}], "+", "y2"}], "\[LessEqual]", "3"}], ",", 
     RowBox[{"y3", "\[LessEqual]", 
      RowBox[{"y1", "+", "y2"}]}], ",", 
     RowBox[{"0", "\[LessEqual]", "x1", "\[LessEqual]", "10"}], ",", 
     RowBox[{"0", "\[LessEqual]", "x2", "\[LessEqual]", "10"}], ",", 
     RowBox[{"0", "\[LessEqual]", "y1", "\[LessEqual]", "1"}], ",", 
     RowBox[{"0", "\[LessEqual]", "y2", "\[LessEqual]", "1"}], ",", 
     RowBox[{"0", "\[LessEqual]", "y3", "\[LessEqual]", "1"}], ",", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"y1", ",", "y2", ",", "y3"}], "}"}], "\[Element]", 
      "Integers"}]}], "\[IndentingNewLine]", "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"{", 
    RowBox[{"x1", ",", "x2", ",", "y1", ",", "y2", ",", "y3"}], "}"}]}], 
  ";"}]}], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[127]:=",
 CellID->245307687],

Cell[TextData[{
 "With the default settings for ",
 Cell[BoxData["DifferentialEvolution"], "InlineFormula"],
 ", an unsatisfactory solution results."
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->302105471],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"f", ",", "c"}], "}"}], ",", "v", ",", 
   RowBox[{"Method", "\[Rule]", "\"\<DifferentialEvolution\>\""}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[130]:=",
 CellID->76487140],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7.930859764401132`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x1", "\[Rule]", "0.49993092379336745`"}], ",", 
     RowBox[{"x2", "\[Rule]", "1.3103326389381322`"}], ",", 
     RowBox[{"y1", "\[Rule]", "1"}], ",", 
     RowBox[{"y2", "\[Rule]", "1"}], ",", 
     RowBox[{"y3", "\[Rule]", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{413, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[130]=",
 CellID->1434938566]
}, Open  ]],

Cell[TextData[{
 "By adjusting ",
 Cell[BoxData["ScalingFactor"], "InlineFormula"],
 ", the results are much better. In this case, the increased ",
 Cell[BoxData["ScalingFactor"], "InlineFormula"],
 " gives ",
 Cell[BoxData["DifferentialEvolution"], "InlineFormula"],
 " better mobility with respect to the integer variables."
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->58027804],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"f", ",", " ", "c"}], "}"}], ",", " ", "v", ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<DifferentialEvolution\>\"", ",", " ", 
      RowBox[{"\"\<ScalingFactor\>\"", "\[Rule]", "1"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[131]:=",
 CellID->103201209],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7.66718006522491`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x1", "\[Rule]", "1.1180339883773303`"}], ",", 
     RowBox[{"x2", "\[Rule]", "1.3103706961567498`"}], ",", 
     RowBox[{"y1", "\[Rule]", "0"}], ",", 
     RowBox[{"y2", "\[Rule]", "1"}], ",", 
     RowBox[{"y3", "\[Rule]", "1"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{406, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[131]=",
 CellID->771122952]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simulated Annealing", "Subsection",
 CellID->7062708],

Cell["\<\
Simulated annealing is a simple stochastic function minimizer. It is \
motivated from the physical process of annealing, where a metal object is \
heated to a high temperature and allowed to cool slowly. The process allows \
the atomic structure of the metal to settle to a lower energy state, thus \
becoming a tougher metal. Using optimization terminology, annealing allows \
the structure to escape from a local minimum, and to explore and settle on a \
better, hopefully global, minimum.\
\>", "Text",
 CellID->51945444],

Cell[TextData[{
 "At each iteration, a new point, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "new"], TraditionalForm]], "InlineMath"],
 ", is generated in the neighborhood of the current point, ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ". The radius of the neighborhood decreases with each iteration. The best \
point found so far, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "best"], TraditionalForm]], "InlineMath"],
 ", is also tracked."
}], "Text",
 CellID->9567416],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "new"], ")"}], "\[LessEqual]", 
    RowBox[{"f", "(", 
     SubscriptBox["x", "best"], ")"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "new"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "best"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ". Otherwise, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "new"], TraditionalForm]], "InlineMath"],
 " replaces ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " with a probability ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["e", 
    RowBox[{"b", "(", 
     RowBox[{"i", ",", " ", 
      RowBox[{"\[CapitalDelta]", "\[InvisibleTimes]", "f"}], " ", ",", " ", 
      SubscriptBox["f", "0"]}], ")"}]], TraditionalForm]], "InlineMath"],
 ". Here ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 " is the function defined by ",
 Cell[BoxData["BoltzmannExponent"], "InlineFormula"],
 ", ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 " is the current iteration, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalDelta]", "\[InvisibleTimes]", "f"}], TraditionalForm]], 
  "InlineMath"],
 " is the change in the objective function value, and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["f", "0"], TraditionalForm]], "InlineMath"],
 " is the value of the objective function from the previous iteration. The \
default function for ",
 Cell[BoxData[
  FormBox["b", TraditionalForm]], "InlineMath"],
 " is ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{
     RowBox[{"-", "\[CapitalDelta]"}], "\[InvisibleSpace]", "f", " ", 
     RowBox[{"log", "(", 
      RowBox[{"i", "+", "1"}], ")"}]}], "10"], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->84070994],

Cell[TextData[{
 "Like the ",
 Cell[BoxData["RandomSearch"], "InlineFormula"],
 " method, ",
 Cell[BoxData["SimulatedAnnealing"], "InlineFormula"],
 " uses multiple starting points, and finds an optimum starting from each of \
them."
}], "Text",
 CellID->234191333],

Cell[TextData[{
 "The default number of starting points, given by the option ",
 Cell[BoxData["SearchPoints"], "InlineFormula"],
 StyleBox[",", "MR"],
 "is ",
 Cell[BoxData[
  FormBox[
   RowBox[{"min", "(", 
    RowBox[{
     RowBox[{"2", " ", "d"}], ",", "50"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is the number of variables."
}], "Text",
 CellID->2029629094],

Cell[TextData[{
 "For each starting point, this is repeated until the maximum number of \
iterations is reached, the method converges to a point, or the method stays \
at the same point consecutively for the number of iterations given by ",
 Cell[BoxData["LevelIterations"], "InlineFormula"],
 "."
}], "Text",
 CellID->1863692427],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<BoltzmannExponent\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "exponent of the probability function ", "TableText"]},
   {"\"\<InitialPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "set of initial points ", "TableText"]},
   {"\"\<LevelIterations\>\"", "50", Cell[
    "maximum number of iterations to stay at a given point ", "TableText"]},
   {"\"\<PenaltyFunction\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "function applied to constraints to penalize invalid points", 
     "TableText"]},
   {"\"\<PerturbationScale\>\"", "1.0", Cell[
    "scale for the random jump ", "TableText"]},
   {"\"\<PostProcess\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "whether to post-process using local search methods ", "TableText"]},
   {"\"\<RandomSeed\>\"", "0", Cell[
    "starting value for the random number generator", "TableText"]},
   {"\"\<SearchPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of initial points ", "TableText"]},
   {"\"\<Tolerance\>\"", "0.001", Cell[
    "tolerance for accepting constraint violations ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->459034107],

Cell[TextData[{
 Cell[BoxData["SimulatedAnnealing"], "InlineFormula"],
 " specific options. "
}], "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->88013361],

Cell[TextData[{
 "Here a function in two variables is minimized using ",
 Cell[BoxData["SimulatedAnnealing"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->136327480],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"100", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "-", 
          SuperscriptBox["x", "2"]}], ")"}], "2"]}], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "x"}], ")"}], "2"]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"-", "2.084"}], "\[LessEqual]", "x", "\[LessEqual]", "2.084"}],
       "&&", 
      RowBox[{
       RowBox[{"-", "2.084"}], "\[LessEqual]", "y", "\[LessEqual]", 
       "2.084"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<SimulatedAnnealing\>\""}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[62]:=",
 CellID->17340756],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{143, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[62]=",
 CellID->1345933910]
}, Open  ]],

Cell["Here is a function with many local minima. ", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->350455679],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Clear", "[", "f", "]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", ",", "y_"}], "]"}], ":=", 
   RowBox[{
    RowBox[{"20", 
     RowBox[{"Sin", "[", 
      RowBox[{
       RowBox[{"\[Pi]", "/", "2"}], 
       RowBox[{"(", 
        RowBox[{"x", "-", 
         RowBox[{"2", "\[Pi]"}]}], ")"}]}], "]"}]}], "+", 
    RowBox[{"20", 
     RowBox[{"Sin", "[", 
      RowBox[{
       RowBox[{"\[Pi]", "/", "2"}], 
       RowBox[{"(", 
        RowBox[{"y", "-", 
         RowBox[{"2", "\[Pi]"}]}], ")"}]}], "]"}]}], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "-", 
       RowBox[{"2", "\[Pi]"}]}], ")"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"y", "-", 
       RowBox[{"2", "\[Pi]"}]}], ")"}], "2"]}]}], ";"}], "\n", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "10"}], "}"}]}], "]"}]}], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[63]:=",
 CellID->409422994],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 298},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[65]=",
 CellID->831528278]
}, Open  ]],

Cell[TextData[{
 "By default, the step size for ",
 Cell[BoxData["SimulatedAnnealing"], "InlineFormula"],
 " is not large enough to escape from the local minima. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->475197752],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<SimulatedAnnealing\>\""}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[68]:=",
 CellID->311100400],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"8.0375008222486`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "1.4809782567471934`"}], ",", 
     RowBox[{"y", "\[Rule]", "1.4809782568539245`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{241, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[68]=",
 CellID->819788825]
}, Open  ]],

Cell[TextData[{
 "By increasing ",
 Cell[BoxData["PerturbationScale"], "InlineFormula"],
 ", larger step sizes are taken to produce a much better solution. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->13451302],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<SimulatedAnnealing\>\"", ",", 
      RowBox[{"\"\<PerturbationScale\>\"", "\[Rule]", "3"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[69]:=",
 CellID->300776615],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "38.07792316615158`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "5.322158592193681`"}], ",", 
     RowBox[{"y", "\[Rule]", "5.322158592193681`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[69]=",
 CellID->2083186270]
}, Open  ]],

Cell[TextData[{
 "Here ",
 Cell[BoxData["BoltzmannExponent"], "InlineFormula"],
 " is set to use an exponential cooling function, that gives faster \
convergence. (Note that the modified ",
 Cell[BoxData["PerturbationScale"], "InlineFormula"],
 " is still being used as well.) "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->273817704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<SimulatedAnnealing\>\"", ",", 
      RowBox[{"\"\<PerturbationScale\>\"", "\[Rule]", "3"}], ",", 
      RowBox[{"\"\<BoltzmannExponent\>\"", "\[Rule]", 
       RowBox[{"Function", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"i", ",", "df", ",", "f0"}], "}"}], ",", 
         RowBox[{
          RowBox[{"-", "df"}], "/", 
          RowBox[{"(", 
           RowBox[{"Exp", "[", 
            RowBox[{"i", "/", "10"}], "]"}], ")"}]}]}], "]"}]}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[70]:=",
 CellID->251756205],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "38.07792316615158`"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "5.322158592193681`"}], ",", 
     RowBox[{"y", "\[Rule]", "5.322158592193681`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{256, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[70]=",
 CellID->1685742849]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Random Search", "Subsection",
 CellID->155930577],

Cell["\<\
The random search algorithm works by generating a population of random \
starting points and uses a local optimization method from each of the \
starting points to converge to a local minimum. The best local minimum is \
chosen to be the solution.\
\>", "Text",
 CellID->417700783],

Cell[TextData[{
 "The possible local search methods are ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", and ",
 Cell[BoxData["\"\<InteriorPoint\>\""], "InlineFormula"],
 ". The default method is ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", which uses ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " with unconstrained methods applied to a system with penalty terms added \
for the constraints. When ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " is set to ",
 Cell[BoxData["\"\<InteriorPoint\>\""], "InlineFormula"],
 ", a nonlinear interior point method is used."
}], "Text",
 CellID->1810634708],

Cell[TextData[{
 "The default number of starting points, given by the option ",
 StyleBox["SearchPoints,", "MR"],
 "is ",
 Cell[BoxData[
  FormBox[
   RowBox[{"min", "(", 
    RowBox[{
     RowBox[{"10", " ", "d"}], ",", "100"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is the number of variables."
}], "Text",
 CellID->1494259366],

Cell[TextData[{
 "Convergence for ",
 Cell[BoxData["RandomSearch"], "InlineFormula"],
 " is determined by convergence of the local method for each starting point."
}], "Text",
 CellID->1343396344],

Cell[TextData[{
 StyleBox["RandomSearch", "MR"],
 " is fast, but does not scale very well with the dimension of the search \
space. It also suffers from many of the same limitations as ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 ". It is not well suited for discrete problems and others where derivatives \
or secants give little useful information about the problem. "
}], "Text",
 CellID->114943451],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<InitialPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "set of initial points", "TableText"]},
   {"\"\<Method\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "which method to use for minimization ", "TableText"]},
   {"\"\<PenaltyFunction\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "function applied to constraints to penalize invalid points", 
     "TableText"]},
   {"\"\<PostProcess\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "whether to post-process using local search methods ", "TableText"]},
   {"\"\<RandomSeed\>\"", "0", Cell[
    "starting value for the random number generator", "TableText"]},
   {"\"\<SearchPoints\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of points to use for starting local searches ", "TableText"]},
   {"\"\<Tolerance\>\"", "0.001", Cell[
    "tolerance for accepting constraint violations ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->13679150],

Cell[TextData[{
 Cell[BoxData["RandomSearch"], "InlineFormula"],
 " specific options. "
}], "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->63134833],

Cell[TextData[{
 "Here the function inside the unit disk is minimized using ",
 Cell[BoxData["RandomSearch"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->385877974],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"100", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "-", 
          SuperscriptBox["x", "2"]}], ")"}], "2"]}], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "x"}], ")"}], "2"]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "\[LessEqual]", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<RandomSearch\>\""}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[71]:=",
 CellID->195334256],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.04567480849953805`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", "0.7864151546826513`"}], ",", 
     RowBox[{"y", "\[Rule]", "0.6176983133341885`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{276, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[71]=",
 CellID->1966772697]
}, Open  ]],

Cell["\<\
Here is a function with several local minima that are all different depths \
and are generally difficult to optimize. \
\>", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->132522388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Clear", "[", 
   RowBox[{"a", ",", "f"}], "]"}], ";", 
  RowBox[{"a", "=", 
   RowBox[{"Reverse", "/@", 
    RowBox[{"Distribute", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "32"}], ",", 
          RowBox[{"-", "16"}], ",", "0", ",", "16", ",", "32"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "32"}], ",", 
          RowBox[{"-", "16"}], ",", "0", ",", "16", ",", "32"}], "}"}]}], 
       "}"}], ",", "List"}], "]"}]}]}], ";", 
  RowBox[{"f", "=", 
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"0.002", "+", 
      RowBox[{"Plus", "@@", 
       RowBox[{"MapIndexed", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"1", "/", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
             "#2", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
             "+", 
             RowBox[{"Plus", "@@", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{"{", 
                   RowBox[{"x", ",", "y"}], "}"}], "-", "#1"}], ")"}], "^", 
                "6"}], ")"}]}]}], ")"}]}], "&"}], ",", "a"}], "]"}]}]}], 
     ")"}]}]}], ";", 
  RowBox[{"Plot3D", "[", 
   RowBox[{"f", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
    RowBox[{"Mesh", "\[Rule]", "None"}], ",", " ", 
    RowBox[{"NormalsFunction", "\[Rule]", "\"\<Weighted\>\""}], ",", " ", 
    RowBox[{"PlotPoints", "\[Rule]", "50"}]}], "]"}]}]], "Input",
 CellLabel->"In[72]:=",
 CellID->282281265],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 288},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[72]=",
 CellID->126164948]
}, Open  ]],

Cell[TextData[{
 "With the default number of ",
 Cell[BoxData["SearchPoints"], "InlineFormula"],
 ", sometimes the minimum is not found. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->448066845],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Do", "[", 
  RowBox[{
   RowBox[{"Print", "[", 
    RowBox[{"NMinimize", "[", 
     RowBox[{"f", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}]}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<RandomSearch\>\"", ",", 
         RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "i"}]}], "}"}]}]}], "]"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[73]:=",
 CellID->227542198],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.992030900220718`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.986400105612574`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.97033574743256`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[73]:=",
 CellID->738196843],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.9920309002207175`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.986401787361332`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.970335513201864`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[73]:=",
 CellID->861311582],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9980038377944502`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.978334835656362`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978334837300377`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[73]:=",
 CellID->2023977894],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.9920309002207175`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "15.98639982481938`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.970335602594332`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[73]:=",
 CellID->855495978],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9980038377944502`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.978334867811007`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978334696450705`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[73]:=",
 CellID->917885581]
}, Open  ]]
}, Open  ]],

Cell[TextData[{
 "By using many more ",
 Cell[BoxData["SearchPoints"], "InlineFormula"],
 ", answers are better. "
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->119506098],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Do", "[", 
  RowBox[{
   RowBox[{"Print", "[", 
    RowBox[{"NMinimize", "[", 
     RowBox[{"f", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "50"}], ",", "50"}], "}"}]}], "}"}], ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<RandomSearch\>\"", ",", 
         RowBox[{"\"\<SearchPoints\>\"", "\[Rule]", "100"}], ",", " ", 
         RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "i"}]}], "}"}]}]}], "]"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[74]:=",
 CellID->4613958],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9980038377944502`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.978336590765686`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978335429688844`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[74]:=",
 CellID->1527923466],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9980038377944502`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.97833755222663`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.97833945273331`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[74]:=",
 CellID->200242236],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.99800383779445`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.9783348977118`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978335199921624`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[74]:=",
 CellID->364385603],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.99800383779445`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.978335005635675`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978334921293637`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[74]:=",
 CellID->1752952024],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.9980038377944502`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.978334867811007`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978334696450705`"}]}]}], "}"}]}], "}"}]], "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"During evaluation of In[74]:=",
 CellID->536514175]
}, Open  ]]
}, Open  ]],

Cell["\<\
Here points are generated on a grid for use as initial points. \
\>", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->107225079],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NMinimize", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "50"}], ",", "50"}], "}"}]}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<RandomSearch\>\"", ",", 
      RowBox[{"\"\<InitialPoints\>\"", "\[Rule]", 
       RowBox[{"Flatten", "[", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"i", ",", "j"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"i", ",", 
             RowBox[{"-", "45"}], ",", "45", ",", "5"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"j", ",", 
             RowBox[{"-", "45"}], ",", "45", ",", "5"}], "}"}]}], "]"}], ",", 
         "1"}], "]"}]}]}], "}"}]}]}], "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[75]:=",
 CellID->61749050],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.99800383779445`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", "31.978335074813447`"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", "31.978334718129744`"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{270, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[75]=",
 CellID->2057389119]
}, Open  ]],

Cell["\<\
This uses nonlinear interior point methods to find the minimum of a sum of \
squares.\
\>", "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->34899127],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "10"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"f", "=", 
   RowBox[{"Sum", "[", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{"x", "[", "i", "]"}], "-", 
        RowBox[{"Sin", "[", "i", "]"}]}], ")"}], "2"], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"c", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "0.5"}], "<", 
      RowBox[{"x", "[", "i", "]"}], "<", "0.5"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"Array", "[", 
    RowBox[{"x", ",", "n"}], "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"Timing", "[", 
  RowBox[{"NMinimize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"f", ",", "c"}], "}"}], ",", "v", ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<RandomSearch\>\"", ",", 
       RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}]}], "}"}]}]}], 
   "]"}], "]"}]}], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[76]:=",
 CellID->659368774],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"8.258761`", ",", 
   RowBox[{"{", 
    RowBox[{"0.826739972462804`", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "[", "1", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "2", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "3", "]"}], "\[Rule]", "0.1411200080598672`"}], ",",
        
       RowBox[{
        RowBox[{"x", "[", "4", "]"}], "\[Rule]", 
        RowBox[{"-", "0.5`"}]}], ",", 
       RowBox[{
        RowBox[{"x", "[", "5", "]"}], "\[Rule]", 
        RowBox[{"-", "0.5`"}]}], ",", 
       RowBox[{
        RowBox[{"x", "[", "6", "]"}], "\[Rule]", 
        RowBox[{"-", "0.27941549819892586`"}]}], ",", 
       RowBox[{
        RowBox[{"x", "[", "7", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "8", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "9", "]"}], "\[Rule]", "0.4121184852417566`"}], ",",
        
       RowBox[{
        RowBox[{"x", "[", "10", "]"}], "\[Rule]", 
        RowBox[{"-", "0.5`"}]}]}], "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{566, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[80]=",
 CellID->1904722686]
}, Open  ]],

Cell[TextData[{
 "For some classes of problems, limiting the number of ",
 Cell[BoxData["SearchPoints"], "InlineFormula"],
 " can be much faster without affecting the quality of the solution."
}], "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->158328164],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"NMinimize", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"f", ",", "c"}], "}"}], ",", "v", ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<RandomSearch\>\"", ",", 
       RowBox[{"Method", "\[Rule]", "\"\<InteriorPoint\>\""}], ",", " ", 
       RowBox[{"\"\<SearchPoints\>\"", "\[Rule]", "1"}]}], "}"}]}]}], "]"}], 
  "]"}]], "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"In[81]:=",
 CellID->376120086],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.3204250000000002`", ",", 
   RowBox[{"{", 
    RowBox[{"0.826739972462804`", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "[", "1", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "2", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "3", "]"}], "\[Rule]", "0.1411200080598672`"}], ",",
        
       RowBox[{
        RowBox[{"x", "[", "4", "]"}], "\[Rule]", 
        RowBox[{"-", "0.5`"}]}], ",", 
       RowBox[{
        RowBox[{"x", "[", "5", "]"}], "\[Rule]", 
        RowBox[{"-", "0.5`"}]}], ",", 
       RowBox[{
        RowBox[{"x", "[", "6", "]"}], "\[Rule]", 
        RowBox[{"-", "0.27941549819892586`"}]}], ",", 
       RowBox[{
        RowBox[{"x", "[", "7", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "8", "]"}], "\[Rule]", "0.5`"}], ",", 
       RowBox[{
        RowBox[{"x", "[", "9", "]"}], "\[Rule]", "0.4121184852417566`"}], ",",
        
       RowBox[{
        RowBox[{"x", "[", "10", "]"}], "\[Rule]", 
        RowBox[{"-", "0.5`"}]}]}], "}"}]}], "}"}]}], "}"}]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{573, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellLabel->"Out[81]=",
 CellID->135457773]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->256102893],

Cell[TextData[ButtonBox["Constrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationOverview"]], "RelatedTutorials",
 CellID->1633042],

Cell[TextData[ButtonBox["Linear Programming",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationLinearProgramming"]], \
"RelatedTutorials",
 CellID->282186215],

Cell[TextData[ButtonBox["Numerical Nonlinear Local Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationLocalNumerical"]], \
"RelatedTutorials",
 CellID->511088424],

Cell[TextData[ButtonBox["Exact Global Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/ConstrainedOptimizationExact"]], "RelatedTutorials",
 CellID->140984286]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Numerical Nonlinear Global Optimization - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 33, 28.0772100}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "Numerical algorithms for constrained nonlinear optimization can be \
broadly categorized into gradient based methods and direct search methods. \
Gradient-based methods use first derivatives (gradients) or second \
derivatives (Hessians). Examples are the sequential quadratic programming \
(SQP) method, the augmented Lagrangian method, and the (nonlinear) interior \
point method. Direct search methods do not use derivative information. \
Examples are Nelder-Mead, genetic algorithm and differential evolution, and \
simulated annealing. Direct search methods tend to converge more slowly, but \
can be more tolerant to the presence of noise in the function and \
constraints. Typically, algorithms only build up a local model of the \
problems. Furthermore, many such algorithms insist on certain decrease of the \
objective function, or decrease of a merit function which is a combination of \
the objective and constraints, to ensure convergence of the iterative \
process. Such algorithms will, if convergent, only find local optima, and are \
called local optimization algorithms. In Mathematica local optimization \
problems can be solved using FindMinimum. Global optimization algorithms, on \
the other hand, attempt to find the global optimum, typically by allowing \
decrease as well as increase of the objective/merit function. Such algorithms \
are usually computationally more expensive. Global optimization problems can \
be solved exactly using Minimize or numerically using NMinimize.", 
    "synonyms" -> {}, "title" -> "Numerical Nonlinear Global Optimization", 
    "type" -> "Tutorial", "uri" -> 
    "tutorial/ConstrainedOptimizationGlobalNumerical"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "NMinimize:Contents"->{
  Cell[8730, 306, 564, 15, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->16616413],
  Cell[9297, 323, 349, 7, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->441818026],
  Cell[9649, 332, 346, 12, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->13823121],
  Cell[10020, 348, 370, 12, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->158556129],
  Cell[10393, 362, 425, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->147730805],
  Cell[10833, 378, 628, 22, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->8449423],
  Cell[11486, 404, 552, 18, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->479319689],
  Cell[12041, 424, 447, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->420543703],
  Cell[12503, 441, 1522, 43, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->55770639],
  Cell[14028, 486, 394, 14, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->278597936],
  Cell[14447, 504, 465, 15, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->105413790],
  Cell[14915, 521, 371, 12, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->218869012],
  Cell[15301, 536, 216, 7, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->869892494],
  Cell[15542, 547, 572, 19, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->434989722],
  Cell[16117, 568, 404, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->386809913],
  Cell[16536, 584, 1356, 36, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->408968279],
  Cell[17895, 622, 183, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->92233752],
  Cell[18103, 631, 286, 9, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->328088122],
  Cell[18392, 642, 309, 10, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->71577405],
  Cell[18716, 655, 146, 2, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->588472900],
  Cell[18887, 661, 334, 10, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->160567602],
  Cell[19224, 673, 310, 10, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->285978531],
  Cell[19549, 686, 738, 22, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->542356352],
  Cell[20312, 712, 528, 17, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->164914745],
  Cell[20843, 731, 415, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->393800755],
  Cell[21273, 747, 586, 22, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->110392765],
  Cell[21884, 773, 372, 11, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->841906160],
  Cell[22259, 786, 330, 11, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->709334044],
  Cell[22604, 800, 211, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->262757712],
  Cell[22840, 810, 488, 15, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->891110009],
  Cell[23331, 827, 349, 11, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->434666499],
  Cell[23695, 841, 1240, 34, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->21544639],
  Cell[24938, 877, 151, 3, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->646694894],
  Cell[25114, 884, 1357, 41, 77, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->167316588],
  Cell[26474, 927, 33567, 555, 162, "Output",
   Evaluatable->False,
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->26605553],
  Cell[60056, 1485, 183, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->583164225],
  Cell[60264, 1495, 290, 8, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->310508122],
  Cell[60557, 1505, 444, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1843386608],
  Cell[61016, 1522, 219, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->178377548],
  Cell[61260, 1532, 398, 12, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->205282400],
  Cell[61661, 1546, 465, 15, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->52145387],
  Cell[73220, 1940, 167, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->125418266],
  Cell[73390, 1947, 217, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->78037682],
  Cell[73632, 1957, 719, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->103075800],
  Cell[74354, 1982, 424, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->593065773],
  Cell[74793, 1998, 175, 4, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->453777393],
  Cell[179794, 3758, 229, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->11406102],
  Cell[180048, 3768, 745, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->45005793],
  Cell[180818, 3795, 375, 10, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2029791019],
  Cell[181196, 3807, 362, 10, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1919923738],
  Cell[181561, 3819, 379, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->68984744],
  Cell[181943, 3832, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1932906675],
  Cell[182342, 3845, 379, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1377358556],
  Cell[182748, 3860, 213, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->100162883],
  Cell[182986, 3869, 951, 26, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->64152227],
  Cell[183962, 3899, 374, 10, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1539019679],
  Cell[184339, 3911, 400, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2010700485],
  Cell[184742, 3924, 397, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1040382203],
  Cell[185142, 3937, 378, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->891831947],
  Cell[185523, 3950, 394, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1731160816],
  Cell[191805, 4158, 178, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->142614438],
  Cell[193273, 4214, 267, 7, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->79261794],
  Cell[193543, 4223, 246, 7, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->103880104],
  Cell[193792, 4232, 1774, 51, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->245307687],
  Cell[195569, 4285, 236, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->302105471],
  Cell[195830, 4295, 302, 9, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->76487140],
  Cell[196135, 4306, 553, 15, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1434938566],
  Cell[196703, 4324, 412, 10, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->58027804],
  Cell[197140, 4338, 427, 12, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->103201209],
  Cell[197570, 4352, 550, 15, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->771122952],
  Cell[204000, 4554, 175, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->88013361],
  Cell[204178, 4561, 220, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->136327480],
  Cell[204423, 4571, 824, 27, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->17340756],
  Cell[205250, 4600, 373, 12, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1345933910],
  Cell[205638, 4615, 134, 2, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->350455679],
  Cell[205797, 4621, 1154, 39, 65, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->409422994],
  Cell[206954, 4662, 92469, 1520, 319, "Output",
   Evaluatable->False,
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->831528278],
  Cell[299438, 6185, 250, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->475197752],
  Cell[299713, 6195, 353, 11, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->311100400],
  Cell[300069, 6208, 420, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->819788825],
  Cell[300504, 6224, 243, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->13451302],
  Cell[300772, 6234, 461, 14, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->300776615],
  Cell[301236, 6250, 440, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2083186270],
  Cell[301691, 6267, 365, 9, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->273817704],
  Cell[302081, 6280, 831, 24, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->251756205],
  Cell[302915, 6306, 440, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1685742849],
  Cell[307291, 6444, 169, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->63134833],
  Cell[307463, 6451, 220, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->385877974],
  Cell[307708, 6461, 714, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->195334256],
  Cell[308425, 6486, 425, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1966772697],
  Cell[308865, 6502, 217, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->132522388],
  Cell[411604, 8225, 225, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->448066845],
  Cell[411854, 8235, 747, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->227542198],
  Cell[412626, 8262, 394, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->738196843],
  Cell[413023, 8275, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->861311582],
  Cell[413422, 8288, 397, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2023977894],
  Cell[413822, 8301, 395, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->855495978],
  Cell[414220, 8314, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->917885581],
  Cell[414643, 8329, 201, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->119506098],
  Cell[414869, 8339, 822, 24, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->4613958],
  Cell[415716, 8367, 397, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1527923466],
  Cell[416116, 8380, 394, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->200242236],
  Cell[416513, 8393, 392, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->364385603],
  Cell[416908, 8406, 395, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1752952024],
  Cell[417306, 8419, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->536514175],
  Cell[417729, 8434, 162, 4, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->107225079],
  Cell[417916, 8442, 1024, 30, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->61749050],
  Cell[418943, 8474, 463, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2057389119],
  Cell[419421, 8491, 183, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->34899127],
  Cell[419629, 8500, 1201, 39, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->659368774],
  Cell[420833, 8541, 1330, 38, 54, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1904722686],
  Cell[422178, 8582, 279, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->158328164],
  Cell[422482, 8592, 510, 14, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->376120086],
  Cell[422995, 8608, 1389, 40, 54, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->135457773]},
 "NMinimize"->{
  Cell[8730, 306, 564, 15, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->16616413],
  Cell[9297, 323, 349, 7, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->441818026],
  Cell[9649, 332, 346, 12, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->13823121],
  Cell[10020, 348, 370, 12, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->158556129],
  Cell[10393, 362, 425, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->147730805],
  Cell[10833, 378, 628, 22, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->8449423],
  Cell[11486, 404, 552, 18, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->479319689],
  Cell[12041, 424, 447, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->420543703],
  Cell[12503, 441, 1522, 43, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->55770639],
  Cell[14028, 486, 394, 14, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->278597936],
  Cell[14447, 504, 465, 15, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->105413790],
  Cell[14915, 521, 371, 12, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->218869012],
  Cell[15301, 536, 216, 7, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->869892494],
  Cell[15542, 547, 572, 19, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->434989722],
  Cell[16117, 568, 404, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->386809913],
  Cell[16536, 584, 1356, 36, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->408968279],
  Cell[17895, 622, 183, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->92233752],
  Cell[18103, 631, 286, 9, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->328088122],
  Cell[18392, 642, 309, 10, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->71577405],
  Cell[18716, 655, 146, 2, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->588472900],
  Cell[18887, 661, 334, 10, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->160567602],
  Cell[19224, 673, 310, 10, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->285978531],
  Cell[19549, 686, 738, 22, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->542356352],
  Cell[20312, 712, 528, 17, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->164914745],
  Cell[20843, 731, 415, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->393800755],
  Cell[21273, 747, 586, 22, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->110392765],
  Cell[21884, 773, 372, 11, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->841906160],
  Cell[22259, 786, 330, 11, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->709334044],
  Cell[22604, 800, 211, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->262757712],
  Cell[22840, 810, 488, 15, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->891110009],
  Cell[23331, 827, 349, 11, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->434666499],
  Cell[23695, 841, 1240, 34, 70, "Text",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->21544639],
  Cell[24938, 877, 151, 3, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->646694894],
  Cell[25114, 884, 1357, 41, 77, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->167316588],
  Cell[26474, 927, 33567, 555, 162, "Output",
   Evaluatable->False,
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->26605553],
  Cell[60056, 1485, 183, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->583164225],
  Cell[60264, 1495, 290, 8, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->310508122],
  Cell[60557, 1505, 444, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1843386608],
  Cell[61016, 1522, 219, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->178377548],
  Cell[61260, 1532, 398, 12, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->205282400],
  Cell[61661, 1546, 465, 15, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->52145387],
  Cell[73220, 1940, 167, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->125418266],
  Cell[73390, 1947, 217, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->78037682],
  Cell[73632, 1957, 719, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->103075800],
  Cell[74354, 1982, 424, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->593065773],
  Cell[74793, 1998, 175, 4, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->453777393],
  Cell[179794, 3758, 229, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->11406102],
  Cell[180048, 3768, 745, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->45005793],
  Cell[180818, 3795, 375, 10, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2029791019],
  Cell[181196, 3807, 362, 10, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1919923738],
  Cell[181561, 3819, 379, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->68984744],
  Cell[181943, 3832, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1932906675],
  Cell[182342, 3845, 379, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1377358556],
  Cell[182748, 3860, 213, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->100162883],
  Cell[182986, 3869, 951, 26, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->64152227],
  Cell[183962, 3899, 374, 10, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1539019679],
  Cell[184339, 3911, 400, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2010700485],
  Cell[184742, 3924, 397, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1040382203],
  Cell[185142, 3937, 378, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->891831947],
  Cell[185523, 3950, 394, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1731160816],
  Cell[191805, 4158, 178, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->142614438],
  Cell[193273, 4214, 267, 7, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->79261794],
  Cell[193543, 4223, 246, 7, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->103880104],
  Cell[193792, 4232, 1774, 51, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->245307687],
  Cell[195569, 4285, 236, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->302105471],
  Cell[195830, 4295, 302, 9, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->76487140],
  Cell[196135, 4306, 553, 15, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1434938566],
  Cell[196703, 4324, 412, 10, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->58027804],
  Cell[197140, 4338, 427, 12, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->103201209],
  Cell[197570, 4352, 550, 15, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->771122952],
  Cell[204000, 4554, 175, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->88013361],
  Cell[204178, 4561, 220, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->136327480],
  Cell[204423, 4571, 824, 27, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->17340756],
  Cell[205250, 4600, 373, 12, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1345933910],
  Cell[205638, 4615, 134, 2, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->350455679],
  Cell[205797, 4621, 1154, 39, 65, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->409422994],
  Cell[206954, 4662, 92469, 1520, 319, "Output",
   Evaluatable->False,
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->831528278],
  Cell[299438, 6185, 250, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->475197752],
  Cell[299713, 6195, 353, 11, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->311100400],
  Cell[300069, 6208, 420, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->819788825],
  Cell[300504, 6224, 243, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->13451302],
  Cell[300772, 6234, 461, 14, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->300776615],
  Cell[301236, 6250, 440, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2083186270],
  Cell[301691, 6267, 365, 9, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->273817704],
  Cell[302081, 6280, 831, 24, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->251756205],
  Cell[302915, 6306, 440, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1685742849],
  Cell[307291, 6444, 169, 5, 70, "Caption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->63134833],
  Cell[307463, 6451, 220, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->385877974],
  Cell[307708, 6461, 714, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->195334256],
  Cell[308425, 6486, 425, 13, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1966772697],
  Cell[308865, 6502, 217, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->132522388],
  Cell[411604, 8225, 225, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->448066845],
  Cell[411854, 8235, 747, 23, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->227542198],
  Cell[412626, 8262, 394, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->738196843],
  Cell[413023, 8275, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->861311582],
  Cell[413422, 8288, 397, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2023977894],
  Cell[413822, 8301, 395, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->855495978],
  Cell[414220, 8314, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->917885581],
  Cell[414643, 8329, 201, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->119506098],
  Cell[414869, 8339, 822, 24, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->4613958],
  Cell[415716, 8367, 397, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1527923466],
  Cell[416116, 8380, 394, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->200242236],
  Cell[416513, 8393, 392, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->364385603],
  Cell[416908, 8406, 395, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1752952024],
  Cell[417306, 8419, 396, 11, 70, "Print",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->536514175],
  Cell[417729, 8434, 162, 4, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->107225079],
  Cell[417916, 8442, 1024, 30, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->61749050],
  Cell[418943, 8474, 463, 14, 36, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->2057389119],
  Cell[419421, 8491, 183, 5, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->34899127],
  Cell[419629, 8500, 1201, 39, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->659368774],
  Cell[420833, 8541, 1330, 38, 54, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->1904722686],
  Cell[422178, 8582, 279, 6, 70, "MathCaption",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->158328164],
  Cell[422482, 8592, 510, 14, 70, "Input",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->376120086],
  Cell[422995, 8608, 1389, 40, 54, "Output",
   CellTags->{"NMinimize:Contents", "NMinimize"},
   CellID->135457773]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"NMinimize:Contents", 428094, 8740},
 {"NMinimize", 441488, 9085}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1222, 27, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1847, 54, 75, 1, 70, "Title",
 CellID->85183321],
Cell[CellGroupData[{
Cell[1947, 59, 51, 1, 70, "Section",
 CellID->469184158],
Cell[2001, 62, 865, 19, 70, "Text",
 CellID->289642637],
Cell[2869, 83, 786, 22, 70, "Text",
 CellID->65223348],
Cell[3658, 107, 783, 25, 70, "Text",
 CellID->166790006],
Cell[4444, 134, 80, 1, 70, "Text",
 CellID->303464556],
Cell[4527, 137, 425, 13, 70, "DisplayMath",
 CellID->401358986],
Cell[4955, 152, 292, 12, 70, "Text",
 CellID->73215332],
Cell[CellGroupData[{
Cell[5272, 168, 496, 17, 30, "Input",
 CellID->213704],
Cell[5771, 187, 336, 12, 36, "Output",
 CellID->423974823]
}, Open  ]],
Cell[6122, 202, 303, 10, 70, "MathCaption",
 CellID->109220874],
Cell[CellGroupData[{
Cell[6450, 216, 499, 17, 70, "Input",
 CellID->38568035],
Cell[6952, 235, 397, 13, 39, "Output",
 CellID->1586759045]
}, Open  ]],
Cell[7364, 251, 328, 10, 70, "MathCaption",
 CellID->583307620],
Cell[CellGroupData[{
Cell[7717, 265, 500, 17, 70, "Input",
 CellID->7703726],
Cell[8220, 284, 397, 13, 39, "Output",
 CellID->2132335936]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8666, 303, 61, 1, 70, "Section",
 CellID->217856200],
Cell[8730, 306, 564, 15, 70, "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->16616413],
Cell[9297, 323, 349, 7, 70, "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->441818026],
Cell[9649, 332, 346, 12, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->13823121],
Cell[CellGroupData[{
Cell[10020, 348, 370, 12, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->158556129],
Cell[10393, 362, 425, 13, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->147730805]
}, Open  ]],
Cell[10833, 378, 628, 22, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->8449423],
Cell[CellGroupData[{
Cell[11486, 404, 552, 18, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->479319689],
Cell[12041, 424, 447, 14, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->420543703]
}, Open  ]],
Cell[12503, 441, 1522, 43, 70, "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->55770639],
Cell[14028, 486, 394, 14, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->278597936],
Cell[CellGroupData[{
Cell[14447, 504, 465, 15, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->105413790],
Cell[14915, 521, 371, 12, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->218869012]
}, Open  ]],
Cell[15301, 536, 216, 7, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->869892494],
Cell[CellGroupData[{
Cell[15542, 547, 572, 19, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->434989722],
Cell[16117, 568, 404, 13, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->386809913]
}, Open  ]],
Cell[16536, 584, 1356, 36, 70, "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->408968279],
Cell[17895, 622, 183, 5, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->92233752],
Cell[CellGroupData[{
Cell[18103, 631, 286, 9, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->328088122],
Cell[18392, 642, 309, 10, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->71577405]
}, Open  ]],
Cell[18716, 655, 146, 2, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->588472900],
Cell[CellGroupData[{
Cell[18887, 661, 334, 10, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->160567602],
Cell[19224, 673, 310, 10, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->285978531]
}, Open  ]],
Cell[19549, 686, 738, 22, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->542356352],
Cell[CellGroupData[{
Cell[20312, 712, 528, 17, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->164914745],
Cell[20843, 731, 415, 13, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->393800755]
}, Open  ]],
Cell[21273, 747, 586, 22, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->110392765],
Cell[CellGroupData[{
Cell[21884, 773, 372, 11, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->841906160],
Cell[22259, 786, 330, 11, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->709334044]
}, Open  ]],
Cell[22604, 800, 211, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->262757712],
Cell[CellGroupData[{
Cell[22840, 810, 488, 15, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->891110009],
Cell[23331, 827, 349, 11, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->434666499]
}, Open  ]],
Cell[23695, 841, 1240, 34, 70, "Text",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->21544639],
Cell[24938, 877, 151, 3, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->646694894],
Cell[CellGroupData[{
Cell[25114, 884, 1357, 41, 77, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->167316588],
Cell[26474, 927, 33567, 555, 162, "Output",
 Evaluatable->False,
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->26605553]
}, Open  ]],
Cell[60056, 1485, 183, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->583164225],
Cell[CellGroupData[{
Cell[60264, 1495, 290, 8, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->310508122],
Cell[60557, 1505, 444, 14, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1843386608]
}, Open  ]],
Cell[61016, 1522, 219, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->178377548],
Cell[CellGroupData[{
Cell[61260, 1532, 398, 12, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->205282400],
Cell[61661, 1546, 465, 15, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->52145387]
}, Open  ]],
Cell[62141, 1564, 726, 18, 70, "Text",
 CellID->413147826],
Cell[62870, 1584, 308, 9, 70, "Text",
 CellID->199644619]
}, Open  ]],
Cell[CellGroupData[{
Cell[63215, 1598, 95, 1, 70, "Section",
 CellID->252245038],
Cell[CellGroupData[{
Cell[63335, 1603, 59, 1, 70, "Subsection",
 CellID->323477034],
Cell[63397, 1606, 620, 17, 70, "Text",
 CellID->202391529],
Cell[64020, 1625, 1011, 34, 70, "Text",
 CellID->829209197],
Cell[65034, 1661, 1120, 40, 70, "Text",
 CellID->500686794],
Cell[66157, 1703, 724, 26, 70, "Text",
 CellID->324409526],
Cell[66884, 1731, 1580, 56, 70, "Text",
 CellID->55919927],
Cell[68467, 1789, 520, 16, 70, "Text",
 CellID->519062731],
Cell[68990, 1807, 1081, 35, 70, "Text",
 CellID->31377205],
Cell[70074, 1844, 851, 31, 70, "Text",
 CellID->281027891],
Cell[70928, 1877, 566, 16, 70, "Text",
 CellID->455205358],
Cell[71497, 1895, 236, 5, 70, "Text",
 CellID->682966497],
Cell[71736, 1902, 1481, 36, 70, "DefinitionBox3Col",
 CellID->555161960],
Cell[73220, 1940, 167, 5, 70, "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->125418266],
Cell[73390, 1947, 217, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->78037682],
Cell[CellGroupData[{
Cell[73632, 1957, 719, 23, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->103075800],
Cell[74354, 1982, 424, 13, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->593065773]
}, Open  ]],
Cell[74793, 1998, 175, 4, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->453777393],
Cell[CellGroupData[{
Cell[74993, 2006, 1707, 53, 83, "Input",
 CellID->75686477],
Cell[76703, 2061, 103076, 1694, 309, "Output",
 Evaluatable->False,
 CellID->174171999]
}, Open  ]],
Cell[179794, 3758, 229, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->11406102],
Cell[CellGroupData[{
Cell[180048, 3768, 745, 23, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->45005793],
Cell[CellGroupData[{
Cell[180818, 3795, 375, 10, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->2029791019],
Cell[181196, 3807, 362, 10, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1919923738],
Cell[181561, 3819, 379, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->68984744],
Cell[181943, 3832, 396, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1932906675],
Cell[182342, 3845, 379, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1377358556]
}, Open  ]]
}, Open  ]],
Cell[182748, 3860, 213, 5, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->100162883],
Cell[CellGroupData[{
Cell[182986, 3869, 951, 26, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->64152227],
Cell[CellGroupData[{
Cell[183962, 3899, 374, 10, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1539019679],
Cell[184339, 3911, 400, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->2010700485],
Cell[184742, 3924, 397, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1040382203],
Cell[185142, 3937, 378, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->891831947],
Cell[185523, 3950, 394, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1731160816]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[185978, 3968, 63, 1, 70, "Subsection",
 CellID->24713453],
Cell[186044, 3971, 108, 3, 70, "Text",
 CellID->15517963],
Cell[186155, 3976, 688, 23, 70, "Text",
 CellID->400968379],
Cell[186846, 4001, 2491, 85, 70, "Text",
 CellID->134958075],
Cell[189340, 4088, 569, 16, 70, "Text",
 CellID->666318363],
Cell[189912, 4106, 203, 5, 70, "Text",
 CellID->373195045],
Cell[190118, 4113, 1684, 43, 70, "DefinitionBox3Col",
 CellID->18427541],
Cell[191805, 4158, 178, 5, 70, "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->142614438],
Cell[191986, 4165, 180, 5, 70, "MathCaption",
 CellID->91972744],
Cell[CellGroupData[{
Cell[192191, 4174, 686, 23, 70, "Input",
 CellID->676994573],
Cell[192880, 4199, 378, 12, 36, "Output",
 CellID->1832991483]
}, Open  ]],
Cell[193273, 4214, 267, 7, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->79261794],
Cell[193543, 4223, 246, 7, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->103880104],
Cell[193792, 4232, 1774, 51, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->245307687],
Cell[195569, 4285, 236, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->302105471],
Cell[CellGroupData[{
Cell[195830, 4295, 302, 9, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->76487140],
Cell[196135, 4306, 553, 15, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1434938566]
}, Open  ]],
Cell[196703, 4324, 412, 10, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->58027804],
Cell[CellGroupData[{
Cell[197140, 4338, 427, 12, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->103201209],
Cell[197570, 4352, 550, 15, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->771122952]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[198169, 4373, 59, 1, 70, "Subsection",
 CellID->7062708],
Cell[198231, 4376, 534, 9, 70, "Text",
 CellID->51945444],
Cell[198768, 4387, 509, 15, 70, "Text",
 CellID->9567416],
Cell[199280, 4404, 1953, 67, 70, "Text",
 CellID->84070994],
Cell[201236, 4473, 265, 8, 70, "Text",
 CellID->234191333],
Cell[201504, 4483, 450, 16, 70, "Text",
 CellID->2029629094],
Cell[201957, 4501, 330, 7, 70, "Text",
 CellID->1863692427],
Cell[202290, 4510, 1707, 42, 70, "DefinitionBox3Col",
 CellID->459034107],
Cell[204000, 4554, 175, 5, 70, "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->88013361],
Cell[204178, 4561, 220, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->136327480],
Cell[CellGroupData[{
Cell[204423, 4571, 824, 27, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->17340756],
Cell[205250, 4600, 373, 12, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1345933910]
}, Open  ]],
Cell[205638, 4615, 134, 2, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->350455679],
Cell[CellGroupData[{
Cell[205797, 4621, 1154, 39, 65, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->409422994],
Cell[206954, 4662, 92469, 1520, 319, "Output",
 Evaluatable->False,
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->831528278]
}, Open  ]],
Cell[299438, 6185, 250, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->475197752],
Cell[CellGroupData[{
Cell[299713, 6195, 353, 11, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->311100400],
Cell[300069, 6208, 420, 13, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->819788825]
}, Open  ]],
Cell[300504, 6224, 243, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->13451302],
Cell[CellGroupData[{
Cell[300772, 6234, 461, 14, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->300776615],
Cell[301236, 6250, 440, 14, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->2083186270]
}, Open  ]],
Cell[301691, 6267, 365, 9, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->273817704],
Cell[CellGroupData[{
Cell[302081, 6280, 831, 24, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->251756205],
Cell[302915, 6306, 440, 14, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1685742849]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[303404, 6326, 55, 1, 70, "Subsection",
 CellID->155930577],
Cell[303462, 6329, 291, 6, 70, "Text",
 CellID->417700783],
Cell[303756, 6337, 917, 28, 70, "Text",
 CellID->1810634708],
Cell[304676, 6367, 415, 15, 70, "Text",
 CellID->1494259366],
Cell[305094, 6384, 196, 5, 70, "Text",
 CellID->1343396344],
Cell[305293, 6391, 486, 11, 70, "Text",
 CellID->114943451],
Cell[305782, 6404, 1506, 38, 70, "DefinitionBox3Col",
 CellID->13679150],
Cell[307291, 6444, 169, 5, 70, "Caption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->63134833],
Cell[307463, 6451, 220, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->385877974],
Cell[CellGroupData[{
Cell[307708, 6461, 714, 23, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->195334256],
Cell[308425, 6486, 425, 13, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1966772697]
}, Open  ]],
Cell[308865, 6502, 217, 5, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->132522388],
Cell[CellGroupData[{
Cell[309107, 6511, 1778, 54, 101, "Input",
 CellID->282281265],
Cell[310888, 6567, 100701, 1655, 309, "Output",
 Evaluatable->False,
 CellID->126164948]
}, Open  ]],
Cell[411604, 8225, 225, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->448066845],
Cell[CellGroupData[{
Cell[411854, 8235, 747, 23, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->227542198],
Cell[CellGroupData[{
Cell[412626, 8262, 394, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->738196843],
Cell[413023, 8275, 396, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->861311582],
Cell[413422, 8288, 397, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->2023977894],
Cell[413822, 8301, 395, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->855495978],
Cell[414220, 8314, 396, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->917885581]
}, Open  ]]
}, Open  ]],
Cell[414643, 8329, 201, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->119506098],
Cell[CellGroupData[{
Cell[414869, 8339, 822, 24, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->4613958],
Cell[CellGroupData[{
Cell[415716, 8367, 397, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1527923466],
Cell[416116, 8380, 394, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->200242236],
Cell[416513, 8393, 392, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->364385603],
Cell[416908, 8406, 395, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1752952024],
Cell[417306, 8419, 396, 11, 70, "Print",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->536514175]
}, Open  ]]
}, Open  ]],
Cell[417729, 8434, 162, 4, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->107225079],
Cell[CellGroupData[{
Cell[417916, 8442, 1024, 30, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->61749050],
Cell[418943, 8474, 463, 14, 36, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->2057389119]
}, Open  ]],
Cell[419421, 8491, 183, 5, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->34899127],
Cell[CellGroupData[{
Cell[419629, 8500, 1201, 39, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->659368774],
Cell[420833, 8541, 1330, 38, 54, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->1904722686]
}, Open  ]],
Cell[422178, 8582, 279, 6, 70, "MathCaption",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->158328164],
Cell[CellGroupData[{
Cell[422482, 8592, 510, 14, 70, "Input",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->376120086],
Cell[422995, 8608, 1389, 40, 54, "Output",
 CellTags->{"NMinimize:Contents", "NMinimize"},
 CellID->135457773]
}, Open  ]],
Cell[CellGroupData[{
Cell[424421, 8653, 72, 1, 70, "RelatedTutorialsSection",
 CellID->256102893],
Cell[424496, 8656, 178, 4, 70, "RelatedTutorials",
 CellID->1633042],
Cell[424677, 8662, 185, 5, 70, "RelatedTutorials",
 CellID->282186215],
Cell[424865, 8669, 202, 5, 70, "RelatedTutorials",
 CellID->511088424],
Cell[425070, 8676, 178, 4, 70, "RelatedTutorials",
 CellID->140984286]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[425299, 8686, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

