(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     62633,       2222]
NotebookOptionsPosition[     43356,       1531]
NotebookOutlinePosition[     54352,       1890]
CellTagsIndexPosition[     54259,       1884]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Lie Symmetry Methods for Solving Nonlinear ODEs", "Title",
 CellTags->{"c:35", "b:1.3"},
 CellID->7278],

Cell[TextData[{
 "Around 1870, Marius Sophus Lie realized that many of the methods for \
solving differential equations could be unified using group theory. Lie \
symmetry methods are central to the modern approach for studying nonlinear \
ODEs. They use the notion of symmetry to generate solutions in a systematic \
manner. Here is a brief introduction to Lie\[CloseCurlyQuote]s approach that \
provides some examples that are solved in this way by ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ". "
}], "Text",
 CellTags->"b:1.3",
 CellID->20799],

Cell[TextData[{
 "A key notion in Lie\[CloseCurlyQuote]s method is that of an ",
 StyleBox["infinitesimal generator",
  FontSlant->"Italic"],
 " for a symmetry group. This concept is illustrated in the following \
example. "
}], "Text",
 CellTags->"b:1.3",
 CellID->18352],

Cell[TextData[{
 "Here is the well-known transformation for rotations in the ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 "-",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " plane. This is a one-parameter group of transformations with parameter ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->20448],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{
    RowBox[{"x", "*", 
     RowBox[{"Cos", "[", "t", "]"}]}], " ", "+", 
    RowBox[{"y", "*", 
     RowBox[{"Sin", "[", "t", "]"}]}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[1]:=",
 CellID->13882],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "x"}], "*", 
     RowBox[{"Sin", "[", "t", "]"}]}], " ", "+", " ", 
    RowBox[{"y", "*", 
     RowBox[{"Cos", "[", "t", "]"}]}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[2]:=",
 CellID->540],

Cell[TextData[{
 "For a fixed value of ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ", the point ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"m", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " (in blue) can be obtained by rotating the line joining ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"x", ",", "y"}], ")"}], TraditionalForm]], "InlineMath"],
 " (in red) to the origin through an angle of ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " in the counterclockwise direction."
}], "Text",
 CellTags->"b:1.3",
 CellID->28315],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Graphics", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"Red", ",", " ", 
          RowBox[{"PointSize", "[", "0.04", "]"}], ",", 
          RowBox[{"Point", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"3", "*", 
              RowBox[{"Cos", "[", 
               RowBox[{"1", "/", "4"}], "]"}]}], ",", " ", 
             RowBox[{"3", "*", 
              RowBox[{"Sin", "[", 
               RowBox[{"1", "/", "4"}], "]"}]}]}], "}"}], "]"}]}], "}"}], ",",
         
        RowBox[{"{", 
         RowBox[{"Blue", ",", " ", 
          RowBox[{"PointSize", "[", "0.04", "]"}], ",", 
          RowBox[{"Point", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"3", "*", 
              RowBox[{"Cos", "[", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"1", "/", "4"}], ")"}], "+", 
                RowBox[{"(", 
                 RowBox[{"Pi", "/", "3"}], ")"}]}], "]"}]}], ",", " ", 
             RowBox[{"3", "*", 
              RowBox[{"Sin", "[", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"1", "/", "4"}], ")"}], "+", 
                RowBox[{"(", 
                 RowBox[{"Pi", "/", "3"}], ")"}]}], "]"}]}]}], "}"}], "]"}]}],
          "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Green", ",", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{"0", ",", "0"}], "}"}], ",", " ", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"3", "*", 
                RowBox[{"Cos", "[", 
                 RowBox[{"1", "/", "4"}], "]"}]}], ",", " ", 
               RowBox[{"3", "*", 
                RowBox[{"Sin", "[", 
                 RowBox[{"1", "/", "4"}], "]"}]}]}], "}"}]}], "}"}], "]"}]}], 
         "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"Green", ",", 
          RowBox[{"Line", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{"0", ",", "0"}], "}"}], ",", " ", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"3", "*", 
                RowBox[{"Cos", "[", 
                 RowBox[{
                  RowBox[{"(", 
                   RowBox[{"1", "/", "4"}], ")"}], "+", 
                  RowBox[{"(", 
                   RowBox[{"Pi", "/", "3"}], ")"}]}], "]"}]}], ",", " ", 
               RowBox[{"3", "*", 
                RowBox[{"Sin", "[", 
                 RowBox[{
                  RowBox[{"(", 
                   RowBox[{"1", "/", "4"}], ")"}], "+", 
                  RowBox[{"(", 
                   RowBox[{"Pi", "/", "3"}], ")"}]}], "]"}]}]}], "}"}]}], 
            "}"}], "]"}]}], "}"}]}], "}"}], "]"}], ",", 
     RowBox[{"ParametricPlot", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"3", "*", 
          RowBox[{"Cos", "[", "t", "]"}]}], ",", " ", 
         RowBox[{"3", "*", 
          RowBox[{"Sin", "[", "t", "]"}]}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"t", ",", " ", "0", ",", 
         RowBox[{"2", "Pi"}]}], "}"}], ",", " ", 
       RowBox[{"DisplayFunction", " ", "\[Rule]", " ", "Identity"}]}], " ", 
      "]"}]}], "}"}], ",", 
   RowBox[{"AspectRatio", " ", "\[Rule]", "1"}], ",", " ", 
   RowBox[{"ImageSize", " ", "\[Rule]", " ", "200"}], ",", " ", 
   RowBox[{"Axes", " ", "\[Rule]", " ", "True"}]}], "]"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[3]:=",
 CellID->4575],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{200, 202},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[3]=",
 CellID->378886835]
}, Open  ]],

Cell[TextData[{
 "A rotation through ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " ",
 "can be represented by the matrix"
}], "Text",
 CellTags->"b:1.3",
 CellID->9567],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["M", "t"], "=", 
   RowBox[{
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        RowBox[{"cos", "(", "t", ")"}], 
        RowBox[{"sin", "(", "t", ")"}]},
       {
        RowBox[{"-", 
         RowBox[{"sin", "(", "t", ")"}]}], 
        RowBox[{"cos", "(", "t", ")"}]}
      },
      GridBoxAlignment->{
       "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
        "RowsIndexed" -> {}},
      GridBoxSpacings->{
       "Columns" -> {0.28, {0.7}, 0.28}, "ColumnsIndexed" -> {}, 
        "Rows" -> {0.2, {0.4}, 0.2}, "RowsIndexed" -> {}}], "\[NoBreak]", 
     ")"}], "."}]}], TraditionalForm]], "DisplayMath",
 CellTags->"b:1.3",
 CellID->22336],

Cell["\<\
This shows that the set of all rotations in the plane satisfies the \
properties for forming a group.\
\>", "MathCaption",
 CellTags->"b:1.3",
 CellID->18397],

Cell[BoxData[
 RowBox[{
  RowBox[{"M", "[", "t_", "]"}], ":=", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Cos", "[", "t", "]"}], ",", 
      RowBox[{"Sin", "[", "t", "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", 
       RowBox[{"Sin", "[", "t", "]"}]}], ",", 
      RowBox[{"Cos", "[", "t", "]"}]}], "}"}]}], "}"}]}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[4]:=",
 CellID->31418],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"M", "[", 
    RowBox[{"a", "+", "b"}], "]"}], "==", 
   RowBox[{
    RowBox[{"M", "[", "a", "]"}], ".", 
    RowBox[{"M", "[", "b", "]"}]}]}], "]"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[5]:=",
 CellID->19897],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[5]=",
 CellID->569618775]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"M", "[", "a", "]"}], ".", 
   RowBox[{"M", "[", "0", "]"}]}], "==", 
  RowBox[{
   RowBox[{"M", "[", "0", "]"}], ".", 
   RowBox[{"M", "[", "a", "]"}]}], "==", 
  RowBox[{"M", "[", "a", "]"}]}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[6]:=",
 CellID->13851],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[6]=",
 CellID->14417335]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"M", "[", "0", "]"}], "==", 
  RowBox[{"IdentityMatrix", "[", "2", "]"}]}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[7]:=",
 CellID->24230],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[7]=",
 CellID->166297553]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"M", "[", "a", "]"}], ".", 
    RowBox[{"M", "[", 
     RowBox[{"-", "a"}], "]"}]}], "==", 
   RowBox[{
    RowBox[{"M", "[", 
     RowBox[{"-", "a"}], "]"}], ".", 
    RowBox[{"M", "[", "a", "]"}]}], "==", 
   RowBox[{"IdentityMatrix", "[", "2", "]"}]}], "//", "Simplify"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[8]:=",
 CellID->16925],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[8]=",
 CellID->30753395]
}, Open  ]],

Cell[TextData[{
 "The Lie symmetry method requires calculating a first-order approximation to \
the expressions for the group. This approximation is called an ",
 StyleBox["infinitesimal generator",
  FontSlant->"Italic"],
 "."
}], "Text",
 CellTags->"b:1.3",
 CellID->229],

Cell[TextData[{
 "This expands the expressions for ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " in a series with respect to ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " and around the origin 0 to obtain linear approximations."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->24520],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Series", "[", 
   RowBox[{"m", " ", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", " ", "0", ",", " ", "1"}], "}"}]}], "]"}], "//", 
  "Normal"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[9]:=",
 CellID->10940],

Cell[BoxData[
 RowBox[{"x", "+", 
  RowBox[{"t", " ", "y"}]}]], "Output",
 ImageSize->{41, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[9]=",
 CellID->375578462]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Series", "[", 
   RowBox[{"n", " ", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", " ", "0", ",", " ", "1"}], "}"}]}], "]"}], "//", 
  "Normal"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[10]:=",
 CellID->15147],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "t"}], " ", "x"}], "+", "y"}]], "Output",
 ImageSize->{49, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[10]=",
 CellID->269638730]
}, Open  ]],

Cell[TextData[{
 "The coefficients of the linear terms in ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " are ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "x"}], TraditionalForm]], "InlineMath"],
 ",",
 " respectively. The infinitesimal generator for the rotation group in the \
plane is defined to be the following differential operator."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->1655],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", " ", "=", " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"y", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "x"}], "]"}]}], " ", "-", " ", 
      RowBox[{"x", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "y"}], "]"}]}]}], ")"}], "&"}]}], 
  ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[11]:=",
 CellID->20675],

Cell[TextData[{
 "Starting from the infinitesimal generator, the original group can be \
recovered by integrating the fundamental system of ",
 StyleBox["Lie equations. ",
  FontSlant->"Italic"],
 StyleBox["For the group of rotations, the Lie equations are given by the",
  FontVariations->{"CompatibilityType"->0}],
 " ",
 StyleBox["first argument to ",
  FontVariations->{"CompatibilityType"->0}],
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 StyleBox[" shown here.",
  FontVariations->{"CompatibilityType"->0}]
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->792],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
      RowBox[{"y", "[", "t", "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", 
      RowBox[{"-", 
       RowBox[{"x", "[", "t", "]"}]}]}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], ",", 
     RowBox[{"y", "[", "t", "]"}]}], "}"}], ",", "t"}], "]"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[12]:=",
 CellID->22361],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], " ", 
       RowBox[{"Cos", "[", "t", "]"}]}], "+", 
      RowBox[{
       RowBox[{"C", "[", "2", "]"}], " ", 
       RowBox[{"Sin", "[", "t", "]"}]}]}]}], ",", 
    RowBox[{
     RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{
       RowBox[{"C", "[", "2", "]"}], " ", 
       RowBox[{"Cos", "[", "t", "]"}]}], "-", 
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], " ", 
       RowBox[{"Sin", "[", "t", "]"}]}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{449, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[12]=",
 CellID->13456374]
}, Open  ]],

Cell[TextData[{
 "The rotation group arises in the study of symmetries of geometrical \
objects; it is an example of a ",
 StyleBox["symmetry group",
  FontSlant->"Italic"],
 ". The infinitesimal generator, a differential operator, is a convenient \
local representation for this symmetric group, which is a set of matrices."
}], "Text",
 CellTags->"b:1.3",
 CellID->16398],

Cell[TextData[{
 "An expression that reduces to 0 under the action of the infinitesimal \
generator is called an",
 StyleBox[" invariant ",
  FontSlant->"Italic"],
 "of the group. "
}], "Text",
 CellTags->"b:1.3",
 CellID->23146],

Cell["Here is an invariant for this group.", "MathCaption",
 CellTags->"b:1.3",
 CellID->8714],

Cell[BoxData[
 RowBox[{
  RowBox[{"invariant", " ", "=", " ", 
   RowBox[{
    RowBox[{"x", "^", "2"}], " ", "+", 
    RowBox[{"y", "^", "2"}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[13]:=",
 CellID->26946],

Cell[TextData[{
 "This states that the distance from the origin to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"x", ",", "y"}], ")"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SqrtBox[
    RowBox[{
     SuperscriptBox["x", "2"], "+", 
     SuperscriptBox["y", "2"]}]], TraditionalForm]], "InlineMath"],
 ", is preserved under rotation."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->21188],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"v", "[", "invariant", "]"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[14]:=",
 CellID->19569],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[14]=",
 CellID->150931950]
}, Open  ]],

Cell["\<\
In the following examples, these ideas are applied to differential equations.\
\
\>", "Text",
 CellTags->"b:1.3",
 CellID->27663],

Cell[TextData[{
 "This is an example of a Riccati equation, from page 103 of [",
 ButtonBox["I99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveReferences#17537"],
 "]."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->15075],

Cell[BoxData[
 RowBox[{
  RowBox[{"Riccatiequation", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "'"}], "[", "x", "]"}], " ", "+", 
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "^", "2"}], "-", " ", 
     RowBox[{"1", "/", 
      RowBox[{"x", "^", "2"}]}]}], "\[Equal]", "0"}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[15]:=",
 CellID->24515],

Cell[TextData[{
 "The equation is invariant under the following ",
 StyleBox["scaling ",
  FontSlant->"Italic"],
 "transformation."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->11521],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", " ", "=", " ", 
   RowBox[{"x", "*", 
    RowBox[{"E", "^", "t"}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[16]:=",
 CellID->475],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", " ", "=", 
   RowBox[{"y", "*", 
    RowBox[{"E", "^", 
     RowBox[{"(", 
      RowBox[{"-", "t"}], ")"}]}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[17]:=",
 CellID->15615],

Cell["\<\
The infinitesimal generator for this one-parameter group of transformations \
is found as before.\
\>", "MathCaption",
 CellTags->"b:1.3",
 CellID->20528],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Series", "[", 
   RowBox[{"m", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", " ", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "Normal"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[18]:=",
 CellID->13234],

Cell[BoxData[
 RowBox[{"x", "+", 
  RowBox[{"t", " ", "x"}]}]], "Output",
 ImageSize->{41, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[18]=",
 CellID->180211428]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Series", "[", 
   RowBox[{"n", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", " ", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "Normal"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[19]:=",
 CellID->905],

Cell[BoxData[
 RowBox[{"y", "-", 
  RowBox[{"t", " ", "y"}]}]], "Output",
 ImageSize->{41, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[19]=",
 CellID->256074322]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", "=", " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "x"}], "]"}]}], " ", "-", 
      RowBox[{"y", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "y"}], "]"}]}]}], ")"}], "&"}]}], 
  ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[20]:=",
 CellID->19557],

Cell[TextData[{
 "Now, the Riccati equation depends on three variables: ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "=", 
    RowBox[{"y", "[", "x", "]"}]}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "=", 
    RowBox[{
     SuperscriptBox["y", "\[Prime]",
      MultilineFunction->None], "[", "x", "]"}]}], TraditionalForm]], 
  "InlineMath"],
 ". Hence, the infinitesimal generator ",
 Cell[BoxData[
  FormBox["v", TraditionalForm]], "InlineMath"],
 " must be ",
 StyleBox["prolonged",
  FontSlant->"Italic"],
 " to act on all three variables in this first-order equation."
}], "Text",
 CellTags->"b:1.3",
 CellID->28962],

Cell[TextData[{
 "It turns out that the required ",
 StyleBox["prolongation",
  FontSlant->"Italic"],
 " is as follows."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->14161],

Cell[BoxData[
 RowBox[{
  RowBox[{"prolongedv", " ", "=", " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "x"}], "]"}]}], " ", "-", 
      RowBox[{"y", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "y"}], "]"}]}], " ", "-", 
      RowBox[{"2", "p", "*", 
       RowBox[{"D", "[", 
        RowBox[{"#", ",", " ", "p"}], "]"}]}]}], " ", ")"}], "&"}]}], 
  ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[21]:=",
 CellID->30524],

Cell[TextData[{
 "This shows that the expression for the Riccati equation in the ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"x", ",", "y", ",", "p"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " coordinates is indeed invariant under ",
 Cell[BoxData["prolongedv"], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->12549],

Cell[BoxData[
 RowBox[{
  RowBox[{"Riccatiexpression", " ", "=", 
   RowBox[{"p", "+", 
    RowBox[{"y", "^", "2"}], "-", 
    RowBox[{"(", 
     RowBox[{"1", "/", 
      RowBox[{"x", "^", "2"}]}], ")"}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[22]:=",
 CellID->17469],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"prolongedv", "[", "Riccatiexpression", "]"}], "/.", 
   RowBox[{"{", 
    RowBox[{"p", "\[Rule]", " ", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "/", 
        RowBox[{"x", "^", "2"}]}], ")"}], "-", 
      RowBox[{"y", "^", "2"}]}]}], "}"}]}], "//", "Together"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[23]:=",
 CellID->330],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[23]=",
 CellID->230072638]
}, Open  ]],

Cell["\<\
Depending on the order of the given equation, the knowledge of a symmetry (in \
the form of an infinitesimal generator) can be used in three ways.\
\>", "Text",
 CellTags->"b:1.3",
 CellID->28350],

Cell[TextData[{
 "If the order of the equation is 1, it gives an integrating factor for the \
ODE that makes the equation ",
 StyleBox["exact",
  FontSlant->"Italic"],
 " and hence solvable."
}], "BulletedText",
 CellTags->"b:1.3",
 CellID->14333],

Cell[TextData[{
 "It gives a set of ",
 StyleBox["canonical coordinates",
  FontSlant->"Italic"],
 " in which the equation has a simple (integrable) form."
}], "BulletedText",
 CellTags->"b:1.3",
 CellID->22925],

Cell[CellGroupData[{

Cell[TextData[{
 "It ",
 StyleBox["reduces",
  FontSlant->"Italic"],
 " the problem of solving an ODE of order ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " to that of solving an ODE of order ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "-", "1"}], TraditionalForm]], "InlineMath"],
 ", which is typically a simpler problem."
}], "BulletedText",
 CellTags->"b:1.3",
 CellID->10910],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " function checks for certain standard types of symmetries in the given ODE \
and uses them to return a solution. Following are three examples of ODEs for \
which ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " uses such a symmetry method."
}], "Text",
 CellTags->"b:1.3",
 CellID->19737],

Cell[TextData[{
 "Here is a nonlinear first-order ODE (equation 1.120, page 315 of [",
 ButtonBox["K59",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveReferences#21388"],
 "])."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->16336],

Cell[BoxData[
 RowBox[{
  RowBox[{"FirstOrderODE", "=", 
   RowBox[{
    RowBox[{"x", "*", 
     RowBox[{
      RowBox[{"y", "'"}], "[", "x", "]"}]}], "\[Equal]", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "*", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"x", "*", 
        RowBox[{"Log", "[", 
         RowBox[{
          RowBox[{"x", "^", "2"}], "/", 
          RowBox[{"y", "[", "x", "]"}]}], "]"}]}], "+", "2"}], ")"}]}]}]}], 
  ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[24]:=",
 CellID->18337],

Cell["\<\
This ODE has a symmetry with the following infinitesimal generator.\
\>", "MathCaption",
 CellTags->"b:1.3",
 CellID->1278],

Cell[BoxData[
 RowBox[{
  RowBox[{"v", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "2"}], "*", 
       RowBox[{"Exp", "[", 
        RowBox[{"-", "x"}], "]"}], "*", "y"}], ")"}], "*", 
     RowBox[{"D", "[", 
      RowBox[{"#", ",", " ", "y"}], "]"}]}], "&"}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[25]:=",
 CellID->12393],

Cell[TextData[{
 "The presence of this symmetry allows ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " to calculate an integrating factor and return the solution."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->7636],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"FirstOrderODE", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[26]:=",
 CellID->30714],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{"-", "2"}], " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "x"}]], " ", 
         RowBox[{"C", "[", "1", "]"}]}]], " ", 
       SuperscriptBox["x", "2"]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{230, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[26]=",
 CellID->333332553]
}, Open  ]],

Cell["This verifies the solution.", "MathCaption",
 CellTags->"b:1.3",
 CellID->31591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"FirstOrderODE", "/.", 
     RowBox[{"sol", "[", 
      RowBox[{"[", "1", "]"}], "]"}]}], "/.", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"x", "\[Rule]", " ", 
       RowBox[{"RandomReal", "[", "]"}]}], ",", " ", 
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], "\[Rule]", " ", 
       RowBox[{"RandomReal", "[", "]"}]}]}], "}"}]}], "//", "Simplify"}], "//",
   "Chop"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[27]:=",
 CellID->19949],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[27]=",
 CellID->184244053]
}, Open  ]],

Cell[TextData[{
 "Here is a second-order nonlinear ODE, based on equation 6.93 on page 213 \
of",
 " ",
 "[",
 ButtonBox["K59",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveReferences#7355"],
 "]."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->2505],

Cell[BoxData[
 RowBox[{
  RowBox[{"SecondOrderODE", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "3"}], "*", 
     RowBox[{
      RowBox[{"y", "''"}], "[", "x", "]"}]}], " ", "\[Equal]", " ", 
    RowBox[{"6", "*", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"x", "*", 
         RowBox[{
          RowBox[{"y", "'"}], "[", "x", "]"}]}], "-", " ", 
        RowBox[{"y", "[", "x", "]"}]}], ")"}], "^", "2"}]}]}]}], 
  ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[28]:=",
 CellID->13337],

Cell["\<\
This equation is invariant under the following scaling transformation.\
\>", "MathCaption",
 CellTags->"b:1.3",
 CellID->10004],

Cell[BoxData[
 RowBox[{
  RowBox[{"m", "=", " ", 
   RowBox[{"x", "*", 
    RowBox[{"E", "^", "t"}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[29]:=",
 CellID->16337],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", "=", 
   RowBox[{"y", "*", 
    RowBox[{"E", "^", "t"}]}]}], ";"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[30]:=",
 CellID->2623],

Cell[TextData[{
 "The presence of this ",
 StyleBox["scaling symmetry",
  FontSlant->"Italic"],
 " allows ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " to find new coordinates in which the independent variable is not \
explicitly present. Hence the problem is solved easily."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->28664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"SecondOrderODE", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[31]:=",
 CellID->9970],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:1.3",
 CellLabel->"During evaluation of In[31]:=",
 CellID->16750299],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "6"]}], " ", "x", " ", 
       RowBox[{"Log", "[", 
        RowBox[{"6", " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            FractionBox[
             RowBox[{"C", "[", "1", "]"}], "x"]}], "-", 
           RowBox[{"C", "[", "2", "]"}]}], ")"}]}], "]"}]}]}], "]"}]}], "}"}],
   "}"}]], "Output",
 ImageSize->{329, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[31]=",
 CellID->251576822]
}, Open  ]],

Cell["This verifies the solution.", "MathCaption",
 CellTags->"b:1.3",
 CellID->19281],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SecondOrderODE", " ", "/.", "sol"}], "//", "Simplify"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[32]:=",
 CellID->7085],

Cell[BoxData[
 RowBox[{"{", "True", "}"}]], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[32]=",
 CellID->287871999]
}, Open  ]],

Cell[TextData[{
 "Finally, here is a system of two nonlinear first-order ODEs that can be \
solved by using a shift: ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"u", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{"u", "[", "x", "]"}], "-", "x"}]}], TraditionalForm]], 
  "InlineMath"],
 ". After the shift, the system becomes autonomous (it does not depend \
explicitly on ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ") and hence it can be solved by reduction to a first-order ODE for ",
 Cell[BoxData[
  FormBox["v", TraditionalForm]], "InlineMath"],
 " as a function of ",
 Cell[BoxData[
  FormBox["u", TraditionalForm]], "InlineMath"],
 ". The ",
 StyleBox["Solve::ifun", "InlineCode"],
 " message can be ignored; it is generated while inverting the expression for \
",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Exp",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Exp"], "[", 
   StyleBox["v", "TI"], "]"}]], "InlineFormula"],
 " to give an expression in terms of ",
 Cell[BoxData[
  ButtonBox["Log",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Log"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:1.3",
 CellID->18373],

Cell[BoxData[
 RowBox[{"Clear", "[", 
  RowBox[{"u", ",", "v"}], "]"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[33]:=",
 CellID->28652],

Cell[BoxData[
 RowBox[{
  RowBox[{"NonlinearSystem", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"u", "'"}], "[", "x", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"Exp", "[", 
        RowBox[{"v", "[", "x", "]"}], "]"}], "+", "1"}]}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"v", "'"}], "[", "x", "]"}], " ", "\[Equal]", " ", 
      RowBox[{
       RowBox[{"u", "[", "x", "]"}], "-", "x"}]}]}], "}"}]}], ";"}]], "Input",\

 CellTags->"b:1.3",
 CellLabel->"In[34]:=",
 CellID->8194],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"NonlinearSystem", ",", " ", 
    RowBox[{"{", 
     RowBox[{"u", ",", "v"}], "}"}], ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[35]:=",
 CellID->9876],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:1.3",
 CellLabel->"During evaluation of In[35]:=",
 CellID->439830118],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"v", "\[Rule]", 
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{"Log", "[", 
        RowBox[{
         RowBox[{"C", "[", "1", "]"}], "+", 
         RowBox[{
          RowBox[{"C", "[", "1", "]"}], " ", 
          SuperscriptBox[
           RowBox[{"Tan", "[", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{
                SqrtBox["2"], " ", "x", " ", 
                SqrtBox[
                 RowBox[{"C", "[", "1", "]"}]]}], "+", 
               RowBox[{"2", " ", 
                SqrtBox["2"], " ", 
                SqrtBox[
                 RowBox[{"C", "[", "1", "]"}]], " ", 
                RowBox[{"C", "[", "2", "]"}]}]}], ")"}]}], "]"}], "2"]}]}], 
        "]"}]}], "]"}]}], ",", 
    RowBox[{"u", "\[Rule]", 
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{"x", "+", 
        RowBox[{
         SqrtBox["2"], " ", 
         SqrtBox[
          RowBox[{"C", "[", "1", "]"}]], " ", 
         RowBox[{"Tan", "[", 
          RowBox[{
           FractionBox["1", "2"], " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              SqrtBox["2"], " ", "x", " ", 
              SqrtBox[
               RowBox[{"C", "[", "1", "]"}]]}], "+", 
             RowBox[{"2", " ", 
              SqrtBox["2"], " ", 
              SqrtBox[
               RowBox[{"C", "[", "1", "]"}]], " ", 
              RowBox[{"C", "[", "2", "]"}]}]}], ")"}]}], "]"}]}]}]}], 
      "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{533, 61},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[35]=",
 CellID->11091606]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NonlinearSystem", "/.", "sol"}], "//", "Simplify"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[36]:=",
 CellID->7572],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"True", ",", "True"}], "}"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.3",
 CellLabel->"Out[36]=",
 CellID->40133574]
}, Open  ]],

Cell[BoxData[
 RowBox[{"Clear", "[", 
  RowBox[{"m", ",", "n", ",", "u", ",", "v"}], "]"}]], "Input",
 CellTags->"b:1.3",
 CellLabel->"In[37]:=",
 CellID->14640],

Cell["\<\
This concludes the discussion of ordinary differential equations. \
\>", "Text",
 CellTags->"b:1.3",
 CellID->29174]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->581124755],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Lie Symmetry Methods for Solving Nonlinear ODEs - Wolfram \
Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/DSolveSystemsOfNonlinearODEs"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolveIntroductionToPDEs"]}], 
      "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 32.7976168}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "Around 1870, Marius Sophus Lie realized that many of the methods for \
solving differential equations could be unified using group theory. Lie \
symmetry methods are central to the modern approach for studying nonlinear \
ODEs. They use the notion of symmetry to generate solutions in a systematic \
manner. Here is a brief introduction to Lie's approach that provides some \
examples that are solved in this way by DSolve. A key notion in Lie's method \
is that of an infinitesimal generator for a symmetry group. This concept is \
illustrated in the following example. Here is the well-known transformation \
for rotations in the x-y plane. This is a one-parameter group of \
transformations with parameter t. ", "synonyms" -> {}, "title" -> 
    "Lie Symmetry Methods for Solving Nonlinear ODEs", "type" -> "Tutorial", 
    "uri" -> "tutorial/DSolveLieSymmetryMethods"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:35"->{
  Cell[1280, 43, 109, 2, 70, "Title",
   CellTags->{"c:35", "b:1.3"},
   CellID->7278]},
 "b:1.3"->{
  Cell[1280, 43, 109, 2, 70, "Title",
   CellTags->{"c:35", "b:1.3"},
   CellID->7278],
  Cell[1392, 47, 622, 14, 70, "Text",
   CellTags->"b:1.3",
   CellID->20799],
  Cell[2017, 63, 272, 8, 70, "Text",
   CellTags->"b:1.3",
   CellID->18352],
  Cell[2292, 73, 415, 13, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->20448],
  Cell[2710, 88, 281, 10, 70, "Input",
   CellTags->"b:1.3",
   CellID->13882],
  Cell[2994, 100, 305, 11, 70, "Input",
   CellTags->"b:1.3",
   CellID->540],
  Cell[3302, 113, 609, 20, 70, "Text",
   CellTags->"b:1.3",
   CellID->28315],
  Cell[3936, 137, 3614, 100, 119, "Input",
   CellTags->"b:1.3",
   CellID->4575],
  Cell[7553, 239, 5934, 102, 223, "Output",
   Evaluatable->False,
   CellTags->"b:1.3",
   CellID->378886835],
  Cell[13502, 344, 192, 8, 70, "Text",
   CellTags->"b:1.3",
   CellID->9567],
  Cell[13697, 354, 727, 22, 70, "DisplayMath",
   CellTags->"b:1.3",
   CellID->22336],
  Cell[14427, 378, 168, 5, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->18397],
  Cell[14598, 385, 446, 16, 70, "Input",
   CellTags->"b:1.3",
   CellID->31418],
  Cell[15069, 405, 282, 10, 70, "Input",
   CellTags->"b:1.3",
   CellID->19897],
  Cell[15354, 417, 181, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->569618775],
  Cell[15572, 428, 310, 11, 70, "Input",
   CellTags->"b:1.3",
   CellID->13851],
  Cell[15885, 441, 180, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->14417335],
  Cell[16102, 452, 178, 6, 70, "Input",
   CellTags->"b:1.3",
   CellID->24230],
  Cell[16283, 460, 181, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->166297553],
  Cell[16501, 471, 403, 14, 70, "Input",
   CellTags->"b:1.3",
   CellID->16925],
  Cell[16907, 487, 180, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->30753395],
  Cell[17102, 496, 273, 8, 70, "Text",
   CellTags->"b:1.3",
   CellID->229],
  Cell[17378, 506, 405, 13, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->24520],
  Cell[17808, 523, 255, 9, 70, "Input",
   CellTags->"b:1.3",
   CellID->10940],
  Cell[18066, 534, 223, 8, 36, "Output",
   CellTags->"b:1.3",
   CellID->375578462],
  Cell[18326, 547, 256, 9, 70, "Input",
   CellTags->"b:1.3",
   CellID->15147],
  Cell[18585, 558, 243, 9, 36, "Output",
   CellTags->"b:1.3",
   CellID->269638730],
  Cell[18843, 570, 489, 16, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->1655],
  Cell[19335, 588, 408, 15, 70, "Input",
   CellTags->"b:1.3",
   CellID->20675],
  Cell[19746, 605, 637, 18, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->792],
  Cell[20408, 627, 600, 20, 70, "Input",
   CellTags->"b:1.3",
   CellID->22361],
  Cell[21011, 649, 804, 27, 36, "Output",
   CellTags->"b:1.3",
   CellID->13456374],
  Cell[21830, 679, 373, 9, 70, "Text",
   CellTags->"b:1.3",
   CellID->16398],
  Cell[22206, 690, 229, 8, 70, "Text",
   CellTags->"b:1.3",
   CellID->23146],
  Cell[22438, 700, 94, 2, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->8714],
  Cell[22535, 704, 224, 8, 70, "Input",
   CellTags->"b:1.3",
   CellID->26946],
  Cell[22762, 714, 434, 16, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->21188],
  Cell[23221, 734, 122, 4, 70, "Input",
   CellTags->"b:1.3",
   CellID->19569],
  Cell[23346, 740, 179, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->150931950],
  Cell[23540, 749, 139, 5, 70, "Text",
   CellTags->"b:1.3",
   CellID->27663],
  Cell[23682, 756, 237, 8, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->15075],
  Cell[23922, 766, 391, 13, 70, "Input",
   CellTags->"b:1.3",
   CellID->24515],
  Cell[24316, 781, 186, 7, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->11521],
  Cell[24505, 790, 184, 7, 70, "Input",
   CellTags->"b:1.3",
   CellID->475],
  Cell[24692, 799, 229, 9, 70, "Input",
   CellTags->"b:1.3",
   CellID->15615],
  Cell[24924, 810, 164, 5, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->20528],
  Cell[25113, 819, 246, 9, 70, "Input",
   CellTags->"b:1.3",
   CellID->13234],
  Cell[25362, 830, 224, 8, 36, "Output",
   CellTags->"b:1.3",
   CellID->180211428],
  Cell[25623, 843, 244, 9, 70, "Input",
   CellTags->"b:1.3",
   CellID->905],
  Cell[25870, 854, 224, 8, 36, "Output",
   CellTags->"b:1.3",
   CellID->256074322],
  Cell[26109, 865, 398, 15, 70, "Input",
   CellTags->"b:1.3",
   CellID->19557],
  Cell[26510, 882, 739, 26, 70, "Text",
   CellTags->"b:1.3",
   CellID->28962],
  Cell[27252, 910, 175, 7, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->14161],
  Cell[27430, 919, 531, 18, 70, "Input",
   CellTags->"b:1.3",
   CellID->30524],
  Cell[27964, 939, 360, 12, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->12549],
  Cell[28327, 953, 285, 10, 70, "Input",
   CellTags->"b:1.3",
   CellID->17469],
  Cell[28637, 967, 390, 13, 70, "Input",
   CellTags->"b:1.3",
   CellID->330],
  Cell[29030, 982, 179, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->230072638],
  Cell[29224, 991, 206, 5, 70, "Text",
   CellTags->"b:1.3",
   CellID->28350],
  Cell[29433, 998, 247, 8, 70, "BulletedText",
   CellTags->"b:1.3",
   CellID->14333],
  Cell[29683, 1008, 211, 7, 70, "BulletedText",
   CellTags->"b:1.3",
   CellID->22925],
  Cell[29919, 1019, 405, 14, 70, "BulletedText",
   CellTags->"b:1.3",
   CellID->10910],
  Cell[30327, 1035, 502, 16, 70, "Text",
   CellTags->"b:1.3",
   CellID->19737],
  Cell[30832, 1053, 244, 8, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->16336],
  Cell[31079, 1063, 521, 19, 70, "Input",
   CellTags->"b:1.3",
   CellID->18337],
  Cell[31603, 1084, 133, 4, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->1278],
  Cell[31739, 1090, 384, 14, 70, "Input",
   CellTags->"b:1.3",
   CellID->12393],
  Cell[32126, 1106, 290, 9, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->7636],
  Cell[32441, 1119, 195, 6, 70, "Input",
   CellTags->"b:1.3",
   CellID->30714],
  Cell[32639, 1127, 614, 20, 43, "Output",
   CellTags->"b:1.3",
   CellID->333332553],
  Cell[33268, 1150, 86, 2, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->31591],
  Cell[33379, 1156, 514, 17, 70, "Input",
   CellTags->"b:1.3",
   CellID->19949],
  Cell[33896, 1175, 182, 6, 36, "Output",
   CellTags->"b:1.3",
   CellID->184244053],
  Cell[34093, 1184, 264, 11, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->2505],
  Cell[34360, 1197, 534, 19, 70, "Input",
   CellTags->"b:1.3",
   CellID->13337],
  Cell[34897, 1218, 137, 4, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->10004],
  Cell[35037, 1224, 181, 7, 70, "Input",
   CellTags->"b:1.3",
   CellID->16337],
  Cell[35221, 1233, 175, 7, 70, "Input",
   CellTags->"b:1.3",
   CellID->2623],
  Cell[35399, 1242, 405, 13, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->28664],
  Cell[35829, 1259, 195, 6, 70, "Input",
   CellTags->"b:1.3",
   CellID->9970],
  Cell[36027, 1267, 517, 10, 70, "Message",
   CellTags->"b:1.3",
   CellID->16750299],
  Cell[36547, 1279, 694, 24, 51, "Output",
   CellTags->"b:1.3",
   CellID->251576822],
  Cell[37256, 1306, 86, 2, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->19281],
  Cell[37367, 1312, 160, 5, 70, "Input",
   CellTags->"b:1.3",
   CellID->7085],
  Cell[37530, 1319, 204, 7, 36, "Output",
   CellTags->"b:1.3",
   CellID->287871999],
  Cell[37749, 1329, 1182, 38, 70, "MathCaption",
   CellTags->"b:1.3",
   CellID->18373],
  Cell[38934, 1369, 141, 5, 70, "Input",
   CellTags->"b:1.3",
   CellID->28652],
  Cell[39078, 1376, 548, 19, 70, "Input",
   CellTags->"b:1.3",
   CellID->8194],
  Cell[39651, 1399, 247, 8, 70, "Input",
   CellTags->"b:1.3",
   CellID->9876],
  Cell[39901, 1409, 518, 10, 70, "Message",
   CellTags->"b:1.3",
   CellID->439830118],
  Cell[40422, 1421, 1836, 58, 82, "Output",
   CellTags->"b:1.3",
   CellID->11091606],
  Cell[42295, 1484, 156, 5, 70, "Input",
   CellTags->"b:1.3",
   CellID->7572],
  Cell[42454, 1491, 254, 9, 36, "Output",
   CellTags->"b:1.3",
   CellID->40133574],
  Cell[42723, 1503, 161, 5, 70, "Input",
   CellTags->"b:1.3",
   CellID->14640],
  Cell[42887, 1510, 126, 4, 70, "Text",
   CellTags->"b:1.3",
   CellID->29174]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:35", 46128, 1590},
 {"b:1.3", 46229, 1594}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 109, 2, 70, "Title",
 CellTags->{"c:35", "b:1.3"},
 CellID->7278],
Cell[1392, 47, 622, 14, 70, "Text",
 CellTags->"b:1.3",
 CellID->20799],
Cell[2017, 63, 272, 8, 70, "Text",
 CellTags->"b:1.3",
 CellID->18352],
Cell[2292, 73, 415, 13, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->20448],
Cell[2710, 88, 281, 10, 70, "Input",
 CellTags->"b:1.3",
 CellID->13882],
Cell[2994, 100, 305, 11, 70, "Input",
 CellTags->"b:1.3",
 CellID->540],
Cell[3302, 113, 609, 20, 70, "Text",
 CellTags->"b:1.3",
 CellID->28315],
Cell[CellGroupData[{
Cell[3936, 137, 3614, 100, 119, "Input",
 CellTags->"b:1.3",
 CellID->4575],
Cell[7553, 239, 5934, 102, 223, "Output",
 Evaluatable->False,
 CellTags->"b:1.3",
 CellID->378886835]
}, Open  ]],
Cell[13502, 344, 192, 8, 70, "Text",
 CellTags->"b:1.3",
 CellID->9567],
Cell[13697, 354, 727, 22, 70, "DisplayMath",
 CellTags->"b:1.3",
 CellID->22336],
Cell[14427, 378, 168, 5, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->18397],
Cell[14598, 385, 446, 16, 70, "Input",
 CellTags->"b:1.3",
 CellID->31418],
Cell[CellGroupData[{
Cell[15069, 405, 282, 10, 70, "Input",
 CellTags->"b:1.3",
 CellID->19897],
Cell[15354, 417, 181, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->569618775]
}, Open  ]],
Cell[CellGroupData[{
Cell[15572, 428, 310, 11, 70, "Input",
 CellTags->"b:1.3",
 CellID->13851],
Cell[15885, 441, 180, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->14417335]
}, Open  ]],
Cell[CellGroupData[{
Cell[16102, 452, 178, 6, 70, "Input",
 CellTags->"b:1.3",
 CellID->24230],
Cell[16283, 460, 181, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->166297553]
}, Open  ]],
Cell[CellGroupData[{
Cell[16501, 471, 403, 14, 70, "Input",
 CellTags->"b:1.3",
 CellID->16925],
Cell[16907, 487, 180, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->30753395]
}, Open  ]],
Cell[17102, 496, 273, 8, 70, "Text",
 CellTags->"b:1.3",
 CellID->229],
Cell[17378, 506, 405, 13, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->24520],
Cell[CellGroupData[{
Cell[17808, 523, 255, 9, 70, "Input",
 CellTags->"b:1.3",
 CellID->10940],
Cell[18066, 534, 223, 8, 36, "Output",
 CellTags->"b:1.3",
 CellID->375578462]
}, Open  ]],
Cell[CellGroupData[{
Cell[18326, 547, 256, 9, 70, "Input",
 CellTags->"b:1.3",
 CellID->15147],
Cell[18585, 558, 243, 9, 36, "Output",
 CellTags->"b:1.3",
 CellID->269638730]
}, Open  ]],
Cell[18843, 570, 489, 16, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->1655],
Cell[19335, 588, 408, 15, 70, "Input",
 CellTags->"b:1.3",
 CellID->20675],
Cell[19746, 605, 637, 18, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->792],
Cell[CellGroupData[{
Cell[20408, 627, 600, 20, 70, "Input",
 CellTags->"b:1.3",
 CellID->22361],
Cell[21011, 649, 804, 27, 36, "Output",
 CellTags->"b:1.3",
 CellID->13456374]
}, Open  ]],
Cell[21830, 679, 373, 9, 70, "Text",
 CellTags->"b:1.3",
 CellID->16398],
Cell[22206, 690, 229, 8, 70, "Text",
 CellTags->"b:1.3",
 CellID->23146],
Cell[22438, 700, 94, 2, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->8714],
Cell[22535, 704, 224, 8, 70, "Input",
 CellTags->"b:1.3",
 CellID->26946],
Cell[22762, 714, 434, 16, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->21188],
Cell[CellGroupData[{
Cell[23221, 734, 122, 4, 70, "Input",
 CellTags->"b:1.3",
 CellID->19569],
Cell[23346, 740, 179, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->150931950]
}, Open  ]],
Cell[23540, 749, 139, 5, 70, "Text",
 CellTags->"b:1.3",
 CellID->27663],
Cell[23682, 756, 237, 8, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->15075],
Cell[23922, 766, 391, 13, 70, "Input",
 CellTags->"b:1.3",
 CellID->24515],
Cell[24316, 781, 186, 7, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->11521],
Cell[24505, 790, 184, 7, 70, "Input",
 CellTags->"b:1.3",
 CellID->475],
Cell[24692, 799, 229, 9, 70, "Input",
 CellTags->"b:1.3",
 CellID->15615],
Cell[24924, 810, 164, 5, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->20528],
Cell[CellGroupData[{
Cell[25113, 819, 246, 9, 70, "Input",
 CellTags->"b:1.3",
 CellID->13234],
Cell[25362, 830, 224, 8, 36, "Output",
 CellTags->"b:1.3",
 CellID->180211428]
}, Open  ]],
Cell[CellGroupData[{
Cell[25623, 843, 244, 9, 70, "Input",
 CellTags->"b:1.3",
 CellID->905],
Cell[25870, 854, 224, 8, 36, "Output",
 CellTags->"b:1.3",
 CellID->256074322]
}, Open  ]],
Cell[26109, 865, 398, 15, 70, "Input",
 CellTags->"b:1.3",
 CellID->19557],
Cell[26510, 882, 739, 26, 70, "Text",
 CellTags->"b:1.3",
 CellID->28962],
Cell[27252, 910, 175, 7, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->14161],
Cell[27430, 919, 531, 18, 70, "Input",
 CellTags->"b:1.3",
 CellID->30524],
Cell[27964, 939, 360, 12, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->12549],
Cell[28327, 953, 285, 10, 70, "Input",
 CellTags->"b:1.3",
 CellID->17469],
Cell[CellGroupData[{
Cell[28637, 967, 390, 13, 70, "Input",
 CellTags->"b:1.3",
 CellID->330],
Cell[29030, 982, 179, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->230072638]
}, Open  ]],
Cell[29224, 991, 206, 5, 70, "Text",
 CellTags->"b:1.3",
 CellID->28350],
Cell[29433, 998, 247, 8, 70, "BulletedText",
 CellTags->"b:1.3",
 CellID->14333],
Cell[29683, 1008, 211, 7, 70, "BulletedText",
 CellTags->"b:1.3",
 CellID->22925],
Cell[CellGroupData[{
Cell[29919, 1019, 405, 14, 70, "BulletedText",
 CellTags->"b:1.3",
 CellID->10910],
Cell[30327, 1035, 502, 16, 70, "Text",
 CellTags->"b:1.3",
 CellID->19737],
Cell[30832, 1053, 244, 8, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->16336],
Cell[31079, 1063, 521, 19, 70, "Input",
 CellTags->"b:1.3",
 CellID->18337],
Cell[31603, 1084, 133, 4, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->1278],
Cell[31739, 1090, 384, 14, 70, "Input",
 CellTags->"b:1.3",
 CellID->12393],
Cell[32126, 1106, 290, 9, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->7636],
Cell[CellGroupData[{
Cell[32441, 1119, 195, 6, 70, "Input",
 CellTags->"b:1.3",
 CellID->30714],
Cell[32639, 1127, 614, 20, 43, "Output",
 CellTags->"b:1.3",
 CellID->333332553]
}, Open  ]],
Cell[33268, 1150, 86, 2, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->31591],
Cell[CellGroupData[{
Cell[33379, 1156, 514, 17, 70, "Input",
 CellTags->"b:1.3",
 CellID->19949],
Cell[33896, 1175, 182, 6, 36, "Output",
 CellTags->"b:1.3",
 CellID->184244053]
}, Open  ]],
Cell[34093, 1184, 264, 11, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->2505],
Cell[34360, 1197, 534, 19, 70, "Input",
 CellTags->"b:1.3",
 CellID->13337],
Cell[34897, 1218, 137, 4, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->10004],
Cell[35037, 1224, 181, 7, 70, "Input",
 CellTags->"b:1.3",
 CellID->16337],
Cell[35221, 1233, 175, 7, 70, "Input",
 CellTags->"b:1.3",
 CellID->2623],
Cell[35399, 1242, 405, 13, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->28664],
Cell[CellGroupData[{
Cell[35829, 1259, 195, 6, 70, "Input",
 CellTags->"b:1.3",
 CellID->9970],
Cell[36027, 1267, 517, 10, 70, "Message",
 CellTags->"b:1.3",
 CellID->16750299],
Cell[36547, 1279, 694, 24, 51, "Output",
 CellTags->"b:1.3",
 CellID->251576822]
}, Open  ]],
Cell[37256, 1306, 86, 2, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->19281],
Cell[CellGroupData[{
Cell[37367, 1312, 160, 5, 70, "Input",
 CellTags->"b:1.3",
 CellID->7085],
Cell[37530, 1319, 204, 7, 36, "Output",
 CellTags->"b:1.3",
 CellID->287871999]
}, Open  ]],
Cell[37749, 1329, 1182, 38, 70, "MathCaption",
 CellTags->"b:1.3",
 CellID->18373],
Cell[38934, 1369, 141, 5, 70, "Input",
 CellTags->"b:1.3",
 CellID->28652],
Cell[39078, 1376, 548, 19, 70, "Input",
 CellTags->"b:1.3",
 CellID->8194],
Cell[CellGroupData[{
Cell[39651, 1399, 247, 8, 70, "Input",
 CellTags->"b:1.3",
 CellID->9876],
Cell[39901, 1409, 518, 10, 70, "Message",
 CellTags->"b:1.3",
 CellID->439830118],
Cell[40422, 1421, 1836, 58, 82, "Output",
 CellTags->"b:1.3",
 CellID->11091606]
}, Open  ]],
Cell[CellGroupData[{
Cell[42295, 1484, 156, 5, 70, "Input",
 CellTags->"b:1.3",
 CellID->7572],
Cell[42454, 1491, 254, 9, 36, "Output",
 CellTags->"b:1.3",
 CellID->40133574]
}, Open  ]],
Cell[42723, 1503, 161, 5, 70, "Input",
 CellTags->"b:1.3",
 CellID->14640],
Cell[42887, 1510, 126, 4, 70, "Text",
 CellTags->"b:1.3",
 CellID->29174]
}, Open  ]],
Cell[CellGroupData[{
Cell[43050, 1519, 72, 1, 70, "RelatedTutorialsSection",
 CellID->581124755],
Cell[43125, 1522, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],
Cell[43329, 1529, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

