(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     43504,       1214]
NotebookOptionsPosition[     34657,        926]
NotebookOutlinePosition[     40386,       1095]
CellTagsIndexPosition[     40213,       1087]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["\<\
Linear Second-Order Equations with Solutions Involving Special Functions\
\>", "Title",
 CellTags->{"special_functions_in_solutions", "c:22", "b:1.1.1.4"},
 CellID->3497],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " can find solutions for most of the standard linear second-order ODEs that \
occur in applied mathematics."
}], "Text",
 CellTags->"b:1.1.1.4",
 CellID->23243],

Cell[TextData[{
 "Here is the solution for ",
 StyleBox["Airy\[CloseCurlyQuote]s equation",
  FontSlant->"Italic"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->27649],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "''"}], "[", "x", "]"}], "-", 
     RowBox[{"x", "*", 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[326]:=",
 CellID->22015],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"AiryAi", "[", "x", "]"}], " ", 
      RowBox[{"C", "[", "1", "]"}]}], "+", 
     RowBox[{
      RowBox[{"AiryBi", "[", "x", "]"}], " ", 
      RowBox[{"C", "[", "2", "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{277, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[326]=",
 CellID->216730397]
}, Open  ]],

Cell[TextData[{
 "Here is a plot that shows the oscillatory behavior of the Airy functions \
for large negative values of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->189],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\n", 
  RowBox[{
   RowBox[{"Plot", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"AiryAi", "[", "x", "]"}], ",", 
       RowBox[{"AiryBi", "[", "x", "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "15"}], ",", 
       RowBox[{"-", "12"}]}], "}"}]}], "]"}], "\n"}]}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[327]:=",
 CellID->16100],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1XW2MHtV1ftn33V2vd73+WnvXa3t3/QWx13aaQo3BfPgDf8UYA8YxiaHE
uCR2k0BCTNPSlIAaxaGiDUoUBdQI1BJoSfLDFZHTpE5LQxuBFIGEgoqSSCko
EVQNigpKJKuRtufe5955Zp65M2Ov1ZVZZnbeufc5zzn3nHPPve/M3sPHj97x
scPHjx05PLHz7sMfP3rsyCcndtx1t/2pfUGrdcHRVqv1ykjLjqfsyP/nfo7Z
f63WsPvdjn+b8v/b13Cl13532YV27hP2u+X+1lN5l7/S4RW0k/9su/JeXMKP
66XVjfOsqbNqa4E76P3Jj3/5ja+//KUv/uCuO5/5y4eee/yxH77yozdbF1dg
7A6yuivd4cJcd6n/ue//7NZDTx+5/ZvWlDX40ou/+IeTr1hrx46ePHjgqW+f
erX1noo2laOOnD/oz8+885tfvPiTX/74563P+g+99davDfGf3vud1177lZe2
88Ybb5/43LPWn4k04e/57Zn/fe6hb9htrYFpcn7I/eo2YUwkE+Z7p39qjbcu
dH+d+avX/staf2LffY/vvveZO7/8zdsfevTKT5w8+vArJ//dOv6Mb8nuNFbs
d6vfnxu3BtC4ao1FqU4df9TuaM0q2JLjuZ3g5mbPtcluLG/f9lVjwA4M2AP3
nzbqjWjDeebMb1vLPG5r/QdfPPn0oRPWwa/fehvMdt780X8a7qcO/PnPvv/y
n/n2Tax7jp96550zQNG2FoxYa6q1JKI0AR2Pg8mxcNC3++TXXsoAdEeNGDwj
4IXnXx/391jfxtDLX3+21ec/Yai+c+9jp+//W2PsXv8JsxQTxskwx59DPne4
KOrUWrB2WrOTWA74diHBo4+8YAe+p7ZhMRkNY2upP3/9+f+wdkyHsI32T0+/
aDKapK0/9ucG2VpwOOb5cxsbZtDucMSfG2a7xR0Sh4Nxk79qXAY9g1FDYrbp
Dkdjbya3620w8hJ6/yN/blZm97veh/y5UWJKcocLge6Rb730tdPucE6Shf0R
hVmIs1iPsW26CSggg2nUrM8YDVzb6AoojvtzszNDYcy1FkS7wPAK52ZAJoU7
nJtEcWO8y5h3Y2BuRGFKpSzsdV7UjLXqUH3Sn5sIdjM/bwPebN1xM9+fv/3G
Wzb2svsrUJgyTR3uUPU5FG082APOzUZt7LjDT7gxaTa4KNpn4BDtYGBk56Xe
bxCZ50dOTCznadGbCWxkukMw++zn/u7VUy+4w7vEHkbFHqA5G+A2prP2Syiu
nwrex3p1rSyIY8IGJ22crSyMnJjzcJ7jTtH/4mhfdr9ZSbAv+6R93mliKIli
X2zF7nJWNSyMQhbTgenT6T+zUiPZHX7Mn9sYtuHkDuGdTK2mXHeIsWQmbSaU
aTYbmfviSLQGOAZoF5ABzmUq8zdGiNHiDj8inMGDm2mbq3CHAzImFiQ5uC7q
02yYvRiHdu6YHBRNQN/Ghjlsp4mPCOoxGRP98fOmCRsZ0GQJxd6kPqmJAdEn
mDavZW7PHR4T5hFvEfrd4UwZQ3UoODbQiwlikNwhZLEmrCF3uFQ0dFQ+PyFW
2SfcDSdRXBu5sOjl7loqGpoZubBWnFWCcfNaZqXu8MPC3XKRaoZIUYeCDI6J
niEL/cS4nN8hUqyQ816RYiSJYk+0RrvLWdO4cDFDZAfjZmdmJ67VO0T2lXLe
I1wQhQOxRziYEA56ReZl0toRkXmVnHcLB4uSHLxXOFgmHEAGZDLUN1s9IjJf
KOcdQV2HglwsFy66hYsV0urt/vyGfU+YIERBLtqCejSJYrdwsUK46Ig9rJRW
D/tzC1UWsNzhu4SLLkFdh4JcrBQu2sIF9M4M7YNTRT/hUfRmXNiZYbVE3nm6
xUkEu6aCZzFCHQ+rhIeuKLe1QrbJy23+nP5yjVdhxgMcJfxfNhp2peyoJ8uG
7Ae8gVNGvd/354xL6AvS/veb/xN84ZIaOcVqejLcmWLRJ2PdrcLH2qixJz//
jH0EPrXU205hxbfaMZgWZC29tSTXY2+bc7GkwB3e4s+Zu62N9nli631uflLX
j4U6S4DYj2P2C/8GOtAP49Uhsft1UZ9f2v0Zl9Wn+9khLKyOY+5D27/w3fv+
pjXpz40+mzG60YFemM+gFwv8p7/ybUMbeqmqL7TjxL+EgNkh5Hpk/4mn/soz
PCnj7QOi77WwlXd+Y3ZsCDFkgq2kqwF5RIpmu4yaNZHFp2550EJ00J5FD1OC
O3y/PzezNePN+OjOtIKhENA010BSdQVXs9gm48qj6oAT+BTgoj/xs+6cv/G4
ejKrzARcXMFTU5UBPNE/ebvpznwSBkhgywa5DXV3eDBpo7wOn2RJiaUm7nB0
muiuEav2fPXY9MQmKfgI7Af4DLKNJ3f4vqR1k1XgQ0WkDl+n1u6IkPEBGs3y
xCnM3xy+rh8+9o/gzezPkubM5nMxqyPaQHyOo7An9/88BliV6cHIcJUU2LoZ
iJkJpTWa7PpUNhJp+75u4epF5uDJFj1SR6w2nTUABePtmiAz7mZMQO+MGTeJ
raB3C142WNk7R3Jd7xJ52iagiUnLor8HCs4q9ksvmdcNURSc07sVs8etwjB6
5+yxEzUUMqDMG4eIiN4tsFh4IQec/6F3eqd0Br1VrB4omIvAdhnBJqWXG8X6
1kaUIT9BBk6PnZ5NbBUbD5Eo+vUgC8ceUDCnRdWCY36daGqGjLg0ii3SC7jg
3LZXUE6KvVwvvawTq50hKOtQkHGgYN3Gj+ycf7he7AS9MrebKajTc9vNsZWC
P2ArwG6JjvVC2ekv9om1AQWyIqKgnRGFA7FZRprmN5hNMiNC75xZo0pR7Q9m
ylir40CzEfbSJyNuUlBmVYo9ux+nPyBHqFKwPpiuuFztzzULoeedKdcnRVNA
wfk9ULACOyDX61DwU2tEn6h1VHvoa+X6OhkrA8JVHQoyphrpF5RAkcX6MENX
D00fOygaT9cBr2rQyEBSI/Ree5IaoQ8dlLFTh6JaI7MaNPLepEaY/6CeSE/N
yqwDcZXcrZqY1aCJ3Q2ayFZPgu9M14WvFKtRTQzGXuw6UVATuxs0ART00HUo
illLXhOzxWpVE7saNIF1B3rsdKX+SrFd1QhQqLeiRnY2aMT16uJMmoMrGjQB
JjV/pCZ2NmgCFX4jwGio5uAKGTkasec2aGJHgybmiybS60ZAweijEVvXkVQT
Ou9UTQwFTZAD1/mmqIGKiA0GNWKbBmw2z16rNTAkGkjLvqlBA2BQIze5uaZB
AwtEA+n1Q6CozpmAgtnHpKDc1qCBhUEDdRzQc64WTQwlNcHr25KaYMTO1hGD
JsBBqs4SD3XGebk/52xNtbSgQUtbk1piNB8WLWGN+dz3TgCnzIxzOBZMMffN
49+S1B/nByPhPjA3/VrMRv93VPxSmlZNqaa3NGgaa3yMP9Pl8TLwIzMZehzV
l+p7c4O+FwkPg+eFk2ucwMmay3CFvq9u0PeotHt++HReyoroSIO+gZOZgup7
VPQEnE21m43SK1hjJU21o7Ml5HLMHFS7i+V+7PVRr7cx6AatcwfAIuHeXXfz
VPTKeKU6WyL3VfearzdAdsYL9E5Pp/NUZE06T6VGlohGiMKBuNRfpSdfIzKA
OUuJTED2TmaRKTBOoXeuFywV5geSHAAFM7tYNQN25iFaPb9CWl8r3IyJBup6
Z3UZHHCuvVhsHigYSzcJw1opGJPraRQbojVV5J/ggh4CstqwtQTIHV4erHe9
jI1x4ai/2HvQ0waxIo2n0KPOxThGEOcSNd3gNybETgRFsNUN8imNlkvFTrR+
eZlY8zqRwqHossuigWDrG4QprdmNSeu6moTetd7O3Hx9YeT9nr8qVf2cLNAb
/YJm+vAbnAmsFTTvLsoIVfleUz5ufCodlzA2OOvROqT0goAcZGP2tUZsdVxG
hEYZ9KpZBdFKr5gCtC4RrFp/Xib61SiyQWxcK7+/w15DzIVlXixo0S898DKx
6hg/dMRrjl7uDz4F/am/os9dJmNZIwa0w9qkzpPK/cIL/a7g1RWW5XId/DFW
QD9a0+R19tv2y6uUyF8pZfv0sCuEEY0T6JlRDMi4rvEeStydeWTc7ttqxxS7
xw6Qyfs7WuEa9kPQT2WfZ2v4vF+FchnQkNjMnCh1J3zQ/6R2tA+IlYw03ukO
eSdZzO6U2UNXoteeYLPjZ3FP7K+TYhXAgwZyn4/nYWxqhoj4wbl+qT85b+es
F0gxY2YlbW7qjty5zjhRpaYPG2m4v6uEALVdzgV9ZtcrFhJoTa8JMycZr7m3
vF7bpiZ6sr9njs1ZbydhM50EjrGaz5b7hZUzRvVV3H123NP/zz3Ldqp0wBgw
r6IltSe1R1xndrboLNsp22VSN/aDFcvJ4t/twO2gb4XVKx0fXOXsKt2H390V
OmLWMqf4iSn4K+0PGmEeMJK8q9wb7uNMb7RBg2oJ4JMrVBMN95ctoIpvpC6T
STnivBURjhlBO/npKo6l3lVjJZ3Q0qzYX2F2fi7WpauUTXy38zLkItr58JzF
CyR4LeHYXHKqJoAshnHubLlG5Y8jfPZZSJw/B+fMs3We1iw5OKfXGy19Iqm3
0jmj3bLGFmrtvGQxzHnfFXjE35GPxPjj/q47fGP+onWlaLGoFzLGDebab+fa
BcvF7zvkPxdHnK5zj06xRtzV0pox64ET8rmePO4kKy67QUaqK1DMQy/y51qL
YRR4d45Fd4j26JUvrtEJay7IjZitrpZesPeTdSd8nrkBKgDMhnVmeEkSBRhk
3WmNyKo7GFYLV3FWsz6iK8xqWMur650RHr1pHZX+aLUwrftnQhVMqmKc49Sh
4K5a9Fq9iwfXmVPqDjyca3VMZkQ+GrgRuz96wRIub7XdmfT4eBgb/JTmahqz
0Uq2ajfNVnpymkrdr5+vbA9BqfT5pnPfXiFyQ1+aKZxVOyV2snpMBTtn3aqN
BPs3Pely51InK2DNxuI0NYlWsnrHNFuBPWBedF72gPRoeowVZkL0dBqlpmMT
riqm0XbaWoWT/v+0ilw9BtGKnlilwHcomGH1y/Vxf85MSmfM6J+Z0iK5jr0r
rGAuT0qlca1JSoxz+lB804PVrLSUrMalpWTGkpaSmQpiQHWNeUMNauZRQH1C
dlgwhoc1hMpMAyh0ryzjYx0KxmKg0HV+9pLF8opMAxrTiinj46UZCgcCvXN0
6nd4dM0B2VQ5w8jyn0KGwbnopUnZ4ae4EuuqHqyE04o1h9PM4iKxN61N1/XO
/BQVQ1bIdR0WFc/qzELtByhYnapDQf2ibqoVc+Y/vi5a2m1EWdV+gJIeZWMS
Ra9Y0XJhfI30AhS614UoV8lYWSdjrRqFsyatHev6K3rXFXG1YmaqmuOx9zj/
Ka9J6J4nWhtq/bpmQWuFHem6G2PAZUnZZ4i1TAiDcf31EpFNdxmtEG505bOu
d+oXveseGdrHxYJOfeEKsQ+g4By3DkVxxlhe/WMvWDXRlXD2AivmiNYV0Mtr
UHCmPy6M6qwLKE4UVsSdptC7rtNRU+nesf+bUWNMmNRZl64c6coSrLk6KtWh
0PVyeh2ddem6ma4zQZPVUWlThsKBQO9cC1oqTOq67KXSOjhgfgEN6k4e2hl7
L9dTKQNW+9UnspKgK6S6JwEodJ8Oo18dCs6SgYK+MYtSof65UbjSKDUm1qqV
iCtqUDBLWSz61nXdy8ReNEotjfcX9ukwftShkO8ll3ZIMMro6v+kSAEUGqWI
Mo2iX/SO+q3ukKDdAEV1tFoq9+sa95U1KDgGgELzRo4h7IjR3XiUAnal+3ao
UaJwINA7xy/mGcW9gM4HYjeQRiuOvMVyn9aE0rJj1ZQRN+68VP+/SfSr+8MX
T+X353CM1PVK/43Kv3xDNRddrhSN6P7HUdEIGGeecVUNCvaC67pbjCiB4oTs
1+L1RXK/7lUDChcTq/YB64x+lmgB9efiTrK89nS3nOYSQKi5BMcXEOJ5Z0RR
Xr1IrSo4GwWDGts4PrM9hYXYRvlgB5plMAJffQ74urOxxfuxeqLfDKKv9O3L
Tgsn10hCs+7vm6fJ16DwjhU0/YYKvSdwaTykXMNJvZLXc+GtjJMWDJwcARox
N0d+C/M6rjAsTOqX9nl+fNLDZk/BKsw7GK22iJ1GPS9M6DmfuwFfHMFx1UVr
s1iTow/ENzt0rYHaBRrd96dWW61doFLvNlt6AQrOQ1R3+v1m1d1Qg+621KAg
80Ch8xBe31ahGf1OhmqGvbvOZ0fZgt9B1Uv315ObbQ0a0O+j6D7gtOxYB2de
gHNGcNXANQ0amJfUAFFurUHBPGFOgwa2V2hAv58UfeC2ml4ZW2c3aGB7gwbm
JjVAD5qWHXcxTxhs0MCOBg3MSWqAeUaai2B9WZ4w2KCBnRUamJMcA8zR63qn
JmY1aGJngyZmJzVBb0cUDoQ+xXCgQQO7GjQwmNQAPf81SQ70+YMDDRrYXaGB
waQGmGPX9U4N9DdoYHdSA/S0s0QDOsvZnkQxXzQxs0ET2XPKCppg1BlIakL8
SAUKagIodK7FVvYkNFH+XrtqRPxKBQo+kyXUaCoj9J4GjTgUbvazLnIY4s6O
wlhADNPnS1RnU9c2aECfFIHrnMfuSMo+JJ9CzVZzJebEexs00CfWrLtC6lCQ
KaDQ+gNb2dugAaDQmb98f7mEQp+Gqk+A0drcdZGrwvyFGs24LDzbQ77DXYGC
No98UmcpHDP7BKVWAoBC13A48tMokPvSj3eEcf2+uj5BRGsxQKH7dBhHdtWg
YEwDCl1Jot7xJBldQaBGlUudkxOFA4HeGc8xzyjWq10ccE/R4fdJGH26hTON
jWmZcc4VAazx6ooBcx3ITJ+nMVKfMKWeeXc1it78rgvasdYk8Syjap8A3vT5
PqzP1iDoyUabncGGVafoXWfD9OZt4U+fwOUjW7IO43Z4pOowfv7fnSGAUQdW
6B3wlCvxDr2Z1GC0UKXlygcwlWe++gy8JLIOLBaRYlJaxpO/pGrVk1mD/YDJ
9WKFVYia5uKLYjtPPPw9twNqUto9ILaBZwJm1hueCajfNt1TgUb3jrix4yo/
w5GFT+980D0LWi34oIwR7NoFa8C2XqzuXBCU+fjX7776la2f4q46joWDKWvu
wFo//9l/tgvrxM6urUBS/oaCWguwmPk+fddfuw1uWZ01jIybowWHcY+R808P
PPnRXQ/Te9K2gKSpHgIbhb9CFd316/wong/JkQpb+fsv/8sjex+Ymir7T/Sn
fgOjM2slvEsAOxn4/DX0pd9vtAHz9G1/4XYtQVbJeyp8VBZPpuL+vIuEmUOR
6fzaUgf+DJ/S3Sbsr53Zrz7JlkxgXZ4+/5CMKjzjEXGTGaWuv1+XlHGhWEFb
ZFspTN0qY8lz0ZPfa4h4rpZf1zslb4seVwpv6J3Zju+9N2N+KvdY2bCjhys5
6XevZG8jKDzXmfIvl1ZuE/mhGz45ExIwp7lIeKhDQR6Agl5gubRym/CwSsYW
UJCLC+X8+hoUlE2fsD0hXOCJzJyTZHtwAheQgjPmVdJeGoXWx/Vp4xPCxWHh
boVw0S2o9bnSNxTGYlZNDvbQJ5j1ie23y+ehKT7ZX58zvkLauyHJwXyRqU/G
xZiMiyP+nHEFuxf0aedErc8ZvzGJYl7Up8ni8phQy8jmJlj7pG+4Q+xnfKro
p5TLZcJNHQrqc0Bk17cafEjGBlBwVtGXtCdysz+JYq7IgroMcwyg4Fzjw6KB
pYKqXz4/LlzUoSDDqFFx5oy1K771Am814BsdlkRrDW/NGIjchTc8wK44e7gp
iQK1Sdaksjd2WK9OA0DBOegxf87qU1w/HhT7WSJoDhRG5uyogfD2krlyN2IJ
PfcfRllNdmMgaIj3oz3O5xeLRg4kZR+UVueLBoaiNVnUdRr4qD8vflM5/7YM
cMm3HozK/e+rQcGqwPzInGnAvXdGz+8UDQzLObhkVWFE7j+YRDFLWlkQZcm/
QyfHcPbml8DdQuFGq5bDopE0CtiwmVPwlcPRhsObf8Awo9DH45gwDTivBo3x
rSlDYs0Lxc5uTqLoj1zkn6mcQzEYUVgm7A7vjrLfc/wU35rDd/lk1Qmjw90/
JNy+vzA2+qIM4b1US2LvdreToT/q09C4N4Td48/1bTl409ULz78e9E+ZA7rs
vUYfIAduAmZWaA7F/LPjE99jtCNTCd7XNRZlMRt3M5G+iM5ksT5anxINZmMM
Bh7GhBlBGNlzIhqTzmXzh4gmfJsGbzWztN3kcQ23OkaMzdTt0IB5j++eTG/W
bhBs5GaVBPekdDs3m7V/TtI/8X2ZaNaSky57nqg16N5EtyRiCyNnULDdUsKG
H5POgoHH1uaXVnx20Gf3mUP56rZ7jC+8Dc6O7S/GWOvTsCP/hjz3HqyZUZPh
HFybFzT8zk4GIvfh+xo+g66sh6RmdNhFblyEd+bd7/WbvTPPOsWzuwyT/cUM
yi5NRA2bBA6113jHuDXrNxMyM/ZvlpjGDPtbub5Mv6aFLq9fOzX9Ao9nsdvs
3jgwsc3ksm/QdKxrU5aFfkPywWli8CXQHhPTFGed2z/r1EzC5/od48hswTRm
XXe5B0R0LIm3Ds2PGaTDFX3GtyG6/7sh9fPYlrFu/+zOP6i4s6lG437tq7i3
SVLeGy0mvk1QrYSfVM/IK+7omBy1Lvg/potqFA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 213},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[327]=",
 CellID->92532786]
}, Open  ]],

Cell[TextData[{
 "The solution to this equation is given in terms of the derivatives of the \
Airy functions, ",
 Cell[BoxData[
  ButtonBox["AiryAiPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AiryAiPrime"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["AiryBiPrime",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AiryBiPrime"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->30648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", " ", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"a", "*", "x"}], "+", "b"}], " ", ")"}], ")"}], "*", 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], "-", 
     RowBox[{"a", "*", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], " ", "-", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{"a", "*", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"a", "*", "x"}], "+", "b"}], ")"}]}], ")"}], "^", "2"}], 
      "*", " ", 
      RowBox[{"y", "[", "x", "]"}]}]}], " ", "==", " ", "0"}], ",", " ", "y", 
   ",", " ", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[329]:=",
 CellID->9523],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{"AiryAiPrime", "[", 
         RowBox[{"b", "+", 
          RowBox[{"a", " ", "x"}]}], "]"}], " ", 
        RowBox[{"C", "[", "1", "]"}]}], "+", 
       RowBox[{
        RowBox[{"AiryBiPrime", "[", 
         RowBox[{"b", "+", 
          RowBox[{"a", " ", "x"}]}], "]"}], " ", 
        RowBox[{"C", "[", "2", "]"}]}]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{488, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[329]=",
 CellID->15260743]
}, Open  ]],

Cell[TextData[{
 "Here is the solution for ",
 StyleBox[ButtonBox["Bessel\[CloseCurlyQuote]s equation",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveIntroductionToODEs#14708"],
  FontSlant->"Italic"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "4"}], TraditionalForm]], "InlineMath"],
 ". Note that the solution is given in terms of Bessel functions of the first \
kind, ",
 Cell[BoxData[
  ButtonBox["BesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJ"]], "InlineFormula"],
 ", as well as those of the second kind, ",
 Cell[BoxData[
  ButtonBox["BesselY",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselY"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->{"BesselODE", "b:1.1.1.4"},
 CellID->24474],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "^", "2"}], "*", 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], " ", "+", " ", 
     RowBox[{"x", "*", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], " ", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"x", "^", "2"}], "-", "16"}], ")"}], "*", 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[330]:=",
 CellID->28633],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"BesselJ", "[", 
       RowBox[{"4", ",", "x"}], "]"}], " ", 
      RowBox[{"C", "[", "1", "]"}]}], "+", 
     RowBox[{
      RowBox[{"BesselY", "[", 
       RowBox[{"4", ",", "x"}], "]"}], " ", 
      RowBox[{"C", "[", "2", "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{327, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[330]=",
 CellID->58145652]
}, Open  ]],

Cell[TextData[{
 "Here is a plot of the ",
 Cell[BoxData[
  ButtonBox["BesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJ"]], "InlineFormula"],
 " functions for specific values of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->27200],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"BesselJ", "[", 
      RowBox[{"1", ",", "x"}], "]"}], ",", " ", 
     RowBox[{"BesselJ", "[", 
      RowBox[{"3", ",", "x"}], "]"}], ",", " ", 
     RowBox[{"BesselJ", "[", 
      RowBox[{"4", ",", "x"}], "]"}]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[331]:=",
 CellID->19854],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 219},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[331]=",
 CellID->608986217]
}, Open  ]],

Cell[TextData[{
 "Here is the general solution for ",
 StyleBox["Legendre\[CloseCurlyQuote]s equation ",
  FontSlant->"Italic"],
 "with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "7"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->25697],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"16", 
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"x", "^", "2"}]}], ")"}], "*", 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], "-", 
     RowBox[{"32", "x", "*", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], "+", 
     RowBox[{"21", "*", 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[332]:=",
 CellID->4919],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"LegendreP", "[", 
       RowBox[{
        FractionBox["3", "4"], ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"LegendreQ", "[", 
       RowBox[{
        FractionBox["3", "4"], ",", "x"}], "]"}]}]}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{364, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[332]=",
 CellID->281958835]
}, Open  ]],

Cell[TextData[{
 "These special functions can be expressed in terms of elementary functions \
for certain values",
 " ",
 "of their parameters. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " performs this conversion automatically wherever it is possible."
}], "Text",
 CellTags->"b:1.1.1.4",
 CellID->22578],

Cell["\<\
These are some of these expressions that are automatically converted.\
\>", "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->12931],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"BesselJ", "[", 
    RowBox[{
     RowBox[{"1", "/", "2"}], ",", "x"}], "]"}], ",", " ", 
   RowBox[{"LegendreP", "[", 
    RowBox[{"4", ",", "x"}], "]"}], ",", " ", 
   RowBox[{"HermiteH", "[", 
    RowBox[{"5", ",", "x"}], "]"}]}], "}"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[333]:=",
 CellID->32544],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    RowBox[{
     SqrtBox[
      FractionBox["2", "\[Pi]"]], " ", 
     RowBox[{"Sin", "[", "x", "]"}]}], 
    SqrtBox["x"]], ",", 
   RowBox[{
    FractionBox["1", "8"], " ", 
    RowBox[{"(", 
     RowBox[{"3", "-", 
      RowBox[{"30", " ", 
       SuperscriptBox["x", "2"]}], "+", 
      RowBox[{"35", " ", 
       SuperscriptBox["x", "4"]}]}], ")"}]}], ",", 
   RowBox[{
    RowBox[{"120", " ", "x"}], "-", 
    RowBox[{"160", " ", 
     SuperscriptBox["x", "3"]}], "+", 
    RowBox[{"32", " ", 
     SuperscriptBox["x", "5"]}]}]}], "}"}]], "Output",
 ImageSize->{362, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[333]=",
 CellID->340205258]
}, Open  ]],

Cell["\<\
As a result of these conversions, the solutions of certain ODEs can be \
partially expressed in terms of elementary functions. Hermite\
\[CloseCurlyQuote]s equation is one such ODE. \
\>", "Text",
 CellTags->"b:1.1.1.4",
 CellID->13487],

Cell[TextData[{
 "Here is the solution for Hermite\[CloseCurlyQuote]s equation with arbitrary \
",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->899],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "''"}], "[", "x", "]"}], "-", 
     RowBox[{"2", "x", "*", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}], "+", 
     RowBox[{"2", "n", "*", 
      RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.1.4",
 CellLabel->"In[334]:=",
 CellID->22525],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{
      RowBox[{"C", "[", "1", "]"}], " ", 
      RowBox[{"HermiteH", "[", 
       RowBox[{"n", ",", "x"}], "]"}]}], "+", 
     RowBox[{
      RowBox[{"C", "[", "2", "]"}], " ", 
      RowBox[{"Hypergeometric1F1", "[", 
       RowBox[{
        RowBox[{"-", 
         FractionBox["n", "2"]}], ",", 
        FractionBox["1", "2"], ",", 
        SuperscriptBox["x", "2"]}], "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{444, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.1.4",
 CellLabel->"Out[334]=",
 CellID->516744808]
}, Open  ]],

Cell[TextData[{
 "With ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " set to 5, the solution is given in terms of polynomials, exponentials, and \
",
 Cell[BoxData[
  ButtonBox["Erfi",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Erfi"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->5538],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Collect", "[", 
  RowBox[{
   RowBox[{"Simplify", "[", 
    RowBox[{"PowerExpand", "[", 
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "/.", 
      RowBox[{
       RowBox[{"DSolve", "[", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"y", "''"}], "[", "x", "]"}], "-", 
           RowBox[{"2", "*", "x", "*", 
            RowBox[{
             RowBox[{"y", "'"}], "[", "x", "]"}]}], "+", 
           RowBox[{"10", "*", 
            RowBox[{"y", "[", "x", "]"}]}]}], "\[Equal]", "0"}], ",", 
         RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}], "[", 
       RowBox[{"[", "1", "]"}], "]"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"C", "[", "1", "]"}], ",", 
     RowBox[{"C", "[", "2", "]"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[335]:=",
 CellID->28703],

Cell[BoxData[
 RowBox[{
  RowBox[{
   FractionBox["1", "8"], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"960", " ", "x"}], "-", 
     RowBox[{"1280", " ", 
      SuperscriptBox["x", "3"]}], "+", 
     RowBox[{"256", " ", 
      SuperscriptBox["x", "5"]}]}], ")"}], " ", 
   RowBox[{"C", "[", "1", "]"}]}], "+", 
  RowBox[{
   FractionBox["1", "8"], " ", 
   RowBox[{"C", "[", "2", "]"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"8", " ", 
      SuperscriptBox["\[ExponentialE]", 
       SuperscriptBox["x", "2"]]}], "-", 
     RowBox[{"18", " ", 
      SuperscriptBox["\[ExponentialE]", 
       SuperscriptBox["x", "2"]], " ", 
      SuperscriptBox["x", "2"]}], "+", 
     RowBox[{"4", " ", 
      SuperscriptBox["\[ExponentialE]", 
       SuperscriptBox["x", "2"]], " ", 
      SuperscriptBox["x", "4"]}], "+", 
     RowBox[{
      SqrtBox["\[Pi]"], " ", "x", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "15"}], "+", 
        RowBox[{"20", " ", 
         SuperscriptBox["x", "2"]}], "-", 
        RowBox[{"4", " ", 
         SuperscriptBox["x", "4"]}]}], ")"}], " ", 
      RowBox[{"Erfi", "[", "x", "]"}]}]}], ")"}]}]}]], "Output",
 ImageSize->{433, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[335]=",
 CellID->16645719]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->409979914],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Linear Second-Order Equations with Solutions Involving Special \
Functions - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/DSolveExactLinearSecondOrderODEs"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/\
DSolveLinearSecondOrderODEsWithRationalCoefficients"]}], "Text", FontFamily -> 
      "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 40.5790660}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "DSolve can find solutions for most of the standard linear second-order \
ODEs that occur in applied mathematics. Here is the solution for Airy's \
equation. Here is a plot that shows the oscillatory behavior of the Airy \
functions for large negative values of x.", "synonyms" -> {}, "title" -> 
    "Linear Second-Order Equations with Solutions Involving Special \
Functions", "type" -> "Tutorial", "uri" -> 
    "tutorial/DSolveLinearSecondOrderSpecialFunctionODEs"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "special_functions_in_solutions"->{
  Cell[1280, 43, 180, 4, 70, "Title",
   CellTags->{"special_functions_in_solutions", "c:22", "b:1.1.1.4"},
   CellID->3497]},
 "c:22"->{
  Cell[1280, 43, 180, 4, 70, "Title",
   CellTags->{"special_functions_in_solutions", "c:22", "b:1.1.1.4"},
   CellID->3497]},
 "b:1.1.1.4"->{
  Cell[1280, 43, 180, 4, 70, "Title",
   CellTags->{"special_functions_in_solutions", "c:22", "b:1.1.1.4"},
   CellID->3497],
  Cell[1463, 49, 292, 9, 70, "Text",
   CellTags->"b:1.1.1.4",
   CellID->23243],
  Cell[1758, 60, 179, 7, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->27649],
  Cell[1962, 71, 359, 12, 28, "Input",
   CellTags->"b:1.1.1.4",
   CellID->22015],
  Cell[2324, 85, 505, 17, 36, "Output",
   CellTags->"b:1.1.1.4",
   CellID->216730397],
  Cell[2844, 105, 250, 8, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->189],
  Cell[3119, 117, 426, 15, 65, "Input",
   CellTags->"b:1.1.1.4",
   CellID->16100],
  Cell[3548, 134, 9001, 152, 234, "Output",
   Evaluatable->False,
   CellTags->"b:1.1.1.4",
   CellID->92532786],
  Cell[12564, 289, 436, 15, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->30648],
  Cell[13025, 308, 779, 27, 70, "Input",
   CellTags->"b:1.1.1.4",
   CellID->9523],
  Cell[13807, 337, 710, 23, 36, "Output",
   CellTags->"b:1.1.1.4",
   CellID->15260743],
  Cell[14532, 363, 758, 24, 70, "MathCaption",
   CellTags->{"BesselODE", "b:1.1.1.4"},
   CellID->24474],
  Cell[15315, 391, 605, 20, 70, "Input",
   CellTags->"b:1.1.1.4",
   CellID->28633],
  Cell[15923, 413, 562, 19, 36, "Output",
   CellTags->"b:1.1.1.4",
   CellID->58145652],
  Cell[16500, 435, 326, 12, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->27200],
  Cell[16851, 451, 471, 15, 28, "Input",
   CellTags->"b:1.1.1.4",
   CellID->19854],
  Cell[17325, 468, 9600, 162, 240, "Output",
   Evaluatable->False,
   CellTags->"b:1.1.1.4",
   CellID->608986217],
  Cell[26940, 633, 290, 11, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->25697],
  Cell[27255, 648, 574, 19, 70, "Input",
   CellTags->"b:1.1.1.4",
   CellID->4919],
  Cell[27832, 669, 624, 22, 51, "Output",
   CellTags->"b:1.1.1.4",
   CellID->281958835],
  Cell[28471, 694, 315, 10, 70, "Text",
   CellTags->"b:1.1.1.4",
   CellID->22578],
  Cell[28789, 706, 140, 4, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->12931],
  Cell[28954, 714, 366, 12, 70, "Input",
   CellTags->"b:1.1.1.4",
   CellID->32544],
  Cell[29323, 728, 768, 28, 80, "Output",
   CellTags->"b:1.1.1.4",
   CellID->340205258],
  Cell[30106, 759, 246, 6, 70, "Text",
   CellTags->"b:1.1.1.4",
   CellID->13487],
  Cell[30355, 767, 224, 8, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->899],
  Cell[30604, 779, 460, 15, 70, "Input",
   CellTags->"b:1.1.1.4",
   CellID->22525],
  Cell[31067, 796, 693, 23, 51, "Output",
   CellTags->"b:1.1.1.4",
   CellID->516744808],
  Cell[31775, 822, 346, 13, 70, "MathCaption",
   CellTags->"b:1.1.1.4",
   CellID->5538]},
 "BesselODE"->{
  Cell[14532, 363, 758, 24, 70, "MathCaption",
   CellTags->{"BesselODE", "b:1.1.1.4"},
   CellID->24474]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"special_functions_in_solutions", 37119, 982},
 {"c:22", 37257, 986},
 {"b:1.1.1.4", 37400, 990},
 {"BesselODE", 40082, 1080}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 180, 4, 70, "Title",
 CellTags->{"special_functions_in_solutions", "c:22", "b:1.1.1.4"},
 CellID->3497],
Cell[1463, 49, 292, 9, 70, "Text",
 CellTags->"b:1.1.1.4",
 CellID->23243],
Cell[1758, 60, 179, 7, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->27649],
Cell[CellGroupData[{
Cell[1962, 71, 359, 12, 28, "Input",
 CellTags->"b:1.1.1.4",
 CellID->22015],
Cell[2324, 85, 505, 17, 36, "Output",
 CellTags->"b:1.1.1.4",
 CellID->216730397]
}, Open  ]],
Cell[2844, 105, 250, 8, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->189],
Cell[CellGroupData[{
Cell[3119, 117, 426, 15, 65, "Input",
 CellTags->"b:1.1.1.4",
 CellID->16100],
Cell[3548, 134, 9001, 152, 234, "Output",
 Evaluatable->False,
 CellTags->"b:1.1.1.4",
 CellID->92532786]
}, Open  ]],
Cell[12564, 289, 436, 15, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->30648],
Cell[CellGroupData[{
Cell[13025, 308, 779, 27, 70, "Input",
 CellTags->"b:1.1.1.4",
 CellID->9523],
Cell[13807, 337, 710, 23, 36, "Output",
 CellTags->"b:1.1.1.4",
 CellID->15260743]
}, Open  ]],
Cell[14532, 363, 758, 24, 70, "MathCaption",
 CellTags->{"BesselODE", "b:1.1.1.4"},
 CellID->24474],
Cell[CellGroupData[{
Cell[15315, 391, 605, 20, 70, "Input",
 CellTags->"b:1.1.1.4",
 CellID->28633],
Cell[15923, 413, 562, 19, 36, "Output",
 CellTags->"b:1.1.1.4",
 CellID->58145652]
}, Open  ]],
Cell[16500, 435, 326, 12, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->27200],
Cell[CellGroupData[{
Cell[16851, 451, 471, 15, 28, "Input",
 CellTags->"b:1.1.1.4",
 CellID->19854],
Cell[17325, 468, 9600, 162, 240, "Output",
 Evaluatable->False,
 CellTags->"b:1.1.1.4",
 CellID->608986217]
}, Open  ]],
Cell[26940, 633, 290, 11, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->25697],
Cell[CellGroupData[{
Cell[27255, 648, 574, 19, 70, "Input",
 CellTags->"b:1.1.1.4",
 CellID->4919],
Cell[27832, 669, 624, 22, 51, "Output",
 CellTags->"b:1.1.1.4",
 CellID->281958835]
}, Open  ]],
Cell[28471, 694, 315, 10, 70, "Text",
 CellTags->"b:1.1.1.4",
 CellID->22578],
Cell[28789, 706, 140, 4, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->12931],
Cell[CellGroupData[{
Cell[28954, 714, 366, 12, 70, "Input",
 CellTags->"b:1.1.1.4",
 CellID->32544],
Cell[29323, 728, 768, 28, 80, "Output",
 CellTags->"b:1.1.1.4",
 CellID->340205258]
}, Open  ]],
Cell[30106, 759, 246, 6, 70, "Text",
 CellTags->"b:1.1.1.4",
 CellID->13487],
Cell[30355, 767, 224, 8, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->899],
Cell[CellGroupData[{
Cell[30604, 779, 460, 15, 70, "Input",
 CellTags->"b:1.1.1.4",
 CellID->22525],
Cell[31067, 796, 693, 23, 51, "Output",
 CellTags->"b:1.1.1.4",
 CellID->516744808]
}, Open  ]],
Cell[31775, 822, 346, 13, 70, "MathCaption",
 CellTags->"b:1.1.1.4",
 CellID->5538],
Cell[CellGroupData[{
Cell[32146, 839, 866, 26, 70, "Input",
 CellID->28703],
Cell[33015, 867, 1299, 42, 81, "Output",
 CellID->16645719]
}, Open  ]],
Cell[CellGroupData[{
Cell[34351, 914, 72, 1, 70, "RelatedTutorialsSection",
 CellID->409979914],
Cell[34426, 917, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],
Cell[34630, 924, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

