(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     35198,       1188]
NotebookOptionsPosition[     26093,        877]
NotebookOutlinePosition[     31803,       1055]
CellTagsIndexPosition[     31710,       1049]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Nonlinear Second-Order ODEs", "Title",
 CellTags->{"c:26", "b:1.1.2"},
 CellID->6538],

Cell["The general form of a nonlinear second-order ODE is", "Text",
 CellTags->"b:1.1.2",
 CellID->31351],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"F", "(", 
    RowBox[{"x", ",", "y", ",", 
     RowBox[{
      SuperscriptBox["y", "\[Prime]"], "(", "x", ")"}], ",", 
     RowBox[{
      SuperscriptBox["y", "\[Prime]\[Prime]"], "(", "x", ")"}]}], ")"}], 
   "\[LongEqual]", "0."}], TraditionalForm]], "DisplayMath",
 CellTags->"b:1.1.2",
 CellID->19447],

Cell[TextData[{
 "For simplicity, assume that the equation can be solved for the \
highest-order derivative ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["y", "\[Prime]\[Prime]",
     MultilineFunction->None], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 " to give "
}], "Text",
 CellTags->"b:1.1.2",
 CellID->24208],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox["y", "\[Prime]\[Prime]"], "(", "x", ")"}], "\[LongEqual]", 
   
   RowBox[{
    RowBox[{"f", "(", 
     RowBox[{"x", ",", 
      RowBox[{"y", "(", "x", ")"}], ",", 
      RowBox[{
       SuperscriptBox["y", "\[Prime]"], "(", "x", ")"}]}], ")"}], "."}]}], 
  TraditionalForm]], "DisplayMath",
 CellTags->"b:1.1.2",
 CellID->9646],

Cell["\<\
There are a few classes of nonlinear second-order ODEs for which solutions \
can be easily found.\
\>", "Text",
 CellTags->"b:1.1.2",
 CellID->22276],

Cell[TextData[{
 "The first class consists of equations that do not explicitly depend on ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 "; that is, equations of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]\[Prime]",
      MultilineFunction->None], "(", "x", ")"}], "=", 
    RowBox[{"f", "(", 
     RowBox[{"x", ",", 
      RowBox[{
       SuperscriptBox["y", "\[Prime]",
        MultilineFunction->None], "(", "x", ")"}]}], ")"}]}], 
   TraditionalForm]], "InlineMath"],
 ". Such equations can be regarded as first-order ODEs in ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"u", "(", "x", ")"}], "=", 
    RowBox[{
     SuperscriptBox["y", "\[Prime]",
      MultilineFunction->None], "(", "x", ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Text",
 CellTags->"b:1.1.2",
 CellID->25759],

Cell["Here is an example of this type.", "MathCaption",
 CellTags->"b:1.1.2",
 CellID->30189],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]\[Prime]",
      MultilineFunction->None], "[", "x", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"5", " ", "x", " ", 
      RowBox[{
       SuperscriptBox["y", "\[Prime]",
        MultilineFunction->None], "[", "x", "]"}]}], "+", 
     SuperscriptBox[
      RowBox[{
       SuperscriptBox["y", "\[Prime]",
        MultilineFunction->None], "[", "x", "]"}], "2"]}]}]}], ";", 
  RowBox[{"sol", "=", 
   RowBox[{"DSolve", "[", 
    RowBox[{"eqn", ",", "y", ",", "x"}], "]"}]}]}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[1]:=",
 CellID->30422],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       RowBox[{"C", "[", "2", "]"}], "-", 
       RowBox[{"Log", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "10"}], " ", 
          RowBox[{"C", "[", "1", "]"}]}], "+", 
         RowBox[{
          SqrtBox[
           RowBox[{"10", " ", "\[Pi]"}]], " ", 
          RowBox[{"Erfi", "[", 
           RowBox[{
            SqrtBox[
             FractionBox["5", "2"]], " ", "x"}], "]"}]}]}], "]"}]}]}], 
     "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{426, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[1]=",
 CellID->513270269]
}, Open  ]],

Cell[TextData[{
 "As in the case of linear second-order ODEs, the solution depends on two \
arbitrary parameters ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["C",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/C"], "[", "1", "]"}]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["C",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/C"], "[", "2", "]"}]], "InlineFormula"],
 ". "
}], "Text",
 CellTags->"b:1.1.2",
 CellID->8486],

Cell["\<\
Here is a plot of the solution for a specific choice of parameters.\
\>", "MathCaption",
 CellTags->"b:1.1.2",
 CellID->3455],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "/.", "sol"}], "/.", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"C", "[", "1", "]"}], "\[Rule]", " ", 
        RowBox[{
         RowBox[{"-", "1"}], "/", "2"}]}], ",", " ", 
       RowBox[{
        RowBox[{"C", "[", "2", "]"}], "\[Rule]", " ", 
        RowBox[{
         RowBox[{"-", "1"}], "/", "8"}]}]}], "}"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "0.4"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[2]:=",
 CellID->2028],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJwVyXs40/0fx/HllMMokpyKiE4rukOhfCSUpIhI3RRFST+6Y1JppGRUqF3l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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{{-0.4, 1}, {-3.714687955618769, 0.8011771687039803}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 221},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[2]=",
 CellID->918253622]
}, Open  ]],

Cell["This verifies the solution.", "MathCaption",
 CellTags->"b:1.1.2",
 CellID->4860],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "/.", "sol"}], "//", "Simplify"}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[3]:=",
 CellID->29253],

Cell[BoxData[
 RowBox[{"{", "True", "}"}]], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[3]=",
 CellID->282667942]
}, Open  ]],

Cell[TextData[{
 "The second class of easily solvable nonlinear second-order equations \
consists of equations that do not depend explicitly on ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["y", "\[Prime]",
     MultilineFunction->None], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 "; that is, equations of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]\[Prime]",
      MultilineFunction->None], "(", "x", ")"}], "\[LongEqual]", 
    RowBox[{"f", "(", 
     RowBox[{"y", "(", "x", ")"}], ")"}]}], TraditionalForm]], "InlineMath"],
 ". These equations can be reduced to first-order ODEs with independent \
variable ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ". Inverse functions are needed to give the final solution for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellTags->"b:1.1.2",
 CellID->12455],

Cell["Here is an example of this type.", "MathCaption",
 CellTags->"b:1.1.2",
 CellID->14197],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"y", "''"}], "[", "x", "]"}], " ", "==", " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"3", "*", 
      RowBox[{"y", "[", "x", "]"}]}], "]"}]}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[4]:=",
 CellID->22106],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:1.1.2",
 CellLabel->"During evaluation of In[4]:=",
 CellID->84756745],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
     RowBox[{"Log", "[", 
      RowBox[{
       RowBox[{"-", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"-", 
           FractionBox["3", "2"]}], ")"}], 
         RowBox[{"1", "/", "3"}]]}], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"C", "[", "1", "]"}]}], "+", 
          RowBox[{
           RowBox[{"C", "[", "1", "]"}], " ", 
           SuperscriptBox[
            RowBox[{"Tanh", "[", 
             RowBox[{
              FractionBox["3", "2"], " ", 
              SqrtBox[
               RowBox[{
                RowBox[{"C", "[", "1", "]"}], " ", 
                SuperscriptBox[
                 RowBox[{"(", 
                  RowBox[{"x", "+", 
                   RowBox[{"C", "[", "2", "]"}]}], ")"}], "2"]}]]}], "]"}], 
            "2"]}]}], ")"}], 
        RowBox[{"1", "/", "3"}]]}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
     RowBox[{"Log", "[", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox["3", "2"], ")"}], 
        RowBox[{"1", "/", "3"}]], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"C", "[", "1", "]"}]}], "+", 
          RowBox[{
           RowBox[{"C", "[", "1", "]"}], " ", 
           SuperscriptBox[
            RowBox[{"Tanh", "[", 
             RowBox[{
              FractionBox["3", "2"], " ", 
              SqrtBox[
               RowBox[{
                RowBox[{"C", "[", "1", "]"}], " ", 
                SuperscriptBox[
                 RowBox[{"(", 
                  RowBox[{"x", "+", 
                   RowBox[{"C", "[", "2", "]"}]}], ")"}], "2"]}]]}], "]"}], 
            "2"]}]}], ")"}], 
        RowBox[{"1", "/", "3"}]]}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
     RowBox[{"Log", "[", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], 
        RowBox[{"2", "/", "3"}]], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox["3", "2"], ")"}], 
        RowBox[{"1", "/", "3"}]], " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", 
           RowBox[{"C", "[", "1", "]"}]}], "+", 
          RowBox[{
           RowBox[{"C", "[", "1", "]"}], " ", 
           SuperscriptBox[
            RowBox[{"Tanh", "[", 
             RowBox[{
              FractionBox["3", "2"], " ", 
              SqrtBox[
               RowBox[{
                RowBox[{"C", "[", "1", "]"}], " ", 
                SuperscriptBox[
                 RowBox[{"(", 
                  RowBox[{"x", "+", 
                   RowBox[{"C", "[", "2", "]"}]}], ")"}], "2"]}]]}], "]"}], 
            "2"]}]}], ")"}], 
        RowBox[{"1", "/", "3"}]]}], "]"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{490, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[4]=",
 CellID->279869398]
}, Open  ]],

Cell[TextData[{
 "The third class consists of equations that do not depend explicitly on ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 "; that is, equations of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]\[Prime]",
      MultilineFunction->None], "(", "x", ")"}], "=", 
    RowBox[{"f", "(", 
     RowBox[{
      RowBox[{"y", "(", "x", ")"}], ",", 
      RowBox[{
       SuperscriptBox["y", "\[Prime]",
        MultilineFunction->None], "(", "x", ")"}]}], ")"}]}], 
   TraditionalForm]], "InlineMath"],
 ". Once again, these equations can be reduced to first-order ODEs with",
 " ",
 "independent variable ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellTags->"b:1.1.2",
 CellID->17296],

Cell[TextData[{
 "This example is based on (equation 6.40, page 550 of [",
 ButtonBox["K59",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveReferences#21388"],
 "]). The underlying first-order ODE is an Abel equation. The hyperbolic \
functions in the solution result from the automatic simplification of Bessel \
functions."
}], "MathCaption",
 CellTags->"b:1.1.2",
 CellID->26281],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "/.", 
    RowBox[{
     RowBox[{"DSolve", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "x", "]"}], " ", "==", 
        RowBox[{
         RowBox[{"3", "*", 
          RowBox[{"y", "[", "x", "]"}], "*", 
          RowBox[{
           RowBox[{"y", "'"}], "[", "x", "]"}]}], " ", "+", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"3", 
            RowBox[{
             RowBox[{"y", "[", "x", "]"}], "^", "2"}]}], "+", 
           RowBox[{"4", "*", 
            RowBox[{"y", "[", "x", "]"}]}], " ", "+", "1"}], ")"}]}]}], ",", 
       " ", 
       RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}], "[", 
     RowBox[{"[", "1", "]"}], "]"}]}], "//", "Simplify"}]}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[5]:=",
 CellID->26021],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"\[ImaginaryI]", " ", 
        SqrtBox["6"], " ", 
        SqrtBox[
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{
           RowBox[{"-", "2"}], " ", "x"}]]], " ", 
        SqrtBox[
         RowBox[{"C", "[", "1", "]"}]]}], "-", 
       RowBox[{"6", " ", 
        RowBox[{"C", "[", "2", "]"}]}]}], ")"}], " ", 
     RowBox[{"Cosh", "[", 
      RowBox[{
       SqrtBox[
        FractionBox["3", "2"]], " ", 
       SqrtBox[
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "2"}], " ", "x"}]]], " ", 
       SqrtBox[
        RowBox[{"C", "[", "1", "]"}]]}], "]"}]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "3"}], " ", "\[ImaginaryI]"}], "+", 
       RowBox[{"2", " ", 
        SqrtBox["6"], " ", 
        SqrtBox[
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{
           RowBox[{"-", "2"}], " ", "x"}]]], " ", 
        SqrtBox[
         RowBox[{"C", "[", "1", "]"}]], " ", 
        RowBox[{"C", "[", "2", "]"}]}]}], ")"}], " ", 
     RowBox[{"Sinh", "[", 
      RowBox[{
       SqrtBox[
        FractionBox["3", "2"]], " ", 
       SqrtBox[
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "2"}], " ", "x"}]]], " ", 
       SqrtBox[
        RowBox[{"C", "[", "1", "]"}]]}], "]"}]}]}], ")"}], "/", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"6", " ", 
     RowBox[{"C", "[", "2", "]"}], " ", 
     RowBox[{"Cosh", "[", 
      RowBox[{
       SqrtBox[
        FractionBox["3", "2"]], " ", 
       SqrtBox[
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "2"}], " ", "x"}]]], " ", 
       SqrtBox[
        RowBox[{"C", "[", "1", "]"}]]}], "]"}]}], "+", 
    RowBox[{"3", " ", "\[ImaginaryI]", " ", 
     RowBox[{"Sinh", "[", 
      RowBox[{
       SqrtBox[
        FractionBox["3", "2"]], " ", 
       SqrtBox[
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "2"}], " ", "x"}]]], " ", 
       SqrtBox[
        RowBox[{"C", "[", "1", "]"}]]}], "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{435, 128},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[5]=",
 CellID->190845684]
}, Open  ]],

Cell[TextData[{
 "The fourth class consists of equations that are homogeneous in some or all \
of the variables ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["y", "\[Prime]\[Prime]",
     MultilineFunction->None], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 ". There are several possibilities in this case, but here only the following \
simple example is considered."
}], "Text",
 CellTags->"b:1.1.2",
 CellID->15947],

Cell[TextData[{
 "In this equation, each term has a total degree of 2 in the variables ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "[", "x", "]"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["y", "\[Prime]",
     MultilineFunction->None], "[", "x", "]"}], TraditionalForm]], 
  "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["y", "\[Prime]\[Prime]",
     MultilineFunction->None], "[", "x", "]"}], TraditionalForm]], 
  "InlineMath"],
 ". This equation can be solved by transforming it to a first-order ODE. "
}], "MathCaption",
 CellTags->"b:1.1.2",
 CellID->27124],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"7", "*", 
      RowBox[{"y", "[", "x", "]"}], "*", 
      RowBox[{
       RowBox[{"y", "''"}], "[", "x", "]"}]}], " ", "-", 
     RowBox[{"11", "*", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], "^", "2"}]}]}], "\[Equal]", 
    "0"}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", " ", "x"}], "]"}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[6]:=",
 CellID->18318],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
    FractionBox[
     RowBox[{"C", "[", "2", "]"}], 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{"4", " ", "x"}], "+", 
        RowBox[{"7", " ", 
         RowBox[{"C", "[", "1", "]"}]}]}], ")"}], 
      RowBox[{"7", "/", "4"}]]]}], "}"}], "}"}]], "Output",
 ImageSize->{178, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[6]=",
 CellID->489637830]
}, Open  ]],

Cell["\<\
The fifth and final class of easily solvable nonlinear second-order ODEs \
consists of equations that are exact or can be made to be exact using an \
integrating factor.\
\>", "Text",
 CellTags->"b:1.1.2",
 CellID->11376],

Cell[TextData[{
 "Here is an example of this type, based on (equation 6.51, page 554 of [",
 ButtonBox["K59",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveReferences#21388"],
 "])."
}], "MathCaption",
 CellTags->"b:1.1.2",
 CellID->1774],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "''"}], "[", "x", "]"}], " ", "+", 
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "*", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "x", "]"}], "^", "2"}]}], "-", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "*", 
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}]}]}], "\[Equal]", "0"}]}], 
  ";"}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[7]:=",
 CellID->29859],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqn", ",", " ", 
    RowBox[{"y", "[", "x", "]"}], ",", " ", "x"}], "]"}]}]], "Input",
 CellTags->"b:1.1.2",
 CellLabel->"In[8]:=",
 CellID->24998],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:1.1.2",
 CellLabel->"During evaluation of In[8]:=",
 CellID->868463575],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "x", "]"}], "\[Rule]", 
    RowBox[{
     RowBox[{"-", "\[ImaginaryI]"}], " ", 
     SqrtBox["2"], " ", 
     RowBox[{"InverseErf", "[", 
      FractionBox[
       RowBox[{"\[ImaginaryI]", " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"3", " ", 
           SqrtBox[
            RowBox[{"2", " ", "\[Pi]"}]], " ", 
           RowBox[{"C", "[", "2", "]"}]}], "-", 
          FractionBox[
           RowBox[{
            SuperscriptBox["3", 
             RowBox[{"1", "/", "3"}]], " ", 
            SqrtBox[
             RowBox[{"2", " ", "\[Pi]"}]], " ", 
            SuperscriptBox[
             RowBox[{"(", 
              RowBox[{"-", 
               SuperscriptBox["x", "3"]}], ")"}], 
             RowBox[{"2", "/", "3"}]], " ", 
            RowBox[{"C", "[", "1", "]"}], " ", 
            RowBox[{"Gamma", "[", 
             RowBox[{
              FractionBox["1", "3"], ",", 
              RowBox[{"-", 
               FractionBox[
                SuperscriptBox["x", "3"], "3"]}]}], "]"}]}], 
           SuperscriptBox["x", "2"]]}], ")"}]}], 
       RowBox[{"3", " ", "\[Pi]"}]], "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{512, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:1.1.2",
 CellLabel->"Out[8]=",
 CellID->286868332]
}, Open  ]],

Cell["\<\
It is important to note that the solutions to fairly simple-looking nonlinear \
ODEs can be complicated. Verifying and applying the solutions in such cases \
is a difficult problem.\
\>", "Text",
 CellTags->"b:1.1.2",
 CellID->6922],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->28277991],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Nonlinear Second-Order ODEs - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/DSolveLinearSecondOrderInhomogeneousEquations"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/DSolveIntroductionToHigherOrderODEs"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 42.2666092}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The general form of a nonlinear second-order ODE is For simplicity, \
assume that the equation can be solved for the highest-order derivative \
y^,,(x) to give There are a few classes of nonlinear second-order ODEs for \
which solutions can be easily found.", "synonyms" -> {}, "title" -> 
    "Nonlinear Second-Order ODEs", "type" -> "Tutorial", "uri" -> 
    "tutorial/DSolveNonlinearSecondOrderODEs"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:26"->{
  Cell[1280, 43, 91, 2, 70, "Title",
   CellTags->{"c:26", "b:1.1.2"},
   CellID->6538]},
 "b:1.1.2"->{
  Cell[1280, 43, 91, 2, 70, "Title",
   CellTags->{"c:26", "b:1.1.2"},
   CellID->6538],
  Cell[1374, 47, 105, 2, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->31351],
  Cell[1482, 51, 352, 11, 70, "DisplayMath",
   CellTags->"b:1.1.2",
   CellID->19447],
  Cell[1837, 64, 340, 12, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->24208],
  Cell[2180, 78, 394, 14, 70, "DisplayMath",
   CellTags->"b:1.1.2",
   CellID->9646],
  Cell[2577, 94, 159, 5, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->22276],
  Cell[2739, 101, 907, 30, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->25759],
  Cell[3649, 133, 93, 2, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->30189],
  Cell[3767, 139, 655, 21, 28, "Input",
   CellTags->"b:1.1.2",
   CellID->30422],
  Cell[4425, 162, 781, 27, 60, "Output",
   CellTags->"b:1.1.2",
   CellID->513270269],
  Cell[5221, 192, 452, 17, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->8486],
  Cell[5676, 211, 135, 4, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->3455],
  Cell[5836, 219, 667, 22, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->2028],
  Cell[6506, 243, 3251, 64, 242, "Output",
   CellTags->"b:1.1.2",
   CellID->918253622],
  Cell[9772, 310, 87, 2, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->4860],
  Cell[9884, 316, 146, 5, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->29253],
  Cell[10033, 323, 205, 7, 36, "Output",
   CellTags->"b:1.1.2",
   CellID->282667942],
  Cell[10253, 333, 1045, 32, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->12455],
  Cell[11301, 367, 93, 2, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->14197],
  Cell[11419, 373, 364, 12, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->22106],
  Cell[11786, 387, 518, 10, 70, "Message",
   CellTags->"b:1.1.2",
   CellID->84756745],
  Cell[12307, 399, 3188, 101, 122, "Output",
   CellTags->"b:1.1.2",
   CellID->279869398],
  Cell[15510, 503, 803, 26, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->17296],
  Cell[16316, 531, 393, 10, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->26281],
  Cell[16734, 545, 895, 28, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->26021],
  Cell[17632, 575, 2361, 81, 149, "Output",
   CellTags->"b:1.1.2",
   CellID->190845684],
  Cell[20008, 659, 617, 20, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->15947],
  Cell[20628, 681, 658, 22, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->27124],
  Cell[21311, 707, 493, 17, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->18318],
  Cell[21807, 726, 542, 19, 54, "Output",
   CellTags->"b:1.1.2",
   CellID->489637830],
  Cell[22364, 748, 231, 6, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->11376],
  Cell[22598, 756, 250, 8, 70, "MathCaption",
   CellTags->"b:1.1.2",
   CellID->1774],
  Cell[22851, 766, 513, 19, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->29859],
  Cell[23389, 789, 221, 7, 70, "Input",
   CellTags->"b:1.1.2",
   CellID->24998],
  Cell[23613, 798, 519, 10, 70, "Message",
   CellTags->"b:1.1.2",
   CellID->868463575],
  Cell[24135, 810, 1371, 42, 81, "Output",
   CellTags->"b:1.1.2",
   CellID->286868332],
  Cell[25521, 855, 242, 6, 70, "Text",
   CellTags->"b:1.1.2",
   CellID->6922]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:26", 28412, 930},
 {"b:1.1.2", 28516, 934}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 91, 2, 70, "Title",
 CellTags->{"c:26", "b:1.1.2"},
 CellID->6538],
Cell[1374, 47, 105, 2, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->31351],
Cell[1482, 51, 352, 11, 70, "DisplayMath",
 CellTags->"b:1.1.2",
 CellID->19447],
Cell[1837, 64, 340, 12, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->24208],
Cell[2180, 78, 394, 14, 70, "DisplayMath",
 CellTags->"b:1.1.2",
 CellID->9646],
Cell[2577, 94, 159, 5, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->22276],
Cell[2739, 101, 907, 30, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->25759],
Cell[3649, 133, 93, 2, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->30189],
Cell[CellGroupData[{
Cell[3767, 139, 655, 21, 28, "Input",
 CellTags->"b:1.1.2",
 CellID->30422],
Cell[4425, 162, 781, 27, 60, "Output",
 CellTags->"b:1.1.2",
 CellID->513270269]
}, Open  ]],
Cell[5221, 192, 452, 17, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->8486],
Cell[5676, 211, 135, 4, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->3455],
Cell[CellGroupData[{
Cell[5836, 219, 667, 22, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->2028],
Cell[6506, 243, 3251, 64, 242, "Output",
 CellTags->"b:1.1.2",
 CellID->918253622]
}, Open  ]],
Cell[9772, 310, 87, 2, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->4860],
Cell[CellGroupData[{
Cell[9884, 316, 146, 5, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->29253],
Cell[10033, 323, 205, 7, 36, "Output",
 CellTags->"b:1.1.2",
 CellID->282667942]
}, Open  ]],
Cell[10253, 333, 1045, 32, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->12455],
Cell[11301, 367, 93, 2, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->14197],
Cell[CellGroupData[{
Cell[11419, 373, 364, 12, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->22106],
Cell[11786, 387, 518, 10, 70, "Message",
 CellTags->"b:1.1.2",
 CellID->84756745],
Cell[12307, 399, 3188, 101, 122, "Output",
 CellTags->"b:1.1.2",
 CellID->279869398]
}, Open  ]],
Cell[15510, 503, 803, 26, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->17296],
Cell[16316, 531, 393, 10, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->26281],
Cell[CellGroupData[{
Cell[16734, 545, 895, 28, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->26021],
Cell[17632, 575, 2361, 81, 149, "Output",
 CellTags->"b:1.1.2",
 CellID->190845684]
}, Open  ]],
Cell[20008, 659, 617, 20, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->15947],
Cell[20628, 681, 658, 22, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->27124],
Cell[CellGroupData[{
Cell[21311, 707, 493, 17, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->18318],
Cell[21807, 726, 542, 19, 54, "Output",
 CellTags->"b:1.1.2",
 CellID->489637830]
}, Open  ]],
Cell[22364, 748, 231, 6, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->11376],
Cell[22598, 756, 250, 8, 70, "MathCaption",
 CellTags->"b:1.1.2",
 CellID->1774],
Cell[22851, 766, 513, 19, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->29859],
Cell[CellGroupData[{
Cell[23389, 789, 221, 7, 70, "Input",
 CellTags->"b:1.1.2",
 CellID->24998],
Cell[23613, 798, 519, 10, 70, "Message",
 CellTags->"b:1.1.2",
 CellID->868463575],
Cell[24135, 810, 1371, 42, 81, "Output",
 CellTags->"b:1.1.2",
 CellID->286868332]
}, Open  ]],
Cell[25521, 855, 242, 6, 70, "Text",
 CellTags->"b:1.1.2",
 CellID->6922],
Cell[CellGroupData[{
Cell[25788, 865, 71, 1, 70, "RelatedTutorialsSection",
 CellID->28277991],
Cell[25862, 868, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],
Cell[26066, 875, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

