(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     34528,       1150]
NotebookOptionsPosition[     24227,        787]
NotebookOutlinePosition[     30497,        987]
CellTagsIndexPosition[     30406,        981]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Differential Equation Solving with DSolve" :> 
         Documentation`HelpLookup["paclet:tutorial/DSolveOverview"]}, #, 2]& ,
        Evaluator -> Automatic], {
      1->"\<\"Differential Equation Solving with DSolve\"\>"}, "\<\"related \
tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Is the Problem Well-Posed?", "Title",
 CellTags->{"c:57", "b:5.6"},
 CellID->13049],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " returns a general solution for a problem if no initial or boundary \
conditions are specified."
}], "Text",
 CellTags->"b:5.6",
 CellID->14118],

Cell["The general solution to this equation is returned.", "MathCaption",
 CellTags->"b:5.6",
 CellID->24065],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"y", "'"}], "[", "x", "]"}], " ", "\[Equal]", 
    RowBox[{"1", "-", 
     RowBox[{"y", "[", "x", "]"}]}]}], ",", " ", "y", ",", "x"}], 
  "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[1]:=",
 CellID->26363],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{"1", "+", 
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"-", "x"}]], " ", 
        RowBox[{"C", "[", "1", "]"}]}]}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{226, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[1]=",
 CellID->115728059]
}, Open  ]],

Cell[TextData[{
 "However, if initial or boundary conditions are specified, the output from \
",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " must satisfy both the underlying differential equation as well as the \
given conditions."
}], "Text",
 CellTags->"b:5.6",
 CellID->28773],

Cell["Here is an example with a boundary condition.", "MathCaption",
 CellTags->"b:5.6",
 CellID->18470],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqns", " ", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}], " ", "\[Equal]", 
      RowBox[{"1", " ", "-", 
       RowBox[{"y", "[", "x", "]"}]}]}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "3", "]"}], "\[Equal]", "5"}]}], "}"}]}], 
  ";"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[2]:=",
 CellID->29731],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"DSolve", "[", 
   RowBox[{"eqns", ",", " ", "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[3]:=",
 CellID->6747],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"-", "x"}]], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"4", " ", 
          SuperscriptBox["\[ExponentialE]", "3"]}], "+", 
         SuperscriptBox["\[ExponentialE]", "x"]}], ")"}]}]}], "]"}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{240, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[3]=",
 CellID->65833916]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqns", "/.", "sol"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[4]:=",
 CellID->5869],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"True", ",", "True"}], "}"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[4]=",
 CellID->410059729]
}, Open  ]],

Cell[TextData[{
 "In such cases, it is useful to check whether ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " has been asked a reasonable question\[LongDash]in other words, to check \
whether the problem is well-posed. An initial or boundary value problem is \
said to be ",
 StyleBox["well-posed",
  FontSlant->"Italic"],
 " if a solution for it is guaranteed to exist in some well-known class of \
functions (for example, analytic functions), if the solution is unique, and \
if the solution depends continuously on the data. Given an ODE of order ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " (or a system of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " first-order equations) and ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " initial conditions, there are standard existence and uniqueness theorems \
that show that the problem is well-posed under a specified set of conditions. \
The right-hand side of the first-order linear ODE in the previous example is \
a polynomial in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "[", "x", "]"}], TraditionalForm]], "InlineMath"],
 " and hence infinitely differentiable. This is sufficient to apply Picard\
\[CloseCurlyQuote]s existence and uniqueness theorem, which only requires \
that the right-hand side be Lipschitz-continuous. "
}], "Text",
 CellTags->"b:5.6",
 CellID->10498],

Cell[TextData[{
 "Most problems that arise in practice are well-posed since they are derived \
from sound theoretical principles. However, as a note of caution, the \
following are examples where ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " might have difficulty finding a satisfactory solution to the problem."
}], "Text",
 CellTags->"b:5.6",
 CellID->7103],

Cell["\<\
Here is the solution to a first-order ODE in which the right-hand side fails \
to satisfy the Lipschitz condition around 0.\
\>", "MathCaption",
 CellTags->"b:5.6",
 CellID->27352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"generalsol", " ", "=", " ", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}], " ", "\[Equal]", " ", 
      RowBox[{"1", "/", 
       RowBox[{"y", "[", "x", "]"}]}]}], "}"}], ",", " ", "y", ",", "x"}], 
   "]"}]}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[5]:=",
 CellID->25679],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"y", "\[Rule]", 
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{
        RowBox[{"-", 
         SqrtBox["2"]}], " ", 
        SqrtBox[
         RowBox[{"x", "+", 
          RowBox[{"C", "[", "1", "]"}]}]]}]}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", "\[Rule]", 
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{
        SqrtBox["2"], " ", 
        SqrtBox[
         RowBox[{"x", "+", 
          RowBox[{"C", "[", "1", "]"}]}]]}]}], "]"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{498, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[5]=",
 CellID->39286122]
}, Open  ]],

Cell["The general solution has two branches.", "MathCaption",
 CellTags->"b:5.6",
 CellID->13043],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "[", "x", "]"}], "/.", "generalsol"}], "/.", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"C", "[", "1", "]"}], " ", "\[Rule]", " ", "1"}], "}"}]}], 
    "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[6]:=",
 CellID->14522],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 223},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[6]=",
 CellID->11483804]
}, Open  ]],

Cell[TextData[{
 "This initial value problem is well posed in a region around the initial \
condition and hence ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " succeeds in picking out the correct branch for the given initial \
condition. "
}], "MathCaption",
 CellTags->"b:5.6",
 CellID->1563],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "x", "]"}], " ", "\[Equal]", " ", 
      RowBox[{"1", "/", 
       RowBox[{"y", "[", "x", "]"}]}]}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", " ", "y",
    ",", "x"}], "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[7]:=",
 CellID->25834],

Cell[BoxData[
 RowBox[{
  RowBox[{"DSolve", "::", "\<\"bvnul\"\>"}], ":", 
  " ", "\<\"For some branches of the general solution, the given boundary \
conditions lead to an empty solution. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\
\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/DSolve/bvnul\\\", ButtonNote -> \
\\\"DSolve::bvnul\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.6",
 CellLabel->"During evaluation of In[7]:=",
 CellID->460469657],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "x", "}"}], ",", 
      SqrtBox[
       RowBox[{"1", "+", 
        RowBox[{"2", " ", "x"}]}]]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{211, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[7]=",
 CellID->496062216]
}, Open  ]],

Cell["\<\
Here is a second-order ODE. The boundary conditions do not allow any solution \
to this problem.\
\>", "MathCaption",
 CellTags->"b:5.6",
 CellID->7240],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "x", "]"}], "+", 
       RowBox[{"y", "[", "x", "]"}]}], "\[Equal]", "0"}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "Pi", "]"}], "\[Equal]", "3"}]}], "}"}], ",", " ", 
   RowBox[{"y", "[", "x", "]"}], ",", "x"}], "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[8]:=",
 CellID->21911],

Cell[BoxData[
 RowBox[{
  RowBox[{"DSolve", "::", "\<\"bvnul\"\>"}], ":", 
  " ", "\<\"For some branches of the general solution, the given boundary \
conditions lead to an empty solution. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\
\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/DSolve/bvnul\\\", ButtonNote -> \
\\\"DSolve::bvnul\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.6",
 CellLabel->"During evaluation of In[8]:=",
 CellID->165520702],

Cell[BoxData[
 RowBox[{"{", "}"}]], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[8]=",
 CellID->74575261]
}, Open  ]],

Cell[TextData[{
 "In this example, ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " returns a pair of solutions. As the table shows, the first solution is \
only valid for values of ",
 StyleBox["x", "MR"],
 " greater than or equal to 2. "
}], "MathCaption",
 CellTags->"b:5.6",
 CellID->28580],

Cell[BoxData[
 RowBox[{
  RowBox[{"eqn", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"y", "'"}], "[", "x", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"y", "[", "x", "]"}], "^", 
     RowBox[{"(", 
      RowBox[{"1", "/", "2"}], ")"}]}]}]}], ";"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[9]:=",
 CellID->15477],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"DSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqn", ",", " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", " ", 
    "y", ",", "x"}], "]"}]}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[10]:=",
 CellID->27616],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"y", "\[Rule]", 
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{
        FractionBox["1", "4"], " ", 
        RowBox[{"(", 
         RowBox[{"4", "-", 
          RowBox[{"4", " ", "x"}], "+", 
          SuperscriptBox["x", "2"]}], ")"}]}]}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", "\[Rule]", 
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", "x", "}"}], ",", 
       RowBox[{
        FractionBox["1", "4"], " ", 
        RowBox[{"(", 
         RowBox[{"4", "+", 
          RowBox[{"4", " ", "x"}], "+", 
          SuperscriptBox["x", "2"]}], ")"}]}]}], "]"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{488, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[10]=",
 CellID->25609640]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"eqn", "/.", "sol"}], "/.", 
    RowBox[{"{", 
     RowBox[{"x", "\[Rule]", " ", "i"}], "}"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"i", ",", " ", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[11]:=",
 CellID->19178],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"False", ",", "True"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"False", ",", "True"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"True", ",", "True"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"True", ",", "True"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"True", ",", "True"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"True", ",", "True"}], "}"}]}], "}"}]], "Output",
 ImageSize->{576, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[11]=",
 CellID->63028327]
}, Open  ]],

Cell[TextData[{
 "Finally, it is possible that a problem has a solution, but that ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " fails to find it because the general solution is in implicit form or \
involves higher transcendental functions. "
}], "Text",
 CellTags->"b:5.6",
 CellID->14615],

Cell["\<\
In this example, a solution is available only after inverting the roles of \
the dependent and independent variables.\
\>", "MathCaption",
 CellTags->"b:5.6",
 CellID->22348],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"y", "'"}], "[", "x", "]"}], " ", "\[Equal]", 
     RowBox[{"1", "/", 
      RowBox[{"(", 
       RowBox[{"x", "-", 
        RowBox[{"y", "[", "x", "]"}]}], ")"}]}]}], "&&", 
    RowBox[{
     RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], ",", " ", "y", ",", 
   "x"}], "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[12]:=",
 CellID->26798],

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used. Values may be lost for \
multivalued inverses. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InverseFunction/ifun\\\", ButtonNote -> \\\
\"InverseFunction::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.6",
 CellLabel->"During evaluation of In[12]:=",
 CellID->689397760],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.6",
 CellLabel->"During evaluation of In[12]:=",
 CellID->613871284],

Cell[BoxData[
 RowBox[{
  RowBox[{"DSolve", "::", "\<\"bvnul\"\>"}], ":", 
  " ", "\<\"For some branches of the general solution, the given boundary \
conditions lead to an empty solution. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\
\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/DSolve/bvnul\\\", ButtonNote -> \
\\\"DSolve::bvnul\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->"b:5.6",
 CellLabel->"During evaluation of In[12]:=",
 CellID->129143153],

Cell[BoxData[
 RowBox[{"{", "}"}]], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[12]=",
 CellID->489650356]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "'"}], "[", "y", "]"}], " ", "\[Equal]", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"x", "[", "y", "]"}], "-", "y"}], ")"}]}], "&&", 
    RowBox[{
     RowBox[{"x", "[", "1", "]"}], "\[Equal]", "0"}]}], ",", " ", "x", ",", 
   "y"}], "]"}]], "Input",
 CellTags->"b:5.6",
 CellLabel->"In[13]:=",
 CellID->1662],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "y", "}"}], ",", 
      FractionBox[
       RowBox[{"\[ExponentialE]", "-", 
        RowBox[{"2", " ", 
         SuperscriptBox["\[ExponentialE]", "y"]}], "+", 
        RowBox[{"\[ExponentialE]", " ", "y"}]}], "\[ExponentialE]"]}], 
     "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{234, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:5.6",
 CellLabel->"Out[13]=",
 CellID->318730300]
}, Open  ]],

Cell[TextData[{
 "This concludes the discussion of the basic principles for effectively \
working with ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ". See the list of \"",
 ButtonBox["references\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/DSolveReferences"],
 " that were found to be useful either during the development of ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " or during the preparation of this documentation."
}], "Text",
 CellTags->"b:5.6",
 CellID->28262],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->35449754],

Cell[TextData[ButtonBox["Differential Equation Solving with DSolve",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DSolveOverview"]], "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Is the Problem Well-Posed? - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Differential Equation Solving with DSolve", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> "paclet:tutorial/DSolveOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/DSolveSymbolicAndInexactQuantities"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DSolveReferences"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 34, 56.5638502}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "DSolve returns a general solution for a problem if no initial or \
boundary conditions are specified. The general solution to this equation is \
returned. However, if initial or boundary conditions are specified, the \
output from DSolve must satisfy both the underlying differential equation as \
well as the given conditions.", "synonyms" -> {}, "title" -> 
    "Is the Problem Well-Posed?", "type" -> "Tutorial", "uri" -> 
    "tutorial/DSolveWellPosedness"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:57"->{
  Cell[1280, 43, 89, 2, 70, "Title",
   CellTags->{"c:57", "b:5.6"},
   CellID->13049]},
 "b:5.6"->{
  Cell[1280, 43, 89, 2, 70, "Title",
   CellTags->{"c:57", "b:5.6"},
   CellID->13049],
  Cell[1372, 47, 277, 9, 70, "Text",
   CellTags->"b:5.6",
   CellID->14118],
  Cell[1652, 58, 109, 2, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->24065],
  Cell[1786, 64, 302, 11, 28, "Input",
   CellTags->"b:5.6",
   CellID->26363],
  Cell[2091, 77, 502, 17, 36, "Output",
   CellTags->"b:5.6",
   CellID->115728059],
  Cell[2608, 97, 353, 11, 70, "Text",
   CellTags->"b:5.6",
   CellID->28773],
  Cell[2964, 110, 104, 2, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->18470],
  Cell[3071, 114, 414, 15, 70, "Input",
   CellTags->"b:5.6",
   CellID->29731],
  Cell[3510, 133, 194, 6, 28, "Input",
   CellTags->"b:5.6",
   CellID->6747],
  Cell[3707, 141, 618, 21, 39, "Output",
   CellTags->"b:5.6",
   CellID->65833916],
  Cell[4362, 167, 113, 4, 28, "Input",
   CellTags->"b:5.6",
   CellID->5869],
  Cell[4478, 173, 254, 9, 36, "Output",
   CellTags->"b:5.6",
   CellID->410059729],
  Cell[4747, 185, 1474, 34, 70, "Text",
   CellTags->"b:5.6",
   CellID->10498],
  Cell[6224, 221, 434, 11, 70, "Text",
   CellTags->"b:5.6",
   CellID->7103],
  Cell[6661, 234, 190, 5, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->27352],
  Cell[6876, 243, 386, 13, 70, "Input",
   CellTags->"b:5.6",
   CellID->25679],
  Cell[7265, 258, 807, 30, 43, "Output",
   CellTags->"b:5.6",
   CellID->39286122],
  Cell[8087, 291, 97, 2, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->13043],
  Cell[8209, 297, 461, 16, 28, "Input",
   CellTags->"b:5.6",
   CellID->14522],
  Cell[8673, 315, 4717, 82, 244, "Output",
   Evaluatable->False,
   CellTags->"b:5.6",
   CellID->11483804],
  Cell[13405, 400, 366, 11, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->1563],
  Cell[13796, 415, 437, 15, 70, "Input",
   CellTags->"b:5.6",
   CellID->25834],
  Cell[14236, 432, 489, 10, 70, "Message",
   CellTags->"b:5.6",
   CellID->460469657],
  Cell[14728, 444, 417, 15, 43, "Output",
   CellTags->"b:5.6",
   CellID->496062216],
  Cell[15160, 462, 162, 5, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->7240],
  Cell[15347, 471, 536, 17, 70, "Input",
   CellTags->"b:5.6",
   CellID->21911],
  Cell[15886, 490, 489, 10, 70, "Message",
   CellTags->"b:5.6",
   CellID->165520702],
  Cell[16378, 502, 194, 7, 36, "Output",
   CellTags->"b:5.6",
   CellID->74575261],
  Cell[16587, 512, 366, 12, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->28580],
  Cell[16956, 526, 319, 12, 70, "Input",
   CellTags->"b:5.6",
   CellID->15477],
  Cell[17300, 542, 313, 11, 70, "Input",
   CellTags->"b:5.6",
   CellID->27616],
  Cell[17616, 555, 897, 31, 51, "Output",
   CellTags->"b:5.6",
   CellID->25609640],
  Cell[18550, 591, 327, 11, 70, "Input",
   CellTags->"b:5.6",
   CellID->19178],
  Cell[18880, 604, 596, 20, 36, "Output",
   CellTags->"b:5.6",
   CellID->63028327],
  Cell[19491, 627, 365, 10, 70, "Text",
   CellTags->"b:5.6",
   CellID->14615],
  Cell[19859, 639, 184, 5, 70, "MathCaption",
   CellTags->"b:5.6",
   CellID->22348],
  Cell[20068, 648, 454, 16, 70, "Input",
   CellTags->"b:5.6",
   CellID->26798],
  Cell[20525, 666, 493, 10, 70, "Message",
   CellTags->"b:5.6",
   CellID->689397760],
  Cell[21021, 678, 518, 10, 70, "Message",
   CellTags->"b:5.6",
   CellID->613871284],
  Cell[21542, 690, 490, 10, 70, "Message",
   CellTags->"b:5.6",
   CellID->129143153],
  Cell[22035, 702, 196, 7, 36, "Output",
   CellTags->"b:5.6",
   CellID->489650356],
  Cell[22268, 714, 424, 15, 70, "Input",
   CellTags->"b:5.6",
   CellID->1662],
  Cell[22695, 731, 559, 18, 52, "Output",
   CellTags->"b:5.6",
   CellID->318730300],
  Cell[23269, 752, 628, 19, 70, "Text",
   CellTags->"b:5.6",
   CellID->28262]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:57", 26563, 840},
 {"b:5.6", 26664, 844}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 655, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1280, 43, 89, 2, 70, "Title",
 CellTags->{"c:57", "b:5.6"},
 CellID->13049],
Cell[1372, 47, 277, 9, 70, "Text",
 CellTags->"b:5.6",
 CellID->14118],
Cell[1652, 58, 109, 2, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->24065],
Cell[CellGroupData[{
Cell[1786, 64, 302, 11, 28, "Input",
 CellTags->"b:5.6",
 CellID->26363],
Cell[2091, 77, 502, 17, 36, "Output",
 CellTags->"b:5.6",
 CellID->115728059]
}, Open  ]],
Cell[2608, 97, 353, 11, 70, "Text",
 CellTags->"b:5.6",
 CellID->28773],
Cell[2964, 110, 104, 2, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->18470],
Cell[3071, 114, 414, 15, 70, "Input",
 CellTags->"b:5.6",
 CellID->29731],
Cell[CellGroupData[{
Cell[3510, 133, 194, 6, 28, "Input",
 CellTags->"b:5.6",
 CellID->6747],
Cell[3707, 141, 618, 21, 39, "Output",
 CellTags->"b:5.6",
 CellID->65833916]
}, Open  ]],
Cell[CellGroupData[{
Cell[4362, 167, 113, 4, 28, "Input",
 CellTags->"b:5.6",
 CellID->5869],
Cell[4478, 173, 254, 9, 36, "Output",
 CellTags->"b:5.6",
 CellID->410059729]
}, Open  ]],
Cell[4747, 185, 1474, 34, 70, "Text",
 CellTags->"b:5.6",
 CellID->10498],
Cell[6224, 221, 434, 11, 70, "Text",
 CellTags->"b:5.6",
 CellID->7103],
Cell[6661, 234, 190, 5, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->27352],
Cell[CellGroupData[{
Cell[6876, 243, 386, 13, 70, "Input",
 CellTags->"b:5.6",
 CellID->25679],
Cell[7265, 258, 807, 30, 43, "Output",
 CellTags->"b:5.6",
 CellID->39286122]
}, Open  ]],
Cell[8087, 291, 97, 2, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->13043],
Cell[CellGroupData[{
Cell[8209, 297, 461, 16, 28, "Input",
 CellTags->"b:5.6",
 CellID->14522],
Cell[8673, 315, 4717, 82, 244, "Output",
 Evaluatable->False,
 CellTags->"b:5.6",
 CellID->11483804]
}, Open  ]],
Cell[13405, 400, 366, 11, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->1563],
Cell[CellGroupData[{
Cell[13796, 415, 437, 15, 70, "Input",
 CellTags->"b:5.6",
 CellID->25834],
Cell[14236, 432, 489, 10, 70, "Message",
 CellTags->"b:5.6",
 CellID->460469657],
Cell[14728, 444, 417, 15, 43, "Output",
 CellTags->"b:5.6",
 CellID->496062216]
}, Open  ]],
Cell[15160, 462, 162, 5, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->7240],
Cell[CellGroupData[{
Cell[15347, 471, 536, 17, 70, "Input",
 CellTags->"b:5.6",
 CellID->21911],
Cell[15886, 490, 489, 10, 70, "Message",
 CellTags->"b:5.6",
 CellID->165520702],
Cell[16378, 502, 194, 7, 36, "Output",
 CellTags->"b:5.6",
 CellID->74575261]
}, Open  ]],
Cell[16587, 512, 366, 12, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->28580],
Cell[16956, 526, 319, 12, 70, "Input",
 CellTags->"b:5.6",
 CellID->15477],
Cell[CellGroupData[{
Cell[17300, 542, 313, 11, 70, "Input",
 CellTags->"b:5.6",
 CellID->27616],
Cell[17616, 555, 897, 31, 51, "Output",
 CellTags->"b:5.6",
 CellID->25609640]
}, Open  ]],
Cell[CellGroupData[{
Cell[18550, 591, 327, 11, 70, "Input",
 CellTags->"b:5.6",
 CellID->19178],
Cell[18880, 604, 596, 20, 36, "Output",
 CellTags->"b:5.6",
 CellID->63028327]
}, Open  ]],
Cell[19491, 627, 365, 10, 70, "Text",
 CellTags->"b:5.6",
 CellID->14615],
Cell[19859, 639, 184, 5, 70, "MathCaption",
 CellTags->"b:5.6",
 CellID->22348],
Cell[CellGroupData[{
Cell[20068, 648, 454, 16, 70, "Input",
 CellTags->"b:5.6",
 CellID->26798],
Cell[20525, 666, 493, 10, 70, "Message",
 CellTags->"b:5.6",
 CellID->689397760],
Cell[21021, 678, 518, 10, 70, "Message",
 CellTags->"b:5.6",
 CellID->613871284],
Cell[21542, 690, 490, 10, 70, "Message",
 CellTags->"b:5.6",
 CellID->129143153],
Cell[22035, 702, 196, 7, 36, "Output",
 CellTags->"b:5.6",
 CellID->489650356]
}, Open  ]],
Cell[CellGroupData[{
Cell[22268, 714, 424, 15, 70, "Input",
 CellTags->"b:5.6",
 CellID->1662],
Cell[22695, 731, 559, 18, 52, "Output",
 CellTags->"b:5.6",
 CellID->318730300]
}, Open  ]],
Cell[23269, 752, 628, 19, 70, "Text",
 CellTags->"b:5.6",
 CellID->28262],
Cell[CellGroupData[{
Cell[23922, 775, 71, 1, 70, "RelatedTutorialsSection",
 CellID->35449754],
Cell[23996, 778, 177, 3, 70, "RelatedTutorials",
 CellID->439091251]
}, Open  ]]
}, Open  ]],
Cell[24200, 785, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

