(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    495041,      11185]
NotebookOptionsPosition[    471756,      10359]
NotebookOutlinePosition[    475584,      10467]
CellTagsIndexPosition[    475240,      10454]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Automatic" :> Documentation`HelpLookup["paclet:ref/Automatic"], 
         "FindRoot" :> Documentation`HelpLookup["paclet:ref/FindRoot"], 
         "Method" :> Documentation`HelpLookup["paclet:ref/Method"]}, #, 2]& , 
       Evaluator -> Automatic], {1->"\<\"Automatic\"\>", 
      2->"\<\"FindRoot\"\>", 3->"\<\"Method\"\>"}, "\<\"functions\"\>", 
      StyleBox["\<\"functions\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["EventLocator Method for NDSolve", "Title",
 CellID->12728],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->24251],

Cell["\<\
It is often useful to be able to detect and precisely locate a change in a \
differential system. For example, with the detection of a singularity or \
state change, the appropriate action can be taken, such as restarting the \
integration. \
\>", "Text",
 CellID->28061],

Cell[TextData[{
 "An ",
 StyleBox["event",
  FontSlant->"Italic"],
 " for a differential system:"
}], "Text",
 CellID->4010],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"Y", "'"}], 
    RowBox[{"(", "t", ")"}]}], " ", "=", " ", 
   RowBox[{"f", "(", 
    RowBox[{"t", ",", " ", 
     RowBox[{"Y", "(", "t", ")"}]}], ")"}]}], TraditionalForm]], "DisplayMath",\

 CellID->164338131],

Cell["\<\
is a point along the solution at which a real valued event function is zero:\
\>", "Text",
 CellID->303513695],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"g", "(", 
    RowBox[{"t", ",", " ", 
     RowBox[{"Y", "(", "t", ")"}]}], ")"}], " ", "=", " ", "0"}], 
  TraditionalForm]], "DisplayMath",
 CellID->451433106],

Cell[TextData[{
 "It is also possible to consider Boolean valued event functions, in which \
case the event occurs when the function changes from ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " or vice-versa."
}], "Text",
 CellID->406756659],

Cell[TextData[{
 "The ",
 StyleBox["EventLocator", "MR"],
 " method which is built into ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " works effectively as a controller method; it handles checking for events \
and taking the appropriate action, but the integration of the differential \
system is otherwise left completely to an underlying method.",
 " "
}], "Text",
 CellID->19480],

Cell[TextData[{
 "In this section, examples are given to demonstrate the basic use of the ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method and options. Subsequent sections show more involved applications of \
event location, such as period detection,",
 " ",
 "Poincar\[EAcute] sections, and discontinuity handling."
}], "Text",
 CellID->17153],

Cell["\<\
These initialization commands load some useful packages that have some \
differential equations to solve and define some utility functions.\
\>", "MathCaption",
 CellID->24027],

Cell[BoxData[{
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveUtilities`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`InterpolatingFunctionAnatomy`\>\"",
    "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Needs", "[", "\"\<Graphics`\>\"", "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Needs", "[", "\"\<GUIKit`\>\"", "]"}], ";"}]}], "Input",
 InitializationCell->True,
 CellLabel->"In[6]:=",
 CellID->16891],

Cell["\<\
A simple example is locating an event, such as when a pendulum started at a \
non-equilibrium position will swing through its lowest point and stopping the \
integration at that point.\
\>", "Text",
 CellID->7491],

Cell[TextData[{
 "This integrates the pendulum equation up to the first point at which the \
solution ",
 Cell[BoxData[
  RowBox[{"y", "[", "t", "]"}]], "InlineFormula"],
 " crosses the axis."
}], "MathCaption",
 CellID->17252],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", "0"}], 
      ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",",
       " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
    ",", "y", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"EventLocator", ",", " ", 
       RowBox[{"\"\<Event\>\"", "\[Rule]", 
        RowBox[{"y", "[", "t", "]"}]}]}], "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->31809],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "1.6749939325948517`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->1081533389]
}, Open  ]],

Cell[TextData[{
 "From the solution you can see that pendulum reaches its lowest point ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    StyleBox["y", "TI"], "[", 
    StyleBox["t", "TI"], "]"}], "=", "0"}]], "InlineFormula"],
 " at about ",
 Cell[BoxData[
  RowBox[{
   StyleBox["t", "TI"], "=", "1.675"}]], "InlineFormula"],
 ".",
 " ",
 "Using the ",
 Cell[BoxData["InterpolatingFunctionAnatomy"], "InlineFormula"],
 " package, it is possible to extract the value from the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object."
}], "Text",
 CellID->5726],

Cell["\<\
This extracts the point at which the event occurs and makes a plot of the \
solution (black) and its derivative (blue) up to that point.\
\>", "MathCaption",
 CellID->26269],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"end", " ", "=", " ", 
   RowBox[{
    RowBox[{"InterpolatingFunctionDomain", "[", 
     RowBox[{"First", "[", 
      RowBox[{"y", " ", "/.", " ", "sol"}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"1", ",", 
      RowBox[{"-", "1"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], ",", " ", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}]}], "}"}], " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "end"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Blue", "}"}]}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[12]:=",
 CellID->5134],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 221},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->83107966]
}, Open  ]],

Cell[TextData[{
 "When you use the event locator method, the events to be located and the \
action to take upon finding an event are specified through method options of \
the ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method."
}], "Text",
 CellID->863],

Cell["\<\
The default action on detecting an event is to stop the integration as \
demonstrated above. The event action can be any expression. It is evaluated \
with numerical values substituted for the problem variables whenever an event \
is detected.\
\>", "Text",
 CellID->6130],

Cell[TextData[{
 "This prints the time and values each time the event",
 " ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     StyleBox["y", "TI"], "'"}], "[", 
    StyleBox["t", "TI"], "]"}], "=", 
   RowBox[{
    StyleBox["y", "TI"], "[", 
    StyleBox["t", "TI"], "]"}]}]], "InlineFormula"],
 " is detected for a damped pendulum."
}], "MathCaption",
 CellID->5721],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
       RowBox[{".1", " ", 
        RowBox[{
         RowBox[{"y", "'"}], "[", "t", "]"}]}], " ", "+", " ", 
       RowBox[{"Sin", "[", 
        RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", "0"}], 
     ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", 
     " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], ",",
    "y", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"EventLocator", ",", " ", 
      RowBox[{"\"\<Event\>\"", "\[Rule]", 
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "t", "]"}], "-", " ", 
        RowBox[{"y", "[", "t", "]"}]}]}], ",", " ", 
      RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
       RowBox[{"Print", "[", 
        RowBox[{
        "\"\<y'[\>\"", ",", " ", "t", ",", " ", "\"\<] = y[\>\"", ",", " ", 
         "t", ",", " ", "\"\<] = \>\"", ",", " ", 
         RowBox[{"y", "[", "t", "]"}]}], "]"}]}]}], "}"}]}]}], "]"}]], "Input",\

 CellLabel->"In[14]:=",
 CellID->31667],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"x'[\"\>", "\[InvisibleSpace]", "2.498535243213883`", 
   "\[InvisibleSpace]", "\<\"] = x[\"\>", "\[InvisibleSpace]", 
   "2.498535243213883`", "\[InvisibleSpace]", "\<\"] = \"\>", 
   "\[InvisibleSpace]", 
   RowBox[{"-", "0.5897530881422002`"}]}],
  SequenceForm[
  "x'[", 2.49853524321388, "] = x[", 2.49853524321388, 
   "] = ", -0.5897530881422],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[14]:=",
 CellID->498535699],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"x'[\"\>", "\[InvisibleSpace]", "5.787597844560016`", 
   "\[InvisibleSpace]", "\<\"] = x[\"\>", "\[InvisibleSpace]", 
   "5.787597844560016`", "\[InvisibleSpace]", "\<\"] = \"\>", 
   "\[InvisibleSpace]", "0.5012282621924667`"}],
  SequenceForm[
  "x'[", 5.78759784456002, "] = x[", 5.78759784456002, "] = ", 
   0.501228262192467],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[14]:=",
 CellID->1132671112],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"x'[\"\>", "\[InvisibleSpace]", "9.034279806380686`", 
   "\[InvisibleSpace]", "\<\"] = x[\"\>", "\[InvisibleSpace]", 
   "9.034279806380686`", "\[InvisibleSpace]", "\<\"] = \"\>", 
   "\[InvisibleSpace]", 
   RowBox[{"-", "0.4266452162251905`"}]}],
  SequenceForm[
  "x'[", 9.03427980638069, "] = x[", 9.03427980638069, 
   "] = ", -0.426645216225191],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[14]:=",
 CellID->1115229109]
}, Open  ]],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->869894129]
}, Open  ]],

Cell[TextData[{
 "Note that in the above example, the ",
 Cell[BoxData["\"\<EventAction\>\""], "InlineFormula"],
 " option was given using ",
 Cell[BoxData[
  ButtonBox["RuleDelayed",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RuleDelayed"]], "InlineFormula"],
 " (",
 Cell[BoxData["\[RuleDelayed]"], "InlineFormula"],
 ") to prevent it from evaluating except when the event is located."
}], "Text",
 CellID->11049],

Cell[TextData[{
 "You can see from the printed output that when the action does not stop the \
integration, multiple instances of an event can be detected. Events are \
detected when the sign of the event expression changes. You can restrict the \
event to be only for a sign change in a particular direction using the ",
 Cell[BoxData["\"\<Direction\>\""], "InlineFormula"],
 " ",
 "option."
}], "Text",
 CellID->15017],

Cell[TextData[{
 "This collects the points at which the velocity changes from negative to \
positive for a damped driven pendulum. ",
 Cell[BoxData[
  ButtonBox["Reap",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reap"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Sow",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sow"]], "InlineFormula"],
 " are programming constructs which are useful for collecting data when you \
don't, at first, know how much data there will be. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Reap",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Reap"], "[", 
   StyleBox["expr", "TI"], "]"}]], "InlineFormula"],
 " gives the value of ",
 Cell[BoxData[
  StyleBox["expr", "TI"]], "InlineFormula"],
 " together with all expressions to which ",
 StyleBox["Sow", "MR"],
 " has been applied during its evaluation.",
 " ",
 "Here ",
 Cell[BoxData[
  ButtonBox["Reap",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reap"]], "InlineFormula"],
 " ",
 StyleBox["encloses the use of ", "Text"],
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 StyleBox[" and ", "Text"],
 Cell[BoxData[
  ButtonBox["Sow",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sow"]], "InlineFormula"],
 " is a part of the event action, which allows us to collect data for each \
instance of an event."
}], "MathCaption",
 CellID->15665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reap", "[", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{".1", " ", 
         RowBox[{
          RowBox[{"y", "'"}], "[", "t", "]"}]}], " ", "+", " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", 
       RowBox[{".1", " ", 
        RowBox[{"Cos", "[", "t", "]"}]}]}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",",
       " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
    ",", "y", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "50"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"EventLocator", ",", " ", 
       RowBox[{"\"\<Event\>\"", "\[Rule]", 
        RowBox[{
         RowBox[{"y", "'"}], "[", "t", "]"}]}], ",", " ", 
       RowBox[{"\"\<Direction\>\"", "\[Rule]", "1"}], ",", " ", 
       RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"t", ",", " ", 
           RowBox[{"y", "[", "t", "]"}], ",", " ", 
           RowBox[{
            RowBox[{"y", "'"}], "[", "t", "]"}]}], "}"}], "]"}]}]}], 
      "}"}]}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->17011],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"x", "\[Rule]", 
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "50.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False]}], "}"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"3.5540689946630035`", ",", 
        RowBox[{"-", "0.8793362332919005`"}], ",", 
        "2.003605614753212`*^-15"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.476207005448906`", ",", 
        RowBox[{"-", "0.832216905499499`"}], ",", 
        "3.3324037973514464`*^-15"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"17.185667623655466`", ",", 
        RowBox[{"-", "0.8749387034665486`"}], ",", 
        RowBox[{"-", "3.157196726277789`*^-15"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"23.772329986042134`", ",", 
        RowBox[{"-", "0.9153517954793174`"}], ",", 
        "8.222589276130066`*^-16"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.28046071844961`", ",", 
        RowBox[{"-", "0.9271860535789481`"}], ",", 
        "1.065380422771156`*^-14"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.721717593150316`", ",", 
        RowBox[{"-", "0.9108169586420904`"}], ",", 
        RowBox[{"-", "1.1596626436904955`*^-14"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"43.101171581356034`", ",", 
        RowBox[{"-", "0.8777079341664088`"}], ",", 
        "2.0816681711721685`*^-17"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"49.42824299080917`", ",", 
        RowBox[{"-", "0.8410832408323995`"}], ",", 
        RowBox[{"-", "1.467857953241225`*^-14"}]}], "}"}]}], "}"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{543, 94},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->493872563]
}, Open  ]],

Cell[TextData[{
 "You may notice from the output of the previous example that the events are \
detected when the derivative is only approximately zero. When the method \
detects the presence of an event in a step of the underlying integrator (by a \
sign change of the event expression), then it uses a numerical method to \
approximately find the position of the root. Since the location process is \
numerical, you should expect only approximate results.",
 " ",
 "Location method options ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  ButtonBox["MaxIterations",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxIterations"]], "InlineFormula"],
 " can be given to those location methods which use ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 " to control tolerances for finding the root."
}], "Text",
 CellID->27725],

Cell["", "Text",
 CellID->24908],

Cell[TextData[{
 "For Boolean valued event functions, an event occurs when the function \
switches from ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " or vice-versa. The ",
 Cell[BoxData["\"\<Direction\>\""], "InlineFormula"],
 " option can be used to restrict the event only from changes from ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " (",
 Cell[BoxData[
  RowBox[{"\"\<Direction\>\"", "->", 
   RowBox[{"-", "1"}]}]], "InlineFormula"],
 ") or only from changes from ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " ",
 "(",
 Cell[BoxData[
  RowBox[{"\"\<Direction\>\"", "->", "1"}]], "InlineFormula"],
 ")."
}], "Text",
 CellID->3162],

Cell[TextData[{
 "This opens up a small window with a button, which when pressed changes the \
value of the variable stop to ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " from its initialized value of ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->2050],

Cell[BoxData[{
 RowBox[{
  RowBox[{"NDSolve`stop", " ", "=", " ", "False"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"GUIRun", "[", 
   RowBox[{"Widget", "[", 
    RowBox[{"\"\<Panel\>\"", ",", " ", 
     RowBox[{"{", 
      RowBox[{"Widget", "[", 
       RowBox[{"\"\<Button\>\"", ",", 
        RowBox[{"{", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"\"\<label\>\"", "\[Rule]", "\"\<Stop\>\""}], ",", 
          "\[IndentingNewLine]", 
          RowBox[{"BindEvent", "[", 
           RowBox[{"\"\<action\>\"", ",", "\[IndentingNewLine]", 
            RowBox[{"Script", "[", 
             RowBox[{"NDSolve`stop", " ", "=", " ", "True"}], "]"}]}], 
           "]"}]}], "}"}]}], "]"}], "}"}]}], "]"}], "]"}], ";"}]}], "Input",
 CellLabel->"In[16]:=",
 CellID->3524],

Cell["\<\
This integrates the pendulum equation up until the button is pushed (or the \
system runs out of memory)\
\>", "MathCaption",
 CellID->1215],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
       RowBox[{"Sin", "[", 
        RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", "0"}], 
     ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
    "}"}], ",", " ", "y", ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", " ", "0", ",", " ", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"EventLocator", ",", " ", 
      RowBox[{"\"\<Event\>\"", "\[RuleDelayed]", " ", "NDSolve`stop"}]}], 
     "}"}]}], ",", " ", 
   RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->28599],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "11407.027069347714`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{339, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->1694059150]
}, Open  ]],

Cell[TextData[{
 "Take note that in this example, the ",
 Cell[BoxData["\"\<Event\>\""], "InlineFormula"],
 " option was specified with ",
 Cell[BoxData[
  ButtonBox["RuleDelayed",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RuleDelayed"]], "InlineFormula"],
 " (",
 Cell[BoxData[":>"], "InlineFormula"],
 ") to prevent the immediate value of ",
 StyleBox["stop", "MR"],
 " from being evaluated and set up as the function. "
}], "Text",
 CellID->17670],

Cell["\<\
You can specify more than one event. If the event function evaluates \
numerically to a list, then each component of the list is considered to be a \
separate event. You can specify different actions, directions, etc. for each \
of these events by specifying the values of these options as lists of the \
appropriate length. \
\>", "Text",
 CellID->12455],

Cell["\<\
This integrates the pendulum equation up until the point at which the above \
\"Stop\" button is pressed. The number of complete swings of the pendulum is \
kept track of during the integration.\
\>", "MathCaption",
 CellID->24711],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"NDSolve`stop", " ", "=", " ", "False"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"swings", " ", "=", " ", "0"}], ";", 
  RowBox[{"{", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", 
          RowBox[{"Sin", "[", 
           RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", "0"}], ",",
         " ", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
        
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], 
       "}"}], ",", " ", "y", ",", " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", " ", "0", ",", " ", "1000000"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"EventLocator", ",", " ", 
         RowBox[{"\"\<Event\>\"", "\[RuleDelayed]", " ", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"y", "[", "t", "]"}], ",", "NDSolve`stop"}], "}"}]}], ",",
          " ", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"swings", "++"}], ",", 
            RowBox[{"Throw", "[", 
             RowBox[{"Null", ",", " ", "\"\<StopIntegration\>\""}], "]"}]}], 
           "}"}]}], ",", 
         RowBox[{"\"\<Direction\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"1", ",", "All"}], "}"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"MaxSteps", "\[Rule]", "Infinity"}]}], "]"}], ",", "swings"}], 
   "}"}]}]}], "Input",
 CellLabel->"In[20]:=",
 CellID->6221],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"x", "\[Rule]", 
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "22742.342476244332`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False]}], "}"}], "}"}], ",", "3372"}], "}"}]], "Output",
 ImageSize->{399, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->1513391367]
}, Open  ]],

Cell["\<\
As you can see from the previous example, it is possible to mix real and \
Boolean valued event functions. The expected number of components and type of \
each component is based on the values at the initial condition and needs to \
be consistent throughout the integration.\
\>", "Text",
 CellID->31680],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option of ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " allows the specification of the numerical method to use in the \
integration."
}], "Text",
 CellID->20037]
}, Open  ]],

Cell[CellGroupData[{

Cell["Event Location Methods", "Section",
 CellTags->"s:2",
 CellID->18569],

Cell[TextData[{
 "The ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method works by taking a step of the underlying method and checking to see \
if the sign (or parity) of any of the event functions is different at the \
step endpoints. Event functions are expected to be real or Boolean valued, so \
if there is a change, there must be an event in the step interval.",
 " ",
 "For each event function which has an event occurrence in a step, a \
refinement procedure is carried out to locate the position of the event \
within the interval."
}], "Text",
 CellID->2807],

Cell["\<\
There are several different methods which can be used refine the position. \
These include simply taking the solution at the beginning or the end of the \
integration interval, a linear interpolation of the event value and using \
bracketed root finding methods. The appropriate method to use depends on a \
trade off between execution speed and location accuracy.\
\>", "Text",
 CellID->23535],

Cell[TextData[{
 "If the event action is to stop the integration then the particular value at \
which the integration is stopped depends on the value obtained from the ",
 Cell[BoxData["EventLocationMethod"], "InlineFormula"],
 " option of ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 "."
}], "Text",
 CellID->23666],

Cell["\<\
Location of a single event is usually fast enough so that the method used \
will not significantly influence the overall computation time. However, when \
an event is detected multiple times, the location refinement method can have \
a substantial effect.\
\>", "Text",
 CellID->3092],

Cell[CellGroupData[{

Cell["StepBegin and StepEnd Methods", "Subsection",
 CellID->1882],

Cell[TextData[{
 "The crudest methods are appropriate for when the exact position of the \
event location doesn't really matter or does not reflect anything with \
precision in the underlying calculation. The stop button example from the \
previous section is such a case: time steps are computed so quickly that \
there is no way that one can time the press of a button to be within a \
particular time step, much less at a particular point within a time step. \
Thus, based on the inherent accuracy of the event, there is no point in \
refining at all. You can specify this by using the ",
 Cell[BoxData[
  FormBox["\"\<StepBegin\>\"", TraditionalForm]], "InlineFormula"],
 " or ",
 Cell[BoxData["\"\<StepEnd\>\""], "InlineFormula"],
 " location methods.",
 " ",
 "In any example where the definition of the event is heuristic or somewhat \
imprecise, this can be an appropriate choice."
}], "Text",
 CellID->24049]
}, Open  ]],

Cell[CellGroupData[{

Cell["LinearInterpolation Method", "Subsection",
 CellID->3106],

Cell["\<\
When event results are needed for the purpose of points to plot in a graph, \
you only need to locate the event to the resolution of the graph. While just \
using the step end is usually too crude for this, a single linear \
interpolation based on the event function values suffices.\
\>", "Text",
 CellID->2594],

Cell["\<\
Denote the event function values at successive mesh points of the numerical \
integration:\
\>", "Text",
 CellID->662609674],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["w", "n"], " ", "=", " ", 
    RowBox[{"g", "(", 
     RowBox[{
      SubscriptBox["t", "n"], ",", " ", 
      SubscriptBox["y", "n"]}], ")"}]}], ",", " ", 
   RowBox[{
    SubscriptBox["w", 
     RowBox[{"n", "+", "1"}]], " ", "=", " ", 
    RowBox[{"g", "(", 
     RowBox[{
      SubscriptBox["t", 
       RowBox[{"n", "+", "1"}]], ",", " ", 
      SubscriptBox["y", 
       RowBox[{"n", "+", "1"}]]}], ")"}]}]}], TraditionalForm]], "DisplayMath",\

 CellID->452965130],

Cell["Linear interpolation gives:", "Text",
 CellID->340096408],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["w", "e"], "=", " ", 
   RowBox[{"|", " ", 
    FractionBox[
     SubscriptBox["w", "n"], 
     RowBox[{
      SubscriptBox["w", 
       RowBox[{"n", "+", "1"}]], "-", " ", 
      SubscriptBox["w", "n"]}]], "|"}]}], TraditionalForm]], "DisplayMath",
 CellID->586491796],

Cell["A linear approximation of the event time is then:", "Text",
 CellID->847793339],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["t", "e"], "=", " ", 
   RowBox[{
    SubscriptBox["t", "n"], "+", " ", 
    RowBox[{
     SubscriptBox["w", "e"], 
     SubscriptBox["h", "n"]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->55447431],

Cell[TextData[{
 "Linear interpolation could also be used to approximate the solution at the \
event time. However, since derivative values ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["f", "n"], " ", "=", " ", 
    RowBox[{"f", "(", 
     RowBox[{
      SubscriptBox["t", "n"], ",", 
      SubscriptBox["y", "n"]}], ")"}], " "}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["f", 
     RowBox[{"n", "+", "1"}]], " ", "=", " ", 
    RowBox[{"f", "(", 
     RowBox[{
      SubscriptBox["t", 
       RowBox[{"n", "+", "1"}]], ",", " ", 
      SubscriptBox["y", 
       RowBox[{"n", "+", "1"}]]}], ")"}]}], TraditionalForm]], "InlineMath"],
 " are available at the mesh points, a better approximation of the solution \
at the event can be computed cheaply using cubic Hermite interpolation as:"
}], "Text",
 CellID->4454476],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", "e"], "=", " ", 
   RowBox[{
    RowBox[{
     SubscriptBox["k", "n"], " ", 
     SubscriptBox["y", "n"]}], "+", " ", 
    RowBox[{
     SubscriptBox["k", 
      RowBox[{"n", "+", "1"}]], " ", 
     SubscriptBox["y", 
      RowBox[{"n", "+", "1"}]]}], "+", " ", 
    RowBox[{
     SubscriptBox["l", "n"], " ", 
     SubscriptBox["f", "n"]}], "+", " ", 
    RowBox[{
     SubscriptBox["l", 
      RowBox[{"n", "+", "1"}]], " ", 
     SubscriptBox["f", 
      RowBox[{"n", "+", "1"}]]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->8389972],

Cell["for suitably defined interpolation weights:", "Text",
 CellID->154414597],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["k", "n"], "=", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["w", "e"], "-", "1"}], ")"}], "2"], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"2", " ", 
         SubscriptBox["w", "e"]}], "+", " ", "1"}], ")"}]}]},
    {
     SubscriptBox["k", 
      RowBox[{"n", "+", "1"}]], "=", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"3", "-", 
        RowBox[{"2", " ", 
         SubscriptBox["w", "e"]}]}], ")"}], " ", 
      SuperscriptBox[
       SubscriptBox["w", "e"], "2"]}]},
    {
     SubscriptBox["l", "n"], "=", 
     RowBox[{
      SubscriptBox["h", "n"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["w", "e"], "-", "1"}], ")"}], "2"], " ", 
      SubscriptBox["w", "e"]}]},
    {
     SubscriptBox["l", 
      RowBox[{"n", "+", "1"}]], "=", 
     RowBox[{
      SubscriptBox["h", "n"], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["w", "e"], " ", "-", " ", "1"}], ")"}], " ", 
      SuperscriptBox[
       SubscriptBox["w", "e"], "2"]}]}
   }], TraditionalForm]], "DisplayMath",
 CellID->526662252],

Cell[TextData[{
 " You can specify refinement based on a single linear interpolation with the \
setting ",
 StyleBox["\"LinearInterpolation\"", "MR"],
 "."
}], "Text",
 CellID->334110284],

Cell["\<\
This computes the solution for a single period of the pendulum equation and \
plots the solution for that period.\
\>", "MathCaption",
 CellID->23324],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", "0"}],
         ",", " ", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "3"}], ",", " ", 
        
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
       "}"}], ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "\[Infinity]"}], "}"}], ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", "\[IndentingNewLine]", " ", 
         RowBox[{"\"\<Event\>\"", "\[Rule]", 
          RowBox[{
           RowBox[{"y", "'"}], "[", "t", "]"}]}], ",", " ", 
         "\[IndentingNewLine]", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
          RowBox[{"Throw", "[", 
           RowBox[{
            RowBox[{"end", " ", "=", " ", "t"}], ",", " ", 
            "\"\<StopIntegration\>\""}], "]"}]}], ",", 
         RowBox[{"\"\<Direction\>\"", "\[Rule]", 
          RowBox[{"-", "1"}]}], ",", " ", "\[IndentingNewLine]", 
         RowBox[{
         "\"\<EventLocationMethod\>\"", "->", "\"\<LinearInterpolation\>\""}],
          ",", 
         RowBox[{"Method", "->", "\"\<ExplicitRungeKutta\>\""}]}], "}"}]}]}], 
     "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], ",", " ", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}]}], "}"}], " ", "/.", " ", "sol"}],
     "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "end"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "Black", "}"}], ",", " ", 
      RowBox[{"{", "Blue", "}"}]}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[21]:=",
 CellID->6444],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 221},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->120101848]
}, Open  ]],

Cell["\<\
At the resolution of the plot over the entire period, you cannot see that the \
endpoint may not be exactly where the derivative hits the axis. However, if \
you zoom in enough, you can see the error.\
\>", "Text",
 CellID->22070],

Cell["This shows a plot just near the endpoint.", "MathCaption",
 CellID->23259],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "'"}], "[", "t", "]"}], " ", "/.", " ", "sol"}], "]"}], 
   ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"end", "*", 
      RowBox[{"(", 
       RowBox[{"1", " ", "-", " ", ".001"}], ")"}]}], ",", "end"}], "}"}], 
   ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", "Blue"}]}], "]"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->30033],

Cell[BoxData[
 GraphicsBox[{{}, {}, {
    {RGBColor[0, 0, 1]}, LineBox[CompressedData["
1:eJwVyX1Q0wUcx/EfBYpOHk1xcQXGgJxNwQBrGN/hYiFzMTYQGPGwBVysbkuJ
3G08K2jE4yDJGTDpioeJLdCERiim0SaI8RSCHSg1YxAIHvFgcH1/f3zufa/7
7JbIBSnPEQQBOLLXlW6HtB4+LCPFkLw8J4EuOquMtGZbytOcf9BjSY9Ihzs6
5dvOopn1Zy+ifRnyOuo0eo02XI9WtLPHAqck0Kmky75BR/7H4+f+LoF2pb+2
GU3dbH5ryw0J6JVc66vo6R9c3VwrJOBcuPL5HXSx3cUHsXsl4Nst+2ISbXei
b+VYrxhqTSz/JXSR4VaEr1wMmROVVRSaD8tDLW3z2ioGF5o+wx0dt5IrHdMl
wSuf8HYeRCcZ+/Y1hSSBSs3h8dBC+/sJzTcTQT1pu0mCLhU5rmZvTgQq82i2
Av1km/zaZ4cSIP7nGEUJ2t5PZuOUEw/zL3bb1qN31Yo6BLffg8Ws5ph29F9s
B43pWRy0OsbT+9AjQulaF8TBo8vlhil0Qe/50ul0EYhd3adW0RTtljN3dbFQ
c01RbO/pwxqbaJ/fvxADRxIFSzT01U5FZY9XDLQGLE28iV66m1I3nhYNxVRq
ER9trWfMetYcgyumL02p6GjhSAV9KArO6x4Wq9BZEW0Vul1RMFets1ejz5Y4
j0p4kWB1Jci5Ab34Gu/4uTIhSGoZlwxoYqo18vSwAF6vqrEZIP+h4FONFAF0
d4xMmtGuM4UroeERcMNSrlpH77R0drqd4oOd5dtL2718WAfaYsvFGeHgQi1M
fBXdGBb6Y97z74JaaP4tCJ2v8n5AUR4F008OvwjRQ6J+o8yBC/x/q/gfoDXs
Fqe3Lx8BDvOCJhtdncUd/Wp/KKy2SFMr0RUt/SUFRg68xDAsNKEzjdqw7w6H
wMb40EYX+rGd9ta+ATYwYsPqB9FS6+t548mHoe8hy8aCTp6wLqsYZAFdQ1g2
0PGKjgS+PgiW44NLd3j7sBoaZ/R18kBwfvLHr3vR99u8pkv93oCZ79kFLLTu
dtrJp3v8wS+HbhWNzh+MVvmO+gIzc2zjQ/TfIvdNDQoGlL5jVuei2VFfh6jE
3iB8Od18Dm1bWbQQ3O8OwzSjQYeWhZ+J/TNrB/QEaPk30fKPl2tT9ljB3LxV
A+nAkICNez0E5PUeGCD9jMPJ2FpNgIl55zFpr9Aoj8ZsAuaPVy+SnvXcfvDk
CQLyS9aXSftbdt9L+IgA7gX9Gukm7qdaQRoBLzS9v04600WYzkkl4H/CbKpt

     "]]}},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  Method->{},
  PlotRange->{{16.1391637395284, 16.155319058587}, {0., 0.00231121367491613}},
  
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 229},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->1679090225]
}, Open  ]],

Cell["\<\
The linear interpolation method is sufficient for most viewing purposes, such \
as the Poincar\[EAcute] section examples shown in the following section. Note \
that for Boolean valued event functions, linear interpolation is effectively \
only one bisection step, so the linear interpolation method may be inadequate \
for graphics. \
\>", "Text",
 CellID->22956]
}, Open  ]],

Cell[CellGroupData[{

Cell["Brent's Method", "Subsection",
 CellID->19093],

Cell[TextData[{
 "The default location method is the event location method ",
 Cell[BoxData["\"\<Brent\>\""], "InlineFormula"],
 ", finding the location of the event using ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 " with ",
 ButtonBox["Brent's",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/UnconstrainedOptimizationBrentsMethod#1562437747"],
 " method. Brent's method starts with a bracketed root and combines steps \
based on interpolation and bisection, guaranteeing a convergence rate at \
least as good as bisection. You can control the accuracy and precision to \
which ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 " tries to get the root of the event function using method options for the \
",
 Cell[BoxData["\"\<Brent\>\""], "InlineFormula"],
 " event location method.",
 " ",
 "The default is to find the root to the same accuracy and precision as ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " is using for local error control."
}], "Text",
 CellID->12891],

Cell[TextData[{
 "For methods which support continuous or dense output, the argument for the \
event function can be found quite efficiently simply by using the continuous \
output formula. However, for methods which do not support continuous output, \
the solution needs to be computed by taking a step of the underlying method, \
which can be relatively expensive.",
 " ",
 "An alternate way of getting a solution approximation which is not accurate \
to the method order, but is consistent with using ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 " on the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object returned from ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " is to use cubic Hermite interpolation, obtaining approximate solution \
values in the middle of the step by interpolation based on the solution \
values and solution derivative values at the step ends."
}], "Text",
 CellID->20022]
}, Open  ]],

Cell[CellGroupData[{

Cell["Comparison", "Subsection",
 CellID->14605],

Cell["\<\
This example integrates the pendulum equation for a number of different event \
location methods and compares the time when the event is found.\
\>", "Text",
 CellID->21887],

Cell["This defines the event location methods to use.", "MathCaption",
 CellID->24757],

Cell[BoxData[
 RowBox[{
  RowBox[{"eventmethods", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
    "\"\<StepBegin\>\"", ",", "\"\<StepEnd\>\"", ",", 
     "\"\<LinearInterpolation\>\"", ",", "Automatic"}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->30498],

Cell["\<\
This integrates the system and prints out the method used and the value of \
the independent variable when the integration is terminated.\
\>", "MathCaption",
 CellID->16028],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Map", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
           RowBox[{"Sin", "[", 
            RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", 
          "0"}], ",", " ", 
         RowBox[{
          RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "3"}], ",", " ",
          
         RowBox[{
          RowBox[{
           RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}],
         "}"}], ",", "y", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "\[Infinity]"}], "}"}], ",", " ", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<EventLocator\>\"", ",", "\[IndentingNewLine]", " ", 
          RowBox[{"\"\<Event\>\"", "\[Rule]", 
           RowBox[{
            RowBox[{"y", "'"}], "[", "t", "]"}]}], ",", " ", 
          "\[IndentingNewLine]", 
          RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
           RowBox[{"Throw", "[", 
            RowBox[{
             RowBox[{"Print", "[", 
              RowBox[{
              "#", ",", "\"\<: t = \>\"", ",", "t", ",", "\"\< y'[t] \>\"", 
               ",", " ", 
               RowBox[{
                RowBox[{"y", "'"}], "[", "t", "]"}]}], "]"}], ",", " ", 
             "\"\<StopIntegration\>\""}], "]"}]}], ",", 
          RowBox[{"\"\<Direction\>\"", "\[Rule]", 
           RowBox[{"-", "1"}]}], ",", 
          RowBox[{"Method", "->", "\"\<ExplicitRungeKutta\>\""}], ",", " ", 
          "\[IndentingNewLine]", 
          RowBox[{"\"\<EventLocationMethod\>\"", "\[Rule]", "#"}]}], 
         "}"}]}]}], "]"}], "&"}], ",", "\[IndentingNewLine]", 
    "eventmethods"}], "\[IndentingNewLine]", "]"}], ";"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->28446],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"StepBegin\"\>", "\[InvisibleSpace]", "\<\": t = \"\>", 
   "\[InvisibleSpace]", "16.148745552432647`"}],
  SequenceForm["StepBegin", ": t = ", 16.1487455524326],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[25]:=",
 CellID->75512204],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"StepEnd\"\>", "\[InvisibleSpace]", "\<\": t = \"\>", 
   "\[InvisibleSpace]", "16.60880483162`"}],
  SequenceForm["StepEnd", ": t = ", 16.60880483162],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[25]:=",
 CellID->596940873],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"LinearInterpolation\"\>", 
   "\[InvisibleSpace]", "\<\": t = \"\>", "\[InvisibleSpace]", 
   "16.15531905858696`"}],
  SequenceForm["LinearInterpolation", ": t = ", 16.155319058587],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[25]:=",
 CellID->1025005775],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
  "Automatic", "\[InvisibleSpace]", "\<\": t = \"\>", "\[InvisibleSpace]", 
   "16.155540649780594`"}],
  SequenceForm[Automatic, ": t = ", 16.1555406497806],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[25]:=",
 CellID->1094793269]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellTags->"s:3",
 CellID->4920],

Cell[CellGroupData[{

Cell["Falling Body", "Subsection",
 CellID->26297],

Cell[TextData[{
 "This system models a body falling under the force of gravity encountering \
air resistance (see [",
 ButtonBox["M04",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25049"],
 "])."
}], "Text",
 CellID->19129],

Cell["\<\
The event action stores the time when the falling body hits the ground and \
stops the integration.\
\>", "MathCaption",
 CellID->26974],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{
   RowBox[{"y", "[", "t", "]"}], "/.", 
   RowBox[{"First", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "''"}], "[", "t", "]"}], "\[Equal]", 
         RowBox[{
          RowBox[{"-", "1"}], "+", 
          RowBox[{
           RowBox[{
            RowBox[{"y", "'"}], "[", "t", "]"}], "^", "2"}]}]}], ",", 
        RowBox[{
         RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
      ",", "y", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "Infinity"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", 
         RowBox[{"\"\<Event\>\"", "\[RuleDelayed]", 
          RowBox[{"y", "[", "t", "]"}]}], ",", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
          RowBox[{"Throw", "[", 
           RowBox[{
            RowBox[{"tend", "=", "t"}], ",", "\"\<StopIntegration\>\""}], 
           "]"}]}]}], "}"}]}]}], "]"}], "]"}]}]}]], "Input",
 CellLabel->"In[26]:=",
 CellID->15089],

Cell[BoxData[
 RowBox[{
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.657454454759687`"}], "}"}], "}"}], 
     ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False], "[", "t", "]"}]], "Output",
 ImageSize->{314, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->2021780425]
}, Open  ]],

Cell["\<\
This plots the solution and highlights the initial and final points (green \
and red) by encircling them.\
\>", "MathCaption",
 CellID->21859],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"plt", "=", 
    RowBox[{"Plot", "[", 
     RowBox[{"sol", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "tend"}], "}"}], ",", 
      RowBox[{"Frame", "\[Rule]", "True"}], ",", 
      RowBox[{"Axes", "\[Rule]", "False"}], ",", 
      RowBox[{"PlotStyle", "\[Rule]", "Blue"}], ",", 
      RowBox[{"DisplayFunction", "\[Rule]", "Identity"}]}], "]"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"grp", "=", 
    RowBox[{"Graphics", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"Green", ",", 
         RowBox[{"Circle", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "1"}], "}"}], ",", "0.025"}], "]"}]}], "}"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"Red", ",", 
         RowBox[{"Circle", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"tend", ",", 
             RowBox[{"sol", "/.", 
              RowBox[{"t", "\[Rule]", "tend"}]}]}], "}"}], ",", "0.025"}], 
          "]"}]}], "}"}]}], "}"}], "]"}]}], ";"}], "\n"}], "\n", 
 RowBox[{"Show", "[", 
  RowBox[{"plt", ",", "grp", ",", 
   RowBox[{"DisplayFunction", "\[Rule]", "$DisplayFunction"}]}], 
  "]"}]}], "Input",
 CellLabel->"In[27]:=",
 CellID->20864],

Cell[BoxData[
 GraphicsBox[{{{}, {}, {
     {RGBColor[0, 0, 1]}, LineBox[CompressedData["
1:eJwV03k41AkYB/CZLjzbfT1yZCRSIqRL5WtnLbIl7K6h3FFqG6bkbEVIrbVW
dD1RuhsUjSMkWxk3OcIMyrivYeaXKxoz7M/7PO/zPp//vt8/XnUPXzuvBRQK
xYPc+evI0P1rQtvDZGZufgg4VI+5RrNNoTMylTNvWb7Ccg+2DRiU9t5Z0tZL
lUbCpQ64t00uRka6fse6vqNsN2i4BjyZIR3kqPqb6ydP7I7dZvSdtHqkJtdH
6g21PI1930hXvdA1vKTFxOItHtljpPcN3603Z7NQbNv/SEx6OUXtmnXReWw2
fyUnJN299jHsP13A3cZ3Xb2kX2/dMuU8EIBTrWsNOkjHmKRneEmDILiVOdlC
eqd3tqq/Vig2mhcIqkiLK5zN6exwaD8UJL+Z79eotqks6zL6DnxO55Aubu+S
HSqKwLFVcTvYpG+NeeXafoqCxVPR7UTSZ+SEZx0HolHm7Nv8J2mo+Gx2l15D
DmP94AnSgz8HJLK0/kFaRVesDumKPaCasuMRaucTLZwlYEintnOzriO4SY5b
Rjr5MDffoigBlqYdTQ9Is9wtWEc/3QAnzSnImrRZQPUWxsAtqK9bc25cRmDD
30c7XKR30PJx57t/SRfnMKx9tO5hGy/eKUNKYBeRKTBhP4B08rhd2gyZv7A6
U7fyIdx5/t3LSU9fHQhXET5CjH2P2WkJgbc0NQ2JzlPI+3lGzE0TCCuwL+Uz
n6NML3A5c4oA3S7uVG5mKkqeKos4k2SfSFk6a2cG6uLUcj6MEnh+bPf0zeMc
BGeEzumJCFTuCFZ3PZ2FsxtcWpuEBIYXvbXSDsyG1emNDoxBAjsyf7z3JiEX
Bokhx8a6Cfg5lhOHM/IQ1bp1N6WdQN7CI/SOygIsY2WpXeYR6FTZW9IyXAT3
JscxnQoCF1J1rzRL30FADVyx5z2BRUmrNIaWfMAu65ZnhXkEotPf1BaacXFe
X//O1ccEFN6eCImLKEWoplG9IJbAUsEx3rq75ZDX9Fh4xZ/ASoqd4T1OJere
E3nLnAhM/PL8HcO4Fkx6mnKqKoEDeuIb4/YNyG/Qzx3liTFXbi/xPdMIH57N
Dy9ixeD0nbcqTmuGe/m5zSY7xehZ225VWcJH7IAaP5EvwkmtvALjplbs6aa1
7A0QgV9otiZi5RcoyefonVokQqUHTfWSlgBnsp218xJGwCvjy7ONOvG5vgZx
K0bge4RhXGbThYuGSx8sujqMii8y7n+O3fjpuMKpb5RhKOrN0jcF96Bg/1HP
i95CbL2mqeR2pRe5Fkc+FjUOweDJs355dh8MzD9MNOsMwZ3jKjxZ2g+viAea
rPBB2JbDPKVuAFXbdlFFbQPYlxJ8m/51EKEO1ReS1AfQFdZm+l42hOZ4L58j
vv3g2Pqpjy8exoEWpxRuYR+C+7ym4jePIOYlizcy3gtqRJimgr4Ik3EC2z6j
XoR3jlmO2IrhR7UkH6UH2vr3H0f5EAieC+HnxXbD0mxv+sz+r/CSKDm8Le5C
kK11mLD6K6ibsvfl9nRCvicwzNxuFIwZTvLkik5sZx5UljWPIuGP8ABWiACH
WzINhjzHwNWqCXdL+4Ka8l0X6MIxqAfduMkcb4PFHW+DhpBx/N6ZxWKuaoWG
KEPpI3UCl2vby7oO8ZFofD1FI3oCHJ0hl7PRzZj1pqVkrptEe+SwfNLtRlQp
c6vYSZNYHxHQhtIGXM9cfY6//RsCxSuX+sTVYWT/bJZb4TeU2F2/uDq5Gs8P
y2gM0ynQ78ydVOkvh4tJqkJk/RSS79cu8G7hwj/He6MKYxqVJYqLefLvQdMo
qV3xeRrTHLtpIjYf5edrvu898x2p8H+t7PAKteaDNnXEd3RP87dYHXqIhplX
m5LCJRBb/Eo9qGiDJS+kcqUREiQkMJ2vKpqCFvVI0hopgQu/xqBDUR+6rEk/
zygJ2lYvZLI20GDUf/9VKmmH7OKXkUorsefhuHUNacdrnBuPVCjIn1W9305a
XORw8wvtq4mxk6VITJpi+EJhtNvN5H8hIjod
      "]]}}, {
    {RGBColor[0, 1, 0], CircleBox[{0, 1}, 0.025]}, 
    {RGBColor[1, 0, 0], CircleBox[{1.6575, 9.2838*^-15}, 0.025]}}},
  AspectRatio->GoldenRatio^(-1),
  Frame->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  Method->{},
  PlotRange->{{0, 1.65745445475969}, {0., 0.999999999999999}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 230},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[29]=",
 CellID->1652505478]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Period of the Van der Pol Oscillator", "Subsection",
 CellID->20479],

Cell[TextData[{
 "The Van der Pol oscillator is an example of an extremely stiff system of \
ODEs.",
 " ",
 "The event locator method can call any method for actually doing the \
integration of the ODE system. The default method, ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", automatically switches to a method appropriate for stiff systems when \
necessary, so that stiffness does not present a problem."
}], "Text",
 CellID->15472],

Cell[TextData[{
 "This integrates the Van der Pol system for a particular value of the \
parameter ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Mu]", "=", "1000"}], TraditionalForm]], "InlineMath"],
 " up to the point where the variable ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "2"], TraditionalForm]], "InlineMath"],
 " reaches its initial value and direction."
}], "MathCaption",
 CellID->15869],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"vsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"(", GridBox[{
       {
        RowBox[{
         RowBox[{
          RowBox[{
           SubscriptBox["y", "1"], "'"}], "[", "t", "]"}], " ", "\[Equal]", 
         " ", 
         RowBox[{
          SubscriptBox["y", "2"], "[", "t", "]"}], " "}], 
        RowBox[{
         RowBox[{
          SubscriptBox["y", "1"], "[", "0", "]"}], "\[Equal]", "2"}]},
       {
        RowBox[{
         RowBox[{
          RowBox[{
           SubscriptBox["y", "2"], "'"}], "[", "t", "]"}], "\[Equal]", 
         RowBox[{
          RowBox[{"1000", " ", 
           RowBox[{"(", 
            RowBox[{"1", " ", "-", " ", 
             RowBox[{
              RowBox[{
               SubscriptBox["y", "1"], "[", "t", "]"}], "^", "2"}]}], ")"}], 
           " ", 
           RowBox[{
            SubscriptBox["y", "2"], "[", "t", "]"}]}], " ", "-", " ", 
          RowBox[{
           SubscriptBox["y", "1"], "[", "t", "]"}]}]}], 
        RowBox[{
         RowBox[{
          SubscriptBox["y", "2"], "[", "0", "]"}], "\[Equal]", "0"}]}
      }], ")"}], ",", "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["y", "1"], ",", " ", 
      SubscriptBox["y", "2"]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "3000"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"EventLocator", ",", " ", 
       RowBox[{"\"\<Event\>\"", "\[Rule]", 
        RowBox[{
         SubscriptBox["y", "2"], "[", "t", "]"}]}], ",", " ", 
       RowBox[{"\"\<Direction\>\"", "\[Rule]", 
        RowBox[{"-", "1"}]}]}], "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[30]:=",
 CellID->10408],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     SubscriptBox["y", "1"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1614.289572418055`"}], "}"}], "}"}], 
        ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{
     SubscriptBox["y", "2"], "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1614.289572418055`"}], "}"}], "}"}], 
        ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{349, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->683934652]
}, Open  ]],

Cell["\<\
Note that the event at the initial condition is not considered.\
\>", "Text",
 CellID->3817],

Cell[TextData[{
 "By selecting the endpoint of the ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " solution, it is possible to write a function which returns the period as a \
function of ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 "."
}], "Text",
 CellID->19663],

Cell[TextData[{
 "This defines a function which returns the period as a function of ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->22222],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"vper", "[", "\[Mu]_", "]"}], " ", ":=", " ", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", "vsol", "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"vsol", " ", "=", " ", 
       RowBox[{"First", "[", 
        RowBox[{
         SubscriptBox["y", "2"], " ", "/.", " ", 
         RowBox[{"NDSolve", "[", 
          RowBox[{
           RowBox[{"(", GridBox[{
              {
               RowBox[{
                RowBox[{
                 RowBox[{
                  SubscriptBox["y", "1"], "'"}], "[", "t", "]"}], " ", 
                "\[Equal]", " ", 
                RowBox[{
                 SubscriptBox["y", "2"], "[", "t", "]"}], " "}], 
               RowBox[{
                RowBox[{
                 SubscriptBox["y", "1"], "[", "0", "]"}], "\[Equal]", "2"}]},
              {
               RowBox[{
                RowBox[{
                 RowBox[{
                  SubscriptBox["y", "2"], "'"}], "[", "t", "]"}], "\[Equal]", 
                
                RowBox[{
                 RowBox[{"\[Mu]", " ", 
                  RowBox[{"(", 
                   RowBox[{"1", " ", "-", " ", 
                    RowBox[{
                    RowBox[{
                    SubscriptBox["y", "1"], "[", "t", "]"}], "^", "2"}]}], 
                   ")"}], " ", 
                  RowBox[{
                   SubscriptBox["y", "2"], "[", "t", "]"}]}], " ", "-", " ", 
                 RowBox[{
                  SubscriptBox["y", "1"], "[", "t", "]"}]}]}], 
               RowBox[{
                RowBox[{
                 SubscriptBox["y", "2"], "[", "0", "]"}], "\[Equal]", "0"}]}
             }], ")"}], ",", "\[IndentingNewLine]", 
           RowBox[{"{", 
            RowBox[{
             SubscriptBox["y", "1"], ",", " ", 
             SubscriptBox["y", "2"]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"t", ",", 
             RowBox[{"Max", "[", 
              RowBox[{"100", ",", " ", 
               RowBox[{"3", " ", "\[Mu]"}]}], "]"}]}], "}"}], ",", " ", 
           RowBox[{"Method", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"EventLocator", ",", " ", 
              RowBox[{"\"\<Event\>\"", "\[Rule]", 
               RowBox[{
                SubscriptBox["y", "2"], "[", "t", "]"}]}], ",", " ", 
              RowBox[{"\"\<Direction\>\"", "\[Rule]", 
               RowBox[{"-", "1"}]}]}], "}"}]}]}], "]"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"InterpolatingFunctionDomain", "[", "vsol", "]"}], "[", 
       RowBox[{"[", 
        RowBox[{"1", ",", 
         RowBox[{"-", "1"}]}], "]"}], "]"}]}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[31]:=",
 CellID->27818],

Cell[TextData[{
 "This uses the function to compute the period at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Mu]", "=", "1000"}], TraditionalForm]], "InlineMath"]
}], "MathCaption",
 CellID->21183],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"vper", "[", "1000", "]"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->8177],

Cell[BoxData["1614.289572418055`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->459458936]
}, Open  ]],

Cell["\<\
Of course, it is easy to generalize this method to any system with periodic \
solutions.\
\>", "Text",
 CellID->5820]
}, Open  ]],

Cell[CellGroupData[{

Cell["Poincar\[EAcute] Sections", "Subsection",
 CellID->16678],

Cell["\<\
Poincar\[EAcute] Sections are a useful technique for visualizing the \
solutions of high dimensional differential systems.\
\>", "Text",
 CellID->198960080],

Cell[TextData[{
 "For an interactive graphical interface see the package ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["EquationTrekker",
    BaseStyle->"Link",
    ButtonData->"paclet:EquationTrekker/tutorial/EquationTrekker"], "MR"]]],
 "."
}], "Text",
 CellID->352748880],

Cell[CellGroupData[{

Cell["The H\[EAcute]non-Heiles system", "Subsubsection",
 CellID->22454],

Cell["\<\
Define the H\[EAcute]non-Heiles system which models stellar motion in a \
galaxy.\
\>", "Text",
 CellID->6847],

Cell[TextData[{
 "This gets the H\[EAcute]non-Heiles system from the ",
 Cell[BoxData["NDSolveProblems"], "InlineFormula"],
 " package."
}], "MathCaption",
 CellID->28803],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<HenonHeiles\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], ";", 
  RowBox[{"eqns", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"system", "[", "\"\<System\>\"", "]"}], ",", " ", 
     RowBox[{"system", "[", "\"\<InitialConditions\>\"", "]"}]}], 
    "}"}]}]}]}], "Input",
 CellLabel->"In[33]:=",
 CellID->6824],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWs1uHEUQHu/u2F5v4hjshMi7SFGchJ9TciEQJC4hGEgMr4AVIWwpCBQi
IUBJeADeBDhxCSe/D+SCeIBmeqZ7d/bbr6f6Z8YOkVfamanu+uvq6uqumtnb
f3Tw5df7jw7v71/afbj/7cHh/e8uffTNw6Kpv5RlS8+yLNu9mBXPqngq//r3
Z/GvHqun28U1Lzr7xb1f3O8Ewj9OuWS5laHKtizr0dYPXwDcH8p7SdGf9mUr
AN/uuP/7UotRce0Vje9C7zLAqG3b/T2jR27uD0rt1gDrPUKlwY9bakcbDQA+
pFrdUmyW7ya3DqO0eYfy6gPWnWPHirPtTVXNjMYemAgQAqOf+UldVbM4o73R
rpI3AW/dSMs98HpGmya8c4Yfyn07ahQI44yMAL5mpJ2D9tyT7pVIeUjXE+jQ
an7WsHsGYqOWFg9H/QbAKw5+jP9J8tP4A7IeXXz8rIuxCbkwaQtzvBAzUNay
0bHqsL9tVfkN6oAykJufzJzKHDtkSt7Uhcz6+mDcUmw7EcZ3UCKvEl7tz/5W
CR8dHRndRvqiYdN/VrAL+uTLJNXuEHFzf16QJsWgr5RPDKif6dryz6vEDl2t
+WsnKIvyPtY13nR60Fyw30+jG8AF9/aT5BI3ovjTyKk127Im5033TVGD6xQr
bjTd84obY4CVB3Vei7FEnwTsPmRPrxq+QbGHHWK7NInLnfQFqw1+lO8b+ffM
fQ/uuLpSagyf0taQGgPK8dPmA4DvAbwnwHh+5FJt9psqzWbd87WlTwy0Z+6f
wf1JiXU3UfZPJZddavv02Xtc3nU1q8njqpFoGqy/6qdfzXjPTFt+hzEMhZ4L
apan1KsWl1VVRRlAO66AK4pXUdjJqQ7baskK0O0oXr3ZBvodNR8pXHJdVR6v
rCe7CK1Ihda4DDBqg/i4lvCELFVv2KmWVWUkvXcEvoj/GsBXAseN/DFDksbN
Mvfp7GcbildjsIrpqqKMHfS+1RyUo/f1POMV6tz4Yoo8iQ6rSb52mDj08q1G
4ayiF6457OKyh69cVybVlX3Lc9vCmmNahHDlueaGYBOJq7QmvPoX9jLmUSla
MdinmiadTfxybMnGoaOT1sEZweaSddv2ga5mr9pTm94YxMzhppqdVI5zZeRG
LkYL1F+qMYbqhW947N4Xa7fQ6r/Ed6RmuxrbBVGetSPbNet2S/VCiX/ovNB5
F9+jSV4VtwvYam3oHhMa/ZF/15EDfT11T/WL967a92mcb4arjLJuTW3d9UAu
Y6CXzv6pPo1wfdZj5LMMNMUeE6CXqj5te2UdP0Z+05ptw77/D697taH/ZfCy
CwCjV+H8vJheVO43ftkN/15wVnHPFr/a2KZ0a5Hc8LyH/euUb96qDn4y6Yk4
QtqY0g0judnvBnFloidxn4iVivAy5RM7H9yrbQ10/tTDKqA9I2XF3DU8ptRD
B/ZoXkc1q0f2jff582IxbpGanesZtwmlXvWUvUksq/E2krjy86jLHjhj1ezi
FwHPAP4c4J8pF/2Ib9vcmG/VerTGTxW+h0Bev0x78D2GrQ7olj/KFvRqd09e
o/2NYqDFu8C0X5ziu4yJmn93NVI8r7B+vQp4ryv+jgPp//bSEndcXDcY99Cv
pC87bfzAnA2jvyRHevdg5Tz3GrXry1HfGviWgz62Fu36UjRUH1vLd41X0uOv
0no4pxJVN9UM6btM1DK2WpyqPY/WXdT7tPbVqjZF08zKb5qt1PGdF0aSWr1J
rfSifjYShFr4HxXj+3YXlrw1zudTaz2sKty9r0vxwW8mfGJ43W9Cc9stoA/N
PVNza5QfmqueWqk9K/Hcs60Mf5PSxeaKZyldW9k01zU282ywvvAVom8eukWp
XZlWWq7IZbnysn/dY2+I0F8ArC83vTjpC89M5HxIX6pcajHv0k9VTpYt/QcU
QHAl\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{440, 71},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->87330506]
}, Open  ]],

Cell[TextData[{
 "The Poincar\[EAcute] section of interest in this case is the collection of \
points in the ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "2"], "-", 
    SubscriptBox["Y", "4"]}], TraditionalForm]], "InlineMath"],
 " plane when the orbit passes through ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "1"], "=", " ", "0"}], TraditionalForm]], 
  "InlineMath"],
 ".",
 " "
}], "Text",
 CellID->1626],

Cell[TextData[{
 "Since the actual result of the numerical integration is not required, it is \
possible to avoid storing all the data in ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " by specifying the output variables list (in the second argument to ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ") as empty, or {}. This means that ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " will produce no ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " as output, avoiding storing a lot of unnecessary data.",
 " ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " does give a message ",
 StyleBox["NDSolve::noout", "InlineCode"],
 " warning there will be no output functions, but it can safely be turned off \
in this case since the data of interest is collected from the event actions."
}], "Text",
 CellID->32663],

Cell["\<\
The linear interpolation event location method is used because the purpose of \
the computation here is to view the results in a graph with relatively low \
resolution. If you were doing an example where you needed to zoom in on the \
graph to great detail or find a feature, such as a fixed point of the Poincar\
\[EAcute] map, it would be more appropriate to use the default location \
method.\
\>", "Text",
 CellID->11991],

Cell["This turns off the message warning about no output.", "MathCaption",
 CellID->8292],

Cell[BoxData[
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{"NDSolve", "::", "noout"}], "]"}], ";"}]], "Input",
 CellLabel->"In[35]:=",
 CellID->7492],

Cell[TextData[{
 "This integrates the H\[EAcute]non-Heiles system using a fourth order \
explicit Runge Kutta method with fixed step size of 0.25.",
 " ",
 "The event action is to ",
 Cell[BoxData[
  ButtonBox["Sow",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sow"]], "InlineFormula"],
 " the values of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["Y", "2"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["Y", "4"], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->954],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", "\[IndentingNewLine]", 
   RowBox[{"Reap", "[", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"NDSolve", "[", 
      RowBox[{"eqns", ",", 
       RowBox[{"{", "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"T", ",", " ", "10000"}], "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"EventLocator", ",", 
          RowBox[{"\"\<Event\>\"", "\[Rule]", 
           RowBox[{
            SubscriptBox["Y", "1"], "[", "T", "]"}]}], ",", 
          RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
           RowBox[{"Sow", "[", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{
               SubscriptBox["Y", "2"], "[", "T", "]"}], ",", " ", 
              RowBox[{
               SubscriptBox["Y", "4"], "[", "T", "]"}]}], "}"}], "]"}]}], ",",
           
          RowBox[{
          "\"\<EventLocationMethod\>\"", "->", 
           "\"\<LinearInterpolation\>\""}], ",", "\[IndentingNewLine]", 
          RowBox[{"\"\<Method\>\"", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<FixedStep\>\"", ",", 
             RowBox[{"\"\<Method\>\"", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
                RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "4"}]}], 
               "}"}]}]}], "}"}]}]}], "}"}]}], ",", "\[IndentingNewLine]", 
       RowBox[{"StartingStepSize", "\[Rule]", "0.25"}], ",", " ", 
       RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}], ";"}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[36]:=",
 CellID->1030],

Cell[TextData[{
 "This plots the Poincar\[EAcute] section.",
 " ",
 "The collected data is found in the last part of the result of ",
 Cell[BoxData[
  ButtonBox["Reap",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reap"]], "InlineFormula"],
 " and the list of points is the first part of that."
}], "MathCaption",
 CellID->18249],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"psdata", "=", 
   RowBox[{"data", "[", 
    RowBox[{"[", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"psdata", ",", 
   RowBox[{"Axes", "\[Rule]", "False"}], ",", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]}], "Input",
 CellLabel->"In[37]:=",
 CellID->21375],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 352},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->80761620]
}, Open  ]],

Cell["\<\
Since the H\[EAcute]non\[Dash]Heiles system is Hamiltonian, a symplectic \
method gives much better qualitative results for this example.\
\>", "Text",
 CellID->1425],

Cell[TextData[{
 "This integrates the H\[EAcute]non\[Dash]Heiles system using a fourth order \
symplectic partitioned Runge\[Dash]Kutta method with fixed step size of \
0.25.",
 " ",
 "The event action is to ",
 Cell[BoxData[
  ButtonBox["Sow",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sow"]], "InlineFormula"],
 " the values of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["Y", "2"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["Y", "4"], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->8493],

Cell[BoxData[
 RowBox[{
  RowBox[{"sdata", "=", "\[IndentingNewLine]", 
   RowBox[{"Reap", "[", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"NDSolve", "[", 
      RowBox[{"eqns", ",", 
       RowBox[{"{", "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"T", ",", " ", "10000"}], "}"}], ",", "\[IndentingNewLine]", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"EventLocator", ",", 
          RowBox[{"\"\<Event\>\"", "\[Rule]", 
           RowBox[{
            SubscriptBox["Y", "1"], "[", "T", "]"}]}], ",", 
          RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
           RowBox[{"Sow", "[", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{
               SubscriptBox["Y", "2"], "[", "T", "]"}], ",", " ", 
              RowBox[{
               SubscriptBox["Y", "4"], "[", "T", "]"}]}], "}"}], "]"}]}], ",",
           
          RowBox[{
          "\"\<EventLocationMethod\>\"", "->", 
           "\"\<LinearInterpolation\>\""}], ",", "\[IndentingNewLine]", 
          RowBox[{"\"\<Method\>\"", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<SymplecticPartitionedRungeKutta\>\"", ",", 
             RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "4"}], ",", 
             RowBox[{"\"\<PositionVariables\>\"", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{
                RowBox[{
                 SubscriptBox["Y", "1"], "[", "T", "]"}], ",", " ", 
                RowBox[{
                 SubscriptBox["Y", "2"], "[", "T", "]"}]}], "}"}]}]}], 
            "}"}]}]}], "}"}]}], ",", "\[IndentingNewLine]", 
       RowBox[{"StartingStepSize", "\[Rule]", "0.25"}], ",", " ", 
       RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}], ";"}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[39]:=",
 CellID->22291],

Cell[TextData[{
 "This plots the Poincar\[EAcute] section.",
 " ",
 "The collected data is found in the last part of the result of ",
 Cell[BoxData[
  ButtonBox["Reap",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reap"]], "InlineFormula"],
 " and the list of points is the first part of that."
}], "MathCaption",
 CellID->162],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"psdata", "=", 
   RowBox[{"sdata", "[", 
    RowBox[{"[", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"psdata", ",", 
   RowBox[{"Axes", "\[Rule]", "False"}], ",", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]}], "Input",
 CellLabel->"In[40]:=",
 CellID->12771],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 352},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->28389329]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The ABC flow", "Subsubsection",
 CellID->9537],

Cell["\<\
This loads an example problem of the Arnold\[Dash]Beltrami\[Dash]Childress \
(ABC) flow which is used to model chaos in laminar flows of the \
three-dimensional Euler equations.\
\>", "Text",
 CellID->17140],

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", "=", 
   RowBox[{
   "GetNDSolveProblem", "[", "\"\<ArnoldBeltramiChildress\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<System\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"icvars", " ", "=", " ", 
   RowBox[{"vars", " ", "/.", " ", 
    RowBox[{"T", "\[Rule]", "0"}]}]}], ";"}]}], "Input",
 CellLabel->"In[42]:=",
 CellID->6888],

Cell[TextData[{
 "This defines a splitting ",
 Cell[BoxData["Y1"], "InlineFormula"],
 ", ",
 Cell[BoxData["Y2"], "InlineFormula"],
 " of the system by setting some of the right-hand side components to zero."
}], "MathCaption",
 CellID->30354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Y1", "=", "eqs"}], ";", " ", 
  RowBox[{
   RowBox[{"Y1", "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "2"}], "]"}], "]"}], " ", "=", " ", "0"}], ";", 
  "Y1"}]], "Input",
 CellLabel->"In[46]:=",
 CellID->5363],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "1"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{
      FractionBox["3", "4"], " ", 
      RowBox[{"Cos", "[", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}], "]"}]}], "+", 
     RowBox[{"Sin", "[", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}], "]"}]}]}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "2"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "3"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"Cos", "[", 
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], "]"}], "+", 
     RowBox[{
      FractionBox["3", "4"], " ", 
      RowBox[{"Sin", "[", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}], "]"}]}]}]}]}], 
  "}"}]], "Output",
 ImageSize->{537, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]=",
 CellID->1040350069]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Y2", "=", "eqs"}], ";", 
  RowBox[{
   RowBox[{"Y2", "[", 
    RowBox[{"[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "3"}], "}"}], ",", "2"}], "]"}], "]"}], " ", "=", 
   " ", "0"}], ";", " ", "Y2"}]], "Input",
 CellLabel->"In[47]:=",
 CellID->20967],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "1"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "2"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"Cos", "[", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}], "]"}], "+", 
     RowBox[{"Sin", "[", 
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], "]"}]}]}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "3"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}]}], 
  "}"}]], "Output",
 ImageSize->{366, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[47]=",
 CellID->1476672042]
}, Open  ]],

Cell["This defines the implicit midpoint method.", "MathCaption",
 CellID->3636],

Cell[BoxData[
 RowBox[{
  RowBox[{"ImplicitMidpoint", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"\"\<ImplicitRungeKutta\>\"", ",", " ", 
     RowBox[{
     "\"\<Coefficients\>\"", "\[Rule]", 
      "\"\<ImplicitRungeKuttaGaussCoefficients\>\""}], ",", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", 
     RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<FixedPoint\>\"", ",", 
        RowBox[{"AccuracyGoal", "\[Rule]", "10"}], ",", " ", 
        RowBox[{"PrecisionGoal", "\[Rule]", "10"}], ",", 
        RowBox[{"\"\<IterationSafetyFactor\>\"", "\[Rule]", "1"}]}], 
       "}"}]}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[48]:=",
 CellID->3461],

Cell["\<\
This constructs a second-order splitting method that retains volume and \
reversing symmetries.\
\>", "MathCaption",
 CellID->2519],

Cell[BoxData[
 RowBox[{
  RowBox[{"ABCSplitting", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"\"\<Splitting\>\"", ",", "\[IndentingNewLine]", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"\"\<Equations\>\"", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"Y2", ",", "Y1", ",", "Y2"}], "}"}]}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"\"\<Method\>\"", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
       "\"\<LocallyExact\>\"", ",", "ImplicitMidpoint", ",", 
        "\"\<LocallyExact\>\""}], "}"}]}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[49]:=",
 CellID->8513],

Cell["\<\
This defines a function which gives the Poincar\[EAcute] section for a \
particular initial condition.\
\>", "MathCaption",
 CellID->1106],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"psect", "[", "ics_", "]"}], " ", ":=", " ", "\[IndentingNewLine]",
    
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", "reapdata", "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"reapdata", " ", "=", " ", "\[IndentingNewLine]", 
       RowBox[{"Reap", "[", "\[IndentingNewLine]", 
        RowBox[{"NDSolve", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"eqs", ",", 
            RowBox[{"Thread", "[", 
             RowBox[{"icvars", " ", "\[Equal]", " ", "ics"}], "]"}]}], "}"}], 
          ",", 
          RowBox[{"{", "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"T", ",", "1000"}], "}"}], ",", "\[IndentingNewLine]", " ",
           
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"EventLocator", ",", "\[IndentingNewLine]", 
             RowBox[{"\"\<Event\>\"", "\[Rule]", 
              RowBox[{
               SubscriptBox["Y", "2"], "[", "T", "]"}]}], ",", " ", 
             RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
              RowBox[{"Sow", "[", 
               RowBox[{"{", 
                RowBox[{
                 RowBox[{
                  SubscriptBox["Y", "1"], "[", "T", "]"}], ",", " ", 
                 RowBox[{
                  SubscriptBox["Y", "3"], "[", "T", "]"}]}], "}"}], "]"}]}], 
             ",", " ", "\[IndentingNewLine]", 
             RowBox[{
             "\"\<EventLocationMethod\>\"", "->", 
              "\"\<LinearInterpolation\>\""}], ",", 
             RowBox[{"Method", "\[Rule]", "ABCSplitting"}]}], "}"}]}], ",", 
          " ", "\[IndentingNewLine]", 
          RowBox[{"StartingStepSize", "\[Rule]", 
           RowBox[{"1", "/", "4"}]}], ",", 
          RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}], 
        "\[IndentingNewLine]", "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"reapdata", "[", 
       RowBox[{"[", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "1"}], "]"}], "]"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[51]:=",
 CellID->13285],

Cell["\<\
This finds the Poincar\[EAcute] sections for several different initial \
conditions and flattens them together into a single list of points.\
\>", "MathCaption",
 CellID->23750],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"data", " ", "=", " ", "\[IndentingNewLine]", 
   RowBox[{"Mod", "[", 
    RowBox[{
     RowBox[{"Map", "[", 
      RowBox[{"psect", ",", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"4.267682454609692", ",", "0", ",", "0.9952906114885919"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"1.6790790859443243", ",", "0", ",", "2.1257099470901704"}],
           "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"2.9189523719753327", ",", "0", ",", "4.939152797323216"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"3.1528896559036776", ",", "0", ",", "4.926744120488727"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"0.9829282640373566", ",", "0", ",", "1.7074633238173198"}],
           "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"0.4090394012299985", ",", "0", ",", "4.170087631574883"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"6.090600411133905", ",", "0", ",", "2.3736566160602277"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"6.261716134007686", ",", "0", ",", "1.4987884558838156"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"1.005126683795467", ",", "0", ",", "1.3745418575363608"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"1.5880780704325377", ",", "0", ",", "1.3039536044289253"}],
           "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"3.622408133554125", ",", "0", ",", "2.289597511313432"}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{
          "0.030948690635763183", ",", "0", ",", "4.306922133429981"}], "}"}],
          ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"5.906038850342371", ",", "0", ",", "5.000045498132029"}], 
          "}"}]}], "}"}]}], "]"}], ",", "\[IndentingNewLine]", 
     RowBox[{"2", "\[Pi]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", "data", "]"}]}], "Input",
 CellLabel->"In[52]:=",
 CellID->13704],

Cell[BoxData[""], "Output",
 ImageSize->{8, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[53]=",
 CellID->310208007]
}, Open  ]],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJwVWWk8lV8XvfdShNxZNCgNlDQY06Czo7c0SIRS/GVIkTJLiJsiSkUJhUop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     "]]}, 
   {Hue[0.9060679774997897, 0.6, 0.6], PointBox[CompressedData["
1:eJwdmXkglc8Xxi9F9rz3Xi2iKJQtVLbITLJV9iSF7Em2ElKWCBGSFFkiZYlS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     "]]}, 
   {Hue[0.14213595499957954`, 0.6, 0.6], PointBox[CompressedData["
1:eJwV1Hk0lWkcB/CrbJd7r+sur2NCkrSgmpMolC/SMJJKMYXKchGlVC6mwe1m
mWyRrWnCVSok2kyRMJbTFSki6siScDSOZUSRZZ754z3Pec7n+S3n/J7nXeF5
cp9gCY1GO0i+/1fx1u3u4m45zNzwW+rWz8QhbpBrzxcFdFikU93GPLza6dmS
bCqPrlfHJ5mNTJSMTC101SvCO7T2G7uTh4B7oyaCe/LI79LzsFZgwZfK+K5R
QceaDBkv1TQ+Omc8++kHFXBqqVx7UzwLfgXSn/tuKUHRVDLdbk1B8GjxV7GR
IpJnrmml1akgZGx1XpiHMpLNt/ucy6JwK2d7YpUhHU/+SL61LYsNj72mcxhQ
Bi21UKjfQGGVpKmBrqKE5Lxj6aX2qsjhyuTq2jPwem2Lrk0tyW+ut6hTqYTX
uaPDt9+oYn6j/cTsZQa0faW+vY8pGMtOllVtU0ZvrWaszlYOMhZ+kto9Z+Co
11fDhtsU/hoySxHlEO8Lv/5LEge2pR4p2SMMVK8zuBB+lYJDQOVAZ78y9vg/
2Mhu4EBr5a6uWSUmqqn2mNhYCsfOvS/K5TEguTtnHD/OQTvL4UShNhMSn37t
1HAK2VmVTt5WDCD4TGSeJhfGzRHlLiZM9ObcZLR5U5CZ1s7MCmaAVhk0bOnK
heEP03fKHZkQqW1sPW9CodRQ54X4Ksn/oLKHLuZivqJjj6snE0dby7rZahQa
2y7WGrSQeAc/ny3NXNTbF9DLzpL+2qwPGTXz8Tg5Z2pwCelnS/rBHWo8iMNW
VawIJPmvlIkDtfmoDGw38trFBDZrVMcV8lA0xiutY8uBlsaviYQqdrQ88/9b
KguawdnMbg02/PJfvfFylINIf/pia4kq4tZf71FvIz4qFNYUquDfE2HhY9HE
VYQNiTQOWlJ0RMGDxIWFrm+MVCCIEnwNf0icMrNoJ/NIKV6W+/s88ZBGl5FG
FkK/nF5v85HU7yiSWgdxYBWid9uVQ/aTAf3NV1gIiUvKjmDJgyZmhZyRcHDA
8dqOIn3iilWxbm4sTE5payaCeOKZx1QrB/HCzD5LK+JF9pcNHViYnw8JGQuW
h8jGwvveHAffh+QkUT7E43Tr3/NZ6KMXl9jWEncz4dVYcPGxXPjCL4l41xG2
jBwLnwtGSuQMFSCq4H4cyuBCQXbMp11KPGHL3mNDTMzrWdS7VRP3fe7LVeVh
9FOMgaUG6Sdo/9BxKRMWjpTNTKQi6SfW9rcbPBzVzk5LiyF+ciYiZ5wJ6kO0
3eIJOmjv+mytD/GBhPFUq+UKoPn7DWEfC7rtSw8HeyqBFru5o4dHoT6yIHzz
Z+KWc6tXuKqAF72oTltP3o9k+ZxzDIWKwn1rBb2KEJUekbh5sLG/seyST5Uy
RE70yQnyfmTUo5xNX5J6Cn3vs9VUocf6Zla7gQFR6GB+eh0FnanW2dlEUu/d
dSchmXc9tV/9bgTx4nDTDc8obIo+cHhAneTLaZpwWcZBVPbCsMMT4kW0gE3F
FC5rqj9cjCL+gXKmh3JgXv1WkeonnpRwVvYaiTed8vZ+Sdz6z5rB+xy4V6zs
71og91mqpXMqk0Iq/+ZXfQY532pw/t0oB2aOS7gJmuT+HsgM+xJM4dvAi61d
5uT8j8NxyVpcWNwsm482YpL5O++udqdgklgYGC8g8YJna3p2cvEpU3w/wI64
vvbbp5YUOtm63W9TifvbrrsTwcWQubRp2J3k96m9FK5OQfi5bKSihvhYV8jT
ci5SLWPtdp8inu3ScrKbD5d15XqG5H9Ba1N55DHOxT/9dcb5oSR/XsDA+Gk+
GlqCnNaYkv3IBDZd4OE/s0hL3A==
     "]]}, 
   {Hue[0.37820393249936934`, 0.6, 0.6], PointBox[CompressedData["
1:eJwVlns4VNsbxyelGSPa2947txJxyshhOCdK4hUHIaGDXBODptSgXH5Ol0lK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     "]]}, 
   {Hue[0.6142719099991583, 0.6, 0.6], PointBox[CompressedData["
1:eJwdWXk41N8XHstYBzNjzBSREPpKaUNK9ySlQki0KAkpSputUpZUKpRosWdL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     "]]}, 
   {Hue[0.8503398874989481, 0.6, 0.6], PointBox[CompressedData["
1:eJwVk3tUjGkcx98218rMvO/MOzMJxVrRhSSLtJ4fo+gy1aKQqTVdVOpYe9oy
naI3ujgUqk2JTHRZxeqyRxyTmmK3MkkXlZrSZaTYWWs6Rslm9/HHc54/Pr/n
+zvP7/M8SwN/3BXyFUEQPnh92UWy1tj2jm4U8qCrcm8bCXOzpj4tbBeAheRc
+w/RXBA/9vTIlQkAgktnbrnREDuU59tgKATYHOJdPERBld1oYFq8AApueo2a
19AQuq8mPAWEMFSmP1fhQ4HMMvRaXKYAiH6kY7H4sGEq2OfSGSFYyEo0zQ9J
SHGKUDZWC0AZV6Ke68yHlsq+Ax7DQiDK/BlkTUK/Hbc3cEwAQ9faGyd286H/
kTOn6pEcMdaX5udmc4CXI3nxeYUQmDeOzkcRH5Idj/TwQ5WI6CgOqJ9hQ02j
orQsBucZu6T8rqEhI9zRrFyqQsSGo40cezb0a7IpTZcQlOnJ12+G0yC/EHy2
lupARO6iipgmFnSKSOvvUnMQlGwTaB/zIKo2eweX6UKMX0rUIRULnrtG9Hnq
apDy/YKi6NU80A/K1Zbkc8Rc1Gb/ZsSGUrv+9+Y3mpBSVbEhMZMLW92VA7KB
XsRM6sLW57MhxXR9YrZ3G1LO8aqqZ3OBvHDYbH9RHyIWm+hN7DkQkbNnp+3f
nYhpmQiEGArikjzka67i8yeUIuFbDnR4f0s1JPUggKHc8askWC67u2w39kfc
Lj29KoYL8X2Ga4exP8ZtcL4G+3NfXrYmfw6e14yn/59qCpzKZQsMTuD61gqP
uk4ayj+mL65zxvOy6Dwm8qBA/f7lq/ZsfP7e9eONi/gwKh3zsMjAfIqxou+T
EBPf+i5Ugfnh81xjTz6IVVezjV7jfNsyV8kyEqReCpVEi/mg1GLamw+z0oev
uWUWISZFsiQjFd9v+lxisS3OWxdyMXYTH+T7n5qzE+sRsc9s4uJrNoz95yIm
EzDvzjou6qXBN7WgKNS1BRF2hnX3lrLh1IHT7jGDuB/xcoSzl4YXOlM/h5+w
v3kZy+uwv9mx+Xt5FQWISIl2yanmQVOLsiO5uQsRvpGrz3SyYESz2nTFP7WI
+HfGoc2cBwH2wmP+Hs8RMVs92Yfzh9+l6qUuzYiRJLhWpnFhRmG48Y+pXkSo
SxQ3r7Bh1SeDapm6DRFPnfZsn6TAcOCzQbK8DzE3DC6Hu3BgSpFJtS18hpiM
pVHfH6FAGuQzLr2D8yVWPx/bSoL/4ZMRk7Y9iJj+1da5moRZgdrOvH7sgzPN
8hVzwfqktIf+4m+byEYtocHINf1lqAm+b6HfifgnFIT2HqwNSsD1y2MbE1U0
BIcni+a5Y+5wO5q7nYL5yqAmr1zMq0oH6s34sI7raVKRg3nYqDq+koTCtana
qlrMrSp9ruzkw7BKdzZSh+ddmyYv4ZMwe0UkWajD/UfFrF1efAg6pBux2V2K
iDd1tW4yDmhXNj8AB1z/zdiOBDs+3NX7n7fJbUDEUNjg+i42lLSIu/JSMd91
dtZgKw3nbV4d3Gz1BBFi4+tCkg1OIW8nyC/vZTz8Sr4zDUuiZGXQgv2Vfhxx
wf4+rJSRcdJixGy/p8kq5oH6wBlHwYJu3D/pr6luFsysyv90oVCJmKoPdJMR
D/ZwMyXiBDxfy2DO4jVs2JIXDbcsHyNiyyE95zgXLjOxNqZC/L9kxr8kNbBB
79u9Vf8R+4v8OqwV+9u0UXtfX475pTt02hYOPLQfT1UkPEPEjXKJiR8FqA9O
ZrGw/yWKqScWJOx/MHEqwAf3MynQfg4g4X8qE0cm
     "]]}, 
   {Hue[0.08640786499873876, 0.6, 0.6], PointBox[CompressedData["
1:eJwV1nlcjNsfB/CnRZSKq595nmyVkitFUVLS92S/WrRRlEyhQqGuqLTMxdVi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     "]]}, 
   {Hue[0.3224758424985268, 0.6, 0.6], PointBox[CompressedData["
1:eJwVmHk8VO8Xx8c+9hlmhtCCiJQlS4t4DkkiLVIpkUgUoaJkSwsSRUX5SciW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     "]]}, 
   {Hue[0.5585438199983166, 0.6, 0.6], PointBox[CompressedData["
1:eJwV13k4VO0bB/AzY99irKOohJKlyBbJc1NRSqTs5VVkS71aJBKjRUULspUl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     "]]}, 
   {Hue[0.7946117974981064, 0.6, 0.6], PointBox[CompressedData["
1:eJwVlnk8VO0bxo+xjMk6GDNFXm1URIuyRM9NmyVvokWIiRghS9ZexSGylVK2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     "]]}, 
   {Hue[0.030679774997896203`, 0.6, 0.6], PointBox[CompressedData["
1:eJwV1nk8VOsfB/BjFnNmzJhEKmRJv7ZbkaRbMuerNFGaSrqyFAkl0k4bjZAl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     "]]}, 
   {Hue[0.266747752497686, 0.6, 0.6], PointBox[CompressedData["
1:eJwdlnk8VesXxrchlzLs7exTR8bIkKE5rql30cCNXCXXPKRBZdalMrUzS2XK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     "]]}, 
   {Hue[0.5028157299974758, 0.6, 0.6], PointBox[CompressedData["
1:eJwVlHlQU1cUh5+gDC5kJS8BLQQoyuaCLEWlvUedUkRpAUW0BEhRYJBNRUtB
oU9GlE4LEllC3UhAFmOgKqAllSZF0SiLiARZHIhiKYJxEGTfevnjzZ0739zz
7pzvd65FaJxfmB5BEAfxt7gaapw/PG7iwm/q/tOrBWyYRv+SeZFccAbjK1Mi
DiRJFx4ELufBvvr+AJ9aFtzb9ioxMYELjIcJSvchDqyfNGmSe/Eg/ciFFYQL
C3I/b/0qMYcL6Y4aYngzCUe6hsKXX+aBus/0VU4BE9hRm8JltVxoER+8XxVA
gtnxqZE3Ezwg9Dt2mi9lgptHZ8LqT7j+D4MGgiASCHn0290JFSgrafuDASED
3P+rshvYwgNJTeqA/yYS+NJQm1zbh4io+ZCfUkyH4JNPm3wu8oAvD7R82cQB
1Wj+SmV1MyKSmvPcemhQJLbN9JvkASNANlbpxIGsVTMNdqIXiP9XEeFURgNC
Nn431LwCaT9L1z3PMgZo+yAdbmpHKl9RopGKBpK9faJedR3S+sULvafYALnW
yWhbJ6JuiOjGLDpI1PkniYRGRMRUKcbC2ZDlKpiL6OtCYFTHeF9KB2qB15IZ
0Yq0ExNMz+cs8FEv0dSc6UZUfInTMw8GSH76ve2fZA3ir0/7he3DAmJzllV1
aScSjpWjwi+ZoO3tCFyT0YkkeWXp5z2YMDp0brfuKheo6FXXnhkaw56nOWZz
J7gAnq8NtX3GIB+xjBrt4gKhywkRO7LBP4jmbRHFBVWVItY+mwP9ufO5KiYP
VBeXhKyTsyDIxszD8RTmH4Ovlo9zIL68VLF2P/YTpmcTz2fBVIou/e9LmNuT
RSQigVkkGgyW8QDCXbakn2ECc7heJqnH9/noO+IWSMLuJwm93oYmQDWcCA57
x4CWbTfMZfP4/OXyuRIBCTCQtKw4tRKp3pO2rl8zQOcy2znvge9Taz78yYIE
rYtTtRXjESJO7MuPSqVD5oV6+wOFmFvXNnCrsL+gW75xVi2Icu6XLWunwdoU
sx0uy01A9Ufx/WJTDvAne/UOWLchSrmw0+g6DVS3J789ZFCJ4PR23x0/GgPR
FqYPO14iojCkJLaBBsLu5GsmBThPg7GzPd3Yb+KmQlE29nukNebJBjrgAB6T
a5oQZLiVKn3YIMx38N/L7EbEHuGvVio6ZK38JqC4uRVRh9VzmnoWUO1X/SAO
8/dbo7wjGMAYV1xfE9iOqC8SVtwwZ0GMH2n5tgz72ciQM/H/JOKCRkU07p8m
a/86ew6I7zaKMnoxf+pgGWLLhqVpyR5xmBPR3lIjOQf8Lcb0O1bzgBIb7JrO
w37WHOdf/BlzRweBpyEJke6Rla8PYe7AmG3TZ0G5qxDCpbj+1FRAqBcJPfn2
MfIa7JdvKMg8zASL9lvR4/g9IFzJmDk8f5XO0vlpngkQ832s4Gbs50/F1h78
HlB2pwTV2J/K+XvNS+U9RKWJBhdsGLA5YqOJch+uN/FiZK8x5t3Ozxn9jxGx
zmJDQBAdhrPzJh7dwed3dineXOIACMXuGm0LIoy2Xj+A+x/3kRvxyQTnpaL0
SskSDkh2WdPNK9sQMaspmJbSgCpoP88fuIuowCqdngDPZ8X+uu23sb/4s2VU
B/YXPJgR4oPz8tbTIbuODVql/sjSvk68143qeWJ/jdXFR1OeIcrUKvIwnoej
oz5x9C2L/laNc8focKvGlbiX9wJRlndMb0pZ0KKjB6dE4/m0mTvbcwznV2E0
LZ5pR8TtGZnzDBO4zZLH56pxvyYm/HgyNuQpvWJnF/2MeN0MXM+B71QsR8HQ
Ird/l2jABt25KqYwFu/T1PTmeg7Q7g+0gS3uV/xZnd1JFvwPKHlmdg==
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{{0., 6.282805996206862}, {0., 5.696454862748169}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Input",
 CellID->1998685447]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Bouncing Ball", "Subsection",
 CellID->640],

Cell[TextData[{
 "This example is a generalization of an example in [",
 ButtonBox["SGT03",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24366"],
 "]. It models a ball bounding down a ramp with a given profile.",
 " ",
 "The example is good for demonstrating how you can use multiple invocation \
of ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " with event location to model some behavior."
}], "Text",
 CellID->24972],

Cell["\<\
This defines a function which computes the solution from one bounce to the \
next. The solution is computed up until the next time the path intersects the \
ramp.\
\>", "MathCaption",
 CellID->20353],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"OneBounce", "[", 
    RowBox[{"k_", ",", " ", "ramp_"}], "]"}], "[", 
   RowBox[{"{", 
    RowBox[{
    "t0_", ",", "x0_", ",", " ", "xp0_", ",", " ", "y0_", ",", " ", "yp0_"}], 
    "}"}], "]"}], " ", ":=", " ", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "sol", ",", " ", "t1", ",", " ", "x1", ",", " ", "xp1", ",", " ", "y1", 
      ",", " ", "yp1", ",", " ", "gramp", ",", " ", "gp"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"sol", " ", "=", " ", 
      RowBox[{"First", "[", 
       RowBox[{"NDSolve", "[", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{"x", "''"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
            "0"}], ",", " ", 
           RowBox[{
            RowBox[{
             RowBox[{"x", "'"}], "[", "t0", "]"}], "\[Equal]", " ", "xp0"}], 
           ",", " ", 
           RowBox[{
            RowBox[{"x", "[", "t0", "]"}], " ", "\[Equal]", " ", "x0"}], ",", 
           "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{
             RowBox[{"y", "''"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
            RowBox[{"-", "9.8"}]}], " ", ",", " ", 
           RowBox[{
            RowBox[{
             RowBox[{"y", "'"}], "[", "t0", "]"}], " ", "\[Equal]", " ", 
            "yp0"}], ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "t0", "]"}], " ", "\[Equal]", " ", "y0"}]}], 
          "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"{", 
          RowBox[{"t", ",", "t0", ",", " ", "\[Infinity]"}], "}"}], ",", " ", 
         
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<EventLocator\>\"", ",", " ", 
            RowBox[{"\"\<Event\>\"", "\[RuleDelayed]", " ", 
             RowBox[{
              RowBox[{"y", "[", "t", "]"}], " ", "-", " ", 
              RowBox[{"ramp", "[", 
               RowBox[{"x", "[", "t", "]"}], "]"}]}]}]}], "}"}]}], ",", 
         RowBox[{"MaxStepSize", "\[Rule]", "0.01"}]}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"t1", " ", "=", " ", 
      RowBox[{
       RowBox[{"InterpolatingFunctionDomain", "[", 
        RowBox[{"x", " ", "/.", " ", "sol"}], "]"}], "[", 
       RowBox[{"[", 
        RowBox[{"1", ",", 
         RowBox[{"-", "1"}]}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x1", ",", "xp1", ",", "y1", ",", " ", "yp1"}], "}"}], " ", 
      "=", " ", 
      RowBox[{
       RowBox[{"Reflection", "[", 
        RowBox[{"k", ",", " ", "ramp"}], "]"}], "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"x", "[", "t1", "]"}], ",", " ", 
          RowBox[{
           RowBox[{"x", "'"}], "[", "t1", "]"}], ",", " ", 
          RowBox[{"y", "[", "t1", "]"}], ",", " ", 
          RowBox[{
           RowBox[{"y", "'"}], "[", "t1", "]"}]}], "}"}], " ", "/.", " ", 
        "sol"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Sow", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"x", "[", "t", "]"}], " ", "/.", "sol"}], ",", 
         RowBox[{
         "t0", " ", "\[LessEqual]", " ", "t", " ", "\[LessEqual]", " ", 
          "t1"}]}], "}"}], ",", " ", "\"\<X\>\""}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Sow", "[", 
      RowBox[{
       RowBox[{"{", " ", 
        RowBox[{
         RowBox[{
          RowBox[{"y", "[", "t", "]"}], " ", "/.", " ", "sol"}], ",", 
         RowBox[{
         "t0", " ", "\[LessEqual]", " ", "t", " ", "\[LessEqual]", " ", 
          "t1"}]}], "}"}], ",", " ", "\"\<Y\>\""}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Sow", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"x1", ",", "y1"}], "}"}], ",", " ", "\"\<Bounces\>\""}], 
      "]"}], ";", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"t1", ",", " ", "x1", ",", "xp1", ",", "y1", ",", " ", "yp1"}], 
      "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[54]:=",
 CellID->3465],

Cell[TextData[{
 "This defines the function for the bounce when the ball hits the ramp. The \
formula is based on reflection about the normal to the ramp assuming only the \
fraction ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 " of energy is left after a bounce."
}], "MathCaption",
 CellID->45],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Reflection", "[", 
    RowBox[{"k_", ",", " ", "ramp_"}], "]"}], "[", 
   RowBox[{"{", 
    RowBox[{"x_", ",", " ", "xp_", ",", " ", "y_", ",", " ", "yp_"}], "}"}], 
   "]"}], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"gramp", ",", " ", "gp", ",", " ", "xpnew", ",", " ", "ypnew"}], 
     "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"gramp", " ", "=", " ", 
      RowBox[{"-", 
       RowBox[{
        RowBox[{"ramp", "'"}], "[", "x", "]"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"Not", "[", 
        RowBox[{"NumberQ", "[", "gramp", "]"}], "]"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Print", "[", "\"\<Could not compute derivative \>\"", "]"}], 
        ";", "\[IndentingNewLine]", 
        RowBox[{"Throw", "[", "$Failed", "]"}]}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"gramp", " ", "=", " ", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{
          RowBox[{"ramp", "'"}], "[", "x", "]"}]}], ",", " ", "1"}], "}"}]}], 
     ";", "\[IndentingNewLine]", 
     RowBox[{"If", "[", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"gramp", ".", 
         RowBox[{"{", 
          RowBox[{"xp", ",", "yp"}], "}"}]}], " ", "\[Equal]", " ", "0"}], 
       ",", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Print", "[", "\"\<No reflection\>\"", "]"}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"Throw", "[", "$Failed", "]"}]}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"gp", "=", " ", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", 
         RowBox[{"-", "1"}]}], "}"}], " ", 
       RowBox[{"Reverse", "[", "gramp", "]"}]}]}], ";", "\[IndentingNewLine]",
      
     RowBox[{
      RowBox[{"{", 
       RowBox[{"xpnew", ",", " ", "ypnew"}], "}"}], " ", "=", " ", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"k", "/", 
         RowBox[{"(", 
          RowBox[{"gramp", ".", "gramp"}], ")"}]}], ")"}], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"gp", " ", 
          RowBox[{"gp", ".", 
           RowBox[{"{", 
            RowBox[{"xp", ",", "yp"}], "}"}]}]}], " ", "-", " ", 
         RowBox[{"gramp", " ", 
          RowBox[{"gramp", ".", 
           RowBox[{"{", 
            RowBox[{"xp", ",", " ", "yp"}], "}"}]}]}]}], ")"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"x", ",", " ", "xpnew", ",", " ", "y", ",", " ", "ypnew"}], 
      "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[55]:=",
 CellID->5306],

Cell["\<\
This defines the function which runs the bouncing ball simulation for a given \
reflection ratio, ramp, and starting position.\
\>", "MathCaption",
 CellID->31126],

Cell[BoxData[
 RowBox[{
  RowBox[{"BouncingBall", "[", 
   RowBox[{"k_", ",", " ", "ramp_", ",", 
    RowBox[{"{", 
     RowBox[{"x0_", ",", "y0_"}], "}"}]}], "]"}], " ", ":=", " ", 
  "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "data", ",", " ", "end", ",", " ", "bounces", ",", " ", "xmin", ",", " ",
       "xmax", ",", " ", "ymin", ",", " ", "ymax"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{"y0", " ", "<", " ", 
        RowBox[{"ramp", "[", "x0", "]"}]}], ",", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Print", "[", "\"\<Start above the ramp\>\"", "]"}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"Return", "[", "$Failed", "]"}]}]}], "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"data", " ", "=", 
      RowBox[{"Reap", "[", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Catch", "[", 
         RowBox[{
          RowBox[{"Sow", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"x0", ",", "y0"}], "}"}], ",", " ", "\"\<Bounces\>\""}], 
           "]"}], ";", "\[IndentingNewLine]", 
          RowBox[{"NestWhile", "[", 
           RowBox[{
            RowBox[{"OneBounce", "[", 
             RowBox[{"k", ",", " ", "ramp"}], "]"}], ",", " ", 
            RowBox[{"{", 
             RowBox[{"0", ",", "x0", ",", "0", ",", "y0", ",", "0"}], "}"}], 
            ",", " ", 
            RowBox[{"Function", "[", 
             RowBox[{
              RowBox[{"1", " ", "-", " ", 
               RowBox[{
                RowBox[{"#1", "[", 
                 RowBox[{"[", "1", "]"}], "]"}], "/", 
                RowBox[{"#2", "[", 
                 RowBox[{"[", "1", "]"}], "]"}]}]}], " ", ">", " ", "0.01"}], 
             "]"}], ",", " ", "2", ",", " ", "25"}], "]"}]}], "]"}], ",", "_",
         ",", " ", "Rule"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"end", " ", "=", " ", 
      RowBox[{"data", "[", 
       RowBox[{"[", 
        RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"data", " ", "=", " ", 
      RowBox[{"Last", "[", "data", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"bounces", " ", "=", " ", 
      RowBox[{"(", 
       RowBox[{"\"\<Bounces\>\"", " ", "/.", " ", "data"}], ")"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"xmax", " ", "=", " ", 
      RowBox[{"Max", "[", 
       RowBox[{"bounces", "[", 
        RowBox[{"[", 
         RowBox[{"All", ",", "1"}], "]"}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"xmin", " ", "=", " ", 
      RowBox[{"Min", "[", 
       RowBox[{"bounces", "[", 
        RowBox[{"[", 
         RowBox[{"All", ",", "1"}], "]"}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"ymax", " ", "=", " ", 
      RowBox[{"Max", "[", 
       RowBox[{"bounces", "[", 
        RowBox[{"[", 
         RowBox[{"All", ",", "2"}], "]"}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"ymin", " ", "=", " ", 
      RowBox[{"Min", "[", 
       RowBox[{"bounces", "[", 
        RowBox[{"[", 
         RowBox[{"All", ",", "2"}], "]"}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Show", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Plot", "[", 
         RowBox[{
          RowBox[{"ramp", "[", "x", "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", "xmin", ",", " ", "xmax"}], "}"}], ",", 
          RowBox[{"PlotRange", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{"xmin", ",", " ", "xmax"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"ymin", ",", " ", "ymax"}], "}"}]}], "}"}]}], ",", 
          RowBox[{"Epilog", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"PointSize", "[", ".025", "]"}], ",", 
             RowBox[{"Map", "[", 
              RowBox[{"Point", ",", "bounces"}], "]"}]}], "}"}]}], ",", " ", 
          RowBox[{"AspectRatio", "\[Rule]", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"ymax", " ", "-", " ", "ymin"}], ")"}], "/", 
            RowBox[{"(", 
             RowBox[{"xmax", " ", "-", " ", "xmin"}], ")"}]}]}]}], "]"}], ",",
         " ", 
        RowBox[{"ParametricPlot", "[", 
         RowBox[{
          RowBox[{"Evaluate", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"Piecewise", "[", 
              RowBox[{"\"\<X\>\"", " ", "/.", " ", "data"}], "]"}], ",", " ", 
             
             RowBox[{"Piecewise", "[", 
              RowBox[{"\"\<Y\>\"", " ", "/.", " ", "data"}], "]"}]}], "}"}], 
           "]"}], ",", " ", 
          RowBox[{"{", 
           RowBox[{"t", ",", "0", ",", "end"}], "}"}], ",", " ", 
          RowBox[{"PlotStyle", "\[Rule]", 
           RowBox[{"RGBColor", "[", 
            RowBox[{"1", ",", "0", ",", "0"}], "]"}]}]}], "]"}]}], "}"}], 
      "]"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[56]:=",
 CellID->27557],

Cell[TextData[{
 "This is the example which is done in",
 " ",
 "[",
 ButtonBox["SGT03",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24366"],
 "]."
}], "MathCaption",
 CellID->23296],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"ramp", "[", "x_", "]"}], " ", ":=", " ", 
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{"x", " ", "<", " ", "1"}], ",", 
     RowBox[{"1", " ", "-", " ", "x"}], ",", " ", "0"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"BouncingBall", "[", 
  RowBox[{".7", ",", " ", "ramp", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1.25"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[57]:=",
 CellID->1546],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 258},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[58]=",
 CellID->382114047]
}, Open  ]],

Cell["The ramp is now defined to be a quarter circle", "MathCaption",
 CellID->31880],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"circle", "[", "x_", "]"}], " ", ":=", " ", 
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{"x", " ", "<", " ", "1"}], ",", " ", 
     RowBox[{"Sqrt", "[", 
      RowBox[{"1", " ", "-", " ", 
       RowBox[{"x", "^", "2"}]}], "]"}], ",", " ", "0"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"BouncingBall", "[", 
  RowBox[{".7", ",", " ", "circle", ",", 
   RowBox[{"{", 
    RowBox[{".1", ",", "1.25"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[59]:=",
 CellID->18796],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 292},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[60]=",
 CellID->628848393]
}, Open  ]],

Cell["This adds a slight waviness to the ramp.", "MathCaption",
 CellID->19484],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"wavyramp", "[", "x_", "]"}], " ", ":=", " ", 
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{"x", " ", "<", " ", "1"}], ",", " ", 
     RowBox[{"1", " ", "-", " ", "x", " ", "+", " ", 
      RowBox[{".05", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{"11", " ", "Pi", " ", "x"}], "]"}]}]}], " ", ",", "0"}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{"BouncingBall", "[", 
  RowBox[{".75", ",", " ", "wavyramp", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1.25"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[61]:=",
 CellID->22693],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 247},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[62]=",
 CellID->11440604]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Event Direction", "Subsection",
 CellID->17194],

Cell[TextData[{
 "This example illustrates the solution of the restricted three body problem, \
a standard nonstiff test system of four equations. The example traces the \
path of a spaceship travelling around the moon and returning to the earth \
(see p. 246 of [",
 ButtonBox["SG75",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#23388"],
 "]). The ability to specify multiple events and the direction of the zero \
crossing is important."
}], "Text",
 CellID->10605],

Cell[TextData[{
 "The initial conditions have been chosen to make the orbit periodic. The \
value of ",
 Cell[BoxData[
  StyleBox["\[Mu]", "TR"]], "InlineFormula"],
 " corresponds to a spaceship travelling around the moon and the earth."
}], "MathCaption",
 CellID->17059],

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Mu]", "=", 
   FractionBox["1", "82.45"]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   SuperscriptBox["\[Mu]", "*"], "=", 
   RowBox[{"1", "-", "\[Mu]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   SubscriptBox["r", "1"], "=", 
   SqrtBox[
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "1"], "[", "t", "]"}], "+", "\[Mu]"}], ")"}], "2"],
      "+", 
     SuperscriptBox[
      RowBox[{
       SubscriptBox["y", "2"], "[", "t", "]"}], "2"]}]]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   SubscriptBox["r", "2"], "=", 
   SqrtBox[
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "1"], "[", "t", "]"}], "-", 
        SuperscriptBox["\[Mu]", "*"]}], ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{
       SubscriptBox["y", "2"], "[", "t", "]"}], "2"]}]]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqns", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox[
          SubscriptBox["y", "1"], "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
        RowBox[{
         SubscriptBox["y", "3"], "[", "t", "]"}]}], ",", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "1"], "[", "0", "]"}], "\[Equal]", "1.2"}]}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox[
          SubscriptBox["y", "2"], "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
        RowBox[{
         SubscriptBox["y", "4"], "[", "t", "]"}]}], ",", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "2"], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
     ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox[
          SubscriptBox["y", "3"], "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
        RowBox[{
         RowBox[{"2", " ", 
          RowBox[{
           SubscriptBox["y", "4"], "[", "t", "]"}]}], "+", 
         RowBox[{
          SubscriptBox["y", "1"], "[", "t", "]"}], "-", 
         FractionBox[
          RowBox[{
           SuperscriptBox["\[Mu]", "*"], " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              SubscriptBox["y", "1"], "[", "t", "]"}], "+", "\[Mu]"}], 
            ")"}]}], 
          SubsuperscriptBox["r", "1", "3"]], "-", 
         FractionBox[
          RowBox[{"\[Mu]", " ", 
           RowBox[{"(", 
            RowBox[{
             RowBox[{
              SubscriptBox["y", "1"], "[", "t", "]"}], "-", 
             SuperscriptBox["\[Mu]", "*"]}], ")"}]}], 
          SubsuperscriptBox["r", "2", "3"]]}]}], ",", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "3"], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
     ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox[
          SubscriptBox["y", "4"], "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "\[Equal]", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "2"}], " ", 
          RowBox[{
           SubscriptBox["y", "3"], "[", "t", "]"}]}], "+", 
         RowBox[{
          SubscriptBox["y", "2"], "[", "t", "]"}], "-", 
         FractionBox[
          RowBox[{
           SuperscriptBox["\[Mu]", "*"], " ", 
           RowBox[{
            SubscriptBox["y", "2"], "[", "t", "]"}]}], 
          SubsuperscriptBox["r", "1", "3"]], "-", 
         FractionBox[
          RowBox[{"\[Mu]", " ", 
           RowBox[{
            SubscriptBox["y", "2"], "[", "t", "]"}]}], 
          SubsuperscriptBox["r", "2", "3"]]}]}], ",", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "4"], "[", "0", "]"}], "\[Equal]", 
        RowBox[{"-", "1.04935750983031990726`20.020923474937767"}]}]}], 
      "}"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[63]:=",
 CellID->6838],

Cell["\<\
The event function is the derivative of the distance from the initial \
conditions. A local maximum or minimum occurs when the value crosses zero.\
\>", "MathCaption",
 CellID->20556],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ddist", " ", "=", " ", 
    RowBox[{"2", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["y", "3"], "[", "t", "]"}], 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           SubscriptBox["y", "1"], "[", "t", "]"}], "-", "1.2"}], ")"}]}], 
       "+", 
       RowBox[{
        RowBox[{
         SubscriptBox["y", "4"], "[", "t", "]"}], 
        RowBox[{
         SubscriptBox["y", "2"], "[", "t", "]"}]}]}], ")"}]}]}], ";"}], 
  " "}]], "Input",
 CellLabel->"In[68]:=",
 CellID->15179],

Cell[TextData[{
 "There are two events, which for this example are the same. The first event \
(with ",
 Cell[BoxData[
  ButtonBox["Direction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Direction"]], "InlineFormula"],
 " 1) corresponds to the point where the distance from the initial point is a \
local",
 " ",
 "minimum, so that the spaceship returns to its original position. The event \
action is to store the time of the event in the variable ",
 Cell[BoxData["tfinal"], "InlineFormula"],
 " and to stop the integration. The second event corresponds to a local \
maximum. The event action is to store the time that the spaceship is farthest \
from the starting position in the variable ",
 Cell[BoxData["tfar"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->18296],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"eqns", ",", 
     RowBox[{"{", 
      RowBox[{
       SubscriptBox["y", "1"], ",", 
       SubscriptBox["y", "2"], ",", 
       SubscriptBox["y", "3"], ",", 
       SubscriptBox["y", "4"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "\[Infinity]"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<EventLocator\>\"", ",", "\[IndentingNewLine]", 
        RowBox[{"\"\<Event\>\"", "->", 
         RowBox[{"{", 
          RowBox[{"ddist", ",", " ", "ddist"}], "}"}]}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"\"\<Direction\>\"", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"1", ",", " ", 
           RowBox[{"-", "1"}]}], "}"}]}], ",", "\[IndentingNewLine]", 
        RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Throw", "[", 
            RowBox[{
             RowBox[{"tfinal", " ", "=", " ", "t"}], ",", 
             "\"\<StopIntegration\>\""}], "]"}], ",", 
           RowBox[{"tfar", " ", "=", " ", "t"}]}], "}"}]}], ",", 
        "\[IndentingNewLine]", " ", 
        RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], 
       "}"}]}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[69]:=",
 CellID->10782],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["y", "1"], "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.192169197330527`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], ",", 
   RowBox[{
    SubscriptBox["y", "2"], "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.192169197330527`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], ",", 
   RowBox[{
    SubscriptBox["y", "3"], "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.192169197330527`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], ",", 
   RowBox[{
    SubscriptBox["y", "4"], "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.192169197330527`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}]}], "}"}]], "Output",
 ImageSize->{335, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[69]=",
 CellID->1089967265]
}, Open  ]],

Cell["\<\
The first two solution components are coordinates of the body of \
infinitesimal mass, so plotting one against the other gives the orbit of the \
body.\
\>", "MathCaption",
 CellID->24729],

Cell["\<\
This displays one half-orbit when the spaceship is at the furthest point from \
the initial position.\
\>", "MathCaption",
 CellID->1556],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox["y", "1"], "[", "t", "]"}], ",", " ", 
      RowBox[{
       SubscriptBox["y", "2"], "[", "t", "]"}]}], "}"}], " ", "/.", " ", 
    "sol"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "tfar"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[70]:=",
 CellID->20042],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[70]=",
 CellID->252930653]
}, Open  ]],

Cell["\<\
This displays one complete orbit when the spaceship returns to the initial \
position.\
\>", "MathCaption",
 CellID->20840],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox["y", "1"], "[", "t", "]"}], ",", " ", 
      RowBox[{
       SubscriptBox["y", "2"], "[", "t", "]"}]}], "}"}], " ", "/.", " ", 
    "sol"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "tfinal"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[71]:=",
 CellID->25536],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 196},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[71]=",
 CellID->284311451]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Discontinuous Equations and Switching Functions", "Subsection",
 CellID->3279],

Cell["\<\
In many applications the function in a differential system may not be \
analytic or continuous everywhere.\
\>", "Text",
 CellID->6094],

Cell[TextData[{
 "A common discontinuous problem that arises in practice involves a ",
 StyleBox["switching function",
  FontSlant->"Italic"],
 " ",
 Cell[BoxData[
  FormBox["g", TraditionalForm]], "InlineMath"],
 ":"
}], "Text",
 CellID->12543],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["y", "'"], "=", 
  RowBox[{"{", GridBox[{
     {
      RowBox[{
       SubscriptBox["f", "I"], 
       RowBox[{"(", 
        RowBox[{"t", ",", "y"}], ")"}]}], "if", 
      RowBox[{
       RowBox[{"g", 
        RowBox[{"(", 
         RowBox[{"t", ",", "y"}], ")"}]}], " ", ">", " ", "0"}]},
     {
      RowBox[{
       SubscriptBox["f", "II"], 
       RowBox[{"(", 
        RowBox[{"t", ",", "y"}], ")"}]}], "if", 
      RowBox[{
       RowBox[{"g", 
        RowBox[{"(", 
         RowBox[{"t", ",", "y"}], ")"}]}], " ", "<", " ", "0"}]}
    }]}]}]], "DisplayMath",
 CellID->4306],

Cell[TextData[{
 "In order to illustrate the difficulty in crossing a discontinuity, consider \
the following example [",
 ButtonBox["G\[CapitalOSlash]84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#12707"],
 "] (see also [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "]):"
}], "Text",
 CellID->8532],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["y", "'"], "=", 
  RowBox[{"{", GridBox[{
     {
      RowBox[{
       SuperscriptBox["t", "2"], "+", " ", 
       RowBox[{"2", " ", 
        SuperscriptBox["y", "2"]}]}], "if", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"t", "+", 
           FractionBox["1", "20"]}], ")"}], "2"], " ", "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", "+", 
           FractionBox["3", "20"]}], ")"}], "2"]}], "\[LessEqual]", " ", 
       "1"}]},
     {
      RowBox[{
       RowBox[{"2", 
        RowBox[{"t", "^", "2"}]}], "+", 
       RowBox[{"3", 
        RowBox[{
         RowBox[{"y", "[", "t", "]"}], "^", "2"}]}], "-", "2", " "}], "if", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"t", "+", 
           FractionBox["1", "20"]}], ")"}], "2"], " ", "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", "+", 
           FractionBox["3", "20"]}], ")"}], "2"]}], " ", ">", " ", "1"}]}
    }]}]}]], "DisplayMath",
 CellID->2379],

Cell[TextData[{
 "Here is the input for the entire system. The switching function is assigned \
to the symbol ",
 Cell[BoxData["event"], "InlineFormula"],
 " and the function defining the system depends on the sign of the switching \
function."
}], "MathCaption",
 CellID->18590],

Cell[BoxData[{
 RowBox[{
  RowBox[{"t0", " ", "=", " ", "0"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ics0", " ", "=", " ", 
   FractionBox["3", "10"]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"event", " ", "=", " ", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"t", "+", 
       FractionBox["1", "20"]}], ")"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], "+", 
       FractionBox["3", "20"]}], ")"}], "2"], "-", "1"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"event", "<=", "0"}], ",", 
        RowBox[{
         SuperscriptBox["t", "2"], "+", 
         RowBox[{"2", 
          SuperscriptBox[
           RowBox[{"y", "[", "t", "]"}], "2"]}]}], ",", 
        RowBox[{
         RowBox[{"2", 
          SuperscriptBox["t", "2"]}], "+", 
         RowBox[{"3", 
          SuperscriptBox[
           RowBox[{"y", "[", "t", "]"}], "2"]}], "-", "2"}]}], "]"}]}], ",", 
     " ", 
     RowBox[{
      RowBox[{"y", "[", "t0", "]"}], "\[Equal]", "ics0"}]}], "}"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[72]:=",
 CellID->6635],

Cell[TextData[{
 "The symbol ",
 Cell[BoxData["odemethod"], "InlineFormula"],
 " is used to indicate the numerical method that should be used for the \
integration. For comparison, you might want to define a different method, \
such as ",
 Cell[BoxData["ExplicitRungeKutta"], "InlineFormula"],
 ", and rerun the computations in this section to see how other methods \
behave."
}], "MathCaption",
 CellID->22881],

Cell[BoxData[
 RowBox[{
  RowBox[{"odemethod", " ", "=", " ", "Automatic"}], ";"}]], "Input",
 CellLabel->"In[76]:=",
 CellID->7516],

Cell[TextData[{
 "This solves the system on the interval [0,1] and collects data for the mesh \
points of the integration using ",
 Cell[BoxData[
  ButtonBox["Reap",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reap"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Sow",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sow"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->13577],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", "\[IndentingNewLine]", 
     RowBox[{"sol", " ", "=", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], " ", "/.", " ", 
       RowBox[{"First", "[", 
        RowBox[{"NDSolve", "[", 
         RowBox[{"system", ",", "y", ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "t0", ",", "1"}], "}"}], ",", 
          RowBox[{"Method", "\[Rule]", "odemethod"}], ",", " ", 
          RowBox[{"MaxStepFraction", "\[Rule]", "1"}], ",", 
          RowBox[{"StepMonitor", "\[RuleDelayed]", 
           RowBox[{"Sow", "[", "t", "]"}]}]}], "]"}], "]"}]}]}], 
     "\[IndentingNewLine]", "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", "sol"}], "Input",
 CellLabel->"In[77]:=",
 CellID->17173],

Cell[BoxData[
 RowBox[{
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False], "[", "t", "]"}]], "Output",
 ImageSize->{279, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[78]=",
 CellID->237754635]
}, Open  ]],

Cell["Here is a plot of the solution.", "MathCaption",
 CellID->23017],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dirsol", " ", "=", " ", 
  RowBox[{"Plot", "[", 
   RowBox[{"sol", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "t0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[79]:=",
 CellID->20978],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 232},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[79]=",
 CellID->740762939]
}, Open  ]],

Cell["\<\
Despite the fact that a solution has been obtained, it is not clear whether \
it has been obtained efficiently.\
\>", "Text",
 CellID->30203],

Cell["\<\
The following example shows that the crossing of the discontinuity presents \
difficulties for the numerical solver.\
\>", "Text",
 CellID->24681],

Cell["\<\
This defines a function that displays the mesh points of the integration \
together with the number of integration steps that are taken.\
\>", "MathCaption",
 CellID->27823],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"StepPlot", "[", 
    RowBox[{"data_", ",", 
     RowBox[{"opts___", "?", "OptionQ"}]}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", "sdata", "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"sdata", " ", "=", " ", 
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"data", ",", 
          RowBox[{"Range", "[", 
           RowBox[{"Length", "[", "data", "]"}], "]"}]}], "}"}], "]"}]}], ";",
       "\[IndentingNewLine]", 
      RowBox[{"ListPlot", "[", 
       RowBox[{"sdata", ",", "opts", ",", 
        RowBox[{"Axes", "\[Rule]", "False"}], ",", 
        RowBox[{"Frame", "\[Rule]", "True"}], ",", 
        RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[80]:=",
 CellID->14232],

Cell[TextData[{
 "As the integration passes the discontinuity (near ",
 Cell[BoxData[
  FormBox["0.6", TraditionalForm]], "InlineMath"],
 " in value) the integration method runs into difficulty and a large number \
of small steps are taken\[LongDash]a number of rejected steps can also \
sometimes be observed."
}], "MathCaption",
 CellID->20575],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"StepPlot", "[", "data", "]"}]], "Input",
 CellLabel->"In[81]:=",
 CellID->32632],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW12IFVUcn/u5amaxSSxCZhCIYJ+kK6Kum5ahqwZSmw89LEuwC7UuJtFD
60pK9dA+Rh8bYZHpW4RQ4VPYS2Ki+WI+iKjgk089BD5Nc2bmzJn5nfOb/1zv
3b13817Y63z8Pv7//zlzzszc4+6RQ2NvvTNyaHx0ZNWOgyOTY+Oj7656+cDB
4FCl5HmlMc/zbvV5wbYfbIV/6jMe/EWbfOuR4LscECpqr6KZvqeO9TvOeWof
z9USXoxwMbXqBsFRHSo7zlVD7V71XTVn4uOZI+usIzXIYL2FqIPjCwV8bBUb
0w+6FQtRaSIvjNpGKJWqk9uJ+fQUa4XcliySl+oPui+2MxsdhzsLfZ21t00a
qaf7TDq2VvYxPXbV43/7YV+PNNgO61LHXTzcj0YeRS0yovbGMQ6r7hnVPOVd
hf0Hhf1lTZ7vNHyr9VQ9S2GlH4hbC3tct+JzVfElvvNazqn9FthfKeyj99Ow
P9AgvpWxo3ej56VY9E3V7ni7SMzRPdWJ7y8ObP5y5tPfE37RnCP+8Gs/BPyX
tn0NfHaFRaxPPv4tYE1OnAYWjoDRaB55RGGqw0MC/twfN4OAorDy8VWNf2P4
5MDmL2L9XRRfc+QsoC31nRl8xdN3ZlnVV+JztmqIrgeqByZO//Lz30kDqpM7
aCyLHR4Cmnhsp6ylDg8BTTwGKeshh4eAJh5bKKvX4SGhEw/ndbXBavM+8HCx
nqV9oM/pmY34eRrxygIRr6XsJx1eEtp4lR1eqyl7jcNLQufn9QRlP+PwktDG
a5HD61Gr1Tc6PJbTVt4IHj0Oj6U0wu0OLwmdn0+JsveCF0b6r8DEHI0SZ+4X
qnMnYapxWE1lqu+55qRR8DTMcKyNJ9fk237j4NTjtbyR6IcxZ2ZZl7YK4G2I
8ZqvZ/pFKUaR6CaEul1JomtU+X2I0SiFFa+n8+TZ2oq8kn9mWkq/KXK18VGI
zTCxXx0V6nOWMmfAQ0Jyj18p83PwqAPzR4FpPPGuI8tUxOPAqIHXCZ+NXKdA
+ySN6pQzKuPxDWX+BB4Sknt8RplnwENC8lrNUOZZ8JCQxqMKHh9ZLXgOtI/R
FguRtUh7eupMrHiMxnLeoZyDtJQ/pPgLIT5S3rZ1NsYfEfCR/p6hb+N4OP5i
uG+eQCSkyfFeNRUQtQ7Tlpj/+NqnCfOf7x67uVOx+dPWGtp5PNAaHJhNaem7
hMg+JZk8YYc9rfLqnu9Uzxyc9T7w9RyMvxmUrbjw6ZnnpGeyohquujaqUaC2
uW2zkPtgJ13Lul9+JSJ1L5SR3YovvJznR5P1NlNx08u6FVcf6SmDe9zrExCP
pTnFZmM086GJLTsfVrzm50F7tUBj83u6Z+tf8NhbgO64ej+Oq647wfwZV0Z2
K77wNO+3+7r/90yW//wjz13tm1fnuvXaf3fZRJ/P/JY67eP1ewlymfLZ9XsJ
cpmmvqgpIDMRHqbYv0BVQCrV/UpVvQqbotjLjvzzkCb/1mnyt5PvWe11FbTx
97oJ2n5XwQt//5ykUV4XohynzNvgISH575CjlHkHPCQk93iTMv8BDwnJa/W6
1aJ3QXufH79Ts7TvCtp7aVTqqwQ+RdD83TxfB6G+cOVEETTPa2su+2HwwkjN
OoJ4oWH80c80bD4w2lgF42W07VkzPTN5njxzqS9co4CVeK5JP/1o+Bjk8ZSv
Z/LW5PG40HvWtLhuqyGfudHn+Zg1D/HS2WRT6mG4goKvnjBo3j9W5LLXg1cR
NB8tl+eyN4FXETT3WubjqKlYL4IHrpPp8dlcaNi4ysaoLc6NWI1oXk/CDvaM
by2XuQt8iqCzPY+j99Gz87fasnOdO22l78Jdmayvwm6FW6tX9tMjlvlfDl7p
PzBad1s=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[81]=",
 CellID->42836257]
}, Open  ]],

Cell["\<\
One of the most efficient methods of crossing a discontinuity is to break the \
integration by restarting at the point of discontinuity.\
\>", "Text",
 CellID->5140],

Cell[TextData[{
 "The following example shows how to use the ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method to accomplish this."
}], "Text",
 CellID->1531],

Cell["\<\
This numerically integrates the first part of the system up to the point of \
discontinuity. The switching function is given as the event. The direction of \
the event is restricted to a change from negative to positive. When the event \
is found, the solution and the time of the event are stored by the event \
action.\
\>", "MathCaption",
 CellID->8386],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"system1", " ", "=", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        SuperscriptBox["t", "2"], "+", 
        RowBox[{"2", 
         SuperscriptBox[
          RowBox[{"y", "[", "t", "]"}], "2"]}]}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "t0", "]"}], "\[Equal]", "ics0"}]}], "}"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"data1", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"sol1", " ", "=", " ", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], " ", "/.", " ", 
       RowBox[{"First", "[", 
        RowBox[{"NDSolve", "[", 
         RowBox[{"system1", ",", "y", ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "t0", ",", "1"}], "}"}], ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<EventLocator\>\"", ",", 
             RowBox[{"\"\<Event\>\"", "->", "event"}], ",", 
             RowBox[{"Direction", "\[Rule]", "1"}], ",", 
             RowBox[{"EventAction", "\[RuleDelayed]", 
              RowBox[{"Throw", "[", 
               RowBox[{
                RowBox[{
                 RowBox[{"t1", " ", "=", " ", "t"}], ";", " ", 
                 RowBox[{"ics1", " ", "=", " ", 
                  RowBox[{"y", "[", "t", "]"}]}], ";"}], " ", ",", 
                "\"\<StopIntegration\>\""}], "]"}]}], ",", 
             RowBox[{"Method", "\[Rule]", "odemethod"}]}], "}"}]}], ",", 
          RowBox[{"MaxStepFraction", "\[Rule]", "1"}], ",", 
          RowBox[{"StepMonitor", "\[RuleDelayed]", 
           RowBox[{"Sow", "[", "t", "]"}]}]}], "]"}], "]"}]}]}], 
     "\[IndentingNewLine]", "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", "sol1"}], "Input",
 CellLabel->"In[82]:=",
 CellID->16361],

Cell[BoxData[
 RowBox[{
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "0.6234178950846722`"}], "}"}], "}"}], 
     ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False], "[", "t", "]"}]], "Output",
 ImageSize->{321, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[84]=",
 CellID->1920518206]
}, Open  ]],

Cell[TextData[{
 "Using the discontinuity found by the ",
 StyleBox["EventLocator", "MR"],
 " method as a new initial condition, the integration can now be continued."
}], "Text",
 CellID->4397],

Cell["\<\
This defines a system and initial condition, solves the system numerically \
and collects the data used for the mesh points.\
\>", "MathCaption",
 CellID->22953],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"system2", " ", "=", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"2", 
         SuperscriptBox["t", "2"]}], "+", 
        RowBox[{"3", 
         SuperscriptBox[
          RowBox[{"y", "[", "t", "]"}], "2"]}], "-", "2"}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "t1", "]"}], "\[Equal]", "ics1"}]}], "}"}]}], ";"}], 
  "\n"}], "\n", 
 RowBox[{
  RowBox[{"data2", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", "\[IndentingNewLine]", 
     RowBox[{"sol2", " ", "=", " ", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], " ", "/.", " ", 
       RowBox[{"First", "[", 
        RowBox[{"NDSolve", "[", 
         RowBox[{"system2", ",", "y", ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "t1", ",", "1"}], "}"}], ",", 
          RowBox[{"Method", "\[Rule]", "odemethod"}], ",", 
          RowBox[{"MaxStepFraction", "\[Rule]", "1"}], ",", 
          RowBox[{"StepMonitor", "\[RuleDelayed]", 
           RowBox[{"Sow", "[", "t", "]"}]}]}], "]"}], "]"}]}]}], 
     "\[IndentingNewLine]", "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", "sol2"}], "Input",
 CellLabel->"In[85]:=",
 CellID->12809],

Cell[BoxData[
 RowBox[{
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.6234178950846722`", ",", "1.`"}], "}"}], "}"}], 
     ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False], "[", "t", "]"}]], "Output",
 ImageSize->{321, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[87]=",
 CellID->1908896991]
}, Open  ]],

Cell["\<\
A plot of the two solutions is very similar to that obtained by solving the \
entire system in one go.\
\>", "MathCaption",
 CellID->8035],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"evsol", " ", "=", " ", 
  RowBox[{"Plot", "[", 
   RowBox[{
    RowBox[{"If", "[", 
     RowBox[{
      RowBox[{"t", " ", "\[LessEqual]", " ", "t1"}], ",", "sol1", ",", 
      "sol2"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[88]:=",
 CellID->4121],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 232},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[88]=",
 CellID->620296164]
}, Open  ]],

Cell["\<\
Examining the mesh points it is clear that far fewer steps were taken by the \
method and that the problematic behavior encountered near the discontinuity \
has been eliminated.\
\>", "MathCaption",
 CellID->32537],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"StepPlot", "[", 
  RowBox[{"Join", "[", 
   RowBox[{"data1", ",", "data2"}], "]"}], "]"}]], "Input",
 CellLabel->"In[89]:=",
 CellID->27743],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 233},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[89]=",
 CellID->216574532]
}, Open  ]],

Cell[TextData[{
 "The value of the discontinuity is given as 0.6234 in [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "], which coincides with the value found by the ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method."
}], "Text",
 CellID->15533],

Cell["\<\
In this example it is possible to analytically solve the system and use a \
numerical method to check the value.\
\>", "Text",
 CellID->28306],

Cell["\<\
The solution of the system up to the discontinuity can be represented in \
terms of Bessel and Gamma functions.\
\>", "MathCaption",
 CellID->28396],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dsol", " ", "=", " ", 
  RowBox[{"FullSimplify", "[", 
   RowBox[{"First", "[", 
    RowBox[{"DSolve", "[", 
     RowBox[{"system1", ",", 
      RowBox[{"y", "[", "t", "]"}], ",", "t"}], "]"}], "]"}], 
   "]"}]}]], "Input",
 CellLabel->"In[90]:=",
 CellID->28786],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
   FractionBox[
    RowBox[{
     SqrtBox["2"], " ", "t", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"3", " ", 
        RowBox[{"BesselJ", "[", 
         RowBox[{
          RowBox[{"-", 
           FractionBox["3", "4"]}], ",", 
          FractionBox[
           SuperscriptBox["t", "2"], 
           SqrtBox["2"]]}], "]"}], " ", 
        RowBox[{"Gamma", "[", 
         FractionBox["1", "4"], "]"}]}], "+", 
       RowBox[{"10", " ", 
        SuperscriptBox["2", 
         RowBox[{"1", "/", "4"}]], " ", 
        RowBox[{"BesselJ", "[", 
         RowBox[{
          FractionBox["3", "4"], ",", 
          FractionBox[
           SuperscriptBox["t", "2"], 
           SqrtBox["2"]]}], "]"}], " ", 
        RowBox[{"Gamma", "[", 
         FractionBox["3", "4"], "]"}]}]}], ")"}]}], 
    RowBox[{
     RowBox[{
      RowBox[{"-", "6"}], " ", 
      RowBox[{"BesselJ", "[", 
       RowBox[{
        FractionBox["1", "4"], ",", 
        FractionBox[
         SuperscriptBox["t", "2"], 
         SqrtBox["2"]]}], "]"}], " ", 
      RowBox[{"Gamma", "[", 
       FractionBox["1", "4"], "]"}]}], "+", 
     RowBox[{"20", " ", 
      SuperscriptBox["2", 
       RowBox[{"1", "/", "4"}]], " ", 
      RowBox[{"BesselJ", "[", 
       RowBox[{
        RowBox[{"-", 
         FractionBox["1", "4"]}], ",", 
        FractionBox[
         SuperscriptBox["t", "2"], 
         SqrtBox["2"]]}], "]"}], " ", 
      RowBox[{"Gamma", "[", 
       FractionBox["3", "4"], "]"}]}]}]]}], "}"}]], "Output",
 ImageSize->{532, 74},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[90]=",
 CellID->1795803233]
}, Open  ]],

Cell["\<\
Substituting in the solution into the switching function, a local \
minimization confirms the value of the discontinuity.\
\>", "MathCaption",
 CellID->22035],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{"event", " ", "/.", " ", "dsol"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", 
     RowBox[{"3", "/", "5"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[91]:=",
 CellID->20673],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"t", "\[Rule]", "0.6234179814117705`"}], "}"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[91]=",
 CellID->1814744480]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Avoiding Wraparound in PDEs", "Subsection",
 CellID->24434],

Cell["\<\
Many evolution equations model behavior on a spatial domain which is infinite \
or sufficiently large to make it impractical to discretize the entire domain \
without using specialized discretization methods. In practice, it is often \
the case that it is possible to use a smaller computational domain for as \
long as the solution of interest remains localized. \
\>", "Text",
 CellID->29407],

Cell[TextData[{
 "In situations where the boundaries of the computational domain are imposed \
by practical considerations rather than the actual model being studied, it is \
possible to pick boundary conditions appropriately. Using a ",
 ButtonBox["pseudospectral",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolvePDE#1761574857"],
 " method with periodic boundary conditions can make it possible to increase \
the extent of the computational domain because of the superb resolution of \
the periodic pseudospectral approximation. The drawback of periodic boundary \
conditions is that signals which propagate past the boundary persist on the \
other side of the domain, affecting the solution through wraparound.",
 " ",
 "It is possible to use an absorbing layer near the boundary to minimize \
these effects, but it is not always possible to completely eliminate them."
}], "Text",
 CellID->1804],

Cell[TextData[{
 "The sine-Gordon equation turns up in differential geometry and relativistic \
field theory. This example integrates the equation starting with a localized \
initial condition which spreads out. The periodic pseudospectral method is \
used for the integration. Since no absorbing layer has been instituted near \
the boundaries, it is most appropriate to stop the integration once \
wraparound becomes significant.",
 " ",
 "This condition is easily detected with event location using the ",
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method."
}], "Text",
 CellID->30807],

Cell["\<\
The integration is stopped when the size of the solution at the periodic \
wraparound point crosses a threshold of 0.01, beyond which, the form of the \
wave would be affected by periodicity.\
\>", "MathCaption",
 CellID->3952],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"sgsol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          SubscriptBox["\[PartialD]", 
           RowBox[{"t", ",", "t"}]], 
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}]}], " ", "\[Equal]", 
         RowBox[{
          RowBox[{
           SubscriptBox["\[PartialD]", 
            RowBox[{"x", ",", "x"}]], 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}]}], " ", "-", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], "]"}]}]}], ",", 
        "\[IndentingNewLine]", " ", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{"-", 
            SuperscriptBox[
             RowBox[{"(", 
              RowBox[{"x", "-", "5"}], ")"}], "2"]}]], "+", 
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{
            RowBox[{"-", 
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"x", "+", "5"}], ")"}], "2"]}], "/", "2"}]]}]}], ",", 
        " ", 
        RowBox[{
         RowBox[{
          SuperscriptBox["u", 
           TagBox[
            RowBox[{"(", 
             RowBox[{"1", ",", "0"}], ")"}],
            Derivative],
           MultilineFunction->None], "[", 
          RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", 
        " ", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", 
           RowBox[{"-", "50"}]}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "50"}], "]"}]}]}], "}"}], ",", " ", "u", ",", " ",
       
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1000"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<MethodOfLines\>\"", ",", "\[IndentingNewLine]", 
         RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
            RowBox[{
            "\"\<DifferenceOrder\>\"", "->", "\"\<Pseudospectral\>\""}]}], 
           "}"}]}], ",", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<EventLocator\>\"", ",", " ", 
            RowBox[{"\"\<Event\>\"", "\[RuleDelayed]", " ", 
             RowBox[{
              RowBox[{"Abs", "[", 
               RowBox[{"u", "[", 
                RowBox[{"t", ",", 
                 RowBox[{"-", "50"}]}], "]"}], "]"}], " ", "-", " ", 
              "0.01"}]}], ",", " ", 
            RowBox[{
            "\"\<EventLocationMethod\>\"", "->", "\"\<StepBegin\>\""}]}], 
           "}"}]}]}], "}"}]}]}], "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[92]:=",
 CellID->27626],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.9510000000000001`", " ", "Second"}], ",", 
   RowBox[{"{", 
    RowBox[{"u", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0.`", ",", "45.81888712328008`"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"\<\"...\"\>", ",", 
            RowBox[{"-", "50.`"}], ",", "50.`", ",", "\<\"...\"\>"}], "}"}]}],
          "}"}], ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}], "}"}]}], "}"}]], "Output",
 ImageSize->{494, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[92]=",
 CellID->1190069951]
}, Open  ]],

Cell[TextData[{
 "This extracts the ending time from the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object and makes a plot of the computed solution.",
 " ",
 "You can see that the integration has been stopped just as the first waves \
begin to reach the boundary."
}], "MathCaption",
 CellID->2331],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"end", " ", "=", " ", 
   RowBox[{
    RowBox[{"InterpolatingFunctionDomain", "[", 
     RowBox[{"u", " ", "/.", " ", "sgsol"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"1", ",", 
      RowBox[{"-", "1"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"DensityPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"u", "[", 
     RowBox[{"t", ",", "x"}], "]"}], " ", "/.", " ", "sgsol"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", " ", "end"}], "}"}], ",", " ", 
   RowBox[{"Mesh", "\[Rule]", "False"}], ",", " ", 
   RowBox[{"PlotPoints", "\[Rule]", "100"}]}], "]"}]}], "Input",
 CellLabel->"In[93]:=",
 CellID->4569],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 362},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[94]=",
 CellID->17599912]
}, Open  ]],

Cell[TextData[{
 "The",
 StyleBox[" ", "MR"],
 StyleBox[ButtonBox["\"DiscretizedMonitorVariables\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolvePDE#1602750198"], "MR"],
 StyleBox[" ", "MR"],
 "option affects the way the event is interpreted for PDEs; with the setting \
",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  RowBox[{
   StyleBox["u", "TI"], "[", 
   RowBox[{
    StyleBox["t", "TI"], ",", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 " is replaced by a vector of discretized values. This is much more efficient \
because it avoids explicitly constructing the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " to evaluate the event."
}], "MathCaption",
 CellID->11808],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"sgsol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          SubscriptBox["\[PartialD]", 
           RowBox[{"t", ",", "t"}]], 
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}]}], " ", "\[Equal]", " ", 
         RowBox[{
          RowBox[{
           SubscriptBox["\[PartialD]", 
            RowBox[{"x", ",", "x"}]], 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}]}], " ", "-", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"u", "[", 
            RowBox[{"t", ",", "x"}], "]"}], "]"}]}]}], ",", 
        "\[IndentingNewLine]", " ", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{"-", 
            SuperscriptBox[
             RowBox[{"(", 
              RowBox[{"x", "-", "5"}], ")"}], "2"]}]], "+", 
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{
            RowBox[{"-", 
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"x", "+", "5"}], ")"}], "2"]}], "/", "2"}]]}]}], ",", 
        " ", 
        RowBox[{
         RowBox[{
          SuperscriptBox["u", 
           TagBox[
            RowBox[{"(", 
             RowBox[{"1", ",", "0"}], ")"}],
            Derivative],
           MultilineFunction->None], "[", 
          RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", 
        " ", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", 
           RowBox[{"-", "50"}]}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "50"}], "]"}]}]}], "}"}], ",", " ", "u", ",", " ",
       
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1000"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "50"}], ",", "50"}], "}"}], ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<MethodOfLines\>\"", ",", "\[IndentingNewLine]", 
         RowBox[{"\"\<DiscretizedMonitorVariables\>\"", "\[Rule]", "True"}], 
         ",", " ", "\[IndentingNewLine]", 
         RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
            RowBox[{
            "\"\<DifferenceOrder\>\"", "->", "\"\<Pseudospectral\>\""}]}], 
           "}"}]}], ",", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<EventLocator\>\"", ",", " ", 
            RowBox[{"\"\<Event\>\"", "\[RuleDelayed]", " ", 
             RowBox[{
              RowBox[{"Abs", "[", 
               RowBox[{"First", "[", 
                RowBox[{"u", "[", 
                 RowBox[{"t", ",", "x"}], "]"}], "]"}], "]"}], " ", "-", " ", 
              "0.01"}]}], ",", " ", 
            RowBox[{
            "\"\<EventLocationMethod\>\"", "->", "\"\<StepBegin\>\""}]}], 
           "}"}]}]}], "}"}]}]}], "]"}], "]"}]}], "]"}]], "Input",
 CellTags->"ELPDEExample",
 CellLabel->"In[95]:=",
 CellID->28531],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.29000000000000004`", " ", "Second"}], ",", 
   RowBox[{"{", 
    RowBox[{"u", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0.`", ",", "45.81888712328008`"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"\<\"...\"\>", ",", 
            RowBox[{"-", "50.`"}], ",", "50.`", ",", "\<\"...\"\>"}], "}"}]}],
          "}"}], ",", "\<\"<>\"\>"}], "]"}],
      False,
      Editable->False]}], "}"}]}], "}"}]], "Output",
 ImageSize->{578, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"ELPDEExample",
 CellLabel->"Out[95]=",
 CellID->190992315]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Performance Comparison", "Subsection",
 CellID->31731],

Cell["\<\
The following example constructs a table making a comparison for two \
different integration methods.\
\>", "Text",
 CellID->27707],

Cell["\<\
This defines a function which returns the time it takes to compute a solution \
of a mildly damped pendulum equation up to the point at which the bob has \
momentarily been at rest 1000 times.\
\>", "MathCaption",
 CellID->1146],

Cell[BoxData[
 RowBox[{
  RowBox[{"EventLocatorTiming", "[", 
   RowBox[{"locmethod_", ",", "odemethod_"}], "]"}], " ", ":=", " ", 
  RowBox[{
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Second", " ", "=", " ", "1"}], ",", " ", "y", ",", " ", "t", 
       ",", " ", 
       RowBox[{"p", " ", "=", " ", "0"}]}], "}"}], ",", "\[IndentingNewLine]",
      
     RowBox[{"First", "[", 
      RowBox[{"Timing", "[", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
             RowBox[{
              FractionBox["1", "1000"], " ", 
              RowBox[{
               RowBox[{"y", "'"}], "[", "t", "]"}]}], " ", "+", " ", 
             RowBox[{"Sin", "[", 
              RowBox[{"y", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", 
            "0"}], ",", " ", 
           RowBox[{
            RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "3"}], ",", 
           " ", 
           RowBox[{
            RowBox[{
             RowBox[{"y", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", 
            "0"}]}], "}"}], ",", " ", "y", ",", " ", 
         RowBox[{"{", 
          RowBox[{"t", ",", "\[Infinity]"}], "}"}], ",", " ", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<EventLocator\>\"", ",", " ", 
            RowBox[{"\"\<Event\>\"", "\[Rule]", 
             RowBox[{
              RowBox[{"y", "'"}], "[", "t", "]"}]}], ",", " ", 
            RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", " ", 
             RowBox[{"If", "[", 
              RowBox[{
               RowBox[{
                RowBox[{"p", "++"}], " ", "\[GreaterEqual]", " ", "1000"}], 
               ",", " ", 
               RowBox[{"Throw", "[", 
                RowBox[{
                 RowBox[{"end", " ", "=", " ", "t"}], ",", " ", 
                 "\"\<StopIntegration\>\""}], "]"}]}], "]"}]}], ",", 
            RowBox[{"\"\<EventLocationMethod\>\"", "\[Rule]", "locmethod"}], 
            ",", " ", 
            RowBox[{"\"\<Method\>\"", "\[Rule]", "odemethod"}]}], "}"}]}], 
         ",", "\[IndentingNewLine]", 
         RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}], "]"}], 
      "]"}]}], "\[IndentingNewLine]", "]"}], ";;"}]}]], "Input",
 CellLabel->"In[96]:=",
 CellID->30942],

Cell["\<\
This uses the function to make a table comparing the different location \
methods for two different ODE integration methods.\
\>", "MathCaption",
 CellID->20145],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"elmethods", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
    "\"\<StepBegin\>\"", ",", "\"\<StepEnd\>\"", ",", " ", 
     "\"\<LinearInterpolation\>\"", ",", " ", 
     RowBox[{"{", 
      RowBox[{"\"\<Brent\>\"", ",", 
       RowBox[{
       "\"\<SolutionApproximation\>\"", "->", 
        "\"\<CubicHermiteInterpolation\>\""}]}], "}"}], ",", " ", 
     "Automatic"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"odemethods", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"Automatic", ",", " ", "\"\<ExplicitRungeKutta\>\""}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{"Outer", "[", 
    RowBox[{
    "EventLocatorTiming", ",", " ", "elmethods", ",", " ", "odemethods", ",", 
     "1"}], "]"}], ",", " ", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"elmethods", ",", " ", "odemethods"}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[97]:=",
 CellID->17330],

Cell[BoxData[
 TagBox[
  TagBox[GridBox[{
     {
      StyleBox["\[Null]",
       ShowStringCharacters->False], 
      TagBox["Automatic",
       HoldForm], 
      TagBox["\<\"ExplicitRungeKutta\"\>",
       HoldForm]},
     {
      TagBox["\<\"StepBegin\"\>",
       HoldForm], "0.33000000000000085`", "0.30000000000000077`"},
     {
      TagBox["\<\"StepEnd\"\>",
       HoldForm], "0.33000000000000007`", "0.33`"},
     {
      TagBox["\<\"LinearInterpolation\"\>",
       HoldForm], "0.36999999999999994`", "0.32000000000000073`"},
     {
      TagBox[
       RowBox[{"{", 
        RowBox[{"\<\"Brent\"\>", ",", 
         RowBox[{"\<\"SolutionApproximation\"\>", 
          "\[Rule]", "\<\"CubicHermiteInterpolation\"\>"}]}], "}"}],
       HoldForm], "0.5200000000000005`", "0.4900000000000009`"},
     {
      TagBox["Automatic",
       HoldForm], "0.5799999999999988`", "0.6000000000000005`"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.28], {
         Offset[0.7]}, 
        Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}],
   {OutputFormsDump`HeadedRows, OutputFormsDump`HeadedColumns}],
  Function[BoxForm`e$, 
   MatrixForm[
   BoxForm`e$, 
    TableHeadings -> {{
      "StepBegin", "StepEnd", "LinearInterpolation", {
       "Brent", "SolutionApproximation" -> "CubicHermiteInterpolation"}, 
       Automatic}, {Automatic, "ExplicitRungeKutta"}}]]]], "Output",
 ImageSize->{609, 89},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[99]//TableForm=",
 CellID->1046106954]
}, Open  ]],

Cell["\<\
While simple step begin/end and linear interpolation location are essentially \
the same low cost, the better location methods are more expensive. The \
default location method is particularly expensive for the explicit \
Runge-Kutta method because that does not yet support a continuous output \
formula - it therefore needs to repeatedly invoke the method with different \
step sizes during the local minimization.\
\>", "Text",
 CellID->30961],

Cell["\<\
It is worth noting that, often, a significant part of the extra time for \
computing events arises from the need to evaluate the event functions at each \
time step to check for the possibility of a sign change.\
\>", "Text",
 CellID->30047],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"Map", "[", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"Block", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"Second", "=", "1"}], ",", "y", ",", "t", ",", 
           RowBox[{"p", "=", "0"}]}], "}"}], ",", "\[IndentingNewLine]", 
         RowBox[{"First", "[", 
          RowBox[{"Timing", "[", 
           RowBox[{"NDSolve", "[", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{
               RowBox[{
                RowBox[{
                 RowBox[{
                  RowBox[{"y", "''"}], "[", "t", "]"}], "+", 
                 RowBox[{
                  FractionBox["1", "1000"], " ", 
                  RowBox[{
                   RowBox[{"y", "'"}], "[", "t", "]"}]}], "+", 
                 RowBox[{"Sin", "[", 
                  RowBox[{"y", "[", "t", "]"}], "]"}]}], "\[Equal]", "0"}], 
               ",", 
               RowBox[{
                RowBox[{"y", "[", "0", "]"}], "\[Equal]", "3"}], ",", 
               RowBox[{
                RowBox[{
                 RowBox[{"y", "'"}], "[", "0", "]"}], "\[Equal]", "0"}]}], 
              "}"}], ",", "y", ",", 
             RowBox[{"{", 
              RowBox[{"t", ",", "end"}], "}"}], ",", 
             RowBox[{"Method", "\[Rule]", "#"}], ",", 
             RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}], "]"}], 
          "]"}]}], "]"}], "&"}], ",", "\[IndentingNewLine]", "odemethods"}], 
     "\[IndentingNewLine]", "]"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"None", ",", "odemethods"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[100]:=",
 CellID->24770],

Cell[BoxData[GridBox[{
   {
    TagBox["Automatic",
     HoldForm], 
    TagBox["\<\"ExplicitRungeKutta\"\>",
     HoldForm]},
   {"0.21999999999999914`", "0.2300000000000003`"}
  },
  GridBoxAlignment->{
   "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}},
  GridBoxDividers->{
   "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
    "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
  GridBoxSpacings->{"Columns" -> {
      Offset[0.28], {
       Offset[0.7]}, 
      Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
      Offset[0.2], {
       Offset[0.4]}, 
      Offset[0.2]}, "RowsIndexed" -> {}}]], "Output",
 ImageSize->{201, 29},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[100]//TableForm=",
 CellID->625624277]
}, Open  ]],

Cell["\<\
An optimization is performed for event functions involving only the \
independent variable. Such events are detected automatically at \
initialization time. For example, this has the advantage that interpolation \
of the solution of the dependent variables is not carried out during at each \
step if the local optimization search - it is deferred until the value of the \
independent variable has been found.\
\>", "Text",
 CellID->7426]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Limitations", "Section",
 CellTags->"s:4",
 CellID->19387],

Cell[TextData[{
 "One limitation of the event locator method is that since the event function \
is only checked for sign changes over a step interval, if the event function \
has multiple roots in a step interval, all or some of the events may be \
missed. This typically only happens when the solution to the ODE varies much \
more slowly than the event function. When you suspect that this may have \
occurred, the simplest solution is to decrease the maximum step size the \
method can take by using the ",
 Cell[BoxData[
  ButtonBox["MaxStepSize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxStepSize"]], "InlineFormula"],
 " option to ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". More sophisticated approaches can be taken, but the best approach depends \
on what is being computed.",
 " ",
 "An example is shown below which demonstrates the problem and shows two \
approaches for fixing it."
}], "Text",
 CellID->14614],

Cell[TextData[{
 "This should compute the number of positive integers less than ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[ExponentialE]", "5"], TraditionalForm]], "InlineMath"],
 " (there are 148).",
 " ",
 "However, most are missed because the method is taking large time steps \
because the solution ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "[", 
   StyleBox["t", "TI"], "]"}]], "InlineFormula"],
 " is so simple."
}], "MathCaption",
 CellID->22686],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"y", "[", "t", "]"}]}], ",", " ", 
        RowBox[{
         RowBox[{"y", "[", 
          RowBox[{"-", "1"}], "]"}], " ", "\[Equal]", " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "1"}]]}]}], "}"}], ",", " ", "y", ",", " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "5"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", " ", 
         RowBox[{"\"\<Event\>\"", "\[Rule]", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"\[Pi]", " ", 
            RowBox[{"y", "[", "t", "]"}]}], "]"}]}], ",", " ", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
          RowBox[{"n", "++"}]}]}], "}"}]}]}], "]"}], ";", "n"}]}], 
  "]"}]], "Input",
 CellLabel->"In[101]:=",
 CellID->1705],

Cell[BoxData["18"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[101]=",
 CellID->220889922]
}, Open  ]],

Cell["\<\
This restricts the maximum step size so that all the events are found.\
\>", "MathCaption",
 CellID->19539],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"y", "[", "t", "]"}]}], ",", " ", 
        RowBox[{
         RowBox[{"y", "[", 
          RowBox[{"-", "1"}], "]"}], " ", "\[Equal]", " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "1"}]]}]}], "}"}], ",", " ", "y", ",", " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "5"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", " ", 
         RowBox[{"\"\<Event\>\"", "\[Rule]", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"\[Pi]", " ", 
            RowBox[{"y", "[", "t", "]"}]}], "]"}]}], ",", " ", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
          RowBox[{"n", "++"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"MaxStepSize", "\[Rule]", "0.001"}]}], "]"}], ";", "n"}]}], 
  "]"}]], "Input",
 CellLabel->"In[102]:=",
 CellID->27365],

Cell[BoxData["148"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[102]=",
 CellID->558720145]
}, Open  ]],

Cell["\<\
It is quite apparent from the nature of the example problem that if the \
endpoint is increased, it is likely that a smaller maximum step size may be \
required. Taking very small steps everywhere is quite inefficient. It is \
possible to introduce an adaptive time step restriction be setting up a \
variable which varies on the same time scale as the event function.\
\>", "Text",
 CellID->25521],

Cell[TextData[{
 "This introduces an additional function to integrate which is the event \
function.",
 " ",
 "With this modification and allowing the method to take as many steps is \
needed, it is possible to find the correct value up to ",
 Cell[BoxData[
  RowBox[{
   StyleBox["t", "TI"], "=", "10"}]], "InlineFormula"],
 " in a reasonable amount of time."
}], "MathCaption",
 CellID->21044],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"n", " ", "=", " ", "0"}], "}"}], ",", 
   RowBox[{
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"y", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"y", "[", "t", "]"}]}], ",", " ", 
        RowBox[{
         RowBox[{"y", "[", 
          RowBox[{"-", "1"}], "]"}], " ", "\[Equal]", " ", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "1"}]]}], ",", " ", 
        RowBox[{
         RowBox[{
          RowBox[{"z", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"Sin", "[", 
            RowBox[{"\[Pi]", " ", 
             RowBox[{"y", "[", "t", "]"}]}], "]"}], ",", "t"}], "]"}]}], ",", 
        " ", 
        RowBox[{
         RowBox[{"z", "[", 
          RowBox[{"-", "1"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"Sin", "[", 
          RowBox[{"\[Pi]", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{"-", "1"}]]}], "]"}]}]}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"y", ",", "z"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "10"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<EventLocator\>\"", ",", " ", 
         RowBox[{"\"\<Event\>\"", "\[Rule]", " ", 
          RowBox[{"z", "[", "t", "]"}]}], ",", " ", 
         RowBox[{"\"\<EventAction\>\"", "\[RuleDelayed]", 
          RowBox[{"n", "++"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"MaxSteps", "\[Rule]", "\[Infinity]"}]}], "]"}], ";", "n"}]}], 
  "]"}]], "Input",
 CellLabel->"In[103]:=",
 CellID->14502],

Cell[BoxData["22026"], "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[103]=",
 CellID->944361772]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:5",
 CellID->31057],

Cell[CellGroupData[{

Cell["EventLocator options", "Subsection",
 CellTags->"EventLocatorOptionSummary",
 CellID->32120],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Direction\>\"", 
    ButtonBox["All",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/All"], Cell["\<\
The direction of zero crossing to allow for the event. 1 means from negative \
to positive, -1 means from positive to negative, and All includes both \
directions.\
\>", "TableText"]},
   {"\"\<Event\>\"", 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell["\<\
An expression that defines the event. An event occurs at points where \
substituting the numerical values of the problem variables makes the \
expression equal to zero.\
\>", "TableText"]},
   {"\"\<EventAction\>\"", Cell["Throw[Null, \"StopIntegration\"]"], Cell[
    TextData[{
     "What to do when an event occurs. Problem variables are substituted with \
their numerical values at the event.",
     " ",
     "In general, you need to use ",
     Cell[BoxData[
      ButtonBox["RuleDelayed",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/RuleDelayed"]], "InlineFormula"],
     " (",
     Cell[BoxData["\[RuleDelayed]"], "InlineFormula"],
     ") to prevent it from being evaluated except with numerical values."
    }], "TableText"]},
   {"\"\<EventLocationMethod\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
The method to use for refining the location of a given event. \
\>", "TableText"]},
   {"\"\<Method\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "The method to use for integrating the system of ODEs. ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->30716],

Cell[TextData[{
 Cell[BoxData["EventLocator"], "InlineFormula"],
 " method options"
}], "Caption",
 CellID->10038]
}, Open  ]],

Cell[CellGroupData[{

Cell["EventLocationMethod options", "Subsection",
 CellTags->"EventLocationMethodOptionSummary",
 CellID->25131],

Cell[BoxData[GridBox[{
   {"\"\<Brent\>\"", Cell[TextData[{
     "Use ",
     Cell[BoxData[
      ButtonBox["FindRoot",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
     " with ",
     Cell[BoxData[
      RowBox[{
       ButtonBox["Method",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Method"], "->", "\"\<Brent\>\""}]], 
      "InlineFormula"],
     " to locate the event. This is the default with the setting ",
     Cell[BoxData[
      ButtonBox["Automatic",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
     "."
    }], "TableText"]},
   {"\"\<LinearInterpolation\>\"", Cell["\<\
Locate the event time using linear interpolation. Cubic Hermite interpolation \
is then used to find the solution at the event time.\
\>", "TableText"]},
   {"\"\<StepBegin\>\"", Cell["\<\
The event is given by the solution at the beginning of the step.\
\>", "TableText"]},
   {"\"\<StepEnd\>\"", Cell[
    "The event is given by the solution at the end of the step.", 
     "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{GridBoxItemSize->{"Columns" -> {
     Scaled[0.4], {
      Scaled[0.6]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->10388],

Cell[TextData[{
 "Settings for the ",
 Cell[BoxData["EventLocationMethod"], "InlineFormula"],
 " option."
}], "Caption",
 CellID->15344]
}, Open  ]],

Cell[CellGroupData[{

Cell["Brent options", "Subsection",
 CellTags->"BrentOptionSummary",
 CellID->10439],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<MaxIterations\>\"", "100", Cell["\<\
The maximum number of iterations to use for locating an event within a step \
of the method. \
\>", "TableText"]},
   {"\"\<AccuracyGoal\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "Accuracy goal setting passed to ",
     Cell[BoxData[
      ButtonBox["FindRoot",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
     ". If ",
     Cell[BoxData[
      ButtonBox["Automatic",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
     ", the value passed to ",
     Cell[BoxData[
      ButtonBox["FindRoot",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
     " is based on the local error setting for ",
     Cell[BoxData[
      ButtonBox["NDSolve",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
     "."
    }], "TableText"]},
   {"\"\<PrecisionGoal\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "Precision goal setting passed to ",
     Cell[BoxData[
      ButtonBox["FindRoot",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
     ". If ",
     Cell[BoxData[
      ButtonBox["Automatic",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
     ", the value passed to ",
     Cell[BoxData[
      ButtonBox["FindRoot",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
     " is based on the local error setting for ",
     Cell[BoxData[
      ButtonBox["NDSolve",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
     "."
    }], "TableText"]},
   {"\"\<SolutionApproximation\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "How to approximate the solution for evaluating the event function \
during the refinement process. Can be ",
     Cell[BoxData[
      ButtonBox["Automatic",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
     " or ",
     Cell[BoxData["CubicHermiteInterpolation"], "InlineFormula"],
     "."
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->7535],

Cell[TextData[{
 "Options for event location method ",
 Cell[BoxData["\"\<Brent\>\""], "InlineFormula"],
 "."
}], "Caption",
 CellID->15901]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"EventLocator Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveDoubleStep"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveExtrapolation"]}], 
      "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 43.5711974}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "It is often useful to be able to detect and precisely locate a change in \
a differential system. For example, with the detection of a singularity or \
state change, the appropriate action can be taken, such as restarting the \
integration. An event for a differential system: is a point along the \
solution at which a real valued event function is zero:", "synonyms" -> {}, 
    "title" -> "EventLocator Method for NDSolve", "type" -> "Tutorial", "uri" -> 
    "tutorial/NDSolveEventLocator"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "s:1"->{
  Cell[1453, 48, 65, 2, 70, "Section",
   CellTags->"s:1",
   CellID->24251]},
 "s:2"->{
  Cell[31902, 994, 75, 2, 70, "Section",
   CellTags->"s:2",
   CellID->18569]},
 "s:3"->{
  Cell[57949, 1713, 60, 2, 70, "Section",
   CellTags->"s:3",
   CellID->4920]},
 "ELPDEExample"->{
  Cell[443845, 9509, 3234, 88, 70, "Input",
   CellTags->"ELPDEExample",
   CellID->28531],
  Cell[447082, 9599, 762, 24, 36, "Output",
   CellTags->"ELPDEExample",
   CellID->190992315]},
 "s:4"->{
  Cell[457752, 9915, 64, 2, 70, "Section",
   CellTags->"s:4",
   CellID->19387]},
 "s:5"->{
  Cell[464931, 10141, 67, 2, 70, "Section",
   CellTags->"s:5",
   CellID->31057]},
 "EventLocatorOptionSummary"->{
  Cell[465023, 10147, 98, 2, 70, "Subsection",
   CellTags->"EventLocatorOptionSummary",
   CellID->32120]},
 "EventLocationMethodOptionSummary"->{
  Cell[467198, 10211, 112, 2, 70, "Subsection",
   CellTags->"EventLocationMethodOptionSummary",
   CellID->25131]},
 "BrentOptionSummary"->{
  Cell[468765, 10263, 84, 2, 70, "Subsection",
   CellTags->"BrentOptionSummary",
   CellID->10439]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"s:1", 474137, 10412},
 {"s:2", 474226, 10416},
 {"s:3", 474317, 10420},
 {"ELPDEExample", 474417, 10424},
 {"s:4", 474616, 10431},
 {"s:5", 474709, 10435},
 {"EventLocatorOptionSummary", 474825, 10439},
 {"EventLocationMethodOptionSummary", 474973, 10443},
 {"BrentOptionSummary", 475115, 10447}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 739, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1364, 43, 64, 1, 70, "Title",
 CellID->12728],
Cell[CellGroupData[{
Cell[1453, 48, 65, 2, 70, "Section",
 CellTags->"s:1",
 CellID->24251],
Cell[1521, 52, 281, 6, 70, "Text",
 CellID->28061],
Cell[1805, 60, 124, 6, 70, "Text",
 CellID->4010],
Cell[1932, 68, 270, 10, 70, "DisplayMath",
 CellID->164338131],
Cell[2205, 80, 120, 3, 70, "Text",
 CellID->303513695],
Cell[2328, 85, 207, 7, 70, "DisplayMath",
 CellID->451433106],
Cell[2538, 94, 432, 14, 70, "Text",
 CellID->406756659],
Cell[2973, 110, 458, 13, 70, "Text",
 CellID->19480],
Cell[3434, 125, 357, 8, 70, "Text",
 CellID->17153],
Cell[3794, 135, 186, 4, 70, "MathCaption",
 CellID->24027],
Cell[3983, 141, 582, 19, 70, "Input",
 InitializationCell->True,
 CellID->16891],
Cell[4568, 162, 223, 5, 70, "Text",
 CellID->7491],
Cell[4794, 169, 227, 7, 70, "MathCaption",
 CellID->17252],
Cell[CellGroupData[{
Cell[5046, 180, 912, 28, 70, "Input",
 CellID->31809],
Cell[5961, 210, 485, 17, 36, "Output",
 CellID->1081533389]
}, Open  ]],
Cell[6461, 230, 657, 22, 70, "Text",
 CellID->5726],
Cell[7121, 254, 183, 4, 70, "MathCaption",
 CellID->26269],
Cell[CellGroupData[{
Cell[7329, 262, 905, 28, 65, "Input",
 CellID->5134],
Cell[8237, 292, 4304, 75, 242, "Output",
 Evaluatable->False,
 CellID->83107966]
}, Open  ]],
Cell[12556, 370, 264, 7, 70, "Text",
 CellID->863],
Cell[12823, 379, 282, 6, 70, "Text",
 CellID->6130],
Cell[13108, 387, 374, 14, 70, "MathCaption",
 CellID->5721],
Cell[CellGroupData[{
Cell[13507, 405, 1353, 40, 70, "Input",
 CellID->31667],
Cell[CellGroupData[{
Cell[14885, 449, 494, 12, 70, "Print",
 CellID->498535699],
Cell[15382, 463, 477, 11, 70, "Print",
 CellID->1132671112],
Cell[15862, 476, 497, 12, 70, "Print",
 CellID->1115229109]
}, Open  ]],
Cell[16374, 491, 468, 17, 36, "Output",
 CellID->869894129]
}, Open  ]],
Cell[16857, 511, 421, 12, 70, "Text",
 CellID->11049],
Cell[17281, 525, 420, 9, 70, "Text",
 CellID->15017],
Cell[17704, 536, 1424, 46, 70, "MathCaption",
 CellID->15665],
Cell[CellGroupData[{
Cell[19153, 586, 1472, 42, 70, "Input",
 CellID->17011],
Cell[20628, 630, 1968, 55, 115, "Output",
 CellID->493872563]
}, Open  ]],
Cell[22611, 688, 1146, 30, 70, "Text",
 CellID->27725],
Cell[23760, 720, 32, 1, 70, "Text",
 CellID->24908],
Cell[23795, 723, 1214, 44, 70, "Text",
 CellID->3162],
Cell[25012, 769, 426, 14, 70, "MathCaption",
 CellID->2050],
Cell[25441, 785, 775, 20, 70, "Input",
 CellID->3524],
Cell[26219, 807, 150, 4, 70, "MathCaption",
 CellID->1215],
Cell[CellGroupData[{
Cell[26394, 815, 930, 27, 70, "Input",
 CellID->28599],
Cell[27327, 844, 485, 17, 36, "Output",
 CellID->1694059150]
}, Open  ]],
Cell[27827, 864, 457, 14, 70, "Text",
 CellID->17670],
Cell[28287, 880, 365, 7, 70, "Text",
 CellID->12455],
Cell[28655, 889, 241, 5, 70, "MathCaption",
 CellID->24711],
Cell[CellGroupData[{
Cell[28921, 898, 1738, 48, 70, "Input",
 CellID->6221],
Cell[30662, 948, 557, 19, 36, "Output",
 CellID->1513391367]
}, Open  ]],
Cell[31234, 970, 314, 6, 70, "Text",
 CellID->31680],
Cell[31551, 978, 314, 11, 70, "Text",
 CellID->20037]
}, Open  ]],
Cell[CellGroupData[{
Cell[31902, 994, 75, 2, 70, "Section",
 CellTags->"s:2",
 CellID->18569],
Cell[31980, 998, 578, 12, 70, "Text",
 CellID->2807],
Cell[32561, 1012, 404, 7, 70, "Text",
 CellID->23535],
Cell[32968, 1021, 324, 8, 70, "Text",
 CellID->23666],
Cell[33295, 1031, 294, 6, 70, "Text",
 CellID->3092],
Cell[CellGroupData[{
Cell[33614, 1041, 66, 1, 70, "Subsection",
 CellID->1882],
Cell[33683, 1044, 917, 18, 70, "Text",
 CellID->24049]
}, Open  ]],
Cell[CellGroupData[{
Cell[34637, 1067, 63, 1, 70, "Subsection",
 CellID->3106],
Cell[34703, 1070, 322, 6, 70, "Text",
 CellID->2594],
Cell[35028, 1078, 134, 4, 70, "Text",
 CellID->662609674],
Cell[35165, 1084, 535, 19, 70, "DisplayMath",
 CellID->452965130],
Cell[35703, 1105, 63, 1, 70, "Text",
 CellID->340096408],
Cell[35769, 1108, 320, 11, 70, "DisplayMath",
 CellID->586491796],
Cell[36092, 1121, 85, 1, 70, "Text",
 CellID->847793339],
Cell[36180, 1124, 254, 9, 70, "DisplayMath",
 CellID->55447431],
Cell[36437, 1135, 891, 27, 70, "Text",
 CellID->4454476],
Cell[37331, 1164, 597, 21, 70, "DisplayMath",
 CellID->8389972],
Cell[37931, 1187, 79, 1, 70, "Text",
 CellID->154414597],
Cell[38013, 1190, 1194, 43, 70, "DisplayMath",
 CellID->526662252],
Cell[39210, 1235, 187, 6, 70, "Text",
 CellID->334110284],
Cell[39400, 1243, 160, 4, 70, "MathCaption",
 CellID->23324],
Cell[CellGroupData[{
Cell[39585, 1251, 2186, 61, 137, "Input",
 CellID->6444],
Cell[41774, 1314, 6541, 111, 242, "Output",
 Evaluatable->False,
 CellID->120101848]
}, Open  ]],
Cell[48330, 1428, 240, 5, 70, "Text",
 CellID->22070],
Cell[48573, 1435, 80, 1, 70, "MathCaption",
 CellID->23259],
Cell[CellGroupData[{
Cell[48678, 1440, 473, 16, 70, "Input",
 CellID->30033],
Cell[49154, 1458, 1803, 41, 250, "Output",
 CellID->1679090225]
}, Open  ]],
Cell[50972, 1502, 373, 7, 70, "Text",
 CellID->22956]
}, Open  ]],
Cell[CellGroupData[{
Cell[51382, 1514, 52, 1, 70, "Subsection",
 CellID->19093],
Cell[51437, 1517, 1212, 33, 70, "Text",
 CellID->12891],
Cell[52652, 1552, 1161, 27, 70, "Text",
 CellID->20022]
}, Open  ]],
Cell[CellGroupData[{
Cell[53850, 1584, 48, 1, 70, "Subsection",
 CellID->14605],
Cell[53901, 1587, 183, 4, 70, "Text",
 CellID->21887],
Cell[54087, 1593, 86, 1, 70, "MathCaption",
 CellID->24757],
Cell[54176, 1596, 275, 9, 70, "Input",
 CellID->30498],
Cell[54454, 1607, 184, 4, 70, "MathCaption",
 CellID->16028],
Cell[CellGroupData[{
Cell[54663, 1615, 1953, 50, 70, "Input",
 CellID->28446],
Cell[CellGroupData[{
Cell[56641, 1669, 304, 7, 70, "Print",
 CellID->75512204],
Cell[56948, 1678, 295, 7, 70, "Print",
 CellID->596940873],
Cell[57246, 1687, 328, 8, 70, "Print",
 CellID->1025005775],
Cell[57577, 1697, 299, 8, 70, "Print",
 CellID->1094793269]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[57949, 1713, 60, 2, 70, "Section",
 CellTags->"s:3",
 CellID->4920],
Cell[CellGroupData[{
Cell[58034, 1719, 50, 1, 70, "Subsection",
 CellID->26297],
Cell[58087, 1722, 249, 8, 70, "Text",
 CellID->19129],
Cell[58339, 1732, 146, 4, 70, "MathCaption",
 CellID->26974],
Cell[CellGroupData[{
Cell[58510, 1740, 1239, 36, 70, "Input",
 CellID->15089],
Cell[59752, 1778, 418, 15, 36, "Output",
 CellID->2021780425]
}, Open  ]],
Cell[60185, 1796, 152, 4, 70, "MathCaption",
 CellID->21859],
Cell[CellGroupData[{
Cell[60362, 1804, 1304, 40, 70, "Input",
 CellID->20864],
Cell[61669, 1846, 2678, 54, 251, "Output",
 CellID->1652505478]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[64396, 1906, 74, 1, 70, "Subsection",
 CellID->20479],
Cell[64473, 1909, 515, 13, 70, "Text",
 CellID->15472],
Cell[64991, 1924, 404, 12, 70, "MathCaption",
 CellID->15869],
Cell[CellGroupData[{
Cell[65420, 1940, 1714, 52, 70, "Input",
 CellID->10408],
Cell[67137, 1994, 855, 30, 54, "Output",
 CellID->683934652]
}, Open  ]],
Cell[68007, 2027, 102, 3, 70, "Text",
 CellID->3817],
Cell[68112, 2032, 359, 12, 70, "Text",
 CellID->19663],
Cell[68474, 2046, 187, 6, 70, "MathCaption",
 CellID->22222],
Cell[68664, 2054, 2735, 72, 70, "Input",
 CellID->27818],
Cell[71402, 2128, 197, 6, 70, "MathCaption",
 CellID->21183],
Cell[CellGroupData[{
Cell[71624, 2138, 99, 3, 70, "Input",
 CellID->8177],
Cell[71726, 2143, 176, 5, 36, "Output",
 CellID->459458936]
}, Open  ]],
Cell[71917, 2151, 127, 4, 70, "Text",
 CellID->5820]
}, Open  ]],
Cell[CellGroupData[{
Cell[72081, 2160, 63, 1, 70, "Subsection",
 CellID->16678],
Cell[72147, 2163, 166, 4, 70, "Text",
 CellID->198960080],
Cell[72316, 2169, 271, 9, 70, "Text",
 CellID->352748880],
Cell[CellGroupData[{
Cell[72612, 2182, 72, 1, 70, "Subsubsection",
 CellID->22454],
Cell[72687, 2185, 120, 4, 70, "Text",
 CellID->6847],
Cell[72810, 2191, 171, 5, 70, "MathCaption",
 CellID->28803],
Cell[CellGroupData[{
Cell[73006, 2200, 521, 15, 65, "Input",
 CellID->6824],
Cell[73530, 2217, 2168, 40, 92, "Output",
 Evaluatable->False,
 CellID->87330506]
}, Open  ]],
Cell[75713, 2260, 447, 17, 70, "Text",
 CellID->1626],
Cell[76163, 2279, 1237, 33, 70, "Text",
 CellID->32663],
Cell[77403, 2314, 435, 8, 70, "Text",
 CellID->11991],
Cell[77841, 2324, 89, 1, 70, "MathCaption",
 CellID->8292],
Cell[77933, 2327, 148, 5, 70, "Input",
 CellID->7492],
Cell[78084, 2334, 533, 19, 70, "MathCaption",
 CellID->954],
Cell[78620, 2355, 1677, 40, 70, "Input",
 CellID->1030],
Cell[80300, 2397, 333, 10, 70, "MathCaption",
 CellID->18249],
Cell[CellGroupData[{
Cell[80658, 2411, 426, 13, 47, "Input",
 CellID->21375],
Cell[81087, 2426, 13872, 231, 373, "Output",
 Evaluatable->False,
 CellID->80761620]
}, Open  ]],
Cell[94974, 2660, 176, 4, 70, "Text",
 CellID->1425],
Cell[95153, 2666, 562, 20, 70, "MathCaption",
 CellID->8493],
Cell[95718, 2688, 1854, 44, 70, "Input",
 CellID->22291],
Cell[97575, 2734, 331, 10, 70, "MathCaption",
 CellID->162],
Cell[CellGroupData[{
Cell[97931, 2748, 427, 13, 47, "Input",
 CellID->12771],
Cell[98361, 2763, 8960, 151, 373, "Output",
 Evaluatable->False,
 CellID->28389329]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[107370, 2920, 52, 1, 70, "Subsubsection",
 CellID->9537],
Cell[107425, 2923, 217, 5, 70, "Text",
 CellID->17140],
Cell[107645, 2930, 566, 18, 70, "Input",
 CellID->6888],
Cell[108214, 2950, 242, 7, 70, "MathCaption",
 CellID->30354],
Cell[CellGroupData[{
Cell[108481, 2961, 247, 9, 70, "Input",
 CellID->5363],
Cell[108731, 2972, 1201, 41, 51, "Output",
 CellID->1040350069]
}, Open  ]],
Cell[CellGroupData[{
Cell[109969, 3018, 304, 11, 70, "Input",
 CellID->20967],
Cell[110276, 3031, 884, 30, 36, "Output",
 CellID->1476672042]
}, Open  ]],
Cell[111175, 3064, 80, 1, 70, "MathCaption",
 CellID->3636],
Cell[111258, 3067, 706, 17, 70, "Input",
 CellID->3461],
Cell[111967, 3086, 141, 4, 70, "MathCaption",
 CellID->2519],
Cell[112111, 3092, 643, 17, 70, "Input",
 CellID->8513],
Cell[112757, 3111, 148, 4, 70, "MathCaption",
 CellID->1106],
Cell[112908, 3117, 2116, 52, 70, "Input",
 CellID->13285],
Cell[115027, 3171, 187, 4, 70, "MathCaption",
 CellID->23750],
Cell[CellGroupData[{
Cell[115239, 3179, 2317, 52, 70, "Input",
 CellID->13704],
Cell[117559, 3233, 157, 5, 36, "Output",
 CellID->310208007]
}, Open  ]],
Cell[117731, 3241, 70898, 1183, 70, "Input",
 CellID->1998685447]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[188678, 4430, 49, 1, 70, "Subsection",
 CellID->640],
Cell[188730, 4433, 520, 15, 70, "Text",
 CellID->24972],
Cell[189253, 4450, 209, 5, 70, "MathCaption",
 CellID->20353],
Cell[189465, 4457, 4242, 113, 70, "Input",
 CellID->3465],
Cell[193710, 4572, 312, 8, 70, "MathCaption",
 CellID->45],
Cell[194025, 4582, 2665, 78, 70, "Input",
 CellID->5306],
Cell[196693, 4662, 173, 4, 70, "MathCaption",
 CellID->31126],
Cell[196869, 4668, 5045, 131, 70, "Input",
 CellID->27557],
Cell[201917, 4801, 208, 9, 70, "MathCaption",
 CellID->23296],
Cell[CellGroupData[{
Cell[202150, 4814, 440, 14, 47, "Input",
 CellID->1546],
Cell[202593, 4830, 4528, 78, 279, "Output",
 Evaluatable->False,
 CellID->382114047]
}, Open  ]],
Cell[207136, 4911, 85, 1, 70, "MathCaption",
 CellID->31880],
Cell[CellGroupData[{
Cell[207246, 4916, 514, 16, 47, "Input",
 CellID->18796],
Cell[207763, 4934, 6745, 115, 313, "Output",
 Evaluatable->False,
 CellID->628848393]
}, Open  ]],
Cell[214523, 5052, 79, 1, 70, "MathCaption",
 CellID->19484],
Cell[CellGroupData[{
Cell[214627, 5057, 582, 17, 47, "Input",
 CellID->22693],
Cell[215212, 5076, 6121, 104, 268, "Output",
 Evaluatable->False,
 CellID->11440604]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[221382, 5186, 53, 1, 70, "Subsection",
 CellID->17194],
Cell[221438, 5189, 493, 11, 70, "Text",
 CellID->10605],
Cell[221934, 5202, 272, 7, 70, "MathCaption",
 CellID->17059],
Cell[222209, 5211, 4005, 131, 70, "Input",
 CellID->6838],
Cell[226217, 5344, 193, 4, 70, "MathCaption",
 CellID->20556],
Cell[226413, 5350, 587, 22, 70, "Input",
 CellID->15179],
Cell[227003, 5374, 782, 19, 70, "MathCaption",
 CellID->18296],
Cell[CellGroupData[{
Cell[227810, 5397, 1397, 36, 70, "Input",
 CellID->10782],
Cell[229210, 5435, 1434, 51, 88, "Output",
 CellID->1089967265]
}, Open  ]],
Cell[230659, 5489, 198, 5, 70, "MathCaption",
 CellID->24729],
Cell[230860, 5496, 147, 4, 70, "MathCaption",
 CellID->1556],
Cell[CellGroupData[{
Cell[231032, 5504, 413, 14, 28, "Input",
 CellID->20042],
Cell[231448, 5520, 4024, 70, 125, "Output",
 Evaluatable->False,
 CellID->252930653]
}, Open  ]],
Cell[235487, 5593, 133, 4, 70, "MathCaption",
 CellID->20840],
Cell[CellGroupData[{
Cell[235645, 5601, 415, 14, 28, "Input",
 CellID->25536],
Cell[236063, 5617, 6671, 113, 217, "Output",
 Evaluatable->False,
 CellID->284311451]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[242783, 5736, 84, 1, 70, "Subsection",
 CellID->3279],
Cell[242870, 5739, 145, 4, 70, "Text",
 CellID->6094],
Cell[243018, 5745, 245, 9, 70, "Text",
 CellID->12543],
Cell[243266, 5756, 621, 23, 70, "DisplayMath",
 CellID->4306],
Cell[243890, 5781, 385, 12, 70, "Text",
 CellID->8532],
Cell[244278, 5795, 1112, 38, 70, "DisplayMath",
 CellID->2379],
Cell[245393, 5835, 279, 7, 70, "MathCaption",
 CellID->18590],
Cell[245675, 5844, 1285, 44, 70, "Input",
 CellID->6635],
Cell[246963, 5890, 411, 10, 70, "MathCaption",
 CellID->22881],
Cell[247377, 5902, 132, 4, 70, "Input",
 CellID->7516],
Cell[247512, 5908, 400, 14, 70, "MathCaption",
 CellID->13577],
Cell[CellGroupData[{
Cell[247937, 5926, 829, 21, 70, "Input",
 CellID->17173],
Cell[248769, 5949, 396, 14, 36, "Output",
 CellID->237754635]
}, Open  ]],
Cell[249180, 5966, 70, 1, 70, "MathCaption",
 CellID->23017],
Cell[CellGroupData[{
Cell[249275, 5971, 223, 7, 28, "Input",
 CellID->20978],
Cell[249501, 5980, 3837, 67, 253, "Output",
 Evaluatable->False,
 CellID->740762939]
}, Open  ]],
Cell[253353, 6050, 151, 4, 70, "Text",
 CellID->30203],
Cell[253507, 6056, 156, 4, 70, "Text",
 CellID->24681],
Cell[253666, 6062, 183, 4, 70, "MathCaption",
 CellID->27823],
Cell[253852, 6068, 896, 25, 70, "Input",
 CellID->14232],
Cell[254751, 6095, 346, 8, 70, "MathCaption",
 CellID->20575],
Cell[CellGroupData[{
Cell[255122, 6107, 104, 3, 28, "Input",
 CellID->32632],
Cell[255229, 6112, 2112, 39, 248, "Output",
 Evaluatable->False,
 CellID->42836257]
}, Open  ]],
Cell[257356, 6154, 175, 4, 70, "Text",
 CellID->5140],
Cell[257534, 6160, 170, 5, 70, "Text",
 CellID->1531],
Cell[257707, 6167, 366, 7, 70, "MathCaption",
 CellID->8386],
Cell[CellGroupData[{
Cell[258098, 6178, 1909, 50, 70, "Input",
 CellID->16361],
Cell[260010, 6230, 419, 15, 36, "Output",
 CellID->1920518206]
}, Open  ]],
Cell[260444, 6248, 194, 5, 70, "Text",
 CellID->4397],
Cell[260641, 6255, 171, 4, 70, "MathCaption",
 CellID->22953],
Cell[CellGroupData[{
Cell[260837, 6263, 1320, 38, 70, "Input",
 CellID->12809],
Cell[262160, 6303, 419, 15, 36, "Output",
 CellID->1908896991]
}, Open  ]],
Cell[262594, 6321, 148, 4, 70, "MathCaption",
 CellID->8035],
Cell[CellGroupData[{
Cell[262767, 6329, 347, 11, 28, "Input",
 CellID->4121],
Cell[263117, 6342, 3886, 68, 253, "Output",
 Evaluatable->False,
 CellID->620296164]
}, Open  ]],
Cell[267018, 6413, 224, 5, 70, "MathCaption",
 CellID->32537],
Cell[CellGroupData[{
Cell[267267, 6422, 164, 5, 28, "Input",
 CellID->27743],
Cell[267434, 6429, 2373, 43, 254, "Output",
 Evaluatable->False,
 CellID->216574532]
}, Open  ]],
Cell[269822, 6475, 314, 9, 70, "Text",
 CellID->15533],
Cell[270139, 6486, 152, 4, 70, "Text",
 CellID->28306],
Cell[270294, 6492, 158, 4, 70, "MathCaption",
 CellID->28396],
Cell[CellGroupData[{
Cell[270477, 6500, 287, 9, 70, "Input",
 CellID->28786],
Cell[270767, 6511, 1711, 57, 95, "Output",
 CellID->1795803233]
}, Open  ]],
Cell[272493, 6571, 168, 4, 70, "MathCaption",
 CellID->22035],
Cell[CellGroupData[{
Cell[272686, 6579, 241, 8, 70, "Input",
 CellID->20673],
Cell[272930, 6589, 229, 7, 36, "Output",
 CellID->1814744480]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[273208, 6602, 65, 1, 70, "Subsection",
 CellID->24434],
Cell[273276, 6605, 404, 7, 70, "Text",
 CellID->29407],
Cell[273683, 6614, 911, 16, 70, "Text",
 CellID->1804],
Cell[274597, 6632, 597, 12, 70, "Text",
 CellID->30807],
Cell[275197, 6646, 237, 5, 70, "MathCaption",
 CellID->3952],
Cell[CellGroupData[{
Cell[275459, 6655, 3068, 85, 70, "Input",
 CellID->27626],
Cell[278530, 6742, 735, 23, 54, "Output",
 CellID->1190069951]
}, Open  ]],
Cell[279280, 6768, 407, 11, 70, "MathCaption",
 CellID->2331],
Cell[CellGroupData[{
Cell[279712, 6783, 756, 22, 65, "Input",
 CellID->4569],
Cell[280471, 6807, 162440, 2667, 383, "Output",
 Evaluatable->False,
 CellID->17599912]
}, Open  ]],
Cell[442926, 9477, 894, 28, 70, "MathCaption",
 CellID->11808],
Cell[CellGroupData[{
Cell[443845, 9509, 3234, 88, 70, "Input",
 CellTags->"ELPDEExample",
 CellID->28531],
Cell[447082, 9599, 762, 24, 36, "Output",
 CellTags->"ELPDEExample",
 CellID->190992315]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[447893, 9629, 60, 1, 70, "Subsection",
 CellID->31731],
Cell[447956, 9632, 141, 4, 70, "Text",
 CellID->27707],
Cell[448100, 9638, 238, 5, 70, "MathCaption",
 CellID->1146],
Cell[448341, 9645, 2440, 62, 70, "Input",
 CellID->30942],
Cell[450784, 9709, 171, 4, 70, "MathCaption",
 CellID->20145],
Cell[CellGroupData[{
Cell[450980, 9717, 958, 29, 70, "Input",
 CellID->17330],
Cell[451941, 9748, 1923, 55, 124, "Output",
 CellID->1046106954]
}, Open  ]],
Cell[453879, 9806, 456, 8, 70, "Text",
 CellID->30961],
Cell[454338, 9816, 251, 5, 70, "Text",
 CellID->30047],
Cell[CellGroupData[{
Cell[454614, 9825, 1801, 46, 70, "Input",
 CellID->24770],
Cell[456418, 9873, 822, 25, 64, "Output",
 CellID->625624277]
}, Open  ]],
Cell[457255, 9901, 448, 8, 70, "Text",
 CellID->7426]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[457752, 9915, 64, 2, 70, "Section",
 CellTags->"s:4",
 CellID->19387],
Cell[457819, 9919, 1012, 23, 70, "Text",
 CellID->14614],
Cell[458834, 9944, 471, 15, 70, "MathCaption",
 CellID->22686],
Cell[CellGroupData[{
Cell[459330, 9963, 1121, 32, 70, "Input",
 CellID->1705],
Cell[460454, 9997, 161, 5, 36, "Output",
 CellID->220889922]
}, Open  ]],
Cell[460630, 10005, 117, 3, 70, "MathCaption",
 CellID->19539],
Cell[CellGroupData[{
Cell[460772, 10012, 1184, 33, 70, "Input",
 CellID->27365],
Cell[461959, 10047, 162, 5, 36, "Output",
 CellID->558720145]
}, Open  ]],
Cell[462136, 10055, 408, 7, 70, "Text",
 CellID->25521],
Cell[462547, 10064, 395, 11, 70, "MathCaption",
 CellID->21044],
Cell[CellGroupData[{
Cell[462967, 10079, 1748, 49, 70, "Input",
 CellID->14502],
Cell[464718, 10130, 164, 5, 36, "Output",
 CellID->944361772]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[464931, 10141, 67, 2, 70, "Section",
 CellTags->"s:5",
 CellID->31057],
Cell[CellGroupData[{
Cell[465023, 10147, 98, 2, 70, "Subsection",
 CellTags->"EventLocatorOptionSummary",
 CellID->32120],
Cell[465124, 10151, 1920, 49, 70, "DefinitionBox3Col",
 CellID->30716],
Cell[467047, 10202, 114, 4, 70, "Caption",
 CellID->10038]
}, Open  ]],
Cell[CellGroupData[{
Cell[467198, 10211, 112, 2, 70, "Subsection",
 CellTags->"EventLocationMethodOptionSummary",
 CellID->25131],
Cell[467313, 10215, 1276, 36, 70, "DefinitionBox",
 CellID->10388],
Cell[468592, 10253, 136, 5, 70, "Caption",
 CellID->15344]
}, Open  ]],
Cell[CellGroupData[{
Cell[468765, 10263, 84, 2, 70, "Subsection",
 CellTags->"BrentOptionSummary",
 CellID->10439],
Cell[468852, 10267, 2695, 78, 70, "DefinitionBox3Col",
 CellID->7535],
Cell[471550, 10347, 140, 5, 70, "Caption",
 CellID->15901]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[471729, 10357, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

