(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    328716,      13525]
NotebookOptionsPosition[    302662,      12683]
NotebookOutlinePosition[    310195,      12884]
CellTagsIndexPosition[    309094,      12853]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["ExplicitRungeKutta Method for NDSolve", "Title",
 CellTags->"c:1",
 CellID->269234798],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->998382023],

Cell[TextData[{
 "Extrapolation methods are a class of arbitrary-order methods with automatic \
order and step size control. The error estimate comes from computing a \
solution over an interval using the same method with a varying number of \
steps and using extrapolation on the polynomial that fits through the \
computed solutions, giving a composite higher-order method [",
 ButtonBox["BS64",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#22705"],
 "]. At the same time, the polynomials give a means of error estimation."
}], "Text",
 CellID->1938414111],

Cell["\<\
Typically, for low precision, the extrapolation methods have not been \
competitive with Runge-Kutta-type methods. For high precision, however, the \
arbitrary order means that they can be arbitrarily faster than fixed-order \
methods for very precise tolerances.\
\>", "Text",
 CellID->1589095302],

Cell[TextData[{
 "The order and step size control is based on the codes ",
 StyleBox["odex.f", "MR"],
 " and ",
 StyleBox["seulex.f", "MR"],
 " described in [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "] and [",
 ButtonBox["HW96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#32197"],
 "]."
}], "Text",
 CellID->992679970],

Cell["\<\
This loads packages that contain some utility functions for plotting step \
sequences and some predefined problems.\
\>", "Text",
 CellID->723849525],

Cell[BoxData[{
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveUtilities`\>\"", "]"}], 
  ";"}]}], "Input",
 InitializationCell->True,
 CellLabel->"In[3]:=",
 CellID->272055467]
}, Open  ]],

Cell[CellGroupData[{

Cell["Extrapolation", "Section",
 CellTags->"s:2",
 CellID->170762999],

Cell[TextData[{
 "The method ",
 StyleBox["DoubleStep", "MR"],
 " performs a single application of Richardson's extrapolation for any one \
step integration method and is described within \"",
 ButtonBox["DoubleStep Method for NDSolve",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveDoubleStep"],
 "\"."
}], "Text",
 CellID->1049443999],

Cell["\<\
Extrapolation generalizes the idea of Richardson's extrapolation to a \
sequence of refinements. A solution is computed over an interval using the \
same method with a varying number of steps and using extrapolation on the \
polynomial that fits through the computed solutions, giving a composite \
higher-order method. The polynomials at the same time provide a means of \
error estimation.\
\>", "Text",
 CellID->462117005],

Cell["Consider a differential system", "Text",
 CellID->1744473514],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]"], "(", "t", ")"}], " ", "=", " ", 
    RowBox[{"f", "(", 
     RowBox[{"t", ",", " ", 
      RowBox[{"y", "(", "t", ")"}]}], ")"}]}], ",", " ", 
   RowBox[{
    RowBox[{"y", "(", 
     SubscriptBox["t", "0"], ")"}], " ", "=", " ", 
    RowBox[{
     SubscriptBox["y", "0"], "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->1304279132],

Cell[TextData[{
 "Let ",
 StyleBox["H",
  FontSlant->"Italic"],
 " > 0 be a basic step size and choose a monotonically increasing sequence of \
positive integers"
}], "Text",
 CellID->1637667014],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["n", "1"], "<", " ", 
   SubscriptBox["n", "2"], "<", " ", 
   SubscriptBox["n", "3"], "<", " ", "\[CenterEllipsis]", " ", "<", 
   SubscriptBox["n", "k"]}], TraditionalForm]], "DisplayMath",
 CellID->593348802],

Cell["and define the corresponding step sizes", "Text",
 CellID->2031094749],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["h", "1"], ">", " ", 
   SubscriptBox["h", "2"], ">", " ", 
   SubscriptBox["h", "3"], ">", " ", "\[CenterEllipsis]", ">", 
   SubscriptBox["h", "k"]}], TraditionalForm]], "DisplayMath",
 CellID->1397129139],

Cell["by", "Text",
 CellID->553099557],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["h", "i"], "=", " ", 
    FractionBox["H", 
     SubscriptBox["n", "i"]]}], ",", 
   RowBox[{"i", "=", "1"}], ",", "2", ",", "\[Ellipsis]", ",", 
   RowBox[{"k", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->69213927],

Cell[TextData[{
 "Choose a numerical method of order ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]]],
 " and compute the solution of the initial value problem by carrying out ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["n", "i"], TraditionalForm]]],
 " steps with step size ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["h", "i"], TraditionalForm]]],
 "to obtain:"
}], "Text",
 CellID->648444595],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["T", 
     RowBox[{"i", ",", "1"}]], "=", " ", 
    RowBox[{
     SubscriptBox["y", 
      SubscriptBox["h", "i"]], "(", 
     RowBox[{
      SubscriptBox["t", "o"], "+", " ", "H"}], ")"}]}], ",", 
   RowBox[{"i", "=", "1"}], ",", "2", ",", "\[Ellipsis]", ",", 
   RowBox[{"k", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->1695389189],

Cell["\<\
Extrapolation is performed using the Aitken-Neville algorithm by building up \
a table of values:\
\>", "Text",
 CellID->2050595844],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["T", 
     RowBox[{"i", ",", " ", "j"}]], "=", " ", 
    RowBox[{
     SubscriptBox["T", 
      RowBox[{"i", ",", " ", 
       RowBox[{"j", "-", "1"}]}]], " ", "+", " ", 
     FractionBox[
      RowBox[{
       SubscriptBox["T", 
        RowBox[{"i", ",", " ", 
         RowBox[{"j", " ", "-", " ", "1"}]}]], " ", "-", " ", 
       SubscriptBox["T", 
        RowBox[{
         RowBox[{"i", "-", "1"}], ",", " ", 
         RowBox[{"j", " ", "-", " ", "1"}]}]]}], 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox[
          SubscriptBox["n", "i"], 
          SubscriptBox["n", 
           RowBox[{"i", " ", "-", " ", "j", " ", "+", " ", "1"}]]], ")"}], 
        "p"], " ", "-", " ", "1"}]]}]}], " ", ",", 
   RowBox[{"i", "=", "2"}], ",", "\[Ellipsis]", ",", "k", ",", 
   RowBox[{"j", " ", "=", " ", "2"}], ",", "\[Ellipsis]", ",", 
   RowBox[{"i", "."}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"AitkenNeville",
 CellID->1289539621],

Cell["\<\
A dependency graph of the values illustrates the relationship:\
\>", "Text",
 CellID->1982698690],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["T", "11"], " ", " ", " ", " ", " ", " ", " "},
    {" ", "\[UpperLeftArrow]", " ", " ", " ", " ", " ", " "},
    {
     SubscriptBox["T", "21"], "\[LongLeftArrow]", 
     SubscriptBox["T", "22"], " ", " ", " ", " ", " "},
    {" ", "\[UpperLeftArrow]", " ", "\[UpperLeftArrow]", " ", " ", " ", " "},
    {
     SubscriptBox["T", "31"], "\[LongLeftArrow]", 
     SubscriptBox["T", "32"], "\[LongLeftArrow]", 
     SubscriptBox["T", "33"], " ", " ", " "},
    {" ", "\[UpperLeftArrow]", "  ", "\[UpperLeftArrow]", " ", 
     "\[UpperLeftArrow]", " ", " "},
    {
     SubscriptBox["T", "41"], "\[LongLeftArrow]", 
     SubscriptBox["T", "42"], "\[LongLeftArrow]", 
     SubscriptBox["T", "43"], "\[LongLeftArrow]", 
     SubscriptBox["T", "44"], " "},
    {"\[CenterEllipsis]", "\[CenterEllipsis]", "\[CenterEllipsis]", 
     "\[CenterEllipsis]", "\[CenterEllipsis]", "\[CenterEllipsis]", 
     "\[CenterEllipsis]", "\[CenterEllipsis]"}
   }], TraditionalForm]], "DisplayMath",
 CellID->823382723],

Cell[TextData[{
 "For ",
 StyleBox["k",
  FontSlant->"Italic"],
 " =2, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["n", "1"], TraditionalForm]]],
 "= 1, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["n", "2"], TraditionalForm]]],
 "= 2 extrapolation is identical to Richardson's extrapolation."
}], "Text",
 CellID->1965840171],

Cell[CellGroupData[{

Cell["Extrapolation sequences", "Subsection",
 CellID->1973820076],

Cell["\<\
Any extrapolation sequence can be specified in the implementation. Some \
common choices are as follows.\
\>", "Text",
 CellID->655573988],

Cell["This is the Romberg sequence.", "Text",
 CellID->1634522524],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`RombergSequenceFunction", "[", 
  RowBox[{"1", ",", " ", "10"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->771343838],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "4", ",", "8", ",", "16", ",", "32", ",", "64", ",", 
   "128", ",", "256", ",", "512"}], "}"}]], "Output",
 ImageSize->{252, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->736392212]
}, Open  ]],

Cell["This is the Bulirsch sequence.", "Text",
 CellID->1060157066],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`BulirschSequenceFunction", "[", 
  RowBox[{"1", ",", " ", "10"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->1040578636],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "3", ",", "4", ",", "6", ",", "8", ",", "12", ",", "16",
    ",", "24", ",", "32"}], "}"}]], "Output",
 ImageSize->{217, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1452765585]
}, Open  ]],

Cell["This is the Harmonic sequence.", "Text",
 CellID->851087529],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`HarmonicSequenceFunction", "[", 
  RowBox[{"1", ",", " ", "10"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->482190291],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8", 
   ",", "9", ",", "10"}], "}"}]], "Output",
 ImageSize->{196, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1284538269]
}, Open  ]],

Cell[TextData[{
 "A sequence that satisfies ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["n", "i"], "/", 
       SubscriptBox["n", 
        RowBox[{"i", " ", "-", " ", "j", " ", "+", " ", "1"}]]}], ")"}], "p"],
     "\[GreaterEqual]", "2", " "}], TraditionalForm]]],
 " has the effect of minimizing the roundoff errors for an order-",
 Cell[BoxData[
  FormBox["p", TraditionalForm]]],
 " base integration method."
}], "Text",
 CellID->1574937055],

Cell["\<\
For a base method of order two the first entries in the sequence are given by \
the following.\
\>", "Text",
 CellID->754245758],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`OptimalRoundingSequenceFunction", "[", 
  RowBox[{"1", ",", " ", "10", ",", " ", "2"}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->107458757],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "2", ",", "3", ",", "5", ",", "8", ",", "12", ",", "17", ",", 
   "25", ",", "36", ",", "51"}], "}"}]], "Output",
 ImageSize->{224, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->20674660]
}, Open  ]],

Cell["\<\
Here is an example of adding a function to define the Harmonic sequence is \
where the method order is an optional pattern.\
\>", "Text",
 CellID->1216362763],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"Default", "[", 
     RowBox[{"myseqfun", ",", " ", "3"}], "]"}], " ", "=", " ", "1"}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"myseqfun", "[", 
   RowBox[{"n1_", ",", " ", "n2_", ",", " ", "p_."}], "]"}], ":=", " ", 
  RowBox[{"Range", "[", 
   RowBox[{"n1", ",", " ", "n2"}], "]"}]}]}], "Input",
 CellLabel->"In[9]:=",
 CellID->1851932271]
}, Open  ]],

Cell[CellGroupData[{

Cell["Rounding error accumulation", "Subsection",
 CellID->1781176538],

Cell[TextData[{
 "For high-order extrapolation an important consideration is the accumulation \
of rounding errors in the Aitken-Neville algorithm (",
 
 CounterBox["NumberedEquation", "AitkenNeville"],
 ")."
}], "Text",
 CellID->706546129],

Cell[TextData[{
 "As an example consider Exercise 5 of Section II.9 in [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "]."
}], "Text",
 CellID->297797426],

Cell[TextData[{
 "Suppose that the entries ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["T", "11"], ",", " ", 
    SubscriptBox["T", "21"], ",", " ", 
    SubscriptBox["T", "31"], ",", " ", "\[Ellipsis]"}], TraditionalForm]]],
 " are disturbed with rounding errors ",
 Cell[BoxData[
  FormBox["\[Epsilon]", TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Epsilon]"}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox["\[Epsilon]", TraditionalForm]]],
 ", \[Ellipsis] and compute the propagation of these errors into the \
extrapolation table."
}], "Text",
 CellID->1664787639],

Cell[TextData[{
 "Due to the linearity of the extrapolation process ",
 "(",
 
 CounterBox["NumberedEquation", "AitkenNeville"],
 "),",
 " suppose that the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "j"}]], TraditionalForm]]],
 " are equal to zero and take ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Epsilon]", " ", "=", " ", "1"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->2103675269],

Cell[TextData[{
 "This shows the evolution of the Aitken-Neville algorithm (",
 
 CounterBox["NumberedEquation", "AitkenNeville"],
 ") on the initial data using the Harmonic sequence and a symmetric order-two \
base integration method."
}], "Text",
 CellID->850896983],

Cell[BoxData[
 FormBox[GridBox[{
    {"1.`", " ", " ", " ", " ", " ", " ", " "},
    {
     RowBox[{"-", "1."}], 
     RowBox[{"-", "1.6666666666666665`"}], " ", " ", " ", " ", " ", " "},
    {"1.", "2.6`", "3.1333333333333333`", " ", " ", " ", " ", " "},
    {
     RowBox[{"-", "1."}], 
     RowBox[{"-", "3.5714285714285716`"}], 
     RowBox[{"-", "5.628571428571429`"}], 
     RowBox[{"-", "6.212698412698413`"}], " ", " ", " ", " "},
    {"1.", "4.555555555555555`", "9.126984126984127`", "11.937566137566137`", 
     "12.693827160493827`", " ", " ", " "},
    {
     RowBox[{"-", "1."}], 
     RowBox[{"-", "5.545454545454546`"}], 
     RowBox[{"-", "13.626262626262626`"}], 
     RowBox[{"-", "21.21067821067821`"}], 
     RowBox[{"-", "25.354208754208756`"}], 
     RowBox[{"-", "26.44129549462883`"}], " ", " "},
    {"1.", "6.538461538461538`", "19.125874125874127`", "35.00569800569801`", 
     "47.65438265438266`", "54.14403522403523`", "55.822896280674065`", " "},
    {
     RowBox[{"-", "1."}], 
     RowBox[{"-", "7.533333333333333`"}], 
     RowBox[{"-", "25.625641025641023`"}], 
     RowBox[{"-", "54.31250971250971`"}], 
     RowBox[{"-", "84.08524561857895`"}], 
     RowBox[{"-", "105.64263933597267`"}], 
     RowBox[{"-", "116.29508430663986`"}], 
     RowBox[{"-", "119.02711574453373`"}]}
   }], TraditionalForm]], "DisplayMath",
 CellID->1734001566],

Cell["\<\
Hence, for an order-sixteen method approximately two decimal digits are lost \
due to rounding error accumulation.\
\>", "Text",
 CellID->604636216],

Cell[TextData[{
 "This model is somewhat crude because, as we will see later, it is more \
likely that rounding errors are made in ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{
     RowBox[{"i", "+", "1"}], ",", " ", "1"}]], TraditionalForm]]],
 "than in ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "1"}]], TraditionalForm]]],
 "for ",
 StyleBox["i",
  FontSlant->"Italic"],
 " \[GreaterEqual] 1."
}], "Text",
 CellID->398802525]
}, Open  ]],

Cell[CellGroupData[{

Cell["Rounding error reduction", "Subsection",
 CellID->1637113763],

Cell["\<\
It seems worthwhile to look for approaches that can reduce the effect of \
rounding errors in high-order extrapolation.\
\>", "Text",
 CellID->1894175838],

Cell[TextData[{
 "Selecting a different step sequence to diminish rounding errors is one \
approach, although the drawback is that the number of integration steps \
needed to form the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", " ", ",", " ", "1"}]], TraditionalForm]]],
 " in the first column of the extrapolation table requires more work."
}], "Text",
 CellID->234017567],

Cell[TextData[{
 "An alternative strategy, which does not appear to have received a great \
deal of attention in the context of extrapolation, is to modify the \
base-integration method in order to reduce the magnitude of the rounding \
errors in floating-point operations. This approach, based on ideas that dated \
back to [",
 ButtonBox["G51",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#10068"],
 "] and used to good effect for the two body problem in [",
 ButtonBox["F96b",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#26897022"],
 "] (",
 "for background ",
 "see also [",
 ButtonBox["K65",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#4309"],
 "], [",
 ButtonBox["M65a",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#28965"],
 "], [",
 ButtonBox["M65b",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#37"],
 "], [",
 ButtonBox["V79",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#11721"],
 "]), are explained next."
}], "Text",
 CellID->313012838]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Base methods", "Section",
 CellTags->"s:3",
 CellID->1712532361],

Cell["\<\
The following methods are the most common choices for base integrators in \
extrapolation.\
\>", "Text",
 CellID->60353995],

Cell[TextData[StyleBox["ExplicitEuler", "MR"]], "BulletedText",
 CellID->968586826],

Cell[TextData[StyleBox["ExplicitMidpoint", "MR"]], "BulletedText",
 CellID->1199571237],

Cell[TextData[{
 StyleBox["ExplicitModifiedMidpoint", "MR"],
 " ",
 "(Gragg smoothing step)"
}], "BulletedText",
 CellID->831697833],

Cell[TextData[StyleBox["LinearlyImplicitEuler", "MR"]], "BulletedText",
 CellID->1061704238],

Cell[TextData[{
 StyleBox["LinearlyImplicitMidpoint", "MR"],
 " (Bader-Deuflhard formulation without smoothing step)"
}], "BulletedText",
 CellID->112244655],

Cell[CellGroupData[{

Cell[TextData[{
 StyleBox["LinearlyImplicitModifiedMidpoint", "MR"],
 " (Bader-Deuflhard formulation with smoothing step)"
}], "BulletedText",
 CellID->1872276470],

Cell[TextData[{
 "For efficiency, these have been built into ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " and are callable via the ",
 StyleBox["Method", "MR"],
 " option."
}], "Text",
 CellID->5720025],

Cell[TextData[{
 "The implementation of these methods has a special interpretation for \
multiple substeps within ",
 StyleBox["DoubleStep", "MR"],
 " and ",
 StyleBox["Extrapolation", "MR"],
 "."
}], "Text",
 CellID->963332185],

Cell[TextData[{
 "The methods are now described together with an increment reformulation that \
is used to reduce rounding error accumulation throughout ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->206983113]
}, Open  ]],

Cell[CellGroupData[{

Cell["Multiple Euler steps", "Subsection",
 CellID->2089899431],

Cell[TextData[{
 "Given ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["t", "0"], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "0"], TraditionalForm]]],
 " and ",
 StyleBox["H",
  FontSlant->"Italic"],
 ", consider a succession of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 " integration steps with step size ",
 Cell[BoxData[
  FormBox[
   RowBox[{"h", " ", "=", " ", 
    RowBox[{"H", "/", "n"}]}], TraditionalForm]]],
 " carried out using Euler's method:"
}], "Text",
 CellID->390785592],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["y", "1"], "=", 
     RowBox[{
      SubscriptBox["y", "0"], "+", " ", 
      RowBox[{"h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "0"], ",", 
         SubscriptBox["y", "0"]}], ")"}]}]}]},
    {
     SubscriptBox["y", "2"], "=", 
     RowBox[{
      SubscriptBox["y", "1"], "+", " ", 
      RowBox[{"h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "1"], ",", 
         SubscriptBox["y", "1"]}], ")"}]}]}]},
    {
     SubscriptBox["y", "3"], "=", 
     RowBox[{
      SubscriptBox["y", "2"], "+", " ", 
      RowBox[{"h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "2"], ",", 
         SubscriptBox["y", "2"]}], ")"}]}]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     SubscriptBox["y", "n"], "=", 
     RowBox[{
      SubscriptBox["y", 
       RowBox[{"n", "-", "1"}]], "+", " ", 
      RowBox[{"h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", 
          RowBox[{"n", "-", "1"}]], ",", 
         SubscriptBox["y", 
          RowBox[{"n", "-", "1"}]]}], ")"}]}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"EulerSteps",
 CellID->961228871],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["t", "i"], "=", " ", 
    RowBox[{
     SubscriptBox["t", "0"], "+", " ", 
     RowBox[{"i", " ", "h"}]}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->49874541]
}, Open  ]],

Cell[CellGroupData[{

Cell["Correspondence with explicit Runge-Kutta methods", "Subsection",
 CellID->867682998],

Cell[TextData[{
 "It is well known that, for certain base integration schemes, the entries ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "j"}]], TraditionalForm]]],
 " in the extrapolation table produced from (",
 
 CounterBox["NumberedEquation", "AitkenNeville"],
 ") correspond to explicit Runge-Kutta methods (see Exercise 1, Section II.9 \
in [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "])."
}], "Text",
 CellID->30107986],

Cell[TextData[{
 "For example, (",
 
 CounterBox["NumberedEquation", "EulerSteps"],
 ") is equivalent to an ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 "-stage explicit Runge-Kutta method:"
}], "Text",
 CellID->1901806812],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["k", "i"], "=", 
     RowBox[{
      RowBox[{"f", "(", 
       RowBox[{
        RowBox[{
         SubscriptBox["t", "0"], "+", " ", 
         RowBox[{
          SubscriptBox["c", "i"], " ", "h"}]}], ",", " ", 
        RowBox[{
         SubscriptBox["y", "0"], "+", " ", 
         RowBox[{"H", " ", 
          RowBox[{
           UnderoverscriptBox["\[Sum]", 
            RowBox[{"i", "=", "1"}], "n"], 
           RowBox[{
            SubscriptBox["a", 
             RowBox[{"i", ",", " ", "j"}]], 
            SubscriptBox["k", "j"]}]}]}]}]}], ")"}], ",", " "}], 
     RowBox[{
      RowBox[{"i", " ", "=", " ", "1"}], ",", "\[Ellipsis]", ",", "n", 
      ","}]},
    {
     SubscriptBox["y", "n"], "=", 
     RowBox[{
      SubscriptBox["y", "0"], "+", " ", 
      RowBox[{"H", " ", 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"i", "=", "1"}], "n"], 
        RowBox[{
         SubscriptBox["b", "i"], 
         SubscriptBox["k", "i"]}]}]}]}], " "}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"ExplicitRungeKuttaMethod",
 CellID->501375888],

Cell["where the coefficients are represented by the Butcher table:", "Text",
 CellID->736654116],

Cell[BoxData[
 FormBox[GridBox[{
    {"0", " ", " ", " ", " "},
    {
     RowBox[{"1", "/", "n"}], 
     RowBox[{"1", "/", "n"}], " ", " ", " "},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[DescendingEllipsis]", 
     " ", " "},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"n", "-", "1"}], ")"}], "/", "n"}], 
     RowBox[{"1", "/", "n"}], "\[CenterEllipsis]", 
     RowBox[{"1", "/", "n"}], " "},
    {" ", 
     RowBox[{"1", "/", "n"}], "\[CenterEllipsis]", 
     RowBox[{"1", "/", "n"}], 
     RowBox[{"1", "/", "n"}]}
   },
   GridBoxDividers->{
    "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
     "Rows" -> {False, False, False, False, {True}, False}, 
     "RowsIndexed" -> {}}], TraditionalForm]], "NumberedEquation",
 CellTags->"ExplicitRungeKuttaCoefficients",
 CellID->52120590]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reformulation", "Subsection",
 CellID->18679880],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]y", "n"], TraditionalForm]]],
 " = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", 
     RowBox[{"n", "+", "1"}]], " ", "-", " ", 
    SubscriptBox["y", "n"]}], TraditionalForm]]],
 ". Then the integration (",
 
 CounterBox["NumberedEquation", "EulerSteps"],
 ") can be rewritten to reflect the correspondence with an explicit \
Runge-Kutta method (",
 
 CounterBox["NumberedEquation", "ExplicitRungeKuttaMethod"],
 ", ",
 
 CounterBox["NumberedEquation", "ExplicitRungeKuttaCoefficients"],
 ") as:"
}], "Text",
 CellID->692845737],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["\[CapitalDelta]y", "0"], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     SubscriptBox["\[CapitalDelta]y", "1"], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "1"], ",", 
        RowBox[{
         SubscriptBox["y", "0"], "+", 
         SubscriptBox["\[CapitalDelta]y", "0"]}]}], ")"}]}]},
    {
     SubscriptBox["\[CapitalDelta]y", "2"], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "2"], ",", 
        RowBox[{
         SubscriptBox["y", "0"], "+", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
           SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}]}], ")"}]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     SubscriptBox["\[CapitalDelta]y", 
      RowBox[{"n", "-", "1"}]], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", 
         RowBox[{"n", "-", "1"}]], ",", 
        RowBox[{
         SubscriptBox["y", "0"], "+", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
           SubscriptBox["\[CapitalDelta]y", "1"], "+", "\[CenterEllipsis]", 
           "+", 
           SubscriptBox["\[CapitalDelta]y", 
            RowBox[{"n", "-", "2"}]]}], ")"}]}]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"EulerIncrements",
 CellID->903017574],

Cell[TextData[{
 "where terms in the right-hand side of (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ") are now considered as departures from the ",
 StyleBox["same",
  FontSlant->"Italic"],
 " ",
 StyleBox["value",
  FontSlant->"Italic"],
 " ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "0"], TraditionalForm]]],
 "."
}], "Text",
 CellID->1752681446],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]y", "i"], TraditionalForm]]],
 " in (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ") correspond to the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "i"], TraditionalForm]]],
 " in (",
 
 CounterBox["NumberedEquation", "ExplicitRungeKuttaMethod"],
 ")."
}], "Text",
 CellID->1297481953],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalSigma]\[CapitalDelta]y", "n"], " ", "=", " ", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"i", "=", "0"}], 
      RowBox[{"n", "-", "1"}]], 
     SubscriptBox["\[CapitalDelta]y", "i"]}]}], TraditionalForm]]],
 ", then the required result can be recovered as:"
}], "Text",
 CellID->1301820099],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", "n"], "=", " ", 
   RowBox[{
    SubscriptBox["y", "0"], "+", " ", 
    SubscriptBox["\[CapitalSigma]\[CapitalDelta]y", "n"]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"EulerIncrementResult",
 CellID->1242311561],

Cell[TextData[{
 "Mathematically the formulations (",
 
 CounterBox["NumberedEquation", "EulerSteps"],
 ") and (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ", ",
 
 CounterBox["NumberedEquation", "EulerIncrementResult"],
 ") are equivalent. For ",
 StyleBox["n",
  FontSlant->"Italic"],
 " > 1, however, the computations in (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ") have the advantage of accumulating a sum of smaller ",
 StyleBox["O",
  FontSlant->"Italic"],
 "(",
 StyleBox["h",
  FontSlant->"Italic"],
 ") quantities, or increments, which reduces rounding error accumulation in \
finite-precision floating-point arithmetic."
}], "Text",
 CellID->1044174143]
}, Open  ]],

Cell[CellGroupData[{

Cell["Multiple linearly implicit Euler steps", "Subsection",
 CellID->1535837666],

Cell["\<\
Next consider a succession of integration steps carried out using the \
linearly implicit Euler method. Increments arise naturally in the description \
of many semi-implicit and implicit methods.\
\>", "Text",
 CellID->1555324399],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      SubscriptBox["\[CapitalDelta]y", "0"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     SubscriptBox["y", "1"], "=", 
     RowBox[{
      SubscriptBox["y", "0"], "+", " ", 
      SubscriptBox["\[CapitalDelta]y", "0"]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      SubscriptBox["\[CapitalDelta]y", "1"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "1"], ",", 
        SubscriptBox["y", "1"]}], ")"}]}]},
    {
     SubscriptBox["y", "2"], "=", 
     RowBox[{
      SubscriptBox["y", "1"], "+", " ", 
      SubscriptBox["\[CapitalDelta]y", "1"]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      SubscriptBox["\[CapitalDelta]y", "2"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "2"], ",", 
        SubscriptBox["y", "2"]}], ")"}]}]},
    {
     SubscriptBox["y", "3"], "=", 
     RowBox[{
      SubscriptBox["y", "2"], "+", 
      SubscriptBox["\[CapitalDelta]y", "2"]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      SubscriptBox["\[CapitalDelta]y", 
       RowBox[{"n", "-", "1"}]]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", 
         RowBox[{"n", "-", "1"}]], ",", 
        SubscriptBox["y", 
         RowBox[{"n", "-", "1"}]]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"LinearlyImplicitEulerSteps",
 CellID->609222856],

Cell[TextData[{
 "Here ",
 StyleBox["I",
  FontSlant->"Italic"],
 " denotes the Identity matrix and ",
 StyleBox["J",
  FontSlant->"Italic"],
 " denotes the Jacobian of ",
 StyleBox["f",
  FontSlant->"Italic"],
 ":"
}], "Text",
 CellID->1596191661],

Cell[BoxData[
 FormBox[
  RowBox[{"J", "=", 
   RowBox[{
    FractionBox[
     RowBox[{"\[PartialD]", "f"}], 
     RowBox[{"\[PartialD]", "y"}]], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["t", "0"], ",", 
       SubscriptBox["y", "0"]}], ")"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->376427577],

Cell[TextData[{
 "The solution of the equations for the increments in (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitEulerSteps"],
 ") is accomplished using a single LU decomposition of the matrix (",
 StyleBox["I",
  FontSlant->"Italic"],
 " - ",
 StyleBox["h J",
  FontSlant->"Italic"],
 ") followed by the solution of triangular linear systems for each right-hand \
side."
}], "Text",
 CellID->1808794093],

Cell[TextData[{
 "The desired result is obtained from (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitEulerSteps"],
 ") as:"
}], "Text",
 CellID->280405846],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", "n"], "=", 
   RowBox[{
    SubscriptBox["y", 
     RowBox[{"n", "-", "1"}]], "+", 
    SubscriptBox["\[CapitalDelta]y", 
     RowBox[{"n", "-", "1"}]]}]}], TraditionalForm]], "DisplayMath",
 CellID->1438131815]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reformulation", "Subsection",
 CellID->1921038749],

Cell[TextData[{
 "Reformulation in terms of increments as departures from ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "0"], TraditionalForm]]],
 " can be accomplished as follows:"
}], "Text",
 CellID->5198668],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      SubscriptBox["\[CapitalDelta]y", "0"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      SubscriptBox["\[CapitalDelta]y", "1"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "1"], ",", 
        RowBox[{
         SubscriptBox["y", "0"], "+", 
         SubscriptBox["\[CapitalDelta]y", "0"]}]}], ")"}]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      SubscriptBox["\[CapitalDelta]y", "2"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "2"], ",", 
        RowBox[{
         SubscriptBox["y", "0"], "+", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
           SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}]}], ")"}]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      SubscriptBox["\[CapitalDelta]y", 
       RowBox[{"n", "-", "1"}]]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", 
         RowBox[{"n", "-", "1"}]], ",", 
        RowBox[{
         SubscriptBox["y", "0"], "+", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
           SubscriptBox["\[CapitalDelta]y", "1"], "+", "\[CenterEllipsis]", 
           "+", 
           SubscriptBox["\[CapitalDelta]y", 
            RowBox[{"n", "-", "2"}]]}], ")"}]}]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"LinearlyImplicitEulerIncrements",
 CellID->1443851841],

Cell[TextData[{
 "The result for ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "n"], TraditionalForm]]],
 " using (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitEulerIncrements"],
 ") is obtained from (",
 
 CounterBox["NumberedEquation", "EulerIncrementResult"],
 ")."
}], "Text",
 CellID->736887286],

Cell[TextData[{
 "Notice that (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitEulerIncrements"],
 ") and (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ") are equivalent when ",
 StyleBox["J",
  FontSlant->"Italic"],
 " = 0."
}], "Text",
 CellID->212181781]
}, Open  ]],

Cell[CellGroupData[{

Cell["Multiple explicit midpoint steps", "Subsection",
 CellID->1386267624],

Cell[TextData[{
 "\"Expansions in even powers of ",
 StyleBox["h",
  FontSlant->"Italic"],
 " are extremely important for an efficient implementation of Richardson's \
extrapolation\" [",
 ButtonBox["S70",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#29276"],
 "]."
}], "Text",
 CellID->1127672878],

Cell["\<\
Consider a succession of integration steps carried out using one Euler step \
followed by multiple explicit midpoint steps:\
\>", "Text",
 CellID->1173410652],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["y", "1"], "=", 
     RowBox[{
      SubscriptBox["y", "0"], "+", " ", 
      RowBox[{"h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "0"], ",", 
         SubscriptBox["y", "0"]}], ")"}]}]}]},
    {
     SubscriptBox["y", "2"], "=", 
     RowBox[{
      SubscriptBox["y", "0"], "+", 
      RowBox[{"2", " ", "h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "1"], ",", 
         SubscriptBox["y", "1"]}], ")"}]}]}]},
    {
     SubscriptBox["y", "3"], "=", 
     RowBox[{
      SubscriptBox["y", "1"], "+", 
      RowBox[{"2", " ", "h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "2"], ",", 
         SubscriptBox["y", "2"]}], ")"}]}]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     SubscriptBox["y", "n"], "=", 
     RowBox[{
      SubscriptBox["y", 
       RowBox[{"n", "-", "2"}]], "+", 
      RowBox[{"2", " ", "h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", 
          RowBox[{"n", "-", "1"}]], ",", 
         SubscriptBox["y", 
          RowBox[{"n", "-", "1"}]]}], ")"}]}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"ExplicitMidpointSteps",
 CellID->1436142165],

Cell[TextData[{
 "If (",
 
 CounterBox["NumberedEquation", "ExplicitMidpointSteps"],
 ") is computed with ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     RowBox[{"2", " ", "m"}], " ", "-", " ", "1"}]}], TraditionalForm]]],
 " midpoint steps, then the method has a symmetric error expansion ([",
 ButtonBox["G65",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#6906"],
 "], [",
 ButtonBox["S70",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#29276"],
 "])."
}], "Text",
 CellID->1995355876]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reformulation", "Subsection",
 CellID->1203518639],

Cell[TextData[{
 "Reformulation of (",
 
 CounterBox["NumberedEquation", "ExplicitMidpointSteps"],
 ") can be accomplished in terms of increments as:"
}], "Text",
 CellID->1190465330],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["\[CapitalDelta]y", "0"], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     SubscriptBox["\[CapitalDelta]y", "1"], "=", 
     RowBox[{
      RowBox[{"2", "h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "1"], ",", 
         RowBox[{
          SubscriptBox["y", "0"], "+", 
          SubscriptBox["\[CapitalDelta]y", "0"]}]}], ")"}]}], "  ", "-", " ", 
      
      SubscriptBox["\[CapitalDelta]y", "0"]}]},
    {
     SubscriptBox["\[CapitalDelta]y", "2"], "=", 
     RowBox[{
      RowBox[{"2", "h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "2"], ",", 
         RowBox[{
          SubscriptBox["y", "0"], "+", 
          RowBox[{"(", 
           RowBox[{
            SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
            SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}]}], ")"}]}], "  ",
       "-", " ", 
      SubscriptBox["\[CapitalDelta]y", "1"]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     SubscriptBox["\[CapitalDelta]y", 
      RowBox[{"n", "-", "1"}]], "=", 
     RowBox[{
      RowBox[{"2", "h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", 
          RowBox[{"n", "-", "1"}]], ",", 
         RowBox[{
          SubscriptBox["y", "0"], "+", 
          RowBox[{"(", 
           RowBox[{
            SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
            SubscriptBox["\[CapitalDelta]y", "1"], "+", "\[CenterEllipsis]", 
            "+", 
            SubscriptBox["\[CapitalDelta]y", 
             RowBox[{"n", "-", "2"}]]}], ")"}]}]}], ")"}]}], "  ", "-", " ", 
      SubscriptBox["\[CapitalDelta]y", 
       RowBox[{"n", "-", "2"}]]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"ExplicitMidpointIncrements",
 CellID->349248116]
}, Open  ]],

Cell[CellGroupData[{

Cell["Gragg's smoothing step", "Subsection",
 CellID->1940172755],

Cell[TextData[{
 "The ",
 StyleBox["smoothing step",
  FontSlant->"Italic"],
 " of Gragg has its historical origins in the weak stability of the explicit \
midpoint rule:"
}], "Text",
 CellID->1242585920],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["Sy", "h"], "(", "n", ")"}], " ", "=", " ", 
   RowBox[{
    RowBox[{"1", "/", "4"}], " ", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["y", 
       RowBox[{"n", "-", "1"}]], "+", " ", 
      RowBox[{"2", " ", 
       SubscriptBox["y", "n"]}], "+", " ", 
      SubscriptBox["y", 
       RowBox[{"n", "+", "1"}]]}], ")"}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"GraggSmoothingStep",
 CellID->367927996],

Cell[TextData[{
 "In order to make use of (",
 
 CounterBox["NumberedEquation", "GraggSmoothingStep"],
 "), the formulation (",
 
 CounterBox["NumberedEquation", "ExplicitMidpointIncrements"],
 ") is computed with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"2", " ", "m"}], TraditionalForm]]],
 " steps. This has the advantage of increasing the stability domain and \
evaluating the function at the end of the basic step [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "]."
}], "Text",
 CellID->485534844],

Cell["\<\
Notice that because of the construction, a sum of increments is available at \
the end of the algorithm together with two consecutive increments. This leads \
to the following formulation:\
\>", "Text",
 CellID->2145603494],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["S\[CapitalDelta]y", "h"], "(", "n", ")"}], "  ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Sy", "h"], "(", "n", ")"}], " ", "-", 
     SubscriptBox["y", "0"]}], "=", " ", 
    RowBox[{
     SubscriptBox["\[CapitalSigma]\[CapitalDelta]y", "n"], " ", "+", " ", 
     RowBox[{
      RowBox[{"1", "/", "4"}], "  ", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["\[CapitalDelta]y", "n"], "-", " ", 
         SubscriptBox["\[CapitalDelta]y", 
          RowBox[{"n", "-", "1"}]]}], ")"}], "."}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ModifiedGraggSmoothingStep",
 CellID->2120609443],

Cell[TextData[{
 "Moreover (",
 
 CounterBox["NumberedEquation", "ModifiedGraggSmoothingStep"],
 ") has an advantage over (",
 
 CounterBox["NumberedEquation", "GraggSmoothingStep"],
 ") in finite-precision arithmetic because the values ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "i"], TraditionalForm]]],
 ", which typically have a larger magnitude than the increments ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]y", "i"], TraditionalForm]]],
 ", do not contribute to the computation."
}], "Text",
 CellID->1783016797],

Cell[TextData[{
 "Gragg's smoothing step is not of great importance if the method is followed \
by extrapolation, and Shampine proposes an alternative smoothing procedure \
that is slightly more efficient [",
 ButtonBox["SB83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#7456"],
 "]."
}], "Text",
 CellID->1299939945],

Cell[TextData[{
 "The method ",
 StyleBox["ExplicitMidpoint", "MR"],
 " uses ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     RowBox[{"2", " ", "m"}], " ", "-", " ", "1"}]}], TraditionalForm]]],
 " steps and ",
 StyleBox["ExplicitModifiedMidpoint", "MR"],
 " uses ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{"2", " ", "m"}]}], TraditionalForm]]],
 " steps followed by the smoothing step (",
 
 CounterBox["NumberedEquation", "ModifiedGraggSmoothingStep"],
 ")."
}], "Text",
 CellID->1215437356]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stability regions", "Subsection",
 CellID->679707293],

Cell[TextData[{
 "This illustrates the effect of the smoothing step on the linear stability \
domain (carried out using the package ",
 StyleBox["FunctionApproximations.m", "MR"],
 ")."
}], "Text",
 CellID->688293963],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .47619 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.47619 0.0113379 0.47619 [
[.5 .48869 -117.688 0 ]
[.5 .48869 117.688 41.625 ]
[ 0 0 0 0 ]
[ 1 .47619 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 .48869 -178.688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 49.625 translate 1 -1 scale
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Linear) show
99.375 13.688 moveto
(stability) show
142.438 13.688 moveto
(regions) show
184.812 13.688 moveto
(for) show
202.000 13.688 moveto
(T) show
209.312 15.562 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 8.500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(i) show
211.188 15.562 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 8.500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(,) show
213.500 15.562 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 8.500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(i) show
216.438 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(,) show
223.062 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(i) show
229.000 13.688 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(=) show
240.000 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1) show
246.625 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(,) show
253.250 13.688 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\274) show
265.250 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(,) show
271.875 13.688 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(5) show
63.000 28.250 moveto
(for) show
80.188 28.250 moveto
(the) show
100.062 28.250 moveto
(explicit) show
143.125 28.250 moveto
(midpoint) show
194.750 28.250 moveto
(rule) show
%%IncludeResource: font Math2
%%IncludeFont: Math2
/Math2 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
217.875 28.250 moveto
(H) show
221.688 28.250 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(left) show
%%IncludeResource: font Math2
%%IncludeFont: Math2
/Math2 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
237.562 28.250 moveto
(L) show
244.688 28.250 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(and) show
267.875 28.250 moveto
(the) show
63.000 42.000 moveto
(explicit) show
106.062 42.000 moveto
(midpoint) show
157.688 42.000 moveto
(rule) show
180.812 42.000 moveto
(with) show
205.312 42.000 moveto
(smoothing) show
%%IncludeResource: font Math2
%%IncludeFont: Math2
/Math2 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
263.625 42.000 moveto
(H) show
267.438 42.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(right) show
%%IncludeResource: font Math2
%%IncludeFont: Math2
/Math2 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
290.562 42.000 moveto
(L) show
294.375 42.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .47619 L
0 .47619 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.0113379 0.477324 0.464853 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0 1 0 1 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of sub-graphic
p
0 0 1 1 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0714286 0.5 0.0714286 [
[.07143 -0.0125 -6 -9 ]
[.07143 -0.0125 6 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.35714 -0.0125 -6 -9 ]
[.35714 -0.0125 6 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.64286 -0.0125 -3 -9 ]
[.64286 -0.0125 3 0 ]
[.78571 -0.0125 -3 -9 ]
[.78571 -0.0125 3 0 ]
[.92857 -0.0125 -3 -9 ]
[.92857 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .07143 -12 -4.5 ]
[-0.0125 .07143 0 4.5 ]
[-0.0125 .21429 -12 -4.5 ]
[-0.0125 .21429 0 4.5 ]
[-0.0125 .35714 -12 -4.5 ]
[-0.0125 .35714 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .64286 -6 -4.5 ]
[-0.0125 .64286 0 4.5 ]
[-0.0125 .78571 -6 -4.5 ]
[-0.0125 .78571 0 4.5 ]
[-0.0125 .92857 -6 -4.5 ]
[-0.0125 .92857 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.07143 0 m
.07143 .00625 L
s
[(-6)] .07143 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(-4)] .21429 -0.0125 0 1 Mshowa
.35714 0 m
.35714 .00625 L
s
[(-2)] .35714 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0)] .5 -0.0125 0 1 Mshowa
.64286 0 m
.64286 .00625 L
s
[(2)] .64286 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(4)] .78571 -0.0125 0 1 Mshowa
.92857 0 m
.92857 .00625 L
s
[(6)] .92857 -0.0125 0 1 Mshowa
.125 Mabswid
.10714 0 m
.10714 .00375 L
s
.14286 0 m
.14286 .00375 L
s
.17857 0 m
.17857 .00375 L
s
.25 0 m
.25 .00375 L
s
.28571 0 m
.28571 .00375 L
s
.32143 0 m
.32143 .00375 L
s
.39286 0 m
.39286 .00375 L
s
.42857 0 m
.42857 .00375 L
s
.46429 0 m
.46429 .00375 L
s
.53571 0 m
.53571 .00375 L
s
.57143 0 m
.57143 .00375 L
s
.60714 0 m
.60714 .00375 L
s
.67857 0 m
.67857 .00375 L
s
.71429 0 m
.71429 .00375 L
s
.75 0 m
.75 .00375 L
s
.82143 0 m
.82143 .00375 L
s
.85714 0 m
.85714 .00375 L
s
.89286 0 m
.89286 .00375 L
s
.03571 0 m
.03571 .00375 L
s
.96429 0 m
.96429 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .07143 m
.00625 .07143 L
s
[(-6)] -0.0125 .07143 1 0 Mshowa
0 .21429 m
.00625 .21429 L
s
[(-4)] -0.0125 .21429 1 0 Mshowa
0 .35714 m
.00625 .35714 L
s
[(-2)] -0.0125 .35714 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(0)] -0.0125 .5 1 0 Mshowa
0 .64286 m
.00625 .64286 L
s
[(2)] -0.0125 .64286 1 0 Mshowa
0 .78571 m
.00625 .78571 L
s
[(4)] -0.0125 .78571 1 0 Mshowa
0 .92857 m
.00625 .92857 L
s
[(6)] -0.0125 .92857 1 0 Mshowa
.125 Mabswid
0 .10714 m
.00375 .10714 L
s
0 .14286 m
.00375 .14286 L
s
0 .17857 m
.00375 .17857 L
s
0 .25 m
.00375 .25 L
s
0 .28571 m
.00375 .28571 L
s
0 .32143 m
.00375 .32143 L
s
0 .39286 m
.00375 .39286 L
s
0 .42857 m
.00375 .42857 L
s
0 .46429 m
.00375 .46429 L
s
0 .53571 m
.00375 .53571 L
s
0 .57143 m
.00375 .57143 L
s
0 .60714 m
.00375 .60714 L
s
0 .67857 m
.00375 .67857 L
s
0 .71429 m
.00375 .71429 L
s
0 .75 m
.00375 .75 L
s
0 .82143 m
.00375 .82143 L
s
0 .85714 m
.00375 .85714 L
s
0 .89286 m
.00375 .89286 L
s
0 .03571 m
.00375 .03571 L
s
0 .96429 m
.00375 .96429 L
s
.25 Mabswid
0 0 m
0 1 L
s
.07143 .99375 m
.07143 1 L
s
.21429 .99375 m
.21429 1 L
s
.35714 .99375 m
.35714 1 L
s
.5 .99375 m
.5 1 L
s
.64286 .99375 m
.64286 1 L
s
.78571 .99375 m
.78571 1 L
s
.92857 .99375 m
.92857 1 L
s
.125 Mabswid
.10714 .99625 m
.10714 1 L
s
.14286 .99625 m
.14286 1 L
s
.17857 .99625 m
.17857 1 L
s
.25 .99625 m
.25 1 L
s
.28571 .99625 m
.28571 1 L
s
.32143 .99625 m
.32143 1 L
s
.39286 .99625 m
.39286 1 L
s
.42857 .99625 m
.42857 1 L
s
.46429 .99625 m
.46429 1 L
s
.53571 .99625 m
.53571 1 L
s
.57143 .99625 m
.57143 1 L
s
.60714 .99625 m
.60714 1 L
s
.67857 .99625 m
.67857 1 L
s
.71429 .99625 m
.71429 1 L
s
.75 .99625 m
.75 1 L
s
.82143 .99625 m
.82143 1 L
s
.85714 .99625 m
.85714 1 L
s
.89286 .99625 m
.89286 1 L
s
.03571 .99625 m
.03571 1 L
s
.96429 .99625 m
.96429 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .07143 m
1 .07143 L
s
.99375 .21429 m
1 .21429 L
s
.99375 .35714 m
1 .35714 L
s
.99375 .5 m
1 .5 L
s
.99375 .64286 m
1 .64286 L
s
.99375 .78571 m
1 .78571 L
s
.99375 .92857 m
1 .92857 L
s
.125 Mabswid
.99625 .10714 m
1 .10714 L
s
.99625 .14286 m
1 .14286 L
s
.99625 .17857 m
1 .17857 L
s
.99625 .25 m
1 .25 L
s
.99625 .28571 m
1 .28571 L
s
.99625 .32143 m
1 .32143 L
s
.99625 .39286 m
1 .39286 L
s
.99625 .42857 m
1 .42857 L
s
.99625 .46429 m
1 .46429 L
s
.99625 .53571 m
1 .53571 L
s
.99625 .57143 m
1 .57143 L
s
.99625 .60714 m
1 .60714 L
s
.99625 .67857 m
1 .67857 L
s
.99625 .71429 m
1 .71429 L
s
.99625 .75 m
1 .75 L
s
.99625 .82143 m
1 .82143 L
s
.99625 .85714 m
1 .85714 L
s
.99625 .89286 m
1 .89286 L
s
.99625 .03571 m
1 .03571 L
s
.99625 .96429 m
1 .96429 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 .5 m
1 .5 L
s
.5 0 m
.5 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
.5 Mabswid
.40816 .37959 m
.42857 .37625 L
.44898 .37959 L
.46683 .38776 L
.46939 .3906 L
.48418 .40816 L
.4898 .41623 L
.49358 .42857 L
.49813 .44898 L
.49983 .46939 L
.50017 .4898 L
.50017 .5102 L
.49983 .53061 L
.49813 .55102 L
.49358 .57143 L
.4898 .58377 L
.48418 .59184 L
.46939 .6094 L
.46683 .61224 L
.44898 .62041 L
.42857 .62375 L
.40816 .62041 L
.39031 .61224 L
.38776 .6094 L
.37296 .59184 L
.36735 .58377 L
.36356 .57143 L
.35901 .55102 L
.35731 .53061 L
.35697 .5102 L
.35697 .4898 L
.35731 .46939 L
.35901 .44898 L
.36356 .42857 L
.36735 .41623 L
.37296 .40816 L
.38776 .3906 L
.39031 .38776 L
.40816 .37959 L
s
.44898 .29717 m
.46939 .29066 L
.4898 .28836 L
.50815 .30612 L
.5102 .3095 L
.51639 .32653 L
.51653 .34694 L
.51123 .36735 L
.5102 .37015 L
.50589 .38776 L
.50204 .40816 L
.50051 .42857 L
.50011 .44898 L
.50006 .46939 L
.50007 .4898 L
.50007 .5102 L
.50006 .53061 L
.50011 .55102 L
.50051 .57143 L
.50204 .59184 L
.50589 .61224 L
.5102 .62985 L
.51123 .63265 L
.51653 .65306 L
.51639 .67347 L
.5102 .6905 L
.50815 .69388 L
.4898 .71164 L
.46939 .70934 L
.44898 .70283 L
.43987 .69388 L
.42857 .68353 L
.41948 .67347 L
.40816 .66354 L
.39238 .65306 L
.38776 .65044 L
.36735 .63897 L
.3567 .63265 L
.34694 .62524 L
.33251 .61224 L
.32653 .60495 L
.31808 .59184 L
.3093 .57143 L
.30612 .56176 L
.30435 .55102 L
.30189 .53061 L
.30098 .5102 L
.30098 .4898 L
.30189 .46939 L
.30435 .44898 L
Mistroke
.30612 .43824 L
.3093 .42857 L
.31808 .40816 L
.32653 .39505 L
.33251 .38776 L
.34694 .37476 L
.3567 .36735 L
.36735 .36103 L
.38776 .34956 L
.39238 .34694 L
.40816 .33646 L
.41948 .32653 L
.42857 .31647 L
.43987 .30612 L
.44898 .29717 L
Mfstroke
.55102 .73954 m
.57143 .75441 L
.57163 .7551 L
.57143 .76496 L
.56448 .77551 L
.55102 .779 L
.54285 .77551 L
.53412 .7551 L
.55102 .73954 L
s
.36735 .28201 m
.38776 .27441 L
.40816 .27213 L
.42857 .2741 L
.44898 .2804 L
.45988 .28571 L
.46939 .2911 L
.48402 .30612 L
.4898 .31627 L
.49315 .32653 L
.49714 .34694 L
.49897 .36735 L
.49973 .38776 L
.49999 .40816 L
.50006 .42857 L
.50007 .44898 L
.50007 .46939 L
.50007 .4898 L
.50007 .5102 L
.50007 .53061 L
.50007 .55102 L
.50006 .57143 L
.49999 .59184 L
.49973 .61224 L
.49897 .63265 L
.49714 .65306 L
.49315 .67347 L
.4898 .68373 L
.48402 .69388 L
.46939 .7089 L
.45988 .71429 L
.44898 .7196 L
.42857 .7259 L
.40816 .72787 L
.38776 .72559 L
.36735 .71799 L
.36055 .71429 L
.34694 .7046 L
.33626 .69388 L
.32653 .6834 L
.31822 .67347 L
.30612 .65876 L
.30141 .65306 L
.28571 .63277 L
.28563 .63265 L
.27167 .61224 L
.26531 .60006 L
.26153 .59184 L
.25419 .57143 L
.24966 .55102 L
Mistroke
.2472 .53061 L
.24618 .5102 L
.24618 .4898 L
.2472 .46939 L
.24966 .44898 L
.25419 .42857 L
.26153 .40816 L
.26531 .39994 L
.27167 .38776 L
.28563 .36735 L
.28571 .36723 L
.30141 .34694 L
.30612 .34124 L
.31822 .32653 L
.32653 .3166 L
.33626 .30612 L
.34694 .2954 L
.36055 .28571 L
.36735 .28201 L
Mfstroke
.55102 .221 m
.56448 .22449 L
.57143 .23504 L
.57163 .2449 L
.57143 .24559 L
.55102 .26046 L
.53412 .2449 L
.54285 .22449 L
.55102 .221 L
s
.42857 .19544 m
.44898 .19488 L
.46939 .20258 L
.47165 .20408 L
.4898 .22441 L
.48983 .22449 L
.49778 .2449 L
.50093 .26531 L
.50149 .28571 L
.50106 .30612 L
.50055 .32653 L
.50025 .34694 L
.50012 .36735 L
.50008 .38776 L
.50007 .40816 L
.50007 .42857 L
.50007 .44898 L
.50007 .46939 L
.50007 .4898 L
.50007 .5102 L
.50007 .53061 L
.50007 .55102 L
.50007 .57143 L
.50007 .59184 L
.50008 .61224 L
.50012 .63265 L
.50025 .65306 L
.50055 .67347 L
.50106 .69388 L
.50149 .71429 L
.50093 .73469 L
.49778 .7551 L
.48983 .77551 L
.4898 .77559 L
.47165 .79592 L
.46939 .79742 L
.44898 .80512 L
.42857 .80456 L
.4092 .79592 L
.40816 .79532 L
.38776 .78253 L
.37826 .77551 L
.36735 .76893 L
.34694 .758 L
.34117 .7551 L
.32653 .7479 L
.30612 .73598 L
.30413 .73469 L
.28571 .72066 L
.27895 .71429 L
Mistroke
.26531 .69922 L
.26112 .69388 L
.24674 .67347 L
.2449 .67055 L
.23394 .65306 L
.22449 .6356 L
.22289 .63265 L
.2128 .61224 L
.20523 .59184 L
.20408 .58826 L
.19927 .57143 L
.19513 .55102 L
.19267 .53061 L
.19155 .5102 L
.19155 .4898 L
.19267 .46939 L
.19513 .44898 L
.19927 .42857 L
.20408 .41174 L
.20523 .40816 L
.2128 .38776 L
.22289 .36735 L
.22449 .3644 L
.23394 .34694 L
.2449 .32945 L
.24674 .32653 L
.26112 .30612 L
.26531 .30078 L
.27895 .28571 L
.28571 .27934 L
.30413 .26531 L
.30612 .26402 L
.32653 .2521 L
.34117 .2449 L
.34694 .242 L
.36735 .23107 L
.37826 .22449 L
.38776 .21747 L
.40816 .20468 L
.4092 .20408 L
.42857 .19544 L
Mfstroke
.46939 .14266 m
.4898 .12667 L
.5102 .12511 L
.52836 .14286 L
.52869 .16327 L
.51955 .18367 L
.5102 .1989 L
.5082 .20408 L
.50211 .22449 L
.5003 .2449 L
.49992 .26531 L
.49993 .28571 L
.5 .30612 L
.50004 .32653 L
.50006 .34694 L
.50006 .36735 L
.50007 .38776 L
.50007 .40816 L
.50007 .42857 L
.50007 .44898 L
.50007 .46939 L
.50007 .4898 L
.50007 .5102 L
.50007 .53061 L
.50007 .55102 L
.50007 .57143 L
.50007 .59184 L
.50007 .61224 L
.50006 .63265 L
.50006 .65306 L
.50004 .67347 L
.5 .69388 L
.49993 .71429 L
.49992 .73469 L
.5003 .7551 L
.50211 .77551 L
.5082 .79592 L
.5102 .8011 L
.51955 .81633 L
.52869 .83673 L
.52836 .85714 L
.5102 .87489 L
.4898 .87333 L
.46939 .85734 L
.46921 .85714 L
.44898 .83962 L
.44419 .83673 L
.42857 .8342 L
.40816 .83337 L
.38776 .83285 L
Mistroke
.36735 .8307 L
.34694 .82568 L
.32655 .81633 L
.32653 .81632 L
.30612 .80394 L
.29596 .79592 L
.28571 .78804 L
.27083 .77551 L
.26531 .77095 L
.24715 .7551 L
.2449 .75304 L
.22621 .73469 L
.22449 .73279 L
.20877 .71429 L
.20408 .70776 L
.19461 .69388 L
.18367 .67471 L
.18298 .67347 L
.17232 .65306 L
.16366 .63265 L
.16327 .6316 L
.15558 .61224 L
.14913 .59184 L
.1445 .57143 L
.14286 .56217 L
.14113 .55102 L
.13882 .53061 L
.13772 .5102 L
.13772 .4898 L
.13882 .46939 L
.14113 .44898 L
.14286 .43783 L
.1445 .42857 L
.14913 .40816 L
.15558 .38776 L
.16327 .3684 L
.16366 .36735 L
.17232 .34694 L
.18298 .32653 L
.18367 .32529 L
.19461 .30612 L
.20408 .29224 L
.20877 .28571 L
.22449 .26721 L
.22621 .26531 L
.2449 .24696 L
.24715 .2449 L
.26531 .22905 L
.27083 .22449 L
.28571 .21196 L
Mistroke
.29596 .20408 L
.30612 .19606 L
.32653 .18368 L
.32655 .18367 L
.34694 .17432 L
.36735 .1693 L
.38776 .16715 L
.40816 .16663 L
.42857 .1658 L
.44419 .16327 L
.44898 .16038 L
.46921 .14286 L
.46939 .14266 L
Mfstroke
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0 0 1 1 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0714286 0.5 0.0714286 [
[.07143 -0.0125 -6 -9 ]
[.07143 -0.0125 6 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.35714 -0.0125 -6 -9 ]
[.35714 -0.0125 6 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.64286 -0.0125 -3 -9 ]
[.64286 -0.0125 3 0 ]
[.78571 -0.0125 -3 -9 ]
[.78571 -0.0125 3 0 ]
[.92857 -0.0125 -3 -9 ]
[.92857 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .07143 -12 -4.5 ]
[-0.0125 .07143 0 4.5 ]
[-0.0125 .21429 -12 -4.5 ]
[-0.0125 .21429 0 4.5 ]
[-0.0125 .35714 -12 -4.5 ]
[-0.0125 .35714 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .64286 -6 -4.5 ]
[-0.0125 .64286 0 4.5 ]
[-0.0125 .78571 -6 -4.5 ]
[-0.0125 .78571 0 4.5 ]
[-0.0125 .92857 -6 -4.5 ]
[-0.0125 .92857 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.07143 0 m
.07143 .00625 L
s
[(-6)] .07143 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(-4)] .21429 -0.0125 0 1 Mshowa
.35714 0 m
.35714 .00625 L
s
[(-2)] .35714 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0)] .5 -0.0125 0 1 Mshowa
.64286 0 m
.64286 .00625 L
s
[(2)] .64286 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(4)] .78571 -0.0125 0 1 Mshowa
.92857 0 m
.92857 .00625 L
s
[(6)] .92857 -0.0125 0 1 Mshowa
.125 Mabswid
.10714 0 m
.10714 .00375 L
s
.14286 0 m
.14286 .00375 L
s
.17857 0 m
.17857 .00375 L
s
.25 0 m
.25 .00375 L
s
.28571 0 m
.28571 .00375 L
s
.32143 0 m
.32143 .00375 L
s
.39286 0 m
.39286 .00375 L
s
.42857 0 m
.42857 .00375 L
s
.46429 0 m
.46429 .00375 L
s
.53571 0 m
.53571 .00375 L
s
.57143 0 m
.57143 .00375 L
s
.60714 0 m
.60714 .00375 L
s
.67857 0 m
.67857 .00375 L
s
.71429 0 m
.71429 .00375 L
s
.75 0 m
.75 .00375 L
s
.82143 0 m
.82143 .00375 L
s
.85714 0 m
.85714 .00375 L
s
.89286 0 m
.89286 .00375 L
s
.03571 0 m
.03571 .00375 L
s
.96429 0 m
.96429 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .07143 m
.00625 .07143 L
s
[(-6)] -0.0125 .07143 1 0 Mshowa
0 .21429 m
.00625 .21429 L
s
[(-4)] -0.0125 .21429 1 0 Mshowa
0 .35714 m
.00625 .35714 L
s
[(-2)] -0.0125 .35714 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(0)] -0.0125 .5 1 0 Mshowa
0 .64286 m
.00625 .64286 L
s
[(2)] -0.0125 .64286 1 0 Mshowa
0 .78571 m
.00625 .78571 L
s
[(4)] -0.0125 .78571 1 0 Mshowa
0 .92857 m
.00625 .92857 L
s
[(6)] -0.0125 .92857 1 0 Mshowa
.125 Mabswid
0 .10714 m
.00375 .10714 L
s
0 .14286 m
.00375 .14286 L
s
0 .17857 m
.00375 .17857 L
s
0 .25 m
.00375 .25 L
s
0 .28571 m
.00375 .28571 L
s
0 .32143 m
.00375 .32143 L
s
0 .39286 m
.00375 .39286 L
s
0 .42857 m
.00375 .42857 L
s
0 .46429 m
.00375 .46429 L
s
0 .53571 m
.00375 .53571 L
s
0 .57143 m
.00375 .57143 L
s
0 .60714 m
.00375 .60714 L
s
0 .67857 m
.00375 .67857 L
s
0 .71429 m
.00375 .71429 L
s
0 .75 m
.00375 .75 L
s
0 .82143 m
.00375 .82143 L
s
0 .85714 m
.00375 .85714 L
s
0 .89286 m
.00375 .89286 L
s
0 .03571 m
.00375 .03571 L
s
0 .96429 m
.00375 .96429 L
s
.25 Mabswid
0 0 m
0 1 L
s
.07143 .99375 m
.07143 1 L
s
.21429 .99375 m
.21429 1 L
s
.35714 .99375 m
.35714 1 L
s
.5 .99375 m
.5 1 L
s
.64286 .99375 m
.64286 1 L
s
.78571 .99375 m
.78571 1 L
s
.92857 .99375 m
.92857 1 L
s
.125 Mabswid
.10714 .99625 m
.10714 1 L
s
.14286 .99625 m
.14286 1 L
s
.17857 .99625 m
.17857 1 L
s
.25 .99625 m
.25 1 L
s
.28571 .99625 m
.28571 1 L
s
.32143 .99625 m
.32143 1 L
s
.39286 .99625 m
.39286 1 L
s
.42857 .99625 m
.42857 1 L
s
.46429 .99625 m
.46429 1 L
s
.53571 .99625 m
.53571 1 L
s
.57143 .99625 m
.57143 1 L
s
.60714 .99625 m
.60714 1 L
s
.67857 .99625 m
.67857 1 L
s
.71429 .99625 m
.71429 1 L
s
.75 .99625 m
.75 1 L
s
.82143 .99625 m
.82143 1 L
s
.85714 .99625 m
.85714 1 L
s
.89286 .99625 m
.89286 1 L
s
.03571 .99625 m
.03571 1 L
s
.96429 .99625 m
.96429 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .07143 m
1 .07143 L
s
.99375 .21429 m
1 .21429 L
s
.99375 .35714 m
1 .35714 L
s
.99375 .5 m
1 .5 L
s
.99375 .64286 m
1 .64286 L
s
.99375 .78571 m
1 .78571 L
s
.99375 .92857 m
1 .92857 L
s
.125 Mabswid
.99625 .10714 m
1 .10714 L
s
.99625 .14286 m
1 .14286 L
s
.99625 .17857 m
1 .17857 L
s
.99625 .25 m
1 .25 L
s
.99625 .28571 m
1 .28571 L
s
.99625 .32143 m
1 .32143 L
s
.99625 .39286 m
1 .39286 L
s
.99625 .42857 m
1 .42857 L
s
.99625 .46429 m
1 .46429 L
s
.99625 .53571 m
1 .53571 L
s
.99625 .57143 m
1 .57143 L
s
.99625 .60714 m
1 .60714 L
s
.99625 .67857 m
1 .67857 L
s
.99625 .71429 m
1 .71429 L
s
.99625 .75 m
1 .75 L
s
.99625 .82143 m
1 .82143 L
s
.99625 .85714 m
1 .85714 L
s
.99625 .89286 m
1 .89286 L
s
.99625 .03571 m
1 .03571 L
s
.99625 .96429 m
1 .96429 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 .5 m
1 .5 L
s
.5 0 m
.5 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
MathSubEnd
P
% End of sub-graphic
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.522676 0.0113379 0.97619 0.464853 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0 1 0 1 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of sub-graphic
p
0 0 1 1 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0714286 0.5 0.0714286 [
[.07143 -0.0125 -6 -9 ]
[.07143 -0.0125 6 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.35714 -0.0125 -6 -9 ]
[.35714 -0.0125 6 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.64286 -0.0125 -3 -9 ]
[.64286 -0.0125 3 0 ]
[.78571 -0.0125 -3 -9 ]
[.78571 -0.0125 3 0 ]
[.92857 -0.0125 -3 -9 ]
[.92857 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .07143 -12 -4.5 ]
[-0.0125 .07143 0 4.5 ]
[-0.0125 .21429 -12 -4.5 ]
[-0.0125 .21429 0 4.5 ]
[-0.0125 .35714 -12 -4.5 ]
[-0.0125 .35714 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .64286 -6 -4.5 ]
[-0.0125 .64286 0 4.5 ]
[-0.0125 .78571 -6 -4.5 ]
[-0.0125 .78571 0 4.5 ]
[-0.0125 .92857 -6 -4.5 ]
[-0.0125 .92857 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.07143 0 m
.07143 .00625 L
s
[(-6)] .07143 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(-4)] .21429 -0.0125 0 1 Mshowa
.35714 0 m
.35714 .00625 L
s
[(-2)] .35714 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0)] .5 -0.0125 0 1 Mshowa
.64286 0 m
.64286 .00625 L
s
[(2)] .64286 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(4)] .78571 -0.0125 0 1 Mshowa
.92857 0 m
.92857 .00625 L
s
[(6)] .92857 -0.0125 0 1 Mshowa
.125 Mabswid
.10714 0 m
.10714 .00375 L
s
.14286 0 m
.14286 .00375 L
s
.17857 0 m
.17857 .00375 L
s
.25 0 m
.25 .00375 L
s
.28571 0 m
.28571 .00375 L
s
.32143 0 m
.32143 .00375 L
s
.39286 0 m
.39286 .00375 L
s
.42857 0 m
.42857 .00375 L
s
.46429 0 m
.46429 .00375 L
s
.53571 0 m
.53571 .00375 L
s
.57143 0 m
.57143 .00375 L
s
.60714 0 m
.60714 .00375 L
s
.67857 0 m
.67857 .00375 L
s
.71429 0 m
.71429 .00375 L
s
.75 0 m
.75 .00375 L
s
.82143 0 m
.82143 .00375 L
s
.85714 0 m
.85714 .00375 L
s
.89286 0 m
.89286 .00375 L
s
.03571 0 m
.03571 .00375 L
s
.96429 0 m
.96429 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .07143 m
.00625 .07143 L
s
[(-6)] -0.0125 .07143 1 0 Mshowa
0 .21429 m
.00625 .21429 L
s
[(-4)] -0.0125 .21429 1 0 Mshowa
0 .35714 m
.00625 .35714 L
s
[(-2)] -0.0125 .35714 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(0)] -0.0125 .5 1 0 Mshowa
0 .64286 m
.00625 .64286 L
s
[(2)] -0.0125 .64286 1 0 Mshowa
0 .78571 m
.00625 .78571 L
s
[(4)] -0.0125 .78571 1 0 Mshowa
0 .92857 m
.00625 .92857 L
s
[(6)] -0.0125 .92857 1 0 Mshowa
.125 Mabswid
0 .10714 m
.00375 .10714 L
s
0 .14286 m
.00375 .14286 L
s
0 .17857 m
.00375 .17857 L
s
0 .25 m
.00375 .25 L
s
0 .28571 m
.00375 .28571 L
s
0 .32143 m
.00375 .32143 L
s
0 .39286 m
.00375 .39286 L
s
0 .42857 m
.00375 .42857 L
s
0 .46429 m
.00375 .46429 L
s
0 .53571 m
.00375 .53571 L
s
0 .57143 m
.00375 .57143 L
s
0 .60714 m
.00375 .60714 L
s
0 .67857 m
.00375 .67857 L
s
0 .71429 m
.00375 .71429 L
s
0 .75 m
.00375 .75 L
s
0 .82143 m
.00375 .82143 L
s
0 .85714 m
.00375 .85714 L
s
0 .89286 m
.00375 .89286 L
s
0 .03571 m
.00375 .03571 L
s
0 .96429 m
.00375 .96429 L
s
.25 Mabswid
0 0 m
0 1 L
s
.07143 .99375 m
.07143 1 L
s
.21429 .99375 m
.21429 1 L
s
.35714 .99375 m
.35714 1 L
s
.5 .99375 m
.5 1 L
s
.64286 .99375 m
.64286 1 L
s
.78571 .99375 m
.78571 1 L
s
.92857 .99375 m
.92857 1 L
s
.125 Mabswid
.10714 .99625 m
.10714 1 L
s
.14286 .99625 m
.14286 1 L
s
.17857 .99625 m
.17857 1 L
s
.25 .99625 m
.25 1 L
s
.28571 .99625 m
.28571 1 L
s
.32143 .99625 m
.32143 1 L
s
.39286 .99625 m
.39286 1 L
s
.42857 .99625 m
.42857 1 L
s
.46429 .99625 m
.46429 1 L
s
.53571 .99625 m
.53571 1 L
s
.57143 .99625 m
.57143 1 L
s
.60714 .99625 m
.60714 1 L
s
.67857 .99625 m
.67857 1 L
s
.71429 .99625 m
.71429 1 L
s
.75 .99625 m
.75 1 L
s
.82143 .99625 m
.82143 1 L
s
.85714 .99625 m
.85714 1 L
s
.89286 .99625 m
.89286 1 L
s
.03571 .99625 m
.03571 1 L
s
.96429 .99625 m
.96429 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .07143 m
1 .07143 L
s
.99375 .21429 m
1 .21429 L
s
.99375 .35714 m
1 .35714 L
s
.99375 .5 m
1 .5 L
s
.99375 .64286 m
1 .64286 L
s
.99375 .78571 m
1 .78571 L
s
.99375 .92857 m
1 .92857 L
s
.125 Mabswid
.99625 .10714 m
1 .10714 L
s
.99625 .14286 m
1 .14286 L
s
.99625 .17857 m
1 .17857 L
s
.99625 .25 m
1 .25 L
s
.99625 .28571 m
1 .28571 L
s
.99625 .32143 m
1 .32143 L
s
.99625 .39286 m
1 .39286 L
s
.99625 .42857 m
1 .42857 L
s
.99625 .46429 m
1 .46429 L
s
.99625 .53571 m
1 .53571 L
s
.99625 .57143 m
1 .57143 L
s
.99625 .60714 m
1 .60714 L
s
.99625 .67857 m
1 .67857 L
s
.99625 .71429 m
1 .71429 L
s
.99625 .75 m
1 .75 L
s
.99625 .82143 m
1 .82143 L
s
.99625 .85714 m
1 .85714 L
s
.99625 .89286 m
1 .89286 L
s
.99625 .03571 m
1 .03571 L
s
.99625 .96429 m
1 .96429 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 .5 m
1 .5 L
s
.5 0 m
.5 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
.5 Mabswid
.40816 .32627 m
.42857 .32473 L
.43968 .32653 L
.44898 .32922 L
.46939 .34206 L
.47404 .34694 L
.48751 .36735 L
.4898 .37266 L
.49464 .38776 L
.4982 .40816 L
.49966 .42857 L
.50006 .44898 L
.5001 .46939 L
.50008 .4898 L
.50008 .5102 L
.5001 .53061 L
.50006 .55102 L
.49966 .57143 L
.4982 .59184 L
.49464 .61224 L
.4898 .62734 L
.48751 .63265 L
.47404 .65306 L
.46939 .65794 L
.44898 .67078 L
.43968 .67347 L
.42857 .67527 L
.40816 .67373 L
.40704 .67347 L
.38776 .66675 L
.36824 .65306 L
.36735 .65237 L
.34754 .63265 L
.34694 .63199 L
.33039 .61224 L
.32653 .60737 L
.31462 .59184 L
.30612 .57945 L
.301 .57143 L
.29037 .55102 L
.28571 .53931 L
.28341 .53061 L
.27986 .5102 L
.27986 .4898 L
.28341 .46939 L
.28571 .46069 L
.29037 .44898 L
.301 .42857 L
.30612 .42055 L
.31462 .40816 L
Mistroke
.32653 .39263 L
.33039 .38776 L
.34694 .36801 L
.34754 .36735 L
.36735 .34763 L
.36824 .34694 L
.38776 .33325 L
.40704 .32653 L
.40816 .32627 L
Mfstroke
.46939 .25725 m
.4898 .25668 L
.50906 .26531 L
.5102 .26627 L
.52254 .28571 L
.52539 .30612 L
.52014 .32653 L
.51142 .34694 L
.5102 .34963 L
.50539 .36735 L
.50201 .38776 L
.50063 .40816 L
.50018 .42857 L
.50007 .44898 L
.50006 .46939 L
.50007 .4898 L
.50007 .5102 L
.50006 .53061 L
.50007 .55102 L
.50018 .57143 L
.50063 .59184 L
.50201 .61224 L
.50539 .63265 L
.5102 .65037 L
.51142 .65306 L
.52014 .67347 L
.52539 .69388 L
.52254 .71429 L
.5102 .73373 L
.50906 .73469 L
.4898 .74332 L
.46939 .74275 L
.4525 .73469 L
.44898 .73166 L
.42857 .71812 L
.42271 .71429 L
.40816 .70717 L
.38776 .70038 L
.36735 .69516 L
.36233 .69388 L
.34694 .68972 L
.32653 .68279 L
.30633 .67347 L
.30612 .67335 L
.28571 .66104 L
.27504 .65306 L
.26531 .64496 L
.25234 .63265 L
.2449 .62495 L
.2335 .61224 L
Mistroke
.22449 .60122 L
.21716 .59184 L
.20408 .57227 L
.20356 .57143 L
.19253 .55102 L
.18552 .53061 L
.18367 .52179 L
.18201 .5102 L
.18201 .4898 L
.18367 .47821 L
.18552 .46939 L
.19253 .44898 L
.20356 .42857 L
.20408 .42773 L
.21716 .40816 L
.22449 .39878 L
.2335 .38776 L
.2449 .37505 L
.25234 .36735 L
.26531 .35504 L
.27504 .34694 L
.28571 .33896 L
.30612 .32665 L
.30633 .32653 L
.32653 .31721 L
.34694 .31028 L
.36233 .30612 L
.36735 .30484 L
.38776 .29962 L
.40816 .29283 L
.42271 .28571 L
.42857 .28188 L
.44898 .26834 L
.4525 .26531 L
.46939 .25725 L
Mfstroke
.57143 .77183 m
.57744 .77551 L
.58445 .79592 L
.57143 .80606 L
.55419 .79592 L
.55746 .77551 L
.57143 .77183 L
s
.36735 .24137 m
.38776 .23797 L
.40816 .23862 L
.42857 .24333 L
.43317 .2449 L
.44898 .25165 L
.46833 .26531 L
.46939 .26613 L
.48499 .28571 L
.4898 .29563 L
.49295 .30612 L
.49685 .32653 L
.49875 .34694 L
.49961 .36735 L
.49994 .38776 L
.50004 .40816 L
.50006 .42857 L
.50007 .44898 L
.50007 .46939 L
.50007 .4898 L
.50007 .5102 L
.50007 .53061 L
.50007 .55102 L
.50006 .57143 L
.50004 .59184 L
.49994 .61224 L
.49961 .63265 L
.49875 .65306 L
.49685 .67347 L
.49295 .69388 L
.4898 .70437 L
.48499 .71429 L
.46939 .73387 L
.46833 .73469 L
.44898 .74835 L
.43317 .7551 L
.42857 .75667 L
.40816 .76138 L
.38776 .76203 L
.36735 .75863 L
.35483 .7551 L
.34694 .7515 L
.32653 .7401 L
.31863 .73469 L
.30612 .72612 L
.29046 .71429 L
.28571 .71078 L
.26531 .69514 L
.26369 .69388 L
.2449 .67884 L
Mistroke
.23855 .67347 L
.22449 .66097 L
.21619 .65306 L
.20408 .64089 L
.19636 .63265 L
.18367 .61876 L
.17785 .61224 L
.16327 .59639 L
.15891 .59184 L
.14286 .57655 L
.13689 .57143 L
.12245 .56013 L
.10963 .55102 L
.10204 .54407 L
.08979 .53061 L
.08163 .51487 L
.08026 .5102 L
.08026 .4898 L
.08163 .48513 L
.08979 .46939 L
.10204 .45593 L
.10963 .44898 L
.12245 .43987 L
.13689 .42857 L
.14286 .42345 L
.15891 .40816 L
.16327 .40361 L
.17785 .38776 L
.18367 .38124 L
.19636 .36735 L
.20408 .35911 L
.21619 .34694 L
.22449 .33903 L
.23855 .32653 L
.2449 .32116 L
.26369 .30612 L
.26531 .30486 L
.28571 .28922 L
.29046 .28571 L
.30612 .27388 L
.31863 .26531 L
.32653 .2599 L
.34694 .2485 L
.35483 .2449 L
.36735 .24137 L
Mfstroke
.57143 .19394 m
.58445 .20408 L
.57744 .22449 L
.57143 .22817 L
.55746 .22449 L
.55419 .20408 L
.57143 .19394 L
s
.40816 .17344 m
.42857 .16644 L
.44898 .16686 L
.46939 .17292 L
.48143 .18367 L
.4898 .19574 L
.49408 .20408 L
.50006 .22449 L
.50209 .2449 L
.50205 .26531 L
.50134 .28571 L
.5007 .30612 L
.50033 .32653 L
.50015 .34694 L
.50009 .36735 L
.50007 .38776 L
.50007 .40816 L
.50007 .42857 L
.50007 .44898 L
.50007 .46939 L
.50007 .4898 L
.50007 .5102 L
.50007 .53061 L
.50007 .55102 L
.50007 .57143 L
.50007 .59184 L
.50007 .61224 L
.50009 .63265 L
.50015 .65306 L
.50033 .67347 L
.5007 .69388 L
.50134 .71429 L
.50205 .73469 L
.50209 .7551 L
.50006 .77551 L
.49408 .79592 L
.4898 .80426 L
.48143 .81633 L
.46939 .82708 L
.44898 .83314 L
.42857 .83356 L
.40816 .82656 L
.3908 .81633 L
.38776 .81493 L
.36735 .80441 L
.34803 .79592 L
.34694 .79551 L
.32653 .788 L
.30612 .77984 L
.29649 .77551 L
Mistroke
.28571 .77019 L
.26531 .75774 L
.26154 .7551 L
.2449 .7424 L
.23624 .73469 L
.22449 .72364 L
.21542 .71429 L
.20408 .70205 L
.19689 .69388 L
.18367 .67803 L
.18002 .67347 L
.16472 .65306 L
.16327 .65098 L
.151 .63265 L
.14286 .619 L
.13908 .61224 L
.12862 .59184 L
.12245 .5781 L
.11974 .57143 L
.11236 .55102 L
.10693 .53061 L
.10397 .5102 L
.10397 .4898 L
.10693 .46939 L
.11236 .44898 L
.11974 .42857 L
.12245 .4219 L
.12862 .40816 L
.13908 .38776 L
.14286 .381 L
.151 .36735 L
.16327 .34902 L
.16472 .34694 L
.18002 .32653 L
.18367 .32197 L
.19689 .30612 L
.20408 .29795 L
.21542 .28571 L
.22449 .27636 L
.23624 .26531 L
.2449 .2576 L
.26154 .2449 L
.26531 .24226 L
.28571 .22981 L
.29649 .22449 L
.30612 .22016 L
.32653 .212 L
.34694 .20449 L
.34803 .20408 L
.36735 .19559 L
Mistroke
.38776 .18507 L
.3908 .18367 L
.40816 .17344 L
Mfstroke
.5102 .09974 m
.51749 .10204 L
.53061 .11318 L
.53463 .12245 L
.53246 .14286 L
.53061 .14708 L
.51947 .16327 L
.5102 .17571 L
.5067 .18367 L
.50144 .20408 L
.50004 .22449 L
.49981 .2449 L
.49987 .26531 L
.49996 .28571 L
.50002 .30612 L
.50005 .32653 L
.50006 .34694 L
.50007 .36735 L
.50007 .38776 L
.50007 .40816 L
.50007 .42857 L
.50007 .44898 L
.50007 .46939 L
.50007 .4898 L
.50007 .5102 L
.50007 .53061 L
.50007 .55102 L
.50007 .57143 L
.50007 .59184 L
.50007 .61224 L
.50007 .63265 L
.50006 .65306 L
.50005 .67347 L
.50002 .69388 L
.49996 .71429 L
.49987 .73469 L
.49981 .7551 L
.50004 .77551 L
.50144 .79592 L
.5067 .81633 L
.5102 .82429 L
.51947 .83673 L
.53061 .85292 L
.53246 .85714 L
.53463 .87755 L
.53061 .88682 L
.51749 .89796 L
.5102 .90026 L
.49695 .89796 L
.4898 .89417 L
Mistroke
.47205 .87755 L
.46939 .87487 L
.44898 .86144 L
.42857 .85879 L
.40816 .85956 L
.38776 .86045 L
.36735 .85978 L
.34969 .85714 L
.34694 .85662 L
.32653 .85095 L
.30612 .84145 L
.29789 .83673 L
.28571 .82982 L
.26553 .81633 L
.26531 .81619 L
.2449 .80236 L
.23597 .79592 L
.22449 .78755 L
.20938 .77551 L
.20408 .77107 L
.18681 .7551 L
.18367 .75197 L
.16774 .73469 L
.16327 .72939 L
.1513 .71429 L
.14286 .70255 L
.13687 .69388 L
.124 .67347 L
.12245 .67079 L
.11235 .65306 L
.10239 .63265 L
.10204 .63188 L
.09353 .61224 L
.0863 .59184 L
.08163 .57541 L
.08062 .57143 L
.07619 .55102 L
.07324 .53061 L
.07178 .5102 L
.07178 .4898 L
.07324 .46939 L
.07619 .44898 L
.08062 .42857 L
.08163 .42459 L
.0863 .40816 L
.09353 .38776 L
.10204 .36812 L
.10239 .36735 L
.11235 .34694 L
.12245 .32921 L
Mistroke
.124 .32653 L
.13687 .30612 L
.14286 .29745 L
.1513 .28571 L
.16327 .27061 L
.16774 .26531 L
.18367 .24803 L
.18681 .2449 L
.20408 .22893 L
.20938 .22449 L
.22449 .21245 L
.23597 .20408 L
.2449 .19764 L
.26531 .18381 L
.26553 .18367 L
.28571 .17018 L
.29789 .16327 L
.30612 .15855 L
.32653 .14905 L
.34694 .14338 L
.34969 .14286 L
.36735 .14022 L
.38776 .13955 L
.40816 .14044 L
.42857 .14121 L
.44898 .13856 L
.46939 .12513 L
.47205 .12245 L
.4898 .10583 L
.49695 .10204 L
.5102 .09974 L
Mfstroke
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0 0 1 1 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0714286 0.5 0.0714286 [
[.07143 -0.0125 -6 -9 ]
[.07143 -0.0125 6 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.35714 -0.0125 -6 -9 ]
[.35714 -0.0125 6 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.64286 -0.0125 -3 -9 ]
[.64286 -0.0125 3 0 ]
[.78571 -0.0125 -3 -9 ]
[.78571 -0.0125 3 0 ]
[.92857 -0.0125 -3 -9 ]
[.92857 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .07143 -12 -4.5 ]
[-0.0125 .07143 0 4.5 ]
[-0.0125 .21429 -12 -4.5 ]
[-0.0125 .21429 0 4.5 ]
[-0.0125 .35714 -12 -4.5 ]
[-0.0125 .35714 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .64286 -6 -4.5 ]
[-0.0125 .64286 0 4.5 ]
[-0.0125 .78571 -6 -4.5 ]
[-0.0125 .78571 0 4.5 ]
[-0.0125 .92857 -6 -4.5 ]
[-0.0125 .92857 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.07143 0 m
.07143 .00625 L
s
[(-6)] .07143 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(-4)] .21429 -0.0125 0 1 Mshowa
.35714 0 m
.35714 .00625 L
s
[(-2)] .35714 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0)] .5 -0.0125 0 1 Mshowa
.64286 0 m
.64286 .00625 L
s
[(2)] .64286 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(4)] .78571 -0.0125 0 1 Mshowa
.92857 0 m
.92857 .00625 L
s
[(6)] .92857 -0.0125 0 1 Mshowa
.125 Mabswid
.10714 0 m
.10714 .00375 L
s
.14286 0 m
.14286 .00375 L
s
.17857 0 m
.17857 .00375 L
s
.25 0 m
.25 .00375 L
s
.28571 0 m
.28571 .00375 L
s
.32143 0 m
.32143 .00375 L
s
.39286 0 m
.39286 .00375 L
s
.42857 0 m
.42857 .00375 L
s
.46429 0 m
.46429 .00375 L
s
.53571 0 m
.53571 .00375 L
s
.57143 0 m
.57143 .00375 L
s
.60714 0 m
.60714 .00375 L
s
.67857 0 m
.67857 .00375 L
s
.71429 0 m
.71429 .00375 L
s
.75 0 m
.75 .00375 L
s
.82143 0 m
.82143 .00375 L
s
.85714 0 m
.85714 .00375 L
s
.89286 0 m
.89286 .00375 L
s
.03571 0 m
.03571 .00375 L
s
.96429 0 m
.96429 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .07143 m
.00625 .07143 L
s
[(-6)] -0.0125 .07143 1 0 Mshowa
0 .21429 m
.00625 .21429 L
s
[(-4)] -0.0125 .21429 1 0 Mshowa
0 .35714 m
.00625 .35714 L
s
[(-2)] -0.0125 .35714 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(0)] -0.0125 .5 1 0 Mshowa
0 .64286 m
.00625 .64286 L
s
[(2)] -0.0125 .64286 1 0 Mshowa
0 .78571 m
.00625 .78571 L
s
[(4)] -0.0125 .78571 1 0 Mshowa
0 .92857 m
.00625 .92857 L
s
[(6)] -0.0125 .92857 1 0 Mshowa
.125 Mabswid
0 .10714 m
.00375 .10714 L
s
0 .14286 m
.00375 .14286 L
s
0 .17857 m
.00375 .17857 L
s
0 .25 m
.00375 .25 L
s
0 .28571 m
.00375 .28571 L
s
0 .32143 m
.00375 .32143 L
s
0 .39286 m
.00375 .39286 L
s
0 .42857 m
.00375 .42857 L
s
0 .46429 m
.00375 .46429 L
s
0 .53571 m
.00375 .53571 L
s
0 .57143 m
.00375 .57143 L
s
0 .60714 m
.00375 .60714 L
s
0 .67857 m
.00375 .67857 L
s
0 .71429 m
.00375 .71429 L
s
0 .75 m
.00375 .75 L
s
0 .82143 m
.00375 .82143 L
s
0 .85714 m
.00375 .85714 L
s
0 .89286 m
.00375 .89286 L
s
0 .03571 m
.00375 .03571 L
s
0 .96429 m
.00375 .96429 L
s
.25 Mabswid
0 0 m
0 1 L
s
.07143 .99375 m
.07143 1 L
s
.21429 .99375 m
.21429 1 L
s
.35714 .99375 m
.35714 1 L
s
.5 .99375 m
.5 1 L
s
.64286 .99375 m
.64286 1 L
s
.78571 .99375 m
.78571 1 L
s
.92857 .99375 m
.92857 1 L
s
.125 Mabswid
.10714 .99625 m
.10714 1 L
s
.14286 .99625 m
.14286 1 L
s
.17857 .99625 m
.17857 1 L
s
.25 .99625 m
.25 1 L
s
.28571 .99625 m
.28571 1 L
s
.32143 .99625 m
.32143 1 L
s
.39286 .99625 m
.39286 1 L
s
.42857 .99625 m
.42857 1 L
s
.46429 .99625 m
.46429 1 L
s
.53571 .99625 m
.53571 1 L
s
.57143 .99625 m
.57143 1 L
s
.60714 .99625 m
.60714 1 L
s
.67857 .99625 m
.67857 1 L
s
.71429 .99625 m
.71429 1 L
s
.75 .99625 m
.75 1 L
s
.82143 .99625 m
.82143 1 L
s
.85714 .99625 m
.85714 1 L
s
.89286 .99625 m
.89286 1 L
s
.03571 .99625 m
.03571 1 L
s
.96429 .99625 m
.96429 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .07143 m
1 .07143 L
s
.99375 .21429 m
1 .21429 L
s
.99375 .35714 m
1 .35714 L
s
.99375 .5 m
1 .5 L
s
.99375 .64286 m
1 .64286 L
s
.99375 .78571 m
1 .78571 L
s
.99375 .92857 m
1 .92857 L
s
.125 Mabswid
.99625 .10714 m
1 .10714 L
s
.99625 .14286 m
1 .14286 L
s
.99625 .17857 m
1 .17857 L
s
.99625 .25 m
1 .25 L
s
.99625 .28571 m
1 .28571 L
s
.99625 .32143 m
1 .32143 L
s
.99625 .39286 m
1 .39286 L
s
.99625 .42857 m
1 .42857 L
s
.99625 .46429 m
1 .46429 L
s
.99625 .53571 m
1 .53571 L
s
.99625 .57143 m
1 .57143 L
s
.99625 .60714 m
1 .60714 L
s
.99625 .67857 m
1 .67857 L
s
.99625 .71429 m
1 .71429 L
s
.99625 .75 m
1 .75 L
s
.99625 .82143 m
1 .82143 L
s
.99625 .85714 m
1 .85714 L
s
.99625 .89286 m
1 .89286 L
s
.99625 .03571 m
1 .03571 L
s
.99625 .96429 m
1 .96429 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 .5 m
1 .5 L
s
.5 0 m
.5 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
MathSubEnd
P
% End of sub-graphic
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{400, 190.25},
 ImageMargins->{{2.25, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"From In[88]:=",
 CellID->623278108]
}, Open  ]],

Cell[CellGroupData[{

Cell["Multiple linearly implicit midpoint steps", "Subsection",
 CellID->1288930149],

Cell[TextData[{
 "Consider one step of the linearly implicit Euler method followed by \
multiple linearly implicit midpoint steps, using the formulation of Bader and \
Deuflhard ",
 "[",
 ButtonBox["BD83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25441"],
 "]",
 ":"
}], "Text",
 CellID->137001976],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      SubscriptBox["\[CapitalDelta]y", "0"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     SubscriptBox["y", "1"], "=", 
     RowBox[{
      SubscriptBox["y", "0"], "+", 
      SubscriptBox["\[CapitalDelta]y", "0"]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", "1"], "-", 
        SubscriptBox["\[CapitalDelta]y", "0"]}], ")"}]}], "=", 
     RowBox[{"2", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"h", " ", 
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["t", "1"], ",", 
           SubscriptBox["y", "1"]}], ")"}]}], "-", 
        SubscriptBox["\[CapitalDelta]y", "0"]}], ")"}]}]},
    {
     SubscriptBox["y", "2"], "=", 
     RowBox[{
      SubscriptBox["y", "1"], "+", 
      SubscriptBox["\[CapitalDelta]y", "1"]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", "2"], "-", 
        SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}], "=", 
     RowBox[{"2", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"h", " ", 
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["t", "2"], ",", 
           RowBox[{
            SubscriptBox["y", "1"], "+", " ", 
            SubscriptBox["\[CapitalDelta]y", "1"]}]}], ")"}]}], "-", 
        SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}]},
    {
     SubscriptBox["y", "3"], "=", 
     RowBox[{
      SubscriptBox["y", "2"], "+", 
      SubscriptBox["\[CapitalDelta]y", "2"]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "1"}]], "-", 
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "2"}]]}], ")"}]}], "=", 
     RowBox[{"2", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"h", " ", 
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["t", 
            RowBox[{"n", "-", "1"}]], ",", 
           SubscriptBox["y", 
            RowBox[{"n", "-", "1"}]]}], ")"}]}], "-", 
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "2"}]]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"LinearlyImplicitMidpointSteps",
 CellID->999705685],

Cell[TextData[{
 "If (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitMidpointSteps"],
 ") is computed for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", " ", "m"}], " ", "-", " ", "1"}], TraditionalForm]]],
 " linearly implicit midpoint steps, then the method has a symmetric error \
expansion ",
 "[",
 ButtonBox["BD83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25441"],
 "]",
 "."
}], "Text",
 CellID->950240595]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reformulation", "Subsection",
 CellID->417407823],

Cell[TextData[{
 "Reformulation of (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitMidpointSteps"],
 ") in terms of increments can be accomplished as follows:"
}], "Text",
 CellID->290353853],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      SubscriptBox["\[CapitalDelta]y", "0"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", "1"], "-", 
        SubscriptBox["\[CapitalDelta]y", "0"]}], ")"}]}], "=", 
     RowBox[{"2", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"h", " ", 
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["t", "1"], ",", 
           RowBox[{
            SubscriptBox["y", "0"], "+", 
            SubscriptBox["\[CapitalDelta]y", "0"]}]}], ")"}]}], "-", 
        SubscriptBox["\[CapitalDelta]y", "0"]}], ")"}]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", "2"], "-", 
        SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}], "=", 
     RowBox[{"2", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"h", " ", 
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["t", "2"], ",", 
           RowBox[{
            SubscriptBox["y", "0"], "+", 
            RowBox[{"(", 
             RowBox[{
              SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
              SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}]}], ")"}]}], 
        "-", 
        SubscriptBox["\[CapitalDelta]y", "1"]}], ")"}]}]},
    {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", "-", 
        RowBox[{"h", " ", "J"}]}], ")"}], 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "1"}]], "-", 
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "2"}]]}], ")"}]}], "=", 
     RowBox[{"2", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"h", " ", 
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["t", 
            RowBox[{"n", "-", "1"}]], ",", 
           RowBox[{
            SubscriptBox["y", "0"], "+", 
            RowBox[{"(", 
             RowBox[{
              SubscriptBox["\[CapitalDelta]y", "0"], "+", " ", 
              SubscriptBox["\[CapitalDelta]y", "1"], "+", "\[CenterEllipsis]",
               "+", 
              SubscriptBox["\[CapitalDelta]y", 
               RowBox[{"n", "-", "2"}]]}], ")"}]}]}], ")"}]}], "-", 
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "2"}]]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"LinearlyImplicitMidpointIncrements",
 CellID->723795696]
}, Open  ]],

Cell[CellGroupData[{

Cell["Smoothing step", "Subsection",
 CellID->422606491],

Cell[TextData[{
 "An appropriate smoothing step for the linearly implicit midpoint rule is \
[",
 ButtonBox["BD83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25441"],
 "]:"
}], "Text",
 CellID->1734205694],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["Sy", "h"], " ", 
    RowBox[{"(", "n", ")"}]}], "=", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["y", 
       RowBox[{"n", "-", "1"}]], "+", 
      SubscriptBox["y", 
       RowBox[{"n", "+", "1"}]]}], ")"}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"BaderSmoothingStep",
 CellID->1460682982],

Cell[TextData[{
 "Bader's smoothing step ",
 "(",
 
 CounterBox["NumberedEquation", "BaderSmoothingStep"],
 ") rewritten ",
 "in terms of increments becomes:"
}], "Text",
 CellID->634788273],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["S\[CapitalDelta]y", "h"], " ", 
    RowBox[{"(", "n", ")"}]}], "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Sy", "h"], " ", 
      RowBox[{"(", "n", ")"}]}], " ", "-", " ", 
     SubscriptBox["y", "0"]}], "=", 
    RowBox[{
     SubscriptBox["\[CapitalSigma]\[CapitalDelta]y", "n"], "+", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["\[CapitalDelta]y", "n"], "-", 
        SubscriptBox["\[CapitalDelta]y", 
         RowBox[{"n", "-", "1"}]]}], ")"}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ModifiedBaderSmoothingStep",
 CellID->972989670],

Cell[TextData[{
 "The required quantities are obtained when (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitMidpointIncrements"],
 ") is run with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"2", " ", "m"}], TraditionalForm]]],
 " steps."
}], "Text",
 CellID->440872212],

Cell[TextData[{
 "The smoothing step for the linearly implicit midpoint rule has a different \
role to Gragg's smoothing for the explicit midpoint rule (see [",
 ButtonBox["BD83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25441"],
 "] and ",
 "[",
 ButtonBox["SB83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#7456"],
 "]",
 "). Since there is no weakly stable term to eliminate, the aim is to improve \
the asymptotic stability."
}], "Text",
 CellID->1808198925],

Cell[TextData[{
 "The method ",
 StyleBox["LinearlyImplicitMidpoint", "MR"],
 " uses ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     RowBox[{"2", " ", "m"}], " ", "-", " ", "1"}]}], TraditionalForm]]],
 " steps and ",
 StyleBox["LinearlyImplicitModifiedMidpoint", "MR"],
 " uses ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{"2", " ", "m"}]}], TraditionalForm]]],
 " steps followed by the smoothing step (",
 
 CounterBox["NumberedEquation", "ModifiedBaderSmoothingStep"],
 ")."
}], "Text",
 CellID->261648188]
}, Open  ]],

Cell[CellGroupData[{

Cell["Polynomial extrapolation in terms of increments", "Subsection",
 CellID->288744440],

Cell[TextData[{
 "We have seen how to modify ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "1"}]], TraditionalForm]]],
 ",",
 " ",
 "the entries in the first column of the extrapolation table, in terms of \
increments."
}], "Text",
 CellID->864233916],

Cell[TextData[{
 "However, for certain base integration methods, ",
 StyleBox["each",
  FontSlant->"Italic"],
 " of the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "j"}]], TraditionalForm]]],
 " corresponds to an explicit Runge-Kutta method."
}], "Text",
 CellID->1452113518],

Cell["\<\
Therefore, it appears that the correspondence has not yet been fully \
exploited and further refinement is possible.\
\>", "Text",
 CellID->637992556],

Cell[TextData[{
 "Since the Aitken-Neville algorithm (",
 
 CounterBox["NumberedEquation", "AitkenNeville"],
 ") involves linear differences, the entire extrapolation process can be \
carried out using increments."
}], "Text",
 CellID->656923023],

Cell["\<\
This leads to the following modification of the Aitken-Neville algorithm:\
\>", "Text",
 CellID->547215790],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["\[CapitalDelta]T", 
     RowBox[{"i", ",", " ", "j"}]], "=", " ", 
    RowBox[{
     SubscriptBox["\[CapitalDelta]T", 
      RowBox[{"i", ",", " ", 
       RowBox[{"j", "-", "1"}]}]], " ", "+", " ", 
     FractionBox[
      RowBox[{
       SubscriptBox["\[CapitalDelta]T", 
        RowBox[{"i", ",", " ", 
         RowBox[{"j", " ", "-", " ", "1"}]}]], " ", "-", " ", 
       SubscriptBox["\[CapitalDelta]T", 
        RowBox[{
         RowBox[{"i", "-", "1"}], ",", " ", 
         RowBox[{"j", " ", "-", " ", "1"}]}]]}], 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         FractionBox[
          SubscriptBox["n", "i"], 
          SubscriptBox["n", 
           RowBox[{"i", " ", "-", " ", "j", " ", "+", " ", "1"}]]], ")"}], 
        "p"], " ", "-", " ", "1"}]]}]}], " ", ",", 
   RowBox[{"i", "=", "2"}], ",", "\[Ellipsis]", ",", "k", ",", 
   RowBox[{"j", " ", "=", " ", "2"}], ",", "\[Ellipsis]", ",", 
   RowBox[{"i", "."}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"ModifiedAitkenNeville",
 CellID->1005920553],

Cell[TextData[{
 "The quantities ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]T", 
     RowBox[{"i", ",", " ", "j"}]], " ", "=", " ", 
    RowBox[{
     SubscriptBox["T", 
      RowBox[{"i", ",", " ", "j"}]], " ", "-", " ", 
     SubscriptBox["y", "0"]}]}], TraditionalForm]]],
 " in (",
 
 CounterBox["NumberedEquation", "ModifiedAitkenNeville"],
 ") can be computed iteratively, starting from the initial quantities ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "1"}]], TraditionalForm]]],
 "that are obtained from the modified base integration schemes without adding \
the contribution from ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "0"], TraditionalForm]]],
 "."
}], "Text",
 CellID->1142457867],

Cell[TextData[{
 "The final desired value ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"k", ",", " ", "k"}]], TraditionalForm]]],
 " can be recovered as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]T", 
     RowBox[{"k", ",", " ", "k"}]], " ", "+", " ", 
    SubscriptBox["y", "0"]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->545335637],

Cell["\<\
The advantage is that the extrapolation table is built up using smaller \
quantities, and so the effect of rounding errors from subtractive \
cancellation is reduced.\
\>", "Text",
 CellID->979046348]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implementation issues", "Section",
 CellTags->"s:4",
 CellID->777991017],

Cell["\<\
There are a number of important implementation issues that should be \
considered, some of which are mentioned here.\
\>", "Text",
 CellID->1845275582],

Cell[CellGroupData[{

Cell["Jacobian reuse", "Subsection",
 CellID->47000056],

Cell[TextData[{
 "The Jacobian is evaluated only once for all entries ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "1"}]], TraditionalForm]]],
 " at each time step by storing it together with the associated time that it \
is evaluated. This also has the advantage that the Jacobian does not need to \
be recomputed for rejected steps."
}], "Text",
 CellID->1457698310]
}, Open  ]],

Cell[CellGroupData[{

Cell["Dense linear algebra", "Subsection",
 CellID->386085898],

Cell[TextData[{
 "For dense systems, the LAPACK routines xyytrf can be used for the LU \
decomposition and the routines xyytrs for solving the resulting triangular \
systems ",
 "[",
 ButtonBox["LAPACK99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#18138"],
 "]",
 "."
}], "Text",
 CellID->670278164]
}, Open  ]],

Cell[CellGroupData[{

Cell["Adaptive order and work estimation", "Subsection",
 CellID->599144811],

Cell["\<\
In order to adaptively change the order of the extrapolation throughout the \
integration, it is important to have a measure of the amount of work required \
by the base scheme and extrapolation sequence.\
\>", "Text",
 CellID->523087874],

Cell["\<\
A measure of the relative cost of function evaluations is advantageous.\
\>", "Text",
 CellID->1669983850],

Cell["\<\
The dimension of the system, preferably with a weighting according to \
structure, needs to be incorporated for linearly implicit schemes in order to \
take account of the expense of solving each linear system.\
\>", "Text",
 CellID->1549385406]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stability check", "Subsection",
 CellID->940495697],

Cell["\<\
Extrapolation methods use a large basic step size that can give rise to some \
difficulties.\
\>", "Text",
 CellID->1960337703],

Cell[TextData[{
 "\"Neither code can solve the van der Pol equation problem in a \
straightforward way because of overflow...\" ",
 "[",
 ButtonBox["S87",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#15837"],
 "]."
}], "Text",
 CellID->125697454],

Cell[TextData[{
 "Two forms of stability check are used for the linearly implicit base \
schemes (for further discussion, see ",
 "[",
 ButtonBox["HW96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#32197"],
 "]",
 ")."
}], "Text",
 CellID->1363102189],

Cell[TextData[{
 "One check is performed during the extrapolation process. Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox[
     StyleBox["err",
      FontSlant->"Italic"], "j"], "=", 
    RowBox[{"||", 
     RowBox[{
      SubscriptBox["T", 
       RowBox[{"j", " ", ",", " ", 
        RowBox[{"j", "-", "1"}]}]], "-", 
      SubscriptBox["T", 
       RowBox[{"j", " ", ",", " ", "j"}]]}], "||"}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1547059749],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"If", " ", 
     SubscriptBox[
      StyleBox["err",
       FontSlant->"Italic"], "j"]}], " ", "\[GreaterEqual]", 
    RowBox[{
     SubscriptBox[
      StyleBox["err",
       FontSlant->"Italic"], 
      RowBox[{"j", "-", "1"}]], " ", "for", " ", "some", " ", "j"}], 
    "\[GreaterEqual]", "3"}], ",", " ", 
   RowBox[{
    RowBox[{
    "then", " ", "recompute", " ", "the", " ", "step", " ", "with", " ", 
     "H"}], " ", "=", " ", 
    RowBox[{"H", "/", "2"}]}]}], TraditionalForm]], "DisplayMath",
 CellID->1586380436],

Cell[TextData[{
 "In order to interrupt computations in the computation of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"1", " ", ",", " ", "1"}]], TraditionalForm]]],
 " Deuflhard suggests checking if the Newton iteration applied to a fully \
implicit scheme would converge."
}], "Text",
 CellID->1997890462],

Cell["\<\
For the implicit Euler method this leads to consideration of:\
\>", "Text",
 CellID->372565771],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      SubscriptBox["\[CapitalDelta]", "0"]}], "=", 
     RowBox[{"h", " ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "0"], ",", " ", 
        SubscriptBox["y", "0"]}], ")"}]}]},
    {
     RowBox[{
      RowBox[{"(", 
       RowBox[{"I", " ", "-", " ", 
        RowBox[{"h", " ", "J"}]}], ")"}], " ", 
      SubscriptBox["\[CapitalDelta]", "1"]}], "=", 
     RowBox[{
      RowBox[{"h", " ", 
       RowBox[{"f", "(", 
        RowBox[{
         SubscriptBox["t", "0"], ",", " ", 
         RowBox[{
          SubscriptBox["y", "0"], "+", " ", 
          SubscriptBox["\[CapitalDelta]", "0"]}]}], ")"}]}], " ", "-", " ", 
      SubscriptBox["\[CapitalDelta]", "0"]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"ImplicitEulerStabilityCheck",
 CellID->2027252648],

Cell[TextData[{
 "Notice that (",
 
 CounterBox["NumberedEquation", "ImplicitEulerStabilityCheck"],
 ") differs from (",
 
 CounterBox["NumberedEquation", "LinearlyImplicitEulerSteps"],
 ") only in the second equation. It requires finding the solution for a \
different right-hand side but no extra function evaluation."
}], "Text",
 CellID->1658605739],

Cell[TextData[{
 "For the implicit midpoint method: ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]", "0"], " ", "=", " ", 
    SubscriptBox["\[CapitalDelta]y", "0"]}], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]", "1"], " ", "=", 
    RowBox[{
     RowBox[{"1", "/", "2"}], 
     RowBox[{"(", " ", 
      RowBox[{
       SubscriptBox["\[CapitalDelta]y", "1"], "-", 
       SubscriptBox["\[CapitalDelta]y", "0"]}], ")"}]}]}], 
   TraditionalForm]]],
 ", which simply requires a few basic arithmetic operations."
}], "Text",
 CellID->634213959],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"If", " ", "||", 
    SubscriptBox["\[CapitalDelta]", "1"], "||", " ", 
    RowBox[{"\[GreaterEqual]", 
     RowBox[{"||", 
      SubscriptBox["\[CapitalDelta]", "0"], "||", " ", 
      RowBox[{
      "then", " ", "the", " ", "implicit", " ", "iteration", " ", 
       "diverges"}]}]}]}], ",", " ", 
   RowBox[{
    RowBox[{
    "so", " ", "recompute", " ", "the", " ", "step", " ", "with", " ", "H"}], 
    " ", "=", " ", 
    RowBox[{"H", "/", "2."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->168513440],

Cell["\<\
Increments are a more accurate formulation for the implementation of both \
forms of stability check.\
\>", "Text",
 CellID->375356008]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellTags->"s:5",
 CellID->2086327477],

Cell[CellGroupData[{

Cell["Stiff systems", "Subsection",
 CellID->806505997],

Cell["\<\
One of the simplest nonlinear equations describing a circuit is van der Pol's \
equation.\
\>", "Text",
 CellID->1032279031],

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<VanderPol\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"time", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<TimeData\>\"", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[11]:=",
 CellID->486059620],

Cell[TextData[{
 "This solves the equations using ",
 StyleBox["Extrapolation", "MR"],
 " with the ",
 StyleBox["LinearlyImplicitEuler", "MR"],
 " base method with the default sub-Harmonic sequence 2,3 4,\[Ellipsis]."
}], "Text",
 CellID->1812426550],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"vdpsol", " ", "=", " ", 
  RowBox[{"Flatten", "[", 
   RowBox[{"vars", " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<Extrapolation\>\"", ",", 
         RowBox[{"Method", "\[Rule]", "\"\<LinearlyImplicitEuler\>\""}]}], 
        "}"}]}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[14]:=",
 CellID->27253251],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "2.5`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "T", "]"}], ",", 
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "2.5`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "T", "]"}]}], "}"}]], "Output",
 ImageSize->{300, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->94940460]
}, Open  ]],

Cell["\<\
Notice that the Jacobian matrix is computed automatically (user specifiable \
by using either numerical differences or symbolic derivatives) and \
appropriate linear algebra routines are selected and invoked at run time.\
\>", "Text",
 CellID->643989250],

Cell["This plots the first solution component over time.", "Text",
 CellID->805244268],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"First", "[", "vdpsol", "]"}], "]"}], ",", 
   RowBox[{"Evaluate", "[", "time", "]"}], ",", " ", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", " ", 
   RowBox[{"Axes", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->729187191],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 230},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->490935639]
}, Open  ]],

Cell["This plots the step sizes taken in computing the solution.", "Text",
 CellID->115458930],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"StepDataPlot", "[", "vdpsol", "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->1115273089],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->606774139]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["High-precision comparison", "Subsection",
 CellID->714603741],

Cell["Select the Lorenz equations.", "Text",
 CellID->1638360964],

Cell[BoxData[
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<Lorenz\>\"", "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->883767673],

Cell[TextData[{
 "This invokes a bigfloat, or software floating-point number, embedded \
explicit Runge-Kutta method of order 9(8) [",
 ButtonBox["V78",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#23842"],
 "]."
}], "Text",
 CellID->116505500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"erksol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
         RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "9"}]}], "}"}]}], ",", 
      
      RowBox[{"WorkingPrecision", "\[Rule]", "32"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->431373013],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5.892104`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->1189623250]
}, Open  ]],

Cell[TextData[{
 "This invokes the ",
 StyleBox["Adams", "MR"],
 " method using a bigfloat version of LSODA. The maximum order of these \
methods is twelve."
}], "Text",
 CellID->242202954],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"adamssol", " ", "=", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", "\"\<Adams\>\""}], ",", " ", 
      RowBox[{"WorkingPrecision", "\[Rule]", "32"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->1794475202],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.3144950000000004`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->1987993033]
}, Open  ]],

Cell[TextData[{
 "This invokes the ",
 StyleBox["Extrapolation", "MR"],
 " method with ",
 StyleBox["ExplicitModifiedMidpoint", "MR"],
 " as the base integration scheme."
}], "Text",
 CellID->1828583391],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"extrapsol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<Extrapolation\>\"", ",", " ", 
         RowBox[{"Method", "->", "\"\<ExplicitModifiedMidpoint\>\""}]}], 
        "}"}]}], ",", 
      RowBox[{"WorkingPrecision", "\[Rule]", "32"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->1644882016],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.2868049999999993`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->1383544515]
}, Open  ]],

Cell["\<\
As many as 14 entries in the first column of the extrapolation table are \
used, which results in a method of order 28. This means that a much larger \
step size can be taken.\
\>", "Text",
 CellID->1708352391],

Cell["Here are the step sizes taken by the various methods.", "Text",
 CellID->1156004108],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"methods", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
    "\"\<ExplicitRungeKutta\>\"", ",", " ", "\"\<Adams\>\"", ",", " ", 
     "\"\<Extrapolation\>\""}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"solutions", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"erksol", ",", "adamssol", ",", " ", "extrapsol"}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{"MapThread", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"StepDataPlot", "[", 
     RowBox[{"#2", ",", " ", 
      RowBox[{"PlotLabel", "\[Rule]", "#1"}]}], "]"}], "&"}], ",", 
   RowBox[{"{", 
    RowBox[{"methods", " ", ",", " ", "solutions"}], "}"}]}], "]"}]}], "Input",\

 CellLabel->"In[21]:=",
 CellID->1102977560],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{386, 276},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->112687912]
}, Open  ]],

Cell["\<\
Here are the solutions obtained by the various methods at the end of the \
integration.\
\>", "Text",
 CellID->1531360116],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DisplayForm", "[", "\[IndentingNewLine]", 
  RowBox[{"GridBox", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"MapThread", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"#1", ",", " ", 
         RowBox[{"FinalSolutions", "[", 
          RowBox[{"system", ",", " ", "#2"}], "]"}]}], "}"}], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{"methods", " ", ",", " ", "solutions"}], "}"}]}], "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{"RowLines", "\[Rule]", "True"}], ",", 
    RowBox[{"ColumnLines", "\[Rule]", "True"}], ",", " ", 
    RowBox[{"GridFrame", "\[Rule]", "True"}]}], "\[IndentingNewLine]", "]"}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->1041821389],

Cell[BoxData[
 TagBox[GridBox[{
    {"ExplicitRungeKutta", 
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
        "-", "9.131313027755842230496037503233988333639411618008`30.\
224687046657845"}], ",", 
        RowBox[{
        "-", "12.476178811599969682958215980805733678039871456862`30.\
343072841888763"}], ",", 
        "22.843338961384550559344064537950420233451887696428`30.\
41875313882122"}], "}"}], "}"}]},
    {"Adams", 
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
        "-", "9.131313013759373146187886878128990622270657916449`30.\
224687046624755"}], ",", 
        RowBox[{
        "-", "12.476178792735678593541117951410428733977303833584`30.\
34307284027132"}], ",", 
        "22.843338946842967198217389853770302217719193805198`30.\
418753141010054"}], "}"}], "}"}]},
    {"Extrapolation", 
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
        "-", "9.131313020651480566479188631787446836838409091547`30.\
22468704664105"}], ",", 
        RowBox[{
        "-", "12.476178802024760667953412332823959606368190796808`30.\
34307284106778"}], ",", 
        "22.843338954003545300133515393033805494785048346862`30.\
418753139932235"}], "}"}], "}"}]}
   },
   GridBoxDividers->{
    "Columns" -> {{True}}, "ColumnsIndexed" -> {}, "Rows" -> {{True}}, 
     "RowsIndexed" -> {}}],
  DisplayForm]], "Output",
 ImageSize->{880, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]//DisplayForm=",
 CellID->1647144482]
}, Open  ]],

Cell[TextData[{
 "For comparison, here is a higher-precision reference solution obtained \
using ",
 StyleBox["Extrapolation", "MR"],
 "."
}], "Text",
 CellID->416155499],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"refsol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", " ", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<Extrapolation\>\"", ",", " ", 
        RowBox[{"Method", "->", "\"\<ExplicitModifiedMidpoint\>\""}]}], 
       "}"}]}], ",", 
     RowBox[{"WorkingPrecision", "\[Rule]", "64"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"SetPrecision", "[", 
  RowBox[{
   RowBox[{"FinalSolutions", "[", 
    RowBox[{"system", ",", " ", "refsol"}], "]"}], ",", " ", "32"}], 
  "]"}]}], "Input",
 CellLabel->"In[25]:=",
 CellID->1527881009],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "9.131313027368752927862669028453995896524315740779`32."}], 
    ",", 
    RowBox[{
    "-", "12.476178811078253333918990893247768951256628973104`31.\
999999999999996"}], ",", 
    "22.843338960982388205501901553235537909596585328078`32."}], "}"}], 
  "}"}]], "Output",
 ImageSize->{513, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->1363978244]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Work-error comparison", "Subsection",
 CellID->443408750],

Cell[TextData[{
 "For comparing different extrapolation schemes consider an example from ",
 "[",
 ButtonBox["HW96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#32197"],
 "]",
 "."
}], "Text",
 CellID->411792618],

Cell[BoxData[{
 RowBox[{
  RowBox[{"t0", " ", "=", " ", 
   RowBox[{"\[Pi]", "/", "6"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"h0", " ", "=", " ", 
   RowBox[{"1", "/", "10"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"y0", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"2", "/", 
     SqrtBox["3"]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"-", 
           RowBox[{"y", "[", "t", "]"}]}], " ", 
          RowBox[{"Sin", "[", "t", "]"}]}], "+", 
         RowBox[{"2", " ", 
          RowBox[{"Tan", "[", "t", "]"}]}]}], ")"}], " ", 
       RowBox[{"y", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"y", "[", "t0", "]"}], "\[Equal]", "y0"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"exactsol", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"y", "[", "t", "]"}], "/.", 
     RowBox[{"First", "[", 
      RowBox[{"DSolve", "[", 
       RowBox[{"eqs", ",", 
        RowBox[{"y", "[", "t", "]"}], ",", "t"}], "]"}], "]"}]}], "/.", 
    RowBox[{"t", "\[Rule]", 
     RowBox[{"t0", "+", "h0"}]}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"idata", " ", "=", "  ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"eqs", ",", 
       RowBox[{"y", "[", "t", "]"}], ",", "t"}], "}"}], ",", "h0", ",", 
     "exactsol"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[27]:=",
 CellID->844820333],

Cell[TextData[{
 "The exact solution is given by ",
 StyleBox["y",
  FontSlant->"Italic"],
 "(",
 StyleBox["t",
  FontSlant->"Italic"],
 ") = 1/cos(",
 StyleBox["t",
  FontSlant->"Italic"],
 ")."
}], "Text",
 CellID->1248653018]
}, Open  ]],

Cell[CellGroupData[{

Cell["Increment formulation", "Subsection",
 CellID->1140979810],

Cell[TextData[{
 "This example involves an eighth-order extrapolation of ",
 StyleBox["ExplicitEuler", "MR"],
 " with the Harmonic sequence. Approximately two digits of accuracy are \
gained by using the increment-based formulation throughout the extrapolation \
process."
}], "Text",
 CellID->1535809639],

Cell[TextData[{
 "The results for the standard formulation (",
 
 CounterBox["NumberedEquation", "EulerSteps"],
 ") are depicted in green."
}], "BulletedText",
 CellID->1364111948],

Cell[TextData[{
 "The results for the increment formulation (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ") followed by standard extrapolation (",
 
 CounterBox["NumberedEquation", "AitkenNeville"],
 ") is depicted in blue."
}], "BulletedText",
 CellID->108769251],

Cell[CellGroupData[{

Cell[TextData[{
 "The results for the increment formulation (",
 
 CounterBox["NumberedEquation", "EulerIncrements"],
 ") with extrapolation carried out on the increments using (",
 
 CounterBox["NumberedEquation", "ModifiedAitkenNeville"],
 ") is depicted in red."
}], "BulletedText",
 CellID->749593462],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10  scalefont  setfont
% Scaling calculations
0.97619 0.0677249 0.0147151 0.402492 [
[.02804 -0.0125 -23.4688 -14.375 ]
[.02804 -0.0125 23.4688 0 ]
[.23122 -0.0125 -23.4688 -14.375 ]
[.23122 -0.0125 23.4688 0 ]
[.43439 -0.0125 -21.5 -14.375 ]
[.43439 -0.0125 21.5 0 ]
[.63757 -0.0125 -19.875 -11.1875 ]
[.63757 -0.0125 19.875 0 ]
[.84074 -0.0125 -11.625 -11.1875 ]
[.84074 -0.0125 11.625 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01472 -9.5 -5.59375 ]
[-0.0125 .01472 0 5.59375 ]
[-0.0125 .08559 -17.75 -5.59375 ]
[-0.0125 .08559 0 5.59375 ]
[-0.0125 .13588 -9.5 -5.59375 ]
[-0.0125 .13588 0 5.59375 ]
[-0.0125 .20675 -9.5 -5.59375 ]
[-0.0125 .20675 0 5.59375 ]
[-0.0125 .29605 -9.5 -5.59375 ]
[-0.0125 .29605 0 5.59375 ]
[-0.0125 .35486 -9.5 -5.59375 ]
[-0.0125 .35486 0 5.59375 ]
[-0.0125 .41721 -15 -5.59375 ]
[-0.0125 .41721 0 5.59375 ]
[-0.0125 .48808 -15 -5.59375 ]
[-0.0125 .48808 0 5.59375 ]
[-0.0125 .53837 -15 -5.59375 ]
[-0.0125 .53837 0 5.59375 ]
[-0.0125 .60924 -15 -5.59375 ]
[-0.0125 .60924 0 5.59375 ]
[ 0 0 -0.125 0 ]
[.02804 .63053 -23.4688 0 ]
[.02804 .63053 23.4688 14.375 ]
[.23122 .63053 -23.4688 0 ]
[.23122 .63053 23.4688 14.375 ]
[.43439 .63053 -21.5 0 ]
[.43439 .63053 21.5 14.375 ]
[.63757 .63053 -19.875 0 ]
[.63757 .63053 19.875 11.1875 ]
[.84074 .63053 -11.625 0 ]
[.84074 .63053 11.625 11.1875 ]
[ 0 .61803 .125 0 ]
[1.0125 .01472 0 -5.59375 ]
[1.0125 .01472 9.5 5.59375 ]
[1.0125 .08559 0 -5.59375 ]
[1.0125 .08559 17.75 5.59375 ]
[1.0125 .13588 0 -5.59375 ]
[1.0125 .13588 9.5 5.59375 ]
[1.0125 .20675 0 -5.59375 ]
[1.0125 .20675 9.5 5.59375 ]
[1.0125 .29605 0 -5.59375 ]
[1.0125 .29605 9.5 5.59375 ]
[1.0125 .35486 0 -5.59375 ]
[1.0125 .35486 9.5 5.59375 ]
[1.0125 .41721 0 -5.59375 ]
[1.0125 .41721 15 5.59375 ]
[1.0125 .48808 0 -5.59375 ]
[1.0125 .48808 15 5.59375 ]
[1.0125 .53837 0 -5.59375 ]
[1.0125 .53837 15 5.59375 ]
[1.0125 .60924 0 -5.59375 ]
[1.0125 .60924 15 5.59375 ]
[ 1 0 .125 0 ]
[.5 .63053 -105.219 14.375 ]
[.5 .63053 105.219 27.6875 ]
[ -0.0005 -0.0005 0 0 ]
[ 1.0005 .61853 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02804 0 m
.02804 .00625 L
s
gsave
.02804 -0.0125 -84.4688 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(14) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.23122 0 m
.23122 .00625 L
s
gsave
.23122 -0.0125 -84.4688 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(11) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.43439 0 m
.43439 .00625 L
s
gsave
.43439 -0.0125 -82.5 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(8) show
100.125 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.63757 0 m
.63757 .00625 L
s
gsave
.63757 -0.0125 -80.875 -15.1875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.00001) show
98.750 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.84074 0 m
.84074 .00625 L
s
gsave
.84074 -0.0125 -72.625 -15.1875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.01) show
82.250 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.16683 0 m
.16683 .00375 L
s
.18708 0 m
.18708 .00375 L
s
.19896 0 m
.19896 .00375 L
s
.2074 0 m
.2074 .00375 L
s
.21395 0 m
.21395 .00375 L
s
.21931 0 m
.21931 .00375 L
s
.22383 0 m
.22383 .00375 L
s
.22776 0 m
.22776 .00375 L
s
.37 0 m
.37 .00375 L
s
.39026 0 m
.39026 .00375 L
s
.40214 0 m
.40214 .00375 L
s
.41058 0 m
.41058 .00375 L
s
.41713 0 m
.41713 .00375 L
s
.42248 0 m
.42248 .00375 L
s
.42701 0 m
.42701 .00375 L
s
.43093 0 m
.43093 .00375 L
s
.57317 0 m
.57317 .00375 L
s
.59343 0 m
.59343 .00375 L
s
.60531 0 m
.60531 .00375 L
s
.61375 0 m
.61375 .00375 L
s
.6203 0 m
.6203 .00375 L
s
.62565 0 m
.62565 .00375 L
s
.63018 0 m
.63018 .00375 L
s
.63411 0 m
.63411 .00375 L
s
.77635 0 m
.77635 .00375 L
s
.7966 0 m
.7966 .00375 L
s
.80849 0 m
.80849 .00375 L
s
.81693 0 m
.81693 .00375 L
s
.82348 0 m
.82348 .00375 L
s
.82883 0 m
.82883 .00375 L
s
.83336 0 m
.83336 .00375 L
s
.83728 0 m
.83728 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01472 m
.00625 .01472 L
s
gsave
-0.0125 .01472 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .08559 m
.00625 .08559 L
s
gsave
-0.0125 .08559 -78.75 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.5) show
76.750 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .13588 m
.00625 .13588 L
s
gsave
-0.0125 .13588 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(2) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .20675 m
.00625 .20675 L
s
gsave
-0.0125 .20675 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(3) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .29605 m
.00625 .29605 L
s
gsave
-0.0125 .29605 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(5) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .35486 m
.00625 .35486 L
s
gsave
-0.0125 .35486 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(7) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .41721 m
.00625 .41721 L
s
gsave
-0.0125 .41721 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .48808 m
.00625 .48808 L
s
gsave
-0.0125 .48808 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(15) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .53837 m
.00625 .53837 L
s
gsave
-0.0125 .53837 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(20) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .60924 m
.00625 .60924 L
s
gsave
-0.0125 .60924 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(30) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .03138 m
.00375 .03138 L
s
0 .04658 m
.00375 .04658 L
s
0 .06058 m
.00375 .06058 L
s
0 .07353 m
.00375 .07353 L
s
0 .09687 m
.00375 .09687 L
s
0 .10747 m
.00375 .10747 L
s
0 .11746 m
.00375 .11746 L
s
0 .12691 m
.00375 .12691 L
s
0 .25704 m
.00375 .25704 L
s
0 .32791 m
.00375 .32791 L
s
0 .3782 m
.00375 .3782 L
s
0 .39879 m
.00375 .39879 L
s
0 .43387 m
.00375 .43387 L
s
0 .44908 m
.00375 .44908 L
s
0 .46307 m
.00375 .46307 L
s
0 .47602 m
.00375 .47602 L
s
0 .49936 m
.00375 .49936 L
s
0 .50996 m
.00375 .50996 L
s
0 .51995 m
.00375 .51995 L
s
0 .5294 m
.00375 .5294 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02804 .61178 m
.02804 .61803 L
s
gsave
.02804 .63053 -84.4688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(14) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.23122 .61178 m
.23122 .61803 L
s
gsave
.23122 .63053 -84.4688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(11) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.43439 .61178 m
.43439 .61803 L
s
gsave
.43439 .63053 -82.5 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(8) show
100.125 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.63757 .61178 m
.63757 .61803 L
s
gsave
.63757 .63053 -80.875 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.00001) show
98.750 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.84074 .61178 m
.84074 .61803 L
s
gsave
.84074 .63053 -72.625 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.01) show
82.250 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.16683 .61428 m
.16683 .61803 L
s
.18708 .61428 m
.18708 .61803 L
s
.19896 .61428 m
.19896 .61803 L
s
.2074 .61428 m
.2074 .61803 L
s
.21395 .61428 m
.21395 .61803 L
s
.21931 .61428 m
.21931 .61803 L
s
.22383 .61428 m
.22383 .61803 L
s
.22776 .61428 m
.22776 .61803 L
s
.37 .61428 m
.37 .61803 L
s
.39026 .61428 m
.39026 .61803 L
s
.40214 .61428 m
.40214 .61803 L
s
.41058 .61428 m
.41058 .61803 L
s
.41713 .61428 m
.41713 .61803 L
s
.42248 .61428 m
.42248 .61803 L
s
.42701 .61428 m
.42701 .61803 L
s
.43093 .61428 m
.43093 .61803 L
s
.57317 .61428 m
.57317 .61803 L
s
.59343 .61428 m
.59343 .61803 L
s
.60531 .61428 m
.60531 .61803 L
s
.61375 .61428 m
.61375 .61803 L
s
.6203 .61428 m
.6203 .61803 L
s
.62565 .61428 m
.62565 .61803 L
s
.63018 .61428 m
.63018 .61803 L
s
.63411 .61428 m
.63411 .61803 L
s
.77635 .61428 m
.77635 .61803 L
s
.7966 .61428 m
.7966 .61803 L
s
.80849 .61428 m
.80849 .61803 L
s
.81693 .61428 m
.81693 .61803 L
s
.82348 .61428 m
.82348 .61803 L
s
.82883 .61428 m
.82883 .61803 L
s
.83336 .61428 m
.83336 .61803 L
s
.83728 .61428 m
.83728 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
gsave
1.0125 .01472 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .08559 m
1 .08559 L
s
gsave
1.0125 .08559 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.5) show
76.750 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .13588 m
1 .13588 L
s
gsave
1.0125 .13588 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(2) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .20675 m
1 .20675 L
s
gsave
1.0125 .20675 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(3) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .29605 m
1 .29605 L
s
gsave
1.0125 .29605 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(5) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .35486 m
1 .35486 L
s
gsave
1.0125 .35486 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(7) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .41721 m
1 .41721 L
s
gsave
1.0125 .41721 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .48808 m
1 .48808 L
s
gsave
1.0125 .48808 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(15) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .53837 m
1 .53837 L
s
gsave
1.0125 .53837 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(20) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .60924 m
1 .60924 L
s
gsave
1.0125 .60924 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(30) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.99625 .03138 m
1 .03138 L
s
.99625 .04658 m
1 .04658 L
s
.99625 .06058 m
1 .06058 L
s
.99625 .07353 m
1 .07353 L
s
.99625 .09687 m
1 .09687 L
s
.99625 .10747 m
1 .10747 L
s
.99625 .11746 m
1 .11746 L
s
.99625 .12691 m
1 .12691 L
s
.99625 .25704 m
1 .25704 L
s
.99625 .32791 m
1 .32791 L
s
.99625 .3782 m
1 .3782 L
s
.99625 .39879 m
1 .39879 L
s
.99625 .43387 m
1 .43387 L
s
.99625 .44908 m
1 .44908 L
s
.99625 .46307 m
1 .46307 L
s
.99625 .47602 m
1 .47602 L
s
.99625 .49936 m
1 .49936 L
s
.99625 .50996 m
1 .50996 L
s
.99625 .51995 m
1 .51995 L
s
.99625 .5294 m
1 .5294 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
gsave
.5 .63053 -166.219 10.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Plot) show
86.875 13.688 moveto
(of) show
100.125 13.688 moveto
(work) show
128.625 13.688 moveto
(vs) show
143.938 13.688 moveto
(error) show
172.312 13.688 moveto
(on) show
188.875 13.688 moveto
(a) show
198.812 13.688 moveto
(log) show
214.688 13.688 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
222.375 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(log) show
241.562 13.688 moveto
(scale) show
269.438 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.5 Mabswid
.83602 .01472 m
.8163 .13588 L
.72509 .13588 L
.83602 .01472 L
s
.8163 .13588 m
.80459 .20675 L
.69286 .25704 L
.8163 .13588 L
s
.80459 .20675 m
.79624 .25704 L
.67248 .32791 L
.80459 .20675 L
s
.79624 .25704 m
.78974 .29605 L
.65744 .3782 L
.79624 .25704 L
s
.78974 .29605 m
.78442 .32791 L
.64551 .41721 L
.78974 .29605 L
s
.78442 .32791 m
.77992 .35486 L
.6356 .44908 L
.78442 .32791 L
s
.77992 .35486 m
.77602 .3782 L
.62714 .47602 L
.77992 .35486 L
s
.72509 .13588 m
.69286 .25704 L
.53423 .25704 L
.72509 .13588 L
s
.69286 .25704 m
.67248 .32791 L
.40147 .35486 L
.69286 .25704 L
s
.67248 .32791 m
.65744 .3782 L
.44516 .41721 L
.67248 .32791 L
s
.65744 .3782 m
.64551 .41721 L
.43699 .46307 L
.65744 .3782 L
s
.64551 .41721 m
.6356 .44908 L
.42639 .49936 L
.64551 .41721 L
s
.6356 .44908 m
.62714 .47602 L
.4161 .5294 L
.6356 .44908 L
s
.53423 .25704 m
.40147 .35486 L
.50317 .35486 L
.53423 .25704 L
s
.40147 .35486 m
.44516 .41721 L
.45739 .43387 L
.40147 .35486 L
s
.44516 .41721 m
.43699 .46307 L
.42563 .48808 L
.44516 .41721 L
s
.43699 .46307 m
.42639 .49936 L
.40101 .5294 L
.43699 .46307 L
s
.42639 .49936 m
.4161 .5294 L
.3808 .5628 L
.42639 .49936 L
s
.50317 .35486 m
.45739 .43387 L
.37672 .43387 L
.50317 .35486 L
s
.45739 .43387 m
.42563 .48808 L
.3205 .49936 L
.45739 .43387 L
s
.42563 .48808 m
.40101 .5294 L
.28173 .5469 L
.42563 .48808 L
s
.40101 .5294 m
.3808 .5628 L
.25163 .58423 L
.40101 .5294 L
s
.37672 .43387 m
.3205 .49936 L
.26526 .49936 L
.37672 .43387 L
s
.3205 .49936 m
.28173 .5469 L
.21236 .55503 L
.3205 .49936 L
s
.28173 .5469 m
.25163 .58423 L
.17318 .59718 L
.28173 .5469 L
s
.26526 .49936 m
.21236 .55503 L
.15843 .55503 L
.26526 .49936 L
s
.21236 .55503 m
.17318 .59718 L
.09502 .60332 L
.21236 .55503 L
s
.15843 .55503 m
.09502 .60332 L
.02381 .60332 L
.15843 .55503 L
s
0 0 1 r
.83602 .01472 m
.8163 .13588 L
.72509 .13588 L
.83602 .01472 L
s
.8163 .13588 m
.80459 .20675 L
.69286 .25704 L
.8163 .13588 L
s
.80459 .20675 m
.79624 .25704 L
.67248 .32791 L
.80459 .20675 L
s
.79624 .25704 m
.78974 .29605 L
.65744 .3782 L
.79624 .25704 L
s
.78974 .29605 m
.78442 .32791 L
.64551 .41721 L
.78974 .29605 L
s
.78442 .32791 m
.77992 .35486 L
.6356 .44908 L
.78442 .32791 L
s
.77992 .35486 m
.77602 .3782 L
.62714 .47602 L
.77992 .35486 L
s
.72509 .13588 m
.69286 .25704 L
.53423 .25704 L
.72509 .13588 L
s
.69286 .25704 m
.67248 .32791 L
.40147 .35486 L
.69286 .25704 L
s
.67248 .32791 m
.65744 .3782 L
.44516 .41721 L
.67248 .32791 L
s
.65744 .3782 m
.64551 .41721 L
.43699 .46307 L
.65744 .3782 L
s
.64551 .41721 m
.6356 .44908 L
.42639 .49936 L
.64551 .41721 L
s
.6356 .44908 m
.62714 .47602 L
.4161 .5294 L
.6356 .44908 L
s
.53423 .25704 m
.40147 .35486 L
.50317 .35486 L
.53423 .25704 L
s
.40147 .35486 m
.44516 .41721 L
.45739 .43387 L
.40147 .35486 L
s
.44516 .41721 m
.43699 .46307 L
.42563 .48808 L
.44516 .41721 L
s
.43699 .46307 m
.42639 .49936 L
.40101 .5294 L
.43699 .46307 L
s
.42639 .49936 m
.4161 .5294 L
.3808 .5628 L
.42639 .49936 L
s
.50317 .35486 m
.45739 .43387 L
.37672 .43387 L
.50317 .35486 L
s
.45739 .43387 m
.42563 .48808 L
.3205 .49936 L
.45739 .43387 L
s
.42563 .48808 m
.40101 .5294 L
.28172 .5469 L
.42563 .48808 L
s
.40101 .5294 m
.3808 .5628 L
.25162 .58423 L
.40101 .5294 L
s
.37672 .43387 m
.3205 .49936 L
.26525 .49936 L
.37672 .43387 L
s
.3205 .49936 m
.28172 .5469 L
.21251 .55503 L
.3205 .49936 L
s
.28172 .5469 m
.25162 .58423 L
.17318 .59718 L
.28172 .5469 L
s
.26525 .49936 m
.21251 .55503 L
.15963 .55503 L
.26525 .49936 L
s
.21251 .55503 m
.17318 .59718 L
.09217 .60332 L
.21251 .55503 L
s
.15963 .55503 m
.09217 .60332 L
.05404 .60332 L
.15963 .55503 L
s
0 1 0 r
.83602 .01472 m
.8163 .13588 L
.72509 .13588 L
.83602 .01472 L
s
.8163 .13588 m
.80459 .20675 L
.69286 .25704 L
.8163 .13588 L
s
.80459 .20675 m
.79624 .25704 L
.67248 .32791 L
.80459 .20675 L
s
.79624 .25704 m
.78974 .29605 L
.65744 .3782 L
.79624 .25704 L
s
.78974 .29605 m
.78442 .32791 L
.64551 .41721 L
.78974 .29605 L
s
.78442 .32791 m
.77992 .35486 L
.6356 .44908 L
.78442 .32791 L
s
.77992 .35486 m
.77602 .3782 L
.62714 .47602 L
.77992 .35486 L
s
.72509 .13588 m
.69286 .25704 L
.53423 .25704 L
.72509 .13588 L
s
.69286 .25704 m
.67248 .32791 L
.40147 .35486 L
.69286 .25704 L
s
.67248 .32791 m
.65744 .3782 L
.44516 .41721 L
.67248 .32791 L
s
.65744 .3782 m
.64551 .41721 L
.43699 .46307 L
.65744 .3782 L
s
.64551 .41721 m
.6356 .44908 L
.42639 .49936 L
.64551 .41721 L
s
.6356 .44908 m
.62714 .47602 L
.4161 .5294 L
.6356 .44908 L
s
.53423 .25704 m
.40147 .35486 L
.50317 .35486 L
.53423 .25704 L
s
.40147 .35486 m
.44516 .41721 L
.45739 .43387 L
.40147 .35486 L
s
.44516 .41721 m
.43699 .46307 L
.42563 .48808 L
.44516 .41721 L
s
.43699 .46307 m
.42639 .49936 L
.40101 .5294 L
.43699 .46307 L
s
.42639 .49936 m
.4161 .5294 L
.3808 .5628 L
.42639 .49936 L
s
.50317 .35486 m
.45739 .43387 L
.37673 .43387 L
.50317 .35486 L
s
.45739 .43387 m
.42563 .48808 L
.32049 .49936 L
.45739 .43387 L
s
.42563 .48808 m
.40101 .5294 L
.28178 .5469 L
.42563 .48808 L
s
.40101 .5294 m
.3808 .5628 L
.25142 .58423 L
.40101 .5294 L
s
.37673 .43387 m
.32049 .49936 L
.2653 .49936 L
.37673 .43387 L
s
.32049 .49936 m
.28178 .5469 L
.2115 .55503 L
.32049 .49936 L
s
.28178 .5469 m
.25142 .58423 L
.1787 .59718 L
.28178 .5469 L
s
.2653 .49936 m
.2115 .55503 L
.15118 .55503 L
.2653 .49936 L
s
.2115 .55503 m
.1787 .59718 L
.14487 .60332 L
.2115 .55503 L
s
.15118 .55503 m
.14487 .60332 L
.14385 .60332 L
.15118 .55503 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{398.812, 246.5},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"From In[27]:=",
 CellID->2078715690],

Cell["\<\
Approximately two decimal digits of accuracy are gained by using the \
increment-based formulation throughout the extrapolation process.\
\>", "Text",
 CellID->1747130215],

Cell["\<\
This compares the relative error in the integration data that forms the \
initial column of the extrapolation table for the previous example.\
\>", "Text",
 CellID->1633361135],

Cell["\<\
Reference values were computed using software arithmetic with 32 decimal \
digits and converted to the nearest IEEE double-precision floating-point \
numbers, where an ULP signifies a Unit in the Last Place or Unit in the Last \
Position.\
\>", "Text",
 CellID->47737542],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   FrameBox[GridBox[{
      {" ", 
       SubscriptBox["T", "11"], 
       SubscriptBox["T", "21"], 
       SubscriptBox["T", "31"], 
       SubscriptBox["T", "41"], 
       SubscriptBox["T", "51"], 
       SubscriptBox["T", "61"], 
       SubscriptBox["T", "71"], 
       SubscriptBox["T", "81"]},
      {
       RowBox[{"Standard", " ", "formulation"}], "0", 
       RowBox[{
        RowBox[{"-", "1"}], " ", "ULP"}], "0", 
       RowBox[{"1", " ", "ULP"}], "0", 
       RowBox[{"1.5", " ", "ULPs"}], "0", 
       RowBox[{"1", " ", "ULP"}]},
      {
       RowBox[{
        RowBox[{"Increment", " ", "formulation"}], "\[IndentingNewLine]", 
        RowBox[{
        "applied", " ", "to", " ", "the", " ", "base", " ", "method"}]}], "0",
        "0", "0", "0", 
       RowBox[{"1", " ", "ULP"}], "0", "0", 
       RowBox[{"1", " ", "ULP"}]}
     },
     GridBoxDividers->{
      "Columns" -> {False, {True}, False}, "ColumnsIndexed" -> {}, 
       "Rows" -> {False, {True}, False}, "RowsIndexed" -> {}}]], 
   TraditionalForm]]],
 Cell[BoxData[
  FormBox[" ", TraditionalForm]]]
}], "Text",
 CellID->31019581],

Cell[TextData[{
 "Notice that the rounding error model that was used to motivate the study of \
rounding error growth is limited because in practice errors in ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "1"}]], TraditionalForm]]],
 " can exceed 1 ULP."
}],
 CellID->182499215],

Cell[TextData[{
 "The increment formulation used throughout the extrapolation process \
produces rounding errors",
 " ",
 "in ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", 
    RowBox[{"i", ",", " ", "1"}]], TraditionalForm]]],
 "that are smaller than 1 ULP."
}], "Text",
 CellID->289940496]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method comparison", "Subsection",
 CellID->1825494783],

Cell[TextData[{
 "This compares the work required for extrapolation based on ",
 StyleBox["ExplicitEuler", "MR"],
 " (red), the ",
 StyleBox["ExplicitMidpoint", "MR"],
 " (blue), and ",
 StyleBox["ExplicitModifiedMidpoint", "MR"],
 " (green)."
}], "Text",
 CellID->278697044],

Cell["\<\
All computations are carried out using software arithmetic with 32 decimal \
digits.\
\>", "Text",
 CellID->2118523887],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10  scalefont  setfont
% Scaling calculations
0.97619 0.0418789 0.0147151 0.33522 [
[.01298 -0.0125 -23.4688 -14.375 ]
[.01298 -0.0125 23.4688 0 ]
[.18049 -0.0125 -23.4688 -14.375 ]
[.18049 -0.0125 23.4688 0 ]
[.34801 -0.0125 -23.4688 -14.375 ]
[.34801 -0.0125 23.4688 0 ]
[.51552 -0.0125 -23.4688 -14.375 ]
[.51552 -0.0125 23.4688 0 ]
[.68304 -0.0125 -21.5 -14.25 ]
[.68304 -0.0125 21.5 0 ]
[.85055 -0.0125 -14.375 -11.1875 ]
[.85055 -0.0125 14.375 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01472 -9.5 -5.59375 ]
[-0.0125 .01472 0 5.59375 ]
[-0.0125 .11563 -9.5 -5.59375 ]
[-0.0125 .11563 0 5.59375 ]
[-0.0125 .24902 -9.5 -5.59375 ]
[-0.0125 .24902 0 5.59375 ]
[-0.0125 .34993 -15 -5.59375 ]
[-0.0125 .34993 0 5.59375 ]
[-0.0125 .45085 -15 -5.59375 ]
[-0.0125 .45085 0 5.59375 ]
[-0.0125 .58424 -15 -5.59375 ]
[-0.0125 .58424 0 5.59375 ]
[ 0 0 -0.125 0 ]
[.01298 .63053 -23.4688 0 ]
[.01298 .63053 23.4688 14.375 ]
[.18049 .63053 -23.4688 0 ]
[.18049 .63053 23.4688 14.375 ]
[.34801 .63053 -23.4688 0 ]
[.34801 .63053 23.4688 14.375 ]
[.51552 .63053 -23.4688 0 ]
[.51552 .63053 23.4688 14.375 ]
[.68304 .63053 -21.5 0 ]
[.68304 .63053 21.5 14.25 ]
[.85055 .63053 -14.375 0 ]
[.85055 .63053 14.375 11.1875 ]
[ 0 .61803 .125 0 ]
[1.0125 .01472 0 -5.59375 ]
[1.0125 .01472 9.5 5.59375 ]
[1.0125 .11563 0 -5.59375 ]
[1.0125 .11563 9.5 5.59375 ]
[1.0125 .24902 0 -5.59375 ]
[1.0125 .24902 9.5 5.59375 ]
[1.0125 .34993 0 -5.59375 ]
[1.0125 .34993 15 5.59375 ]
[1.0125 .45085 0 -5.59375 ]
[1.0125 .45085 15 5.59375 ]
[1.0125 .58424 0 -5.59375 ]
[1.0125 .58424 15 5.59375 ]
[ 1 0 .125 0 ]
[.5 .63053 -105.219 14.375 ]
[.5 .63053 105.219 27.6875 ]
[ -0.0005 -0.0005 0 0 ]
[ 1.0005 .61853 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.01298 0 m
.01298 .00625 L
s
gsave
.01298 -0.0125 -84.4688 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(23) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.18049 0 m
.18049 .00625 L
s
gsave
.18049 -0.0125 -84.4688 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(19) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.34801 0 m
.34801 .00625 L
s
gsave
.34801 -0.0125 -84.4688 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(15) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.51552 0 m
.51552 .00625 L
s
gsave
.51552 -0.0125 -84.4688 -18.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(11) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.68304 0 m
.68304 .00625 L
s
gsave
.68304 -0.0125 -82.5 -18.25 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.25 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.188 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.188 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.188 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.188 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.188 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 9.875 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 9.875 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 9.875 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(7) show
100.125 9.875 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.85055 0 m
.85055 .00625 L
s
gsave
.85055 -0.0125 -75.375 -15.1875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.001) show
87.750 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.14054 0 m
.14054 .00375 L
s
.15314 0 m
.15314 .00375 L
s
.16051 0 m
.16051 .00375 L
s
.16574 0 m
.16574 .00375 L
s
.1698 0 m
.1698 .00375 L
s
.17312 0 m
.17312 .00375 L
s
.17592 0 m
.17592 .00375 L
s
.17835 0 m
.17835 .00375 L
s
.30806 0 m
.30806 .00375 L
s
.32066 0 m
.32066 .00375 L
s
.32803 0 m
.32803 .00375 L
s
.33326 0 m
.33326 .00375 L
s
.33732 0 m
.33732 .00375 L
s
.34063 0 m
.34063 .00375 L
s
.34344 0 m
.34344 .00375 L
s
.34586 0 m
.34586 .00375 L
s
.47557 0 m
.47557 .00375 L
s
.48817 0 m
.48817 .00375 L
s
.49554 0 m
.49554 .00375 L
s
.50078 0 m
.50078 .00375 L
s
.50483 0 m
.50483 .00375 L
s
.50815 0 m
.50815 .00375 L
s
.51095 0 m
.51095 .00375 L
s
.51338 0 m
.51338 .00375 L
s
.64309 0 m
.64309 .00375 L
s
.65569 0 m
.65569 .00375 L
s
.66306 0 m
.66306 .00375 L
s
.66829 0 m
.66829 .00375 L
s
.67235 0 m
.67235 .00375 L
s
.67566 0 m
.67566 .00375 L
s
.67847 0 m
.67847 .00375 L
s
.6809 0 m
.6809 .00375 L
s
.81061 0 m
.81061 .00375 L
s
.8232 0 m
.8232 .00375 L
s
.83058 0 m
.83058 .00375 L
s
.83581 0 m
.83581 .00375 L
s
.83986 0 m
.83986 .00375 L
s
.84318 0 m
.84318 .00375 L
s
.84598 0 m
.84598 .00375 L
s
.84841 0 m
.84841 .00375 L
s
.97812 0 m
.97812 .00375 L
s
.99072 0 m
.99072 .00375 L
s
.99809 0 m
.99809 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01472 m
.00625 .01472 L
s
gsave
-0.0125 .01472 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .11563 m
.00625 .11563 L
s
gsave
-0.0125 .11563 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(2) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .24902 m
.00625 .24902 L
s
gsave
-0.0125 .24902 -70.5 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(5) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .34993 m
.00625 .34993 L
s
gsave
-0.0125 .34993 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .45085 m
.00625 .45085 L
s
gsave
-0.0125 .45085 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(20) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .58424 m
.00625 .58424 L
s
gsave
-0.0125 .58424 -76 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(50) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .17466 m
.00375 .17466 L
s
0 .21654 m
.00375 .21654 L
s
0 .27557 m
.00375 .27557 L
s
0 .29801 m
.00375 .29801 L
s
0 .31745 m
.00375 .31745 L
s
0 .3346 m
.00375 .3346 L
s
0 .50988 m
.00375 .50988 L
s
0 .55176 m
.00375 .55176 L
s
0 .61079 m
.00375 .61079 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.01298 .61178 m
.01298 .61803 L
s
gsave
.01298 .63053 -84.4688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(23) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.18049 .61178 m
.18049 .61803 L
s
gsave
.18049 .63053 -84.4688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(19) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.34801 .61178 m
.34801 .61803 L
s
gsave
.34801 .63053 -84.4688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(15) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.51552 .61178 m
.51552 .61803 L
s
gsave
.51552 .63053 -84.4688 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.312 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.312 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.312 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 10.000 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(11) show
104.062 10.000 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.68304 .61178 m
.68304 .61803 L
s
gsave
.68304 .63053 -82.5 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 22.25 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 15.188 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
%%BeginResource: font Helvetica-MISO
%%BeginFont: Helvetica-MISO
/Helvetica /Helvetica-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 15.188 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
71.250 15.188 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
72.750 15.188 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
80.625 15.188 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
91.625 9.875 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
91.625 9.875 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
96.188 9.875 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(7) show
100.125 9.875 moveto
%%IncludeResource: font Helvetica-MISO
%%IncludeFont: Helvetica-MISO
/Helvetica-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.85055 .61178 m
.85055 .61803 L
s
gsave
.85055 .63053 -75.375 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.001) show
87.750 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.14054 .61428 m
.14054 .61803 L
s
.15314 .61428 m
.15314 .61803 L
s
.16051 .61428 m
.16051 .61803 L
s
.16574 .61428 m
.16574 .61803 L
s
.1698 .61428 m
.1698 .61803 L
s
.17312 .61428 m
.17312 .61803 L
s
.17592 .61428 m
.17592 .61803 L
s
.17835 .61428 m
.17835 .61803 L
s
.30806 .61428 m
.30806 .61803 L
s
.32066 .61428 m
.32066 .61803 L
s
.32803 .61428 m
.32803 .61803 L
s
.33326 .61428 m
.33326 .61803 L
s
.33732 .61428 m
.33732 .61803 L
s
.34063 .61428 m
.34063 .61803 L
s
.34344 .61428 m
.34344 .61803 L
s
.34586 .61428 m
.34586 .61803 L
s
.47557 .61428 m
.47557 .61803 L
s
.48817 .61428 m
.48817 .61803 L
s
.49554 .61428 m
.49554 .61803 L
s
.50078 .61428 m
.50078 .61803 L
s
.50483 .61428 m
.50483 .61803 L
s
.50815 .61428 m
.50815 .61803 L
s
.51095 .61428 m
.51095 .61803 L
s
.51338 .61428 m
.51338 .61803 L
s
.64309 .61428 m
.64309 .61803 L
s
.65569 .61428 m
.65569 .61803 L
s
.66306 .61428 m
.66306 .61803 L
s
.66829 .61428 m
.66829 .61803 L
s
.67235 .61428 m
.67235 .61803 L
s
.67566 .61428 m
.67566 .61803 L
s
.67847 .61428 m
.67847 .61803 L
s
.6809 .61428 m
.6809 .61803 L
s
.81061 .61428 m
.81061 .61803 L
s
.8232 .61428 m
.8232 .61803 L
s
.83058 .61428 m
.83058 .61803 L
s
.83581 .61428 m
.83581 .61803 L
s
.83986 .61428 m
.83986 .61803 L
s
.84318 .61428 m
.84318 .61803 L
s
.84598 .61428 m
.84598 .61803 L
s
.84841 .61428 m
.84841 .61803 L
s
.97812 .61428 m
.97812 .61803 L
s
.99072 .61428 m
.99072 .61803 L
s
.99809 .61428 m
.99809 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
gsave
1.0125 .01472 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .11563 m
1 .11563 L
s
gsave
1.0125 .11563 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(2) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .24902 m
1 .24902 L
s
gsave
1.0125 .24902 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(5) show
68.500 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .34993 m
1 .34993 L
s
gsave
1.0125 .34993 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .45085 m
1 .45085 L
s
gsave
1.0125 .45085 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(20) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.99375 .58424 m
1 .58424 L
s
gsave
1.0125 .58424 -61 -9.59375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.1875 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(50) show
74.000 12.125 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.99625 .17466 m
1 .17466 L
s
.99625 .21654 m
1 .21654 L
s
.99625 .27557 m
1 .27557 L
s
.99625 .29801 m
1 .29801 L
s
.99625 .31745 m
1 .31745 L
s
.99625 .3346 m
1 .3346 L
s
.99625 .50988 m
1 .50988 L
s
.99625 .55176 m
1 .55176 L
s
.99625 .61079 m
1 .61079 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
gsave
.5 .63053 -166.219 10.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Plot) show
86.875 13.688 moveto
(of) show
100.125 13.688 moveto
(work) show
128.625 13.688 moveto
(vs) show
143.938 13.688 moveto
(error) show
172.312 13.688 moveto
(on) show
188.875 13.688 moveto
(a) show
198.812 13.688 moveto
(log) show
214.688 13.688 moveto
%%IncludeResource: font Mathematica1
%%IncludeFont: Mathematica1
/Mathematica1 findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
222.375 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(log) show
241.562 13.688 moveto
(scale) show
269.438 13.688 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 12.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 .271 0 r
.5 Mabswid
.88951 .01472 m
.87732 .11563 L
.82092 .11563 L
.88951 .01472 L
s
.87732 .11563 m
.87008 .17466 L
.80099 .21654 L
.87732 .11563 L
s
.87008 .17466 m
.86492 .21654 L
.78838 .27557 L
.87008 .17466 L
s
.86492 .21654 m
.8609 .24902 L
.77909 .31745 L
.86492 .21654 L
s
.8609 .24902 m
.85761 .27557 L
.77171 .34993 L
.8609 .24902 L
s
.85761 .27557 m
.85482 .29801 L
.76558 .37648 L
.85761 .27557 L
s
.82092 .11563 m
.80099 .21654 L
.70289 .21654 L
.82092 .11563 L
s
.80099 .21654 m
.78838 .27557 L
.6208 .29801 L
.80099 .21654 L
s
.78838 .27557 m
.77909 .31745 L
.64782 .34993 L
.78838 .27557 L
s
.77909 .31745 m
.77171 .34993 L
.64277 .38813 L
.77909 .31745 L
s
.77171 .34993 m
.76558 .37648 L
.63621 .41836 L
.77171 .34993 L
s
.70289 .21654 m
.6208 .29801 L
.68369 .29801 L
.70289 .21654 L
s
.6208 .29801 m
.64782 .34993 L
.65538 .36381 L
.6208 .29801 L
s
.64782 .34993 m
.64277 .38813 L
.63574 .40896 L
.64782 .34993 L
s
.64277 .38813 m
.63621 .41836 L
.62051 .44338 L
.64277 .38813 L
s
.68369 .29801 m
.65538 .36381 L
.6055 .36381 L
.68369 .29801 L
s
.65538 .36381 m
.63574 .40896 L
.57073 .41836 L
.65538 .36381 L
s
.63574 .40896 m
.62051 .44338 L
.54676 .45795 L
.63574 .40896 L
s
.6055 .36381 m
.57073 .41836 L
.53657 .41836 L
.6055 .36381 L
s
.57073 .41836 m
.54676 .45795 L
.50386 .46472 L
.57073 .41836 L
s
.53657 .41836 m
.50386 .46472 L
.47047 .46472 L
.53657 .41836 L
s
.274 .51 .706 r
.82092 .11563 m
.79577 .21654 L
.69818 .24902 L
.82092 .11563 L
s
.79577 .21654 m
.78103 .27557 L
.6584 .3346 L
.79577 .21654 L
s
.78103 .27557 m
.77057 .31745 L
.63323 .38813 L
.78103 .27557 L
s
.77057 .31745 m
.76245 .34993 L
.61466 .42719 L
.77057 .31745 L
s
.76245 .34993 m
.75582 .37648 L
.59992 .45795 L
.76245 .34993 L
s
.75582 .37648 m
.75022 .39892 L
.58768 .48333 L
.75582 .37648 L
s
.69818 .24902 m
.6584 .3346 L
.57678 .34993 L
.69818 .24902 L
s
.6584 .3346 m
.63323 .38813 L
.52667 .41836 L
.6584 .3346 L
s
.63323 .38813 m
.61466 .42719 L
.49341 .46472 L
.63323 .38813 L
s
.61466 .42719 m
.59992 .45795 L
.46822 .49983 L
.61466 .42719 L
s
.59992 .45795 m
.58768 .48333 L
.44788 .5281 L
.59992 .45795 L
s
.57678 .34993 m
.52667 .41836 L
.4538 .42719 L
.57678 .34993 L
s
.52667 .41836 m
.49341 .46472 L
.39575 .48333 L
.52667 .41836 L
s
.49341 .46472 m
.46822 .49983 L
.3559 .52375 L
.49341 .46472 L
s
.46822 .49983 m
.44788 .5281 L
.32512 .55535 L
.46822 .49983 L
s
.4538 .42719 m
.39575 .48333 L
.33037 .48904 L
.4538 .42719 L
s
.39575 .48333 m
.3559 .52375 L
.26589 .53642 L
.39575 .48333 L
s
.3559 .52375 m
.32512 .55535 L
.22049 .5721 L
.3559 .52375 L
s
.33037 .48904 m
.26589 .53642 L
.20663 .54041 L
.33037 .48904 L
s
.26589 .53642 m
.22049 .5721 L
.13678 .5813 L
.26589 .53642 L
s
.20663 .54041 m
.13678 .5813 L
.08267 .58424 L
.20663 .54041 L
s
.486 .988 0 r
.80835 .17466 m
.78317 .24902 L
.6726 .29801 L
.80835 .17466 L
s
.78317 .24902 m
.76843 .29801 L
.63274 .36381 L
.78317 .24902 L
s
.76843 .29801 m
.75797 .3346 L
.60755 .40896 L
.76843 .29801 L
s
.75797 .3346 m
.74985 .36381 L
.58897 .44338 L
.75797 .3346 L
s
.74985 .36381 m
.74322 .38813 L
.57423 .47119 L
.74985 .36381 L
s
.74322 .38813 m
.73761 .40896 L
.56199 .49454 L
.74322 .38813 L
s
.6726 .29801 m
.63274 .36381 L
.54064 .38813 L
.6726 .29801 L
s
.63274 .36381 m
.60755 .40896 L
.49044 .44338 L
.63274 .36381 L
s
.60755 .40896 m
.58897 .44338 L
.45715 .48333 L
.60755 .40896 L
s
.58897 .44338 m
.57423 .47119 L
.43196 .51465 L
.58897 .44338 L
s
.57423 .47119 m
.56199 .49454 L
.41161 .54041 L
.57423 .47119 L
s
.54064 .38813 m
.49044 .44338 L
.41107 .45795 L
.54064 .38813 L
s
.49044 .44338 m
.45715 .48333 L
.35288 .50494 L
.49044 .44338 L
s
.45715 .48333 m
.43196 .51465 L
.313 .54041 L
.45715 .48333 L
s
.43196 .51465 m
.41161 .54041 L
.28221 .5689 L
.43196 .51465 L
s
.41107 .45795 m
.35288 .50494 L
.28181 .51465 L
.41107 .45795 L
s
.35288 .50494 m
.313 .54041 L
.21715 .55535 L
.35288 .50494 L
s
.313 .54041 m
.28221 .5689 L
.17171 .58713 L
.313 .54041 L
s
.28181 .51465 m
.21715 .55535 L
.15268 .56229 L
.28181 .51465 L
s
.21715 .55535 m
.17171 .58713 L
.08262 .59812 L
.21715 .55535 L
s
.15268 .56229 m
.08262 .59812 L
.02381 .60332 L
.15268 .56229 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{401, 247.75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"From In[141]:=",
 CellID->1322893152]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Fine-tuning", "Section",
 CellTags->"s:6",
 CellID->747460645],

Cell[CellGroupData[{

Cell["StepSizeSafetyFactors", "Subsection",
 CellID->1679392631],

Cell[TextData[{
 "As with most methods, you want a balance between taking too small a step \
and trying to take too big a step that will be frequently rejected. The \
option ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["StepSizeSafetyFactors", "MR"], "->", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["s", "1"], ",", 
      SubscriptBox["s", "2"]}], "}"}]}], TraditionalForm]]],
 " constrains the choice of step size as follows. The step size chosen by the \
method for order ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]]],
 " satisfies:"
}], "Text",
 CellID->331413612],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["h", 
    RowBox[{"n", "+", "1"}]], "=", " ", 
   RowBox[{
    SubscriptBox["h", "n"], 
    SubscriptBox["s", "1"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["s", "2"], 
       FormBox[
        FractionBox[
         StyleBox["Tol",
          FontSlant->"Italic"], 
         RowBox[{"\[LeftDoubleBracketingBar]", 
          SubscriptBox[
           StyleBox["err",
            FontSlant->"Italic"], "n"], "\[RightDoubleBracketingBar]"}]],
        TraditionalForm]}], ")"}], 
     FractionBox["1", 
      RowBox[{"p", "+", "1"}]]]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"ExtrapolationStepControl",
 CellID->1850438205],

Cell["\<\
This includes both an order-dependent factor and an order-independent factor.\
\
\>", "Text",
 CellID->1408774815]
}, Open  ]],

Cell[CellGroupData[{

Cell["StepSizeRatioBounds", "Subsection",
 CellID->1862773728],

Cell[TextData[{
 "The option ",
 StyleBox["StepSizeRatioBounds\[Rule]{", "MR"],
 StyleBox[Cell[BoxData[
  FormBox[
   SubscriptBox["sr", "min"], TraditionalForm]], "MR"], "MR"],
 StyleBox[",", "MR"],
 StyleBox[Cell[BoxData[
  FormBox[
   SubscriptBox["sr", "max"], TraditionalForm]], "MR"], "MR"],
 StyleBox["}", "MR"],
 " specifies bounds on the next step size to take such that:"
}], "Text",
 CellID->744775946],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox[
    StyleBox["sr",
     FontSlant->"Italic"], 
    StyleBox["min",
     FontSlant->"Italic"]], "\[LessEqual]", 
   RowBox[{"\[LeftBracketingBar]", 
    FractionBox[
     SubscriptBox["h", 
      RowBox[{"n", "+", "1"}]], 
     SubscriptBox["h", "n"]], "\[RightBracketingBar]"}], "\[LessEqual]", 
   SubscriptBox[
    StyleBox["sr",
     FontSlant->"Italic"], 
    StyleBox["max",
     FontSlant->"Italic"]]}], TraditionalForm]], "DisplayMath",
 CellID->1944662996]
}, Open  ]],

Cell[CellGroupData[{

Cell["OrderSafetyFactors", "Subsection",
 CellID->131445579],

Cell[TextData[{
 "An important aspect in ",
 StyleBox["Extrapolation", "MR"],
 " is the choice of order. "
}], "Text",
 CellID->125173308],

Cell[TextData[{
 "Each extrapolation of order ",
 Cell[BoxData[
  FormBox["k", TraditionalForm]]],
 " has an associated work estimate ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[ScriptCapitalA]", "k"], TraditionalForm]]],
 " which is based on, for example, the number of function evaluations of the \
base method and the step sequence used."
}], "Text",
 CellID->2145494079],

Cell["\<\
The work estimate for explicit base methods is based on the number of \
function evaluations and the step sequence used.\
\>", "Text",
 CellID->574854330],

Cell["\<\
The work estimate for linearly implicit base methods also includes an \
estimate of the cost of evaluating the Jacobian, the cost of an LU \
decomposition, and the cost of back solving the linear equations.\
\>", "Text",
 CellID->536965926],

Cell[TextData[{
 "Estimates for the work per unit step are formed from the work estimate ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[ScriptCapitalA]", "k"], TraditionalForm]]],
 " and the expected new step size to take for a method of order ",
 Cell[BoxData[
  FormBox["k", TraditionalForm]]],
 " (computed from (",
 
 CounterBox["NumberedEquation", "ExtrapolationStepControl"],
 ")): ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[ScriptCapitalW]", "k"], " ", "=", " ", 
    RowBox[{
     SubscriptBox["\[ScriptCapitalA]", "k"], "/", " ", 
     SubsuperscriptBox["h", 
      RowBox[{"n", "+", "1"}], "k"]}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->842830764],

Cell[TextData[{
 "Comparing consecutive estimates, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[ScriptCapitalW]", "k"], " "}], TraditionalForm]]],
 " allows a decision about when a different order method will be more \
efficient."
}], "Text",
 CellID->1823507348],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["OrderSafetyFactors", "MR"], "->", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["f", "1"], ",", " ", 
      SubscriptBox["f", "2"]}], "}"}]}], TraditionalForm]]],
 " specifies safety factors to be included in the comparison of estimates ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[ScriptCapitalW]", "k"], TraditionalForm]]],
 "."
}], "Text",
 CellID->1677945736],

Cell[TextData[{
 "An order decrease is made when ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["\[ScriptCapitalW]", 
      RowBox[{"k", "-", "1"}]], "<"}], " "}], TraditionalForm]]],
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["f", "1"], 
    SubscriptBox["\[ScriptCapitalW]", "k"]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->231156755],

Cell[TextData[{
 "An order increase is made when ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[ScriptCapitalW]", 
     RowBox[{"k", "+", "1"}]], "<", " ", 
    FormBox[
     RowBox[{
      SubscriptBox["f", "2"], 
      SubscriptBox["\[ScriptCapitalW]", "k"]}],
     TraditionalForm]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1040135649],

Cell["\<\
There are some additional restrictions, such as when the maximal order \
increase per step is one (two for symmetric methods), and when an increase in \
order is prevented immediately after a rejected step.\
\>", "Text",
 CellID->1786714988],

Cell["\<\
For a nonstiff base method the default values are {4/5, 9/10} whereas for a \
stiff method they are {7/10, 9/10}.\
\>", "Text",
 CellID->980750218]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:7",
 CellID->971367691],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{"option", " ", "name"}], "TableHeader"], 
    StyleBox[
     RowBox[{"default", " ", "value"}], "TableHeader"], ""},
   {
    StyleBox["ExtrapolationSequence", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies the sequence to use in extrapolation", "TableText"]},
   {
    StyleBox["MaxDifferenceOrder", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies the maximum order to use", "TableText"]},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    StyleBox["ExplicitModifiedMidpoint", "InlineCode"], Cell[
    "specifies the base integration method to use", "TableText"]},
   {
    StyleBox["MinDifferenceOrder", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies the minimum order to use", "TableText"]},
   {
    StyleBox["OrderSafetyFactors", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
specifies the safety factors to use in the estimates for adaptive order \
selection\
\>", "TableText"]},
   {
    StyleBox["StartingDifferenceOrder", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies the initial order to use", "TableText"]},
   {
    StyleBox["StepSizeRatioBounds", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "specifies the bounds on a relative change in the new step size ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["h",
        FontSlant->"Italic"], 
       RowBox[{
        StyleBox["n",
         FontSlant->"Italic"], "+", "1"}]]], "InlineFormula"],
     " from the current step size ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["h",
        FontSlant->"Italic"], 
       StyleBox["n",
        FontSlant->"Italic"]]], "InlineFormula"],
     " as low <= ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["h",
        FontSlant->"Italic"], 
       RowBox[{
        StyleBox["n",
         FontSlant->"Italic"], "+", "1"}]]], "InlineFormula"],
     Cell[BoxData["/"], "InlineFormula"],
     Cell[BoxData[
      SubscriptBox[
       StyleBox["h",
        FontSlant->"Italic"], 
       StyleBox["n",
        FontSlant->"Italic"]]], "InlineFormula"],
     " <= high"
    }], "TableText"]},
   {
    StyleBox["StepSizeSafetyFactors", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
specifies the safety factors to incorporate into the error estimate used for \
adaptive step sizes\
\>", "TableText"]},
   {
    StyleBox["StiffnessTest", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies whether to use the stiffness detection capability", 
     "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->1386361555],

Cell[TextData[{
 "Options of the method ",
 StyleBox["Extrapolation", "MR"],
 "."
}], "Caption",
 CellID->466627705],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["ExtrapolationSequence", "MR"],
 " selects a sequence based on the stiffness and symmetry of the base \
method."
}], "Text",
 CellID->37295877],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["MaxDifferenceOrder", "MR"],
 " bounds the maximum order by two times the decimal working precision."
}], "Text",
 CellID->704418088],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["MinDifferenceOrder", "MR"],
 " selects the minimum number of two extrapolations starting from the order \
of the base method. This also depends on whether the base method is \
symmetric."
}], "Text",
 CellID->497606648],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["OrderSafetyFactors", "MR"],
 " uses the values ",
 StyleBox["{7/10,9/10}", "MR"],
 " for a stiff base method and ",
 StyleBox["{4/5,9/10}", "MR"],
 " for a non-stiff base method."
}], "Text",
 CellID->32145147],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StartingDifferenceOrder", "MR"],
 " depends on the setting of ",
 StyleBox["MinDifferenceOrder", "MR"],
 " ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["p", "min"], TraditionalForm]]],
 ". It is set to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["p", "min"], "+", "1"}], TraditionalForm]]],
 " or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["p", "min"], "+", "2"}], TraditionalForm]]],
 " depending on whether the base method is symmetric."
}], "Text",
 CellID->790341517],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StepSizeRatioBounds", "MR"],
 " uses the values ",
 StyleBox["{1/10,4}", "MR"],
 " for a stiff base method and ",
 StyleBox["{1/50,4}", "MR"],
 " for a non-stiff base method."
}], "Text",
 CellID->156208096],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StepSizeSafetyFactors", "MR"],
 " uses the values ",
 StyleBox["{9/10,4/5}", "MR"],
 " for a stiff base method and ",
 StyleBox["{9/10,13/20}", "MR"],
 " for a non-stiff base method."
}], "Text",
 CellID->352536813],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StiffnessTest", "MR"],
 " indicates that the stiffness test is activate if a non-stiff base method \
is used."
}], "Text",
 CellID->9411404],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{"option", " ", "name"}], "TableHeader"], 
    StyleBox[
     RowBox[{"default", " ", "value"}], "TableHeader"], ""},
   {
    StyleBox["StabilityCheck", "InlineCode"], 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell[TextData[{
     "specifies whether to carry out a stability check on consecutive \
implicit solutions (see e.g. (",
     
     CounterBox["NumberedEquation", "ImplicitEulerStabilityCheck"],
     "))"
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->1019105233],

Cell[TextData[{
 "Options of the method ",
 StyleBox["LinearlyImplicitEuler", "MR"],
 ", ",
 StyleBox["LinearlyImplicitMidpoint", "MR"],
 ", and ",
 StyleBox["LinearlyImplicitModifiedMidpoint", "MR"],
 "."
}], "Caption",
 CellID->1417381136]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ExplicitRungeKutta Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveEventLocator"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveFixedStep"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 48.8682080}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "Extrapolation methods are a class of arbitrary-order methods with \
automatic order and step size control. The error estimate comes from \
computing a solution over an interval using the same method with a varying \
number of steps and using extrapolation on the polynomial that fits through \
the computed solutions, giving a composite higher-order method [BS64]. At the \
same time, the polynomials give a means of error estimation. Typically, for \
low precision, the extrapolation methods have not been competitive with \
Runge-Kutta-type methods. For high precision, however, the arbitrary order \
means that they can be arbitrarily faster than fixed-order methods for very \
precise tolerances. The order and step size control is based on the codes \
odex.f and seulex.f described in [HNW93] and [HW96].", "synonyms" -> {}, 
    "title" -> "ExplicitRungeKutta Method for NDSolve", "type" -> "Tutorial", 
    "uri" -> "tutorial/NDSolveExtrapolation"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[728, 29, 92, 2, 70, "Title",
   CellTags->"c:1",
   CellID->269234798]},
 "s:1"->{
  Cell[845, 35, 69, 2, 70, "Section",
   CellTags->"s:1",
   CellID->998382023]},
 "s:2"->{
  Cell[2734, 99, 70, 2, 70, "Section",
   CellTags->"s:2",
   CellID->170762999]},
 "AitkenNeville"->{
  Cell[6195, 225, 1041, 31, 70, "NumberedEquation",
   CellTags->"AitkenNeville",
   CellID->1289539621]},
 "s:3"->{
  Cell[18165, 650, 70, 2, 70, "Section",
   CellTags->"s:3",
   CellID->1712532361]},
 "EulerSteps"->{
  Cell[20585, 753, 1287, 44, 70, "NumberedEquation",
   CellTags->"EulerSteps",
   CellID->961228871]},
 "ExplicitRungeKuttaMethod"->{
  Cell[23008, 847, 1139, 37, 70, "NumberedEquation",
   CellTags->"ExplicitRungeKuttaMethod",
   CellID->501375888]},
 "ExplicitRungeKuttaCoefficients"->{
  Cell[24249, 889, 836, 24, 70, "NumberedEquation",
   CellTags->"ExplicitRungeKuttaCoefficients",
   CellID->52120590]},
 "EulerIncrements"->{
  Cell[25811, 947, 1648, 50, 70, "NumberedEquation",
   CellTags->"EulerIncrements",
   CellID->903017574]},
 "EulerIncrementResult"->{
  Cell[28628, 1051, 283, 9, 70, "NumberedEquation",
   CellTags->"EulerIncrementResult",
   CellID->1242311561]},
 "LinearlyImplicitEulerSteps"->{
  Cell[29972, 1102, 2014, 67, 70, "NumberedEquation",
   CellTags->"LinearlyImplicitEulerSteps",
   CellID->609222856]},
 "LinearlyImplicitEulerIncrements"->{
  Cell[33747, 1249, 2095, 66, 70, "NumberedEquation",
   CellTags->"LinearlyImplicitEulerIncrements",
   CellID->1443851841]},
 "ExplicitMidpointSteps"->{
  Cell[37049, 1371, 1314, 44, 70, "NumberedEquation",
   CellTags->"ExplicitMidpointSteps",
   CellID->1436142165]},
 "ExplicitMidpointIncrements"->{
  Cell[39200, 1453, 1981, 59, 70, "NumberedEquation",
   CellTags->"ExplicitMidpointIncrements",
   CellID->349248116]},
 "GraggSmoothingStep"->{
  Cell[41493, 1529, 491, 17, 70, "NumberedEquation",
   CellTags->"GraggSmoothingStep",
   CellID->367927996]},
 "ModifiedGraggSmoothingStep"->{
  Cell[42784, 1575, 723, 22, 70, "NumberedEquation",
   CellTags->"ModifiedGraggSmoothingStep",
   CellID->2120609443]},
 "LinearlyImplicitMidpointSteps"->{
  Cell[85964, 4598, 2862, 92, 70, "NumberedEquation",
   CellTags->"LinearlyImplicitMidpointSteps",
   CellID->999705685]},
 "LinearlyImplicitMidpointIncrements"->{
  Cell[89586, 4726, 2923, 90, 70, "NumberedEquation",
   CellTags->"LinearlyImplicitMidpointIncrements",
   CellID->723795696]},
 "BaderSmoothingStep"->{
  Cell[92841, 4834, 426, 16, 70, "NumberedEquation",
   CellTags->"BaderSmoothingStep",
   CellID->1460682982]},
 "ModifiedBaderSmoothingStep"->{
  Cell[93463, 4862, 708, 23, 70, "NumberedEquation",
   CellTags->"ModifiedBaderSmoothingStep",
   CellID->972989670]},
 "ModifiedAitkenNeville"->{
  Cell[96767, 4991, 1109, 31, 70, "NumberedEquation",
   CellTags->"ModifiedAitkenNeville",
   CellID->1005920553]},
 "s:4"->{
  Cell[99300, 5080, 78, 2, 70, "Section",
   CellTags->"s:4",
   CellID->777991017]},
 "ImplicitEulerStabilityCheck"->{
  Cell[103486, 5242, 957, 30, 70, "NumberedEquation",
   CellTags->"ImplicitEulerStabilityCheck",
   CellID->2027252648]},
 "s:5"->{
  Cell[106187, 5337, 66, 2, 70, "Section",
   CellTags->"s:5",
   CellID->2086327477]},
 "s:6"->{
  Cell[289107, 12182, 68, 2, 70, "Section",
   CellTags->"s:6",
   CellID->747460645]},
 "ExtrapolationStepControl"->{
  Cell[289862, 12211, 721, 24, 70, "NumberedEquation",
   CellTags->"ExtrapolationStepControl",
   CellID->1850438205]},
 "s:7"->{
  Cell[295418, 12426, 71, 2, 70, "Section",
   CellTags->"s:7",
   CellID->971367691]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 305508, 12742},
 {"s:1", 305598, 12746},
 {"s:2", 305690, 12750},
 {"AitkenNeville", 305793, 12754},
 {"s:3", 305910, 12758},
 {"EulerSteps", 306013, 12762},
 {"ExplicitRungeKuttaMethod", 306148, 12766},
 {"ExplicitRungeKuttaCoefficients", 306303, 12770},
 {"EulerIncrements", 306447, 12774},
 {"EulerIncrementResult", 306583, 12778},
 {"LinearlyImplicitEulerSteps", 306730, 12782},
 {"LinearlyImplicitEulerIncrements", 306889, 12786},
 {"ExplicitMidpointSteps", 307044, 12790},
 {"ExplicitMidpointIncrements", 307194, 12794},
 {"GraggSmoothingStep", 307340, 12798},
 {"ModifiedGraggSmoothingStep", 307485, 12802},
 {"LinearlyImplicitMidpointSteps", 307642, 12806},
 {"LinearlyImplicitMidpointIncrements", 307807, 12810},
 {"BaderSmoothingStep", 307961, 12814},
 {"ModifiedBaderSmoothingStep", 308107, 12818},
 {"ModifiedAitkenNeville", 308255, 12822},
 {"s:4", 308382, 12826},
 {"ImplicitEulerStabilityCheck", 308502, 12830},
 {"s:5", 308635, 12834},
 {"s:6", 308733, 12838},
 {"ExtrapolationStepControl", 308852, 12842},
 {"s:7", 308983, 12846}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 92, 2, 70, "Title",
 CellTags->"c:1",
 CellID->269234798],
Cell[CellGroupData[{
Cell[845, 35, 69, 2, 70, "Section",
 CellTags->"s:1",
 CellID->998382023],
Cell[917, 39, 583, 11, 70, "Text",
 CellID->1938414111],
Cell[1503, 52, 308, 6, 70, "Text",
 CellID->1589095302],
Cell[1814, 60, 407, 15, 70, "Text",
 CellID->992679970],
Cell[2224, 77, 159, 4, 70, "Text",
 CellID->723849525],
Cell[2386, 83, 311, 11, 70, "Input",
 InitializationCell->True,
 CellID->272055467]
}, Open  ]],
Cell[CellGroupData[{
Cell[2734, 99, 70, 2, 70, "Section",
 CellTags->"s:2",
 CellID->170762999],
Cell[2807, 103, 349, 10, 70, "Text",
 CellID->1049443999],
Cell[3159, 115, 435, 8, 70, "Text",
 CellID->462117005],
Cell[3597, 125, 67, 1, 70, "Text",
 CellID->1744473514],
Cell[3667, 128, 435, 14, 70, "DisplayMath",
 CellID->1304279132],
Cell[4105, 144, 195, 7, 70, "Text",
 CellID->1637667014],
Cell[4303, 153, 262, 7, 70, "DisplayMath",
 CellID->593348802],
Cell[4568, 162, 76, 1, 70, "Text",
 CellID->2031094749],
Cell[4647, 165, 258, 7, 70, "DisplayMath",
 CellID->1397129139],
Cell[4908, 174, 38, 1, 70, "Text",
 CellID->553099557],
Cell[4949, 177, 287, 9, 70, "DisplayMath",
 CellID->69213927],
Cell[5239, 188, 400, 14, 70, "Text",
 CellID->648444595],
Cell[5642, 204, 405, 13, 70, "DisplayMath",
 CellID->1695389189],
Cell[6050, 219, 142, 4, 70, "Text",
 CellID->2050595844],
Cell[6195, 225, 1041, 31, 70, "NumberedEquation",
 CellTags->"AitkenNeville",
 CellID->1289539621],
Cell[7239, 258, 107, 3, 70, "Text",
 CellID->1982698690],
Cell[7349, 263, 1053, 24, 70, "DisplayMath",
 CellID->823382723],
Cell[8405, 289, 327, 14, 70, "Text",
 CellID->1965840171],
Cell[CellGroupData[{
Cell[8757, 307, 66, 1, 70, "Subsection",
 CellID->1973820076],
Cell[8826, 310, 148, 4, 70, "Text",
 CellID->655573988],
Cell[8977, 316, 66, 1, 70, "Text",
 CellID->1634522524],
Cell[CellGroupData[{
Cell[9068, 321, 156, 4, 70, "Input",
 CellID->771343838],
Cell[9227, 327, 300, 9, 36, "Output",
 CellID->736392212]
}, Open  ]],
Cell[9542, 339, 67, 1, 70, "Text",
 CellID->1060157066],
Cell[CellGroupData[{
Cell[9634, 344, 158, 4, 70, "Input",
 CellID->1040578636],
Cell[9795, 350, 296, 9, 36, "Output",
 CellID->1452765585]
}, Open  ]],
Cell[10106, 362, 66, 1, 70, "Text",
 CellID->851087529],
Cell[CellGroupData[{
Cell[10197, 367, 157, 4, 70, "Input",
 CellID->482190291],
Cell[10357, 373, 293, 9, 36, "Output",
 CellID->1284538269]
}, Open  ]],
Cell[10665, 385, 516, 17, 70, "Text",
 CellID->1574937055],
Cell[11184, 404, 138, 4, 70, "Text",
 CellID->754245758],
Cell[CellGroupData[{
Cell[11347, 412, 179, 4, 70, "Input",
 CellID->107458757],
Cell[11529, 418, 295, 9, 36, "Output",
 CellID->20674660]
}, Open  ]],
Cell[11839, 430, 168, 4, 70, "Text",
 CellID->1216362763],
Cell[12010, 436, 424, 13, 70, "Input",
 CellID->1851932271]
}, Open  ]],
Cell[CellGroupData[{
Cell[12471, 454, 70, 1, 70, "Subsection",
 CellID->1781176538],
Cell[12544, 457, 240, 7, 70, "Text",
 CellID->706546129],
Cell[12787, 466, 211, 7, 70, "Text",
 CellID->297797426],
Cell[13001, 475, 616, 21, 70, "Text",
 CellID->1664787639],
Cell[13620, 498, 425, 17, 70, "Text",
 CellID->2103675269],
Cell[14048, 517, 268, 7, 70, "Text",
 CellID->850896983],
Cell[14319, 526, 1377, 33, 70, "DisplayMath",
 CellID->1734001566],
Cell[15699, 561, 158, 4, 70, "Text",
 CellID->604636216],
Cell[15860, 567, 481, 18, 70, "Text",
 CellID->398802525]
}, Open  ]],
Cell[CellGroupData[{
Cell[16378, 590, 67, 1, 70, "Subsection",
 CellID->1637113763],
Cell[16448, 593, 164, 4, 70, "Text",
 CellID->1894175838],
Cell[16615, 599, 397, 10, 70, "Text",
 CellID->234017567],
Cell[17015, 611, 1101, 33, 70, "Text",
 CellID->313012838]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[18165, 650, 70, 2, 70, "Section",
 CellTags->"s:3",
 CellID->1712532361],
Cell[18238, 654, 133, 4, 70, "Text",
 CellID->60353995],
Cell[18374, 660, 83, 1, 70, "BulletedText",
 CellID->968586826],
Cell[18460, 663, 87, 1, 70, "BulletedText",
 CellID->1199571237],
Cell[18550, 666, 132, 5, 70, "BulletedText",
 CellID->831697833],
Cell[18685, 673, 92, 1, 70, "BulletedText",
 CellID->1061704238],
Cell[18780, 676, 157, 4, 70, "BulletedText",
 CellID->112244655],
Cell[CellGroupData[{
Cell[18962, 684, 163, 4, 70, "BulletedText",
 CellID->1872276470],
Cell[19128, 690, 280, 10, 70, "Text",
 CellID->5720025],
Cell[19411, 702, 228, 8, 70, "Text",
 CellID->963332185],
Cell[19642, 712, 309, 9, 70, "Text",
 CellID->206983113]
}, Open  ]],
Cell[CellGroupData[{
Cell[19988, 726, 63, 1, 70, "Subsection",
 CellID->2089899431],
Cell[20054, 729, 528, 22, 70, "Text",
 CellID->390785592],
Cell[20585, 753, 1287, 44, 70, "NumberedEquation",
 CellTags->"EulerSteps",
 CellID->961228871],
Cell[21875, 799, 246, 11, 70, "Text",
 CellID->49874541]
}, Open  ]],
Cell[CellGroupData[{
Cell[22158, 815, 90, 1, 70, "Subsection",
 CellID->867682998],
Cell[22251, 818, 519, 16, 70, "Text",
 CellID->30107986],
Cell[22773, 836, 232, 9, 70, "Text",
 CellID->1901806812],
Cell[23008, 847, 1139, 37, 70, "NumberedEquation",
 CellTags->"ExplicitRungeKuttaMethod",
 CellID->501375888],
Cell[24150, 886, 96, 1, 70, "Text",
 CellID->736654116],
Cell[24249, 889, 836, 24, 70, "NumberedEquation",
 CellTags->"ExplicitRungeKuttaCoefficients",
 CellID->52120590]
}, Open  ]],
Cell[CellGroupData[{
Cell[25122, 918, 54, 1, 70, "Subsection",
 CellID->18679880],
Cell[25179, 921, 629, 24, 70, "Text",
 CellID->692845737],
Cell[25811, 947, 1648, 50, 70, "NumberedEquation",
 CellTags->"EulerIncrements",
 CellID->903017574],
Cell[27462, 999, 370, 16, 70, "Text",
 CellID->1752681446],
Cell[27835, 1017, 386, 17, 70, "Text",
 CellID->1297481953],
Cell[28224, 1036, 401, 13, 70, "Text",
 CellID->1301820099],
Cell[28628, 1051, 283, 9, 70, "NumberedEquation",
 CellTags->"EulerIncrementResult",
 CellID->1242311561],
Cell[28914, 1062, 694, 25, 70, "Text",
 CellID->1044174143]
}, Open  ]],
Cell[CellGroupData[{
Cell[29645, 1092, 81, 1, 70, "Subsection",
 CellID->1535837666],
Cell[29729, 1095, 240, 5, 70, "Text",
 CellID->1555324399],
Cell[29972, 1102, 2014, 67, 70, "NumberedEquation",
 CellTags->"LinearlyImplicitEulerSteps",
 CellID->609222856],
Cell[31989, 1171, 248, 12, 70, "Text",
 CellID->1596191661],
Cell[32240, 1185, 338, 13, 70, "DisplayMath",
 CellID->376427577],
Cell[32581, 1200, 416, 13, 70, "Text",
 CellID->1808794093],
Cell[33000, 1215, 163, 6, 70, "Text",
 CellID->280405846],
Cell[33166, 1223, 267, 9, 70, "DisplayMath",
 CellID->1438131815]
}, Open  ]],
Cell[CellGroupData[{
Cell[33470, 1237, 56, 1, 70, "Subsection",
 CellID->1921038749],
Cell[33529, 1240, 215, 7, 70, "Text",
 CellID->5198668],
Cell[33747, 1249, 2095, 66, 70, "NumberedEquation",
 CellTags->"LinearlyImplicitEulerIncrements",
 CellID->1443851841],
Cell[35845, 1317, 313, 13, 70, "Text",
 CellID->736887286],
Cell[36161, 1332, 276, 12, 70, "Text",
 CellID->212181781]
}, Open  ]],
Cell[CellGroupData[{
Cell[36474, 1349, 75, 1, 70, "Subsection",
 CellID->1386267624],
Cell[36552, 1352, 323, 11, 70, "Text",
 CellID->1127672878],
Cell[36878, 1365, 168, 4, 70, "Text",
 CellID->1173410652],
Cell[37049, 1371, 1314, 44, 70, "NumberedEquation",
 CellTags->"ExplicitMidpointSteps",
 CellID->1436142165],
Cell[38366, 1417, 552, 20, 70, "Text",
 CellID->1995355876]
}, Open  ]],
Cell[CellGroupData[{
Cell[38955, 1442, 56, 1, 70, "Subsection",
 CellID->1203518639],
Cell[39014, 1445, 183, 6, 70, "Text",
 CellID->1190465330],
Cell[39200, 1453, 1981, 59, 70, "NumberedEquation",
 CellTags->"ExplicitMidpointIncrements",
 CellID->349248116]
}, Open  ]],
Cell[CellGroupData[{
Cell[41218, 1517, 65, 1, 70, "Subsection",
 CellID->1940172755],
Cell[41286, 1520, 204, 7, 70, "Text",
 CellID->1242585920],
Cell[41493, 1529, 491, 17, 70, "NumberedEquation",
 CellTags->"GraggSmoothingStep",
 CellID->367927996],
Cell[41987, 1548, 558, 18, 70, "Text",
 CellID->485534844],
Cell[42548, 1568, 233, 5, 70, "Text",
 CellID->2145603494],
Cell[42784, 1575, 723, 22, 70, "NumberedEquation",
 CellTags->"ModifiedGraggSmoothingStep",
 CellID->2120609443],
Cell[43510, 1599, 543, 17, 70, "Text",
 CellID->1783016797],
Cell[44056, 1618, 344, 9, 70, "Text",
 CellID->1299939945],
Cell[44403, 1629, 523, 21, 70, "Text",
 CellID->1215437356]
}, Open  ]],
Cell[CellGroupData[{
Cell[44963, 1655, 59, 1, 70, "Subsection",
 CellID->679707293],
Cell[45025, 1658, 217, 6, 70, "Text",
 CellID->688293963],
Cell[45245, 1666, 40265, 2911, 70, "Graphics",
 CellID->623278108]
}, Open  ]],
Cell[CellGroupData[{
Cell[85547, 4582, 84, 1, 70, "Subsection",
 CellID->1288930149],
Cell[85634, 4585, 327, 11, 70, "Text",
 CellID->137001976],
Cell[85964, 4598, 2862, 92, 70, "NumberedEquation",
 CellTags->"LinearlyImplicitMidpointSteps",
 CellID->999705685],
Cell[88829, 4692, 461, 18, 70, "Text",
 CellID->950240595]
}, Open  ]],
Cell[CellGroupData[{
Cell[89327, 4715, 55, 1, 70, "Subsection",
 CellID->417407823],
Cell[89385, 4718, 198, 6, 70, "Text",
 CellID->290353853],
Cell[89586, 4726, 2923, 90, 70, "NumberedEquation",
 CellTags->"LinearlyImplicitMidpointIncrements",
 CellID->723795696]
}, Open  ]],
Cell[CellGroupData[{
Cell[92546, 4821, 56, 1, 70, "Subsection",
 CellID->422606491],
Cell[92605, 4824, 233, 8, 70, "Text",
 CellID->1734205694],
Cell[92841, 4834, 426, 16, 70, "NumberedEquation",
 CellTags->"BaderSmoothingStep",
 CellID->1460682982],
Cell[93270, 4852, 190, 8, 70, "Text",
 CellID->634788273],
Cell[93463, 4862, 708, 23, 70, "NumberedEquation",
 CellTags->"ModifiedBaderSmoothingStep",
 CellID->972989670],
Cell[94174, 4887, 272, 10, 70, "Text",
 CellID->440872212],
Cell[94449, 4899, 518, 15, 70, "Text",
 CellID->1808198925],
Cell[94970, 4916, 538, 21, 70, "Text",
 CellID->261648188]
}, Open  ]],
Cell[CellGroupData[{
Cell[95545, 4942, 89, 1, 70, "Subsection",
 CellID->288744440],
Cell[95637, 4945, 283, 11, 70, "Text",
 CellID->864233916],
Cell[95923, 4958, 309, 11, 70, "Text",
 CellID->1452113518],
Cell[96235, 4971, 160, 4, 70, "Text",
 CellID->637992556],
Cell[96398, 4977, 246, 7, 70, "Text",
 CellID->656923023],
Cell[96647, 4986, 117, 3, 70, "Text",
 CellID->547215790],
Cell[96767, 4991, 1109, 31, 70, "NumberedEquation",
 CellTags->"ModifiedAitkenNeville",
 CellID->1005920553],
Cell[97879, 5024, 768, 26, 70, "Text",
 CellID->1142457867],
Cell[98650, 5052, 388, 15, 70, "Text",
 CellID->545335637],
Cell[99041, 5069, 210, 5, 70, "Text",
 CellID->979046348]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[99300, 5080, 78, 2, 70, "Section",
 CellTags->"s:4",
 CellID->777991017],
Cell[99381, 5084, 161, 4, 70, "Text",
 CellID->1845275582],
Cell[CellGroupData[{
Cell[99567, 5092, 55, 1, 70, "Subsection",
 CellID->47000056],
Cell[99625, 5095, 401, 10, 70, "Text",
 CellID->1457698310]
}, Open  ]],
Cell[CellGroupData[{
Cell[100063, 5110, 62, 1, 70, "Subsection",
 CellID->386085898],
Cell[100128, 5113, 327, 11, 70, "Text",
 CellID->670278164]
}, Open  ]],
Cell[CellGroupData[{
Cell[100492, 5129, 76, 1, 70, "Subsection",
 CellID->599144811],
Cell[100571, 5132, 248, 5, 70, "Text",
 CellID->523087874],
Cell[100822, 5139, 116, 3, 70, "Text",
 CellID->1669983850],
Cell[100941, 5144, 255, 5, 70, "Text",
 CellID->1549385406]
}, Open  ]],
Cell[CellGroupData[{
Cell[101233, 5154, 57, 1, 70, "Subsection",
 CellID->940495697],
Cell[101293, 5157, 137, 4, 70, "Text",
 CellID->1960337703],
Cell[101433, 5163, 271, 9, 70, "Text",
 CellID->125697454],
Cell[101707, 5174, 277, 10, 70, "Text",
 CellID->1363102189],
Cell[101987, 5186, 472, 17, 70, "Text",
 CellID->1547059749],
Cell[102462, 5205, 583, 19, 70, "DisplayMath",
 CellID->1586380436],
Cell[103048, 5226, 327, 9, 70, "Text",
 CellID->1997890462],
Cell[103378, 5237, 105, 3, 70, "Text",
 CellID->372565771],
Cell[103486, 5242, 957, 30, 70, "NumberedEquation",
 CellTags->"ImplicitEulerStabilityCheck",
 CellID->2027252648],
Cell[104446, 5274, 353, 10, 70, "Text",
 CellID->1658605739],
Cell[104802, 5286, 628, 21, 70, "Text",
 CellID->634213959],
Cell[105433, 5309, 557, 16, 70, "DisplayMath",
 CellID->168513440],
Cell[105993, 5327, 145, 4, 70, "Text",
 CellID->375356008]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[106187, 5337, 66, 2, 70, "Section",
 CellTags->"s:5",
 CellID->2086327477],
Cell[CellGroupData[{
Cell[106278, 5343, 55, 1, 70, "Subsection",
 CellID->806505997],
Cell[106336, 5346, 134, 4, 70, "Text",
 CellID->1032279031],
Cell[106473, 5352, 431, 13, 70, "Input",
 CellID->486059620],
Cell[106907, 5367, 250, 7, 70, "Text",
 CellID->1812426550],
Cell[CellGroupData[{
Cell[107182, 5378, 444, 12, 70, "Input",
 CellID->27253251],
Cell[107629, 5392, 728, 27, 54, "Output",
 CellID->94940460]
}, Open  ]],
Cell[108372, 5422, 264, 5, 70, "Text",
 CellID->643989250],
Cell[108639, 5429, 86, 1, 70, "Text",
 CellID->805244268],
Cell[CellGroupData[{
Cell[108750, 5434, 342, 9, 28, "Input",
 CellID->729187191],
Cell[109095, 5445, 3939, 69, 251, "Output",
 Evaluatable->False,
 CellID->490935639]
}, Open  ]],
Cell[113049, 5517, 94, 1, 70, "Text",
 CellID->115458930],
Cell[CellGroupData[{
Cell[113168, 5522, 115, 3, 28, "Input",
 CellID->1115273089],
Cell[113286, 5527, 10291, 173, 246, "Output",
 Evaluatable->False,
 CellID->606774139]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[123626, 5706, 67, 1, 70, "Subsection",
 CellID->714603741],
Cell[123696, 5709, 65, 1, 70, "Text",
 CellID->1638360964],
Cell[123764, 5712, 187, 6, 70, "Input",
 CellID->883767673],
Cell[123954, 5720, 269, 8, 70, "Text",
 CellID->116505500],
Cell[CellGroupData[{
Cell[124248, 5732, 520, 14, 70, "Input",
 CellID->431373013],
Cell[124771, 5748, 217, 7, 36, "Output",
 CellID->1189623250]
}, Open  ]],
Cell[125003, 5758, 189, 6, 70, "Text",
 CellID->242202954],
Cell[CellGroupData[{
Cell[125217, 5768, 388, 10, 70, "Input",
 CellID->1794475202],
Cell[125608, 5780, 227, 7, 36, "Output",
 CellID->1987993033]
}, Open  ]],
Cell[125850, 5790, 203, 7, 70, "Text",
 CellID->1828583391],
Cell[CellGroupData[{
Cell[126078, 5801, 540, 14, 70, "Input",
 CellID->1644882016],
Cell[126621, 5817, 227, 7, 36, "Output",
 CellID->1383544515]
}, Open  ]],
Cell[126863, 5827, 220, 5, 70, "Text",
 CellID->1708352391],
Cell[127086, 5834, 90, 1, 70, "Text",
 CellID->1156004108],
Cell[CellGroupData[{
Cell[127201, 5839, 701, 22, 65, "Input",
 CellID->1102977560],
Cell[127905, 5863, 24853, 411, 297, "Output",
 Evaluatable->False,
 CellID->112687912]
}, Open  ]],
Cell[152773, 6277, 132, 4, 70, "Text",
 CellID->1531360116],
Cell[CellGroupData[{
Cell[152930, 6285, 760, 19, 70, "Input",
 CellID->1041821389],
Cell[153693, 6306, 1542, 47, 86, "Output",
 CellID->1647144482]
}, Open  ]],
Cell[155250, 6356, 170, 6, 70, "Text",
 CellID->416155499],
Cell[CellGroupData[{
Cell[155445, 6366, 611, 17, 70, "Input",
 CellID->1527881009],
Cell[156059, 6385, 476, 15, 54, "Output",
 CellID->1363978244]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[156584, 6406, 63, 1, 70, "Subsection",
 CellID->443408750],
Cell[156650, 6409, 238, 9, 70, "Text",
 CellID->411792618],
Cell[156891, 6420, 1533, 52, 70, "Input",
 CellID->844820333],
Cell[158427, 6474, 228, 12, 70, "Text",
 CellID->1248653018]
}, Open  ]],
Cell[CellGroupData[{
Cell[158692, 6491, 64, 1, 70, "Subsection",
 CellID->1140979810],
Cell[158759, 6494, 305, 7, 70, "Text",
 CellID->1535809639],
Cell[159067, 6503, 180, 6, 70, "BulletedText",
 CellID->1364111948],
Cell[159250, 6511, 278, 9, 70, "BulletedText",
 CellID->108769251],
Cell[CellGroupData[{
Cell[159553, 6524, 305, 9, 70, "BulletedText",
 CellID->749593462],
Cell[159861, 6535, 64497, 2852, 70, "Graphics",
 CellID->2078715690],
Cell[224361, 9389, 181, 4, 70, "Text",
 CellID->1747130215],
Cell[224545, 9395, 186, 4, 70, "Text",
 CellID->1633361135],
Cell[224734, 9401, 281, 6, 70, "Text",
 CellID->47737542],
Cell[225018, 9409, 1152, 36, 70, "Text",
 CellID->31019581],
Cell[226173, 9447, 310, 9, 70, InheritFromParent,
 CellID->182499215],
Cell[226486, 9458, 295, 11, 70, "Text",
 CellID->289940496]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[226830, 9475, 60, 1, 70, "Subsection",
 CellID->1825494783],
Cell[226893, 9478, 275, 9, 70, "Text",
 CellID->278697044],
Cell[227171, 9489, 129, 4, 70, "Text",
 CellID->2118523887],
Cell[227303, 9495, 61755, 2681, 70, "Graphics",
 CellID->1322893152]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[289107, 12182, 68, 2, 70, "Section",
 CellTags->"s:6",
 CellID->747460645],
Cell[CellGroupData[{
Cell[289200, 12188, 64, 1, 70, "Subsection",
 CellID->1679392631],
Cell[289267, 12191, 592, 18, 70, "Text",
 CellID->331413612],
Cell[289862, 12211, 721, 24, 70, "NumberedEquation",
 CellTags->"ExtrapolationStepControl",
 CellID->1850438205],
Cell[290586, 12237, 124, 4, 70, "Text",
 CellID->1408774815]
}, Open  ]],
Cell[CellGroupData[{
Cell[290747, 12246, 62, 1, 70, "Subsection",
 CellID->1862773728],
Cell[290812, 12249, 413, 13, 70, "Text",
 CellID->744775946],
Cell[291228, 12264, 516, 18, 70, "DisplayMath",
 CellID->1944662996]
}, Open  ]],
Cell[CellGroupData[{
Cell[291781, 12287, 60, 1, 70, "Subsection",
 CellID->131445579],
Cell[291844, 12290, 138, 5, 70, "Text",
 CellID->125173308],
Cell[291985, 12297, 378, 11, 70, "Text",
 CellID->2145494079],
Cell[292366, 12310, 164, 4, 70, "Text",
 CellID->574854330],
Cell[292533, 12316, 250, 5, 70, "Text",
 CellID->536965926],
Cell[292786, 12323, 685, 22, 70, "Text",
 CellID->842830764],
Cell[293474, 12347, 279, 9, 70, "Text",
 CellID->1823507348],
Cell[293756, 12358, 456, 16, 70, "Text",
 CellID->1677945736],
Cell[294215, 12376, 378, 15, 70, "Text",
 CellID->231156755],
Cell[294596, 12393, 359, 14, 70, "Text",
 CellID->1040135649],
Cell[294958, 12409, 251, 5, 70, "Text",
 CellID->1786714988],
Cell[295212, 12416, 157, 4, 70, "Text",
 CellID->980750218]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[295418, 12426, 71, 2, 70, "Section",
 CellTags->"s:7",
 CellID->971367691],
Cell[295492, 12430, 3354, 101, 70, "DefinitionBox3Col",
 CellID->1386361555],
Cell[298849, 12533, 116, 5, 70, "Caption",
 CellID->466627705],
Cell[298968, 12540, 248, 8, 70, "Text",
 CellID->37295877],
Cell[299219, 12550, 238, 7, 70, "Text",
 CellID->704418088],
Cell[299460, 12559, 325, 9, 70, "Text",
 CellID->497606648],
Cell[299788, 12570, 316, 11, 70, "Text",
 CellID->32145147],
Cell[300107, 12583, 606, 23, 70, "Text",
 CellID->790341517],
Cell[300716, 12608, 313, 11, 70, "Text",
 CellID->156208096],
Cell[301032, 12621, 321, 11, 70, "Text",
 CellID->352536813],
Cell[301356, 12634, 246, 8, 70, "Text",
 CellID->9411404],
Cell[301605, 12644, 759, 22, 70, "DefinitionBox3Col",
 CellID->1019105233],
Cell[302367, 12668, 241, 9, 70, "Caption",
 CellID->1417381136]
}, Open  ]]
}, Open  ]],
Cell[302635, 12681, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

