(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     60982,       1573]
NotebookOptionsPosition[     54748,       1379]
NotebookOutlinePosition[     58127,       1466]
CellTagsIndexPosition[     57991,       1458]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["LocallyExact Method for NDSolve", "Title",
 CellTags->"c:1",
 CellID->1330753532],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->65952254],

Cell[TextData[{
 "A differential system can sometimes be solved by analytic means.",
 " The function ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " implements many of the known algorithmic techniques."
}], "Text",
 CellID->1507843601],

Cell[TextData[{
 "However, differential systems that can be solved in closed form constitute \
only a small subset. Despite this fact, when a closed-form solution does not \
exist for the entire vector field, it is often possible to analytically solve \
a system of differential equations for part of the vector field. An example \
of this is the method ",
 StyleBox["Splitting", "MR"],
 ", which breaks up a vector field ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]]],
 " into subfields",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     SubscriptBox["f", "1"], " ", ",", "\[Ellipsis]", ",", " ", 
     SubscriptBox["f", "n"]}]}], TraditionalForm]]],
 " such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " ", "=", " ", 
    RowBox[{
     SubscriptBox["f", "1"], " ", "+", " ", "\[CenterEllipsis]", " ", "+", 
     " ", 
     SubscriptBox["f", "n"]}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1277659127],

Cell[TextData[{
 "The idea underlying the method ",
 StyleBox["LocallyExact", "MR"],
 " is that rather than using a standard numerical integration scheme, when a \
solution can be found by ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ", direct numerical evaluation can be used to locally advance the solution."
}], "Text",
 CellID->278328714],

Cell[TextData[{
 "Since the method ",
 StyleBox["LocallyExact", "MR"],
 " makes no attempt to adaptively adjust step sizes, it is primarily intended \
for use as a submethod between integration steps."
}], "Text",
 CellID->1105721628]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellTags->"s:2",
 CellID->1717072260],

Cell["Load a package with some predefined problems.", "Text",
 CellID->1690513279],

Cell[BoxData[
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1306710614],

Cell[CellGroupData[{

Cell["Harmonic oscillator", "Subsection",
 CellID->185670201],

Cell[TextData[{
 "Numerically solve the equations of motion for a harmonic oscillator using \
the method ",
 StyleBox["LocallyExact", "MR"],
 ". The result is two interpolating functions that approximate the solution \
and the first derivative."
}], "Text",
 CellID->768545419],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<HarmonicOscillator\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], ";"}], "\n", 

 RowBox[{
  RowBox[{
   RowBox[{"tdata", " ", "=", " ", 
    RowBox[{"system", "[", "\"\<TimeData\>\"", "]"}]}], ";"}], "\n"}], "\n", 
 RowBox[{"sols", " ", "=", " ", 
  RowBox[{"vars", " ", "/.", " ", 
   RowBox[{"First", "[", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"StartingStepSize", "\[Rule]", 
       RowBox[{"1", "/", "10"}]}], ",", " ", 
      RowBox[{"Method", "\[Rule]", "\"\<LocallyExact\>\""}]}], "]"}], 
    "]"}]}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->1933010432],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "T", "]"}], ",", 
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "T", "]"}]}], "}"}]], "Output",
 ImageSize->{300, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]="]
}, Open  ]],

Cell["The solution evolves on the unit circle.", "Text",
 CellID->687375868],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", "sols", "]"}], ",", 
   RowBox[{"Evaluate", "[", "tdata", "]"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->344745854],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 357},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->46275211]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Global versus local", "Subsection",
 CellID->2111583195],

Cell[TextData[{
 "The method ",
 StyleBox["LocallyExact", "MR"],
 " is not intended as a substitute for a closed-form (global) solution."
}], "Text",
 CellID->1485049789],

Cell[TextData[{
 "Despite the fact that the method ",
 StyleBox["LocallyExact", "MR"],
 " uses the analytic solution to advance the solution, it only produces \
solutions at the grid points in the numerical integration (or even inside \
grid points if called appropriately). Therefore, there can be errors due to \
sampling at interpolation points that do not lie exactly on the numerical \
integration grid."
}], "Text",
 CellID->2126845481],

Cell["\<\
Plot the error in the first solution component of the harmonic oscillator and \
compare it with the exact flow.\
\>", "Text",
 CellID->80106032],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"First", "[", "sols", "]"}], "-", 
     RowBox[{"Cos", "[", "T", "]"}]}], "]"}], ",", 
   RowBox[{"Evaluate", "[", "tdata", "]"}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1919918225],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 197},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->221420782]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simplification", "Subsection",
 CellID->2117841846],

Cell[TextData[{
 "The method ",
 StyleBox["LocallyExact", "MR"],
 " has an option ",
 StyleBox["SimplificationFunction", "MR"],
 " that can be used to simplify the results of ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1989880484],

Cell["\<\
Here is the linearized component of the differential system that turns up in \
the splitting of the Lorenz equations using standard values for the \
parameters.\
\>", "Text",
 CellID->2021562841],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqs", "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["Y", "1"], "'"}], "[", "T", "]"}], "\[Equal]", 
       RowBox[{"\[Sigma]", " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           SubscriptBox["Y", "2"], "[", "T", "]"}], "-", 
          RowBox[{
           SubscriptBox["Y", "1"], "[", "T", "]"}]}], ")"}]}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["Y", "2"], "'"}], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"r", " ", 
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}]}], "-", 
        RowBox[{
         SubscriptBox["Y", "2"], "[", "T", "]"}]}]}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["Y", "3"], "'"}], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"-", "b"}], " ", 
        RowBox[{
         SubscriptBox["Y", "3"], "[", "T", "]"}]}]}]}], "}"}], "/.", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Sigma]", "\[Rule]", "10"}], ",", 
      RowBox[{"r", "\[Rule]", "28"}], ",", 
      RowBox[{"b", "\[Rule]", 
       RowBox[{"8", "/", "3"}]}]}], "}"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ics", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "1"], "[", "0", "]"}], "\[Equal]", 
      RowBox[{"-", "8"}]}], ",", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "2"], "[", "0", "]"}], "\[Equal]", "8"}], ",", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "3"], "[", "0", "]"}], "\[Equal]", "27"}]}], "}"}]}],
   ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], ",", 
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], ",", 
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[8]:=",
 CellID->1856863696],

Cell[TextData[{
 "This subsystem is exactly solvable by ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->1434493352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{"eqs", ",", "vars", ",", "T"}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->1361232667],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{
       FractionBox["1", "2402"], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "-", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}]}], ")"}], " ", 
        RowBox[{"C", "[", "1", "]"}]}]}], "-", 
      FractionBox[
       RowBox[{"10", " ", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{
            FractionBox["1", "2"], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "11"}], "-", 
              SqrtBox["1201"]}], ")"}], " ", "T"}]], "-", 
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{
            FractionBox["1", "2"], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "11"}], "+", 
              SqrtBox["1201"]}], ")"}], " ", "T"}]]}], ")"}], " ", 
        RowBox[{"C", "[", "2", "]"}]}], 
       SqrtBox["1201"]]}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"28", " ", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]], "-", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], ")"}], " ", 
         RowBox[{"C", "[", "1", "]"}]}], 
        SqrtBox["1201"]]}], "+", 
      RowBox[{
       FractionBox["1", "2402"], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "-", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}]}], ")"}], " ", 
        RowBox[{"C", "[", "2", "]"}]}]}]}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "8"}], " ", "T"}], "/", "3"}]], " ", 
      RowBox[{"C", "[", "3", "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{534, 251},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]="]
}, Open  ]],

Cell[TextData[{
 "Often the results of ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " can be simplified. This defines a function to simplify an expression and \
also prints out the input and the result."
}], "Text",
 CellID->2034852465],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"myfun", "[", "x_", "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", "simpx", "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Print", "[", 
       RowBox[{"\"\<Before simplification \>\"", ",", "x"}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"simpx", " ", "=", " ", 
       RowBox[{"FullSimplify", "[", 
        RowBox[{"ExpToTrig", "[", "x", "]"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Print", "[", 
       RowBox[{"\"\<After simplification \>\"", ",", " ", "simpx"}], "]"}], 
      ";", "\[IndentingNewLine]", "simpx"}]}], "\[IndentingNewLine]", "]"}]}],
   ";"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->242054611],

Cell[TextData[{
 "The function can be passed as an option to the method ",
 StyleBox["LocallyExact", "MR"],
 "."
}], "Text",
 CellID->1917497536],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"eqs", ",", " ", "ics"}], "}"}], ",", " ", "vars", ",", " ", 
   RowBox[{"{", 
    RowBox[{"T", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"StartingStepSize", "\[Rule]", 
    RowBox[{"1", "/", "10"}]}], ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<LocallyExact\>\"", ",", " ", 
      RowBox[{"\"\<SimplificationFunction\>\"", "\[Rule]", "myfun"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->2100804719],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Before simplification \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       FractionBox["1", "2402"], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "-", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}]}], ")"}], " ", 
        RowBox[{
         SubscriptBox["Y", "1"], "[", "T", "]"}]}]}], "-", 
      FractionBox[
       RowBox[{"10", " ", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{
            FractionBox["1", "2"], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "11"}], "-", 
              SqrtBox["1201"]}], ")"}], " ", "T"}]], "-", 
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{
            FractionBox["1", "2"], " ", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{"-", "11"}], "+", 
              SqrtBox["1201"]}], ")"}], " ", "T"}]]}], ")"}], " ", 
        RowBox[{
         SubscriptBox["Y", "2"], "[", "T", "]"}]}], 
       SqrtBox["1201"]]}], ",", 
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"28", " ", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]], "-", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], ")"}], " ", 
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}]}], 
        SqrtBox["1201"]]}], "+", 
      RowBox[{
       FractionBox["1", "2402"], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "-", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "-", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"1201", " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}], "+", 
          RowBox[{"9", " ", 
           SqrtBox["1201"], " ", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{
             FractionBox["1", "2"], " ", 
             RowBox[{"(", 
              RowBox[{
               RowBox[{"-", "11"}], "+", 
               SqrtBox["1201"]}], ")"}], " ", "T"}]]}]}], ")"}], " ", 
        RowBox[{
         SubscriptBox["Y", "2"], "[", "T", "]"}]}]}]}], ",", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "8"}], " ", "T"}], "/", "3"}]], " ", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}]}]}], "}"}]}],
  SequenceForm[
  "Before simplification ", {
   Rational[1, 2402] (
      1201 E^(Rational[1, 2] (-11 - 1201^Rational[1, 2]) 
         DifferentialEquations`NDSolveProblems`T) + 
      9 1201^Rational[1, 2] 
       E^(Rational[1, 2] (-11 - 1201^Rational[1, 2]) 
         DifferentialEquations`NDSolveProblems`T) + 
      1201 E^(Rational[1, 2] (-11 + 1201^Rational[1, 2]) 
         DifferentialEquations`NDSolveProblems`T) - 9 1201^Rational[1, 2] 
      E^(Rational[1, 2] (-11 + 1201^Rational[1, 2]) 
        DifferentialEquations`NDSolveProblems`T)) 
     Subscript[DifferentialEquations`NDSolveProblems`Y, 1][
      DifferentialEquations`NDSolveProblems`T] - 10 
    1201^Rational[-1, 2] (
     E^(Rational[1, 2] (-11 - 1201^Rational[1, 2]) 
       DifferentialEquations`NDSolveProblems`T) - 
     E^(Rational[1, 2] (-11 + 1201^Rational[1, 2]) 
      DifferentialEquations`NDSolveProblems`T)) 
    Subscript[DifferentialEquations`NDSolveProblems`Y, 2][
     DifferentialEquations`NDSolveProblems`T], (-28) 
     1201^Rational[-1, 2] (
      E^(Rational[1, 2] (-11 - 1201^Rational[1, 2]) 
        DifferentialEquations`NDSolveProblems`T) - 
      E^(Rational[1, 2] (-11 + 1201^Rational[1, 2]) 
       DifferentialEquations`NDSolveProblems`T)) 
     Subscript[DifferentialEquations`NDSolveProblems`Y, 1][
      DifferentialEquations`NDSolveProblems`T] + 
    Rational[1, 2402] (
      1201 E^(Rational[1, 2] (-11 - 1201^Rational[1, 2]) 
         DifferentialEquations`NDSolveProblems`T) - 9 1201^Rational[1, 2] 
      E^(Rational[1, 2] (-11 - 1201^Rational[1, 2]) 
        DifferentialEquations`NDSolveProblems`T) + 
      1201 E^(Rational[1, 2] (-11 + 1201^Rational[1, 2]) 
         DifferentialEquations`NDSolveProblems`T) + 
      9 1201^Rational[1, 2] 
       E^(Rational[1, 2] (-11 + 1201^Rational[1, 2]) 
         DifferentialEquations`NDSolveProblems`T)) 
     Subscript[DifferentialEquations`NDSolveProblems`Y, 2][
      DifferentialEquations`NDSolveProblems`T], 
    E^(Rational[-8, 3] DifferentialEquations`NDSolveProblems`T) 
    Subscript[DifferentialEquations`NDSolveProblems`Y, 3][
     DifferentialEquations`NDSolveProblems`T]}],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[13]:="],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"After simplification \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{
     FractionBox[
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "11"}], " ", "T"}], "/", "2"}]], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"1201", " ", 
          RowBox[{"Cosh", "[", 
           FractionBox[
            RowBox[{
             SqrtBox["1201"], " ", "T"}], "2"], "]"}], " ", 
          RowBox[{
           SubscriptBox["Y", "1"], "[", "T", "]"}]}], "+", 
         RowBox[{
          SqrtBox["1201"], " ", 
          RowBox[{"Sinh", "[", 
           FractionBox[
            RowBox[{
             SqrtBox["1201"], " ", "T"}], "2"], "]"}], " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{
             RowBox[{"-", "9"}], " ", 
             RowBox[{
              SubscriptBox["Y", "1"], "[", "T", "]"}]}], "+", 
            RowBox[{"20", " ", 
             RowBox[{
              SubscriptBox["Y", "2"], "[", "T", "]"}]}]}], ")"}]}]}], ")"}]}],
       "1201"], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "11"}], " ", "T"}], "/", "2"}]], " ", 
       RowBox[{"Cosh", "[", 
        FractionBox[
         RowBox[{
          SqrtBox["1201"], " ", "T"}], "2"], "]"}], " ", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}]}], "+", 
      FractionBox[
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{
           RowBox[{"-", "11"}], " ", "T"}], "/", "2"}]], " ", 
        RowBox[{"Sinh", "[", 
         FractionBox[
          RowBox[{
           SqrtBox["1201"], " ", "T"}], "2"], "]"}], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"56", " ", 
           RowBox[{
            SubscriptBox["Y", "1"], "[", "T", "]"}]}], "+", 
          RowBox[{"9", " ", 
           RowBox[{
            SubscriptBox["Y", "2"], "[", "T", "]"}]}]}], ")"}]}], 
       SqrtBox["1201"]]}], ",", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "8"}], " ", "T"}], "/", "3"}]], " ", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}]}]}], "}"}]}],
  SequenceForm[
  "After simplification ", {
   Rational[1, 1201] 
    E^(Rational[-11, 2] DifferentialEquations`NDSolveProblems`T) (
     1201 Cosh[
       Rational[1, 2] 1201^Rational[1, 2] 
        DifferentialEquations`NDSolveProblems`T] 
      Subscript[DifferentialEquations`NDSolveProblems`Y, 1][
       DifferentialEquations`NDSolveProblems`T] + 
     1201^Rational[1, 2] 
      Sinh[Rational[1, 2] 1201^Rational[1, 2] 
        DifferentialEquations`NDSolveProblems`T] ((-9) 
        Subscript[DifferentialEquations`NDSolveProblems`Y, 1][
         DifferentialEquations`NDSolveProblems`T] + 
       20 Subscript[DifferentialEquations`NDSolveProblems`Y, 2][
         DifferentialEquations`NDSolveProblems`T])), 
    E^(Rational[-11, 2] DifferentialEquations`NDSolveProblems`T) 
     Cosh[Rational[1, 2] 1201^Rational[1, 2] 
       DifferentialEquations`NDSolveProblems`T] 
     Subscript[DifferentialEquations`NDSolveProblems`Y, 2][
      DifferentialEquations`NDSolveProblems`T] + 
    1201^Rational[-1, 2] 
     E^(Rational[-11, 2] DifferentialEquations`NDSolveProblems`T) 
     Sinh[Rational[1, 2] 1201^Rational[1, 2] 
       DifferentialEquations`NDSolveProblems`T] (
      56 Subscript[DifferentialEquations`NDSolveProblems`Y, 1][
        DifferentialEquations`NDSolveProblems`T] + 
      9 Subscript[DifferentialEquations`NDSolveProblems`Y, 2][
        DifferentialEquations`NDSolveProblems`T]), 
    E^(Rational[-8, 3] DifferentialEquations`NDSolveProblems`T) 
    Subscript[DifferentialEquations`NDSolveProblems`Y, 3][
     DifferentialEquations`NDSolveProblems`T]}],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[13]:="]
}, Open  ]],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{356, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]="]
}, Open  ]],

Cell["\<\
The simplification is performed only once during the initialization phase \
that constructs the data object for the numerical integration method.\
\>", "Text",
 CellID->708136193]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:3",
 CellID->617261628],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"SimplificationFunction", 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "function to use in simplifying the result of ",
     Cell[BoxData[
      ButtonBox["DSolve",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/DSolve"]], "InlineFormula"]
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->1922163065],

Cell[TextData[{
 "Options of the method ",
 Cell[BoxData["LocallyExact"], "InlineFormula"],
 ". "
}], "Caption",
 CellID->2014846807]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"LocallyExact Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/NDSolveExtrapolation#1712532361"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolvePlugIns"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 55.4465014}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "A differential system can sometimes be solved by analytic means. The \
function DSolve implements many of the known algorithmic techniques. However, \
differential systems that can be solved in closed form constitute only a \
small subset. Despite this fact, when a closed-form solution does not exist \
for the entire vector field, it is often possible to analytically solve a \
system of differential equations for part of the vector field. An example of \
this is the method Splitting, which breaks up a vector field f into subfields \
f_1 ,..., f_n such that f = f_1 + ... + f_n. The idea underlying the method \
LocallyExact is that rather than using a standard numerical integration \
scheme, when a solution can be found by DSolve, direct numerical evaluation \
can be used to locally advance the solution.", "synonyms" -> {}, "title" -> 
    "LocallyExact Method for NDSolve", "type" -> "Tutorial", "uri" -> 
    "tutorial/NDSolveLocallyExact"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[728, 29, 87, 2, 70, "Title",
   CellTags->"c:1",
   CellID->1330753532]},
 "s:1"->{
  Cell[840, 35, 68, 2, 70, "Section",
   CellTags->"s:1",
   CellID->65952254]},
 "s:2"->{
  Cell[2848, 103, 66, 2, 70, "Section",
   CellTags->"s:2",
   CellID->1717072260]},
 "s:3"->{
  Cell[53811, 1344, 71, 2, 70, "Section",
   CellTags->"s:3",
   CellID->617261628]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 57605, 1439},
 {"s:1", 57696, 1443},
 {"s:2", 57787, 1447},
 {"s:3", 57882, 1451}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 87, 2, 70, "Title",
 CellTags->"c:1",
 CellID->1330753532],
Cell[CellGroupData[{
Cell[840, 35, 68, 2, 70, "Section",
 CellTags->"s:1",
 CellID->65952254],
Cell[911, 39, 308, 9, 70, "Text",
 CellID->1507843601],
Cell[1222, 50, 933, 27, 70, "Text",
 CellID->1277659127],
Cell[2158, 79, 416, 11, 70, "Text",
 CellID->278328714],
Cell[2577, 92, 234, 6, 70, "Text",
 CellID->1105721628]
}, Open  ]],
Cell[CellGroupData[{
Cell[2848, 103, 66, 2, 70, "Section",
 CellTags->"s:2",
 CellID->1717072260],
Cell[2917, 107, 82, 1, 70, "Text",
 CellID->1690513279],
Cell[3002, 110, 171, 6, 70, "Input",
 CellID->1306710614],
Cell[CellGroupData[{
Cell[3198, 120, 61, 1, 70, "Subsection",
 CellID->185670201],
Cell[3262, 123, 277, 7, 70, "Text",
 CellID->768545419],
Cell[CellGroupData[{
Cell[3564, 134, 802, 23, 70, "Input",
 CellID->1933010432],
Cell[4369, 159, 708, 26, 54, "Output"]
}, Open  ]],
Cell[5092, 188, 76, 1, 70, "Text",
 CellID->687375868],
Cell[CellGroupData[{
Cell[5193, 193, 262, 7, 28, "Input",
 CellID->344745854],
Cell[5458, 202, 7434, 126, 378, "Output",
 Evaluatable->False,
 CellID->46275211]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[12941, 334, 62, 1, 70, "Subsection",
 CellID->2111583195],
Cell[13006, 337, 170, 5, 70, "Text",
 CellID->1485049789],
Cell[13179, 344, 442, 9, 70, "Text",
 CellID->2126845481],
Cell[13624, 355, 154, 4, 70, "Text",
 CellID->80106032],
Cell[CellGroupData[{
Cell[13803, 363, 296, 9, 28, "Input",
 CellID->1919918225],
Cell[14102, 374, 17451, 290, 218, "Output",
 Evaluatable->False,
 CellID->221420782]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[31602, 670, 57, 1, 70, "Subsection",
 CellID->2117841846],
Cell[31662, 673, 330, 12, 70, "Text",
 CellID->1989880484],
Cell[31995, 687, 205, 5, 70, "Text",
 CellID->2021562841],
Cell[32203, 694, 1994, 67, 70, "Input",
 CellID->1856863696],
Cell[34200, 763, 212, 8, 70, "Text",
 CellID->1434493352],
Cell[CellGroupData[{
Cell[34437, 775, 142, 4, 70, "Input",
 CellID->1361232667],
Cell[34582, 781, 4878, 145, 272, "Output"]
}, Open  ]],
Cell[39475, 929, 310, 9, 70, "Text",
 CellID->2034852465],
Cell[39788, 940, 768, 20, 70, "Input",
 CellID->242054611],
Cell[40559, 962, 145, 5, 70, "Text",
 CellID->1917497536],
Cell[CellGroupData[{
Cell[40729, 971, 561, 15, 70, "Input",
 CellID->2100804719],
Cell[CellGroupData[{
Cell[41315, 990, 6923, 183, 70, "Print"],
Cell[48241, 1175, 3976, 107, 70, "Print"]
}, Open  ]],
Cell[52232, 1285, 1326, 46, 71, "Output"]
}, Open  ]],
Cell[53573, 1334, 189, 4, 70, "Text",
 CellID->708136193]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[53811, 1344, 71, 2, 70, "Section",
 CellTags->"s:3",
 CellID->617261628],
Cell[53885, 1348, 673, 18, 70, "DefinitionBox3Col",
 CellID->1922163065],
Cell[54561, 1368, 133, 5, 70, "Caption",
 CellID->2014846807]
}, Open  ]]
}, Open  ]],
Cell[54721, 1377, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

