(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[   1408151,      35668]
NotebookOptionsPosition[   1272189,      31559]
NotebookOutlinePosition[   1346173,      33647]
CellTagsIndexPosition[   1344617,      33595]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Numerical Differential Equation Solving in \
Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/NDSolveOverview"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Advanced Numerical Differential Equation Solving in \
Mathematica\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Derivative" :> 
          Documentation`HelpLookup["paclet:ref/Derivative"], "False" :> 
          Documentation`HelpLookup["paclet:ref/False"], "Flatten" :> 
          Documentation`HelpLookup["paclet:ref/Flatten"], "Length" :> 
          Documentation`HelpLookup["paclet:ref/Length"], "True" :> 
          Documentation`HelpLookup["paclet:ref/True"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Derivative\"\>", 2->"\<\"False\"\>", 
       3->"\<\"Flatten\"\>", 4->"\<\"Length\"\>", 
       5->"\<\"True\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Numerical Solution of Partial Differential Equations", "Title",
 CellID->2023759877],

Cell[CellGroupData[{

Cell["The Numerical Method Of Lines", "Section",
 CellTags->{"c:1", "PDE:1"},
 CellID->855348500],

Cell[CellGroupData[{

Cell["Introduction", "Subsection",
 CellTags->{"c:2", "b:7.0", "ndsg:2.0.0", "PDE:1.1"},
 CellID->1699661144],

Cell["\<\
The numerical methods of lines is a technique for solving partial \
differential equations discretizing in all but one dimension, and then \
integrating the semi-discrete problem as a system of ODEs or DAEs. A \
significant advantage of the method is that it allows the solution to take \
advantage of the sophisticated general purpose methods and software that have \
been developed for numerically integrating ODEs and DAEs. For the PDEs to \
which the method of lines is applicable, the method typically proves to be \
quite efficient. \
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->178642212],

Cell["\<\
It is necessary that the PDE problem be well-posed as an initial value \
(Cauchy) problem in at least one dimension, since the ODE and DAE integrators \
used are initial value problem solvers. This rules out purely elliptic \
equations such as Laplace's equation, but leaves a large class of evolution \
equations that can be solved quite efficiently.\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->31579405],

Cell["\<\
A simple example illustrates better than mere words the fundamental idea of \
the method. Consider the following problem (a simple model for seasonal \
variation of heat in soil). \
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1206238451],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["u", "t"], "==", 
    RowBox[{
     FractionBox["1", "8"], 
     SubscriptBox["u", "xx"]}]}], ",", 
   RowBox[{
    RowBox[{"u", "(", 
     RowBox[{"0", ",", "t"}], ")"}], "==", 
    RowBox[{"sin", "(", 
     RowBox[{"2", "\[Pi]", " ", "t"}], ")"}]}], ",", 
   RowBox[{
    RowBox[{
     SubscriptBox["u", "x"], "(", 
     RowBox[{"1", ",", "t"}], ")"}], "==", "0"}], ",", " ", 
   RowBox[{
    RowBox[{"u", "(", 
     RowBox[{"x", ",", "0"}], ")"}], " ", "\[Equal]", " ", "0"}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->749156851],

Cell["\<\
This is a candidate for the method of lines since we have the initial value \
u(x,0) == 0.\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->184931984],

Cell["\<\
Problem (1) will be discretized with respect to the variable x using second \
order finite differences, in particular using the approximation\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->984024416],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["u", "xx"], "(", 
    RowBox[{"x", ",", "t"}], ")"}], " ", "\[TildeEqual]", 
   FractionBox[
    RowBox[{
     RowBox[{"u", "(", 
      RowBox[{
       RowBox[{"x", " ", "+", " ", "h"}], ",", " ", "t"}], ")"}], " ", "-", 
     " ", 
     RowBox[{"2", " ", 
      RowBox[{"u", "(", 
       RowBox[{"x", ",", " ", "t"}], ")"}]}], " ", "-", " ", 
     RowBox[{"u", "(", 
      RowBox[{
       RowBox[{"x", " ", "-", " ", "h"}], ",", " ", "t"}], ")"}]}], 
    SuperscriptBox["h", "2"]]}], TraditionalForm]], "NumberedEquation",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->223999859],

Cell["\<\
Even though finite difference discretizations are the most common, there is \
certainly no requirement that discretizations for the method of lines be done \
with finite differences; finite volume or even finite element discretizations \
can also be used.\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1396176391],

Cell[TextData[{
 "To use the discretization above, choose a uniform grid ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "i"], ",", " ", 
    RowBox[{
    "0", " ", "\[LessEqual]", " ", "i", " ", "\[LessEqual]", " ", "n"}]}], 
   TraditionalForm]]],
 "with spacing h == 1/n such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "i"], " ", "\[Equal]", " ", 
    RowBox[{"i", " ", "h"}]}], TraditionalForm]]],
 ". Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["u", "i"], "[", "t", "]"}], TraditionalForm]]],
 " be the value of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"u", "(", 
    RowBox[{
     SubscriptBox["x", "i"], ",", "t"}], ")"}], TraditionalForm]]],
 ".",
 " ",
 "For the purposes of illustrating the problem set-up, a particular value of \
n is chosen."
}], "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1792546714],

Cell["\<\
This defines a particular value of n and the corresponding value of h used in \
the subsequent commands. This can be changed to make a finer or coarser \
spatial approximation.\
\>", "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->917104564],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", " ", "=", " ", "10"}], ";", 
  RowBox[{
   SubscriptBox["h", "n"], " ", "=", " ", 
   FractionBox["1", "n"]}], ";"}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[1]:=",
 CellID->92453843],

Cell[TextData[{
 "This defines the vector of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["u", "i"], TraditionalForm]]],
 "."
}], "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1230812222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"U", "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     SubscriptBox["u", "i"], "[", "t", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[2]:=",
 CellID->117310191],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["u", "0"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "1"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "2"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "3"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "4"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "5"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "6"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "7"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "8"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "9"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "10"], "[", "t", "]"}]}], "}"}]], "Output",
 ImageSize->{504, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[2]=",
 CellID->1955190738]
}, Open  ]],

Cell["\<\
For 1 \[LessEqual] i \[LessEqual] 9, we can use the centered difference \
formula (2) to obtain a system of ODEs. However, before doing this, it is \
useful to incorporate the boundary conditions first.\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->248954016],

Cell[TextData[{
 "The Dirichlet boundary condition at x == 0 can easily be handled by simply \
defining ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["u", "0"], TraditionalForm]]],
 "as a function of t. An alternative option is to differentiate the boundary \
condition with respect to time and use the corresponding differential \
equation.",
 " ",
 "In this example, the latter method will be used."
}], "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->24061468],

Cell[TextData[{
 "The Neumann boundary condition at x == 1 is a little more difficult. With \
second order differences, one way to handle it is with reflection: imagine we \
are solving the problem on the interval 0 \[LessEqual] x \[LessEqual] 2 with \
the same boundary conditions at x == 0 and x == 2.",
 " ",
 "Since the initial condition and boundary conditions are symmetric with \
respect to x, the solution should be symmetric with respect to x for all \
time, and so symmetry is equivalent to the Neumann boundary condition at x == \
1. Thus,",
 StyleBox[" ", "DisplayMath"],
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"u", "(", 
     RowBox[{
      RowBox[{"1", " ", "+", " ", "h"}], ",", " ", "t"}], ")"}], " ", 
    "\[Equal]", " ", 
    RowBox[{"u", "(", 
     RowBox[{
      RowBox[{"1", " ", "-", " ", "h"}], ",", " ", "t"}], ")"}]}], 
   TraditionalForm]]],
 ", so ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["u", 
      RowBox[{"n", "+", "1"}]], "[", "t", "]"}], " ", "\[Equal]", " ", 
    RowBox[{
     SubscriptBox["u", 
      RowBox[{"n", "-", "1"}]], "[", "t", "]"}]}], TraditionalForm]]]
}], "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1450594402],

Cell[TextData[{
 "This uses ",
 Cell[BoxData[
  ButtonBox["ListCorrelate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ListCorrelate"]], "InlineFormula"],
 " to apply the difference formula. The padding ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox["u", 
     RowBox[{"n", "-", "1"}]], "[", "t", "]"}], "}"}]]],
 "implements the Neumann boundary condition."
}], "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1860087577],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqns", " ", "=", " ", 
  RowBox[{"Thread", "[", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"U", "[", "t", "]"}], ",", "t"}], "]"}], " ", "\[Equal]", " ", 
    RowBox[{"Join", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"Sin", "[", 
          RowBox[{"2", "\[Pi]", " ", "t"}], "]"}], ",", "t"}], "]"}], "}"}], 
      ",", 
      RowBox[{
       RowBox[{"ListCorrelate", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{"-", "2"}], ",", "1"}], "}"}], "/", 
          SuperscriptBox[
           SubscriptBox["h", "n"], "2"]}], ",", " ", 
         RowBox[{"U", "[", "t", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "2"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           SubscriptBox["u", 
            RowBox[{"n", "-", "1"}]], "[", "t", "]"}], "}"}]}], "]"}], "/", 
       "8"}]}], "]"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[3]:=",
 CellID->1514066680],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{530, 270},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[3]=",
 CellID->693390441]
}, Open  ]],

Cell["This sets up the zero initial condition.", "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->640535500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"initc", " ", "=", " ", 
  RowBox[{"Thread", "[", 
   RowBox[{
    RowBox[{"U", "[", "0", "]"}], " ", "\[Equal]", " ", 
    RowBox[{"Table", "[", 
     RowBox[{"0", ",", 
      RowBox[{"{", 
       RowBox[{"n", "+", "1"}], "}"}]}], "]"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[4]:=",
 CellID->1442769105],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWetOwjAULgwYiOA1xARI4GX0h/JHn2AhJvDDaJBH0ufE0q1zHk63tpyW
BiHZ1uu5fOe03T5myXrx+pasl/Nk8rBKPhbL+efk/n3Fm6IaY7Uvxtj0jvHy
hpfEtf198ystpqXL7T2SvbzIb3Ve7WbPMejvZ+2tinEXhnJd2dHjtyav1g3l
RwKdohf8zhpgVBfUh5k0OK+pOQ9DwXSeiZ1X++iL8/ZUq6buEaiLyLBOJjkD
nuVFOBJYGyHo3iraY6kjznX8sRyOt5Zf8AGzeR8d7QofIPpUPtRd4qOZSab6
0sxqw9YSj8SMJrBmx45Wpi/tkL+Wcm5dhkthRcvKiiZqRWxsBZQLM9YdJlhk
sPVx2AiV7eN+IyWw0FwrbnBSrK6dsTCH7Ky7QUf1tDTaRcpWo30EcI197xr1
fHSPKn2O49qhJ7VNeTZHXtaYGhP708pk91FngR//8bz3fyaEGwfsxPZzRuru
/R0ra0z2iDPnGih9wHP6GFDqE2rAM0x+McjMp0TtHMimxAvKpo72MWDiPnsG
hBpQ1sJS1jBQuyhlTQhlqaMrT0P5drKtD7L6uKJ/tOf8qv5pRT96iosK5GjN
S1A23COq2E/JmjYqxvWBb2VssHxzqhqnI+9JeAnfe4peYnmjy7rasLwm865B
O/x2pWZrZyhaMgd8M4lliAXNuDrUkfvx+y7OCw2CKDyisd/h+BgVu0DFvEGL
7awhZd68f3GpONLDMbZhcaTpKYCuxQC4M5oMPjGSu6jaczC4pmfx1FvToXBR
/v470stu9ziEzQ3ScHImcTHJ2mNgm06cnDL+jWLfMXBDJ75Mtb7ht2oofM9/
4LTSiMCxktsIlYd6EVZvTdVhjljtB3VJngA=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{420, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[4]=",
 CellID->549380745]
}, Open  ]],

Cell[TextData[{
 "Now the PDE has been partially discretized into an ODE initial value \
problem that can be solved by the ODE integrators in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->812336098],

Cell["This solves the ODE initial value problem.", "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1605955158],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"lines", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqns", ",", " ", "initc"}], "}"}], ",", 
    RowBox[{"U", "[", "t", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[5]:=",
 CellID->1794569415],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{357, 186},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[5]=",
 CellID->313657449]
}, Open  ]],

Cell[TextData[{
 "This shows the solutions ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"u", "(", 
     RowBox[{
      SubscriptBox["x", "i"], ",", "t"}], ")"}], " "}], TraditionalForm]]],
 "plotted as a function of x and t."
}], "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1069182557],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"i", " ", 
         SubscriptBox["h", "n"]}], ",", "t", ",", " ", 
        RowBox[{"First", "[", 
         RowBox[{
          RowBox[{
           SubscriptBox["u", "i"], "[", "t", "]"}], " ", "/.", "lines"}], 
         "]"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", " ", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<x\>\"", ",", "\"\<t\>\"", ",", "\"\<u\>\""}], "}"}]}]}], 
  "]"}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[6]:=",
 CellID->868401424],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{328, 432},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[6]=",
 CellID->535237732]
}, Open  ]],

Cell["\<\
The plot above indicates why this technique is called the numerical \"method \
of lines\". \
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->621360053],

Cell[TextData[{
 "The solution in between lines can be found by interpolation. When ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " computes the solution for the PDE, the result is a two-dimensional ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["InterpolatingFunction",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterpolatingFunction"],
   FontFamily->"Courier"]], "InlineFormula"],
 StyleBox[".",
  FontFamily->"Courier"]
}], "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1047043637],

Cell[TextData[{
 "This uses ",
 StyleBox["NDSolve", "MR"],
 " to compute the solution of the heat equation (1) directly."
}], "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1193123751],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"solution", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"x", ",", "t"}], "]"}], ",", "t"}], "]"}], " ", "\[Equal]", 
       " ", 
       RowBox[{
        FractionBox["1", "8"], 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"x", ",", "t"}], "]"}], ",", "x", ",", "x"}], "]"}]}]}], 
      ",", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"x", ",", "0"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", " ",
       
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "t"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"Sin", "[", 
        RowBox[{"2", " ", "\[Pi]", " ", "t"}], "]"}]}], ",", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"x", ",", "t"}], "]"}], ",", "x"}], "]"}], " ", "/.", 
         RowBox[{"x", "\[Rule]", "1"}]}], ")"}], " ", "\[Equal]", " ", 
       "0"}]}], "}"}], ",", "u", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[7]:=",
 CellID->1827598504],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "4.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{373, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[7]=",
 CellID->1528854640]
}, Open  ]],

Cell["This creates a surface plot of the solution.", "MathCaption",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1378055735],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"First", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"x", ",", "t"}], "]"}], " ", "/.", " ", "solution"}], "]"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}], ",", " ", 
   RowBox[{"PlotPoints", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"14", ",", "36"}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"In[8]:=",
 CellID->664139272],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 292},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellLabel->"Out[8]=",
 CellID->470571780]
}, Open  ]],

Cell[TextData[{
 "The setting n == 10 used above did not give a very accurate solution. When \
",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " computes the solution, it uses spatial error estimates on the initial \
condition to determine what the grid spacing should be. The error in the \
temporal (or at least time-like) variable is handled by the adaptive ODE \
integrator. "
}], "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->626748478],

Cell["\<\
In the example (1), the distinction between time and space was quite clear \
from the problem context. Even when the distinction is not explicit, this \
document will refer to \"spatial\" and \"temporal\" variables. The \
\"spatial\" variables are those to which the discretization is done. The \
\"temporal\" variable is the one left in the ODE system to be integrated.\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->309202338],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{"option", " ", "name"}], "TableHeader"], 
    StyleBox[
     RowBox[{"default", " ", "value"}], "TableHeader"], ""},
   {
    StyleBox["TemporalVariable", "InlineCode"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
what variable to keep derivatives with respect to the derived ODE or DAE \
system\
\>", "TableText"]},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "what method to use for integrating the ODEs or DAEs", "TableText"]},
   {
    StyleBox["SpatialDiscretization", "InlineCode"], 
    StyleBox["TensorProductGrid", "InlineCode"], Cell[
    "what method to use for spatial discretization ", "TableText"]},
   {
    StyleBox["DifferentiateBoundaryConditions", "InlineCode"], 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell["\<\
whether to differentiate the boundary conditions with respect to the temporal \
variable\
\>", "TableText"]},
   {
    StyleBox["ExpandFunctionSymbolically", "InlineCode"], 
    ButtonBox["False",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/False"], Cell[
    "whether to expand the effective function symbolically or not", 
     "TableText"]},
   {
    StyleBox["DiscretizedMonitorVariables", "InlineCode"], 
    ButtonBox["False",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/False"], Cell[TextData[{
     "whether to ",
     ButtonBox["interpret",
      BaseStyle->"Link",
      ButtonData->"paclet:tutorial/NDSolvePDE#1602750198"],
     " dependent variables given in monitors like ",
     ButtonBox["StepMonitor",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/StepMonitor"],
     " or in method options for methods like ",
     StyleBox["EventLocator", "InlineCode"],
     " and ",
     ButtonBox["Projection",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Projection"],
     " as functions of the spatial variables or vectors representing the \
spatially discretized values."
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->789821263],

Cell["Options for NDSolve`MethodOfLines. ", "Caption",
 CellID->719202321],

Cell["\<\
Use of some of these options requires further knowledge of how the method of \
lines works and will be explained in the sections that follow.\
\>", "Text",
 CellTags->{"b:7.0", "ndsg:2.0.0"},
 CellID->1540014560],

Cell[TextData[{
 "Currently, the only method implemented for spatial discretization is the ",
 StyleBox["TensorProductGrid", "MR"],
 " method, which uses discretization methods for one spatial dimension and \
uses an outer tensor product to derive methods for multiple spatial \
dimensions on rectangular regions.",
 " ",
 StyleBox["TensorProductGrid", "MR"],
 " has its own set of options that you can use to control the grid selection \
process. The following sections give sufficient background information so \
that you will be able to use these options if necessary."
}], "Text",
 CellID->907131455]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spatial Derivative Approximations", "Subsection",
 CellTags->{"c:3", "PDE:1.2"},
 CellID->1106740037],

Cell[CellGroupData[{

Cell["Finite Differences", "Subsubsection",
 CellTags->{"c:4", "b:7.1.0", "ndsg:2.0.1.0", "PDE:1.2.1"},
 CellID->1788968576],

Cell["\<\
The essence of the concept of finite differences is embodied in the standard \
definition of the derivative\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->931192923],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "'"}], 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}], " ", "==", 
   RowBox[{
    UnderscriptBox["lim", 
     RowBox[{"h", "\[Rule]", "0"}]], "\[ThinSpace]", 
    FractionBox[
     RowBox[{
      RowBox[{"f", "(", 
       RowBox[{"h", "+", 
        SubscriptBox["x", "i"]}], ")"}], "-", 
      RowBox[{"f", "(", 
       SubscriptBox["x", "i"], ")"}]}], "h"]}]}], 
  TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->409850792],

Cell[TextData[{
 "where instead of passing to the limit as h approaches zero, the finite \
spacing to the next adjacent point, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", 
     RowBox[{"i", "+", "1"}]], "\[Equal]", 
    RowBox[{
     SubscriptBox["x", "i"], "+", "h"}]}], TraditionalForm]]],
 ", is used so that we get an approximation."
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1501572505],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "'"}], 
    SubscriptBox[
     RowBox[{"(", 
      SubscriptBox["x", "i"], ")"}], "approx"]}], " ", "==", "\[ThinSpace]", 
   FractionBox[
    RowBox[{
     RowBox[{"f", "(", 
      SubscriptBox["x", 
       RowBox[{"i", "+", "1"}]], ")"}], "-", 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}], "h"]}], 
  TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->297775955],

Cell["\<\
The difference formula can also be derived from Taylor's formula, \
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1103992639],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", 
      RowBox[{"i", "+", "1"}]], ")"}], " ", "\[Equal]", " ", 
    RowBox[{
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}], " ", "+", " ", 
     RowBox[{"h", " ", 
      RowBox[{"f", "'"}], 
      RowBox[{"(", 
       SubscriptBox["x", "i"], ")"}]}], "+", 
     RowBox[{
      FractionBox[
       RowBox[{" ", 
        SuperscriptBox["h", "2"]}], "2"], 
      RowBox[{"f", "''"}], 
      RowBox[{"(", 
       SubscriptBox["\[Xi]", "i"], ")"}]}]}]}], ";", " ", 
   RowBox[{
    SubscriptBox["x", 
     RowBox[{"i", " "}]], "<", " ", 
    SubscriptBox["\[Xi]", "i"], " ", "<", " ", 
    SubscriptBox["x", 
     RowBox[{"i", "+", "1"}]]}]}], TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->2142108005],

Cell["\<\
which is more useful since it provides an error estimate (assuming sufficient \
smoothness)\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1740545060],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "'"}], 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}], " ", "\[Equal]", " \[ThinSpace]", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"i", "+", "1"}]], ")"}], "-", 
      RowBox[{"f", "(", 
       SubscriptBox["x", "i"], ")"}]}], "h"], "-", " ", 
    RowBox[{
     FractionBox[
      RowBox[{" ", "h"}], "2"], 
     RowBox[{"f", "''"}], 
     RowBox[{"(", 
      SubscriptBox["\[Xi]", "i"], ")"}], " "}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1344786428],

Cell[TextData[{
 "An important aspect of this formula is that ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Xi]", "i"], TraditionalForm]]],
 "must lie between ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]]],
 "and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"i", "+", "1"}]], TraditionalForm]]],
 "so that the error is local to the interval enclosing the sampling points. \
It is generally true for finite difference formulas that the error is local \
to the stencil, or set of sample points.",
 " ",
 "Typically, for convergence and other analysis, the error is expressed in \
asymptotic form:"
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->806960456],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "'"}], 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}], "  ", "\[Equal]", " \[ThinSpace]", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"i", "+", "1"}]], ")"}], "-", 
      RowBox[{"f", "(", 
       SubscriptBox["x", "i"], ")"}]}], "h"], "+", 
    RowBox[{"O", "(", "h", ")"}], " "}]}], TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1199016570],

Cell[TextData[{
 "This formula is most commonly referred to as the first order forward \
difference.",
 " ",
 "The backward difference would use ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"i", "-", "1"}]], TraditionalForm]]],
 "."
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->991872195],

Cell["\<\
Taylor's formula can easily be used to derive higher order approximations. \
For example, subtracting \
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1113156301],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"f", "(", 
    SubscriptBox["x", 
     RowBox[{"i", "+", "1"}]], ")"}], " ", "\[Equal]", " ", 
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "i"], ")"}], " ", "+", " ", 
    RowBox[{"h", " ", 
     RowBox[{"f", "'"}], 
     RowBox[{"(", 
      SubscriptBox["x", "i"], ")"}]}], " ", "+", 
    RowBox[{
     FractionBox[
      RowBox[{" ", 
       SuperscriptBox["h", "2"]}], "2"], 
     RowBox[{"f", "''"}], 
     RowBox[{"(", 
      SubscriptBox["x", "i"], ")"}]}], " ", "+", " ", 
    RowBox[{"O", "(", 
     SuperscriptBox["h", "3"], ")"}]}]}], TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->120715479],

Cell["from", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1860273620],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"f", "(", 
    SubscriptBox["x", 
     RowBox[{"i", "-", "1"}]], ")"}], " ", "\[Equal]", " ", 
   RowBox[{
    RowBox[{"f", "(", 
     SubscriptBox["x", "i"], ")"}], " ", "-", " ", 
    RowBox[{"h", " ", 
     RowBox[{"f", "'"}], 
     RowBox[{"(", 
      SubscriptBox["x", "i"], ")"}]}], " ", "+", 
    RowBox[{
     FractionBox[
      RowBox[{" ", 
       SuperscriptBox["h", "2"]}], "2"], 
     RowBox[{"f", "''"}], 
     RowBox[{"(", 
      SubscriptBox["x", "i"], ")"}]}], " ", "+", " ", 
    RowBox[{"O", "(", 
     SuperscriptBox["h", "3"], ")"}]}]}], TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->127216999],

Cell[TextData[{
 "and solving for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "'"}], 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}], TraditionalForm]]],
 "gives the second order centered difference formula for the first \
derivative,"
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->742075532],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "'"}], 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}], "  ", "\[Equal]", " \[ThinSpace]", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"i", "+", "1"}]], ")"}], "-", 
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"i", "-", "1"}]], ")"}]}], 
     RowBox[{"2", "h"}]], "+", 
    RowBox[{"O", "(", 
     SuperscriptBox["h", "2"], ")"}], " "}]}], 
  TraditionalForm]], "DisplayMath",
 FontWeight->"Plain",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->759833609],

Cell["\<\
If the Taylor formulas above are expanded out one order farther and added and \
then combined with the formula just above, it is not difficult to derive a \
centered formula for the second derivative.\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1320340750],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "''"}], 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}], "  ", "\[Equal]", " \[ThinSpace]", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"i", "+", "1"}]], ")"}], "-", 
      RowBox[{"2", " ", 
       RowBox[{"f", "(", 
        SubscriptBox["x", "i"], ")"}]}], " ", "+", " ", 
      RowBox[{"f", "(", 
       SubscriptBox["x", 
        RowBox[{"i", "-", "1"}]], ")"}]}], 
     SuperscriptBox["h", "2"]], "+", 
    RowBox[{"O", "(", 
     SuperscriptBox["h", "2"], ")"}], " "}]}], 
  TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->422190388],

Cell["\<\
Note that the while having a uniform step size h between points makes it \
convenient to write out the formulas, it is certainly not a requirement. For \
example, the approximation to the second derivative is in general\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->408550449],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox["f", "\[Prime]\[Prime]",
     MultilineFunction->None], "(", 
    SubscriptBox["x", "i"], ")"}], "==", 
   RowBox[{
    FractionBox[
     RowBox[{"2", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["x", 
            RowBox[{"i", "-", "1"}]], "-", 
           SubscriptBox["x", "i"]}], ")"}]}], "+", 
        RowBox[{
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["x", "i"], "-", 
           SubscriptBox["x", 
            RowBox[{"i", "+", "1"}]]}], ")"}]}], "+", 
        RowBox[{
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}], " ", 
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["x", 
            RowBox[{"i", "+", "1"}]], "-", 
           SubscriptBox["x", 
            RowBox[{"i", "-", "1"}]]}], ")"}]}]}], ")"}]}], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["x", 
         RowBox[{"i", "-", "1"}]], "-", 
        SubscriptBox["x", "i"]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["x", 
         RowBox[{"i", "-", "1"}]], "-", 
        SubscriptBox["x", 
         RowBox[{"i", "+", "1"}]]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["x", "i"], "-", 
        SubscriptBox["x", 
         RowBox[{"i", "+", "1"}]]}], ")"}]}]], "+", 
    RowBox[{"O", "(", "h", ")"}]}]}], TraditionalForm]], "DisplayMath",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"Out[65]=",
 CellID->550912837],

Cell["\<\
where h corresponds to the maximum local grid spacing. Note that the \
asymptotic order of the three-point formula has dropped to first order; that \
it was second order on a uniform grid is due to fortuitous cancellations.\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1086329660],

Cell["\<\
In general, formulas for any given derivative with asymptotic error of any \
chosen order can be derived from the Taylor formulas as long as a sufficient \
number of sample points are used. However, this method becomes cumbersome and \
inefficient beyond the simple examples shown above. An alternate formulation \
is based on polynomial interpolation: since the Taylor formulas are exact (no \
error term) for polynomials of sufficiently low order, so are the finite \
difference formulas. It is not difficult to show that the finite difference \
formulas are equivalent to the derivatives of interpolating polynomials. For \
example, a simple way of deriving the formula shown above for the second \
derivative is to interpolate a quadratic and find its second derivative \
(which is essentially just the leading coefficient).\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->281267149],

Cell[TextData[{
 "This finds the three point finite difference formula for the second \
derivative by differentiating the polynomial interpolating the three points \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["x", 
      RowBox[{"i", "-", "1"}]], ",", " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", 
       RowBox[{"i", "-", "1"}]], ")"}]}], ")"}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["x", "i"], ",", " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}], ")"}], TraditionalForm]]],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["x", 
      RowBox[{"i", "+", "1"}]], ",", " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", 
       RowBox[{"i", "+", "1"}]], ")"}]}], ")"}], TraditionalForm]]]
}], "MathCaption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1177661315],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"InterpolatingPolynomial", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", " ", 
        RowBox[{
         SubscriptBox["x", 
          RowBox[{"i", "+", "k"}]], ",", 
         RowBox[{"f", "[", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "k"}]], "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"k", ",", 
         RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}], ",", "z"}], "]"}], 
   ",", "z", ",", "z"}], "]"}]], "Input",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"In[9]:=",
 CellID->1395531999],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      FractionBox[
       RowBox[{
        RowBox[{"-", 
         RowBox[{"f", "[", 
          SubscriptBox["x", 
           RowBox[{
            RowBox[{"-", "1"}], "+", "i"}]], "]"}]}], "+", 
        RowBox[{"f", "[", 
         SubscriptBox["x", "i"], "]"}]}], 
       RowBox[{
        RowBox[{"-", 
         SubscriptBox["x", 
          RowBox[{
           RowBox[{"-", "1"}], "+", "i"}]]}], "+", 
        SubscriptBox["x", "i"]}]]}], "+", 
     FractionBox[
      RowBox[{
       RowBox[{"-", 
        RowBox[{"f", "[", 
         SubscriptBox["x", "i"], "]"}]}], "+", 
       RowBox[{"f", "[", 
        SubscriptBox["x", 
         RowBox[{"1", "+", "i"}]], "]"}]}], 
      RowBox[{
       RowBox[{"-", 
        SubscriptBox["x", "i"]}], "+", 
       SubscriptBox["x", 
        RowBox[{"1", "+", "i"}]]}]]}], ")"}]}], 
  RowBox[{
   RowBox[{"-", 
    SubscriptBox["x", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "i"}]]}], "+", 
   SubscriptBox["x", 
    RowBox[{"1", "+", "i"}]]}]]], "Output",
 ImageSize->{208, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"Out[9]=",
 CellID->350740614]
}, Open  ]],

Cell[TextData[{
 "In this form of the formula, it is easy to see that it is effectively a \
difference of the forward and backward first order derivative approximations. \
Sometimes it is advantageous to use finite differences in this way, \
particularly for terms with coefficients inside of derivatives, such as (",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    RowBox[{
     RowBox[{
      RowBox[{"a", "(", "x", ")"}], " ", 
      SubscriptBox["u", "x"]}], ")"}], "x"], TraditionalForm]]],
 ", which commonly appear in PDEs.",
 " "
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1896863636],

Cell["\<\
Another property made apparent by considering interpolation formulas is that \
the point at which you get the derivative approximation need not be on the \
grid. A common use of this is with staggered grids where the derivative may \
be wanted at the midpoints between grid points. \
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->788062911],

Cell[TextData[{
 "This generates a fourth order approximation for the first derivative on a \
uniform staggered grid, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]]],
 ", where the main grid points ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    RowBox[{"i", "+", 
     RowBox[{"k", "/", "2"}]}]], TraditionalForm]]],
 "are at ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]]],
 "+h k/2,",
 " ",
 "for odd k. "
}], "MathCaption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1978219867],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"InterpolatingPolynomial", "[", 
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", " ", 
          RowBox[{
           RowBox[{
            SubscriptBox["x", "i"], "+", 
            RowBox[{"k", " ", 
             RowBox[{"h", "/", "2"}]}]}], ",", 
           RowBox[{"f", "[", 
            SubscriptBox["x", 
             RowBox[{"i", "+", 
              RowBox[{"k", "/", "2"}]}]], "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", 
           RowBox[{"-", "3"}], ",", "3", ",", "2"}], "}"}]}], "]"}], ",", 
       "z"}], "]"}], ",", "z"}], "]"}], " ", "/.", " ", 
   RowBox[{"z", "\[Rule]", 
    SubscriptBox["x", "i"]}]}], "]"}]], "Input",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"In[10]:=",
 CellID->856120026],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"f", "[", 
    SubscriptBox["x", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["3", "2"]}], "+", "i"}]], "]"}], "-", 
   RowBox[{"27", " ", 
    RowBox[{"f", "[", 
     SubscriptBox["x", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "2"]}], "+", "i"}]], "]"}]}], "+", 
   RowBox[{"27", " ", 
    RowBox[{"f", "[", 
     SubscriptBox["x", 
      RowBox[{
       FractionBox["1", "2"], "+", "i"}]], "]"}]}], "-", 
   RowBox[{"f", "[", 
    SubscriptBox["x", 
     RowBox[{
      FractionBox["3", "2"], "+", "i"}]], "]"}]}], 
  RowBox[{"24", " ", "h"}]]], "Output",
 ImageSize->{276, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"Out[10]=",
 CellID->809050559]
}, Open  ]],

Cell[TextData[{
 "The fourth order error coefficient for this formula is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["3", "640"], " ", 
    SuperscriptBox["h", "4"], " ", 
    RowBox[{
     SuperscriptBox["f", 
      TagBox[
       RowBox[{"(", "5", ")"}],
       Derivative],
      MultilineFunction->None], "(", 
     SubscriptBox["x", "i"], ")"}]}], TraditionalForm]]],
 " versus",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["1", "30"], " ", 
    SuperscriptBox["h", "4"], " ", 
    RowBox[{
     SuperscriptBox["f", 
      TagBox[
       RowBox[{"(", "5", ")"}],
       Derivative],
      MultilineFunction->None], "(", 
     SubscriptBox["x", "i"], ")"}]}], TraditionalForm]]],
 " for the standard fourth order formula derived below.",
 " ",
 "Much of the reduced error can be attributed to the reduced stencil size."
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->761929142],

Cell["\<\
This generates a fourth order approximation for the first derivative at a \
point on a uniform grid.\
\>", "MathCaption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1265970818],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"InterpolatingPolynomial", "[", 
      RowBox[{
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", " ", 
          RowBox[{
           RowBox[{
            SubscriptBox["x", "i"], "+", 
            RowBox[{"k", " ", "h"}]}], ",", 
           RowBox[{"f", "[", 
            SubscriptBox["x", 
             RowBox[{"i", "+", "k"}]], "]"}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", 
           RowBox[{"-", "2"}], ",", "2", ",", "1"}], "}"}]}], "]"}], ",", 
       "z"}], "]"}], ",", "z"}], "]"}], " ", "/.", " ", 
   RowBox[{"z", "\[Rule]", 
    SubscriptBox["x", "i"]}]}], "]"}]], "Input",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"In[11]:=",
 CellID->1931120345],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"f", "[", 
    SubscriptBox["x", 
     RowBox[{
      RowBox[{"-", "2"}], "+", "i"}]], "]"}], "-", 
   RowBox[{"8", " ", 
    RowBox[{"f", "[", 
     SubscriptBox["x", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "i"}]], "]"}]}], "+", 
   RowBox[{"8", " ", 
    RowBox[{"f", "[", 
     SubscriptBox["x", 
      RowBox[{"1", "+", "i"}]], "]"}]}], "-", 
   RowBox[{"f", "[", 
    SubscriptBox["x", 
     RowBox[{"2", "+", "i"}]], "]"}]}], 
  RowBox[{"12", " ", "h"}]]], "Output",
 ImageSize->{250, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"Out[11]=",
 CellID->1491349620]
}, Open  ]],

Cell["\<\
In general, a finite difference formula using n points will be exact for \
functions that are polynomials of degree n - 1 and have asymptotic order at \
least n - m. On uniform grids, you can expect higher asymptotic order, \
especially for centered differences.\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->222479809],

Cell[TextData[{
 "Using efficient polynomial interpolation techniques is a reasonable way to \
generate coefficients, but B. Fornberg has developed a fast algorithm for \
finite difference weight generation [",
 ButtonBox["F92",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21013"],
 "], [",
 ButtonBox["F98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#9836"],
 "], which is substantially faster.",
 " "
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1925744703],

Cell[TextData[{
 "In [",
 ButtonBox["F98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#9836"],
 "], he presents a one-line ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " formula for explicit finite differences."
}], "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->652766509],

Cell["\<\
This is the simple formula of Fornberg for generating weights on a uniform \
grid. Here it has been modified slightly by making it a function definition.\
\>", "MathCaption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1567266237],

Cell[BoxData[
 RowBox[{
  RowBox[{"UFDWeights", "[", 
   RowBox[{"m_", ",", " ", "n_", ",", " ", "s_"}], "]"}], " ", ":=", " ", 
  RowBox[{"CoefficientList", "[", 
   RowBox[{
    RowBox[{"Normal", "[", 
     RowBox[{
      RowBox[{"Series", "[", 
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "s"], 
         SuperscriptBox[
          RowBox[{"Log", "[", "x", "]"}], "m"]}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "1", ",", "n"}], "}"}]}], "]"}], "/", 
      SuperscriptBox["h", "m"]}], "]"}], ",", "x"}], "]"}]}]], "Input",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"In[12]:=",
 CellID->585221511],

Cell["\<\
Here m is the order of the derivative, n is the number of grid intervals \
enclosed in the stencil, and s is the number of grid intervals between the \
point at which the derivative is approximated and the leftmost edge of the \
stencil. There is no requirement that s be an integer; noninteger values \
simply lead to staggered grid approximations. Setting s to be n/2 always \
generates a centered formula. \
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1851783080],

Cell[TextData[{
 "This uses the Fornberg formula to generate the weights for a staggered \
fourth order approximation to the first derivative. This is the same one \
computed above with ",
 Cell[BoxData[
  ButtonBox["InterpolatingPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingPolynomial"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->411654785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"UFDWeights", "[", 
  RowBox[{"1", ",", " ", "3", ",", 
   RowBox[{"3", "/", "2"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"In[13]:=",
 CellID->1698377812],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", 
    RowBox[{"24", " ", "h"}]], ",", 
   RowBox[{"-", 
    FractionBox["9", 
     RowBox[{"8", " ", "h"}]]}], ",", 
   FractionBox["9", 
    RowBox[{"8", " ", "h"}]], ",", 
   RowBox[{"-", 
    FractionBox["1", 
     RowBox[{"24", " ", "h"}]]}]}], "}"}]], "Output",
 ImageSize->{166, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"Out[13]=",
 CellID->198697353]
}, Open  ]],

Cell["\<\
A table of some commonly used finite difference formulas is given below for \
reference.\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->124444757],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"FDFormula", "[", 
    RowBox[{"m_", ",", " ", "n_", ",", " ", "s_"}], "]"}], " ", ":=", " ", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", "}"}], ",", 
     RowBox[{
      RowBox[{"F", "=", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"f", "[", 
          RowBox[{"Subscript", "[", 
           RowBox[{"x", ",", 
            RowBox[{"i", "+", "k"}]}], "]"}], "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", 
           RowBox[{"-", "s"}], ",", 
           RowBox[{"n", "-", "s"}]}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"W", "=", 
       RowBox[{"PadRight", "[", 
        RowBox[{
         RowBox[{"CoefficientList", "[", 
          RowBox[{
           RowBox[{"Normal", "[", 
            RowBox[{
             RowBox[{"Series", "[", 
              RowBox[{
               RowBox[{
                RowBox[{"x", "^", "s"}], " ", 
                RowBox[{
                 RowBox[{"Log", "[", "x", "]"}], "^", "m"}]}], ",", 
               RowBox[{"{", 
                RowBox[{"x", ",", "1", ",", "n"}], "}"}]}], "]"}], "/", 
             RowBox[{"h", "^", "m"}]}], "]"}], ",", "x"}], "]"}], ",", " ", 
         RowBox[{"Length", "[", "F", "]"}], ",", " ", "0"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Print", "[", "W", "]"}], ";", "\[IndentingNewLine]", 
      RowBox[{"Wfact", "=", 
       RowBox[{"1", "/", 
        RowBox[{"Apply", "[", 
         RowBox[{"PolynomialGCD", ",", "W"}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"W", "=", 
       RowBox[{"Simplify", "[", 
        RowBox[{"W", " ", "Wfact"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"Print", "[", 
       RowBox[{"{", 
        RowBox[{"F", ",", "W"}], "}"}], "]"}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"taylor", "[", "h_", "]"}], "=", 
       RowBox[{"Normal", "[", 
        RowBox[{"Series", "[", 
         RowBox[{
          RowBox[{"f", "[", 
           RowBox[{
            RowBox[{"Subscript", "[", 
             RowBox[{"x", ",", "i"}], "]"}], "+", "h"}], "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"h", ",", "0", ",", 
            RowBox[{"n", "+", "2"}]}], "}"}]}], "]"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"error", "=", 
       RowBox[{"Drop", "[", 
        RowBox[{
         RowBox[{"CoefficientList", "[", 
          RowBox[{
           RowBox[{"Expand", "[", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{
               RowBox[{"Table", "[", 
                RowBox[{
                 RowBox[{"taylor", "[", 
                  RowBox[{"h", " ", "k"}], "]"}], ",", 
                 RowBox[{"{", 
                  RowBox[{"k", ",", 
                   RowBox[{"-", "s"}], ",", 
                   RowBox[{"n", "-", "s"}]}], "}"}]}], "]"}], ".", "W"}], 
              ")"}], "/", "Wfact"}], "]"}], ",", "h"}], "]"}], ",", "1"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"do", "=", 
       RowBox[{
        RowBox[{"Position", "[", 
         RowBox[{"error", ",", 
          RowBox[{"e_", "/;", 
           RowBox[{"e", "\[NotEqual]", "0"}]}]}], "]"}], "[", 
        RowBox[{"[", 
         RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
      
      RowBox[{"error", "=", 
       RowBox[{"error", "[", 
        RowBox[{"[", "do", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"error", "=", 
       RowBox[{"error", "/.", 
        RowBox[{
         RowBox[{"f_", "[", 
          RowBox[{"Subscript", "[", 
           RowBox[{"x", ",", "i"}], "]"}], "]"}], "\[Rule]", "f"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"error", "=", 
       RowBox[{
        RowBox[{"h", "^", "do"}], " ", "error"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"Derivative", "[", "m", "]"}], "[", "f", "]"}], "[", 
          SubscriptBox["x", "i"], "]"}], " ", "\[TildeEqual]", " ", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"F", ".", "W"}], ")"}], "/", "Wfact"}]}], ",", "error"}], 
       "}"}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{"Flatten", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Join", "[", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"FDFormula", "[", 
            RowBox[{"2", ",", "n", ",", " ", "s"}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"s", ",", "n", ",", " ", 
             RowBox[{"Ceiling", "[", 
              RowBox[{"n", "/", "2"}], "]"}], ",", " ", 
             RowBox[{"-", " ", "1"}]}], "}"}]}], "]"}], ",", 
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"FDFormula", "[", 
            RowBox[{"2", ",", "n", ",", " ", "s"}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"s", ",", 
             RowBox[{
              RowBox[{"Floor", "[", 
               RowBox[{"n", "/", "2"}], "]"}], "  ", "-", " ", "1"}], ",", 
             "0", ",", " ", 
             RowBox[{"-", " ", "1"}]}], "}"}]}], "]"}]}], "]"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"n", ",", "3", ",", "7", ",", "2"}], "}"}]}], "]"}], ",", 
     "1"}], "]"}], ",", " ", 
   RowBox[{"TableAlignments", "\[Rule]", "Center"}], ",", " ", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<Formula\>\"", ",", " ", "\"\<Error Term\>\""}], "}"}]}], 
     "}"}]}]}], "]"}]}], "Input", "ConversionLockdown",
 CellOpen->False,
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellLabel->"In[74]:=",
 CellID->1825594811],

Cell[TextData[Cell[BoxData[
 FormBox[GridBox[{
    {"Formula", 
     RowBox[{"Error", " ", "term"}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "2"}]], ")"}], "-", 
        RowBox[{"4", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"3", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}]}], 
       RowBox[{"2", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "3"], " ", 
      SuperscriptBox["h", "2"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "3", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "1"}]], ")"}], "-", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "1"}]], ")"}]}], 
       RowBox[{"2", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "6"], " ", 
      SuperscriptBox["h", "2"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "3", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "3"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"4", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "2"}]], ")"}]}], 
       RowBox[{"2", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "3"], " ", 
      SuperscriptBox["h", "2"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "3", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {" ", " "},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"3", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"16", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"36", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"48", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"25", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}]}], 
       RowBox[{"12", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "5"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "5", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"-", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"6", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"18", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"3", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}]}], 
       RowBox[{"12", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "20"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "5", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "2"}]], ")"}], "-", 
        RowBox[{"8", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"8", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "2"}]], ")"}]}], 
       RowBox[{"12", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "30"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "5", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "3"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"18", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"6", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "3"}]], ")"}]}], 
       RowBox[{"12", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "20"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "5", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "25"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"48", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"36", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"16", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "-", 
        RowBox[{"3", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}]}], 
       RowBox[{"12", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "5"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "5", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {" ", " "},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "6"}]], ")"}]}], "-", 
        RowBox[{"72", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "5"}]], ")"}]}], "+", 
        RowBox[{"225", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"400", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"450", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"360", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"147", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "7"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "2"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "5"}]], ")"}]}], "+", 
        RowBox[{"15", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"50", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"100", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"150", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"77", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "42"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "4"}]], ")"}], "-", 
        RowBox[{"8", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"30", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"80", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"35", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"24", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "105"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"-", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"9", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"45", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"45", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"9", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "3"}]], ")"}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "140"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"24", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"35", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"80", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"30", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"8", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "-", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "4"}]], ")"}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "105"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "10"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"77", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"150", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"100", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"50", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "-", 
        RowBox[{"15", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}], "+", 
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "5"}]], ")"}]}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "42"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "147"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"360", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"450", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"400", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "-", 
        RowBox[{"225", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}], "+", 
        RowBox[{"72", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "5"}]], ")"}]}], "-", 
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "6"}]], ")"}]}]}], 
       RowBox[{"60", " ", "h"}]]}], 
     RowBox[{
      FractionBox["1", "7"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "7", ")"}],
        Derivative],
       MultilineFunction->None]}]}
   },
   GridBoxAlignment->{
    "Columns" -> {{Center}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.28], {
        Offset[2.1]}, 
       Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}], TraditionalForm]]]], "FramedBox",
 Background->RGBColor[0.964706, 0.929412, 0.839216],
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->567090444],

Cell["\<\
Finite difference formulas on uniform grids for the first derivative.\
\>", "Caption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->884278366],

Cell[TextData[Cell[BoxData[
 FormBox[GridBox[{
    {"Formula", 
     RowBox[{"Error", " ", "Term"}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"-", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"4", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"5", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}]}], 
       SuperscriptBox["h", "2"]]}], 
     RowBox[{
      FractionBox["11", "12"], "  ", 
      SuperscriptBox["h", "2"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "4", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "1"}]], ")"}], "-", 
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "1"}]], ")"}]}], 
       SuperscriptBox["h", "2"]]}], 
     RowBox[{
      FractionBox["1", "12"], " ", 
      SuperscriptBox["h", "2"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "4", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "-", 
        RowBox[{"5", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "+", 
        RowBox[{"4", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "-", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "3"}]], ")"}]}], 
       SuperscriptBox["h", "2"]]}], 
     RowBox[{
      FractionBox["11", "12"], " ", 
      SuperscriptBox["h", "2"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "4", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {" ", " "},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "10"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "5"}]], ")"}]}], "+", 
        RowBox[{"61", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"156", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"214", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"154", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"45", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}]}], 
       RowBox[{"12", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["137", "180"], "  ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "6", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "4"}]], ")"}], "-", 
        RowBox[{"6", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"14", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"4", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"15", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}]}], 
       RowBox[{"12", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["13", "180"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "6", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"-", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "+", 
        RowBox[{"16", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"30", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"16", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "2"}]], ")"}]}], 
       RowBox[{"12", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["1", "90"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "6", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"15", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "-", 
        RowBox[{"4", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "+", 
        RowBox[{"14", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "-", 
        RowBox[{"6", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "+", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", "+", "4"}]], ")"}]}], 
       RowBox[{"12", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["13", "180"], " ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "6", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"45", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "-", 
        RowBox[{"154", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "+", 
        RowBox[{"214", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "-", 
        RowBox[{"156", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "+", 
        RowBox[{"61", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}], "-", 
        RowBox[{"10", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "5"}]], ")"}]}]}], 
       RowBox[{"12", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["137", "180"], "  ", 
      SuperscriptBox["h", "4"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "6", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {" ", " "},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "126"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "7"}]], ")"}]}], "+", 
        RowBox[{"1019", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "6"}]], ")"}]}], "-", 
        RowBox[{"3618", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "5"}]], ")"}]}], "+", 
        RowBox[{"7380", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"9490", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"7911", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"4014", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "+", 
        RowBox[{"938", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["363", "560"], "  ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"11", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "6"}]], ")"}]}], "-", 
        RowBox[{"90", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "5"}]], ")"}]}], "+", 
        RowBox[{"324", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"670", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"855", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "-", 
        RowBox[{"486", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"70", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"126", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["29", "560"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "2"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "5"}]], ")"}]}], "+", 
        RowBox[{"16", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "4"}]], ")"}]}], "-", 
        RowBox[{"54", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "+", 
        RowBox[{"85", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "+", 
        RowBox[{"130", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"378", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"214", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"11", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["47", "5040"], 
      SuperscriptBox["h", "6"], "  ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "3"}]], ")"}]}], "-", 
        RowBox[{"27", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "+", 
        RowBox[{"270", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"490", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"270", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "-", 
        RowBox[{"27", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "+", 
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["1", "560"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{
         RowBox[{"-", "11"}], " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "2"}]], ")"}]}], "+", 
        RowBox[{"214", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"378", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "+", 
        RowBox[{"130", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "+", 
        RowBox[{"85", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "-", 
        RowBox[{"54", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "+", 
        RowBox[{"16", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}], "-", 
        RowBox[{"2", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "5"}]], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["47", "5040"], 
      SuperscriptBox["h", "6"], "  ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"126", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "-", "1"}]], ")"}]}], "-", 
        RowBox[{"70", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "-", 
        RowBox[{"486", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "+", 
        RowBox[{"855", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "-", 
        RowBox[{"670", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "+", 
        RowBox[{"324", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}], "-", 
        RowBox[{"90", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "5"}]], ")"}]}], "+", 
        RowBox[{"11", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "6"}]], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["29", "560"], " ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]},
    {
     RowBox[{
      RowBox[{
       SuperscriptBox["f", "\[Prime]\[Prime]",
        MultilineFunction->None], "(", 
       SubscriptBox["x", "i"], ")"}], "\[TildeEqual]", 
      FractionBox[
       RowBox[{
        RowBox[{"938", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", "i"], ")"}]}], "-", 
        RowBox[{"4014", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "1"}]], ")"}]}], "+", 
        RowBox[{"7911", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "2"}]], ")"}]}], "-", 
        RowBox[{"9490", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "3"}]], ")"}]}], "+", 
        RowBox[{"7380", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "4"}]], ")"}]}], "-", 
        RowBox[{"3618", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "5"}]], ")"}]}], "+", 
        RowBox[{"1019", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "6"}]], ")"}]}], "-", 
        RowBox[{"126", " ", 
         RowBox[{"f", "(", 
          SubscriptBox["x", 
           RowBox[{"i", "+", "7"}]], ")"}]}]}], 
       RowBox[{"180", " ", 
        SuperscriptBox["h", "2"]}]]}], 
     RowBox[{
      FractionBox["363", "560"], "  ", 
      SuperscriptBox["h", "6"], " ", 
      SuperscriptBox["f", 
       TagBox[
        RowBox[{"(", "8", ")"}],
        Derivative],
       MultilineFunction->None]}]}
   },
   GridBoxAlignment->{
    "Columns" -> {{Center}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{"Columns" -> {
       Offset[0.28], {
        Offset[2.1]}, 
       Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
       Offset[0.2], {
        Offset[0.4]}, 
       Offset[0.2]}, "RowsIndexed" -> {}}], TraditionalForm]]]], "FramedBox",
 Background->RGBColor[0.964706, 0.929412, 0.839216],
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->998451913],

Cell["\<\
Finite difference formulas on uniform grids for the second derivative.\
\>", "Caption",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->989280832],

Cell["\<\
One thing to notice from this table is that the farther the formulas get from \
centered, the larger the error term coefficient, sometimes by factors of \
hundreds. For this reason, sometimes where one-sided derivative formulas are \
required (such as at boundaries), formulas of higher order are used to offset \
the extra error.\
\>", "Text",
 CellTags->{"b:7.1.0", "ndsg:2.0.1.0"},
 CellID->1292828815]
}, Open  ]],

Cell[CellGroupData[{

Cell["NDSolve`FiniteDifferenceDerivative", "Subsubsection",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1", "PDE:1.2.2"},
 CellID->1549364750],

Cell[TextData[{
 "Fornberg [",
 ButtonBox["F92",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21013"],
 "], [",
 ButtonBox["F98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#9836"],
 "] also gives an algorithm that, though not quite so elegant and simple, is \
more general and, in particular, is applicable to nonuniform grids.",
 " ",
 "It is not difficult to program in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", but to make it as efficient as possible, a new kernel function has been \
provided as a simpler interface (along with some additional features)."
}], "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->2075610493],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     StyleBox["NDSolve`FiniteDifferenceDerivative", "InlineCode"], "[", 
     ButtonBox["Derivative",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Derivative"], "[", 
     StyleBox["m", "TI"], "]", ",", 
     StyleBox["grid", "TI"], ",", 
     StyleBox["values", "TI"], "]"}], Cell[TextData[{
     "approximate the ",
     Cell[BoxData[
      SuperscriptBox["m", "th"]], "InlineFormula"],
     " order derivative for the function that takes on ",
     StyleBox["values", "TI"],
     " on the ",
     StyleBox["grid", "TI"],
     "."
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["NDSolve`FiniteDifferenceDerivative", "InlineCode"], "[", 
     ButtonBox["Derivative",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Derivative"], "[", 
     StyleBox["m1", "TI"], ",", 
     StyleBox["m2", "TI"], ",", 
     StyleBox["...", "TI"], ",", 
     StyleBox["mn", "TI"], "]", ",", "{", 
     StyleBox["grid1", "TI"], ",", 
     StyleBox["grid2", "TI"], ",", 
     StyleBox["...", "TI"], ",", 
     StyleBox["gridn", "TI"], "}", ",", 
     StyleBox["values", "TI"], "]"}], Cell[TextData[{
     "approximate the partial derivative of ",
     "order ",
     Cell[BoxData[
      RowBox[{"(", 
       StyleBox["m1", "TI"], ",", 
       StyleBox["m2", "TI"], ",", 
       StyleBox["...", "TI"], ",", 
       StyleBox["mn", "TI"], ")"}]], "InlineCode"],
     " for the function of n variables which takes on ",
     StyleBox["values", "TI"],
     " on the tensor product grid defined by the outer product of ",
     Cell[BoxData[
      RowBox[{"(", 
       StyleBox["grid1", "TI"], ",", 
       StyleBox["grid2", "TI"], ",", 
       StyleBox["...", "TI"], ",", 
       StyleBox["gridn", "TI"], ")"}]], "InlineCode"],
     "."
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["NDSolve`FiniteDifferenceDerivative", "InlineCode"], "[", 
     ButtonBox["Derivative",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Derivative"], "[", 
     StyleBox["m1", "TI"], ",", 
     StyleBox["m2", "TI"], ",", 
     StyleBox["...", "TI"], ",", 
     StyleBox["mn", "TI"], "]", ",", "{", 
     StyleBox["grid1", "TI"], ",", 
     StyleBox["grid2", "TI"], ",", 
     StyleBox["...", "TI"], ",", 
     StyleBox["gridn", "TI"], "}", "]"}], Cell[TextData[{
     "compute the finite difference weights needed to approximate the partial \
derivative of order ",
     Cell[BoxData[
      RowBox[{"(", 
       StyleBox["m1", "TI"], ",", 
       StyleBox["m2", "TI"], ",", 
       StyleBox["...", "TI"], ",", 
       StyleBox["mn", "TI"], ")"}]], "InlineCode"],
     " for the function of n variables on the tensor product grid defined by \
the outer product of ",
     Cell[BoxData[
      RowBox[{"(", 
       StyleBox["grid1", "TI"], ",", 
       StyleBox["grid2", "TI"], ",", 
       StyleBox["...", "TI"], ",", 
       StyleBox["gridn", "TI"], ")"}]], "InlineCode"],
     ".  The result is returned as an ",
     StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "InlineCode"],
     ", ",
     "which can be repeatedly applied to values on the grid."
    }], "TableText"]}
  }]], "DefinitionBox",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->1574095964],

Cell["Finding finite difference approximations to derivatives.", "Caption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->579542417],

Cell[TextData[StyleBox["This defines a uniform grid with points spaced apart \
by a symbolic distance h.", "MathCaption"]], "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->1323658844],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ugrid", " ", "=", " ", 
  RowBox[{"h", " ", 
   RowBox[{"Range", "[", 
    RowBox[{"0", ",", "8"}], "]"}]}]}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[14]:=",
 CellID->497700729],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "h", ",", 
   RowBox[{"2", " ", "h"}], ",", 
   RowBox[{"3", " ", "h"}], ",", 
   RowBox[{"4", " ", "h"}], ",", 
   RowBox[{"5", " ", "h"}], ",", 
   RowBox[{"6", " ", "h"}], ",", 
   RowBox[{"7", " ", "h"}], ",", 
   RowBox[{"8", " ", "h"}]}], "}"}]], "Output",
 ImageSize->{241, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[14]=",
 CellID->636217446]
}, Open  ]],

Cell["\<\
This gives the first derivative formulas on the grid for a symbolic function \
f.\
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->2111721755],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
  RowBox[{
   RowBox[{"Derivative", "[", "1", "]"}], ",", "ugrid", ",", " ", 
   RowBox[{"Map", "[", 
    RowBox[{"f", ",", " ", "ugrid"}], "]"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[15]:=",
 CellID->328436948],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{501, 186},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[15]=",
 CellID->19541732]
}, Open  ]],

Cell[TextData[{
 "The derivatives at the endpoints are computed using one-sided formulas. The \
formulas shown in the example above are fourth order accurate, which is the \
default. In general, when you use symbolic grid and/or data, you will get \
symbolic formulas. This is often useful for doing analysis on the methods; \
however, for actual numerical grids, it is usually faster and more accurate \
to give the numerical grid to ",
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 " rather than using the symbolic formulas."
}], "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->393785947],

Cell["\<\
This defines a randomly spaced grid between 0 and 2 \[Pi].\
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->1090366091],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"rgrid", " ", "=", " ", 
  RowBox[{"Sort", "[", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", " ", 
       RowBox[{"2", " ", "\[Pi]"}]}], "}"}], ",", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"2", "\[Pi]", " ", 
        RowBox[{"RandomReal", "[", "]"}]}], ",", 
       RowBox[{"{", "10", "}"}]}], "]"}]}], "]"}], "]"}]}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[16]:=",
 CellID->303529602],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0.9436696735617042`", ",", "1.0049988763341355`", ",", 
   "1.0887270822598292`", ",", "1.7205159930876173`", ",", 
   "1.787759121148989`", ",", "2.4157408467548636`", ",", 
   "2.4911873107400315`", ",", "2.9324750337231738`", ",", 
   "4.445080086163461`", ",", "6.2062082855683105`", ",", 
   RowBox[{"2", " ", "\[Pi]"}]}], "}"}]], "Output",
 ImageSize->{337, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[16]=",
 CellID->80198218]
}, Open  ]],

Cell["\<\
This approximates the derivative of the sine function at each point on the \
grid.\
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->2587540],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
  RowBox[{
   RowBox[{"Derivative", "[", "1", "]"}], ",", "rgrid", ",", " ", 
   RowBox[{"Sin", "[", "rgrid", "]"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[17]:=",
 CellID->526009411],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9898909038815946`", ",", "0.5868516049179241`", ",", 
   "0.5360718544911441`", ",", "0.46360055844536724`", ",", 
   RowBox[{"-", "0.14915190939619183`"}], ",", 
   RowBox[{"-", "0.2152124559876297`"}], ",", 
   RowBox[{"-", "0.747841603562488`"}], ",", 
   RowBox[{"-", "0.7955016402301233`"}], ",", 
   RowBox[{"-", "0.970650214406914`"}], ",", 
   RowBox[{"-", "0.24750295561624655`"}], ",", "0.9976904894568318`", ",", 
   "0.9991307625577804`"}], "}"}]], "Output",
 ImageSize->{427, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[17]=",
 CellID->1627271874]
}, Open  ]],

Cell["This shows the error in the approximations.", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->655354050],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Cos", "[", "rgrid", "]"}]}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[18]:=",
 CellID->2093275648],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "0.01010909611840538`"}], ",", "0.00003101903311042431`", ",",
    
   RowBox[{"-", "0.000017308774474367183`"}], ",", 
   RowBox[{"-", "0.000013036566470836508`"}], ",", "9.031347163235282`*^-6", 
   ",", "0.00005216386557133079`", ",", "0.00009268362326653179`", ",", 
   "0.0003367849439039894`", ",", "0.007564260422777136`", ",", 
   "0.016633908933810587`", ",", "0.00065175771140491`", ",", 
   RowBox[{"-", "0.0008692374422195615`"}]}], "}"}]], "Output",
 ImageSize->{435, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[18]=",
 CellID->585718310]
}, Open  ]],

Cell[TextData[{
 "In multiple dimensions, ",
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 " works on tensor product grids, and you only need to specify the grid \
points for each dimension. "
}], "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->359653482],

Cell["\<\
This defines grids xgrid and ygrid for the x and y direction, gives an \
approximation for the mixed xy partial derivative of the Gaussian on the \
tensor product of xgrid and ygrid, and makes a surface plot of the error. \
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->357446785],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"xgrid", " ", "=", " ", 
   RowBox[{"Range", "[", 
    RowBox[{"0", ",", "8"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ygrid", " ", "=", " ", 
   RowBox[{"Range", "[", 
    RowBox[{"0", ",", "10"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"gaussian", "[", 
    RowBox[{"x_", ",", " ", "y_"}], "]"}], " ", "=", " ", 
   RowBox[{"Exp", "[", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", " ", "-", " ", "4"}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", " ", "-", " ", "5"}], ")"}], "2"]}], ")"}]}], "/", 
     "10"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"values", " ", "=", "  ", 
   RowBox[{"Outer", "[", 
    RowBox[{"gaussian", ",", "xgrid", ",", " ", "ygrid"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"ListPlot3D", "[", 
  RowBox[{
   RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}], ",", "values"}], "]"}], 
   " ", "-", " ", 
   RowBox[{"Outer", "[", 
    RowBox[{
     RowBox[{"Function", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"x", ",", "y"}], "}"}], ",", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"gaussian", "[", 
           RowBox[{"x", ",", "y"}], "]"}], ",", "x", ",", "y"}], "]"}], 
        "]"}]}], "]"}], ",", " ", "xgrid", ",", " ", "ygrid"}], "]"}]}], 
  "]"}]}], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[19]:=",
 CellID->91799375],

Cell[BoxData[
 Graphics3DBox[GraphicsComplex3DBox[CompressedData["
1:eJx1mnt0U1UWxmMpEEqAkKQlhDRN0gcv0SogDJTeA7RSQKSUKjioUxwQUNEK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   "], {{
     {EdgeForm[None], GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJwtlnegz1UYxs85MhJlRGTU1eXebNfmWtktLU1pDyoaUmlQt4k0JO09haJk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         "]], Polygon3DBox[CompressedData["
1:eJwtlwcUl2MUxu/7b2hIpSKVVEpKqWhoadA0UlYyUvZMHDtURlYqUlS2jGyi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         "]], Polygon3DBox[CompressedData["
1:eJwtlGVsFlEQRfctFCvWIsUCtFCKFXco0tKmuP0AEkIICZbgHtzdgjUUCRoo
7lLc3YtLcQ3uzpnM/DjpvdPdb3dn7rzQjr1a9vQ9z+sAAQLmuvO8IrANvxVC
qL3AN4F7+LvgU7uKLwDr8evgH/oS5IJV+JWQh+te4pvDA3wqpKGWgi8Em/Ab
YST6CuSDtfhqnjzE8y7j88Aa7GqYgr4JxWE3PgbScd0NfDHYid8Bf9AXIAiW
4ZdCBq67hS8Jyfg9kJHabXwk7Mfvg7HoaxAKW/BR0A19Av6iZ0AYDMZfhByw
Al8BfqPPQ1ZYgl8MP9FnIRMk4hdAf/Q5yAyL8JHwA30G0kMCfj70QJ+GAJiH
j4Cv6JPgwSz+zIT8fMde/GN0Z+gCraWX8ALdW/oE3/CnIA3Mwc+W70HfgbJw
EB8Pn9DH4Ad6MkyCAjxjH7Vn6O7ybvAZfxx+o6fBVCjp63ekc3qf3N9XZg2B
MB0fCqV87UtGp/fJ/fGWtcYyP/xFaGAZagaX8VegjK/zze70G+RbuqA3wB/J
E+S2GW2GYJhr/Yv0dUZZnPZOeliL2l18OaeZl+xHUbuHr+g0U5KtOGrP8PEy
K/xZebb8HyrDLnw01OW6B/gaTjMl2UpC74EWcBXfRrIgmYVacMB6X4d7U/FV
nWZUsjpcsgh5YSG+NNSw7JZ2ukOyS+V83cGcTvOTYPPdDmWkF5bn6rYLJZzu
ruxwRdvp/E5zK/mtZLta0OkOyS6Vt70McZrlRNuZrVAYltsuSHY3QlqYIPmR
/vua3X/o0TAGivqate/ogTAIwn3N1S/0MPl+qGJnU5jTs0XOmKq2++FOzwc5
J4r7mnEfxuHHS37R2yACkuQ+mIe+D1Vkt/FNIdByUB4O4w9BLpt5A7iFv2lz
ewhRsu/4VpDZ8lJJdgd/FIKpPcZHO82t5DeI2iN8Haf5lpznpPYUHwfX8ddg
Jfo5NJL9xLeHjuj18BXdR34LCnHvQWpv0ENgKIRRO0LtM7on9JKdo3aY2gd0
V+gG2SynNZ1mWTKd1fJXTc4K/EkoSO0A/hW6PwyAdpIdeIceIb2Td8Qfgvfo
URAiOZK9gBg5K/BN4K3Td35rs5UZv3H6jNeWgYHWg93QEM5Zn2OtV7FO303e
Mdp6WtvpzGR2Mdb7ek5nITN55fScfGLfL32Qc3I/vET3s54ulWugPqTg28Jr
p+ffc+un9PW9055+tCyPthkdhW/oiZb7VvhkeITuJFmBD05n9MUyKln9Dy0D
BEE=
         "]]}]}, {}, {}, {}}, {
     Line3DBox[{100, 1, 339, 12, 350, 23, 361, 34, 372, 45, 383, 56, 394, 67, 
      405, 78, 416, 89, 212, 90, 213, 214, 91, 215, 92, 216, 217, 93, 218, 94,
       219, 95, 220, 221, 96, 222, 97, 223, 224, 98, 225, 99, 425, 88, 414, 
      77, 403, 66, 392, 55, 381, 44, 370, 33, 359, 22, 348, 11, 126, 10, 123, 
      122, 9, 120, 8, 117, 116, 7, 114, 6, 112, 5, 109, 108, 4, 106, 3, 103, 
      102, 2, 100}], {{}, 
      Line3DBox[{12, 101, 13, 104, 105, 14, 107, 15, 110, 111, 16, 113, 17, 
       115, 18, 118, 119, 19, 121, 20, 124, 125, 21, 127, 22}], 
      Line3DBox[{339, 442, 538, 338, 490, 443, 539, 340, 229, 341, 491, 444, 
       540, 342, 492, 426, 343, 445, 541, 344, 493, 446, 542, 345, 236, 346, 
       494, 447, 543, 347, 495, 427, 348}], 
      Line3DBox[{23, 128, 24, 129, 130, 25, 131, 26, 132, 133, 27, 134, 28, 
       135, 29, 136, 137, 30, 138, 31, 139, 140, 32, 141, 33}], 
      Line3DBox[{350, 448, 544, 349, 496, 449, 545, 351, 243, 352, 497, 450, 
       546, 353, 498, 428, 354, 451, 547, 355, 499, 452, 548, 356, 250, 357, 
       500, 453, 549, 358, 501, 429, 359}], 
      Line3DBox[{34, 142, 35, 143, 144, 36, 145, 37, 146, 147, 38, 148, 39, 
       149, 40, 150, 151, 41, 152, 42, 153, 154, 43, 155, 44}], 
      Line3DBox[{361, 454, 550, 360, 502, 455, 551, 362, 257, 363, 503, 456, 
       552, 364, 504, 430, 365, 457, 553, 366, 505, 458, 554, 367, 264, 368, 
       506, 459, 555, 369, 507, 431, 370}], 
      Line3DBox[{45, 156, 46, 157, 158, 47, 159, 48, 160, 161, 49, 162, 50, 
       163, 51, 164, 165, 52, 166, 53, 167, 168, 54, 169, 55}], 
      Line3DBox[{372, 460, 556, 371, 508, 461, 557, 373, 271, 374, 509, 462, 
       558, 375, 510, 432, 376, 463, 559, 377, 511, 464, 560, 378, 278, 379, 
       512, 465, 561, 380, 513, 433, 381}], 
      Line3DBox[{56, 170, 57, 171, 172, 58, 173, 59, 174, 175, 60, 176, 61, 
       177, 62, 178, 179, 63, 180, 64, 181, 182, 65, 183, 66}], 
      Line3DBox[{383, 466, 562, 382, 514, 467, 563, 384, 285, 385, 515, 468, 
       564, 386, 516, 434, 387, 469, 565, 388, 517, 470, 566, 389, 292, 390, 
       518, 471, 567, 391, 519, 435, 392}], 
      Line3DBox[{67, 184, 68, 185, 186, 69, 187, 70, 188, 189, 71, 190, 72, 
       191, 73, 192, 193, 74, 194, 75, 195, 196, 76, 197, 77}], 
      Line3DBox[{394, 472, 568, 393, 520, 473, 569, 395, 299, 396, 521, 474, 
       570, 397, 522, 436, 398, 475, 571, 399, 523, 476, 572, 400, 306, 401, 
       524, 477, 573, 402, 525, 437, 403}], 
      Line3DBox[{78, 198, 79, 199, 200, 80, 201, 81, 202, 203, 82, 204, 83, 
       205, 84, 206, 207, 85, 208, 86, 209, 210, 87, 211, 88}], 
      Line3DBox[{405, 478, 574, 404, 526, 479, 575, 406, 313, 407, 527, 480, 
       576, 408, 528, 438, 409, 481, 577, 410, 529, 482, 578, 411, 320, 412, 
       530, 483, 579, 413, 531, 439, 414}], 
      Line3DBox[{416, 484, 580, 415, 532, 485, 581, 417, 327, 418, 533, 486, 
       582, 419, 534, 440, 420, 487, 583, 421, 535, 488, 584, 422, 334, 423, 
       536, 489, 585, 424, 537, 441, 425}]}, {{}, 
      Line3DBox[{100, 538, 226, 101, 544, 240, 128, 550, 254, 142, 556, 268, 
       156, 562, 282, 170, 568, 296, 184, 574, 310, 198, 580, 324, 212}], 
      Line3DBox[{102, 227, 490, 104, 241, 496, 129, 255, 502, 143, 269, 508, 
       157, 283, 514, 171, 297, 520, 185, 311, 526, 199, 325, 532, 213}], 
      Line3DBox[{106, 229, 107, 243, 131, 257, 145, 271, 159, 285, 173, 299, 
       187, 313, 201, 327, 215}], 
      Line3DBox[{108, 230, 491, 110, 244, 497, 132, 258, 503, 146, 272, 509, 
       160, 286, 515, 174, 300, 521, 188, 314, 527, 202, 328, 533, 216}], 
      Line3DBox[{112, 232, 492, 113, 246, 498, 134, 260, 504, 148, 274, 510, 
       162, 288, 516, 176, 302, 522, 190, 316, 528, 204, 330, 534, 218}], 
      Line3DBox[{6, 343, 17, 354, 28, 365, 39, 376, 50, 387, 61, 398, 72, 409,
        83, 420, 94}], 
      Line3DBox[{114, 541, 233, 115, 547, 247, 135, 553, 261, 149, 559, 275, 
       163, 565, 289, 177, 571, 303, 191, 577, 317, 205, 583, 331, 219}], 
      Line3DBox[{116, 234, 493, 118, 248, 499, 136, 262, 505, 150, 276, 511, 
       164, 290, 517, 178, 304, 523, 192, 318, 529, 206, 332, 535, 220}], 
      Line3DBox[{120, 236, 121, 250, 138, 264, 152, 278, 166, 292, 180, 306, 
       194, 320, 208, 334, 222}], 
      Line3DBox[{122, 237, 494, 124, 251, 500, 139, 265, 506, 153, 279, 512, 
       167, 293, 518, 181, 307, 524, 195, 321, 530, 209, 335, 536, 223}], 
      Line3DBox[{126, 239, 495, 127, 253, 501, 141, 267, 507, 155, 281, 513, 
       169, 295, 519, 183, 309, 525, 197, 323, 531, 211, 337, 537, 225}], 
      Line3DBox[{214, 326, 581, 200, 312, 575, 186, 298, 569, 172, 284, 563, 
       158, 270, 557, 144, 256, 551, 130, 242, 545, 105, 228, 539, 103}], 
      Line3DBox[{217, 329, 582, 203, 315, 576, 189, 301, 570, 175, 287, 564, 
       161, 273, 558, 147, 259, 552, 133, 245, 546, 111, 231, 540, 109}], 
      Line3DBox[{221, 333, 584, 207, 319, 578, 193, 305, 572, 179, 291, 566, 
       165, 277, 560, 151, 263, 554, 137, 249, 548, 119, 235, 542, 117}], 
      Line3DBox[{224, 336, 585, 210, 322, 579, 196, 308, 573, 182, 294, 567, 
       168, 280, 561, 154, 266, 555, 140, 252, 549, 125, 238, 543, 
       123}]}, {}, {}}},
   VertexNormals->CompressedData["
1:eJxdWnlYTtv3L0MZSqLpKjRPmudB72qeNapUrpIhhCSJ0G2UIYlEykwyDylK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    "]],
  Axes->True,
  BoxRatios->{1, 1, 0.4},
  ImageMargins->0.,
  ImageSize->Medium,
  Method->{
   "RotationControl" -> "Globe", "SpherePoints" -> Automatic, 
    "CylinderPoints" -> Automatic},
  PlotRange->{{1., 11.}, {1., 9.}, {-0.00315111641114599, 
   0.00315111641114599}},
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02], 
    Scaled[0.02]},
  ViewAngle->Automatic,
  ViewCenter->Automatic,
  ViewPoint->{1.3, -2.4, 2.},
  ViewVertical->{0., 0., 1.}]], "Output",
 ImageSize->{364, 277},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[23]=",
 CellID->88989795,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell["\<\
Note that the values need to be given in a matrix corresponding to the outer \
product of the grid coordinates. \
\>", "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->481891542],

Cell[TextData[{
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 " does not compute weights for sums of derivatives. This means that for \
common operators like the Laplacian, you need to combine two approximations."
}], "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->1917394187],

Cell["\<\
This makes a function that approximates the Laplacian operator on a tensor \
product grid.\
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->751758837],

Cell[BoxData[
 RowBox[{
  RowBox[{"lap", "[", 
   RowBox[{"values_", ",", " ", 
    RowBox[{"{", 
     RowBox[{"xgrid_", ",", " ", "ygrid_"}], "}"}]}], "]"}], " ", ":=", " ", 
  RowBox[{
   RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"2", ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}], ",", " ", "values"}], 
    "]"}], "+", " ", 
   RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}], ",", " ", "values"}], 
    "]"}]}]}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[24]:=",
 CellID->1366169908],

Cell["\<\
This uses the function to approximate the Laplacian for the same grid and \
Gaussian function used in the previous example.\
\>", "MathCaption",
 CellID->768362452],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot3D", "[", 
  RowBox[{"lap", "[", 
   RowBox[{"values", ",", " ", 
    RowBox[{"{", 
     RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->1741039670],

Cell[BoxData[
 Graphics3DBox[GraphicsComplex3DBox[CompressedData["
1:eJx1mnt0VNUVxocwQMAJhMlkmFwmd+6dTMJDRBYiipDMLqK8tKQIBZqK4SWi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   "], {{
     {EdgeForm[None], GraphicsGroup3DBox[{Polygon3DBox[CompressedData["
1:eJwtlnegz1UYxs85MhJlRGTU1eXebNfmWtktLU1pDyoaUmlQt4k0JO09haJk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         "]], Polygon3DBox[CompressedData["
1:eJwtlwcUl2MUxu/7b2hIpSKVVEpKqWhoadA0UlYyUvZMHDtURlYqUlS2jGyi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         "]], Polygon3DBox[CompressedData["
1:eJwtlGVsFlEQRfctFCvWIsUCtFCKFXco0tKmuP0AEkIICZbgHtzdgjUUCRoo
7lLc3YtLcQ3uzpnM/DjpvdPdb3dn7rzQjr1a9vQ9z+sAAQLmuvO8IrANvxVC
qL3AN4F7+LvgU7uKLwDr8evgH/oS5IJV+JWQh+te4pvDA3wqpKGWgi8Em/Ab
YST6CuSDtfhqnjzE8y7j88Aa7GqYgr4JxWE3PgbScd0NfDHYid8Bf9AXIAiW
4ZdCBq67hS8Jyfg9kJHabXwk7Mfvg7HoaxAKW/BR0A19Av6iZ0AYDMZfhByw
Al8BfqPPQ1ZYgl8MP9FnIRMk4hdAf/Q5yAyL8JHwA30G0kMCfj70QJ+GAJiH
j4Cv6JPgwSz+zIT8fMde/GN0Z+gCraWX8ALdW/oE3/CnIA3Mwc+W70HfgbJw
EB8Pn9DH4Ad6MkyCAjxjH7Vn6O7ybvAZfxx+o6fBVCjp63ekc3qf3N9XZg2B
MB0fCqV87UtGp/fJ/fGWtcYyP/xFaGAZagaX8VegjK/zze70G+RbuqA3wB/J
E+S2GW2GYJhr/Yv0dUZZnPZOeliL2l18OaeZl+xHUbuHr+g0U5KtOGrP8PEy
K/xZebb8HyrDLnw01OW6B/gaTjMl2UpC74EWcBXfRrIgmYVacMB6X4d7U/FV
nWZUsjpcsgh5YSG+NNSw7JZ2ukOyS+V83cGcTvOTYPPdDmWkF5bn6rYLJZzu
ruxwRdvp/E5zK/mtZLta0OkOyS6Vt70McZrlRNuZrVAYltsuSHY3QlqYIPmR
/vua3X/o0TAGivqate/ogTAIwn3N1S/0MPl+qGJnU5jTs0XOmKq2++FOzwc5
J4r7mnEfxuHHS37R2yACkuQ+mIe+D1Vkt/FNIdByUB4O4w9BLpt5A7iFv2lz
ewhRsu/4VpDZ8lJJdgd/FIKpPcZHO82t5DeI2iN8Haf5lpznpPYUHwfX8ddg
Jfo5NJL9xLeHjuj18BXdR34LCnHvQWpv0ENgKIRRO0LtM7on9JKdo3aY2gd0
V+gG2SynNZ1mWTKd1fJXTc4K/EkoSO0A/hW6PwyAdpIdeIceIb2Td8Qfgvfo
URAiOZK9gBg5K/BN4K3Td35rs5UZv3H6jNeWgYHWg93QEM5Zn2OtV7FO303e
Mdp6WtvpzGR2Mdb7ek5nITN55fScfGLfL32Qc3I/vET3s54ulWugPqTg28Jr
p+ffc+un9PW9055+tCyPthkdhW/oiZb7VvhkeITuJFmBD05n9MUyKln9Dy0D
BEE=
         "]]}]}, {}, {}, {}}, {
     Line3DBox[{100, 1, 339, 12, 350, 23, 361, 34, 372, 45, 383, 56, 394, 67, 
      405, 78, 416, 89, 212, 90, 213, 214, 91, 215, 92, 216, 217, 93, 218, 94,
       219, 95, 220, 221, 96, 222, 97, 223, 224, 98, 225, 99, 425, 88, 414, 
      77, 403, 66, 392, 55, 381, 44, 370, 33, 359, 22, 348, 11, 126, 10, 123, 
      122, 9, 120, 8, 117, 116, 7, 114, 6, 112, 5, 109, 108, 4, 106, 3, 103, 
      102, 2, 100}], {{}, 
      Line3DBox[{12, 101, 13, 104, 105, 14, 107, 15, 110, 111, 16, 113, 17, 
       115, 18, 118, 119, 19, 121, 20, 124, 125, 21, 127, 22}], 
      Line3DBox[{339, 442, 538, 338, 490, 443, 539, 340, 229, 341, 491, 444, 
       540, 342, 492, 426, 343, 445, 541, 344, 493, 446, 542, 345, 236, 346, 
       494, 447, 543, 347, 495, 427, 348}], 
      Line3DBox[{23, 128, 24, 129, 130, 25, 131, 26, 132, 133, 27, 134, 28, 
       135, 29, 136, 137, 30, 138, 31, 139, 140, 32, 141, 33}], 
      Line3DBox[{350, 448, 544, 349, 496, 449, 545, 351, 243, 352, 497, 450, 
       546, 353, 498, 428, 354, 451, 547, 355, 499, 452, 548, 356, 250, 357, 
       500, 453, 549, 358, 501, 429, 359}], 
      Line3DBox[{34, 142, 35, 143, 144, 36, 145, 37, 146, 147, 38, 148, 39, 
       149, 40, 150, 151, 41, 152, 42, 153, 154, 43, 155, 44}], 
      Line3DBox[{361, 454, 550, 360, 502, 455, 551, 362, 257, 363, 503, 456, 
       552, 364, 504, 430, 365, 457, 553, 366, 505, 458, 554, 367, 264, 368, 
       506, 459, 555, 369, 507, 431, 370}], 
      Line3DBox[{45, 156, 46, 157, 158, 47, 159, 48, 160, 161, 49, 162, 50, 
       163, 51, 164, 165, 52, 166, 53, 167, 168, 54, 169, 55}], 
      Line3DBox[{372, 460, 556, 371, 508, 461, 557, 373, 271, 374, 509, 462, 
       558, 375, 510, 432, 376, 463, 559, 377, 511, 464, 560, 378, 278, 379, 
       512, 465, 561, 380, 513, 433, 381}], 
      Line3DBox[{56, 170, 57, 171, 172, 58, 173, 59, 174, 175, 60, 176, 61, 
       177, 62, 178, 179, 63, 180, 64, 181, 182, 65, 183, 66}], 
      Line3DBox[{383, 466, 562, 382, 514, 467, 563, 384, 285, 385, 515, 468, 
       564, 386, 516, 434, 387, 469, 565, 388, 517, 470, 566, 389, 292, 390, 
       518, 471, 567, 391, 519, 435, 392}], 
      Line3DBox[{67, 184, 68, 185, 186, 69, 187, 70, 188, 189, 71, 190, 72, 
       191, 73, 192, 193, 74, 194, 75, 195, 196, 76, 197, 77}], 
      Line3DBox[{394, 472, 568, 393, 520, 473, 569, 395, 299, 396, 521, 474, 
       570, 397, 522, 436, 398, 475, 571, 399, 523, 476, 572, 400, 306, 401, 
       524, 477, 573, 402, 525, 437, 403}], 
      Line3DBox[{78, 198, 79, 199, 200, 80, 201, 81, 202, 203, 82, 204, 83, 
       205, 84, 206, 207, 85, 208, 86, 209, 210, 87, 211, 88}], 
      Line3DBox[{405, 478, 574, 404, 526, 479, 575, 406, 313, 407, 527, 480, 
       576, 408, 528, 438, 409, 481, 577, 410, 529, 482, 578, 411, 320, 412, 
       530, 483, 579, 413, 531, 439, 414}], 
      Line3DBox[{416, 484, 580, 415, 532, 485, 581, 417, 327, 418, 533, 486, 
       582, 419, 534, 440, 420, 487, 583, 421, 535, 488, 584, 422, 334, 423, 
       536, 489, 585, 424, 537, 441, 425}]}, {{}, 
      Line3DBox[{100, 538, 226, 101, 544, 240, 128, 550, 254, 142, 556, 268, 
       156, 562, 282, 170, 568, 296, 184, 574, 310, 198, 580, 324, 212}], 
      Line3DBox[{102, 227, 490, 104, 241, 496, 129, 255, 502, 143, 269, 508, 
       157, 283, 514, 171, 297, 520, 185, 311, 526, 199, 325, 532, 213}], 
      Line3DBox[{106, 229, 107, 243, 131, 257, 145, 271, 159, 285, 173, 299, 
       187, 313, 201, 327, 215}], 
      Line3DBox[{108, 230, 491, 110, 244, 497, 132, 258, 503, 146, 272, 509, 
       160, 286, 515, 174, 300, 521, 188, 314, 527, 202, 328, 533, 216}], 
      Line3DBox[{112, 232, 492, 113, 246, 498, 134, 260, 504, 148, 274, 510, 
       162, 288, 516, 176, 302, 522, 190, 316, 528, 204, 330, 534, 218}], 
      Line3DBox[{6, 343, 17, 354, 28, 365, 39, 376, 50, 387, 61, 398, 72, 409,
        83, 420, 94}], 
      Line3DBox[{114, 541, 233, 115, 547, 247, 135, 553, 261, 149, 559, 275, 
       163, 565, 289, 177, 571, 303, 191, 577, 317, 205, 583, 331, 219}], 
      Line3DBox[{116, 234, 493, 118, 248, 499, 136, 262, 505, 150, 276, 511, 
       164, 290, 517, 178, 304, 523, 192, 318, 529, 206, 332, 535, 220}], 
      Line3DBox[{120, 236, 121, 250, 138, 264, 152, 278, 166, 292, 180, 306, 
       194, 320, 208, 334, 222}], 
      Line3DBox[{122, 237, 494, 124, 251, 500, 139, 265, 506, 153, 279, 512, 
       167, 293, 518, 181, 307, 524, 195, 321, 530, 209, 335, 536, 223}], 
      Line3DBox[{126, 239, 495, 127, 253, 501, 141, 267, 507, 155, 281, 513, 
       169, 295, 519, 183, 309, 525, 197, 323, 531, 211, 337, 537, 225}], 
      Line3DBox[{214, 326, 581, 200, 312, 575, 186, 298, 569, 172, 284, 563, 
       158, 270, 557, 144, 256, 551, 130, 242, 545, 105, 228, 539, 103}], 
      Line3DBox[{217, 329, 582, 203, 315, 576, 189, 301, 570, 175, 287, 564, 
       161, 273, 558, 147, 259, 552, 133, 245, 546, 111, 231, 540, 109}], 
      Line3DBox[{221, 333, 584, 207, 319, 578, 193, 305, 572, 179, 291, 566, 
       165, 277, 560, 151, 263, 554, 137, 249, 548, 119, 235, 542, 117}], 
      Line3DBox[{224, 336, 585, 210, 322, 579, 196, 308, 573, 182, 294, 567, 
       168, 280, 561, 154, 266, 555, 140, 252, 549, 125, 238, 543, 
       123}]}, {}, {}}},
   VertexNormals->CompressedData["
1:eJxdWnk0Vd/br1BJGqRBlAYqDUoIKU9SadJAhQYZEmUoU1Iyz5GZkMxknjN7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    "]],
  Axes->True,
  BoxRatios->{1, 1, 0.4},
  ImageMargins->0.,
  ImageSize->Automatic,
  Method->{
   "RotationControl" -> "Globe", "SpherePoints" -> Automatic, 
    "CylinderPoints" -> Automatic},
  PlotRange->{{1., 11.}, {1., 9.}, {-0.397640452486763, 0.0571163757528755}},
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02], 
    Scaled[0.02]},
  ViewAngle->Automatic,
  ViewCenter->Automatic,
  ViewPoint->{1.3, -2.4, 2.},
  ViewVertical->{0., 0., 1.}]], "Output",
 ImageSize->{364, 281},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->1244269874,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{"option", " ", "name"}], "TableHeader"], 
    StyleBox[
     RowBox[{"default", " ", "value"}], "TableHeader"], " "},
   {
    StyleBox["DifferenceOrder", "InlineCode"], 
    StyleBox["4", "InlineCode"], Cell[
    "asymptotic order of the error", "TableText"]},
   {
    StyleBox["PeriodicInterpolation", "InlineCode"], 
    ButtonBox["False",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/False"], Cell["\<\
whether to consider the values as those of a periodic function with the \
period equal to the interval enclosed by the grid\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellTags->"b:7.1.1ndsg:2.0.1.1",
 CellID->170243555],

Cell[TextData[{
 "Options for ",
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 "."
}], "Caption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->1669166515],

Cell["\<\
This approximates the derivatives for the sine function on the random grid \
defined above, assuming that the function repeats periodically.\
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->2085611040],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
  RowBox[{"1", ",", "rgrid", ",", " ", 
   RowBox[{"Sin", "[", "rgrid", "]"}], ",", " ", 
   RowBox[{"PeriodicInterpolation", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[26]:=",
 CellID->749785972],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.9989496974212897`", ",", "0.5867647603237631`", ",", 
   "0.5360718544911441`", ",", "0.46360055844536724`", ",", 
   RowBox[{"-", "0.14915190939619183`"}], ",", 
   RowBox[{"-", "0.2152124559876297`"}], ",", 
   RowBox[{"-", "0.747841603562488`"}], ",", 
   RowBox[{"-", "0.7955016402301233`"}], ",", 
   RowBox[{"-", "0.970650214406914`"}], ",", 
   RowBox[{"-", "0.24750295561624655`"}], ",", "0.994584731949651`", ",", 
   "0.9989496974212897`"}], "}"}]], "Output",
 ImageSize->{427, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[26]=",
 CellID->787779803]
}, Open  ]],

Cell[TextData[{
 "When using ",
 StyleBox["PeriodicInterpolation\[Rule]True", "MR"],
 ", you can omit the last point in the values since it should always be the \
same as the first. This feature is useful when solving a PDE with periodic \
boundary conditions."
}], "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->435828121],

Cell["\<\
This generates second order finite difference formulas for the first \
derivative of a symbolic function.\
\>", "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->1078708378],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
  RowBox[{"1", ",", " ", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", 
      RowBox[{"-", "1"}]], ",", 
     SubscriptBox["x", "0"], ",", 
     SubscriptBox["x", "1"]}], "}"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["f", 
      RowBox[{"-", "1"}]], ",", 
     SubscriptBox["f", "0"], ",", " ", 
     SubscriptBox["f", "1"]}], "}"}], ",", " ", 
   RowBox[{"DifferenceOrder", "\[Rule]", "2"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"In[27]:=",
 CellID->809579818],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{508, 146},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellLabel->"Out[27]=",
 CellID->333133293]
}, Open  ]],

Cell[TextData[{
 "Fourth order differences typically provide a good balance between \
truncation (approximation) error and roundoff error for machine precision. \
However, there are some applications where fourth order differences produce \
excessive oscillation (Gibb's phenomena), so second order differences are \
better. Also, for high precision, higher order differences may be \
appropriate. Even values of ",
 StyleBox["DifferenceOrder", "MR"],
 " use centered formulas, which typically have smaller error coefficients \
than non-centered formulas, so even values are recommended when appropriate."
}], "Text",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->116267162]
}, Open  ]],

Cell[CellGroupData[{

Cell["NDSolve`FiniteDifferenceDerivativeFunction", "Subsubsection",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1", "PDE:1.2.3"},
 CellID->1203365766],

Cell[TextData[{
 "When computing the solution to a PDE, it is common to repeatedly apply the \
same finite difference approximation to different values on the same grid. A \
significant savings can be made by storing the necessary weight computations \
and applying them to the changing data.",
 " ",
 "When you omit the (third) argument with function values in ",
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 ", the result will be an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 ", which is a data object that stores the weight computations in a efficient \
form for future repeated use."
}], "Text",
 CellID->1854631161],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     StyleBox["NDSolve`FiniteDifferenceDerivative", "InlineCode"], "[", "{", 
     StyleBox["m1", "TI"], ",", 
     StyleBox["m2", "TI"], ",", 
     StyleBox["...", "TI"], "}", ",", "{", 
     StyleBox["grid1", "TI"], ",", 
     StyleBox["grid2", "TI"], ",", 
     StyleBox["...", "TI"], "}", "]"}], Cell[TextData[{
     "compute the finite difference weights needed to approximate the partial \
derivative of order ",
     Cell[BoxData[
      RowBox[{"(", 
       StyleBox["m1", "TI"], ",", 
       StyleBox["m2", "TI"], ",", 
       StyleBox["...", "TI"], ")"}]], "InlineCode"],
     " for the function of n variables on the tensor product grid defined by \
the outer product of ",
     Cell[BoxData[
      RowBox[{"(", 
       StyleBox["grid1", "TI"], ",", 
       StyleBox["grid2", "TI"], ",", 
       StyleBox["...", "TI"], ")"}]], "InlineCode"],
     ".  The result is returned as an ",
     StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "InlineCode"],
     " object."
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "InlineCode"], 
     "[", 
     ButtonBox["Derivative",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Derivative"], "[", 
     StyleBox["m", "TI"], "]", ",", 
     StyleBox["data", "TI"], "]"}], Cell[TextData[{
     "a data object that contains the weights and other data needed to \
quickly approximate the ",
     Cell[BoxData[
      SuperscriptBox["m", "th"]], "InlineCode"],
     " order derivative of a function. In the standard output form, only the \
",
     Cell[BoxData[
      RowBox[{
       ButtonBox["Derivative",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Derivative"], "[", 
       StyleBox["m", "TI"], "]"}]], "InlineCode"],
     " operator it approximates is shown. "
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "InlineCode"], 
     "[", 
     StyleBox["data", "TI"], "]", "[", 
     StyleBox["values", "TI"], "]"}], Cell[TextData[{
     "approximate the derivative of the function that takes on ",
     StyleBox["values", "TI"],
     " on the grid used to determine ",
     StyleBox["data", "TI"],
     "."
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->419796764],

Cell["Computing finite difference weights for repeated use. ", "Caption",
 CellID->1380788411],

Cell["\<\
This defines a uniform grid with 25 points on the unit interval and evaluates \
the sine function with one period on the grid.\
\>", "MathCaption",
 CellID->1857218701],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"n", " ", "=", " ", "24"}], ";"}], " "}], "\n", 
 RowBox[{
  RowBox[{"grid", " ", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"Range", "[", 
      RowBox[{"0", ",", "n"}], "]"}], "/", "n"}], "]"}]}], ";"}], "\n", 
 RowBox[{"values", " ", "=", " ", 
  RowBox[{"Sin", "[", 
   RowBox[{"2", " ", "\[Pi]", " ", "grid"}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->945806175],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.`", ",", "0.25881904510252074`", ",", "0.49999999999999994`", ",", 
   "0.7071067811865475`", ",", "0.8660254037844386`", ",", 
   "0.9659258262890683`", ",", "1.`", ",", "0.9659258262890683`", ",", 
   "0.8660254037844387`", ",", "0.7071067811865476`", ",", 
   "0.49999999999999994`", ",", "0.258819045102521`", ",", 
   "1.2246467991473532`*^-16", ",", 
   RowBox[{"-", "0.25881904510252035`"}], ",", 
   RowBox[{"-", "0.5000000000000001`"}], ",", 
   RowBox[{"-", "0.7071067811865475`"}], ",", 
   RowBox[{"-", "0.8660254037844384`"}], ",", 
   RowBox[{"-", "0.9659258262890683`"}], ",", 
   RowBox[{"-", "1.`"}], ",", 
   RowBox[{"-", "0.9659258262890684`"}], ",", 
   RowBox[{"-", "0.8660254037844386`"}], ",", 
   RowBox[{"-", "0.7071067811865477`"}], ",", 
   RowBox[{"-", "0.5000000000000004`"}], ",", 
   RowBox[{"-", "0.2588190451025207`"}], ",", 
   RowBox[{"-", "2.4492935982947064`*^-16"}]}], "}"}]], "Output",
 ImageSize->{560, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->127136990]
}, Open  ]],

Cell[TextData[{
 "This defines an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 ", which can be repeatedly applied to different values on the grid to \
approximate the second derivative."
}], "MathCaption",
 CellTags->{"b:7.1.1", "ndsg:2.0.1.1"},
 CellID->365089103],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fddf", " ", "=", " ", 
  RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
   RowBox[{
    RowBox[{"Derivative", "[", "2", "]"}], ",", " ", "grid"}], 
   "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->891598176],

Cell[BoxData[
 TagBox[
  RowBox[{"NDSolve`FiniteDifferenceDerivativeFunction", "[", 
   RowBox[{
    RowBox[{"Derivative", "[", "2", "]"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{429, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->222425732]
}, Open  ]],

Cell["\<\
Note that the standard output form is abbreviated and only shows the \
derivative operators that are approximated. \
\>", "Text",
 CellID->724742585],

Cell["\<\
This computes the approximation to the second derivative of the sine \
function.\
\>", "MathCaption",
 CellID->1249044961],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fddf", "[", "values", "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->1969176697],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.07202673938179843`", ",", 
   RowBox[{"-", "10.224832144498144`"}], ",", 
   RowBox[{"-", "19.73818479164852`"}], ",", 
   RowBox[{"-", "27.91400862897582`"}], ",", 
   RowBox[{"-", "34.187538908318665`"}], ",", 
   RowBox[{"-", "38.13124490863909`"}], ",", 
   RowBox[{"-", "39.47636958329719`"}], ",", 
   RowBox[{"-", "38.13124490863901`"}], ",", 
   RowBox[{"-", "34.18753890831874`"}], ",", 
   RowBox[{"-", "27.914008628975914`"}], ",", 
   RowBox[{"-", "19.73818479164831`"}], ",", 
   RowBox[{"-", "10.217236279663524`"}], ",", "3.3968661217187446`*^-13", ",",
    "10.21723627966258`", ",", "19.738184791649115`", ",", 
   "27.914008628975846`", ",", "34.18753890831835`", ",", 
   "38.13124490863927`", ",", "39.476369583297085`", ",", 
   "38.13124490863926`", ",", "34.18753890831844`", ",", 
   "27.914008628975765`", ",", "19.738184791649203`", ",", 
   "10.224832144497753`", ",", 
   RowBox[{"-", "0.07202673938650653`"}]}], "}"}]], "Output",
 ImageSize->{564, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->16600336]
}, Open  ]],

Cell[TextData[{
 "This function is only applicable for values defined on the particular grid \
used to construct it. If your problem requires changing the grid, you will \
need to use ",
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 " to generate weights each time the grid changes.",
 " ",
 "However, when you can use ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 " objects, evaluation will be substantially faster. "
}], "Text",
 CellID->1730936504],

Cell["\<\
This compares timings for computing the Laplacian with the function just \
defined and with the definition of the previous section. A loop is used to \
repeat the calculation in each case because it is too fast for the \
differences to show up with Timing. \
\>", "MathCaption",
 CellID->1739087236],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"repeats", " ", "=", " ", "10000"}], ";"}], "\n", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{"First", "[", 
    RowBox[{"Timing", "[", 
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{"fddf", "[", "values", "]"}], ",", " ", 
       RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"First", "[", 
    RowBox[{"Timing", "[", 
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
        RowBox[{
         RowBox[{"Derivative", "[", "2", "]"}], ",", " ", "grid", ",", " ", 
         "values"}], "]"}], ",", 
       RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], "]"}]}], 
  "}"}]}], "Input",
 CellLabel->"In[9]:=",
 CellID->1661169816],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.047000000000000014`", ",", "2.2499999999999996`"}], 
  "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->562473107]
}, Open  ]],

Cell[TextData[{
 "An ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 " can be used repeatedly in many situations.",
 " ",
 "As a simple example, consider a collocation method for solving the boundary \
value problem"
}], "Text",
 CellID->359966041],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["u", 
      StyleBox["xx",
       FontSlant->"Italic"]], " ", "+", " ", 
     RowBox[{
      RowBox[{"sin", "(", "x", ")"}], " ", "u"}]}], " ", "=", " ", 
    RowBox[{"\[Lambda]", " ", "u"}]}], ";", " ", 
   RowBox[{
    RowBox[{"u", "(", "0", ")"}], " ", "=", " ", 
    RowBox[{
     RowBox[{"u", "(", "1", ")"}], " ", "=", " ", "0"}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->1254725838],

Cell["\<\
on the unit interval. (This simple method is not necessarily the best way to \
solve this particular problem, but it is useful as an example.)\
\>", "Text",
 CellID->1461137840],

Cell[TextData[{
 "This defines a function that will have all components zero at an \
approximate solution of the boundary value problem. Using the intermediate \
vector v and setting its endpoints (parts {1,-1}) to 0 is a fast and simple \
trick to enforce the boundary conditions. Evaluation is prevented except for \
numbers \[Lambda] because this would not work otherwise. (Also, because ",
 Cell[BoxData[
  ButtonBox["Times",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Times"]], "InlineFormula"],
 " is ",
 Cell[BoxData[
  ButtonBox["Listable",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Listable"]], "InlineFormula"],
 ", Sin[2 Pi grid] u would thread componentwise.)"
}], "MathCaption",
 CellID->530209596],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Clear", "[", "fun", "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"fun", "[", 
   RowBox[{"u_", ",", " ", 
    RowBox[{"\[Lambda]_", "?", "NumberQ"}]}], "]"}], " ", ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"n", " ", "=", " ", 
       RowBox[{"Length", "[", "u", "]"}]}], ",", " ", 
      RowBox[{"v", " ", "=", "  ", 
       RowBox[{
        RowBox[{"fddf", "[", "u", "]"}], " ", "+", 
        RowBox[{
         RowBox[{"(", " ", 
          RowBox[{
           RowBox[{"Sin", "[", "grid", "]"}], " ", "-", " ", "\[Lambda]"}], 
          ")"}], " ", "u"}]}]}]}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"v", "[", 
       RowBox[{"[", 
        RowBox[{"{", 
         RowBox[{"1", ",", 
          RowBox[{"-", "1"}]}], "}"}], "]"}], "]"}], " ", "=", " ", "0."}], 
     ";", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"v", ",", " ", 
       RowBox[{
        RowBox[{"u", ".", "u"}], " ", "-", " ", "1"}]}], "}"}]}]}], 
   "]"}]}]}], "Input",
 CellLabel->"In[11]:=",
 CellID->776408705],

Cell[TextData[{
 "This uses ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 " to find an approximate eigenfunction using the constant coefficient case \
for a starting value and shows a plot of the eigenfunction."
}], "MathCaption",
 CellID->1399265233],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"s4", " ", "=", " ", 
   RowBox[{"FindRoot", "[", 
    RowBox[{
     RowBox[{"fun", "[", 
      RowBox[{"u", ",", " ", "\[Lambda]"}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"u", ",", "values"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"\[Lambda]", ",", " ", 
       RowBox[{
        RowBox[{"-", "4"}], " ", 
        SuperscriptBox["\[Pi]", "2"]}]}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"grid", ",", " ", 
      RowBox[{"u", " ", "/.", " ", "s4"}]}], "}"}], "]"}], ",", " ", 
   RowBox[{"PlotLabel", "\[Rule]", 
    RowBox[{"ToString", "[", 
     RowBox[{"Last", "[", "s4", "]"}], "]"}]}]}], "]"}]}], "Input",
 CellLabel->"In[13]:=",
 CellID->1279995568],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], 
    PointBox[{{0., -0.008890569689287585}, {0.041666666666666664`, 
     0.06641815875611598}, {0.08333333333333333, 0.1372076694472921}, {0.125, 
     0.19864451477918124`}, {0.16666666666666666`, 0.24652685705710012`}, {
     0.20833333333333334`, 0.277570314603884}, {0.25, 0.2896352461419325}, {
     0.2916666666666667, 0.2818767917923523}, {0.3333333333333333, 
     0.254806759216142}, {0.375, 0.2102626186435709}, {0.4166666666666667, 
     0.15128524957464515`}, {0.4583333333333333, 0.08191338558501098}, {0.5, 
     0.0069085059819111614`}, {0.5416666666666666, -0.06857118236962573}, {
     0.5833333333333334, -0.1393253911807572}, {
     0.625, -0.20047062481240438`}, {
     0.6666666666666666, -0.2477784453468528}, {
     0.7083333333333334, -0.2779695807702858}, {
     0.75, -0.28894326756305394`}, {
     0.7916666666666666, -0.2799256158508527}, {
     0.8333333333333334, -0.25152634309560934`}, {
     0.875, -0.205699560861234}, {0.9166666666666666, -0.145610992515709}, {
     0.9583333333333334, -0.07542077024999388}, {
     1., -2.4492935982947064`*^-16}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotLabel->FormBox["\"\[Lambda] -> -39.4004\"", TraditionalForm],
  PlotRange->{{0., 1.}, {-0.28894326756305394`, 0.2896352461419325}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 238},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->796501582]
}, Open  ]],

Cell[TextData[{
 "With the setup for this problem is so simple, it is easy to compare various \
alternatives. For example, to compare the solution above, which used the \
default fourth order differences, to the usual use of second order \
differences, all that needs to be changed is the ",
 StyleBox["DifferenceOrder", "MR"],
 "."
}], "Text",
 CellID->211220299],

Cell["\<\
This solves the boundary value problem using second order differences and \
shows a plot of the difference between it and the fourth order solution.\
\>", "MathCaption",
 CellID->283846587],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"fddf", " ", "=", " ", 
   RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
    RowBox[{
     RowBox[{"Derivative", "[", "2", "]"}], ",", " ", "grid", ",", " ", 
     RowBox[{"DifferenceOrder", "\[Rule]", "2"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"s2", " ", "=", " ", 
   RowBox[{"FindRoot", "[", 
    RowBox[{
     RowBox[{"fun", "[", 
      RowBox[{"u", ",", " ", "\[Lambda]"}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"u", ",", "values"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"\[Lambda]", ",", " ", 
       RowBox[{
        RowBox[{"-", "4"}], " ", 
        SuperscriptBox["\[Pi]", "2"]}]}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"grid", ",", " ", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"u", " ", "/.", " ", "s4"}], ")"}], " ", "-", " ", 
      RowBox[{"(", 
       RowBox[{"u", " ", "/.", " ", "s2"}], ")"}]}]}], "}"}], "]"}], 
  "]"}]}], "Input",
 CellLabel->"In[39]:=",
 CellID->451874776],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWTtvE0EQXvvOdkSALgIpFAYJCpACAhLTmIcQb0EFoqCyIiSngKAQ0VKk
JQUlCFEBoqBDgKhdBCoQoqCioOU3oOX2Hp7b2bm5vfXZuYAl3+3efDM7M7vz
3Xp9tbfav32nt7q02GtfWOnd6y8t3m+fX14JHnk1IWp9IcS33SJoy6AVfvdI
KYWnWuEnvF1WDeEimQ0lL198CYTrjwah5EpKcvPGq0By7uwzQ/L5069A8uH9
j5SkXN+UJBGMZiUt2ZZEfKr7JIm4yyCDDATIJAMcMsjI8t23SUZopI3HreBa
DwQeQiTP1L2ZqR1KfJAI4atbM/0kutaRDTwe2IxE0UeNLhpDO7GgyBiNuHtx
bP5m2Gxgm55LJrUn2KdG7qw247sf37M1yq8m1wrcJc2aOULqRki9ZjikXjM0
sooZqZpP/2Ls+6W56vYxSH3VcUh91UVIjnV99Az3IQYbzsAjZFuz5F2D13yW
VzJ8nDIZN7oqvBVnG8zo9KYqaIXiVbvs/n8VVTWJa+xHpckMMwxSZwYOqTMD
jdxauZq8hGNSVc9O7JS56xKsNUsGzeXn4vyHeafwbrVcayPtpzHH5v8K2fxV
OErNnJYmv7QYpM4vHFLnFxq5ubmaRrHj/lymrWmUC9zP04Tc4D6t+Qd5dolB
gRd4V7mX0QIP8A6T1lKXA0gzG0nPyglpsqfiKdVcSMtg1xcjOoSmataxJpJ1
4tnHfDDcTQ3Xh0Lq3JJmXuXDvIHAHHHcQJhsZ4PpILuegfDI/Jox2Hnok9GZ
fh2U+vpqWejMIR3skXtsLYQ4RnuTmsEF6xihGswY1bpQ6+cwQk4yssQHOiKl
ekjqtWo/V6BTXkTmWOZ6y44J74HGWUMJRyX7gYR58JzMp57X43goPe232wTe
emckUW9iJo75uioDQ29HTn9njny7jCtBROfpefg8edE+aS9+h+C3fdlRw38F
TXJOyh6lSrnF+6Hyosb9Bpnbkzn9Luq3k1zG/Dabo1/OqHl4ehR12Tq/Klwk
38M+1OcKg4KVxqFgZmlUFbPgIrmFMvebQUHmOBRkjkaZ+dlAPjwk9TaQDxwK
fKBR48rulMw/08k+abXZV+EzndGsYT/5s42ipy/KV/1EyNVKIZ8sznCybSYz
iM9w7P9//Cj11bzOoGA106g+snVN6qub0wLbPxkU2AJU1TkrW/IG5espg4L8
cCjID6D4/DxAPnxlUOADhwIfaJRLpp4jL18zKPCSQ4GXNGoNjTjQ8rmGRhow
NvAJGNh8R2o9RrY5FNimUS65Bomo/QUs4Kvj\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 210},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->402972293]
}, Open  ]],

Cell["\<\
One way to determine which is the better solution is to study the convergence \
as the grid is refined. This will be considered to some extent in the section \
on differentiation matrices below. \
\>", "Text",
 CellID->1487212353],

Cell[TextData[{
 "While the most vital information about an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 " object, the derivative order, is displayed in its output form, sometimes \
it is useful to extract this and other information from an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 ", say for use in a program. The structure of the way the data is stored may \
change between versions of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", so extracting the information by using parts of the expression is not \
recommended.",
 " ",
 "A better alternative is to use any of the several method functions provided \
for this purpose. "
}], "Text",
 CellID->159022289],

Cell[TextData[{
 "Let ",
 StyleBox["FDDF", "MR",
  FontSlant->"Italic"],
 " represent an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction[", "MR"],
 StyleBox["data", "MR",
  FontSlant->"Italic"],
 StyleBox["]", "MR"],
 " object."
}], "Text",
 CellID->747284226],

Cell[BoxData[GridBox[{
   {Cell[TextData[{
     StyleBox["FDDF", "TI"],
     StyleBox["@\"DerivativeOrder\"", "InlineCode"]
    }], "TableText"], Cell[TextData[{
     "get the derivative order which ",
     StyleBox["FDDF", "TI"],
     " approximates."
    }], "TableText"]},
   {Cell[TextData[{
     StyleBox["FDDF", "TI"],
     StyleBox["@\"DifferenceOrder\"", "InlineCode"]
    }], "TableText"], Cell["\<\
get the list with the difference order used for the approximation in each \
dimension.  \
\>", "TableText"]},
   {Cell[TextData[{
     StyleBox["FDDF", "TI"],
     StyleBox["@\"PeriodicInterpolation\"", "InlineCode"]
    }], "TableText"], Cell[TextData[{
     "get the list with elements ",
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"],
     " or ",
     ButtonBox["False",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/False"],
     " indicating whether periodic interpolation is used for each dimension."
    }], "TableText"]},
   {Cell[TextData[{
     StyleBox["FDDF", "TI"],
     StyleBox["@\"Coordinates\"", "InlineCode"]
    }], "TableText"], Cell[
    "get the list with the grid coordinates in each dimension.", 
     "TableText"]},
   {Cell[TextData[{
     StyleBox["FDDF", "TI"],
     StyleBox["@\"Grid\"", "InlineCode"]
    }]], Cell["\<\
form the tensor of the grid points. This is the outer product of the grid \
coordinates. \
\>", "TableText"]},
   {Cell[TextData[{
     StyleBox["FDDF", "TI"],
     StyleBox["@\"DifferentiationMatrix\"", "InlineCode"]
    }], "TableText"], Cell[TextData[{
     "compute the sparse differentiation matrix ",
     StyleBox["mat", "TI"],
     " such that ",
     Cell[BoxData[
      RowBox[{
       StyleBox["mat", "TI"], 
       StyleBox[".", "InlineCode"], 
       ButtonBox["Flatten",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Flatten"], "[", 
       StyleBox["values", "TI"], "]"}]], "InlineCode"],
     " is equivalent to ",
     Cell[BoxData[
      RowBox[{
       ButtonBox["Flatten",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Flatten"], "[", 
       StyleBox["FDDF", "TI"], "[", 
       StyleBox["values", "TI"], "]", "]"}]], "InlineCode"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{GridBoxItemSize->{"Columns" -> {
     Scaled[0.4], {
      Scaled[0.6]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->720517116],

Cell[TextData[{
 "Method functions for exacting information from an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction[", "MR"],
 StyleBox["data", "MR",
  FontSlant->"Italic"],
 StyleBox["]", "MR"],
 " object."
}], "Caption",
 CellID->2016240991],

Cell[TextData[{
 "Any of the method functions that return a list with an element for each of \
the dimensions can be used with an integer argument ",
 StyleBox["dim",
  FontSlant->"Italic"],
 ", which will return only the value for that particular dimension such that \
",
 StyleBox["FDDF", "MR",
  FontSlant->"Italic"],
 StyleBox["@", "MR"],
 StyleBox["method", "MR",
  FontSlant->"Italic"],
 StyleBox["[dim]", "MR"],
 " = ",
 StyleBox["(FDDF@", "MR"],
 StyleBox["method", "MR",
  FontSlant->"Italic"],
 StyleBox[")[[", "MR"],
 StyleBox["dim", "MR",
  FontSlant->"Italic"],
 StyleBox["]]", "MR"],
 "."
}], "Text",
 CellID->1693090401],

Cell["\<\
The examples below show how you might use some of these methods.\
\>", "Text",
 CellID->2056989227],

Cell["\<\
Here is an NDSolve`FiniteDifferenceDerivativeFunction object created with \
random grids having between 10 and 16 points in each dimension.\
\>", "MathCaption",
 CellID->233846446],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fddf", " ", "=", " ", 
  RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
   RowBox[{
    RowBox[{"Derivative", "[", 
     RowBox[{"0", ",", "1", ",", "2"}], "]"}], ",", " ", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"Sort", "[", 
       RowBox[{"Join", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.", ",", "1."}], "}"}], ",", " ", 
         RowBox[{"Table", "[", 
          RowBox[{
           RowBox[{"RandomReal", "[", "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"RandomInteger", "[", 
             RowBox[{"{", 
              RowBox[{"8", ",", "14"}], "}"}], "]"}], "}"}]}], "]"}]}], "]"}],
        "]"}], ",", 
      RowBox[{"{", "3", "}"}]}], "]"}], ",", " ", 
    RowBox[{"PeriodicInterpolation", "\[Rule]", "True"}]}], "]"}]}]], "Input",\

 CellLabel->"In[15]:=",
 CellID->437204929],

Cell[BoxData[
 TagBox[
  RowBox[{"NDSolve`FiniteDifferenceDerivativeFunction", "[", 
   RowBox[{
    RowBox[{"Derivative", "[", 
     RowBox[{"0", ",", "1", ",", "2"}], "]"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{465, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->423455823]
}, Open  ]],

Cell["This shows the dimensions of the outer product grid. ", "MathCaption",
 CellID->958589032],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Dimensions", "[", 
  RowBox[{"tpg", " ", "=", " ", 
   RowBox[{"fddf", "@", "\"\<Grid\>\""}]}], "]"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->1686249891],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"15", ",", "10", ",", "11", ",", "3"}], "}"}]], "Output",
 ImageSize->{101, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->233783400]
}, Open  ]],

Cell[TextData[{
 "Note that the rank of the grid point tensor is one more than the \
dimensionality of the tensor product. This is because the 3 coordinates \
defining each point are in a list themselves. If you have a function that \
depends on the grid variables, you can use ",
 StyleBox["Apply[f, fddf[\"Grid\"], {n}]", "MR"],
 " where n = ",
 StyleBox["Length[fddf[\"DerivativeOrder\"]]", "MR"],
 " is the dimensionality of the space in which you are approximating the \
derivative."
}], "Text",
 CellID->1868000011],

Cell[TextData[{
 "This defines a Gaussian function of 3 variables and applies it to the grid \
on which the ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 " is defined."
}], "MathCaption",
 CellID->1269702747],

Cell[BoxData[{
 RowBox[{
  RowBox[{"f", " ", "=", " ", 
   RowBox[{"Function", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "y", ",", "z"}], "}"}], ",", " ", 
     RowBox[{"Exp", "[", 
      RowBox[{"-", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{"x", "-", ".5"}], ")"}], "^", "2"}], " ", "+", " ", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"y", "-", ".5"}], ")"}], "^", "2"}], " ", "+", " ", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"z", "-", ".5"}], ")"}], "^", "2"}]}], ")"}]}], "]"}]}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"values", " ", "=", " ", 
   RowBox[{"Apply", "[", 
    RowBox[{"f", ",", " ", 
     RowBox[{"fddf", "@", "\"\<Grid\>\""}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"Length", "[", 
       RowBox[{"fddf", "[", "\"\<DerivativeOrder\>\"", "]"}], "]"}], "}"}]}], 
    "]"}]}], ";"}]}], "Input",
 CellLabel->"In[21]:=",
 CellID->1200179539],

Cell["\<\
This shows a 3-dimensional plot of the grid points colored according to the \
scaled value of the derivative.\
\>", "MathCaption",
 CellID->1381686179],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Module", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"dvals", " ", "=", " ", 
      RowBox[{"fddf", "[", "values", "]"}]}], ",", "maxval", ",", " ", 
     "minval"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"maxval", " ", "=", " ", 
     RowBox[{"Max", "[", "dvals", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"minval", " ", "=", " ", 
     RowBox[{"Min", "[", "dvals", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"Graphics3D", "[", 
     RowBox[{"MapThread", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Hue", "[", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"#2", " ", "-", " ", "minval"}], ")"}], "/", 
            RowBox[{"(", 
             RowBox[{"maxval", " ", "-", " ", "minval"}], ")"}]}], "]"}], ",",
           " ", 
          RowBox[{"Point", "[", "#1", "]"}]}], "}"}], "&"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"fddf", "[", "\"\<Grid\>\"", "]"}], ",", 
         RowBox[{"fddf", "[", "values", "]"}]}], "}"}], ",", " ", 
       RowBox[{"Length", "[", 
        RowBox[{"fddf", "[", "\"\<DerivativeOrder\>\"", "]"}], "]"}]}], "]"}],
      "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->71841863],

Cell[GraphicsData["CompressedBitmap", "\<\
eJytfQm4X0WR78l/udkTEhJCgBjWhICAAiLg4AICA7iGRZZAJuAVHHAUZOYb
5TGAC76HhGHRp058oDNhFx6gszx1SILkA97j3puHotluQth0dEaYBKMJ9Dev
q36/6tOnzrk3N8zLd3LvPVt3/aqqq6urq/t8cNGVl/R+atGVn7xo0d4nX7Ho
M5d88qLP7n3Sp6+Il9qjimLUJfH/V3Yv4t//URT4/x8hhE/G/0XxavzZ6u3t
/Qs9+7f4sx3Pinb8o1f/FX+ud37D5zr8jav/wrMeOZMX8Vpxpd59maWNkbMO
7uKBz+r9F/n2+CHuP8/7E+Wsm+6zjiv0mY2sY/IwzwyyHHmmVVwWDJ0SnVEu
f35a31jLUqfIb32jfMK/8Wf6xmrWMV2xxFcSEZ3G53/O5/dwd33pn9Knf0p6
ZqL0O8vSu43P/1+WvqfSf6cUWywhYv/GZfrGAN+YpTy6E4TgpWK0e+NSfeNp
0rSfu+vLx9P/m+XPcXd73Pmf6tNPsOy5erdTmLQ8JXh6Jcs+KHs6/izGuKeh
8T/h04e4u/7pS/Tp5aTksBE9/SjLPtzdHevOL9anf8Syj3R3x7nzT+jT/4tl
v3NET/9jyHX+gnCge2q8O1f2Fj8gRbvI7xG/9QjrmqZ342u8O9c9PcGdf1zf
fohvv28HT1+kTz9ACndXXHeYnp4oP7xuNZdwfyjbRqWE4sRaCRPd+YVawj2k
YdabLuEu0nCauzvJnS/Sp5fy6Q+6u5Pd+Z/o098NZdtp7V8ET1HzW3eEsg01
vSX63rq6CL69LtS3/wfffqveja/z7gENpbQbSrlAS/kbUn6Wuzul8elvss5z
3N2p7vx8ffrrLPs8dxctszjNKPaU4e3bWNcFTXjiy94aLNC3buFbf9JQZyu+
5a2C0lbcREovdHenNT59I+vodXenu/Nz9ekb+PTFI3r6K6Tkk+7ubu5cJVB8
mWV/agdPn61Pf4Flf3pET1/Lsicp5+YHz5td9PzYpHe+z/2YlvJXWSndWMoi
95S0DCnE92B4+6pQehrt+Da66vn6xElS8kNFmNlAV+vYur6rhhefy+gp2ihK
/mwR5SksdbqerwwtV4qv7Uwt9S8cnVp3p1gSrPji/freQ4lb0/TJlRW/SFpR
e26dF6jjSlI+lZQqv7uoI/7kH9HpgJZ2tIsOKB3v9IbW3DpfztDSr8j4IgjG
qLyWiFiLAhDicYKWc3Qzz2P5vnWdri9+OuQ9HMpuxbKV5/H+SSx1N/3dV+HJ
Xo0lforU7prxogVqjdR4nMZyp2u9Um7xz+qi/nMo8OZlNSsyX9+/NOT9lS9X
5NN+dxHOgwT69F6lZPFfvd+Eki8hL2Y4HggdPbFMb+121fPk29a8sY9qqZ8I
mdeeafU4pWVJaLMGodq3wLe4849oiReFUqOFw+MdvSKi1nuK8AnU+K+K/l9D
MbuxtEWhqruq/fHtXvf23u7tD+vbC/k2erb3pDbUdufeHntaUNoCIoNdO121
TgoRi9q+sAi+5/OlfEhLOZc0TWEpXXKamjIKx7gdlO3xqreh9tfGKfUy0Y9c
mFDvMqIyzyRq7bs+WATvkezjzj+gb50eSr1qxTLGhkp7GI3jzABbCf/wrMby
93Xn6oWpdrRLvT1Lb50ZrXH8MbqKnnUJn1vHFeFDAfZzxk7X+aFQeulSZY4L
1h6tTP7ssLZLWIuXnx+Hnao1fKCiG2eFLwaMBzx9zW+fQp5or9wpvh3Sv9ju
pqrtfFcR3q7vrklasK+i6Wej+LFe222Ievd356dovSeT6pn61reN71NxqPWI
9cqYTKrdj/W1Y11D1eM90T/Wek5kPbtbPdrdfVsOOkU5zkPl99pC65yjdf44
+DFOcz3Hk497sB5arunvYomnu7e8932ylvJeUittsr0O+FvF5hoNaHXF5SYP
H1NAaceRJqG4/c4iHM5SYT3mEWG99Nl6/fKaZ3CSlvquUPFpFhgNamWPKrQm
GXm3Yk2i6d1Y09whcIzVmlaEXVmj9xZ00FccE/I2tAA97FHUDtbSirV4HuP6
ijCrsdSjyJ2pDsfJpL1NLoG38Q/en+9Km+fO1fcqjiTNH2cpfgzDUcmglQoP
4e6ah3CClvb2kPciC83juyOkf1HXjlFdLw6Er5ZRfLYrE2OaQe0Xh67zsJDb
FdQpY2Fq9jFHy+/YTtDrHxgWDlGL79GO19IPCbmtXxQm7GTpBzWWelDI/YhF
4VSWIvaq9cUi/LF762B3rhGK4sCQ+w+LqAnfMRt1Gg7o6oHJJiq/ni1Ux7sN
db3Vnb9X65oTSjtYr0NbaqzibSy7x0lWvKl2rOvUEdW1X8hsVBwCFlJhLq2s
JrMR+uyXCtWR9qVFONGV7GNs79Ga9gml1W0X3zVU5+I4MSu7/SXU1Yr9wsjL
914HrPt32VYxug2B46x9YCmK9kAlatEmKqkZBvQUPTA++Hil/ENdfe9WKmYT
5f6sRZ6SSuBLHKw6LV7DxAxn+7L4O1YDHlRrQUt7LPhRyHGpNrsqDuwseXov
9M3tvqJSs1iXnlgzbPSXEu6WcTpSYZjL6/MDxo0attM6dt0xRZWo/l6Opt3I
AS1H6uyWfD4lyAVRDBmpz9OalgcfLYcPFEnguafkj0pKetLVYAM6GXPtTppg
HeHDFV1Uj0M5EC/0kAv5uN/ogob9bW201Cp+EvZspMmPdFIfLbMNl8Gzb0X9
nky6UPsALGDxQ1q+U5RvHVLW5u+TGqiB9qyO79QpelcjRewn0mjvVK3xUKVq
LKnq8Hdb5TQgvc4Pwxei4H4YMhphLear3FvFc43cwkzLaqnxJ1ajp/PYRjq9
H3IK6RxH7YIXMcDQnhKGPyKdE0M5Zs/Hqm/Td56rxRZ8XL+ZIsjhzjJ2UCwR
P3IJbL6aC31uLCnsZLKN938YroHWSXtt/an1cvPDO8k/zxf0RL8Ino5jhqRO
54RgdZfgONe41Elcigckt7VCxRkNtbdj7Rj9hPVeHkcPyyPvmwqHW/9UhL09
T+T3gLWTkkdyv0sKfTSB8YpfGL8RexkphT5+9HZSNruRU9o+LzG7uVWPdmwz
F7lS3koueV9WZ3XUansvqdUpQh6FMks1vpEO8HUrmkWkB4dcgJadXuOSn30a
nhIfg1Tf/YECngBbW+tzsb/5vPU3W0H9JZBXl1RobKkTB23412szbMUvTVpe
SkcNSZfO+fHc0zeH9MF+HhXoz68M2ftCr7Ln81aMcBD1C/k6OYDWGCk/P2Sj
W8rkEL32y1qc7R2NNPtZHvTdSyuxJKEZre8obXnC2ynGW+XdRYXp2VaiOyMg
+rEiYCTQX5MWtO+XNZ++mU4/hpYW0opkTiAvzX51jSMVTVRedBYViPqESK5w
FTpxhloRGdv5KIaf3TyykTIfZ5pFymjHjzVOTiHnmimkhJW8+DP28h8MrYvM
7yu56WeUmmn0vod/ax+l4VhaaVg0k3Y31tyL68mXYRxqi7paW2jNs/KOaKx1
V3cOO3psmJppRKv4EbzOWCvsxZnhj1DbuaHy3rogBE1ylBzr6ji8kRI/X4Ux
zHHJ7x1TkcmPeMA+bKUMzopPnBmAXKeZdIjQij/AzXXqf3o99Zxqps/Pee1O
+sw7RtaKGE8zUsK3ngCbNpr0wd6fqYdGIiKZQh/iaZFAlu411vPw7Y00+tmw
/cg77YsjadNIxTszDoVgHvs5jKysqXEIvA8Db3PX39ZIxQx3juhUYkthfZ/1
PX+spZ8TDtbfO1u79268D7Z/MPu7sDC9UWnMh4XROFaxjLXWeQJbcHvYfwje
MPYSBnKqsvuHNVKZRiZXFyEe2n66qt2RvkQo7dMZiSbSLGX2RJonkXOY8bo9
HOVqedeIaNGZoqvNhyrbvHKMz8AbOAP6G6mIB+JIkYop5BS8/NuDb09om/8a
vDd8aCM1Xptmydtnp2jF78KphRz0CM9gVHOZ8WqZUTydcnv3ENQc4a4f0kjN
rAbq2mebXxOpKT4W5GdLfrRPdTPKZRzwYPIK2vQCY5RDU2djXT+CbaYSsy2n
VXwMlLRS45tF9wS4U78LJNnTCy9/gUZxhZuwai9QF86lN387LchA8JGUvUmv
p0xjR4VHsSfp9f5Gl6XoDPYJoutCpU6o631Y0QWMNS9jn/CCyr5NKkUDpw9B
ped2M3XQwDLHo4lK4Sz6yHPAvEjtCUWS+0J795BQamcrUjqe/MScVTFgz/kY
G+8/bfc9Xw9upNzPGu1OCtELLQxHkgq0nfOCj+VqpC1WOstdP6ixtr3JB58f
KTaE464fGf0TM0ok/nsYKRlLSti6b7fnMc9TbLdzn1n25ihVSZYxptDkUWCm
fxH7wmUcN71AD+O8Sn8gBPqIvc9Lm9dIEc6r4wq7f2wh7bPOw8mkzLR/PHm4
S8h7qO2hMZqdnR/YSBH8LfgKej2SIZS8l3yAF7+AfLidfFlRixY31+Z75T1D
GQlSGahz/WMe8rfkwUCbNtNWa3Kl3kPOxvn0phboYXRpNkIDXSe487mNdO1d
0QmjJ9ebuWozhSp4Dsvpg75A63M+5XI7rdOK4PPwwOFik6HxvXYzZRgxW+/8
Y1jkuUWNEtFVcHeF9YkrrKYWr/vZBFC+KRztrs/ZISUyp8la/l1v9oru9kIy
YBXiu0IW3wd/7uCYYgWpHlTKOBcZHuezBwxBqfcxDhiWUrakH+d8EILbklfk
3oGlKXVtNKUr1r4bqR5NqqFnmJHqRoL30+f60KppDU8aEZVot//OPuKy+huk
Jv4sJHpsvd4U8hGt4I5EbcnTFcpXol9viDSuHinGe8rqoon6kxvoLLRP8dGI
VvEqtb+aLb+/8mg5c6tKDdiFlOOdFUQ+qBSjh1jPqG94vM3fqouSsZSMpgwv
3xyNXubIQFjOlnth2JvcA4cG9TC6cG29HkYje69XDZ+fj/d6sF8jlbCFKxlF
Lkvz8R/qxPN2HzK/UPtCa0cdJ/MWqWdM83GMGF5ln9GH+ig1bx32HZZaP+u5
T0Zd/Bk19nn2GRdSY1ew7xis2SYheI8RUCh9TGt2PW+zmVI/7kCLfZ6S7NVj
JinblVTA3yrtUIsa1ObvOYkquauEIWtjdj0SuU8jVejxi382/N7D0ehZpBK+
ba9Fn75rzyNCUPJvAimfQwoPJoVe1+GfRDJ57nkIav0YvpOVJkR7H4NRqotz
aUo/gFbz3WDZE9YnjiO1GE+ElTgv+evXT5zizvceEZVeN3X0Fokc76iyHkgo
Q6QorBxLijC3U83shPf6JPU5POLjuLMbqfNxQHjlyzmOuJhW8G8R4+wkyerV
VRTUII/9yTessChtxWE7oDYie8TPJTRTayNRP1sNb2d5mElqOWsSCYz2cpW0
91WMJISVsM7FK9o8XrE5zvWM/pf88/HltwxLkdeN2aSkJ9Mx6U26kRK0nnWs
Maz0llpzOYqXhpgjW1/ziGYNS9khlKi3k7m3h5L/LukeHlihh1EuPITfsS5Z
IvZ6BRC9GtBn1SnXvIFYWGMko1EnQXFwY7hNmGcnpdYzi6Dpv4btGSI5f4Ln
mE2A5S6ERF5nv/Zb1YbfIkLTKtaG3Vm+j/zoeS2m6ccCyr5uOgeMhAFZBXlb
Kvv1VXqg7a9LiOLPqMNhO1theFKvPREPSOcVeiYDZcuIYOTPacTj23kzDvPZ
/bzhvuRyO2nyOrb9SJhSKNRhzBeebDm62g1RD20fZ8AraYfv1yz+niPiM2zq
c/AsZCxUXB5woKdar1RPIn3QnfDkeNJG2Q8Z7/A0+j50j0YajyZNkGFJCV5f
ZZWtM+8tbP/MEPV6K9dcn842FxuDX6XFUcs6q01k1mWNiHWZdDAPq/OEfNf3
pB925zOHpcT7hsKBjs7DhO3wc17RWGKXI3up2PeKmE8sAoMv6l9Gyk9QBL1Y
SVDU5j+OIAWIC9xdsSCwXStow1ZxDqeix6TPdCPFkeDdswQb7dZp5ojiSeMi
sgrC7XNIO8pCbkyrt1yhXMrW43kfcTRH+1Zw3Lcq0O9K1o/yDH0Z99qMbLQS
1/tr/Sg8oSdpZcPtc1nOCb1I6pFxim+puzdS7kcX8EbKmcE2pfAW8KhvTqjO
w9ITTfMzTStdpbeWOSi5jPJGSt1e5B9qfaypb1ynhvgxwfwY42qhzyLKHMnY
oI8ZHP3BNIT940a772eIx7n7RxKFX+05UupD0iNEhUi9HkI/uJUs4KoDiWMs
6TV9AHgE5BHt31ibOR7L64eTZmK5M+fFyDHs06AXsee0cwBrQDXOIZrXIBFB
xJaR9Eii98iv3xj8PDNmS8Xx2MjjsEwu7YZMH2Dzs8TvoVxg36vyoeasNXpG
Ewfnp8MzGf2CC+3h3yjrfrbM31nfNWjloE97umYpIGFkHbbCR5Cj1mv5w3cO
0V48IoxRBmvSApXrKLbH0F4UZnpX31xbsU5YqxeeGUds05zsxhIrbPBr7KUG
abGfrvBzDnFNIi4/oh0ez36Gh3olWPxIRLwnIR6eYXiGEvyNUYvYQGlTPdV+
1tzaTeMM9RC6tKpCJWzdmtLrSZwu+/bfhD1J1xzS43srtKCNwXKmfRb78PT0
kAbvE+3uuNMiNfBDVwa/dhlc20huhdkdUuNnjZqpeS9qU6cqEmOtK7wuJ2vq
K6GR3/eblOkFFP0BKx6KX6ux+bW1po1hJqmSDQFqsbIdUvc+fXsNR/7hdT+i
3M1Rg1b0Kz3MfukK0WuyeFskL4TUw28wPpPvYbY8coFo7QW19e+g0lu94ynJ
+PIbQmVjzqCQlFFnM2o6D3NtPL+mHhGEZDdg1pg9dJHtrHDAiKhDdvJjJok3
1MulzhdhfTz2J/dgNX6lHCs611ksSa5eKwssr0nRJe8poyVswCrKeNl7RcNT
dhAoWy+2sadYSWn/KuQSbF9Xz4PZi7XBQt9RWzm0W2OtyH77CXe1iNVmpVmu
aVoHTb/VKKIEUl94nLxznbWa6yrRt2hJbvJZJYgSrIWP1osdcySy1dkpBOyp
V2sq7IBo6k9q85U2mgWKgTxrdqXp4Knkq/Sg3WGobVX4vLPU6kqZSKzyNlKK
GdhfVvqgJmo75PlxoPop43meHSAj+ZuC/ZMT31ckDPGSz5sBvd47O8hxd57W
V6V3Cum1nG2MJvrZykv9OFo5+xRb1gucFw03eU0WzkuT9JkMzRTCspejbnhA
vwgHkrPex0I+0QDjo29UKDyGFBbtv0raS193sWmxp3YC9QGe1Zrgs9iaqd6H
VEB+v4glv17rUSeRn5gleAMxl46Q18cD/tJTUI9IMg7wNyz20bGxpBSI1oSR
aYBROha1a+Z4JDYePr7Rwych/19x7KUrIN7QB94IjSjA2WWNXNcxbxhVW1ff
EyqjheT/jgzTnuQso3qv29seUZfP9YS8z3pDW4Ku7WgDmZWZx0DhlT5l+0Ao
3qtCcZVCjBeWWYwxzXGpD61r9+t42yEfOaylHSjvTx8Sp2lZq9hWi9rZWhXD
J+1Bnn9LhiX+LCTKOJ14lCcRBUhcxoN/h5uofc+jfdlOBPE36lhdG1HIn7s4
KSLG3Bf8+BUo/eh3VqhaAOjhNmYKotMSN8ZsqNkpeB1vpHfRewyop9uNiNHS
S1uLfm8ZW90r5ofdNCZDLDs5jMuQCpkYY63Ra2/dKWRzKbd3hKwXzkbmggbd
7Rs82gnXACOEAzz0nWK7zWUkPQWaTRaDSDEEzcS5UgRY5hvmHGiTA+I1sNex
+b6k0V3KHmWHwlo3Lf8ouS6WHm0FHEJkpuizMvz+Ws2cwmzMH1Tm1gcV3TQG
DJVxf1/JkrbKH3wqKbMVJPH3djy1nfM4m7ieAXySUb349dDtp/D7injtylCU
mZDhZtGWidSSHmoJR+rCvlGobT4P32Pl+uPt2465kWw375uW57113noEKzii
0AvDLXR3I9bRHit1ArYy3LxLhtPLXDyU0o8W9MzPGCXIO/Kjm2QhAgtPcFQD
vWgVD42w5eh+AOI9m6/C3kaL18i+tYuOSZyY5cBTZMA2OTCbv4non2Mb28i4
0VPMTjI8kGSKyfI8tUuZR/pPofswSwI1v6S2/jJgpZA0njSOJs4mKYOebXx7
G49xRAor+hxjwk8FZOaFW3X8Fd+znPCC87VtO+/Vf4WsixudeAu4IdhYeB37
pNVcZVFimzYk4nbqiTPE6omM4V3Ysu2m0ymzEjq6jStrIqhMJ1sZRujwMtYS
bqvh7ZR4B4OufO3m8RJTYOzpU0UAvq4LPpINvN5TR+vsD6c5uVEzBxS72shW
hh9c2c6jQ/mSG9usjMmU7HSHlm3xVnumFcaw9YsJj4BFiKrKtvuUj/aNo2xH
jrHNaIlocz7iFwQtYhDSOaedeq2JxAAJUnILLFpc9lgaYY84LJ6H59fbeCnt
ACZXlzTvCTViWb3HyWoCpWPtXkeUFmvQtuYlA/3bxL7mKTaqp/QwhJCx+V0Y
/USALCHvWctRyXqOtZYE35sig7R4pqJfO0R8vCEVn4T9CSTyuvkEiQuTKcGJ
GbYu+4+iswDusGLRSGeYzhGa7xXHUnK2pxqLX1Lf0xKrUft1bC3QfNSuGRPs
0Rb1kcW3HxdK399sS25TxhMXZZg0kz5Q8qMmUpbtj7JfDjPRS0Sk8TA/oGMj
GtufDrk6S2rx3v88OsQeVnIEXCKsWg8ff4MN3URbsxFvn2x2P4KTiY7lpsNh
iUbYIlaRKOxr6ODh0/VopdHMYG3mhHPNaVwOHi4Pfhcsfa+2JlMjZMXLbHfb
OcexvbaqqbUDyY0jwmgnXg5ARzoMo86ydiJIvrHO5hnVq0UveHrw0V1kWy3n
6vBv16zO8KgESd3ON2kobWbF7vsxZTlvFYsWrC9La35Z6WO/sIRrRaKRWRds
gluK4Yjh9JCVJxoLfVqvPUHTbm3t4nvMobBdInaGB1EaTwchUx2blvNWc17I
PRsLQv831eqBVm9k3tjL5OzLhBRBf4d5XuNgGo0J+q4scMzKGr8T+A8kbour
yuF702Ye6G5SxUvBYhPgxwAHKxMdbutRUPgmHk1ciOWY5us7T5JnL9OPeFm9
n79LTxhv4u9xkT9jWtyrzfaF81r/wYBYw8hQ6jx8RFn1zjFG49xm0mvBLON1
P9PA9pDN1T5J6l62Fvx3mGMJx1sLtsyHcan9ynpW3Z5P7/cQHbK76/K1TF6f
X9CMEhG/auSiVbxI6l6qjM2q/ew2tqBNlM7GmkzLFR5PBuSQtBLuEr/mAkf4
5h8BX/hHjnKtp4U/2IAXtvUXXKXZZ/MIafTuoz3gg+9bMAsxQOm8mPreNApH
Ww/wZ8eG0psX5NANzLHT5j0pK5Sx1LSfOvQys3heQn8QMY8PZSRfME9kebOG
wDjZYZtDzE3j9eExiozTXhUWpVGPUHkWf89y2Ew7IqCErV8zscGjsnd6mdx7
idx7iYdeDR9TDTfU3WiBILmNtcxExJ5+mnLTvWc4tRElViDl48xttt4l9aut
Go50Jf2TzHe23whti74Hb+El8uQlvQaquJMGyxaU1AP1KUMo+6wJGeomWSPK
+XPuMfTTWky+GbXt7giP6vfas7eLP8A7ryN1GYfVVruFtvslG6MAqMpQvfkI
FmPJMBMtYkFDrz5Qm4f4/4NM7GVPRLYXr5PP2mHa2h2LCj4iGyNsEQglJlDn
dbNDXKgl+lMaoVnAZW/tAI/cI+LTWxiJ+HmoXO+rZaM1I8RMZ7/Z2tRbdHm9
nSEXmY6m5uxGDrQziebeE3U6cWFLws6aXsp1wOs185H2YdRsIb880pvmvRfk
b/vZQs8pzjRIgH6LHDPIIbzzdT1Gpg97EjWjIZbhyav7kjczcLc/v8v8a2HR
y0LMa7Tn4ZG8DSgbMj2hzdV+RdHGMb/wxSLhkTOFrLU1P4HcSfbnOEc/RpHL
uHrw66arFieurWoCF/wM/yzqBsZCW4mkn7kxgxxxmOe0nGheC/lXXzLNj78n
Gs6IT46O4Ezxw4V6dIkRa31KTfUyMozIgekLtorJz1APjU0GRe0GGZZIkWOy
gX1xiRi6/Fo51mIb9zqvY/Zuifc5xOeLRVLnhrzOAdJ/h2bPSVZk0z69bc4j
5rL0eKcMizflKvZjZbLPzt2NiMdRxtS0ZNsnNnBAALfC91O0ymaEYP02yMcs
4iGltoiVGGyfUstuKH1YUgVq9XLNq/S7kTXjzuf4tOSI3eMWiU8m7tHEDev6
pLZ3xlNfK8t4iSMQh5ujRSknYd8A+X6He+AlS5qwel/zID7nV5E149urWurv
AhhGeZd4JhCP6ToUdiqlp35CRJLH/aUS2o2UL5XOxWf7rvXC32Fc33qRfs4I
3ctaqztmHfym8FkUFZrxO+L6HeJGZVuK49EuEVu8cnRo6ov9uK1ryClDBZl5
peLYvZ00cNSwNkSEcgyHNJd0c0v1cb39hkAKWa1PGZPAtyKI6yQxrzGUZU8o
+xTkz1fHimpvOzBF0FF6YAOlvnbVJy9XKB4WSi+gJZDV0K214976nlfNrdNj
tVV+zstILa2d8cC020ozXpi+CS/QIuXeFo6yXko+2STjRxcSwpKC3jRjDutU
dCnwbDZ5I3VnI5VBtAtz1fcGyTPyGYwjxy72fHpF3q8Ry2sBuw+Vq0LJmy12
zkyMNMtDP+ME4L8smMfUTtbI8rMtIpaPKC2vt8xrvzdJG1JaTZtXX3N9pjvf
ZVhZ70l8wP9asBYO+WxhjOpFng+Sdus/m2I6wFJ6Q7RSlZFYN+Ed1MNa2Vhi
HZthVO71cu/7LALjM7Wace7j5MS+Luk0c06zeE0/R6dbKPsXSdWL7IUGK5QD
8XP1jOsg2XCZdMtVpkNcGeRouKTFbKatC/yXyBn507iDNvVs6h/sXfkTI4dt
nB/aEZdgrV6l9m+xiHJa5T6N8jEPQI7JjrPCIdx/UQ+dRe4RoIMlzPiTo7cN
qUUbLbIX6p0hRYihkRtT79GNv20kwFV6/4J5nWfL3rGb+47Ghx1rjZ+hsjXK
8L/Meq3neT/59CI90MGsDZerGbqgx+w4AA+Jtp3aVm7XbP+GXOPysWOLbUV4
gT7vWbPaz9p3I6Xt2LhSvtyzJDEHc7Fb6VfXZzqaeWOWY47TEfDmRdK8OVkO
yGowmG3g3NN9mMvbkOXF1luP8pMR5LbjyRzi3524lbA0YqXVuTlpZzCfqFiS
ewXYl+DN8EB6C8yHreU4/UXFDTCb9RBL0pNii4M2Ck86Yv5plN+9beNJRnGT
bdXIegNfZirP+hlBv4/Z5s9WvMduZIjpguyE7PuOqW9KIw4lF2ZQ8p4Dwhmb
dZuQIdZeQVBvrK9yAGfqSHPuzQgYm9UQU+MFrXzyyq9Z+Kg7nzykjFsp0lq8
YrXq3vtpV6LNXBuLFswM0nsZh0l+MK9kvWCKbBKb4RVJvoUSGk9cLO1RffFR
OUROnV7LLLg52G/IeGVtVuIjO4XYMu3eTqTIv67Ghbl35L09IfdeYKHQOjaZ
pU/y2pt4xiuVj2K0Ss30KGwPK/ndNIqUdWJ+vwKg8t/hhJxesXVfKZJOLzbl
DaJPLPfSGSKCPCjSxMHWHP/emGTaZgtNM06U7xzFfB/HA48yvvAorfezaZQO
3t1cixEZT2z/eY/SeIIIf73lNvMGnsRvGUPrYz/3PFvrC1zxNKgS6GhuAFGX
83alVQMTmLnrrVeOX3DDn0RbHU3M6P9W8kC+z0Ate2TWm0YqWyTsQ6Q+axR2
qYq47JNN4jY2adOPqUg6Qypa3qakO/w9xhAHtF1YxpXleC/985FV761xfUSK
5/vvbAwv68Md8h53bj7NRGIdU8M8WMOM0XA/R46PcmzybL5mKvXEeLYeNfUR
RaxNWkEf6/5almIzyv0oX8jz1WCZ4T38zTVng1W52/jdogxw1ya6Z70mGupd
KzJTA40oVK/lQtXbs80V/YHnZW74QPCrPQ4kJ1DfznBCzN34Bk60Sk5sMKvN
mY37ysjChnL/sFDquvXi9VmRvtC0igj7jxc/k+/F4jBbR57cYjzpOE0RdtFb
/EObOfJ5rcfhoftzWeb3JzXyZS/Suiv5YthoxVI7mOr4U41Rqq/m5m025GM6
erwW1W7i2jhKiDWvD8X39Dpa6qNltkWWh6DZ68JMWYWYefiQzC01PbPVFsbH
XDI63rSZAc2G83s7goPem5hFujkHljyj8RUteZ4c3kDLZzGVcF9PqLe1dqZl
PjOrS4nl/kk+8sDucWBeCJbR8KhaGbTvn2VrkG5J6wsZSQ5vZJqBPhuZjiPj
BmPaFb9CmAJf+nnj0mCgaoBLkqkT7rdRia3eRNYMn9Wt8PxacOPFGJaS75pn
+kLL+mjefzS1N/ueAWyURV/CG/4LHH6PsKH50Ob4TvBb9jq+8jvI0oF7Ssjn
P8o9/zxSXTeR/oFr36MepXzatbS5ISSJW79irQNv3JpaSBz7nDjJofZrx/1u
fcOjFt9iKlGlOFEtwjMukz3ty6YARS5aQ0g7InygzgPaDF29JByQyGcaI6jY
LZ+l9CpvVe3XFQoRv9kz9kTJlvl9c0bGB6zY+y2/IyfyzK3mumQDZxCDt+Wc
+QMrHpDDpCz09UR0Iledos8sWJdytXHPaMqU6zNTm/S+hd/dshnTCaSAI4Xw
AHMRDYvtmernTmBfRE4P6HyjTXcFVVSRFCzZWlrln9kIb6AavyhtPyJu9rWn
8U5iyIjrC5axmsc3vPTUpyn8iPhkSonZ+A8aYB+vgGV9zmX2+ZkUk5wUsUkL
3CQ7f21SmxRKfZ0YSttc+j/VKON4ShSZTCVGH7k4Tq9/rRaRbsaLHZI2hgNM
T0yyETt58EBIvae15HU80JqeI51YsDgzlN6W12tRAemPkuautZy+n9qM90At
16ISP/dcQR5Vv65jaV9d/S4R4m13c9Q+Uo4A6dOcv69zJsrvQWuN2At1HXcU
Mg8zbOJq5U3YKwXqLj/HECtarzDga3qYjluce4JDKj+x9iwKnN9z8OPKSUR7
0E7JHxJ8mv7qRs61WG/al/huFgnSPzyU9kBU2+bPy5b80xRty6WoWOVHJbPw
o9TkptxUyfzz0Y1mJBj3b7BcjvSdE9tvtWPIKhqMdrzOpPyQvdVOPIitNR4m
UVu3N9ZQ9trOCQPK9eR1cnmG3/8e/f1IMaGWX3PP2A3Ui3VcO2RYbOFuKTeg
sx4H2vlQxVs0S+R5bqsSTHLt4r+n7Aj0MStr34adQjlhnNpfkyOweX2Et/Fr
2y8oRQiBea1SPo2ya2c9xTwiRZt5yFZ51pBIOWUv+Q1INUNkUpI/F7l33xVg
a32ZExqRjKOU0Pus5ZyIWv2NNhXtqaevsSzthyEPdfPYh5Cu9qLWB35DD4uD
2bjGfFe/Jw9yBVdwfrO/lknajGl/6o6OGTvFGujM38eDu1ZCi3SKppthy+0E
ZFb0lCgtldrbrrrtt9i8R5zvzTQU0n13CiH0UHfM+XuMVDYmf1Keg8Rg4wqu
bbN+W3lZkVrxTOOeWezXbYcia0lm7z/egCSfY/eRvWZEsyiHKTWZcd40tbEe
PslIcLKVtmNWW/uuZeQ+NBh91zOph7URhe+lTXIWE96cyViuI/a9DDv7ZNT7
+dmhMYpwbP3rPyXK1zLnawN7QPPhSk+1xbXzkhdKT5RfnVAfMsNkbcnmmzcr
JzeLxdxsMhNKZB+J9zsKQcWNtf05mvFwzFzZQU5g5Zz3uCIVz7cowTa/pFNk
3wpN87+pNWlWnau5HcpRX4+Ay/xo5nA8mpfYSvqAJnCiK2/SsLj9rNZ+5Cw/
PZL2+srlidpW63mXPDgC/Hk+y/QBSdRpW8Nu6xatWqzHKL5NRL22O2Opn4xm
bYamhK2YYX/UYghpTavIW74cknuafi69GTOyucOPIOM1CZ8cOT7NhROMik04
McawtYHN9BRs66eLiF1/vp3mfzF7wFH9Zq4n28o4/lYciPKXsXk9f6jQPCJB
2TS68P7CziGlH9tv2k07kbTXZgWsnWUYi+z7kZwdobQ2e2sj53G0sXU6pTjB
SVG5EnHS0qb250fjjIwl64b9AB+ozcQ18+GQUFpicABzpuEFvzoK80zz3dy9
enC6BLK/1meavjKeuZUe6tbpxrcsBj8uwFZhTPZQkdcaqVhq5xhjrsX88dym
0dQA55ZHygFk6K3mqDW8SO17IR4B9md0g/x1M4oaF/h94AJHqol7S+T8iL+3
zqjwIc8SLu2YHJg7X8rxdInda77tB7XnTmmA4d+XpaMVLldeWO5oD2VvswEq
rbyvyHbPEGSm60z/t/yjpNup5WTIwZVlwfrBnANTnRbA2i/lM3rZdvlJM20e
fdPqkKK2nr1Dbnyoxo1+G7fNz/vhpP/8nWYwHEd2I6py/sqyBZsRYxS2lPeW
8gvKy3f43c7hUWEcu4alApWtcsvbusqi3INmc9PuWrl8baeIHFXHyRxxP/uG
1lJGGZdy/djy2spB/9WBZmSIDq3jrohrk69czVfgLBnfbw2DbDeTScpcW0ZK
SzlJO+tEyo9z72KX4wdr7W5cI+UYRYYnpjkE4ygjxdvJfSTzW1uUjs2J/J4l
evubY7G/p5H73pedu9O0yyY1x5BWztCvNlmnvsJ54vV5uBJNlxJh2b/3I9sZ
Tp+mEtEC9xxtxT+E6vmDOW0jl86xRBj//hXaMjJdq/Y/eejVtzdzlVYNC2IY
y5h3Z9/XW0YLv7T2tTQ89w+ctRhKSr4WfFMWdOfS0Raq473SFu1GSn3JY0K9
PexJKjk2u97K9bMRuziqUdc3anG/oakXrueUg6+np36IPXGWRxrb9qhy7ULY
MnEEuDg6SZrFsUXqbXTH5QvxbfZuhJvGD1zFcPBO4XoPKPo1vYo+8zpFBzrZ
yI6+xCgc9I1fo3eyBceEIdBxRJD0kXxKCG0mzXYgW8b43VLI9sKCGT3Xl+vU
mc980E5hfbeWCqwtzkLLbdtxm/N7Jbp2ic6vN7bR+VQnrU4oLVsutVny/Afl
24DX03Jfr1lyeZn++7djG1EcDep+Qwv1a+pLWvNvcWT6AqN3IRr6u3/w37oz
JIhKLuPs852WpZvak1Hvv424c1TTNp2VqJVdKSo+vNqDSPdU0ksp/EEOT3ve
VsRPaaUv0C5jnhgTpZkzfYDeux7XIp54MI9nVW2FMaNz3zRK62vKm5Fiz8Rf
MOINxPjOuLSuM9NYdCLlwz38/zA1VFpFwAyJ70XREtJunEnvsKvEncEQGiqg
x87paC33NfS03wzlikSPz1NwCPFJDygW4hijhD5mZwiqvU9Ju3anIXgbKca8
1yrOQt1XGVMZtflsy8ioRkbqz4N9AcBKUIqhcVNVk3tKJH2Uu68BeUlrinIz
0bIfsi9pXc9Y/Cq0cklgTP+keu488S1D78dBzSiwF8/Pg+1/YXkttM7TuW4+
YrExj3lj+4ayj6+1/rX8THKEhKNj+1u5PJGZ1K5JxNamzA6nZvkxIDIav1Wz
z8PLyMYAGIOJNhXTsQcV3CfY2PD7/ch5v3uUWsAME8S0G/HYvBba3XKVFUcQ
ydZNJD7kaPyAMxT3VfRa0k4P2Cl84r11aOHKbCTIzHRA+s+3kLIOZZbwrStC
POgdrq1EJeQ7OfDb8EVJ+fyWAOLap/ThpVlExOumlSWtAutma2l13R/TiAzj
nrDFdrun7d7GnR9et1UBEyixxC8imkYEXScjjincF6+Wh7SLyPVyYGwWQfEp
zERfniyH4WnaFwR4vN5yFyX6awswb9+LdS8RzzaOCl5nL5+yRdqOXvOk9Qax
Jo7yHNK7x3Tw7rwsWzkkRYJHq8LJRNfU1uq07CzSal/M/RS309fYZjnQ4yhP
le+6ombvgfueYF/A3KUit/8qB0swVB9wJRwwrP55FLKOpNyZN2yHh/cG10OU
9PYYHU3yYFk9lEeHvy3W4Wn/CtGtCrpO1LIhddYOfcHDwTzNkaGANVmYtI3f
st0+hmimszTLZOk4VN4KdSkF9gVpnnEW39uV9OP3XcgJKW6hHcAMJL7v8nBt
jDc8gqoWqUcWimNgcV8Plkc7gXQbDm/Hc0kI+aL5XdLZShmroFX+3INcHxmt
nElaZFwxro931Jb9RB/tUymB9hpLLMFugh4B2vZ6GyUku5C+0Cm/+YVOWuS7
7BnEiu5i+XoZswnd4taQ/gl677mqHhS534RWvihZRPNBjzEcEdNUSmNHmMwu
KQ++WDDCBm3qEofAeL/+vrVmG3zOWzO17VJvTkvfPaRfhpy8Z1P/Z72Cz343
SseRUr0Qf8ufaFNfTUjl57RMKvLnFCLpOMmcQGRpToJ0++941JHNqurYaVpS
BNJx+uG/F8b7qQWPNUTdhCj+u7TAjkYdYjuY9Jt/8N7Asbzz/4eneY9QGU+e
i3HNNnz9JtKO2db7U2aQl7f2lV8ixp4vmtQuxZfoF4fWpUUF+9sctzHiAfUW
oav3e8MhEL0wvyty/Vz4ldsCvsK7nDPs95Nv9XnwJjRqc0H5YsEif85xlHN3
vtSjtygHFHq3HvKVpPra4x1LZVdKZSalgTI1As6927iPz0DBPJvhcbUqUiql
sTxcpoJqFTchPmKYc36kWa67guUQ9hArPOmP85vN9Vm5/XeI1HaAgy++FfEu
cW/7Ch5Kf8S5D+XXtKLesLWsJ4+wcAiy6lzdXcxLvJtv3M1xTlVyQHgDY1Yf
Zy5rGdllRK2yF05OFXyEO2prTHvIgfKKrQjCGG4rRyNPI8bUU6yocCLywUc+
ldLLqlYd1FyWbCjjCdWdu7PzXcgZXn8kf6+9A05hRH9DwGrfB2t79e4xYk7A
E7gs5Ur+ETkBD0a+ItyXEPleUeUUAU9xsoZxfZiHQDN7aZhM0jmOVsThe8b9
RkS/fqczXjoLnEw5Szqa/UJR8+JMerYDY8dJhjMRD9s5esOHKQEVFP3xKqJ2
ROJzvH0mWB2Bcf7aIajdlVSqh9OB7YMawBd+2OhP9ApdpkVGV5sc9+tofF5X
A329zC8NxZJPaCm2r8sXKv5CmXta9Uy6SXehEZ1QevJtjrFzm94lxbAYNwZJ
yvPZLfChn66NAkB9ikBHintJsfLjmiLN9yoZf1rwGIp2a3/C3G7G0ZzeWXr9
q0Eyj0ROfuYNI74vUxJD0dyyKEakWdC2/qng3pRlZkiPOx/rMXVzTJLk3Bxp
LLHYb1rWG6xk7h31VTsfUIQijRYRtuRjebXcjC/XRgpE567qiDsiRIbTWo4I
+M3ySwo5Cu/B+XbBr/vck+tN3hJ1Jl33/PgqD7nAHZNxsliO9xGNp3Ey5eZt
axc65q6eRJkhJrYmZaRqG/18wfHyCsACwnigL++3HVJTHwBf+h5GvO+pyQdt
6AZaMPO57ft3X6XOij4uDic6Sv2KMuDREmn/Ww8U9OAwS936HMURceBoE02H
v8G7SyrxM8FF///eHJfAFBl1M3wms6mGp1viiceNNvK4kWqxOH29Z4A7I3yz
8SvhdZzek5HxWjvine3xthNe1r0Cu1rq6BjrIWCZ7630C2iLj9A238OY9Q22
x3pqUS3iwZxjicLnf8xqRDGa0tIo8dKCNumBSk+dpLbILrcuMo9lhR4zXNle
WoKDlkBbWfxJXRQdYyTxRnvavs4ilzoN96Az1RW0JzgK/Jwg0Pq1YJrRFFHr
t54WEVPEJvD2cM+iF7mX7e8R5W83xaJMn8xCWOZsdXWIoUr7hgVoXpuy8/m3
PoMPKMZRZhiNLU2yegdRNElaZgA6RADv9h72mzeqDLgLq/HZOKzU38jDvnfo
vZwWqdf6i75GH7KOwmdXfNGdg1f3kUePWG+ZdGpPpxljHW/xe3GiWkaRVWmY
Hh0dEM/cEd36ci1b7AsNdLdsbizSLUcn/kAPeS/ncs/h/XP1mNjAUcGCdxZz
ta2n/wgt6fPMqO2rtXjMLn+g5rsDh87B9uLrnbLiRT3rS8wHWFFb1TOWchhD
LZragINzvovLlSC2Dmwxv9KSRqfkPccsabU/pHY/XsVliwYGsof7T38+ePlM
HhavHxEcTdza8xG32Ga/1/MEytPkym+x/3WO4xyiPIfcWVz2Lnp0UltfnFbw
Yrx0v+Vk3288Q0w7DMygdL0N8KuJgW4KpXmSXkr95/zCOJzGMv57Aj0hj7DY
Xi72+6/Zjs6hDi4O5c7zi1Nb68mwSFmY0QkD73B1+X4CtPudAdTSXl0gz3Yh
RletiOTIYSm/33ybyrxZO1LeylDgy0nQwuImOSY6DH79CiNFq/NS6ximkf9D
0T5HS3+gtpK0CUObuyYJ7dgl/SbOxQyEXvc++pnVtbEAqPJRV7SMq1OPoRmV
ZyOHp3VqwYxuW6X9/cZRf04to6wW86bdullbZjdS3U1W9v5kbf+sAYHse9h1
15XPNY05NKO4HSk+YgeUgp+3pijS1BAaqfq0e99HeUHN7pSy+lgnoG0J35DX
9YPaDD77rGTdINVzlaLx5FUro6zpy9JCJaJLtwW/x5nfFRVUeruosZ1ILfbW
/UEtNjKBVMHDsVUKN/OYYZyKBO2v3LuNX39Vl1dL8DbJt2jQtSe5dyi591Zy
rcPfQPtgbUcV+hvnWW2TQhnjL/ev1A0wCqM25+Jscg/cv433btPnscfsMcFn
LzUjmOWunk2Ojh2CctR4HjKCuqiSt/rRcGj9MQy4RTzlWyx7PO18MpqagdpT
KWxtt2l2sWDwcvf7BQODH82cTe4PhQHSOY95gLelfdvzdg8P85Z0D+ffo+i+
x2MOS+g4HGMzHsifH3IUNOXFFJBqL3O1ji10plfoZy+wIF9pWbeYpTbkez13
I5Xou25jf9cf/C57nhotvxb9GueoIubz9aZ8XVDn87GUN2jsg/OQX/PSNZ5O
clqRVlelf0Itvsq2rqbPvs2D6n3JQ6P2HK3z/NDmtzLqfPsac/j60xrd/cgv
+LVXpFaZa7b8+c6AmY6RUebnHpTCuUU4S2s5vxbb5OjSuBdKPcxpsBzm/Ssy
viLxmN9dTGtIPa2N333DzLpxMdKIndr/gfMqfey7ywxqWPPwuMQK2UOktjCG
XIR2XlGhfqGrvem7ekVtfuooUoMZx29SPkbVA4kq5gk9zpzOx+UQChlVvcKo
aJNCz1u/7tyPk0Cdn2G+MKPK92xC3QEqIfBqHHnScRR90r3rLRhqnkcpXRTU
Flt2X8CsHjJSH0yrmmfhqa+HkhlfN+zh8cmk5HJXk/fuULOf7Xi/lnIhZzm+
VcMOz+BBzpnFmhXw13lARqCsrLO6WiI//8wOuDNKaXwruWO0Yd7nW0lb0Nr/
p/E+zaRxtOJoxBjHtLg//KWczw7FcLwCHX70xfyxXqtvjqOHI43HJ1Ei7Vx+
bRW6vvcxuT87VGYvmyk4jJyAveot98Fhziks7GdDviLNayRym+bxmXl6nEMe
eGn7mBSo8PnnmNe6mD1rb7AcQOaIrcSY+LPBVqXn2BGfmRcucGV6nxk1H+6u
Imf6Ys6s9THv8hsi9ZXTGngBjZhHScwL3i4013oEuX4IcfpRKixxrDLDJFQI
dmjblfwS4jxq3bxwsSvD9xmo2Y9RT9NSHq75QYjHf0txU97WezMa/efx/5X0
cuahv7Zd3wbl8OuPhJ5R8VToAC1HkQvD0aAWI9IwmXWC91fq/bnkgNjenoYa
e2o1+rVzZ7lz+5rQE8b34m/YY3wr2P6QB+CZg4wbttsq5DAY/sSV2a1R4dcm
+q8dcHXUk0II8P6NHpgv7gvINj6I3+ModxPHaDl8v2CLr+M/lhxHlOfy4PcQ
Yzbpk+wVn5BjAnGndTl8lhHJg+28k1ElfNAZgjM4dxaJ8l52lTL/NXNQcnnt
i9jGG/jvn0v7Ke6vzx/M8dugfvPMapfv0DTtoVCl4DjyZuwQvJneUGMr1gjp
nRHsS3relrVqNfmZzg+7c2rhVcbZLmumpTvErpvmXT1MjUXx3pC3sjtq/S/G
w1eRg4cGqWA6S75Kf182TOnHu9KOh3z6OkPUBo/janLxKq4SOZTt+pCEX2r/
Yqo9pF6sWrufB+GcZF9zfORqrlDT7kWvT85qiv7PZSH9kx6wWtf7yUf20WFV
peyh6zyCddyW3ZfSUarX7yOVH2FVGnfJR2qftvjtoNud9TbShLJODlnEoQjP
dIYoi/sbHmVcEDsru1uqdxo+onlvJW6/c+bBLN1HQdBTHsX9H8v99i5qLPVU
0mqlodXV58HRDx1r1ifRLEGmbi/3Ay18zjb2GvnyEL3asUFSPiTzBTleflUp
szy/bHV5zwkzDcdRh49lz7AuXKO/L9CxhZSOldkfItL3uVJgJ49jFtC6gJ75
WrUj18jc/zWW33kBIozxVeyS7Mfojd8DaatB0/MxpM/yIjUKFCtqX2Pe+QXB
MhPxFfSPkmbuirfe27Q9SLNq5XUcaV4rkyKgU3VAP3DtfaGm0jDXa/uEcY86
K/W6oJ9sFNpmaan+W0h+/wIrTT3a62SvZrytMQvtawUZImEfqNknRELLXJGj
WMo5RKa9v/rX+VuHD1HaXaxb8+AQe6KvXbNaf4URb4ftt1BeesvQ9BZiB2GU
9T9HkFL1bDWMKHXCrqzG7N9VXJOvA4RR2jI7RpcGbfJ6dN5Vt2NbzSW5VobG
QjjIaFuuJAhRP1pe0j1jCu+L+9k2nF+Z+tBDEnWKrbDr84hNe1Dll2DzpcHK
X6GlXSkt6UquIsr2o4Bmz09tVeeia6N6vyOSRhViwVKicXASS7IdfuSw9qpZ
bTrqFzp9aTmd8uc7WFK7oURrn2qrCj9r4/O9ta+PZTDXoLK/lJx23PVdWbpG
3HX4KfQ2ldqOpU7OShE6pxOtekKFH4f4rGjM0ixIqHUtaBjDtVLz02pO1f/i
Evf2Ae5cR4IL2CvFUmz9jWqcxkpa1mprHtZHEw3qYS1AVl8rTGc/MV/Xhwiy
C7W0S10pTbvVSaG61jDMZHuYXsQjlbg76VMuqcMh9DVlO7ZOpk89RInIgTrd
Ymo1WSq1hZ9/8jtE6zzgyczUjjXswlJnELnOFBV+vshH1Y1eaPCZOmoWGq7U
tz9DlP4t7CFxVpCV1zNZ31/qG1e4J/2ssEbvwvHUmbPYUs7UYzrrvlpL+izr
biqhzRLyVeDGvev07T93b6mXMrse7cO8XfhYRZ9i6QcR1Ze1tL8I2QijoRQd
78ZCTIdv0Lf+0j3l5+rUGwmX2Be9Ew0YKS3Q2JCUdpOW9jn3tp9Ts9Km8u3d
ieBWffvzRODfQo7NQtvvJH1j2b7x8w19+7+4t/zcAtcdpXVQsFYLky4u0VKu
Jg3+bcSEFpltW+hp+I6+fY17C3NVcYAY7J/0dBhFLkpe0VJ999pMfu2GMeWB
pPMefboxUya+1bTuQOrQzZpreUH+yw978+mH9ekvkSLEii/T7Amh4O/17vUN
FKAn1w9VxsNs7Q/1+a+wtDzXT56bxqce1af+myu1aeZEaHhMn77B3W2K/krZ
T+jTXyUFTaNGKfP/6FOLs7u93EpPylild0XPe+LZz/SsGPX/AOqTDVs=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 391},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->31506406]
}, Open  ]],

Cell[TextData[{
 "For a moderate sized tensor product grid like the example above, using ",
 Cell[BoxData[
  ButtonBox["Apply",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Apply"]], "InlineFormula"],
 " is reasonably fast. However, as the grid size gets larger, this approach \
may not be the fastest because ",
 Cell[BoxData[
  ButtonBox["Apply",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Apply"]], "InlineFormula"],
 " can only be used in limited ways with the ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " compiler and hence, with packed arrays. If you can define your function so \
you can use ",
 Cell[BoxData[
  ButtonBox["Map",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Map"]], "InlineFormula"],
 " instead of ",
 StyleBox["Apply", "MR"],
 ", you may be able to use a ",
 Cell[BoxData[
  ButtonBox["CompiledFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CompiledFunction"]], "InlineFormula"],
 " since ",
 StyleBox["Map", "MR"],
 " has greater applicability within the ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " compiler than does ",
 StyleBox["Apply", "MR"],
 "."
}], "Text",
 CellID->488928370],

Cell[TextData[{
 "This defines a ",
 Cell[BoxData[
  ButtonBox["CompiledFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CompiledFunction"]], "InlineFormula"],
 " which uses ",
 StyleBox["Map", "MR"],
 " to get the values on the grid. (If the first grid dimension is greater \
than the SystemOption \"",
 StyleBox["MapCompileLength", "MR"],
 "\", then you do not need to construct the ",
 StyleBox["CompiledFunction", "MR"],
 " since the compilation is done automatically when grid is a packed array.)"
}], "MathCaption",
 CellID->1532979704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", " ", "=", " ", 
  RowBox[{"Compile", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"grid", ",", " ", "_Real", ",", " ", "4"}], "}"}], "}"}], ",", 
    " ", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{"Function", "[", 
       RowBox[{
        RowBox[{"{", "X", "}"}], ",", " ", 
        RowBox[{"Module", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"Xs", " ", "=", " ", 
            RowBox[{"X", " ", "-", " ", ".5"}]}], "}"}], ",", " ", 
          RowBox[{"Exp", "[", 
           RowBox[{"-", 
            RowBox[{"(", 
             RowBox[{"Xs", ".", "Xs"}], ")"}]}], "]"}]}], "]"}]}], "]"}], ",",
       "grid", ",", " ", 
      RowBox[{"{", "3", "}"}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[24]:=",
 CellID->2090355176],

Cell[BoxData[
 TagBox[
  RowBox[{"CompiledFunction", "[", 
   RowBox[{
    RowBox[{"{", "grid", "}"}], ",", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{"Function", "[", 
       RowBox[{
        RowBox[{"{", "X", "}"}], ",", 
        RowBox[{"Module", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"Xs", "=", 
            RowBox[{"X", "-", "0.5`"}]}], "}"}], ",", 
          SuperscriptBox["\[ExponentialE]", 
           RowBox[{"-", 
            RowBox[{"Xs", ".", "Xs"}]}]]}], "]"}]}], "]"}], ",", "grid", ",", 
      
      RowBox[{"{", "3", "}"}]}], "]"}], ",", "\<\"-CompiledCode-\"\>"}], 
   "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{524, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->277511222]
}, Open  ]],

Cell[TextData[{
 "An even better approach, when possible, is to take advantage of listability \
when your function consists of operations and functions which have the ",
 Cell[BoxData[
  ButtonBox["Listable",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Listable"]], "InlineFormula"],
 " attribute. The trick is to separate the x, y, and z values at each of the \
points on the tensor product grid. The fastest way to do this is using ",
 StyleBox["Transpose[fddf[\"Grid\"]], RotateLeft[Range[n + 1]]]", "MR"],
 ", where n = ",
 StyleBox["Length[fddf[\"DerivativeOrder\"]]", "MR"],
 " is the dimensionality of the space in which you are approximating the \
derivative. This will return a list of length n, which has the values on the \
grid for each of the component dimensions separately. With the ",
 Cell[BoxData[
  ButtonBox["Listable",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Listable"]], "InlineFormula"],
 " attribute, functions applied to this will thread over the grid."
}], "Text",
 CellID->784761289],

Cell[TextData[{
 "This defines a function that takes advantage of the fact that ",
 Cell[BoxData[
  ButtonBox["Exp",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exp"]], "InlineFormula"],
 " has the ",
 Cell[BoxData[
  ButtonBox["Listable",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Listable"]], "InlineFormula"],
 " attribute to find the values on the grid."
}], "MathCaption",
 CellID->1222867096],

Cell[BoxData[
 RowBox[{
  RowBox[{"fgrid", "[", "grid_", "]"}], " ", ":=", " ", 
  RowBox[{"Apply", "[", 
   RowBox[{"f", ",", " ", 
    RowBox[{"Transpose", "[", 
     RowBox[{"grid", ",", " ", 
      RowBox[{"RotateLeft", "[", 
       RowBox[{
        RowBox[{"Range", "[", 
         RowBox[{"TensorRank", "[", "grid", "]"}], "]"}], ",", " ", "1"}], 
       "]"}]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[25]:=",
 CellID->739603844],

Cell["\<\
This compares timings for the three methods. The commands are repeated \
several times to get more accurate timings.\
\>", "MathCaption",
 CellID->472370995],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Module", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"repeats", " ", "=", " ", "100"}], ",", 
     RowBox[{"grid", " ", "=", " ", 
      RowBox[{"fddf", "[", "\"\<Grid\>\"", "]"}]}], ",", " ", 
     RowBox[{"n", " ", "=", " ", 
      RowBox[{"Length", "[", 
       RowBox[{"fddf", "[", "\"\<DerivativeOrder\>\"", "]"}], "]"}]}]}], 
    "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"First", "[", 
      RowBox[{"Timing", "[", 
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{"Apply", "[", 
          RowBox[{"f", ",", " ", "grid", ",", " ", 
           RowBox[{"{", "n", "}"}]}], "]"}], ",", 
         RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"First", "[", 
      RowBox[{"Timing", "[", 
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{"cf", "[", "grid", "]"}], ",", 
         RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"First", "[", 
      RowBox[{"Timing", "[", 
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{"fgrid", "[", "grid", "]"}], ",", 
         RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], "]"}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->1434087395],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.7660000000000016`", ",", "0.12499999999999971`", ",", 
   "0.04699999999999971`"}], "}"}]], "Output",
 ImageSize->{146, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->37262781]
}, Open  ]],

Cell[TextData[{
 "The example timings show that using the ",
 Cell[BoxData[
  ButtonBox["CompiledFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CompiledFunction"]], "InlineFormula"],
 " is typically much faster than using ",
 Cell[BoxData[
  ButtonBox["Apply",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Apply"]], "InlineFormula"],
 " and taking advantage of listability is a little faster yet."
}], "Text",
 CellID->924245772]
}, Open  ]],

Cell[CellGroupData[{

Cell["Pseudospectral Derivatives", "Subsubsection",
 CellTags->{"c:5", "b:7.1.2", "ndsg:2.0.1.2", "PDE:1.2.4"},
 CellID->1761574857],

Cell["\<\
The maximum value the difference order can take on is determined by the \
number of points in the grid. If you exceed this, a warning message will be \
given and the order reduced automatically.\
\>", "Text",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->363179136],

Cell["\<\
This uses maximal order to approximate the first derivative of the sine \
function on a random grid. \
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->1083268061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
  RowBox[{"1", ",", " ", "rgrid", ",", " ", 
   RowBox[{"Sin", "[", "rgrid", "]"}], ",", 
   RowBox[{"DifferenceOrder", "\[Rule]", 
    RowBox[{"Length", "[", "rgrid", "]"}]}]}], "]"}]], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[50]:=",
 CellID->361375435],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve`FiniteDifferenceDerivative", "::", "\<\"ordred\"\>"}], ":",
   " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"There are insufficient points in \
dimension \\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(1\\)\\!\\(\\*StyleBox[\\\"\\\\\\\
\" to achieve the requested approximation order. Order will be reduced to \
\\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(11\\)\\!\\(\\*StyleBox[\\\"\\\\\\\".\\\\\\\
\"\\\", \\\"MSG\\\"]\\)\"\>"}]], "Message", "MSG",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"During evaluation of In[50]:=",
 CellID->1467854779],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.0000118928134494`", ",", "0.5868205830975541`", ",", 
   "0.5360891645117325`", ",", "0.4636135930076364`", ",", 
   RowBox[{"-", "0.14916093919479526`"}], ",", 
   RowBox[{"-", "0.21526462140177718`"}], ",", 
   RowBox[{"-", "0.7479342839123092`"}], ",", 
   RowBox[{"-", "0.7958384290138014`"}], ",", 
   RowBox[{"-", "0.9782143761709053`"}], ",", 
   RowBox[{"-", "0.26415470676580216`"}], ",", "0.9970886938643753`", ",", 
   "0.9999405113544253`"}], "}"}]], "Output",
 ImageSize->{441, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[50]=",
 CellID->608281247]
}, Open  ]],

Cell[TextData[{
 "Using a limiting order is commonly referred to as a ",
 StyleBox["pseudospectral",
  FontWeight->"Bold"],
 " derivative. A common problem with these is that artificial oscillations \
(Runge's phenomena) can be extreme. However, there are two instances where \
this is not the case: a uniform grid with periodic repetition and a grid with \
points at the zeros of the Chebyshev polynomials, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["T", "n"], TraditionalForm]]],
 ", or Chebyshev-Gauss-Lobatto points [",
 ButtonBox["F96a",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#22110"],
 "], [",
 ButtonBox["QV94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#14672"],
 "].",
 " ",
 "The computation in both of these cases can be done using a fast Fourier \
transform, which is efficient and minimizes roundoff error."
}], "Text",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->2054465836],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     StyleBox["DifferenceOrder", "InlineCode"], "->", 
     StyleBox["n", "TI"]}], Cell[TextData[{
     "use ",
     Cell[BoxData[
      SuperscriptBox["n", "th"]], "InlineCode"],
     "order finite differences to approximate the derivative"
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["DifferenceOrder", "InlineCode"], "->", 
     ButtonBox["Length",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Length"], "[", 
     StyleBox["grid", "TI"], "]"}], Cell[TextData[{
     "use the highest possible order finite differences to approximate \
derivative on ",
     StyleBox["grid", "TI"],
     "  (Not generally recommended.)"
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["DifferenceOrder", "InlineCode"], "->", 
     StyleBox["\"\<Pseudospectral\>\"", "InlineCode"]}], Cell[TextData[{
     "use a pseudospectral derivative approximation.  Only applicable when \
the grid points are spaced corresponding to the Chebyshev-Gauss-Lobatto \
points or when the grid is uniform with ",
     StyleBox["PeriodicInterpolation", "InlineCode"],
     " -> ",
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"],
     "."
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["DifferenceOrder", "InlineCode"], "->", "{", 
     StyleBox["n1", "TI"], ",", 
     StyleBox["n2", "TI"], ",", 
     StyleBox["...", "TI"], "}"}], Cell[TextData[{
     "use difference orders ",
     StyleBox["n1", "TI"],
     ", ",
     StyleBox["n2", "TI"],
     ", ",
     "... in dimensions 1, 2, ... respectively"
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{GridBoxItemSize->{"Columns" -> {
     Scaled[0.45], {
      Scaled[0.55]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->993201831],

Cell["Settings for the DifferenceOrder option. ", "Caption",
 CellID->1185871851],

Cell["\<\
This gives a pseudospectral approximation for the second derivative of the \
sine function on a uniform grid.\
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->344187118],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"ugrid", " ", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{"2", " ", "\[Pi]", " ", 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"0", ",", "10"}], "]"}], "/", "10"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
  RowBox[{"1", ",", " ", "ugrid", ",", " ", 
   RowBox[{"Sin", "[", "ugrid", "]"}], ",", " ", 
   RowBox[{"PeriodicInterpolation", "\[Rule]", "True"}], ",", " ", 
   RowBox[{"DifferenceOrder", "->", "\"\<Pseudospectral\>\""}]}], 
  "]"}]}], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[27]:=",
 CellID->632601084],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.0000000000000007`", ",", "0.8090169943749467`", ",", 
   "0.30901699437494795`", ",", 
   RowBox[{"-", "0.30901699437494723`"}], ",", 
   RowBox[{"-", "0.8090169943749477`"}], ",", 
   RowBox[{"-", "0.9999999999999996`"}], ",", 
   RowBox[{"-", "0.8090169943749479`"}], ",", 
   RowBox[{"-", "0.30901699437494723`"}], ",", "0.30901699437494684`", ",", 
   "0.8090169943749472`", ",", "1.0000000000000007`"}], "}"}]], "Output",
 ImageSize->{349, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[28]=",
 CellID->61745455]
}, Open  ]],

Cell["\<\
This computes the error at each point. The approximation is accurate to \
roundoff because the effective basis for the pseudospectral derivative on a \
uniform grid for a periodic function are the trigonometric functions.\
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->2030437009],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", " ", "-", " ", 
  RowBox[{"Cos", "[", "ugrid", "]"}]}]], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[29]:=",
 CellID->93693386],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6.661338147750939`*^-16", ",", 
   RowBox[{"-", "7.771561172376096`*^-16"}], ",", "4.996003610813204`*^-16", 
   ",", "1.1102230246251565`*^-16", ",", 
   RowBox[{"-", "3.3306690738754696`*^-16"}], ",", "4.440892098500626`*^-16", 
   ",", 
   RowBox[{"-", "3.3306690738754696`*^-16"}], ",", "3.3306690738754696`*^-16",
    ",", 
   RowBox[{"-", "3.885780586188048`*^-16"}], ",", 
   RowBox[{"-", "1.1102230246251565`*^-16"}], ",", 
   "6.661338147750939`*^-16"}], "}"}]], "Output",
 ImageSize->{426, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[29]=",
 CellID->17165946]
}, Open  ]],

Cell[TextData[{
 "The Chebyshev-Gauss-Lobatto points are the zeros of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", " ", "-", " ", 
      SuperscriptBox["x", "2"]}], ")"}], 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["T", "n"], "'"], "(", "x", ")"}]}], TraditionalForm]]],
 ".",
 " ",
 "Using the property ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["T", "n"], "(", "x", ")"}], " ", "=", " ", 
    RowBox[{
     RowBox[{
      SubscriptBox["T", "n"], "(", 
      RowBox[{"cos", "(", "\[Theta]", ")"}], ")"}], "==", 
     RowBox[{"cos", "(", 
      RowBox[{"n", " ", "\[Theta]"}], ")"}]}]}], TraditionalForm]]],
 ", these can be shown to be at ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "j"], "=", 
    RowBox[{"cos", "(", 
     FractionBox[
      RowBox[{"\[Pi]", " ", "j"}], "n"], ")"}]}], TraditionalForm]]],
 " ."
}], "Text",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->1152656108],

Cell["\<\
This defines a simple function that generates a grid of n points with \
leftmost point at x0 and interval length L having the spacing of the \
Chebyshev-Gauss-Lobatto points\
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->1293872925],

Cell[BoxData[
 RowBox[{
  RowBox[{"CGLGrid", "[", 
   RowBox[{"x0_", ",", " ", "L_", ",", " ", 
    RowBox[{"n_Integer", " ", "/;", " ", 
     RowBox[{"n", " ", ">", " ", "1"}]}]}], "]"}], " ", ":=", " ", 
  RowBox[{"x0", " ", "+", 
   RowBox[{
    FractionBox["1", "2"], " ", "L", 
    RowBox[{"(", 
     RowBox[{"1", " ", "-", "  ", 
      RowBox[{"Cos", "[", 
       RowBox[{"\[Pi]", " ", 
        RowBox[{
         RowBox[{"Range", "[", 
          RowBox[{"0", ",", 
           RowBox[{"n", "-", "1"}]}], "]"}], "/", 
         RowBox[{"(", 
          RowBox[{"n", "-", "1"}], ")"}]}]}], "]"}]}], ")"}]}]}]}]], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[30]:=",
 CellID->1099179777],

Cell["\<\
This computes the pseudospectral derivative for a Gaussian function.\
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->1224497971],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"cgrid", " ", "=", " ", 
   RowBox[{"CGLGrid", "[", 
    RowBox[{
     RowBox[{"-", "5"}], ",", "10.", ",", "16"}], "]"}]}], ";", 
  RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
   RowBox[{"1", ",", "cgrid", ",", " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"-", 
      SuperscriptBox["cgrid", "2"]}], "]"}], ",", 
    RowBox[{"DifferenceOrder", "->", "\"\<Pseudospectral\>\""}]}], 
   "]"}]}]], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[31]:=",
 CellID->706534857],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.04024259748482862`", ",", 
   RowBox[{"-", "0.020992247826051122`"}], ",", "0.023915096104594222`", ",", 
   
   RowBox[{"-", "0.03005892260765349`"}], ",", "0.042555322789605075`", ",", 
   RowBox[{"-", "0.05908708518729317`"}], ",", "0.40662970477176`", ",", 
   "0.6033596780293696`", ",", 
   RowBox[{"-", "0.6033596780293703`"}], ",", 
   RowBox[{"-", "0.40662970477175886`"}], ",", "0.05908708518729259`", ",", 
   RowBox[{"-", "0.042555322789604964`"}], ",", "0.030058922607653433`", ",", 
   
   RowBox[{"-", "0.02391509610459439`"}], ",", "0.02099224782605101`", ",", 
   RowBox[{"-", "0.04024259748482619`"}]}], "}"}]], "Output",
 ImageSize->{419, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[31]=",
 CellID->657113483]
}, Open  ]],

Cell["This shows a plot of the approximation and the exact values.", \
"MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->609994027],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"ListPlot", "[", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"cgrid", ",", " ", "%"}], "}"}], "]"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"PointSize", "[", "0.025", "]"}]}]}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"Evaluate", "[", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"-", 
            SuperscriptBox["x", "2"]}], "]"}], ",", "x"}], "]"}], "]"}], ",", 
       
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[32]:=",
 CellID->649406385],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 223},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[32]=",
 CellID->406539086]
}, Open  ]],

Cell["\<\
This shows a plot of the derivative computed using a uniform grid with the \
same number of points with maximal difference order.\
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->1394755316],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"ugrid", " ", "=", " ", 
   RowBox[{
    RowBox[{"-", "5"}], " ", "+", " ", 
    RowBox[{"10.", " ", 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"0", ",", "15"}], "]"}], "/", "15"}]}]}]}], ";"}], "\n", 
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"ListPlot", "[", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"ugrid", ",", " ", 
          RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
           RowBox[{"1", ",", "ugrid", ",", " ", 
            RowBox[{"Exp", "[", 
             RowBox[{"-", 
              SuperscriptBox["ugrid", "2"]}], "]"}], ",", 
            RowBox[{"DifferenceOrder", "\[Rule]", 
             RowBox[{
              RowBox[{"Length", "[", "ugrid", "]"}], " ", "-", " ", "1"}]}]}],
            "]"}]}], "}"}], "]"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"PointSize", "[", "0.025", "]"}]}]}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"Evaluate", "[", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"-", 
            SuperscriptBox["x", "2"]}], "]"}], ",", "x"}], "]"}], "]"}], ",", 
       
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[35]:=",
 CellID->1872273482],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWk1oXFUUvpn3M4k2ie7aJGI3lVoX2kpxITFpqm1jjVrxZyEuxkHICFqp
caMb3ehGrdVNKiWN1CSNdpFaqAgVxIJYEAtiQVQoFaQuLILFRUAYzzn3zn3v
nTn3vulMp81ghpebuT/n537n55735k2Upiafe6E0VSmX1u/cV3ppslJ+ef2O
vftgKOhSqmtSKXV+rYLvVfhGf/ipwJ9SJ7ANamNVtYBNcXn534PTZ/Y8cmRk
+ODu8cPvvH368uVlFVoKJJgnSlhZmVyCZSrEJnjqiXlYilMFkb+mmvv4LKxG
goioPnj/GzPf5aHa++KJDBXIaoBK7wIJYmgCz8rXXv0iw7/87LEG+H956tcM
1aef/OClIoRxx4YqoAZkm3kZt6PYRGe+/e2N108BDrAc5KrIrg1h6XGR8rMV
NdopeuJge6XMGzTQ3gX4H4mryFsiPWpb7SOptbyvaWeNBIrbMJmV6VHIEk0v
pmYbl9YcFcVQnNbNraGMDmIXeDH8f3nuqp4rN+bl0QUj258J6ATIjeKrlwWa
lXet6VZzQOfqmRdxg9g6qtEDIsUAjbqr0m5GNWrQGmXjI2Z8Oxu/PSXFXcUW
GdW2VF8J0kbZvE+qXAUjVS2HoN73VlNRz/bl0iyNx6aUxGwFXWS6PsA43Mf6
Yx4NkP6OlCR3BR5XdXRL0lBfaR8Fxt1dqde4k66ZHMQtdUvV7ZGDddRjGS1o
3lP5c9/cZqj53mKPDv1Vv/8PGJ73eHjEovcNYVt3B7LRw6fM+EQ5uvVW/VHV
65E1nMP7LtrDrpxVfR4JD4m7kbW9dOkfFZ79/ncVgY8dnvlOX5+f/EkFOBqU
S8dGhqelO9k1LWh4Ww5+d6YwcK9qRYOtIkZJxgpN86HuA3kIkyoC6Ralt978
ikhUsH3skF5KeUdFMHl86RwACBci3NPGnfQzzbmdNJZbWuCwpkXtN3hk8/OH
KqcQMDv99XlA8f6xQ7FJQOiESezx/Lvn0SOwHJiqAlkkOQmoH1u7QVoAo4Bb
F6/yLrVe2RNoM2G/uQnKvhZt4stAeZmXYqEHbPLLz3+CG4ObwxwwAcwBwOcr
S6rIzjytjfvUQo4FzCexzTKvTJ2E3UHgdNc01+MqoAZ46fBRUZKHWvBO93mt
4wPPwAoe8nUe2cf6/awfeiyF/Eh+ZLHBWayAGEI3pLiCrhrhyGoJn4sX/9Y5
GW98NDZwQZioAiG2a8eMYU+0+KQClIJLQwuiyAIBxFY61iKLXEazBL+0ZhhK
xL179/gsWBCcAi74AgKQuTKVAascwqRyUBpPczpD15zOaZSxX2T48Oow8lg1
Qd3GlNkbejrNJLUPqKppEx35fVmeB+hnm835TTbqlX7oq6zfhgmSUDEzPW/M
wYzvI8zBTPuHjlCQazKn1RCyBQw8/tgc2RS90GIL3m7iDEYvXPhr4sFZo609
QY1fgd21A+qtprQrpvwiqdr4Ko5q3xVbU/flXJV4jkUpExVJ1qCo6LFc4cM1
5zVyY1606QpsZrnWoXZzVYjCFCa8bg49mNjdBud+/AM8ZHznjNI5G04UnWFU
kWoj6ynPPL348MRH1gOmaycx053H9FZxLxzHOGdvvTl7GWT8Nnpwd3PZIOra
I/panq6CdeEkxmDspnxvM3a1sdzI72PlyMrzkSERpy05VMOirJty0OzN4cpj
uCxK8d3hkdXV3SLdkznS+xvyy/3Ud1dV60iD/SKvW0Ve7ppmLfGqPT97l1a7
a8Vkdb2M7Pkz4FnpzkE+/m6by1Tv5dhinUh1QER1SFy7Up+td/4z0vbr2d7f
H9pLd/1+r7iWvzt0SkytPI1WY/96v4XQbHw18j5Bq+8sdM7bB6uR1S49qc53
1LdHRXp99+yuSec9VO7qc85D5a5CfVTZKnTBs9Jdhfr4u6tQmUrfUbmr0Fou
0nahaKqrPhdFzvhVv8+ouv4DJ8ZeCg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 223},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[36]=",
 CellID->140299997]
}, Open  ]],

Cell["\<\
Even though the approximation is somewhat better in the center (because the \
points are more closely spaced there in the uniform grid), the plot clearly \
shows the disastrous oscillation typical of overly high order finite \
difference approximations. Using the Chebyshev-Gauss-Lobatto spacing has \
minimized this.\
\>", "Text",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->1158877229],

Cell["\<\
This shows a plot of the pseudospectral derivative approximation computed \
using a uniform grid with periodic repetition.\
\>", "MathCaption",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->321532202],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"ugrid", " ", "=", " ", 
   RowBox[{
    RowBox[{"-", "5"}], " ", "+", " ", 
    RowBox[{"10.", " ", 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"0", ",", "15"}], "]"}], "/", "15"}]}]}]}], ";"}], "\n", 
 RowBox[{"Show", "[", " ", 
  RowBox[{
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"ListPlot", "[", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"ugrid", ",", " ", 
          RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
           RowBox[{"1", ",", "ugrid", ",", " ", 
            RowBox[{"Exp", "[", 
             RowBox[{"-", 
              SuperscriptBox["ugrid", "2"]}], "]"}], ",", 
            RowBox[{"DifferenceOrder", "\[Rule]", "\"\<Pseudospectral\>\""}], 
            ",", " ", 
            RowBox[{"PeriodicInterpolation", "\[Rule]", "True"}]}], "]"}]}], 
         "}"}], "]"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"PointSize", "[", "0.025", "]"}]}]}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{"Evaluate", "[", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"-", 
            SuperscriptBox["x", "2"]}], "]"}], ",", "x"}], "]"}], "]"}], ",", 
       
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "5"}], ",", "5"}], "}"}]}], "]"}]}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "Input",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"In[70]:=",
 CellID->643888436],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 223},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellLabel->"Out[71]=",
 CellID->878682385]
}, Open  ]],

Cell["\<\
With the assumption of periodicity, the approximation is significantly \
improved. The accuracy of the periodic pseudospectral approximations is \
sufficiently high to justify, in some cases, using a larger computational \
domain to simulate periodicity, say for a pulse like the above example. \
Despite the great accuracy of these approximations, they are not without \
pitfalls: one of the worst is probably aliasing error, whereby an oscillatory \
function component with too great a frequency can be misapproximated or \
disappear entirely.\
\>", "Text",
 CellTags->{"b:7.1.2", "ndsg:2.0.1.2"},
 CellID->684711338]
}, Open  ]],

Cell[CellGroupData[{

Cell["Accuracy and Convergence of Finite Difference Approximations", \
"Subsubsection",
 CellTags->{"b:7.1.3", "ndsg:2.0.1.3", "PDE:1.2.5"},
 CellID->1727156497],

Cell[TextData[{
 "When using finite differences, it is important to keep in mind that the \
truncation error, or the asymptotic approximation error induced by cutting \
off the Taylor series approximation, is not the only source of error. There \
are two other sources of error in applying finite difference formulas; \
condition error and roundoff error [",
 ButtonBox["GMW81",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#16479"],
 "]. Roundoff error comes from roundoff in the arithmetic computations \
required. Condition error comes from magnification of any errors in the \
function values, typically from the division by a power of the step size, and \
so grows with decreasing step size. This means that in practice, even though \
the truncation error approaches zero as h does, the actual error will start \
growing beyond some point.",
 " ",
 "The figures below demonstrate the typical behavior as h becomes small for a \
smooth function."
}], "Text",
 CellID->1134343873],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.293651 0.263645 0.576668 0.0407324 [
[.23364 .3009 -11 -9.8125 ]
[.23364 .3009 11 0 ]
[.49728 .3009 -14 -9.8125 ]
[.49728 .3009 14 0 ]
[.76093 .3009 -17 -9.8125 ]
[.76093 .3009 17 0 ]
[-0.0125 .00641 -51.6875 -6.15625 ]
[-0.0125 .00641 0 6.15625 ]
[-0.0125 .12861 -51.6875 -6.15625 ]
[-0.0125 .12861 0 6.15625 ]
[-0.0125 .25081 -47.4375 -6.15625 ]
[-0.0125 .25081 0 6.15625 ]
[-0.0125 .37301 -46 -4.90625 ]
[-0.0125 .37301 0 4.90625 ]
[-0.0125 .4952 -28 -4.90625 ]
[-0.0125 .4952 0 4.90625 ]
[-0.0125 .6174 -16 -4.90625 ]
[-0.0125 .6174 0 4.90625 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.23364 .3134 m
.23364 .31965 L
s
gsave
.23364 .3009 -72 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(100) show
81.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.49728 .3134 m
.49728 .31965 L
s
gsave
.49728 .3009 -75 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1000) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.76093 .3134 m
.76093 .31965 L
s
gsave
.76093 .3009 -78 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10000) show
93.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.04936 .3134 m
.04936 .31715 L
s
.09578 .3134 m
.09578 .31715 L
s
.12872 .3134 m
.12872 .31715 L
s
.15427 .3134 m
.15427 .31715 L
s
.17515 .3134 m
.17515 .31715 L
s
.1928 .3134 m
.1928 .31715 L
s
.20809 .3134 m
.20809 .31715 L
s
.22158 .3134 m
.22158 .31715 L
s
.313 .3134 m
.313 .31715 L
s
.35943 .3134 m
.35943 .31715 L
s
.39237 .3134 m
.39237 .31715 L
s
.41792 .3134 m
.41792 .31715 L
s
.4388 .3134 m
.4388 .31715 L
s
.45645 .3134 m
.45645 .31715 L
s
.47173 .3134 m
.47173 .31715 L
s
.48522 .3134 m
.48522 .31715 L
s
.57665 .3134 m
.57665 .31715 L
s
.62308 .3134 m
.62308 .31715 L
s
.65601 .3134 m
.65601 .31715 L
s
.68156 .3134 m
.68156 .31715 L
s
.70244 .3134 m
.70244 .31715 L
s
.72009 .3134 m
.72009 .31715 L
s
.73538 .3134 m
.73538 .31715 L
s
.74887 .3134 m
.74887 .31715 L
s
.84029 .3134 m
.84029 .31715 L
s
.88672 .3134 m
.88672 .31715 L
s
.91966 .3134 m
.91966 .31715 L
s
.94521 .3134 m
.94521 .31715 L
s
.96609 .3134 m
.96609 .31715 L
s
.98374 .3134 m
.98374 .31715 L
s
.99902 .3134 m
.99902 .31715 L
s
.25 Mabswid
0 .3134 m
1 .3134 L
s
0 .00641 m
.00625 .00641 L
s
gsave
-0.0125 .00641 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(14) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .12861 m
.00625 .12861 L
s
gsave
-0.0125 .12861 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(11) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .25081 m
.00625 .25081 L
s
gsave
-0.0125 .25081 -108.438 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(8) show
105.125 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .37301 m
.00625 .37301 L
s
gsave
-0.0125 .37301 -107 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.00001) show
105.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .4952 m
.00625 .4952 L
s
gsave
-0.0125 .4952 -89 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.01) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .6174 m
.00625 .6174 L
s
gsave
-0.0125 .6174 -77 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
75.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .08988 m
.00375 .08988 L
s
0 .10207 m
.00375 .10207 L
s
0 .10921 m
.00375 .10921 L
s
0 .11429 m
.00375 .11429 L
s
0 .11823 m
.00375 .11823 L
s
0 .12145 m
.00375 .12145 L
s
0 .12417 m
.00375 .12417 L
s
0 .12653 m
.00375 .12653 L
s
0 .21208 m
.00375 .21208 L
s
0 .22426 m
.00375 .22426 L
s
0 .23141 m
.00375 .23141 L
s
0 .23649 m
.00375 .23649 L
s
0 .24043 m
.00375 .24043 L
s
0 .24365 m
.00375 .24365 L
s
0 .24637 m
.00375 .24637 L
s
0 .24873 m
.00375 .24873 L
s
0 .33428 m
.00375 .33428 L
s
0 .34646 m
.00375 .34646 L
s
0 .35361 m
.00375 .35361 L
s
0 .35868 m
.00375 .35868 L
s
0 .36262 m
.00375 .36262 L
s
0 .36584 m
.00375 .36584 L
s
0 .36857 m
.00375 .36857 L
s
0 .37093 m
.00375 .37093 L
s
0 .45648 m
.00375 .45648 L
s
0 .46866 m
.00375 .46866 L
s
0 .4758 m
.00375 .4758 L
s
0 .48088 m
.00375 .48088 L
s
0 .48482 m
.00375 .48482 L
s
0 .48804 m
.00375 .48804 L
s
0 .49076 m
.00375 .49076 L
s
0 .49312 m
.00375 .49312 L
s
0 .57867 m
.00375 .57867 L
s
0 .59086 m
.00375 .59086 L
s
0 .598 m
.00375 .598 L
s
0 .60308 m
.00375 .60308 L
s
0 .60702 m
.00375 .60702 L
s
0 .61024 m
.00375 .61024 L
s
0 .61296 m
.00375 .61296 L
s
0 .61532 m
.00375 .61532 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
1 0 0 r
.015 w
.02381 .60082 Mdot
.04936 .60063 Mdot
.06027 .59915 Mdot
.0794 .59514 Mdot
.10317 .58832 Mdot
.12285 .58159 Mdot
.14473 .57558 Mdot
.16309 .57106 Mdot
.18254 .565 Mdot
.20222 .55899 Mdot
.22158 .55333 Mdot
.24245 .54693 Mdot
.2619 .54079 Mdot
.28158 .53489 Mdot
.30221 .52855 Mdot
.32182 .5225 Mdot
.34127 .51657 Mdot
.36095 .51047 Mdot
.38094 .50431 Mdot
.40065 .49824 Mdot
.42063 .49206 Mdot
.44031 .48598 Mdot
.46031 .4798 Mdot
.48028 .47364 Mdot
.5 .46754 Mdot
.51986 .4614 Mdot
.53967 .45529 Mdot
.55951 .44916 Mdot
.57937 .44302 Mdot
.59923 .43688 Mdot
.61904 .43076 Mdot
.63888 .42463 Mdot
.65873 .4185 Mdot
.67855 .41238 Mdot
.6984 .40624 Mdot
.71824 .40011 Mdot
.7381 .39398 Mdot
.75794 .38784 Mdot
.77779 .38171 Mdot
.79763 .37558 Mdot
.81746 .36945 Mdot
.8373 .36332 Mdot
.85714 .35719 Mdot
.87698 .35106 Mdot
.89683 .34493 Mdot
.91667 .3388 Mdot
.93651 .33267 Mdot
.95635 .32654 Mdot
.97619 .32041 Mdot
0 1 0 r
.02381 .58786 Mdot
.04936 .58737 Mdot
.06027 .58489 Mdot
.0794 .57831 Mdot
.10317 .56719 Mdot
.12285 .55619 Mdot
.14473 .54262 Mdot
.16309 .53229 Mdot
.18254 .522 Mdot
.20222 .51009 Mdot
.22158 .49824 Mdot
.24245 .48589 Mdot
.2619 .47395 Mdot
.28158 .4618 Mdot
.30221 .44926 Mdot
.32182 .43721 Mdot
.34127 .42519 Mdot
.36095 .41312 Mdot
.38094 .40075 Mdot
.40065 .38859 Mdot
.42063 .37628 Mdot
.44031 .36412 Mdot
.46031 .35178 Mdot
.48028 .33944 Mdot
.5 .32725 Mdot
.51986 .31498 Mdot
.53967 .30274 Mdot
.55951 .29048 Mdot
.57937 .27821 Mdot
.59923 .26593 Mdot
.61904 .2537 Mdot
.63888 .24143 Mdot
.65873 .22916 Mdot
.67855 .21692 Mdot
.6984 .20465 Mdot
.71824 .1924 Mdot
.7381 .18015 Mdot
.75794 .16799 Mdot
.77779 .15586 Mdot
.79763 .14397 Mdot
.81746 .13272 Mdot
.8373 .12242 Mdot
.85714 .11427 Mdot
.87698 .10907 Mdot
.89683 .1079 Mdot
.91667 .1081 Mdot
.93651 .11122 Mdot
.95635 .11425 Mdot
.97619 .11677 Mdot
0 0 1 r
.02381 .57707 Mdot
.04936 .57764 Mdot
.06027 .57452 Mdot
.0794 .56598 Mdot
.10317 .55123 Mdot
.12285 .53644 Mdot
.14473 .51799 Mdot
.16309 .50128 Mdot
.18254 .48543 Mdot
.20222 .46888 Mdot
.22158 .45117 Mdot
.24245 .43251 Mdot
.2619 .41478 Mdot
.28158 .39688 Mdot
.30221 .37782 Mdot
.32182 .3597 Mdot
.34127 .3418 Mdot
.36095 .32371 Mdot
.38094 .30523 Mdot
.40065 .28699 Mdot
.42063 .26849 Mdot
.44031 .25024 Mdot
.46031 .23173 Mdot
.48028 .21323 Mdot
.5 .19496 Mdot
.51986 .17657 Mdot
.53967 .15825 Mdot
.55951 .13992 Mdot
.57937 .12215 Mdot
.59923 .10524 Mdot
.61904 .09218 Mdot
.63888 .08328 Mdot
.65873 .08324 Mdot
.67855 .08435 Mdot
.6984 .09002 Mdot
.71824 .08872 Mdot
.7381 .09409 Mdot
.75794 .09755 Mdot
.77779 .10294 Mdot
.79763 .10427 Mdot
.81746 .10665 Mdot
.8373 .10986 Mdot
.85714 .113 Mdot
.87698 .11722 Mdot
.89683 .11997 Mdot
.91667 .12254 Mdot
.93651 .12528 Mdot
.95635 .13013 Mdot
.97619 .13225 Mdot
1 0 1 r
.02381 .57009 Mdot
.04936 .57034 Mdot
.06027 .56639 Mdot
.0794 .55634 Mdot
.10317 .53851 Mdot
.12285 .52034 Mdot
.14473 .49739 Mdot
.16309 .47642 Mdot
.18254 .45319 Mdot
.20222 .43227 Mdot
.22158 .40968 Mdot
.24245 .38388 Mdot
.2619 .3612 Mdot
.28158 .33684 Mdot
.30221 .31202 Mdot
.32182 .288 Mdot
.34127 .26409 Mdot
.36095 .23987 Mdot
.38094 .21521 Mdot
.40065 .19092 Mdot
.42063 .16626 Mdot
.44031 .14204 Mdot
.46031 .11768 Mdot
.48028 .09485 Mdot
.5 .07772 Mdot
.51986 .07338 Mdot
.53967 .07081 Mdot
.55951 .07338 Mdot
.57937 .0795 Mdot
.59923 .08421 Mdot
.61904 .08295 Mdot
.63888 .08756 Mdot
.65873 .09135 Mdot
.67855 .08805 Mdot
.6984 .09596 Mdot
.71824 .09967 Mdot
.7381 .10334 Mdot
.75794 .10772 Mdot
.77779 .11637 Mdot
.79763 .11855 Mdot
.81746 .11566 Mdot
.8373 .11984 Mdot
.85714 .11804 Mdot
.87698 .12463 Mdot
.89683 .12886 Mdot
.91667 .13259 Mdot
.93651 .13021 Mdot
.95635 .13722 Mdot
.97619 .14108 Mdot
0 0 0 r
.02381 .57368 Mdot
.04936 .54539 Mdot
.06027 .52869 Mdot
.0794 .49069 Mdot
.10317 .42161 Mdot
.12285 .3381 Mdot
.14473 .20469 Mdot
.16309 .04716 Mdot
.18254 .01544 Mdot
.20222 .01638 Mdot
.22158 .01472 Mdot
.24245 .02126 Mdot
.2619 .0297 Mdot
.28158 .0326 Mdot
.30221 .03898 Mdot
.32182 .03758 Mdot
.34127 .04996 Mdot
.36095 .04604 Mdot
.38094 .0694 Mdot
.40065 .04065 Mdot
.42063 .06475 Mdot
.44031 .05831 Mdot
.46031 .07947 Mdot
.48028 .08711 Mdot
.5 .08394 Mdot
.51986 .06495 Mdot
.53967 .09455 Mdot
.55951 .08223 Mdot
.57937 .09303 Mdot
.59923 .08236 Mdot
.61904 .11172 Mdot
.63888 .09676 Mdot
.65873 .10484 Mdot
.67855 .12688 Mdot
.6984 .12621 Mdot
.71824 .11358 Mdot
.7381 .1158 Mdot
.75794 .13582 Mdot
.77779 .13743 Mdot
.79763 .14667 Mdot
.81746 .13272 Mdot
.8373 .1545 Mdot
.85714 .15453 Mdot
.87698 .15286 Mdot
.89683 .14739 Mdot
.91667 .15887 Mdot
.93651 .16789 Mdot
.95635 .16761 Mdot
.97619 .16052 Mdot
.5 .5 .5 r
.02381 .60332 Mdot
.04936 .59227 Mdot
.06027 .58672 Mdot
.0794 .57234 Mdot
.10317 .54806 Mdot
.12285 .519 Mdot
.14473 .47224 Mdot
.16309 .41707 Mdot
.18254 .33739 Mdot
.20222 .22759 Mdot
.22158 .07825 Mdot
.24245 .04551 Mdot
.2619 .10182 Mdot
.28158 .09222 Mdot
.30221 .08655 Mdot
.32182 .08231 Mdot
.34127 .12348 Mdot
.36095 .12002 Mdot
.38094 .14031 Mdot
.40065 .13902 Mdot
.42063 .14632 Mdot
.44031 .13873 Mdot
.46031 .177 Mdot
.48028 .17383 Mdot
.5 .16317 Mdot
.51986 .15996 Mdot
.53967 .16018 Mdot
.55951 .17994 Mdot
.57937 .20926 Mdot
.59923 .16285 Mdot
.61904 .19572 Mdot
.63888 .20974 Mdot
.65873 .20302 Mdot
.67855 .23237 Mdot
.6984 .23536 Mdot
.71824 .23738 Mdot
.7381 .24347 Mdot
.75794 .2388 Mdot
.77779 .24013 Mdot
.79763 .27487 Mdot
.81746 .26774 Mdot
.8373 .27885 Mdot
.85714 .30323 Mdot
.87698 .2847 Mdot
.89683 .27299 Mdot
.91667 .29928 Mdot
.93651 .28457 Mdot
.95635 .2992 Mdot
.97619 .30105 Mdot
1 0 0 r
.5 Mabswid
.02381 .60082 m
.04936 .60063 L
.06027 .59915 L
.0794 .59514 L
.10317 .58832 L
.12285 .58159 L
.14473 .57558 L
.16309 .57106 L
.18254 .565 L
.20222 .55899 L
.22158 .55333 L
.24245 .54693 L
.2619 .54079 L
.28158 .53489 L
.30221 .52855 L
.32182 .5225 L
.34127 .51657 L
.36095 .51047 L
.38094 .50431 L
.40065 .49824 L
.42063 .49206 L
.44031 .48598 L
.46031 .4798 L
.48028 .47364 L
.5 .46754 L
.51986 .4614 L
.53967 .45529 L
.55951 .44916 L
.57937 .44302 L
.59923 .43688 L
.61904 .43076 L
.63888 .42463 L
.65873 .4185 L
.67855 .41238 L
.6984 .40624 L
.71824 .40011 L
.7381 .39398 L
.75794 .38784 L
.77779 .38171 L
.79763 .37558 L
.81746 .36945 L
.8373 .36332 L
.85714 .35719 L
.87698 .35106 L
.89683 .34493 L
.91667 .3388 L
.93651 .33267 L
.95635 .32654 L
.97619 .32041 L
s
0 1 0 r
.02381 .58786 m
.04936 .58737 L
.06027 .58489 L
.0794 .57831 L
.10317 .56719 L
.12285 .55619 L
.14473 .54262 L
.16309 .53229 L
.18254 .522 L
.20222 .51009 L
.22158 .49824 L
.24245 .48589 L
.2619 .47395 L
.28158 .4618 L
.30221 .44926 L
.32182 .43721 L
.34127 .42519 L
.36095 .41312 L
.38094 .40075 L
.40065 .38859 L
.42063 .37628 L
.44031 .36412 L
.46031 .35178 L
.48028 .33944 L
.5 .32725 L
.51986 .31498 L
.53967 .30274 L
.55951 .29048 L
.57937 .27821 L
.59923 .26593 L
.61904 .2537 L
.63888 .24143 L
.65873 .22916 L
.67855 .21692 L
.6984 .20465 L
.71824 .1924 L
.7381 .18015 L
.75794 .16799 L
.77779 .15586 L
.79763 .14397 L
.81746 .13272 L
.8373 .12242 L
.85714 .11427 L
.87698 .10907 L
.89683 .1079 L
.91667 .1081 L
.93651 .11122 L
.95635 .11425 L
.97619 .11677 L
s
0 0 1 r
.02381 .57707 m
.04936 .57764 L
.06027 .57452 L
.0794 .56598 L
.10317 .55123 L
.12285 .53644 L
.14473 .51799 L
.16309 .50128 L
.18254 .48543 L
.20222 .46888 L
.22158 .45117 L
.24245 .43251 L
.2619 .41478 L
.28158 .39688 L
.30221 .37782 L
.32182 .3597 L
.34127 .3418 L
.36095 .32371 L
.38094 .30523 L
.40065 .28699 L
.42063 .26849 L
.44031 .25024 L
.46031 .23173 L
.48028 .21323 L
.5 .19496 L
.51986 .17657 L
.53967 .15825 L
.55951 .13992 L
.57937 .12215 L
.59923 .10524 L
.61904 .09218 L
.63888 .08328 L
.65873 .08324 L
.67855 .08435 L
.6984 .09002 L
.71824 .08872 L
.7381 .09409 L
.75794 .09755 L
.77779 .10294 L
.79763 .10427 L
.81746 .10665 L
.8373 .10986 L
.85714 .113 L
.87698 .11722 L
.89683 .11997 L
.91667 .12254 L
.93651 .12528 L
.95635 .13013 L
.97619 .13225 L
s
1 0 1 r
.02381 .57009 m
.04936 .57034 L
.06027 .56639 L
.0794 .55634 L
.10317 .53851 L
.12285 .52034 L
.14473 .49739 L
.16309 .47642 L
.18254 .45319 L
.20222 .43227 L
.22158 .40968 L
.24245 .38388 L
.2619 .3612 L
.28158 .33684 L
.30221 .31202 L
.32182 .288 L
.34127 .26409 L
.36095 .23987 L
.38094 .21521 L
.40065 .19092 L
.42063 .16626 L
.44031 .14204 L
.46031 .11768 L
.48028 .09485 L
.5 .07772 L
.51986 .07338 L
.53967 .07081 L
.55951 .07338 L
.57937 .0795 L
.59923 .08421 L
.61904 .08295 L
.63888 .08756 L
.65873 .09135 L
.67855 .08805 L
.6984 .09596 L
.71824 .09967 L
.7381 .10334 L
.75794 .10772 L
.77779 .11637 L
.79763 .11855 L
.81746 .11566 L
.8373 .11984 L
.85714 .11804 L
.87698 .12463 L
.89683 .12886 L
.91667 .13259 L
.93651 .13021 L
.95635 .13722 L
.97619 .14108 L
s
0 0 0 r
.02381 .57368 m
.04936 .54539 L
.06027 .52869 L
.0794 .49069 L
.10317 .42161 L
.12285 .3381 L
.14473 .20469 L
.16309 .04716 L
.18254 .01544 L
.20222 .01638 L
.22158 .01472 L
.24245 .02126 L
.2619 .0297 L
.28158 .0326 L
.30221 .03898 L
.32182 .03758 L
.34127 .04996 L
.36095 .04604 L
.38094 .0694 L
.40065 .04065 L
.42063 .06475 L
.44031 .05831 L
.46031 .07947 L
.48028 .08711 L
.5 .08394 L
.51986 .06495 L
.53967 .09455 L
.55951 .08223 L
.57937 .09303 L
.59923 .08236 L
.61904 .11172 L
.63888 .09676 L
.65873 .10484 L
.67855 .12688 L
.6984 .12621 L
.71824 .11358 L
.7381 .1158 L
.75794 .13582 L
.77779 .13743 L
.79763 .14667 L
.81746 .13272 L
.8373 .1545 L
.85714 .15453 L
.87698 .15286 L
.89683 .14739 L
.91667 .15887 L
.93651 .16789 L
.95635 .16761 L
.97619 .16052 L
s
.5 .5 .5 r
.02381 .60332 m
.04936 .59227 L
.06027 .58672 L
.0794 .57234 L
.10317 .54806 L
.12285 .519 L
.14473 .47224 L
.16309 .41707 L
.18254 .33739 L
.20222 .22759 L
.22158 .07825 L
.24245 .04551 L
.2619 .10182 L
.28158 .09222 L
.30221 .08655 L
.32182 .08231 L
.34127 .12348 L
.36095 .12002 L
.38094 .14031 L
.40065 .13902 L
.42063 .14632 L
.44031 .13873 L
.46031 .177 L
.48028 .17383 L
.5 .16317 L
.51986 .15996 L
.53967 .16018 L
.55951 .17994 L
.57937 .20926 L
.59923 .16285 L
.61904 .19572 L
.63888 .20974 L
.65873 .20302 L
.67855 .23237 L
.6984 .23536 L
.71824 .23738 L
.7381 .24347 L
.75794 .2388 L
.77779 .24013 L
.79763 .27487 L
.81746 .26774 L
.8373 .27885 L
.85714 .30323 L
.87698 .2847 L
.89683 .27299 L
.91667 .29928 L
.93651 .28457 L
.95635 .2992 L
.97619 .30105 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{288, 177.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->1768407590,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {0.222812, -15.9116, \
0.0163206, 0.105637}}],

Cell[TextData[{
 "A logarithmic plot of the maximum error for approximating the first \
derivative of the Gaussian ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", "x", ")"}], " ", "=", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"15", " ", 
         RowBox[{"(", 
          RowBox[{"x", " ", "-", " ", 
           RowBox[{"1", "/", "2"}]}], ")"}]}], ")"}], "2"]}]]}], 
   TraditionalForm]]],
 "at points on a grid covering the interval [0,1] as a function of the number \
of grid points, n, using machine precision.",
 " ",
 "Finite differences of order 2,4,6, and 8 on a uniform grid are shown in \
red, green, blue, and magenta respectively. Pseudospectral derivatives with \
uniform (periodic) and Chebyshev spacing are shown in black and grey \
respectively. "
}], "Caption",
 CellID->531698254],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.293651 0.263645 0.570493 0.0237705 [
[.23364 .40435 -11 -9.8125 ]
[.23364 .40435 11 0 ]
[.49728 .40435 -14 -9.8125 ]
[.49728 .40435 14 0 ]
[.76093 .40435 -17 -9.8125 ]
[.76093 .40435 17 0 ]
[-0.0125 0 -51.6875 -6.15625 ]
[-0.0125 0 0 6.15625 ]
[-0.0125 .11885 -51.6875 -6.15625 ]
[-0.0125 .11885 0 6.15625 ]
[-0.0125 .23771 -51.6875 -6.15625 ]
[-0.0125 .23771 0 6.15625 ]
[-0.0125 .35656 -47.4375 -6.15625 ]
[-0.0125 .35656 0 6.15625 ]
[-0.0125 .47541 -40 -4.90625 ]
[-0.0125 .47541 0 4.90625 ]
[-0.0125 .59426 -16 -4.90625 ]
[-0.0125 .59426 0 4.90625 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.23364 .41685 m
.23364 .4231 L
s
gsave
.23364 .40435 -72 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(100) show
81.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.49728 .41685 m
.49728 .4231 L
s
gsave
.49728 .40435 -75 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1000) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.76093 .41685 m
.76093 .4231 L
s
gsave
.76093 .40435 -78 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10000) show
93.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.04936 .41685 m
.04936 .4206 L
s
.09578 .41685 m
.09578 .4206 L
s
.12872 .41685 m
.12872 .4206 L
s
.15427 .41685 m
.15427 .4206 L
s
.17515 .41685 m
.17515 .4206 L
s
.1928 .41685 m
.1928 .4206 L
s
.20809 .41685 m
.20809 .4206 L
s
.22158 .41685 m
.22158 .4206 L
s
.313 .41685 m
.313 .4206 L
s
.35943 .41685 m
.35943 .4206 L
s
.39237 .41685 m
.39237 .4206 L
s
.41792 .41685 m
.41792 .4206 L
s
.4388 .41685 m
.4388 .4206 L
s
.45645 .41685 m
.45645 .4206 L
s
.47173 .41685 m
.47173 .4206 L
s
.48522 .41685 m
.48522 .4206 L
s
.57665 .41685 m
.57665 .4206 L
s
.62308 .41685 m
.62308 .4206 L
s
.65601 .41685 m
.65601 .4206 L
s
.68156 .41685 m
.68156 .4206 L
s
.70244 .41685 m
.70244 .4206 L
s
.72009 .41685 m
.72009 .4206 L
s
.73538 .41685 m
.73538 .4206 L
s
.74887 .41685 m
.74887 .4206 L
s
.84029 .41685 m
.84029 .4206 L
s
.88672 .41685 m
.88672 .4206 L
s
.91966 .41685 m
.91966 .4206 L
s
.94521 .41685 m
.94521 .4206 L
s
.96609 .41685 m
.96609 .4206 L
s
.98374 .41685 m
.98374 .4206 L
s
.99902 .41685 m
.99902 .4206 L
s
.25 Mabswid
0 .41685 m
1 .41685 L
s
0 0 m
.00625 0 L
s
gsave
-0.0125 0 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(24) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .11885 m
.00625 .11885 L
s
gsave
-0.0125 .11885 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(19) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .23771 m
.00625 .23771 L
s
gsave
-0.0125 .23771 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(14) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .35656 m
.00625 .35656 L
s
gsave
-0.0125 .35656 -108.438 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(9) show
105.125 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .47541 m
.00625 .47541 L
s
gsave
-0.0125 .47541 -101 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.0001) show
99.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .59426 m
.00625 .59426 L
s
gsave
-0.0125 .59426 -77 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
75.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .09617 m
.00375 .09617 L
s
0 .10333 m
.00375 .10333 L
s
0 .10751 m
.00375 .10751 L
s
0 .11048 m
.00375 .11048 L
s
0 .11278 m
.00375 .11278 L
s
0 .11467 m
.00375 .11467 L
s
0 .11626 m
.00375 .11626 L
s
0 .11764 m
.00375 .11764 L
s
0 .21502 m
.00375 .21502 L
s
0 .22218 m
.00375 .22218 L
s
0 .22636 m
.00375 .22636 L
s
0 .22933 m
.00375 .22933 L
s
0 .23164 m
.00375 .23164 L
s
0 .23352 m
.00375 .23352 L
s
0 .23511 m
.00375 .23511 L
s
0 .23649 m
.00375 .23649 L
s
0 .33388 m
.00375 .33388 L
s
0 .34103 m
.00375 .34103 L
s
0 .34522 m
.00375 .34522 L
s
0 .34819 m
.00375 .34819 L
s
0 .35049 m
.00375 .35049 L
s
0 .35237 m
.00375 .35237 L
s
0 .35396 m
.00375 .35396 L
s
0 .35534 m
.00375 .35534 L
s
0 .45273 m
.00375 .45273 L
s
0 .45988 m
.00375 .45988 L
s
0 .46407 m
.00375 .46407 L
s
0 .46704 m
.00375 .46704 L
s
0 .46934 m
.00375 .46934 L
s
0 .47123 m
.00375 .47123 L
s
0 .47282 m
.00375 .47282 L
s
0 .47419 m
.00375 .47419 L
s
0 .57158 m
.00375 .57158 L
s
0 .57874 m
.00375 .57874 L
s
0 .58292 m
.00375 .58292 L
s
0 .58589 m
.00375 .58589 L
s
0 .5882 m
.00375 .5882 L
s
0 .59008 m
.00375 .59008 L
s
0 .59167 m
.00375 .59167 L
s
0 .59305 m
.00375 .59305 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.015 w
.02381 .58459 Mdot
.04936 .58447 Mdot
.06027 .58361 Mdot
.0794 .58127 Mdot
.10317 .57729 Mdot
.12285 .57337 Mdot
.14473 .56985 Mdot
.16309 .56722 Mdot
.18254 .56369 Mdot
.20222 .56018 Mdot
.22158 .55687 Mdot
.24245 .55314 Mdot
.2619 .54955 Mdot
.28158 .54611 Mdot
.30221 .54241 Mdot
.32182 .53888 Mdot
.34127 .53542 Mdot
.36095 .53186 Mdot
.38094 .52827 Mdot
.40065 .52472 Mdot
.42063 .52112 Mdot
.44031 .51757 Mdot
.46031 .51397 Mdot
.48028 .51037 Mdot
.5 .50681 Mdot
.51986 .50323 Mdot
.53967 .49966 Mdot
.55951 .49608 Mdot
.57937 .4925 Mdot
.59923 .48892 Mdot
.61904 .48535 Mdot
.63888 .48177 Mdot
.65873 .47819 Mdot
.67855 .47461 Mdot
.6984 .47104 Mdot
.71824 .46746 Mdot
.7381 .46388 Mdot
.75794 .4603 Mdot
.77779 .45672 Mdot
.79763 .45314 Mdot
.81746 .44957 Mdot
.8373 .44599 Mdot
.85714 .44241 Mdot
.87698 .43883 Mdot
.89683 .43525 Mdot
.91667 .43168 Mdot
.93651 .4281 Mdot
.95635 .42452 Mdot
.97619 .42094 Mdot
0 1 0 r
.02381 .57702 Mdot
.04936 .57674 Mdot
.06027 .57529 Mdot
.0794 .57145 Mdot
.10317 .56496 Mdot
.12285 .55854 Mdot
.14473 .55062 Mdot
.16309 .5446 Mdot
.18254 .53859 Mdot
.20222 .53164 Mdot
.22158 .52472 Mdot
.24245 .51751 Mdot
.2619 .51055 Mdot
.28158 .50346 Mdot
.30221 .49614 Mdot
.32182 .48911 Mdot
.34127 .48209 Mdot
.36095 .47505 Mdot
.38094 .46783 Mdot
.40065 .46074 Mdot
.42063 .45355 Mdot
.44031 .44646 Mdot
.46031 .43925 Mdot
.48028 .43205 Mdot
.5 .42494 Mdot
.51986 .41777 Mdot
.53967 .41063 Mdot
.55951 .40348 Mdot
.57937 .39632 Mdot
.59923 .38916 Mdot
.61904 .38201 Mdot
.63888 .37486 Mdot
.65873 .3677 Mdot
.67855 .36055 Mdot
.6984 .35339 Mdot
.71824 .34623 Mdot
.7381 .33907 Mdot
.75794 .33192 Mdot
.77779 .32476 Mdot
.79763 .31761 Mdot
.81746 .31045 Mdot
.8373 .3033 Mdot
.85714 .29614 Mdot
.87698 .28899 Mdot
.89683 .28183 Mdot
.91667 .27467 Mdot
.93651 .26752 Mdot
.95635 .26036 Mdot
.97619 .25321 Mdot
0 0 1 r
.02381 .57073 Mdot
.04936 .57106 Mdot
.06027 .56924 Mdot
.0794 .56425 Mdot
.10317 .55565 Mdot
.12285 .54702 Mdot
.14473 .53625 Mdot
.16309 .5265 Mdot
.18254 .51725 Mdot
.20222 .50759 Mdot
.22158 .49725 Mdot
.24245 .48636 Mdot
.2619 .47602 Mdot
.28158 .46557 Mdot
.30221 .45445 Mdot
.32182 .44388 Mdot
.34127 .43343 Mdot
.36095 .42287 Mdot
.38094 .41209 Mdot
.40065 .40145 Mdot
.42063 .39065 Mdot
.44031 .38 Mdot
.46031 .36919 Mdot
.48028 .3584 Mdot
.5 .34773 Mdot
.51986 .33699 Mdot
.53967 .32628 Mdot
.55951 .31554 Mdot
.57937 .3048 Mdot
.59923 .29406 Mdot
.61904 .28334 Mdot
.63888 .27261 Mdot
.65873 .26187 Mdot
.67855 .25115 Mdot
.6984 .24041 Mdot
.71824 .22968 Mdot
.7381 .21894 Mdot
.75794 .2082 Mdot
.77779 .19747 Mdot
.79763 .18673 Mdot
.81746 .176 Mdot
.8373 .16527 Mdot
.85714 .15454 Mdot
.87698 .1438 Mdot
.89683 .13307 Mdot
.91667 .12234 Mdot
.93651 .1116 Mdot
.95635 .10087 Mdot
.97619 .09014 Mdot
1 0 1 r
.02381 .56665 Mdot
.04936 .5668 Mdot
.06027 .56449 Mdot
.0794 .55863 Mdot
.10317 .54823 Mdot
.12285 .53762 Mdot
.14473 .52423 Mdot
.16309 .51199 Mdot
.18254 .49843 Mdot
.20222 .48622 Mdot
.22158 .47304 Mdot
.24245 .45798 Mdot
.2619 .44475 Mdot
.28158 .43054 Mdot
.30221 .41605 Mdot
.32182 .40204 Mdot
.34127 .38808 Mdot
.36095 .37395 Mdot
.38094 .35955 Mdot
.40065 .34537 Mdot
.42063 .33097 Mdot
.44031 .3168 Mdot
.46031 .30238 Mdot
.48028 .28799 Mdot
.5 .27377 Mdot
.51986 .25945 Mdot
.53967 .24517 Mdot
.55951 .23085 Mdot
.57937 .21654 Mdot
.59923 .20221 Mdot
.61904 .18792 Mdot
.63888 .17361 Mdot
.65873 .15929 Mdot
.67855 .145 Mdot
.6984 .13068 Mdot
.71824 .11637 Mdot
.7381 .10205 Mdot
.75794 .08774 Mdot
.77779 .07342 Mdot
.79763 .05911 Mdot
.81746 .04571 Mdot
.8373 .04571 Mdot
.85714 .04571 Mdot
.87698 .04571 Mdot
.89683 .04571 Mdot
.91667 .04571 Mdot
.93651 .04571 Mdot
.95635 .04571 Mdot
.97619 .04571 Mdot
0 0 0 r
.02381 .56875 Mdot
.04936 .55224 Mdot
.06027 .54249 Mdot
.0794 .52032 Mdot
.10317 .48001 Mdot
.12285 .43127 Mdot
.14473 .35341 Mdot
.16309 .26081 Mdot
.18254 .1244 Mdot
.20222 .04571 Mdot
.22158 .04571 Mdot
.24245 .04571 Mdot
.2619 .04571 Mdot
.28158 .04571 Mdot
.30221 .04571 Mdot
.32182 .04571 Mdot
.34127 .04571 Mdot
.36095 .04571 Mdot
.38094 .04571 Mdot
.40065 .04571 Mdot
.42063 .04571 Mdot
.44031 .04571 Mdot
.46031 .04571 Mdot
.48028 .04571 Mdot
.5 .04571 Mdot
.51986 .04571 Mdot
.53967 .04571 Mdot
.55951 .04571 Mdot
.57937 .04571 Mdot
.59923 .04571 Mdot
.61904 .04571 Mdot
.63888 .04571 Mdot
.65873 .04571 Mdot
.67855 .04571 Mdot
.6984 .04571 Mdot
.71824 .04571 Mdot
.7381 .04571 Mdot
.75794 .04571 Mdot
.77779 .04571 Mdot
.79763 .04571 Mdot
.81746 .04571 Mdot
.8373 .04571 Mdot
.85714 .04571 Mdot
.87698 .04571 Mdot
.89683 .04571 Mdot
.91667 .04571 Mdot
.93651 .04571 Mdot
.95635 .04571 Mdot
.97619 .04571 Mdot
.5 .5 .5 r
.02381 .58605 Mdot
.04936 .5775 Mdot
.06027 .57636 Mdot
.0794 .56796 Mdot
.10317 .5538 Mdot
.12285 .53684 Mdot
.14473 .50955 Mdot
.16309 .47736 Mdot
.18254 .43086 Mdot
.20222 .36678 Mdot
.22158 .28185 Mdot
.24245 .1585 Mdot
.2619 .00517 Mdot
1 0 0 r
.5 Mabswid
.04936 .12684 m
.06027 .14115 L
s
.06027 .14115 m
.07914 0 L
s
.07999 0 m
.10317 .07675 L
.12285 .21271 L
.14473 .1984 L
.16309 .21987 L
.18254 .22702 L
.20222 .21987 L
.22158 .21987 L
.24245 .23418 L
.2619 .23418 L
.28158 .23121 L
.30221 .2328 L
.32182 .23836 L
.34127 .23995 L
.36095 .23836 L
.38094 .23995 L
.40065 .24635 L
.42063 .24782 L
.44031 .2497 L
.46031 .2497 L
.48028 .2534 L
.5 .25498 L
.51986 .25564 L
.53967 .25795 L
.55951 .26025 L
.57937 .26256 L
.59923 .26327 L
.61904 .2662 L
.63888 .26741 L
.65873 .26971 L
.67855 .27145 L
.6984 .27264 L
.71824 .27446 L
.7381 .27662 L
.75794 .27854 L
.77779 .28049 L
.79763 .28207 L
.81746 .28356 L
.8373 .28578 L
.85714 .28764 L
.87698 .28953 L
.89683 .29108 L
.91667 .293 L
.93651 .2946 L
.95635 .2967 L
.97619 .29841 L
s
0 1 0 r
.02381 .21987 m
.04936 .21271 L
.06027 .21271 L
.0794 .21987 L
.10317 .21987 L
.12285 .21987 L
.14473 .21987 L
.16309 .21987 L
.18254 .22702 L
.20222 .21987 L
.22158 .22702 L
.24245 .23418 L
.2619 .23648 L
.28158 .23539 L
.30221 .23836 L
.32182 .24133 L
.34127 .24364 L
.36095 .24255 L
.38094 .24364 L
.40065 .24849 L
.42063 .25026 L
.44031 .25267 L
.46031 .25289 L
.48028 .25619 L
.5 .25869 L
.51986 .25916 L
.53967 .26051 L
.55951 .26343 L
.57937 .26591 L
.59923 .26688 L
.61904 .2692 L
.63888 .27084 L
.65873 .27279 L
.67855 .27477 L
.6984 .27627 L
.71824 .27766 L
.7381 .28004 L
.75794 .2823 L
.77779 .28419 L
.79763 .28462 L
.81746 .28689 L
.8373 .28943 L
.85714 .29127 L
.87698 .29283 L
.89683 .2947 L
.91667 .29672 L
.93651 .29849 L
.95635 .30046 L
.97619 .30214 L
s
0 0 1 r
.02381 .2158 m
.04936 .21987 L
.06027 .21987 L
.0794 .2213 L
.10317 .22702 L
.12285 .22702 L
.14473 .23121 L
.16309 .23418 L
.18254 .23836 L
.20222 .23539 L
.22158 .23836 L
.24245 .24364 L
.2619 .24552 L
.28158 .24635 L
.30221 .24849 L
.32182 .25026 L
.34127 .25224 L
.36095 .25389 L
.38094 .25528 L
.40065 .25808 L
.42063 .26015 L
.44031 .26114 L
.46031 .26387 L
.48028 .26615 L
.5 .26823 L
.51986 .2682 L
.53967 .27159 L
.55951 .27043 L
.57937 .27526 L
.59923 .27619 L
.61904 .27892 L
.63888 .27889 L
.65873 .2822 L
.67855 .283 L
.6984 .28653 L
.71824 .28576 L
.7381 .28887 L
.75794 .29089 L
.77779 .29404 L
.79763 .29481 L
.81746 .2962 L
.8373 .29807 L
.85714 .29991 L
.87698 .30237 L
.89683 .30397 L
.91667 .30547 L
.93651 .30707 L
.95635 .3099 L
.97619 .31114 L
s
1 0 1 r
.02381 .22865 m
.04936 .22857 L
.06027 .23121 L
.0794 .23176 L
.10317 .2344 L
.12285 .23836 L
.14473 .24191 L
.16309 .24133 L
.18254 .24133 L
.20222 .24364 L
.22158 .24133 L
.24245 .24911 L
.2619 .24999 L
.28158 .25246 L
.30221 .25154 L
.32182 .25564 L
.34127 .25755 L
.36095 .26046 L
.38094 .25928 L
.40065 .26268 L
.42063 .26542 L
.44031 .26704 L
.46031 .2672 L
.48028 .27041 L
.5 .27342 L
.51986 .27519 L
.53967 .27505 L
.55951 .27689 L
.57937 .28037 L
.59923 .2831 L
.61904 .28237 L
.63888 .28506 L
.65873 .28727 L
.67855 .28535 L
.6984 .28996 L
.71824 .29213 L
.7381 .29427 L
.75794 .29683 L
.77779 .30187 L
.79763 .30315 L
.81746 .30146 L
.8373 .3039 L
.85714 .30285 L
.87698 .3067 L
.89683 .30916 L
.91667 .31134 L
.93651 .30995 L
.95635 .31404 L
.97619 .31629 L
s
0 0 0 r
.02381 .21987 m
.04936 .22217 L
.06027 .22339 L
.0794 .23203 L
.10317 .22746 L
.12285 .23869 L
.14473 .23924 L
.16309 .23772 L
.18254 .24297 L
.20222 .24352 L
.22158 .24255 L
.24245 .24637 L
.2619 .2513 L
.28158 .25299 L
.30221 .25671 L
.32182 .25589 L
.34127 .26312 L
.36095 .26083 L
.38094 .27446 L
.40065 .25769 L
.42063 .27175 L
.44031 .26799 L
.46031 .28034 L
.48028 .2848 L
.5 .28295 L
.51986 .27186 L
.53967 .28914 L
.55951 .28195 L
.57937 .28825 L
.59923 .28203 L
.61904 .29916 L
.63888 .29043 L
.65873 .29514 L
.67855 .308 L
.6984 .30762 L
.71824 .30024 L
.7381 .30154 L
.75794 .31322 L
.77779 .31416 L
.79763 .31956 L
.81746 .31142 L
.8373 .32412 L
.85714 .32414 L
.87698 .32317 L
.89683 .31998 L
.91667 .32668 L
.93651 .33194 L
.95635 .33178 L
.97619 .32764 L
s
.5 .5 .5 r
.02381 .23747 m
.04936 .24149 L
.06027 .23351 L
.0794 .24084 L
.10317 .24545 L
.12285 .25958 L
.14473 .26309 L
.16309 .26458 L
.18254 .26187 L
.20222 .26208 L
.22158 .2649 L
.24245 .26052 L
.2619 .29338 L
.28158 .28778 L
.30221 .28447 L
.32182 .28199 L
.34127 .30602 L
.36095 .304 L
.38094 .31584 L
.40065 .31509 L
.42063 .31935 L
.44031 .31492 L
.46031 .33725 L
.48028 .33541 L
.5 .32918 L
.51986 .32731 L
.53967 .32744 L
.55951 .33897 L
.57937 .35608 L
.59923 .329 L
.61904 .34818 L
.63888 .35636 L
.65873 .35244 L
.67855 .36957 L
.6984 .37131 L
.71824 .37249 L
.7381 .37604 L
.75794 .37332 L
.77779 .37409 L
.79763 .39437 L
.81746 .39021 L
.8373 .39669 L
.85714 .41092 L
.87698 .40011 L
.89683 .39327 L
.91667 .40862 L
.93651 .40003 L
.95635 .40857 L
.97619 .40965 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{288, 177.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->1041326062,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {0.222812, -27.5715, \
0.0163206, 0.181016}}],

Cell[TextData[{
 "A logarithmic plot of the truncation error (dotted) and the condition and \
roundoff error (solid line) for approximating the first derivative of the \
Gaussian ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", "x", ")"}], " ", "=", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"15", " ", 
         RowBox[{"(", 
          RowBox[{"x", " ", "-", " ", 
           RowBox[{"1", "/", "2"}]}], ")"}]}], ")"}], "2"]}]]}], 
   TraditionalForm]]],
 "at points on a grid covering the interval [0,1] as a function of the number \
of grid points, n. Finite differences of order 2, 4, 6, and 8 on a uniform \
grid are shown in red, green, blue, and magenta, respectively. Pseudospectral \
derivatives with uniform (periodic) and Chebyshev spacing are shown in black \
and grey, respectively. The truncation error was computed by computing the \
approximations with very high precision. The roundoff and condition error was \
estimated by subtracting the machine precision approximation from the high \
precision approximation. The roundoff and condition error tends to increase \
linearly (because of the 1/h factor common to finite difference formulas for \
the first derivative) and tends to be a little bit higher for higher order \
derivatives. The pseudospectral derivatives show more variations because the \
error of the FFT computations vary with length. Note that the truncation \
error for the uniform (periodic) pseudospectral derivative does not decrease \
below about ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "22"}]], TraditionalForm]]],
 ".",
 " ",
 "This is because, mathematically, the Gaussian is not a periodic function; \
this error in essence gives the deviation from periodicity."
}], "Caption",
 CellID->614125773],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.602776 0.0375672 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21429 -0.0125 -9 -9 ]
[.21429 -0.0125 9 0 ]
[.40476 -0.0125 -9 -9 ]
[.40476 -0.0125 9 0 ]
[.59524 -0.0125 -9 -9 ]
[.59524 -0.0125 9 0 ]
[.78571 -0.0125 -9 -9 ]
[.78571 -0.0125 9 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[-0.0125 .0017 -51.6875 -6.15625 ]
[-0.0125 .0017 0 6.15625 ]
[-0.0125 .1144 -51.6875 -6.15625 ]
[-0.0125 .1144 0 6.15625 ]
[-0.0125 .2271 -51.6875 -6.15625 ]
[-0.0125 .2271 0 6.15625 ]
[-0.0125 .33981 -47.4375 -6.125 ]
[-0.0125 .33981 0 6.125 ]
[-0.0125 .45251 -40 -4.90625 ]
[-0.0125 .45251 0 4.90625 ]
[-0.0125 .56521 -22 -4.90625 ]
[-0.0125 .56521 0 4.90625 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(0.2)] .21429 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(0.4)] .40476 -0.0125 0 1 Mshowa
.59524 0 m
.59524 .00625 L
s
[(0.6)] .59524 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(0.8)] .78571 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(1)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.2619 0 m
.2619 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.5 0 m
.5 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.7381 0 m
.7381 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .0017 m
.00625 .0017 L
s
gsave
-0.0125 .0017 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(16) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .1144 m
.00625 .1144 L
s
gsave
-0.0125 .1144 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(13) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .2271 m
.00625 .2271 L
s
gsave
-0.0125 .2271 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .33981 m
.00625 .33981 L
s
gsave
-0.0125 .33981 -108.438 -10.125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.25 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.688 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.688 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.688 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.688 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.688 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.312 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.312 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.312 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(7) show
105.125 9.312 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .45251 m
.00625 .45251 L
s
gsave
-0.0125 .45251 -101 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.0001) show
99.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .56521 m
.00625 .56521 L
s
gsave
-0.0125 .56521 -83 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.1) show
81.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .07868 m
.00375 .07868 L
s
0 .08992 m
.00375 .08992 L
s
0 .09651 m
.00375 .09651 L
s
0 .10119 m
.00375 .10119 L
s
0 .10483 m
.00375 .10483 L
s
0 .10779 m
.00375 .10779 L
s
0 .11031 m
.00375 .11031 L
s
0 .11248 m
.00375 .11248 L
s
0 .19139 m
.00375 .19139 L
s
0 .20262 m
.00375 .20262 L
s
0 .20921 m
.00375 .20921 L
s
0 .21389 m
.00375 .21389 L
s
0 .21753 m
.00375 .21753 L
s
0 .2205 m
.00375 .2205 L
s
0 .22301 m
.00375 .22301 L
s
0 .22518 m
.00375 .22518 L
s
0 .30409 m
.00375 .30409 L
s
0 .31532 m
.00375 .31532 L
s
0 .32191 m
.00375 .32191 L
s
0 .3266 m
.00375 .3266 L
s
0 .33023 m
.00375 .33023 L
s
0 .3332 m
.00375 .3332 L
s
0 .33571 m
.00375 .33571 L
s
0 .33789 m
.00375 .33789 L
s
0 .41679 m
.00375 .41679 L
s
0 .42802 m
.00375 .42802 L
s
0 .43462 m
.00375 .43462 L
s
0 .4393 m
.00375 .4393 L
s
0 .44293 m
.00375 .44293 L
s
0 .4459 m
.00375 .4459 L
s
0 .44841 m
.00375 .44841 L
s
0 .45059 m
.00375 .45059 L
s
0 .52949 m
.00375 .52949 L
s
0 .54073 m
.00375 .54073 L
s
0 .54732 m
.00375 .54732 L
s
0 .552 m
.00375 .552 L
s
0 .55563 m
.00375 .55563 L
s
0 .5586 m
.00375 .5586 L
s
0 .56111 m
.00375 .56111 L
s
0 .56329 m
.00375 .56329 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
1 0 0 r
.015 w
.02381 .01472 Mdot
.04497 .01472 Mdot
.06614 .01472 Mdot
.0873 .03092 Mdot
.10847 .09573 Mdot
.12963 .15683 Mdot
.15079 .2142 Mdot
.17196 .26781 Mdot
.19312 .31763 Mdot
.21429 .36363 Mdot
.23545 .40577 Mdot
.25661 .44397 Mdot
.27778 .47816 Mdot
.29894 .50821 Mdot
.32011 .53394 Mdot
.34127 .55505 Mdot
.36243 .57102 Mdot
.3836 .58063 Mdot
.40476 .57982 Mdot
.42593 .56665 Mdot
.44709 .59762 Mdot
.46825 .60332 Mdot
.48942 .59155 Mdot
.51058 .59155 Mdot
.53175 .60332 Mdot
.55291 .59762 Mdot
.57407 .56665 Mdot
.59524 .57982 Mdot
.6164 .58063 Mdot
.63757 .57102 Mdot
.65873 .55505 Mdot
.67989 .53394 Mdot
.70106 .50821 Mdot
.72222 .47816 Mdot
.74339 .44397 Mdot
.76455 .40577 Mdot
.78571 .36363 Mdot
.80688 .31763 Mdot
.82804 .26781 Mdot
.84921 .2142 Mdot
.87037 .15683 Mdot
.89153 .09573 Mdot
.9127 .03092 Mdot
.93386 .01472 Mdot
.95503 .01472 Mdot
.97619 .01472 Mdot
0 1 0 r
.02381 .01472 Mdot
.04497 .01472 Mdot
.06614 .01472 Mdot
.0873 .06685 Mdot
.10847 .12792 Mdot
.12963 .18522 Mdot
.15079 .23871 Mdot
.17196 .28836 Mdot
.19312 .3341 Mdot
.21429 .37584 Mdot
.23545 .41348 Mdot
.25661 .44684 Mdot
.27778 .47567 Mdot
.29894 .49955 Mdot
.32011 .51766 Mdot
.34127 .52787 Mdot
.36243 .51796 Mdot
.3836 .54266 Mdot
.40476 .55919 Mdot
.42593 .55547 Mdot
.44709 .55769 Mdot
.46825 .57389 Mdot
.48942 .56469 Mdot
.51058 .56469 Mdot
.53175 .57389 Mdot
.55291 .55769 Mdot
.57407 .55547 Mdot
.59524 .55919 Mdot
.6164 .54266 Mdot
.63757 .51796 Mdot
.65873 .52787 Mdot
.67989 .51766 Mdot
.70106 .49955 Mdot
.72222 .47567 Mdot
.74339 .44684 Mdot
.76455 .41348 Mdot
.78571 .37584 Mdot
.80688 .3341 Mdot
.82804 .28836 Mdot
.84921 .23871 Mdot
.87037 .18522 Mdot
.89153 .12792 Mdot
.9127 .06685 Mdot
.93386 .01472 Mdot
.95503 .01472 Mdot
.97619 .01472 Mdot
0 0 1 r
.02381 .01472 Mdot
.04497 .01472 Mdot
.06614 .04019 Mdot
.0873 .10114 Mdot
.10847 .15828 Mdot
.12963 .21157 Mdot
.15079 .26093 Mdot
.17196 .30629 Mdot
.19312 .34751 Mdot
.21429 .38442 Mdot
.23545 .41675 Mdot
.25661 .44402 Mdot
.27778 .46534 Mdot
.29894 .47834 Mdot
.32011 .46738 Mdot
.34127 .50121 Mdot
.36243 .51824 Mdot
.3836 .50803 Mdot
.40476 .53124 Mdot
.42593 .54097 Mdot
.44709 .5026 Mdot
.46825 .54994 Mdot
.48942 .54335 Mdot
.51058 .54335 Mdot
.53175 .54994 Mdot
.55291 .5026 Mdot
.57407 .54097 Mdot
.59524 .53124 Mdot
.6164 .50803 Mdot
.63757 .51824 Mdot
.65873 .50121 Mdot
.67989 .46738 Mdot
.70106 .47834 Mdot
.72222 .46534 Mdot
.74339 .44402 Mdot
.76455 .41675 Mdot
.78571 .38442 Mdot
.80688 .34751 Mdot
.82804 .30629 Mdot
.84921 .26093 Mdot
.87037 .21157 Mdot
.89153 .15828 Mdot
.9127 .10114 Mdot
.93386 .04019 Mdot
.95503 .01472 Mdot
.97619 .01472 Mdot
1 0 1 r
.02381 .01472 Mdot
.04497 .01472 Mdot
.06614 .07532 Mdot
.0873 .13228 Mdot
.10847 .18532 Mdot
.12963 .23436 Mdot
.15079 .27928 Mdot
.17196 .31991 Mdot
.19312 .35598 Mdot
.21429 .38706 Mdot
.23545 .41233 Mdot
.25661 .42979 Mdot
.27778 .42955 Mdot
.29894 .45418 Mdot
.32011 .4768 Mdot
.34127 .47171 Mdot
.36243 .49539 Mdot
.3836 .5074 Mdot
.40476 .49232 Mdot
.42593 .52368 Mdot
.44709 .50691 Mdot
.46825 .52956 Mdot
.48942 .52567 Mdot
.51058 .52567 Mdot
.53175 .52956 Mdot
.55291 .50691 Mdot
.57407 .52368 Mdot
.59524 .49232 Mdot
.6164 .5074 Mdot
.63757 .49539 Mdot
.65873 .47171 Mdot
.67989 .4768 Mdot
.70106 .45418 Mdot
.72222 .42955 Mdot
.74339 .42979 Mdot
.76455 .41233 Mdot
.78571 .38706 Mdot
.80688 .35598 Mdot
.82804 .31991 Mdot
.84921 .27928 Mdot
.87037 .23436 Mdot
.89153 .18532 Mdot
.9127 .13228 Mdot
.93386 .07532 Mdot
.95503 .01472 Mdot
.97619 .01472 Mdot
0 0 0 r
.02381 .04123 Mdot
.04497 .24311 Mdot
.06614 .25447 Mdot
.0873 .26119 Mdot
.10847 .26602 Mdot
.12963 .26983 Mdot
.15079 .273 Mdot
.17196 .27575 Mdot
.19312 .27817 Mdot
.21429 .28036 Mdot
.23545 .28236 Mdot
.25661 .28419 Mdot
.27778 .28587 Mdot
.29894 .28742 Mdot
.32011 .28884 Mdot
.34127 .29013 Mdot
.36243 .29128 Mdot
.3836 .29229 Mdot
.40476 .29314 Mdot
.42593 .29384 Mdot
.44709 .29438 Mdot
.46825 .29473 Mdot
.48942 .29491 Mdot
.51058 .29491 Mdot
.53175 .29473 Mdot
.55291 .29438 Mdot
.57407 .29384 Mdot
.59524 .29314 Mdot
.6164 .29229 Mdot
.63757 .29128 Mdot
.65873 .29013 Mdot
.67989 .28884 Mdot
.70106 .28742 Mdot
.72222 .28587 Mdot
.74339 .28419 Mdot
.76455 .28236 Mdot
.78571 .28036 Mdot
.80688 .27817 Mdot
.82804 .27575 Mdot
.84921 .273 Mdot
.87037 .26983 Mdot
.89153 .26602 Mdot
.9127 .26119 Mdot
.93386 .25447 Mdot
.95503 .24311 Mdot
.97619 .04123 Mdot
.5 .5 .5 r
.02381 .49665 Mdot
.02497 .48541 Mdot
.02844 .48562 Mdot
.03422 .48597 Mdot
.04226 .48647 Mdot
.05253 .48712 Mdot
.06498 .48792 Mdot
.07955 .48888 Mdot
.09617 .49001 Mdot
.11475 .4913 Mdot
.13522 .49277 Mdot
.15746 .49442 Mdot
.18137 .49627 Mdot
.20683 .49831 Mdot
.23372 .50055 Mdot
.2619 .50297 Mdot
.29125 .50554 Mdot
.32162 .50821 Mdot
.35285 .51088 Mdot
.3848 .5134 Mdot
.41731 .51558 Mdot
.45022 .5172 Mdot
.48338 .51806 Mdot
.51662 .51806 Mdot
.54978 .5172 Mdot
.58269 .51558 Mdot
.6152 .5134 Mdot
.64715 .51088 Mdot
.67838 .50821 Mdot
.70875 .50554 Mdot
.7381 .50297 Mdot
.76628 .50055 Mdot
.79317 .49831 Mdot
.81863 .49627 Mdot
.84254 .49442 Mdot
.86478 .49277 Mdot
.88525 .4913 Mdot
.90383 .49001 Mdot
.92045 .48888 Mdot
.93502 .48792 Mdot
.94747 .48712 Mdot
.95774 .48647 Mdot
.96578 .48597 Mdot
.97156 .48562 Mdot
.97503 .48541 Mdot
.97619 .49665 Mdot
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{288, 177.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->1717570105,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.271653, -18.6343, \
0.00451797, 0.114537}}],

Cell[TextData[{
 "A semilogarithmic plot of the error for approximating the first derivative \
of the Gaussian ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", "x", ")"}], " ", "=", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"x", " ", "-", " ", 
         RowBox[{"1", "/", "2"}]}], ")"}], "2"]}]]}], TraditionalForm]]],
 "as a function of x at points on a 45 point grid covering the interval \
[0,1]. Finite differences of order 2, 4, 6, and 8 on a uniform grid are shown \
in red, green, blue, and magenta, respectively. Pseudospectral derivatives \
with uniform (periodic) and Chebyshev spacing are shown in black and grey, \
respectively. All but the pseudospectral derivative with Chebyshev spacing \
were computed using uniform spacing 1/45. It is apparent that the error for \
the pseudospectral derivatives is not so localized; not surprising since the \
approximation at any point is based on the values over the whole grid.",
 " ",
 "The error for the finite difference approximations are localized and the \
magnitude of the errors follows the size of the Gaussian (which is parabolic \
on a semilogarithmic plot)."
}], "Caption",
 CellID->1385513180],

Cell[TextData[{
 "From the second plot, it is apparent that there is a size for which the \
best possible derivative approximation is found; for larger h, the truncation \
error dominates, and for smaller h, the condition and roundoff error \
dominate. The optimal h tends to give better approximations for higher order \
differences. This is not typically an issue for spatial discretization of \
PDEs because computing to that level of accuracy would be prohibitively \
expensive. However, this error balance is a vitally important issue when \
using low order differences to approximate, for example, Jacobian matrices. \
To avoid extra function evaluations, first order forward differences are \
usually used, and the error balance is proportional to the square root of \
unit roundoff, so picking a good value of h is important [",
 ButtonBox["GMW81",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#16479"],
 "].",
 " "
}], "Text",
 CellID->1246726857],

Cell["\<\
The plots above showed the situation typical for smooth functions where there \
were no real boundary effects. If the parameter in the Gaussian is changed so \
the function is flatter, boundary effects begin to appear.\
\>", "Text",
 CellID->1714547340],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.697995 0.0436502 [
[.02381 .46707 -3 -9 ]
[.02381 .46707 3 0 ]
[.21429 .46707 -9 -9 ]
[.21429 .46707 9 0 ]
[.40476 .46707 -9 -9 ]
[.40476 .46707 9 0 ]
[.59524 .46707 -9 -9 ]
[.59524 .46707 9 0 ]
[.78571 .46707 -9 -9 ]
[.78571 .46707 9 0 ]
[.97619 .46707 -3 -9 ]
[.97619 .46707 3 0 ]
[-0.0125 .13054 -51.6875 -6.15625 ]
[-0.0125 .13054 0 6.15625 ]
[-0.0125 .26149 -51.6875 -6.15625 ]
[-0.0125 .26149 0 6.15625 ]
[-0.0125 .39244 -47.4375 -6.125 ]
[-0.0125 .39244 0 6.125 ]
[-0.0125 .52339 -40 -4.90625 ]
[-0.0125 .52339 0 4.90625 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 .47957 m
.02381 .48582 L
s
[(0)] .02381 .46707 0 1 Mshowa
.21429 .47957 m
.21429 .48582 L
s
[(0.2)] .21429 .46707 0 1 Mshowa
.40476 .47957 m
.40476 .48582 L
s
[(0.4)] .40476 .46707 0 1 Mshowa
.59524 .47957 m
.59524 .48582 L
s
[(0.6)] .59524 .46707 0 1 Mshowa
.78571 .47957 m
.78571 .48582 L
s
[(0.8)] .78571 .46707 0 1 Mshowa
.97619 .47957 m
.97619 .48582 L
s
[(1)] .97619 .46707 0 1 Mshowa
.125 Mabswid
.07143 .47957 m
.07143 .48332 L
s
.11905 .47957 m
.11905 .48332 L
s
.16667 .47957 m
.16667 .48332 L
s
.2619 .47957 m
.2619 .48332 L
s
.30952 .47957 m
.30952 .48332 L
s
.35714 .47957 m
.35714 .48332 L
s
.45238 .47957 m
.45238 .48332 L
s
.5 .47957 m
.5 .48332 L
s
.54762 .47957 m
.54762 .48332 L
s
.64286 .47957 m
.64286 .48332 L
s
.69048 .47957 m
.69048 .48332 L
s
.7381 .47957 m
.7381 .48332 L
s
.83333 .47957 m
.83333 .48332 L
s
.88095 .47957 m
.88095 .48332 L
s
.92857 .47957 m
.92857 .48332 L
s
.25 Mabswid
0 .47957 m
1 .47957 L
s
0 .13054 m
.00625 .13054 L
s
gsave
-0.0125 .13054 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(13) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .26149 m
.00625 .26149 L
s
gsave
-0.0125 .26149 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .39244 m
.00625 .39244 L
s
gsave
-0.0125 .39244 -108.438 -10.125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.25 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.688 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.688 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.688 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.688 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.688 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.312 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.312 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.312 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(7) show
105.125 9.312 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .52339 m
.00625 .52339 L
s
gsave
-0.0125 .52339 -101 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.0001) show
99.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .08904 m
.00375 .08904 L
s
0 .1021 m
.00375 .1021 L
s
0 .10975 m
.00375 .10975 L
s
0 .11519 m
.00375 .11519 L
s
0 .11942 m
.00375 .11942 L
s
0 .12287 m
.00375 .12287 L
s
0 .12578 m
.00375 .12578 L
s
0 .12831 m
.00375 .12831 L
s
0 .21999 m
.00375 .21999 L
s
0 .23305 m
.00375 .23305 L
s
0 .24071 m
.00375 .24071 L
s
0 .24614 m
.00375 .24614 L
s
0 .25037 m
.00375 .25037 L
s
0 .25382 m
.00375 .25382 L
s
0 .25674 m
.00375 .25674 L
s
0 .25926 m
.00375 .25926 L
s
0 .35094 m
.00375 .35094 L
s
0 .364 m
.00375 .364 L
s
0 .37166 m
.00375 .37166 L
s
0 .3771 m
.00375 .3771 L
s
0 .38132 m
.00375 .38132 L
s
0 .38477 m
.00375 .38477 L
s
0 .38769 m
.00375 .38769 L
s
0 .39021 m
.00375 .39021 L
s
0 .48189 m
.00375 .48189 L
s
0 .49495 m
.00375 .49495 L
s
0 .50261 m
.00375 .50261 L
s
0 .50805 m
.00375 .50805 L
s
0 .51227 m
.00375 .51227 L
s
0 .51572 m
.00375 .51572 L
s
0 .51864 m
.00375 .51864 L
s
0 .52116 m
.00375 .52116 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
1 0 0 r
.015 w
.02381 .55852 Mdot
.04497 .54535 Mdot
.06614 .54514 Mdot
.0873 .54486 Mdot
.10847 .54448 Mdot
.12963 .54402 Mdot
.15079 .54346 Mdot
.17196 .54279 Mdot
.19312 .54201 Mdot
.21429 .5411 Mdot
.23545 .54006 Mdot
.25661 .53886 Mdot
.27778 .53748 Mdot
.29894 .5359 Mdot
.32011 .53408 Mdot
.34127 .53196 Mdot
.36243 .52946 Mdot
.3836 .52649 Mdot
.40476 .52284 Mdot
.42593 .5182 Mdot
.44709 .51191 Mdot
.46825 .50229 Mdot
.48942 .4815 Mdot
.51058 .4815 Mdot
.53175 .50229 Mdot
.55291 .51191 Mdot
.57407 .5182 Mdot
.59524 .52284 Mdot
.6164 .52649 Mdot
.63757 .52946 Mdot
.65873 .53196 Mdot
.67989 .53408 Mdot
.70106 .5359 Mdot
.72222 .53748 Mdot
.74339 .53886 Mdot
.76455 .54006 Mdot
.78571 .5411 Mdot
.80688 .54201 Mdot
.82804 .54279 Mdot
.84921 .54346 Mdot
.87037 .54402 Mdot
.89153 .54448 Mdot
.9127 .54486 Mdot
.93386 .54514 Mdot
.95503 .54535 Mdot
.97619 .55852 Mdot
0 1 0 r
.02381 .44485 Mdot
.04497 .4186 Mdot
.06614 .41093 Mdot
.0873 .41097 Mdot
.10847 .4109 Mdot
.12963 .41071 Mdot
.15079 .41041 Mdot
.17196 .40998 Mdot
.19312 .40941 Mdot
.21429 .4087 Mdot
.23545 .40784 Mdot
.25661 .4068 Mdot
.27778 .40558 Mdot
.29894 .40413 Mdot
.32011 .40242 Mdot
.34127 .4004 Mdot
.36243 .398 Mdot
.3836 .3951 Mdot
.40476 .39151 Mdot
.42593 .38693 Mdot
.44709 .38068 Mdot
.46825 .37108 Mdot
.48942 .3503 Mdot
.51058 .3503 Mdot
.53175 .37108 Mdot
.55291 .38068 Mdot
.57407 .38693 Mdot
.59524 .39151 Mdot
.6164 .3951 Mdot
.63757 .398 Mdot
.65873 .4004 Mdot
.67989 .40242 Mdot
.70106 .40413 Mdot
.72222 .40558 Mdot
.74339 .4068 Mdot
.76455 .40784 Mdot
.78571 .4087 Mdot
.80688 .40941 Mdot
.82804 .40998 Mdot
.84921 .41041 Mdot
.87037 .41071 Mdot
.89153 .4109 Mdot
.9127 .41097 Mdot
.93386 .41093 Mdot
.95503 .4186 Mdot
.97619 .44485 Mdot
0 0 1 r
.02381 .34122 Mdot
.04497 .30731 Mdot
.06614 .28999 Mdot
.0873 .28458 Mdot
.10847 .28485 Mdot
.12963 .28498 Mdot
.15079 .28496 Mdot
.17196 .28478 Mdot
.19312 .28445 Mdot
.21429 .28395 Mdot
.23545 .28328 Mdot
.25661 .28241 Mdot
.27778 .28134 Mdot
.29894 .28003 Mdot
.32011 .27844 Mdot
.34127 .27653 Mdot
.36243 .27422 Mdot
.3836 .27139 Mdot
.40476 .26787 Mdot
.42593 .26333 Mdot
.44709 .25712 Mdot
.46825 .24755 Mdot
.48942 .22679 Mdot
.51058 .22678 Mdot
.53175 .24755 Mdot
.55291 .25712 Mdot
.57407 .26333 Mdot
.59524 .26787 Mdot
.6164 .27139 Mdot
.63757 .27422 Mdot
.65873 .27653 Mdot
.67989 .27844 Mdot
.70106 .28002 Mdot
.72222 .28134 Mdot
.74339 .28241 Mdot
.76455 .28328 Mdot
.78571 .28395 Mdot
.80688 .28445 Mdot
.82804 .28478 Mdot
.84921 .28496 Mdot
.87037 .28498 Mdot
.89153 .28485 Mdot
.9127 .28458 Mdot
.93386 .28999 Mdot
.95503 .30731 Mdot
.97619 .34122 Mdot
1 0 1 r
.02381 .24435 Mdot
.04497 .20505 Mdot
.06614 .18129 Mdot
.0873 .16822 Mdot
.10847 .16429 Mdot
.12963 .1648 Mdot
.15079 .16495 Mdot
.17196 .16509 Mdot
.19312 .16513 Mdot
.21429 .16476 Mdot
.23545 .16437 Mdot
.25661 .16368 Mdot
.27778 .16262 Mdot
.29894 .16163 Mdot
.32011 .16026 Mdot
.34127 .15831 Mdot
.36243 .15622 Mdot
.3836 .15344 Mdot
.40476 .14995 Mdot
.42593 .14584 Mdot
.44709 .13988 Mdot
.46825 .13101 Mdot
.48942 .11172 Mdot
.51058 .10366 Mdot
.53175 .12848 Mdot
.55291 .13834 Mdot
.57407 .14473 Mdot
.59524 .14906 Mdot
.6164 .15271 Mdot
.63757 .1556 Mdot
.65873 .15776 Mdot
.67989 .15977 Mdot
.70106 .16117 Mdot
.72222 .16219 Mdot
.74339 .16328 Mdot
.76455 .16399 Mdot
.78571 .16439 Mdot
.80688 .16477 Mdot
.82804 .16474 Mdot
.84921 .1646 Mdot
.87037 .16445 Mdot
.89153 .16394 Mdot
.9127 .16837 Mdot
.93386 .1814 Mdot
.95503 .205 Mdot
.97619 .24443 Mdot
0 0 0 r
.02381 .59459 Mdot
.04497 .53179 Mdot
.06614 .46938 Mdot
.0873 .42356 Mdot
.10847 .39185 Mdot
.12963 .31781 Mdot
.15079 .30335 Mdot
.17196 .2175 Mdot
.19312 .25679 Mdot
.21429 .22461 Mdot
.23545 .2011 Mdot
.25661 .13341 Mdot
.27778 .11762 Mdot
.29894 .15542 Mdot
.32011 .12664 Mdot
.34127 .10079 Mdot
.36243 .11017 Mdot
.3836 .09631 Mdot
.40476 .10708 Mdot
.42593 .09787 Mdot
.44709 .10998 Mdot
.46825 .12219 Mdot
.48942 .10012 Mdot
.51058 .10804 Mdot
.53175 .10738 Mdot
.55291 .08688 Mdot
.57407 .1255 Mdot
.59524 .08374 Mdot
.6164 .11938 Mdot
.63757 .12081 Mdot
.65873 .03554 Mdot
.67989 .12463 Mdot
.70106 .15521 Mdot
.72222 .15597 Mdot
.74339 .16154 Mdot
.76455 .15522 Mdot
.78571 .19705 Mdot
.80688 .24838 Mdot
.82804 .24432 Mdot
.84921 .284 Mdot
.87037 .29995 Mdot
.89153 .35167 Mdot
.9127 .42598 Mdot
.93386 .46947 Mdot
.95503 .52442 Mdot
.97619 .60332 Mdot
.5 .5 .5 r
.02381 .15317 Mdot
.02497 .06897 Mdot
.02844 .08156 Mdot
.03422 .09837 Mdot
.04226 .0883 Mdot
.05253 .06951 Mdot
.06498 .06334 Mdot
.07955 .07456 Mdot
.09617 .07374 Mdot
.11475 .08729 Mdot
.13522 .07415 Mdot
.15746 .06951 Mdot
.18137 .041 Mdot
.20683 .07611 Mdot
.23372 .07822 Mdot
.2619 .05637 Mdot
.29125 .07103 Mdot
.32162 .07288 Mdot
.35285 .06017 Mdot
.3848 .08012 Mdot
.41731 .01895 Mdot
.45022 .04269 Mdot
.48338 .02118 Mdot
.51662 .04487 Mdot
.54978 .04546 Mdot
.58269 .06017 Mdot
.6152 .06405 Mdot
.64715 .03846 Mdot
.67838 .04868 Mdot
.70875 .03977 Mdot
.7381 .05837 Mdot
.76628 .01472 Mdot
.79317 .0502 Mdot
.81863 .06541 Mdot
.84254 .06017 Mdot
.86478 .06182 Mdot
.88525 .07496 Mdot
.90383 .08184 Mdot
.92045 .06786 Mdot
.93502 .081 Mdot
.94747 .0516 Mdot
.95774 .10662 Mdot
.96578 .10458 Mdot
.97156 .09755 Mdot
.97503 .12177 Mdot
.97619 .15943 Mdot
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{288, 177.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->1268466541,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.271653, -17.6321, \
0.00451797, 0.0985753}}],

Cell[TextData[{
 "A semilogarithmic plot of the error for approximating the first derivative \
of the Gaussian ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", "x", ")"}], " ", "=", " ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"15", " ", 
         RowBox[{"(", 
          RowBox[{"x", " ", "-", " ", 
           RowBox[{"1", "/", "2"}]}], ")"}]}], ")"}], "2"]}]]}], 
   TraditionalForm]]],
 "as a function of x at points on a 45 point grid covering the interval \
[0,1]. Finite differences of order 2, 4, 6, and 8 on a uniform grid are shown \
in red, green, blue, and magenta, respectively. Pseudospectral derivatives \
with uniform (nonperiodic) and Chebyshev spacing are shown in black and grey, \
respectively. All but the pseudospectral derivative with Chebyshev spacing \
were computed using uniform spacing 1/45. The error for the finite difference \
approximations are localized, and the magnitude of the errors follows the \
magnitude of the first derivative of the Gaussian. The error near the \
boundary for the uniform spacing pseudospectral (order 45 polynomial) \
approximation becomes enormous; as h decreases, this is not bounded.",
 " ",
 "On the other hand, the error for the Chebyshev spacing pseudospectral is \
more uniform and overall quite small."
}], "Caption",
 CellID->1340420243],

Cell[TextData[{
 "From what has so far been shown, it would appear that the higher the order \
of the approximation, the better. However, there are two additional issues to \
consider. The higher order approximations lead to more expensive function \
evaluations, and if implicit iteration is needed (as for a stiff problem), \
then not only is computing the Jacobian more expensive, but the eigenvalues \
of the matrix also tend to be larger, leading to more stiffness and more \
difficultly for iterative solvers. This is at an extreme for pseudospectral \
methods, where the Jacobian has essentially no nonzero entries [",
 ButtonBox["F96a",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#22110"],
 "]. Of course, these problems are a trade-off for smaller system (and hence \
matrix) size.",
 " "
}], "Text",
 CellID->1432040938],

Cell["\<\
The other issue is associated with discontinuities. Typically, the higher \
order the polynomial approximation, the worse the approximation. To make \
matters even worse, for a true discontinuity, the errors magnify as the grid \
spacing is reduced. \
\>", "Text",
 CellID->273639001],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.309017 0.00328449 [
[.21429 .29652 -9 -9 ]
[.21429 .29652 9 0 ]
[.40476 .29652 -9 -9 ]
[.40476 .29652 9 0 ]
[.59524 .29652 -9 -9 ]
[.59524 .29652 9 0 ]
[.78571 .29652 -9 -9 ]
[.78571 .29652 9 0 ]
[.97619 .29652 -3 -9 ]
[.97619 .29652 3 0 ]
[.01131 .06268 -18 -4.5 ]
[.01131 .06268 0 4.5 ]
[.01131 .14479 -18 -4.5 ]
[.01131 .14479 0 4.5 ]
[.01131 .2269 -18 -4.5 ]
[.01131 .2269 0 4.5 ]
[.01131 .39113 -12 -4.5 ]
[.01131 .39113 0 4.5 ]
[.01131 .47324 -12 -4.5 ]
[.01131 .47324 0 4.5 ]
[.01131 .55535 -12 -4.5 ]
[.01131 .55535 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .30902 m
.21429 .31527 L
s
[(0.2)] .21429 .29652 0 1 Mshowa
.40476 .30902 m
.40476 .31527 L
s
[(0.4)] .40476 .29652 0 1 Mshowa
.59524 .30902 m
.59524 .31527 L
s
[(0.6)] .59524 .29652 0 1 Mshowa
.78571 .30902 m
.78571 .31527 L
s
[(0.8)] .78571 .29652 0 1 Mshowa
.97619 .30902 m
.97619 .31527 L
s
[(1)] .97619 .29652 0 1 Mshowa
.125 Mabswid
.07143 .30902 m
.07143 .31277 L
s
.11905 .30902 m
.11905 .31277 L
s
.16667 .30902 m
.16667 .31277 L
s
.2619 .30902 m
.2619 .31277 L
s
.30952 .30902 m
.30952 .31277 L
s
.35714 .30902 m
.35714 .31277 L
s
.45238 .30902 m
.45238 .31277 L
s
.5 .30902 m
.5 .31277 L
s
.54762 .30902 m
.54762 .31277 L
s
.64286 .30902 m
.64286 .31277 L
s
.69048 .30902 m
.69048 .31277 L
s
.7381 .30902 m
.7381 .31277 L
s
.83333 .30902 m
.83333 .31277 L
s
.88095 .30902 m
.88095 .31277 L
s
.92857 .30902 m
.92857 .31277 L
s
.25 Mabswid
0 .30902 m
1 .30902 L
s
.02381 .06268 m
.03006 .06268 L
s
[(-75)] .01131 .06268 1 0 Mshowa
.02381 .14479 m
.03006 .14479 L
s
[(-50)] .01131 .14479 1 0 Mshowa
.02381 .2269 m
.03006 .2269 L
s
[(-25)] .01131 .2269 1 0 Mshowa
.02381 .39113 m
.03006 .39113 L
s
[(25)] .01131 .39113 1 0 Mshowa
.02381 .47324 m
.03006 .47324 L
s
[(50)] .01131 .47324 1 0 Mshowa
.02381 .55535 m
.03006 .55535 L
s
[(75)] .01131 .55535 1 0 Mshowa
.125 Mabswid
.02381 .0791 m
.02756 .0791 L
s
.02381 .09553 m
.02756 .09553 L
s
.02381 .11195 m
.02756 .11195 L
s
.02381 .12837 m
.02756 .12837 L
s
.02381 .16122 m
.02756 .16122 L
s
.02381 .17764 m
.02756 .17764 L
s
.02381 .19406 m
.02756 .19406 L
s
.02381 .21048 m
.02756 .21048 L
s
.02381 .24333 m
.02756 .24333 L
s
.02381 .25975 m
.02756 .25975 L
s
.02381 .27617 m
.02756 .27617 L
s
.02381 .29259 m
.02756 .29259 L
s
.02381 .32544 m
.02756 .32544 L
s
.02381 .34186 m
.02756 .34186 L
s
.02381 .35828 m
.02756 .35828 L
s
.02381 .37471 m
.02756 .37471 L
s
.02381 .40755 m
.02756 .40755 L
s
.02381 .42397 m
.02756 .42397 L
s
.02381 .4404 m
.02756 .4404 L
s
.02381 .45682 m
.02756 .45682 L
s
.02381 .48966 m
.02756 .48966 L
s
.02381 .50609 m
.02756 .50609 L
s
.02381 .52251 m
.02756 .52251 L
s
.02381 .53893 m
.02756 .53893 L
s
.02381 .04626 m
.02756 .04626 L
s
.02381 .02984 m
.02756 .02984 L
s
.02381 .01341 m
.02756 .01341 L
s
.02381 .57178 m
.02756 .57178 L
s
.02381 .5882 m
.02756 .5882 L
s
.02381 .60462 m
.02756 .60462 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 0 r
.015 w
.02381 .01472 Mdot
.03131 .44221 Mdot
.03881 .22323 Mdot
.04631 .37274 Mdot
.05381 .25793 Mdot
.0613 .35195 Mdot
.0688 .27176 Mdot
.0763 .3421 Mdot
.0838 .27912 Mdot
.0913 .3364 Mdot
.0988 .28365 Mdot
.1063 .33272 Mdot
.1138 .28669 Mdot
.1213 .33017 Mdot
.1288 .28885 Mdot
.1363 .32833 Mdot
.14379 .29044 Mdot
.15129 .32696 Mdot
.15879 .29163 Mdot
.16629 .32592 Mdot
.17379 .29254 Mdot
.18129 .32513 Mdot
.18879 .29323 Mdot
.19629 .32453 Mdot
.20379 .29374 Mdot
.21129 .32409 Mdot
.21879 .29411 Mdot
.22628 .32379 Mdot
.23378 .29436 Mdot
.24128 .3236 Mdot
.24878 .29449 Mdot
.25628 .32352 Mdot
.26378 .29451 Mdot
.27128 .32355 Mdot
.27878 .29443 Mdot
.28628 .32368 Mdot
.29378 .29425 Mdot
.30127 .32392 Mdot
.30877 .29394 Mdot
.31627 .32429 Mdot
.32377 .2935 Mdot
.33127 .32481 Mdot
.33877 .29291 Mdot
.34627 .3255 Mdot
.35377 .29212 Mdot
.36127 .3264 Mdot
.36877 .29108 Mdot
.37627 .32759 Mdot
.38376 .2897 Mdot
.39126 .32918 Mdot
.39876 .28786 Mdot
.40626 .33134 Mdot
.41376 .28531 Mdot
.42126 .33438 Mdot
.42876 .28163 Mdot
.43626 .33891 Mdot
.44376 .27593 Mdot
.45126 .34627 Mdot
.45876 .26608 Mdot
.46625 .3601 Mdot
.47375 .24529 Mdot
.48125 .3948 Mdot
.48875 .17583 Mdot
.49625 .60332 Mdot
.50375 .593 Mdot
.51125 .18615 Mdot
.51875 .38446 Mdot
.52625 .25565 Mdot
.53375 .34972 Mdot
.54124 .2765 Mdot
.54874 .33582 Mdot
.55624 .28643 Mdot
.56374 .32836 Mdot
.57124 .29224 Mdot
.57874 .32371 Mdot
.58624 .29606 Mdot
.59374 .32051 Mdot
.60124 .29878 Mdot
.60874 .31816 Mdot
.61624 .30083 Mdot
.62373 .31635 Mdot
.63123 .30244 Mdot
.63873 .3149 Mdot
.64623 .30376 Mdot
.65373 .31369 Mdot
.66123 .30488 Mdot
.66873 .31266 Mdot
.67623 .30585 Mdot
.68373 .31174 Mdot
.69123 .30672 Mdot
.69873 .31091 Mdot
.70622 .30752 Mdot
.71372 .31013 Mdot
.72122 .30828 Mdot
.72872 .30938 Mdot
.73622 .30902 Mdot
.74372 .30865 Mdot
.75122 .30976 Mdot
.75872 .3079 Mdot
.76622 .31051 Mdot
.77372 .30713 Mdot
.78121 .31132 Mdot
.78871 .30629 Mdot
.79621 .31218 Mdot
.80371 .30538 Mdot
.81121 .31316 Mdot
.81871 .30434 Mdot
.82621 .31427 Mdot
.83371 .30313 Mdot
.84121 .31559 Mdot
.84871 .30168 Mdot
.85621 .3172 Mdot
.8637 .29987 Mdot
.8712 .31925 Mdot
.8787 .29753 Mdot
.8862 .32197 Mdot
.8937 .29433 Mdot
.9012 .32579 Mdot
.9087 .28967 Mdot
.9162 .3316 Mdot
.9237 .28221 Mdot
.9312 .34154 Mdot
.9387 .26831 Mdot
.94619 .36238 Mdot
.95369 .23357 Mdot
.96119 .43188 Mdot
.96869 .02503 Mdot
.97619 .01472 Mdot
1 0 0 r
.02381 .30902 Mdot
.03131 .30902 Mdot
.03881 .30902 Mdot
.04631 .30902 Mdot
.05381 .30902 Mdot
.0613 .30902 Mdot
.0688 .30902 Mdot
.0763 .30902 Mdot
.0838 .30902 Mdot
.0913 .30902 Mdot
.0988 .30902 Mdot
.1063 .30902 Mdot
.1138 .30902 Mdot
.1213 .30902 Mdot
.1288 .30902 Mdot
.1363 .30902 Mdot
.14379 .30902 Mdot
.15129 .30902 Mdot
.15879 .30902 Mdot
.16629 .30902 Mdot
.17379 .30902 Mdot
.18129 .30902 Mdot
.18879 .30902 Mdot
.19629 .30902 Mdot
.20379 .30902 Mdot
.21129 .30902 Mdot
.21879 .30902 Mdot
.22628 .30902 Mdot
.23378 .30902 Mdot
.24128 .30902 Mdot
.24878 .30902 Mdot
.25628 .30902 Mdot
.26378 .30902 Mdot
.27128 .30902 Mdot
.27878 .30902 Mdot
.28628 .30902 Mdot
.29378 .30902 Mdot
.30127 .30902 Mdot
.30877 .30902 Mdot
.31627 .30902 Mdot
.32377 .30902 Mdot
.33127 .30902 Mdot
.33877 .30902 Mdot
.34627 .30902 Mdot
.35377 .30902 Mdot
.36127 .30902 Mdot
.36877 .30902 Mdot
.37627 .30902 Mdot
.38376 .30902 Mdot
.39126 .30902 Mdot
.39876 .30902 Mdot
.40626 .30902 Mdot
.41376 .30902 Mdot
.42126 .30902 Mdot
.42876 .30902 Mdot
.43626 .30902 Mdot
.44376 .30902 Mdot
.45126 .30902 Mdot
.45876 .30902 Mdot
.46625 .30902 Mdot
.47375 .30902 Mdot
.48125 .30902 Mdot
.48875 .30902 Mdot
.49625 .51758 Mdot
.50375 .51758 Mdot
.51125 .30902 Mdot
.51875 .30902 Mdot
.52625 .30902 Mdot
.53375 .30902 Mdot
.54124 .30902 Mdot
.54874 .30902 Mdot
.55624 .30902 Mdot
.56374 .30902 Mdot
.57124 .30902 Mdot
.57874 .30902 Mdot
.58624 .30902 Mdot
.59374 .30902 Mdot
.60124 .30902 Mdot
.60874 .30902 Mdot
.61624 .30902 Mdot
.62373 .30902 Mdot
.63123 .30902 Mdot
.63873 .30902 Mdot
.64623 .30902 Mdot
.65373 .30902 Mdot
.66123 .30902 Mdot
.66873 .30902 Mdot
.67623 .30902 Mdot
.68373 .30902 Mdot
.69123 .30902 Mdot
.69873 .30902 Mdot
.70622 .30902 Mdot
.71372 .30902 Mdot
.72122 .30902 Mdot
.72872 .30902 Mdot
.73622 .30902 Mdot
.74372 .30902 Mdot
.75122 .30902 Mdot
.75872 .30902 Mdot
.76622 .30902 Mdot
.77372 .30902 Mdot
.78121 .30902 Mdot
.78871 .30902 Mdot
.79621 .30902 Mdot
.80371 .30902 Mdot
.81121 .30902 Mdot
.81871 .30902 Mdot
.82621 .30902 Mdot
.83371 .30902 Mdot
.84121 .30902 Mdot
.84871 .30902 Mdot
.85621 .30902 Mdot
.8637 .30902 Mdot
.8712 .30902 Mdot
.8787 .30902 Mdot
.8862 .30902 Mdot
.8937 .30902 Mdot
.9012 .30902 Mdot
.9087 .30902 Mdot
.9162 .30902 Mdot
.9237 .30902 Mdot
.9312 .30902 Mdot
.9387 .30902 Mdot
.94619 .30902 Mdot
.95369 .30902 Mdot
.96119 .30902 Mdot
.96869 .30902 Mdot
.97619 .30902 Mdot
0 1 0 r
.02381 .30902 Mdot
.03131 .30902 Mdot
.03881 .30902 Mdot
.04631 .30902 Mdot
.05381 .30902 Mdot
.0613 .30902 Mdot
.0688 .30902 Mdot
.0763 .30902 Mdot
.0838 .30902 Mdot
.0913 .30902 Mdot
.0988 .30902 Mdot
.1063 .30902 Mdot
.1138 .30902 Mdot
.1213 .30902 Mdot
.1288 .30902 Mdot
.1363 .30902 Mdot
.14379 .30902 Mdot
.15129 .30902 Mdot
.15879 .30902 Mdot
.16629 .30902 Mdot
.17379 .30902 Mdot
.18129 .30902 Mdot
.18879 .30902 Mdot
.19629 .30902 Mdot
.20379 .30902 Mdot
.21129 .30902 Mdot
.21879 .30902 Mdot
.22628 .30902 Mdot
.23378 .30902 Mdot
.24128 .30902 Mdot
.24878 .30902 Mdot
.25628 .30902 Mdot
.26378 .30902 Mdot
.27128 .30902 Mdot
.27878 .30902 Mdot
.28628 .30902 Mdot
.29378 .30902 Mdot
.30127 .30902 Mdot
.30877 .30902 Mdot
.31627 .30902 Mdot
.32377 .30902 Mdot
.33127 .30902 Mdot
.33877 .30902 Mdot
.34627 .30902 Mdot
.35377 .30902 Mdot
.36127 .30902 Mdot
.36877 .30902 Mdot
.37627 .30902 Mdot
.38376 .30902 Mdot
.39126 .30902 Mdot
.39876 .30902 Mdot
.40626 .30902 Mdot
.41376 .30902 Mdot
.42126 .30902 Mdot
.42876 .30902 Mdot
.43626 .30902 Mdot
.44376 .30902 Mdot
.45126 .30902 Mdot
.45876 .30902 Mdot
.46625 .30902 Mdot
.47375 .30902 Mdot
.48125 .30902 Mdot
.48875 .27426 Mdot
.49625 .55234 Mdot
.50375 .55234 Mdot
.51125 .27426 Mdot
.51875 .30902 Mdot
.52625 .30902 Mdot
.53375 .30902 Mdot
.54124 .30902 Mdot
.54874 .30902 Mdot
.55624 .30902 Mdot
.56374 .30902 Mdot
.57124 .30902 Mdot
.57874 .30902 Mdot
.58624 .30902 Mdot
.59374 .30902 Mdot
.60124 .30902 Mdot
.60874 .30902 Mdot
.61624 .30902 Mdot
.62373 .30902 Mdot
.63123 .30902 Mdot
.63873 .30902 Mdot
.64623 .30902 Mdot
.65373 .30902 Mdot
.66123 .30902 Mdot
.66873 .30902 Mdot
.67623 .30902 Mdot
.68373 .30902 Mdot
.69123 .30902 Mdot
.69873 .30902 Mdot
.70622 .30902 Mdot
.71372 .30902 Mdot
.72122 .30902 Mdot
.72872 .30902 Mdot
.73622 .30902 Mdot
.74372 .30902 Mdot
.75122 .30902 Mdot
.75872 .30902 Mdot
.76622 .30902 Mdot
.77372 .30902 Mdot
.78121 .30902 Mdot
.78871 .30902 Mdot
.79621 .30902 Mdot
.80371 .30902 Mdot
.81121 .30902 Mdot
.81871 .30902 Mdot
.82621 .30902 Mdot
.83371 .30902 Mdot
.84121 .30902 Mdot
.84871 .30902 Mdot
.85621 .30902 Mdot
.8637 .30902 Mdot
.8712 .30902 Mdot
.8787 .30902 Mdot
.8862 .30902 Mdot
.8937 .30902 Mdot
.9012 .30902 Mdot
.9087 .30902 Mdot
.9162 .30902 Mdot
.9237 .30902 Mdot
.9312 .30902 Mdot
.9387 .30902 Mdot
.94619 .30902 Mdot
.95369 .30902 Mdot
.96119 .30902 Mdot
.96869 .30902 Mdot
.97619 .30902 Mdot
0 0 1 r
.02381 .30902 Mdot
.03131 .30902 Mdot
.03881 .30902 Mdot
.04631 .30902 Mdot
.05381 .30902 Mdot
.0613 .30902 Mdot
.0688 .30902 Mdot
.0763 .30902 Mdot
.0838 .30902 Mdot
.0913 .30902 Mdot
.0988 .30902 Mdot
.1063 .30902 Mdot
.1138 .30902 Mdot
.1213 .30902 Mdot
.1288 .30902 Mdot
.1363 .30902 Mdot
.14379 .30902 Mdot
.15129 .30902 Mdot
.15879 .30902 Mdot
.16629 .30902 Mdot
.17379 .30902 Mdot
.18129 .30902 Mdot
.18879 .30902 Mdot
.19629 .30902 Mdot
.20379 .30902 Mdot
.21129 .30902 Mdot
.21879 .30902 Mdot
.22628 .30902 Mdot
.23378 .30902 Mdot
.24128 .30902 Mdot
.24878 .30902 Mdot
.25628 .30902 Mdot
.26378 .30902 Mdot
.27128 .30902 Mdot
.27878 .30902 Mdot
.28628 .30902 Mdot
.29378 .30902 Mdot
.30127 .30902 Mdot
.30877 .30902 Mdot
.31627 .30902 Mdot
.32377 .30902 Mdot
.33127 .30902 Mdot
.33877 .30902 Mdot
.34627 .30902 Mdot
.35377 .30902 Mdot
.36127 .30902 Mdot
.36877 .30902 Mdot
.37627 .30902 Mdot
.38376 .30902 Mdot
.39126 .30902 Mdot
.39876 .30902 Mdot
.40626 .30902 Mdot
.41376 .30902 Mdot
.42126 .30902 Mdot
.42876 .30902 Mdot
.43626 .30902 Mdot
.44376 .30902 Mdot
.45126 .30902 Mdot
.45876 .30902 Mdot
.46625 .30902 Mdot
.47375 .30902 Mdot
.48125 .31597 Mdot
.48875 .2534 Mdot
.49625 .56625 Mdot
.50375 .56625 Mdot
.51125 .2534 Mdot
.51875 .31597 Mdot
.52625 .30902 Mdot
.53375 .30902 Mdot
.54124 .30902 Mdot
.54874 .30902 Mdot
.55624 .30902 Mdot
.56374 .30902 Mdot
.57124 .30902 Mdot
.57874 .30902 Mdot
.58624 .30902 Mdot
.59374 .30902 Mdot
.60124 .30902 Mdot
.60874 .30902 Mdot
.61624 .30902 Mdot
.62373 .30902 Mdot
.63123 .30902 Mdot
.63873 .30902 Mdot
.64623 .30902 Mdot
.65373 .30902 Mdot
.66123 .30902 Mdot
.66873 .30902 Mdot
.67623 .30902 Mdot
.68373 .30902 Mdot
.69123 .30902 Mdot
.69873 .30902 Mdot
.70622 .30902 Mdot
.71372 .30902 Mdot
.72122 .30902 Mdot
.72872 .30902 Mdot
.73622 .30902 Mdot
.74372 .30902 Mdot
.75122 .30902 Mdot
.75872 .30902 Mdot
.76622 .30902 Mdot
.77372 .30902 Mdot
.78121 .30902 Mdot
.78871 .30902 Mdot
.79621 .30902 Mdot
.80371 .30902 Mdot
.81121 .30902 Mdot
.81871 .30902 Mdot
.82621 .30902 Mdot
.83371 .30902 Mdot
.84121 .30902 Mdot
.84871 .30902 Mdot
.85621 .30902 Mdot
.8637 .30902 Mdot
.8712 .30902 Mdot
.8787 .30902 Mdot
.8862 .30902 Mdot
.8937 .30902 Mdot
.9012 .30902 Mdot
.9087 .30902 Mdot
.9162 .30902 Mdot
.9237 .30902 Mdot
.9312 .30902 Mdot
.9387 .30902 Mdot
.94619 .30902 Mdot
.95369 .30902 Mdot
.96119 .30902 Mdot
.96869 .30902 Mdot
.97619 .30902 Mdot
1 0 1 r
.02381 .30902 Mdot
.03131 .30902 Mdot
.03881 .30902 Mdot
.04631 .30902 Mdot
.05381 .30902 Mdot
.0613 .30902 Mdot
.0688 .30902 Mdot
.0763 .30902 Mdot
.0838 .30902 Mdot
.0913 .30902 Mdot
.0988 .30902 Mdot
.1063 .30902 Mdot
.1138 .30902 Mdot
.1213 .30902 Mdot
.1288 .30902 Mdot
.1363 .30902 Mdot
.14379 .30902 Mdot
.15129 .30902 Mdot
.15879 .30902 Mdot
.16629 .30902 Mdot
.17379 .30902 Mdot
.18129 .30902 Mdot
.18879 .30902 Mdot
.19629 .30902 Mdot
.20379 .30902 Mdot
.21129 .30902 Mdot
.21879 .30902 Mdot
.22628 .30902 Mdot
.23378 .30902 Mdot
.24128 .30902 Mdot
.24878 .30902 Mdot
.25628 .30902 Mdot
.26378 .30902 Mdot
.27128 .30902 Mdot
.27878 .30902 Mdot
.28628 .30902 Mdot
.29378 .30902 Mdot
.30127 .30902 Mdot
.30877 .30902 Mdot
.31627 .30902 Mdot
.32377 .30902 Mdot
.33127 .30902 Mdot
.33877 .30902 Mdot
.34627 .30902 Mdot
.35377 .30902 Mdot
.36127 .30902 Mdot
.36877 .30902 Mdot
.37627 .30902 Mdot
.38376 .30902 Mdot
.39126 .30902 Mdot
.39876 .30902 Mdot
.40626 .30902 Mdot
.41376 .30902 Mdot
.42126 .30902 Mdot
.42876 .30902 Mdot
.43626 .30902 Mdot
.44376 .30902 Mdot
.45126 .30902 Mdot
.45876 .30902 Mdot
.46625 .30902 Mdot
.47375 .30753 Mdot
.48125 .32342 Mdot
.48875 .23999 Mdot
.49625 .5737 Mdot
.50375 .5737 Mdot
.51125 .23999 Mdot
.51875 .32342 Mdot
.52625 .30753 Mdot
.53375 .30902 Mdot
.54124 .30902 Mdot
.54874 .30902 Mdot
.55624 .30902 Mdot
.56374 .30902 Mdot
.57124 .30902 Mdot
.57874 .30902 Mdot
.58624 .30902 Mdot
.59374 .30902 Mdot
.60124 .30902 Mdot
.60874 .30902 Mdot
.61624 .30902 Mdot
.62373 .30902 Mdot
.63123 .30902 Mdot
.63873 .30902 Mdot
.64623 .30902 Mdot
.65373 .30902 Mdot
.66123 .30902 Mdot
.66873 .30902 Mdot
.67623 .30902 Mdot
.68373 .30902 Mdot
.69123 .30902 Mdot
.69873 .30902 Mdot
.70622 .30902 Mdot
.71372 .30902 Mdot
.72122 .30902 Mdot
.72872 .30902 Mdot
.73622 .30902 Mdot
.74372 .30902 Mdot
.75122 .30902 Mdot
.75872 .30902 Mdot
.76622 .30902 Mdot
.77372 .30902 Mdot
.78121 .30902 Mdot
.78871 .30902 Mdot
.79621 .30902 Mdot
.80371 .30902 Mdot
.81121 .30902 Mdot
.81871 .30902 Mdot
.82621 .30902 Mdot
.83371 .30902 Mdot
.84121 .30902 Mdot
.84871 .30902 Mdot
.85621 .30902 Mdot
.8637 .30902 Mdot
.8712 .30902 Mdot
.8787 .30902 Mdot
.8862 .30902 Mdot
.8937 .30902 Mdot
.9012 .30902 Mdot
.9087 .30902 Mdot
.9162 .30902 Mdot
.9237 .30902 Mdot
.9312 .30902 Mdot
.9387 .30902 Mdot
.94619 .30902 Mdot
.95369 .30902 Mdot
.96119 .30902 Mdot
.96869 .30902 Mdot
.97619 .30902 Mdot
.5 .5 .5 r
.02381 .30245 Mdot
.02396 .3123 Mdot
.02439 .30573 Mdot
.02512 .31231 Mdot
.02614 .30572 Mdot
.02745 .31233 Mdot
.02904 .3057 Mdot
.03093 .31235 Mdot
.0331 .30567 Mdot
.03556 .31238 Mdot
.0383 .30563 Mdot
.04133 .31243 Mdot
.04464 .30558 Mdot
.04822 .31248 Mdot
.05208 .30553 Mdot
.05622 .31254 Mdot
.06062 .30546 Mdot
.0653 .31261 Mdot
.07024 .30538 Mdot
.07544 .3127 Mdot
.08091 .30529 Mdot
.08663 .3128 Mdot
.0926 .30518 Mdot
.09883 .31291 Mdot
.10529 .30506 Mdot
.112 .31305 Mdot
.11895 .30491 Mdot
.12613 .3132 Mdot
.13354 .30475 Mdot
.14117 .31337 Mdot
.14903 .30456 Mdot
.15709 .31358 Mdot
.16537 .30434 Mdot
.17385 .31381 Mdot
.18254 .30409 Mdot
.19141 .31408 Mdot
.20048 .3038 Mdot
.20972 .3144 Mdot
.21915 .30345 Mdot
.22875 .31478 Mdot
.23851 .30304 Mdot
.24843 .31523 Mdot
.25851 .30254 Mdot
.26874 .31578 Mdot
.2791 .30194 Mdot
.28961 .31645 Mdot
.30024 .30119 Mdot
.31099 .31728 Mdot
.32186 .30025 Mdot
.33284 .31836 Mdot
.34392 .29901 Mdot
.35509 .31979 Mdot
.36636 .29734 Mdot
.3777 .32178 Mdot
.38912 .29495 Mdot
.40061 .3247 Mdot
.41216 .29129 Mdot
.42376 .32941 Mdot
.43541 .28499 Mdot
.4471 .33824 Mdot
.45882 .27175 Mdot
.47057 .36034 Mdot
.48233 .22752 Mdot
.49411 .49308 Mdot
.50589 .49308 Mdot
.51767 .22752 Mdot
.52943 .36034 Mdot
.54118 .27175 Mdot
.5529 .33824 Mdot
.56459 .28499 Mdot
.57624 .32941 Mdot
.58784 .29129 Mdot
.59939 .3247 Mdot
.61088 .29495 Mdot
.6223 .32178 Mdot
.63364 .29734 Mdot
.64491 .31979 Mdot
.65608 .29901 Mdot
.66716 .31836 Mdot
.67814 .30025 Mdot
.68901 .31728 Mdot
.69976 .30119 Mdot
.71039 .31645 Mdot
.7209 .30194 Mdot
.73126 .31578 Mdot
.74149 .30254 Mdot
.75157 .31523 Mdot
.76149 .30304 Mdot
.77125 .31478 Mdot
.78085 .30345 Mdot
.79028 .3144 Mdot
.79952 .3038 Mdot
.80859 .31408 Mdot
.81746 .30409 Mdot
.82615 .31381 Mdot
.83463 .30434 Mdot
.84291 .31358 Mdot
.85097 .30456 Mdot
.85883 .31337 Mdot
.86646 .30475 Mdot
.87387 .3132 Mdot
.88105 .30491 Mdot
.888 .31305 Mdot
.89471 .30506 Mdot
.90117 .31291 Mdot
.9074 .30518 Mdot
.91337 .3128 Mdot
.91909 .30529 Mdot
.92456 .3127 Mdot
.92976 .30538 Mdot
.9347 .31261 Mdot
.93938 .30546 Mdot
.94378 .31254 Mdot
.94792 .30553 Mdot
.95178 .31248 Mdot
.95536 .30558 Mdot
.95867 .31243 Mdot
.9617 .30563 Mdot
.96444 .31238 Mdot
.9669 .30567 Mdot
.96907 .31235 Mdot
.97096 .3057 Mdot
.97255 .31233 Mdot
.97386 .30572 Mdot
.97488 .31231 Mdot
.97561 .30573 Mdot
.97604 .3123 Mdot
.97619 .30245 Mdot
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{288, 177.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellID->486809521,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0825959, -99.0004, \
0.00385924, 1.11904}}],

Cell[TextData[{
 "A plot of approximations for the first derivative of the discontinuous unit \
step function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", "x", ")"}], " ", "=", " ", 
    RowBox[{"UnitStep", "(", 
     RowBox[{"x", " ", "-", " ", 
      RowBox[{"1", "/", "2"}]}], ")"}]}], TraditionalForm]]],
 "as a function of x at points on a 128 point grid covering the interval \
[0,1]. Finite differences of order 2, 4, 6, and 8 on a uniform grid are shown \
in red, green, blue, and magenta, respectively. Pseudospectral derivatives \
with uniform (periodic) and Chebyshev spacing are shown in black and grey, \
respectively. All but the pseudospectral derivative with Chebyshev spacing \
were computed using uniform spacing 1/128.",
 " ",
 "All show oscillatory behavior, but it is apparent that the Chebyshev \
pseudospectral derivative does better in this regard."
}], "Caption",
 CellID->383737067],

Cell[TextData[{
 "There are numerous alternatives that are used around known discontinuities, \
such as front tracking. First order forward differences minimize oscillation, \
but introduce artificial viscosity terms. One good alternative are the \
so-called essentially nonoscillatory (ENO) schemes, which have full order \
away from discontinuities but introduce limits near discontinuities that \
limit the approximation order and the oscillatory behavior. At this time, ENO \
schemes are not implemented in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->1498136972],

Cell["\<\
In summary, choosing an appropriate difference order depends greatly on the \
problem structure. The default of 4 was chosen to be generally reasonable for \
a wide variety of PDEs, but you may want to try other settings for a \
particular problem to get better results.\
\>", "Text",
 CellID->1193344378]
}, Open  ]],

Cell[CellGroupData[{

Cell["Differentiation Matrices", "Subsubsection",
 CellTags->{"c:5", "b:7.1.4", "ndsg:2.0.1.4", "PDE:1.2.6"},
 CellID->1971845215],

Cell[TextData[{
 "Since differentiation, and naturally finite difference approximation, is a \
linear operation, an alternative way of expressing the action of a ",
 StyleBox["FiniteDifferenceDerivativeFunction", "MR"],
 " is with a matrix. A matrix that represents an approximation to the \
differential operator is referred to as a ",
 StyleBox["differentiation matrix ",
  FontWeight->"Bold"],
 "[",
 ButtonBox["F96a",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#22110"],
 "]. While differentiation matrices may not always be the optimal way of \
applying finite difference approximations (particularly in cases where an FFT \
can be used to reduce complexity and error), they are invaluable as aids for \
analysis and, sometimes, for use in the linear solvers often needed to solve \
PDEs. "
}], "Text",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->2108131000],

Cell[TextData[{
 "Let ",
 StyleBox["FDDF", "MR",
  FontSlant->"Italic"],
 " represent an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction[", "MR"],
 StyleBox["data", "MR",
  FontSlant->"Italic"],
 StyleBox["]", "MR"],
 " object."
}], "Text",
 CellID->514205085],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     StyleBox["FDDF", "TI"], 
     StyleBox["@", "InlineCode"], 
     StyleBox[
      RowBox[{"\"\<\>", "DifferentiationMatrix", "\"\<\>"}], "InlineCode"]}], 
    Cell[TextData[{
     "recast the linear operation of ",
     StyleBox["FDDF", "TI"],
     " as a matrix which represents the linear operator."
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->181409121],

Cell["Forming a differentiation matrix. ", "Caption",
 CellID->282967651],

Cell[TextData[{
 "This creates a ",
 StyleBox["FiniteDifferenceDerivativeFunction", "MR"],
 " object."
}], "MathCaption",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->1567540598],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fdd", " ", "=", " ", 
  RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
   RowBox[{"2", ",", " ", 
    RowBox[{"Range", "[", 
     RowBox[{"0", ",", "10"}], "]"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"In[37]:=",
 CellID->1975888561],

Cell[BoxData[
 TagBox[
  RowBox[{"NDSolve`FiniteDifferenceDerivativeFunction", "[", 
   RowBox[{
    RowBox[{"Derivative", "[", "2", "]"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{429, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"Out[37]=",
 CellID->26375331]
}, Open  ]],

Cell["\<\
This makes a matrix representing the underlying linear operator.\
\>", "MathCaption",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->578934179],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"smat", " ", "=", " ", 
  RowBox[{"fdd", "[", "\"\<DifferentiationMatrix\>\"", "]"}]}]], "Input",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"In[38]:=",
 CellID->149937115],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "59", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 59, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"11", ",", "11"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"Out[38]=",
 CellID->478765308]
}, Open  ]],

Cell["\<\
The matrix is given in a sparse form because, in general, differentiation \
matrices have relatively few nonzero entries. \
\>", "Text",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->1351902618],

Cell[TextData[{
 "This converts to a normal dense matrix and displays it using ",
 Cell[BoxData[
  ButtonBox["MatrixForm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixForm"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->834648453],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixForm", "[", 
  RowBox[{"mat", " ", "=", " ", 
   RowBox[{"Normal", "[", "smat", "]"}]}], "]"}]], "Input",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"In[39]:=",
 CellID->1298606618],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      FractionBox["15", "4"], 
      RowBox[{"-", 
       FractionBox["77", "6"]}], 
      FractionBox["107", "6"], 
      RowBox[{"-", "13"}], 
      FractionBox["61", "12"], 
      RowBox[{"-", 
       FractionBox["5", "6"]}], "0", "0", "0", "0", "0"},
     {
      FractionBox["5", "6"], 
      RowBox[{"-", 
       FractionBox["5", "4"]}], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      FractionBox["7", "6"], 
      RowBox[{"-", 
       FractionBox["1", "2"]}], 
      FractionBox["1", "12"], "0", "0", "0", "0", "0"},
     {
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}], "0", "0", "0", "0", "0", "0"},
     {"0", 
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}], "0", "0", "0", "0", "0"},
     {"0", "0", 
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}], "0", "0", "0", "0"},
     {"0", "0", "0", 
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}], "0", "0", "0"},
     {"0", "0", "0", "0", 
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}], "0", "0"},
     {"0", "0", "0", "0", "0", 
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}], "0"},
     {"0", "0", "0", "0", "0", "0", 
      RowBox[{"-", 
       FractionBox["1", "12"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["5", "2"]}], 
      FractionBox["4", "3"], 
      RowBox[{"-", 
       FractionBox["1", "12"]}]},
     {"0", "0", "0", "0", "0", 
      FractionBox["1", "12"], 
      RowBox[{"-", 
       FractionBox["1", "2"]}], 
      FractionBox["7", "6"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{"-", 
       FractionBox["5", "4"]}], 
      FractionBox["5", "6"]},
     {"0", "0", "0", "0", "0", 
      RowBox[{"-", 
       FractionBox["5", "6"]}], 
      FractionBox["61", "12"], 
      RowBox[{"-", "13"}], 
      FractionBox["107", "6"], 
      RowBox[{"-", 
       FractionBox["77", "6"]}], 
      FractionBox["15", "4"]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{346, 261},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"Out[39]//MatrixForm=",
 CellID->273826563]
}, Open  ]],

Cell["\<\
This shows that all three of the representations are roughly equivalent in \
terms of their action on data.\
\>", "MathCaption",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->455572395],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Exp", "[", 
       RowBox[{"-", 
        SuperscriptBox["#", "2"]}], "]"}], "&"}], ",", " ", 
     RowBox[{"N", "[", 
      RowBox[{"Range", "[", 
       RowBox[{"0", ",", "10"}], "]"}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{"fdd", "[", "data", "]"}], ",", " ", 
   RowBox[{"smat", ".", "data"}], ",", " ", 
   RowBox[{"mat", ".", "data"}]}], "}"}]}], "Input",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"In[40]:=",
 CellID->1830304872],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.6460943569286459`"}], ",", "0.36752274074445684`", ",", 
     "0.361548028034272`", ",", 
     RowBox[{"-", "0.0065441427104490245`"}], ",", 
     RowBox[{"-", "0.001362038154698501`"}], ",", 
     RowBox[{"-", "0.000010134138160481343`"}], ",", 
     RowBox[{"-", "9.359414547998316`*^-9"}], ",", 
     RowBox[{"-", "1.1570193870136904`*^-12"}], ",", 
     RowBox[{"-", "1.9328657867675584`*^-17"}], ",", 
     "1.1572126798838262`*^-12", ",", 
     RowBox[{"-", "1.1572107470180393`*^-11"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.6460943569286459`"}], ",", "0.36752274074445684`", ",", 
     "0.361548028034272`", ",", 
     RowBox[{"-", "0.0065441427104490245`"}], ",", 
     RowBox[{"-", "0.001362038154698501`"}], ",", 
     RowBox[{"-", "0.000010134138160481343`"}], ",", 
     RowBox[{"-", "9.359414547998316`*^-9"}], ",", 
     RowBox[{"-", "1.1570193870136904`*^-12"}], ",", 
     RowBox[{"-", "1.9328657867675584`*^-17"}], ",", 
     "1.1572126798838262`*^-12", ",", 
     RowBox[{"-", "1.1572107470180393`*^-11"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.6460943569286459`"}], ",", "0.36752274074445684`", ",", 
     "0.361548028034272`", ",", 
     RowBox[{"-", "0.0065441427104490245`"}], ",", 
     RowBox[{"-", "0.001362038154698501`"}], ",", 
     RowBox[{"-", "0.000010134138160481343`"}], ",", 
     RowBox[{"-", "9.359414547998316`*^-9"}], ",", 
     RowBox[{"-", "1.1570193870136904`*^-12"}], ",", 
     RowBox[{"-", "1.9328657867675584`*^-17"}], ",", 
     "1.1572126798838262`*^-12", ",", 
     RowBox[{"-", "1.1572107470180393`*^-11"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{552, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"Out[41]=",
 CellID->37859650]
}, Open  ]],

Cell["\<\
As mentioned previously, the matrix form is useful for analysis. For example, \
it can be used in a direct solver or to find the eigenvalues that could, for \
example, be used for linear stability analysis.\
\>", "Text",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->1069698168],

Cell["This computes the eigenvalues of the differentiation matrix.", \
"MathCaption",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->1400391329],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Eigenvalues", "[", 
  RowBox[{"N", "[", "smat", "]"}], "]"}]], "Input",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"In[42]:=",
 CellID->618118437],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.906973351462473`"}], ",", 
   RowBox[{"-", "3.7923175494651775`"}], ",", 
   RowBox[{"-", "2.38894559451046`"}], ",", 
   RowBox[{"-", "1.1243491172014914`"}], ",", 
   RowBox[{"-", "0.28741438736033226`"}], ",", 
   RowBox[{"8.123174953076054`*^-6", "+", 
    RowBox[{"0.00001406983289220111`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"8.123174953076054`*^-6", "-", 
    RowBox[{"0.00001406983289220111`", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"-", "0.00001624634990577809`"}], ",", 
   RowBox[{"-", "8.451044886863057`*^-6"}], ",", 
   RowBox[{"4.225522409963901`*^-6", "+", 
    RowBox[{"7.317794048118555`*^-6", " ", "\[ImaginaryI]"}]}], ",", 
   RowBox[{"4.225522409963901`*^-6", "-", 
    RowBox[{"7.317794048118555`*^-6", " ", "\[ImaginaryI]"}]}]}], 
  "}"}]], "Output",
 ImageSize->{520, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"Out[42]=",
 CellID->36266919]
}, Open  ]],

Cell["\<\
For pseudospectral derivatives, which can be computed using fast Fourier \
transforms, it may be faster to use the differentiation matrix for small \
size, but ultimately, on a larger grid, the better complexity and numerical \
properties of the FFT make this the much better choice. \
\>", "Text",
 CellID->967455021],

Cell[TextData[{
 "For multidimensional derivatives, the matrix is formed so that it is \
operating on the flattened data, the ",
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 " of the matrices for the 1-dimensional derivatives.",
 " ",
 "It is easiest to understand this through an example."
}], "Text",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->1886584978],

Cell["\<\
This evaluates a Gaussian function on the grid that is the outer product of \
grids in the x and y direction.\
\>", "MathCaption",
 CellID->1509361620],

Cell[BoxData[{
 RowBox[{
  RowBox[{"xgrid", " ", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{"Range", "[", 
     RowBox[{
      RowBox[{"-", "2"}], ",", "2", ",", 
      RowBox[{"1", "/", "10"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ygrid", " ", "=", " ", 
   RowBox[{"N", "[", 
    RowBox[{"Range", "[", 
     RowBox[{
      RowBox[{"-", "2"}], ",", "2", ",", 
      RowBox[{"1", "/", "8"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"Outer", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Exp", "[", 
       RowBox[{"-", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{"(", "#1", ")"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{"(", "#2", ")"}], "2"]}], ")"}]}], "]"}], "&"}], ",", 
     "xgrid", ",", " ", "ygrid"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->252012311],

Cell[TextData[{
 "This defines an ",
 StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
 " which computes the mixed x-y partial of the function using fourth order \
differences."
}], "MathCaption",
 CellID->12740331],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fdd", " ", "=", " ", 
  RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1996171141],

Cell[BoxData[
 TagBox[
  RowBox[{"NDSolve`FiniteDifferenceDerivativeFunction", "[", 
   RowBox[{
    RowBox[{"Derivative", "[", 
     RowBox[{"1", ",", "1"}], "]"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{447, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->188067610]
}, Open  ]],

Cell["This computes the associated differentiation matrix.", "MathCaption",
 CellID->1510877304],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dm", " ", "=", " ", 
  RowBox[{"fdd", "[", "\"\<DifferentiationMatrix\>\"", "]"}]}]], "Input",
 CellLabel->"In[8]:=",
 CellID->1042031872],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "22848", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 22848, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"1353", ",", "1353"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{239, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->294894963]
}, Open  ]],

Cell["\<\
Note that the differentiation matrix is a 1353x1353 matrix. The number 1353 \
is the total number of points on the tensor product grid, that, of course, is \
the product of the number of points on the x and y grids. The differentiation \
matrix operates on a vector of data which comes from flattening data on the \
tensor product grid. The matrix is also very sparse; only about a half of a \
percent of the entries are nonzero. This is easily seen with a plot of the \
positions with nonzero values.\
\>", "Text",
 CellID->460110945],

Cell["\<\
Show a plot of the positions with nonzero values for the differentiation \
matrix.\
\>", "MathCaption",
 CellID->1471524656],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", 
  RowBox[{"Unitize", "[", "dm", "]"}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->196196081],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 164},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->5165663]
}, Open  ]],

Cell["\<\
This compares the computation of the mixed x-y partial with the two methods.\
\>", "MathCaption",
 CellID->157355938],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{
   RowBox[{"dm", ".", 
    RowBox[{"Flatten", "[", "data", "]"}]}], " ", "-", " ", 
   RowBox[{"Flatten", "[", 
    RowBox[{"fdd", "[", "data", "]"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[53]:=",
 CellID->818839000],

Cell[BoxData["3.608224830031759`*^-15"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[53]=",
 CellID->121896906]
}, Open  ]],

Cell[TextData[{
 "The matrix is the ",
 Cell[BoxData[
  ButtonBox["KroneckerProduct",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/KroneckerProduct"]], "InlineFormula"],
 ", or direct matrix product of the 1-dimensional matrices."
}], "Text",
 CellID->56799984],

Cell["\<\
Get the 1-dimensional differentiation matrices and form their direct matrix \
product.\
\>", "MathCaption",
 CellID->270094880],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"fddx", " ", "=", " ", 
   RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "xgrid", "}"}]}], "]"}]}], ";", 
  RowBox[{"fddy", " ", "=", " ", 
   RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "ygrid", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"dmk", " ", "=", 
   RowBox[{"KroneckerProduct", "[", 
    RowBox[{
     RowBox[{"fddx", "@", "\"\<DifferentiationMatrix\>\""}], ",", " ", 
     RowBox[{"fddy", "@", "\"\<DifferentiationMatrix\>\""}]}], "]"}]}], ";", 
  RowBox[{"dmk", "\[Equal]", "dm"}]}]}], "Input",
 CellLabel->"In[16]:=",
 CellID->609028615],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->910250617]
}, Open  ]],

Cell["\<\
Using the differentiation matrix results in slightly different values for \
machine numbers because the order of operations is different which, in turn, \
leads to different roundoff errors. \
\>", "Text",
 CellID->91285780],

Cell["\<\
The differentation matrix can be advantageous when what is desired is a \
linear combination of derivatives. For example, the computation of the \
Laplacian operator can be put into a single matrix.\
\>", "Text",
 CellID->477062099],

Cell["\<\
This makes a function that approximates the Laplacian operator on a the \
tensor product grid.\
\>", "MathCaption",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellID->279471931],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"flap", " ", "=", " ", 
  RowBox[{"Function", "[", 
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}]}], "]"}], "[", "#", 
      "]"}], "+", 
     RowBox[{
      RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "2"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"xgrid", ",", " ", "ygrid"}], "}"}]}], "]"}], "[", "#", 
      "]"}]}], "]"}], "]"}]}]], "Input",
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"In[18]:=",
 CellID->111513315],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    TagBox[
     RowBox[{"NDSolve`FiniteDifferenceDerivativeFunction", "[", 
      RowBox[{
       RowBox[{"Derivative", "[", 
        RowBox[{"0", ",", "2"}], "]"}], ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False], "[", "#1", "]"}], "+", 
   RowBox[{
    TagBox[
     RowBox[{"NDSolve`FiniteDifferenceDerivativeFunction", "[", 
      RowBox[{
       RowBox[{"Derivative", "[", 
        RowBox[{"2", ",", "0"}], "]"}], ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False], "[", "#1", "]"}]}], "&"}]], "Output",
 ImageSize->{499, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.1.4", "ndsg:2.0.1.4"},
 CellLabel->"Out[18]=",
 CellID->253463560]
}, Open  ]],

Cell["\<\
This computes the differentiation matrices associated with the derivatives in \
the x and y direction.\
\>", "MathCaption",
 CellID->1659560836],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dmlist", " ", "=", " ", 
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Head", "[", "#", "]"}], "[", "\"\<DifferentiationMatrix\>\"", 
       "]"}], ")"}], "&"}], ",", " ", 
    RowBox[{"List", "@@", " ", 
     RowBox[{"First", "[", "flap", "]"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[19]:=",
 CellID->1181211483],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"SparseArray", "[", 
     RowBox[{
      InterpretationBox[
       RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "6929", 
        "\[InvisibleSpace]", "\<\">\"\>"}],
       SequenceForm["<", 6929, ">"],
       Editable->False], ",", 
      RowBox[{"{", 
       RowBox[{"1353", ",", "1353"}], "}"}]}], "]"}],
    False,
    Editable->False], ",", 
   TagBox[
    RowBox[{"SparseArray", "[", 
     RowBox[{
      InterpretationBox[
       RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "6897", 
        "\[InvisibleSpace]", "\<\">\"\>"}],
       SequenceForm["<", 6897, ">"],
       Editable->False], ",", 
      RowBox[{"{", 
       RowBox[{"1353", ",", "1353"}], "}"}]}], "]"}],
    False,
    Editable->False]}], "}"}]], "Output",
 ImageSize->{480, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->184290304]
}, Open  ]],

Cell["\<\
This adds the two sparse matrices together resulting in a single matrix for \
the Laplacian operator.\
\>", "MathCaption",
 CellID->130195626],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"slap", " ", "=", " ", 
  RowBox[{"Total", "[", "dmlist", "]"}]}]], "Input",
 CellLabel->"In[68]:=",
 CellID->1350152860],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "12473", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 12473, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"1353", ",", "1353"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{239, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[68]=",
 CellID->76316015]
}, Open  ]],

Cell["\<\
This shows a plot of the positions with nonzero values for the \
differentiation matrix.\
\>", "MathCaption",
 CellID->2016780604],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", 
  RowBox[{"Unitize", "[", "slap", "]"}], "]"}]], "Input",
 CellLabel->"In[69]:=",
 CellID->712030833],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 344},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[69]=",
 CellID->80025461]
}, Open  ]],

Cell["\<\
This compares the values and timings for the two different ways of \
approximating the Laplacian.\
\>", "MathCaption",
 CellID->2029520936],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"repeats", " ", "=", " ", "1000"}], ",", " ", "l1", ",", " ", 
     "l2"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"data", " ", "=", " ", 
     RowBox[{"Developer`ToPackedArray", "[", "data", "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"fdata", " ", "=", " ", 
     RowBox[{"Flatten", "[", "data", "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"Map", "[", 
     RowBox[{"First", ",", 
      RowBox[{"{", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Timing", "[", 
         RowBox[{"Do", "[", 
          RowBox[{
           RowBox[{"l1", " ", "=", " ", 
            RowBox[{"flap", "[", "data", "]"}]}], ",", 
           RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Timing", "[", 
         RowBox[{"Do", "[", 
          RowBox[{
           RowBox[{"l2", " ", "=", 
            RowBox[{"slap", ".", "fdata"}]}], ",", " ", 
           RowBox[{"{", "repeats", "}"}]}], "]"}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"Max", "[", 
          RowBox[{
           RowBox[{"Flatten", "[", "l1", "]"}], " ", "-", " ", "l2"}], "]"}], 
         "}"}]}], "\[IndentingNewLine]", "}"}]}], "]"}]}]}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellLabel->"In[64]:=",
 CellID->560718326],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.13999999999999985`", ",", "0.04699999999999928`", ",", 
   "1.3988810110276972`*^-14"}], "}"}]], "Output",
 ImageSize->{194, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[64]=",
 CellID->140848314]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Interpretation of Discretized Dependent Variables", "Subsubsection",
 CellTags->{"c:5", "b:7.1.5", "ndsg:2.0.1.5", "PDE:1.2.7"},
 CellID->1602750198],

Cell["\<\
When a dependent variable is given in a monitor (e.g. StepMonitor) option or \
in a method where interpretation of the dependent variable is needed (e.g. \
EventLocator and Projection), for ODEs, the interpretation is generally \
clear: at a particular value of time (or the independent variable), use the \
value for that component of the solution for the dependent variable.\
\>", "Text",
 CellID->1728118660],

Cell[TextData[{
 "For PDEs, the interpretation to use is not so obvious. Mathematically \
speaking, the dependent variable at a particular time is a function of \
space.",
 " ",
 "This leads to the default interpretation, which is to represent the \
dependent variable as an approximate function across the spatial domain using \
an ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 "."
}], "Text",
 CellID->716955600],

Cell[TextData[{
 "Another possible interpretation for PDEs is to consider the dependent \
variable at a particular time as representing the spatially discretized \
values at that time -- i.e. discretized both in time and space.",
 " ",
 "You can request that monitors and methods use this fully discretized \
interpretation by using the ",
 StyleBox["MethodOfLines", "MR"],
 " option ",
 StyleBox["DiscretizedMonitorVariables->True", "MR"],
 "."
}], "Text",
 CellID->192565538],

Cell["\<\
The best way to see the difference between the two interpretations is with an \
example.\
\>", "Text",
 CellID->571069100],

Cell[TextData[{
 "This solves Burgers' equation. The ",
 StyleBox["StepMonitor", "MR"],
 " is set so that it makes a plot of the solution at the step time of every \
tenth time step, producing a sequence of curves of gradated color.",
 " ",
 "You can animate the motion by replacing ",
 Cell[BoxData[
  ButtonBox["Show",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Show"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  ButtonBox["ListAnimate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ListAnimate"]], "InlineFormula"],
 "; note that the motion of the wave in the animation does not reflect actual \
wave speed since it effectively includes the step size used by ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->1396399276],

Cell[BoxData[
 RowBox[{
  RowBox[{"curves", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"Block", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"count", " ", "=", " ", "0"}], "}"}], ",", 
       RowBox[{"Timing", "[", 
        RowBox[{"NDSolve", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{
             RowBox[{"D", "[", 
              RowBox[{
               RowBox[{"u", "[", 
                RowBox[{"t", ",", "x"}], "]"}], " ", ",", "t"}], "]"}], 
             "\[Equal]", "  ", 
             RowBox[{
              RowBox[{"0.01", "  ", 
               RowBox[{"D", "[", 
                RowBox[{
                 RowBox[{"u", "[", 
                  RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], 
                "]"}]}], " ", "+", " ", 
              RowBox[{
               RowBox[{"u", "[", 
                RowBox[{"t", ",", "x"}], "]"}], " ", 
               RowBox[{"D", "[", 
                RowBox[{
                 RowBox[{"u", "[", 
                  RowBox[{"t", ",", "x"}], "]"}], ",", "x"}], "]"}]}]}]}], 
            ",", " ", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", 
             RowBox[{"Cos", "[", 
              RowBox[{"2", " ", "Pi", " ", "x"}], "]"}]}], ",", " ", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "0"}], "]"}], " ", "\[Equal]", " ", 
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "1"}], "]"}]}]}], "}"}], ",", "u", ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
          RowBox[{"StepMonitor", "\[RuleDelayed]", 
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"Mod", "[", 
               RowBox[{
                RowBox[{"count", "++"}], ",", " ", "10"}], "]"}], " ", 
              "\[Equal]", " ", "0"}], ",", 
             RowBox[{"Sow", "[", 
              RowBox[{"Plot", "[", 
               RowBox[{
                RowBox[{"u", "[", 
                 RowBox[{"t", ",", "x"}], "]"}], ",", 
                RowBox[{"{", 
                 RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
                RowBox[{"PlotRange", "\[Rule]", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"0", ",", "1"}], "}"}], ",", 
                   RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}]}], ",", 
                " ", 
                RowBox[{"PlotStyle", "\[Rule]", 
                 RowBox[{"Hue", "[", "t", "]"}]}]}], "]"}], "]"}]}], "]"}]}], 
          ",", " ", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
             RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
                RowBox[{"\"\<MinPoints\>\"", "\[Rule]", "100"}], ",", " ", 
                RowBox[{
                "\"\<DifferenceOrder\>\"", "\[Rule]", 
                 "\"\<Pseudospectral\>\""}]}], "}"}]}]}], "}"}]}]}], "]"}], 
        "]"}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->349921476],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", "curves", "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->116827693],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 105},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->223730093]
}, Open  ]],

Cell[TextData[{
 "In executing the command above, u[t,x] in the ",
 StyleBox["StepMonitor", "MR"],
 " is effectively a function of x, so it can be plotted with plot.",
 " ",
 "You could do other operations on it, such as numerical integration."
}], "Text",
 CellID->1003404637],

Cell[TextData[{
 "This solves Burgers' equation. The ",
 StyleBox["StepMonitor", "MR"],
 " is set so that it makes a list plot of the spatially discretized solution \
at the step time every tenth step.",
 " ",
 "You can animate the motion by replacing ",
 Cell[BoxData[
  ButtonBox["Show",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Show"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  ButtonBox["ListAnimate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ListAnimate"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->1086211594],

Cell[BoxData[
 RowBox[{
  RowBox[{"discretecurves", " ", "=", " ", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"Block", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"count", " ", "=", " ", "0"}], "}"}], ",", " ", 
       RowBox[{"Timing", "[", 
        RowBox[{"NDSolve", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{
             RowBox[{"D", "[", 
              RowBox[{
               RowBox[{"u", "[", 
                RowBox[{"t", ",", "x"}], "]"}], " ", ",", "t"}], "]"}], 
             "\[Equal]", "  ", 
             RowBox[{
              RowBox[{"0.01", "  ", 
               RowBox[{"D", "[", 
                RowBox[{
                 RowBox[{"u", "[", 
                  RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], 
                "]"}]}], " ", "+", " ", 
              RowBox[{
               RowBox[{"u", "[", 
                RowBox[{"t", ",", "x"}], "]"}], " ", 
               RowBox[{"D", "[", 
                RowBox[{
                 RowBox[{"u", "[", 
                  RowBox[{"t", ",", "x"}], "]"}], ",", "x"}], "]"}]}]}]}], 
            ",", " ", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", 
             RowBox[{"Cos", "[", 
              RowBox[{"2", " ", "Pi", " ", "x"}], "]"}]}], ",", " ", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "0"}], "]"}], " ", "\[Equal]", " ", 
             RowBox[{"u", "[", 
              RowBox[{"t", ",", "1"}], "]"}]}]}], "}"}], ",", "u", ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", " ", 
          RowBox[{"StepMonitor", "\[RuleDelayed]", 
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"Mod", "[", 
               RowBox[{
                RowBox[{"count", "++"}], ",", " ", "10"}], "]"}], " ", 
              "\[Equal]", " ", "0"}], ",", 
             RowBox[{
              RowBox[{"Sow", "[", 
               RowBox[{"ListPlot", "[", 
                RowBox[{
                 RowBox[{"u", "[", 
                  RowBox[{"t", ",", "x"}], "]"}], ",", 
                 RowBox[{"PlotRange", "\[Rule]", 
                  RowBox[{"{", 
                   RowBox[{
                    RowBox[{"-", "1"}], ",", "1"}], "}"}]}], ",", " ", 
                 RowBox[{"PlotStyle", "\[Rule]", 
                  RowBox[{"Hue", "[", "t", "]"}]}]}], "]"}], "]"}], ";"}]}], 
            "]"}]}], ",", " ", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
             RowBox[{
             "\"\<DiscretizedMonitorVariables\>\"", "\[Rule]", "True"}], ",", 
             " ", 
             RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
                RowBox[{"\"\<MinPoints\>\"", "\[Rule]", "100"}], ",", " ", 
                RowBox[{
                "\"\<DifferenceOrder\>\"", "\[Rule]", 
                 "\"\<Pseudospectral\>\""}]}], "}"}]}]}], "}"}]}]}], "]"}], 
        "]"}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->211200568],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", "discretecurves", "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->31329639],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9XAmYFNW1PlXdPT09Mwwz7JuKrCIoRFAT3CASISGoJGoUUUSGRVxAkRf0
KRoZxQ2NK2pQRNHoS1zI0wAKfs9EMEETRWQkKvpIIL4gIhECA8PUeef8996q
6uqeUcyX8GlvU111lv//z7lL9ajxV06ZeOn4Ky+aML7r8CvGT59y0YQZXU+Z
doV8lPKI6AX5/9IOJK9ZXuH/F5jlufAxK48pOcDXd/qC8Y98efk4jihN/CWl
D+noPdFj5illnvSf7942+72MvfZDX/E6mcQR39aHFnzLXM4Qp4lLPPuirIS/
cST3PYw7tOHhxDXE56uHNUQ5nu7zTcTziG/x+AHixcQLPX1cVE1DvqYd+F4H
tUOuXd0Cpvg88wprxKKFvGQujyG+qC1PhDWXk7XhVnI2EC8h/h/i34idbxKV
8HrizcRbiD/y8Fg9+Gtad5I+lERRkuCcPESMSpnoiWnjYJT8N9sY5RUaZawq
4/eIdxAHtK+B/sG0m2lrQOuZ3mJaJ5iRJ0q9z1l75UEhwhQPJbHnuH0nhPbF
o2ftK7Ghm5DVoF1PPMfzxUZKq4lPwMpXjG2t1TZEbN8ea9vf2Nr2DtNvmV5k
eonpKab7mO5kuhuPPxMEDoSd7a19J+q7DPfoGg+WiXN5FK0whbe7FC7xebWN
U5o3IHWf076AfLGEyj4sZshDTPOZ5kjkZsox6WlME5guZrqRs0eFNsVjdTyy
bWN1cKciUTIAqzVRasEP+jaFq4j/QGIb17kMNmp8xKwNch1kLv3bmFH+nWq1
mHYJ04Ww60ym7zCNhoHyx5U8ZUBRGweb+MXzWVFaEL8QbfN8G79fpcP45dRK
id+OlMTPJLIO8fut2irho/KnA9+k0Zgo9k1mOkO5KAZInOS4wYrGobD6LHnZ
rLXFst2dp1bbeF7h822w9gFKxvM938YzsPF8z+X410wLHNLEypkw9AzYczKs
PILpIKbOamsrtbULPj2pSVu/VTz7ZU5ecpYjN6dtSJ/I8GuJiKabimjlgsAz
Vk6LZXyIM7S9XscnzqQ5rU+UynJKrO6SZ63yZ7CRJRNRl/nuUeane5Y580MG
+18ay5C1YbprmIa7XHdlElPaGBNhXZa4NXF3/NdePhBjfyRM7180rsfGETtq
ZBwDpREGap0sPh5qTstmomqYdH8s9eRPUOSOCNTqwbA6DCnM7UPc3+N+6kF/
0XR5Jc/csy83Z3cy0ofxvOFq7qS0RcMdJfyYE6g38iXgK4Q5gYFW1mYT3J4w
8CTi0cQ/kE/PVatH48P+7Zuy+pgCFKP36KMImZCvrfM9y7bX080oQmGkDcmG
OIZ1cIH2uJWJMvGJxGcRX0g8lXiyRnyq2l6jPvCppZxqxvZkxPsoRuSbYvzN
JRFGXjOQ/jqxzkdGVxgsYPg+rBtP/B/oHO7SGPn6LAfNhSc1HveMLFc2Hq3v
eiSwrb6EmmFk+I60Rckrni1ef9ES8aWhTnBQzAb/qp3Z/WG2ifMNsPlnxD8n
vdiTePEIPhTzT24CL4OKxbyvjbmISUKY3/B4k+kCvC8N+IiYBhdDyAUwe6aL
9kPSRhOvIV6rqfX1WUx63blxYXH7BxbH+yDlaUJZlqSkBCpq1nvSxDQV+xmu
3A1vkpiH5YOlFmJr4v0qzN9KvId4Fx53kkZsLTy5arBSxzuiaU86hpmIUFTj
J7XmNYeiHbQHvdiGgizMcIqYHVGEqF2NClbmw8dgJ0wCvNjdQNuYdjT6+lRC
u/YRb4NHt5nyVNyXb5h2qpAXIa5CCXrFENkLifyhTYmXSElNLCUVSf4WhZNJ
xofEf4cn++hjpjeZXmdaw7QaL9ayS9LDpo5F/ii/j0qwA9hqHWlpiK3nyar/
JsvrD5vOSLcRjs/leWIfYsow+UGkYhWAszVy4B2ER078CMLzMB4XM/0Of9rU
iOO1sDWTmRQvW2rrsB9qbCHG6gzTSw3TXysgSaGixgheCCxJx0dAz3aLqnfh
hy8jYaqgW5guh3JMx+N/wKnXkTCFm9S9k41HbsQTHwG7UXFy9IPe6QSLw6lT
8pS5NonAhCgsQ6duEheWvK6RFoSFY0wTwNvq3Aws2CR8t8AzEffxTKOYhjF9
F4+n4nOTSeXYVI2e8Rc+5Y1Bswkvk3MBA4yGx5Uk5N18B9c6O4bami/jj7DX
dHK7unbkvAKQbohYts4566tDGbqG6YdwUupYN/20HVNOi0I77VzVdZPvjxha
epX1OznjYYbesTAUjYO+7W/kNFYT8hgb4ttUskZVT1MEFnx5vovCemue50sd
NxOOi7tVnEtJBITylGvpK/XLMwiCgYBiXc644GtnHp4flcy86UEeSCL9Q/ZM
zhe4juOs4jk3wnSFA/izUYHY3Gj1yCjDrVDpMzHKgsO5Uj6YuC/xAOLU8er1
scTyfJRx/AjtshXyiqDlTXqd4eJzG/1N92W07LozlIhXZhIIF1c/zE+vuHoO
0jvUpTfD1U61xsQqumHx/0UsfpGtVl3j+uDDbFY7+tbJ07Xf5eni7X+qt7OI
5XkG8TdILyNHKtB5rSfVojlv414m3+N7R+dpmmF3LTloq8RKMd0K0Y4n+ByX
4Oq8bmZMcURv3m/L5VMcufwtpK1t5PI4+HultOdurka+T/4a9X8lProdITge
RyvMeY8nBuaruXsu6m0M0ULk6ShU91ME50bbAz0NFNa6BA9wjUIx5RIUv6/V
aPfuSLCMOk9iOrG4k7UIz0sosX8m9qXD86XgUlngGr4tOEIgcCaB1wKcXQ35
PcWRjP7UVWBNnilFzyUh+xzTTS5tA9DCtcgToitg0X0RUhN0nOocaaeyI9nu
6xVx5HOFCu9XtLyFzuhln+lJ6U0UOX+UTwKq24I83mpYi7ZpBGepX9EOo18C
nyZj8y0y9+3O647mFs+VUZzrHSDX0+56y8Gn8pW1D+eySc+WQp82EtfDs0b6
gOkXIMG9TJcyXSAqJcpJF4jGaYzEhnvFx3fRwilRUYa7cPP+hYiU/N3jsPg3
W05fgxaaMVxB5sbEIPiJ5Zk0G6Is8yL9zJUXd2ubJbaE4hU4NBdTSkO0s1M5
aqlFQcqrrzaX63WpFxg/Cl7yP8BFU3FlONucj/0Vo6O6qb3XpUJB/bRRuxkj
KpehUxiUlzqjJfdYUO7eGbHLSMhQTCVluI8fuSaC9a7VrA8AwV8yXYep0B9q
iqmTyLTxKK2TRuqh/JfWJ82h+aQaPsqX60RvbEHl0cX9O1wfjowwagR0fgTQ
12PNwKAogcdCQ6bh0BguVwR0rdPH9upcxxSPhAzcUMS5hxwKh6rEIF3Ot0zo
Gz7JcFknuJc2H5oYZOGrBOeaN0xVkXf841gmVWX6xnvAAWm+FhKzUiG6EURf
4Lr4HrYASin4Hjh3F9CMClDXaAl3tq11HdNfxbEjEzjMOQ/UCfZKOZXj4aN4
yLd54aNmBFLBS5fp++OGhc4i7b0bqU4U+GBPXFxQPJd99KFX1O1NKLdiU0cb
A3rUZbFHVOoTEN2lcDOiMlTlUqjXD+Xq+jwffyOqyHSDLowp19oV4tF+kuvE
qTKuaivOlaltOT7/Yk630OGeHiYv7hXYT5Tcifw241N3zZ4ZXwj/VqrAfLqP
no/lLcWHOEW5B4DcQnWBYvenUJFhnGtR1JXnkf7Lkag8biXwJ0WjO1e15wlT
pXMRi7Ls5Vxyjn8JV6Rd9U340FYe0nbtUxFZoR9Wqaacl1a2/Rft22mpNtOl
Bw5dpr29+CuJ2YZvUqV605H7eHmu7FPYPQQFPAu8C5mUjXuCDGW5S99w3VSf
K9WaUq6t1b/pcVUBCriUNo7Gl/rclqM13PjnqaJet0PLFo5K2Y5NE0ehjz8k
0p+7UwrYdVprRVl/nJfc2eqs/Kvbr5oo//qpdspzfiBqIQn9E8zL5jFPHPZL
NZuwpyTedZrHpJ0VNutZdwKBzdM0q8CnYj625mRnW+wbuqDu6XOeenVo9rth
BCusvl3oGXpsbNTYGW54PIL4bsGonoZyHUsQMk+bPnmxX0M2A2LVsUhlsZ/4
7RXwZdXC5MyBRax1e+CqG6TiBOr5L41Y/BwdvuQaWQNS033eQ5/u1Xbldsc/
j88mXqFQe12/LfIhj1M9LQbyAsLxHT1XpH4FipHpqFp34vdF9g4oYsYiHU4p
GQd4+Jvvjs+LStt/Q5wwLqtUho6H5GyhRagkbbQNuBRdy2a6O1D5zLBECDqV
DZQgnRChnD2diAEiUxJJUXUPU/rU81RRD6G2GSdN6Sfp34aftl8xazY+wr/l
yrzHAhMZ6QHvoroGhc8wNsAJ9L8ZiE9ZLCzp2AtEpffgVhxpZ3j5gitXmtgo
w1ozjQjopX+1Grlnp/xAdjnXVq5pVLmG4y9rK3FJMMbTlnavqkvnUFcin/UT
RCKlXdCEqS7iDuf6XG38Ez7oHFlKeWmZEK9IyUrUdEXqYUiGWvNpvSQOZaWu
oU9aq2mjctpcJ5eKcApOe23VzZb97Cy4Q2MT9R7XyemOg8WqI7nKQAYwgXdK
2OQWVqSsXsIrl9SblbQUp/vqJb1uzV0j5QFggZcf4bgKBcb0dG+zqh+b7y5+
3u5Ah5zRnSfj58fCnjDXB/tXUu7IVmYkIRBgGaq6Kcrk2dOJ98k5jqb/Xhi3
OGUCr3fBDGDyXF//2v/E+ZRngfdNiZiY+M9alHxfUjTGeJ/l4ae5DOZc/NIh
StqEYWuT+HbnxPu2BjWuPyuIcqevZFMbY35513xUlUWoIqCKejV4y+lN9jbu
3vG59z1zbd80tNVmLFBw7pbmqLk3g6gtjI9VhX2vuURAvT/2dNaCTgs8nnuv
jLAbSehgrxUyO7qW6kkL89deg+I4bJXsKPUa4uExAY2Wa1SmONuBaeNu4kk+
nzY/wMrarWUjTap1NqRc+Wbi2ruobzlz3Ta94tdtG7tuC3dd6atmBPTTZR4f
IwNYUYHbAmxTe72cO1+EwRp/QHbtzljQoHPz+h37aXELMiZzMTTBhg4xG1rr
FDm1Ceg4pkX7abm1oURDrNOmG/dgE+ThYslPmP4S0A4ZXczyjAwZW54lflsV
+CDksFUTtpjuwEUjRFHHmC1tENSDA509ensPbRJbziU+wcds7UKmXweIw+3E
M1ty5UWYXNncgNm/F4lrhtIogPhkjy8mDtEgCeqVsAhwM/k4pMgYyxrSHdMh
aoj0BnLG0z3MhY7FQG9jPWbKXvD4tBJOd2B/su52o7clVb/HHPLVyIuxiPqi
P6SB1qhMplceQlPWktAABSLTqICeDOjjL+jz5TozwuM9zgp+vod+/8WA/r6f
6rcJ/gWjxKeJDacyTWRazrS3kRq2Y3J5SgWf2VErhrVkljrA9IwrNunOBiiG
HWZc7LWOK0F30ymzHY5m0WTLZWY10vJ9Ol3LT8O6C4nbVICcIzAFreBpxLr1
o8S3eXySidNETPf9nGlHPTK3HKEyPDpVr1fqLQsk29toJ66a084Bfzb9stWK
CM2paF6pwOqRGLtIKK9qoDVxYycajB+sk/I6gBJ0vRBQ/XYkdSkOkXFX/3Yu
pj/HMu/mAFB/hPinxN+tsNwzVtP2ffR+8LwCbyiu2gZGtWyV5GCyJoODAree
Ceu30yeCp4A25RmdViQpVsYxCSEfkCALA77A4vLjpPNjk1LcWULdnv3RTPew
NvybzPaAzdjecZVOTUSdBKwvwYDyi3HSMOpqsQQka2N7+EDL37K4UpdWF59r
yapflsZYHJ4nEI75daj4VYtVrQfgTk8fSlsFUF+O0f9qg+tdyIQg5w7iM4gP
T3PnHpgfHo1pIaHaG0wNuxyEaj2emNNGUFB0Omqop+FY0c2Trv0T8PIIRUgq
4VU6galUBfxqWVgd0F4aMM3CFPeivbRpH+Iewx264kNEQXsLVysw6ToWewGW
hITY7DB2DXh7hngmNbInPLsWzq0WoRV+BDozq2C7HxkLd2/AvxwSOS/TTgm2
T1E3W8ILMc+43Phl+V4VjnNbhkXIac4pAV0NzCypp02fYmvafQBfTZ6fyKSu
jx6M2bex2E7xDFYM6nc6PD4FrRZ2D/O4c5YrwwTeyrSCaVUcmMuNaMq4Y1C4
3G08rfB4lEZqhE6DPK+jsFU4yTEwOR3zJh3PGtDYyq57Kya/CSJLzVTnAloj
yhn3T9B4mMc9PG5TxjE/O+tQ6m6AU8rws0wfGHx+jlnoRzFjOQVicbR4aBI5
CpOwN2F5QVTjz3Gg3m/vMYjtxhmt1+nlcTuBOV29X0j4Y6zAOSqe7hLbyqWq
VQySh+oygG6/MM4tDjVDhGo1sJPnXBopO1qNjHnZVbP5UaCcWsu0tcGhdAX4
pyj1uEuG0xXs98Cy+CiwIFQXrTe7AFezrXF2ji+p4kcmhDIDH1vKKElOQouD
bintMn5NITnnY92hBok9GrT5rqJ4gvZLhm73B7pXuohvQo2hBN9SWOEYhFWT
6dD0Zwu42cWpy7sOn7ONwBB3KUe3ahJ4Ea67KFQaIePHcHC5o6QQwjXWPwRO
z+Vu0tx4q4JMhis9noMk7rM+LsPJFopPt1GHQPc7KQwbdW7/40b6/K8FPg0j
tjJSFWOZgeFzVkxiGTxYnVNV+QOAeS/OcRbx8GYSZ3Rmm3DwM3Qtr8nlPUtD
yZ0D6Jl6/h7pz3ZLZyfFZxwS97bu/eQGXXHl3bTTePEFPpHg/gkHvEL8k0J3
Kp07Y+DOT5AlC7y9kI4/x/zqAL9uclS7lPReHU1Wy2LJClUlAuRyLFDWGtap
3IeYhF9lnqjA+Vp/pFRchcL+hNsYIiHx3xHZ6/Aq9o0+VtSdchTkTqKcSNBU
BHdhXCp24WRCpaeNX6Xc9M4vdOwH4Y+ng3jSAA82Kaxiv39BCldFCuNZgC7z
LUBnVloShuO/s5DJnLh3SyCxz1TqhsNzIQ83wPWbMdcscb6uGUdPRrU+B0hc
CNVfz9L60FaBwl8dAB/EQt1s4zFma5vdAeS7XOtiri72HEZcJR73xFArTPF8
l+INkmLDynWehe7jWNmvVWKaObFIYY3fJdJGPQHqDeKqNPcQt4hPIR6L1VAJ
w0gsVFh3s87dIc7d67Eh6nGYIPT7k/SLu4HXLfB4EZoaLeoen3Egfutb6uTD
33iG74a/v8CGyPcMT+tB1Tcdqu8HZKf5UZLdJlXjsScweCXQdcHzQbpy3Wza
3lPXu+Cxk3G0HD3oAJSR0Vj9FF/viBOzAZtLtwPJK1GG70SzeRn4KPnqfUCZ
bivvD8LEXKf8BN8PVP/CadNnRYFdSzE2uySfja4vLe3GR/vV6pswDhmEIZuv
o4jOts+Oezo1JkFrmH4fcnZvAZSv0/sDraIeYvoZaspnt9Mr4zqIkzjXE2so
o7DNIcys0yvP6lUCzDdKgD1L4phw0dm2wz2f9u6hvzfSy4HmbCzWt33dXtxZ
nUu4+IRjq3BHvlNvhkNbsH8ohO9siM55xMfFS2Iv9pr1tWCeS/M7gStGwaLL
kAyT1lXsWb0KcO3NMTTP9+zYZHzGMThUrnOA56xW+g+Vc5Kpu4DSa3H+aVht
TPi5NkHTv2Lrfxy9M4h/JAM58TZeJU9QifPztdrt60qb3ptoJpdMxqVvjBUe
3CWHO/mUpga1K1sqau/DEKUGN1+8ujCpT+dYfeqld1HyH7UT+2948QSWKh+O
OfUmKCkXUogaSm4oAOp4A1TsBsxrbIYCgec478KuqMU1zpFVMbmJiqgB5WLl
3TzcH1uiw/3xmvvkHVHwRUaPLYh/qS3A3npdmhOb/6DY/L1ic62TlU8a88m2
Gt+JI3Gc0xdpX6viGfo2oDVZU+9Rol94iqvD6rgh1JGdpTYjqzwnn65iTPS5
dojkJa4n5yIn0rGiSVyk6dy7V4kjZe59TAXsanDWJ3j0IBqF2dhMPdbgi3S5
orPUtEQqzgZ2zWgh5oUibB0PXAEdtMkwnNls737M8gsdLLBuTPIm7z4S48cw
9jxtnmajeftf8OFTAOgfzvwPQA/p2X4VQ1KNa0KsHpSyn9GF0WQWDBEQ9VTM
jwrxYy2n30KP/5nxoB6gXe/ZAdFisvfqTvPsLZ+F/BgLTC3lOSp5um1H2ttl
QMwavQPkLSpz91quxDxtEkFyzBHUMuR4JyRggNZ1L54DMMBboVDtlMzGJ0HP
SLg+80Ioka4TycEhlmqsYJvuLtLs8/R991L+Drh5I/ZeiZ3PuJA/HIOOkaYQ
Or1D6JiCGTYJNcnYC1w8E+zEuAeqVK6BX+crmyVaEX78CD+phO6OA6fKtFfo
h21uN2NEeI+L8U0xyfkR+igDlJ4hXQ91wR7FHu4byc3MD/Yq7ON6V3lalYz6
/vpTIhEKC4XToTRfng41NZXADCyvKFEd6gzbLgciZjli1uSjo0rQkeZ0ijOG
oT3cbbYmzKY7uTMWZnMfu5Tu3bh1KW451p23VEf17WW/ACVkBDRVwFfY3bbS
jJR1p94YGDremTuyEBVtsDbZAxv0T4DF5E+Shwpj8p0Fkd4EkzWye/vE7G6j
dq+vsNhWwfdsxxVC5IJ0pJT5N29fgF7ncOxgwlzGt2DoUGdunwQuQhImhPBG
zA+HYTZ3nZeYOO/RdtDXuG5JJSpYmUW19hEltndKoqPgDm7Y3P1cLhmiuw9F
IdvDyO6hqXFEdHHEG5Yveq6ZayLCuwACCWmdH8dIucb6j+MV0c8gxHc4FXfQ
oEJsXKiftNzASx/gIYfoFgIFiUhyaGYxJCjnsmFsYw22jW0uL7abUQtMyV+d
KUD109V2hmee+f0AzGQ5UFCSgxP1/aEDMU/VFsgwBh5akPtLAqXYdKdkD8a6
4yYDqvrrawBXDU3qBs+vcMiVmPoa0HFeaGHBXdyws+QHWKDpifk0Y54v2aZM
It/mVw/SizC99yJM9Ncq24yJReIosHw+jVvt47WxVO28s73r881tPUMcVtPR
b3FUlE5G/VvAJ5+LeI2MwdAwvcPMAu68GmtKTQe0NyiMnxdKqsZrfrIvTEeY
tBWtyG9wTAHz7+FuJoOTcetBwqBEQtex9lDbYBMKKnqSTbGAPWcra8pq5tw8
LGJ3zaS0bRgemUCRbfm/X3Kxfl62jAfchQHDgzFjXsJKmcteK8levQwVJHt7
XI9S5+6GXOHC9FAoiGDrtZ2iUUJTv4+TCUdMS+aYzsBvIo6X4dsf8PkrOb06
qCBfLKR2plN+B/eVSR63xw183+VxtZteftzNTs5zA8hpGCsbO3MJ+wp/FyYd
apP5Z4wcfGwirHSFHtdqfzBnL1ftbexZH3gSvIZdGZ3N/xSR+whTrK8S/w6D
mmfxSzGL1Db9mQwx6Fq3FDU+ZeJyQDaC5ca68Hd9XCDpaj2ikj9bym8P4dXH
8XKff+nxzyrUhEeFdJ7G5jYKTfD1p3/S2ped0s00PgdiC+ZvMtEP+pgIzUH/
oUmfnOPJJXz9QJ5UymNL+OLhertFaHFF6Ve91qIDOs6NXN0v6pjfXyr8bSfy
/h/FaMoC\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->188891842]
}, Open  ]],

Cell["\<\
In this case, u[t,x] is given at each step as a vector with the discretized \
values of the solution on the spatial grid. Showing the discretization points \
makes for a more informative monitor in this example since it allows you to \
see how well the front is resolved as it forms.\
\>", "Text",
 CellID->1026920681],

Cell[TextData[{
 "The vector of values contains no information about the grid itself; in the \
example, the plot is made versus the index values which shows the correct \
spacing for a uniform grid.",
 " ",
 "Note that when u is interpreted as a function, the grid will be contained \
in the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " used to represent the spatial solution, so if you need the grid, the \
easiest way to get it is to extract it from the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " which represents u[t,x]. "
}], "Text",
 CellID->2014625136],

Cell[TextData[{
 "Finally note that using the discretized representation is significantly \
faster. This may be an important issue if you are using the representation in \
solution method such as ",
 StyleBox["Projection", "MR"],
 " or ",
 StyleBox["EventLocator", "MR"],
 ".",
 " ",
 "An ",
 ButtonBox["example",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveEventLocator#28531"],
 " where event detection is used to prevent solutions from going beyond a \
computational domain is computed much more quickly by using the discretized \
interpretation."
}], "Text",
 CellID->658922218]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Boundary Conditions", "Subsection",
 CellTags->{"c:6", "b:7.2", "ndsg:2.0.2", "PDE:1.3"},
 CellID->1306392612],

Cell["\<\
Often, with PDEs, it is possible to determine a good numerical way to apply \
boundary conditions for a particular equation and boundary condition. The \
example given previously in the method of lines introduction section is such \
a case. However, the problem of finding a general algorithm is much more \
difficult and is complicated somewhat by the effect that boundary conditions \
can have on stiffness and overall stability.\
\>", "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->2126138452],

Cell["\<\
Periodic boundary conditions are particularly simple to deal with: periodic \
interpolation is used for the finite differences. Since pseudospectral \
approximations are accurate with uniform grids, solutions can often be found \
quite efficiently.\
\>", "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->818805542],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      StyleBox["NDSolve", "MR"], 
      StyleBox["[", "MR"], 
      RowBox[{
       RowBox[{
        StyleBox["{", "MR"], 
        RowBox[{
         StyleBox[
          SubscriptBox[
           StyleBox["eqn", "MR",
            FontSlant->"Italic"], "1"], "MR"], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         StyleBox[
          SubscriptBox[
           StyleBox["eqn", "MR",
            FontSlant->"Italic"], "2"], "MR"], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         StyleBox["...", "MR"], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         RowBox[{
          RowBox[{
           SubscriptBox[
            StyleBox["u", "MR"], "1"], 
           StyleBox["[", "MR"], 
           StyleBox[
            RowBox[{"t", ",", " ", 
             StyleBox["xmin",
              FontSlant->"Italic"]}], "MR"], 
           StyleBox["]", "MR"]}], 
          StyleBox[" ", "MR"], 
          StyleBox["==", "MR"], 
          StyleBox[" ", "MR"], 
          RowBox[{
           SubscriptBox[
            StyleBox["u", "MR"], "1"], 
           StyleBox["[", "MR"], 
           StyleBox[
            RowBox[{"t", ",", 
             StyleBox["xmax",
              FontSlant->"Italic"]}], "MR"], 
           StyleBox["]", "MR"]}]}], 
         StyleBox[",", "MR"], 
         RowBox[{
          RowBox[{
           SubscriptBox[
            StyleBox["u", "MR"], "2"], 
           StyleBox["[", "MR"], 
           StyleBox[
            RowBox[{"t", ",", " ", 
             StyleBox["xmin",
              FontSlant->"Italic"]}], "MR"], 
           StyleBox["]", "MR"]}], 
          StyleBox[" ", "MR"], 
          StyleBox["==", "MR"], 
          StyleBox[" ", "MR"], 
          StyleBox[
           RowBox[{
            SubscriptBox["u", "2"], "[", 
            RowBox[{"t", ",", 
             StyleBox["xmax",
              FontSlant->"Italic"]}], "]"}], "MR"]}], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         StyleBox["...", "MR"]}], 
        StyleBox["}", "MR"]}], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           SubscriptBox["u", "1"], "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", " ", 
          RowBox[{
           SubscriptBox["u", "2"], "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", " ", "..."}], "}"}], "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"{", 
         RowBox[{"t", ",", " ", 
          StyleBox["tmin",
           FontSlant->"Italic"], ",", " ", 
          StyleBox["tmax",
           FontSlant->"Italic"]}], "}"}], "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"{", 
         RowBox[{"x", ",", " ", 
          StyleBox["xmin",
           FontSlant->"Italic"], ",", " ", 
          StyleBox["xmax",
           FontSlant->"Italic"]}], "}"}], "MR"]}], 
      StyleBox["]", "MR"]}], "\[SpanFromLeft]"},
    {"", Cell[TextData[{
      "solve a system of partial differential equations for function ",
      Cell[BoxData[
       SubscriptBox[
        StyleBox["u", "MR"], "1"]]],
      ", ",
      Cell[BoxData[
       SubscriptBox[
        StyleBox["u", "MR"], "2"]]],
      ", ... with periodic boundary conditions in the spatial variable x \
(assuming that t is a temporal variable)."
     }], "Text"]},
    {
     RowBox[{
      StyleBox["NDSolve", "MR"], 
      StyleBox["[", "MR"], 
      RowBox[{
       RowBox[{
        StyleBox["{", "MR"], 
        RowBox[{
         StyleBox[
          SubscriptBox[
           StyleBox["eqn", "MR",
            FontSlant->"Italic"], "1"], "MR"], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         StyleBox[
          SubscriptBox[
           StyleBox["eqn", "MR",
            FontSlant->"Italic"], "2"], "MR"], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         StyleBox["...", "MR"], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         RowBox[{
          RowBox[{
           StyleBox[
            SubscriptBox["u", "1"], "MR"], 
           StyleBox["[", "MR"], 
           RowBox[{
            StyleBox["t", "MR"], 
            StyleBox[",", "MR"], 
            StyleBox[" ", "MR"], 
            RowBox[{
             SubscriptBox[
              StyleBox["x", "MR",
               FontSlant->"Italic"], "1"], 
             StyleBox["min", "MR",
              FontSlant->"Italic"]}], 
            StyleBox[",", "MR",
             FontSlant->"Italic"], 
            StyleBox[" ", "MR",
             FontSlant->"Italic"], 
            StyleBox[
             SubscriptBox[
              StyleBox["x", "MR",
               FontSlant->"Italic"], "2"], "MR"], 
            StyleBox[",", "MR"], 
            StyleBox[" ", "MR"], 
            StyleBox["...", "MR"]}], 
           StyleBox["]", "MR"]}], 
          StyleBox[" ", "MR"], 
          StyleBox["==", "MR"], 
          StyleBox[" ", "MR"], 
          RowBox[{
           StyleBox[
            SubscriptBox["u", "1"], "MR"], 
           StyleBox["[", "MR"], 
           RowBox[{
            StyleBox["t", "MR"], 
            StyleBox[",", "MR"], 
            RowBox[{
             SubscriptBox[
              StyleBox["x", "MR",
               FontSlant->"Italic"], "1"], 
             StyleBox["max", "MR",
              FontSlant->"Italic"], 
             StyleBox[" ", "MR",
              FontSlant->"Italic"], 
             StyleBox[
              SubscriptBox[
               StyleBox["x", "MR",
                FontSlant->"Italic"], "2"], "MR"]}], 
            StyleBox[",", "MR"], 
            StyleBox[" ", "MR"], 
            StyleBox["...", "MR"]}], 
           StyleBox["]", "MR"]}]}], 
         StyleBox[",", "MR"], 
         RowBox[{
          RowBox[{
           StyleBox[
            SubscriptBox["u", "2"], "MR"], 
           StyleBox["[", "MR"], 
           RowBox[{
            StyleBox["t", "MR"], 
            StyleBox[",", "MR"], 
            StyleBox[" ", "MR"], 
            RowBox[{
             SubscriptBox[
              StyleBox["x", "MR",
               FontSlant->"Italic"], "1"], 
             StyleBox["min", "MR",
              FontSlant->"Italic"]}], 
            StyleBox[",", "MR",
             FontSlant->"Italic"], 
            StyleBox[" ", "MR",
             FontSlant->"Italic"], 
            StyleBox[
             SubscriptBox[
              StyleBox["x", "MR",
               FontSlant->"Italic"], "2"], "MR"], 
            StyleBox[",", "MR"], 
            StyleBox[" ", "MR"], 
            StyleBox["...", "MR"]}], 
           StyleBox["]", "MR"]}], 
          StyleBox[" ", "MR"], 
          StyleBox["==", "MR"], 
          StyleBox[" ", "MR"], 
          RowBox[{
           StyleBox[
            SubscriptBox["u", "2"], "MR"], 
           StyleBox["[", "MR"], 
           RowBox[{
            StyleBox["t", "MR"], 
            StyleBox[",", "MR"], 
            RowBox[{
             SubscriptBox[
              StyleBox["x", "MR",
               FontSlant->"Italic"], "1"], 
             StyleBox["max", "MR",
              FontSlant->"Italic"], 
             StyleBox[" ", "MR",
              FontSlant->"Italic"], 
             StyleBox[
              SubscriptBox[
               StyleBox["x", "MR",
                FontSlant->"Italic"], "2"], "MR"]}], 
            StyleBox[",", "MR"], 
            StyleBox[" ", "MR"], 
            StyleBox["...", "MR"]}], 
           StyleBox["]", "MR"]}]}], 
         StyleBox[",", "MR"], 
         StyleBox[" ", "MR"], 
         StyleBox["...", "MR"]}], 
        StyleBox["}", "MR"]}], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           SubscriptBox["u", "1"], "[", 
           RowBox[{"t", ",", 
            SubscriptBox[
             StyleBox["x", "MR",
              FontSlant->"Italic"], "1"], 
            StyleBox[",",
             FontSlant->"Italic"], 
            StyleBox[" ",
             FontSlant->"Italic"], 
            SubscriptBox[
             StyleBox["x", "MR",
              FontSlant->"Italic"], "2"], ",", " ", "..."}], "]"}], ",", " ", 
          
          RowBox[{
           SubscriptBox["u", "2"], "[", 
           RowBox[{"t", ",", 
            SubscriptBox[
             StyleBox["x", "MR",
              FontSlant->"Italic"], "1"], 
            StyleBox[",",
             FontSlant->"Italic"], 
            StyleBox[" ",
             FontSlant->"Italic"], 
            SubscriptBox[
             StyleBox["x", "MR",
              FontSlant->"Italic"], "2"], ",", " ", "..."}], "]"}], ",", " ", 
          "..."}], "}"}], "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"{", 
         RowBox[{"t", ",", " ", 
          StyleBox["tmin",
           FontSlant->"Italic"], ",", " ", 
          StyleBox["tmax",
           FontSlant->"Italic"]}], "}"}], "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"{", 
         RowBox[{"x", ",", " ", 
          StyleBox["xmin",
           FontSlant->"Italic"], ",", " ", 
          StyleBox["xmax",
           FontSlant->"Italic"]}], "}"}], "MR"]}], 
      StyleBox["]", "MR"]}], "\[SpanFromLeft]"},
    {"", Cell[TextData[{
      "solve a system of partial differential equations for function ",
      Cell[BoxData[
       SubscriptBox[
        StyleBox["u", "MR"], "1"]]],
      ", ",
      Cell[BoxData[
       SubscriptBox[
        StyleBox["u", "MR"], "2"]]],
      ", ... with periodic boundary conditions in the spatial variable ",
      Cell[BoxData[
       FormBox[
        SubscriptBox["x", "1"], TraditionalForm]]],
      " (assuming that t is a temporal variable)."
     }], "Text"]}
   }], TextForm]], "DefinitionBox",
 CellID->818469801],

Cell["\<\
Giving boundary conditions for partial differential equations. \
\>", "Caption",
 CellID->1159866287],

Cell[TextData[{
 "If you are solving for several functions ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["u", "1"], ",", " ", 
    SubscriptBox["u", "2"], ",", " ", "..."}], TraditionalForm]]],
 " then for any of the functions to have periodic boundary conditions, all of \
them must (the condition need only be specified for one function). If you are \
working with more than one spatial dimension, you can have periodic boundary \
conditions in some independent variable dimensions and not in others.",
 " "
}], "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->231596546],

Cell["\<\
This solves a generalization of the sine-Gordon equation to two spatial \
dimensions with periodic boundary conditions using a pseudospectral method. \
Without the pseudospectral method enabled by the periodicity, the problem \
could take much longer to solve.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->948665427],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", "=", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", " ", "x", ",", " ", "y"}], "]"}], ",", " ", "t", 
         ",", " ", "t"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", " ", "x", ",", " ", "y"}], "]"}], ",", " ", "x", 
          ",", " ", "x"}], "]"}], " ", "+", " ", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", " ", "x", ",", " ", "y"}], "]"}], ",", " ", "y", 
          ",", " ", "y"}], "]"}], " ", "-", " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"u", "[", 
          RowBox[{"t", ",", " ", "x", ",", " ", "y"}], "]"}], "]"}]}]}], ",", 
      " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", " ", "x", ",", " ", "y"}], "]"}], " ", "\[Equal]", 
       " ", 
       RowBox[{"Exp", "[", 
        RowBox[{"-", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"]}], ")"}]}], "]"}]}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"Derivative", "[", 
          RowBox[{"1", ",", "0", ",", "0"}], "]"}], "[", "u", "]"}], "[", 
        RowBox[{"0", ",", "x", ",", "y"}], "]"}], " ", "\[Equal]", " ", "0"}],
       ",", " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", 
         RowBox[{"-", "10"}], ",", " ", "y"}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "10", ",", " ", "y"}], "]"}]}], ",", " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "x", ",", " ", 
         RowBox[{"-", "10"}]}], "]"}], " ", "\[Equal]", " ", 
       RowBox[{"u", "[", 
        RowBox[{"t", ",", " ", "x", ",", " ", "10"}], "]"}]}]}], "}"}], ",", 
    "u", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "6"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
       RowBox[{"\"\<SpatialDiscretization\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<TensorProductGrid\>\"", ",", " ", 
          RowBox[{
          "\"\<DifferenceOrder\>\"", "->", "\"\<Pseudospectral\>\""}]}], 
         "}"}]}]}], "}"}]}]}], "]"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[2]:=",
 CellID->362535500],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "6.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "10.`"}], ",", "10.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "10.`"}], ",", "10.`"}], "}"}]}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{481, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[2]=",
 CellID->654262509]
}, Open  ]],

Cell[TextData[{
 "In the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object returned as a solution, the ellipses in the notation ",
 StyleBox["{..., xmin, xmax, ...}",
  FontFamily->"Courier"],
 " are used to indicate that this dimension repeats periodically"
}], "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->817962383],

Cell["\<\
This makes a surface plot of a part of the solution derived from periodic \
continuation at t==6.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1074566333],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"First", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"6", ",", "x", ",", "y"}], "]"}], " ", "/.", " ", "sol"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "20", ",", " ", "40"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "15"}], ",", "15"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "40"}]}], "]"}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[7]:=",
 CellID->1244100908],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 289},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[7]=",
 CellID->29001616]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " uses two methods for nonperiodic boundary conditions. Both have their \
merits and drawbacks. The first method is to differentiate the boundary \
conditions with respect to the temporal variable and solve for the resulting \
differential equation(s) at the boundary. The second method is to discretize \
each boundary condition as it is. This typically results in an algebraic \
equation for the boundary solution component, so the equations must be solved \
with a DAE solver. This is controlled with the ",
 StyleBox["DifferentiateBoundaryConditions",
  FontFamily->"Courier"],
 " option to ",
 StyleBox["MethodOfLines.",
  FontFamily->"Courier"]
}], "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1635284659],

Cell["\<\
To see how the differentiation method works, consider again the simple \
example of the method of lines introduction section. In the first \
formulation, the Dirichlet boundary condition at x == 0 was handled by \
differentiation with respect to t. The Neumann boundary condition was handled \
using the idea of reflection, which worked fine for a second order finite \
difference approximation, but does not generalize quite as easily to higher \
order (though it can be done easily for this problem by computing the entire \
reflection). The differentiation method, however, can be used for any order \
differences on the Neumann boundary condition at x == 1. As an example, a \
solution to the problem will be developed using fourth order differences.\
\>", "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->364624975],

Cell["\<\
This is a setting for the number of and spacing between spatial points. It is \
purposely set small so you can see the resulting equations. You can change it \
later to improve the accuracy of the approximations.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->2092914263],

Cell[BoxData[
 RowBox[{
  RowBox[{"n", " ", "=", " ", "10"}], ";", 
  RowBox[{
   SubscriptBox["h", "n"], " ", "=", " ", 
   RowBox[{"1", "/", "n"}]}], ";"}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[8]:=",
 CellID->1090551210],

Cell[TextData[{
 "This defines the vector of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["u", "i"], TraditionalForm]]],
 "."
}], "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->2092743636],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"U", "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{
     SubscriptBox["u", "i"], "[", "t", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "0", ",", "n"}], "}"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[9]:=",
 CellID->662386215],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["u", "0"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "1"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "2"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "3"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "4"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "5"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "6"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "7"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "8"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "9"], "[", "t", "]"}], ",", 
   RowBox[{
    SubscriptBox["u", "10"], "[", "t", "]"}]}], "}"}]], "Output",
 ImageSize->{504, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[9]=",
 CellID->17447031]
}, Open  ]],

Cell["\<\
This discretizes the Neumann boundary condition at x == 1 in the spatial \
direction.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->516329088],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"bc", " ", "=", " ", 
  RowBox[{
   RowBox[{"Last", "[", 
    RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
     RowBox[{"1", ",", 
      RowBox[{
       SubscriptBox["h", "n"], 
       RowBox[{"Range", "[", 
        RowBox[{"0", ",", "n"}], "]"}]}], ",", " ", 
      RowBox[{"U", "[", "t", "]"}]}], "]"}], "]"}], " ", "\[Equal]", " ", 
   "0"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[10]:=",
 CellID->2058785492],

Cell[BoxData[
 RowBox[{
  RowBox[{
   FractionBox[
    RowBox[{"5", " ", 
     RowBox[{
      SubscriptBox["u", "6"], "[", "t", "]"}]}], "2"], "-", 
   FractionBox[
    RowBox[{"40", " ", 
     RowBox[{
      SubscriptBox["u", "7"], "[", "t", "]"}]}], "3"], "+", 
   RowBox[{"30", " ", 
    RowBox[{
     SubscriptBox["u", "8"], "[", "t", "]"}]}], "-", 
   RowBox[{"40", " ", 
    RowBox[{
     SubscriptBox["u", "9"], "[", "t", "]"}]}], "+", 
   FractionBox[
    RowBox[{"125", " ", 
     RowBox[{
      SubscriptBox["u", "10"], "[", "t", "]"}]}], "6"]}], "\[Equal]", 
  "0"}]], "Output",
 ImageSize->{349, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[10]=",
 CellID->86454675]
}, Open  ]],

Cell["\<\
This differentiates the discretized boundary condition with respect to t.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1906237188],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"bcprime", " ", "=", " ", 
  RowBox[{"D", "[", 
   RowBox[{"bc", ",", "t"}], "]"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[11]:=",
 CellID->914706481],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    FractionBox["5", "2"], " ", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["u", "6"], "\[Prime]",
      MultilineFunction->None], "[", "t", "]"}]}], "-", 
   RowBox[{
    FractionBox["40", "3"], " ", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["u", "7"], "\[Prime]",
      MultilineFunction->None], "[", "t", "]"}]}], "+", 
   RowBox[{"30", " ", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["u", "8"], "\[Prime]",
      MultilineFunction->None], "[", "t", "]"}]}], "-", 
   RowBox[{"40", " ", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["u", "9"], "\[Prime]",
      MultilineFunction->None], "[", "t", "]"}]}], "+", 
   RowBox[{
    FractionBox["125", "6"], " ", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["u", "10"], "\[Prime]",
      MultilineFunction->None], "[", "t", "]"}]}]}], "\[Equal]", 
  "0"}]], "Output",
 ImageSize->{366, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[11]=",
 CellID->246439146]
}, Open  ]],

Cell[TextData[{
 "Technically, it is not necessary that the discretization of the boundary \
condition be done with the same difference order as the rest of the DE; in \
fact, since the error terms for the one-sided derivatives are much larger, it \
may sometimes be desirable to increase the order near the boundaries. ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " does not do this because it is desirable that the difference order and the \
InterpolatingFunction interpolation order be consistent across the spatial \
direction. "
}], "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->2117437756],

Cell[TextData[{
 "This is another way of generating the equations using ",
 StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
 ". The first and last will have to be replaced with the appropriate \
equations from the boundary conditions."
}], "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1096566600],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"eqns", " ", "=", " ", 
  RowBox[{"Thread", "[", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"U", "[", "t", "]"}], ",", "t"}], "]"}], " ", "\[Equal]", " ", 
    RowBox[{
     FractionBox["1", "8"], 
     RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
      RowBox[{"2", ",", 
       RowBox[{
        SubscriptBox["h", "n"], 
        RowBox[{"Range", "[", 
         RowBox[{"0", ",", "n"}], "]"}]}], ",", " ", 
       RowBox[{"U", "[", "t", "]"}]}], "]"}]}]}], "]"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[12]:=",
 CellID->1598686852],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{567, 330},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[12]=",
 CellID->408090353]
}, Open  ]],

Cell["\<\
Now we can replace the first and last equation with the boundary condition.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1755488819],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"eqns", "[", 
    RowBox[{"[", 
     RowBox[{"1", ",", "2"}], "]"}], "]"}], " ", "=", " ", 
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"2", "\[Pi]", " ", "t"}], "]"}], ",", "t"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"eqns", "[", 
    RowBox[{"[", 
     RowBox[{"-", "1"}], "]"}], "]"}], " ", "=", " ", "bcprime"}], 
  ";"}], "\n", "eqns"}], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[13]:=",
 CellID->757595817],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{512, 322},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[15]=",
 CellID->660666478]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " is capable of solving the system as is for the appropriate derivatives, so \
it is ready for the ODEs."
}], "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->426810713],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"diffsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqns", ",", " ", 
      RowBox[{"Thread", "[", 
       RowBox[{
        RowBox[{"U", "[", "0", "]"}], " ", "\[Equal]", " ", 
        RowBox[{"Table", "[", 
         RowBox[{"0", ",", " ", 
          RowBox[{"{", "11", "}"}]}], "]"}]}], "]"}]}], "}"}], ",", " ", 
    RowBox[{"U", "[", "t", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[16]:=",
 CellID->1576065618],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{357, 186},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[16]=",
 CellID->52685007]
}, Open  ]],

Cell["\<\
This shows a plot of how well the boundary condition at x == 1 was satisfied\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->658407259],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"Apply", "[", 
      RowBox[{"Subtract", ",", " ", "bc"}], "]"}], " ", "/.", " ", 
     "diffsol"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[17]:=",
 CellID->377247397],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxN13k4lGvYAHBEKIkWRQtJpVKcEuI+JWlFUUQdSSqtJBQqmRwke6WSpSZb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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{{0, 4}, {-8.118505867571457*^-16, 1.250735626179278*^-15}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 194},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[17]=",
 CellID->277514825]
}, Open  ]],

Cell["\<\
Treating the boundary conditions as algebraic conditions saves a couple of \
steps in the processing at the expense of using a DAE solver.\
\>", "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1270249831],

Cell["\<\
This replaces the first and last equations (from before) with algebraic \
conditions corresponding to the boundary conditions.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1195209780],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"eqns", "[", 
    RowBox[{"[", "1", "]"}], "]"}], " ", "=", " ", 
   RowBox[{
    RowBox[{
     SubscriptBox["u", "0"], "[", "t", "]"}], " ", "\[Equal]", " ", 
    RowBox[{"Sin", "[", 
     RowBox[{"2", " ", "\[Pi]", " ", "t"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"eqns", "[", 
    RowBox[{"[", 
     RowBox[{"-", "1"}], "]"}], "]"}], " ", "=", " ", "bc"}], 
  ";"}], "\n", "eqns"}], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[18]:=",
 CellID->265234873],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztnd+PXDcVxyc7s7vdhmbT/E6TNAs0bdSkLaFVhEgjNlW7uxFpoflV5RdS
qJCa0pIq9KEtD0BTBKKE8qMSKiDgBd55Rw2i5H+Av4UHs/eHZzzfe2yfY/vO
zO66UnPjO7728bGvfc752DcvXn/7te+8ef3tG69eX1i+df2t1268+v2FpZu3
Vm91N3U6U51OZ9Nzezqrf1erfy3/1/9Nrf4/SFap/at/Tq9menb12l29/ru+
/rf/a2faLGP1750T5N1/kXf/U14PFH92+791FiH9T8ezD0Per7Ge/UrRwtXr
n8r0QXjm2frXz1avPd0i4/c/lE8twN2TVM1QarMUiQ4/gxJouQptFjdP1td7
9fWrZO6/wN1e3fbN9fVwfZ2B+2+Spf3ZuFtIPA25NkP6MKS7nvzfJWv9Idyd
qVvcq6+ztczlXzrlf9VfLFL2jOfN+w/W96ew3G6/3PK3KUGZM5C+XOfD5+mW
lrka42SKvIslTmnBQcpGfzRKwqe2lY8O0s0cmJ7SUjrGCq9u1N/Xa/1VTdH/
zbI08vvy/lx59+7du/Wzh8tqFxcXDbmm6zfCzPeUJd8cTzt0L/r0g2MB+nK6
fm7GGKWl/kcqAWrqBGhqG1k79uwninoDQrXLGvmOtm2xa4weZ6QE1Ezhqzn9
iEed6nmN1hzdmvuFrZhX2G8S3UlrQ51dIkvFMVDNuR82JG3MZrW+9BWlK+5P
E+VvNdL20dMLLP3bgjb+lMxLy7NDqHteK6WlSlq3Tw3eNNOGeZKo0/y9kMm0
I3bC79vV4D2hfv9S3TM426KNtUe51vx9oD9sI9pIRz2axPx/L/6YBU0qer7A
Z9H6riTerHh2F7bEZiPtUjy7y9VqW9m/Nvpo2uhLax1MTdnqozVWvtMsi81m
X/JWud3KN5dRlhq2gWc9YEm/pUqJtV2ZvcGTkO6bLayyqHdcMp8Vc0kaK01a
M/YL5p/l61LSGw6JYvpBz+Eh+o+z5/TsFKt/n4doHwVUu6XS7FdmZETP5DNk
nXPCsss1Ptjmk9aGem3MIc62zQtr+0dfX5jLnLWxDTbbbaeRto9DtAy5paNm
aA3MQumoEVvptCak9h1PA9uFpfLWICzlIWEtAw30OgMr0bT2TKtxB/z+sRqs
8Ho+KcrZC/k+LWvRclPR2+HUqbq8x+rr7fq6Ul8/qq9PlbkXSc1gr+4mc/2Y
vLtE3r1D3l2AejY7ZEJ/aD+kb0N6BdIfQfogpO8fe92651D7xbgsRsIH9fV0
ff1Vff2CoCU4d2BMXtoS7ME5Qd17yVzxowr1MTekXezLh5T7PfkiWdq8GszF
5tt6CHJfU7RPes9y/wI8T9rtZYTaZTvgCMI2XDPSnY7f17wHabRYMf85Quq+
1dLXGRVPvw+etPlX2KIlNezDp/JJ0SouImkSP9Jmq3HqOk081+iFhsaomDvm
4flJemUajjjuhlwvkGXx/bs4Txd19kdKR5HSeb0usZRUz/Z93aq3vP4f2ac+
q9yIzvdKQVfTlvxdyN8m99A6kLRnWtHU4bhFTl8+TTFIP9jRl+1467bWIVNB
LVarEz2C9CwsHTm8eaAc04GEJsST9735Yf51KmnwDXf1ywPCumxrn4TjSL1s
n7ZlXnds7bR20ZNFvdo8WUqf/pES6pX/DdI8r5xbOlpelWZ8XnnIiPNrSOq1
o2Z4Xrs04sDXkMuuR7+JYkq9jt37fxqeR1b0jOd3XR5GDzAKcUzRbArlxTWh
0spOuHsc0qYfrN8Jl19wi2iVK/8xKN/Hqd4t/rBEp3114fow8GT7Vg5I7rP/
KX7M8QPOqWFfpuHTOHgY19fYBvKn3kN1yVc+k7dx20P33nPUXdTW2HyXs4La
+TL5LBeebDgfJqF4jnHKk4rez9EmYQyTM2w0+qyedvyoi9TzY9jHFj3ikkt0
meqP0D1okvHlkIk3sjDG6SvF3qPUXDvKEdWdiJHEmr0cEl2h9B+oEfYc75AH
0xUx8I0iHDVpfbzzRtu6HZk3K90nmH6ESCW4atGlpNXkTonkvU7vMA31Ty8K
2he6IzKmd0PrvBbRLuxHWx28/kO/ph0vmrZdebOYtK6Y/pTWRVuYvHbtEdZF
eqQlp9S7TvXqS3n2F+B37Wnh7lSdNiMFVL6rlvJwP0Kp0X7D+fsR9Hmli/X1
Tn1drq83ytyLpGZxTD9A5jpP3v2QvEtTK+T6px0y4cywDdJoBd2B9DKk90F6
Zex12/YjFDu9i5HwSn39ZX1dqa+4r8/VElw3MGYobQn24LKg7nkyV/yoQn0s
D2kX+3Krcr8nB8nStsJdvT7b4peHFH3e7weQD+OEnxC/m+lTpTTmHrJCAxhd
9O1POATP++J+70B+X6zvd5ZWlmvpYH+CJQYWukfhoKdcbmwN4wLfUGGxvNB9
CfgOYv7ybSP3JYSx1LB9CfjWmRYXnwX69N5SNMrxhoXJSvVZUdupQV+NfceB
uafGfG4zPMeLg0mjBmH7CUDjCeJmo94H8QyRr7m+2HcUeN9nj+zMN9lK6qX9
jDK+DOnQ2AzrvRXIRVo6Le0goDUsjWlgC85COiTGEbtTgNZiqp0CB6CcUUQL
uJEWtHGqlqfeCfCE8PkY/1q6n4CvAZel/AT8juSeIv/ainLtAPiyoi3wI4p3
mhRX46p1PmJ/06NDtJTfE+YvpA89ZYor6sB3owj8ZjVs9epv15jUYsgKdlBz
Kr5PWcJXhXVQ6xOXyLd1MpXWso9Nln3FtGrLL+A0V9AxnUe9lkgaFk30SkN/
A2Ocp2HDxsPnII3rbjHvkOdcx2AhX2pRFqn1iWvY6E/d8vobLRnsbyzluF1b
0QyfmsXC7Iw0JDjE60CrY7SnfTFdRWR9fU5ZxUWy/MIY43xvm6d+L7cgg9Tf
QVvLzcOlJ43D+oz+XpH5zTrpexDqf1xJWCfXr9pFlhZ6wpnXA2jFzpEyYG+b
3/STakfqB8X0hJSf0vYNPiU9aY09U3E5TUr1CU7tcZ1Qg9NUppWOHti3LPko
gmqWj0S0jHP0GxxPRJ+vr2+UuRdJjaYnos+Rd7GnXnLIFEsln4f0Xki/OPa6
pUT0hfqKVNLVklgiii3BHjwjqDueiNKjCvVxZki7XCKq3xM8i1GVJiWij0Ia
4y0+Mvox8buZXrJK5VpNYsnoux6pMP9vLK2svATbyW1Xi8R01Cjb55cnI6QJ
YkNY58+I58x0uVeApKThJxdxDS2S65OSpokNcfqsqG1p0FeZkm5ISkqdfm+u
OfYR0g0YGTF0NHTvfFt0NDSO43tDXXrPVFSmxY1ARdG+qVq+kagoXwMSKqpp
5jTc59JQ33noo4qmpTinVq3xUdDvGen+BOuwinF/oc/qln6bF1fJgY+WhHxG
7v/zUs/I/Yxc8imhqbRGJ5Fy2lo2WrpJeTlyaWLpZhrGGtb34yKaVm171ou1
TTRja+X18aRQTL2aThLFjBl/YTQzlJ1iuoqeZno5fLddeiktPazPMr3000tu
abweyPTSb7/gU/jmSOllxdDGRS+xnFdKaXR7M70cGgtGOtNLTksyvUxDL7nf
l35UDd5uM997io7H8Chl7PnNxzy1xJ7fZFLKts5v9sxye8STk3pyM5xJ2uyn
UbBIr584IUwyLpKTmkUWf51sFhnjD46bQeoeaV/uNAxS63qUDNKnn3GxyNBo
TGaR1N3MIjOL1Lkmg0WmOqHpY5KoWd8JzXbYJO7w8+V/HMqXnspMzCYNxtfg
UkTLKCtXeirTpS0xo4xkn5lTyqRg2RStc8o0tDRzylBZqH7JnDKeIYXu0muL
U4buYgznkzGjPXNKjgyZU1J3M6d090DmlH77JXNKM5U5ZZXOnJJqSeaU4zll
eUjR+7jTnK5cJ9zSERNBSzbmdGVoLCTmdGVojGftk8w0kYtMMmntTc6pSkqn
mWiuj1OVsRGCfLrSp+FMNDPRzESTIpp6FIeQTNu3ZtshmG9BGleZ2NOVRwip
XflbJpiN3YDUKkHNzpNyrjLEB1hb3NLlo42WX+oZNYxIxHLLND7I5HNLPU9O
Cq+k6sq8Mp4lYf+izsfFLfX4Gw23DKWkmVdyZMi80vbmZV5p74HMK/12S+aV
ZirzyiqdeSXVkswr0/HK4iaHU2qbCjllt9OMxPD45INwF+07yddfFTEj+v71
Hl8khsknGxGPlN98HRmVDI7bpGWR5hpMsowWmSSLZ0wQkzR1lUbWEDap34rM
JHU5KZlkaOxmnCzSFx3diEwyNjaT2SR1N7PJzCZ1rslgk2vry69ts8nJPV3p
m39TnK70vT35dCWlFcraIXtrg1DKuNonn06yPBnPmrF+KGVIrWuLUvreuHFR
ylBumL/+Onw3U8pMKe11Zkpp3s2UMlPKTCnHUXemlGlG1SR9/fURNZj9zXxx
X3+NpZTUv6Tper4lSumIfaxJXpkglrN+/o3KtUQu08R2Nt6/VRlLtibhVGV/
cLcud7rvxLpGXZsk0xn3c9TZ5ndiXXJkgpkJpr6fCSbj3ZlggtntNK1oCcGk
nm//+7C6VIl1jTrOBLO5E5Brw3NOWIYy0bXDLlnr5EjZpWv8Zoa53hhmKFnK
7NK8u7bYJesks0OufNJy+G5mmJlh2uvMDNO8mxlmZpiZYY6j7sww04yqUTHM
4uZ9yh2NoWghskz99stZ5jzclX4R1pS20JwvGvKOarbGlX8CWGbkHuu1zTFZ
vKNFjpkmWjEajpkmzrPxOCarjx3SZo5J1zepHDOWW2eemXlm5pmhozTzzGNq
sFqF8EzX12IpSzyMY9706CqWYx4V5l8HHNNRRzTRDGakmWOGScMiHCPhmPH+
yeTzzBSca23/S5ehFHVt8Uxto0hb2eaXYmN8hTCOGUtPM8/kyJB5pm3sZZ5p
74HMM/32S+aZZkpzmyNqmM+s1Nf36+uTZe5FUqM46vaRuWjGtETe/RF5F2nY
VodM+F4tQBrJ2wqk34c0MsX5sddt45lFtMvkmKfr621F80xXS3B+/XxkS7AH
twjqPkDmih9VqI8tQ9rFvnxYud8T5JlVacWfRcW+L8e+rujoy11Ffzn2NDyP
c1cZXSH/nU3XqoFRsDc8teDzn0Ia4wmYf5kov7+6t8oxz9T6NhlNt5M28oIR
+JeVJQLTiJKEEs2TxHON/GMnmi9h/WTZoyaM2FvfjJSKxcHEBIrq4X40tSWq
JeWeyDtt5M4nR9gYDPFKY3gj5uspI/Y7AV+yRR6Ks1m1TtmJOeVtpeGgOA+E
nQ4Pjc3g+34+sL9Y57QdcvjecFf/xPJR7AFqB0O3Y4slSEeFbT0cJTeltZuK
m6I+ab1hbdxYxl+ptieLlKAtVmkmNVflrf1YqtT/52lKGsHga8hl6T/iya/9
ej2Han9/O+TT50/xay1PK5rD7lBufkuRTxevrVrv47Vm/1OzpW8npG/nJH4H
2OehvC6UB9eggd9K8V299mMsx/dlWi4VveQrPwHL3Q5lt1HHWeX3uhp0k1k2
3WM+aliWknzvYxof4AqWHCkVix4lZcR82dLw67ORUjhtSjG35o1A0nIZw/7L
6NGWnEzvgnQvkWakcpyjekhUa196Qa280YTRSrKUMexwSD+aYtl3qtEUK8d5
SMvOEodEB8x0Ffv3jSKUaRQ7XdOPGCnz1paxtjAkrZXWdcGi4TQed1ivzwra
m4LL63zj4fK8dnHruEiWFurx8/oPfRaeT53GJ47px3aoPt0uaV20BRkbl8D+
rTi23kFgel4uT/2qakYIiqtvx0DXku+CckcaSgrdVwS1l0C/LcWd/5V3UBOj
+oWafzHKUVwXFG3hF398QJZsvrN6hnC9hY9DesEiWZH4CVmjzhEbNSjf6d7g
uU6fZVTN1//pxyjPm5qdbPSPkgt1fLus2BcrT+dDzyiKSOgYGcpBpnmRf+b6
Qq+xlE2px0g1KnWEROs7vf+HlCZMFyTLG7Ky0Xui3sSfK+soSe6p0COEWvfl
2nBx+45IF2RULNpO1Gd8KP7DeT9cPmtA+3pme0LsJrRiTEse44w+f7vIOlvX
QdVl7xvcfePrBZS6kBS/iEH3N357nT9+9Dcw9Pq4CzREya3zaq3uVYNZ6hdk
LcUfbaz+TbtDa6WySzqb/g9UZI4M\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{512, 322},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[20]=",
 CellID->177608103]
}, Open  ]],

Cell[TextData[{
 "This solves the system of DAEs with ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1895209452],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"daesol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqns", ",", " ", 
      RowBox[{"Thread", "[", 
       RowBox[{
        RowBox[{"U", "[", "0", "]"}], " ", "\[Equal]", " ", 
        RowBox[{"Table", "[", 
         RowBox[{"0", ",", " ", 
          RowBox[{"{", "11", "}"}]}], "]"}]}], "]"}]}], "}"}], ",", " ", 
    RowBox[{"U", "[", "t", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[21]:=",
 CellID->1900519533],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{357, 186},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[21]=",
 CellID->58579361]
}, Open  ]],

Cell["This shows how well the boundary condition was satisfied.", \
"MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1840640067],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"Apply", "[", 
      RowBox[{"Subtract", ",", " ", "bc"}], "]"}], " ", "/.", " ", "daesol"}],
     "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[22]:=",
 CellID->843587095],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 195},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[22]=",
 CellID->51013059]
}, Open  ]],

Cell["\<\
For this example, the boundary condition was satisfied well within tolerances \
in both cases, but the differentiation method did very slightly better. This \
is not always true; in some cases, with the differentiation method, the \
boundary condition can experience cumulative drift since the error control in \
this case is only local. The Dirichlet boundary condition at x == 0 in this \
example shows some drift.\
\>", "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->355542635],

Cell["\<\
This makes a plot which compares how well the Dirichlet boundary condition at \
x == 0 was satisfied with the two methods. The solution with the \
differentiated boundary condition is shown in black.\
\>", "MathCaption",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->2126703843],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["u", "0"], "[", "t", "]"}], " ", "/.", " ", "diffsol"}],
        ",", " ", 
       RowBox[{
        RowBox[{
         SubscriptBox["u", "0"], "[", "t", "]"}], " ", "/.", " ", 
        "daesol"}]}], "}"}], "-", " ", 
     RowBox[{"Sin", "[", 
      RowBox[{"2", " ", "\[Pi]", " ", "t"}], "]"}]}], " ", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "4"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "Black", "}"}], ",", " ", 
      RowBox[{"{", "Blue", "}"}]}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"In[23]:=",
 CellID->1193081143],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 197},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellLabel->"Out[23]=",
 CellID->421302604]
}, Open  ]],

Cell[TextData[{
 "When using ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ", it is easy to switch between the two methods by using the ",
 StyleBox["DifferentiateBoundaryConditions", "MR"],
 " option. Remember that when you use ",
 StyleBox["DifferentiateBoundaryConditions->False", "MR"],
 ", you are not as free to choose integration methods; the method needs to be \
a DAE solver."
}], "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->1612258420],

Cell["\<\
With systems of PDEs or equations with higher order derivatives having more \
complicated boundary conditions, both methods can be made to work in general. \
When there are multiple boundary conditions at one end, it may be necessary \
to attach some conditions to interior points. Here is an example of a PDE \
with two boundary conditions at each end of the spatial interval.\
\>", "Text",
 CellTags->{"b:7.2", "ndsg:2.0.2"},
 CellID->951834684],

Cell["\<\
This solves a differential equation with two boundary conditions at each end \
of the spatial interval. The StiffnessSwitching integration method is used to \
avoid potential problems with stability from the fourth order derivative.\
\>", "MathCaption",
 CellID->1181550960],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"x", ",", "t"}], "]"}], ",", "t", ",", "t"}], "]"}], " ", 
       "\[Equal]", " ", 
       RowBox[{"-", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"x", ",", "t"}], "]"}], ",", "x", ",", "x", ",", "x", ",", 
          "x"}], "]"}]}]}], ",", " ", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"x", ",", "t"}], "]"}], " ", "\[Equal]", " ", 
          RowBox[{
           FractionBox[
            SuperscriptBox["x", "2"], "2"], "-", 
           FractionBox[
            SuperscriptBox["x", "3"], "3"], "+", 
           FractionBox[
            SuperscriptBox["x", "4"], "12"]}]}], ",", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"D", "[", 
           RowBox[{
            RowBox[{"u", "[", 
             RowBox[{"x", ",", "t"}], "]"}], ",", "t"}], "]"}], " ", 
          "\[Equal]", " ", "0"}]}], "}"}], " ", "/.", " ", 
       RowBox[{"t", "\[Rule]", "0"}]}], ",", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"x", ",", "t"}], "]"}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "d"}], "}"}]}], "]"}], " ", "\[Equal]", " ", 
           "0"}], ")"}], " ", "/.", " ", 
         RowBox[{"x", "\[Rule]", "b"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"b", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"d", ",", 
          RowBox[{"2", " ", "b"}], ",", " ", 
          RowBox[{
           RowBox[{"2", " ", "b"}], " ", "+", " ", "1"}]}], "}"}]}], "]"}]}], 
     "\[IndentingNewLine]", "}"}], ",", "\[IndentingNewLine]", "u", ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "2"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "\"\<StiffnessSwitching\>\""}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[25]:=",
 CellID->36540943],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"eerr\"\>"}], ":", 
  " ", "\<\"Warning: Scaled local spatial error estimate of \
\\!\\(298.4502973277475`\\) at \\!\\(t\\) = \\!\\(2.`\\) in the direction of \
independent variable \\!\\(x\\) is much greater than prescribed error \
tolerance. Grid spacing with \\!\\(25\\) points may be too large to achieve \
the desired accuracy or precision.  A singularity may have formed or you may \
want to specify a smaller grid spacing using the MaxStepSize or MinPoints \
method options.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[25]:=",
 CellID->5097741],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "2.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{373, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->124685242]
}, Open  ]],

Cell["\<\
Understanding the message about spatial error will be addressed in the next \
section. For now, ignore the message and consider the boundary conditions.\
\>", "Text",
 CellID->1635227796],

Cell[TextData[{
 "This forms a list of ",
 StyleBox["InterpolatingFunctions", "MR"],
 " differentiated to the same order as each of the boundary conditions."
}], "MathCaption",
 CellID->758884389],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"D", "[", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"x", ",", "t"}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "d"}], "}"}]}], "]"}], "  ", "/.", " ", 
      RowBox[{"x", "\[Rule]", "b"}]}], ")"}], "/.", " ", 
    RowBox[{"First", "[", "dsol", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"b", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"d", ",", 
     RowBox[{"2", " ", "b"}], ",", " ", 
     RowBox[{
      RowBox[{"2", " ", "b"}], " ", "+", " ", "1"}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->1886122732],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWttu1EAMzeayS1sKtCptQUIUoT4iAY9IPCHBEx+xqpDaBwQq/QH48ZCZ
xNvJiTOXXJYl8UolGcdjezz2XA7+ur67/vZ9fXdztb74crv+eX1z9evi84/b
gpQsomjxO4qiT+dR8Z4Xb/pP/f4Uf+Vr+fau+DcpPr6tnh+LZ1w8l+prQr3y
KIX2E2ifOvj3e/bH9qlhp3oewPfHFT1zyEm0F94DFdsfAq07qrSfM9oUHb3R
xs9Z68O3X/EdBdoda28o32UROY9+xKpsoGdaPOOKO67aqfE9g2di8PPSY0Nq
UjGpkVQTpX+KXppOP5Pb1GHqbPZKAr1D0tKcz5GlodUnhx6MLJ/NwfS+HTUs
4OIhI7eB5QlD94svjEpNzTbtaor07KyAWn1bMZ5rSuDlsjbx2nmqs69FItps
ymrtuS0LO80ma4G5Mu8x426LXo7fT8vKyeUaGemOLRno46GY8so5N4NniyW6
cB8cKWMsGnvnimff2GFHH7vY04PrdNGQgmedh9A+9JLy1NELbVFaKDZjZiyu
9qNAv2L/Rd4vmrMWep+5S0eM2017B+O228qOccuduftLQVvoJN91TXbFbfdV
uH3Hzlgq7Uap2bd2HmN370FWXl4ffTHP9HxMWext8OPprD1SN1ItY/6XsXoG
XHhv4+59YSfsEaM1NUfTuEPl9RsSxaZxQ2J60d2OO00khtQYpYIMpJlzjrHB
W6UIePvEGyBq6eTLkSILb/B46kVbXgTafpjzN7duqx3G/SvH92fsmDGbnu/Q
fJywVNfeFDorw2Q34TNv8jLGT4BLYWvLovkS6Me53w5KuUQYG55Pub2Q+BNP
r5l3N1oRzvIQzE69IpoZ/oa2vs7nhYdqOxrWXEJ7qngo9T9gvWBHKgUHFRzU
FwfdY+MrAFcRHNQuT3DQ/P/GQTe7QGPcfriH4J81qsUOwT/HwT/1Ca7PGg8j
FrxzuDgVvLNLnAq+KfhmvT0+vmmJTsEzQYvgmb6rmeCXw8yCX/ZqrbVsnTJa
eazHrFj7o5FSjam8MNdqTPP8JFWYNi8l0ZzQR7IIs0eqMG19BX2cC/rY7sd+
tZh0c5QazFBrBYPkbQnFIHkPhFZi6gyQysudiNYpI5HdopX/v1KptxQ8cgw8
cvAYFVQStEwZleS1+NZa8iubYJNDzkW3TJ5iRaWpj6/6U6/9MUtXZM2zotJ1
K5lnhaXLK3ZUUjBPwTzDKi5D/SwYqJbV2lMw0KljoPd1mK57odRlhlkrmChv
y3Yw0bIyYcD9APwgGOlw0SsY6XDRK5ipYKb19rYw04CYFQwVtAiG2m/lE0yV
t2gcTLWcI3dmTxlhDasCjRZ/AcoRoA4=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{385, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->623610076]
}, Open  ]],

Cell[TextData[{
 "This makes a logarithmic plot of how well each of the four boundary \
conditions is satisfied by the solution computed with ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " as a function of t."
}], "MathCaption",
 CellID->574634790],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", " ", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"LogPlot", "[", 
      RowBox[{
       RowBox[{"Abs", "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"x", ",", "t"}], "]"}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "d"}], "}"}]}], "]"}], "  ", "/.", " ", 
           RowBox[{"x", "\[Rule]", "b"}]}], ")"}], "/.", " ", 
         RowBox[{"First", "[", "dsol", "]"}]}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "2"}], "}"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"Hue", "[", 
         RowBox[{"d", "/", "4"}], "]"}]}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"b", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"d", ",", 
       RowBox[{"2", " ", "b"}], ",", " ", 
       RowBox[{
        RowBox[{"2", " ", "b"}], " ", "+", " ", "1"}]}], "}"}]}], "]"}], ",", 
   " ", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[27]:=",
 CellID->1821405583],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[27]=",
 CellID->557260415]
}, Open  ]],

Cell[TextData[{
 "It is clear that the boundary conditions are satisfied to well within the \
tolerances allowed by ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " options. It is typical that conditions with higher order derivatives will \
not be satisfied as well as those with lower order derivatives."
}], "Text",
 CellID->823857056],

Cell[CellGroupData[{

Cell["Inconsistent Boundary Conditions", "Subsubsection",
 CellTags->{"b:7.3", "ndsg:2.0.3.0"},
 CellID->51169332],

Cell[TextData[{
 "It is important that the boundary conditions you specify be consistent with \
both the initial condition and the PDE. If this is not the case, ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " will issue a message warning about the inconsistency. When this happens, \
the solution may not satisfy the boundary conditions, and in the worst cases, \
instability may appear."
}], "Text",
 CellID->740920019],

Cell["\<\
In this example for the heat equation, the boundary condition at x = 0 is \
clearly inconsistent with the initial condition.\
\>", "MathCaption",
 CellID->2094106888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], " ", "\[Equal]", 
       " ", 
       RowBox[{"D", "[", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], ",", 
      " ", 
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "0"}], "]"}], " ", "\[Equal]", " ", "1"}], ",", " ",
       
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"t", ",", "1"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", " ",
       
      RowBox[{
       RowBox[{"u", "[", 
        RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", ".5"}]}], "}"}],
     ",", " ", "u", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1246379112],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ibcinc\"\>"}], ":", 
  " ", "\<\"Warning: Boundary and initial conditions are inconsistent. \\!\\(\
\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"MessageLink\\\", \
ButtonFrame->None, ButtonData:>{\\\"System`.`\\\", \\\"NDSolve\\\", \
\\\"ibcinc\\\"}, ButtonNote -> \\\"NDSolve::ibcinc\\\"]\\)\"\>"}]], "Message",\
 "MSG",
 CellLabel->"During evaluation of In[2]:=",
 CellID->180362994],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"u", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{373, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->11175235]
}, Open  ]],

Cell[TextData[{
 "This shows a plot of the solution at x = 0 as a function of t.",
 " ",
 "The boundary condition ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"u", "(", 
     RowBox[{"t", ",", "0"}], ")"}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 " is clearly not satisfied."
}], "MathCaption",
 CellID->994104916],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"First", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], " ", "/.", " ", "sol"}], "]"}], "]"}], 
   ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->759190778],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztGrtOwzDQiZMiBhAjE+ofMAEtEgIWWChTxYqiCqkdEKh0gi/hU/ghfqP4
EluYsx3bbazQ1JHsnn0P353t013SUbGYPj0Xi9mk6N/Oi9fpbPLWv3mZsyma
EJJMCSHfh4TBSwaVDZ4ZaxVohkasTxkDhRGtOAmMWRvIuFxIXXKKcw0ngDA3
tOBgKtXgslL2HfTZL4aQHGl3plD0kDSVIlNmXGiGSC5VKKhBZ6zRqXa1jPu5
XV13EOeJfjVpB4Yr6Qz7mPKzE0pTsYZeQ3Guw/nSxw+qjjlat8l9F3e9J62e
anw6kOZTKQ5gPjGubu0BkgLdQwDMLrfpkbVES7tvGe950m+avHX58+jV1rx6
7Cn1qGEv+cqz6au3+soyvvak168CXYgItCpGIP6TTtH26BUXDE8hykdXK2Rm
ztpM3pUPZ9O4sjDx4dwGqg5Ryowd6V3WUbOyGsskX4ocCud9ofZxm89w9Eq0
3RWjxDvl/Yf5xjYTucTbAfiti1i4MnfVS9SgDdJLPsNx1Gx1+3vd/dO8lu2X
F5/3rBH68f7FqTYj344nBGNMWRyAPrHAK9sJymmLjNQzwsl0bhmceEtmpm5/
37tyfqNXou2umE2OdXKuZc39wusW69WOYLbZ9i54hX9q44/uC7r5XmJq9Suh
OQKE5/0b5drStalIBxD+pwVJfgBLwn3S\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 230},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->283759706]
}, Open  ]],

Cell[TextData[{
 "The reason the boundary condition is not satisfied is because once it is \
differentiated, it becomes ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["u", "t"], "(", 
     RowBox[{"t", ",", "0"}], ")"}], "=", " ", "0"}], TraditionalForm]]],
 ", so the solution will be whatever constant value comes from the initial \
condition."
}], "Text",
 CellID->687261336],

Cell["\<\
When the boundary conditions are not differentiated, the DAE solver in effect \
modifies the initial conditions so that the boundary condition is satisfied. \
\
\>", "MathCaption",
 CellID->1602777873],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"daesol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], " ", "\[Equal]",
         " ", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], ",",
        " ", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "0"}], "]"}], " ", "\[Equal]", " ", "1"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "1"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}],
      ",", " ", "u", ",", " ", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
        RowBox[{
        "\"\<DifferentiateBoundaryConditions\>\"", "\[Rule]", "False"}]}], 
       "}"}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"First", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0"}], "]"}], " ", "/.", " ", "daesol"}], "]"}], " ",
     "-", " ", "1"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->1721734455],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ibcinc\"\>"}], ":", 
  " ", "\<\"Warning: Boundary and initial conditions are inconsistent. \\!\\(\
\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"MessageLink\\\", \
ButtonFrame->None, ButtonData:>{\\\"System`.`\\\", \\\"NDSolve\\\", \
\\\"ibcinc\\\"}, ButtonNote -> \\\"NDSolve::ibcinc\\\"]\\)\"\>"}]], "Message",\
 "MSG",
 CellLabel->"During evaluation of In[4]:=",
 CellID->208051534],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ivcon\"\>"}], ":", 
  " ", "\<\"The given initial conditions were not consistent with the \
differential-algebraic equations.  NDSolve will attempt to correct the \
values. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"MessageLink\\\", ButtonFrame->None, \
ButtonData:>{\\\"System`.`\\\", \\\"NDSolve\\\", \\\"ivcon\\\"}, ButtonNote \
-> \\\"NDSolve::ivcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[4]:=",
 CellID->453499949],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ivres\"\>"}], ":", 
  " ", "\<\"NDSolve has computed initial values that give a zero residual for \
the differential-algebraic system, but some components are different from \
those specified.  If you need those to be satisfied, it is recommended that \
you give initial conditions for all dependent variables and derivatives of \
them. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"MessageLink\\\", ButtonFrame->None, \
ButtonData:>{\\\"System`.`\\\", \\\"NDSolve\\\", \\\"ivres\\\"}, ButtonNote \
-> \\\"NDSolve::ivres\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[4]:=",
 CellID->524125413],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJxN1Xk0lP/3APBJkuRjT1GS9aNS+qiIRtbKni3RQiRFhSQpyRJSoc1OkX2n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     "]]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{364, 194},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->91662772]
}, Open  ]],

Cell["\<\
It is not always the case that the DAE solver will find good initial \
conditions which lead to an effectively correct solution like this. A better \
way to handle this problem is to give an initial condition that is consistent \
with the boundary conditions, even if it is discontinuous. In this case the \
unit step function does what is needed:\
\>", "Text",
 CellID->1046772841],

Cell["\<\
This uses a discontinuous initial condition to match the boundary condition, \
giving a solution correct to the resolution of the spatial discretization.\
\>", "MathCaption",
 CellID->1719166634],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"usol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", "t"}], "]"}], " ", "\[Equal]",
         " ", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x"}], "]"}], ",", "x", ",", "x"}], "]"}]}], ",",
        " ", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "0"}], "]"}], " ", "\[Equal]", " ", "1"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"t", ",", "1"}], "]"}], " ", "\[Equal]", " ", "0"}], ",", 
       " ", 
       RowBox[{
        RowBox[{"u", "[", 
         RowBox[{"0", ",", "x"}], "]"}], " ", "\[Equal]", 
        RowBox[{"UnitStep", "[", 
         RowBox[{"-", "x"}], "]"}]}]}], "}"}], ",", " ", "u", ",", " ", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"First", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", " ", "x"}], "]"}], " ", "/.", " ", "usol"}], "]"}], 
    "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->343715762],

Cell[GraphicsData["CompressedBitmap", "\<\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