(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     91734,       2766]
NotebookOptionsPosition[     82240,       2469]
NotebookOutlinePosition[     85509,       2554]
CellTagsIndexPosition[     85373,       2546]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["NDSolve Method Plug-in Framework", "Title",
 CellTags->"c:1",
 CellID->94258360],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->483130645],

Cell[TextData[{
 "The control mechanisms set up for ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " enable you to define your own numerical integration algorithms and use \
them as specifications for the ",
 StyleBox["Method", "MR"],
 " option of ",
 StyleBox["NDSolve", "MR"],
 "."
}], "Text",
 CellID->777382341],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " accesses its numerical algorithms and the information it needs from them \
in an object-oriented manner. At each step of a numerical integration, ",
 StyleBox["NDSolve", "MR"],
 " keeps the method in a form so that it can keep private data as needed."
}], "Text",
 CellID->2065479094],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      StyleBox["AlgorithmIdentifier", "TI"], 
      StyleBox["[", "MR"], 
      StyleBox["data", "TI"], 
      StyleBox["]", "MR"]}], Cell[TextData[{
      "An algorithm object that contains any ",
      StyleBox["data", "TI"],
      " that a particular numerical ODE integration algorithm may need to \
use. The data is effectively private to the algorithm. ",
      StyleBox["AlgorithmIdentifier", "TI"],
      " should be a ",
      StyleBox["Mathematica",
       FontSlant->"Italic"],
      " symbol. The algorithm is accessed from ",
      StyleBox["NDSolve", "MR"],
      " by using the option ",
      StyleBox["Method->", "MR"],
      StyleBox["AlgorithmIdentifier.", "TI"]
     }], "Text"]}
   }], TextForm]], "DefinitionBox",
 CellID->883038498],

Cell[TextData[{
 "The structure for method data used in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->2054400571],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " does not access the data associated with an algorithm directly, so you can \
keep the information needed in any form that is convenient or efficient to \
use. The algorithm and information which might be saved in its private data \
are accessed only through method functions of the algorithm object."
}], "Text",
 CellID->1084643037],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     StyleBox["AlgorithmObject", "TI"], "[", "Step", "[", 
     RowBox[{
      StyleBox["rhs", "TI"], ",", 
      StyleBox["t", "TI"], ",", 
      StyleBox["h", "TI"], ",", 
      StyleBox["y", "TI"], ",", 
      StyleBox["yp", "TI"]}], "]", "]"}], Cell[TextData[{
     "Attempt to take a single time step of size ",
     Cell[BoxData[
      StyleBox["h", "TI"]], "InlineFormula"],
     " from time ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     " to time ",
     Cell[BoxData[
      RowBox[{
       StyleBox["t", "TI"], "+", 
       StyleBox["h", "TI"]}]], "InlineFormula"],
     " using the numerical algorithm. ",
     Cell[BoxData[
      StyleBox["y", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["yp", "TI"]], "InlineFormula"],
     " are the approximate solution vector and its time derivative, \
respectively, at time ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     ". The function should generally return a list ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        StyleBox["newh", "TI"], ",", 
        StyleBox["\[CapitalDelta]y", "TI"]}], "}"}]], "InlineFormula"],
     " where ",
     Cell[BoxData[
      StyleBox["newh", "TI"]], "InlineFormula"],
     " is the best size for the next step determined by the algorithm and ",
     Cell[BoxData[
      StyleBox["\[CapitalDelta]y", "TI"]], "InlineFormula"],
     " is the ",
     StyleBox["increment",
      FontSlant->"Italic"],
     " such that the approximate solution at time ",
     Cell[BoxData[
      RowBox[{
       StyleBox["t", "TI"], "+", 
       StyleBox["h", "TI"]}]], "InlineFormula"],
     " is given by ",
     Cell[BoxData[
      RowBox[{
       StyleBox["y", "TI"], "+", 
       StyleBox["\[CapitalDelta]y", "TI"]}]], "InlineFormula"],
     ". If the time step is too large, the function should only return the \
value ",
     Cell[BoxData[
      RowBox[{"{", 
       StyleBox["hnew", "TI"], "}"}]], "InlineFormula"],
     " where ",
     StyleBox["hnew",
      FontSlant->"Italic"],
     " should be small enough for an acceptable step. See below for complete \
descriptions of possible return values."
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["AlgorithmObject", "TI"], "[", "DifferenceOrder", "]"}], 
    Cell["\<\
Return the current asymptotic difference order of the algorithm.\
\>", "TableText"]},
   {
    RowBox[{
     StyleBox["AlgorithmObject", "TI"], "[", "StepMode", "]"}], Cell[
    TextData[{
     "Return the step mode for the algorithm object. The step mode should \
either be ",
     StyleBox["Automatic", "MR"],
     " or ",
     StyleBox["Fixed", "MR"],
     ". ",
     StyleBox["Automatic", "MR"],
     " means that the algorithm has a means to estimate error and determines \
an appropriate size ",
     Cell[BoxData[
      StyleBox["newh", "TI"]], "InlineFormula"],
     " for the next time step. ",
     StyleBox["Fixed", "MR"],
     " means that the algorithm will be called from a time step controller \
and is not expected to do any error estimation."
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->807124616],

Cell[TextData[{
 "Required method functions for algorithms used from ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->486741098],

Cell[TextData[{
 "These method functions must be defined for the algorithm work with ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". The Step method function should always return a list, but the length of \
the list depends on whether the step was successful or not. Also, some \
methods may need to compute the function value rhs[",
 StyleBox["t",
  FontSlant->"Italic"],
 " + ",
 StyleBox["h",
  FontSlant->"Italic"],
 ", ",
 StyleBox["y",
  FontSlant->"Italic"],
 " + \[CapitalDelta]",
 StyleBox["y",
  FontSlant->"Italic"],
 "] at the step end, so to avoid recomputation, you can add that to the \
list."
}], "Text",
 CellID->1396850147],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{
       StyleBox["Step", "MR"], 
       StyleBox["[", "MR"], 
       RowBox[{
        StyleBox["rhs", "TI"], 
        StyleBox[",", "TableHeader"], 
        StyleBox[" ", "TableHeader"], 
        StyleBox["t", "TI"], 
        StyleBox[",", "TableHeader"], 
        StyleBox[" ", "TableHeader"], 
        StyleBox["h", "TI"], 
        StyleBox[",", "TableHeader"], 
        StyleBox[" ", "TableHeader"], 
        StyleBox["y", "TI"], 
        StyleBox[",", "TableHeader"], 
        StyleBox[" ", "TableHeader"], 
        StyleBox["yp", "TI"]}], 
       StyleBox["]", "MR"]}], 
      StyleBox[" ", "TableHeader"], 
      StyleBox["method", "TableHeader"], 
      StyleBox[" ", "TableHeader"], 
      StyleBox["output", "TableHeader"]}], 
     StyleBox["Interpretation", "TableHeader"]},
    {
     RowBox[{
      StyleBox["{", "MR"], 
      RowBox[{
       StyleBox["newh", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["\[CapitalDelta]y", "TI"]}], 
      StyleBox["}", "MR"]}], Cell[TextData[{
      "successful step with computed solution increment ",
      StyleBox["\[CapitalDelta]y", "TI"],
      " and recommended next step ",
      StyleBox["newh.", "TI"]
     }], "Text"]},
    {
     RowBox[{
      StyleBox["{", "MR"], 
      RowBox[{
       StyleBox["newh", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["\[CapitalDelta]y", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["yph", "TI"]}], 
      StyleBox["}", "MR"]}], Cell[TextData[{
      "successful step with computed solution increment ",
      StyleBox["\[CapitalDelta]y", "TI"],
      " and recommended next step ",
      StyleBox["newh", "TI"],
      " and time derivatives computed at the step endpoint,",
      " ",
      StyleBox["yph", "TI"],
      StyleBox[" = ", "MR"],
      StyleBox["rhs", "TI"],
      StyleBox["[", "MR"],
      StyleBox["t", "TI"],
      StyleBox[" + ", "MR"],
      StyleBox["h", "TI"],
      StyleBox[", ", "MR"],
      StyleBox["y", "TI"],
      StyleBox[" + ", "MR"],
      StyleBox["\[CapitalDelta]y", "TI"],
      StyleBox["]", "MR"],
      "."
     }], "Text"]},
    {
     RowBox[{
      StyleBox["{", "MR"], 
      RowBox[{
       StyleBox["newh", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["\[CapitalDelta]y", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["yph", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["newobj", "TI"]}], 
      StyleBox["}", "MR"]}], Cell[TextData[{
      "successful step with computed solution increment ",
      StyleBox["\[CapitalDelta]y", "TI"],
      " and recommended next step ",
      StyleBox["newh", "TI"],
      " and time derivatives computed at the step endpoint, ",
      StyleBox["yph", "TI"],
      StyleBox[" = ", "MR"],
      StyleBox["rhs", "TI"],
      StyleBox["[", "MR"],
      StyleBox["t", "TI"],
      StyleBox[" + ", "MR"],
      StyleBox["h", "TI"],
      StyleBox[", ", "MR"],
      StyleBox["y", "TI"],
      StyleBox[" + ", "MR"],
      StyleBox["\[CapitalDelta]y", "TI"],
      StyleBox["]", "MR"],
      ". Any changes in the object data are returned in the new instance of \
the method object, ",
      StyleBox["newobj.", "TI"]
     }], "Text"]},
    {
     RowBox[{
      StyleBox["{", "MR"], 
      RowBox[{
       StyleBox["newh", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["\[CapitalDelta]y", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["None", "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["newobj", "TI"]}], 
      StyleBox["}", "MR"]}], 
     RowBox[{
      RowBox[{
      "successful", " ", "step", " ", "with", " ", "computed", " ", 
       "solution", " ", "increment", " ", 
       StyleBox["\[CapitalDelta]y", "TI"], 
       StyleBox[" ", "TI"], "and", " ", "recommended", " ", "next", " ", 
       "step", " ", 
       RowBox[{
        StyleBox["newh", "TI"], ".", " ", "Any"}], " ", "changes", " ", "in", 
       " ", "the", " ", "object", " ", "data", " ", "are", " ", "returned", 
       " ", "in", " ", "the", " ", "new", " ", "instance", " ", "of", " ", 
       "the", " ", "method", " ", "object"}], ",", " ", 
      RowBox[{
       StyleBox["newobj", "TI"], "."}]}]},
    {
     RowBox[{
      StyleBox["{", "MR"], 
      StyleBox["newh", "TI"], 
      StyleBox["}", "MR"]}], 
     RowBox[{
      RowBox[{
      "rejected", " ", "step", " ", "with", " ", "recommended", " ", "next", 
       " ", "step", " ", 
       StyleBox["newh", "TI"], " ", "such", " ", "that"}], " ", "|", 
      StyleBox["newh", "TI"], "|", " ", 
      RowBox[{"<", " ", 
       RowBox[{"|", 
        StyleBox["h", "TI"], "|", "."}]}]}]},
    {
     RowBox[{
      StyleBox["{", "MR"], 
      RowBox[{
       StyleBox["newh", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["$Failed", "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["None", "MR"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["newobj", "TI"]}], 
      StyleBox["}", "MR"]}], 
     RowBox[{
      RowBox[{
       RowBox[{
       "rejected", " ", "step", " ", "with", " ", "recommended", " ", "next", 
        " ", "step", " ", 
        StyleBox["newh", "TI"], " ", "such", " ", "that"}], " ", "|", 
       StyleBox["newh", "TI"], "|", " ", 
       RowBox[{"<", " ", 
        RowBox[{"|", 
         StyleBox["h", "TI"], "|", 
         RowBox[{
          RowBox[{".", " ", "Any"}], " ", "changes", " ", "in", " ", "the", 
          " ", "object", " ", "data", " ", "are", " ", "returned", " ", "in", 
          " ", "the", " ", "new", " ", "instance", " ", "of", " ", "the", " ",
           "method", " ", "object"}]}]}]}], ",", " ", 
      RowBox[{
       StyleBox["newobj", "TI"], "."}]}]}
   }], TextForm]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->1711530781],

Cell[TextData[{
 "Interpretation of ",
 StyleBox["Step", "MR"],
 " method output."
}], "Caption",
 CellID->861559550]
}, Open  ]],

Cell[CellGroupData[{

Cell["Classical Runge-Kutta", "Section",
 CellTags->"s:2",
 CellID->1701730745],

Cell["\<\
Here is an example of how to set up and access a simple numerical algorithm.\
\>", "Text",
 CellID->1781383456],

Cell["\<\
This defines a method function to take a single step towards integrating an \
ODE using the classical fourth order RungeKutta method. Since the method is \
so simple, it is not necessary to save any private data.\
\>", "MathCaption",
 CellID->1521747525],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"CRK4", "[", "]"}], "[", 
   RowBox[{"\"\<Step\>\"", "[", 
    RowBox[{
    "rhs_", ",", " ", "t_", ",", " ", "h_", ",", " ", "y_", ",", " ", "yp_"}],
     "]"}], "]"}], " ", ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"k0", ",", "k1", ",", "k2", ",", "k3"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"k0", "=", 
      RowBox[{"h", " ", "yp"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"k1", "=", 
      RowBox[{"h", " ", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"t", "+", 
          RowBox[{"h", "/", "2"}]}], ",", 
         RowBox[{"y", "+", 
          RowBox[{"k0", "/", "2"}]}]}], "]"}]}]}], ";", "\[IndentingNewLine]",
      
     RowBox[{"k2", "=", 
      RowBox[{"h", " ", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"t", "+", 
          RowBox[{"h", "/", "2"}]}], ",", 
         RowBox[{"y", "+", 
          RowBox[{"k1", "/", "2"}]}]}], "]"}]}]}], ";", "\[IndentingNewLine]",
      
     RowBox[{"k3", "=", 
      RowBox[{"h", " ", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"t", "+", "h"}], ",", 
         RowBox[{"y", "+", "k2"}]}], "]"}]}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"h", ",", " ", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"k0", "+", 
          RowBox[{"2", "  ", "k1"}], "+", 
          RowBox[{"2", "  ", "k2"}], "+", "k3"}], ")"}], "/", "6"}]}], 
      "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->34770196],

Cell[TextData[{
 "This defines a method function so that ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " can obtain the proper difference order to use for the method. The ___ \
template is used because the difference order for the method is always 4."
}], "MathCaption",
 CellID->1989270633],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"CRK4", "[", "___", "]"}], "[", "\"\<DifferenceOrder\>\"", "]"}], 
  " ", ":=", " ", "4"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1728676204],

Cell[TextData[{
 "This defines a method function for the step mode so that ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " will know how to control time steps. This algorithm method does not have \
any step control, so we define the step mode to be ",
 StyleBox["Fixed", "MR"],
 "."
}], "MathCaption",
 CellID->343094845],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"CRK4", "[", "___", "]"}], "[", "\"\<StepMode\>\"", "]"}], " ", ":=",
   " ", "Fixed"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1819449731],

Cell["\<\
This integrates the simple harmonic oscillator equation with fixed step size.\
\
\>", "MathCaption",
 CellID->1198933315],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"fixed", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"x", "[", "t", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", " ",
       
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
     "}"}], ",", "x", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", 
      RowBox[{"2", " ", "\[Pi]"}]}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "CRK4"}]}], "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->467251457],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.283185307179586`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1881782482]
}, Open  ]],

Cell[TextData[{
 "Generally using a fixed step size is less efficient than allowing the step \
size to vary with the local difficulty of the integration. Modern explicit \
Runge-Kutta methods (accessed in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " with ",
 StyleBox["Method->ExplicitRungeKutta", "MR"],
 ") have a so-called embedded error estimator that makes it possible to very \
efficiently determine appropriate step sizes. An alternative is to use \
built-in step controller methods that use extrapolation. The method ",
 StyleBox["DoubleStep", "MR"],
 " uses an extrapolation based on integrating a time step with a single step \
of size ",
 StyleBox["h",
  FontSlant->"Italic"],
 " and two steps of size ",
 StyleBox["h",
  FontSlant->"Italic"],
 "/2. The method ",
 StyleBox["Extrapolation", "MR"],
 " does a more sophisticated extrapolation and modifies the degree of \
extrapolation automatically as the integration is performed, but is generally \
used with base methods of difference orders 1 and 2."
}], "Text",
 CellID->1148943482],

Cell[TextData[{
 "This integrates the simple harmonic oscillator using the classical \
fourth-order Runge-Kutta method with steps controlled by using the ",
 StyleBox["DoubleStep", "MR"],
 " method."
}], "MathCaption",
 CellID->73770162],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dstep", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"x", "[", "t", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", " ",
       
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
     "}"}], ",", "x", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", 
      RowBox[{"2", " ", "\[Pi]"}]}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"DoubleStep", ",", " ", 
       RowBox[{"Method", "\[Rule]", "CRK4"}]}], "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->600422781],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.283185307179586`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1748875186]
}, Open  ]],

Cell[TextData[{
 "This makes a plot comparing the error in the computed solutions at the step \
ends.",
 " ",
 "The error for the ",
 StyleBox["DoubleStep", "MR"],
 " method is shown in blue."
}], "MathCaption",
 CellID->556900808],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"ploterror", "[", 
    RowBox[{
     RowBox[{"{", "sol_", "}"}], ",", " ", "opts___"}], "]"}], " ", ":=", " ",
    
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"points", " ", "=", " ", 
        RowBox[{
         RowBox[{"x", "@", 
          RowBox[{"\"\<Coordinates\>\"", "[", "1", "]"}]}], " ", "/.", " ", 
         "sol"}]}], ",", " ", "\[IndentingNewLine]", 
       RowBox[{"values", " ", "=", " ", 
        RowBox[{
         RowBox[{"x", "@", "\"\<ValuesOnGrid\>\""}], " ", "/.", " ", 
         "sol"}]}]}], "}"}], ",", " ", "\[IndentingNewLine]", 
     RowBox[{"ListPlot", "[", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"points", ",", " ", 
          RowBox[{"values", " ", "-", " ", 
           RowBox[{"Cos", "[", "points", "]"}]}]}], "}"}], "]"}], ",", " ", 
       "opts"}], "]"}]}], "\[IndentingNewLine]", "]"}]}], ";"}], "\n", 
 RowBox[{"Show", "[", 
  RowBox[{"{", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"ploterror", "[", "fixed", "]"}], ",", "\[IndentingNewLine]", 
    RowBox[{"ploterror", "[", 
     RowBox[{"dstep", ",", " ", 
      RowBox[{"PlotStyle", "\[Rule]", 
       RowBox[{"RGBColor", "[", 
        RowBox[{"0", ",", "0", ",", "1"}], "]"}]}]}], "]"}]}], 
   "\[IndentingNewLine]", "}"}], "]"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->1377805122],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 197},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->65782193]
}, Open  ]],

Cell["\<\
The fixed step size ended up with smaller overall error mostly because the \
steps are so much smaller; it required more than three times as many steps. \
For a problem where the local solution structure changes more significantly, \
the difference can be even greater.\
\>", "Text",
 CellID->1439939306],

Cell[TextData[{
 "A facility for stiffness detection is described within ",
 StyleBox[ButtonBox["DoubleStep",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveDoubleStep"], "MR"],
 "."
}], "Text",
 CellID->333282649],

Cell[TextData[{
 "For more sophisticated methods, it may be necessary or more efficient to \
set up some data for the method to use. When ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " uses a particular numerical algorithm for the first time, it calls an \
initialization function. You can define rules for the initialization that \
will set up appropriate data for your method. "
}], "Text",
 CellID->2030977303],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      StyleBox["InitializeMethod", "MR"], 
      StyleBox["[", "MR"], 
      RowBox[{
       StyleBox[
        RowBox[{"Algorithm", " ", "Identifier"}], "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["stepmode", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox["state", "TI"], 
       StyleBox[",", "MR"], 
       StyleBox[" ", "MR"], 
       StyleBox[
        RowBox[{"Algorithm", " ", "Options"}], "TI"]}], 
      StyleBox["]", "MR"]}], Cell[TextData[{
      "The expression that ",
      StyleBox["NDSolve", "MR"],
      " evaluates for initialization when it first uses an algorithm for a \
particular integration. ",
      StyleBox["stepmode", "TI"],
      " is either ",
      StyleBox["Automatic", "MR"],
      " or ",
      StyleBox["Fixed", "MR"],
      " depending on whether your method is expected to called within the \
framework of a step controller or another method or not. ",
      StyleBox["state", "TI"],
      " is the ",
      StyleBox["NDSolveState", "MR"],
      " object used by ",
      StyleBox["NDSolve", "MR"],
      ". ",
      StyleBox["Algorithm Options", "TI"],
      " is a list that contains any options given specifically with the \
specification to use the particular algorithm, for example, ",
      StyleBox["{", "MR"],
      StyleBox["opts", "TI"],
      StyleBox["}", "MR"],
      " in ",
      StyleBox["Method", "MR"],
      "\[Rule]",
      StyleBox["{", "MR"],
      StyleBox["Algorithm Identifier", "TI"],
      StyleBox[", ", "MR"],
      StyleBox["opts", "TI"],
      StyleBox["}", "MR"],
      "."
     }], "Text"]},
    {
     RowBox[{
      StyleBox[
       RowBox[{"Algorithm", " ", "Identifier"}], "TI"], 
      StyleBox[" ", "TI"], 
      StyleBox["/:", "MR"], 
      StyleBox[" ", "TI"], 
      RowBox[{
       StyleBox["InitializeMethod", "MR"], 
       StyleBox["[", "MR"], 
       RowBox[{
        StyleBox[
         RowBox[{"Algorithm", " ", "Identifier"}], "TI"], 
        StyleBox[",", "MR"], 
        StyleBox[" ", "MR"], 
        StyleBox["\n", "MR"], 
        StyleBox["stepmode_", "MR"], 
        StyleBox[",", "MR"], 
        StyleBox[" ", "MR"], 
        StyleBox["rhs_NumericalFunction", "MR"], 
        StyleBox[",", "MR"], 
        StyleBox[" ", "MR"], 
        StyleBox["state_NDSolveState", "MR"], 
        StyleBox[",", "MR"], 
        StyleBox[" ", "MR"], 
        StyleBox[
         RowBox[{"{", 
          RowBox[{"opts___", "?", "OptionQ"}], "}"}], "MR"]}], 
       StyleBox["]", "MR"]}], 
      StyleBox[" ", "MR"], 
      StyleBox[":=", "MR"], 
      StyleBox[" ", "MR"], 
      StyleBox["initialization", "TI"]}], Cell[TextData[{
      "Definition of the initialization so that the rule is associated with \
the algorithm. ",
      StyleBox["initialization", "TI"],
      " should return an algorithm object in the form ",
      StyleBox["Algorithm Identifier", "TI"],
      StyleBox["[", "MR"],
      StyleBox["data", "TI"],
      StyleBox["]", "MR"],
      "."
     }], "Text"]}
   }], TextForm]], "DefinitionBox",
 CellID->1659409838],

Cell["Initializing a method from NDSolve.", "Caption",
 CellID->1169361004],

Cell[TextData[{
 "As a system symbol, ",
 StyleBox["InitializeMethod", "MR"],
 " is protected, so to attach rules to it, you would need to unprotect it \
first. It is better to keep the rules associated with your method. A tidy way \
to do this is to make the initialization definition using ",
 Cell[BoxData[
  ButtonBox["TagSet",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TagSet"]], "InlineFormula"],
 " as shown above."
}], "Text",
 CellID->440394463],

Cell["\<\
As an example, suppose we want to redefine the Runge-Kutta method shown above \
so that instead of using the exact coefficients 2, 1/2, and 1/6, numerical \
values with the appropriate precision are used instead to make the \
computation slightly faster.\
\>", "Text",
 CellID->889731312],

Cell["\<\
This defines a method function to take a single step towards integrating an \
ODE using the classical fourth-order RungeKutta method using saved numerical \
values for the required coefficients. \
\>", "MathCaption",
 CellID->183074644],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"CRK4", "[", 
    RowBox[{"{", 
     RowBox[{"two_", ",", " ", "half_", ",", " ", "sixth_"}], "}"}], "]"}], 
   "[", 
   RowBox[{"\"\<Step\>\"", "[", 
    RowBox[{
    "rhs_", ",", " ", "t_", ",", " ", "h_", ",", " ", "y_", ",", " ", "yp_"}],
     "]"}], "]"}], " ", ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"k0", ",", "k1", ",", "k2", ",", "k3"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"k0", "=", 
      RowBox[{"h", " ", "yp"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"k1", "=", 
      RowBox[{"h", " ", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"t", "+", 
          RowBox[{"half", " ", "h"}]}], ",", 
         RowBox[{"y", "+", 
          RowBox[{"half", " ", "k0"}]}]}], "]"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"k2", "=", 
      RowBox[{"h", " ", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"t", "+", 
          RowBox[{"half", " ", "h"}]}], ",", 
         RowBox[{"y", "+", 
          RowBox[{"half", " ", "k1"}]}]}], "]"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"k3", "=", 
      RowBox[{"h", " ", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"t", "+", "h"}], ",", 
         RowBox[{"y", "+", "k2"}]}], "]"}]}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"h", ",", " ", 
       RowBox[{"sixth", " ", 
        RowBox[{"(", 
         RowBox[{"k0", "+", 
          RowBox[{"two", " ", 
           RowBox[{"(", 
            RowBox[{"k1", "+", "k2"}], ")"}]}], "+", "k3"}], ")"}]}]}], 
      "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[15]:=",
 CellID->1880333770],

Cell["\<\
This defines a rule that initializes the algorithm object with the data to be \
used later.\
\>", "MathCaption",
 CellID->26969710],

Cell[BoxData[
 RowBox[{"CRK4", " ", "/:", " ", 
  RowBox[{"NDSolve`InitializeMethod", "[", 
   RowBox[{
   "CRK4", ",", " ", "stepmode_", ",", " ", "rhs_", ",", " ", "state_", ",", 
    " ", "opts___"}], "]"}], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", "prec", "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"prec", " ", "=", " ", 
      RowBox[{"state", "@", "\"\<WorkingPrecision\>\""}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"CRK4", "[", 
      RowBox[{"N", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", " ", 
          RowBox[{"1", "/", "2"}], ",", " ", 
          RowBox[{"1", "/", "6"}]}], "}"}], ",", " ", "prec"}], "]"}], 
      "]"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[16]:=",
 CellID->281431321],

Cell[TextData[{
 "Saving the numerical values of the numbers gives between 5 and 10 percent \
speedup for a longer integration using ",
 StyleBox["DoubleStep", "MR"],
 ". "
}], "Text",
 CellID->436915611]
}, Open  ]],

Cell[CellGroupData[{

Cell["Adams methods", "Section",
 CellTags->"s:4",
 CellID->51030643],

Cell[TextData[{
 "In terms of the ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " framework, it is not really any more difficult to write an algorithm that \
controls steps automatically. However, the requirements for estimating error \
and determining an appropriate step size usually make this much more \
difficult from both the mathematical and programming standpoints. The \
following example is a partial adaptation of the FORTRAN DEABM code of \
Shampine and Watts to fit into the ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " framework. The algorithm adaptively chooses both step size and order based \
on criteria described in [",
 ButtonBox["SG75",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#23388"],
 "]."
}], "Text",
 CellID->162754684],

Cell[TextData[{
 "The first stage is to define the coefficients. The integration method uses \
variable step-size coefficients. Given a sequence of step sizes ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["h", 
      RowBox[{"n", "-", "k", "+", "1"}]], ",", " ", 
     SubscriptBox["h", 
      RowBox[{"n", "-", "k", "+", "2"}]], ",", " ", "...", ",", " ", 
     SubscriptBox["h", "n"]}], "}"}], TraditionalForm]]],
 ", where ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["h", "n"], TraditionalForm]]],
 "is the current step to take, the coefficients for the method with \
Adams-Bashforth predictor of order ",
 StyleBox["k",
  FontSlant->"Italic"],
 " and Adams-Moulton corrector of order ",
 StyleBox["k",
  FontSlant->"Italic"],
 " + 1, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["g", "j"], "(", "n", ")"}], TraditionalForm]]],
 "such that"
}], "Text",
 CellID->2006408190],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", 
    RowBox[{"n", "+", "1"}]], "=", 
   RowBox[{
    SubscriptBox["p", 
     RowBox[{"n", "+", "1"}]], "+", 
    RowBox[{
     SubscriptBox["h", "n"], " ", 
     RowBox[{
      SubscriptBox["g", "k"], "(", "n", ")"}], " ", 
     RowBox[{
      SubscriptBox["\[CapitalPhi]", "k"], "(", 
      RowBox[{"n", "+", "1"}], ")"}]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->212024040],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["p", 
    RowBox[{"n", "+", "1"}]], "=", 
   RowBox[{
    SubscriptBox["y", "n"], "+", 
    RowBox[{
     SubscriptBox["h", "n"], " ", 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"j", "=", "0"}], 
       RowBox[{"k", "-", "1"}]], 
      RowBox[{
       RowBox[{
        SubscriptBox["g", "j"], "(", "n", ")"}], " ", 
       RowBox[{
        SubsuperscriptBox["\[CapitalPhi]", "k", "*"], 
        "\[InvisibleApplication]", 
        RowBox[{"(", "n", ")"}]}]}]}]}]}]}], TraditionalForm]], "DisplayMath",\

 CellID->2070688222],

Cell[TextData[{
 "where the ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalPhi]", "j"], "(", "n", ")"}], TraditionalForm]]],
 "are the divided differences."
}], "Text",
 CellID->1915158219],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["\[CapitalPhi]", "j"], "(", "n", ")"}], "==", 
   RowBox[{
    UnderoverscriptBox["\[Product]", 
     RowBox[{"i", "=", "0"}], 
     RowBox[{"j", "-", "1"}]], 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["t", "n"], "-", 
       SubscriptBox["t", 
        RowBox[{"n", "-", "i"}]]}], ")"}], " ", 
     SuperscriptBox["\[Delta]", "k"], " ", 
     RowBox[{"f", "[", 
      RowBox[{
       SubscriptBox["t", "n"], ",", "\[Ellipsis]", ",", 
       SubscriptBox["t", 
        RowBox[{"n", "-", "j"}]]}], "]"}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellLabel->"In[19]:=",
 CellID->1769867585],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      SubscriptBox["\[CapitalPhi]", "j"], ")"}], "*"], 
    "\[InvisibleApplication]", 
    RowBox[{"(", "n", ")"}]}], "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["\[Beta]", "j"], "(", "n", ")"}], " ", 
     RowBox[{
      SubscriptBox["\[CapitalPhi]", "j"], "(", "n", ")"}], "        ", "with",
      "      ", 
     RowBox[{
      SubscriptBox["\[Beta]", "j"], "(", "n", ")"}]}], "=", 
    RowBox[{
     UnderoverscriptBox["\[Product]", 
      RowBox[{"i", "=", "0"}], 
      RowBox[{"j", "-", "1"}]], 
     FractionBox[
      RowBox[{
       SubscriptBox["t", 
        RowBox[{"n", "+", "1"}]], "-", 
       SubscriptBox["t", 
        RowBox[{"n", "-", "i"}]]}], 
      RowBox[{
       SubscriptBox["t", "n"], "-", 
       SubscriptBox["t", 
        RowBox[{
         RowBox[{"-", "i"}], "+", "n", "-", "1"}]]}]]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->1395185246],

Cell[TextData[{
 "This defines a function that computes the coefficients ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "j"], TraditionalForm]]],
 "and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Beta]", "j"], TraditionalForm]]],
 ", along with ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Sigma]", "j"], TraditionalForm]]],
 " that are used in error estimation. The formulas are from [",
 ButtonBox["HNW93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24148"],
 "] and use essentially the same notation."
}], "MathCaption",
 CellID->537984150],

Cell[BoxData[
 RowBox[{
  RowBox[{"AdamsBMCoefficients", "[", "hlist_List", "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "k", ",", "h", ",", "\[CapitalDelta]h", ",", "brat", ",", "\[Beta]", ",",
       "\[Alpha]", ",", "\[Sigma]", ",", "c"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"k", "=", 
      RowBox[{"Length", "[", "hlist", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"h", "=", 
      RowBox[{"Last", "[", "hlist", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"\[CapitalDelta]h", "=", 
      RowBox[{"Drop", "[", 
       RowBox[{
        RowBox[{"FoldList", "[", 
         RowBox[{"Plus", ",", "0", ",", 
          RowBox[{"Reverse", "[", "hlist", "]"}]}], "]"}], ",", "1"}], 
       "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"brat", "=", 
      FractionBox[
       RowBox[{"Drop", "[", 
        RowBox[{"\[CapitalDelta]h", ",", 
         RowBox[{"-", "1"}]}], "]"}], 
       RowBox[{
        RowBox[{"Drop", "[", 
         RowBox[{"\[CapitalDelta]h", ",", "1"}], "]"}], "-", "h"}]]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"\[Beta]", "=", 
      RowBox[{"FoldList", "[", 
       RowBox[{"Times", ",", "1", ",", "brat"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"\[Alpha]", "=", 
      FractionBox["h", "\[CapitalDelta]h"]}], ";", "\[IndentingNewLine]", 
     RowBox[{"\[Sigma]", "=", 
      RowBox[{"FoldList", "[", 
       RowBox[{"Times", ",", "1", ",", 
        RowBox[{"\[Alpha]", " ", 
         RowBox[{"Range", "[", 
          RowBox[{"Length", "[", "\[Alpha]", "]"}], "]"}]}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"c", "[", "0", "]"}], "=", 
      RowBox[{"Table", "[", 
       RowBox[{
        FractionBox["1", "q"], ",", 
        RowBox[{"{", 
         RowBox[{"q", ",", "1", ",", "k"}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"c", "[", "1", "]"}], "=", 
      RowBox[{"Table", "[", 
       RowBox[{
        FractionBox["1", 
         RowBox[{"q", " ", 
          RowBox[{"(", 
           RowBox[{"q", "+", "1"}], ")"}]}]], ",", 
        RowBox[{"{", 
         RowBox[{"q", ",", "1", ",", "k"}], "}"}]}], "]"}]}], ";", 
     RowBox[{"Do", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"c", "[", "j", "]"}], "=", 
        RowBox[{
         RowBox[{"Drop", "[", 
          RowBox[{
           RowBox[{"c", "[", 
            RowBox[{"j", "-", "1"}], "]"}], ",", 
           RowBox[{"-", "1"}]}], "]"}], "-", 
         FractionBox[
          RowBox[{
           RowBox[{"Drop", "[", 
            RowBox[{
             RowBox[{"c", "[", 
              RowBox[{"j", "-", "1"}], "]"}], ",", "1"}], "]"}], " ", "h"}], 
          RowBox[{
          "\[CapitalDelta]h", "\[LeftDoubleBracket]", "j", 
           "\[RightDoubleBracket]"}]]}]}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "2", ",", "k"}], "}"}]}], "]"}], ";", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"First", "[", 
           RowBox[{"c", "[", "#1", "]"}], "]"}], "&"}], ")"}], "/@", 
        RowBox[{"Range", "[", 
         RowBox[{"0", ",", "k"}], "]"}]}], ",", "\[Beta]", ",", "\[Sigma]"}], 
      "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[17]:=",
 CellID->734474579],

Cell[TextData[{
 StyleBox["hlist", "MR"],
 " is the list of step sizes ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["h", 
      RowBox[{"n", "-", "k"}]], ",", " ", 
     RowBox[{
      SubscriptBox["h", 
       RowBox[{
        RowBox[{"n", "-", "k", "+", "1"}], ","}]], "\[Ellipsis]"}], ",", " ", 
     
     SubscriptBox["h", "n"]}], "}"}], TraditionalForm]]],
 "from past steps. The constant-coefficient Adams coefficients can be \
computed once, and much more easily. Since the constant step size \
Adams-Moulton coefficients are used in error prediction for changing the \
method order, it makes sense to define them once with rules that save the \
values."
}], "Text",
 CellID->353582079],

Cell["\<\
This defines a function that computes and saves the values of the constant \
step size Adams-Moulton coefficients.\
\>", "MathCaption",
 CellID->682557607],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"Moulton", "[", "0", "]"}], "=", "1"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Moulton", "[", "m_", "]"}], ":=", 
  RowBox[{
   RowBox[{"Moulton", "[", "m", "]"}], "=", 
   RowBox[{"-", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Moulton", "[", "k", "]"}], "/", 
       RowBox[{"(", 
        RowBox[{"1", "+", "m", "-", "k"}], ")"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"k", ",", "0", ",", 
        RowBox[{"m", "-", "1"}]}], "}"}]}], "]"}]}]}]}]}], "Input",
 CellLabel->"In[18]:=",
 CellID->1881335417],

Cell[TextData[{
 "The next stage is to set up a data structure that will keep the necessary \
information between steps and define how that data should be initialized. The \
key information that needs to be saved is the list of past step sizes, ",
 StyleBox["hlist", "MR"],
 ", and the divided differences, \[CapitalPhi]. Since the method does the \
error estimation, it needs to get the correct norm to use from the ",
 StyleBox[ButtonBox["NDSolve`StateData",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveStateData"], "MR"],
 " object. Some other data such as precision is saved for optimization and \
convenience. "
}], "Text",
 CellID->1301249544],

Cell[TextData[{
 "This defines a rule for initializing the ",
 StyleBox["AdamsBM", "MR"],
 " method from ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->408323619],

Cell[BoxData[
 RowBox[{
  RowBox[{"AdamsBM", "/:", "\[IndentingNewLine]", 
   RowBox[{"NDSolve`InitializeMethod", "[", 
    RowBox[{"AdamsBM", ",", 
     RowBox[{"{", 
      RowBox[{"Automatic", ",", "DenseQ_"}], "}"}], ",", "rhs_", ",", 
     "ndstate_", ",", "opts___"}], "]"}], ":=", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "prec", ",", "norm", ",", "hlist", ",", "\[CapitalPhi]", ",", "mord"}], 
      "}"}], ",", 
     RowBox[{
      RowBox[{"mord", "=", 
       RowBox[{"MaxDifferenceOrder", "/.", "\[InvisibleSpace]", 
        RowBox[{"Flatten", "[", 
         RowBox[{"{", 
          RowBox[{"opts", ",", 
           RowBox[{"Options", "[", "AdamsBM", "]"}]}], "}"}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"mord", " ", "\[NotEqual]", " ", "\[Infinity]"}], " ", "&&", 
         " ", 
         RowBox[{"!", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"IntegerQ", "[", "mord", "]"}], " ", "&&", " ", 
            RowBox[{"mord", " ", ">", " ", "0"}]}], ")"}]}]}], ",", 
        RowBox[{"Return", "[", "$Failed", "]"}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"prec", "=", 
       RowBox[{"ndstate", "[", "\"\<WorkingPrecision\>\"", "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"norm", "=", 
       RowBox[{"ndstate", "[", "\"\<Norm\>\"", "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"hlist", "=", 
       RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"\[CapitalPhi]", "=", 
       RowBox[{"{", 
        RowBox[{"ndstate", "[", 
         RowBox[{
         "\"\<SolutionDerivativeVector\>\"", "[", "\"\<Active\>\"", "]"}], 
         "]"}], "}"}]}], ";", 
      RowBox[{"AdamsBM", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"hlist", ",", "\[CapitalPhi]", ",", 
           RowBox[{
            RowBox[{"N", "[", 
             RowBox[{"0", ",", "prec"}], "]"}], " ", 
            RowBox[{
            "\[CapitalPhi]", "\[LeftDoubleBracket]", "1", 
             "\[RightDoubleBracket]"}]}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"norm", ",", "prec", ",", "mord", ",", "0", ",", "True"}], 
          "}"}]}], "}"}], "]"}]}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->1223807452],

Cell[TextData[{
 StyleBox["hlist", "MR"],
 " is initialized to {} since at initialization time there have been no \
steps. \[CapitalPhi] is initialized to the derivative of the solution vector \
at the initial condition since the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["0", "th"], TraditionalForm]]],
 "divided difference is just the function value. Note that \[CapitalPhi] is a \
matrix. The third element, which is initialized to a zero vector, is used for \
determining the best order for the next step. It is effectively an additional \
divided difference. The use of the other parts of the data is clarified in \
the definition of the stepping function."
}], "Text",
 CellID->1920883093],

Cell[TextData[{
 "The initialization is also set up to get the value of an option that can be \
used to limit the maximum order of the method to use. In the code DEABM, this \
is limited to 12, because this is a practical limit for machine-precision \
calculations. However, in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", computations can be done in higher precision where higher-order methods \
may be of significant advantage, so there is no good reason for an absolute \
limit of this sort. Thus, we set the default of the option to be \
\[Infinity]."
}], "Text",
 CellID->1333022947],

Cell[TextData[{
 "This sets the default for the ",
 StyleBox["MaxDifferenceOrder", "MR"],
 " option of the ",
 StyleBox["AdamsBM", "MR"],
 " method."
}], "MathCaption",
 CellID->1033530945],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Options", "[", "AdamsBM", "]"}], " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"MaxDifferenceOrder", "\[Rule]", "\[Infinity]"}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->832386502],

Cell[TextData[{
 "Before proceeding to the more complicated ",
 StyleBox["Step", "MR"],
 " method functions, it makes sense to define the simple ",
 StyleBox["StepMode", "MR"],
 " and ",
 StyleBox["DifferenceOrder", "MR"],
 " method functions."
}], "Text",
 CellID->1259182109],

Cell[TextData[{
 "This defines the step mode for the ",
 StyleBox["AdamsBM", "MR"],
 " method to always be ",
 StyleBox["Automatic", "MR"],
 ". This means that it cannot be called from step controller methods that \
request fixed step sizes of possibly varying sizes. "
}], "MathCaption",
 CellID->917024601],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"AdamsBM", "[", "___", "]"}], "[", "\"\<StepMode\>\"", "]"}], "=",
    "Automatic"}], ";"}]], "Input",
 CellLabel->"In[22]:=",
 CellID->344312692],

Cell[TextData[{
 "This defines the difference order for the ",
 StyleBox["AdamsBM", "MR"],
 " method.",
 " ",
 "This varies with the number of past values saved."
}], "MathCaption",
 CellID->281059465],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"AdamsBM", "[", "data_", "]"}], "[", "\"\<DifferenceOrder\>\"", 
    "]"}], ":=", 
   RowBox[{"Length", "[", 
    RowBox[{"data", "[", 
     RowBox[{"[", 
      RowBox[{"1", ",", "2"}], "]"}], "]"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->1357419064],

Cell[TextData[{
 "Finally, the definition of the ",
 StyleBox["Step", "MR"],
 " function. The actual process of taking a step is only a few lines. The \
rest of the code handles the automatic step size and order selection \
following very closely the DEABM code of Shampine and Watts."
}], "Text",
 CellID->1234044004],

Cell[TextData[{
 "This defines the ",
 StyleBox["Step", "MR"],
 " method function for ",
 StyleBox["AdamsBM", "MR"],
 " that returns step data according to the templates described above. "
}], "MathCaption",
 CellID->464134109],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"AdamsBM", "[", "data_", "]"}], "[", 
    RowBox[{"\"\<Step\>\"", "[", 
     RowBox[{"rhs_", ",", "t_", ",", "h_", ",", "y_", ",", "yp_"}], "]"}], 
    "]"}], ":=", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "prec", ",", "norm", ",", "hlist", ",", "\[CapitalPhi]", ",", 
       "\[CapitalPhi]1", ",", "ns", ",", "starting", ",", "k", ",", "zero", 
       ",", "g", ",", "\[Beta]", ",", "\[Sigma]", ",", "p", ",", "f", ",", 
       "\[CapitalDelta]y", ",", "normh", ",", "ev", ",", "err", ",", "PE", 
       ",", "knew", ",", "hnew", ",", "temp"}], "}"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"hlist", ",", "\[CapitalPhi]", ",", "\[CapitalPhi]1"}], 
          "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
          "norm", ",", "prec", ",", "mord", ",", "ns", ",", "starting"}], 
          "}"}]}], "}"}], "=", "data"}], ";", "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "Norm", " ", "scaling", " ", "will", " ", "be", " ", "based", " ", 
        "on", " ", "current", " ", "solution", " ", 
        RowBox[{"y", "."}]}], " ", "*)"}], 
      RowBox[{"normh", "=", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"Abs", "[", "h", "]"}], " ", 
           RowBox[{"temp", "[", 
            RowBox[{"#1", ",", "y"}], "]"}]}], "&"}], ")"}], "/.", 
        "\[InvisibleSpace]", 
        RowBox[{"{", 
         RowBox[{"temp", "\[Rule]", "norm"}], "}"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"k", "=", 
       RowBox[{"Length", "[", "\[CapitalPhi]", "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"zero", "=", 
       RowBox[{"N", "[", 
        RowBox[{"0", ",", "prec"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "Keep", " ", "track", " ", "of", " ", "number", " ", "of", " ", 
        "steps", " ", "at", " ", "this", " ", "stepsize", " ", 
        RowBox[{"h", "."}]}], " ", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"Length", "[", "hlist", "]"}], ">", "0"}], "&&", 
         RowBox[{
          RowBox[{"Last", "[", "hlist", "]"}], "==", "h"}]}], ",", 
        RowBox[{"ns", "++"}], ",", 
        RowBox[{"ns", "=", "1"}]}], "]"}], ";", 
      RowBox[{"hlist", "=", 
       RowBox[{"Join", "[", 
        RowBox[{"hlist", ",", 
         RowBox[{"{", "h", "}"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"g", ",", "\[Beta]", ",", "\[Sigma]"}], "}"}], "=", 
       RowBox[{"AdamsBMCoefficients", "[", "hlist", "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"Convert", " ", "\[CapitalPhi]", " ", "to", " ", 
        SuperscriptBox["\[CapitalPhi]", "*"]}], " ", "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"\[CapitalPhi]", "=", 
       RowBox[{"\[CapitalPhi]", " ", 
        RowBox[{"Reverse", "[", "\[Beta]", "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"PE", ":", " ", 
        RowBox[{"Predict", " ", "and", " ", "evaluate"}]}], " ", "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"p", "=", 
       RowBox[{
        RowBox[{"Reverse", "[", 
         RowBox[{"Drop", "[", 
          RowBox[{"g", ",", 
           RowBox[{"-", "1"}]}], "]"}], "]"}], ".", "\[CapitalPhi]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"f", "=", 
       RowBox[{"rhs", "[", 
        RowBox[{
         RowBox[{"h", "+", "t"}], ",", 
         RowBox[{
          RowBox[{"h", " ", "p"}], "+", "y"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"Update", " ", "divided", " ", "differences"}], " ", "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"\[CapitalPhi]", "=", 
       RowBox[{"FoldList", "[", 
        RowBox[{"Plus", ",", 
         RowBox[{"zero", " ", "\[CapitalPhi]1"}], ",", "\[CapitalPhi]"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"Compute", " ", "scaled", " ", "error", " ", "estimate"}], " ",
        "*)"}], "\[IndentingNewLine]", 
      RowBox[{"ev", "=", 
       RowBox[{"f", "-", 
        RowBox[{"Last", "[", "\[CapitalPhi]", "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"err", "=", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"g", "\[LeftDoubleBracket]", 
           RowBox[{"-", "2"}], "\[RightDoubleBracket]"}], "-", 
          RowBox[{"g", "\[LeftDoubleBracket]", 
           RowBox[{"-", "1"}], "\[RightDoubleBracket]"}]}], ")"}], " ", 
        RowBox[{"normh", "[", "ev", "]"}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"First", " ", "order", " ", 
        RowBox[{"check", ":", " ", 
         RowBox[{
         "determines", " ", "if", " ", "order", " ", "should", " ", "be", " ",
           "lowered"}]}], "\[IndentingNewLine]", 
        RowBox[{
        "even", " ", "in", " ", "the", " ", "case", " ", "of", " ", "a", " ", 
         "rejected", " ", "step"}]}], " ", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"knew", "=", 
       RowBox[{"OrderCheck", "[", 
        RowBox[{
        "PE", ",", "k", ",", "\[CapitalPhi]", ",", "ev", ",", "normh", ",", 
         "\[Sigma]"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"err", ">", "1"}], ",", "\[IndentingNewLine]", 
        RowBox[{"(*", " ", 
         RowBox[{
          RowBox[{"Rejected", " ", 
           RowBox[{"step", ":", " ", 
            RowBox[{"reduce", " ", "h", " ", "by", " ", "half"}]}]}], ",", 
          " ", 
          RowBox[{
          "make", " ", "sure", " ", "starting", " ", "mode", " ", "flag", " ",
            "is", " ", "unset", " ", "and", " ", "reset", " ", 
           "\[CapitalPhi]", " ", "to", " ", "previous", " ", "values"}]}], 
         " ", "*)"}], "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"hnew", "=", 
          FractionBox["h", "2"]}], ";", 
         RowBox[{"\[CapitalDelta]y", "=", "$Failed"}], ";", 
         RowBox[{"f", "=", "None"}], ";", 
         RowBox[{"starting", "=", "False"}], ";", 
         RowBox[{"\[CapitalPhi]", "=", 
          RowBox[{"data", "\[LeftDoubleBracket]", 
           RowBox[{"1", ",", "2"}], "\[RightDoubleBracket]"}]}]}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"(*", " ", 
         RowBox[{"Sucessful", " ", 
          RowBox[{"step", ":", "\[IndentingNewLine]", 
           RowBox[{"CE", ":", " ", 
            RowBox[{"Correct", " ", "and", " ", "evaluate"}]}]}]}], " ", 
         "*)"}], "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"\[CapitalDelta]y", "=", 
          RowBox[{"h", " ", 
           RowBox[{"(", 
            RowBox[{"p", "+", 
             RowBox[{"ev", " ", 
              RowBox[{"Last", "[", "g", "]"}]}]}], ")"}]}]}], ";", 
         "\[IndentingNewLine]", 
         RowBox[{"f", "=", 
          RowBox[{"rhs", "[", 
           RowBox[{
            RowBox[{"h", "+", "t"}], ",", 
            RowBox[{"y", "+", "\[CapitalDelta]y"}]}], "]"}]}], ";", 
         RowBox[{"temp", "=", 
          RowBox[{"f", "-", 
           RowBox[{"Last", "[", "\[CapitalPhi]", "]"}]}]}], ";", 
         "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{"Update", " ", "the", " ", "divided", " ", "differences"}], 
          " ", "*)"}], "\[IndentingNewLine]", 
         RowBox[{"\[CapitalPhi]", "=", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             RowBox[{"temp", "+", "#1"}], "&"}], ")"}], "/@", 
           "\[CapitalPhi]"}]}], ";", "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{
          "Determine", " ", "best", " ", "order", " ", "and", " ", "stepsize",
            " ", "for", " ", "the", " ", "next", " ", "step"}], " ", "*)"}], 
         "\[IndentingNewLine]", 
         RowBox[{"\[CapitalPhi]1", "=", 
          RowBox[{"temp", "-", "\[CapitalPhi]1"}]}], ";", 
         RowBox[{"knew", "=", 
          RowBox[{"ChooseNextOrder", "[", 
           RowBox[{
           "starting", ",", "PE", ",", "k", ",", "knew", ",", 
            "\[CapitalPhi]1", ",", "normh", ",", "\[Sigma]", ",", "mord", ",",
             "ns"}], "]"}]}], ";", 
         RowBox[{"hnew", "=", 
          RowBox[{"ChooseNextStep", "[", 
           RowBox[{"PE", ",", "knew", ",", "h"}], "]"}]}]}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "Truncate", " ", "hlist", " ", "and", " ", "\[CapitalPhi]", " ", "to", 
        " ", "the", " ", "appropriate", " ", "length", " ", "for", " ", "the",
         " ", "chosen", " ", 
        RowBox[{"order", "."}]}], " ", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"hlist", "=", 
       RowBox[{"Take", "[", 
        RowBox[{"hlist", ",", 
         RowBox[{"1", "-", "knew"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Length", "[", "\[CapitalPhi]", "]"}], ">", "knew"}], ",", 
        RowBox[{
         RowBox[{"\[CapitalPhi]1", "=", 
          RowBox[{"\[CapitalPhi]", "\[LeftDoubleBracket]", 
           RowBox[{
            RowBox[{"Length", "[", "\[CapitalPhi]", "]"}], "-", "knew"}], 
           "\[RightDoubleBracket]"}]}], ";", 
         RowBox[{"\[CapitalPhi]", "=", 
          RowBox[{"Take", "[", 
           RowBox[{"\[CapitalPhi]", ",", 
            RowBox[{"-", "knew"}]}], "]"}]}], ";"}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "Return", " ", "step", " ", "data", " ", "along", " ", "with", " ", 
        "updated", " ", "method", " ", "data"}], " ", "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"hnew", ",", "\[CapitalDelta]y", ",", "f", ",", 
        RowBox[{"AdamsBM", "[", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"hlist", ",", "\[CapitalPhi]", ",", "\[CapitalPhi]1"}], 
            "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
            "norm", ",", "prec", ",", "mord", ",", "ns", ",", "starting"}], 
            "}"}]}], "}"}], "]"}]}], "}"}]}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->1090269186],

Cell[TextData[{
 "There are a few deviations from DEABM in the code here. The most \
significant is that coefficients are recomputed at each step, whereas DEABM \
computes only those that need updating. This modification was made to keep \
the code simpler, but does incur a clear performance loss, particularly for \
small to moderately sized systems. A second significant modification is that \
much of the code for limiting rejected steps is left to ",
 StyleBox["NDSolve", "MR"],
 ", so there are no checks in this code to see if the step size is too small \
or the tolerances are too large. The stiffness detection heuristic has also \
been left out. The order and step size determination code has been \
modularized into separate functions."
}], "Text",
 CellID->1261013714],

Cell[TextData[{
 "This defines a function that constructs error estimates ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["PE", "j"], TraditionalForm]]],
 " for ",
 StyleBox["j",
  FontSlant->"Italic"],
 " == ",
 StyleBox["k",
  FontSlant->"Italic"],
 "-2, ",
 StyleBox["k",
  FontSlant->"Italic"],
 "-1, and ",
 StyleBox["k",
  FontSlant->"Italic"],
 " and determines if the order should be lowered or not."
}], "MathCaption",
 CellID->745565430],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"OrderCheck", "[", 
    RowBox[{
    "PE_", ",", "k_", ",", "\[CapitalPhi]_", ",", "ev_", ",", "normh_", ",", 
     "\[Sigma]_"}], "]"}], ":=", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"knew", "=", "k"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       SubscriptBox["PE", "k"], "=", 
       RowBox[{"Abs", "[", 
        RowBox[{
         RowBox[{"\[Sigma]", "\[LeftDoubleBracket]", 
          RowBox[{"k", "+", "1"}], "\[RightDoubleBracket]"}], " ", 
         RowBox[{"Moulton", "[", "k", "]"}], " ", 
         RowBox[{"normh", "[", "ev", "]"}]}], "]"}]}], ";", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"k", ">", "1"}], ",", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{
          SubscriptBox["PE", 
           RowBox[{"k", "-", "1"}]], "=", 
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{
            "\[Sigma]", "\[LeftDoubleBracket]", "k", 
             "\[RightDoubleBracket]"}], " ", 
            RowBox[{"Moulton", "[", 
             RowBox[{"k", "-", "1"}], "]"}], " ", 
            RowBox[{"normh", "[", 
             RowBox[{"ev", "+", 
              RowBox[{
              "\[CapitalPhi]", "\[LeftDoubleBracket]", "2", 
               "\[RightDoubleBracket]"}]}], "]"}]}], "]"}]}], ";", 
         "\[IndentingNewLine]", 
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{"k", ">", "2"}], ",", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{
             SubscriptBox["PE", 
              RowBox[{"k", "-", "2"}]], "=", 
             RowBox[{"Abs", "[", 
              RowBox[{
               RowBox[{"\[Sigma]", "\[LeftDoubleBracket]", 
                RowBox[{"k", "-", "1"}], "\[RightDoubleBracket]"}], " ", 
               RowBox[{"Moulton", "[", 
                RowBox[{"k", "-", "2"}], "]"}], " ", 
               RowBox[{"normh", "[", 
                RowBox[{"ev", "+", 
                 RowBox[{
                 "\[CapitalPhi]", "\[LeftDoubleBracket]", "3", 
                  "\[RightDoubleBracket]"}]}], "]"}]}], "]"}]}], ";", 
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"Max", "[", 
                RowBox[{
                 SubscriptBox["PE", 
                  RowBox[{"k", "-", "1"}]], ",", 
                 SubscriptBox["PE", 
                  RowBox[{"k", "-", "2"}]]}], "]"}], "<", 
               SubscriptBox["PE", "k"]}], ",", 
              RowBox[{"knew", "=", 
               RowBox[{"k", "-", "1"}]}]}], "]"}]}]}], "]"}]}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{
            SubscriptBox["PE", 
             RowBox[{"k", "-", "1"}]], "<", 
            FractionBox[
             SubscriptBox["PE", "k"], "2"]}], ",", 
           RowBox[{"knew", "=", 
            RowBox[{"k", "-", "1"}]}]}], "]"}], ";"}]}], 
       "\[IndentingNewLine]", "]"}], ";", "\[IndentingNewLine]", "knew"}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->1527184798],

Cell["\<\
This defines a function that determines the best order to use after a \
successful step.\
\>", "MathCaption",
 CellID->1312044358],

Cell[BoxData[{
 RowBox[{
  RowBox[{"SetAttributes", "[", 
   RowBox[{"ChooseNextOrder", ",", "HoldFirst"}], "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"ChooseNextOrder", "[", 
    RowBox[{
    "starting_", ",", "PE_", ",", "k_", ",", "knw_", ",", "\[CapitalPhi]1_", 
     ",", "normh_", ",", "\[Sigma]_", ",", "mord_", ",", "ns_"}], "]"}], ":=",
    
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"knew", "=", "knw"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"starting", "=", 
       RowBox[{"starting", "&&", 
        RowBox[{"knew", "\[GreaterEqual]", "k"}], "&&", 
        RowBox[{"k", "<", "mord"}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"If", "[", 
       RowBox[{"starting", ",", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"knew", "=", 
          RowBox[{"k", "+", "1"}]}], ";", 
         RowBox[{
          SubscriptBox["PE", 
           RowBox[{"k", "+", "1"}]], "=", "0"}]}], ",", "\[IndentingNewLine]",
         
        RowBox[{
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"knew", "\[GreaterEqual]", "k"}], "&&", 
            RowBox[{"ns", "\[GreaterEqual]", 
             RowBox[{"k", "+", "1"}]}]}], ",", "\[IndentingNewLine]", 
           RowBox[{
            RowBox[{
             SubscriptBox["PE", 
              RowBox[{"k", "+", "1"}]], "=", 
             RowBox[{"Abs", "[", 
              RowBox[{
               RowBox[{"Moulton", "[", 
                RowBox[{"k", "+", "1"}], "]"}], " ", 
               RowBox[{"normh", "[", "\[CapitalPhi]1", "]"}]}], "]"}]}], ";", 
            "\[IndentingNewLine]", 
            RowBox[{"If", "[", 
             RowBox[{
              RowBox[{"k", ">", "1"}], ",", "\[IndentingNewLine]", 
              RowBox[{"If", "[", 
               RowBox[{
                RowBox[{
                 SubscriptBox["PE", 
                  RowBox[{"k", "-", "1"}]], "\[LessEqual]", 
                 RowBox[{"Min", "[", 
                  RowBox[{
                   SubscriptBox["PE", "k"], ",", 
                   SubscriptBox["PE", 
                    RowBox[{"k", "+", "1"}]]}], "]"}]}], ",", 
                "\[IndentingNewLine]", 
                RowBox[{"knew", "=", 
                 RowBox[{"k", "-", "1"}]}], ",", "\[IndentingNewLine]", 
                RowBox[{"If", "[", 
                 RowBox[{
                  RowBox[{
                   RowBox[{
                    SubscriptBox["PE", 
                    RowBox[{"k", "+", "1"}]], "<", 
                    SubscriptBox["PE", "k"]}], "&&", 
                   RowBox[{"k", "<", "mord"}]}], ",", 
                  RowBox[{"knew", "=", 
                   RowBox[{"k", "+", "1"}]}]}], "]"}]}], 
               "\[IndentingNewLine]", "]"}], ",", "\[IndentingNewLine]", 
              RowBox[{"If", "[", 
               RowBox[{
                RowBox[{
                 SubscriptBox["PE", 
                  RowBox[{"k", "+", "1"}]], "<", 
                 FractionBox[
                  SubscriptBox["PE", "k"], "2"]}], ",", 
                RowBox[{"knew", "=", 
                 RowBox[{"k", "+", "1"}]}]}], "]"}]}], "\[IndentingNewLine]", 
             "]"}], ";"}]}], "\[IndentingNewLine]", "]"}], ";"}]}], 
       "\[IndentingNewLine]", "]"}], ";", "\[IndentingNewLine]", "knew"}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[26]:=",
 CellID->908320114],

Cell[TextData[{
 "This defines a function that determines the best step size to use after a \
successful step of size ",
 StyleBox["h",
  FontSlant->"Italic"],
 ". "
}], "MathCaption",
 CellID->1386109340],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ChooseNextStep", "[", 
    RowBox[{"PE_", ",", "k_", ",", "h_"}], "]"}], ":=", "\[IndentingNewLine]",
    
   RowBox[{"If", "[", 
    RowBox[{
     RowBox[{
      SubscriptBox["PE", "k"], "<", 
      SuperscriptBox["2", 
       RowBox[{"-", 
        RowBox[{"(", 
         RowBox[{"k", "+", "2"}], ")"}]}]]}], ",", "\[IndentingNewLine]", 
     RowBox[{"2", " ", "h"}], ",", "\[IndentingNewLine]", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["PE", "k"], "<", 
        FractionBox["1", "2"]}], ",", "h", ",", 
       RowBox[{"h", " ", 
        RowBox[{"Max", "[", 
         RowBox[{
          FractionBox["1", "2"], ",", 
          RowBox[{"Min", "[", 
           RowBox[{
            FractionBox["9", "10"], ",", 
            SuperscriptBox[
             RowBox[{"(", 
              FractionBox["1", 
               RowBox[{"2", " ", 
                SubscriptBox["PE", "k"]}]], ")"}], 
             FractionBox["1", 
              RowBox[{"k", "+", "1"}]]]}], "]"}]}], "]"}]}]}], "]"}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[28]:=",
 CellID->1524068398],

Cell[TextData[{
 "Once these definitions are entered, you can access the method in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " by simply using ",
 StyleBox["Method->AdamsBM", "MR"],
 "."
}], "Text",
 CellID->831524689],

Cell["\<\
This solves the harmonic oscillator equation with the Adams method defined \
earlier. \
\>", "MathCaption",
 CellID->1153783911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"asol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"x", "[", "t", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", " ",
       
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "0"}]}], 
     "}"}], ",", "x", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", 
      RowBox[{"2", " ", "\[Pi]"}]}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "AdamsBM"}]}], "]"}]}]], "Input",
 CellLabel->"In[29]:=",
 CellID->1146452335],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "6.283185307179586`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[29]=",
 CellID->1258323235]
}, Open  ]],

Cell["\<\
This shows the error of the computed solution. It is apparent that the error \
is kept within reasonable bounds. Note that after the first few points, the \
step size has been increased.\
\>", "MathCaption",
 CellID->1880926914],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ploterror", "[", "asol", "]"}]], "Input",
 CellLabel->"In[30]:=",
 CellID->432808366],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1WEtPFEEQHmZnH6zuLosQjAQCqJGDETmQSMLLEDCY4AOCRAOJGzBAjGIQ
gzF6WA6e9KYHiYkXNPHkQSI/wMToyatn/8jYj5nunm/KnZ1ddpOdmeqpqu/r
qu6ampktba/fe1Da3lgp9VzZKj1a31h53DO9ucWGEk2WZR2w/+JJi1277Er8
D1yXnc3jRX5M8HviZ5XFaIodHTZocymp7op7lpXgZ8ccTXi6NjsXvfN94fnj
/u/x0XevX3232oR12sDjvlBOBn37HAyGeZBX+CGrcNQFG15c+MSupybfS2xJ
XP5sXzQ8pUF2auOSU7Dqgg3/+vl38+HXw29/dESKwmkz+EjWFBEuCvSUAlIX
Yh4pRUYRyJP4Dpnz6FikvLynYPwurINskKWilVcrL4hTiEQORyhJclBQVk7I
ioPMo8i/7SH668M/OyoVBkwIZ0ksI3MJaowiFQer34gaPy8D03aQpdW5yBkv
wQzbQa7Wzw6wTpOs74BWJ7CWVlhrboNWJ3CsZKW1ugHbIa2eAZbW4vwXwUcv
YNA+b4HP/gpa2tdpwLLV2vOraK0V9wXwkZ4zwMfcn7VV3OcwI4nT+Oo6B5E7
36D5zUEcaZwkmal4dZIPz0E0abR8pN/oKiif7fVXOekHa8CuW11FuQ5ZHIhl
p7MyEGJhahe86FoJWYf9GHKIFu9em4+pgu8/SVs9r7JK+lWi6GN4DIoGJoc6
ARyKILdG5rRZ4Zl64mmV0hGzLMkW/aFc8OaLvKrFFaMZsqFS60+zaiHQbS+W
ZpTaYrBROciQrZRioXkhC1pmrL1K5We66GVeZ9y0ErvRkXOdu7mveiqloSjl
QrPB6Bd8xgaDSquGzk0G5u245j4Idhnm/CSapOuvWduIA40mvCal150nh170
jwOHtUDe0nA3byBz6zUSySGRngpdseMTl8f35Ba0bH5IbKx9YWd29GS5Rriq
tLrAji9FpIV99trVD+x6YnRvYX5/eemz3E1v3/yQARFPLfr5S9f6KnWPaDQL
o25wTzYUO0ZUmkS8MbdR9RArPr6ZHHODK8iJ8I8y3U+ecoM7ZlVp8VWMnexq
BR+aWT/4HCatpsD3cAB5GnzSPso1jerndXWWXTCfPpBHhFWghw3UQOwFbZfu
YLshIn0gSxyyEzOeMOF3yKiOrwti3QcyjYs9R/xOs9yg+dB+6XoVzRo71jHI
/gSJdhQd6xjkXyLV38GOQX6l31o72nIsvf91rMHdOQIxnlR3TZte0OqAeFWy
0lodEI3JABPcC8ukz0vAZLqClkaupKURpVbgm2RN1WVXrZ7Q7mrAbqu/OgxB
TGeEGfm9LjZ/Lg5BNmj/9FM+XtUIz0pnd0Zl9+i/+w1CBGfVCqivfgxC5KRf
rB+i39TvbWrKUju6XpwB9thR34jjJxf+/suGw+Q8rziXs3w0G/5q6xoN6LxL
7WbBL60sXfP9jbaQ+8lq+gf5u7Lz\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->429524975]
}, Open  ]],

Cell["\<\
Where this method has the potential to outperform some of the built-in \
methods is with high-precision computations with strict tolerances. This is \
because the built-in methods are adapted from codes with the restriction to \
order 12. \
\>", "Text",
 CellID->1614778683],

Cell[BoxData[{
 RowBox[{
  RowBox[{"LorenzEquations", " ", "=", " ", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "'"}], "[", "t", "]"}], " ", "==", " ", 
        RowBox[{
         RowBox[{"-", "3"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"x", "[", "t", "]"}], " ", "-", " ", 
           RowBox[{"y", "[", "t", "]"}]}], ")"}]}]}], ",", " ", 
       RowBox[{
        RowBox[{"x", "[", "0", "]"}], " ", "==", " ", "0"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "'"}], "[", "t", "]"}], " ", "==", " ", 
        RowBox[{
         RowBox[{
          RowBox[{"-", 
           RowBox[{"x", "[", "t", "]"}]}], " ", 
          RowBox[{"z", "[", "t", "]"}]}], " ", "+", " ", 
         RowBox[{
          RowBox[{"53", "/", "2"}], " ", 
          RowBox[{"x", "[", "t", "]"}]}], " ", "-", " ", 
         RowBox[{"y", "[", "t", "]"}]}]}], ",", 
       RowBox[{
        RowBox[{"y", "[", "0", "]"}], " ", "==", " ", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"z", "'"}], "[", "t", "]"}], " ", "==", " ", 
        RowBox[{
         RowBox[{
          RowBox[{"x", "[", "t", "]"}], " ", 
          RowBox[{"y", "[", "t", "]"}]}], " ", "-", " ", 
         RowBox[{"z", "[", "t", "]"}]}]}], ",", " ", 
       RowBox[{
        RowBox[{"z", "[", "0", "]"}], " ", "==", " ", "0"}]}], "}"}]}], 
    "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], ",", 
     RowBox[{"y", "[", "t", "]"}], ",", 
     RowBox[{"z", "[", "t", "]"}]}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[31]:=",
 CellID->1734057911],

Cell["A lot of time is required for coefficient computation.", "Text",
 CellID->635165640],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"LorenzEquations", ",", " ", "vars", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "20"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "AdamsBM"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->691102488],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"7.04`", " ", "Second"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], "\[Rule]", 
       RowBox[{
        TagBox[
         RowBox[{"InterpolatingFunction", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
          "]"}],
         False,
         Editable->False], "[", "t", "]"}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
       RowBox[{
        TagBox[
         RowBox[{"InterpolatingFunction", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
          "]"}],
         False,
         Editable->False], "[", "t", "]"}]}], ",", 
      RowBox[{
       RowBox[{"z", "[", "t", "]"}], "\[Rule]", 
       RowBox[{
        TagBox[
         RowBox[{"InterpolatingFunction", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
          "]"}],
         False,
         Editable->False], "[", "t", "]"}]}]}], "}"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{442, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->815928492]
}, Open  ]],

Cell["\<\
This is not using as high an order as might be expected.
In any case, about half the time is spent generating coefficients, so to make \
it better, we need to figure out the coefficient update.\
\>", "Text",
 CellID->1724633433],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"LorenzEquations", ",", " ", "vars", ",", " ", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "20"}], "}"}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "AdamsBM"}], ",", " ", 
    RowBox[{"WorkingPrecision", "\[Rule]", "32"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[34]:=",
 CellID->192360210],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"11.109`", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], "\[Rule]", 
       RowBox[{
        TagBox[
         RowBox[{"InterpolatingFunction", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"0", ",", "20.`32."}], "}"}], "}"}], ",", "\<\"<>\"\>"}],
           "]"}],
         False,
         Editable->False], "[", "t", "]"}]}], ",", 
      RowBox[{
       RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
       RowBox[{
        TagBox[
         RowBox[{"InterpolatingFunction", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"0", ",", "20.`32."}], "}"}], "}"}], ",", "\<\"<>\"\>"}],
           "]"}],
         False,
         Editable->False], "[", "t", "]"}]}], ",", 
      RowBox[{
       RowBox[{"z", "[", "t", "]"}], "\[Rule]", 
       RowBox[{
        TagBox[
         RowBox[{"InterpolatingFunction", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"{", 
             RowBox[{"0", ",", "20.`32."}], "}"}], "}"}], ",", "\<\"<>\"\>"}],
           "]"}],
         False,
         Editable->False], "[", "t", "]"}]}]}], "}"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{575, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->2053349190]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NDSolve Method Plug-in Framework - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveLocallyExact"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolvePDE#1699661144"]}], 
      "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 40, 15.9626516}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The control mechanisms set up for NDSolve enable you to define your own \
numerical integration algorithms and use them as specifications for the \
Method option of NDSolve. NDSolve accesses its numerical algorithms and the \
information it needs from them in an object-oriented manner. At each step of \
a numerical integration, NDSolve keeps the method in a form so that it can \
keep private data as needed. An algorithm object that contains any data that \
a particular numerical ODE integration algorithm may need to use. The data is \
effectively private to the algorithm. AlgorithmIdentifier should be a \
Mathematica symbol. The algorithm is accessed from NDSolve by using the \
option Method->AlgorithmIdentifier.", "synonyms" -> {}, "title" -> 
    "NDSolve Method Plug-in Framework", "type" -> "Tutorial", "uri" -> 
    "tutorial/NDSolvePlugIns"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[728, 29, 86, 2, 70, "Title",
   CellTags->"c:1",
   CellID->94258360]},
 "s:1"->{
  Cell[839, 35, 69, 2, 70, "Section",
   CellTags->"s:1",
   CellID->483130645]},
 "s:2"->{
  Cell[13745, 445, 79, 2, 70, "Section",
   CellTags->"s:2",
   CellID->1701730745]},
 "s:4"->{
  Cell[35358, 1081, 69, 2, 70, "Section",
   CellTags->"s:4",
   CellID->51030643]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 84988, 2527},
 {"s:1", 85077, 2531},
 {"s:2", 85169, 2535},
 {"s:4", 85265, 2539}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 86, 2, 70, "Title",
 CellTags->"c:1",
 CellID->94258360],
Cell[CellGroupData[{
Cell[839, 35, 69, 2, 70, "Section",
 CellTags->"s:1",
 CellID->483130645],
Cell[911, 39, 389, 13, 70, "Text",
 CellID->777382341],
Cell[1303, 54, 420, 10, 70, "Text",
 CellID->2065479094],
Cell[1726, 66, 808, 23, 70, "DefinitionBox",
 CellID->883038498],
Cell[2537, 91, 216, 8, 70, "Caption",
 CellID->2054400571],
Cell[2756, 101, 469, 10, 70, "Text",
 CellID->1084643037],
Cell[3228, 113, 3166, 94, 70, "DefinitionBox",
 CellID->807124616],
Cell[6397, 209, 228, 8, 70, "Caption",
 CellID->486741098],
Cell[6628, 219, 717, 23, 70, "Text",
 CellID->1396850147],
Cell[7348, 244, 6240, 189, 70, "DefinitionBox",
 CellID->1711530781],
Cell[13591, 435, 117, 5, 70, "Caption",
 CellID->861559550]
}, Open  ]],
Cell[CellGroupData[{
Cell[13745, 445, 79, 2, 70, "Section",
 CellTags->"s:2",
 CellID->1701730745],
Cell[13827, 449, 121, 3, 70, "Text",
 CellID->1781383456],
Cell[13951, 454, 264, 5, 70, "MathCaption",
 CellID->1521747525],
Cell[14218, 461, 1560, 49, 70, "Input",
 CellID->34770196],
Cell[15781, 512, 366, 9, 70, "MathCaption",
 CellID->1989270633],
Cell[16150, 523, 191, 6, 70, "Input",
 CellID->1728676204],
Cell[16344, 531, 396, 11, 70, "MathCaption",
 CellID->343094845],
Cell[16743, 544, 188, 6, 70, "Input",
 CellID->1819449731],
Cell[16934, 552, 131, 4, 70, "MathCaption",
 CellID->1198933315],
Cell[CellGroupData[{
Cell[17090, 560, 758, 23, 70, "Input",
 CellID->467251457],
Cell[17851, 585, 483, 17, 36, "Output",
 CellID->1881782482]
}, Open  ]],
Cell[18349, 605, 1125, 27, 70, "Text",
 CellID->1148943482],
Cell[19477, 634, 237, 6, 70, "MathCaption",
 CellID->73770162],
Cell[CellGroupData[{
Cell[19739, 644, 864, 26, 70, "Input",
 CellID->600422781],
Cell[20606, 672, 483, 17, 36, "Output",
 CellID->1748875186]
}, Open  ]],
Cell[21104, 692, 231, 8, 70, "MathCaption",
 CellID->556900808],
Cell[CellGroupData[{
Cell[21360, 704, 1444, 39, 173, "Input",
 CellID->1377805122],
Cell[22807, 745, 4388, 76, 218, "Output",
 Evaluatable->False,
 CellID->65782193]
}, Open  ]],
Cell[27210, 824, 314, 6, 70, "Text",
 CellID->1439939306],
Cell[27527, 832, 226, 7, 70, "Text",
 CellID->333282649],
Cell[27756, 841, 489, 11, 70, "Text",
 CellID->2030977303],
Cell[28248, 854, 3137, 97, 70, "DefinitionBox",
 CellID->1659409838],
Cell[31388, 953, 75, 1, 70, "Caption",
 CellID->1169361004],
Cell[31466, 956, 461, 12, 70, "Text",
 CellID->440394463],
Cell[31930, 970, 298, 6, 70, "Text",
 CellID->889731312],
Cell[32231, 978, 246, 5, 70, "MathCaption",
 CellID->183074644],
Cell[32480, 985, 1695, 53, 70, "Input",
 CellID->1880333770],
Cell[34178, 1040, 141, 4, 70, "MathCaption",
 CellID->26969710],
Cell[34322, 1046, 792, 22, 70, "Input",
 CellID->281431321],
Cell[35117, 1070, 204, 6, 70, "Text",
 CellID->436915611]
}, Open  ]],
Cell[CellGroupData[{
Cell[35358, 1081, 69, 2, 70, "Section",
 CellTags->"s:4",
 CellID->51030643],
Cell[35430, 1085, 927, 23, 70, "Text",
 CellID->162754684],
Cell[36360, 1110, 921, 30, 70, "Text",
 CellID->2006408190],
Cell[37284, 1142, 439, 15, 70, "DisplayMath",
 CellID->212024040],
Cell[37726, 1159, 598, 21, 70, "DisplayMath",
 CellID->2070688222],
Cell[38327, 1182, 210, 8, 70, "Text",
 CellID->1915158219],
Cell[38540, 1192, 679, 23, 70, "DisplayMath",
 CellID->1769867585],
Cell[39222, 1217, 986, 34, 70, "DisplayMath",
 CellID->1395185246],
Cell[40211, 1253, 592, 19, 70, "MathCaption",
 CellID->537984150],
Cell[40806, 1274, 3331, 94, 70, "Input",
 CellID->734474579],
Cell[44140, 1370, 726, 21, 70, "Text",
 CellID->353582079],
Cell[44869, 1393, 165, 4, 70, "MathCaption",
 CellID->682557607],
Cell[45037, 1399, 580, 19, 70, "Input",
 CellID->1881335417],
Cell[45620, 1420, 664, 13, 70, "Text",
 CellID->1301249544],
Cell[46287, 1435, 268, 10, 70, "MathCaption",
 CellID->408323619],
Cell[46558, 1447, 2368, 63, 70, "Input",
 CellID->1223807452],
Cell[48929, 1512, 700, 14, 70, "Text",
 CellID->1920883093],
Cell[49632, 1528, 599, 12, 70, "Text",
 CellID->1333022947],
Cell[50234, 1542, 189, 7, 70, "MathCaption",
 CellID->1033530945],
Cell[50426, 1551, 247, 8, 70, "Input",
 CellID->832386502],
Cell[50676, 1561, 277, 9, 70, "Text",
 CellID->1259182109],
Cell[50956, 1572, 308, 8, 70, "MathCaption",
 CellID->917024601],
Cell[51267, 1582, 205, 7, 70, "Input",
 CellID->344312692],
Cell[51475, 1591, 201, 7, 70, "MathCaption",
 CellID->281059465],
Cell[51679, 1600, 329, 11, 70, "Input",
 CellID->1357419064],
Cell[52011, 1613, 318, 7, 70, "Text",
 CellID->1234044004],
Cell[52332, 1622, 227, 7, 70, "MathCaption",
 CellID->464134109],
Cell[52562, 1631, 10396, 254, 70, "Input",
 CellID->1090269186],
Cell[62961, 1887, 780, 13, 70, "Text",
 CellID->1261013714],
Cell[63744, 1902, 444, 19, 70, "MathCaption",
 CellID->745565430],
Cell[64191, 1923, 3156, 84, 70, "Input",
 CellID->1527184798],
Cell[67350, 2009, 140, 4, 70, "MathCaption",
 CellID->1312044358],
Cell[67493, 2015, 3436, 86, 70, "Input",
 CellID->908320114],
Cell[70932, 2103, 205, 7, 70, "MathCaption",
 CellID->1386109340],
Cell[71140, 2112, 1168, 36, 70, "Input",
 CellID->1524068398],
Cell[72311, 2150, 297, 10, 70, "Text",
 CellID->831524689],
Cell[72611, 2162, 138, 4, 70, "MathCaption",
 CellID->1153783911],
Cell[CellGroupData[{
Cell[72774, 2170, 762, 23, 70, "Input",
 CellID->1146452335],
Cell[73539, 2195, 484, 17, 36, "Output",
 CellID->1258323235]
}, Open  ]],
Cell[74038, 2215, 238, 5, 70, "MathCaption",
 CellID->1880926914],
Cell[CellGroupData[{
Cell[74301, 2224, 109, 3, 28, "Input",
 CellID->432808366],
Cell[74413, 2229, 1705, 32, 108, "Output",
 Evaluatable->False,
 CellID->429524975]
}, Open  ]],
Cell[76133, 2264, 284, 6, 70, "Text",
 CellID->1614778683],
Cell[76420, 2272, 1819, 55, 70, "Input",
 CellID->1734057911],
Cell[78242, 2329, 90, 1, 70, "Text",
 CellID->635165640],
Cell[CellGroupData[{
Cell[78357, 2334, 317, 8, 70, "Input",
 CellID->691102488],
Cell[78677, 2344, 1430, 48, 71, "Output",
 CellID->815928492]
}, Open  ]],
Cell[80122, 2395, 238, 5, 70, "Text",
 CellID->1724633433],
Cell[CellGroupData[{
Cell[80385, 2404, 379, 9, 70, "Input",
 CellID->192360210],
Cell[80767, 2415, 1407, 47, 88, "Output",
 CellID->2053349190]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[82213, 2467, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

