(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     98841,       3231]
NotebookOptionsPosition[     85369,       2771]
NotebookOutlinePosition[     88555,       2856]
CellTagsIndexPosition[     88419,       2848]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Composition and Splitting Methods for NDSolve", "Title",
 CellTags->"c:1",
 CellID->174109254],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->1525338532],

Cell[TextData[{
 "In some cases it is useful to split the differential system into subsystems \
and solve each subsystem using appropriate integration methods. Recombining \
the individual solutions often allows certain dynamical properties, such as \
volume, to be conserved. ",
 "More information on splitting and composition can be found in [",
 ButtonBox["MQ02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#31673"],
 ", ",
 ButtonBox["HLW02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2433"],
 "], and specific aspects related to ",
 StyleBox["NDSolve", "MR"],
 " are discussed in [",
 ButtonBox["SS05",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#17586"],
 ", ",
 ButtonBox["SS06",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#146830885"],
 "]."
}], "Text",
 CellID->1630494851],

Cell[CellGroupData[{

Cell["Definitions", "Subsection",
 CellID->649779944],

Cell[TextData[{
 "We are concerned with initial value problems ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     RowBox[{
      RowBox[{"y", "'"}], 
      RowBox[{"(", "t", ")"}]}], "=", " ", 
     RowBox[{"f", "(", 
      RowBox[{"y", "(", "t", ")"}], ")"}]}]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"y", "(", "0", ")"}], "=", 
    RowBox[{
     SubscriptBox["y", "0"], "\[Element]", 
     SuperscriptBox["\[DoubleStruckCapitalR]", "n"]}]}], TraditionalForm]]],
 "given."
}], "Text",
 CellID->898893221]
}, Open  ]],

Cell[CellGroupData[{

Cell["Composition", "Subsection",
 CellID->406958863],

Cell["\<\
Composition is a useful device for raising the order of a numerical \
integration scheme.\
\>", "Text",
 CellID->786614357],

Cell["\<\
In contrast to the Aitken-Neville algorithm used in extrapolation, \
composition can conserve geometric properties of the base integration method \
(e.g. symplecticity).\
\>", "Text",
 CellID->774074422],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   SubsuperscriptBox["\[CapitalPhi]", 
    RowBox[{
     SubscriptBox["f", ","], 
     SubscriptBox["\[Gamma]", "i"], "h"}], 
    RowBox[{"(", "i", ")"}]], TraditionalForm]]],
 " be a basic integration method that takes a step of size ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Gamma]", "i"], "h"}], TraditionalForm]]],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Gamma]", "1"], ",", "\[Ellipsis]", ",", 
    SubscriptBox["\[Gamma]", "s"]}], TraditionalForm]]],
 " given real numbers."
}], "Text",
 CellID->1577685144],

Cell[TextData[{
 "Then the ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]]],
 " stage composition method ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPsi]", 
    RowBox[{"f", ",", " ", "h"}]], TraditionalForm]]],
 " is given by:"
}], "Text",
 CellID->12509670],

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["\[CapitalPsi]", 
    RowBox[{"f", ",", " ", "h"}]], "=", " ", 
   RowBox[{
    SubsuperscriptBox["\[CapitalPhi]", 
     RowBox[{"f", ",", " ", 
      RowBox[{
       SubscriptBox["\[Gamma]", "s"], "h"}]}], 
     RowBox[{"(", "s", ")"}]], "\[EmptySmallCircle]", " ", 
    "\[CenterEllipsis]", " ", "\[EmptySmallCircle]", " ", 
    SubsuperscriptBox["\[CapitalPhi]", 
     RowBox[{"f", ",", " ", 
      RowBox[{
       SubscriptBox["\[Gamma]", "1"], "h"}]}], 
     RowBox[{"(", "1", ")"}]]}]}], TraditionalForm]]]], "Text",
 CellID->1734802721],

Cell[TextData[{
 "Often we are interested in composition methods ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPsi]", 
    RowBox[{"f", ",", " ", "h"}]], TraditionalForm]]],
 " that involve the same base method ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[CapitalPhi]", " ", "=", " ", 
     SuperscriptBox["\[CapitalPhi]", 
      RowBox[{"(", "i", ")"}]]}], ",", " ", 
    RowBox[{"i", " ", "=", "1"}], ",", "\[Ellipsis]", ",", 
    RowBox[{"s", "."}]}], TraditionalForm]]]
}], "Text",
 CellID->394448947],

Cell[TextData[{
 "An interesting special case is symmetric composition: ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Gamma]", "i"], " ", "=", " ", 
    SubscriptBox["\[Gamma]", 
     RowBox[{"s", "-", "i", "+", "1"}]]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"i", "=", "1"}], ",", "\[Ellipsis]", ",", 
    RowBox[{"\[LeftFloor]", 
     RowBox[{"s", "/", "2"}], "\[RightFloor]"}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1210214738],

Cell["The most common types of composition are:", "Text",
 CellID->797495146],

Cell["Symmetric composition of symmetric second-order methods.", \
"BulletedText",
 CellID->1134072999],

Cell[TextData[{
 "Symmetric composition of first-order methods (e.g. a method ",
 Cell[BoxData[
  FormBox["\[CapitalPhi]", TraditionalForm]]],
 " with its adjoint ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[CapitalPhi]", "*"], TraditionalForm]]],
 ")."
}], "BulletedText",
 CellID->2115316660],

Cell["Composition of first-order methods.", "BulletedText",
 CellID->1517362642]
}, Open  ]],

Cell[CellGroupData[{

Cell["Splitting", "Subsection",
 CellID->280476626],

Cell[TextData[{
 "An ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]]],
 "-stage splitting method is a generalization of a composition method in \
which ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]]],
 " is broken up in an additive fashion:"
}], "Text",
 CellID->1356823117],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " ", "=", " ", 
    RowBox[{
     SubscriptBox["f", "1"], " ", "+", " ", "\[CenterEllipsis]", " ", "+", 
     " ", 
     SubscriptBox["f", "k"]}]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", " ", "\[LessEqual]", " ", "s"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->48180321],

Cell[TextData[{
 "The essential point is that there can often be computational advantages in \
solving problems involving ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["f", "i"], TraditionalForm]]],
 " instead of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]]],
 "."
}], "Text",
 CellID->752549885],

Cell[TextData[{
 "An ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]]],
 " stage splitting method is a composition of the form:"
}], "Text",
 CellID->1687764060],

Cell[TextData[Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["\[CapitalPsi]", 
    RowBox[{"f", ",", " ", "h"}]], "=", " ", 
   RowBox[{
    SubsuperscriptBox["\[CapitalPhi]", 
     RowBox[{
      SubscriptBox["f", "s"], ",", 
      RowBox[{
       SubscriptBox["\[Gamma]", "s"], "h"}]}], 
     RowBox[{"(", "s", ")"}]], "\[EmptySmallCircle]", " ", 
    "\[CenterEllipsis]", " ", "\[EmptySmallCircle]", " ", 
    SubsuperscriptBox["\[CapitalPhi]", 
     RowBox[{
      SubscriptBox["f", "1"], ",", " ", 
      RowBox[{
       SubscriptBox["\[Gamma]", "1"], "h"}]}], 
     RowBox[{"(", "1", ")"}]]}]}], TraditionalForm]]]], "Text",
 CellID->110167493],

Cell[TextData[{
 "with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["f", "1"], ",", "\[Ellipsis]", ",", 
    SubscriptBox["f", "s"]}], TraditionalForm]]],
 " not necessarily distinct."
}], "Text",
 CellID->1565446313],

Cell["\<\
Each base integration method now only solves part of the problem, but a \
suitable composition can still give rise to a numerical scheme with \
advantageous properties.\
\>", "Text",
 CellID->200748626],

Cell[TextData[{
 "If the vector field ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["f", "i"], TraditionalForm]]],
 " is integrable, then the exact solution or flow ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CurlyPhi]", 
    RowBox[{
     SubscriptBox["f", "i"], ",", " ", "h"}]], TraditionalForm]]],
 " can be used in place of a numerical integration method."
}], "Text",
 CellID->2069068573],

Cell["\<\
A splitting method may also use a mixture of flows and numerical methods.\
\>", "Text",
 CellID->1571823932],

Cell[TextData[{
 "An example is Lie-Trotter splitting [",
 ButtonBox["T59",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#8091"],
 "]:\nSplit ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " ", "=", " ", 
    RowBox[{
     SubscriptBox["f", "1"], "+", 
     SubscriptBox["f", "2"]}]}], TraditionalForm]]],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Gamma]", "1"], "=", " ", 
    RowBox[{
     SubscriptBox["\[Gamma]", "2"], " ", "=", " ", "1"}]}], 
   TraditionalForm]]],
 "; then ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalPsi]", 
     RowBox[{"f", ",", " ", "h"}]], "=", " ", 
    RowBox[{
     SubsuperscriptBox["\[CurlyPhi]", 
      RowBox[{
       SubscriptBox["f", "2"], ",", "h"}], 
      RowBox[{"(", "2", ")"}]], "\[EmptySmallCircle]", " ", 
     SubsuperscriptBox["\[CurlyPhi]", 
      RowBox[{
       SubscriptBox["f", "1"], ",", " ", "h"}], 
      RowBox[{"(", "1", ")"}]]}]}], TraditionalForm]]],
 " yields a first-order integration method."
}], "Text",
 CellID->2116694905],

Cell["\<\
Computationally it can be advantageous to combine flows using the group \
property:\
\>", "Text",
 CellID->1134364391],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CurlyPhi]", 
     RowBox[{
      SubscriptBox["f", "i"], ",", " ", 
      RowBox[{
       SubscriptBox["h", "1"], "+", 
       SubscriptBox["h", "2"]}]}]], "=", " ", 
    RowBox[{
     SubscriptBox["\[CurlyPhi]", 
      RowBox[{
       SubscriptBox["f", "i"], ",", " ", 
       SubscriptBox["h", "2"]}]], " ", "\[EmptySmallCircle]", " ", 
     SubscriptBox["\[CurlyPhi]", 
      RowBox[{
       SubscriptBox["f", "i"], ",", " ", 
       SubscriptBox["h", "1"]}]]}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1043697469]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implementation", "Subsection",
 CellID->226936990],

Cell[TextData[{
 "Several changes to the new ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " framework were needed in order to implement splitting and composition \
methods."
}], "Text",
 CellID->1308473645],

Cell["Allow a method to call an arbitrary number of submethods.", \
"BulletedText",
 CellID->421552353],

Cell["\<\
The ability to pass around a function for numerically evaluating a subfield, \
instead of the entire vector field.\
\>", "BulletedText",
 CellID->1857431841],

Cell[TextData[{
 "A ",
 StyleBox["LocallyExact", "MR"],
 " method to compute the flow; analytically solve a subsystem and advance the \
(local) solution numerically."
}], "BulletedText",
 CellID->1958253590],

Cell[CellGroupData[{

Cell["\<\
Cache data for identical methods to avoid repeated initialization. Data for \
numerically evaluating identical subfields is also cached.\
\>", "BulletedText",
 CellID->1320445575],

Cell["\<\
A simplified input syntax allows omitted vector fields and methods to be \
filled in cyclically. These must be defined unambiguously:\
\>", "Text",
 CellID->116907056],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{"{", 
        RowBox[{
         SubscriptBox["f", "1"], ","}]}],
       TraditionalForm], 
      SubscriptBox["f", "2"]}], ",", 
     SubscriptBox["f", "1"], ",", 
     SubscriptBox["f", "2"]}], "}"}], TraditionalForm]]],
 " can be input as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     FormBox[
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["f", "1"], ","}]}],
      TraditionalForm], 
     SubscriptBox["f", "2"]}], "}"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->597384299],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{"{", 
        RowBox[{
         SubscriptBox["f", "1"], ","}]}],
       TraditionalForm], 
      SubscriptBox["f", "2"]}], ",", 
     SubscriptBox["f", "3"], ",", 
     SubscriptBox["f", "2"], ",", 
     SubscriptBox["f", "1"]}], "}"}], TraditionalForm]]],
 "cannot be input as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{"{", 
        RowBox[{
         SubscriptBox["f", "1"], ","}]}],
       TraditionalForm], 
      SubscriptBox["f", "2"]}], ",", 
     SubscriptBox["f", "3"]}], "}"}], TraditionalForm]]],
 " since this corresponds to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{"{", 
        RowBox[{
         SubscriptBox["f", "1"], ","}]}],
       TraditionalForm], 
      SubscriptBox["f", "2"]}], ",", 
     SubscriptBox["f", "3"], ",", 
     SubscriptBox["f", "1"], ",", 
     SubscriptBox["f", "2"]}], "}"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->2094519997]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Nested methods", "Subsection",
 CellID->1694592200],

Cell["\<\
The following example constructs a high-order splitting method from a \
low-order splitting using composition.\
\>", "Text",
 CellID->609893969],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   FrameBox[
    StyleBox["NDSolve", "MR"],
    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], TraditionalForm]]],
 " ",
 Cell[BoxData[
  FormBox[GridBox[{
     {"\[LongRightArrow]", 
      RowBox[{
       FrameBox[
        StyleBox["Composition", "MR"],
        FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", 
       FormBox[GridBox[{
          {"\[UpperRightArrow]", 
           RowBox[{
            FrameBox[
             RowBox[{
              RowBox[{
               StyleBox["Splitting", "MR"], 
               StyleBox[" ", "Text"], 
               StyleBox["f", "Text"]}], 
              StyleBox["=", "Text"], 
              RowBox[{
               SubscriptBox[
                StyleBox["f", "Text"], "1"], "+", 
               SubscriptBox["f", "2"]}]}],
             FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", " ", 
            FormBox[GridBox[{
               {"\[UpperRightArrow]", 
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
               {"\[LongRightArrow]", GridBox[{
                  {
                   FrameBox[
                    RowBox[{
                    StyleBox["ImplicitMidpoint", "MR"], " ", 
                    SubscriptBox["f", "2"]}],
                    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], ""}
                 }]},
               {"\[LowerRightArrow]", GridBox[{
                  {
                   FrameBox[
                    RowBox[{
                    StyleBox["LocallyExact", "MR"], " ", 
                    SubscriptBox["f", "1"]}],
                    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], ""}
                 }]}
              }],
             TraditionalForm]}]},
          {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
          {"\[LongRightArrow]", 
           RowBox[{
            FrameBox[
             RowBox[{
              RowBox[{
               StyleBox["Splitting", "MR"], 
               StyleBox[" ", "Text"], 
               StyleBox["f", "Text"]}], 
              StyleBox["=", "Text"], 
              RowBox[{
               SubscriptBox[
                StyleBox["f", "Text"], "1"], "+", 
               SubscriptBox["f", "2"]}]}],
             FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", " ", 
            FormBox[GridBox[{
               {"\[UpperRightArrow]", 
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
               {"\[LongRightArrow]", GridBox[{
                  {
                   FrameBox[
                    RowBox[{
                    StyleBox["ImplicitMidpoint", "MR"], " ", 
                    SubscriptBox["f", "2"]}],
                    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], ""}
                 }]},
               {"\[LowerRightArrow]", GridBox[{
                  {
                   FrameBox[
                    RowBox[{
                    StyleBox["LocallyExact", "MR"], " ", 
                    SubscriptBox["f", "1"]}],
                    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], ""}
                 }]}
              }],
             TraditionalForm]}]},
          {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
          {"\[LowerRightArrow]", 
           RowBox[{
            FrameBox[
             RowBox[{
              RowBox[{
               StyleBox["Splitting", "MR"], 
               StyleBox[" ", "Text"], 
               StyleBox["f", "Text"]}], 
              StyleBox["=", "Text"], 
              RowBox[{
               SubscriptBox[
                StyleBox["f", "Text"], "1"], "+", 
               SubscriptBox["f", "2"]}]}],
             FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", " ", 
            FormBox[GridBox[{
               {"\[UpperRightArrow]", 
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
               {"\[LongRightArrow]", GridBox[{
                  {
                   FrameBox[
                    RowBox[{
                    StyleBox["ImplicitMidpoint", "MR"], " ", 
                    SubscriptBox["f", "2"]}],
                    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], ""}
                 }]},
               {"\[LowerRightArrow]", GridBox[{
                  {
                   FrameBox[
                    RowBox[{
                    StyleBox["LocallyExact", "MR"], " ", 
                    SubscriptBox["f", "1"]}],
                    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], ""}
                 }]}
              }],
             TraditionalForm]}]}
         }],
        TraditionalForm]}]}
    }], TraditionalForm]]]
}], "Text",
 CellID->1681839070]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simplification", "Subsection",
 CellID->2089041147],

Cell[TextData[{
 "A more efficient integrator can be obtained in the previous example using \
the group property of flows and calling the ",
 StyleBox["Splitting", "MR"],
 " method directly."
}], "Text",
 CellID->1820108707],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   FrameBox[
    StyleBox["NDSolve", "MR"],
    FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], TraditionalForm]]],
 " ",
 Cell[BoxData[
  FormBox[GridBox[{
     {"\[LongRightArrow]", 
      RowBox[{
       FrameBox[
        RowBox[{
         RowBox[{
          StyleBox["Splitting", "MR"], 
          StyleBox[" ", "Text"], 
          StyleBox["f", "Text"]}], 
         StyleBox["=", "Text"], 
         RowBox[{
          SubscriptBox[
           StyleBox["f", "Text"], "1"], "+", 
          SubscriptBox["f", "2"]}]}],
        FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], 
       FormBox[GridBox[{
          {"\[UpperRightArrow]", 
           RowBox[{GridBox[{
              {
               RowBox[{
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " "}]},
              {
               FrameBox[
                RowBox[{
                 StyleBox["ImplicitMidpoint", "MR"], " ", 
                 SubscriptBox["f", "2"]}],
                FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}]}
             }], 
            FormBox["",
             TraditionalForm]}]},
          {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
          {"\[LongRightArrow]", 
           RowBox[{GridBox[{
              {
               RowBox[{
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " "}]},
              {
               FrameBox[
                RowBox[{
                 StyleBox["ImplicitMidpoint", "MR"], " ", 
                 SubscriptBox["f", "2"]}],
                FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
              {
               RowBox[{
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " "}]}
             }], " ", 
            FormBox["",
             TraditionalForm]}]},
          {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
          {"\[LowerRightArrow]", 
           RowBox[{GridBox[{
              {
               FrameBox[
                RowBox[{
                 StyleBox["ImplicitMidpoint", "MR"], " ", 
                 SubscriptBox["f", "2"]}],
                FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
              {
               RowBox[{
                FrameBox[
                 RowBox[{
                  StyleBox["LocallyExact", "MR"], "  ", 
                  SubscriptBox["f", "1"]}],
                 FrameMargins->{{0.2, 0.2}, {0.4, 0.4}}], " "}]}
             }], "  "}]}
         }],
        TraditionalForm]}]}
    }], TraditionalForm]]]
}], "Text",
 CellID->331850569]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellTags->"s:2",
 CellID->1075630498],

Cell[TextData[{
 "The following examples will use a second-order symmetric splitting known as \
the Strang splitting [",
 ButtonBox["S68",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25"],
 "], [",
 ButtonBox["M68",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#24932"],
 "]. ",
 "The splitting coefficients are automatically determined from the structure \
of the equations."
}], "Text",
 CellID->1787941719],

Cell[TextData[{
 "This defines a method known as symplectic leapfrog in terms of the method \
",
 StyleBox["SymplecticPartitionedRungeKutta", "MR"],
 "."
}], "MathCaption",
 CellID->1356107125],

Cell[BoxData[
 RowBox[{
  RowBox[{"SymplecticLeapfrog", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"\"\<SymplecticPartitionedRungeKutta\>\"", ",", " ", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", " ", 
     RowBox[{"\"\<PositionVariables\>\"", ":>", "qvars"}]}], "}"}]}], 
  ";"}]], "Input",
 InitializationCell->True,
 CellLabel->"In[2]:=",
 CellID->997281188],

Cell["Load a package with some useful example problems.", "MathCaption",
 CellID->1897393532],

Cell[BoxData[
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}]], "Input",
 InitializationCell->True,
 CellLabel->"In[3]:=",
 CellID->2108657010]
}, Open  ]],

Cell[CellGroupData[{

Cell["Symplectic splitting", "Section",
 CellID->537561600],

Cell[CellGroupData[{

Cell["Symplectic leapfrog", "Subsection",
 CellID->2007561025],

Cell[TextData[{
 StyleBox["SymplecticPartitionedRungeKutta", "MR"],
 " is an efficient method for solving separable Hamiltonian systems ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"H", "(", 
     RowBox[{"p", ",", " ", "q"}], ")"}], " ", "=", " ", 
    RowBox[{
     RowBox[{"T", "(", "p", ")"}], " ", "+", " ", 
     RowBox[{"V", "(", "q", ")"}]}]}], TraditionalForm]]],
 " with favorable long time dynamics."
}], "Text",
 CellID->1526619675],

Cell[TextData[{
 StyleBox["Splitting", "MR"],
 " is a more general-purpose method, but it can be used to construct \
partitioned symplectic methods (though it is somewhat less efficient than ",
 StyleBox["SymplecticPartitionedRungeKutta", "MR"],
 ")."
}], "Text",
 CellID->738310227],

Cell[TextData[{
 "Consider the harmonic oscillator that arises from a linear differential \
system that is governed by the separable Hamiltonian ",
 StyleBox["H",
  FontSlant->"Italic"],
 "(",
 StyleBox["p",
  FontSlant->"Italic"],
 ",",
 StyleBox["q",
  FontSlant->"Italic"],
 ") = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["p", "2"], "/", "2"}], TraditionalForm]]],
 "+ ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["q", "2"], "/", "2"}], TraditionalForm]]]
}], "Text",
 CellID->1929145950],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"system", " ", "=", " ", 
  RowBox[{
  "GetNDSolveProblem", "[", "\"\<HarmonicOscillator\>\"", "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->950959959],

Cell[BoxData[
 RowBox[{"NDSolveProblem", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "1"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "2"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{"-", 
        RowBox[{
         SubscriptBox["Y", "1"], "[", "T", "]"}]}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "1"], "[", "0", "]"}], "\[Equal]", "1"}], ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "2"], "[", "0", "]"}], "\[Equal]", "0"}]}], "}"}], 
    ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], ",", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"T", ",", "0", ",", "10"}], "}"}], ",", 
    RowBox[{"{", "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{
          SubscriptBox["Y", "2"], "[", "T", "]"}], "2"]}], ")"}]}], "}"}]}], 
   "}"}], "]"}]], "Output",
 ImageSize->{521, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1695476445]
}, Open  ]],

Cell["\<\
Split the Hamiltonian vector field into independent components governing \
momentum and position. This is done by setting the relevant right-hand sides \
of the equations to zero.\
\>", "Text",
 CellID->916026693],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<System\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Y1", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y1", ",", " ", "1", ",", "2"}], "]"}], " ", "=", " ", "0"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"Y2", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y2", ",", " ", "2", ",", "2"}], "]"}], " ", "=", " ", "0"}], 
  ";"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->1994657428],

Cell["\<\
This composition of weighted (first-order) Euler integration steps correspond \
to the symplectic (second-order) leapfrog method.\
\>", "Text",
 CellID->934458474],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"tfinal", " ", "=", " ", "1"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"time", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"T", ",", " ", "0", ",", " ", "tfinal"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"qvars", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Subscript", "[", 
      RowBox[{"Y", ",", "1"}], "]"}], "[", "T", "]"}], "}"}]}], ";"}], "\n", 
 RowBox[{"splittingsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", " ", "time", ",", " ", 
    RowBox[{"StartingStepSize", "\[Rule]", 
     RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<Splitting\>\"", ",", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", " ", 
       RowBox[{"\"\<Equations\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"Y1", ",", "Y2", ",", "Y1"}], "}"}]}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"\"\<Method\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
         "\"\<ExplicitEuler\>\"", ",", " ", "\"\<ExplicitEuler\>\"", ",", 
          " ", "\"\<ExplicitEuler\>\""}], "}"}]}]}], "}"}]}]}], 
   "]"}]}]}], "Input",
 CellLabel->"In[11]:=",
 CellID->77016691],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{356, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->2114824137]
}, Open  ]],

Cell[TextData[{
 "The method ",
 StyleBox["ExplicitEuler", "MR"],
 " could only have been specified once, since the second and third instances \
would have been filled in cyclically."
}], "Text",
 CellID->644406667],

Cell["This is the result at the end of the integration step.", "Text",
 CellID->2035270281],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", 
  RowBox[{"splittingsol", " ", "/.", " ", 
   RowBox[{"T", "\[Rule]", "tfinal"}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->1589171709],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
 ImageSize->{609, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]//InputForm=",
 CellID->27267255]
}, Open  ]],

Cell["\<\
This invokes the built-in integration method corresponding to the symplectic \
leapfrog integrator.\
\>", "Text",
 CellID->485170932],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sprksol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", " ", "time", ",", " ", 
    RowBox[{"StartingStepSize", "\[Rule]", 
     RowBox[{"1", "/", "10"}]}], ",", " ", 
    RowBox[{"Method", "\[Rule]", "SymplecticLeapfrog"}]}], "]"}]}]], "Input",
 CellLabel->"In[16]:=",
 CellID->1536208058],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{356, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->579565883]
}, Open  ]],

Cell["\<\
The result at the end of the integration step is identical to the result of \
the splitting method.\
\>", "Text",
 CellID->1095064901],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", 
  RowBox[{"sprksol", " ", "/.", " ", 
   RowBox[{"T", "\[Rule]", "tfinal"}]}], "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->1070563481],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
 ImageSize->{609, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]//InputForm=",
 CellID->526978384]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Composition of symplectic leapfrog", "Subsection",
 CellID->767689960],

Cell[TextData[{
 "This takes the symplectic leapfrog scheme as the base integration method \
and constructs a fourth-order symplectic integrator using a symmetric \
composition of Yoshida [",
 ButtonBox["Y90",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21389"],
 "]."
}], "Text",
 CellID->1402414050],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"YoshidaCoefficients", " ", "=", " ", 
    RowBox[{"RootReduce", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1", "/", 
        RowBox[{"(", 
         RowBox[{"2", "-", 
          RowBox[{"2", "^", 
           RowBox[{"(", 
            RowBox[{"1", "/", "3"}], ")"}]}]}], ")"}]}], ",", 
       RowBox[{
        RowBox[{"-", 
         RowBox[{"2", "^", 
          RowBox[{"(", 
           RowBox[{"1", "/", "3"}], ")"}]}]}], "/", 
        RowBox[{"(", 
         RowBox[{"2", "-", 
          RowBox[{"2", "^", 
           RowBox[{"(", 
            RowBox[{"1", "/", "3"}], ")"}]}]}], ")"}]}], ",", 
       RowBox[{"1", "/", 
        RowBox[{"(", 
         RowBox[{"2", "-", 
          RowBox[{"2", "^", 
           RowBox[{"(", 
            RowBox[{"1", "/", "3"}], ")"}]}]}], ")"}]}]}], "}"}], "]"}]}], 
   ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"YoshidaCompositionCoefficients", "[", 
     RowBox[{"4", ",", " ", "p_"}], "]"}], ":=", " ", 
    RowBox[{"N", "[", 
     RowBox[{"YoshidaCoefficients", ",", " ", "p"}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"splittingsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", "time", ",", " ", 
    RowBox[{"StartingStepSize", "\[Rule]", 
     RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<Composition\>\"", ",", " ", 
       RowBox[{
       "\"\<Coefficients\>\"", "\[Rule]", "YoshidaCompositionCoefficients"}], 
       ",", "\[IndentingNewLine]", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "4"}], ",", " ", 
       RowBox[{"\"\<Method\>\"", "\[Rule]", 
        RowBox[{"{", "SymplecticLeapfrog", "}"}]}]}], "}"}]}]}], 
   "]"}]}]}], "Input",
 CellLabel->"In[18]:=",
 CellID->1325610273],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{356, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->1729215438]
}, Open  ]],

Cell["This is the result at the end of the integration step.", "Text",
 CellID->1104143613],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", 
  RowBox[{"splittingsol", " ", "/.", " ", 
   RowBox[{"T", "\[Rule]", "tfinal"}]}], "]"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->534233750],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5403078808898406, Subscript[Y, 2][1] -> \
-0.8414706295697821}}\
\>", "Output",
 ImageSize->{609, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]//InputForm=",
 CellID->1882198361]
}, Open  ]],

Cell["\<\
This invokes the built-in symplectic integration method using coefficients \
for the fourth-order methods of Ruth, Yoshida.\
\>", "Text",
 CellID->854053497],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"SPRK4", "[", 
     RowBox[{"4", ",", "prec_"}], "]"}], ":=", 
    RowBox[{"N", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"-", "1"}], "+", 
              RowBox[{"12", "*", "#1"}], "-", 
              RowBox[{"48", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"48", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}], ",", 
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"1", "-", 
              RowBox[{"24", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"48", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}], ",", 
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"1", "-", 
              RowBox[{"24", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"48", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}], ",", 
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"-", "1"}], "+", 
              RowBox[{"12", "*", "#1"}], "-", 
              RowBox[{"48", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"48", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"-", "1"}], "+", 
              RowBox[{"6", "*", "#1"}], "-", 
              RowBox[{"12", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"6", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}], ",", 
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"1", "-", 
              RowBox[{"3", "*", "#1"}], "+", 
              RowBox[{"3", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"3", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}], ",", 
          RowBox[{"Root", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"-", "1"}], "+", 
              RowBox[{"6", "*", "#1"}], "-", 
              RowBox[{"12", "*", 
               RowBox[{"#1", "^", "2"}]}], "+", 
              RowBox[{"6", "*", 
               RowBox[{"#1", "^", "3"}]}]}], "&"}], ",", "1", ",", "0"}], 
           "]"}], ",", "0"}], "}"}]}], "}"}], ",", "prec"}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{" ", 
  RowBox[{"sprksol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", "time", ",", " ", 
     RowBox[{"StartingStepSize", "\[Rule]", 
      RowBox[{"1", "/", "10"}]}], ",", " ", "\[IndentingNewLine]", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<SymplecticPartitionedRungeKutta\>\"", ",", " ", 
        RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "SPRK4"}], ",", 
        RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "4"}], ",", 
        RowBox[{"\"\<PositionVariables\>\"", "\[Rule]", "qvars"}]}], 
       "}"}]}]}], "]"}]}]}]}], "Input",
 CellLabel->"In[22]:=",
 CellID->495407112],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{356, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->941950873]
}, Open  ]],

Cell["\<\
The result at the end of the integration step is identical to the result of \
the composition method.\
\>", "Text",
 CellID->714130875],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", 
  RowBox[{"sprksol", " ", "/.", " ", 
   RowBox[{"T", "\[Rule]", "tfinal"}]}], "]"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->2022026788],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5403078808898406, Subscript[Y, 2][1] -> \
-0.8414706295697821}}\
\>", "Output",
 ImageSize->{609, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]//InputForm=",
 CellID->240885230]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Hybrid methods", "Subsection",
 CellID->495793177],

Cell["\<\
While a closed-form solution often does not exist for the entire vector \
field, in some cases it is possible to analytically solve a system of \
differential equations for part of the vector field.\
\>", "Text",
 CellID->825503099],

Cell[TextData[{
 "When a solution can be found by ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ", direct numerical evaluation can be used to locally advance the solution."
}], "Text",
 CellID->1241338884],

Cell[TextData[{
 "This idea is implemented in the method ",
 StyleBox[ButtonBox["LocallyExact",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveLocallyExact"], "MR"],
 "."
}], "Text",
 CellID->1411819871]
}, Open  ]],

Cell[CellGroupData[{

Cell["Harmonic oscillator test example", "Subsection",
 CellID->672676880],

Cell["\<\
This example checks that the solution for the exact flows of split components \
of the harmonic oscillator equations is the same as applying Euler's method \
to each of the split components.\
\>", "Text",
 CellID->28313711],

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<HarmonicOscillator\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<System\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Y1", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y1", ",", " ", "1", ",", "2"}], "]"}], " ", "=", " ", "0"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"Y2", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y2", ",", " ", "2", ",", "2"}], "]"}], " ", "=", " ", "0"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"tfinal", " ", "=", " ", "1"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"time", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"T", ",", " ", "0", ",", " ", "tfinal"}], "}"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[25]:=",
 CellID->1488836562],

Cell[BoxData[
 RowBox[{
  RowBox[{"solexact", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", " ", "time", ",", " ", 
     RowBox[{"StartingStepSize", "\[Rule]", 
      RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"NDSolve`Splitting", ",", " ", 
        RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", " ", 
        RowBox[{"\"\<Equations\>\"", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"Y1", ",", "Y2", ",", "Y1"}], "}"}]}], ",", " ", 
        RowBox[{"\"\<Method\>\"", "\[Rule]", 
         RowBox[{"{", "\"\<LocallyExact\>\"", "}"}]}]}], "}"}]}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->941403014],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", 
  RowBox[{"solexact", " ", "/.", " ", 
   RowBox[{"T", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[34]:=",
 CellID->672720378],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
 ImageSize->{609, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]//InputForm=",
 CellID->1722591709]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"soleuler", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", "time", ",", " ", 
      RowBox[{"StartingStepSize", "\[Rule]", 
       RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"NDSolve`Splitting", ",", " ", 
         RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", " ", 
         RowBox[{"\"\<Equations\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"Y1", ",", "Y2", ",", "Y1"}], "}"}]}], ",", " ", 
         RowBox[{"\"\<Method\>\"", "\[Rule]", 
          RowBox[{"{", "\"\<ExplicitEuler\>\"", "}"}]}]}], "}"}]}]}], "]"}]}],
    ";"}], "\n"}], "\n", 
 RowBox[{"InputForm", "[", 
  RowBox[{"soleuler", " ", "/.", " ", 
   RowBox[{"T", "\[Rule]", "tfinal"}]}], "]"}]}], "Input",
 CellLabel->"In[37]:=",
 CellID->383091075],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
 ImageSize->{609, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]//InputForm=",
 CellID->368457738]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Hybrid numeric-symbolic splitting methods (ABC flow)", "Subsection",
 CellID->1861794127],

Cell["\<\
Consider the Arnold, Beltrami, and Childress flow, a widely studied model for \
volume-preserving three-dimensional flows.\
\>", "Text",
 CellID->1919299133],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"system", " ", "=", " ", 
  RowBox[{
  "GetNDSolveProblem", "[", "\"\<ArnoldBeltramiChildress\>\"", 
   "]"}]}]], "Input",
 CellLabel->"In[39]:=",
 CellID->1742456377],

Cell[BoxData[
 RowBox[{"NDSolveProblem", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "1"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{
         FractionBox["3", "4"], " ", 
         RowBox[{"Cos", "[", 
          RowBox[{
           SubscriptBox["Y", "2"], "[", "T", "]"}], "]"}]}], "+", 
        RowBox[{"Sin", "[", 
         RowBox[{
          SubscriptBox["Y", "3"], "[", "T", "]"}], "]"}]}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "2"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"Cos", "[", 
         RowBox[{
          SubscriptBox["Y", "3"], "[", "T", "]"}], "]"}], "+", 
        RowBox[{"Sin", "[", 
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}], "]"}]}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "3"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"Cos", "[", 
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}], "]"}], "+", 
        RowBox[{
         FractionBox["3", "4"], " ", 
         RowBox[{"Sin", "[", 
          RowBox[{
           SubscriptBox["Y", "2"], "[", "T", "]"}], "]"}]}]}]}]}], "}"}], ",",
     
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "1"], "[", "0", "]"}], "\[Equal]", 
       FractionBox["1", "4"]}], ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "2"], "[", "0", "]"}], "\[Equal]", 
       FractionBox["1", "3"]}], ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "3"], "[", "0", "]"}], "\[Equal]", 
       FractionBox["1", "2"]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], ",", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}], ",", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"T", ",", "0", ",", "100"}], "}"}], ",", 
    RowBox[{"{", "}"}], ",", 
    RowBox[{"{", "}"}]}], "}"}], "]"}]], "Output",
 ImageSize->{535, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->791842832]
}, Open  ]],

Cell["\<\
When applied directly, a volume preserving integrator would not in general \
preserve symmetries. A symmetry-preserving integrator, such as the implicit \
midpoint rule, would not preserve volume.\
\>", "Text",
 CellID->842378966],

Cell["\<\
This defines a splitting of the system by setting some of the right hand side \
components to zero.\
\>", "Text",
 CellID->1992436152],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<System\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Y1", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y1", ",", " ", "2", ",", "2"}], "]"}], " ", "=", " ", "0"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"Y2", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y2", ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", "3"}], "}"}], ",", "2"}], "]"}], " ", "=", " ", "0"}],
   ";"}]}], "Input",
 CellLabel->"In[40]:=",
 CellID->1577065340],

Cell[CellGroupData[{

Cell[BoxData["Y1"], "Input",
 CellLabel->"In[45]:=",
 CellID->97309368],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "1"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{
      FractionBox["3", "4"], " ", 
      RowBox[{"Cos", "[", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}], "]"}]}], "+", 
     RowBox[{"Sin", "[", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}], "]"}]}]}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "2"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "3"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"Cos", "[", 
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], "]"}], "+", 
     RowBox[{
      FractionBox["3", "4"], " ", 
      RowBox[{"Sin", "[", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}], "]"}]}]}]}]}], 
  "}"}]], "Output",
 ImageSize->{537, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[45]=",
 CellID->1320340019]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData["Y2"], "Input",
 CellLabel->"In[46]:=",
 CellID->1985213372],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "1"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "2"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
    RowBox[{
     RowBox[{"Cos", "[", 
      RowBox[{
       SubscriptBox["Y", "3"], "[", "T", "]"}], "]"}], "+", 
     RowBox[{"Sin", "[", 
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], "]"}]}]}], ",", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["Y", "3"], "\[Prime]",
      MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}]}], 
  "}"}]], "Output",
 ImageSize->{366, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]=",
 CellID->1685009027]
}, Open  ]],

Cell[TextData[{
 "The system for ",
 StyleBox["Y1", "MR"],
 " is solvable exactly by ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " so that we can use the ",
 StyleBox["LocallyExact", "MR"],
 " method."
}], "Text",
 CellID->1704796528],

Cell[TextData[{
 StyleBox["Y2", "MR"],
 " is not solvable, however, so we need to use a suitable numerical \
integrator in order to obtain the desired properties in the splitting \
method."
}], "Text",
 CellID->1243158945],

Cell[TextData[{
 "This defines a method for computing the implicit midpoint rule in terms of \
the built-in ",
 StyleBox["ImplicitRungeKutta", "MR"],
 " method."
}], "Text",
 CellID->2055506479],

Cell[BoxData[
 RowBox[{
  RowBox[{"ImplicitMidpoint", " ", "=", 
   RowBox[{"{", 
    RowBox[{"\"\<FixedStep\>\"", ",", 
     RowBox[{"Method", "\[Rule]", " ", 
      RowBox[{"{", 
       RowBox[{"\"\<ImplicitRungeKutta\>\"", ",", " ", 
        RowBox[{
        "\"\<Coefficients\>\"", "\[Rule]", 
         "\"\<ImplicitRungeKuttaGaussCoefficients\>\""}], ",", 
        RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", " ", 
        RowBox[{"ImplicitSolver", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"FixedPoint", ",", 
           RowBox[{"AccuracyGoal", "\[Rule]", "MachinePrecision"}], ",", " ", 
           
           RowBox[{"PrecisionGoal", "\[Rule]", "MachinePrecision"}], ",", 
           RowBox[{"\"\<IterationSafetyFactor\>\"", "\[Rule]", "1"}]}], 
          "}"}]}]}], "}"}]}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[47]:=",
 CellID->52719993],

Cell[TextData[{
 "This defines a second-order, volume-preserving, reversing symmetry-group \
integrator [",
 ButtonBox["MQ02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#31673"],
 "]."
}], "Text",
 CellID->1038666118],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"splittingsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", "\[IndentingNewLine]", 
    RowBox[{"StartingStepSize", "\[Rule]", 
     RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<Splitting\>\"", ",", " ", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"\"\<Equations\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"Y2", ",", "Y1", ",", "Y2"}], "}"}]}], ",", " ", 
       RowBox[{"\"\<Method\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
         "\"\<LocallyExact\>\"", ",", "ImplicitMidpoint", ",", 
          "\"\<LocallyExact\>\""}], "}"}]}]}], "}"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[48]:=",
 CellID->622153706],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "100.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "100.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "100.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{370, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[48]=",
 CellID->1620970739]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Lotka-Volterra equations", "Subsection",
 CellID->1572483518],

Cell[TextData[{
 "Various numerical integrators for this system are compared within \"",
 ButtonBox["Numerical methods for solving the Lotka-Volterra equations",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveLotkaVolterra"],
 "\"."
}], "Text",
 CellID->1117946883]
}, Open  ]],

Cell[CellGroupData[{

Cell["Euler's equations", "Subsection",
 CellID->752766232],

Cell[TextData[{
 "Various numerical integrators for Euler's equations are compared within \
\"",
 ButtonBox["Rigid body solvers",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveRigidBody"],
 "\"."
}], "Text",
 CellID->666338754]
}, Open  ]],

Cell[CellGroupData[{

Cell["Non-autonomous vector fields", "Subsection",
 CellID->382283106],

Cell["\<\
Consider the Duffing oscillator, a forced planar non-autonomous differential \
system.\
\>", "Text",
 CellID->1425443112],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"system", " ", "=", " ", 
  RowBox[{
  "GetNDSolveProblem", "[", "\"\<DuffingOscillator\>\"", "]"}]}]], "Input",
 CellLabel->"In[49]:=",
 CellID->694652465],

Cell[BoxData[
 RowBox[{"NDSolveProblem", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "1"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "2"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        FractionBox[
         RowBox[{"3", " ", 
          RowBox[{"Cos", "[", "T", "]"}]}], "10"], "+", 
        RowBox[{
         SubscriptBox["Y", "1"], "[", "T", "]"}], "-", 
        SuperscriptBox[
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}], "3"], "+", 
        FractionBox[
         RowBox[{
          SubscriptBox["Y", "2"], "[", "T", "]"}], "4"]}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "1"], "[", "0", "]"}], "\[Equal]", "0"}], ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "2"], "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], 
    ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], ",", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"T", ",", "0", ",", "10"}], "}"}], ",", 
    RowBox[{"{", "}"}], ",", 
    RowBox[{"{", "}"}]}], "}"}], "]"}]], "Output",
 ImageSize->{494, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[49]=",
 CellID->1390290835]
}, Open  ]],

Cell["This defines a splitting of the system.", "Text",
 CellID->219362478],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Y1", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "1"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "2"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
      FractionBox[
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}], "4"]}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"Y2", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "1"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "2"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
      RowBox[{
       FractionBox[
        RowBox[{"3", " ", 
         RowBox[{"Cos", "[", "T", "]"}]}], "10"], "+", 
       RowBox[{
        SubscriptBox["Y", "1"], "[", "T", "]"}], "-", 
       SuperscriptBox[
        RowBox[{
         SubscriptBox["Y", "1"], "[", "T", "]"}], "3"]}]}]}], "}"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[50]:=",
 CellID->1367372844],

Cell["\<\
The splitting of the time component among the vector fields is ambiguous, so \
the method issues an error message.\
\>", "Text",
 CellID->1100259215],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"splittingsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", " ", 
    RowBox[{"StartingStepSize", "\[Rule]", 
     RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<Splitting\>\"", ",", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"\"\<Equations\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"Y2", ",", "Y1", ",", "Y1"}], "}"}]}], ",", 
       RowBox[{"\"\<Method\>\"", "\[Rule]", 
        RowBox[{"{", "\"\<LocallyExact\>\"", "}"}]}]}], "}"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[52]:=",
 CellID->602453553],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"spltdep\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"The differential system \
\\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\({0, \\(\\(\\(3\\\\ \
\\(\\(Cos[T]\\)\\)\\)\\/10 + \\(\\(Y\\_1[T]\\)\\) - \\(Y\\_1[T]\\)\\^3\\)\\)}\
\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" in the method \\\\\\\"\\\", \\\"MSG\\\"]\\)\
\\!\\(\\\"Splitting\\\"\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" depends on \
\\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(T\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" which \
is ambiguous. The differential system should be in autonomous form.\\\\\\\"\\\
\", \\\"MSG\\\"]\\) \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet://Messages/NDSolve/spltdep\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellLabel->"During evaluation of In[52]:=",
 CellID->761636065],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"initf\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"The initialization of the method \\\
\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(NDSolve`Splitting\\)\\!\\(\\*StyleBox[\\\"\\\
\\\\\" failed.\\\\\\\"\\\", \\\"MSG\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[52]:=",
 CellID->9944190],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{356, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[52]=",
 CellID->738573746]
}, Open  ]],

Cell[TextData[{
 "The equations can be extended by introducing a new \
\[OpenCurlyDoubleQuote]dummy\[CloseCurlyDoubleQuote] variable ",
 StyleBox["Z[T]", "Input"],
 " such that ",
 StyleBox["Z[T]==T", "Input"],
 " and specifying how it should be distributed in the split differential \
systems."
}], "Text",
 CellID->248896027],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Y1", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "1"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}]}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "2"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
      FractionBox[
       RowBox[{
        SubscriptBox["Y", "2"], "[", "T", "]"}], "4"]}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"Z", "'"}], "[", "T", "]"}], "\[Equal]", "1"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"Y2", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "1"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{
       SuperscriptBox[
        SubscriptBox["Y", "2"], "\[Prime]",
        MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
      RowBox[{
       FractionBox[
        RowBox[{"3", " ", 
         RowBox[{"Cos", "[", 
          RowBox[{"Z", "[", "T", "]"}], "]"}]}], "10"], "+", 
       RowBox[{
        SubscriptBox["Y", "1"], "[", "T", "]"}], "-", 
       SuperscriptBox[
        RowBox[{
         SubscriptBox["Y", "1"], "[", "T", "]"}], "3"]}]}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"Z", "'"}], "[", "T", "]"}], "\[Equal]", "0"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"system", "[", "\"\<System\>\"", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"Z", "'"}], "[", "T", "]"}], "\[Equal]", "1"}], "}"}]}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ics", " ", "=", " ", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"system", "[", "\"\<InitialConditions\>\"", "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Z", "[", "0", "]"}], "\[Equal]", "1"}], "}"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"Z", "[", "T", "]"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"time", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<TimeData\>\"", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[53]:=",
 CellID->1623945074],

Cell[TextData[{
 "This defines a geometric splitting method that satisfies ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["\[Lambda]", "1"], " ", "+", " ", 
     SubscriptBox["\[Lambda]", "2"]}], " ", "=", 
    RowBox[{"-", "\[Delta]"}]}], TraditionalForm]]],
 " for any finite time interval, where ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Lambda]", "1"], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[Lambda]", "2"], TraditionalForm]]],
 " are the Lyapunov exponents [",
 ButtonBox["MQ02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#31673"],
 "]."
}], "Text",
 CellID->1216648005],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"splittingsol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"eqs", ",", " ", "ics"}], "}"}], ",", " ", "vars", ",", "time", 
    ",", " ", 
    RowBox[{"StartingStepSize", "\[Rule]", 
     RowBox[{"1", "/", "10"}]}], ",", "\[IndentingNewLine]", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"NDSolve`Splitting", ",", " ", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "2"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{"\"\<Equations\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"Y2", ",", "Y1", ",", "Y2"}], "}"}]}], ",", " ", 
       RowBox[{"\"\<Method\>\"", "\[Rule]", 
        RowBox[{"{", "\"\<LocallyExact\>\"", "}"}]}]}], "}"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[59]:=",
 CellID->93848532],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{"Z", "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{358, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[59]=",
 CellID->1922726336]
}, Open  ]],

Cell["Here is a plot of the solution.", "Text",
 CellID->1313957374],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"system", "[", 
      RowBox[{"\"\<DependentVariables\>\"", "[", "]"}], "]"}], " ", "/.", " ",
      
     RowBox[{"First", "[", "splittingsol", "]"}]}], "]"}], ",", 
   RowBox[{"Evaluate", "[", "time", "]"}], ",", 
   RowBox[{"AspectRatio", "->", "1"}]}], "]"}]], "Input",
 CellLabel->"In[60]:=",
 CellID->1264411310],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 360},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[60]=",
 CellID->280095711]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:3",
 CellID->367926707],

Cell[TextData[{
 "The default coefficient choice in ",
 StyleBox["Composition", "MR"],
 " tries to automatically select between ",
 StyleBox["SymmetricCompositionCoefficients", "MR"],
 " and ",
 StyleBox["SymmetricCompositionSymmetricMethodCoefficients", "MR"],
 " depending on the properties of the methods specified using the ",
 StyleBox["Method", "MR"],
 " option."
}], "Text",
 CellID->821724190],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {"Coefficients", "Automatic", Cell[
    "specifies the coefficients to use in the composition method", 
     "TableText"]},
   {"DifferenceOrder", "Automatic", Cell[
    "specifies the order of local accuracy of the method", "TableText"]},
   {"Method", "None", Cell["\<\
specifies the base methods to use in the numerical integration\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->2134415436],

Cell[TextData[{
 "Options of the method ",
 Cell[BoxData[
  ButtonBox["Composition",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Composition"]], "InlineFormula"],
 ". "
}], "Caption",
 CellID->275949538],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {"Coefficients", 
    RowBox[{"{", "}"}], Cell[
    "specifies the coefficients to use in the splitting method", 
     "TableText"]},
   {"DifferenceOrder", "Automatic", Cell[
    "specifies the order of local accuracy of the method", "TableText"]},
   {"Equations", 
    RowBox[{"{", "}"}], Cell[
    "specifies the way in which the equations should be split", 
     "TableText"]},
   {"Method", "None", Cell["\<\
specifies the base methods to use in the numerical integration\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->874444183],

Cell[TextData[{
 "Options of the method ",
 Cell[BoxData["Splitting"], "InlineFormula"],
 "."
}], "Caption",
 CellID->1025597906]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Composition and Splitting Methods for NDSolve - Wolfram \
Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveSPRK"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveDoubleStep"]}], "Text",
       FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 40, 19.9627540}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "In some cases it is useful to split the differential system into \
subsystems and solve each subsystem using appropriate integration methods. \
Recombining the individual solutions often allows certain dynamical \
properties, such as volume, to be conserved. More information on splitting \
and composition can be found in [MQ02, HLW02], and specific aspects related \
to NDSolve are discussed in [SS05, SS06]. We are concerned with initial value \
problems y'(t)= f(y(t)), y(0)=y_0\\[Element]\\[DoubleStruckCapitalR]^ngiven. \
Composition is a useful device for raising the order of a numerical \
integration scheme.", "synonyms" -> {}, "title" -> 
    "Composition and Splitting Methods for NDSolve", "type" -> "Tutorial", 
    "uri" -> "tutorial/NDSolveSplitting"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[728, 29, 100, 2, 70, "Title",
   CellTags->"c:1",
   CellID->174109254]},
 "s:1"->{
  Cell[853, 35, 70, 2, 70, "Section",
   CellTags->"s:1",
   CellID->1525338532]},
 "s:2"->{
  Cell[22142, 797, 66, 2, 70, "Section",
   CellTags->"s:2",
   CellID->1075630498]},
 "s:3"->{
  Cell[82961, 2692, 71, 2, 70, "Section",
   CellTags->"s:3",
   CellID->367926707]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 88030, 2829},
 {"s:1", 88121, 2833},
 {"s:2", 88214, 2837},
 {"s:3", 88310, 2841}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 100, 2, 70, "Title",
 CellTags->"c:1",
 CellID->174109254],
Cell[CellGroupData[{
Cell[853, 35, 70, 2, 70, "Section",
 CellTags->"s:1",
 CellID->1525338532],
Cell[926, 39, 888, 25, 70, "Text",
 CellID->1630494851],
Cell[CellGroupData[{
Cell[1839, 68, 53, 1, 70, "Subsection",
 CellID->649779944],
Cell[1895, 71, 558, 21, 70, "Text",
 CellID->898893221]
}, Open  ]],
Cell[CellGroupData[{
Cell[2490, 97, 53, 1, 70, "Subsection",
 CellID->406958863],
Cell[2546, 100, 133, 4, 70, "Text",
 CellID->786614357],
Cell[2682, 106, 213, 5, 70, "Text",
 CellID->774074422],
Cell[2898, 113, 612, 22, 70, "Text",
 CellID->1577685144],
Cell[3513, 137, 273, 11, 70, "Text",
 CellID->12509670],
Cell[3789, 150, 608, 17, 70, "Text",
 CellID->1734802721],
Cell[4400, 169, 525, 16, 70, "Text",
 CellID->394448947],
Cell[4928, 187, 494, 17, 70, "Text",
 CellID->1210214738],
Cell[5425, 206, 77, 1, 70, "Text",
 CellID->797495146],
Cell[5505, 209, 103, 2, 70, "BulletedText",
 CellID->1134072999],
Cell[5611, 213, 299, 10, 70, "BulletedText",
 CellID->2115316660],
Cell[5913, 225, 80, 1, 70, "BulletedText",
 CellID->1517362642]
}, Open  ]],
Cell[CellGroupData[{
Cell[6030, 231, 51, 1, 70, "Subsection",
 CellID->280476626],
Cell[6084, 234, 281, 10, 70, "Text",
 CellID->1356823117],
Cell[6368, 246, 364, 14, 70, "Text",
 CellID->48180321],
Cell[6735, 262, 301, 11, 70, "Text",
 CellID->752549885],
Cell[7039, 275, 163, 6, 70, "Text",
 CellID->1687764060],
Cell[7205, 283, 654, 19, 70, "Text",
 CellID->110167493],
Cell[7862, 304, 230, 9, 70, "Text",
 CellID->1565446313],
Cell[8095, 315, 212, 5, 70, "Text",
 CellID->200748626],
Cell[8310, 322, 397, 13, 70, "Text",
 CellID->2069068573],
Cell[8710, 337, 118, 3, 70, "Text",
 CellID->1571823932],
Cell[8831, 342, 1060, 37, 70, "Text",
 CellID->2116694905],
Cell[9894, 381, 128, 4, 70, "Text",
 CellID->1134364391],
Cell[10025, 387, 599, 21, 70, "Text",
 CellID->1043697469]
}, Open  ]],
Cell[CellGroupData[{
Cell[10661, 413, 56, 1, 70, "Subsection",
 CellID->226936990],
Cell[10720, 416, 282, 9, 70, "Text",
 CellID->1308473645],
Cell[11005, 427, 103, 2, 70, "BulletedText",
 CellID->421552353],
Cell[11111, 431, 167, 4, 70, "BulletedText",
 CellID->1857431841],
Cell[11281, 437, 207, 6, 70, "BulletedText",
 CellID->1958253590],
Cell[CellGroupData[{
Cell[11513, 447, 189, 4, 70, "BulletedText",
 CellID->1320445575],
Cell[11705, 453, 177, 4, 70, "Text",
 CellID->116907056],
Cell[11885, 459, 616, 27, 70, "Text",
 CellID->597384299],
Cell[12504, 488, 1095, 45, 70, "Text",
 CellID->2094519997]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13648, 539, 57, 1, 70, "Subsection",
 CellID->1694592200],
Cell[13708, 542, 154, 4, 70, "Text",
 CellID->609893969],
Cell[13865, 548, 4977, 138, 70, "Text",
 CellID->1681839070]
}, Open  ]],
Cell[CellGroupData[{
Cell[18879, 691, 57, 1, 70, "Subsection",
 CellID->2089041147],
Cell[18939, 694, 224, 6, 70, "Text",
 CellID->1820108707],
Cell[19166, 702, 2927, 89, 70, "Text",
 CellID->331850569]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[22142, 797, 66, 2, 70, "Section",
 CellTags->"s:2",
 CellID->1075630498],
Cell[22211, 801, 459, 14, 70, "Text",
 CellID->1787941719],
Cell[22673, 817, 193, 6, 70, "MathCaption",
 CellID->1356107125],
Cell[22869, 825, 383, 10, 70, "Input",
 InitializationCell->True,
 CellID->997281188],
Cell[23255, 837, 93, 1, 70, "MathCaption",
 CellID->1897393532],
Cell[23351, 840, 198, 7, 70, "Input",
 InitializationCell->True,
 CellID->2108657010]
}, Open  ]],
Cell[CellGroupData[{
Cell[23586, 852, 59, 1, 70, "Section",
 CellID->537561600],
Cell[CellGroupData[{
Cell[23670, 857, 62, 1, 70, "Subsection",
 CellID->2007561025],
Cell[23735, 860, 452, 13, 70, "Text",
 CellID->1526619675],
Cell[24190, 875, 283, 7, 70, "Text",
 CellID->738310227],
Cell[24476, 884, 524, 22, 70, "Text",
 CellID->1929145950],
Cell[CellGroupData[{
Cell[25025, 910, 179, 5, 70, "Input",
 CellID->950959959],
Cell[25207, 917, 1641, 55, 73, "Output",
 CellID->1695476445]
}, Open  ]],
Cell[26863, 975, 223, 5, 70, "Text",
 CellID->916026693],
Cell[27089, 982, 574, 19, 70, "Input",
 CellID->1994657428],
Cell[27666, 1003, 173, 4, 70, "Text",
 CellID->934458474],
Cell[CellGroupData[{
Cell[27864, 1011, 1251, 33, 70, "Input",
 CellID->77016691],
Cell[29118, 1046, 968, 34, 54, "Output",
 CellID->2114824137]
}, Open  ]],
Cell[30101, 1083, 215, 6, 70, "Text",
 CellID->644406667],
Cell[30319, 1091, 91, 1, 70, "Text",
 CellID->2035270281],
Cell[CellGroupData[{
Cell[30435, 1096, 187, 5, 70, "Input",
 CellID->1589171709],
Cell[30625, 1103, 257, 8, 50, "Output",
 CellID->27267255]
}, Open  ]],
Cell[30897, 1114, 143, 4, 70, "Text",
 CellID->485170932],
Cell[CellGroupData[{
Cell[31065, 1122, 337, 8, 70, "Input",
 CellID->1536208058],
Cell[31405, 1132, 967, 34, 54, "Output",
 CellID->579565883]
}, Open  ]],
Cell[32387, 1169, 144, 4, 70, "Text",
 CellID->1095064901],
Cell[CellGroupData[{
Cell[32556, 1177, 182, 5, 70, "Input",
 CellID->1070563481],
Cell[32741, 1184, 258, 8, 50, "Output",
 CellID->526978384]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[33048, 1198, 76, 1, 70, "Subsection",
 CellID->767689960],
Cell[33127, 1201, 327, 9, 70, "Text",
 CellID->1402414050],
Cell[CellGroupData[{
Cell[33479, 1214, 1875, 54, 70, "Input",
 CellID->1325610273],
Cell[35357, 1270, 968, 34, 54, "Output",
 CellID->1729215438]
}, Open  ]],
Cell[36340, 1307, 91, 1, 70, "Text",
 CellID->1104143613],
Cell[CellGroupData[{
Cell[36456, 1312, 186, 5, 70, "Input",
 CellID->534233750],
Cell[36645, 1319, 259, 8, 50, "Output",
 CellID->1882198361]
}, Open  ]],
Cell[36919, 1330, 167, 4, 70, "Text",
 CellID->854053497],
Cell[CellGroupData[{
Cell[37111, 1338, 3619, 101, 70, "Input",
 CellID->495407112],
Cell[40733, 1441, 967, 34, 54, "Output",
 CellID->941950873]
}, Open  ]],
Cell[41715, 1478, 145, 4, 70, "Text",
 CellID->714130875],
Cell[CellGroupData[{
Cell[41885, 1486, 182, 5, 70, "Input",
 CellID->2022026788],
Cell[42070, 1493, 258, 8, 50, "Output",
 CellID->240885230]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[42377, 1507, 56, 1, 70, "Subsection",
 CellID->495793177],
Cell[42436, 1510, 242, 5, 70, "Text",
 CellID->825503099],
Cell[42681, 1517, 278, 8, 70, "Text",
 CellID->1241338884],
Cell[42962, 1527, 215, 7, 70, "Text",
 CellID->1411819871]
}, Open  ]],
Cell[CellGroupData[{
Cell[43214, 1539, 74, 1, 70, "Subsection",
 CellID->672676880],
Cell[43291, 1542, 233, 5, 70, "Text",
 CellID->28313711],
Cell[43527, 1549, 921, 30, 70, "Input",
 CellID->1488836562],
Cell[44451, 1581, 754, 18, 70, "Input",
 CellID->941403014],
Cell[CellGroupData[{
Cell[45230, 1603, 177, 5, 70, "Input",
 CellID->672720378],
Cell[45410, 1610, 259, 8, 50, "Output",
 CellID->1722591709]
}, Open  ]],
Cell[CellGroupData[{
Cell[45706, 1623, 913, 22, 70, "Input",
 CellID->383091075],
Cell[46622, 1647, 258, 8, 50, "Output",
 CellID->368457738]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[46929, 1661, 95, 1, 70, "Subsection",
 CellID->1861794127],
Cell[47027, 1664, 167, 4, 70, "Text",
 CellID->1919299133],
Cell[CellGroupData[{
Cell[47219, 1672, 190, 6, 70, "Input",
 CellID->1742456377],
Cell[47412, 1680, 2429, 77, 111, "Output",
 CellID->791842832]
}, Open  ]],
Cell[49856, 1760, 240, 5, 70, "Text",
 CellID->842378966],
Cell[50099, 1767, 144, 4, 70, "Text",
 CellID->1992436152],
Cell[50246, 1773, 628, 21, 70, "Input",
 CellID->1577065340],
Cell[CellGroupData[{
Cell[50899, 1798, 71, 2, 70, "Input",
 CellID->97309368],
Cell[50973, 1802, 1201, 41, 51, "Output",
 CellID->1320340019]
}, Open  ]],
Cell[CellGroupData[{
Cell[52211, 1848, 73, 2, 70, "Input",
 CellID->1985213372],
Cell[52287, 1852, 884, 30, 36, "Output",
 CellID->1685009027]
}, Open  ]],
Cell[53186, 1885, 309, 12, 70, "Text",
 CellID->1704796528],
Cell[53498, 1899, 222, 6, 70, "Text",
 CellID->1243158945],
Cell[53723, 1907, 194, 6, 70, "Text",
 CellID->2055506479],
Cell[53920, 1915, 880, 21, 70, "Input",
 CellID->52719993],
Cell[54803, 1938, 244, 8, 70, "Text",
 CellID->1038666118],
Cell[CellGroupData[{
Cell[55072, 1950, 845, 20, 70, "Input",
 CellID->622153706],
Cell[55920, 1972, 1353, 47, 71, "Output",
 CellID->1620970739]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[57322, 2025, 67, 1, 70, "Subsection",
 CellID->1572483518],
Cell[57392, 2028, 277, 7, 70, "Text",
 CellID->1117946883]
}, Open  ]],
Cell[CellGroupData[{
Cell[57706, 2040, 59, 1, 70, "Subsection",
 CellID->752766232],
Cell[57768, 2043, 240, 8, 70, "Text",
 CellID->666338754]
}, Open  ]],
Cell[CellGroupData[{
Cell[58045, 2056, 70, 1, 70, "Subsection",
 CellID->382283106],
Cell[58118, 2059, 131, 4, 70, "Text",
 CellID->1425443112],
Cell[CellGroupData[{
Cell[58274, 2067, 179, 5, 70, "Input",
 CellID->694652465],
Cell[58456, 2074, 1645, 53, 73, "Output",
 CellID->1390290835]
}, Open  ]],
Cell[60116, 2130, 75, 1, 70, "Text",
 CellID->219362478],
Cell[60194, 2133, 1367, 46, 70, "Input",
 CellID->1367372844],
Cell[61564, 2181, 159, 4, 70, "Text",
 CellID->1100259215],
Cell[CellGroupData[{
Cell[61748, 2189, 726, 18, 70, "Input",
 CellID->602453553],
Cell[62477, 2209, 853, 15, 70, "Message",
 CellID->761636065],
Cell[63333, 2226, 367, 7, 70, "Message",
 CellID->9944190],
Cell[63703, 2235, 967, 34, 54, "Output",
 CellID->738573746]
}, Open  ]],
Cell[64685, 2272, 327, 9, 70, "Text",
 CellID->248896027],
Cell[65015, 2283, 2539, 82, 70, "Input",
 CellID->1623945074],
Cell[67557, 2367, 669, 23, 70, "Text",
 CellID->1216648005],
Cell[CellGroupData[{
Cell[68251, 2394, 826, 21, 70, "Input",
 CellID->93848532],
Cell[69080, 2417, 1324, 46, 71, "Output",
 CellID->1922726336]
}, Open  ]],
Cell[70419, 2466, 68, 1, 70, "Text",
 CellID->1313957374],
Cell[CellGroupData[{
Cell[70512, 2471, 431, 12, 47, "Input",
 CellID->1264411310],
Cell[70946, 2485, 11954, 200, 381, "Output",
 Evaluatable->False,
 CellID->280095711]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[82961, 2692, 71, 2, 70, "Section",
 CellTags->"s:3",
 CellID->367926707],
Cell[83035, 2696, 401, 11, 70, "Text",
 CellID->821724190],
Cell[83439, 2709, 694, 16, 70, "DefinitionBox3Col",
 CellID->2134415436],
Cell[84136, 2727, 208, 8, 70, "Caption",
 CellID->275949538],
Cell[84347, 2737, 836, 21, 70, "DefinitionBox3Col",
 CellID->874444183],
Cell[85186, 2760, 129, 5, 70, "Caption",
 CellID->1025597906]
}, Open  ]]
}, Open  ]],
Cell[85342, 2769, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

