(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    723516,      20518]
NotebookOptionsPosition[    670964,      18751]
NotebookOutlinePosition[    679218,      18982]
CellTagsIndexPosition[    677848,      18945]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Numerical Integration in Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/NIntegrateOverview"], 
          "NIntegrate Introduction" :> 
          Documentation`HelpLookup["paclet:tutorial/NIntegrateIntroduction"], 
          "NIntegrate Integration Strategies" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NIntegrateIntegrationStrategies"], 
          "NIntegrate References" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NIntegrateReferences"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Advanced Numerical Integration in Mathematica\"\>", 
       2->"\<\"NIntegrate Introduction\"\>", 
       3->"\<\"NIntegrate Integration Strategies\"\>", 
       4->"\<\"NIntegrate References\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"], "Calculus" :> 
          Documentation`HelpLookup["paclet:guide/Calculus"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Numerical Evaluation & Precision\"\>", 
       2->"\<\"Calculus\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Random" :> Documentation`HelpLookup["paclet:ref/Random"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Random\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["NIntegrate Integration Rules", "Title",
 CellID->358747687],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellID->235359397],

Cell[TextData[{
 "An integration rule computes an estimate of an integral over a region using \
a weighted sum. In the context of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " usage, an integration rule object provides both an integral estimate and \
an error estimate as a measure of the integral estimate's accuracy."
}], "Text",
 CellID->170702081],

Cell[TextData[{
 "An integration rule samples the integrand with a set of points. These \
points are called sampling points. In the literature these are usually called \
abscissas. Corresponding to each sampling point ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 " there is a weight number ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["w", "i"], TraditionalForm]], "InlineMath"],
 ". An integration rule estimates the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "b"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " ",
 "with the weighted sum ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Sum]", 
    RowBox[{
     SubscriptBox["w", "i"], 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ". An integration rule is a functional, that is, it maps functions over the \
interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"a", ",", "b"}], "]"}], TraditionalForm]], "InlineMath"],
 " (or a more general region) into real numbers. "
}], "Text",
 CellTags->"Def:SaplingPointsAndAbscissas",
 CellID->348387628],

Cell[TextData[{
 "If a rule is applied over the region ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 " this will be denoted as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["R", "V"], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " "}], TraditionalForm]], "InlineMath"],
 " is the integrand."
}], "Text",
 CellID->106200629],

Cell[TextData[{
 "The sampling points of the rules considered below are chosen to compute \
estimates for integrals either over the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 ", or the unit cube ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"[", 
     RowBox[{"0", ",", "1"}], "]"}], "d"], TraditionalForm]], "InlineMath"],
 ", or the \"centered\" unit cube ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"[", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "2"]}], ",", 
      FractionBox["1", "2"]}], "]"}], "d"], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is the dimension of the integral. So if ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 " is one of these regions, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " will be used instead of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["R", "V"], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 ". When these rules are applied to other regions, their abscissas and \
estimates need to be scaled accordingly."
}], "Text",
 CellID->106091508],

Cell[TextData[{
 "The integration rule ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]], "InlineMath"],
 " is said to be exact for the function ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["R", 
      RowBox[{"[", 
       RowBox[{"a", ",", "b"}], "]"}]], "(", "f", ")"}], "=", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "Text",
 CellTags->"Def:ExactRule",
 CellID->98815734],

Cell[TextData[{
 "The application of an integration rule ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]], "InlineMath"],
 " to a function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " "}], TraditionalForm]], "InlineMath"],
 " will be referred as an integration of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ", for example,",
 " \"when ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is integrated by ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]], "InlineMath"],
 ", we get ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 ".\""
}], "Text",
 CellID->382170004],

Cell[TextData[{
 "An one-dimensional integration rule is said to be of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " if it integrates exactly all polynomials of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " or less, and will fail to do so for at least one polynomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "+", "1"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellTags->"Def:OneDimRuleDegree",
 CellID->471065697],

Cell[TextData[{
 "A multi-dimensional integration rule is said to be of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " if it integrates exactly all monomials of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " or less, and will fail to do so for at least one monomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "+", "1"}], TraditionalForm]], "InlineMath"],
 ", that is, the rule is exact for all monomials of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Product]", 
     RowBox[{"i", "=", "1"}], "d"], 
    SubsuperscriptBox["x", "i", 
     SubscriptBox["\[Alpha]", "i"]]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is the dimension, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Alpha]", "i"], "\[GreaterEqual]", " ", "0"}], 
   TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"i", "=", "1"}], "d"], 
     SubscriptBox["\[Alpha]", "i"]}], "\[LessEqual]", "n"}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellTags->"Def:MultiDimRuleDegree",
 CellID->47792312],

Cell[TextData[{
 "A ",
 "null rule",
 " of degree ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " will integrate to zero all monomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", " ", "m"}], TraditionalForm]], "InlineMath"],
 " and will fail to do so for at least one monomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "+", "1"}], TraditionalForm]], "InlineMath"],
 ". Each null rule may be thought of as the difference between a basic \
integration rule and an appropriate integration rule of a lower degree."
}], "Text",
 CellTags->"Def:NullRule",
 CellID->84757638],

Cell[TextData[{
 "If the set of sampling points of a rule ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "1"], TraditionalForm]], "InlineMath"],
 " of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 ", contains the set of sampling points of a rule ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "2"], TraditionalForm]], "InlineMath"],
 " of a lower degree ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 ", that is, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", ">", "m"}], TraditionalForm]], "InlineMath"],
 ", then ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "2"], TraditionalForm]], "InlineMath"],
 " is said to be imbedded in ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "1"], TraditionalForm]], "InlineMath"],
 ". This will be denoted as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["R", "2"], "\[Subset]", 
    SubscriptBox["R", "1"]}], TraditionalForm]], "InlineMath"]
}], "Text",
 CellTags->"Def:ImbededRule",
 CellID->139010942],

Cell[TextData[{
 "An integration rule of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " that is a member of a family of rules with a common derivation and \
properties, but different degrees will be denoted as ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", " "}], TraditionalForm]], "InlineMath"],
 " might be chosen to identify the family. (For example, trapezoidal rule of \
degree 4 might be referred to as ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", "4"}], ")"}], TraditionalForm]], "InlineMath"],
 ".)"
}], "Text",
 CellID->655009440],

Cell[TextData[{
 "If each rule in a family is imbedded in another rule in the same family, \
then the rules of that family are called progressive. (For any given ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "\[Element]", "\[DoubleStruckCapitalN]"}], TraditionalForm]], 
  "InlineMath"],
 " there exists ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"n", "\[Element]", "\[DoubleStruckCapitalN]"}], ",", " ", 
    RowBox[{"n", ">", "m"}]}], TraditionalForm]], "InlineMath"],
 ", for which ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"R", "(", 
     RowBox[{"f", ",", "m"}], ")"}], "\[Subset]", 
    RowBox[{"R", "(", 
     RowBox[{"f", ",", "n"}], ")"}]}], TraditionalForm]], "InlineMath"],
 ")."
}], "Text",
 CellTags->"Def:ProgressiveRule",
 CellID->65611042],

Cell["\<\
An integration rule is of open type if the integrand is not evaluated at the \
end points of the interval. It is called to be of closed type if it uses \
integrand evaluations at the interval end points.\
\>", "Text",
 CellTags->"Def:OpenAndClosedType",
 CellID->68931625],

Cell[TextData[{
 "An ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " integration rule object has one integration rule for the integral estimate \
and one or several null rules for the error estimate. The sampling points of \
the integration rule and the null rules coincide. It should be clear from the \
context whether \"integration rule\" or \"rule\" would mean an ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " integration rule object, or an integration rule in the usual mathematical \
sense."
}], "Text",
 CellID->453813706]
}, Open  ]],

Cell[CellGroupData[{

Cell["Integration Rule Specification", "Section",
 CellID->135207409],

Cell[TextData[{
 "All integration rules described below, except ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ", ",
 "are to be used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s",
 ButtonBox[" adaptive strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#155948475"],
 ". All of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s Monte Carlo strategies, ",
 ButtonBox["crude",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#20795380"],
 " and ",
 ButtonBox["adaptive",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#65285686"],
 ", use ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ". Changing the integration rule component of an integration strategy will \
make a different integration algorithm. "
}], "Text",
 CellID->275460191],

Cell[TextData[{
 "The way to specify what integration rule ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s adaptive strategies ",
 ButtonBox["Global Adaptive",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#137878679"],
 " and ",
 ButtonBox["Local Adaptive",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#19475523"],
 " should use is through a ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " suboption."
}], "Text",
 CellID->107496655],

Cell[TextData[{
 "Here is an example of using an integration rule with a strategy (",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ")."
}], "MathCaption",
 CellID->439197840],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"Method", "\[Rule]", "\"\<ClenshawCurtisRule\>\""}]}], 
      "}"}]}]}], "]"}], "//", "InputForm"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->90688061],

Cell["1.9999999999193905", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]//InputForm=",
 CellID->793054161]
}, Open  ]],

Cell[TextData[{
 "Here is an example of using the same integration rule as in the example \
above through a different strategy (",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 ")."
}], "MathCaption",
 CellID->13137204],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<LocalAdaptive\>\"", ",", 
       RowBox[{"Method", "\[Rule]", "\"\<ClenshawCurtisRule\>\""}]}], 
      "}"}]}]}], "]"}], "//", "InputForm"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->201810221],

Cell["1.9999999976742142", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//InputForm=",
 CellID->550159177]
}, Open  ]],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is given a method option that has only an integration rule specification \
other than ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ", then that rule is used with the ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " strategy. The two inputs below are equivalent."
}], "Text",
 CellID->10639524],

Cell["For this integration only integration rule is specified.", "MathCaption",
 CellID->887361652],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<LobattoKronrodRule\>\""}]}], "]"}], "//",
   "InputForm"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->14623201],

Cell["2.0000000000019873", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//InputForm=",
 CellID->85441193]
}, Open  ]],

Cell["\<\
For this integration an integration strategy and an integration rule are \
specified.\
\>", "MathCaption",
 CellID->730821796],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"Method", "\[Rule]", "\"\<LobattoKronrodRule\>\""}]}], 
      "}"}]}]}], "]"}], "//", "InputForm"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->44418322],

Cell["2.0000000000019873", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]//InputForm=",
 CellID->354958004]
}, Open  ]],

Cell[TextData[{
 "Similarly for ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ", the adaptive Monte Carlo strategy is going to be used when the following \
two equivalent commands are executed."
}], "Text",
 CellID->449396829],

Cell[TextData[{
 "For this Monte Carlo integration only the ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " is specified."
}], "MathCaption",
 CellID->23614898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<MonteCarloRule\>\""}]}], "]"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->438368767],

Cell["1.9923900530424228", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]//InputForm=",
 CellID->13273171]
}, Open  ]],

Cell[TextData[{
 "For this Monte Carlo integration a Monte Carlo integration strategy and ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " are specified."
}], "MathCaption",
 CellID->23122508],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", " ", 
       RowBox[{"Method", "\[Rule]", "\"\<MonteCarloRule\>\""}]}], "}"}]}]}], 
   "]"}], "//", "InputForm"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->521627535],

Cell["1.9745771611582486", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]//InputForm=",
 CellID->444563884]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"TrapezoidalRule\"", "Section",
 CellTags->"TrapezoidalRule",
 CellID->618158740],

Cell["\<\
The trapezoidal rule for integral estimation is one of the simplest and \
oldest rules (possibly used by the Babylonians and certainly by the ancient \
Greek mathematicians):\
\>", "Text",
 CellID->353686003],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     FormBox[
      RowBox[{"(", 
       RowBox[{"b", "-", "a"}]}],
      TraditionalForm], ")"}], " ", 
    FractionBox[
     FormBox[
      RowBox[{
       RowBox[{"f", "(", "a", ")"}], "+", 
       RowBox[{"f", "(", "b", ")"}]}],
      TraditionalForm], "2"]}], "\[TildeTilde]", 
   RowBox[{
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "a", "b"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "."}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"TrapezoidalRule1",
 CellID->833180539],

Cell["The compounded trapezoidal rule is a Riemann sum of the form", "Text",
 CellID->263555821],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     FormBox[
      RowBox[{
       RowBox[{"T", "(", 
        RowBox[{"f", ",", "n"}], ")"}], "=", 
       RowBox[{
        FractionBox["1", "2"], " ", "h", " ", 
        RowBox[{"f", "(", "a", ")"}]}]}],
      TraditionalForm], "+", 
     FormBox[
      RowBox[{"h", " ", 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"i", "=", "1"}], 
         RowBox[{"n", "-", "1"}]], 
        RowBox[{"f", "(", 
         RowBox[{"a", "+", 
          RowBox[{"h", " ", "i"}]}], ")"}]}]}],
      TraditionalForm], "+", 
     FormBox[
      RowBox[{
       FractionBox["1", "2"], " ", "h", " ", 
       RowBox[{"f", "(", "b", ")"}]}],
      TraditionalForm]}], "\[TildeTilde]", 
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "a", "b"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm]}], ","}], TraditionalForm]], "NumberedEquation",
 CellTags->"TrapezoidalRule2",
 CellID->9107376],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"h", "=", 
    FractionBox[
     RowBox[{"b", "-", "a"}], 
     RowBox[{"n", "-", "1"}]]}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->314463073],

Cell[TextData[{
 "If the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option is given the value ",
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 ", the compounded trapezoidal rule is used to estimate each subinterval \
formed by the integration strategy."
}], "Text",
 CellID->675496772],

Cell[TextData[{
 StyleBox["A ", "Text"],
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 " integration:"
}], "MathCaption",
 CellID->56571889],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"x", "+", "5"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "7"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<TrapezoidalRule\>\""}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->7867874],

Cell[BoxData["59.5`"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->412717973]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", "5", Cell[
    "number of coarse trapezoidal points", "TableText"]},
   {"\"\<RombergQuadrature\>\"", 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell[
    "should Romberg quadrature be used or not", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->245149443],

Cell[TextData[{
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->626574275],

Cell[TextData[{
 "The trapezoidal rule and its compounded (multipanel) extension are not very \
accurate. (The compounded trapezoidal rule is exact for linear functions and \
converges at least as fast ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["n", 
    RowBox[{"-", "2"}]], TraditionalForm]], "InlineMath"],
 ", if the integrand has continuous second derivative [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "].) The accuracy of the multipanel trapezoidal rule can be increased using \
\"",
 ButtonBox["Romberg quadrature",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#101738258"],
 "\"."
}], "Text",
 CellID->10033282],

Cell[TextData[{
 "Since the abscissas of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " are a subset of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", the difference ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{
     RowBox[{"T", "(", 
      RowBox[{"f", ",", 
       RowBox[{
        RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], "-", 
     RowBox[{"T", "(", 
      RowBox[{"f", ",", "n"}], ")"}]}], "|"}], TraditionalForm]], 
  "InlineMath"],
 ", can be taken to be an error estimate of the integral estimate ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", and can be computed without extra integrand evaluations."
}], "Text",
 CellID->107355692],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  RowBox[{"\"\<Points\>\"", "->", 
   StyleBox["k", "TI"]}]], "InlineFormula"],
 " can be used to specify how many ",
 "coarse",
 " points are used. The total number of points used by ",
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 " is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", "k"}], "-", "1"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->341442099],

Cell[TextData[{
 "This verifies that the sampling points are as in (",
 
 CounterBox["NumberedEquation", "TrapezoidalRule2"],
 ")."
}], "MathCaption",
 CellID->92628213],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "4"}], ";"}], "\n", 
 RowBox[{"Reap", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"x", "+", "5"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "7"}], "}"}], ",", 
    RowBox[{"Method", "->", 
     RowBox[{"{", 
      RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
       RowBox[{"\"\<Points\>\"", "\[Rule]", "k"}], ",", 
       RowBox[{"\"\<RombergQuadrature\>\"", "\[Rule]", "False"}]}], "}"}]}], 
    ",", 
    RowBox[{"EvaluationMonitor", ":>", 
     RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}]}]}], "Input",
 CellLabel->"In[8]:=",
 CellID->331363070],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"54.`", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
     "1.`", ",", "2.`", ",", "3.`", ",", "4.`", ",", "5.`", ",", "6.`", ",", 
      "7.`"}], "}"}], "}"}]}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->167807418]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 " can be used for multi-dimensional integrals too."
}], "Text",
 CellID->381982768],

Cell[TextData[{
 "Here is a multi-dimensional integration with ",
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 ". The exact result is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["x", "2"], "+", "y"}], ")"}], 
       RowBox[{"\[DifferentialD]", "y"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}], "=", 
    RowBox[{
     RowBox[{"5", "/", "6"}], "=", 
     RowBox[{"0.8333333", "\[Ellipsis]"}]}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->99879431],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "+", "y"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<TrapezoidalRule\>\""}]}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->494333022],

Cell[BoxData["0.8333333334372267`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->492529597]
}, Open  ]],

Cell[TextData[{
 StyleBox["Remark:",
  FontWeight->"Bold"],
 " ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " has both a trapezoidal rule and a trapezoidal strategy; see ",
 ButtonBox["\"Trapezoidal\" Strategy",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#403370786"],
 " in the tutorial ",
 ButtonBox["Integration Strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies"],
 ". All internally implemented integration rules of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " have the suffix -",
 Cell[BoxData[
  ButtonBox["Rule",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Rule"]], "InlineFormula"],
 ". So ",
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 " is used to specify the trapezoidal integration rule, and ",
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " is used to specify the trapezoidal strategy."
}], "Text",
 CellID->354787860],

Cell[CellGroupData[{

Cell["Romberg quadrature", "Subsection",
 CellTags->"RombergQuadrature",
 CellID->101738258],

Cell[TextData[{
 "The idea of the Romberg quadrature is to use a linear combination of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " a",
 "nd ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " that eliminates the same order terms of truncation approximation errors of \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " ",
 "and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"T", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->224041229],

Cell[TextData[{
 "From the Euler\[Dash]Maclaurin formula [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "] we have"
}], "Text",
 CellID->241294357],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "a", "b"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "=", 
    RowBox[{
     FormBox[
      RowBox[{
       FractionBox["1", "2"], " ", "h", " ", 
       RowBox[{"f", "(", "a", ")"}]}],
      TraditionalForm], "+", 
     FormBox[
      RowBox[{"h", " ", 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"i", "=", "1"}], 
         RowBox[{"n", "-", "1"}]], 
        RowBox[{"f", "(", 
         RowBox[{"a", "+", 
          RowBox[{"h", " ", "i"}]}], ")"}]}]}],
      TraditionalForm], "+", 
     FormBox[
      RowBox[{
       FractionBox["1", "2"], " ", "h", " ", 
       RowBox[{"f", "(", "b", ")"}]}],
      TraditionalForm], "-", 
     FormBox[
      RowBox[{
       FractionBox["1", "12"], " ", 
       SuperscriptBox["h", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          SuperscriptBox["f", "\[Prime]",
           MultilineFunction->None], "(", "b", ")"}], "-", 
         RowBox[{
          SuperscriptBox["f", "\[Prime]",
           MultilineFunction->None], "(", "a", ")"}]}], ")"}]}],
      TraditionalForm], "+", 
     FormBox[
      RowBox[{
       FractionBox["1", "720"], " ", 
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}], " ", 
       SuperscriptBox["h", "4"], " ", 
       RowBox[{
        SuperscriptBox["f", "4"], "[", "\[Xi]", "]"}]}],
      TraditionalForm]}]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->504426728],

Cell["where", "Text",
 CellID->666192100],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{"h", "=", 
     FractionBox[
      RowBox[{"b", "-", "a"}], 
      RowBox[{"n", "-", "1"}]]}],
    TraditionalForm], ",", 
   RowBox[{
    FormBox[
     RowBox[{"a", "<", "\[Xi]", "<", "b"}],
     TraditionalForm], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->71003172],

Cell["Hence we can write", "Text",
 CellID->151244],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{"f", 
      RowBox[{"(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}], "=", 
    RowBox[{
     RowBox[{"T", "(", 
      RowBox[{"f", ",", "n"}], ")"}], "+", 
     RowBox[{"A", " ", 
      SuperscriptBox["h", "2"]}], "+", 
     RowBox[{"O", "(", 
      SuperscriptBox["h", "4"], ")"}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->21442028],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "b"], 
    RowBox[{"f", 
     RowBox[{"(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], "=", 
   RowBox[{
    RowBox[{"T", "(", 
     RowBox[{"f", ",", 
      RowBox[{
       RowBox[{"2", "n"}], "-", "1"}]}], ")"}], "+", 
    RowBox[{"A", " ", 
     SuperscriptBox[
      RowBox[{"(", 
       FractionBox["h", "2"], ")"}], "2"]}], "+", 
    RowBox[{
     RowBox[{"O", "(", 
      SuperscriptBox["h", "4"], ")"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->195266292],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["h", "2"], TraditionalForm]], "InlineMath"],
 " terms of the equations above can be eliminated if from the second equation \
multiplied by ",
 Cell[BoxData[
  FormBox["4", TraditionalForm]], "InlineMath"],
 " the first one is subtracted. The result is",
 " ",
 " "
}], "Text",
 CellID->325738640],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "b"], 
    RowBox[{"f", 
     RowBox[{"(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], "=", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"4", 
       RowBox[{"T", "(", 
        RowBox[{"f", ",", 
         RowBox[{
          RowBox[{"2", "n"}], "-", "1"}]}], ")"}]}], "-", 
      RowBox[{"T", 
       RowBox[{"(", 
        RowBox[{"f", ",", "n"}], ")"}]}]}], "3"], "+", 
    RowBox[{
     RowBox[{"O", "(", 
      SuperscriptBox["h", "4"], ")"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->717613353],

Cell[TextData[{
 "Here is an example that shows that a trapezoidal rule that uses Romberg \
quadrature gives better performance than the standard trapezoidal rule. Also, \
the result of the former is closer to the exact result, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       SqrtBox["x"], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "=", 
    RowBox[{
     FractionBox["2", "3"], "=", 
     RowBox[{"0.6666666", "\[Ellipsis]"}]}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->32589623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
           RowBox[{"\"\<Points\>\"", "\[Rule]", "5"}], ",", 
           RowBox[{"\"\<RombergQuadrature\>\"", "\[Rule]", "True"}]}], 
          "}"}]}], ",", 
        RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", "\[Infinity]"}]}], 
       "}"}]}], ",", 
     RowBox[{"MaxRecursion", "\[Rule]", "100"}], ",", 
     RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}], "//", "InputForm"}], 
  "//", "Timing"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->309570571],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.06399`", ",", 
   InterpretationBox[
    StyleBox["0.6666666666571325",
     ShowStringCharacters->True,
     NumberMarks->True],
    InputForm[0.6666666666571325],
    AutoDelete->True,
    Editable->True]}], "}"}]], "Output",
 ImageSize->{205, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->360285275]
}, Open  ]],

Cell["\<\
Here is an integration with a trapezoidal rule that does not use Romberg \
quadrature.\
\>", "MathCaption",
 CellID->919715312],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
           RowBox[{"\"\<Points\>\"", "\[Rule]", "5"}], ",", 
           RowBox[{"\"\<RombergQuadrature\>\"", "\[Rule]", "False"}]}], 
          "}"}]}], ",", 
        RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", "\[Infinity]"}]}], 
       "}"}]}], ",", 
     RowBox[{"MaxRecursion", "\[Rule]", "100"}], ",", 
     RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}], "//", "InputForm"}], 
  "//", "Timing"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->294291419],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.10998300000000004`", ",", 
   InterpretationBox[
    StyleBox["0.6666666644416138",
     ShowStringCharacters->True,
     NumberMarks->True],
    InputForm[0.6666666644416138],
    AutoDelete->True,
    Editable->True]}], "}"}]], "Output",
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->996818232]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"TrapezoidalRule\" sampling points and weights", "Subsection",
 CellID->12731638],

Cell["\<\
The following calculates the trapezoidal sampling points, weights, and error \
weights for a given precision.\
\>", "MathCaption",
 CellID->869822652],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";", 
  RowBox[{"precision", "=", "MachinePrecision"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`TrapezoidalRuleData", "[", 
   RowBox[{"n", ",", "precision"}], "]"}]}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->436027268],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.`", ",", "0.125`", ",", "0.25`", ",", "0.375`", ",", "0.5`", ",", 
     "0.625`", ",", "0.75`", ",", "0.875`", ",", "1.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.0625`", ",", "0.125`", ",", "0.125`", ",", "0.125`", ",", "0.125`", 
     ",", "0.125`", ",", "0.125`", ",", "0.125`", ",", "0.0625`"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.0625`"}], ",", "0.125`", ",", 
     RowBox[{"-", "0.125`"}], ",", "0.125`", ",", 
     RowBox[{"-", "0.125`"}], ",", "0.125`", ",", 
     RowBox[{"-", "0.125`"}], ",", "0.125`", ",", 
     RowBox[{"-", "0.0625`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{489, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1959317245]
}, Open  ]],

Cell["\<\
Here is how the Romberg quadrature weights and error weights can be derived.\
\>", "MathCaption",
 CellID->309374735],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"rombergAbsc", "=", "absc"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"lowOrderWeights", "=", 
   RowBox[{"-", 
    RowBox[{"(", 
     RowBox[{"errweights", "-", "weights"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"rombergWeights", "=", 
   FractionBox[
    RowBox[{
     RowBox[{"4", "weights"}], "-", "lowOrderWeights"}], "3"]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"rombergErrorWeights", "=", 
   RowBox[{"rombergWeights", "-", "weights"}]}], ";"}], "\n", 
 RowBox[{"{", 
  RowBox[{"rombergAbsc", ",", "rombergWeights", ",", "rombergErrorWeights"}], 
  "}"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->232745638],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.`", ",", "0.125`", ",", "0.25`", ",", "0.375`", ",", "0.5`", ",", 
     "0.625`", ",", "0.75`", ",", "0.875`", ",", "1.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.041666666666666664`", ",", "0.16666666666666666`", ",", 
     "0.08333333333333333`", ",", "0.16666666666666666`", ",", 
     "0.08333333333333333`", ",", "0.16666666666666666`", ",", 
     "0.08333333333333333`", ",", "0.16666666666666666`", ",", 
     "0.041666666666666664`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.020833333333333336`"}], ",", "0.04166666666666666`", ",",
      
     RowBox[{"-", "0.04166666666666667`"}], ",", "0.04166666666666666`", ",", 
     
     RowBox[{"-", "0.04166666666666667`"}], ",", "0.04166666666666666`", ",", 
     
     RowBox[{"-", "0.04166666666666667`"}], ",", "0.04166666666666666`", ",", 
     
     RowBox[{"-", "0.020833333333333336`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{489, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->1707758716]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"NewtonCotesRule\"", "Section",
 CellTags->"NewtonCotesRule",
 CellID->81663330],

Cell["\<\
Newton\[Dash]Cotes integration formulas are formulas of interpolatory type \
with sampling points that are equally spaced. \
\>", "Text",
 CellID->386092692],

Cell[TextData[{
 "The Newton-Cotes quadrature for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can be specified with the ",
 ButtonBox["Method",
  BaseStyle->"Link",
  ButtonData->"paclet:ref/Method"],
 " option value ",
 Cell[BoxData["\"\<NewtonCotesRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->337187050],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<NewtonCotesRule\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[20]:=",
 CellID->646414825],

Cell[BoxData["2.0000000019587723`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->135573007]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", "3", Cell["\<\
number of coarse Newton\[Dash]Cotes points\
\>", "TableText"]},
   {"\"\<Type\>\"", "Closed", Cell[
    "type of the Newton\[Dash]Cotes rule", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->226067643],

Cell[TextData[{
 Cell[BoxData["\"\<NewtonCotesRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->447898499],

Cell[TextData[{
 "Let the interval of integration, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"a", ",", "b"}], "]"}], TraditionalForm]], "InlineMath"],
 ", be divided into ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "-", "1"}], TraditionalForm]], "InlineMath"],
 " subintervals of equal length by the points"
}], "Text",
 CellID->806097146],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox["a",
    TraditionalForm], ",", 
   FormBox[
    RowBox[{"a", "+", "h"}],
    TraditionalForm], ",", 
   FormBox[
    RowBox[{"a", "+", 
     RowBox[{"2", " ", "h"}]}],
    TraditionalForm], ",", "\[Ellipsis]", ",", 
   FormBox[
    RowBox[{
     RowBox[{"a", "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"n", "-", "1"}], ")"}], " ", "h"}]}], "=", "b"}],
    TraditionalForm], ",", " ", 
   RowBox[{
    FormBox[
     RowBox[{"h", "=", 
      FractionBox[
       RowBox[{"b", "-", "a"}], 
       RowBox[{"n", "-", "1"}]]}],
     TraditionalForm], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->310285549],

Cell["\<\
Then the integration formula of interpolatory type is given by\
\>", "Text",
 CellID->102306622],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "a", "b"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "\[TildeTilde]", 
    RowBox[{
     FormBox[
      FractionBox[
       RowBox[{"b", "-", "a"}], 
       RowBox[{"n", "-", "1"}]],
      TraditionalForm], 
     FormBox[
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{"k", "=", "0"}], 
        RowBox[{"n", "-", "1"}]], 
       RowBox[{
        SubscriptBox["B", 
         RowBox[{
          RowBox[{"n", "-", "1"}], ",", "k"}]], " ", 
        RowBox[{"f", "(", 
         RowBox[{"a", "+", 
          RowBox[{"h", " ", "k"}]}], ")"}]}]}],
      TraditionalForm]}]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->574268937],

Cell["where", "Text",
 CellID->436311501],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["B", 
     RowBox[{
      RowBox[{"n", "-", "1"}], ",", "k"}]], "=", 
    RowBox[{
     FormBox[
      FractionBox[
       RowBox[{"n", "-", "1"}], 
       RowBox[{"b", "-", "a"}]],
      TraditionalForm], 
     FormBox[
      RowBox[{
       SubsuperscriptBox["\[Integral]", "a", "b"], 
       RowBox[{
        FractionBox[
         RowBox[{"w", "(", "x", ")"}], 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", "a"}], "-", 
            RowBox[{"k", " ", "h"}], "+", "x"}], ")"}], " ", 
          RowBox[{
           SuperscriptBox["w", "\[Prime]",
            MultilineFunction->None], "(", 
           RowBox[{"a", "+", 
            RowBox[{"h", " ", "k"}]}], ")"}]}]], 
        RowBox[{"\[DifferentialD]", "x"}]}]}],
      TraditionalForm]}]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->197244546],

Cell["with", "Text",
 CellID->585197279],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{"w", "(", "x", ")"}],
    TraditionalForm], "=", 
   RowBox[{
    FormBox[
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "a"}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{"x", "-", "a", "-", "h"}], ")"}]}],
     TraditionalForm], 
    RowBox[{
     RowBox[{"\[Ellipsis]", "(", 
      RowBox[{
       FormBox[
        RowBox[{"x", "-", "a"}],
        TraditionalForm], "-", 
       FormBox[
        RowBox[{
         RowBox[{"(", 
          RowBox[{"n", "-", "1"}], ")"}], " ", "h"}],
        TraditionalForm]}], ")"}], "."}]}]}], TraditionalForm]], "DisplayMath",\

 CellID->221787720],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " is large, the Newton\[Dash]Cotes ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 "-point coefficients are large and are of mixed sign."
}], "Text",
 CellID->274109754],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate`NewtonCotesRuleData", "[", 
   RowBox[{"25", ",", "MachinePrecision"}], "]"}], "[", 
  RowBox[{"[", "2", "]"}], "]"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->43642413],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.004211688103018862`", ",", "0.07120021233785076`", ",", 
   RowBox[{"-", "0.49996530490860375`"}], ",", "5.170283332910602`", ",", 
   RowBox[{"-", "43.21780224418538`"}], ",", "306.5281855878435`", ",", 
   RowBox[{"-", "1854.4368365284338`"}], ",", "9697.72990242269`", ",", 
   RowBox[{"-", "44332.375221568545`"}], ",", "178882.34730662667`", ",", 
   RowBox[{"-", "642291.4660636157`"}], ",", "2.0662034181067916`*^6", ",", 
   RowBox[{"-", "5.989338018589833`*^6"}], ",", "1.571992416818574`*^7", ",", 
   
   RowBox[{"-", "3.751166782484619`*^7"}], ",", "8.166460515677029`*^7", ",", 
   
   RowBox[{"-", "1.626782402514811`*^8"}], ",", "2.9725618150922334`*^8", ",",
    
   RowBox[{"-", "4.992777559365902`*^8"}], ",", "7.721713535309594`*^8", ",", 
   
   RowBox[{"-", "1.101177481497187`*^9"}], ",", "1.4496362492196114`*^9", ",",
    
   RowBox[{"-", "1.7631444527510812`*^9"}], ",", "1.9824528961193094`*^9", 
   ",", 
   RowBox[{"-", "2.0613776923951828`*^9"}], ",", "1.9824528961193094`*^9", 
   ",", 
   RowBox[{"-", "1.7631444527510812`*^9"}], ",", "1.4496362492196114`*^9", 
   ",", 
   RowBox[{"-", "1.101177481497187`*^9"}], ",", "7.721713535309594`*^8", ",", 
   
   RowBox[{"-", "4.992777559365902`*^8"}], ",", "2.9725618150922334`*^8", ",",
    
   RowBox[{"-", "1.626782402514811`*^8"}], ",", "8.166460515677029`*^7", ",", 
   
   RowBox[{"-", "3.751166782484619`*^7"}], ",", "1.571992416818574`*^7", ",", 
   
   RowBox[{"-", "5.989338018589833`*^6"}], ",", "2.0662034181067916`*^6", ",",
    
   RowBox[{"-", "642291.4660636157`"}], ",", "178882.34730662667`", ",", 
   RowBox[{"-", "44332.375221568545`"}], ",", "9697.72990242269`", ",", 
   RowBox[{"-", "1854.4368365284338`"}], ",", "306.5281855878435`", ",", 
   RowBox[{"-", "43.21780224418538`"}], ",", "5.170283332910602`", ",", 
   RowBox[{"-", "0.49996530490860375`"}], ",", "0.07120021233785076`", ",", 
   "0.004211688103018862`"}], "}"}]], "Output",
 ImageSize->{576, 139},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->481638253]
}, Open  ]],

Cell["\<\
Since this may lead to large losses of significance by cancelation, a \
high-order Newton\[Dash]Cotes rule must be used with caution.\
\>", "Text",
 CellID->187839623],

Cell[CellGroupData[{

Cell["\"NewtonCotesRule\" sampling points and weights", "Subsection",
 CellID->617627612],

Cell["\<\
The following calculates the Newton\[Dash]Cotes sampling points, weights, and \
error weights for a given precision.\
\>", "MathCaption",
 CellID->242967120],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";", 
  RowBox[{"precision", "=", "MachinePrecision"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`NewtonCotesRuleData", "[", 
   RowBox[{"n", ",", "precision"}], "]"}]}]}], "Input",
 CellLabel->"In[22]:=",
 CellID->137233356],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.`", ",", "0.125`", ",", "0.25`", ",", "0.375`", ",", "0.5`", ",", 
     "0.625`", ",", "0.75`", ",", "0.875`", ",", "1.`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.03488536155202822`", ",", "0.20768959435626103`", ",", 
     RowBox[{"-", "0.0327336860670194`"}], ",", "0.37022927689594354`", ",", 
     RowBox[{"-", "0.16014109347442682`"}], ",", "0.37022927689594354`", ",", 
     
     RowBox[{"-", "0.0327336860670194`"}], ",", "0.20768959435626103`", ",", 
     "0.03488536155202822`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "0.04289241622574956`"}], ",", "0.20768959435626103`", ",", 
     
     RowBox[{"-", "0.38828924162257494`"}], ",", "0.37022927689594354`", ",", 
     
     RowBox[{"-", "0.29347442680776015`"}], ",", "0.37022927689594354`", ",", 
     
     RowBox[{"-", "0.38828924162257494`"}], ",", "0.20768959435626103`", ",", 
     
     RowBox[{"-", "0.04289241622574956`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{461, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->74978554]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"GaussBerntsenEspelidRule\"", "Section",
 CellTags->"GaussBerntsenEspelidRule",
 CellID->141675390],

Cell["\<\
Gaussian quadrature uses optimal sampling points (through polynomial \
interpolation) to form a weighted sum of the integrand values over these \
points. On a subset of these sampling points a lower order quadrature rule \
can be made. The difference between the two rules can be used to estimate the \
error. Berntsen and Espelid derived error estimation rules by removing the \
central point of Gaussian rules with odd number of sampling points.\
\>", "Text",
 CellID->366043185],

Cell[TextData[{
 "The Gaussian quadrature for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can be specified with the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option value ",
 Cell[BoxData["\"\<GaussBerntsenEspelidRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->188552690],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<GaussBerntsenEspelidRule\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->594944],

Cell[BoxData["2.0000000000000036`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->83729302]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of Gauss points", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->788578913],

Cell[TextData[{
 Cell[BoxData["\"\<GaussBerntsenEspelidRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->85891900],

Cell[TextData[{
 "A Gaussian rule ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " points for integrand ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", "n"}], "-", "1"}], TraditionalForm]], "InlineMath"],
 ". (I.e. ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"G", "(", 
     RowBox[{"f", ",", "n"}], ")"}], "=", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " is polynomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", 
    RowBox[{
     RowBox[{"2", "n"}], "-", "1"}]}], TraditionalForm]], "InlineMath"],
 ".) "
}], "Text",
 CellID->771349169],

Cell[TextData[{
 "Gaussian rules are of open type since the integrand is not evaluated at the \
end points of the interval. (",
 ButtonBox["Lobatto rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#210486704"],
 ", ",
 ButtonBox["Clenshaw\[Dash]Curtis rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#486402291"],
 ", and the ",
 ButtonBox["trapezoidal rule",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#618158740"],
 " are of closed type since they use integrand evaluations at the interval \
end points.)"
}], "Text",
 CellID->18936690],

Cell[TextData[{
 "This defines the divided differences functional [",
 ButtonBox["Ehrich2000",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#78468907"],
 "]"
}], "Text",
 CellID->23167374],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     FormBox[
      RowBox[{"dvd", "(", 
       RowBox[{
        SubscriptBox["t", "1"], ",", 
        SubscriptBox["t", "2"], ",", "\[Ellipsis]", ",", 
        SubscriptBox["t", 
         RowBox[{"s", "+", "1"}]]}], ")"}],
      TraditionalForm], " ", "[", "f", "]"}], "=", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"\[Nu]", "=", "1"}], 
      RowBox[{"s", "+", "1"}]], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        UnderoverscriptBox["\[Product]", GridBox[{
           {
            RowBox[{"\[Mu]", "=", "1"}]},
           {
            RowBox[{"\[Mu]", "\[NotEqual]", "\[Nu]"}]}
          }], 
         RowBox[{"s", "+", "1"}]], 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SubscriptBox["t", "\[Nu]"], "-", 
           SubscriptBox["t", "\[Mu]"]}], ")"}], 
         RowBox[{"-", "1"}]]}], ")"}], 
      FormBox[
       RowBox[{"f", "(", 
        SubscriptBox["t", "\[Nu]"], ")"}],
       TraditionalForm]}]}]}], ",", " ", 
   RowBox[{"0", "\[LessEqual]", 
    SubscriptBox["t", "1"], "<", 
    SubscriptBox["t", "2"], "<", "\[Ellipsis]", "<", 
    SubscriptBox["t", 
     RowBox[{"s", "+", "1"}]], "\[LessEqual]", "1."}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->521791750],

Cell[TextData[{
 "If the sampling points of the Gaussian rule ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", "n"}], "+", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", with sampling points ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], ",", 
    SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
    SubscriptBox["x", 
     RowBox[{
      RowBox[{"2", "n"}], "+", "1"}]]}], TraditionalForm]], "InlineMath"],
 ", Berntsen and Espelid have derived the following error estimate functional \
(see [",
 ButtonBox["Ehrich2000",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#78468907"],
 "])"
}], "Text",
 CellID->101767876],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"E", "(", 
    RowBox[{"G", "(", 
     RowBox[{"f", ",", 
      RowBox[{
       RowBox[{"2", "m"}], "+", "1"}]}], ")"}], ")"}], "=", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "n"], 
    FractionBox[
     RowBox[{
      SuperscriptBox["2", 
       RowBox[{
        RowBox[{"2", "n"}], "+", "1"}]], 
      SuperscriptBox[
       RowBox[{"n", "!"}], "2"], 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"2", "n"}], ")"}], "!"}]}], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"4", "n"}], "+", "1"}], ")"}], "!"}]], 
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{"dvd", "(", 
        RowBox[{
         SubscriptBox["x", "1"], ",", 
         SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
         SubscriptBox["x", 
          RowBox[{
           RowBox[{"2", "n"}], "+", "1"}]]}], ")"}],
       TraditionalForm], " ", "[", "f", "]"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->46242465],

Cell[TextData[{
 "(The original formula in [",
 ButtonBox["Ehrich2000",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#78468907"],
 "] is for sampling points in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 ". The formula above is for sampling points in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 ".)"
}], "Text",
 CellID->98024343],

Cell[TextData[{
 "This example shows the number of sampling points used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with various values of ",
 Cell[BoxData["\"\<GaussBerntsenEspelidRule\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<Points\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->8631885],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"k", "=", "0"}], ";", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"x", "^", 
        RowBox[{"(", 
         RowBox[{"1", "/", "2"}], ")"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<GaussBerntsenEspelidRule\>\"", ",", 
          RowBox[{"\"\<Points\>\"", "\[Rule]", "i"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}], ")"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "2", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->13965209],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "164", ",", "106", ",", "110", ",", "128", ",", "146", ",", "164", ",", 
   "182", ",", "200", ",", "218", ",", "236", ",", "225", ",", "243", ",", 
   "261", ",", "279", ",", "231", ",", "245", ",", "259", ",", "273", ",", 
   "287"}], "}"}]], "Output",
 ImageSize->{361, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->605619980]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"GaussBerntsenEspelidRule\" sampling points and weights", "Subsection",
 CellID->165210023],

Cell["\<\
The following calculates the Gaussian abscissas, weights, and \
Bernsen\[Dash]Espelid error weights for a given number of coarse points and \
precision.\
\>", "MathCaption",
 CellID->122899051],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";", 
  RowBox[{"precision", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`GaussBerntsenEspelidRuleData", "[", 
   RowBox[{"n", ",", "precision"}], "]"}]}]}], "Input",
 CellLabel->"In[26]:=",
 CellID->857995281],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.0108856709269715035980309994385668454`20.", ",", 
     "0.0564687001159523504624211153480357162`20.", ",", 
     "0.13492399721297533795329187398442245282`20.", ",", 
     "0.24045193539659409203713716527069519536`20.", ",", 
     "0.36522842202382751383423400729956921333`20.", ",", "0.5`20.", ",", 
     "0.63477157797617248616576599270043078667`20.", ",", 
     "0.75954806460340590796286283472930480464`20.", ",", 
     "0.86507600278702466204670812601557754718`20.", ",", 
     "0.9435312998840476495375788846519642838`20.", ",", 
     "0.9891143290730284964019690005614331546`20."}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.0278342835580868332418069770353173528`19.69897000433602", ",", 
     "0.06279018473245231231803506176509164987`19.69897000433602", ",", 
     "0.09314510546386712571487243829803460201`19.69897000433602", ",", 
     "0.11659688229599523996159488349777627855`19.69897000433602", ",", 
     "0.1314022722551233310903444349485395124`19.69897000433602", ",", 
     "0.13646254338895031535724176416817109458`19.69897000433602", ",", 
     "0.1314022722551233310903444349485395124`19.69897000433602", ",", 
     "0.11659688229599523996159488349777627855`19.69897000433602", ",", 
     "0.09314510546386712571487243829803460201`19.69897000433602", ",", 
     "0.06279018473245231231803506176509164987`19.69897000433602", ",", 
     "0.0278342835580868332418069770353173528`19.69897000433602"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
     "-", "0.02558041542407929977267670473756403132`18.678008365695064"}], 
     ",", "0.08546625092175164372912494887791114845`18.62168586473116", ",", 
     RowBox[{
     "-", "0.15407013862509290813252755596515072392`18.551694159442984"}], 
     ",", "0.21562641393186216188031380737`18.478665206161885", ",", 
     RowBox[{"-", "0.25790465419339191306147625972`18.405268146458695"}], ",",
      "0.27292508677790063071448352834`18.33140699930036", ",", 
     RowBox[{"-", "0.25790465419339191306147625972`18.2556981955688"}], ",", 
     "0.21562641393186216188031380737`18.175892721423754", ",", 
     RowBox[{"-", "0.15407013862509290813252755597`18.089300601364606"}], ",",
      "0.08546625092175164372912494887`17.99780955302844", ",", 
     RowBox[{
     "-", "0.02558041542407929977267670473756403132`18.046414819744665"}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{517, 203},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[27]=",
 CellID->268189062]
}, Open  ]],

Cell["The Berntsen-Espelid error weights are implemented below.", "Text",
 CellID->369875033],

Cell["This implements the divided differences.", "MathCaption",
 CellID->267299906],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"polyd", "[", 
    RowBox[{"vec_List", ",", " ", "nu_"}], "]"}], " ", ":=", " ", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"Times", "@@", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"vec", "[", 
         RowBox[{"[", "nu", "]"}], "]"}], "-", 
        RowBox[{"Drop", "[", 
         RowBox[{"vec", ",", 
          RowBox[{"{", "nu", "}"}]}], "]"}]}], ")"}]}], ")"}], "^", 
    RowBox[{"(", 
     RowBox[{"-", "1"}], ")"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"dvdWeights", "[", "vec_List", "]"}], " ", ":=", " ", 
    RowBox[{
     RowBox[{"dvdWeights", "[", "vec", "]"}], " ", "=", " ", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"polyd", "[", 
        RowBox[{"vec", ",", " ", "nu"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"nu", ",", " ", "1", ",", " ", 
         RowBox[{"Length", "[", "vec", "]"}]}], "}"}]}], "]"}]}]}], ";"}], 
  " "}]}], "Input",
 CellLabel->"In[28]:=",
 CellID->404712197],

Cell[TextData[{
 "This computes the abscissas and the weights of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", 
    RowBox[{"(", 
     RowBox[{"f", ",", 
      RowBox[{
       RowBox[{"2", "n"}], "+", "1"}]}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->202744913],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
   RowBox[{"NIntegrate`GaussRuleData", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"2", "n"}], "+", "1"}], ",", "precision"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[30]:=",
 CellID->14344216],

Cell["\<\
This computes the error Berntsen\[Dash]Espelid error weights.\
\>", "MathCaption",
 CellID->766006120],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], "n"], " ", 
     SqrtBox["\[Pi]"], " ", 
     SuperscriptBox[
      RowBox[{"Gamma", "[", 
       RowBox[{"1", "+", "n"}], "]"}], "2"]}], ")"}], 
   RowBox[{
    RowBox[{"Gamma", "[", 
     RowBox[{
      FractionBox["3", "2"], "+", 
      RowBox[{"2", " ", "n"}]}], "]"}], " ", 
    SuperscriptBox["2", 
     RowBox[{"2", " ", "n"}]]}]], " ", 
  RowBox[{"dvdWeights", "[", "absc", "]"}]}]], "Input",
 CellLabel->"In[31]:=",
 CellID->226502069],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
   "-", "0.02558041542407929977267670473765354598`18.959042042908774"}], ",", 
   "0.0854662509217516437291249488782102242`18.857129238498175", ",", 
   RowBox[{
   "-", "0.15407013862509290813252755596346474011`18.742943322267838"}], ",", 
   "0.21562641393186216188031380737`18.634230543595425", ",", 
   RowBox[{"-", "0.25790465419339191306147625972`18.532677334001413"}], ",", 
   "0.27292508677790063071448352834`18.436255267436454", ",", 
   RowBox[{"-", "0.25790465419339191306147625972`18.341977843309255"}], ",", 
   "0.21562641393186216188031380737`18.246438769076427", ",", 
   RowBox[{"-", "0.15407013862509290813252755597`18.146216292185624"}], ",", 
   "0.08546625092175164372912494887`18.043324341164848", ",", 
   RowBox[{
   "-", "0.02558041542407929977267670473765354598`18.097649578338427"}]}], 
  "}"}]], "Output",
 ImageSize->{497, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->33431475]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"GaussKronrodRule\"", "Section",
 CellTags->"GaussKronrodRule",
 CellID->381359375],

Cell["\<\
Gaussian quadrature uses optimal sampling points (through polynomial \
interpolation) to form a weighted sum of the integrand values over these \
points. The Kronrod extension of a Gaussian rule adds new sampling points in \
between the Gaussian points and forms a higher-order rule that reuses the \
Gaussian rule integrand evaluations.\
\>", "Text",
 CellID->270397252],

Cell[TextData[{
 "The Gauss\[Dash]Kronrod quadrature for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can be specified with the ",
 ButtonBox["Method",
  BaseStyle->"Link",
  ButtonData->"paclet:ref/Method"],
 " option value ",
 Cell[BoxData["\"\<GaussKronrodRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->205758888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<GaussKronrodRule\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->340866141],

Cell[BoxData["2.000000000000003`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->6236080]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
number of Gauss points that will be extended with Kronrod points\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->40598151],

Cell[TextData[{
 Cell[BoxData["\"\<GaussKronrodRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->489455538],

Cell[TextData[{
 "A Gaussian rule ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " points for integrand ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", "n"}], "-", "1"}], TraditionalForm]], "InlineMath"],
 ", that is, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"G", "(", 
     RowBox[{"f", ",", "n"}], ")"}], "=", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " is polynomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", 
    RowBox[{
     RowBox[{"2", "n"}], "-", "1"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->34338313],

Cell["\<\
Gauss\[Dash]Kronrod rules are of open type since the integrand is not \
evaluated at the end points of the interval.\
\>", "Text",
 CellID->642176838],

Cell[TextData[{
 "The Kronrod extension ",
 Cell[BoxData[
  FormBox[
   RowBox[{"GK", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " of a Gaussian rule with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", " "}], TraditionalForm]], "InlineMath"],
 " points ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " adds ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "+", "1"}], TraditionalForm]], "InlineMath"],
 " points to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " and the extended rule is exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"3", "n"}], "+", "1"}], TraditionalForm]], "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", " "}], TraditionalForm]], "InlineMath"],
 " is even, or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"3", "n"}], "+", "2"}], TraditionalForm]], "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " is odd. The weights associated with a Gaussian rule change in its Kronrod \
extension."
}], "Text",
 CellTags->"GaussKronrodRuleDegree",
 CellID->411028157],

Cell[TextData[{
 "Since the abscissas of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " are a subset of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"GK", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 ", the difference ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LeftBracketingBar]", 
    RowBox[{
     RowBox[{"GK", "(", 
      RowBox[{"f", ",", "n"}], ")"}], "-", 
     RowBox[{"G", "(", 
      RowBox[{"f", ",", "n"}], ")"}]}], "\[RightBracketingBar]"}], 
   TraditionalForm]], "InlineMath"],
 " can be taken to be an error estimate of the integral estimate ",
 Cell[BoxData[
  FormBox[
   RowBox[{"GK", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 ", and can be computed without extra integrand evaluations."
}], "Text",
 CellID->369924265],

Cell[TextData[{
 "This example shows the number of sampling point used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with various values of ",
 Cell[BoxData["\"\<GaussKronrodRule\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<Points\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->1330256],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"k", "=", "0"}], ";", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"x", "^", "10"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<GaussKronrodRule\>\"", ",", 
          RowBox[{"\"\<Points\>\"", "\[Rule]", "i"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}], ")"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "2", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->481448378],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "284", ",", "91", ",", "63", ",", "33", ",", "13", ",", "15", ",", "17", 
   ",", "19", ",", "21", ",", "23", ",", "25", ",", "27", ",", "29", ",", 
   "31", ",", "33", ",", "35", ",", "37", ",", "39", ",", "41"}], 
  "}"}]], "Output",
 ImageSize->{493, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->386419301]
}, Open  ]],

Cell[TextData[{
 "For an implementation description of Kronrod extensions of Gaussian rules, \
see [",
 ButtonBox["PiesBrand74",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#186112679"],
 "]. "
}], "Text",
 CellID->371546905],

Cell[CellGroupData[{

Cell["\"GaussKronrodRule\" sampling points and weights", "Subsection",
 CellID->360328261],

Cell["\<\
The following calculates the Gauss\[Dash]Kronrod abscissas, weights, and \
error weights for given number of coarse points and precision.\
\>", "MathCaption",
 CellID->510007182],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";", 
  RowBox[{"precision", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
   RowBox[{"n", ",", "precision"}], "]"}]}]}], "Input",
 CellLabel->"In[34]:=",
 CellID->242297082],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.00795731995257876775191353268192826317`17.904236463836078", ",", 
     "0.04691007703066800360118656085030137242`18.692132268328738", ",", 
     "0.12291663671457538977959141652694210503`19.14656979514744", ",", 
     "0.23076534494715845448184278964989549204`19.47711174790053", ",", 
     "0.36018479341910840329326673862551127484`19.75047081565451", ",", 
     "0.5`20.", ",", "0.63981520658089159670673326137448872516`20.", ",", 
     "0.76923465505284154551815721035010450796`20.", ",", 
     "0.87708336328542461022040858347305789497`20.", ",", 
     "0.95308992296933199639881343914969862758`20.", ",", 
     "0.99204268004742123224808646731807173683`20."}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.02129101837554091643225472542383229017`20.", ",", 
     "0.05761665831123669701249019502504849095`19.205913010778012", ",", 
     "0.09340039827824632873390001343924299055`20.", ",", 
     "0.12052016961432379334997130561150594331`19.226015107998496", ",", 
     "0.13642490095627946117049663224222775871`20.", ",", 
     "0.1414937089287456066021278006863347476`19.10537347739902", ",", 
     "0.13642490095627946117049663224222775871`20.", ",", 
     "0.12052016961432379334997130561150594331`19.226015107998496", ",", 
     "0.09340039827824632873390001343924299055`20.", ",", 
     "0.05761665831123669701249019502504849095`19.205913010778012", ",", 
     "0.02129101837554091643225472542383229017`20."}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.02129101837554091643225472542383229017`20.", ",", 
     RowBox[{
     "-", "0.06084678421685784674500527976148545076`19.698970004336026"}], 
     ",", "0.09340039827824632873390001343924299055`20.", ",", 
     RowBox[{
     "-", "0.11879416563535944067067445180648348827`19.69897000433602"}], ",",
      "0.13642490095627946117049663224222775871`20.", ",", 
     RowBox[{
     "-", "0.14295073551569883784231664375810969685`19.42181650942328"}], ",",
      "0.13642490095627946117049663224222775871`20.", ",", 
     RowBox[{
     "-", "0.11879416563535944067067445180648348827`19.69897000433602"}], ",",
      "0.09340039827824632873390001343924299055`20.", ",", 
     RowBox[{
     "-", "0.06084678421685784674500527976148545076`19.698970004336026"}], 
     ",", "0.02129101837554091643225472542383229017`20."}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{532, 203},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]=",
 CellID->10911793]
}, Open  ]],

Cell[TextData[{
 "The calculations below demonstrate the degree of the Gauss-Kronrod \
integration rule (see ",
 ButtonBox["above",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#411028157"],
 ")."
}], "Text",
 CellID->336934172],

Cell["\<\
This computes the degree of the Gauss\[Dash]Kronrod integration rule.\
\>", "MathCaption",
 CellID->7329969],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "=", 
  RowBox[{"If", "[", 
   RowBox[{
    RowBox[{"OddQ", "[", "n", "]"}], ",", 
    RowBox[{
     RowBox[{"3", "*", "n"}], "+", "2"}], ",", 
    RowBox[{
     RowBox[{"3", "*", "n"}], "+", "1"}]}], "]"}]}]], "Input",
 CellLabel->"In[36]:=",
 CellID->451066042],

Cell[BoxData["17"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]=",
 CellID->491184162]
}, Open  ]],

Cell["This defines a function.", "MathCaption",
 CellID->90881048],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  SuperscriptBox["x", "p"]}]], "Input",
 CellLabel->"In[37]:=",
 CellID->543769040],

Cell[TextData[{
 "The command below implements the integration rule weighted sums for the \
integral estimate, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", " ", "n"}], "+", "1"}]], 
    RowBox[{
     SubscriptBox["w", "i"], " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ", and the error estimate, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", " ", "n"}], "+", "1"}]], 
    RowBox[{
     SubscriptBox["e", "i"], " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     RowBox[{"{", 
      SubscriptBox["x", "i"], " ", "}"}], 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", "n"}], "+", "1"}]], " "}], TraditionalForm]], 
  "InlineMath"],
 " are the abscissas, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     RowBox[{"{", 
      SubscriptBox["w", "i"], " ", "}"}], 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", "n"}], "+", "1"}]], " "}], TraditionalForm]], 
  "InlineMath"],
 " are the weights, and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     RowBox[{"{", 
      SubscriptBox["e", "i"], " ", "}"}], 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", "n"}], "+", "1"}]], " "}], TraditionalForm]], 
  "InlineMath"],
 "are the error weights."
}], "Text",
 CellID->3201818],

Cell[TextData[{
 "These are the integral and error estimates for",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " computed with the rule."
}], "MathCaption",
 CellID->10533031],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "@", 
  RowBox[{"MapThread", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[38]:=",
 CellID->51837532],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.05555555555555555555563366249220808965`18.687863323141702", ",", 
   "0.00044344096276720964104819151419322905`16.624162954964568"}], 
  "}"}]], "Output",
 ImageSize->{324, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->42626316]
}, Open  ]],

Cell["The integral estimate above coincides with the exact result.", \
"MathCaption",
 CellID->189440317],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", "precision"}], 
  "]"}]], "Input",
 CellLabel->"In[39]:=",
 CellID->274742322],

Cell[BoxData["0.05555555555555555555555555555555555555`20."], "Output",
 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->86169558]
}, Open  ]],

Cell[TextData[{
 "The error estimate above is not zero since the imbedded Gauss rule is exact \
for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", 
    RowBox[{
     RowBox[{"2", " ", "n"}], "-", "1"}]}], TraditionalForm]], "InlineMath"],
 ". If we integrate a polynomial of that degree, the error estimate becomes \
zero."
}], "Text",
 CellID->215160853],

Cell["This defines a function.", "MathCaption",
 CellID->192354571],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  SuperscriptBox["x", 
   RowBox[{
    RowBox[{"2", "n"}], "-", "1"}]]}]], "Input",
 CellLabel->"In[40]:=",
 CellID->264566629],

Cell[TextData[{
 "These are the integral and error estimates for",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " computed with the rule."
}], "MathCaption",
 CellID->342265653],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "@", 
  RowBox[{"MapThread", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[41]:=",
 CellID->11345905],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.10000000000000000000011471457015603679`18.901873930403898", ",", 
   "0``19.970238497884363"}], "}"}]], "Output",
 ImageSize->{231, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->9175640]
}, Open  ]],

Cell[TextData[{
 "Here is the exact result using ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->664975092],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", "precision"}], 
  "]"}]], "Input",
 CellLabel->"In[42]:=",
 CellID->179491808],

Cell[BoxData["0.1`20."], "Output",
 ImageSize->{158, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->431190145]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"LobattoKronrodRule\"", "Section",
 CellTags->"LobattoKronrodRule",
 CellID->210486704],

Cell["\<\
Lobatto integration rule is a Gauss type rule with preassigned abscissas. It \
uses the end points of the integration interval and optimal sampling points \
inside the interval to form a weighted sum the integrand values over these \
points. The Kronrod extension of a Lobatto rule adds new sampling points in \
between the Lobatto rule points and forms a higher-order rule that reuses the \
Lobatto rule integrand evaluations.\
\>", "Text",
 CellID->221379897],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses the Kronrod extension of the Lobatto rule if the ",
 ButtonBox["Method",
  BaseStyle->"Link",
  ButtonData->"paclet:ref/Method"],
 " option is given the value ",
 Cell[BoxData["\"\<LobattoKronrodRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->244831656],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<LobattoKronrodRule\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[43]:=",
 CellID->211318625],

Cell[BoxData["2.0000000000019873`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[43]=",
 CellID->119532472]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", "5", Cell["\<\
number of Gauss\[Dash]Lobatto points that will be extended with Kronrod \
points\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->384885018],

Cell[TextData[{
 Cell[BoxData["\"\<LobattoKronrodRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->8046501],

Cell[TextData[{
 "A Lobatto rule ",
 Cell[BoxData[
  FormBox[
   RowBox[{"L", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " points for integrand ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " is exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", "n"}], "-", "3"}], TraditionalForm]], "InlineMath"],
 ". (I.e. ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"L", "(", 
     RowBox[{"f", ",", "n"}], ")"}], "=", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " is polynomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", 
    RowBox[{
     RowBox[{"2", "n"}], "-", "3"}]}], TraditionalForm]], "InlineMath"],
 ".) "
}], "Text",
 CellID->62664843],

Cell[TextData[{
 "The Kronrod extension ",
 Cell[BoxData[
  FormBox[
   RowBox[{"LK", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " of a Lobatto rule with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", " "}], TraditionalForm]], "InlineMath"],
 " points ",
 Cell[BoxData[
  FormBox[
   RowBox[{"L", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " adds ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "-", "1"}], TraditionalForm]], "InlineMath"],
 " points to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"L", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " and the extended rule is exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"3", "n"}], "-", "2"}], TraditionalForm]], "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " is even, or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"3", "n"}], "-", "3"}], TraditionalForm]], "InlineMath"],
 " if ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " is odd. The weights associated with a Lobatto rule change in its Kronrod \
extension."
}], "Text",
 CellTags->"LobattoKronrodRuleDegree",
 CellID->701694722],

Cell[TextData[{
 "As with ",
 Cell[BoxData["\"\<GaussKronrodRule\>\""], "InlineFormula"],
 ", the number of Gauss points is specified with the option ",
 Cell[BoxData["\"\<GaussPoints\>\""], "InlineFormula"],
 ". If ",
 Cell[BoxData["\"\<LobattoKronrodRule\>\""], "InlineFormula"],
 " is invoked with ",
 Cell[BoxData[
  RowBox[{"\"\<Points\>\"", "->", 
   StyleBox["n", "TI"]}]], "InlineFormula"],
 ", the total number of rule points will be ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", " ", "n"}], " ", "-", "1"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->491447804],

Cell[TextData[{
 "This example shows the number of sampling point used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with various values of ",
 Cell[BoxData["\"\<LobattoKronrodRule\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<Points\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->233311261],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"k", "=", "0"}], ";", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"x", "^", "10"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<LobattoKronrodRule\>\"", ",", 
          RowBox[{"\"\<Points\>\"", "\[Rule]", "i"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}], ")"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "3", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[44]:=",
 CellID->209034893],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "304", ",", "91", ",", "63", ",", "33", ",", "13", ",", "15", ",", "17", 
   ",", "19", ",", "21", ",", "23", ",", "25", ",", "27", ",", "29", ",", 
   "31", ",", "33", ",", "35", ",", "37", ",", "39"}], "}"}]], "Output",
 ImageSize->{468, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[44]=",
 CellID->147922501]
}, Open  ]],

Cell[TextData[{
 "Since the Lobatto rule is a closed rule, the integrand need to be evaluated \
at the end points of the interval. If there is a singularity at these end \
points, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " will ignore it."
}], "Text",
 CellID->161877433],

Cell[TextData[{
 "For an implementation description of Kronrod extensions of Lobatto rules, \
see [",
 ButtonBox["PiesBrand74",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#186112679"],
 "]. "
}], "Text",
 CellID->102460424],

Cell[CellGroupData[{

Cell["\"LobattoKronrodRule\" sampling points and weights", "Subsection",
 CellID->49911002],

Cell["\<\
The following calculates the Lobatto\[Dash]Kronrod abscissas, weights, and \
error weights for given number of coarse points and precision.\
\>", "MathCaption",
 CellID->82668669],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";", 
  RowBox[{"precision", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`LobattoKronrodRuleData", "[", 
   RowBox[{"n", ",", "precision"}], "]"}]}]}], "Input",
 CellLabel->"In[45]:=",
 CellID->718180355],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0``20.", ",", 
     "0.05479723624366560671485462905167405247`18.763233672105393", ",", 
     "0.1726731646460114328348953191778507959`19.31954773713266", ",", 
     "0.32950886704450351424346591086062216636`19.691474067089153", ",", 
     "0.5`20.", ",", "0.67049113295549648575653408913937783364`20.", ",", 
     "0.8273268353539885671651046808221492041`20.", ",", 
     "0.94520276375633439328514537094832594753`20.", ",", "1.`20."}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
    "0.01532186948853615520282186948841580683`19.517709808790155", ",", 
     "0.08963134977660367799014201009129994927`20.", ",", 
     "0.14198938902406054910680583230290178369`19.286940082580557", ",", 
     "0.167116869908208841791754397360942431`20.", ",", 
     "0.17188104360518153621601897463958901676`19.154964296478276", ",", 
     "0.167116869908208841791754397360942431`20.", ",", 
     "0.14198938902406054910680583230290178369`19.286940082580557", ",", 
     "0.08963134977660367799014201009129994927`20.", ",", 
     "0.01532186948853615520282186948841580683`19.517709808790155"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
     "-", "0.03467813051146384479717813051158419317`19.872453726823327"}], 
     ",", "0.08963134977660367799014201009129994927`20.", ",", 
     RowBox[{
     "-", "0.13023283319816167311541638992262274663`19.838458244365444"}], 
     ",", "0.167116869908208841791754397360942431`20.", ",", 
     RowBox[{
     "-", "0.18367451195037401933953658091596653879`19.57218663127779"}], ",",
      "0.167116869908208841791754397360942431`20.", ",", 
     RowBox[{
     "-", "0.13023283319816167311541638992262274663`19.838458244365444"}], 
     ",", "0.08963134977660367799014201009129994927`20.", ",", 
     RowBox[{
     "-", "0.03467813051146384479717813051158419317`19.872453726823327"}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{546, 158},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]=",
 CellID->106416111]
}, Open  ]],

Cell[TextData[{
 "The calculations below demonstrate the degree of the Lobatto-Kronrod \
integration rule (see ",
 ButtonBox["above",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#701694722"],
 ")."
}], "Text",
 CellID->610949573],

Cell["\<\
This computes the degree of the Lobatto\[Dash]Kronrod integration rule.\
\>", "MathCaption",
 CellID->483828678],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "=", 
  RowBox[{"If", "[", 
   RowBox[{
    RowBox[{"OddQ", "[", "n", "]"}], ",", 
    RowBox[{
     RowBox[{"3", "*", "n"}], "-", "3"}], ",", 
    RowBox[{
     RowBox[{"3", "*", "n"}], "-", "2"}]}], "]"}]}]], "Input",
 CellLabel->"In[47]:=",
 CellID->49463485],

Cell[BoxData["12"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[47]=",
 CellID->442533813]
}, Open  ]],

Cell["This defines a function.", "MathCaption",
 CellID->380705553],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  SuperscriptBox["x", "p"]}]], "Input",
 CellLabel->"In[48]:=",
 CellID->227232811],

Cell[TextData[{
 "The command below implements the integration rule weighted sums for the \
integral estimate, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]], 
    RowBox[{
     SubscriptBox["w", "i"], " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ", and the error estimate, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]], 
    RowBox[{
     SubscriptBox["e", "i"], " ", 
     RowBox[{"f", "(", 
      SubscriptBox["x", "i"], ")"}]}]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     RowBox[{"{", 
      SubscriptBox["x", "i"], " ", "}"}], 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", "n"}], "-", "1"}]], " "}], TraditionalForm]], 
  "InlineMath"],
 " are the abscissas, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     RowBox[{"{", 
      SubscriptBox["w", "i"], " ", "}"}], 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", "n"}], "-", "1"}]], " "}], TraditionalForm]], 
  "InlineMath"],
 " are the weights, and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox[
     RowBox[{"{", 
      SubscriptBox["e", "i"], " ", "}"}], 
     RowBox[{"i", "=", "1"}], 
     RowBox[{
      RowBox[{"2", "n"}], "-", "1"}]], " "}], TraditionalForm]], 
  "InlineMath"],
 "are the error weights. "
}], "Text",
 CellID->59389584],

Cell[TextData[{
 "These are the integral and error estimates for",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " computed with the rule."
}], "MathCaption",
 CellID->430422070],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "@", 
  RowBox[{"MapThread", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[49]:=",
 CellID->445772403],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.0769230769230769212722146989339503492`18.84773046179871", ",", 
   RowBox[{
   "-", "0.00115669452631916177024302662977731328`16.964954659325002"}]}], 
  "}"}]], "Output",
 ImageSize->{331, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[49]=",
 CellID->104276203]
}, Open  ]],

Cell["The integral estimate above coincides with the exact result.", \
"MathCaption",
 CellID->482734384],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", "precision"}], 
  "]"}]], "Input",
 CellLabel->"In[50]:=",
 CellID->572404606],

Cell[BoxData["0.07692307692307692307692307692307692308`20."], "Output",
 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[50]=",
 CellID->261997679]
}, Open  ]],

Cell[TextData[{
 "The error estimate above is not zero since the imbedded Lobatto rule is \
exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", 
    RowBox[{
     RowBox[{"2", " ", "n"}], "-", "3"}]}], TraditionalForm]], "InlineMath"],
 ". If we integrate a polynomial of that degree, the error estimate becomes \
zero."
}], "Text",
 CellID->271407587],

Cell["This defines a function.", "MathCaption",
 CellID->233526],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  SuperscriptBox["x", 
   RowBox[{
    RowBox[{"2", "n"}], "-", "3"}]]}]], "Input",
 CellLabel->"In[51]:=",
 CellID->349925812],

Cell[TextData[{
 "These are the integral and error estimates for",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " computed with the rule."
}], "MathCaption",
 CellID->120734638],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "@", 
  RowBox[{"MapThread", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[52]:=",
 CellID->27301135],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.12499999999999999642175420257116319545`19.018935209396282", ",", 
   RowBox[{"-", "6.8569863295318188262`1.7713104707243892*^-19"}]}], 
  "}"}]], "Output",
 ImageSize->{239, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[52]=",
 CellID->196573900]
}, Open  ]],

Cell[TextData[{
 "The exact result using ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->808380556],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", "precision"}], 
  "]"}]], "Input",
 CellLabel->"In[53]:=",
 CellID->735325913],

Cell[BoxData["0.125`20."], "Output",
 ImageSize->{158, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[53]=",
 CellID->126387969]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"ClenshawCurtisRule\"", "Section",
 CellTags->"ClenshawCurtisRule",
 CellID->486402291],

Cell["\<\
A Clenshaw\[Dash]Curtis rule uses sampling points derived from the Chebyshev \
polynomial approximation of the integrand. \
\>", "Text",
 CellID->340439718],

Cell[TextData[{
 "The Clenshaw-Curtis quadrature for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can specified with the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option value ",
 Cell[BoxData["\"\<ClenshawCurtisRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->303679401],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ClenshawCurtisRule\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[54]:=",
 CellID->32016025],

Cell[BoxData["1.9999999999193905`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[54]=",
 CellID->143959292]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", "5", Cell["\<\
number of coarse Clenshaw\[Dash]Curtis points\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->91740512],

Cell[TextData[{
 Cell[BoxData["\"\<ClenshawCurtisRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->689040563],

Cell[TextData[{
 "Theoretically a Clenshaw\[Dash]Curtis rule with ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " sampling points is exact for polynomials of degree ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " or less. In practice, though, Clenshaw\[Dash]Curtis rules achieve the \
accuracy of the ",
 ButtonBox["Gaussian rules ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#381359375"],
 "[",
 ButtonBox["Evans93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#42112923"],
 "][",
 ButtonBox["OHaraSmith68",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#28772993"],
 "]. The error of the Clenshaw\[Dash]Curtis formula is analyzed in [",
 ButtonBox["OHaraSmith68",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#28772993"],
 "]."
}], "Text",
 CellID->74612686],

Cell[TextData[{
 "The sampling points of the classical Clenshaw\[Dash]Curtis rule are zeroes \
of Chebyshev polynomials. The sampling points of a practical \
Clenshaw\[Dash]Curtis rule are chosen to be Chebyshev polynomial extremum \
points. The classical Clenshaw\[Dash]Curtis rules are not progressive, but \
the practical Clenshaw\[Dash]Curtis rules are [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "][",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "]."
}], "Text",
 CellID->79196589],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{"PCC", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " denote a practical Clenshaw\[Dash]Curtis rule of ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " sampling points for the function ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ". \nThe progressive property means that the sampling points of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"PCC", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " are a subset of the sampling points of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"PCC", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ". Hence",
 " ",
 "the difference ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"|", 
     RowBox[{
      RowBox[{"PCC", "(", 
       RowBox[{"f", ",", 
        RowBox[{
         RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], "-", 
      RowBox[{"PCC", "(", 
       RowBox[{"f", ",", "n"}], ")"}]}]}], "\[RightBracketingBar]"}], 
   TraditionalForm]], "InlineMath"],
 " can be taken to be an error estimate of the integral estimate ",
 Cell[BoxData[
  FormBox[
   RowBox[{"PCC", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", and can be computed without extra integrand evaluations."
}], "Text",
 CellID->8642916],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " option ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   RowBox[{"{", 
    RowBox[{"\"\<ClenshawCurtisRule\>\"", ",", 
     RowBox[{"\"\<Points\>\"", "->", 
      StyleBox["k", "TI"]}]}], "}"}]}]], "InlineFormula"],
 " uses a practical Clenshaw\[Dash]Curtis rule with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", "n"}], "-", "1"}], TraditionalForm]], "InlineMath"],
 " points ",
 Cell[BoxData[
  FormBox[
   RowBox[{"PCC", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->503678959],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", 
    RowBox[{"{", 
     RowBox[{"\"\<ClenshawCurtisRule\>\"", ",", 
      RowBox[{"\"\<Points\>\"", "->", "10"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[55]:=",
 CellID->305820777],

Cell[BoxData["0.6666666666666696`"], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[55]=",
 CellID->74347468]
}, Open  ]],

Cell[TextData[{
 "This example shows the number of sampling point used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with various values of ",
 Cell[BoxData["\"\<ClenshawCurtisRule\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<Points\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->593420888],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"k", "=", "0"}], ";", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"x", "^", "10"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<ClenshawCurtisRule\>\"", ",", 
          RowBox[{"\"\<Points\>\"", "\[Rule]", "i"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}], ")"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "3", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[56]:=",
 CellID->53244339],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "208", ",", "226", ",", "79", ",", "83", ",", "35", ",", "41", ",", "17", 
   ",", "19", ",", "21", ",", "23", ",", "25", ",", "27", ",", "29", ",", 
   "31", ",", "33", ",", "35", ",", "37", ",", "39"}], "}"}]], "Output",
 ImageSize->{475, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[56]=",
 CellID->408857200]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"ClenshawCurtisRule\" sampling points and weights", "Subsection",
 CellID->123359709],

Cell["\<\
Here are the sampling points and the weights of the Clenshaw\[Dash]Curtis \
rule for given coarse number of points and precision.\
\>", "MathCaption",
 CellID->283484080],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "5"}], ";", 
  RowBox[{"precision", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`ClenshawCurtisRuleData", "[", 
   RowBox[{"n", ",", "precision"}], "]"}]}]}], "Input",
 CellLabel->"In[57]:=",
 CellID->136232968],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0``20.30102999566398", ",", 
     "0.03806023374435662193590840530160585675`18.859503860513893", ",", 
     "0.14644660940672623779957781894757547975`19.245013178576308", ",", 
     "0.30865828381745511413577000798480056622`19.53186269682501", ",", 
     "0.5`19.803880122969847", ",", 
     "0.69134171618254488586422999201519943378`20.103926272714936", ",", 
     "0.85355339059327376220042218105242452025`20.382775685337865", ",", 
     "0.96193976625564337806409159469839414325`20.318562528431798", ",", 
     "1.`20.30102999566398"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.00793650793650793650793650793640776369`18.831207979685818", ",", 
     "0.07310932460800907750596741368053460451`19.008146043681208", ",", 
     "0.13968253968253968253968253968235436283`19.34077800674589", ",", 
     "0.18085892936024489074800084028451383371`19.517936243165916", ",", 
     "0.1968253968253968253968253968253968254`19.92462966484806", ",", 
     "0.18085892936024489074800084028451383371`19.67748119478277", ",", 
     "0.13968253968253968253968253968235436283`19.65629482113401", ",", 
     "0.07310932460800907750596741368053460451`19.513743734625503", ",", 
     "0.00793650793650793650793650793640776369`18.831207979685818"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
     "-", "0.02539682539682539682539682539729370473`18.881383978185973"}], 
     ",", "0.07310932460800907750596741368053460451`19.008146043681208", ",", 
     
     RowBox[{
     "-", "0.12698412698412698412698412698410194092`18.876018369477894"}], 
     ",", "0.18085892936024489074800084028451383371`19.517936243165916", ",", 
     
     RowBox[{
     "-", "0.2031746031746031746031746031746031746`19.483443969513942"}], ",",
      "0.18085892936024489074800084028451383371`19.67748119478277", ",", 
     RowBox[{
     "-", "0.12698412698412698412698412698410194092`19.181191885596625"}], 
     ",", "0.07310932460800907750596741368053460451`19.513743734625503", ",", 
     
     RowBox[{
     "-", "0.02539682539682539682539682539729370473`18.881383978185973"}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{532, 158},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[58]=",
 CellID->146115799]
}, Open  ]],

Cell[TextData[{
 "Here is another way to compute the sampling points of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"PCC", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "-", "1"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->128574619],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"nn", "=", 
   RowBox[{
    RowBox[{"2", "n"}], "-", "1"}]}], ";"}], "\n", 
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     FractionBox["1", "2"], 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Cos", "[", 
        RowBox[{
         FractionBox["\[Pi]", 
          RowBox[{"nn", "-", "1"}]], "i"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", 
         RowBox[{"nn", "-", "1"}], ",", "0", ",", 
         RowBox[{"-", "1"}]}], "}"}]}], "]"}]}], "+", 
    FractionBox["1", "2"]}], ",", "precision"}], "]"}]}], "Input",
 CellLabel->"In[59]:=",
 CellID->15132135],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0.03806023374435662193590840530160585659`20.", ",", 
   "0.1464466094067262377995778189475779585`20.", ",", 
   "0.30865828381745511413577000798480056622`20.", ",", "0.5`20.", ",", 
   "0.69134171618254488586422999201519943378`20.", ",", 
   "0.8535533905932737622004221810524220415`20.", ",", 
   "0.96193976625564337806409159469839414341`20.", ",", "1.`20."}], 
  "}"}]], "Output",
 ImageSize->{503, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[60]=",
 CellID->777636271]
}, Open  ]],

Cell["This defines a function.", "MathCaption",
 CellID->216857284],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  SuperscriptBox["x", 
   RowBox[{
    RowBox[{"2", "n"}], "-", "1"}]]}]], "Input",
 CellLabel->"In[61]:=",
 CellID->129525173],

Cell[TextData[{
 "These are the integral and error estimates for",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " computed with the rule."
}], "MathCaption",
 CellID->263836066],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Total", "@", 
  RowBox[{"MapThread", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
       RowBox[{
        RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
    RowBox[{"{", 
     RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[62]:=",
 CellID->548907976],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.09999999999999999999999999999949477581`19.097515784512982", ",", 
   "0.00175781249999999999999999999917722807`17.1618838313196"}], 
  "}"}]], "Output",
 ImageSize->{331, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[62]=",
 CellID->778105720]
}, Open  ]],

Cell[TextData[{
 "The exact value by ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->308066929],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[63]:=",
 CellID->996374849],

Cell[BoxData[
 FractionBox["1", "10"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[63]=",
 CellID->439137555]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"MultiPanelRule\"", "Section",
 CellTags->"MultiPanelRule",
 CellID->106389821],

Cell[TextData[{
 Cell[BoxData["\"\<MultiPanelRule\>\""], "InlineFormula"],
 " combines into one rule the applications of a one-dimensional integration \
rule over two or more adjacent intervals. An application of the original rule \
to any of the adjacent intervals is called a panel."
}], "Text",
 CellID->279020568],

Cell[TextData[{
 "Here is an example of an integration with ",
 Cell[BoxData["\"\<MultiPanelRule\>\""], "InlineFormula"],
 StyleBox[".", "InlineCode"]
}], "MathCaption",
 CellID->122516479],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<MultiPanelRule\>\"", ",", 
      RowBox[{"Method", "->", "\"\<GaussKronrodRule\>\""}], ",", 
      RowBox[{"\"\<Panels\>\"", "->", "3"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[64]:=",
 CellID->203034866],

Cell[BoxData["2.000000000000002`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[64]=",
 CellID->241538296]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], "\"\<NewtonCotesRule\>\"", Cell["\<\
integration rule specification that provides the abscissas, weights, and \
error weights for a single panel\
\>", "TableText"]},
   {"\"\<Panels\>\"", "5", Cell["number of panels", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->284199955],

Cell[TextData[{
 Cell[BoxData["\"\<MultiPanelRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->597817729],

Cell[TextData[{
 "Let the unit interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 " be partitioned into ",
 Cell[BoxData[
  FormBox["k", TraditionalForm]], "InlineMath"],
 " sub-intervals with the points ",
 Cell[BoxData[
  FormBox[
   RowBox[{"0", "=", 
    RowBox[{
     RowBox[{
      SubscriptBox["y", "0"], "<", 
      SubscriptBox["y", "1"], "<", "\[Ellipsis]", "<", 
      SubscriptBox["y", "k"]}], "=", "1"}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->55345536],

Cell["If we have the rule", "Text",
 CellID->584112888],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"R", "(", "f", ")"}], "=", 
    RowBox[{
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "n"], 
      RowBox[{
       SubscriptBox["w", "i"], " ", 
       RowBox[{"f", "(", 
        SubscriptBox["x", "i"], ")"}]}]}], "\[TildeTilde]", 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->17051204],

Cell[TextData[{
 "it can be transformed into a rule for the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     SubscriptBox["y", 
      RowBox[{"j", "-", "1"}]], ",", 
     SubscriptBox["y", "j"]}], "]"}], TraditionalForm]], "InlineMath"],
 ","
}], "Text",
 CellID->361764049],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     FractionBox["1", 
      RowBox[{
       SubscriptBox["y", "j"], "-", 
       SubscriptBox["y", 
        RowBox[{"j", "-", "1"}]]}]], 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "n"], 
      RowBox[{
       SubscriptBox["w", "i"], " ", 
       RowBox[{"f", "(", 
        RowBox[{
         RowBox[{
          SubscriptBox["x", "i"], " ", 
          RowBox[{"(", 
           RowBox[{
            SubscriptBox["y", "j"], "-", 
            SubscriptBox["y", 
             RowBox[{"j", "-", "1"}]]}], ")"}]}], "+", 
         SubscriptBox["y", 
          RowBox[{"j", "-", "1"}]]}], ")"}]}]}]}], "\[TildeTilde]", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      SubscriptBox["y", 
       RowBox[{"j", "-", "1"}]], 
      SubscriptBox["y", "j"]], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], ",", 
   RowBox[{"j", "=", "1"}], ",", "\[Ellipsis]", ",", 
   RowBox[{"k", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->387201495],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", 
     RowBox[{"i", " ", "j"}]], "=", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "i"], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["y", "j"], "-", 
        SubscriptBox["y", 
         RowBox[{"j", "-", "1"}]]}], ")"}]}], "+", 
     SubscriptBox["y", 
      RowBox[{"j", "-", "1"}]]}]}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["y", "j"], "-", 
     SubscriptBox["y", 
      RowBox[{"j", "-", "1"}]]}], "=", 
    RowBox[{"1", "/", "k"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"j", " ", "=", "1"}], ",", "\[Ellipsis]", ",", "k"}], 
   TraditionalForm]], "InlineMath"],
 ". Then the ",
 Cell[BoxData[
  FormBox["k", TraditionalForm]], "InlineMath"],
 "-panel integration rule based on ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " can be written explicitly as"
}], "Text",
 CellID->459098833],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"k", "\[Times]", 
    RowBox[{"R", "(", "f", ")"}]}], "=", 
   RowBox[{
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"j", "=", "1"}], "k"], 
     RowBox[{
      FractionBox["1", 
       RowBox[{
        SubscriptBox["y", "j"], "-", 
        SubscriptBox["y", 
         RowBox[{"j", "-", "1"}]]}]], 
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{"i", "=", "1"}], "n"], 
       RowBox[{
        SubscriptBox["w", "i"], " ", 
        RowBox[{"f", "(", 
         SubscriptBox["x", 
          RowBox[{"i", " ", "j"}]], " ", ")"}]}]}]}]}], "=", 
    UnderoverscriptBox[
     RowBox[{
      FractionBox["1", "k"], "\[Sum]"}], 
     RowBox[{"j", "=", "1"}], "k"], 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"i", "=", "1"}], "n"], 
     RowBox[{
      SubscriptBox["w", "i"], " ", 
      RowBox[{
       RowBox[{"f", "(", 
        SubscriptBox["x", 
         RowBox[{"i", " ", "j"}]], " ", ")"}], "."}]}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->83082974],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " is closed, that is, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " has ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 " as sampling points, then ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", 
     RowBox[{
      RowBox[{"n", " ", "j"}], "-", "1"}]], "=", 
    SubscriptBox["x", 
     RowBox[{"1", " ", "j"}]]}], TraditionalForm]], "InlineMath"],
 ", and the number of sampling points of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "\[Times]", 
    RowBox[{"R", "(", "f", ")"}]}], TraditionalForm]], "InlineMath"],
 " can be reduced to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"k", " ", 
     RowBox[{"(", 
      RowBox[{"n", "-", "1"}], ")"}]}], "+", "1"}], TraditionalForm]], 
  "InlineMath"],
 ". (This is done in the implementation of ",
 Cell[BoxData["\"\<MultiPanelRule\>\""], "InlineFormula"],
 ".)"
}], "Text",
 CellID->36114295],

Cell[TextData[{
 "More about the theory of multi-panel rules, also referred to as compounded \
or composite rules, can be found in [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "] and [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "]."
}], "Text",
 CellID->102589039],

Cell[CellGroupData[{

Cell["\"MultiPanelRule\" sampling points and weights", "Subsection",
 CellID->362787930],

Cell[TextData[{
 "The sampling points and the weights of the ",
 Cell[BoxData["\"\<MultiPanelRule\>\""], "InlineFormula"],
 " can be obtained with this command."
}], "MathCaption",
 CellID->299816157],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"npanels", "=", "3"}], ";"}], "\n", 
 RowBox[{"NIntegrate`MultiPanelRuleData", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\"\<GaussKronrodRule\>\"", ",", 
     RowBox[{"\"\<Points\>\"", "->", "2"}]}], "}"}], ",", "npanels", ",", 
   "MachinePrecision"}], "]"}]}], "Input",
 CellLabel->"In[65]:=",
 CellID->387652139],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.012363316704574762`", ",", "0.07044162180172904`", ",", 
     "0.16666666666666666`", ",", "0.2628917115316043`", ",", 
     "0.32097001662875857`", ",", "0.34569665003790806`", ",", 
     "0.40377495513506234`", ",", "0.5`", ",", "0.5962250448649375`", ",", 
     "0.6543033499620918`", ",", "0.6790299833712414`", ",", 
     "0.7371082884683957`", ",", "0.8333333333333333`", ",", 
     "0.929558378198271`", ",", "0.9876366832954253`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.03299663299663301`", ",", "0.08181818181818185`", ",", 
     "0.1037037037037037`", ",", "0.08181818181818185`", ",", 
     "0.03299663299663301`", ",", "0.03299663299663301`", ",", 
     "0.08181818181818185`", ",", "0.1037037037037037`", ",", 
     "0.08181818181818185`", ",", "0.03299663299663301`", ",", 
     "0.03299663299663301`", ",", "0.08181818181818185`", ",", 
     "0.1037037037037037`", ",", "0.08181818181818185`", ",", 
     "0.03299663299663301`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.03299663299663301`", ",", 
     RowBox[{"-", "0.08484848484848485`"}], ",", "0.1037037037037037`", ",", 
     RowBox[{"-", "0.08484848484848485`"}], ",", "0.03299663299663301`", ",", 
     "0.03299663299663301`", ",", 
     RowBox[{"-", "0.08484848484848485`"}], ",", "0.1037037037037037`", ",", 
     RowBox[{"-", "0.08484848484848485`"}], ",", "0.03299663299663301`", ",", 
     "0.03299663299663301`", ",", 
     RowBox[{"-", "0.08484848484848485`"}], ",", "0.1037037037037037`", ",", 
     RowBox[{"-", "0.08484848484848485`"}], ",", "0.03299663299663301`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{524, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[66]=",
 CellID->266235973]
}, Open  ]],

Cell["\<\
Here are the abscissas and weights of a Gauss\[Dash]Kronrod rule.\
\>", "MathCaption",
 CellID->729040440],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
  RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
   RowBox[{"2", ",", "MachinePrecision"}], "]"}]}]], "Input",
 CellLabel->"In[67]:=",
 CellID->256564232],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0.03708995011372429`", ",", "0.21132486540518713`", ",", "0.5`", ",", 
     "0.7886751345948129`", ",", "0.9629100498862757`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0.09898989898989903`", ",", "0.24545454545454554`", ",", 
     "0.3111111111111111`", ",", "0.24545454545454554`", ",", 
     "0.09898989898989903`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0.09898989898989903`", ",", 
     RowBox[{"-", "0.2545454545454546`"}], ",", "0.3111111111111111`", ",", 
     RowBox[{"-", "0.2545454545454546`"}], ",", "0.09898989898989903`"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{387, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[67]=",
 CellID->130952621]
}, Open  ]],

Cell[TextData[{
 "The multi-panel rule abscissas can be obtained using ",
 Cell[BoxData[
  ButtonBox["Rescale",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Rescale"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->725321628],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Join", "@@", 
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Rescale", "[", 
      RowBox[{"absc", ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "1"}], "}"}], ",", "#"}], "]"}], "&"}], ",", 
    RowBox[{"Partition", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Range", "[", 
        RowBox[{"0", ",", "npanels"}], "]"}], 
       FractionBox["1", "npanels"]}], ",", "2", ",", "1"}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[68]:=",
 CellID->25604383],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.012363316704574762`", ",", "0.07044162180172904`", ",", 
   "0.16666666666666666`", ",", "0.2628917115316043`", ",", 
   "0.32097001662875857`", ",", "0.34569665003790806`", ",", 
   "0.40377495513506234`", ",", "0.5`", ",", "0.5962250448649375`", ",", 
   "0.6543033499620918`", ",", "0.6790299833712414`", ",", 
   "0.7371082884683957`", ",", "0.8333333333333333`", ",", 
   "0.929558378198271`", ",", "0.9876366832954253`"}], "}"}]], "Output",
 ImageSize->{503, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[68]=",
 CellID->214226]
}, Open  ]],

Cell["\<\
This shows how to derive the multi-panel rule weights from the original \
weights.\
\>", "MathCaption",
 CellID->164000480],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  FractionBox["1", "npanels"], 
  RowBox[{"Join", "@@", 
   RowBox[{"Table", "[", 
    RowBox[{"weights", ",", 
     RowBox[{"{", "npanels", "}"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[69]:=",
 CellID->169686619],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.03299663299663301`", ",", "0.08181818181818185`", ",", 
   "0.1037037037037037`", ",", "0.08181818181818185`", ",", 
   "0.03299663299663301`", ",", "0.03299663299663301`", ",", 
   "0.08181818181818185`", ",", "0.1037037037037037`", ",", 
   "0.08181818181818185`", ",", "0.03299663299663301`", ",", 
   "0.03299663299663301`", ",", "0.08181818181818185`", ",", 
   "0.1037037037037037`", ",", "0.08181818181818185`", ",", 
   "0.03299663299663301`"}], "}"}]], "Output",
 ImageSize->{371, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[69]=",
 CellID->274563372]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"CartesianRule\"", "Section",
 CellTags->"CartesianProductRule",
 CellID->650745857],

Cell[TextData[{
 "A ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional Cartesian rule has sampling points that are a Cartesian \
product of the sampling points of ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " one-dimensional rules. The weight associated with a Cartesian rule \
sampling point is the product of the one-dimensional rule weights that \
correspond to its coordinates."
}], "Text",
 CellTags->"Def:ProductRule",
 CellID->140598913],

Cell[TextData[{
 "The Cartesian product integration for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can be specified with the ",
 ButtonBox["Method",
  BaseStyle->"Link",
  ButtonData->"paclet:ref/Method"],
 " option value ",
 Cell[BoxData["\"\<CartesianRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->300155745],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"x", "+", "y", "+", "z"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<CartesianRule\>\""}]}], "]"}]], "Input",
 CellLabel->"In[70]:=",
 CellID->393423482],

Cell[BoxData["0.8628770763645799`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[70]=",
 CellID->2121472]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], "\"\<GaussKronrodRule\>\"", Cell["\<\
a rule or a list of rules with which the Cartesian product rule will be \
formed\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->530404128],

Cell[TextData[{
 Cell[BoxData["\"\<CartesianRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->61382795],

Cell["For example, suppose we have the formulae:", "Text",
 CellID->204881985],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       RowBox[{
        SubscriptBox["f", "1"], "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "\[TildeTilde]", 
    FormBox[
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], 
       SubscriptBox["n", "1"]], 
      RowBox[{
       SubsuperscriptBox["w", "i", "1"], " ", 
       RowBox[{
        SubscriptBox["f", "1"], "(", 
        SubsuperscriptBox["\[Alpha]", "i", "1"], ")"}]}]}],
     TraditionalForm]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->524096338],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       RowBox[{
        SubscriptBox["f", "2"], "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "\[TildeTilde]", 
    FormBox[
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], 
       SubscriptBox["n", "2"]], 
      RowBox[{
       SubsuperscriptBox["w", "i", "2"], " ", 
       RowBox[{
        SubscriptBox["f", "2"], "(", 
        SubsuperscriptBox["\[Alpha]", "i", "2"], ")"}]}]}],
     TraditionalForm]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->180444258],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      RowBox[{
       SubscriptBox["f", "3"], "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}],
    TraditionalForm], "\[TildeTilde]", 
   RowBox[{
    FormBox[
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], 
       SubscriptBox["n", "3"]], 
      RowBox[{
       SubsuperscriptBox["w", "i", "3"], " ", 
       RowBox[{
        SubscriptBox["f", "3"], "(", 
        SubsuperscriptBox["\[Alpha]", "i", "3"], ")"}]}]}],
     TraditionalForm], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->619143603],

Cell[TextData[{
 "that are exact for polynomials of degree ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["d", "1"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["d", "2"], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["d", "3"], TraditionalForm]], "InlineMath"],
 ", respectively. Then it is not difficult to see that the formula with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["n", "1"], "\[Times]", 
    SubscriptBox["n", "2"], "\[Times]", 
    SubscriptBox["n", "3"]}], TraditionalForm]], "InlineMath"],
 " points,"
}], "Text",
 CellID->121385605],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       SubsuperscriptBox["\[Integral]", "0", "1"], 
       RowBox[{
        RowBox[{"f", "(", 
         RowBox[{"x", ",", "y", ",", "z"}], ")"}], 
        RowBox[{"\[DifferentialD]", "x"}], 
        RowBox[{"\[DifferentialD]", "y"}], 
        RowBox[{"\[DifferentialD]", "z"}]}]}]}]}],
    TraditionalForm], "\[TildeTilde]", 
   FormBox[
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"i", "=", "1"}], 
      SubscriptBox["n", "1"]], 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"j", "=", "1"}], 
       SubscriptBox["n", "2"]], 
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{"k", "=", "1"}], 
        SubscriptBox["n", "3"]], 
       RowBox[{
        SubscriptBox["w", 
         RowBox[{"1", " ", "i"}]], " ", 
        SubscriptBox["w", 
         RowBox[{"2", " ", "j"}]], " ", 
        SubscriptBox["w", 
         RowBox[{"3", " ", "k"}]], " ", 
        RowBox[{"f", "(", 
         RowBox[{
          SubscriptBox["\[Alpha]", 
           RowBox[{"1", " ", "i"}]], ",", 
          SubscriptBox["\[Alpha]", 
           RowBox[{"2", " ", "j"}]], ",", 
          SubscriptBox["\[Alpha]", 
           RowBox[{"3", " ", "k"}]]}], ")"}]}]}]}]}],
    TraditionalForm]}], TraditionalForm]], "DisplayMath",
 CellID->555797250],

Cell[TextData[{
 "is exact for polynomials in ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], ",", " ", 
    SubscriptBox["x", "2"], ",", " ", 
    SubscriptBox["x", "3"]}], TraditionalForm]], "InlineMath"],
 " of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"min", "(", 
    RowBox[{
     SubscriptBox["d", "1"], ",", 
     SubscriptBox["d", "2"], ",", 
     SubscriptBox["d", "3"]}], ")"}], TraditionalForm]], "InlineMath"],
 ". Note that the weight associated with the abscissa ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubsuperscriptBox["\[Alpha]", "i", "1"], ",", 
     SubsuperscriptBox["\[Alpha]", "i", "2"], ",", 
     SubsuperscriptBox["\[Alpha]", "i", "3"]}], "}"}], TraditionalForm]], 
  "InlineMath"],
 " is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["w", "i", "1"], 
    SubsuperscriptBox["w", "i", "2"], 
    SubsuperscriptBox["w", "i", "3"]}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->588880804],

Cell[TextData[{
 "The general Cartesian product formula for ",
 Cell[BoxData[
  FormBox["D", TraditionalForm]], "InlineMath"],
 " one-dimensional rules the ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 " of which has ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["n", "i"], TraditionalForm]], "InlineMath"],
 " sampling points ",
 Cell[BoxData[
  FormBox[
   SubsuperscriptBox[
    RowBox[{"{", 
     SubsuperscriptBox["\[Alpha]", "j", "i"], "}"}], 
    RowBox[{"j", "=", "1"}], 
    SubscriptBox["n", "i"]], TraditionalForm]], "InlineMath"],
 " and weights ",
 Cell[BoxData[
  FormBox[
   SubsuperscriptBox[
    RowBox[{"{", 
     SubsuperscriptBox["w", "j", "i"], "}"}], 
    RowBox[{"j", "=", "1"}], 
    SubscriptBox["n", "i"]], TraditionalForm]], "InlineMath"],
 " "
}], "Text",
 CellID->91227601],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["\[Integral]", "V"], 
      FormBox[
       RowBox[{
        RowBox[{"f", "(", 
         RowBox[{
          SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
          SubscriptBox["x", "d"]}], ")"}], 
        RowBox[{"\[DifferentialD]", 
         SubscriptBox["x", "1"]}], "\[Ellipsis]", 
        RowBox[{"\[DifferentialD]", 
         SubscriptBox["x", "D"], " "}]}],
       TraditionalForm]}], "\[TildeTilde]", 
     FormBox[
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{
         SubscriptBox["i", "1"], "=", "1"}], 
        SubscriptBox["n", "1"]], 
       RowBox[{"(", 
        RowBox[{"\[Ellipsis]", 
         RowBox[{
          UnderoverscriptBox["\[Sum]", 
           RowBox[{
            SubscriptBox["i", "D"], "=", "1"}], 
           SubscriptBox["n", "D"]], 
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             UnderoverscriptBox["\[Product]", 
              RowBox[{"k", "=", "1"}], "D"], 
             SubsuperscriptBox["w", 
              SubscriptBox["i", "k"], "k"]}], ")"}], " ", 
           RowBox[{"f", "(", 
            RowBox[{
             SubsuperscriptBox["\[Alpha]", 
              SubscriptBox["i", "1"], "1"], ",", "\[Ellipsis]", ",", 
             SubsuperscriptBox["\[Alpha]", 
              SubscriptBox["i", "D"], "D"]}], ")"}]}]}]}]}]}],
      TraditionalForm]}], ")"}], "."}], TraditionalForm]], "NumberedEquation",\

 CellTags->"CartesianRule1",
 CellID->537222873],

Cell[TextData[{
 "Clearly (",
 
 CounterBox["NumberedEquation", "CartesianRule1"],
 ") can be written as "
}], "Text",
 CellID->260414201],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["\[Integral]", "V"], 
     RowBox[{
      FormBox[
       RowBox[{"f", "(", "x", ")"}],
       TraditionalForm], 
      RowBox[{"\[DifferentialD]", 
       FormBox["x",
        TraditionalForm]}]}]}], "\[TildeTilde]", 
    FormBox[
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{
        SubscriptBox["i", "D"], "=", "1"}], "n"], 
      RowBox[{
       SubscriptBox["w", "i"], " ", 
       RowBox[{"f", "(", 
        SubscriptBox["\[Alpha]", "i"], ")"}]}]}],
     TraditionalForm]}], ","}], TraditionalForm]], "NumberedEquation",
 CellTags->"CartesianRule2",
 CellID->42116383],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", 
    RowBox[{
     UnderoverscriptBox["\[Product]", 
      RowBox[{"k", "=", "1"}], "D"], 
     SubscriptBox["n", "k"], " "}]}], TraditionalForm]], "InlineMath"],
 ", and for each integer ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "\[Element]", 
    RowBox[{"[", 
     RowBox[{"1", ",", "n"}], "]"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Alpha]", "k"], "=", 
    RowBox[{"{", 
     RowBox[{
      SubsuperscriptBox["\[Alpha]", "i", "1"], ",", "\[Ellipsis]", ",", 
      SubsuperscriptBox["\[Alpha]", "i", "k"]}], "}"}]}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["w", "k"], "=", 
    RowBox[{
     UnderoverscriptBox["\[Product]", 
      RowBox[{"k", "=", "1"}], "D"], 
     SubsuperscriptBox["w", 
      SubscriptBox["i", "k"], "k"]}]}], TraditionalForm]], "InlineMath"],
 ",."
}], "Text",
 CellID->25296496],

Cell[TextData[{
 "Here is a visualization of a Cartesian product rule integration. Along the \
",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 "-axis ",
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 " is used; along the ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 "-axis ",
 Cell[BoxData["\"\<GaussKronrodRule\>\""], "InlineFormula"],
 " is used."
}], "MathCaption",
 CellID->39398941],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pnts", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"x", "+", 
        RowBox[{"y", "^", "9"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "->", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
            RowBox[{"\"\<Points\>\"", "->", "4"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"\"\<GaussKronrodRule\>\"", ",", 
            RowBox[{"\"\<Points\>\"", "->", "5"}]}], "}"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"Point", "/@", "pnts"}], ",", 
   RowBox[{"AspectRatio", "->", "1"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}], ",", 
   RowBox[{"AxesOrigin", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "0.02"}], ",", 
      RowBox[{"-", "0.02"}]}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[71]:=",
 CellID->723150784],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWs1OwkAQXrpFuZigCJpoTO/6EJz05EM0xKQcjAZ5Bx6DaLx4J+Hx6s52
SzezW5yWFlpcEurOz3amM/PN7laew3n08hrOp5MweJqF79F08hE8vs0Ei3cY
Yyv4XjMxjsVIfldxLP7q1wdx9YSIA8VBB7hMDge6rJvIkg9oDC0zYegRZMDy
LDKpz+7h6mcSxrrIuwtD4wTdzdTwDc6AoDNE9+WGhpfjM/bo3GrNJ/pRr6+n
aGbfbk3LwGUpnyGPkOdRjZ6mNuwecqLv5WNp3susxXwfu8hulXlPsX6iWfcs
MR1ofF0vnTdEdIJavav00R0/C3PPxPV7c28zy9hnH9E4jph2+va43iG9G0Tf
/kFj/cwOmGE8iiIlSap2sVgItuCqnpDRVwXlmNaeUK1dQRCIQa/XU7oZPUJ0
UTmmTdvr9Xo8Hi+XS6Wb0SNEF5VjWrO9EwJ178tnpZ1Z82SttjRnNXEhJl+u
XlqNclIOa+CqZVx+bCcNvDqpWVvPAJQ5eA8OtoG06eIdBT6fVDXH3L/lPIkW
s3SnJWPgMNYQjB0KTW1BPdkjV9ENqWhC5zbeA+GTBrUPU/qkz7JTe24v1dYE
ii/p2bsCXS02eCWwP+Gx4vmQCE726g6/x7iC5O0bYUjFMHm/RZi1tQ+VssEL
dKJt73PsHcnL9eTQqP2v624De3ejEO/eP7SpopvWn4vs56rv5Tud6puHJYew
BiDsOPr2LvXg6qW2s70fb3qR/fcY9s6ENc3/N/taBPad+WLe7bs2aN5xiZkk
scDFtVBP7RAjV7YeSGvhj2SZv1NinV+p3Oqc\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[72]=",
 CellID->543011096]
}, Open  ]],

Cell[TextData[{
 "Cartesian rules are applicable for relatively low dimensions (",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", " ", "4"}], TraditionalForm]], "InlineMath"],
 "), since for higher dimensions they are subject to \"combinatorial \
explosion\". For example, a ",
 Cell[BoxData[
  FormBox["5", TraditionalForm]], "InlineMath"],
 "-dimensional Cartesian product of ",
 Cell[BoxData[
  FormBox["5", TraditionalForm]], "InlineMath"],
 " identical one-dimensional rules each having ",
 Cell[BoxData[
  FormBox["10", TraditionalForm]], "InlineMath"],
 " sampling points, would have ",
 Cell[BoxData[
  FormBox[
   RowBox[{"10", "^", "5"}], TraditionalForm]], "InlineMath"],
 " sampling points."
}], "Text",
 CellID->9994829],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses Cartesian product rule if the integral is multidimensional and the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option is given a one-dimensional rule or a list of one-dimensional \
rules."
}], "Text",
 CellID->24408754],

Cell[TextData[{
 "Here is an example specifying Cartesian product rule integration with ",
 Cell[BoxData["GaussKronrodRule"], "InlineFormula"],
 StyleBox[".", "InlineCode"]
}], "MathCaption",
 CellID->898295519],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"x", "+", "y"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<GaussKronrodRule\>\""}]}], "]"}]], "Input",
 CellLabel->"In[73]:=",
 CellID->674937957],

Cell[BoxData["1.0000000000000022`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[73]=",
 CellID->65238187]
}, Open  ]],

Cell["\<\
Here is an example specifying Cartesian product rule integration with a list \
of one-dimensional integration rules.\
\>", "MathCaption",
 CellID->40900188],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"x", "+", "y"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", 
    RowBox[{"{", 
     RowBox[{"\"\<LobattoKronrodRule\>\"", ",", "\"\<GaussKronrodRule\>\""}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[74]:=",
 CellID->68962419],

Cell[BoxData["0.999999999999952`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[74]=",
 CellID->401605760]
}, Open  ]],

Cell["\<\
Another example of specifying Cartesian product rule integration with a list \
of one-dimensional integration rules.\
\>", "MathCaption",
 CellID->382128625],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"x", "+", 
    RowBox[{"y", "^", "3"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
        RowBox[{"\"\<Points\>\"", "->", "8"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<GaussKronrodRule\>\"", ",", 
        RowBox[{"\"\<GaussPoints\>\"", "->", "12"}]}], "}"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[75]:=",
 CellID->101435058],

Cell[BoxData["0.7500000000000018`"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[75]=",
 CellID->668996031]
}, Open  ]],

Cell[TextData[{
 "More about Cartesian rules can be found in [",
 ButtonBox["Stroud71",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#44510618"],
 "]."
}], "Text",
 CellID->336461069],

Cell[CellGroupData[{

Cell["\"CartesianRule\" sampling points and weights", "Subsection",
 CellID->643860566],

Cell[TextData[{
 "The sampling points and the weights of the ",
 Cell[BoxData["\"\<CartesianRule\>\""], "InlineFormula"],
 " rule can be obtained with the command ",
 Cell[BoxData["NIntegrate`CartesianRuleData"], "InlineFormula"],
 StyleBox[".", "Text"]
}], "MathCaption",
 CellID->167416858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"crule", "=", 
  RowBox[{"NIntegrate`CartesianRuleData", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\"\<GaussKronrodRule\>\"", ",", 
        RowBox[{"\"\<GaussPoints\>\"", "\[Rule]", "2"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<TrapezoidalRule\>\"", ",", 
        RowBox[{"\"\<Points\>\"", "\[Rule]", "2"}]}], "}"}]}], "}"}], ",", 
    "MachinePrecision"}], "]"}]}]], "Input",
 CellLabel->"In[76]:=",
 CellID->53627707],

Cell[BoxData[
 RowBox[{"NIntegrate`CartesianRule", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
       "0.03708995011372429`", ",", "0.21132486540518713`", ",", "0.5`", ",", 
        "0.7886751345948129`", ",", "0.9629100498862757`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "0.5`", ",", "1.`"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
       "0.09898989898989903`", ",", "0.24545454545454554`", ",", 
        "0.3111111111111111`", ",", "0.24545454545454554`", ",", 
        "0.09898989898989903`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
       "0.16666666666666666`", ",", "0.6666666666666666`", ",", 
        "0.16666666666666666`"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.09898989898989903`", ",", 
        RowBox[{"-", "0.2545454545454546`"}], ",", "0.3111111111111111`", ",",
         
        RowBox[{"-", "0.2545454545454546`"}], ",", "0.09898989898989903`"}], 
       "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.08333333333333334`"}], ",", "0.16666666666666663`", 
        ",", 
        RowBox[{"-", "0.08333333333333334`"}]}], "}"}]}], "}"}]}], "}"}], 
  "]"}]], "Output",
 ImageSize->{426, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[76]=",
 CellID->18274478]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["NIntegrate`CartesianRuleData"], "InlineFormula"],
 " keeps the abscissas and the weights of each rule separated. Otherwise, as \
it can be seen from (",
 
 CounterBox["NumberedEquation", "CartesianRule2"],
 ") the result might be too big for higher dimensions. "
}], "Text",
 CellID->26637091],

Cell[TextData[{
 "The results of ",
 Cell[BoxData["NIntegrate`CartesianRuleData"], "InlineFormula"],
 " can be put into the form of (",
 
 CounterBox["NumberedEquation", "CartesianRule2"],
 ") with this function."
}], "MathCaption",
 CellID->241482162],

Cell[BoxData[
 RowBox[{
  RowBox[{"productFunc", "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"MapAt", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"Flatten", "[", 
            RowBox[{"Outer", "[", 
             RowBox[{"Times", ",", 
              RowBox[{"Sequence", "@@", "#"}]}], "]"}], "]"}], "&"}], ",", 
          "#", ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", "3"}], "}"}]}], "]"}], "&"}], "@", 
       RowBox[{"MapAt", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Flatten", "[", 
           RowBox[{"Outer", "[", 
            RowBox[{"Times", ",", 
             RowBox[{"Sequence", "@@", "#"}]}], "]"}], "]"}], "&"}], ",", "#",
          ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "2"}], "}"}]}], "]"}]}], "&"}], "@", 
     RowBox[{"MapAt", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Flatten", "[", 
         RowBox[{
          RowBox[{"Outer", "[", 
           RowBox[{"List", ",", 
            RowBox[{"Sequence", "@@", "#"}]}], "]"}], ",", 
          RowBox[{
           RowBox[{"Length", "[", "#", "]"}], "-", "1"}]}], "]"}], "&"}], ",",
        "#", ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "]"}]}], "&"}]}], ";"}]], "Input",
 CellLabel->"In[77]:=",
 CellID->661477249],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"productFunc", "[", "crule", "]"}]], "Input",
 CellLabel->"In[78]:=",
 CellID->188665271],

Cell[BoxData[
 RowBox[{"NIntegrate`CartesianRule", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.03708995011372429`", ",", "0.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.03708995011372429`", ",", "0.5`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.03708995011372429`", ",", "1.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.21132486540518713`", ",", "0.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.21132486540518713`", ",", "0.5`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.21132486540518713`", ",", "1.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.5`", ",", "0.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.5`", ",", "0.5`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.5`", ",", "1.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.7886751345948129`", ",", "0.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.7886751345948129`", ",", "0.5`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.7886751345948129`", ",", "1.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.9629100498862757`", ",", "0.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.9629100498862757`", ",", "0.5`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.9629100498862757`", ",", "1.`"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
     "0.016498316498316505`", ",", "0.06599326599326602`", ",", 
      "0.016498316498316505`", ",", "0.04090909090909092`", ",", 
      "0.1636363636363637`", ",", "0.04090909090909092`", ",", 
      "0.05185185185185185`", ",", "0.2074074074074074`", ",", 
      "0.05185185185185185`", ",", "0.04090909090909092`", ",", 
      "0.1636363636363637`", ",", "0.04090909090909092`", ",", 
      "0.016498316498316505`", ",", "0.06599326599326602`", ",", 
      "0.016498316498316505`"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "0.008249158249158254`"}], ",", "0.0164983164983165`", ",",
       
      RowBox[{"-", "0.008249158249158254`"}], ",", "0.021212121212121217`", 
      ",", 
      RowBox[{"-", "0.04242424242424242`"}], ",", "0.021212121212121217`", 
      ",", 
      RowBox[{"-", "0.02592592592592593`"}], ",", "0.05185185185185184`", ",",
       
      RowBox[{"-", "0.02592592592592593`"}], ",", "0.021212121212121217`", 
      ",", 
      RowBox[{"-", "0.04242424242424242`"}], ",", "0.021212121212121217`", 
      ",", 
      RowBox[{"-", "0.008249158249158254`"}], ",", "0.0164983164983165`", ",",
       
      RowBox[{"-", "0.008249158249158254`"}]}], "}"}]}], "}"}], 
  "]"}]], "Output",
 ImageSize->{553, 152},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[78]=",
 CellID->89200106]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"MultiDimensionalRule\"", "Section",
 CellTags->"FullySymmetricRule",
 CellID->89177434],

Cell[TextData[{
 "A fully symmetric integration rule for the cube ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"[", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "2"]}], ",", 
      FractionBox["1", "2"]}], "]"}], "d"], TraditionalForm]], "InlineMath"],
 ",",
 " ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"d", "\[Element]", "\[DoubleStruckCapitalN]"}], ",", " ", 
    RowBox[{"d", ">", "1"}]}], TraditionalForm]], "InlineMath"],
 ", consists of sets of points with the properties: (i) all points in a set \
can be generated by permutations and/or sign changes of the coordinates of \
any fixed point from that set, (ii) all points in a set have the same weight \
associated with them."
}], "Text",
 CellID->232587503],

Cell[TextData[{
 "The fully symmetric multi-dimensional integration (fully symmetric \
cubature) for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can be specified with the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option value ",
 Cell[BoxData["\"\<MultiDimensionalRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->552950062],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"x", "+", "y"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<MultiDimensionalRule\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[79]:=",
 CellID->347703773],

Cell[BoxData["1.1045695042415093`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[79]=",
 CellID->313102913]
}, Open  ]],

Cell[TextData[{
 "A set of points of a fully symmetric integration rule that satisfies the \
properties above is called an orbit. A point of an orbit, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["x", "d"]}], "}"}], TraditionalForm]], "InlineMath"],
 ", ",
 "for the coordinates of which the inequality ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], "\[GreaterEqual]", 
    SubscriptBox["x", "2"], "\[GreaterEqual]", "\[Ellipsis]", 
    "\[GreaterEqual]", " ", 
    SubscriptBox["x", "d"]}], TraditionalForm]], "InlineMath"],
 " holds, is called a generator. (See [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "][",
 ButtonBox["GenzMalik83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#444100062"],
 "].) "
}], "Text",
 CellID->456247437],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Generators\>\"", "5", Cell[
    "number of generators of the fully symmetric rule", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->190249742],

Cell[TextData[{
 Cell[BoxData["\"\<MultiDimensionalRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->417675764],

Cell[TextData[{
 "If an integration rule has ",
 Cell[BoxData[
  FormBox["K", TraditionalForm]], "InlineMath"],
 " orbits denoted ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalOmega]", "1"], ",", " ", 
    SubscriptBox["\[CapitalOmega]", "2"], ",", "\[Ellipsis]", ",", 
    SubscriptBox["\[CapitalOmega]", "K"]}], TraditionalForm]], "InlineMath"],
 ", and the ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " of them, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalOmega]", "i"], TraditionalForm]], "InlineMath"],
 ", has a weight ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["w", "i"], TraditionalForm]], "InlineMath"],
 " associated with it, then the integral estimate is calculated with the \
formula"
}], "Text",
 CellID->414457384],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["\[Integral]", 
     SuperscriptBox[
      RowBox[{"[", 
       RowBox[{
        RowBox[{"-", 
         FractionBox["1", "2"]}], ",", 
        FractionBox["1", "2"]}], "]"}], 
      RowBox[{"d", " "}]]], 
    RowBox[{
     RowBox[{"f", "(", "X", ")"}], 
     RowBox[{"\[DifferentialD]", "X"}]}]}], "\[TildeTilde]", 
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], "K"], 
    RowBox[{
     SubscriptBox["w", "i"], " ", 
     RowBox[{
      UnderscriptBox["\[Sum]", 
       RowBox[{
        SubscriptBox["X", "j"], "\[Element]", 
        SubscriptBox["\[CapitalOmega]", "i"]}]], 
      RowBox[{
       RowBox[{"f", "(", 
        SubscriptBox["X", "j"], ")"}], "."}]}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->101011595],

Cell[TextData[{
 "A null rule of degree ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " will integrate to zero all monomials of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LessEqual]", " ", "m"}], TraditionalForm]], "InlineMath"],
 " and will fail to do so for at least one monomial of degree ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "+", "1"}], TraditionalForm]], "InlineMath"],
 ". Each null rule may be thought of as the difference between a basic \
integration rule an appropriate integration of lower degree."
}], "Text",
 CellID->316383435],

Cell[TextData[{
 StyleBox["Nintegrate'", "InlineCode"],
 "s ",
 Cell[BoxData["\"\<MultiDimensionalRule\>\""], "InlineFormula"],
 " object is basically an interface to three different integration rule \
objects that combine an integration rule and one or several null rules. Their \
number of generators and orders are summarized in the table below. The rule \
objects with 6 and 9 generators use three null rules, each of which is a \
linear combination of two null rules. The null rule linear combinations are \
used in order to avoid ",
 ButtonBox["phase errors",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#599865576"],
 ". See",
 " ",
 "[",
 ButtonBox["BerntEspGenz91",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#133940620"],
 "] for more details how the null rules are used."
}], "Text",
 CellID->351238863],

Cell[TextData[{
 StyleBox["Number of generators and orders of ", "Text"],
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s fully-symmetric rules:"
}], "MathCaption",
 CellID->593623189],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{
      RowBox[{"Number", " ", "of"}], "Generators"}],
     FontFamily->"Times"], 
    StyleBox[
     RowBox[{
      RowBox[{"Integration", " ", "Rule"}], "\[IndentingNewLine]", "Order"}],
     FontFamily->"Times"], 
    StyleBox[
     RowBox[{
     "Order", " ", "of", " ", "Each", " ", "of", " ", "the", " ", "Null", " ",
       "Rules"}],
     FontFamily->"Times"], 
    StyleBox[
     RowBox[{"Described", " ", "in"}],
     FontFamily->"Times"]},
   {"5", "7", "5", 
    RowBox[{"[", 
     ButtonBox["GenzMalik80",
      BaseStyle->"Link",
      ButtonData->"paclet:tutorial/NIntegrateReferences#67891716"], "]"}]},
   {"6", "7", 
    RowBox[{"5", ",", "3", ",", "1"}], 
    RowBox[{
     RowBox[{"[", 
      ButtonBox["GenzMalik83",
       BaseStyle->"Link",
       ButtonData->"paclet:tutorial/NIntegrateReferences#444100062"], "]"}], 
     "[", 
     ButtonBox["BerntEspGenz91",
      BaseStyle->"Link",
      ButtonData->"paclet:tutorial/NIntegrateReferences#133940620"], "]"}]},
   {"9", "9", 
    RowBox[{"7", ",", "5", ",", "3"}], 
    RowBox[{
     RowBox[{"[", 
      ButtonBox["GenzMalik83",
       BaseStyle->"Link",
       ButtonData->"paclet:tutorial/NIntegrateReferences#444100062"], "]"}], 
     "[", 
     ButtonBox["BerntEspGenz91",
      BaseStyle->"Link",
      ButtonData->"paclet:tutorial/NIntegrateReferences#133940620"], "]"}]}
  },
  GridBoxDividers->{
   "Columns" -> {{True}}, "ColumnsIndexed" -> {}, "Rows" -> {{True}}, 
    "RowsIndexed" -> {}}]], "Text",
 CellID->84920445],

Cell[TextData[{
 "These are the number of sampling points used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with its fully symmetric multi-dimensional integration rules for integrals \
of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox["x", "m"], "+", 
        SuperscriptBox["y", "m"]}], ")"}], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"m", "=", "1"}], ",", "\[Ellipsis]", ",", "20"}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->589519280],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Prepend", "[", 
      RowBox[{
       RowBox[{"Table", "[", "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"k", "=", "0"}], ";", 
           RowBox[{"NIntegrate", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"x", "^", "m"}], "+", 
              RowBox[{"y", "^", "m"}]}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"Method", "->", 
              RowBox[{"{", 
               RowBox[{"\"\<MultiDimensionalRule\>\"", ",", 
                RowBox[{"\"\<Generators\>\"", "->", "gen"}]}], "}"}]}], ",", 
             RowBox[{"EvaluationMonitor", ":>", 
              RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}], ")"}], ",", 
         RowBox[{"{", 
          RowBox[{"gen", ",", 
           RowBox[{"{", 
            RowBox[{"5", ",", "6", ",", "9"}], "}"}]}], "}"}]}], "]"}], ",", 
       "m"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"m", ",", "1", ",", "20"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Grid", "[", 
  RowBox[{
   RowBox[{"Join", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
        "\"\<Monomial\>\"", ",", "\"\<Number of generators\>\"", ",", 
         "SpanFromLeft", ",", "SpanFromLeft"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
        "\"\<degree\>\"", ",", "\"\<5\>\"", ",", "\"\<6\>\"", ",", 
         "\"\<9\>\""}], "}"}]}], "}"}], ",", "tbl"}], "]"}], ",", 
   RowBox[{"Dividers", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"False", ",", "True", ",", "False"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"False", ",", "False", ",", "True", ",", "False"}], "}"}]}], 
     "}"}]}], ",", 
   RowBox[{"Alignment", "->", 
    RowBox[{"{", "Center", "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[80]:=",
 CellID->9963956],

Cell[BoxData[
 TagBox[GridBox[{
    {"\<\"Monomial\"\>", "\<\"Number of generators\"\>", "\[SpanFromLeft]", 
     "\[SpanFromLeft]"},
    {"\<\"degree\"\>", "\<\"5\"\>", "\<\"6\"\>", "\<\"9\"\>"},
    {"1", "17", "21", "33"},
    {"2", "17", "426615", "33"},
    {"3", "17", "206157", "33"},
    {"4", "17", "21", "21417"},
    {"5", "17", "21", "39897"},
    {"6", "527", "651", "33"},
    {"7", "1003", "903", "33"},
    {"8", "1241", "1281", "231"},
    {"9", "1445", "1617", "429"},
    {"10", "1717", "1785", "561"},
    {"11", "3145", "3045", "561"},
    {"12", "3689", "3297", "561"},
    {"13", "3825", "3843", "561"},
    {"14", "3825", "3843", "825"},
    {"15", "4063", "3591", "957"},
    {"16", "3893", "2247", "1089"},
    {"17", "3961", "2205", "1155"},
    {"18", "3995", "3297", "1155"},
    {"19", "4403", "3255", "1155"},
    {"20", "6035", "4137", "1155"}
   },
   GridBoxAlignment->{"Columns" -> {{Center}}},
   GridBoxDividers->{
    "Columns" -> {False, True, False}, 
     "Rows" -> {False, False, True, False}}],
  "Grid"]], "Output",
 ImageSize->{209, 329},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[81]=",
 CellID->478127139]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"MultiDimensionalRule\" sampling points and weights", "Subsection",
 CellID->810793694],

Cell["\<\
This subsection gives an example of a calculation of an integral estimate \
with a fully symmetric multi-dimensional rule. \
\>", "Text",
 CellID->308119359],

Cell["Here is parameter for the number of generators.", "MathCaption",
 CellID->85577757],

Cell[BoxData[
 RowBox[{
  RowBox[{"numberOfGenerators", "=", "9"}], ";"}]], "Input",
 CellLabel->"In[82]:=",
 CellID->871976899],

Cell["This function takes a generator point and creates its orbit.", \
"MathCaption",
 CellID->622343018],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"MakeOrbit", "[", "generator_", "]"}], ":=", "\[IndentingNewLine]",
    
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"perms", ",", "signs", ",", "gperms", ",", 
       RowBox[{"len", "=", 
        RowBox[{"Length", "[", "generator", "]"}]}]}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"perms", "=", 
       RowBox[{"Permutations", "[", 
        RowBox[{"Range", "[", "len", "]"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"signs", "=", 
       RowBox[{"Flatten", "[", 
        RowBox[{
         RowBox[{"Outer", "[", 
          RowBox[{"List", ",", 
           RowBox[{"Sequence", "@@", 
            RowBox[{"Table", "[", 
             RowBox[{
              RowBox[{"{", 
               RowBox[{"1", ",", 
                RowBox[{"-", "1"}]}], "}"}], ",", 
              RowBox[{"{", "len", "}"}]}], "]"}]}]}], "]"}], ",", 
         RowBox[{"len", "-", "1"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"gperms", "=", 
       RowBox[{"Map", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Part", "[", 
           RowBox[{"generator", ",", "#1"}], "]"}], "&"}], ",", "perms"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"Union", "[", 
       RowBox[{"Flatten", "[", 
        RowBox[{
         RowBox[{"Outer", "[", 
          RowBox[{"Times", ",", "gperms", ",", "signs", ",", "1"}], "]"}], 
         ",", "1"}], "]"}], "]"}]}]}], "\[IndentingNewLine]", "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[83]:=",
 CellID->194618994],

Cell["The generators and weights for given number of generators.", \
"MathCaption",
 CellID->7824105],

Cell[BoxData[{
 RowBox[{
  RowBox[{"dimension", "=", "2"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"precision", "=", "MachinePrecision"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"rdata", "=", 
   RowBox[{"NIntegrate`MultiDimensionalRuleData", "[", 
    RowBox[{"numberOfGenerators", ",", "precision", ",", "dimension"}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"generators", "=", 
   RowBox[{"rdata", "[", 
    RowBox[{"[", 
     RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"weights", "=", 
   RowBox[{"rdata", "[", 
    RowBox[{"[", 
     RowBox[{"1", ",", "2"}], "]"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[84]:=",
 CellID->332011626],

Cell["This computes the orbit of each generator.", "MathCaption",
 CellID->277639999],

Cell[BoxData[
 RowBox[{
  RowBox[{"orbits", "=", 
   RowBox[{"MakeOrbit", "/@", "generators"}]}], ";"}]], "Input",
 CellLabel->"In[89]:=",
 CellID->172683935],

Cell["This defines a function.", "MathCaption",
 CellID->230195363],

Cell[BoxData[{
 RowBox[{"Clear", "[", "f", "]"}], "\n", 
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{"x_", ",", "y_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"x", "^", "3"}], "*", 
   RowBox[{"y", "^", "3"}]}]}]}], "Input",
 CellLabel->"In[90]:=",
 CellID->222259366],

Cell["This applies the multidimensional rule.", "MathCaption",
 CellID->695694313],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Total", "@", 
   RowBox[{"MapThread", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Total", "[", 
       RowBox[{
        RowBox[{"Map", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"f", "@@", 
            RowBox[{"(", 
             RowBox[{"#1", "+", 
              RowBox[{"1", "/", "2"}]}], ")"}]}], "&"}], ",", "#1"}], "]"}], 
        "*", "#2"}], "]"}], "&"}], ",", 
     RowBox[{"{", 
      RowBox[{"orbits", ",", "weights"}], "}"}]}], "]"}]}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[92]:=",
 CellID->537035195],

Cell["0.06250000000000001", "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[92]//InputForm=",
 CellID->11859086]
}, Open  ]],

Cell["Here is the exact result.", "MathCaption",
 CellID->101276330],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "y"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", "N"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[93]:=",
 CellID->366086263],

Cell["0.0625", "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[93]//InputForm=",
 CellID->727749007]
}, Open  ]],

Cell["This makes graphics primitives for points of the orbits.", "MathCaption",
 CellID->481732818],

Cell[BoxData[
 RowBox[{
  RowBox[{"graphs", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Graphics", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"Red", ",", 
         RowBox[{"AbsolutePointSize", "[", "4", "]"}], ",", 
         RowBox[{"Point", "/@", "#1"}]}], "}"}], ",", 
       RowBox[{"Axes", "->", "False"}], ",", 
       RowBox[{"AspectRatio", "->", "1"}], ",", 
       RowBox[{"Frame", "->", "True"}], ",", 
       RowBox[{"FrameTicks", "\[Rule]", "None"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}], "/", "2"}]}], 
       ",", 
       RowBox[{"ImageSize", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"75", ",", "75"}], "}"}]}]}], "]"}], "&"}], "/@", 
    "orbits"}]}], ";"}]], "Input",
 CellLabel->"In[94]:=",
 CellID->392564065],

Cell["Here is how the different orbits look.", "MathCaption",
 CellID->409154864],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", "graphs", "]"}]], "Input",
 CellLabel->"In[95]:=",
 CellID->223653060],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWktOIzEQdT78FYEESANswmfBJuIAaAQIBGIxh4jQSLAYzQhYsKKXHAJB
jsGSZY7AmhVHYOnxL6G7Um2XnX9wS0l3PVzPr91lV7XDr/rt5e8/9duri3r1
7Lr+7/Lq4qZ6+vdaQKUCY4V7xtjzDyauubhSH3kk4sPYDFcndsfkIS1WarXg
7Ex8FYV5bs6szGdnuSLhzeaRbFLmHx9tULZZluhUGhVN2RqKLqf4C6iCTWXz
tzeeJPzlhW0DRSWuvh4eTPsDranRMJ1kNKVQ4dHSBFC3pip6L1udY7W7K/+2
tCSamrH6/GyDKV0pVDRt6QIoWRe4nw5dJdn49VXwsxOtShvqWNXjrRB5uQFs
sgqgHqrAzjjfFlAQzgSjebw958HM2AP2fDDzHBrfNRSd67YXEK01FO26FxCN
NRQN72UBxOgesMOZxysmB+mZjvVjEPuBjITs5s9Ky1BBap35JWhUeX526Ofz
3QbPEGZ6OpPKGM787M3nyKvefB4Z0s63w+kZcvTndc89p9ue7eA6ALFWNJ7r
AF8E9jqtx7LpUUVgUc//ZtOo31eaQHQaHrxKrqDomt+4NRodK5FRko3rrBIQ
8xUUJSqZMUpA3P/UKrJoVgXwqKAoUUUqGrhZ6A751zqnYekHK99FYHuO/3jM
lbiid8M30St6rLcmq9767p7x/dmjl/j+3BNP124mnYm2B+nN5947tOaAFYBt
AHuFqsORSQe/D+3ahRzGPjSMw8HsOMsb1b8WSCNRgP/Vu/B/0mRB9V7a69HF
Y4lqLx5LVNJ4bHUdjeE8ekSPb+eBvUn6MeSvKEE8yIoSxIOsKH482Ioyes8v
egzXI6/u7EU9Z+/ZXqf2o44j6cmpCPpTxZEU5dQW/ajp7HryKt3u/29g+PMg
eoy2R8zzMc9Hj+gR7mH7zYnGMAl7D1+7KxJMNHfhPxjesJ8=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{382, 158},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[95]=",
 CellID->414096072]
}, Open  ]],

Cell["Here are all rule points together.", "MathCaption",
 CellID->375109605],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"First", "/@", "graphs"}], ",", 
   RowBox[{"Frame", "->", "True"}], ",", 
   RowBox[{"FrameTicks", "\[Rule]", "None"}]}], "]"}]], "Input",
 CellLabel->"In[96]:=",
 CellID->498852237],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWc1OwkAQ3lIVCB4ETQQvIIkRjdG30JPxH87EmMjBaJQQbx77GH0FDiYe
PfIIPIKP4HHc3S7UfoyFkpCgbkOb/b6dndnZnZ1p6HGzfXtz12y3rpuVo8fm
w23r+qlyeP8oKdcRQryquyhkm2QruF0i1VBXV95OAF3Vpy/RlQ8nEFgastTv
08sLvb3tKxF3CEUV8CZglE+F2heM9o8PymRSehbU65lZ7KrHYtBHeu6yT1RY
tsyye4pdAja0PvDY96VZbdvzorZ93wyTP88b2Aa2zLLGdpQNbaeN7c9PqtVU
58qKnGcwahHZCsuWWdbYjbLsfsrr/V0KHgT7o9uqWQW8CVjv58IQj66n7K7D
njdYqSt21+pxsrDKsbKwMrxsA7zjpbqWTcwWIQaeAW+wo9bZmOiwbClOA0RK
h2VjNUD8dFiW11CCqHoGzI+ah12z7IQsUxkvKBrh54AnrXwnxEX7KU1bzYw+
iH2jL2GFOiPuFBhdCarOJUVPhF6rMVVlziJgStbWRsvOOhOlIMZ2AOcArwFe
HSOP+tDe+EynocDMt0Vc1GdZtsCy+QQathWbxowazsyR0Bld/UiG1Ydu4Mcw
4xo/4ERmWbbAsvkEGrQfI5nc913tg+d982FZcFkd4DZxOSLLsgWWzSfQoK3l
4mekRH6uJS5F89QO4BzgNcCrY+RRn472f1Grfhtra+vcsPb92L4fW3Yi1v5T
Yv8pmT37J2qj/R5hv0cAnuR7RHgiXoM+5wuXP43w\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[96]=",
 CellID->618496341]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"MonteCarloRule\"", "Section",
 CellTags->"MonteCarloRule",
 CellID->430697921],

Cell["\<\
A Monte Carlo rule estimates an integral by forming an uniformly weighted sum \
of integrand evaluations over random (quasi-random) sampling points.\
\>", "Text",
 CellID->445079933],

Cell[TextData[{
 "Here is an example of using ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox["1000", TraditionalForm]], "InlineMath"],
 " sampling points."
}], "MathCaption",
 CellID->125960922],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "x"], "-", "1"}], 
    RowBox[{"\[ExponentialE]", "-", "1"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<MonteCarloRule\>\"", ",", 
      RowBox[{"\"\<Points\>\"", "\[Rule]", "1000"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[97]:=",
 CellID->87523634],

Cell[BoxData["0.413393531493587`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[97]=",
 CellID->79918751]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Points\>\"", "100", Cell["number of sampling points", "TableText"]},
   {"\"\<PointGenerator\>\"", 
    ButtonBox["Random",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Random"], Cell[
    "sampling points coordinates generator", "TableText"]},
   {"\"\<AxisSelector\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
selection algorithm of the splitting axis when global adaptive Monte Carlo \
integration is used\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->787116549],

Cell[TextData[{
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->156226993],

Cell[TextData[{
 "In Monte Carlo methods [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "], the ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Integral]", "V"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " is interpreted as the following expected (mean) value"
}], "Text",
 CellID->621655033],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     FormBox[
      RowBox[{
       SubscriptBox["\[Integral]", "V"], 
       RowBox[{
        RowBox[{"f", "(", "x", ")"}], 
        RowBox[{"\[DifferentialD]", "x"}]}]}],
      TraditionalForm],
     TraditionalForm], "=", 
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{"vol", "(", "V", ")"}],
       TraditionalForm], 
      RowBox[{
       SubscriptBox["\[Integral]", 
        FormBox[
         SuperscriptBox["R", "d"],
         TraditionalForm]], 
       RowBox[{
        FormBox[
         FractionBox["1", 
          RowBox[{"vol", "(", "V", ")"}]],
         TraditionalForm], 
        FormBox[
         RowBox[{
          FormBox["Boole",
           TraditionalForm], "(", 
          FormBox[
           RowBox[{"x", "\[Element]", "V"}],
           TraditionalForm], ")"}],
         TraditionalForm], 
        FormBox[
         RowBox[{"f", "(", "x", ")"}],
         TraditionalForm], 
        RowBox[{"\[DifferentialD]", 
         FormBox["x",
          TraditionalForm]}]}]}]}], "=", 
     FormBox[
      RowBox[{
       RowBox[{"vol", "(", "V", ")"}], " ", 
       RowBox[{"E", "(", "f", ")"}]}],
      TraditionalForm]}]}], ","}], TraditionalForm]], "NumberedEquation",
 CellTags->"MonteCarloRule1",
 CellID->242788534],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"E", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " is the mean value of the function ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " interpreted as a random variable, with respect to the uniform distribution \
on ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 ", that is, the distribution with probability density ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"vol", "(", "V", ")"}], 
     RowBox[{"-", "1"}]], 
    RowBox[{"Boole", "(", 
     RowBox[{"x", "\[Element]", "V"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". With ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Boole", "(", 
    RowBox[{"x", "\[Element]", "V"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " is denoted the characteristic function of the region ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 "; with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"vol", "(", "V", ")"}], TraditionalForm]], "InlineMath"],
 " is denoted the volume of ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->202442289],

Cell[TextData[{
 "The crude Monte Carlo estimate of the expected value ",
 Cell[BoxData[
  FormBox[
   RowBox[{"E", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " is obtained by taking ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " independent random vectors ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox[
     StyleBox["x",
      FontSlant->"Plain"], "1"], ",", 
    SubscriptBox[
     StyleBox["x",
      FontSlant->"Plain"], "2"], ",", "\[Ellipsis]", ",", 
    RowBox[{
     SubscriptBox[
      StyleBox["x",
       FontSlant->"Plain"], "n"], "\[Element]", 
     SuperscriptBox["\[DoubleStruckCapitalR]", "d"]}]}], TraditionalForm]], 
  "InlineMath"],
 " with density ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"vol", "(", "V", ")"}], 
     RowBox[{"-", "1"}]], 
    RowBox[{"Boole", "(", 
     RowBox[{"x", "\[Element]", "V"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ", (that is the vectors are uniformly distributed on ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 ") and making the estimate"
}], "Text",
 CellID->41316462],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{"MC", "(", 
     RowBox[{"f", ",", "n"}], ")"}],
    TraditionalForm], "=", 
   RowBox[{
    FractionBox["1", "n"], 
    RowBox[{
     FormBox[
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{"i", "=", "1"}], "n"], 
       RowBox[{"f", "(", 
        SubscriptBox["x", "i"], ")"}]}],
      TraditionalForm], "."}]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"MonteCarloRule2",
 CellID->38840336],

Cell[TextData[{
 StyleBox["Remark",
  FontWeight->"Bold"],
 ": The function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     RowBox[{"vol", "(", "V", ")"}], 
     RowBox[{"-", "1"}]], 
    RowBox[{"Boole", "(", 
     RowBox[{"x", "\[Element]", "V"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 " is a valid probability density function because it is non-negative on \
whole of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", "d"], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["\[Integral]", 
      RowBox[{
       SuperscriptBox["\[DoubleStruckCapitalR]", "d"], " "}]], " ", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"vol", "(", "V", ")"}], 
       RowBox[{"-", "1"}]], 
      RowBox[{"Boole", "(", 
       RowBox[{"x", "\[Element]", "V"}], ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->361534288],

Cell["\<\
According to the strong law of large numbers, the convergence\
\>", "Text",
 CellID->432244465],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     RowBox[{"MC", "(", 
      RowBox[{"f", ",", "n"}], ")"}],
     TraditionalForm], "\[Rule]", 
    FormBox[
     RowBox[{"\[Mu]", "(", "f", ")"}],
     TraditionalForm]}], ",", 
   FormBox[
    RowBox[{"n", "\[Rule]", "\[Infinity]"}],
    TraditionalForm], ","}], TraditionalForm]], "DisplayMath",
 CellID->69332026],

Cell[TextData[{
 "happens with probability ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 ". The strong law of large numbers does not provide information for the \
error ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"MC", "(", 
     RowBox[{"f", ",", "n"}], ")"}], "-", 
    RowBox[{
     SubscriptBox["\[Integral]", "V"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", so a probabilistic estimate is used."
}], "Text",
 CellID->188740124],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox["\[CurlyTheta]", TraditionalForm]], "InlineMath"],
 " be defined as"
}], "Text",
 CellID->45118777],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox["\[CurlyTheta]",
    TraditionalForm], "=", 
   RowBox[{
    FormBox[
     RowBox[{
      SubscriptBox["\[Integral]", "V"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}],
     TraditionalForm], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->30141812],

Cell[TextData[{
 "Formula (",
 
 CounterBox["NumberedEquation", "MonteCarloRule2"],
 ") is an unbiased estimator of ",
 Cell[BoxData[
  FormBox["\[CurlyTheta]", TraditionalForm]], "InlineMath"],
 " (that is, the expectation of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " for various sets of ",
 Cell[BoxData[
  FormBox[
   SubsuperscriptBox[
    RowBox[{"{", 
     SubscriptBox["x", "i"], "}"}], 
    RowBox[{"i", "=", "1"}], "n"], TraditionalForm]], "InlineMath"],
 " is ",
 Cell[BoxData[
  FormBox["\[CurlyTheta]", TraditionalForm]], "InlineMath"],
 ") and its variance is "
}], "Text",
 CellID->1707266],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     FormBox[
      FractionBox["1", "n"],
      TraditionalForm], 
     FormBox[
      RowBox[{
       SubscriptBox["\[Integral]", 
        FormBox["V",
         TraditionalForm]], 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"f", "(", "x", ")"}], "-", "\[CurlyTheta]"}], ")"}], "2"], 
        
        RowBox[{"\[DifferentialD]", "x"}]}]}],
      TraditionalForm]}], "=", 
    FormBox[
     FractionBox[
      RowBox[{"Var", "(", "f", ")"}], "n"],
     TraditionalForm]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->248999231],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Var", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " denotes the variance of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ", The standard error of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " is thus ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    SqrtBox[
     RowBox[{"Var", "(", "f", ")"}]], 
    SqrtBox["n"]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->566360629],

Cell[TextData[{
 "In practice the ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Var", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " is not known, so it is estimated with the formula "
}], "Text",
 CellID->211802050],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    SuperscriptBox["s", "2"],
    TraditionalForm], "=", 
   RowBox[{
    FormBox[
     FractionBox["1", 
      RowBox[{"n", "-", "1"}]],
     TraditionalForm], 
    RowBox[{
     FormBox[
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{"i", "=", "1"}], "n"], 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          RowBox[{"f", "(", 
           SubscriptBox["x", "i"], ")"}], "-", 
          RowBox[{"MC", "(", 
           RowBox[{"f", ",", "n"}], ")"}]}], ")"}], "2"]}],
      TraditionalForm], "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->41092199],

Cell[TextData[{
 "The standard error of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " is then"
}], "Text",
 CellID->642292911],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{"SD", "(", 
     RowBox[{"f", ",", "n"}], ")"}],
    TraditionalForm], "=", 
   RowBox[{
    FormBox[
     FractionBox["s", 
      SqrtBox["n"]],
     TraditionalForm], "=", 
    RowBox[{
     FormBox[
      FractionBox["1", 
       SqrtBox[
        RowBox[{"n", " ", 
         RowBox[{"(", 
          RowBox[{"n", "-", "1"}], ")"}]}]]],
      TraditionalForm], 
     RowBox[{
      FormBox[
       SqrtBox[
        RowBox[{
         UnderoverscriptBox["\[Sum]", 
          RowBox[{"i", "=", "1"}], "n"], 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{
            RowBox[{"f", "(", 
             SubscriptBox["x", "i"], ")"}], "-", 
            RowBox[{"MC", "(", 
             RowBox[{"f", ",", "n"}], ")"}]}], ")"}], "2"]}]],
       TraditionalForm], "."}]}]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"MonteCarloRule3",
 CellID->598600020],

Cell[TextData[{
 "The result of the Monte Carlo estimation can be written as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"MC", "(", 
     RowBox[{"f", ",", "n"}], ")"}], "\[PlusMinus]", 
    RowBox[{"SD", "(", 
     RowBox[{"f", ",", "n"}], ")"}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->763425414],

Cell[TextData[{
 "It can be seen from Equation (",
 
 CounterBox["NumberedEquation", "MonteCarloRule3"],
 ") that the convergence rate of the crude Monte Carlo estimation does not \
depend on the dimension ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " of the integral, and if ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " sampling points are used then the convergence rate is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SqrtBox["n"], " "}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->446452961],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s integration rule ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " calculates the estimates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"SD", "(", 
    RowBox[{"f", ",", "n"}], ")"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->78621487],

Cell[TextData[{
 "The estimates can be improved incrementally. That is, if we have the \
estimates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", 
     SubscriptBox["n", "0"]}], ")"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"SD", "(", 
    RowBox[{"f", ",", 
     SubscriptBox["n", "0"]}], ")"}], TraditionalForm]], "InlineMath"],
 ", and a new additional set of sample function values ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["f", "1"], ",", 
     SubscriptBox["f", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["f", 
      SubscriptBox["n", "1"]]}], "}"}], TraditionalForm]], "InlineMath"],
 ", then using (",
 
 CounterBox["NumberedEquation", "MonteCarloRule2"],
 ") and (",
 
 CounterBox["NumberedEquation", "MonteCarloRule3"],
 ") we have"
}], "Text",
 CellID->352176520],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox[
     RowBox[{"MC", "(", 
      RowBox[{"f", ",", 
       RowBox[{
        SubscriptBox["n", "0"], "+", 
        SubscriptBox["n", "1"]}]}], ")"}],
     TraditionalForm], 
    StyleBox["=", "MR"], 
    RowBox[{
     FormBox[
      FractionBox["1", 
       RowBox[{
        SubscriptBox["n", "0"], "+", 
        SubscriptBox["n", "1"]}]],
      TraditionalForm], 
     RowBox[{
      StyleBox["(", "MR"], 
      FormBox[
       RowBox[{
        RowBox[{
         RowBox[{"MC", "(", 
          RowBox[{"f", ",", 
           SubscriptBox["n", "0"]}], ")"}], " ", 
         SubscriptBox["n", "0"]}], "+", 
        RowBox[{
         UnderoverscriptBox["\[Sum]", 
          RowBox[{"i", "=", "1"}], 
          SubscriptBox["n", "1"]], 
         SubscriptBox["f", "i"]}]}],
       TraditionalForm], ")"}]}]}], ","}], TraditionalForm]], "DisplayMath",
 CellID->239818919],

Cell[BoxData[
 FormBox[
  RowBox[{
   FormBox[
    RowBox[{"SD", "(", 
     RowBox[{"f", ",", 
      RowBox[{
       SubscriptBox["n", "0"], "+", 
       SubscriptBox["n", "1"]}]}], ")"}],
    TraditionalForm], 
   StyleBox["=", "MR"], 
   RowBox[{
    FormBox[
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          SubscriptBox["n", "0"], "+", 
          SubscriptBox["n", "1"]}], ")"}], " ", 
        RowBox[{"(", 
         RowBox[{
          SubscriptBox["n", "0"], "+", 
          SubscriptBox["n", "1"], "-", "1"}], ")"}]}]]],
     TraditionalForm], 
    RowBox[{
     SuperscriptBox[
      RowBox[{
       StyleBox["(", "MR"], 
       FormBox[
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            SubscriptBox["n", "0"], "-", "1"}], ")"}], " ", 
          SubscriptBox["n", "0"], " ", 
          SuperscriptBox[
           RowBox[{"SD", "(", 
            RowBox[{"f", ",", 
             SubscriptBox["n", "0"]}], ")"}], "2"]}], "+", 
         RowBox[{
          UnderoverscriptBox["\[Sum]", 
           RowBox[{"i", "=", "1"}], 
           SubscriptBox["n", "1"]], 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{
             SubscriptBox["f", "i"], "-", 
             RowBox[{"MC", "(", 
              RowBox[{"f", ",", 
               RowBox[{
                SubscriptBox["n", "0"], "+", 
                SubscriptBox["n", "1"]}]}], ")"}]}], ")"}], "2"]}]}],
        TraditionalForm], ")"}], 
      FractionBox["1", "2"]], "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->288417924],

Cell[TextData[{
 "To compute the estimates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      SubscriptBox["n", "0"], "+", 
      SubscriptBox["n", "1"]}]}], ")"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"SD", "(", 
    RowBox[{"f", ",", 
     RowBox[{
      SubscriptBox["n", "0"], "+", 
      SubscriptBox["n", "1"]}]}], ")"}], TraditionalForm]], "InlineMath"],
 ", it is not necessary to know the random points used to compute the \
estimates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"MC", "(", 
    RowBox[{"f", ",", 
     SubscriptBox["n", "0"]}], ")"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"SD", "(", 
    RowBox[{"f", ",", 
     SubscriptBox["n", "0"]}], ")"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->128955055],

Cell[CellGroupData[{

Cell["\"AxisSelector\"", "Subsection",
 CellTags->"MonteCarloRuleAxisChoice",
 CellID->364207621],

Cell[TextData[{
 "When used for multi-dimensional global adaptive integration, ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " chooses the splitting axis of an integration sub-region it is applied to \
in two ways: (i) by random selection or (ii) by minimizing the sum of the \
variances of the integral estimates of each half of the sub-region, if the \
sub-region is divided along that axis. The splitting axis is selected after \
the integral estimation."
}], "Text",
 CellID->376198576],

Cell[TextData[{
 "The random axis selection is done in the following way. ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 " keeps a set of axes for selection, ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 ".",
 " Initially ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 " contains all axes. An element of ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 " is randomly selected. The selected axis is excluded from ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 ". After the next integral estimation, an axis is selected from ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 " and excluded from it, and so forth. If ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 " is empty, it is filled up with all axes."
}], "Text",
 CellID->4046376],

Cell["\<\
The minimization of variance axis selection is done in the following way. \
During the integration over the region, a subset of the sampling points and \
their integrand values are stored. Then for each axis, the variances of the \
two subregions the splitting along this axis will produce are estimated using \
the stored sampling point and corresponding integrand values. The axis for \
which the sum of these variances is minimal is chosen to be the splitting \
axis, since this would mean that if the region is split on that axis, the new \
integration error estimate will be minimal. If it happens that for some axis \
all stored points are clustered in one of the half-regions, then that axis is \
selected for splitting. \
\>", "Text",
 CellID->545958628],

Cell[BoxData[GridBox[{
   {Cell["option value", "TableHeader"], " "},
   {
    ButtonBox["Random",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Random"], Cell[
    "random splitting axis election", "TableText"]},
   {
    RowBox[{"MinVariance", "|", 
     RowBox[{"{", 
      RowBox[{"MinVariance", ",", " ", 
       RowBox[{"\"\<SubsampleFraction\>\"", "->", 
        StyleBox["frac", "TI"]}]}], "}"}]}], Cell["\<\
splitting axis selection that minimizes the sum of variances of the new \
regions\
\>", "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->419965355],

Cell[TextData[{
 Cell[BoxData["\"\<AxisSelector\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->138741263],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<SubsampleFraction\>\"", 
    RowBox[{"1", "/", "10"}], Cell["\<\
fraction of the sampling points used to determine the splitting axis\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->275325445],

Cell[TextData[{
 Cell[BoxData["MinVariance"], "InlineFormula"],
 " options."
}], "Caption",
 CellID->807023952],

Cell[TextData[{
 "This is an example of using ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<AxisSelector\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->19134768],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"t", "=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Exp", "[", 
     RowBox[{"-", 
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", "-", 
           RowBox[{"1", "/", "2"}]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", "-", 
           RowBox[{"1", "/", "2"}]}], ")"}], "2"]}], ")"}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<MonteCarloRule\>\"", ",", 
       RowBox[{"\"\<AxisSelector\>\"", "\[Rule]", "Random"}]}], "}"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[98]:=",
 CellID->506232340],

Cell[BoxData["0.853540246229783`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[98]=",
 CellID->33153669]
}, Open  ]],

Cell[TextData[{
 "In the examples below the two axis selection algorithms are compared. In \
general, the minimization of variance selection uses less number of sampling \
points. Nevertheless, using the minimization of variance axis selection slows \
down the application of ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ". So for integrals for which both axis selection methods would result in \
the same number of sampling points, it is faster to use random axis \
selection. Also, using larger fraction sampling points to determine the \
splitting axis in minimization of variance selection makes the integration \
slower."
}], "Text",
 CellID->80599949],

Cell["Consider the following function.", "MathCaption",
 CellID->172687242],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{
     RowBox[{"x_", "?", "NumberQ"}], ",", 
     RowBox[{"y_", "?", "NumberQ"}]}], "]"}], ":=", 
   RowBox[{
    RowBox[{"Boole", "[", 
     RowBox[{
      RowBox[{"Abs", "[", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"x", "-", 
           FractionBox["1", "2"]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"y", "-", 
           FractionBox["1", "2"]}], ")"}], "2"]}], "]"}], "<", 
      FractionBox["1", "6"]}], "]"}], " ", 
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{"-", "3"}], " ", 
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"]}], ")"}]}]], "+", 
      FractionBox["1", "2"]}], ")"}]}]}], ";"}], "\n", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"PlotRange", "->", "All"}], ",", 
   RowBox[{"PlotPoints", "->", "20"}]}], "]"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->226078],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->185328952]
}, Open  ]],

Cell["\<\
These are the adaptive Monte Carlo integration sampling points for the \
function above with random choice of splitting axis.\
\>", "MathCaption",
 CellID->220307675],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<MonteCarloRule\>\"", ",", 
             RowBox[{"\"\<AxisSelector\>\"", "\[Rule]", "Random"}]}], 
            "}"}]}]}], "}"}]}], ",", 
       RowBox[{"MinRecursion", "\[Rule]", "1"}], ",", 
       RowBox[{"PrecisionGoal", "\[Rule]", "2.8"}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"PointSize", "[", "0.006", "]"}], ",", 
     RowBox[{"Point", "[", "t", "]"}]}], "}"}], ",", 
   RowBox[{"AspectRatio", "->", "1"}], ",", 
   RowBox[{"Frame", "->", "True"}], ",", 
   RowBox[{"PlotLabel", "->", 
    RowBox[{"\"\<Number of sampling points = \>\"", "<>", 
     RowBox[{"ToString", "[", 
      RowBox[{"Length", "[", "t", "]"}], "]"}]}]}]}], "]"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->191743341],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNXdmOHLcVpaenu2ek0WpJtiTLUuxsdhYkQDYECJCHIMuD85IvEIIA9kOQ
IPGv+CPybiCfxwyr6rIuzzmXrGkJcBqY7mF1FXl594Vkf/b6y8//9vfXX37x
19ev/vCv1//8/Iu//vvV7//xr+tLu3dSSl9f//33/XT9f77+b/r7OufrT/9+
5/r97PqrXWntyj3TK5VrD8V3qbTxu319brlDPWm9vjsYsVw6E9/dm/q+Ku/n
6zcpndOV+3RlDzN4QHccYMR7G8bhXvieh9Dvju64+wbzQqj5jtLLuXz2/3E+
x21U6FJyy7wKPxgvfpOzMTj0LEzOvlma3ASf+hsP29vkMdNdh+XzIbRN0yAd
7rvr6jlsz5qHdad/n4GcX0qzFTx8VfsaYYCvoKSfv2Eb6fKm/b3t9luBr8Hg
qZh/Dj0/g/bTLd+fCMkONF7591aeudJz3G3XLu8X1lx6nN+V1X2QTEjtVf6/
hP5Lf1dtfwtkZwLC3QL5cfm8Aea91Cy97+lO5Y808ifndJUWkNZLtQcby/fy
SdJ+yKWYrYcGZrb0fAueunMyTpqZl8+jH6v97myZw6ECZnN/tnzz/cTeEdpY
0L959igQD4QrPWOHYdOv0/jkQxoXeb5GfHi+Tws6fXs/MW993uPHem+w8un1
x21B8ysJIeINZWue8f0teGrk9HzpC6XwfKHZEa7vluZPAFtm3w5Ze9Vkp+Qs
0fLoWRnM/pqN62G9NaDYeUWsF9EzefVTmO9lXjm+lSPAseO9Rmtum6nDKmLP
KHQBs0JK/iKB9jwwg3qNUZ5BOFboADvL/Rdtv/VrlJkHgjO+A/cc0qpPcN5g
ZwQfoNQa7k3r3pN4lvql0UK3lz6Qh1CTT+1zjx9tIX6UWPdNbZIJ5iepqfVz
Qx050hhSbhvM3IWxytj3HBZS9Qc8/z8CvD2E7xXHc8RjtkFzpab1NIiQAYXV
C4AKNcwvoX2RlA+BdNb0wrsubjivEu2Y5j0ImiCnHqD98+u/93Ksb8YcF8Nl
9678/2fAA88bNSTKKHrtmp/RFzo46szvn6VV30S8gVGdx09FY5eiLYRnbkwn
J4ljWLbYdxbM7QUvIt0uJIa36QuPqfkF+rPbQwWOesC70F79TmABaQ+aMq3S
KXWjiIiUVvB2wiIi9NaNQQK/wc3jN2nNVfjrI17CdhRdKs1dxir5ir84COdX
+f8q4BzwTY1GhxUG+6djQ4jOSp8ZRn8IEHy3TrwlKVpb8w041mytjfJAO/aO
JGS/9FHG+gHA8DNoXyShawEqh72NlESO2C1Q/XTpB+3nXUlx5L37Alatu1bI
vFQoPwnHQK/7BbQR8m8zZpim9gjESj1tVORA+Za+/Rja30lOL4+8vK3+hqQ8
5RXEqCinrwUUIy/vNK9PZVkanQSQfABtjKVH+EQ/yvyXvZihm8EwSi6SgzHI
veTyFJDHYo67B09/OJiJaQ2jDXqM40yFyjiNMpDeaqmYK+Kq8j5F+Jd+3ut3
mnooVVUDrtkhfn7VTY7jR1mqXrYo4u6nAuNGFeJLRSFFMfI1UAI+hfbU6wHm
nNkDimauIl8FvWUB0Ed5hfe2NEirR0B3NZYUswso6Rt9CaBZaapMLfpGwFWe
sWbPvrEZz+UcK3f2Y7LlLi/vdks7H5WxRCztJSSqvm30G/GDxEyv3YW7/PuM
aM05SvSjkOKenuwRo7VFyfK8q3h4ygkMvMIstOEN8t0Bb/ksseOR688P0uod
WlZNyWRMfbQo+8S6EDHlvTs3b4E3w/PbyIjj3I4Oh/P7kwHs9pVvo8bAHEvN
OjdZtIvEslPa2Nvl0kQvVFubjhdB2vh5ivxUHy15GnGkci77uGvkWJmlSp/P
3N53WElr5IMZbY45NuWZheUl7Vp6uzMRbR2zFZItFvWyItn3gDkr4wUVw5D/
4nEQzljV543PleekfLtAP66xcn33eexhLIHYEVqvxr4y7/IoxbpUjeJilQvo
r+E/lJtdbmPPbBIvIFaazeNwvoQSXKC/C9cKjUEnJ2cjiOONLrF15ZzFeYp0
YenhfYK70l7n16hKwb7VOV1BrXU7tVpkfmEErXyw0+K+iYhRHYckFn1gNa72
hNAzKA9YBbHq/wM+tdLU5//W71Hv4EoFpsBZZ5ybYa8OBdHnHRoTqbVvYajg
NvV0sv3IKZfyLuWrtRYJcWY5QqocdKs5N8+1Ve7imE7NLPAnpufT9nyb8jK3
xH9c1btFVw4pUd4aJeQK2ujJqli/56mAbkizYZ501W3mKHiXnhvmLqYZCeu2
J56Y+DSSpUYbliveilesHxjIiv2N2TZf8VjvRY5GSz/5oEeGvbx7XVR7FHyE
tDWP9ZI4ZWSbH7r5rjBrPMaeTjejX98R6jsD2MIMis49yypJjZvkt7VHkuWq
+YWFUvyk7TDXDjQn8Tqmd2GMY4osOz9rku1rSn341nkjzi9cH+Wy9oG83Ugt
F401taQoWu37Yv7zu/J7L0J4lMYPIhJvXdPqM5D0CHtKKzdiTDTeFnv9Pgdm
nWCs6fN//r1CTNiN7Fqvhm9ctE+6Zo+xUlCH2KhVtF6VuYqOd9rTKNEcb/I8
+EsBpNMkEbt51tZ0fSKH54BJS3Zz5kfoZyBr6BMQd0RZI6yey4gzc17TVnlc
JfY4Shv9yyjG8/WOzooMliigA9ohlrojPEVVUom3GoOGK6gMQ92aclq5Q65Y
de278knRe8CJLebI05azRKsdxRQq1u7xomx3YhVv3VgGKT6C3qt9b7hF6jBR
T9DehLPtTa+YsX+0NNmvVVXZmuXllX6dVd99Hjh3uK04WyAqqkavnleR9CbL
lrSueAJQqQoARi2WnWmi9yNSw/FvwGH1e4qXb2Xt4VkmHjXgFi2EeUPMahmd
R/nr43LfxQK4z3isvO99PoJYQH1mFCdNon3jbTl/nT9Eva1zNnL9e5CnjOtK
853o/bbz6K9jMexwTD6Nc+VHjt7PU6J1VkhXtLi4zsdheFi5RcqiBD1Orf4E
LWQN6X9p3ad4LNLcwod3NMIcEvDQcvXVRuycZxfHDbDybnIr6ChykNU35itx
l1/pkauI7JdBqJ6Q1hVYpKkg7rBZaB3Ur0ArXYZjoI5CHjZ6KZ5TUXMcuVc9
Lq04yoqas+JAlfkpnqZaC2CUh5w8wPgkrVKAENCeGeOOcP8P6pVldqVxrP34
d+mNqH0mzJW+ukfxaycOjPipwddGvWq7Go7uLvaWjft3Ym5GB4xDcHVVxA21
6neJFO5pMxwNs9YOR95SI4dC+wOHo5U3tM01ylGE7q73M3/GxZE/bljqYVHx
RqQTbe/NUTyjI1Ncualyfy2n7NsrjLms/ANdX/BcId8JW7VKCPIocm2Z5e7d
zXiR9Tf2RDqxqs9iIkw6KwjZs7T6oqaP/Ux1JIVRIno42/IVHyieC6nY4oUq
hjjiEJN1VwzoGlibAn2X9vtd+vYs7fRvYk2g58C6gqw09C3bVCn2ejlX1cZ5
EosXSY90c9lGQZ0LrzTvSpD3PrR3TlEQ8Ys9ez4cza+TqF0sl9CT7NO+p/Nq
Jn1EL+ENKfsAusLuCFbOMrchPNNdgDsPy/O0deaRXHu6Ue7LfUexIWBI+esV
YpkNRwk2TzzCt7/3cWLJ1PP+Vrk9yNrIPGHHg5CrYdG+r55kM5yKxJZe8uon
mYf30mGtMk4gve8p6sjVmpbH6OX4evn8yrOj7IbAS/VlRJwix2h0h1qJGPP5
EH4ZbeoTLsL1Akj3bvYviMOT1iA6BrcMxaMWO6mnARBSWjcJ7VofEesdMd6i
jHCcX2CLGqyn3KU21zjx8tH3247fW42BbYxMtTQ9dCPOr3VQrDbMLy+7Pa+w
id6D3XfsXXjtr+NkZSU9LuMsgIcWsWWx41ji/HoGgAzzSt62QC9xXUFq5YYC
FdZwnTNoHoldzISp3HsB7qGAKcbMJp1a3gK7EectlK9QbWUQfXD0z3GEzsfJ
eSBlJIzqVAiNr2dyZFmRo9WBdXZoh+S6+nidoGeQch13mBvHqFM/VL4HtcDj
dCO5CvW88RHwjn/vftfT5j6GzyQnuxTIJHpSMksVec6COzPHtHEdBLGH+8R8
zdTT2PJ0qIsgO+q4LLIx9mTk/UR6Ooodjc05M6UrBX29hFBN1afAzimfTmlD
4clIyFQ+WVnvXdjDmftEifPxGNs55ANt9R/JZ33eRfOC2rcV3y0ytXmV1KgK
RpjKQcwaZvpxTfU6xs1smYKy7uEVFkf72j7axMyWtiAqZu3FHAbXWH6DagCM
N8gdhqv3G/yMT/BJ6xpCjFiUvh2tz/e4VlkspauiPnp+tlFjOy9VG0gxU1Pl
gKgIa4Xre5w/M5zieq3SDa2o0zwosFTvguhiXKNQXEs5bTfHCkKlDOw4UHzk
uG193nyWUT7D+wHcC/qbFAOKSJn0d4PblzHetuzwBLyJDEuO1taUeNqfQmTg
q5VosGrP3W149bquglh5FnPEGIvaqT+nYQF5EWvcfF6X1Cmplcr5fsXr7EGD
N+AoXiuAQX1aWbkkoMNqRtEhp50Gaiv3pzWJ4U7HI/Wj9DVZnFF2sJ8lEtfM
Vwrr/sKHfnUDbrKo1t+L9VfvC6+YAi5IrGEtqm6sENUPm+wg1xsvgDAN9pRH
4mOUylnk+WzVO6PTg/1e5vnqo+6cqEqtuCSw1By961g41lYYr+iddi9i3Hiu
yFw18E83OTzlzwRVERUtYC4VOR8lAzVgUJWW66pK3zEG1Pk/FGdi1pHstfLr
KTrenuPo5PGIv1Jr468IM2ibYs+Gvd/niXhy6WifWyva0lOtt6izG+5yUXkX
k77D0juuCMZzc+I1Jv6UrF5VH3PrvWyZtSmOy60fDPY4Q7RLVeT+Pq127Omb
TgYALRPWzMnLkPJhkkQnDTPEyz+RX2cU9XszWwoYbibogxpX37NHy2enSY31
wIdt34npXSUAdBlnUTC+naYDkMXRSryWxPe9XcO9cP0B5PpMmG59S9WOR7GV
9py8jR31ij0+F99vwwbuZPH7QeZb9ol5mCrNSck2VHNqp71Zem8HZwnUuH59
+2ZUD/b7gjX1o4eaxkvwnzZD8TKRR58pm1onSHEKYB0rPH04kIfQgrwHvSlL
299FZrERPol77H4F7WjXi8qSGuTlOu21zxS5GUYGVXqb307CYGc2+zMfhGeP
XhTprGKjMUKHc2tT/5RIs6O7zGtYLaKSdOt45OAzu5GMv2tWZ131iqQX/TaZ
F3f9eWKqlvfRmWHxKQ1Kv47OmVcrL/W5eTUj0fXbrHuVl8KZELfk9bwitqD4
qwvxvsz6vKgL2e5nzHzqMac7O5U2jQ//awrc5wgrl8uTuPfz0kCt08F5yD2/
yVlwyLMYb5ufjf7n3IGQ9BzbNOXtX7RPp1n6EaKaZyYZvNpE+Z1NM9tr/h/9
9Ssj2fC8CQ3NraxPOEVq4JpXo7WPAzxcT0Sft4nmUita70Q76TmAr68lXmQN
M51O4Mcd7hv1efumAtm5T62DRDzvRJ+ttY8rDLzr1ywHW0Dy0mBMpYHxNPzb
1KusFOT513d6ecHyr9nhnt+g80V+HXGFS/KTrGIEaypwtqNM9MKDmW04n79k
9kTB1DvPvHT6XuJMfZOZ6J2AK/1h5Dfas5Rbjcr59/IE+oMq19FkomFUvWZA
coODHbUcrEfJq3bDmiec7yg4FPGCOsDnyHDHlDpJUF3TVg9zEJuqcRixC2hI
oglbmHFS5xFGucGa4dLas7uSHvUR+gtcMdG/XrSpzkJ74JTnitYFazuIKToF
I6XOb3GpLGnRW/Y7ZZEmHfvSmzAgLWRPWnGN06dJ6YFIQnxPBSWYvcVqpv7t
p3ivIUKPp4O/sHmLFa3RGjk/DzyjX3lCscUpHZ1WBVR7HXrZW8xffoJwprTB
t4F15suVp2m1APdO5zyMp4P1ZtGZEj+2y/rES8V7lDcw2SNZ40pqYI8a3ap7
yKzjMPeu8lyYhUKrihGwUcX3oalDp9QNev5IUKDnfaN+fOnaBoXymDV0Hy/z
r6ua+ru+oP1jR/f5id4a/VH1NPLI9BkVKIOoO+w81ZEFxBxuJ1rpaACOoi6S
XhEY71xscYAjIt0/HsxL/TZC5MuUF9oK24HVZBlyP3ZA64UwoS2n836S9p/w
uYlG62oNz2WZYy70GNie9qJL4Mz4XI6AG9YxzTu0+DnKXJMv6J5BLnji++pI
agH7KOHfpieUJIFcLv+YVuV8kK58i3ozWZJRHRv3eek1KuhN3HL4WOHfiA+a
z9QI9m2o3UiWM1D8P1G5y9/lLvRKlPeE1lp5zzpjBb+eZ49t0NeWi4zW4sQ7
HhFPuO/IcKdkTu31O+Kz5VPs/ME4X/+qsjpJv1cvyDkT9LbezLJFv01CM2Pb
4ZO9PZNM/Rt+fftS63ebdan0CGR92d/166S0LkSraeTbGQ/189kWHY9ynmyL
TaZU1cxbJvaT0O59Am3MoBiVlZ+m6hs6nhjVBHoZLrKmG05G96tNfL0dpbOn
qTEC/Qj6VDrLvkdKa265iY9FGt1DnnO4M1OvXdJXd8tFO5cAsS9ztsht6Lc1
FtLXNXs5rw5/BLsheW2vXLPSO/+vu8IGbWB5oR/6BNp2AgjaToVDtZJs6iP3
zu6aMHLl+UpMbnpXJx/5qLk+3XAT0q6JppOOWjB/bKxmK9V68UnlTbBLKq+L
Y2D2b5tlBI/C8YOCtXKa0EFWrVDeEvgSefWSKcuW2xV59jT6HAYl8sXBQRGt
hpArAzqYFOsJgQ+2n7C/c5+2w0X7v8hbClKF59K7OhWyehBUQawjhbn/OsUg
Qp5vUTtztqx52WeKL9y3B8ehdahQ53KmrslnSd/Z/wKvz4bJ3+kQ/q3VxjbV
c06w+lM/AS9HPgiswYdpR9kCmi1phrtp1QJVfru7CSyerVqcqLMxnicc4Roj
bCvc7BLHGvrXKXB1cs46tzTN4FaE6Vx1wa6ChD6HrIAF/gxCdCVhH/+CsWoH
tCSZQQpXi0fZYs61eb45dzxaH6vv98NnA4h0voSy9gqzaj+ZWpcx2MEneA7v
UXVKtDeXOfDYM+uGfrYerSvl15LOT+gV3tKrH81VVyWGu2pnuZeSMaipeF1T
68vdSFXJ1VnAKQwBVttUj+WFkef0THDCFUqYecgY0644kDYjoI3x0qn41npC
RzvkQUS/X8+Vpq6vgfNrdl7Ak6piTxFYanNJBm9pvp99xfIUnnwOzzyD9tMt
3wt9dAosaGF7FdYt7ZGX80233wp8J+P+q9SrkP5nuvQ1vX+WZ457kNvfKPrT
9O2fYeSp/2G14I9v/KxZFJWt4P2U7NH77NEurZoT27u4P5GBv6Q7rS/OLdlo
w9Nur1+P3CxL2+vgnfjUeAksHWu76JdqPQT05HidazYOFv0HkEgvbuu96FdH
WcJNfXVtgnFJr3Kr8aja71x/vlM+V0rqDP0NcUDRkJjFmIJ1RY4+ZW0A003h
ztmdOxH8FvdJfeY132/7O0d02UCnwW88n0qpMDLbJG99j+3NaaTG3b1teqXZ
AgW/TRJplA4Eb0I9yLPVB2ROTsxEr6a9ka6EHuNdc9twK/ZvATW3a8+t2qHc
ckzKL9S2ZMRTgqIDOgi724V7vhUrT1hP8H4E7s/p40fluHVGHCM8X2VTtRIc
t8WLh/gB4M2vUECs31zGVLyDK+KeQPsltL8H7ceC0nWcng4a6/NORkFC3XDx
hwO4n+Q4Tjwdbj97ixRL++nSfhGMinHhq3Y+ueUS7XVy9JDe+R8vo6T4\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->443928057]
}, Open  ]],

Cell[TextData[{
 "These are the sampling points with choice of splitting axes that minimizes \
the variance. Compared to the previous Monte Carlo integration, the sampling \
points of this one are more concentrated around the circle ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "-", 
        RowBox[{"1", "/", "2"}]}], ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"y", "-", 
        RowBox[{"1", "/", "2"}]}], ")"}], "2"]}], "=", 
    RowBox[{"1", "/", "6"}]}], TraditionalForm]], "InlineMath"],
 ", and their number is nearly twice as small."
}], "MathCaption",
 CellID->196890053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{"x", ",", "y"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<MonteCarloRule\>\"", ",", 
             RowBox[{"\"\<AxisSelector\>\"", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<MinVariance\>\"", ",", 
                RowBox[{"\"\<SubsampleFraction\>\"", "\[Rule]", 
                 RowBox[{"1", "/", "3"}]}]}], "}"}]}]}], "}"}]}]}], "}"}]}], 
       ",", 
       RowBox[{"MinRecursion", "\[Rule]", "1"}], ",", 
       RowBox[{"PrecisionGoal", "\[Rule]", "2.8"}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";", 
  RowBox[{"Graphics", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "0.006", "]"}], ",", 
      RowBox[{"Point", "[", "t", "]"}]}], "}"}], ",", 
    RowBox[{"AspectRatio", "->", "1"}], ",", 
    RowBox[{"Frame", "->", "True"}], ",", 
    RowBox[{"PlotLabel", "->", 
     RowBox[{"\"\<Number of sampling points = \>\"", "<>", 
      RowBox[{"ToString", "[", 
       RowBox[{"Length", "[", "t", "]"}], "]"}]}]}]}], "]"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->367184567],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->373733288]
}, Open  ]],

Cell["\<\
Here is an adaptive Monte Carlo integration that uses random axis selection.\
\>", "MathCaption",
 CellID->626558770],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}]]], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<MonteCarloRule\>\"", ",", 
            RowBox[{"\"\<Points\>\"", "->", "500"}], ",", 
            RowBox[{"\"\<AxisSelector\>\"", "\[Rule]", "Random"}]}], 
           "}"}]}]}], "}"}]}]}], "]"}], ",", 
    RowBox[{"{", "100", "}"}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[104]:=",
 CellID->167416277],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.2103600000000005`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[104]=",
 CellID->60397028]
}, Open  ]],

Cell["\<\
Here is an adaptive Monte Carlo integration for the integral above that uses \
the minimization of variance axis selection and is slower than using random \
axis selection. \
\>", "MathCaption",
 CellID->124917769],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}]]], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<MonteCarloRule\>\"", ",", 
            RowBox[{"\"\<Points\>\"", "->", "500"}], ",", 
            RowBox[{"\"\<AxisSelector\>\"", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{"\"\<MinVariance\>\"", ",", 
               RowBox[{"\"\<SubsampleFraction\>\"", "\[Rule]", "0.3"}]}], 
              "}"}]}]}], "}"}]}]}], "}"}]}]}], "]"}], ",", 
    RowBox[{"{", "100", "}"}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[105]:=",
 CellID->430097928],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4.206359999999998`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[105]=",
 CellID->196842352]
}, Open  ]],

Cell["\<\
Using larger fraction of stored points for the minimization of variance axis \
choice slows down the integration. \
\>", "MathCaption",
 CellID->82415329],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}]]], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", 
        RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", 
        RowBox[{"-", "1"}], ",", "2"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<MonteCarloRule\>\"", ",", 
            RowBox[{"\"\<Points\>\"", "->", "500"}], ",", 
            RowBox[{"\"\<AxisSelector\>\"", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{"\"\<MinVariance\>\"", ",", 
               RowBox[{"\"\<SubsampleFraction\>\"", "\[Rule]", "0.6"}]}], 
              "}"}]}]}], "}"}]}]}], "}"}]}]}], "]"}], ",", 
    RowBox[{"{", "100", "}"}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[106]:=",
 CellID->108345155],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5.086227999999999`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[106]=",
 CellID->135829003]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Comparisons of the Rules", "Section",
 CellID->233261223],

Cell[TextData[{
 "All integration rules, except ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ", are to be used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 ButtonBox["adaptive strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#155948475"],
 ". Changing the type and the number of points of the integration rule \
component for an integration strategy will make a different integration \
algorithm. In general these different integration algorithms will perform \
differently for different integrals. Naturally the following questions \
arise."
}], "Text",
 CellID->265640948],

Cell["\<\
Is there a type of rule that is better than other types for any integral or \
for integrals of a certain type?\
\>", "ItemizedText",
 CellID->351937731],

Cell["\<\
Given an integration strategy, what rules perform better with it? For what \
integrals?\
\>", "ItemizedText",
 CellID->858066548],

Cell[CellGroupData[{

Cell["\<\
Given an integral, an integration strategy, and an integration rule, what \
number of points in the rule will minimize the total number of sampling \
points used to reach an integral estimate that satisfies the precision goal?\
\>", "ItemizedText",
 CellID->169742927],

Cell["\<\
For a given integral and integration strategy the integration rule which \
achieves a result that satisfies the precision goal with the smallest number \
of sampling points is called the best integration rule. There are several \
factors that determine the best integration rule.\
\>", "Text",
 CellTags->"Def:NullRule",
 CellID->596623745]
}, Open  ]],

Cell["\<\
In general the higher the degree of the rule the faster the integration will \
be for smooth integrands and for higher-precision goals. On the other hand, \
the rule degree might be too high for the integrand and hence too many \
sampling points might be used when the adaptive strategies work around, for \
example, the integrand's discontinuities.\
\>", "ItemizedText",
 CounterAssignments->{"ItemizedText", 0},
 CellTags->"Def:NullRule",
 CellID->501247729],

Cell[TextData[{
 "The error estimation functional of a rule influences significantly the \
total amount of work by the integration strategy. Rules with smaller number \
of points might lead (i) to wrong result because of underestimation of the \
integral, or (ii) to applying too many sampling points because of \
overestimation of the integrand. (See ",
 ButtonBox["Examples of Pathological Behavior",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#8354729"],
 ".) Further, the error estimation functional might be computed with one or \
several imbedded ",
 ButtonBox["null rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#84757638"],
 ". In general, the larger the number of the null rules the better the error \
estimation\[LongDash]less ",
 ButtonBox["phase errors",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#599865576"],
 " are expected. The number of the null rules and the weights assigned to \
them in the sum that computes the error estimate determines the sets of \
pathological integrals and integrals hard to compute for that rule. (Some of \
",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s multi-dimensional rules use several imbedded null rules to compute the \
error estimate. All of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s one-dimensional integration rules use only one null rule.) "
}], "ItemizedText",
 CellTags->"Def:NullRule",
 CellID->64836516],

Cell[TextData[{
 "Local adaptive strategies are more effective with closed rules that have \
their sampling points more uniformly distributed (for example, ",
 Cell[BoxData["\"\<ClenshawCurtisRule\>\""], "InlineFormula"],
 ") than with open rules (for example, ",
 Cell[BoxData["GaussKronrodRule"], "InlineFormula"],
 ") and closed rules that have sampling points distributed in a non-uniform \
way (for example, ",
 Cell[BoxData["\"\<LobattoKronrodRule\>\""], "InlineFormula"],
 ")."
}], "ItemizedText",
 CellTags->"Def:NullRule",
 CellID->76485970],

Cell[TextData[{
 "The percent of reused points by the strategy might greatly determine what \
is the best rule. For one-dimensional integrals, ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " reuses all points of the closed rules. ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " throws away almost all points of the regions that need improvement of \
their error estimate."
}], "ItemizedText",
 CellTags->"Def:NullRule",
 CellID->372932111],

Cell[CellGroupData[{

Cell["Number of points in a rule", "Subsection",
 CellID->6741536],

Cell["\<\
This subsection demonstrates with examples that the higher the degree of the \
rule the faster the integration will be for smooth integrands and for \
higher-precision goals. It also shows examples in which the degree of the \
rule is too high for the integrand and hence too many sampling points are \
used when the adaptive strategies work around the integrand's \
discontinuities. All examples use Gaussian rules with Berntsen\[Dash]Espelid \
error estimate.\
\>", "Text",
 CellID->263334898],

Cell[TextData[{
 "Here is the error of a Gaussian rule in the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"a", ",", "b"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->33375855],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"E", "[", 
     RowBox[{"G", "(", 
      RowBox[{"f", ",", "n"}], ")"}], "]"}], "=", 
    RowBox[{
     FractionBox[
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"b", "-", "a"}], ")"}], 
        RowBox[{
         RowBox[{"2", "n"}], "+", "1"}]], 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"n", "!"}], ")"}], "4"]}], 
      SuperscriptBox[
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"2", "n"}], "+", "1"}], ")"}], "[", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{"2", "n"}], ")"}], "!"}], "]"}], "3"]], 
     RowBox[{
      SuperscriptBox["f", 
       RowBox[{"(", 
        RowBox[{"2", "n"}], ")"}]], "(", "\[Xi]", ")"}]}]}], ",", " ", 
   RowBox[{"a", "<", "\[Xi]", "<", 
    RowBox[{"b", "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->489783266],

Cell[TextData[{
 "(See [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "].)"
}], "Text",
 CellID->5802283],

Cell[TextData[{
 "Here is function that calculates the error of a rule for the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " using the exact value by ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->97608709],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RuleError", "[", 
    RowBox[{"f_", ",", "rule_String", ",", "prec_", ",", 
     RowBox[{"pnts_", "?", "NumberQ"}]}], "]"}], ":=", "\[IndentingNewLine]", 
   
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
       RowBox[{
        RowBox[{"ToExpression", "[", 
         RowBox[{"\"\<NIntegrate`\>\"", "<>", "rule", "<>", "\"\<Data\>\""}], 
         "]"}], "[", 
        RowBox[{"pnts", ",", "prec"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"Abs", "[", 
       RowBox[{
        RowBox[{"Total", "[", 
         RowBox[{"MapThread", "[", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{"f", "[", "#1", "]"}], "#2"}], "&"}], ",", 
           RowBox[{"{", 
            RowBox[{"absc", ",", "weights"}], "}"}]}], "]"}], "]"}], "-", 
        RowBox[{"Integrate", "[", 
         RowBox[{
          RowBox[{"f", "[", "x", "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[107]:=",
 CellID->51461333],

Cell["This defines a list of functions.", "MathCaption",
 CellID->508919412],

Cell[BoxData[
 RowBox[{
  RowBox[{"funcs", "=", 
   RowBox[{"{", 
    RowBox[{
     SqrtBox["x"], ",", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", "-", 
       FractionBox["1", "\[ExponentialE]"]}], "]"}], ",", 
     RowBox[{"\[Piecewise]", GridBox[{
        {"2", 
         RowBox[{"x", "\[LessEqual]", 
          FractionBox["1", "\[ExponentialE]"]}]},
        {"3", 
         RowBox[{"x", ">", 
          FractionBox["1", "\[ExponentialE]"]}]}
       },
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}},
       GridBoxItemSize->{
        "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
         "RowsIndexed" -> {}},
       GridBoxSpacings->{"Columns" -> {
           Offset[0.28], {
            Offset[0.84]}, 
           Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
           Offset[0.2], {
            Offset[0.4]}, 
           Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
     FractionBox["1", 
      RowBox[{
       RowBox[{
        SuperscriptBox["10", "4"], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}], "+", 
       "1"}]]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[108]:=",
 CellID->447064306],

Cell[TextData[{
 "Here are plots of the functions in the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->70008065],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Plot", "[", 
      RowBox[{"#", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"PlotRange", "->", "All"}], ",", 
       RowBox[{"Frame", "->", "True"}], ",", 
       RowBox[{"FrameTicks", "->", 
        RowBox[{"{", 
         RowBox[{"None", ",", "Automatic"}], "}"}]}], ",", 
       RowBox[{"ImageSize", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"120", ",", "120"}], "}"}]}]}], "]"}], "&"}], "/@", 
    "funcs"}], ",", "\"\<  \>\""}], "]"}]], "Input",
 CellLabel->"In[109]:=",
 CellID->95153325],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{498, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[109]=",
 CellID->701900742]
}, Open  ]],

Cell[TextData[{
 "Here is the computation of the errors of ",
 Cell[BoxData["\"\<GaussBerntsenEspelidRule\>\""], "InlineFormula"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SqrtBox["x"], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"\[LeftBracketingBar]", 
      RowBox[{"x", "-", 
       FractionBox["1", "2"]}], "\[RightBracketingBar]"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"\[Piecewise]", GridBox[{
        {"2", 
         RowBox[{"x", "\[LessEqual]", 
          FractionBox["1", "2"]}]},
        {"3", 
         RowBox[{"x", ">", 
          FractionBox["1", "2"]}]}
       },
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}},
       GridBoxItemSize->{
        "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
         "RowsIndexed" -> {}},
       GridBoxSpacings->{"Columns" -> {
           Offset[0.28], {
            Offset[0.84]}, 
           Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
           Offset[0.2], {
            Offset[0.4]}, 
           Offset[0.2]}, "RowsIndexed" -> {}}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     FractionBox["1", 
      RowBox[{
       RowBox[{
        SuperscriptBox["10", "4"], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           FractionBox["1", "3"], "-", "x"}], ")"}], "2"]}], "+", "1"}]], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " for a range of points."
}], "MathCaption",
 CellID->12446453],

Cell[BoxData[
 RowBox[{
  RowBox[{"errors", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"pnts", ",", 
         RowBox[{"RuleError", "[", 
          RowBox[{
          "#", ",", "\"\<GaussBerntsenEspelidRule\>\"", ",", "30", ",", 
           "pnts"}], "]"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"pnts", ",", "4", ",", "100", ",", "1"}], "}"}]}], "]"}], 
     "&"}], "/@", 
    RowBox[{"Function", "/@", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"Function", "[", 
        RowBox[{
         RowBox[{"{", "f", "}"}], ",", 
         RowBox[{"f", "/.", 
          RowBox[{"x", "->", "#"}]}]}], "]"}], "/@", "funcs"}], ")"}]}]}]}], 
  ";"}]], "Input",
 CellLabel->"In[110]:=",
 CellID->318514032],

Cell["\<\
Here are plots of how the logarithm of the error decreases for each of the \
functions. It can be seen that the integral estimates of discontinuous \
functions and functions with discontinuous derivatives improve slowly when \
the number of points is increased.\
\>", "MathCaption",
 CellID->564454320],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"ListLinePlot", "[", 
    RowBox[{
     RowBox[{"MapThread", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Tooltip", "[", 
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"#", "[", 
               RowBox[{"[", "1", "]"}], "]"}], ",", 
              RowBox[{"Log", "[", 
               RowBox[{"10", ",", 
                RowBox[{"#", "[", 
                 RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "&"}], "/@",
            "#1"}], ",", "#2"}], "]"}], "&"}], ",", 
       RowBox[{"{", 
        RowBox[{"errors", ",", "funcs"}], "}"}]}], "]"}], ",", 
     RowBox[{"PlotRange", "->", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "100"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"-", "9"}]}], "}"}]}], "}"}]}], ",", 
     RowBox[{"AxesOrigin", "->", 
      RowBox[{"{", 
       RowBox[{"0", ",", "0"}], "}"}]}], ",", 
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", "300", "}"}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xc", "=", "110"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xcSq", "=", "106"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"legend", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "2"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "2", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "4"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "3", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "6"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "4", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "8"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"legendSq", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "2"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "4"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "6"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "8"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"legendSq", "=", 
   RowBox[{"MapThread", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Append", "[", 
       RowBox[{"#1", ",", 
        RowBox[{"Background", "->", "#2"}]}], "]"}], "&"}], ",", 
     RowBox[{"{", 
      RowBox[{"legendSq", ",", 
       RowBox[{"Cases", "[", 
        RowBox[{"gr", ",", 
         RowBox[{"Hue", "[", "s__", "]"}], ",", "\[Infinity]"}], "]"}]}], 
      "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Row", "[", 
  RowBox[{"{", 
   RowBox[{"gr", ",", "\"\< \>\"", ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", " ", 
       RowBox[{"legend", ",", "legendSq"}], "}"}], ",", 
      RowBox[{"ImageSize", "->", 
       RowBox[{"{", 
        RowBox[{"200", ",", "200"}], "}"}]}], ",", 
      RowBox[{"AspectRatio", "->", "5"}]}], "]"}]}], "}"}], "]"}]}], "Input",
 CellLabel->"In[111]:=",
 CellID->89675936],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   GraphicsBox[{{{}, {}, 
      {Hue[0.67, 0.6, 0.6], LineBox[CompressedData["
1:eJw1ywlQ1GUcxvFVKVdRXFpALkUpCgE52pRL4FkOWViuZffPLoKyUjAoIcKQ
YOKwg4GYOkRgyg2ZxiAOJoiQATuERJENCEYYrkEexc3YGJDZMb6/d+addz7z
vN/N8amRCct5PN7e/+7/74sjwKQ6JVGnXKd9YSO07NKzabwkYDbF1Znhkxa5
hsyWaMot7G33eIV5E+oTb7ncHyVbo9M+xSAxRcj8Gv4u66jomSW/jrQ07pvj
iUbMtviwddG5aJBsh/t1A8UxbsbMDjjS0aIv+ITsiJkL3esqp8nOaDw6Otvt
Y8LsgjhdurntafKb0D/nWhjyI1mEgKIx9XmL9cxvoXny55FTu8nbUPP8+sWY
CvJ2rBULgpU/kV2xwJmu6jA0ZXZDv+X+9B4J2R0mn0nP1hwle0Bc6rQq8wrZ
E1vLqutSx8g7IGo/HN4gMGP2wkTQ6j1Sb7I3bhzqc07aT/aBTYHkmMUZMtA/
9PnXRzqYNcDz07XRlY9oF8Pu4fpfP15rznYxch4aJaWImHm+4B6UHd+hot0X
1T2pVQbZtPuhT3jnt7kq2v3wRsWT6lkt7f7o1lUJTcZp90dy1oL2wHILtgeg
V5qQuMKaWROAvDqR8T3QvhPLPvUV68XRvhP6sTM/aLJpD4Sq/32rXaW0B+IL
vqPl2Wu0S5Bz2Q/eA7RL0DB4r5aboj0IiieGS+MvW7I9CObXHRomNzHzgiHX
+n2f4UF7MPjpW2yOyWmXItdryMviXdqlmGyS7BF/QHsIltqEJYvltIfA85y6
JbCJ9lBo3K/tc/qO9lAMXNn9avMvtIchzbDQYeQp7WHwbZVcqF2zge3hKFUt
5QusmTXhKE7+46KtK+0R4OvVtU5JmRGByzlfhsep6X8EnhnNj+VmMGsj8Pj2
n73RBdTL4BG00v9ROfUyZLQnH7ZrpF6GZ3lzmbZd1MsQ9sDFbXyI+kgcyh6u
jH5MfST8FW1ZHy1SH4kSk/m8Av2NrI9E1taRg34bmHly+Gz+a7DLkRly8E/M
uPPBrJFj4saWjcYy6uW4E3VzemIv9QpwxSL+yXTqFcjnpjqf5lKvgHm3QYxT
MfUKxNsvCrefp55D/Oiw2Zom6jkcrLcvutpFPYfKCkmJ3W3qOTjG3hK9N0Z9
FGp0wsziOeqjUNZZHpv/D/VRiGk5cDfMwIr1Ueg9kaObtmTmKdGnWxH+jj0z
lBja9u3qJndmjRILCXm/3w2kXokknfVXoxz1Klya54e2vU29Cj771Oq0NOpV
qC9XNL+UQ70KZp6yZZmnqI9Gq4HX0s1SK+2/b6fBnQ==
        "]]}, 
      {Hue[0.9060679774997897, 0.6, 0.6], LineBox[CompressedData["
1:eJw1Ug1Qk2UcRwSbAgPW+BJG2AGSTsACW5jjBxuMwQS2sQ8KOAV2RIeAV0p6
1Y1QuOwI211DPMNyDMFhZIKAhUxcXhfnJWbGhYqHFxgBJThBNPo4n//u3tv9
7vf5Ps+7rrBCZXB1cXHZ+d/z///Tnw8WqwKe4ym87E8xHxpu9tQDuxvDgbAJ
xza+ELuG4RBcGTq/MNlCOAwzipOhlhOrGX4epadu1ruZPBgOx+QWTg3yvRmO
RP8Nn1RtOumj0B6irioQ+DC8AaZgeeMXAVyGhXg/0jtWIaF90UgfvFlXZaC8
WPw5/uiyYAf5NyNmn/DDpGZPhl/EQN/ASOp3PIZfQt7uI1x9COnjUDgpG/ft
oLx4JDjeDp9Y58vwFkRUWtRBv5L+ZZi79vRI1pNehNKDtglpdBDDr8DZVhJh
Gid9Ai42OK2zFwlvhSNSFexfQntehfPro2H8eOrbhlmRIVJuJb0YneLLc6n7
+Qwn4tJXbckrjKQHlnfylsoOMmwEHJzfrbruZxmfhP6e0d6Wd4lPgq1saC33
GPmTUW0PNHFdWL4xGWm1pjaFG+2TYLOiYoUllmGjBP7T1pneQj/GS5E/UGFb
2UP5UnzS2rOKW0f+FHxre+Nx6D22x5iCxWc+7nJ/h/hUaHdd44ivUH4qPOpi
5A5HMONlePP69cbxScqX4dMPYJa+Re+XhoKUTvfH9yk/DUOdXcEldyhfjvLR
iA7PQuLl4HnnLh9I92d8OvaUuyY2+1J/Ol67NXhkpJLOOwPR77UUh3rQ+WRg
4VzT4BSoX4Hitk0hgw8oX4FSgQFdO+h8tsPrfoPMkUj527GhRLU3rJn4TPiX
L81sFVJ+Jh5VRNVwj1F+Fjauf5gk+oj4LOj5030/zNG+bEgPf36o6BrLRzaO
asocorwAps/G0N484aKS6e3ZOINL3RfmKV+J7rG/F+aFbA+U8Pvs8HG/s9Sn
hHbaVVBbxPR2JaJ+co3rdqxlfhUOaL8UnKTvEyqIvVTVZjn5Vcjc7WmOmmD5
dhVGvznVWHaI9qvxfWA01jxk+VCDPzU6/Fc97VdjnjOrK7fSfjVOn/ZI8DxO
/hyc4Oz3aLrK7hM50PMy+pfLqT8Hhr7ceEU9+XOwq+eGzH2I7l+DuyL1rbvD
tF8Dcf69ELOT/Bqs3lfvFi8OZH4NzjgbCn5+nfq1WOlm+mV4E52fFglLcyW/
VbJ8oxa53NbBsX+oXwvzH9tqrmbS/evwJDnIUtEkYH4dTLVTceGZ1K9D8e2l
EcttOj8dzv4omRhPpv16nGs4zzfEUL8eea3Vdzo7GDbq0VzEe9JroP16FKy6
oBdJaX8uJrPa220d/vZ/AaqBues=
        "]]}, 
      {Hue[0.14213595499957954`, 0.6, 0.6], LineBox[CompressedData["
1:eJw1k3tQlFUYxjdYQRBDLhkUI5e4GLvjGnJfFx7uLOwuywK7yyVHuYVBiaFM
isBayGrEVGaFAzgQoGSQICYbDrhRUqFi2Cj80QCSWEuWMCILQlAN5/1mvvnm
mfc8v/N873mPa+Y+RY4Jh8PZ89/7/3ft2YS8ibis7om5q2vaHs1a3hGj02Om
HXC5Nnfuu3fnmXaCS+T8QAJ3mWkX+DzvlSF+aK5f027wiC96Pe7YCqu7Y6Nj
3vhrTk+Z9sTRkzd+FDw0Mr0V9YN5vPnUJaa9oROlrw7u5zAeH51ORtcDQkum
t2H7T1Z3gkqovh3phcFjN/spzysQuFbdMbEjng9WhQfrr5dSnh0wtPxS9Zev
CfP7wu3RcN1SMeX3w+2C7I5lv2eY9keTRPvOlcPkD8CTrMVqv7x/mA5Eo9+e
Ae1ByhMEW1lX39FuU6aDwX1VfLyy2oxpIWQv9RX0XyL+TgSOV9qb6VYZTwQ7
YWWVuZR0CD567kXTshJaH4q08g8WN3mtYxpwbHrD+St7xtcACxULPQk29H9h
+EQhKTqTwvJpwnD2BbuG3GLKG47Ii8L9E9lsvSYcfze6TN9qprwR+GF1rtpj
kO2niQB/uvOIr5r4kRip2fKg6TLxI9HqX8XLE1DeKHzRYn7NX8T6oYlC+Gnd
XkPaelaPxtb4uuz8z4gfjVHR5sy9HxI/Bocte2rFZoyniUGftfup06PEj8Xn
v01VBllxWT0WD3aMp59ooXkRw3Gd8WbtW8QXo9Omd9Kzj/hx8GkeEDiIiR8H
/6aYW23FVI8Hf/cN7u8jxI+HpDz0wCGhNatL8LZ12NNzSuJL4Dah+tLjPvml
KBosm7n/HvGlSODd+3XFiuZDhnGv48b2FvLLYD5rcHA++SyrJ6D0UmjbRVA9
Ad9K+jeIuOSXQzDRNKpaz/aDHIX6KZ37Geq3HL2mDeeUJWZrWi+HxJgdY5u5
gfkTYV1fM7S8jfGRiMlZqZ/OmfyJuNJtCHhSzvj6RJx3enNGxGf94Cggcg2S
2u9m9wcKtAhG5ENGC+ZX4ON40WKOC+PrFXj//GaBNoDyJ6FRGNHh9ZjyJ0Ee
klPh/Q31OwkX0gtFsl1sXvRJONFTM6NwZHxOMiwDPVt59pQ/GV+nBRdlSCh/
MkRn1WMOWUzrk1ExqrPriqb7k4KX+aeWyrSMhxQc+j4qMr2M7adJQcTkwt1r
FpQ/BeXXYxqu7qL8SlyQ2RoGfmYaSrSPjUjb7tJ5KWHz57DFvRU2j3olZlMf
5ddZ0vyrUKDZeew2h/KrEDfL63LcR/lVyO3ImLITsX7oVQjaMjPklM/Ok6NG
+x8hfIPbRuZXwzd6mu/dwc5Do4YW6liLRfKrsVJmEdVbSvlT8WmDeeNwK1f/
L+gMvYo=
        "]]}, 
      {Hue[0.37820393249936934`, 0.6, 0.6], LineBox[CompressedData["
1:eJw1kglQlGUYxz9ATrl3BQbBODYULESiOGThzw2i7C57NkxYBCSk6XCWMbWD
hGQO0mAlWwZyaNMA47hgVANs1hYQDKeJSMlgJAoSFR5cbjW8zzfzzTf/+b2/
93ne5/08M46kZhlzHPfKf+//343HHlGfjklcTDbpNjIfmuDclPTDK90b2QWm
Nmd6pZ9RdkOebDd/uMyOrfdAaZ7X591ljxj3wuSM2btlb62zLEBmc/Wpj2zX
WPaB58HBgNFUE+bvQO+u9lI3n8eM+6HEf2ZSP27E+DNoGljILLlK+/kjzyD9
5C8D8QB0+Ht7J15eZXw3at260kUHzBkPxMe88Z4/VgyMP4eM9eLr9e3kB0H4
y6O4X104lp+HYPDYHVOVMcsvIMN9j7NpOa0PRnqgXJg4SzkEd6fD+7xHaH0o
1i6kFJ0bJB6GpvLVgqf1dN49yB+O/FpSQOvDoa01593rMmNZiCyB7kPLl4hH
4H272wkOWjpPJHbWlXXPSmg/gDdwY+2Bie1GVgPzUS87Lr9DPAoXWwfLvru2
mfEozLk7vG5WTvcdjS+lyydKhBaMR4MXE1bJdzBlPAYnF1YWvn3I6qtj0Fg1
NWTiRf3GwvXadpugfOKxWOWcLhe7Ub9x6FxqOxV9lHgcIn/2kaslrB4Xj9/T
1m751hKPx5tjDS1PzVkyngB3j6X66WHiCQi4cF0vamTn4RJxZfBAdIeI+k+E
n29o9I4t9H8modLS+e03OognQd5yX/Kb5xbG92LWwarui2xWT70X/GEjq7Ya
8pMhy588Xm5hxXgyNM/6Np72ZvPm9kH/QV/T1mni+xAYolo9K7NhfD+GLIKd
1vQ0//3QhJpscncinoL+41X+4+PWjKfATxs2f9SOuAgH0+t+sHWl+xWhNKey
50EQcTHOhdo5xmtYvxCjWDtVUVjGuFoMcaugOWHMfiPrxNh2YkUQOUu+BFUr
SdbBmY7Ml8Ax26ju1hGqJ4Fvc3xEywCbl06CmsVF/YQ5zScVu2xzCo9Vk5+K
f3T1Zjk6xtWpsA/93jr3Jwfmp2LieEPszgTWDyfF4qtnKsyLGIcUPV2j98Td
jKul+FsfcmlKTr4U0UPaLJ8kljkZbl4MMzwkDhmUijudI3Msq2Xovl28LDpE
vgya5IIbuW2sX06Ob4w7W7dVky/H+SseaROdPObLUbjczlP2ki9HUcd7W7kS
+n8UiJ0XiC5tpvMr8KOkr2F0xJn5CmyfCGl9LY1xnQKPWw5FZQjJV+LJ+Zk4
56/IVyIz3DBVcZLPfCXu9/N4f3qwfnRKnHadzmkRMs6pkN2okK3XMA4VNGcP
G9K8yFfhav6SQuHDsk6FuxFZY08E5L8I7U3/uPR+vu5fH5bCwQ==
        "]]}}, {
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{4., -3.917904663289041}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{5., -4.1676603111179045`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{6., -4.377058205656498}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{7., -4.557364591368855}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{8., -4.7156951696030465`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{9., -4.856834626591932}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{10., -4.984156083234436}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{11., -5.100127329435382}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{12., -5.20660854839492}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{13., -5.305036840125665}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{14., -5.396545584706128}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{15., -5.482044454934551}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{16., -5.56227469237585}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{17., -5.637848294507051}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{18., -5.70927642609423}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{19., -5.776990426795526}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{20., -5.841357616470237}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{21., -5.902693371753827}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{22., -5.961270482294151}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{23., -6.01732649050745}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{24., -6.071069514967533}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{25., -6.122682918540724}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{26., -6.172329085855666}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{27., -6.2201525065816385`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{28., -6.266282312207568}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{29., -6.310834378602957}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{30., -6.353913080614514}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{31., -6.3956127656001955`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{32., -6.436018998259172}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{33., -6.475209618077768}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{34., -6.513255642255254}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{35., -6.550222040439052}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{36., -6.586168402508595}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{37., -6.621149516651399}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{38., -6.655215871815879}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{39., -6.688414096111081}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{40., -6.720787340709368}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{41., -6.752375617185048}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{42., -6.783216094906443}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{43., -6.813343364026996}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{44., -6.842789668743081}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{45., -6.871585114763557}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{46., -6.89975785433861}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{47., -6.927334251699151}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{48., -6.954339031344147}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{49., -6.980795411266552}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{50., -7.006725222917186}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{51., -7.032149019459917}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{52., -7.057086173663339}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{53., -7.081554966597203}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{54., -7.105572668151014}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{55., -7.129155610263278}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{56., -7.152319253639229}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{57., -7.175078248639745}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{58., -7.19744649094207}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{59., -7.21943717250197}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{60., -7.241062828285394}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{61., -7.26233537918421}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{62., -7.28326617148391}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{63., -7.303866013210512}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{64., -7.324145207648151}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{65., -7.344113584287624}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{66., -7.363780527438617}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{67., -7.383155002714125}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{68., -7.402245581574158}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{69., -7.421060464096957}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{70., -7.43960750012911}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{71., -7.457894208951144}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{72., -7.475927797581894}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{73., -7.49371517783317}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{74., -7.511262982215744}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{75., -7.528577578788251}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{76., -7.545665085032225}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{77., -7.562531380828894}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{78., -7.579182120606655}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{79., -7.595622744722012}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{80., -7.611858490131302}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{81., -7.6278944004055855`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{82., -7.643735335136649}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{83., -7.659385978777978}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{84., -7.674850848960974}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{85., -7.690134304323337}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{86., -7.705240551883574}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{87., -7.720173653992829}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{88., -7.734937534892794}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{89., -7.749535986906176}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{90., -7.7639726762841645`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{91., -7.778251148733439}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{92., -7.792374834643577}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{93., -7.806347054034138}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{94., -7.820171021239268}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{95., -7.833849849346372}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{96., -7.847386554404163}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{97., -7.860784059414356}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{98., -7.8740451981201804`}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{99., -7.887172718604028}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{100., -7.900169286705641}}]},
        SqrtBox["x"]],
       Annotation[#, $CellContext`x^Rational[1, 2], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{4., -3.662143938926416}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{5., -2.5942171388322017`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{6., -3.2735160281641433`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{7., -3.330520568264069}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{8., -3.2044126887136515`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{9., -3.4462977488486115`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{10., -3.9205332028276305`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{11., -3.161297366513138}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{12., -4.028680672118903}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{13., -3.761538679045267}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{14., -3.657860160314896}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{15., -3.924544392990616}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{16., -4.093863591195926}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{17., -3.57727343718502}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{18., -4.692653071119309}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{19., -4.027679660848971}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{20., -3.95901848585065}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{21., -4.283104486109602}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{22., -4.212987238741367}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{23., -3.894172127990326}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{24., -6.295184711664981}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{25., -4.221260628177551}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{26., -4.1893869334343785`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{27., -4.6006702679530385`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{28., -4.301834592287141}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{29., -4.161411857536434}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{30., -5.119411271428972}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{31., -4.375004807992982}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{32., -4.380298526165005}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{33., -4.927092163418084}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{34., -4.371722114954477}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{35., -4.402397555032773}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{36., -5.000055582772007}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{37., -4.504198678916119}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{38., -4.547496817033832}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{39., -5.346408570530022}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{40., -4.428724936194922}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{41., -4.631889443358561}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{42., -4.976680687202073}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{43., -4.617209263962643}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{44., -4.700431955188227}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{45., -6.942155021215295}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{46., -4.476450321545693}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{47., -4.862566325843223}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{48., -4.986309136221591}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{49., -4.719143326667061}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{50., -4.845751466570489}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{51., -5.572772955117914}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{52., -4.517205223582654}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{53., -5.110201250928492}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{54., -5.010858128108607}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{55., -4.813395538932801}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{56., -4.989021721159677}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{57., -5.34443760506779}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{58., -4.562272184525011}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{59., -5.404420708446006}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{60., -5.043197363290821}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{61., -4.902399110167601}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{62., -5.135956793450343}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{63., -5.237101464304111}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{64., -4.70545784573557}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{65., -5.84006029998036}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{66., -5.080048268455083}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{67., -4.988032188455848}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{68., -5.293899416799765}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{69., -5.174887173844827}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{70., -4.84719128923742}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{71., -6.692097485229887}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{72., -5.119778121724838}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{73., -5.071863194115677}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{74., -5.4747641987890585`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{75., -5.1351798723180435`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{76., -4.99027347833016}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{77., -5.8886139669871405`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{78., -5.161553739455701}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{79., -5.15532007881431}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{80., -5.703708819697711}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{81., -5.10839362879468}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{82., -5.137986024476366}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{83., -5.6982623314669185`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{84., -5.204970796780091}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{85., -5.239830447847735}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{86., -6.060749121794523}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{87., -5.08968112710629}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{88., -5.294745796477403}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{89., -5.61025011463658}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{90., -5.24987701548124}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{91., -5.3269672876876}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{92., -7.399839924256181}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{93., -5.076309992613135}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{94., -5.467421903907596}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{95., -5.5653186777962445`}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{96., -5.296284030806353}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{97., -5.418639667428206}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{98., -6.099338472959884}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{99., -5.066621195409963}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{100., -5.668623591383926}}]},
        RowBox[{"Abs", "[", 
          RowBox[{
            RowBox[{"-", 
              FractionBox["1", "\[ExponentialE]"]}], "+", "x"}], "]"}]],
       Annotation[#, 
        Abs[(-1)/E + $CellContext`x], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{4., -1.3048623922932396`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{5., -1.194571957625592}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{6., -1.4071692988306785`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{7., -1.7510526439621188`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{8., -2.9903742980412966`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{9., -1.9075416715829274`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{10., -1.6319349982091496`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{11., -1.4951627582278493`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{12., -1.7094632812192352`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{13., -2.0555629682273455`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{14., -3.279030099252474}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{15., -2.060168899967856}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{16., -1.7975596123497073`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{17., -1.6921413747350689`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{18., -1.9057117605877918`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{19., -2.2758437084164496`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{20., -2.933097100407296}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{21., -2.1513608723304083`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{22., -1.9049639796037716`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{23., -1.8379416986338104`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{24., -2.0573878763595563`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{25., -2.4646029375524074`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{26., -2.8191042563809123`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{27., -2.2132614608312116`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{28., -1.9824276735143496`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{29., -1.956306023613881}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{30., -2.1852989428177314`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{31., -2.6440753456990196`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{32., -2.7600977378209675`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{33., -2.25845285483285}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{34., -2.0418061263720326`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{35., -2.0578129297094447`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{36., -2.29925621474675}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{37., -2.8304746905525837`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{38., -2.7237044420616914`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{39., -2.2930688611789005`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{40., -2.0891797998511765`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{41., -2.1481479074344243`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{42., -2.4048896210631097`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{43., -3.044390512414936}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{44., -2.6989353046605125`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{45., -2.320513502410659}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{46., -2.1280700842046913`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{47., -2.2307616780762762`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{48., -2.5059690989405263`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{49., -3.330828397511584}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{50., -2.680961236096334}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{51., -2.342847606081897}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{52., -2.160690663575701}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{53., -2.307946795819954}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{54., -2.605421102034617}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{55., -3.904040249049933}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{56., -2.6673125792603365`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{57., -2.3613999735062103`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{58., -2.192819479817723}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{59., -2.381345487233751}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{60., -2.7058982218195378`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{61., -3.8213483759301456`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{62., -2.6565907914105638`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{63., -2.377069560877857}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{64., -2.2540738034258747`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{65., -2.452218479644266}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{66., -2.8102238746364194`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{67., -3.4228864267051105`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{68., -2.6479431286167094`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{69., -2.390488068047703}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{70., -2.312479435749052}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{71., -2.521602514610625}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{72., -2.9219137223300593`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{73., -3.2459989848775472`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{74., -2.640819499190408}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{75., -2.402113311652391}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{76., -2.3687280436056004`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{77., -2.590413126315788}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{78., -3.046047668448828}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{79., -3.1373652264222227`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{80., -2.6348488938010677`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{81., -2.4122857095893218`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{82., -2.423387276726116}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{83., -2.6595212498460246`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{84., -3.1911601690858515`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{85., -3.061697514895567}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{86., -2.6297718805983847`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{87., -2.4212639352712695`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{88., -2.47694189154739}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{89., -2.7298208314421886`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{90., -3.3744526641070327`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{91., -3.0051811630648833`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{92., -2.6254015359009397`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{93., -2.4292482373986433`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{94., -2.5298231047545867`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{95., -2.802301041319677}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{96., -3.6420463043878812`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{97., -2.961017943286402}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{98., -2.6215997363979713`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{99., -2.4363961595080346`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{100., -2.5824310856297172`}}]},
        RowBox[{"\[Piecewise]", 
          GridBox[{{"2", 
             RowBox[{"x", "\[LessEqual]", 
               FractionBox["1", "\[ExponentialE]"]}]}, {"3", 
             RowBox[{"x", ">", 
               FractionBox["1", "\[ExponentialE]"]}]}}, 
           ColumnAlignments -> {Left}, ColumnSpacings -> 1.2, ColumnWidths -> 
           Automatic]}]],
       Annotation[#, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{4., -2.5015112298018782`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{5., -1.651701265629049}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{6., -1.6631658434060352`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{7., -3.9398519113115578`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{8., -1.4699093033915749`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{9., -1.842167373847011}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{10., -1.7535605899127171`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{11., -2.415441876725723}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{12., -1.509304533555211}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{13., -2.230846143478457}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{14., -1.8599819128693096`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{15., -2.2499725172528366`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{16., -1.7310258266979144`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{17., -2.92153810517955}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{18., -1.9987552495451602`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{19., -2.213689502095506}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{20., -2.012140795350915}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{21., -2.2929800114014984`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{22., -2.1904393973269842`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{23., -2.2384190360665306`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{24., -2.3526073264914555`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{25., -2.2263745539429447`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{26., -2.471262849737132}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{27., -2.306796500454021}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{28., -2.843219898862992}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{29., -2.295428408313955}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{30., -2.962435304276316}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{31., -2.414994713053621}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{32., -3.751934028050195}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{33., -2.4369985042214832`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{34., -3.47937301405712}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{35., -2.565441929302087}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{36., -3.0300226805652146`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{37., -2.6333399879007486`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{38., -2.9954756422355944`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{39., -2.7666088408023644`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{40., -2.9312516238957302`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{41., -2.8889001763310467`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{42., -2.929814433357562}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{43., -3.0397955563034027`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{44., -2.9527165888680305`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{45., -3.2398065152761397`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{46., -2.977483056140058}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{47., -3.4552302795825867`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{48., -3.038925665425235}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{49., -3.88533069359723}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{50., -3.090054475441497}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{51., -5.283074646121652}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{52., -3.175126314456282}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{53., -3.9500504285625855`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{54., -3.254162817158809}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{55., -3.7673638176981514`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{56., -3.360404296153748}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{57., -3.666307414655443}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{58., -3.4716722236199544`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{59., -3.635797530506534}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{60., -3.60588108423733}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{61., -3.6325396018422866`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{62., -3.762723480362002}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{63., -3.650859419675897}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{64., -3.950328961457855}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{65., -3.689677304608661}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{66., -4.207304658640695}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{67., -3.7384016884760873`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{68., -4.597866241469674}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{69., -3.8046257496287206`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{70., -5.450856183112628}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{71., -3.878835245528603}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{72., -4.644021830117295}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{73., -3.968950321046685}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{74., -4.444740485587559}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{75., -4.069500992279211}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{76., -4.365263113076239}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{77., -4.1868321124781644`}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{78., -4.333864930596507}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{79., -4.321435536479898}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{80., -4.333950031237658}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{81., -4.480296103710848}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{82., -4.355773828451617}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{83., -4.676188848231414}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{84., -4.393671732221884}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{85., -4.935403274413745}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{86., -4.446621234876479}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{87., -5.371094160114913}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{88., -4.511414753559681}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{89., -5.955885466788862}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{90., -4.589262697429638}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{91., -5.309945127733492}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{92., -4.678799717083619}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{93., -5.135287776544917}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{94., -4.782161996921087}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{95., -5.060231832909849}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{96., -4.900380437648982}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{97., -5.033554068626609}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{98., -5.037438734693696}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{99., -5.03612834804492}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        {Opacity[0.], PointBox[{{100., -5.199578362450597}}]},
        FractionBox["1", 
         RowBox[{"1", "+", 
           RowBox[{"10000", " ", 
             SuperscriptBox[
              RowBox[{"(", 
                RowBox[{
                  FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}]}]]],
       Annotation[#, (1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), 
        "Tooltip"]& ]}},
    AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
    Axes->True,
    AxesOrigin->{0, 0},
    ImageSize->{300},
    PlotRange->{{0, 100}, {0, -9}},
    PlotRangeClipping->True,
    PlotRangePadding->{Automatic, Automatic}], 
   "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
   GraphicsBox[{{InsetBox[
       SqrtBox["x"], {110, -2}, {-1, 0}], InsetBox[
       RowBox[{"\[LeftBracketingBar]", 
        RowBox[{"x", "-", 
         FractionBox["1", "\[ExponentialE]"]}], 
        "\[RightBracketingBar]"}], {110, -4}, {-1, 0}], InsetBox[
       RowBox[{"\[Piecewise]", GridBox[{
          {"2", 
           RowBox[{"x", "\[LessEqual]", 
            FractionBox["1", "\[ExponentialE]"]}]},
          {"3", 
           RowBox[{"x", ">", 
            FractionBox["1", "\[ExponentialE]"]}]}
         },
         GridBoxAlignment->{
          "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{Baseline}}, "RowsIndexed" -> {}},
         GridBoxItemSize->{
          "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{1.}}, "RowsIndexed" -> {}},
         GridBoxSpacings->{"Columns" -> {
             Offset[0.27999999999999997`], {
              Offset[0.84]}, 
             Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, 
           "Rows" -> {
             Offset[0.2], {
              Offset[0.4]}, 
             Offset[0.2]}, "RowsIndexed" -> {}}]}], {110, -6}, {-1, 0}], 
      InsetBox[
       FractionBox["1", 
        RowBox[{
         RowBox[{"10000", " ", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{
             FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}], "+", 
         "1"}]], {110, -8}, {-1, 0}]}, {
      InsetBox["\<\"  \"\>", {106, -2}, {-1, 0},
       Background->Hue[0.67, 0.6, 0.6]], 
      InsetBox["\<\"  \"\>", {106, -4}, {-1, 0},
       Background->Hue[0.9060679774997897, 0.6, 0.6]], 
      InsetBox["\<\"  \"\>", {106, -6}, {-1, 0},
       Background->Hue[0.14213595499957954`, 0.6, 0.6]], 
      InsetBox["\<\"  \"\>", {106, -8}, {-1, 0},
       Background->Hue[0.37820393249936934`, 0.6, 0.6]]}},
    AspectRatio->5,
    ImageSize->{200, 200}]}],
  Row[{
    Graphics[{{{}, {}, {
        Hue[0.67, 0.6, 0.6], 
        Line[CompressedData["
1:eJw1ywlQ1GUcxvFVKVdRXFpALkUpCgE52pRL4FkOWViuZffPLoKyUjAoIcKQ
YOKwg4GYOkRgyg2ZxiAOJoiQATuERJENCEYYrkEexc3YGJDZMb6/d+addz7z
vN/N8amRCct5PN7e/+7/74sjwKQ6JVGnXKd9YSO07NKzabwkYDbF1Znhkxa5
hsyWaMot7G33eIV5E+oTb7ncHyVbo9M+xSAxRcj8Gv4u66jomSW/jrQ07pvj
iUbMtviwddG5aJBsh/t1A8UxbsbMDjjS0aIv+ITsiJkL3esqp8nOaDw6Otvt
Y8LsgjhdurntafKb0D/nWhjyI1mEgKIx9XmL9cxvoXny55FTu8nbUPP8+sWY
CvJ2rBULgpU/kV2xwJmu6jA0ZXZDv+X+9B4J2R0mn0nP1hwle0Bc6rQq8wrZ
E1vLqutSx8g7IGo/HN4gMGP2wkTQ6j1Sb7I3bhzqc07aT/aBTYHkmMUZMtA/
9PnXRzqYNcDz07XRlY9oF8Pu4fpfP15rznYxch4aJaWImHm+4B6UHd+hot0X
1T2pVQbZtPuhT3jnt7kq2v3wRsWT6lkt7f7o1lUJTcZp90dy1oL2wHILtgeg
V5qQuMKaWROAvDqR8T3QvhPLPvUV68XRvhP6sTM/aLJpD4Sq/32rXaW0B+IL
vqPl2Wu0S5Bz2Q/eA7RL0DB4r5aboj0IiieGS+MvW7I9CObXHRomNzHzgiHX
+n2f4UF7MPjpW2yOyWmXItdryMviXdqlmGyS7BF/QHsIltqEJYvltIfA85y6
JbCJ9lBo3K/tc/qO9lAMXNn9avMvtIchzbDQYeQp7WHwbZVcqF2zge3hKFUt
5QusmTXhKE7+46KtK+0R4OvVtU5JmRGByzlfhsep6X8EnhnNj+VmMGsj8Pj2
n73RBdTL4BG00v9ROfUyZLQnH7ZrpF6GZ3lzmbZd1MsQ9sDFbXyI+kgcyh6u
jH5MfST8FW1ZHy1SH4kSk/m8Av2NrI9E1taRg34bmHly+Gz+a7DLkRly8E/M
uPPBrJFj4saWjcYy6uW4E3VzemIv9QpwxSL+yXTqFcjnpjqf5lKvgHm3QYxT
MfUKxNsvCrefp55D/Oiw2Zom6jkcrLcvutpFPYfKCkmJ3W3qOTjG3hK9N0Z9
FGp0wsziOeqjUNZZHpv/D/VRiGk5cDfMwIr1Ueg9kaObtmTmKdGnWxH+jj0z
lBja9u3qJndmjRILCXm/3w2kXokknfVXoxz1Klya54e2vU29Cj771Oq0NOpV
qC9XNL+UQ70KZp6yZZmnqI9Gq4HX0s1SK+2/b6fBnQ==
         "]]}, {
        Hue[0.9060679774997897, 0.6, 0.6], 
        Line[CompressedData["
1:eJw1Ug1Qk2UcRwSbAgPW+BJG2AGSTsACW5jjBxuMwQS2sQ8KOAV2RIeAV0p6
1Y1QuOwI211DPMNyDMFhZIKAhUxcXhfnJWbGhYqHFxgBJThBNPo4n//u3tv9
7vf5Ps+7rrBCZXB1cXHZ+d/z///Tnw8WqwKe4ym87E8xHxpu9tQDuxvDgbAJ
xza+ELuG4RBcGTq/MNlCOAwzipOhlhOrGX4epadu1ruZPBgOx+QWTg3yvRmO
RP8Nn1RtOumj0B6irioQ+DC8AaZgeeMXAVyGhXg/0jtWIaF90UgfvFlXZaC8
WPw5/uiyYAf5NyNmn/DDpGZPhl/EQN/ASOp3PIZfQt7uI1x9COnjUDgpG/ft
oLx4JDjeDp9Y58vwFkRUWtRBv5L+ZZi79vRI1pNehNKDtglpdBDDr8DZVhJh
Gid9Ai42OK2zFwlvhSNSFexfQntehfPro2H8eOrbhlmRIVJuJb0YneLLc6n7
+Qwn4tJXbckrjKQHlnfylsoOMmwEHJzfrbruZxmfhP6e0d6Wd4lPgq1saC33
GPmTUW0PNHFdWL4xGWm1pjaFG+2TYLOiYoUllmGjBP7T1pneQj/GS5E/UGFb
2UP5UnzS2rOKW0f+FHxre+Nx6D22x5iCxWc+7nJ/h/hUaHdd44ivUH4qPOpi
5A5HMONlePP69cbxScqX4dMPYJa+Re+XhoKUTvfH9yk/DUOdXcEldyhfjvLR
iA7PQuLl4HnnLh9I92d8OvaUuyY2+1J/Ol67NXhkpJLOOwPR77UUh3rQ+WRg
4VzT4BSoX4Hitk0hgw8oX4FSgQFdO+h8tsPrfoPMkUj527GhRLU3rJn4TPiX
L81sFVJ+Jh5VRNVwj1F+Fjauf5gk+oj4LOj5030/zNG+bEgPf36o6BrLRzaO
asocorwAps/G0N484aKS6e3ZOINL3RfmKV+J7rG/F+aFbA+U8Pvs8HG/s9Sn
hHbaVVBbxPR2JaJ+co3rdqxlfhUOaL8UnKTvEyqIvVTVZjn5Vcjc7WmOmmD5
dhVGvznVWHaI9qvxfWA01jxk+VCDPzU6/Fc97VdjnjOrK7fSfjVOn/ZI8DxO
/hyc4Oz3aLrK7hM50PMy+pfLqT8Hhr7ceEU9+XOwq+eGzH2I7l+DuyL1rbvD
tF8Dcf69ELOT/Bqs3lfvFi8OZH4NzjgbCn5+nfq1WOlm+mV4E52fFglLcyW/
VbJ8oxa53NbBsX+oXwvzH9tqrmbS/evwJDnIUtEkYH4dTLVTceGZ1K9D8e2l
EcttOj8dzv4omRhPpv16nGs4zzfEUL8eea3Vdzo7GDbq0VzEe9JroP16FKy6
oBdJaX8uJrPa220d/vZ/AaqBues=
         "]]}, {
        Hue[0.14213595499957954`, 0.6, 0.6], 
        Line[CompressedData["
1:eJw1k3tQlFUYxjdYQRBDLhkUI5e4GLvjGnJfFx7uLOwuywK7yyVHuYVBiaFM
isBayGrEVGaFAzgQoGSQICYbDrhRUqFi2Cj80QCSWEuWMCILQlAN5/1mvvnm
mfc8v/N873mPa+Y+RY4Jh8PZ89/7/3ft2YS8ibis7om5q2vaHs1a3hGj02Om
HXC5Nnfuu3fnmXaCS+T8QAJ3mWkX+DzvlSF+aK5f027wiC96Pe7YCqu7Y6Nj
3vhrTk+Z9sTRkzd+FDw0Mr0V9YN5vPnUJaa9oROlrw7u5zAeH51ORtcDQkum
t2H7T1Z3gkqovh3phcFjN/spzysQuFbdMbEjng9WhQfrr5dSnh0wtPxS9Zev
CfP7wu3RcN1SMeX3w+2C7I5lv2eY9keTRPvOlcPkD8CTrMVqv7x/mA5Eo9+e
Ae1ByhMEW1lX39FuU6aDwX1VfLyy2oxpIWQv9RX0XyL+TgSOV9qb6VYZTwQ7
YWWVuZR0CD567kXTshJaH4q08g8WN3mtYxpwbHrD+St7xtcACxULPQk29H9h
+EQhKTqTwvJpwnD2BbuG3GLKG47Ii8L9E9lsvSYcfze6TN9qprwR+GF1rtpj
kO2niQB/uvOIr5r4kRip2fKg6TLxI9HqX8XLE1DeKHzRYn7NX8T6oYlC+Gnd
XkPaelaPxtb4uuz8z4gfjVHR5sy9HxI/Bocte2rFZoyniUGftfup06PEj8Xn
v01VBllxWT0WD3aMp59ooXkRw3Gd8WbtW8QXo9Omd9Kzj/hx8GkeEDiIiR8H
/6aYW23FVI8Hf/cN7u8jxI+HpDz0wCGhNatL8LZ12NNzSuJL4Dah+tLjPvml
KBosm7n/HvGlSODd+3XFiuZDhnGv48b2FvLLYD5rcHA++SyrJ6D0UmjbRVA9
Ad9K+jeIuOSXQzDRNKpaz/aDHIX6KZ37Geq3HL2mDeeUJWZrWi+HxJgdY5u5
gfkTYV1fM7S8jfGRiMlZqZ/OmfyJuNJtCHhSzvj6RJx3enNGxGf94Cggcg2S
2u9m9wcKtAhG5ENGC+ZX4ON40WKOC+PrFXj//GaBNoDyJ6FRGNHh9ZjyJ0Ee
klPh/Q31OwkX0gtFsl1sXvRJONFTM6NwZHxOMiwDPVt59pQ/GV+nBRdlSCh/
MkRn1WMOWUzrk1ExqrPriqb7k4KX+aeWyrSMhxQc+j4qMr2M7adJQcTkwt1r
FpQ/BeXXYxqu7qL8SlyQ2RoGfmYaSrSPjUjb7tJ5KWHz57DFvRU2j3olZlMf
5ddZ0vyrUKDZeew2h/KrEDfL63LcR/lVyO3ImLITsX7oVQjaMjPklM/Ok6NG
+x8hfIPbRuZXwzd6mu/dwc5Do4YW6liLRfKrsVJmEdVbSvlT8WmDeeNwK1f/
L+gMvYo=
         "]]}, {
        Hue[0.37820393249936934`, 0.6, 0.6], 
        Line[CompressedData["
1:eJw1kglQlGUYxz9ATrl3BQbBODYULESiOGThzw2i7C57NkxYBCSk6XCWMbWD
hGQO0mAlWwZyaNMA47hgVANs1hYQDKeJSMlgJAoSFR5cbjW8zzfzzTf/+b2/
93ne5/08M46kZhlzHPfKf+//343HHlGfjklcTDbpNjIfmuDclPTDK90b2QWm
Nmd6pZ9RdkOebDd/uMyOrfdAaZ7X591ljxj3wuSM2btlb62zLEBmc/Wpj2zX
WPaB58HBgNFUE+bvQO+u9lI3n8eM+6HEf2ZSP27E+DNoGljILLlK+/kjzyD9
5C8D8QB0+Ht7J15eZXw3at260kUHzBkPxMe88Z4/VgyMP4eM9eLr9e3kB0H4
y6O4X104lp+HYPDYHVOVMcsvIMN9j7NpOa0PRnqgXJg4SzkEd6fD+7xHaH0o
1i6kFJ0bJB6GpvLVgqf1dN49yB+O/FpSQOvDoa01593rMmNZiCyB7kPLl4hH
4H272wkOWjpPJHbWlXXPSmg/gDdwY+2Bie1GVgPzUS87Lr9DPAoXWwfLvru2
mfEozLk7vG5WTvcdjS+lyydKhBaMR4MXE1bJdzBlPAYnF1YWvn3I6qtj0Fg1
NWTiRf3GwvXadpugfOKxWOWcLhe7Ub9x6FxqOxV9lHgcIn/2kaslrB4Xj9/T
1m751hKPx5tjDS1PzVkyngB3j6X66WHiCQi4cF0vamTn4RJxZfBAdIeI+k+E
n29o9I4t9H8modLS+e03OognQd5yX/Kb5xbG92LWwarui2xWT70X/GEjq7Ya
8pMhy588Xm5hxXgyNM/6Np72ZvPm9kH/QV/T1mni+xAYolo9K7NhfD+GLIKd
1vQ0//3QhJpscncinoL+41X+4+PWjKfATxs2f9SOuAgH0+t+sHWl+xWhNKey
50EQcTHOhdo5xmtYvxCjWDtVUVjGuFoMcaugOWHMfiPrxNh2YkUQOUu+BFUr
SdbBmY7Ml8Ax26ju1hGqJ4Fvc3xEywCbl06CmsVF/YQ5zScVu2xzCo9Vk5+K
f3T1Zjk6xtWpsA/93jr3Jwfmp2LieEPszgTWDyfF4qtnKsyLGIcUPV2j98Td
jKul+FsfcmlKTr4U0UPaLJ8kljkZbl4MMzwkDhmUijudI3Msq2Xovl28LDpE
vgya5IIbuW2sX06Ob4w7W7dVky/H+SseaROdPObLUbjczlP2ki9HUcd7W7kS
+n8UiJ0XiC5tpvMr8KOkr2F0xJn5CmyfCGl9LY1xnQKPWw5FZQjJV+LJ+Zk4
56/IVyIz3DBVcZLPfCXu9/N4f3qwfnRKnHadzmkRMs6pkN2okK3XMA4VNGcP
G9K8yFfhav6SQuHDsk6FuxFZY08E5L8I7U3/uPR+vu5fH5bCwQ==
         "]]}}, {
       Tooltip[{
         Opacity[0.], 
         Point[{{4., -3.917904663289041}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{5., -4.1676603111179045`}}]}, $CellContext`x^Rational[1, 2]],
        
       Tooltip[{
         Opacity[0.], 
         Point[{{6., -4.377058205656498}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{7., -4.557364591368855}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{8., -4.7156951696030465`}}]}, $CellContext`x^Rational[1, 2]],
        
       Tooltip[{
         Opacity[0.], 
         Point[{{9., -4.856834626591932}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{10., -4.984156083234436}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{11., -5.100127329435382}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{12., -5.20660854839492}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{13., -5.305036840125665}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{14., -5.396545584706128}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{15., -5.482044454934551}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{16., -5.56227469237585}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{17., -5.637848294507051}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{18., -5.70927642609423}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{19., -5.776990426795526}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{20., -5.841357616470237}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{21., -5.902693371753827}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{22., -5.961270482294151}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{23., -6.01732649050745}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{24., -6.071069514967533}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{25., -6.122682918540724}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{26., -6.172329085855666}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{27., -6.2201525065816385`}}]}, $CellContext`x^
        Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{28., -6.266282312207568}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{29., -6.310834378602957}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{30., -6.353913080614514}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{31., -6.3956127656001955`}}]}, $CellContext`x^
        Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{32., -6.436018998259172}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{33., -6.475209618077768}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{34., -6.513255642255254}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{35., -6.550222040439052}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{36., -6.586168402508595}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{37., -6.621149516651399}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{38., -6.655215871815879}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{39., -6.688414096111081}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{40., -6.720787340709368}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{41., -6.752375617185048}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{42., -6.783216094906443}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{43., -6.813343364026996}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{44., -6.842789668743081}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{45., -6.871585114763557}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{46., -6.89975785433861}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{47., -6.927334251699151}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{48., -6.954339031344147}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{49., -6.980795411266552}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{50., -7.006725222917186}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{51., -7.032149019459917}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{52., -7.057086173663339}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{53., -7.081554966597203}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{54., -7.105572668151014}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{55., -7.129155610263278}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{56., -7.152319253639229}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{57., -7.175078248639745}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{58., -7.19744649094207}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{59., -7.21943717250197}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{60., -7.241062828285394}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{61., -7.26233537918421}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{62., -7.28326617148391}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{63., -7.303866013210512}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{64., -7.324145207648151}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{65., -7.344113584287624}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{66., -7.363780527438617}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{67., -7.383155002714125}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{68., -7.402245581574158}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{69., -7.421060464096957}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{70., -7.43960750012911}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{71., -7.457894208951144}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{72., -7.475927797581894}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{73., -7.49371517783317}}]}, $CellContext`x^Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{74., -7.511262982215744}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{75., -7.528577578788251}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{76., -7.545665085032225}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{77., -7.562531380828894}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{78., -7.579182120606655}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{79., -7.595622744722012}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{80., -7.611858490131302}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{81., -7.6278944004055855`}}]}, $CellContext`x^
        Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{82., -7.643735335136649}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{83., -7.659385978777978}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{84., -7.674850848960974}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{85., -7.690134304323337}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{86., -7.705240551883574}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{87., -7.720173653992829}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{88., -7.734937534892794}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{89., -7.749535986906176}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{90., -7.7639726762841645`}}]}, $CellContext`x^
        Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{91., -7.778251148733439}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{92., -7.792374834643577}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{93., -7.806347054034138}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{94., -7.820171021239268}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{95., -7.833849849346372}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{96., -7.847386554404163}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{97., -7.860784059414356}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{98., -7.8740451981201804`}}]}, $CellContext`x^
        Rational[1, 2]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{99., -7.887172718604028}}]}, $CellContext`x^Rational[1, 2]], 
       
       Tooltip[{
         Opacity[0.], 
         Point[{{100., -7.900169286705641}}]}, $CellContext`x^Rational[1, 2]],
        
       Tooltip[{
         Opacity[0.], 
         Point[{{4., -3.662143938926416}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{5., -2.5942171388322017`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{6., -3.2735160281641433`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{7., -3.330520568264069}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{8., -3.2044126887136515`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{9., -3.4462977488486115`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{10., -3.9205332028276305`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{11., -3.161297366513138}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{12., -4.028680672118903}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{13., -3.761538679045267}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{14., -3.657860160314896}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{15., -3.924544392990616}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{16., -4.093863591195926}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{17., -3.57727343718502}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{18., -4.692653071119309}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{19., -4.027679660848971}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{20., -3.95901848585065}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{21., -4.283104486109602}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{22., -4.212987238741367}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{23., -3.894172127990326}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{24., -6.295184711664981}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{25., -4.221260628177551}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{26., -4.1893869334343785`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{27., -4.6006702679530385`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{28., -4.301834592287141}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{29., -4.161411857536434}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{30., -5.119411271428972}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{31., -4.375004807992982}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{32., -4.380298526165005}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{33., -4.927092163418084}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{34., -4.371722114954477}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{35., -4.402397555032773}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{36., -5.000055582772007}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{37., -4.504198678916119}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{38., -4.547496817033832}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{39., -5.346408570530022}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{40., -4.428724936194922}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{41., -4.631889443358561}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{42., -4.976680687202073}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{43., -4.617209263962643}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{44., -4.700431955188227}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{45., -6.942155021215295}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{46., -4.476450321545693}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{47., -4.862566325843223}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{48., -4.986309136221591}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{49., -4.719143326667061}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{50., -4.845751466570489}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{51., -5.572772955117914}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{52., -4.517205223582654}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{53., -5.110201250928492}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{54., -5.010858128108607}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{55., -4.813395538932801}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{56., -4.989021721159677}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{57., -5.34443760506779}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{58., -4.562272184525011}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{59., -5.404420708446006}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{60., -5.043197363290821}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{61., -4.902399110167601}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{62., -5.135956793450343}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{63., -5.237101464304111}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{64., -4.70545784573557}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{65., -5.84006029998036}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{66., -5.080048268455083}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{67., -4.988032188455848}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{68., -5.293899416799765}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{69., -5.174887173844827}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{70., -4.84719128923742}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{71., -6.692097485229887}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{72., -5.119778121724838}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{73., -5.071863194115677}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{74., -5.4747641987890585`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{75., -5.1351798723180435`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{76., -4.99027347833016}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{77., -5.8886139669871405`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{78., -5.161553739455701}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{79., -5.15532007881431}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{80., -5.703708819697711}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{81., -5.10839362879468}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{82., -5.137986024476366}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{83., -5.6982623314669185`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{84., -5.204970796780091}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{85., -5.239830447847735}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{86., -6.060749121794523}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{87., -5.08968112710629}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{88., -5.294745796477403}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{89., -5.61025011463658}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{90., -5.24987701548124}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{91., -5.3269672876876}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{92., -7.399839924256181}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{93., -5.076309992613135}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{94., -5.467421903907596}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{95., -5.5653186777962445`}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{96., -5.296284030806353}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{97., -5.418639667428206}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{98., -6.099338472959884}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{99., -5.066621195409963}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{100., -5.668623591383926}}]}, 
        Abs[(-1)/E + $CellContext`x]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{4., -1.3048623922932396`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{5., -1.194571957625592}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{6., -1.4071692988306785`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{7., -1.7510526439621188`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{8., -2.9903742980412966`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{9., -1.9075416715829274`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{10., -1.6319349982091496`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{11., -1.4951627582278493`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{12., -1.7094632812192352`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{13., -2.0555629682273455`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{14., -3.279030099252474}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{15., -2.060168899967856}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{16., -1.7975596123497073`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{17., -1.6921413747350689`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{18., -1.9057117605877918`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{19., -2.2758437084164496`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{20., -2.933097100407296}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{21., -2.1513608723304083`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{22., -1.9049639796037716`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{23., -1.8379416986338104`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{24., -2.0573878763595563`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{25., -2.4646029375524074`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{26., -2.8191042563809123`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{27., -2.2132614608312116`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{28., -1.9824276735143496`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{29., -1.956306023613881}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{30., -2.1852989428177314`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{31., -2.6440753456990196`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{32., -2.7600977378209675`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{33., -2.25845285483285}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{34., -2.0418061263720326`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{35., -2.0578129297094447`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{36., -2.29925621474675}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{37., -2.8304746905525837`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{38., -2.7237044420616914`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{39., -2.2930688611789005`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{40., -2.0891797998511765`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{41., -2.1481479074344243`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{42., -2.4048896210631097`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{43., -3.044390512414936}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{44., -2.6989353046605125`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{45., -2.320513502410659}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{46., -2.1280700842046913`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{47., -2.2307616780762762`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{48., -2.5059690989405263`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{49., -3.330828397511584}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{50., -2.680961236096334}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{51., -2.342847606081897}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{52., -2.160690663575701}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{53., -2.307946795819954}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{54., -2.605421102034617}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{55., -3.904040249049933}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{56., -2.6673125792603365`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{57., -2.3613999735062103`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{58., -2.192819479817723}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{59., -2.381345487233751}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{60., -2.7058982218195378`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{61., -3.8213483759301456`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{62., -2.6565907914105638`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{63., -2.377069560877857}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{64., -2.2540738034258747`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{65., -2.452218479644266}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{66., -2.8102238746364194`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{67., -3.4228864267051105`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{68., -2.6479431286167094`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{69., -2.390488068047703}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{70., -2.312479435749052}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{71., -2.521602514610625}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{72., -2.9219137223300593`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{73., -3.2459989848775472`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{74., -2.640819499190408}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{75., -2.402113311652391}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{76., -2.3687280436056004`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{77., -2.590413126315788}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{78., -3.046047668448828}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{79., -3.1373652264222227`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{80., -2.6348488938010677`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{81., -2.4122857095893218`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{82., -2.423387276726116}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{83., -2.6595212498460246`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{84., -3.1911601690858515`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{85., -3.061697514895567}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{86., -2.6297718805983847`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{87., -2.4212639352712695`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{88., -2.47694189154739}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{89., -2.7298208314421886`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{90., -3.3744526641070327`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{91., -3.0051811630648833`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{92., -2.6254015359009397`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{93., -2.4292482373986433`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{94., -2.5298231047545867`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{95., -2.802301041319677}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{96., -3.6420463043878812`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{97., -2.961017943286402}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{98., -2.6215997363979713`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{99., -2.4363961595080346`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{100., -2.5824310856297172`}}]}, 
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0]], 
       Tooltip[{
         Opacity[0.], 
         Point[{{4., -2.5015112298018782`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{5., -1.651701265629049}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{6., -1.6631658434060352`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{7., -3.9398519113115578`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{8., -1.4699093033915749`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{9., -1.842167373847011}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{10., -1.7535605899127171`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{11., -2.415441876725723}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{12., -1.509304533555211}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{13., -2.230846143478457}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{14., -1.8599819128693096`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{15., -2.2499725172528366`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{16., -1.7310258266979144`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{17., -2.92153810517955}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{18., -1.9987552495451602`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{19., -2.213689502095506}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{20., -2.012140795350915}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{21., -2.2929800114014984`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{22., -2.1904393973269842`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{23., -2.2384190360665306`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{24., -2.3526073264914555`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{25., -2.2263745539429447`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{26., -2.471262849737132}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{27., -2.306796500454021}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{28., -2.843219898862992}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{29., -2.295428408313955}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{30., -2.962435304276316}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{31., -2.414994713053621}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{32., -3.751934028050195}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{33., -2.4369985042214832`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{34., -3.47937301405712}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{35., -2.565441929302087}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{36., -3.0300226805652146`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{37., -2.6333399879007486`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{38., -2.9954756422355944`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{39., -2.7666088408023644`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{40., -2.9312516238957302`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{41., -2.8889001763310467`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{42., -2.929814433357562}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{43., -3.0397955563034027`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{44., -2.9527165888680305`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{45., -3.2398065152761397`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{46., -2.977483056140058}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{47., -3.4552302795825867`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{48., -3.038925665425235}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{49., -3.88533069359723}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{50., -3.090054475441497}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{51., -5.283074646121652}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{52., -3.175126314456282}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{53., -3.9500504285625855`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{54., -3.254162817158809}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{55., -3.7673638176981514`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{56., -3.360404296153748}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{57., -3.666307414655443}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{58., -3.4716722236199544`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{59., -3.635797530506534}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{60., -3.60588108423733}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{61., -3.6325396018422866`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{62., -3.762723480362002}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{63., -3.650859419675897}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{64., -3.950328961457855}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{65., -3.689677304608661}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{66., -4.207304658640695}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{67., -3.7384016884760873`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{68., -4.597866241469674}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{69., -3.8046257496287206`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{70., -5.450856183112628}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{71., -3.878835245528603}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{72., -4.644021830117295}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{73., -3.968950321046685}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{74., -4.444740485587559}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{75., -4.069500992279211}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{76., -4.365263113076239}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{77., -4.1868321124781644`}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{78., -4.333864930596507}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{79., -4.321435536479898}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{80., -4.333950031237658}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{81., -4.480296103710848}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{82., -4.355773828451617}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{83., -4.676188848231414}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{84., -4.393671732221884}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{85., -4.935403274413745}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{86., -4.446621234876479}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{87., -5.371094160114913}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{88., -4.511414753559681}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{89., -5.955885466788862}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{90., -4.589262697429638}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{91., -5.309945127733492}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{92., -4.678799717083619}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{93., -5.135287776544917}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{94., -4.782161996921087}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{95., -5.060231832909849}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{96., -4.900380437648982}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{97., -5.033554068626609}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{98., -5.037438734693696}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{99., -5.03612834804492}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)], 
       Tooltip[{
         Opacity[0.], 
         Point[{{100., -5.199578362450597}}]}, (1 + 
         10000 (Pi^(-1) - $CellContext`x)^2)^(-1)]}}, {
     AspectRatio -> GoldenRatio^(-1), Axes -> True, AxesOrigin -> {0, 0}, 
      ImageSize -> {300}, PlotRange -> {{0, 100}, {0, -9}}, PlotRangeClipping -> 
      True, PlotRangePadding -> {Automatic, Automatic}}], " ", 
    Graphics[{{
       Text[$CellContext`x^Rational[1, 2], {110, -2}, {-1, 0}], 
       Text[
        Abs[(-1)/E + $CellContext`x], {110, -4}, {-1, 0}], 
       Text[
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], {110, -6}, {-1, 0}], 
       Text[(1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), {110, -8}, {-1, 
        0}]}, {
       Text["  ", {106, -2}, {-1, 0}, Background -> Hue[0.67, 0.6, 0.6]], 
       Text[
       "  ", {106, -4}, {-1, 0}, Background -> 
        Hue[0.9060679774997897, 0.6, 0.6]], 
       Text[
       "  ", {106, -6}, {-1, 0}, Background -> 
        Hue[0.14213595499957954`, 0.6, 0.6]], 
       Text[
       "  ", {106, -8}, {-1, 0}, Background -> 
        Hue[0.37820393249936934`, 0.6, 0.6]]}}, ImageSize -> {200, 200}, 
     AspectRatio -> 5]}]]], "Output",
 ImageSize->{511, 202},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[117]=",
 CellID->103941512]
}, Open  ]],

Cell[CellGroupData[{

Cell["Minimal number of sampling points", "Subsubsection",
 CellID->870923766],

Cell["\<\
Here is a function that finds the number of sampling points used in an \
integration.\
\>", "MathCaption",
 CellID->416488103],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"Attributes", "[", "SamplingPoints", "]"}], "=", 
   RowBox[{"{", "HoldFirst", "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"SamplingPoints", "[", "expr_", "]"}], ":=", "\[IndentingNewLine]", 
  
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"k", "=", "0"}], ",", "res"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"res", "=", 
      RowBox[{
       RowBox[{"Hold", "[", "expr", "]"}], "/.", "\[InvisibleSpace]", 
       RowBox[{
        RowBox[{"HoldPattern", "[", 
         RowBox[{"NIntegrate", "[", "s___", "]"}], "]"}], "\[RuleDelayed]", 
        RowBox[{"NIntegrate", "[", 
         RowBox[{"s", ",", 
          RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
           RowBox[{"k", "++"}]}]}], "]"}]}]}]}], ";", 
     RowBox[{"ReleaseHold", "[", "res", "]"}], ";", "k"}]}], 
   "]"}]}]}], "Input",
 CellLabel->"In[118]:=",
 CellID->305297791],

Cell["\<\
This finds the number of sampling points used for a range of precision goals \
and a range of integration rule coarse points.\
\>", "MathCaption",
 CellID->320095499],

Cell[BoxData[
 RowBox[{
  RowBox[{"tblga", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"pg", ",", "pnts", ",", 
         RowBox[{"SamplingPoints", "[", 
          RowBox[{"NIntegrate", "[", 
           RowBox[{"#", ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"Method", "->", 
             RowBox[{"{", 
              RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
               RowBox[{"\"\<SymbolicProcessing\>\"", "->", "0"}], ",", 
               RowBox[{"Method", "->", 
                RowBox[{"{", 
                 RowBox[{"\"\<GaussBerntsenEspelidRule\>\"", ",", 
                  RowBox[{"\"\<Points\>\"", "->", "pnts"}]}], "}"}]}]}], 
              "}"}]}], ",", 
            RowBox[{"MaxRecursion", "->", "100"}], ",", 
            RowBox[{"WorkingPrecision", "->", "35"}], ",", 
            RowBox[{"PrecisionGoal", "->", "pg"}]}], "]"}], "]"}]}], "}"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"pg", ",", "4", ",", "30"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"pnts", ",", "4", ",", "25"}], "}"}]}], "]"}], "&"}], "/@", 
    "funcs"}]}], ";"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->2205647],

Cell["\<\
This finds the for each precision the minimum total number of sampling \
points. This way the number of coarse integration rule points used is also \
found.\
\>", "MathCaption",
 CellID->147910307],

Cell[BoxData[
 RowBox[{
  RowBox[{"minPnts", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"#", "[", 
           RowBox[{"[", 
            RowBox[{
             RowBox[{"Position", "[", 
              RowBox[{"#", ",", 
               RowBox[{"Min", "[", 
                RowBox[{
                 RowBox[{
                  RowBox[{"#", "[", 
                   RowBox[{"[", "3", "]"}], "]"}], "&"}], "/@", "#"}], 
                "]"}]}], "]"}], "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}], "]"}], "]"}], "&"}], "[",
          "#", "]"}], "&"}], "/@", "#"}], ")"}], "&"}], "/@", "tblga"}]}], 
  ";"}]], "Input",
 CellLabel->"In[121]:=",
 CellID->190038355],

Cell["\<\
This is a plot of the precision goal and the number of integration rule \
points with which the minimum number of total sampling points was used.\
\>", "MathCaption",
 CellID->318278572],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"ListLinePlot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"Drop", "[", 
           RowBox[{"#", ",", 
            RowBox[{"-", "1"}]}], "]"}], "&"}], "/@", "#"}], ")"}], "&"}], "/@",
       "minPnts"}], ",", 
     RowBox[{"PlotRange", "->", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "30"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "26"}], "}"}]}], "}"}]}], ",", 
     RowBox[{"PlotStyle", "->", 
      RowBox[{"Thickness", "[", "0.003", "]"}]}], ",", 
     RowBox[{"AxesOrigin", "->", 
      RowBox[{"{", 
       RowBox[{"3", ",", "0"}], "}"}]}], ",", 
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", 
       RowBox[{"300", ",", "200"}], "}"}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xc", "=", "110"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"xcSq", "=", "106"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"legend", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "1", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "2"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "2", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "4"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "3", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "6"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{
       RowBox[{"funcs", "[", 
        RowBox[{"[", "4", "]"}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"xc", ",", 
         RowBox[{"-", "8"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"legendSq", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "2"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "4"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "6"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
     RowBox[{"Text", "[", 
      RowBox[{"\"\<  \>\"", ",", 
       RowBox[{"{", 
        RowBox[{"xcSq", ",", 
         RowBox[{"-", "8"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"legendSq", "=", 
   RowBox[{"MapThread", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Append", "[", 
       RowBox[{"#1", ",", 
        RowBox[{"Background", "->", "#2"}]}], "]"}], "&"}], ",", 
     RowBox[{"{", 
      RowBox[{"legendSq", ",", 
       RowBox[{"Cases", "[", 
        RowBox[{"gr", ",", 
         RowBox[{"Hue", "[", "s__", "]"}], ",", "\[Infinity]"}], "]"}]}], 
      "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Row", "[", 
  RowBox[{"{", 
   RowBox[{"gr", ",", "\"\< \>\"", ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", " ", 
       RowBox[{"legend", ",", "legendSq"}], "}"}], ",", 
      RowBox[{"ImageSize", "->", 
       RowBox[{"{", 
        RowBox[{"200", ",", "200"}], "}"}]}], ",", 
      RowBox[{"AspectRatio", "->", "5"}]}], "]"}]}], "}"}], "]"}]}], "Input",
 CellLabel->"In[122]:=",
 CellID->201325054],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   GraphicsBox[{{}, {}, 
     {Hue[0.67, 0.6, 0.6], Thickness[0.003], 
      LineBox[{{4., 11.}, {5., 21.}, {6., 4.}, {7., 4.}, {8., 4.}, {9., 5.}, {
       10., 6.}, {11., 6.}, {12., 7.}, {13., 8.}, {14., 8.}, {15., 9.}, {16., 
       10.}, {17., 10.}, {18., 11.}, {19., 11.}, {20., 12.}, {21., 13.}, {22.,
        13.}, {23., 14.}, {24., 15.}, {25., 15.}, {26., 16.}, {27., 17.}, {
       28., 17.}, {29., 18.}, {30., 18.}}]}, 
     {Hue[0.9060679774997897, 0.6, 0.6], Thickness[0.003], 
      LineBox[{{4., 14.}, {5., 4.}, {6., 4.}, {7., 5.}, {8., 4.}, {9., 4.}, {
       10., 4.}, {11., 4.}, {12., 4.}, {13., 4.}, {14., 4.}, {15., 4.}, {16., 
       4.}, {17., 4.}, {18., 4.}, {19., 4.}, {20., 4.}, {21., 4.}, {22., 
       4.}, {23., 4.}, {24., 4.}, {25., 4.}, {26., 4.}, {27., 4.}, {28., 
       4.}, {29., 4.}, {30., 4.}}]}, 
     {Hue[0.14213595499957954`, 0.6, 0.6], Thickness[0.003], 
      LineBox[{{4., 4.}, {5., 4.}, {6., 4.}, {7., 4.}, {8., 4.}, {9., 4.}, {
       10., 4.}, {11., 4.}, {12., 4.}, {13., 4.}, {14., 4.}, {15., 4.}, {16., 
       4.}, {17., 4.}, {18., 4.}, {19., 4.}, {20., 4.}, {21., 4.}, {22., 
       4.}, {23., 4.}, {24., 4.}, {25., 4.}, {26., 4.}, {27., 4.}, {28., 
       4.}, {29., 4.}, {30., 4.}}]}, 
     {Hue[0.37820393249936934`, 0.6, 0.6], Thickness[0.003], 
      LineBox[{{4., 5.}, {5., 6.}, {6., 10.}, {7., 12.}, {8., 10.}, {9., 
       11.}, {10., 12.}, {11., 13.}, {12., 14.}, {13., 15.}, {14., 17.}, {15.,
        18.}, {16., 19.}, {17., 20.}, {18., 21.}, {19., 22.}, {20., 23.}, {
       21., 23.}, {22., 25.}, {23., 24.}, {24., 25.}, {25., 21.}, {26., 
       20.}, {27., 25.}, {28., 22.}, {29., 23.}, {30., 24.}}]}},
    AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
    Axes->True,
    AxesOrigin->{3, 0},
    ImageSize->{300, 200},
    PlotRange->{{0, 30}, {0, 26}},
    PlotRangeClipping->True,
    PlotRangePadding->{Automatic, Automatic}], 
   "\[InvisibleSpace]", "\<\" \"\>", "\[InvisibleSpace]", 
   GraphicsBox[{{InsetBox[
       SqrtBox["x"], {110, -2}, {-1, 0}], InsetBox[
       RowBox[{"\[LeftBracketingBar]", 
        RowBox[{"x", "-", 
         FractionBox["1", "\[ExponentialE]"]}], 
        "\[RightBracketingBar]"}], {110, -4}, {-1, 0}], InsetBox[
       RowBox[{"\[Piecewise]", GridBox[{
          {"2", 
           RowBox[{"x", "\[LessEqual]", 
            FractionBox["1", "\[ExponentialE]"]}]},
          {"3", 
           RowBox[{"x", ">", 
            FractionBox["1", "\[ExponentialE]"]}]}
         },
         GridBoxAlignment->{
          "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{Baseline}}, "RowsIndexed" -> {}},
         GridBoxItemSize->{
          "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{1.}}, "RowsIndexed" -> {}},
         GridBoxSpacings->{"Columns" -> {
             Offset[0.27999999999999997`], {
              Offset[0.84]}, 
             Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, 
           "Rows" -> {
             Offset[0.2], {
              Offset[0.4]}, 
             Offset[0.2]}, "RowsIndexed" -> {}}]}], {110, -6}, {-1, 0}], 
      InsetBox[
       FractionBox["1", 
        RowBox[{
         RowBox[{"10000", " ", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{
             FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}], "+", 
         "1"}]], {110, -8}, {-1, 0}]}, {
      InsetBox["\<\"  \"\>", {106, -2}, {-1, 0},
       Background->Hue[0.67, 0.6, 0.6]], 
      InsetBox["\<\"  \"\>", {106, -4}, {-1, 0},
       Background->Hue[0.9060679774997897, 0.6, 0.6]], 
      InsetBox["\<\"  \"\>", {106, -6}, {-1, 0},
       Background->Hue[0.14213595499957954`, 0.6, 0.6]], 
      InsetBox["\<\"  \"\>", {106, -8}, {-1, 0},
       Background->Hue[0.37820393249936934`, 0.6, 0.6]]}},
    AspectRatio->5,
    ImageSize->{200, 200}]}],
  Row[{
    Graphics[{{{}, {}, {
        Hue[0.67, 0.6, 0.6], 
        Thickness[0.003], 
        Line[{{4., 11.}, {5., 21.}, {6., 4.}, {7., 4.}, {8., 4.}, {9., 5.}, {
         10., 6.}, {11., 6.}, {12., 7.}, {13., 8.}, {14., 8.}, {15., 9.}, {
         16., 10.}, {17., 10.}, {18., 11.}, {19., 11.}, {20., 12.}, {21., 
         13.}, {22., 13.}, {23., 14.}, {24., 15.}, {25., 15.}, {26., 16.}, {
         27., 17.}, {28., 17.}, {29., 18.}, {30., 18.}}]}, {
        Hue[0.9060679774997897, 0.6, 0.6], 
        Thickness[0.003], 
        Line[{{4., 14.}, {5., 4.}, {6., 4.}, {7., 5.}, {8., 4.}, {9., 4.}, {
         10., 4.}, {11., 4.}, {12., 4.}, {13., 4.}, {14., 4.}, {15., 4.}, {
         16., 4.}, {17., 4.}, {18., 4.}, {19., 4.}, {20., 4.}, {21., 4.}, {
         22., 4.}, {23., 4.}, {24., 4.}, {25., 4.}, {26., 4.}, {27., 4.}, {
         28., 4.}, {29., 4.}, {30., 4.}}]}, {
        Hue[0.14213595499957954`, 0.6, 0.6], 
        Thickness[0.003], 
        Line[{{4., 4.}, {5., 4.}, {6., 4.}, {7., 4.}, {8., 4.}, {9., 4.}, {
         10., 4.}, {11., 4.}, {12., 4.}, {13., 4.}, {14., 4.}, {15., 4.}, {
         16., 4.}, {17., 4.}, {18., 4.}, {19., 4.}, {20., 4.}, {21., 4.}, {
         22., 4.}, {23., 4.}, {24., 4.}, {25., 4.}, {26., 4.}, {27., 4.}, {
         28., 4.}, {29., 4.}, {30., 4.}}]}, {
        Hue[0.37820393249936934`, 0.6, 0.6], 
        Thickness[0.003], 
        Line[{{4., 5.}, {5., 6.}, {6., 10.}, {7., 12.}, {8., 10.}, {9., 
         11.}, {10., 12.}, {11., 13.}, {12., 14.}, {13., 15.}, {14., 17.}, {
         15., 18.}, {16., 19.}, {17., 20.}, {18., 21.}, {19., 22.}, {20., 
         23.}, {21., 23.}, {22., 25.}, {23., 24.}, {24., 25.}, {25., 21.}, {
         26., 20.}, {27., 25.}, {28., 22.}, {29., 23.}, {30., 24.}}]}}}, {
     AspectRatio -> GoldenRatio^(-1), Axes -> True, AxesOrigin -> {3, 0}, 
      ImageSize -> {300, 200}, PlotRange -> {{0, 30}, {0, 26}}, 
      PlotRangeClipping -> True, PlotRangePadding -> {Automatic, Automatic}}],
     " ", 
    Graphics[{{
       Text[$CellContext`x^Rational[1, 2], {110, -2}, {-1, 0}], 
       Text[
        Abs[(-1)/E + $CellContext`x], {110, -4}, {-1, 0}], 
       Text[
        Piecewise[{{2, $CellContext`x <= E^(-1)}, {
          3, $CellContext`x > E^(-1)}}, 0], {110, -6}, {-1, 0}], 
       Text[(1 + 10000 (Pi^(-1) - $CellContext`x)^2)^(-1), {110, -8}, {-1, 
        0}]}, {
       Text["  ", {106, -2}, {-1, 0}, Background -> Hue[0.67, 0.6, 0.6]], 
       Text[
       "  ", {106, -4}, {-1, 0}, Background -> 
        Hue[0.9060679774997897, 0.6, 0.6]], 
       Text[
       "  ", {106, -6}, {-1, 0}, Background -> 
        Hue[0.14213595499957954`, 0.6, 0.6]], 
       Text[
       "  ", {106, -8}, {-1, 0}, Background -> 
        Hue[0.37820393249936934`, 0.6, 0.6]]}}, ImageSize -> {200, 200}, 
     AspectRatio -> 5]}]]], "Output",
 ImageSize->{511, 202},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[128]=",
 CellID->87888892]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Rule comparison", "Subsection",
 CellID->870846129],

Cell[TextData[{
 "Here is function that calculates the error of a rule for the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " using the exact value by ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->164356145],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RuleErrors", "[", 
    RowBox[{"f_", ",", "rule_String", ",", "prec_", ",", 
     RowBox[{"pnts_", "?", "NumberQ"}]}], "]"}], ":=", "\[IndentingNewLine]", 
   
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"absc", ",", "weights", ",", "errweights", ",", "exact"}], 
      "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
       RowBox[{
        RowBox[{"ToExpression", "[", 
         RowBox[{"\"\<NIntegrate`\>\"", "<>", "rule", "<>", "\"\<Data\>\""}], 
         "]"}], "[", 
        RowBox[{"pnts", ",", "prec"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Abs", "[", 
         RowBox[{
          RowBox[{"Total", "[", 
           RowBox[{"MapThread", "[", 
            RowBox[{
             RowBox[{
              RowBox[{
               RowBox[{"f", "[", "#1", "]"}], "#2"}], "&"}], ",", 
             RowBox[{"{", 
              RowBox[{"absc", ",", "weights"}], "}"}]}], "]"}], "]"}], "-", 
          RowBox[{"Integrate", "[", 
           RowBox[{
            RowBox[{"f", "[", "x", "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", 
        
        RowBox[{"Abs", "[", 
         RowBox[{"Total", "[", 
          RowBox[{"MapThread", "[", 
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#2"}], "&"}], ",", 
            RowBox[{"{", 
             RowBox[{"absc", ",", "errweights"}], "}"}]}], "]"}], "]"}], 
         "]"}]}], "}"}]}]}], "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[129]:=",
 CellID->267469494],

Cell["This defines a list of functions.", "MathCaption",
 CellID->630579378],

Cell[BoxData[
 RowBox[{
  RowBox[{"funcs", "=", 
   RowBox[{"{", 
    RowBox[{
     SqrtBox["x"], ",", 
     RowBox[{"Abs", "[", 
      RowBox[{"x", "-", 
       FractionBox["1", "\[ExponentialE]"]}], "]"}], ",", 
     RowBox[{"\[Piecewise]", GridBox[{
        {"2", 
         RowBox[{"x", "\[LessEqual]", 
          FractionBox["1", "\[ExponentialE]"]}]},
        {"3", 
         RowBox[{"x", ">", 
          FractionBox["1", "\[ExponentialE]"]}]}
       },
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}},
       GridBoxItemSize->{
        "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
         "RowsIndexed" -> {}},
       GridBoxSpacings->{"Columns" -> {
           Offset[0.28], {
            Offset[0.84]}, 
           Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
           Offset[0.2], {
            Offset[0.4]}, 
           Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
     FractionBox["1", 
      RowBox[{
       RowBox[{
        SuperscriptBox["10", "4"], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           FractionBox["1", "\[Pi]"], "-", "x"}], ")"}], "2"]}], "+", 
       "1"}]]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[130]:=",
 CellID->888976217],

Cell[TextData[{
 "Here are plots of the functions in the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->537957296],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Plot", "[", 
      RowBox[{"#", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"PlotRange", "->", "All"}], ",", 
       RowBox[{"Frame", "->", "True"}], ",", 
       RowBox[{"FrameTicks", "->", 
        RowBox[{"{", 
         RowBox[{"None", ",", "Automatic"}], "}"}]}], ",", 
       RowBox[{"ImageSize", "->", 
        RowBox[{"{", 
         RowBox[{"120", ",", "120"}], "}"}]}]}], "]"}], "&"}], "/@", 
    "funcs"}], ",", "\"\<  \>\""}], "]"}]], "Input",
 CellLabel->"In[131]:=",
 CellID->86280932],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{498, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[131]=",
 CellID->842272163]
}, Open  ]],

Cell[TextData[{
 "This is the computation of the errors of ",
 Cell[BoxData["\"\<GaussKronrodRule\>\""], "InlineFormula"],
 StyleBox[", ", "Text"],
 Cell[BoxData["\"\<LobattoKronrodRule\>\""], "InlineFormula"],
 StyleBox[", ", "Text"],
 Cell[BoxData["\"\<TrapezoidalRule\>\""], "InlineFormula"],
 StyleBox[", and ", "Text"],
 Cell[BoxData["\"\<ClenshawCurtisRule\>\""], "InlineFormula"],
 " for each of the integrals ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SqrtBox["x"], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"\[LeftBracketingBar]", 
      RowBox[{"x", "-", 
       FractionBox["1", "2"]}], "\[RightBracketingBar]"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ",",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{"\[Piecewise]", GridBox[{
        {"2", 
         RowBox[{"x", "\[LessEqual]", 
          FractionBox["1", "2"]}]},
        {"3", 
         RowBox[{"x", ">", 
          FractionBox["1", "2"]}]}
       },
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}},
       GridBoxItemSize->{
        "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
         "RowsIndexed" -> {}},
       GridBoxSpacings->{"Columns" -> {
           Offset[0.28], {
            Offset[0.84]}, 
           Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
           Offset[0.2], {
            Offset[0.4]}, 
           Offset[0.2]}, "RowsIndexed" -> {}}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     FractionBox["1", 
      RowBox[{
       RowBox[{
        SuperscriptBox["10", "4"], " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           FractionBox["1", "3"], "-", "x"}], ")"}], "2"]}], "+", "1"}]], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " for a range of points."
}], "MathCaption",
 CellID->121789434],

Cell[BoxData[{
 RowBox[{
  RowBox[{"rules", "=", 
   RowBox[{"{", 
    RowBox[{
    "\"\<GaussKronrodRule\>\"", ",", "\"\<LobattoKronrodRule\>\"", ",", 
     "\"\<TrapezoidalRule\>\"", ",", "\"\<ClenshawCurtisRule\>\""}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"errors", "=", 
   RowBox[{"Outer", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"pnts", ",", 
          RowBox[{"RuleErrors", "[", 
           RowBox[{"#2", ",", "#1", ",", "30", ",", "pnts"}], "]"}]}], "}"}], 
        ",", 
        RowBox[{"{", 
         RowBox[{"pnts", ",", "4", ",", "100", ",", "1"}], "}"}]}], "]"}], 
      "&"}], ",", "rules", ",", 
     RowBox[{"Function", "/@", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"Function", "[", 
         RowBox[{
          RowBox[{"{", "f", "}"}], ",", 
          RowBox[{"f", "/.", 
           RowBox[{"x", "->", "#"}]}]}], "]"}], "/@", "funcs"}], ")"}]}]}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"exactErrors", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"#", "[", 
       RowBox[{"[", "1", "]"}], "]"}], "&"}], ",", "errors", ",", 
     RowBox[{"{", 
      RowBox[{"-", "2"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ruleErrors", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"#", "[", 
       RowBox[{"[", "2", "]"}], "]"}], "&"}], ",", "errors", ",", 
     RowBox[{"{", 
      RowBox[{"-", "2"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[132]:=",
 CellID->299139976],

Cell["\<\
Here are plots of how the logarithms of the errors decrease for each rule and \
each function.\
\>", "MathCaption",
 CellID->41515502],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Grid", "[", 
     RowBox[{
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"\"\<exact errors\>\"", ",", "\"\<error estimates\>\""}], 
          "}"}], "}"}], ",", 
        RowBox[{"Flatten", "[", 
         RowBox[{
          RowBox[{"Transpose", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{
               RowBox[{"{", 
                RowBox[{"#", ",", "SpanFromLeft"}], "}"}], "&"}], "/@", 
              "rules"}], ",", 
             RowBox[{"Transpose", "[", 
              RowBox[{"Map", "[", "\[IndentingNewLine]", 
               RowBox[{
                RowBox[{"Function", "[", 
                 RowBox[{
                  RowBox[{"{", "d", "}"}], ",", "\[IndentingNewLine]", 
                  RowBox[{"(", 
                   RowBox[{
                    RowBox[{"gr", "=", 
                    RowBox[{"ListLinePlot", "[", 
                    RowBox[{
                    RowBox[{"Map", "[", 
                    RowBox[{
                    RowBox[{
                    RowBox[{
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"#", "[", 
                    RowBox[{"[", "1", "]"}], "]"}], ",", 
                    RowBox[{"Log", "[", 
                    RowBox[{"10", ",", 
                    RowBox[{"#", "[", 
                    RowBox[{"[", "2", "]"}], "]"}]}], "]"}]}], "}"}], "&"}], "/@",
                     "#"}], "&"}], ",", "d"}], "]"}], ",", 
                    RowBox[{"ImageSize", "->", 
                    RowBox[{"{", 
                    RowBox[{"200", ",", "100"}], "}"}]}], ",", 
                    RowBox[{"PlotRange", "->", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{"0", ",", "100"}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"0", ",", 
                    RowBox[{"-", "9"}]}], "}"}]}], "}"}]}], ",", 
                    RowBox[{"AxesOrigin", "->", 
                    RowBox[{"{", 
                    RowBox[{"0", ",", "0"}], "}"}]}]}], "]"}]}], ";", "\n", 
                    RowBox[{"xc", "=", "110"}], ";", "\n", 
                    RowBox[{"xcSq", "=", "106"}], ";", "\n", 
                    RowBox[{"legend", "=", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"Text", "[", 
                    RowBox[{
                    RowBox[{"funcs", "[", 
                    RowBox[{"[", "1", "]"}], "]"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"xc", ",", 
                    RowBox[{"-", "1.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
                    RowBox[{"Text", "[", 
                    RowBox[{
                    RowBox[{"funcs", "[", 
                    RowBox[{"[", "2", "]"}], "]"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"xc", ",", 
                    RowBox[{"-", "3.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
                    RowBox[{"Text", "[", 
                    RowBox[{
                    RowBox[{"funcs", "[", 
                    RowBox[{"[", "3", "]"}], "]"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"xc", ",", 
                    RowBox[{"-", "5.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
                    RowBox[{"Text", "[", 
                    RowBox[{
                    RowBox[{"funcs", "[", 
                    RowBox[{"[", "4", "]"}], "]"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"xc", ",", 
                    RowBox[{"-", "7.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}]}], "}"}]}],
                     ";", "\n", 
                    RowBox[{"legendSq", "=", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"Text", "[", 
                    RowBox[{"\"\<  \>\"", ",", 
                    RowBox[{"{", 
                    RowBox[{"xcSq", ",", 
                    RowBox[{"-", "1.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
                    RowBox[{"Text", "[", 
                    RowBox[{"\"\<  \>\"", ",", 
                    RowBox[{"{", 
                    RowBox[{"xcSq", ",", 
                    RowBox[{"-", "3.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
                    RowBox[{"Text", "[", 
                    RowBox[{"\"\<  \>\"", ",", 
                    RowBox[{"{", 
                    RowBox[{"xcSq", ",", 
                    RowBox[{"-", "5.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
                    RowBox[{"Text", "[", 
                    RowBox[{"\"\<  \>\"", ",", 
                    RowBox[{"{", 
                    RowBox[{"xcSq", ",", 
                    RowBox[{"-", "7.5"}]}], "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}]}], "}"}]}],
                     ";", "\n", 
                    RowBox[{"legendSq", "=", 
                    RowBox[{"MapThread", "[", 
                    RowBox[{
                    RowBox[{
                    RowBox[{"Append", "[", 
                    RowBox[{"#1", ",", 
                    RowBox[{"Background", "->", "#2"}]}], "]"}], "&"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"legendSq", ",", 
                    RowBox[{"Cases", "[", 
                    RowBox[{"gr", ",", 
                    RowBox[{"Hue", "[", "s__", "]"}], ",", "\[Infinity]"}], 
                    "]"}]}], "}"}]}], "]"}]}], ";", "\n", "gr"}], ")"}]}], 
                 "]"}], ",", 
                RowBox[{"{", 
                 RowBox[{"exactErrors", ",", "ruleErrors"}], "}"}], ",", 
                RowBox[{"{", "2", "}"}]}], "]"}], "]"}]}], "}"}], "]"}], ",", 
          "1"}], "]"}]}], "]"}], ",", 
      RowBox[{"Dividers", "->", "All"}]}], "]"}], ",", 
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"legend", ",", "legendSq"}], "}"}], ",", 
      RowBox[{"ImageSize", "->", 
       RowBox[{"{", 
        RowBox[{"200", ",", "200"}], "}"}]}], ",", 
      RowBox[{"AspectRatio", "->", "5"}]}], "]"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[136]:=",
 CellID->640793172],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{425, 717},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[136]=",
 CellID->286380649]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples of Pathological Behavior", "Section",
 CellTags->"ExamplesOfPathologicalBehavior",
 CellID->8354729],

Cell[CellGroupData[{

Cell["Tricking the error estimator", "Subsection",
 CellTags->"Tricking the error estimator",
 CellID->81420002],

Cell[TextData[{
 "In this subsection will be discussed an integral which ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " underestimates with its default settings since it fails to detect part of \
the integrand. The part is detected if the precision goal is increased."
}], "Text",
 CellID->65273665],

Cell[CellGroupData[{

Cell["The wrong estimation", "Subsubsection",
 CellID->594163809],

Cell["Consider the following function.", "Text",
 CellID->587795730],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  RowBox[{
   RowBox[{
    RowBox[{"Sech", "[", 
     RowBox[{"10", "*", 
      RowBox[{"(", 
       RowBox[{"x", "-", "0.2"}], ")"}]}], "]"}], "^", "2"}], "+", 
   RowBox[{
    RowBox[{"Sech", "[", 
     RowBox[{"100", "*", 
      RowBox[{"(", 
       RowBox[{"x", "-", "0.4"}], ")"}]}], "]"}], "^", "4"}], "+", 
   RowBox[{
    RowBox[{"Sech", "[", 
     RowBox[{"1000", "*", 
      RowBox[{"(", 
       RowBox[{"x", "-", "0.6"}], ")"}]}], "]"}], "^", "6"}]}]}]], "Input",
 CellLabel->"In[13]:=",
 CellID->957759846],

Cell[TextData[{
 "Here is its exact integral over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->18894011],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[138]:=",
 CellID->283837762],

Cell[BoxData["0.21080273550054926`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[138]=",
 CellID->295853054]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " gives the estimate."
}], "Text",
 CellID->205592170],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"est", "=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[139]:=",
 CellID->314533975],

Cell[BoxData["0.20973606883567553`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[139]=",
 CellID->19696595]
}, Open  ]],

Cell["This is too inaccurate when compared to the exact value.", "Text",
 CellID->294095742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"exact", "-", "est"}], "]"}]], "Input",
 CellLabel->"In[140]:=",
 CellID->389546213],

Cell[BoxData["0.0010666666648737333`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[140]=",
 CellID->500929119]
}, Open  ]],

Cell["\<\
Here is the plot of function, which is also wrong (see below).\
\>", "Text",
 CellID->350117999],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[141]:=",
 CellID->642089480],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[141]=",
 CellID->92603440]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Better results", "Subsubsection",
 CellID->26508302],

Cell[TextData[{
 "Better results can be achieved using ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s option ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " option and increasing the recursion depth."
}], "Text",
 CellID->95164517],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "20"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "12"}]}], "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->288824240],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[17]:=",
 CellID->9393131],

Cell[BoxData["0.2108027355005495`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->940999677]
}, Open  ]],

Cell["\<\
This is table that finds the precision goal for which no good result are \
computed.\
\>", "Text",
 CellID->769920255],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"pg", ",", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"f", "[", "x", "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}], ",", 
       RowBox[{"MaxRecursion", "\[Rule]", "20"}], ",", 
       RowBox[{"PrecisionGoal", "\[Rule]", "pg"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"pg", ",", "6", ",", "12"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->560881033],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[18]:=",
 CellID->617882725],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[18]:=",
 CellID->524580207],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[18]:=",
 CellID->50110422],

Cell[BoxData[
 RowBox[{
  RowBox[{"General", "::", "\<\"stop\"\>"}], ":", 
  " ", "\<\"Further output of \\!\\(NIntegrate :: \\\"slwcon\\\"\\) will be \
suppressed during this calculation. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\
\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/stop\\\", ButtonNote -> \
\\\"General::stop\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[18]:=",
 CellID->503101152],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"6", ",", "0.20973606883567553`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"7", ",", "0.20973606883391976`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"8", ",", "0.2097360688339216`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"9", ",", "0.2108027355005495`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"10", ",", "0.21080273550054957`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"11", ",", "0.2108027355005495`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"12", ",", "0.2108027355005495`"}], "}"}]}], "}"}]], "Output",
 ImageSize->{426, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->822624168]
}, Open  ]],

Cell["\<\
If the plot points are increased, the plot of the function looks different.\
\>", "Text",
 CellID->17871884],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "100"}]}], "]"}]], "Input",
 CellLabel->"In[144]:=",
 CellID->64721228],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1WkuIHEUY7kzPTvb9yM7Ozm42utGoiYqI4ilqhPhARMRD8CAe1kXYPWgk
WS960Yte9KIXA2IQo+DjEDFEhAhiQAyIATEgKkgCEg8GweBhQVi/qr+6/9qv
e3qqe2aHZLaqq+r7/vpf9eh5dGlt5Zlnl9ZWl5cWHzqy9PzK6vLRxQcPH8Gj
eFsURafw/2g7QnkDJfv/1MYG/ma/9+G7hg6xqcWmp3ka2eKM3zYgbfKJuDVO
W6Nal7YWCuZRLafN9o/2mu+6tkTRAEnXzPRoENqOTI965sl0QJ8W4caZHrUO
MrNE+Wz1QDm2VtbtYdortECIzMaOxs6zWyhpwpEvYRwoe3VdZrGyfJ1lHCDe
fto9ifWG+9tyf1mn097zRMq8cUldotbPKpOE+F7B0zF8n0hRuA/rg+v1kv3L
1i3+aFq/8NOfn3z842uvfn34uc8fOPgOnj/+2Psov7B2un783e/tyPM//IF/
Ud18xeiNR1+d+c3Vr1z5V0yS0Xq/Z5KLl9HvAtV3Ud1aZQhSYyIvvfjlgXuO
ra6cfOvNb784/TPms77+XzRhu5nJRdHly/9kEJghX477bBNQXnn5DFQKhrPf
/G7XEYsNXoN9f+7YA7YO9UM0o+l2x1HGW6U30DEdU5wr5sAUTfFeW8cQDAwY
hY5Xr667UZiPM3vBKBOv5767BF/CwLttPxQeefi4Kc4XziiKf/3lLxkY7bc9
L178+6knPyoe6a3ooozoroR1+elPoUq3emMssBUL2FmsRoqFD4wHb7nTckB9
iA1gABNSulUfE8VTxQRaFnO7jwkciGQQ7jCztrGGArzFWFxQ4ZOIzC6o/o4j
mQ9kM558u7UDQhxM0sXwSK8PTpw/9va5Yuyay85mlMVqyKQFS6SrufadQRKK
BSAbrGl63Wbrn5288MbrZ9MZ1FNtFmK7BGs/ebs9zhd1ipNbE1meOPQh3Fkw
Gql34IPpYZId2Av3c/m5L9VedIutw6ngWqleGr6lwrl5X2aw4lQmne/NNF8r
YwOoqMtQyQYhnJzree/boPnuo/lmuaF0qL5/3Bo7wg2HQ2Aod5qLSnFn9yps
6QZpeS9ZwbYPgMKnjrp7eLI34VVRtIFQRkCb4k22jvBBEOXwiRo28RkpN8f8
oG3Fc8zB5JEbk95iLeFUTDw1MvuYppifAQRb150bSH7BFshq2MifyKKmuIds
MdgzNrCw+Jl1aQ/ZtSy20fOwbZVIN8XrbV2WAMVESkBiQFGW3UB5h2xd17rr
Evnh9camtr0u7ocisBaCcEVmgEAJpribdN4brupiN+miN1y10yLZcbgErrHZ
COlxkfTs40mOCZNzhOLiWluXnYfiij4kEvNxM6ti5p6DM4gwa24WZs2XwiyW
KWTuuiplWTWnXEP1kUqsyQ1EcjIetWPSqHR7eZ1rdRZ/tWeNjpLtdpFthRVb
Hmx8wlmTM3Y+m3r4AnmoaZf9pe6i83NIGT7FFz6NAOXTZx34PF/lVZ39ZczW
0zh1iLpuhLMmkTzuRoj8ulcph5SVkc8XeuoZ6xFZdT5PNrftddnzCHZY1hkn
rc6R55bBNTqdcDOcJ532Jp9qq00aHfdwZQ0qg6v6bJM+fVz5DsOdIH3Okj57
wTV6bZNey+AZ+0ySPlukh4me5q12aRGP4IqFinA7nelMMS8XTZI+5H5F87vw
ihzyNDi/F+7yJ8muwqsZ3ueF2dIsU4U3Tu2mZxrh011YP/my89RVuUn1ycq8
cY41p2xdb5emadbV2LrdI06R7YRVVudSrLSa1XL0OUX+OU3+G8qVxLN9W+DF
mdQ1DqdK4GXl1LzAPIIr9xVlcHfQfA2OyWvy1kP8qjyexiHbcsrDlb1CsT7F
HnqaETyNMx9PUmQZOSluzC0nVJG2V8AVeXV/K3lY79N6w9XTl+Dq6aw3XMrP
nvzSDggBjYqt1SQtjpOWm8FovoxNsrnsvTQLpajurrY/K1mTPHeMPDtldTdG
/cnvTbKHsGp89o81zrGXnCQ0W81UYOt0SuEVUs6busOrwtVdn7wPGCErbiUr
nWozu6KurAG3jTNkwWGyoOz7sILjnNmZKYnfWbLRENnIvpdFLNuVx0kw3xHR
l7NFdhgiO0g7oN19a1hmkFH65kRQ9aakF1TNWoNUnw1GNTptJ71dRpVzta4I
4Wi+jDKKbmm9W2Zph3tVQE3ferg7IzgDHEhPZdVQ9daX3wdUQ22TBw0kKO6k
56OaYrGV5Jyt9zF1iqu5YDRfxjny+phkDkMVYZJP3q+aOCvMkfQ1z4P1toJ5
0bKJl5myv/rgvGdxG+muBB/xyJ3EJ/dv6fuK6GAvfO4NMFgxW91VYDOwiU1M
UJbNdFhI/NVlb6mDz+XY5YrS709wXF7pL07I6iEj4Y/O62VkEhPyGwFs/d0Z
Q1uzGEhDAb0QX27/mt+r6Fc80bb/AdrqJx4=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[144]=",
 CellID->48052908]
}, Open  ]],

Cell[TextData[{
 "Here is the zoomed plot of the spike that ",
 Cell[BoxData[
  ButtonBox["Plot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Plot"]], "InlineFormula"],
 " is missing with the default options."
}], "Text",
 CellID->576771419],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"eps", "=", "0.0015"}], ";", 
  RowBox[{"Plot", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"0.6", "-", "eps"}], ",", 
      RowBox[{"0.6", "+", "eps"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[145]:=",
 CellID->64999389],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[145]=",
 CellID->246760415]
}, Open  ]],

Cell[TextData[{
 "If this part of the function is integrated, the result fits the quantity \
that is \"lost\" (or \"missed\") by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with the default option settings."
}], "Text",
 CellID->346156256],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"0.6", "-", "eps"}], ",", 
     RowBox[{"0.6", "+", "eps"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[146]:=",
 CellID->366120302],

Cell[BoxData["0.0010685732905488772`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[146]=",
 CellID->460257124]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Abs", "[", 
  RowBox[{"exact", "-", "est"}], "]"}]], "Input",
 CellLabel->"In[147]:=",
 CellID->744098211],

Cell[BoxData["0.0010666666648737333`"], "Output",
 ImageSize->{74, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[147]=",
 CellID->193582369]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Why the estimator is misled", "Subsubsection",
 CellID->276439645],

Cell[TextData[{
 "These are the abscissas and weights of a Gauss\[Dash]Kronrod rule used by \
default by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "."
}], "Text",
 CellID->26596856],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
   RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
    RowBox[{"5", ",", "MachinePrecision"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[146]:=",
 CellID->20167425],

Cell["This defines a function for application of the rule above.", "Text",
 CellID->788890286],

Cell[BoxData[
 RowBox[{
  RowBox[{"IRuleEstimate", "[", 
   RowBox[{"f_", ",", 
    RowBox[{"{", 
     RowBox[{"a_", ",", "b_"}], "}"}]}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"integral", ",", "error"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"integral", ",", "error"}], "}"}], "=", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}], 
       RowBox[{"Total", "@", 
        RowBox[{"MapThread", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Rescale", "[", 
             RowBox[{"absc", ",", 
              RowBox[{"{", 
               RowBox[{"0", ",", "1"}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"a", ",", "b"}], "}"}]}], "]"}], ",", "weights", ",", 
            "errweights"}], "}"}]}], "]"}]}]}]}], ";", "\[IndentingNewLine]", 
     
     RowBox[{"{", 
      RowBox[{"integral", ",", 
       RowBox[{"Abs", "[", "error", "]"}]}], "}"}]}]}], "\[IndentingNewLine]",
    "]"}]}]], "Input",
 CellLabel->"In[147]:=",
 CellID->387478236],

Cell["\<\
This finds the points at which the adaptive strategy samples the integrand.\
\>", "Text",
 CellID->595901578],

Cell[BoxData[
 RowBox[{
  RowBox[{"cTbl", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"f", "[", "x", "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]",
         
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
     RowBox[{"[", "2", "]"}], "]"}], "//", "Flatten"}]}], ";"}]], "Input",
 CellLabel->"In[148]:=",
 CellID->47765883],

Cell["\<\
This is a plot of the sampling points. The vertical axis is for the order at \
which the points have been used to evaluate the integrand. \
\>", "Text",
 CellID->336487250],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"cTbl", ",", 
      RowBox[{"Range", "[", 
       RowBox[{"1", ",", 
        RowBox[{"Length", "[", "cTbl", "]"}]}], "]"}]}], "}"}], "]"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "0.5"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", 
        RowBox[{"Length", "[", "cTbl", "]"}]}], "}"}]}], "}"}]}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Hue", "[", "0.7", "]"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[149]:=",
 CellID->807428914],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.7], PointBox[CompressedData["
1:eJw1l3tszXcYxo+uE5Uto5pFZJtDqmND6m5j+qK0ekEvejl6OffTc9rT039W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     "]]}, {}},
  AspectRatio->0.5,
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0, 1}, {0, 253}},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[149]=",
 CellID->1479857]
}, Open  ]],

Cell[TextData[{
 "It can be seen on the plot above that ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " works a lot around ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0.4"}], TraditionalForm]], "InlineMath"],
 ", where the top of the second spike is. And it can be seen that the last \
set of points is over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0.6"}], TraditionalForm]], "InlineMath"],
 " where the top of the unintegrated spike is\[LongDash]",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " does not work much around that point, as for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0.4"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->100168402],

Cell[TextData[{
 "These are Gauss\[Dash]Kronrod and Gauss abscissas in the last set of \
sampling points, which is over the region ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0.5", ",", "0.75"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->35774155],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"gk", "=", " ", 
  RowBox[{"Sort", "[", 
   RowBox[{"Take", "[", 
    RowBox[{"cTbl", ",", 
     RowBox[{"-", "11"}]}], "]"}], "]"}]}], "\n", 
 RowBox[{"g", "=", 
  RowBox[{"Take", "[", 
   RowBox[{"gk", ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", 
      RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]}]}], "Input",
 CellLabel->"In[150]:=",
 CellID->425392635],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.5019893299881447`", ",", "0.511727519257667`", ",", 
   "0.5307291591786438`", ",", "0.5576913362367897`", ",", 
   "0.5900461983547771`", ",", "0.625`", ",", "0.6599538016452229`", ",", 
   "0.6923086637632103`", ",", "0.7192708408213562`", ",", 
   "0.738272480742333`", ",", "0.7480106700118553`"}], "}"}]], "Output",
 ImageSize->{389, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[150]=",
 CellID->266628682],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.511727519257667`", ",", "0.5576913362367897`", ",", "0.625`", ",", 
   "0.6923086637632103`", ",", "0.738272480742333`"}], "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[151]=",
 CellID->154930106]
}, Open  ]],

Cell["Here the integrand is applied over the abscissas.", "Text",
 CellID->7317078],

Cell[BoxData[{
 RowBox[{
  RowBox[{"fgk", "=", 
   RowBox[{"f", "/@", "gk"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"fg", "=", 
   RowBox[{"f", "/@", "g"}]}], ";"}]}], "Input",
 CellLabel->"In[152]:=",
 CellID->23331306],

Cell["\<\
Here is a polynomial approximation of the integrand over the abscissas.\
\>", "Text",
 CellID->308356064],

Cell[BoxData[{
 RowBox[{
  RowBox[{"gkf", "[", "x_", "]"}], ":=", 
  RowBox[{"Evaluate", "[", 
   RowBox[{"InterpolatingPolynomial", "[", 
    RowBox[{
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"gk", ",", "fgk"}], "}"}], "]"}], ",", "x"}], "]"}], 
   "]"}]}], "\n", 
 RowBox[{
  RowBox[{"gf", "[", "x_", "]"}], ":=", 
  RowBox[{"Evaluate", "[", 
   RowBox[{"InterpolatingPolynomial", "[", 
    RowBox[{
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"g", ",", "fg"}], "}"}], "]"}], ",", "x"}], "]"}], 
   "]"}]}]}], "Input",
 CellLabel->"In[154]:=",
 CellID->185124695],

Cell[TextData[{
 "These plots show that the two polynomial approximations almost coincide \
over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0.6"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->377548952],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"gkf", "[", "x", "]"}], ",", 
     RowBox[{"gf", "[", "x", "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"Min", "[", "gk", "]"}], ",", 
     RowBox[{"Max", "[", "gk", "]"}]}], "}"}]}], "]"}], "\n", 
 RowBox[{
  RowBox[{"eps", "=", "0.01"}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"gkf", "[", "x", "]"}], ",", 
     RowBox[{"gf", "[", "x", "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"0.6", "-", "eps"}], ",", 
     RowBox[{"0.6", "+", "eps"}]}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[156]:=",
 CellID->733072035],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[156]=",
 CellID->57978769],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[158]=",
 CellID->72932622]
}, Open  ]],

Cell[TextData[{
 "If the polynomials are integrated over the region where ",
 Cell[BoxData[
  FormBox["0.6", TraditionalForm]], "InlineMath"],
 " is placed, the difference between them, which ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses as an error estimate, is really small."
}], "Text",
 CellID->63256172],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"gkf", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0.5", ",", "0.75"}], "}"}]}], "]"}], "\n", 
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"gf", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0.5", ",", "0.75"}], "}"}]}], "]"}], "\n", 
 RowBox[{
  RowBox[{"%", "-", "%%"}], "//", "FullForm"}]}], "Input",
 CellLabel->"In[159]:=",
 CellID->413656717],

Cell[BoxData["0.0004911843469210808`"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[159]=",
 CellID->494884295],

Cell[BoxData["0.0004911839803963813`"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[160]=",
 CellID->510395348],

Cell[BoxData[
 TagBox[
  StyleBox[
   RowBox[{"-", "3.6652469947995314`*^-10"}],
   ShowSpecialCharacters->False,
   ShowStringCharacters->True,
   NumberMarks->True],
  FullForm]], "Output",
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[161]//FullForm=",
 CellID->290260353]
}, Open  ]],

Cell[TextData[{
 "Since the difference above is the error estimate assigned for the region ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0.5", ",", "0.75"}], "]"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with the default precision goal never picks it up for further integration \
refinement."
}], "Text",
 CellID->592263786]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Phase errors ", "Subsection",
 CellTags->"Phase Errors",
 CellID->599865576],

Cell[TextData[{
 "In this subsection are discussed causes why integration rules might \
seriously underestimate or overestimate the actual error of their integral \
estimates. Similar discussion is given in [",
 ButtonBox["LynKag76",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#406642067"],
 "]."
}], "Text",
 CellID->471141279],

Cell["This defines a function.", "Text",
 CellID->220215642],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{"x_", ",", "\[Lambda]_", ",", "\[Mu]_"}], "]"}], ":=", 
  FractionBox[
   SuperscriptBox["10", 
    RowBox[{"-", "\[Mu]"}]], 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"x", "-", "\[Lambda]"}], ")"}], "2"], "+", 
    SuperscriptBox["10", 
     RowBox[{
      RowBox[{"-", "2"}], "\[Mu]"}]]}]]}]], "Input",
 CellLabel->"In[162]:=",
 CellID->288788974],

Cell[TextData[{
 "Consider the numerical and symbolic evaluations of the integral of ",
 Cell[BoxData[
  RowBox[{"f", "[", 
   RowBox[{"x", ",", "0.415", ",", "1.25"}], "]"}]], "InlineFormula"],
 " over the region ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->642958260],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"num", "=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"x", ",", "0.415", ",", "1.25"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "->", "2"}]}], "]"}]}]], "Input",
 CellLabel->"In[163]:=",
 CellID->20537278],

Cell[BoxData["1.7229514480420216`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[163]=",
 CellID->268377181]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"x", ",", "0.415", ",", "1.25"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[164]:=",
 CellID->58834672],

Cell[BoxData[
 RowBox[{"3.006039848056518`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[164]=",
 CellID->278366612]
}, Open  ]],

Cell[TextData[{
 "They differ significantly. The precision goal requested is 2, but relative \
error is much higher than ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "2"}]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->154978428],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"num", "-", "exact"}], "]"}], "/", 
  RowBox[{"Abs", "[", "exact", "]"}]}]], "Input",
 CellLabel->"In[165]:=",
 CellID->9700234],

Cell[BoxData["0.4268367902188808`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[165]=",
 CellID->980781]
}, Open  ]],

Cell[TextData[{
 "(Note that ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " gives right results for higher precision goals.)"
}], "Text",
 CellID->702892424],

Cell["Below is an explanation why this happens.", "Text",
 CellID->115797403],

Cell[TextData[{
 "Let the integration rule ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "2"], TraditionalForm]], "InlineMath"],
 " be imbedded in the rule ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["R", "1"], TraditionalForm]], "InlineMath"],
 ". Accidentally, the error estimate ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{
     RowBox[{
      SuperscriptBox[
       SubscriptBox["R", "1"], "V"], "[", "f", "]"}], "-", 
     RowBox[{
      SuperscriptBox[
       SubscriptBox["R", "2"], "V"], "[", "f", "]"}]}], "|"}], 
   TraditionalForm]], "InlineMath"],
 " of the integral estimate ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     SubscriptBox["R", "1"], "V"], "[", "f", "]"}], TraditionalForm]], 
  "InlineMath"],
 ",",
 " where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"V", "=", 
    RowBox[{"[", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "1"}], "]"}]}], TraditionalForm]], 
  "InlineMath"],
 ", can be too small compared with the exact error ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{
     RowBox[{
      SuperscriptBox[
       SubscriptBox["R", "2"], "V"], "[", "f", "]"}], "-", 
     RowBox[{
      SubscriptBox["\[Integral]", "V"], 
      RowBox[{
       RowBox[{"f", "(", "x", ")"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}], "|"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->43904657],

Cell[TextData[{
 "To demonstrate this, consider the Gauss\[Dash]Kronrod rule ",
 Cell[BoxData[
  RowBox[{"GK", "[", 
   RowBox[{"f", ",", "5"}], "]"}]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox["11", TraditionalForm]], "InlineMath"],
 " sampling points that has an imbedded Gauss rule ",
 StyleBox["G[f,5]", "InlineCode"],
 " with ",
 Cell[BoxData[
  FormBox["5", TraditionalForm]], "InlineMath"],
 " sampling points. (This is the rule used in the two integrations above.) "
}], "Text",
 CellID->117125857],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
   RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
    RowBox[{"5", ",", "MachinePrecision"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[166]:=",
 CellID->173926475],

Cell["This defines a function that applies the rule.", "MathCaption",
 CellID->83936180],

Cell[BoxData[
 RowBox[{
  RowBox[{"IRuleEstimate", "[", 
   RowBox[{"f_", ",", 
    RowBox[{"{", 
     RowBox[{"a_", ",", "b_"}], "}"}]}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"integral", ",", "error"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"integral", ",", "error"}], "}"}], "=", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}], 
       RowBox[{"Total", "@", 
        RowBox[{"MapThread", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Rescale", "[", 
             RowBox[{"absc", ",", 
              RowBox[{"{", 
               RowBox[{"0", ",", "1"}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"a", ",", "b"}], "}"}]}], "]"}], ",", "weights", ",", 
            "errweights"}], "}"}]}], "]"}]}]}]}], ";", "\[IndentingNewLine]", 
     
     RowBox[{"{", 
      RowBox[{"integral", ",", 
       RowBox[{"Abs", "[", "error", "]"}]}], "}"}]}]}], "\[IndentingNewLine]",
    "]"}]}]], "Input",
 CellLabel->"In[167]:=",
 CellID->107661966],

Cell[TextData[{
 "This is the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " of ",
 Cell[BoxData[
  RowBox[{"f", "[", 
   RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}]], "InlineFormula"],
 " defined above."
}], "MathCaption",
 CellID->155242890],

Cell[BoxData[
 RowBox[{
  RowBox[{"exact", "=", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"ArcTan", "[", 
      RowBox[{
       SuperscriptBox["10", "\[Mu]"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"-", "1"}], "+", "\[Lambda]"}], ")"}]}], "]"}]}], "+", 
    RowBox[{"ArcTan", "[", 
     RowBox[{
      SuperscriptBox["10", "\[Mu]"], " ", 
      RowBox[{"(", 
       RowBox[{"1", "+", "\[Lambda]"}], ")"}]}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[168]:=",
 CellID->5012208],

Cell[TextData[{
 "We can plot a graph with the estimated error of ",
 StyleBox["GK[f,5]", "InlineCode"],
 "and the real error for different values of ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 ". That is, we plot ",
 StyleBox["|GK[f,5]-G[f,5]|", "InlineCode"],
 " and ",
 StyleBox["|GK[f,5]-", "InlineCode"],
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 StyleBox["|", "InlineCode"],
 "."
}], "Text",
 CellID->338113705],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\[Lambda]", ",", " ", 
     RowBox[{"\[Mu]", "=", "1.15"}], ",", 
     RowBox[{"pnts", "=", "1000"}], ",", "rres", ",", "errres", ",", 
     "exactres", ",", "lambdas"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
    "the", " ", "plot", " ", "uses", " ", "1000", " ", "values", " ", 
     "\[Lambda]"}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"lambdas", "=", 
     RowBox[{"Table", "[", 
      RowBox[{"\[Lambda]", ",", 
       RowBox[{"{", 
        RowBox[{"\[Lambda]", ",", 
         RowBox[{"-", "1"}], ",", "1", ",", 
         FractionBox["2", 
          RowBox[{"pnts", "-", "1"}]]}], "}"}]}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
     "this", " ", "computes", " ", "the", " ", "integral", " ", "and", " ", 
      "error", " ", "esitmates", " ", "over", " ", "the", " ", 
      RowBox[{"\[Lambda]", "'"}], 
      StyleBox["s",
       StyleForm -> "Text"]}], " ", "*)"}], "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"rres", ",", "errres"}], "}"}], "=", 
     RowBox[{"Transpose", "@", 
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{"Function", "[", 
         RowBox[{
          RowBox[{"{", "\[Lambda]", "}"}], ",", 
          RowBox[{"IRuleEstimate", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"f", "[", 
              RowBox[{"#1", ",", "\[Lambda]", ",", 
               RowBox[{"Evaluate", "[", "\[Mu]", "]"}]}], "]"}], "&"}], ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]}], "]"}], ",", 
        "lambdas"}], "]"}]}]}], ";", "\[IndentingNewLine]", 
    "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
     "this", " ", "computes", " ", "the", " ", "exact", " ", "integrals", " ",
       "over", " ", "the", " ", 
      RowBox[{"\[Lambda]", "'"}], 
      StyleBox["s",
       StyleForm -> "Text"]}], " ", "*)"}], "\[IndentingNewLine]", 
    RowBox[{"exactres", " ", "=", 
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"exact", "/.", 
         RowBox[{"\[Lambda]", "->", "#1"}]}], "&"}], ",", "lambdas"}], 
      "]"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
     "this", " ", "finds", " ", "the", " ", "number", " ", "underestimating", 
      " ", "error", " ", "estimates"}], " ", "*)"}], "\[IndentingNewLine]", 
    RowBox[{"Print", "[", 
     RowBox[{"\"\<Percent of underestimation: \>\"", ",", 
      RowBox[{
       RowBox[{"100", "*", 
        RowBox[{
         RowBox[{"Length", "[", 
          RowBox[{"Select", "[", 
           RowBox[{
            RowBox[{"errres", "-", 
             RowBox[{"Abs", "[", 
              RowBox[{"exactres", "-", "rres"}], "]"}]}], ",", 
            RowBox[{
             RowBox[{"#1", "<", "0"}], "&"}]}], "]"}], "]"}], "/", 
         RowBox[{"Length", "[", "lambdas", "]"}]}]}], "//", "N"}], ",", 
      "\"\<%\>\"", ",", " ", "\"\< \>\""}], "]"}], ";", "\[IndentingNewLine]",
     "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
      RowBox[{"the", " ", "plots"}], ",", " ", 
      RowBox[{
       RowBox[{"blue", " ", "is", " ", "for"}], " ", "|", 
       RowBox[{
        RowBox[{"GK", "[", 
         RowBox[{"f", ",", "5"}], "]"}], "-", 
        RowBox[{"GK", "[", 
         RowBox[{"f", ",", "5"}], "]"}]}], "|"}], ",", " ", 
      RowBox[{
       RowBox[{"red", " ", "is", " ", "for"}], " ", "|", 
       RowBox[{
        RowBox[{"GK", "[", 
         RowBox[{"f", ",", "5"}], "]"}], "-", 
        FormBox[
         RowBox[{
          SubsuperscriptBox["\[Integral]", 
           RowBox[{"-", "1"}], "1"], 
          RowBox[{"f", "[", 
           RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}]}],
         TraditionalForm]}], "|"}]}], " ", "*)"}], "\[IndentingNewLine]", 
    RowBox[{"ListLinePlot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"{", 
          RowBox[{"lambdas", ",", "errres"}], "}"}], "]"}], ",", 
        RowBox[{"Transpose", "[", 
         RowBox[{"{", 
          RowBox[{"lambdas", ",", 
           RowBox[{"Abs", "[", 
            RowBox[{"exactres", "-", "rres"}], "]"}]}], "}"}], "]"}]}], "}"}],
       ",", 
      RowBox[{"PlotRange", "->", "All"}], ",", 
      RowBox[{"PlotStyle", "->", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"Hue", "[", "0.7", "]"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"Hue", "[", "0", "]"}], "}"}]}], "}"}]}], ",", 
      RowBox[{"AxesLabel", "->", 
       RowBox[{"{", 
        RowBox[{"\[Lambda]", ",", "\"\<error\>\""}], "}"}]}]}], "]"}]}]}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellLabel->"In[169]:=",
 CellID->748699075],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Percent of underestimation: \"\>", "\[InvisibleSpace]", 
   "23.8`", "\[InvisibleSpace]", "\<\"%\"\>", 
   "\[InvisibleSpace]", "\<\" \"\>"}],
  SequenceForm["Percent of underestimation: ", 23.8, "%", " "],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[169]:=",
 CellID->753783567],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 129},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[169]=",
 CellID->14808626]
}, Open  ]],

Cell[TextData[{
 "In the plot above the blue graph is for the estimated error, ",
 StyleBox["|GK[f,5]-G[f,5]|", "InlineCode"],
 ".",
 " The graph of the actual error ",
 StyleBox["|GK[f,5]-", "InlineCode"],
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 StyleBox["|", "InlineCode"],
 "is red."
}], "Text",
 CellID->296855786],

Cell[TextData[{
 "We can see that the value 0.415 of the parameter ",
 Cell[BoxData[
  FormBox["\[Lambda]", TraditionalForm]], "InlineMath"],
 " is very close to one of the ",
 StyleBox["|GK[f,5]-G[f,5]|", "InlineCode"],
 " local ",
 "minimals",
 "."
}], "Text",
 CellID->71375821],

Cell[TextData[{
 "A one-dimensional quadrature rule can be seen as the result of the \
integration of a polynomial that is fitted through the rule's abscissas and \
the integrand values over them. We can further try to see the actual fitting \
polynomials for the integration of ",
 Cell[BoxData[
  RowBox[{"f", "[", 
   RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}]], "InlineFormula"],
 "."
}], "Text",
 CellID->496100613],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Clear", "[", "FitPlots", "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"FitPlots", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"a_", ",", "b_"}], "}"}], ",", "abscArg_"}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"absc", "=", 
       RowBox[{"Rescale", "[", 
        RowBox[{"abscArg", ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"a", ",", "b"}], "}"}]}], "]"}]}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"(*", " ", 
      RowBox[{
      "this", " ", "finds", " ", "the", " ", "interpolating", " ", 
       "polynomial", " ", "through", " ", "the", " ", "Gauss", " ", 
       "abscissas", " ", "and", " ", "the", " ", "values", " ", "of", " ", 
       "f", " ", "over", " ", "them"}], " ", "*)"}], "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"polyGauss", "[", "x_", "]"}], ":=", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"InterpolatingPolynomial", "[", 
         RowBox[{
          RowBox[{"Transpose", "[", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"Take", "[", 
              RowBox[{"absc", ",", 
               RowBox[{"{", 
                RowBox[{"2", ",", 
                 RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], ",", 
             RowBox[{
              RowBox[{
               RowBox[{"f", "[", "#1", "]"}], "&"}], "/@", 
              RowBox[{"(", 
               RowBox[{"Take", "[", 
                RowBox[{"absc", ",", 
                 RowBox[{"{", 
                  RowBox[{"2", ",", 
                   RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], ")"}]}]}], 
            "}"}], "]"}], ",", "x"}], "]"}], "]"}]}], ";", 
      "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
        RowBox[{
        "this", " ", "finds", " ", "the", " ", "interpolating", " ", 
         "polynomial", " ", "through", " ", "the", " ", "Gauss"}], "-", 
        RowBox[{
        "Kronrod", " ", "abscissas", " ", "and", " ", "the", " ", "values", 
         " ", "of", " ", "f", " ", "over", " ", "them"}]}], " ", "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"polyGaussKronrod", "[", "x_", "]"}], ":=", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"InterpolatingPolynomial", "[", 
         RowBox[{
          RowBox[{"Transpose", "[", 
           RowBox[{"{", 
            RowBox[{"absc", ",", 
             RowBox[{
              RowBox[{
               RowBox[{"f", "[", "#1", "]"}], "&"}], "/@", "absc"}]}], "}"}], 
           "]"}], ",", "x"}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "plot", " ", "of", " ", "the", " ", "Gauss", " ", "interpolating", " ",
         "points"}], " ", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"samplPointsGauss", "=", 
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"GrayLevel", "[", "0", "]"}], ",", 
          RowBox[{"PointSize", "[", "0.02", "]"}], ",", 
          RowBox[{"Point", "/@", 
           RowBox[{"Transpose", "[", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"Take", "[", 
               RowBox[{"absc", ",", 
                RowBox[{"{", 
                 RowBox[{"2", ",", 
                  RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}], ",", 
              RowBox[{
               RowBox[{
                RowBox[{"f", "[", "#1", "]"}], "&"}], "/@", 
               RowBox[{"Take", "[", 
                RowBox[{"absc", ",", 
                 RowBox[{"{", 
                  RowBox[{"2", ",", 
                   RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]}]}], "}"}], 
            "]"}]}]}], "}"}], "]"}]}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
        RowBox[{"plot", " ", "of", " ", "the", " ", "Gauss"}], "-", 
        RowBox[{"Kronrod", " ", "interpolating", " ", "points"}]}], " ", 
       "*)"}], "\[IndentingNewLine]", 
      RowBox[{"samplPointsGaussKronrod", "=", 
       RowBox[{"Graphics", "[", 
        RowBox[{"{", 
         RowBox[{"Red", ",", 
          RowBox[{"PointSize", "[", "0.012", "]"}], ",", 
          RowBox[{"Point", "/@", 
           RowBox[{"Transpose", "[", 
            RowBox[{"{", 
             RowBox[{"absc", ",", 
              RowBox[{
               RowBox[{
                RowBox[{"f", "[", "#1", "]"}], "&"}], "/@", "absc"}]}], "}"}],
             "]"}]}]}], "}"}], "]"}]}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "interpolating", " ", "polynomials", " ", "and", " ", "f", " ", 
        "plots"}], " ", "*)"}], "\[IndentingNewLine]", 
      RowBox[{"Block", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"$DisplayFunction", "=", "Identity"}], "}"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{"funcPlots", "=", 
          RowBox[{"Plot", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"polyGauss", "[", "x", "]"}], ",", 
              RowBox[{"polyGaussKronrod", "[", "x", "]"}], ",", 
              RowBox[{"f", "[", "x", "]"}]}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "a", ",", "b"}], "}"}], ",", 
            RowBox[{"PlotRange", "->", "All"}], ",", 
            RowBox[{"PlotStyle", "->", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{"{", 
                RowBox[{"Hue", "[", "0.7", "]"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"Hue", "[", "0.8", "]"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"Hue", "[", "0", "]"}], "}"}]}], "}"}]}]}], "]"}]}], 
         ";"}]}], "\[IndentingNewLine]", "]"}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", 
      RowBox[{"exact", "=", 
       RowBox[{"Integrate", "[", 
        RowBox[{
         RowBox[{"f", "[", "x", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "a", ",", "b"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"r1", "=", 
       RowBox[{"Integrate", "[", 
        RowBox[{
         RowBox[{"polyGauss", "[", "x", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "a", ",", "b"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"r2", "=", 
       RowBox[{"Integrate", "[", 
        RowBox[{
         RowBox[{"polyGaussKronrod", "[", "x", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"x", ",", "a", ",", "b"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Print", "[", 
       RowBox[{
        RowBox[{"\"\<estimated integral:\>\"", "<>", 
         RowBox[{"ToString", "@", "r2"}]}], ",", 
        RowBox[{"\"\<  exact integral:\>\"", "<>", 
         RowBox[{"ToString", "@", 
          RowBox[{"Re", "@", "exact"}]}]}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Print", "[", 
       RowBox[{
        RowBox[{"\"\<estimated error:\>\"", "<>", 
         RowBox[{"ToString", "@", 
          RowBox[{"Abs", "[", 
           RowBox[{"r1", "-", "r2"}], "]"}]}]}], ",", 
        RowBox[{"\"\<  actual error:\>\"", "<>", 
         RowBox[{"ToString", "@", 
          RowBox[{"Abs", "[", 
           RowBox[{"r2", "-", "exact"}], "]"}]}]}]}], "]"}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Show", "[", 
       RowBox[{"{", 
        RowBox[{
        "funcPlots", ",", "samplPointsGauss", ",", 
         "samplPointsGaussKronrod"}], "}"}], "]"}]}]}], "\[IndentingNewLine]",
     "]"}]}], ";"}]}], "Input",
 CellLabel->"In[170]:=",
 CellID->227199784],

Cell[TextData[{
 "In the plots below the function ",
 Cell[BoxData[
  RowBox[{"f", "[", 
   RowBox[{"x", ",", "\[Lambda]", ",", "\[Mu]"}], "]"}]], "InlineFormula"],
 " is plotted in red, the Gauss polynomial is plotted in blue, the Gauss\
\[Dash]Kronrod polynomial is plotted in violet, the Gauss sampling points are \
in black, and the Gauss\[Dash]Kronrod sampling points are in red."
}], "Text",
 CellID->336526354],

Cell[TextData[{
 "We can see that since the peak of ",
 Cell[BoxData[
  RowBox[{"f", "[", 
   RowBox[{"x", ",", "0.415", ",", "1.25"}], "]"}]], "InlineFormula"],
 " falls approximately halfway between two abscissas its approximation is an \
underestimate."
}], "Text",
 CellID->86027269],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FitPlots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"#1", ",", "0.415", ",", "1.25"}], "]"}], "&"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", "absc"}], "]"}]], "Input",
 CellLabel->"In[172]:=",
 CellID->10869174],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"estimated integral:1.72295\"\>", 
   "\[InvisibleSpace]", "\<\"  exact integral:3.00604\"\>"}],
  SequenceForm["estimated integral:1.72295", "  exact integral:3.00604"],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[172]:=",
 CellID->498831657],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"estimated error:0.0133177\"\>", 
   "\[InvisibleSpace]", "\<\"  actual error:1.28309\"\>"}],
  SequenceForm["estimated error:0.0133177", "  actual error:1.28309"],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[172]:=",
 CellID->11278681]
}, Open  ]],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[172]=",
 CellID->8847780]
}, Open  ]],

Cell[TextData[{
 "Conversely, we can see that since the peak of ",
 Cell[BoxData[
  RowBox[{"f", "[", 
   RowBox[{"x", ",", "0.53", ",", "1.25"}], "]"}]], "InlineFormula"],
 " falls approximately on one of the abscissas and its approximation is an \
overestimate."
}], "Text",
 CellID->5456465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FitPlots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"#1", ",", "0.53", ",", "1.25"}], "]"}], "&"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", "absc"}], "]"}]], "Input",
 CellLabel->"In[173]:=",
 CellID->110555663],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"estimated integral:4.77891\"\>", 
   "\[InvisibleSpace]", "\<\"  exact integral:2.98577\"\>"}],
  SequenceForm["estimated integral:4.77891", "  exact integral:2.98577"],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[173]:=",
 CellID->351767316],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"estimated error:3.77834\"\>", 
   "\[InvisibleSpace]", "\<\"  actual error:1.79313\"\>"}],
  SequenceForm["estimated error:3.77834", "  actual error:1.79313"],
  Editable->False]], "Print",
 CellLabel->"During evaluation of In[173]:=",
 CellID->5237898]
}, Open  ]],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[173]=",
 CellID->29824625]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Index of Technical Terms", "Section",
 CellID->17191326],

Cell[TextData[ButtonBox["Abscissas",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#348387628"]], "Text",\

 CellID->6544852],

Cell[TextData[ButtonBox["Degree of a one-dimensional integration rule",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#471065697"]], "Text",\

 CellID->91255008],

Cell[TextData[{
 ButtonBox["Degree of a multi-dimensional integration rule",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#47792312"],
 " "
}], "Text",
 CellID->22951571],

Cell[TextData[ButtonBox["Exact rule",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#98815734"]], "Text",
 CellID->78773278],

Cell[TextData[ButtonBox["Imbedded rule",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#139010942"]], "Text",\

 CellID->694134032],

Cell[TextData[ButtonBox["Null rule",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#84757638"]], "Text",
 CellID->357176027],

Cell[TextData[ButtonBox["Product rule",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#140598913"]], "Text",\

 CellID->484789141],

Cell[TextData[ButtonBox["Progressive rule",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#65611042"]], "Text",
 CellID->40390016],

Cell[TextData[ButtonBox["Sampling points",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#348387628"]], "Text",\

 CellID->585015044],

Cell[CellGroupData[{

Cell["MORE ABOUT", "TutorialMoreAboutSection",
 CellID->171462907],

Cell[TextData[ButtonBox["Numerical Evaluation & Precision",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NumericalEvaluationAndPrecision"]], "TutorialMoreAbout",
 CellID->341702418],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Calculus"]], "TutorialMoreAbout",
 CellID->168618367]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->289585108],

Cell[TextData[{
 ButtonBox["Advanced Numerical Integration in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateOverview"],
  FontSlant->"Italic"]
}], "RelatedTutorials",
 CellID->655978384],

Cell[TextData[ButtonBox["NIntegrate Introduction",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateIntroduction"]], "RelatedTutorials",
 CellID->40399306],

Cell[TextData[ButtonBox["NIntegrate Integration Strategies",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NIntegrateIntegrationStrategies"]], "RelatedTutorials",
 CellID->129348187],

Cell[TextData[ButtonBox["NIntegrate References",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/NIntegrateReferences"]], "RelatedTutorials",
 CellID->20913699]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"NIntegrate Integration Rules - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 40, 27.6973270}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "An integration rule computes an estimate of an integral over a region \
using a weighted sum. In the context of NIntegrate usage, an integration rule \
object provides both an integral estimate and an error estimate as a measure \
of the integral estimate's accuracy. An integration rule samples the \
integrand with a set of points. These points are called sampling points. In \
the literature these are usually called abscissas. Corresponding to each \
sampling point x_i there is a weight number w_i. An integration rule \
estimates the integral SubsuperscriptBox[\\[Integral], a, b]f(x)dx with the \
weighted sum \\[Sum]w_if(x_i). An integration rule is a functional, that is, \
it maps functions over the interval [a,b] (or a more general region) into \
real numbers. If a rule is applied over the region V this will be denoted as \
R^V(f), where f is the integrand.", "synonyms" -> {}, "title" -> 
    "NIntegrate Integration Rules", "type" -> "Tutorial", "uri" -> 
    "tutorial/NIntegrateIntegrationRules"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "Def:SaplingPointsAndAbscissas"->{
  Cell[3681, 106, 1243, 37, 70, "Text",
   CellTags->"Def:SaplingPointsAndAbscissas",
   CellID->348387628]},
 "Def:ExactRule"->{
  Cell[6636, 206, 644, 24, 70, "Text",
   CellTags->"Def:ExactRule",
   CellID->98815734]},
 "Def:OneDimRuleDegree"->{
  Cell[7949, 258, 502, 14, 70, "Text",
   CellTags->"Def:OneDimRuleDegree",
   CellID->471065697]},
 "Def:MultiDimRuleDegree"->{
  Cell[8454, 274, 1259, 40, 70, "Text",
   CellTags->"Def:MultiDimRuleDegree",
   CellID->47792312]},
 "Def:NullRule"->{
  Cell[9716, 316, 625, 18, 70, "Text",
   CellTags->"Def:NullRule",
   CellID->84757638],
  Cell[250028, 8067, 350, 7, 70, "Text",
   CellTags->"Def:NullRule",
   CellID->596623745],
  Cell[250393, 8077, 470, 9, 70, "ItemizedText",
   CounterAssignments->{"ItemizedText", 0},
   CellTags->"Def:NullRule",
   CellID->501247729],
  Cell[250866, 8088, 1647, 36, 70, "ItemizedText",
   CellTags->"Def:NullRule",
   CellID->64836516],
  Cell[252516, 8126, 550, 12, 70, "ItemizedText",
   CellTags->"Def:NullRule",
   CellID->76485970],
  Cell[253069, 8140, 472, 10, 70, "ItemizedText",
   CellTags->"Def:NullRule",
   CellID->372932111]},
 "Def:ImbededRule"->{
  Cell[10344, 336, 1013, 35, 70, "Text",
   CellTags->"Def:ImbededRule",
   CellID->139010942]},
 "Def:ProgressiveRule"->{
  Cell[12090, 397, 775, 24, 70, "Text",
   CellTags->"Def:ProgressiveRule",
   CellID->65611042]},
 "Def:OpenAndClosedType"->{
  Cell[12868, 423, 282, 6, 70, "Text",
   CellTags->"Def:OpenAndClosedType",
   CellID->68931625]},
 "TrapezoidalRule"->{
  Cell[21403, 726, 88, 2, 70, "Section",
   CellTags->"TrapezoidalRule",
   CellID->618158740]},
 "TrapezoidalRule1"->{
  Cell[21715, 737, 653, 24, 70, "NumberedEquation",
   CellTags->"TrapezoidalRule1",
   CellID->833180539]},
 "TrapezoidalRule2"->{
  Cell[22470, 766, 1047, 36, 70, "NumberedEquation",
   CellTags->"TrapezoidalRule2",
   CellID->9107376]},
 "RombergQuadrature"->{
  Cell[31649, 1096, 92, 2, 70, "Subsection",
   CellTags->"RombergQuadrature",
   CellID->101738258]},
 "NewtonCotesRule"->{
  Cell[43830, 1536, 87, 2, 70, "Section",
   CellTags->"NewtonCotesRule",
   CellID->81663330]},
 "GaussBerntsenEspelidRule"->{
  Cell[54256, 1891, 106, 2, 70, "Section",
   CellTags->"GaussBerntsenEspelidRule",
   CellID->141675390]},
 "GaussKronrodRule"->{
  Cell[70805, 2432, 90, 2, 70, "Section",
   CellTags->"GaussKronrodRule",
   CellID->381359375]},
 "GaussKronrodRuleDegree"->{
  Cell[74347, 2559, 1260, 45, 70, "Text",
   CellTags->"GaussKronrodRuleDegree",
   CellID->411028157]},
 "LobattoKronrodRule"->{
  Cell[88553, 3062, 94, 2, 70, "Section",
   CellTags->"LobattoKronrodRule",
   CellID->210486704]},
 "LobattoKronrodRuleDegree"->{
  Cell[91959, 3181, 1241, 44, 70, "Text",
   CellTags->"LobattoKronrodRuleDegree",
   CellID->701694722]},
 "ClenshawCurtisRule"->{
  Cell[105836, 3682, 94, 2, 70, "Section",
   CellTags->"ClenshawCurtisRule",
   CellID->486402291]},
 "MultiPanelRule"->{
  Cell[120538, 4186, 86, 2, 70, "Section",
   CellTags->"MultiPanelRule",
   CellID->106389821]},
 "CartesianProductRule"->{
  Cell[135314, 4690, 91, 2, 70, "Section",
   CellTags->"CartesianProductRule",
   CellID->650745857]},
 "Def:ProductRule"->{
  Cell[135408, 4694, 499, 13, 70, "Text",
   CellTags->"Def:ProductRule",
   CellID->140598913]},
 "CartesianRule1"->{
  Cell[143965, 4996, 1542, 47, 70, "NumberedEquation",
   CellTags->"CartesianRule1",
   CellID->537222873]},
 "CartesianRule2"->{
  Cell[145651, 5053, 673, 24, 70, "NumberedEquation",
   CellTags->"CartesianRule2",
   CellID->42116383]},
 "FullySymmetricRule"->{
  Cell[161504, 5568, 95, 2, 70, "Section",
   CellTags->"FullySymmetricRule",
   CellID->89177434]},
 "MonteCarloRule"->{
  Cell[183490, 6286, 86, 2, 70, "Section",
   CellTags->"MonteCarloRule",
   CellID->430697921]},
 "MonteCarloRule1"->{
  Cell[186430, 6387, 1302, 48, 70, "NumberedEquation",
   CellTags->"MonteCarloRule1",
   CellID->242788534]},
 "MonteCarloRule2"->{
  Cell[190046, 6520, 484, 18, 70, "NumberedEquation",
   CellTags->"MonteCarloRule2",
   CellID->38840336]},
 "MonteCarloRule3"->{
  Cell[196091, 6766, 936, 35, 70, "NumberedEquation",
   CellTags->"MonteCarloRule3",
   CellID->598600020]},
 "MonteCarloRuleAxisChoice"->{
  Cell[202787, 7019, 97, 2, 70, "Subsection",
   CellTags->"MonteCarloRuleAxisChoice",
   CellID->364207621]},
 "ExamplesOfPathologicalBehavior"->{
  Cell[585456, 16419, 115, 2, 70, "Section",
   CellTags->"ExamplesOfPathologicalBehavior",
   CellID->8354729]},
 "Tricking the error estimator"->{
  Cell[585596, 16425, 112, 2, 70, "Subsection",
   CellTags->"Tricking the error estimator",
   CellID->81420002]},
 "Phase Errors"->{
  Cell[623070, 17486, 82, 2, 70, "Subsection",
   CellTags->"Phase Errors",
   CellID->599865576]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"Def:SaplingPointsAndAbscissas", 673072, 18794},
 {"Def:ExactRule", 673202, 18798},
 {"Def:OneDimRuleDegree", 673321, 18802},
 {"Def:MultiDimRuleDegree", 673450, 18806},
 {"Def:NullRule", 673571, 18810},
 {"Def:ImbededRule", 674227, 18830},
 {"Def:ProgressiveRule", 674350, 18834},
 {"Def:OpenAndClosedType", 674477, 18838},
 {"TrapezoidalRule", 674599, 18842},
 {"TrapezoidalRule1", 674719, 18846},
 {"TrapezoidalRule2", 674851, 18850},
 {"RombergQuadrature", 674983, 18854},
 {"NewtonCotesRule", 675108, 18858},
 {"GaussBerntsenEspelidRule", 675236, 18862},
 {"GaussKronrodRule", 675367, 18866},
 {"GaussKronrodRuleDegree", 675495, 18870},
 {"LobattoKronrodRule", 675625, 18874},
 {"LobattoKronrodRuleDegree", 675757, 18878},
 {"ClenshawCurtisRule", 675889, 18882},
 {"MultiPanelRule", 676012, 18886},
 {"CartesianProductRule", 676137, 18890},
 {"Def:ProductRule", 676263, 18894},
 {"CartesianRule1", 676382, 18898},
 {"CartesianRule2", 676513, 18902},
 {"FullySymmetricRule", 676646, 18906},
 {"MonteCarloRule", 676768, 18910},
 {"MonteCarloRule1", 676888, 18914},
 {"MonteCarloRule2", 677021, 18918},
 {"MonteCarloRule3", 677152, 18922},
 {"MonteCarloRuleAxisChoice", 677293, 18926},
 {"ExamplesOfPathologicalBehavior", 677441, 18930},
 {"Tricking the error estimator", 677590, 18934},
 {"Phase Errors", 677725, 18938}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 2475, 59, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3100, 86, 65, 1, 70, "Title",
 CellID->358747687],
Cell[CellGroupData[{
Cell[3190, 91, 51, 1, 70, "Section",
 CellID->235359397],
Cell[3244, 94, 434, 10, 70, "Text",
 CellID->170702081],
Cell[3681, 106, 1243, 37, 70, "Text",
 CellTags->"Def:SaplingPointsAndAbscissas",
 CellID->348387628],
Cell[4927, 145, 423, 16, 70, "Text",
 CellID->106200629],
Cell[5353, 163, 1280, 41, 70, "Text",
 CellID->106091508],
Cell[6636, 206, 644, 24, 70, "Text",
 CellTags->"Def:ExactRule",
 CellID->98815734],
Cell[7283, 232, 663, 24, 70, "Text",
 CellID->382170004],
Cell[7949, 258, 502, 14, 70, "Text",
 CellTags->"Def:OneDimRuleDegree",
 CellID->471065697],
Cell[8454, 274, 1259, 40, 70, "Text",
 CellTags->"Def:MultiDimRuleDegree",
 CellID->47792312],
Cell[9716, 316, 625, 18, 70, "Text",
 CellTags->"Def:NullRule",
 CellID->84757638],
Cell[10344, 336, 1013, 35, 70, "Text",
 CellTags->"Def:ImbededRule",
 CellID->139010942],
Cell[11360, 373, 727, 22, 70, "Text",
 CellID->655009440],
Cell[12090, 397, 775, 24, 70, "Text",
 CellTags->"Def:ProgressiveRule",
 CellID->65611042],
Cell[12868, 423, 282, 6, 70, "Text",
 CellTags->"Def:OpenAndClosedType",
 CellID->68931625],
Cell[13153, 431, 693, 17, 70, "Text",
 CellID->453813706]
}, Open  ]],
Cell[CellGroupData[{
Cell[13883, 453, 69, 1, 70, "Section",
 CellID->135207409],
Cell[13955, 456, 1053, 31, 70, "Text",
 CellID->275460191],
Cell[15011, 489, 666, 21, 70, "Text",
 CellID->107496655],
Cell[15680, 512, 189, 5, 70, "MathCaption",
 CellID->439197840],
Cell[CellGroupData[{
Cell[15894, 521, 472, 14, 70, "Input",
 CellID->90688061],
Cell[16369, 537, 178, 5, 50, "Output",
 CellID->793054161]
}, Open  ]],
Cell[16562, 545, 232, 6, 70, "MathCaption",
 CellID->13137204],
Cell[CellGroupData[{
Cell[16819, 555, 472, 14, 70, "Input",
 CellID->201810221],
Cell[17294, 571, 178, 5, 50, "Output",
 CellID->550159177]
}, Open  ]],
Cell[17487, 579, 477, 13, 70, "Text",
 CellID->10639524],
Cell[17967, 594, 99, 1, 70, "MathCaption",
 CellID->887361652],
Cell[CellGroupData[{
Cell[18091, 599, 355, 11, 70, "Input",
 CellID->14623201],
Cell[18449, 612, 177, 5, 50, "Output",
 CellID->85441193]
}, Open  ]],
Cell[18641, 620, 136, 4, 70, "MathCaption",
 CellID->730821796],
Cell[CellGroupData[{
Cell[18802, 628, 472, 14, 70, "Input",
 CellID->44418322],
Cell[19277, 644, 178, 5, 50, "Output",
 CellID->354958004]
}, Open  ]],
Cell[19470, 652, 243, 6, 70, "Text",
 CellID->449396829],
Cell[19716, 660, 177, 5, 70, "MathCaption",
 CellID->23614898],
Cell[CellGroupData[{
Cell[19918, 669, 352, 11, 70, "Input",
 CellID->438368767],
Cell[20273, 682, 177, 5, 50, "Output",
 CellID->13273171]
}, Open  ]],
Cell[20465, 690, 208, 5, 70, "MathCaption",
 CellID->23122508],
Cell[CellGroupData[{
Cell[20698, 699, 475, 14, 70, "Input",
 CellID->521627535],
Cell[21176, 715, 178, 5, 50, "Output",
 CellID->444563884]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[21403, 726, 88, 2, 70, "Section",
 CellTags->"TrapezoidalRule",
 CellID->618158740],
Cell[21494, 730, 218, 5, 70, "Text",
 CellID->353686003],
Cell[21715, 737, 653, 24, 70, "NumberedEquation",
 CellTags->"TrapezoidalRule1",
 CellID->833180539],
Cell[22371, 763, 96, 1, 70, "Text",
 CellID->263555821],
Cell[22470, 766, 1047, 36, 70, "NumberedEquation",
 CellTags->"TrapezoidalRule2",
 CellID->9107376],
Cell[23520, 804, 226, 10, 70, "Text",
 CellID->314463073],
Cell[23749, 816, 378, 11, 70, "Text",
 CellID->675496772],
Cell[24130, 829, 155, 5, 70, "MathCaption",
 CellID->56571889],
Cell[CellGroupData[{
Cell[24310, 838, 272, 8, 70, "Input",
 CellID->7867874],
Cell[24585, 848, 162, 5, 36, "Output",
 CellID->412717973]
}, Open  ]],
Cell[24762, 856, 795, 20, 70, "DefinitionBox3Col",
 CellID->245149443],
Cell[25560, 878, 123, 4, 70, "Caption",
 CellID->626574275],
Cell[25686, 884, 727, 19, 70, "Text",
 CellID->10033282],
Cell[26416, 905, 995, 36, 70, "Text",
 CellID->107355692],
Cell[27414, 943, 447, 16, 70, "Text",
 CellID->341442099],
Cell[27864, 961, 169, 6, 70, "MathCaption",
 CellID->92628213],
Cell[CellGroupData[{
Cell[28058, 971, 620, 18, 70, "Input",
 CellID->331363070],
Cell[28681, 991, 359, 12, 36, "Output",
 CellID->167807418]
}, Open  ]],
Cell[29055, 1006, 160, 4, 70, "Text",
 CellID->381982768],
Cell[29218, 1012, 699, 23, 70, "MathCaption",
 CellID->99879431],
Cell[CellGroupData[{
Cell[29942, 1039, 369, 11, 70, "Input",
 CellID->494333022],
Cell[30314, 1052, 177, 5, 36, "Output",
 CellID->492529597]
}, Open  ]],
Cell[30506, 1060, 1118, 32, 70, "Text",
 CellID->354787860],
Cell[CellGroupData[{
Cell[31649, 1096, 92, 2, 70, "Subsection",
 CellTags->"RombergQuadrature",
 CellID->101738258],
Cell[31744, 1100, 823, 32, 70, "Text",
 CellID->224041229],
Cell[32570, 1134, 214, 7, 70, "Text",
 CellID->241294357],
Cell[32787, 1143, 1603, 54, 70, "DisplayMath",
 CellID->504426728],
Cell[34393, 1199, 41, 1, 70, "Text",
 CellID->666192100],
Cell[34437, 1202, 335, 13, 70, "DisplayMath",
 CellID->71003172],
Cell[34775, 1217, 51, 1, 70, "Text",
 CellID->151244],
Cell[34829, 1220, 486, 17, 70, "DisplayMath",
 CellID->21442028],
Cell[35318, 1239, 581, 21, 70, "DisplayMath",
 CellID->195266292],
Cell[35902, 1262, 365, 13, 70, "Text",
 CellID->325738640],
Cell[36270, 1277, 628, 23, 70, "DisplayMath",
 CellID->717613353],
Cell[36901, 1302, 632, 20, 70, "MathCaption",
 CellID->32589623],
Cell[CellGroupData[{
Cell[37558, 1326, 872, 23, 70, "Input",
 CellID->309570571],
Cell[38433, 1351, 401, 14, 36, "Output",
 CellID->360285275]
}, Open  ]],
Cell[38849, 1368, 137, 4, 70, "MathCaption",
 CellID->919715312],
Cell[CellGroupData[{
Cell[39011, 1376, 873, 23, 70, "Input",
 CellID->294291419],
Cell[39887, 1401, 413, 14, 36, "Output",
 CellID->996818232]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[40349, 1421, 88, 1, 70, "Subsection",
 CellID->12731638],
Cell[40440, 1424, 160, 4, 70, "MathCaption",
 CellID->869822652],
Cell[CellGroupData[{
Cell[40625, 1432, 365, 10, 70, "Input",
 CellID->436027268],
Cell[40993, 1444, 832, 23, 71, "Output",
 CellID->1959317245]
}, Open  ]],
Cell[41840, 1470, 127, 3, 70, "MathCaption",
 CellID->309374735],
Cell[CellGroupData[{
Cell[41992, 1477, 652, 21, 70, "Input",
 CellID->232745638],
Cell[42647, 1500, 1122, 29, 88, "Output",
 CellID->1707758716]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[43830, 1536, 87, 2, 70, "Section",
 CellTags->"NewtonCotesRule",
 CellID->81663330],
Cell[43920, 1540, 167, 4, 70, "Text",
 CellID->386092692],
Cell[44090, 1546, 408, 14, 70, "MathCaption",
 CellID->337187050],
Cell[CellGroupData[{
Cell[44523, 1564, 315, 10, 70, "Input",
 CellID->646414825],
Cell[44841, 1576, 177, 5, 36, "Output",
 CellID->135573007]
}, Open  ]],
Cell[45033, 1584, 713, 18, 70, "DefinitionBox3Col",
 CellID->226067643],
Cell[45749, 1604, 123, 4, 70, "Caption",
 CellID->447898499],
Cell[45875, 1610, 356, 12, 70, "Text",
 CellID->806097146],
Cell[46234, 1624, 668, 26, 70, "DisplayMath",
 CellID->310285549],
Cell[46905, 1652, 106, 3, 70, "Text",
 CellID->102306622],
Cell[47014, 1657, 843, 30, 70, "DisplayMath",
 CellID->574268937],
Cell[47860, 1689, 41, 1, 70, "Text",
 CellID->436311501],
Cell[47904, 1692, 912, 31, 70, "DisplayMath",
 CellID->197244546],
Cell[48819, 1725, 40, 1, 70, "Text",
 CellID->585197279],
Cell[48862, 1728, 662, 26, 70, "DisplayMath",
 CellID->221787720],
Cell[49527, 1756, 280, 9, 70, "Text",
 CellID->274109754],
Cell[CellGroupData[{
Cell[49832, 1769, 217, 6, 70, "Input",
 CellID->43642413],
Cell[50052, 1777, 2112, 44, 160, "Output",
 CellID->481638253]
}, Open  ]],
Cell[52179, 1824, 177, 4, 70, "Text",
 CellID->187839623],
Cell[CellGroupData[{
Cell[52381, 1832, 89, 1, 70, "Subsection",
 CellID->617627612],
Cell[52473, 1835, 167, 4, 70, "MathCaption",
 CellID->242967120],
Cell[CellGroupData[{
Cell[52665, 1843, 366, 10, 70, "Input",
 CellID->137233356],
Cell[53034, 1855, 1161, 29, 88, "Output",
 CellID->74978554]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[54256, 1891, 106, 2, 70, "Section",
 CellTags->"GaussBerntsenEspelidRule",
 CellID->141675390],
Cell[54365, 1895, 491, 8, 70, "Text",
 CellID->366043185],
Cell[54859, 1905, 450, 15, 70, "MathCaption",
 CellID->188552690],
Cell[CellGroupData[{
Cell[55334, 1924, 322, 10, 70, "Input",
 CellID->594944],
Cell[55659, 1936, 176, 5, 36, "Output",
 CellID->83729302]
}, Open  ]],
Cell[55850, 1944, 685, 18, 70, "DefinitionBox3Col",
 CellID->788578913],
Cell[56538, 1964, 131, 4, 70, "Caption",
 CellID->85891900],
Cell[56672, 1970, 1093, 41, 70, "Text",
 CellID->771349169],
Cell[57768, 2013, 648, 17, 70, "Text",
 CellID->18936690],
Cell[58419, 2032, 215, 7, 70, "Text",
 CellID->23167374],
Cell[58637, 2041, 1311, 43, 70, "DisplayMath",
 CellID->521791750],
Cell[59951, 2086, 736, 25, 70, "Text",
 CellID->101767876],
Cell[60690, 2113, 1028, 38, 70, "DisplayMath",
 CellID->46242465],
Cell[61721, 2153, 524, 18, 70, "Text",
 CellID->98024343],
Cell[62248, 2173, 406, 12, 70, "MathCaption",
 CellID->8631885],
Cell[CellGroupData[{
Cell[62679, 2189, 759, 22, 70, "Input",
 CellID->13965209],
Cell[63441, 2213, 428, 11, 54, "Output",
 CellID->605619980]
}, Open  ]],
Cell[CellGroupData[{
Cell[63906, 2229, 98, 1, 70, "Subsection",
 CellID->165210023],
Cell[64007, 2232, 203, 5, 70, "MathCaption",
 CellID->122899051],
Cell[CellGroupData[{
Cell[64235, 2241, 361, 10, 70, "Input",
 CellID->857995281],
Cell[64599, 2253, 2565, 50, 224, "Output",
 CellID->268189062]
}, Open  ]],
Cell[67179, 2306, 93, 1, 70, "Text",
 CellID->369875033],
Cell[67275, 2309, 83, 1, 70, "MathCaption",
 CellID->267299906],
Cell[67361, 2312, 1008, 32, 70, "Input",
 CellID->404712197],
Cell[68372, 2346, 298, 12, 70, "MathCaption",
 CellID->202744913],
Cell[68673, 2360, 323, 11, 70, "Input",
 CellID->14344216],
Cell[68999, 2373, 112, 3, 70, "MathCaption",
 CellID->766006120],
Cell[CellGroupData[{
Cell[69136, 2380, 588, 21, 70, "Input",
 CellID->226502069],
Cell[69727, 2403, 1017, 22, 88, "Output",
 CellID->33431475]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[70805, 2432, 90, 2, 70, "Section",
 CellTags->"GaussKronrodRule",
 CellID->381359375],
Cell[70898, 2436, 381, 7, 70, "Text",
 CellID->270397252],
Cell[71282, 2445, 416, 14, 70, "MathCaption",
 CellID->205758888],
Cell[CellGroupData[{
Cell[71723, 2463, 317, 10, 70, "Input",
 CellID->340866141],
Cell[72043, 2475, 174, 5, 36, "Output",
 CellID->6236080]
}, Open  ]],
Cell[72232, 2483, 726, 19, 70, "DefinitionBox3Col",
 CellID->40598151],
Cell[72961, 2504, 124, 4, 70, "Caption",
 CellID->489455538],
Cell[73088, 2510, 1093, 41, 70, "Text",
 CellID->34338313],
Cell[74184, 2553, 160, 4, 70, "Text",
 CellID->642176838],
Cell[74347, 2559, 1260, 45, 70, "Text",
 CellTags->"GaussKronrodRuleDegree",
 CellID->411028157],
Cell[75610, 2606, 878, 28, 70, "Text",
 CellID->369924265],
Cell[76491, 2636, 397, 12, 70, "MathCaption",
 CellID->1330256],
Cell[CellGroupData[{
Cell[76913, 2652, 694, 20, 70, "Input",
 CellID->481448378],
Cell[77610, 2674, 409, 11, 36, "Output",
 CellID->386419301]
}, Open  ]],
Cell[78034, 2688, 253, 8, 70, "Text",
 CellID->371546905],
Cell[CellGroupData[{
Cell[78312, 2700, 90, 1, 70, "Subsection",
 CellID->360328261],
Cell[78405, 2703, 188, 4, 70, "MathCaption",
 CellID->510007182],
Cell[CellGroupData[{
Cell[78618, 2711, 353, 10, 70, "Input",
 CellID->242297082],
Cell[78974, 2723, 2523, 50, 224, "Output",
 CellID->10911793]
}, Open  ]],
Cell[81512, 2776, 261, 8, 70, "Text",
 CellID->336934172],
Cell[81776, 2786, 118, 3, 70, "MathCaption",
 CellID->7329969],
Cell[CellGroupData[{
Cell[81919, 2793, 291, 10, 70, "Input",
 CellID->451066042],
Cell[82213, 2805, 160, 5, 36, "Output",
 CellID->491184162]
}, Open  ]],
Cell[82388, 2813, 66, 1, 70, "MathCaption",
 CellID->90881048],
Cell[82457, 2816, 147, 5, 70, "Input",
 CellID->543769040],
Cell[82607, 2823, 1636, 61, 70, "Text",
 CellID->3201818],
Cell[84246, 2886, 346, 11, 70, "MathCaption",
 CellID->10533031],
Cell[CellGroupData[{
Cell[84617, 2901, 430, 15, 70, "Input",
 CellID->51837532],
Cell[85050, 2918, 330, 10, 36, "Output",
 CellID->42626316]
}, Open  ]],
Cell[85395, 2931, 105, 2, 70, "MathCaption",
 CellID->189440317],
Cell[CellGroupData[{
Cell[85525, 2937, 288, 10, 70, "Input",
 CellID->274742322],
Cell[85816, 2949, 202, 5, 36, "Output",
 CellID->86169558]
}, Open  ]],
Cell[86033, 2957, 386, 11, 70, "Text",
 CellID->215160853],
Cell[86422, 2970, 67, 1, 70, "MathCaption",
 CellID->192354571],
Cell[86492, 2973, 191, 7, 70, "Input",
 CellID->264566629],
Cell[86686, 2982, 347, 11, 70, "MathCaption",
 CellID->342265653],
Cell[CellGroupData[{
Cell[87058, 2997, 430, 15, 70, "Input",
 CellID->11345905],
Cell[87491, 3014, 288, 9, 39, "Output",
 CellID->9175640]
}, Open  ]],
Cell[87794, 3026, 216, 8, 70, "MathCaption",
 CellID->664975092],
Cell[CellGroupData[{
Cell[88035, 3038, 288, 10, 70, "Input",
 CellID->179491808],
Cell[88326, 3050, 166, 5, 36, "Output",
 CellID->431190145]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[88553, 3062, 94, 2, 70, "Section",
 CellTags->"LobattoKronrodRule",
 CellID->210486704],
Cell[88650, 3066, 471, 8, 70, "Text",
 CellID->221379897],
Cell[89124, 3076, 415, 13, 70, "MathCaption",
 CellID->244831656],
Cell[CellGroupData[{
Cell[89564, 3093, 319, 10, 70, "Input",
 CellID->211318625],
Cell[89886, 3105, 177, 5, 36, "Output",
 CellID->119532472]
}, Open  ]],
Cell[90078, 3113, 657, 17, 70, "DefinitionBox3Col",
 CellID->384885018],
Cell[90738, 3132, 124, 4, 70, "Caption",
 CellID->8046501],
Cell[90865, 3138, 1091, 41, 70, "Text",
 CellID->62664843],
Cell[91959, 3181, 1241, 44, 70, "Text",
 CellTags->"LobattoKronrodRuleDegree",
 CellID->701694722],
Cell[93203, 3227, 603, 19, 70, "Text",
 CellID->491447804],
Cell[93809, 3248, 401, 12, 70, "MathCaption",
 CellID->233311261],
Cell[CellGroupData[{
Cell[94235, 3264, 696, 20, 70, "Input",
 CellID->209034893],
Cell[94934, 3286, 395, 10, 36, "Output",
 CellID->147922501]
}, Open  ]],
Cell[95344, 3299, 357, 10, 70, "Text",
 CellID->161877433],
Cell[95704, 3311, 252, 8, 70, "Text",
 CellID->102460424],
Cell[CellGroupData[{
Cell[95981, 3323, 91, 1, 70, "Subsection",
 CellID->49911002],
Cell[96075, 3326, 189, 4, 70, "MathCaption",
 CellID->82668669],
Cell[CellGroupData[{
Cell[96289, 3334, 355, 10, 70, "Input",
 CellID->718180355],
Cell[96647, 3346, 2070, 46, 179, "Output",
 CellID->106416111]
}, Open  ]],
Cell[98732, 3395, 263, 8, 70, "Text",
 CellID->610949573],
Cell[98998, 3405, 122, 3, 70, "MathCaption",
 CellID->483828678],
Cell[CellGroupData[{
Cell[99145, 3412, 290, 10, 70, "Input",
 CellID->49463485],
Cell[99438, 3424, 160, 5, 36, "Output",
 CellID->442533813]
}, Open  ]],
Cell[99613, 3432, 67, 1, 70, "MathCaption",
 CellID->380705553],
Cell[99683, 3435, 147, 5, 70, "Input",
 CellID->227232811],
Cell[99833, 3442, 1638, 61, 70, "Text",
 CellID->59389584],
Cell[101474, 3505, 347, 11, 70, "MathCaption",
 CellID->430422070],
Cell[CellGroupData[{
Cell[101846, 3520, 431, 15, 70, "Input",
 CellID->445772403],
Cell[102280, 3537, 345, 10, 36, "Output",
 CellID->104276203]
}, Open  ]],
Cell[102640, 3550, 105, 2, 70, "MathCaption",
 CellID->482734384],
Cell[CellGroupData[{
Cell[102770, 3556, 288, 10, 70, "Input",
 CellID->572404606],
Cell[103061, 3568, 203, 5, 36, "Output",
 CellID->261997679]
}, Open  ]],
Cell[103279, 3576, 388, 11, 70, "Text",
 CellID->271407587],
Cell[103670, 3589, 64, 1, 70, "MathCaption",
 CellID->233526],
Cell[103737, 3592, 191, 7, 70, "Input",
 CellID->349925812],
Cell[103931, 3601, 347, 11, 70, "MathCaption",
 CellID->120734638],
Cell[CellGroupData[{
Cell[104303, 3616, 430, 15, 70, "Input",
 CellID->27301135],
Cell[104736, 3633, 332, 10, 39, "Output",
 CellID->196573900]
}, Open  ]],
Cell[105083, 3646, 208, 8, 70, "MathCaption",
 CellID->808380556],
Cell[CellGroupData[{
Cell[105316, 3658, 288, 10, 70, "Input",
 CellID->735325913],
Cell[105607, 3670, 168, 5, 36, "Output",
 CellID->126387969]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[105836, 3682, 94, 2, 70, "Section",
 CellTags->"ClenshawCurtisRule",
 CellID->486402291],
Cell[105933, 3686, 166, 4, 70, "Text",
 CellID->340439718],
Cell[106102, 3692, 448, 15, 70, "MathCaption",
 CellID->303679401],
Cell[CellGroupData[{
Cell[106575, 3711, 318, 10, 70, "Input",
 CellID->32016025],
Cell[106896, 3723, 177, 5, 36, "Output",
 CellID->143959292]
}, Open  ]],
Cell[107088, 3731, 618, 16, 70, "DefinitionBox3Col",
 CellID->91740512],
Cell[107709, 3749, 126, 4, 70, "Caption",
 CellID->689040563],
Cell[107838, 3755, 921, 26, 70, "Text",
 CellID->74612686],
Cell[108762, 3783, 620, 15, 70, "Text",
 CellID->79196589],
Cell[109385, 3800, 1485, 50, 70, "Text",
 CellID->8642916],
Cell[110873, 3852, 835, 31, 70, "MathCaption",
 CellID->503678959],
Cell[CellGroupData[{
Cell[111733, 3887, 379, 11, 70, "Input",
 CellID->305820777],
Cell[112115, 3900, 226, 7, 36, "Output",
 CellID->74347468]
}, Open  ]],
Cell[112356, 3910, 401, 12, 70, "MathCaption",
 CellID->593420888],
Cell[CellGroupData[{
Cell[112782, 3926, 695, 20, 70, "Input",
 CellID->53244339],
Cell[113480, 3948, 396, 10, 36, "Output",
 CellID->408857200]
}, Open  ]],
Cell[CellGroupData[{
Cell[113913, 3963, 92, 1, 70, "Subsection",
 CellID->123359709],
Cell[114008, 3966, 180, 4, 70, "MathCaption",
 CellID->283484080],
Cell[CellGroupData[{
Cell[114213, 3974, 355, 10, 70, "Input",
 CellID->136232968],
Cell[114571, 3986, 2308, 50, 179, "Output",
 CellID->146115799]
}, Open  ]],
Cell[116894, 4039, 294, 11, 70, "MathCaption",
 CellID->128574619],
Cell[CellGroupData[{
Cell[117213, 4054, 635, 22, 70, "Input",
 CellID->15132135],
Cell[117851, 4078, 568, 14, 71, "Output",
 CellID->777636271]
}, Open  ]],
Cell[118434, 4095, 67, 1, 70, "MathCaption",
 CellID->216857284],
Cell[118504, 4098, 191, 7, 70, "Input",
 CellID->129525173],
Cell[118698, 4107, 347, 11, 70, "MathCaption",
 CellID->263836066],
Cell[CellGroupData[{
Cell[119070, 4122, 431, 15, 70, "Input",
 CellID->548907976],
Cell[119504, 4139, 329, 10, 36, "Output",
 CellID->778105720]
}, Open  ]],
Cell[119848, 4152, 204, 8, 70, "MathCaption",
 CellID->308066929],
Cell[CellGroupData[{
Cell[120077, 4164, 217, 7, 70, "Input",
 CellID->996374849],
Cell[120297, 4173, 180, 6, 51, "Output",
 CellID->439137555]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[120538, 4186, 86, 2, 70, "Section",
 CellTags->"MultiPanelRule",
 CellID->106389821],
Cell[120627, 4190, 317, 6, 70, "Text",
 CellID->279020568],
Cell[120947, 4198, 189, 5, 70, "MathCaption",
 CellID->122516479],
Cell[CellGroupData[{
Cell[121161, 4207, 470, 13, 70, "Input",
 CellID->203034866],
Cell[121634, 4222, 176, 5, 36, "Output",
 CellID->241538296]
}, Open  ]],
Cell[121825, 4230, 840, 21, 70, "DefinitionBox3Col",
 CellID->284199955],
Cell[122668, 4253, 122, 4, 70, "Caption",
 CellID->597817729],
Cell[122793, 4259, 567, 21, 70, "Text",
 CellID->55345536],
Cell[123363, 4282, 55, 1, 70, "Text",
 CellID->584112888],
Cell[123421, 4285, 558, 19, 70, "DisplayMath",
 CellID->17051204],
Cell[123982, 4306, 298, 11, 70, "Text",
 CellID->361764049],
Cell[124283, 4319, 1090, 36, 70, "DisplayMath",
 CellID->387201495],
Cell[125376, 4357, 1083, 41, 70, "Text",
 CellID->459098833],
Cell[126462, 4400, 1063, 37, 70, "DisplayMath",
 CellID->83082974],
Cell[127528, 4439, 1133, 41, 70, "Text",
 CellID->36114295],
Cell[128664, 4482, 400, 12, 70, "Text",
 CellID->102589039],
Cell[CellGroupData[{
Cell[129089, 4498, 88, 1, 70, "Subsection",
 CellID->362787930],
Cell[129180, 4501, 200, 5, 70, "MathCaption",
 CellID->299816157],
Cell[CellGroupData[{
Cell[129405, 4510, 355, 10, 70, "Input",
 CellID->387652139],
Cell[129763, 4522, 1809, 37, 139, "Output",
 CellID->266235973]
}, Open  ]],
Cell[131587, 4562, 116, 3, 70, "MathCaption",
 CellID->729040440],
Cell[CellGroupData[{
Cell[131728, 4569, 265, 7, 70, "Input",
 CellID->256564232],
Cell[131996, 4578, 790, 21, 71, "Output",
 CellID->130952621]
}, Open  ]],
Cell[132801, 4602, 234, 8, 70, "MathCaption",
 CellID->725321628],
Cell[CellGroupData[{
Cell[133060, 4614, 508, 17, 70, "Input",
 CellID->25604383],
Cell[133571, 4633, 620, 14, 54, "Output",
 CellID->214226]
}, Open  ]],
Cell[134206, 4650, 133, 4, 70, "MathCaption",
 CellID->164000480],
Cell[CellGroupData[{
Cell[134364, 4658, 238, 8, 70, "Input",
 CellID->169686619],
Cell[134605, 4668, 648, 15, 71, "Output",
 CellID->274563372]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[135314, 4690, 91, 2, 70, "Section",
 CellTags->"CartesianProductRule",
 CellID->650745857],
Cell[135408, 4694, 499, 13, 70, "Text",
 CellTags->"Def:ProductRule",
 CellID->140598913],
Cell[135910, 4709, 412, 14, 70, "MathCaption",
 CellID->300155745],
Cell[CellGroupData[{
Cell[136347, 4727, 480, 14, 70, "Input",
 CellID->393423482],
Cell[136830, 4743, 175, 5, 36, "Output",
 CellID->2121472]
}, Open  ]],
Cell[137020, 4751, 750, 20, 70, "DefinitionBox3Col",
 CellID->530404128],
Cell[137773, 4773, 120, 4, 70, "Caption",
 CellID->61382795],
Cell[137896, 4779, 78, 1, 70, "Text",
 CellID->204881985],
Cell[137977, 4782, 683, 23, 70, "DisplayMath",
 CellID->524096338],
Cell[138663, 4807, 683, 23, 70, "DisplayMath",
 CellID->180444258],
Cell[139349, 4832, 675, 23, 70, "DisplayMath",
 CellID->619143603],
Cell[140027, 4857, 643, 22, 70, "Text",
 CellID->121385605],
Cell[140673, 4881, 1466, 46, 70, "DisplayMath",
 CellID->555797250],
Cell[142142, 4929, 994, 34, 70, "Text",
 CellID->588880804],
Cell[143139, 4965, 823, 29, 70, "Text",
 CellID->91227601],
Cell[143965, 4996, 1542, 47, 70, "NumberedEquation",
 CellTags->"CartesianRule1",
 CellID->537222873],
Cell[145510, 5045, 138, 6, 70, "Text",
 CellID->260414201],
Cell[145651, 5053, 673, 24, 70, "NumberedEquation",
 CellTags->"CartesianRule2",
 CellID->42116383],
Cell[146327, 5079, 1010, 37, 70, "Text",
 CellID->25296496],
Cell[147340, 5118, 444, 14, 70, "MathCaption",
 CellID->39398941],
Cell[CellGroupData[{
Cell[147809, 5136, 1348, 39, 83, "Input",
 CellID->723150784],
Cell[149160, 5177, 1043, 21, 201, "Output",
 Evaluatable->False,
 CellID->543011096]
}, Open  ]],
Cell[150218, 5201, 739, 21, 70, "Text",
 CellID->9994829],
Cell[150960, 5224, 443, 13, 70, "Text",
 CellID->24408754],
Cell[151406, 5239, 211, 5, 70, "MathCaption",
 CellID->898295519],
Cell[CellGroupData[{
Cell[151642, 5248, 345, 10, 70, "Input",
 CellID->674937957],
Cell[151990, 5260, 176, 5, 36, "Output",
 CellID->65238187]
}, Open  ]],
Cell[152181, 5268, 166, 4, 70, "MathCaption",
 CellID->40900188],
Cell[CellGroupData[{
Cell[152372, 5276, 426, 13, 70, "Input",
 CellID->68962419],
Cell[152801, 5291, 176, 5, 36, "Output",
 CellID->401605760]
}, Open  ]],
Cell[152992, 5299, 167, 4, 70, "MathCaption",
 CellID->382128625],
Cell[CellGroupData[{
Cell[153184, 5307, 648, 20, 70, "Input",
 CellID->101435058],
Cell[153835, 5329, 177, 5, 36, "Output",
 CellID->668996031]
}, Open  ]],
Cell[154027, 5337, 210, 7, 70, "Text",
 CellID->336461069],
Cell[CellGroupData[{
Cell[154262, 5348, 87, 1, 70, "Subsection",
 CellID->643860566],
Cell[154352, 5351, 292, 7, 70, "MathCaption",
 CellID->167416858],
Cell[CellGroupData[{
Cell[154669, 5362, 502, 14, 70, "Input",
 CellID->53627707],
Cell[155174, 5378, 1438, 41, 122, "Output",
 CellID->18274478]
}, Open  ]],
Cell[156627, 5422, 324, 8, 70, "Text",
 CellID->26637091],
Cell[156954, 5432, 252, 8, 70, "MathCaption",
 CellID->241482162],
Cell[157209, 5442, 1333, 42, 70, "Input",
 CellID->661477249],
Cell[CellGroupData[{
Cell[158567, 5488, 112, 3, 70, "Input",
 CellID->188665271],
Cell[158682, 5493, 2761, 68, 173, "Output",
 CellID->89200106]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[161504, 5568, 95, 2, 70, "Section",
 CellTags->"FullySymmetricRule",
 CellID->89177434],
Cell[161602, 5572, 756, 22, 70, "Text",
 CellID->232587503],
Cell[162361, 5596, 501, 16, 70, "MathCaption",
 CellID->552950062],
Cell[CellGroupData[{
Cell[162887, 5616, 416, 13, 70, "Input",
 CellID->347703773],
Cell[163306, 5631, 177, 5, 36, "Output",
 CellID->313102913]
}, Open  ]],
Cell[163498, 5639, 977, 29, 70, "Text",
 CellID->456247437],
Cell[164478, 5670, 626, 15, 70, "DefinitionBox3Col",
 CellID->190249742],
Cell[165107, 5687, 128, 4, 70, "Caption",
 CellID->417675764],
Cell[165238, 5693, 869, 28, 70, "Text",
 CellID->414457384],
Cell[166110, 5723, 821, 29, 70, "DisplayMath",
 CellID->101011595],
Cell[166934, 5754, 578, 15, 70, "Text",
 CellID->316383435],
Cell[167515, 5771, 877, 21, 70, "Text",
 CellID->351238863],
Cell[168395, 5794, 264, 8, 70, "MathCaption",
 CellID->593623189],
Cell[168662, 5804, 1560, 49, 70, "Text",
 CellID->84920445],
Cell[170225, 5855, 885, 30, 70, "MathCaption",
 CellID->589519280],
Cell[CellGroupData[{
Cell[171135, 5889, 2082, 59, 70, "Input",
 CellID->9963956],
Cell[173220, 5950, 1192, 35, 350, "Output",
 CellID->478127139]
}, Open  ]],
Cell[CellGroupData[{
Cell[174449, 5990, 94, 1, 70, "Subsection",
 CellID->810793694],
Cell[174546, 5993, 167, 4, 70, "Text",
 CellID->308119359],
Cell[174716, 5999, 89, 1, 70, "MathCaption",
 CellID->85577757],
Cell[174808, 6002, 128, 4, 70, "Input",
 CellID->871976899],
Cell[174939, 6008, 105, 2, 70, "MathCaption",
 CellID->622343018],
Cell[175047, 6012, 1590, 45, 70, "Input",
 CellID->194618994],
Cell[176640, 6059, 101, 2, 70, "MathCaption",
 CellID->7824105],
Cell[176744, 6063, 671, 21, 70, "Input",
 CellID->332011626],
Cell[177418, 6086, 85, 1, 70, "MathCaption",
 CellID->277639999],
Cell[177506, 6089, 158, 5, 70, "Input",
 CellID->172683935],
Cell[177667, 6096, 67, 1, 70, "MathCaption",
 CellID->230195363],
Cell[177737, 6099, 263, 9, 70, "Input",
 CellID->222259366],
Cell[178003, 6110, 82, 1, 70, "MathCaption",
 CellID->695694313],
Cell[CellGroupData[{
Cell[178110, 6115, 582, 20, 70, "Input",
 CellID->537035195],
Cell[178695, 6137, 179, 5, 50, "Output",
 CellID->11859086]
}, Open  ]],
Cell[178889, 6145, 68, 1, 70, "MathCaption",
 CellID->101276330],
Cell[CellGroupData[{
Cell[178982, 6150, 385, 13, 70, "Input",
 CellID->366086263],
Cell[179370, 6165, 166, 5, 50, "Output",
 CellID->727749007]
}, Open  ]],
Cell[179551, 6173, 99, 1, 70, "MathCaption",
 CellID->481732818],
Cell[179653, 6176, 1030, 31, 70, "Input",
 CellID->392564065],
Cell[180686, 6209, 81, 1, 70, "MathCaption",
 CellID->409154864],
Cell[CellGroupData[{
Cell[180792, 6214, 105, 3, 28, "Input",
 CellID->223653060],
Cell[180900, 6219, 1177, 23, 179, "Output",
 Evaluatable->False,
 CellID->414096072]
}, Open  ]],
Cell[182092, 6245, 77, 1, 70, "MathCaption",
 CellID->375109605],
Cell[CellGroupData[{
Cell[182194, 6250, 246, 7, 28, "Input",
 CellID->498852237],
Cell[182443, 6259, 986, 20, 201, "Output",
 Evaluatable->False,
 CellID->618496341]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[183490, 6286, 86, 2, 70, "Section",
 CellTags->"MonteCarloRule",
 CellID->430697921],
Cell[183579, 6290, 192, 4, 70, "Text",
 CellID->445079933],
Cell[183774, 6296, 245, 8, 70, "MathCaption",
 CellID->125960922],
Cell[CellGroupData[{
Cell[184044, 6308, 484, 15, 70, "Input",
 CellID->87523634],
Cell[184531, 6325, 175, 5, 36, "Output",
 CellID->79918751]
}, Open  ]],
Cell[184721, 6333, 1030, 26, 70, "DefinitionBox3Col",
 CellID->787116549],
Cell[185754, 6361, 122, 4, 70, "Caption",
 CellID->156226993],
Cell[185879, 6367, 548, 18, 70, "Text",
 CellID->621655033],
Cell[186430, 6387, 1302, 48, 70, "NumberedEquation",
 CellTags->"MonteCarloRule1",
 CellID->242788534],
Cell[187735, 6437, 1174, 40, 70, "Text",
 CellID->202442289],
Cell[188912, 6479, 1131, 39, 70, "Text",
 CellID->41316462],
Cell[190046, 6520, 484, 18, 70, "NumberedEquation",
 CellTags->"MonteCarloRule2",
 CellID->38840336],
Cell[190533, 6540, 1007, 37, 70, "Text",
 CellID->361534288],
Cell[191543, 6579, 105, 3, 70, "Text",
 CellID->432244465],
Cell[191651, 6584, 379, 14, 70, "DisplayMath",
 CellID->69332026],
Cell[192033, 6600, 560, 19, 70, "Text",
 CellID->188740124],
Cell[192596, 6621, 149, 6, 70, "Text",
 CellID->45118777],
Cell[192748, 6629, 354, 13, 70, "DisplayMath",
 CellID->30141812],
Cell[193105, 6644, 687, 24, 70, "Text",
 CellID->1707266],
Cell[193795, 6670, 652, 25, 70, "DisplayMath",
 CellID->248999231],
Cell[194450, 6697, 561, 22, 70, "Text",
 CellID->566360629],
Cell[195014, 6721, 218, 7, 70, "Text",
 CellID->211802050],
Cell[195235, 6730, 645, 24, 70, "DisplayMath",
 CellID->41092199],
Cell[195883, 6756, 205, 8, 70, "Text",
 CellID->642292911],
Cell[196091, 6766, 936, 35, 70, "NumberedEquation",
 CellTags->"MonteCarloRule3",
 CellID->598600020],
Cell[197030, 6803, 330, 11, 70, "Text",
 CellID->763425414],
Cell[197363, 6816, 560, 18, 70, "Text",
 CellID->446452961],
Cell[197926, 6836, 538, 19, 70, "Text",
 CellID->78621487],
Cell[198467, 6857, 885, 31, 70, "Text",
 CellID->352176520],
Cell[199355, 6890, 922, 34, 70, "DisplayMath",
 CellID->239818919],
Cell[200280, 6926, 1609, 55, 70, "DisplayMath",
 CellID->288417924],
Cell[201892, 6983, 870, 32, 70, "Text",
 CellID->128955055],
Cell[CellGroupData[{
Cell[202787, 7019, 97, 2, 70, "Subsection",
 CellTags->"MonteCarloRuleAxisChoice",
 CellID->364207621],
Cell[202887, 7023, 507, 9, 70, "Text",
 CellID->376198576],
Cell[203397, 7034, 872, 24, 70, "Text",
 CellID->4046376],
Cell[204272, 7060, 772, 12, 70, "Text",
 CellID->545958628],
Cell[205047, 7074, 726, 21, 70, "DefinitionBox",
 CellID->419965355],
Cell[205776, 7097, 120, 4, 70, "Caption",
 CellID->138741263],
Cell[205899, 7103, 479, 12, 70, "DefinitionBox3Col",
 CellID->275325445],
Cell[206381, 7117, 111, 4, 70, "Caption",
 CellID->807023952],
Cell[206495, 7123, 222, 7, 70, "MathCaption",
 CellID->19134768],
Cell[CellGroupData[{
Cell[206742, 7134, 822, 26, 70, "Input",
 CellID->506232340],
Cell[207567, 7162, 175, 5, 36, "Output",
 CellID->33153669]
}, Open  ]],
Cell[207757, 7170, 675, 12, 70, "Text",
 CellID->80599949],
Cell[208435, 7184, 75, 1, 70, "MathCaption",
 CellID->172687242],
Cell[CellGroupData[{
Cell[208535, 7189, 1274, 42, 61, "Input",
 CellID->226078],
Cell[209812, 7233, 12681, 212, 162, "Output",
 Evaluatable->False,
 CellID->185328952]
}, Open  ]],
Cell[222508, 7448, 176, 4, 70, "MathCaption",
 CellID->220307675],
Cell[CellGroupData[{
Cell[222709, 7456, 1552, 42, 101, "Input",
 CellID->191743341],
Cell[224264, 7500, 8077, 136, 204, "Output",
 Evaluatable->False,
 CellID->443928057]
}, Open  ]],
Cell[232356, 7639, 682, 19, 70, "MathCaption",
 CellID->196890053],
Cell[CellGroupData[{
Cell[233063, 7662, 1740, 46, 155, "Input",
 CellID->367184567],
Cell[234806, 7710, 9257, 156, 204, "Output",
 Evaluatable->False,
 CellID->373733288]
}, Open  ]],
Cell[244078, 7869, 127, 3, 70, "MathCaption",
 CellID->626558770],
Cell[CellGroupData[{
Cell[244230, 7876, 955, 28, 70, "Input",
 CellID->167416277],
Cell[245188, 7906, 226, 7, 36, "Output",
 CellID->60397028]
}, Open  ]],
Cell[245429, 7916, 224, 5, 70, "MathCaption",
 CellID->124917769],
Cell[CellGroupData[{
Cell[245678, 7925, 1108, 31, 70, "Input",
 CellID->430097928],
Cell[246789, 7958, 226, 7, 36, "Output",
 CellID->196842352]
}, Open  ]],
Cell[247030, 7968, 164, 4, 70, "MathCaption",
 CellID->82415329],
Cell[CellGroupData[{
Cell[247219, 7976, 1108, 31, 70, "Input",
 CellID->108345155],
Cell[248330, 8009, 226, 7, 36, "Output",
 CellID->135829003]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[248617, 8023, 63, 1, 70, "Section",
 CellID->233261223],
Cell[248683, 8026, 732, 18, 70, "Text",
 CellID->265640948],
Cell[249418, 8046, 162, 4, 70, "ItemizedText",
 CellID->351937731],
Cell[249583, 8052, 139, 4, 70, "ItemizedText",
 CellID->858066548],
Cell[CellGroupData[{
Cell[249747, 8060, 278, 5, 70, "ItemizedText",
 CellID->169742927],
Cell[250028, 8067, 350, 7, 70, "Text",
 CellTags->"Def:NullRule",
 CellID->596623745]
}, Open  ]],
Cell[250393, 8077, 470, 9, 70, "ItemizedText",
 CounterAssignments->{"ItemizedText", 0},
 CellTags->"Def:NullRule",
 CellID->501247729],
Cell[250866, 8088, 1647, 36, 70, "ItemizedText",
 CellTags->"Def:NullRule",
 CellID->64836516],
Cell[252516, 8126, 550, 12, 70, "ItemizedText",
 CellTags->"Def:NullRule",
 CellID->76485970],
Cell[253069, 8140, 472, 10, 70, "ItemizedText",
 CellTags->"Def:NullRule",
 CellID->372932111],
Cell[CellGroupData[{
Cell[253566, 8154, 66, 1, 70, "Subsection",
 CellID->6741536],
Cell[253635, 8157, 505, 9, 70, "Text",
 CellID->263334898],
Cell[254143, 8168, 222, 8, 70, "Text",
 CellID->33375855],
Cell[254368, 8178, 911, 32, 70, "DisplayMath",
 CellID->489783266],
Cell[255282, 8212, 172, 7, 70, "Text",
 CellID->5802283],
Cell[255457, 8221, 499, 16, 70, "MathCaption",
 CellID->97608709],
Cell[255959, 8239, 1351, 38, 70, "Input",
 CellID->51461333],
Cell[257313, 8279, 76, 1, 70, "MathCaption",
 CellID->508919412],
Cell[257392, 8282, 1318, 40, 70, "Input",
 CellID->447064306],
Cell[258713, 8324, 224, 8, 70, "MathCaption",
 CellID->70008065],
Cell[CellGroupData[{
Cell[258962, 8336, 642, 19, 47, "Input",
 CellID->95153325],
Cell[259607, 8357, 4269, 74, 143, "Output",
 Evaluatable->False,
 CellID->701900742]
}, Open  ]],
Cell[263891, 8434, 2073, 66, 70, "MathCaption",
 CellID->12446453],
Cell[265967, 8502, 780, 26, 70, "Input",
 CellID->318514032],
Cell[266750, 8530, 312, 6, 70, "MathCaption",
 CellID->564454320],
Cell[CellGroupData[{
Cell[267087, 8540, 4580, 148, 70, "Input",
 CellID->89675936],
Cell[271670, 8690, 219998, 5806, 70, "Output",
 CellID->103941512]
}, Open  ]],
Cell[CellGroupData[{
Cell[491705, 14501, 78, 1, 70, "Subsubsection",
 CellID->870923766],
Cell[491786, 14504, 136, 4, 70, "MathCaption",
 CellID->416488103],
Cell[491925, 14510, 958, 28, 70, "Input",
 CellID->305297791],
Cell[492886, 14540, 176, 4, 70, "MathCaption",
 CellID->320095499],
Cell[493065, 14546, 1264, 33, 70, "Input",
 CellID->2205647],
Cell[494332, 14581, 207, 5, 70, "MathCaption",
 CellID->147910307],
Cell[494542, 14588, 787, 26, 70, "Input",
 CellID->190038355],
Cell[495332, 14616, 196, 4, 70, "MathCaption",
 CellID->318278572],
Cell[CellGroupData[{
Cell[495553, 14624, 4318, 141, 70, "Input",
 CellID->201325054],
Cell[499874, 14767, 6850, 141, 223, "Output",
 CellID->87888892]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[506785, 14915, 57, 1, 70, "Subsection",
 CellID->870846129],
Cell[506845, 14918, 500, 16, 70, "MathCaption",
 CellID->164356145],
Cell[507348, 14936, 1796, 50, 70, "Input",
 CellID->267469494],
Cell[509147, 14988, 76, 1, 70, "MathCaption",
 CellID->630579378],
Cell[509226, 14991, 1318, 40, 70, "Input",
 CellID->888976217],
Cell[510547, 15033, 225, 8, 70, "MathCaption",
 CellID->537957296],
Cell[CellGroupData[{
Cell[510797, 15045, 637, 19, 47, "Input",
 CellID->86280932],
Cell[511437, 15066, 4269, 74, 143, "Output",
 Evaluatable->False,
 CellID->842272163]
}, Open  ]],
Cell[515721, 15143, 2352, 72, 70, "MathCaption",
 CellID->121789434],
Cell[518076, 15217, 1561, 51, 70, "Input",
 CellID->299139976],
Cell[519640, 15270, 144, 4, 70, "MathCaption",
 CellID->41515502],
Cell[CellGroupData[{
Cell[519809, 15278, 7183, 171, 299, "Input",
 CellID->640793172],
Cell[526995, 15451, 58400, 961, 738, "Output",
 Evaluatable->False,
 CellID->286380649]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[585456, 16419, 115, 2, 70, "Section",
 CellTags->"ExamplesOfPathologicalBehavior",
 CellID->8354729],
Cell[CellGroupData[{
Cell[585596, 16425, 112, 2, 70, "Subsection",
 CellTags->"Tricking the error estimator",
 CellID->81420002],
Cell[585711, 16429, 381, 9, 70, "Text",
 CellID->65273665],
Cell[CellGroupData[{
Cell[586117, 16442, 65, 1, 70, "Subsubsection",
 CellID->594163809],
Cell[586185, 16445, 68, 1, 70, "Text",
 CellID->587795730],
Cell[586256, 16448, 582, 20, 70, "Input",
 CellID->957759846],
Cell[586841, 16470, 201, 8, 70, "Text",
 CellID->18894011],
Cell[CellGroupData[{
Cell[587067, 16482, 249, 8, 70, "Input",
 CellID->283837762],
Cell[587319, 16492, 179, 5, 36, "Output",
 CellID->295853054]
}, Open  ]],
Cell[587513, 16500, 194, 7, 70, "Text",
 CellID->205592170],
Cell[CellGroupData[{
Cell[587732, 16511, 248, 8, 70, "Input",
 CellID->314533975],
Cell[587983, 16521, 178, 5, 36, "Output",
 CellID->19696595]
}, Open  ]],
Cell[588176, 16529, 92, 1, 70, "Text",
 CellID->294095742],
Cell[CellGroupData[{
Cell[588293, 16534, 130, 4, 70, "Input",
 CellID->389546213],
Cell[588426, 16540, 181, 5, 36, "Output",
 CellID->500929119]
}, Open  ]],
Cell[588622, 16548, 106, 3, 70, "Text",
 CellID->350117999],
Cell[CellGroupData[{
Cell[588753, 16555, 213, 7, 28, "Input",
 CellID->642089480],
Cell[588969, 16564, 2597, 47, 136, "Output",
 Evaluatable->False,
 CellID->92603440]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[591615, 16617, 58, 1, 70, "Subsubsection",
 CellID->26508302],
Cell[591676, 16620, 402, 13, 70, "Text",
 CellID->95164517],
Cell[CellGroupData[{
Cell[592103, 16637, 389, 10, 70, "Input",
 CellID->288824240],
Cell[592495, 16649, 565, 10, 70, "Message",
 CellID->9393131],
Cell[593063, 16661, 177, 5, 36, "Output",
 CellID->940999677]
}, Open  ]],
Cell[593255, 16669, 128, 4, 70, "Text",
 CellID->769920255],
Cell[CellGroupData[{
Cell[593408, 16677, 586, 16, 70, "Input",
 CellID->560881033],
Cell[593997, 16695, 567, 10, 70, "Message",
 CellID->617882725],
Cell[594567, 16707, 567, 10, 70, "Message",
 CellID->524580207],
Cell[595137, 16719, 566, 10, 70, "Message",
 CellID->50110422],
Cell[595706, 16731, 470, 9, 70, "Message",
 CellID->503101152],
Cell[596179, 16742, 730, 21, 54, "Output",
 CellID->822624168]
}, Open  ]],
Cell[596924, 16766, 118, 3, 70, "Text",
 CellID->17871884],
Cell[CellGroupData[{
Cell[597067, 16773, 263, 8, 28, "Input",
 CellID->64721228],
Cell[597333, 16783, 2690, 48, 136, "Output",
 Evaluatable->False,
 CellID->48052908]
}, Open  ]],
Cell[600038, 16834, 246, 8, 70, "Text",
 CellID->576771419],
Cell[CellGroupData[{
Cell[600309, 16846, 331, 11, 28, "Input",
 CellID->64999389],
Cell[600643, 16859, 2695, 48, 135, "Output",
 Evaluatable->False,
 CellID->246760415]
}, Open  ]],
Cell[603353, 16910, 324, 9, 70, "Text",
 CellID->346156256],
Cell[CellGroupData[{
Cell[603702, 16923, 279, 9, 70, "Input",
 CellID->366120302],
Cell[603984, 16934, 181, 5, 36, "Output",
 CellID->460257124]
}, Open  ]],
Cell[CellGroupData[{
Cell[604202, 16944, 130, 4, 70, "Input",
 CellID->744098211],
Cell[604335, 16950, 181, 5, 36, "Output",
 CellID->193582369]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[604565, 16961, 72, 1, 70, "Subsubsection",
 CellID->276439645],
Cell[604640, 16964, 266, 9, 70, "Text",
 CellID->26596856],
Cell[604909, 16975, 287, 8, 70, "Input",
 CellID->20167425],
Cell[605199, 16985, 94, 1, 70, "Text",
 CellID->788890286],
Cell[605296, 16988, 1400, 42, 70, "Input",
 CellID->387478236],
Cell[606699, 17032, 119, 3, 70, "Text",
 CellID->595901578],
Cell[606821, 17037, 544, 16, 70, "Input",
 CellID->47765883],
Cell[607368, 17055, 182, 4, 70, "Text",
 CellID->336487250],
Cell[CellGroupData[{
Cell[607575, 17063, 727, 22, 70, "Input",
 CellID->807428914],
Cell[608305, 17087, 2606, 51, 119, "Output",
 CellID->1479857]
}, Open  ]],
Cell[610926, 17141, 849, 26, 70, "Text",
 CellID->100168402],
Cell[611778, 17169, 287, 9, 70, "Text",
 CellID->35774155],
Cell[CellGroupData[{
Cell[612090, 17182, 387, 13, 70, "Input",
 CellID->425392635],
Cell[612480, 17197, 498, 12, 54, "Output",
 CellID->266628682],
Cell[612981, 17211, 318, 9, 36, "Output",
 CellID->154930106]
}, Open  ]],
Cell[613314, 17223, 83, 1, 70, "Text",
 CellID->7317078],
Cell[613400, 17226, 218, 8, 70, "Input",
 CellID->23331306],
Cell[613621, 17236, 115, 3, 70, "Text",
 CellID->308356064],
Cell[613739, 17241, 614, 20, 70, "Input",
 CellID->185124695],
Cell[614356, 17263, 226, 8, 70, "Text",
 CellID->377548952],
Cell[CellGroupData[{
Cell[614607, 17275, 721, 24, 65, "Input",
 CellID->733072035],
Cell[615331, 17301, 2800, 50, 127, "Output",
 Evaluatable->False,
 CellID->57978769],
Cell[618134, 17353, 2811, 50, 120, "Output",
 Evaluatable->False,
 CellID->72932622]
}, Open  ]],
Cell[620960, 17406, 396, 11, 70, "Text",
 CellID->63256172],
Cell[CellGroupData[{
Cell[621381, 17421, 455, 14, 70, "Input",
 CellID->413656717],
Cell[621839, 17437, 181, 5, 36, "Output",
 CellID->494884295],
Cell[622023, 17444, 181, 5, 36, "Output",
 CellID->510395348],
Cell[622207, 17451, 334, 12, 50, "Output",
 CellID->290260353]
}, Open  ]],
Cell[622556, 17466, 465, 14, 70, "Text",
 CellID->592263786]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[623070, 17486, 82, 2, 70, "Subsection",
 CellTags->"Phase Errors",
 CellID->599865576],
Cell[623155, 17490, 357, 9, 70, "Text",
 CellID->471141279],
Cell[623515, 17501, 60, 1, 70, "Text",
 CellID->220215642],
Cell[623578, 17504, 428, 15, 70, "Input",
 CellID->288788974],
Cell[624009, 17521, 387, 13, 70, "Text",
 CellID->642958260],
Cell[CellGroupData[{
Cell[624421, 17538, 360, 11, 70, "Input",
 CellID->20537278],
Cell[624784, 17551, 178, 5, 36, "Output",
 CellID->268377181]
}, Open  ]],
Cell[CellGroupData[{
Cell[624999, 17561, 313, 10, 70, "Input",
 CellID->58834672],
Cell[625315, 17573, 257, 7, 36, "Output",
 CellID->278366612]
}, Open  ]],
Cell[625587, 17583, 270, 9, 70, "Text",
 CellID->154978428],
Cell[CellGroupData[{
Cell[625882, 17596, 186, 6, 70, "Input",
 CellID->9700234],
Cell[626071, 17604, 175, 5, 36, "Output",
 CellID->980781]
}, Open  ]],
Cell[626261, 17612, 239, 8, 70, "Text",
 CellID->702892424],
Cell[626503, 17622, 77, 1, 70, "Text",
 CellID->115797403],
Cell[626583, 17625, 1380, 53, 70, "Text",
 CellID->43904657],
Cell[627966, 17680, 519, 15, 70, "Text",
 CellID->117125857],
Cell[628488, 17697, 288, 8, 70, "Input",
 CellID->173926475],
Cell[628779, 17707, 88, 1, 70, "MathCaption",
 CellID->83936180],
Cell[628870, 17710, 1400, 42, 70, "Input",
 CellID->107661966],
Cell[630273, 17754, 504, 17, 70, "MathCaption",
 CellID->155242890],
Cell[630780, 17773, 502, 17, 70, "Input",
 CellID->5012208],
Cell[631285, 17792, 835, 28, 70, "Text",
 CellID->338113705],
Cell[CellGroupData[{
Cell[632145, 17824, 4883, 131, 70, "Input",
 CellID->748699075],
Cell[637031, 17957, 351, 8, 23, "Print",
 CellID->753783567],
Cell[637385, 17967, 9575, 161, 150, "Output",
 Evaluatable->False,
 CellID->14808626]
}, Open  ]],
Cell[646975, 18131, 563, 18, 70, "Text",
 CellID->296855786],
Cell[647541, 18151, 281, 10, 70, "Text",
 CellID->71375821],
Cell[647825, 18163, 430, 10, 70, "Text",
 CellID->496100613],
Cell[648258, 18175, 7807, 201, 70, "Input",
 CellID->227199784],
Cell[656068, 18378, 417, 9, 70, "Text",
 CellID->336526354],
Cell[656488, 18389, 287, 8, 70, "Text",
 CellID->86027269],
Cell[CellGroupData[{
Cell[656800, 18401, 304, 10, 70, "Input",
 CellID->10869174],
Cell[CellGroupData[{
Cell[657129, 18415, 314, 7, 70, "Print",
 CellID->498831657],
Cell[657446, 18424, 307, 7, 23, "Print",
 CellID->11278681]
}, Open  ]],
Cell[657768, 18434, 3604, 63, 128, "Output",
 Evaluatable->False,
 CellID->8847780]
}, Open  ]],
Cell[661387, 18500, 294, 8, 70, "Text",
 CellID->5456465],
Cell[CellGroupData[{
Cell[661706, 18512, 304, 10, 70, "Input",
 CellID->110555663],
Cell[CellGroupData[{
Cell[662035, 18526, 314, 7, 70, "Print",
 CellID->351767316],
Cell[662352, 18535, 302, 7, 23, "Print",
 CellID->5237898]
}, Open  ]],
Cell[662669, 18545, 5207, 89, 130, "Output",
 Evaluatable->False,
 CellID->29824625]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[667937, 18641, 62, 1, 70, "Section",
 CellID->17191326],
Cell[668002, 18644, 155, 4, 70, "Text",
 CellID->6544852],
Cell[668160, 18650, 191, 4, 70, "Text",
 CellID->91255008],
Cell[668354, 18656, 203, 6, 70, "Text",
 CellID->22951571],
Cell[668560, 18664, 154, 3, 70, "Text",
 CellID->78773278],
Cell[668717, 18669, 161, 4, 70, "Text",
 CellID->694134032],
Cell[668881, 18675, 154, 3, 70, "Text",
 CellID->357176027],
Cell[669038, 18680, 160, 4, 70, "Text",
 CellID->484789141],
Cell[669201, 18686, 160, 3, 70, "Text",
 CellID->40390016],
Cell[669364, 18691, 163, 4, 70, "Text",
 CellID->585015044],
Cell[CellGroupData[{
Cell[669552, 18699, 66, 1, 70, "TutorialMoreAboutSection",
 CellID->171462907],
Cell[669621, 18702, 186, 4, 70, "TutorialMoreAbout",
 CellID->341702418],
Cell[669810, 18708, 136, 3, 70, "TutorialMoreAbout",
 CellID->168618367]
}, Open  ]],
Cell[CellGroupData[{
Cell[669983, 18716, 72, 1, 70, "RelatedTutorialsSection",
 CellID->289585108],
Cell[670058, 18719, 314, 9, 70, "RelatedTutorials",
 CellID->655978384],
Cell[670375, 18730, 166, 3, 70, "RelatedTutorials",
 CellID->40399306],
Cell[670544, 18735, 189, 4, 70, "RelatedTutorials",
 CellID->129348187],
Cell[670736, 18741, 162, 3, 70, "RelatedTutorials",
 CellID->20913699]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[670937, 18749, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

