(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     50936,       1892]
NotebookOptionsPosition[     41036,       1560]
NotebookOutlinePosition[     44136,       1637]
CellTagsIndexPosition[     44043,       1631]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Symbolic Mathematics" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/SymbolicMathematicsOverview"], 
          "Complex Polynomial Systems" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ComplexPolynomialSystems"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Symbolic Mathematics\"\>", 
       2->"\<\"Complex Polynomial Systems\"\>"}, "\<\"related tutorials\"\>", 
       
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Eliminate" :> Documentation`HelpLookup["paclet:ref/Eliminate"],
           "Reduce" :> Documentation`HelpLookup["paclet:ref/Reduce"], "Solve" :> 
          Documentation`HelpLookup["paclet:ref/Solve"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Eliminate\"\>", 2->"\<\"Reduce\"\>", 
       3->"\<\"Solve\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Solving Equations", "Title",
 CellTags->{"S1.5.7", "7.1"},
 CellID->30806],

Cell[TextData[{
 "An expression like ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "+", 
    RowBox[{"2", "x"}], "-", "7"}], "==", "0"}]], "InlineFormula"],
 " represents an ",
 StyleBox["equation",
  FontSlant->"Italic"],
 " in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ". You will often need to ",
 StyleBox["solve",
  FontSlant->"Italic"],
 " equations like this, to find out for what values of ",
 Cell[BoxData["x"], "InlineFormula"],
 " they are true."
}], "Text",
 CellID->29849],

Cell[TextData[{
 "This gives the two solutions to the quadratic equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["x", "2"], "+", "2", "x", "-", "7", "=", "0"}], 
   TraditionalForm]], "InlineMath"],
 ". The solutions are given as replacements for\[NonBreakingSpace]",
 Cell[BoxData["x"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->14854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "+", 
     RowBox[{"2", "x"}], "-", "7"}], "==", "0"}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->24272],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "1"}], "-", 
      RowBox[{"2", " ", 
       SqrtBox["2"]}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      RowBox[{"2", " ", 
       SqrtBox["2"]}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{226, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->301653470]
}, Open  ]],

Cell["Here are the numerical values of the solutions.", "MathCaption",
 CellID->12294],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->12054],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "3.8284271247461903`"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "1.8284271247461903`"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{209, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->671513081]
}, Open  ]],

Cell[TextData[{
 "You can get a list of the actual solutions for ",
 Cell[BoxData["x"], "InlineFormula"],
 " by applying the rules generated by ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " to ",
 Cell[BoxData["x"], "InlineFormula"],
 " using the replacement operator. "
}], "MathCaption",
 CellID->146],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"x", "/.", "%"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->4229],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "3.8284271247461903`"}], ",", "1.8284271247461903`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->591447060]
}, Open  ]],

Cell[TextData[{
 "You can equally well apply the rules to any other expression involving ",
 Cell[BoxData["x"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->10428],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"x", "^", "2"}], "+", 
   RowBox[{"3", " ", "x"}]}], "/.", "%%"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->10559],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.17157287525381`", ",", "8.828427124746192`"}], "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->449655680]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Solve",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Solve"], "[", 
     StyleBox["lhs", "TI"], "==", 
     StyleBox["rhs", "TI"], ",", 
     StyleBox["x", "TI"], "]"}], Cell[TextData[{
     "solve an equation, giving a list of rules for ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["x", "TI"], "/.", 
     StyleBox["solution", "TI"]}], Cell[TextData[{
     "use the list of rules to get values for ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["expr", "TI"], "/.", 
     StyleBox["solution", "TI"]}], Cell[
    "use the list of rules to get values for an expression", "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, False, {False}, False}, "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->29421],

Cell["Finding and using solutions to equations. ", "Caption",
 CellID->22688],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " always tries to give you explicit ",
 StyleBox["formulas",
  FontSlant->"Italic"],
 " for the solutions to equations. However, it is a basic mathematical result \
that, for sufficiently complicated equations, explicit algebraic formulas in \
terms of radicals cannot be given. If you have an algebraic equation in one \
variable, and the highest power of the variable is at most four, then ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can always give you formulas for the solutions. However, if the highest \
power is five or more, it may be mathematically impossible to give explicit \
algebraic formulas for all the solutions. "
}], "Text",
 CellID->16366],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can always solve algebraic equations in one variable when the highest \
power is less than five. "
}], "MathCaption",
 CellID->12033],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "4"}], "-", 
     RowBox[{"5", " ", 
      RowBox[{"x", "^", "2"}]}], "-", "3"}], "==", "0"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->26685],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", 
      SqrtBox[
       RowBox[{
        FractionBox["5", "2"], "+", 
        FractionBox[
         SqrtBox["37"], "2"]}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     SqrtBox[
      RowBox[{
       FractionBox["5", "2"], "+", 
       FractionBox[
        SqrtBox["37"], "2"]}]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "\[ImaginaryI]"}], " ", 
      SqrtBox[
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "5"}], "+", 
          SqrtBox["37"]}], ")"}]}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"\[ImaginaryI]", " ", 
      SqrtBox[
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "5"}], "+", 
          SqrtBox["37"]}], ")"}]}]]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{346, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->151059938]
}, Open  ]],

Cell["It can solve some equations that involve higher powers. ", "MathCaption",
 CellID->25329],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "6"}], "==", "1"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->32261],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "1"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"-", "1"}], ")"}], 
       RowBox[{"1", "/", "3"}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], 
      RowBox[{"1", "/", "3"}]]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"-", "1"}], ")"}], 
       RowBox[{"2", "/", "3"}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"-", "1"}], ")"}], 
      RowBox[{"2", "/", "3"}]]}], "}"}]}], "}"}]], "Output",
 ImageSize->{510, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->297368981]
}, Open  ]],

Cell[TextData[{
 "There are some equations, however, for which it is mathematically \
impossible to find explicit formulas for the solutions. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects to represent the solutions in this case."
}], "MathCaption",
 CellID->2208],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"2", "-", 
     RowBox[{"4", " ", "x"}], "+", 
     RowBox[{"x", "^", "5"}]}], "==", "0"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->8982],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"2", "-", 
         RowBox[{"4", " ", "#1"}], "+", 
         SuperscriptBox["#1", "5"]}], "&"}], ",", "1"}], "]"}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"2", "-", 
         RowBox[{"4", " ", "#1"}], "+", 
         SuperscriptBox["#1", "5"]}], "&"}], ",", "2"}], "]"}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"2", "-", 
         RowBox[{"4", " ", "#1"}], "+", 
         SuperscriptBox["#1", "5"]}], "&"}], ",", "3"}], "]"}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"2", "-", 
         RowBox[{"4", " ", "#1"}], "+", 
         SuperscriptBox["#1", "5"]}], "&"}], ",", "4"}], "]"}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"2", "-", 
         RowBox[{"4", " ", "#1"}], "+", 
         SuperscriptBox["#1", "5"]}], "&"}], ",", "5"}], "]"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{591, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->59714926]
}, Open  ]],

Cell["\<\
Even though you cannot get explicit formulas, you can still evaluate the \
solutions numerically. \
\>", "MathCaption",
 CellID->32506],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->6878],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "1.518512152784912`"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "0.5084994846573329`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", "1.2435963905735432`"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "0.11679186122298203`"}], "-", 
      RowBox[{"1.4384476953291772`", " ", "\[ImaginaryI]"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "0.11679186122298203`"}], "+", 
      RowBox[{"1.4384476953291772`", " ", "\[ImaginaryI]"}]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{376, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->245492487]
}, Open  ]],

Cell[TextData[{
 "In addition to being able to solve purely algebraic equations, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can also solve some equations involving other functions. "
}], "Text",
 CellID->1897],

Cell[TextData[{
 "After printing a warning, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " returns one solution to this equation. "
}], "MathCaption",
 CellID->7887],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "==", "a"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->6538],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used by \\!\\(Solve\\), so some \
solutions may not be found; use Reduce for complete solution information. \\!\
\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/ifun\\\", \
ButtonNote -> \\\"Solve::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[9]:=",
 CellID->405388678],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{"ArcSin", "[", "a", "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{118, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->88756058]
}, Open  ]],

Cell[TextData[{
 "It is important to realize that an equation such as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"sin", "(", "x", ")"}], "=", "a"}], TraditionalForm]], 
  "InlineMath"],
 " actually has an infinite number of possible solutions, in this case \
differing by multiples of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"2", "\[Pi]"}], TraditionalForm]], "InlineMath"],
 ". However, ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " by default returns just one solution, but prints a message telling you \
that other solutions may exist. You can use ",
 Cell[BoxData[
  ButtonBox["Reduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reduce"]], "InlineFormula"],
 " to get more information. "
}], "Text",
 CellID->17540],

Cell["\<\
There is no explicit \[OpenCurlyDoubleQuote]closed form\
\[CloseCurlyDoubleQuote] solution for a transcendental equation like this. \
\>", "MathCaption",
 CellID->22795],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", "x", "]"}], "==", "x"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->7266],

Cell[BoxData[
 RowBox[{
  RowBox[{"Solve", "::", "\<\"tdep\"\>"}], ":", 
  " ", "\<\"The equations appear to involve the variables to be solved for in \
an essentially non-algebraic way. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/Solve/tdep\\\", \
ButtonNote -> \\\"Solve::tdep\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[10]:=",
 CellID->728791810],

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", "x", "]"}], "\[Equal]", "x"}], ",", "x"}], 
  "]"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->77651796]
}, Open  ]],

Cell[TextData[{
 "You can find an approximate numerical solution using ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 ", and giving a starting value for ",
 Cell[BoxData["x"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->24557],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Cos", "[", "x", "]"}], "==", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->2317],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"x", "\[Rule]", "0.7390851332151607`"}], "}"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->92464891]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " can also handle equations involving symbolic functions. In such cases, it \
again prints a warning, then gives results in terms of formal inverse \
functions. "
}], "Text",
 CellID->32053],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " returns a result in terms of the formal inverse function of ",
 Cell[BoxData["f"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->18135],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"x", "^", "2"}], "]"}], "==", "a"}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[12]:=",
 CellID->12258],

Cell[BoxData[
 RowBox[{
  RowBox[{"InverseFunction", "::", "\<\"ifun\"\>"}], ":", 
  " ", "\<\"Inverse functions are being used. Values may be lost for \
multivalued inverses. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/InverseFunction/ifun\\\", ButtonNote -> \\\
\"InverseFunction::ifun\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[12]:=",
 CellID->602935356],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", 
      SqrtBox[
       RowBox[{
        TagBox[
         SuperscriptBox["f", 
          RowBox[{"(", 
           RowBox[{"-", "1"}], ")"}]],
         {InverseFunction, 1, 1},
         Editable->False], "[", "a", "]"}]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     SqrtBox[
      RowBox[{
       TagBox[
        SuperscriptBox["f", 
         RowBox[{"(", 
          RowBox[{"-", "1"}], ")"}]],
        {InverseFunction, 1, 1},
        Editable->False], "[", "a", "]"}]]}], "}"}]}], "}"}]], "Output",
 ImageSize->{246, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->176468237]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Solve",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Solve"], "[", "{", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "1"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "2"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["y", "TI"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "solve a set of simultaneous equations for ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", ",", "y", ",", "\[Ellipsis]"}], TraditionalForm]], 
      "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxItemSize->{"Columns" -> {18, {
      Scaled[0.6]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->21693],

Cell["Solving sets of simultaneous equations. ", "Caption",
 CellID->16466],

Cell[TextData[{
 "You can also use ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " to solve sets of simultaneous equations. You simply give the list of \
equations, and specify the list of variables to solve for. "
}], "Text",
 CellID->22951],

Cell[TextData[{
 "Here is a list of two simultaneous equations, to be solved for the \
variables ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->18140],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"a", " ", "x"}], "+", "y"}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"2", " ", "x"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", "a"}], ")"}], " ", "y"}]}], "==", "1"}]}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->11094],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", 
      FractionBox["1", 
       RowBox[{
        RowBox[{"-", "2"}], "+", "a", "-", 
        SuperscriptBox["a", "2"]}]]}]}], ",", 
    RowBox[{"y", "\[Rule]", 
     RowBox[{"-", 
      FractionBox["a", 
       RowBox[{"2", "-", "a", "+", 
        SuperscriptBox["a", "2"]}]]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{224, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->515410693]
}, Open  ]],

Cell[TextData[{
 "Here are some more complicated simultaneous equations. The two solutions \
are given as two lists of replacements for ",
 Cell[BoxData["x"], "InlineFormula"],
 " and ",
 Cell[BoxData["y"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->5174],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "^", "2"}], "+", 
       RowBox[{"y", "^", "2"}]}], "==", "1"}], ",", 
     RowBox[{
      RowBox[{"x", "+", 
       RowBox[{"3", " ", "y"}]}], "==", "0"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->2512],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"-", 
       FractionBox["3", 
        SqrtBox["10"]]}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      FractionBox["1", 
       SqrtBox["10"]]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      FractionBox["3", 
       SqrtBox["10"]]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{"-", 
       FractionBox["1", 
        SqrtBox["10"]]}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{327, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->218676234]
}, Open  ]],

Cell[TextData[{
 "This uses the solutions to evaluate the expression ",
 Cell[BoxData[
  RowBox[{"x", "+", "y"}]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->9551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"x", "+", "y"}], "/.", "%"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->2695],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", 
    SqrtBox[
     FractionBox["2", "5"]]}], ",", 
   SqrtBox[
    FractionBox["2", "5"]]}], "}"}]], "Output",
 ImageSize->{99, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->18727050]
}, Open  ]],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can solve any set of simultaneous ",
 StyleBox["linear",
  FontSlant->"Italic"],
 " or polynomial equations. "
}], "Text",
 CellID->23492],

Cell["\<\
When you are working with sets of equations in several variables, it is often \
convenient to reorganize the equations by eliminating some variables between \
them.\
\>", "Text",
 CellID->21292],

Cell[TextData[{
 "This eliminates ",
 Cell[BoxData["y"], "InlineFormula"],
 " between the two equations, giving a single equation \
for\[NonBreakingSpace]",
 Cell[BoxData["x"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->20237],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Eliminate", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"a", " ", "x"}], "+", "y"}], "==", "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"2", " ", "x"}], "+", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", "a"}], ")"}], " ", "y"}]}], "==", "1"}]}], "}"}], 
   ",", "y"}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->27530],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{"2", "-", "a", "+", 
     SuperscriptBox["a", "2"]}], ")"}], " ", "x"}], "\[Equal]", 
  "1"}]], "Output",
 ImageSize->{104, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->413032785]
}, Open  ]],

Cell[TextData[{
 "If you have several equations, there is no guarantee that there exists ",
 StyleBox["any",
  FontSlant->"Italic"],
 " consistent solution for a particular variable."
}], "Text",
 CellID->25296],

Cell[TextData[{
 "There is no consistent solution to these equations, so ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " returns ",
 Cell[BoxData[
  RowBox[{"{", "}"}]], "InlineFormula"],
 ", indicating that the set of solutions is empty. "
}], "MathCaption",
 CellID->21710],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "==", "1"}], ",", 
     RowBox[{"x", "==", "2"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->32389],

Cell[BoxData[
 RowBox[{"{", "}"}]], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->10857541]
}, Open  ]],

Cell[TextData[{
 "There is also no consistent solution to these equations for almost all \
values of\[NonBreakingSpace]",
 Cell[BoxData["a"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->5809],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "==", "1"}], ",", 
     RowBox[{"x", "==", "a"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->28786],

Cell[BoxData[
 RowBox[{"{", "}"}]], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->122589386]
}, Open  ]],

Cell[TextData[{
 "The general question of whether a set of equations has any consistent \
solution is quite a subtle one. For example, for most values of ",
 Cell[BoxData["a"], "InlineFormula"],
 ", the equations ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "==", "1"}], ",", 
    RowBox[{"x", "==", "a"}]}], "}"}]], "InlineFormula"],
 " are inconsistent, so there is no possible solution for ",
 Cell[BoxData["x"], "InlineFormula"],
 ". However, if ",
 Cell[BoxData["a"], "InlineFormula"],
 " is equal to ",
 Cell[BoxData["1"], "InlineFormula"],
 ", then the equations ",
 StyleBox["do",
  FontSlant->"Italic"],
 " have a solution. ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " is set up to give you ",
 StyleBox["generic",
  FontSlant->"Italic"],
 " solutions to equations. It discards any solutions that exist only when \
special constraints between parameters are satisfied."
}], "Text",
 CellID->18840],

Cell[TextData[{
 "If you use ",
 Cell[BoxData[
  ButtonBox["Reduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reduce"]], "InlineFormula"],
 " instead of ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 ", ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will however keep ",
 StyleBox["all",
  FontSlant->"Italic"],
 " the possible solutions to a set of equations, including those that require \
special conditions on parameters."
}], "Text",
 CellID->8182],

Cell[TextData[{
 "This shows that the equations have a solution only when ",
 Cell[BoxData[
  RowBox[{"a", "==", "1"}]], "InlineFormula"],
 ". The notation ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"a", "==", "1"}], "&&", 
   RowBox[{"x", "==", "1"}]}]], "InlineFormula"],
 " represents the requirement that ",
 StyleBox["both",
  FontSlant->"Italic"],
 " ",
 Cell[BoxData[
  RowBox[{"a", "==", "1"}]], "InlineFormula"],
 " ",
 StyleBox["and",
  FontSlant->"Italic"],
 " ",
 Cell[BoxData[
  RowBox[{" ", 
   RowBox[{"x", "==", "1"}]}]], "InlineFormula"],
 " should be ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->28569],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "==", "a"}], ",", 
     RowBox[{"x", "==", "1"}]}], "}"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->6120],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "\[Equal]", "1"}], "&&", 
  RowBox[{"x", "\[Equal]", "1"}]}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->289087492]
}, Open  ]],

Cell[TextData[{
 "This gives the complete set of possible solutions to the equation. The \
answer is stated in terms of a combination of simpler equations. ",
 Cell[BoxData["&&"], "InlineFormula"],
 " indicates equations that must simultaneously be true; ",
 Cell[BoxData["||"], "InlineFormula"],
 " indicates alternatives. "
}], "MathCaption",
 CellID->6292],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"a", " ", "x"}], "-", "b"}], "==", "0"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->14962],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"b", "\[Equal]", "0"}], "&&", 
    RowBox[{"a", "\[Equal]", "0"}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"a", "\[NotEqual]", "0"}], "&&", 
    RowBox[{"x", "\[Equal]", 
     FractionBox["b", "a"]}]}], ")"}]}]], "Output",
 ImageSize->{216, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->446938638]
}, Open  ]],

Cell["This gives a more complicated combination of equations. ", "MathCaption",
 CellID->12527],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"a", " ", 
      RowBox[{"x", "^", "2"}]}], "-", "b"}], "==", "0"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->26807],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"b", "\[Equal]", "0"}], "&&", 
    RowBox[{"a", "\[Equal]", "0"}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"a", "\[NotEqual]", "0"}], "&&", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "\[Equal]", 
       RowBox[{"-", 
        FractionBox[
         SqrtBox["b"], 
         SqrtBox["a"]]}]}], "||", 
      RowBox[{"x", "\[Equal]", 
       FractionBox[
        SqrtBox["b"], 
        SqrtBox["a"]]}]}], ")"}]}], ")"}]}]], "Output",
 ImageSize->{327, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->33137622]
}, Open  ]],

Cell["This gives a symbolic representation of all solutions. ", "MathCaption",
 CellID->8705],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "==", "a"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[22]:=",
 CellID->14713],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"C", "[", "1", "]"}], "\[Element]", "Integers"}], "&&", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", 
     RowBox[{"\[Pi]", "-", 
      RowBox[{"ArcSin", "[", "a", "]"}], "+", 
      RowBox[{"2", " ", "\[Pi]", " ", 
       RowBox[{"C", "[", "1", "]"}]}]}]}], "||", 
    RowBox[{"x", "\[Equal]", 
     RowBox[{
      RowBox[{"ArcSin", "[", "a", "]"}], "+", 
      RowBox[{"2", " ", "\[Pi]", " ", 
       RowBox[{"C", "[", "1", "]"}]}]}]}]}], ")"}]}]], "Output",
 ImageSize->{474, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->462710354]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Solve",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Solve"], "[", 
     StyleBox["lhs", "TI"], "==", 
     StyleBox["rhs", "TI"], ",", 
     StyleBox["x", "TI"], "]"}], Cell[TextData[{
     "solve an equation for ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["Solve",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Solve"], "[", "{", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "1"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "2"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["y", "TI"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "solve a set of simultaneous equations for ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", ",", "y", ",", "\[Ellipsis]"}], TraditionalForm]], 
      "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["Eliminate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Eliminate"], "[", "{", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "1"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "2"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "eliminate ",
     Cell[BoxData[
      FormBox[
       RowBox[{"x", ",", "\[Ellipsis]"}], TraditionalForm]], "InlineMath"],
     " in a set of simultaneous equations"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["Reduce",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Reduce"], "[", "{", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "1"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "1"], ",", 
     SubscriptBox[
      StyleBox["lhs", "TI"], "2"], "==", 
     SubscriptBox[
      StyleBox["rhs", "TI"], "2"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["y", "TI"], ",", 
     StyleBox["\[Ellipsis]", "TR"], "}", "]"}], "\[SpanFromLeft]"},
   {"", Cell["\<\
give a set of simplified equations, including all possible solutions\
\>", "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, False, False, False, False, False, {False}, False}, 
   "RowsIndexed" -> {}},
 GridBoxItemSize->{"Columns" -> {18, {
      Scaled[0.6]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->18913],

Cell["Functions for solving and manipulating equations. ", "Caption",
 CellID->6693],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Reduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reduce"]], "InlineFormula"],
 " also has powerful capabilities for handling equations specifically over \
real numbers or integers. \"",
 ButtonBox["Equations and Inequalities over Domains",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/EquationsAndInequalitiesOverDomains"],
 "\" discusses this in more detail. "
}], "Text",
 CellID->19664],

Cell[TextData[{
 "This reduces the equation assuming ",
 Cell[BoxData["x"], "InlineFormula"],
 " and ",
 Cell[BoxData["y"], "InlineFormula"],
 " are complex. "
}], "MathCaption",
 CellID->26203],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "+", 
     RowBox[{"y", "^", "2"}]}], "==", "1"}], ",", "y"}], "]"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->28995],

Cell[BoxData[
 RowBox[{
  RowBox[{"y", "\[Equal]", 
   RowBox[{"-", 
    SqrtBox[
     RowBox[{"1", "-", 
      SuperscriptBox["x", "2"]}]]}]}], "||", 
  RowBox[{"y", "\[Equal]", 
   SqrtBox[
    RowBox[{"1", "-", 
     SuperscriptBox["x", "2"]}]]}]}]], "Output",
 ImageSize->{182, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->52409804]
}, Open  ]],

Cell[TextData[{
 "This includes the conditions for ",
 Cell[BoxData["x"], "InlineFormula"],
 " and ",
 Cell[BoxData["y"], "InlineFormula"],
 " to be real. "
}], "MathCaption",
 CellID->14381],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "+", 
     RowBox[{"y", "^", "2"}]}], "==", "1"}], ",", "y", ",", "Reals"}], 
  "]"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->26946],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "1"}], "\[LessEqual]", "x", "\[LessEqual]", "1"}], "&&", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"y", "\[Equal]", 
     RowBox[{"-", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]]}]}], "||", 
    RowBox[{"y", "\[Equal]", 
     SqrtBox[
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}]]}]}], ")"}]}]], "Output",
 ImageSize->{272, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->120106569]
}, Open  ]],

Cell["This gives only the integer solutions. ", "MathCaption",
 CellID->1048],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "^", "2"}], "+", 
     RowBox[{"y", "^", "2"}]}], "==", "1"}], ",", "y", ",", "Integers"}], 
  "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->21768],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", 
     RowBox[{"-", "1"}]}], "&&", 
    RowBox[{"y", "\[Equal]", "0"}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", "0"}], "&&", 
    RowBox[{"y", "\[Equal]", 
     RowBox[{"-", "1"}]}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", "0"}], "&&", 
    RowBox[{"y", "\[Equal]", "1"}]}], ")"}], "||", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"x", "\[Equal]", "1"}], "&&", 
    RowBox[{"y", "\[Equal]", "0"}]}], ")"}]}]], "Output",
 ImageSize->{461, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->245846069]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->45405261],

Cell[TextData[ButtonBox["Symbolic Mathematics",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/SymbolicMathematicsOverview"]], "RelatedTutorials",
 CellID->391267452],

Cell[TextData[ButtonBox["Complex Polynomial Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ComplexPolynomialSystems"]], "RelatedTutorials",\

 CellID->65437048]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Solving Equations - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Symbolic Mathematics", "OverviewNavText"], BaseStyle -> 
        "Link", ButtonData -> "paclet:tutorial/SymbolicMathematicsOverview"]],
       "Text", FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/RelationalAndLogicalOperators"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/Inequalities-SymbolicMathematics"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 42, 44.9820914}", "context" -> "", 
    "keywords" -> {
     "algebraic equations", "eliminate variables", "eliminating variables", 
      "elimination", "equations", "FindRoot", "InverseFunction", 
      "inverse functions", "linear equations", 
      "numerical solution of equations", "polynomial equations", "reduce", 
      "Root", "roots", "roots of equations", "rules", 
      "simultaneous equations", "solutions", "solve", "solving equations", 
      "transcendental equations", "trigonometric equations"}, "index" -> True,
     "label" -> "Mathematica Tutorial", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "An expression like x^2+2x-7==0 represents an equation in Mathematica. \
You will often need to solve equations like this, to find out for what values \
of x they are true. This gives the two solutions to the quadratic equation \
x^2+2x-7=0. The solutions are given as replacements for x. Here are the \
numerical values of the solutions.", "synonyms" -> {"1.5.7"}, "title" -> 
    "Solving Equations", "type" -> "Tutorial", "uri" -> 
    "tutorial/SolvingEquations"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S1.5.7"->{
  Cell[2172, 64, 80, 2, 70, "Title",
   CellTags->{"S1.5.7", "7.1"},
   CellID->30806]},
 "7.1"->{
  Cell[2172, 64, 80, 2, 70, "Title",
   CellTags->{"S1.5.7", "7.1"},
   CellID->30806]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S1.5.7", 43832, 1620},
 {"7.1", 43931, 1624}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1547, 37, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2172, 64, 80, 2, 70, "Title",
 CellTags->{"S1.5.7", "7.1"},
 CellID->30806],
Cell[2255, 68, 520, 20, 70, "Text",
 CellID->29849],
Cell[2778, 90, 366, 11, 70, "MathCaption",
 CellID->14854],
Cell[CellGroupData[{
Cell[3169, 105, 229, 9, 28, "Input",
 CellID->24272],
Cell[3401, 116, 498, 19, 43, "Output",
 CellID->301653470]
}, Open  ]],
Cell[3914, 138, 86, 1, 70, "MathCaption",
 CellID->12294],
Cell[CellGroupData[{
Cell[4025, 143, 93, 3, 28, "Input",
 CellID->12054],
Cell[4121, 148, 373, 13, 36, "Output",
 CellID->671513081]
}, Open  ]],
Cell[4509, 164, 376, 12, 70, "MathCaption",
 CellID->146],
Cell[CellGroupData[{
Cell[4910, 180, 88, 3, 28, "Input",
 CellID->4229],
Cell[5001, 185, 262, 9, 36, "Output",
 CellID->591447060]
}, Open  ]],
Cell[5278, 197, 170, 5, 70, "MathCaption",
 CellID->10428],
Cell[CellGroupData[{
Cell[5473, 206, 161, 6, 70, "Input",
 CellID->10559],
Cell[5637, 214, 237, 7, 36, "Output",
 CellID->449655680]
}, Open  ]],
Cell[5889, 224, 1151, 34, 70, "DefinitionBox",
 CellID->29421],
Cell[7043, 260, 77, 1, 70, "Caption",
 CellID->22688],
Cell[7123, 263, 800, 18, 70, "Text",
 CellID->16366],
Cell[7926, 283, 201, 6, 70, "MathCaption",
 CellID->12033],
Cell[CellGroupData[{
Cell[8152, 293, 262, 10, 70, "Input",
 CellID->26685],
Cell[8417, 305, 1148, 43, 111, "Output",
 CellID->151059938]
}, Open  ]],
Cell[9580, 351, 95, 1, 70, "MathCaption",
 CellID->25329],
Cell[CellGroupData[{
Cell[9700, 356, 170, 6, 70, "Input",
 CellID->32261],
Cell[9873, 364, 1061, 38, 39, "Output",
 CellID->297368981]
}, Open  ]],
Cell[10949, 405, 402, 12, 70, "MathCaption",
 CellID->2208],
Cell[CellGroupData[{
Cell[11376, 421, 231, 8, 70, "Input",
 CellID->8982],
Cell[11610, 431, 1457, 52, 58, "Output",
 CellID->59714926]
}, Open  ]],
Cell[13082, 486, 145, 4, 70, "MathCaption",
 CellID->32506],
Cell[CellGroupData[{
Cell[13252, 494, 92, 3, 70, "Input",
 CellID->6878],
Cell[13347, 499, 834, 25, 54, "Output",
 CellID->245492487]
}, Open  ]],
Cell[14196, 527, 221, 6, 70, "Text",
 CellID->1897],
Cell[14420, 535, 173, 6, 70, "MathCaption",
 CellID->7887],
Cell[CellGroupData[{
Cell[14618, 545, 176, 6, 70, "Input",
 CellID->6538],
Cell[14797, 553, 497, 9, 70, "Message",
 CellID->405388678],
Cell[15297, 564, 268, 9, 36, "Output",
 CellID->88756058]
}, Open  ]],
Cell[15580, 576, 809, 25, 70, "Text",
 CellID->17540],
Cell[16392, 603, 179, 4, 70, "MathCaption",
 CellID->22795],
Cell[CellGroupData[{
Cell[16596, 611, 177, 6, 70, "Input",
 CellID->7266],
Cell[16776, 619, 464, 9, 70, "Message",
 CellID->728791810],
Cell[17243, 630, 279, 10, 36, "Output",
 CellID->77651796]
}, Open  ]],
Cell[17537, 643, 310, 10, 70, "MathCaption",
 CellID->24557],
Cell[CellGroupData[{
Cell[17872, 657, 229, 8, 70, "Input",
 CellID->2317],
Cell[18104, 667, 227, 7, 36, "Output",
 CellID->92464891]
}, Open  ]],
Cell[18346, 677, 320, 9, 70, "Text",
 CellID->32053],
Cell[18669, 688, 209, 7, 70, "MathCaption",
 CellID->18135],
Cell[CellGroupData[{
Cell[18903, 699, 204, 8, 70, "Input",
 CellID->12258],
Cell[19110, 709, 473, 9, 70, "Message",
 CellID->602935356],
Cell[19586, 720, 758, 28, 48, "Output",
 CellID->176468237]
}, Open  ]],
Cell[20359, 751, 1230, 36, 70, "DefinitionBox",
 CellID->21693],
Cell[21592, 789, 75, 1, 70, "Caption",
 CellID->16466],
Cell[21670, 792, 249, 7, 70, "Text",
 CellID->22951],
Cell[21922, 801, 278, 10, 70, "MathCaption",
 CellID->18140],
Cell[CellGroupData[{
Cell[22225, 815, 474, 18, 70, "Input",
 CellID->11094],
Cell[22702, 835, 543, 19, 52, "Output",
 CellID->515410693]
}, Open  ]],
Cell[23260, 857, 264, 8, 70, "MathCaption",
 CellID->5174],
Cell[CellGroupData[{
Cell[23549, 869, 410, 15, 70, "Input",
 CellID->2512],
Cell[23962, 886, 651, 25, 55, "Output",
 CellID->218676234]
}, Open  ]],
Cell[24628, 914, 172, 6, 70, "MathCaption",
 CellID->9551],
Cell[CellGroupData[{
Cell[24825, 924, 112, 4, 70, "Input",
 CellID->2695],
Cell[24940, 930, 296, 12, 60, "Output",
 CellID->18727050]
}, Open  ]],
Cell[25251, 945, 206, 8, 70, "Text",
 CellID->23492],
Cell[25460, 955, 204, 5, 70, "Text",
 CellID->21292],
Cell[25667, 962, 235, 8, 70, "MathCaption",
 CellID->20237],
Cell[CellGroupData[{
Cell[25927, 974, 429, 16, 70, "Input",
 CellID->27530],
Cell[26359, 992, 301, 11, 39, "Output",
 CellID->413032785]
}, Open  ]],
Cell[26675, 1006, 211, 6, 70, "Text",
 CellID->25296],
Cell[26889, 1014, 282, 9, 70, "MathCaption",
 CellID->21710],
Cell[CellGroupData[{
Cell[27196, 1027, 224, 8, 70, "Input",
 CellID->32389],
Cell[27423, 1037, 175, 6, 36, "Output",
 CellID->10857541]
}, Open  ]],
Cell[27613, 1046, 199, 6, 70, "MathCaption",
 CellID->5809],
Cell[CellGroupData[{
Cell[27837, 1056, 224, 8, 70, "Input",
 CellID->28786],
Cell[28064, 1066, 176, 6, 36, "Output",
 CellID->122589386]
}, Open  ]],
Cell[28255, 1075, 998, 30, 70, "Text",
 CellID->18840],
Cell[29256, 1107, 539, 20, 70, "Text",
 CellID->8182],
Cell[29798, 1129, 720, 29, 70, "MathCaption",
 CellID->28569],
Cell[CellGroupData[{
Cell[30543, 1162, 224, 8, 70, "Input",
 CellID->6120],
Cell[30770, 1172, 242, 8, 36, "Output",
 CellID->289087492]
}, Open  ]],
Cell[31027, 1183, 359, 8, 70, "MathCaption",
 CellID->6292],
Cell[CellGroupData[{
Cell[31411, 1195, 201, 8, 70, "Input",
 CellID->14962],
Cell[31615, 1205, 434, 15, 51, "Output",
 CellID->446938638]
}, Open  ]],
Cell[32064, 1223, 95, 1, 70, "MathCaption",
 CellID->12527],
Cell[CellGroupData[{
Cell[32184, 1228, 228, 9, 70, "Input",
 CellID->26807],
Cell[32415, 1239, 647, 24, 61, "Output",
 CellID->33137622]
}, Open  ]],
Cell[33077, 1266, 93, 1, 70, "MathCaption",
 CellID->8705],
Cell[CellGroupData[{
Cell[33195, 1271, 179, 6, 70, "Input",
 CellID->14713],
Cell[33377, 1279, 651, 20, 36, "Output",
 CellID->462710354]
}, Open  ]],
Cell[34043, 1302, 3029, 92, 70, "DefinitionBox",
 CellID->18913],
Cell[37075, 1396, 84, 1, 70, "Caption",
 CellID->6693],
Cell[37162, 1399, 449, 12, 70, "Text",
 CellID->19664],
Cell[37614, 1413, 194, 7, 70, "MathCaption",
 CellID->26203],
Cell[CellGroupData[{
Cell[37833, 1424, 224, 8, 70, "Input",
 CellID->28995],
Cell[38060, 1434, 394, 15, 45, "Output",
 CellID->52409804]
}, Open  ]],
Cell[38469, 1452, 191, 7, 70, "MathCaption",
 CellID->14381],
Cell[CellGroupData[{
Cell[38685, 1463, 241, 9, 70, "Input",
 CellID->26946],
Cell[38929, 1474, 535, 19, 52, "Output",
 CellID->120106569]
}, Open  ]],
Cell[39479, 1496, 77, 1, 70, "MathCaption",
 CellID->1048],
Cell[CellGroupData[{
Cell[39581, 1501, 244, 9, 70, "Input",
 CellID->21768],
Cell[39828, 1512, 695, 24, 36, "Output",
 CellID->245846069]
}, Open  ]],
Cell[CellGroupData[{
Cell[40560, 1541, 71, 1, 70, "RelatedTutorialsSection",
 CellID->45405261],
Cell[40634, 1544, 172, 4, 70, "RelatedTutorials",
 CellID->391267452],
Cell[40809, 1550, 173, 4, 70, "RelatedTutorials",
 CellID->65437048]
}, Open  ]]
}, Open  ]],
Cell[41009, 1558, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

