(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     47668,       1796]
NotebookOptionsPosition[     38153,       1475]
NotebookOutlinePosition[     40814,       1539]
CellTagsIndexPosition[     40721,       1533]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"LeastSquares" :> 
         Documentation`HelpLookup["paclet:ref/LeastSquares"], "LinearSolve" :> 
         Documentation`HelpLookup["paclet:ref/LinearSolve"], "MatrixRank" :> 
         Documentation`HelpLookup["paclet:ref/MatrixRank"], "NullSpace" :> 
         Documentation`HelpLookup["paclet:ref/NullSpace"], "RowReduce" :> 
         Documentation`HelpLookup["paclet:ref/RowReduce"]}, #, 2]& , 
       Evaluator -> Automatic], {1->"\<\"LeastSquares\"\>", 
      2->"\<\"LinearSolve\"\>", 3->"\<\"MatrixRank\"\>", 
      4->"\<\"NullSpace\"\>", 5->"\<\"RowReduce\"\>"}, "\<\"functions\"\>", 
      StyleBox["\<\"functions\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Solving Linear Systems", "Title",
 CellTags->{"S3.7.8", "8.1"},
 CellID->22511],

Cell[TextData[{
 "Many calculations involve solving systems of linear equations. In many \
cases, you will find it convenient to write down the equations explicitly, \
and then solve them using ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->7954],

Cell["\<\
In some cases, however, you may prefer to convert the system of linear \
equations into a matrix equation, and then apply matrix manipulation \
operations to solve it. This approach is often useful when the system of \
equations arises as part of a general algorithm, and you do not know in \
advance how many variables will be involved. \
\>", "Text",
 CellID->4235],

Cell[TextData[{
 "A system of linear equations can be stated in matrix form as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["m", "TB"], ".", 
    StyleBox["x", "TB"], "=", 
    StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   StyleBox["x", "TB"], TraditionalForm]], "InlineMath"],
 " is the vector of variables. "
}], "Text",
 CellID->1127],

Cell[TextData[{
 "Note that if your system of equations is sparse, so that most of the \
entries in the matrix ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 " are zero, then it is best to represent the matrix as a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " object. As discussed in\[NonBreakingSpace]\"",
 ButtonBox["Sparse Arrays: Linear Algebra",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/SparseArrays-LinearAlgebra"],
 "\", you can convert from symbolic equations to ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects using ",
 Cell[BoxData[
  ButtonBox["CoefficientArrays",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CoefficientArrays"]], "InlineFormula"],
 ". All the functions described here work on ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " objects as well as ordinary matrices. "
}], "Text",
 CellID->14633],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["LinearSolve",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/LinearSolve"], "[", 
     StyleBox["m", "TI"], ",", 
     StyleBox["b", "TI"], "]"}], Cell[TextData[{
     "a vector ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"],
     " which solves the matrix equation ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        RowBox[{
         StyleBox["m",
          FontWeight->"Bold"], ".", "x"}], "==", 
        StyleBox["b",
         FontWeight->"Bold"]}], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["NullSpace",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NullSpace"], "[", 
     StyleBox["m", "TI"], "]"}], Cell[TextData[{
     "a list of linearly independent vectors whose linear combinations span \
all solutions to the matrix equation ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        RowBox[{
         StyleBox["m",
          FontWeight->"Bold"], ".", "x"}], "==", 
        StyleBox["0",
         FontWeight->"Bold"]}], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["MatrixRank",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/MatrixRank"], "[", 
     StyleBox["m", "TI"], "]"}], Cell[TextData[{
     "the number of linearly independent rows or columns of ",
     Cell[BoxData[
      FormBox[
       StyleBox["m",
        FontWeight->"Bold"], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["RowReduce",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/RowReduce"], "[", 
     StyleBox["m", "TI"], "]"}], Cell[TextData[{
     "a simplified form of ",
     Cell[BoxData[
      FormBox[
       StyleBox["m",
        FontWeight->"Bold"], TraditionalForm]], "InlineMath"],
     " obtained by making linear combinations of rows"
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, False, False, {False}, False}, "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->28240],

Cell["Solving and analyzing linear systems. ", "Caption",
 CellID->1862],

Cell["Here is a 2x2 matrix. ", "MathCaption",
 CellID->15127],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "5"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "1"}], "}"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27577],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->193083219]
}, Open  ]],

Cell["This gives two linear equations. ", "MathCaption",
 CellID->2340],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"m", ".", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "==", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b"}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->10142],

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "+", 
     RowBox[{"5", " ", "y"}]}], ",", 
    RowBox[{
     RowBox[{"2", " ", "x"}], "+", "y"}]}], "}"}], "\[Equal]", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b"}], "}"}]}]], "Output",
 ImageSize->{159, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->577854366]
}, Open  ]],

Cell[TextData[{
 "You can use ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 " directly to solve these equations. "
}], "MathCaption",
 CellID->28400],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Solve", "[", 
  RowBox[{"%", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "y"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->6558],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      FractionBox["1", "9"], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "a"}], "+", 
        RowBox[{"5", " ", "b"}]}], ")"}]}]}], ",", 
    RowBox[{"y", "\[Rule]", 
     RowBox[{
      FractionBox["1", "9"], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"2", " ", "a"}], "-", "b"}], ")"}]}]}]}], "}"}], 
  "}"}]], "Output",
 ImageSize->{227, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->26226321]
}, Open  ]],

Cell[TextData[{
 "You can also get the vector of solutions by calling ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 ". The result is equivalent to the one you get from ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->3154],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{"m", ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->10935],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["1", "9"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "a"}], "+", 
      RowBox[{"5", " ", "b"}]}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "9"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", "a"}], "-", "b"}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{167, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->569073261]
}, Open  ]],

Cell[TextData[{
 "Another way to solve the equations is to invert the matrix ",
 Cell[BoxData["m"], "InlineFormula"],
 ", and then multiply ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"a", ",", "b"}], "}"}]], "InlineFormula"],
 " by the inverse. This is not as efficient as using ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->5831],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Inverse", "[", "m", "]"}], ".", 
  RowBox[{"{", 
   RowBox[{"a", ",", "b"}], "}"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->21295],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     FractionBox["a", "9"]}], "+", 
    FractionBox[
     RowBox[{"5", " ", "b"}], "9"]}], ",", 
   RowBox[{
    FractionBox[
     RowBox[{"2", " ", "a"}], "9"], "-", 
    FractionBox["b", "9"]}]}], "}"}]], "Output",
 ImageSize->{127, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->617260752]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RowReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RowReduce"]], "InlineFormula"],
 " performs a version of Gaussian elimination and can also be used to solve \
the equations. "
}], "MathCaption",
 CellID->15623],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RowReduce", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "5", ",", "a"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "1", ",", "b"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->17922],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", 
     RowBox[{
      FractionBox["1", "9"], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "a"}], "+", 
        RowBox[{"5", " ", "b"}]}], ")"}]}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", 
     RowBox[{
      FractionBox["1", "9"], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"2", " ", "a"}], "-", "b"}], ")"}]}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{268, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->15190666]
}, Open  ]],

Cell[TextData[{
 "If you have a square matrix ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 " with a non\[Hyphen]zero determinant, then you can always find a unique \
solution to the matrix equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["m", "TB"], ".", 
    StyleBox["x", "TB"], "=", 
    StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"],
 " for any ",
 Cell[BoxData[
  FormBox[
   StyleBox["b", "TB"], TraditionalForm]], "InlineMath"],
 ". If, however, the matrix ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 " has determinant zero, then there may be either no vector, or an infinite \
number of vectors ",
 Cell[BoxData[
  FormBox[
   StyleBox["x", "TB"], TraditionalForm]], "InlineMath"],
 " which satisfy ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["m", "TB"], ".", 
    StyleBox["x", "TB"], "=", 
    StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"],
 " for a particular ",
 Cell[BoxData[
  FormBox[
   StyleBox["b", "TB"], TraditionalForm]], "InlineMath"],
 ". This occurs when the linear equations embodied in ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 " are not independent."
}], "Text",
 CellID->28029],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 " has determinant zero, it is nevertheless always possible to find non\
\[Hyphen]zero vectors ",
 Cell[BoxData[
  FormBox[
   StyleBox["x", "TB"], TraditionalForm]], "InlineMath"],
 " that satisfy ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["m", "TB"], ".", 
    StyleBox["x", "TB"], "=", 
    StyleBox["0", "TB"]}], TraditionalForm]], "InlineMath"],
 ". The set of vectors ",
 Cell[BoxData[
  FormBox[
   StyleBox["x", "TB"], TraditionalForm]], "InlineMath"],
 " satisfying this equation form the ",
 StyleBox["null space",
  FontSlant->"Italic"],
 " or ",
 StyleBox["kernel",
  FontSlant->"Italic"],
 " of the matrix ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 ". Any of these vectors can be expressed as a linear combination of a \
particular set of basis vectors, which can be obtained using ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NullSpace",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NullSpace"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 ". "
}], "Text",
 CellID->30211],

Cell["\<\
Here is a simple matrix, corresponding to two identical linear equations. \
\>", "MathCaption",
 CellID->16211],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}]}], "}"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->11754],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->971122310]
}, Open  ]],

Cell["The matrix has determinant zero. ", "MathCaption",
 CellID->12253],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Det", "[", "m", "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->3293],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->35055773]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 " cannot find a solution to the equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     StyleBox["m", "TB"], ".", "x"}], "\[Equal]", 
    StyleBox["b",
     FontWeight->"Bold"]}], TraditionalForm]], "InlineMath"],
 " in this case. "
}], "MathCaption",
 CellID->20785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{"m", ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->26528],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearSolve", "::", "\<\"nosol\"\>"}], ":", 
  " ", "\<\"Linear equation encountered that has no solution. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/LinearSolve/nosol\\\", \
ButtonNote -> \\\"LinearSolve::nosol\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[9]:=",
 CellID->123244089],

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}]}], "]"}]], "Output",
 ImageSize->{249, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->6048113]
}, Open  ]],

Cell[TextData[{
 "There is a single basis vector for the null space of ",
 Cell[BoxData["m"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->10555],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NullSpace", "[", "m", "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->24671],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "2"}], ",", "1"}], "}"}], "}"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->24574712]
}, Open  ]],

Cell[TextData[{
 "Multiplying the basis vector for the null space by ",
 Cell[BoxData["m"], "InlineFormula"],
 " gives the zero vector. "
}], "MathCaption",
 CellID->7452],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", ".", 
  RowBox[{"%", "[", 
   RowBox[{"[", "1", "]"}], "]"}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->9375],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->114808124]
}, Open  ]],

Cell[TextData[{
 "There is only ",
 Cell[BoxData["1"], "InlineFormula"],
 " linearly independent row in ",
 Cell[BoxData["m"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->13640],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixRank", "[", "m", "]"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->8934],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->256138648]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NullSpace",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NullSpace"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MatrixRank",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixRank"]], "InlineFormula"],
 " have to determine whether particular combinations of matrix elements are \
zero. For approximate numerical matrices, the ",
 Cell[BoxData[
  ButtonBox["Tolerance",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Tolerance"]], "InlineFormula"],
 " option can be used to specify how close to zero is considered good enough. \
For exact symbolic matrices, you may sometimes need to specify something like \
",
 Cell[BoxData[
  RowBox[{
   ButtonBox["ZeroTest",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ZeroTest"], "->", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{
       ButtonBox["FullSimplify",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/FullSimplify"], "[", "#", "]"}], "==", "0"}], 
     "&"}], ")"}]}]], "InlineFormula"],
 " to force more to be done to test whether symbolic expressions are zero."
}], "Text",
 CellID->7986],

Cell["Here is a simple symbolic matrix with determinant zero. ", "MathCaption",
 CellID->28834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"2", " ", "a"}], ",", 
      RowBox[{"2", " ", "b"}], ",", 
      RowBox[{"2", " ", "c"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"3", " ", "a"}], ",", 
      RowBox[{"3", " ", "b"}], ",", 
      RowBox[{"3", " ", "c"}]}], "}"}]}], "}"}]}]], "Input",
 CellLabel->"In[13]:=",
 CellID->14248],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", " ", "a"}], ",", 
     RowBox[{"2", " ", "b"}], ",", 
     RowBox[{"2", " ", "c"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"3", " ", "a"}], ",", 
     RowBox[{"3", " ", "b"}], ",", 
     RowBox[{"3", " ", "c"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{273, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->8790580]
}, Open  ]],

Cell[TextData[{
 "The basis for the null space of ",
 Cell[BoxData["m"], "InlineFormula"],
 " contains two vectors. "
}], "MathCaption",
 CellID->28765],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NullSpace", "[", "m", "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->12796],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["c", "a"]}], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["b", "a"]}], ",", "1", ",", "0"}], "}"}]}], "}"}]], "Output",\

 ImageSize->{168, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->73055370]
}, Open  ]],

Cell[TextData[{
 "Multiplying ",
 Cell[BoxData["m"], "InlineFormula"],
 " by any linear combination of these vectors gives zero. "
}], "MathCaption",
 CellID->7262],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{"m", ".", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"x", " ", 
      RowBox[{"%", "[", 
       RowBox[{"[", "1", "]"}], "]"}]}], "+", 
     RowBox[{"y", " ", 
      RowBox[{"%", "[", 
       RowBox[{"[", "2", "]"}], "]"}]}]}], ")"}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->25052],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0", ",", "0"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->421728913]
}, Open  ]],

Cell[TextData[{
 "An important feature of functions like ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["NullSpace",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NullSpace"]], "InlineFormula"],
 " is that they work with ",
 StyleBox["rectangular",
  FontSlant->"Italic"],
 ", as well as ",
 StyleBox["square",
  FontSlant->"Italic"],
 ", matrices."
}], "Text",
 CellID->23987],

Cell[TextData[{
 "When you represent a system of linear equations by a matrix equation of the \
form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["m", "TB"], ".", 
    StyleBox["x", "TB"], "=", 
    StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"],
 ", the number of columns in ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 " gives the number of variables, and the number of rows gives the number of \
equations. There are a number of cases. "
}], "Text",
 CellID->24464],

Cell[BoxData[GridBox[{
   {Cell["Underdetermined", "TableHeader"], Cell["\<\
number of equations less than the number of variables; no solutions or many \
solutions may exist\
\>", "TableText"]},
   {Cell["Overdetermined", "TableHeader"], Cell["\<\
number of equations more than the number of variables; solutions may or may \
not exist\
\>", "TableText"]},
   {Cell["Nonsingular", "TableHeader"], Cell["\<\
number of independent equations equal to the number of variables, and \
determinant non\[Hyphen]zero; a unique solution exists\
\>", "TableText"]},
   {Cell["Consistent", "TableHeader"], Cell[
    "at least one solution exists", "TableText"]},
   {Cell["Inconsistent", "TableHeader"], Cell[
    "no solutions exist", "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, False, True, False, {False}, False}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->32557],

Cell["\<\
Classes of linear systems represented by rectangular matrices. \
\>", "Caption",
 CellID->29556],

Cell[TextData[{
 "This asks for the solution to the inconsistent set of equations ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "1"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->20170],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "1", "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->5450],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearSolve", "::", "\<\"nosol\"\>"}], ":", 
  " ", "\<\"Linear equation encountered that has no solution. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/LinearSolve/nosol\\\", \
ButtonNote -> \\\"LinearSolve::nosol\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[16]:=",
 CellID->389534161],

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "1", "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0"}], "}"}]}], "]"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->756216478]
}, Open  ]],

Cell["This matrix represents two equations, for three variables. ", \
"MathCaption",
 CellID->13293],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", "=", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "3", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", "1", ",", "3"}], "}"}]}], "}"}]}]], "Input",
 CellLabel->"In[17]:=",
 CellID->24099],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "3", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "1", ",", "3"}], "}"}]}], "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->389824106]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 " gives one of the possible solutions to this underdetermined set of \
equations. "
}], "MathCaption",
 CellID->19044],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"v", "=", 
  RowBox[{"LinearSolve", "[", 
   RowBox[{"m", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[18]:=",
 CellID->32331],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["2", "5"], ",", 
   FractionBox["1", "5"], ",", "0"}], "}"}]], "Output",
 ImageSize->{70, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->232856376]
}, Open  ]],

Cell["\<\
When a matrix represents an underdetermined system of equations, the matrix \
has a non\[Hyphen]trivial null space. In this case, the null space is spanned \
by a single vector. \
\>", "MathCaption",
 CellID->31004],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NullSpace", "[", "m", "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->19050],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "1"}], ",", 
    RowBox[{"-", "1"}], ",", "1"}], "}"}], "}"}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->131089621]
}, Open  ]],

Cell[TextData[{
 "If you take the solution you get from ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 ", and add any linear combination of the basis vectors for the null space, \
you still get a solution. "
}], "MathCaption",
 CellID->14066],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"m", ".", 
  RowBox[{"(", 
   RowBox[{"v", "+", 
    RowBox[{"4", " ", 
     RowBox[{"%", "[", 
      RowBox[{"[", "1", "]"}], "]"}]}]}], ")"}]}]], "Input",
 CellLabel->"In[20]:=",
 CellID->11738],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "1"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->97411633]
}, Open  ]],

Cell[TextData[{
 "The number of independent equations is the ",
 StyleBox["rank",
  FontSlant->"Italic"],
 " of the matrix ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["MatrixRank",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MatrixRank"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 ". The number of redundant equations is ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Length",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Length"], "[", 
   RowBox[{
    ButtonBox["NullSpace",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/NullSpace"], "[", 
    StyleBox["m", "TI"], "]"}], "]"}]], "InlineFormula"],
 ". Note that the sum of these quantities is always equal to the number of \
columns in ",
 Cell[BoxData[
  StyleBox["m", "TI"]], "InlineFormula"],
 "."
}], "Text",
 CellID->19623],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["LinearSolve",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/LinearSolve"], "[", 
     StyleBox["m", "TI"], "]"}], Cell[TextData[{
     "generate a function for solving equations of the form ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        StyleBox["m", "TB"], ".", 
        StyleBox["x", "TB"], "=", 
        StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, False}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.28, {0.56}, 0.28}, "ColumnsIndexed" -> {}, 
   "Rows" -> {0.2, {0.16}, 0.2}, "RowsIndexed" -> {}}},
 CellID->13843],

Cell[TextData[{
 "Generating ",
 StyleBox["LinearSolveFunction", "MR"],
 " objects. "
}], "Caption",
 CellID->441],

Cell[TextData[{
 "In some applications, you will want to solve equations of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["m", "TB"], ".", 
    StyleBox["x", "TB"], "=", 
    StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"],
 " many times with the same ",
 Cell[BoxData[
  FormBox[
   StyleBox["m", "TB"], TraditionalForm]], "InlineMath"],
 ", but different ",
 Cell[BoxData[
  FormBox[
   StyleBox["b", "TB"], TraditionalForm]], "InlineMath"],
 ". You can do this efficiently in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " by using ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["LinearSolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/LinearSolve"], "[", 
   StyleBox["m", "TI"], "]"}]], "InlineFormula"],
 " to create a single ",
 Cell[BoxData[
  ButtonBox["LinearSolveFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolveFunction"]], "InlineFormula"],
 " that you can apply to as many vectors as you want. "
}], "Text",
 CellID->3380],

Cell[TextData[{
 "This creates a ",
 Cell[BoxData[
  ButtonBox["LinearSolveFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolveFunction"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->19695],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  RowBox[{"LinearSolve", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "}"}], "]"}]}]], "Input",
 CellLabel->"In[21]:=",
 CellID->7918],

Cell[BoxData[
 TagBox[
  RowBox[{"LinearSolveFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"2", ",", "2"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{216, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->98198351]
}, Open  ]],

Cell["You can apply this to a vector. ", "MathCaption",
 CellID->1363],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "[", 
  RowBox[{"{", 
   RowBox[{"5", ",", "7"}], "}"}], "]"}]], "Input",
 CellLabel->"In[22]:=",
 CellID->12628],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["13", "5"], ",", 
   FractionBox["3", "5"]}], "}"}]], "Output",
 ImageSize->{59, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->791808058]
}, Open  ]],

Cell[TextData[{
 "You get the same result by giving the vector as an explicit second argument \
to ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->1088],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"5", ",", "7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->28854],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["13", "5"], ",", 
   FractionBox["3", "5"]}], "}"}]], "Output",
 ImageSize->{59, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->15481553]
}, Open  ]],

Cell[TextData[{
 "But you can apply ",
 Cell[BoxData["f"], "InlineFormula"],
 " to any vector you want. "
}], "MathCaption",
 CellID->24979],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"-", "5"}], ",", "9"}], "}"}], "]"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->18569],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["51", "5"], ",", 
   RowBox[{"-", 
    FractionBox["19", "5"]}]}], "}"}]], "Output",
 ImageSize->{74, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->165320319]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["LeastSquares",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/LeastSquares"], "[", 
     RowBox[{
      StyleBox["m", "TI"], ",", 
      StyleBox["b", "TI"]}], "]"}], Cell[TextData[{
     "give a vector ",
     Cell[BoxData[
      FormBox["x", TraditionalForm]], "InlineMath"],
     " that solves the least-squares problem ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        RowBox[{
         StyleBox["m", "TB"], ".", 
         StyleBox["x", "TB"]}], "==", 
        StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->2096742444],

Cell["Solving least-squares problems.", "Caption",
 CellID->1891092685],

Cell["This linear system is inconsistent. ", "MathCaption",
 CellID->128712739],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "6"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->89698293],

Cell[BoxData[
 RowBox[{
  RowBox[{"LinearSolve", "::", "\<\"nosol\"\>"}], ":", 
  " ", "\<\"Linear equation encountered that has no solution. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/LinearSolve/nosol\\\", \
ButtonNote -> \\\"LinearSolve::nosol\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[25]:=",
 CellID->887817749],

Cell[BoxData[
 RowBox[{"LinearSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "6"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "0", ",", "2"}], "}"}]}], "]"}]], "Output",
 ImageSize->{326, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->23084840]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["LeastSquares",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LeastSquares"]], "InlineFormula"],
 " finds a vector ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " that minimizes ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     StyleBox["m", "TB"], ".", 
     StyleBox["x", "TB"]}], "-", 
    StyleBox["b", "TB"]}], TraditionalForm]], "InlineMath"],
 " in the least-squares sense. "
}], "MathCaption",
 CellID->169712276],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LeastSquares", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5", ",", "6"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->373251134],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["8", "3"], ",", 
   RowBox[{"-", 
    FractionBox["23", "12"]}]}], "}"}]], "Output",
 ImageSize->{67, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->70911698]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Solving Linear Systems - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 42, 45.5133550}", "context" -> "", 
    "keywords" -> {
     "consistency", "Cramers rule", "division", "echelon form", "elimination",
       "equations", "Gaussian elimination", "inconsistent linear systems", 
      "kernel", "ladder form", "LinearSolve", "linear systems", "LINPACK", 
      "matrices", "matrix equations", "MatrixRank", 
      "Nonsingular linear systems", "NullSpace", 
      "overdetermined linear systems", "rank", "redundant linear equations", 
      "RowReduce", "solve", "tolerance", "Underdetermined linear systems", 
      "ZeroTest", "linalg", "lin algebra"}, "index" -> True, "label" -> 
    "Mathematica Tutorial", "language" -> "en", "paclet" -> "Mathematica", 
    "status" -> "None", "summary" -> 
    "Many calculations involve solving systems of linear equations. In many \
cases, you will find it convenient to write down the equations explicitly, \
and then solve them using Solve. In some cases, however, you may prefer to \
convert the system of linear equations into a matrix equation, and then apply \
matrix manipulation operations to solve it. This approach is often useful \
when the system of equations arises as part of a general algorithm, and you \
do not know in advance how many variables will be involved. A system of \
linear equations can be stated in matrix form as m.x=b, where x is the vector \
of variables. ", "synonyms" -> {"3.7.8"}, "title" -> "Solving Linear Systems",
     "type" -> "Tutorial", "uri" -> "tutorial/SolvingLinearSystems"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S3.7.8"->{
  Cell[1609, 47, 85, 2, 70, "Title",
   CellTags->{"S3.7.8", "8.1"},
   CellID->22511]},
 "8.1"->{
  Cell[1609, 47, 85, 2, 70, "Title",
   CellTags->{"S3.7.8", "8.1"},
   CellID->22511]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S3.7.8", 40510, 1522},
 {"8.1", 40609, 1526}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 984, 20, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1609, 47, 85, 2, 70, "Title",
 CellTags->{"S3.7.8", "8.1"},
 CellID->22511],
Cell[1697, 51, 342, 10, 70, "Text",
 CellID->7954],
Cell[2042, 63, 377, 7, 70, "Text",
 CellID->4235],
Cell[2422, 72, 399, 14, 70, "Text",
 CellID->1127],
Cell[2824, 88, 1137, 32, 70, "Text",
 CellID->14633],
Cell[3964, 122, 2257, 71, 70, "DefinitionBox",
 CellID->28240],
Cell[6224, 195, 72, 1, 70, "Caption",
 CellID->1862],
Cell[6299, 198, 61, 1, 70, "MathCaption",
 CellID->15127],
Cell[CellGroupData[{
Cell[6385, 203, 237, 9, 28, "Input",
 CellID->27577],
Cell[6625, 214, 302, 11, 36, "Output",
 CellID->193083219]
}, Open  ]],
Cell[6942, 228, 71, 1, 70, "MathCaption",
 CellID->2340],
Cell[CellGroupData[{
Cell[7038, 233, 208, 8, 70, "Input",
 CellID->10142],
Cell[7249, 243, 384, 14, 36, "Output",
 CellID->577854366]
}, Open  ]],
Cell[7648, 260, 220, 8, 70, "MathCaption",
 CellID->28400],
Cell[CellGroupData[{
Cell[7893, 272, 168, 6, 70, "Input",
 CellID->6558],
Cell[8064, 280, 586, 22, 51, "Output",
 CellID->26226321]
}, Open  ]],
Cell[8665, 305, 406, 13, 70, "MathCaption",
 CellID->3154],
Cell[CellGroupData[{
Cell[9096, 322, 175, 6, 70, "Input",
 CellID->10935],
Cell[9274, 330, 476, 18, 51, "Output",
 CellID->569073261]
}, Open  ]],
Cell[9765, 351, 448, 14, 70, "MathCaption",
 CellID->5831],
Cell[CellGroupData[{
Cell[10238, 369, 169, 6, 70, "Input",
 CellID->21295],
Cell[10410, 377, 417, 16, 51, "Output",
 CellID->617260752]
}, Open  ]],
Cell[10842, 396, 266, 8, 70, "MathCaption",
 CellID->15623],
Cell[CellGroupData[{
Cell[11133, 408, 270, 9, 70, "Input",
 CellID->17922],
Cell[11406, 419, 618, 23, 51, "Output",
 CellID->15190666]
}, Open  ]],
Cell[12039, 445, 1282, 43, 70, "Text",
 CellID->28029],
Cell[13324, 490, 1169, 41, 70, "Text",
 CellID->30211],
Cell[14496, 533, 121, 3, 70, "MathCaption",
 CellID->16211],
Cell[CellGroupData[{
Cell[14642, 540, 237, 9, 70, "Input",
 CellID->11754],
Cell[14882, 551, 302, 11, 36, "Output",
 CellID->971122310]
}, Open  ]],
Cell[15199, 565, 72, 1, 70, "MathCaption",
 CellID->12253],
Cell[CellGroupData[{
Cell[15296, 570, 94, 3, 70, "Input",
 CellID->3293],
Cell[15393, 575, 157, 5, 36, "Output",
 CellID->35055773]
}, Open  ]],
Cell[15565, 583, 421, 15, 70, "MathCaption",
 CellID->20785],
Cell[CellGroupData[{
Cell[16011, 602, 175, 6, 70, "Input",
 CellID->26528],
Cell[16189, 610, 432, 8, 70, "Message",
 CellID->123244089],
Cell[16624, 620, 421, 15, 36, "Output",
 CellID->6048113]
}, Open  ]],
Cell[17060, 638, 152, 5, 70, "MathCaption",
 CellID->10555],
Cell[CellGroupData[{
Cell[17237, 647, 102, 3, 70, "Input",
 CellID->24671],
Cell[17342, 652, 247, 9, 36, "Output",
 CellID->24574712]
}, Open  ]],
Cell[17604, 664, 171, 5, 70, "MathCaption",
 CellID->7452],
Cell[CellGroupData[{
Cell[17800, 673, 140, 5, 70, "Input",
 CellID->9375],
Cell[17943, 680, 204, 7, 36, "Output",
 CellID->114808124]
}, Open  ]],
Cell[18162, 690, 185, 7, 70, "MathCaption",
 CellID->13640],
Cell[CellGroupData[{
Cell[18372, 701, 102, 3, 70, "Input",
 CellID->8934],
Cell[18477, 706, 159, 5, 36, "Output",
 CellID->256138648]
}, Open  ]],
Cell[18651, 714, 1142, 34, 70, "Text",
 CellID->7986],
Cell[19796, 750, 95, 1, 70, "MathCaption",
 CellID->28834],
Cell[CellGroupData[{
Cell[19916, 755, 491, 17, 70, "Input",
 CellID->14248],
Cell[20410, 774, 546, 19, 36, "Output",
 CellID->8790580]
}, Open  ]],
Cell[20971, 796, 152, 5, 70, "MathCaption",
 CellID->28765],
Cell[CellGroupData[{
Cell[21148, 805, 102, 3, 70, "Input",
 CellID->12796],
Cell[21253, 810, 416, 16, 51, "Output",
 CellID->73055370]
}, Open  ]],
Cell[21684, 829, 164, 5, 70, "MathCaption",
 CellID->7262],
Cell[CellGroupData[{
Cell[21873, 838, 345, 12, 70, "Input",
 CellID->25052],
Cell[22221, 852, 214, 7, 36, "Output",
 CellID->421728913]
}, Open  ]],
Cell[22450, 862, 498, 19, 70, "Text",
 CellID->23987],
Cell[22951, 883, 529, 16, 70, "Text",
 CellID->24464],
Cell[23483, 901, 1090, 26, 70, "DefinitionBox",
 CellID->32557],
Cell[24576, 929, 106, 3, 70, "Caption",
 CellID->29556],
Cell[24685, 934, 311, 11, 70, "MathCaption",
 CellID->20170],
Cell[CellGroupData[{
Cell[25021, 949, 276, 10, 70, "Input",
 CellID->5450],
Cell[25300, 961, 433, 8, 70, "Message",
 CellID->389534161],
Cell[25736, 971, 370, 13, 36, "Output",
 CellID->756216478]
}, Open  ]],
Cell[26121, 987, 100, 2, 70, "MathCaption",
 CellID->13293],
Cell[CellGroupData[{
Cell[26246, 993, 258, 9, 70, "Input",
 CellID->24099],
Cell[26507, 1004, 323, 11, 36, "Output",
 CellID->389824106]
}, Open  ]],
Cell[26845, 1018, 260, 8, 70, "MathCaption",
 CellID->19044],
Cell[CellGroupData[{
Cell[27130, 1030, 202, 7, 70, "Input",
 CellID->32331],
Cell[27335, 1039, 258, 9, 51, "Output",
 CellID->232856376]
}, Open  ]],
Cell[27608, 1051, 225, 5, 70, "MathCaption",
 CellID->31004],
Cell[CellGroupData[{
Cell[27858, 1060, 102, 3, 70, "Input",
 CellID->19050],
Cell[27963, 1065, 278, 10, 36, "Output",
 CellID->131089621]
}, Open  ]],
Cell[28256, 1078, 324, 9, 70, "MathCaption",
 CellID->14066],
Cell[CellGroupData[{
Cell[28605, 1091, 219, 8, 70, "Input",
 CellID->11738],
Cell[28827, 1101, 203, 7, 36, "Output",
 CellID->97411633]
}, Open  ]],
Cell[29045, 1111, 809, 28, 70, "Text",
 CellID->19623],
Cell[29857, 1141, 783, 23, 70, "DefinitionBox",
 CellID->13843],
Cell[30643, 1166, 114, 5, 70, "Caption",
 CellID->441],
Cell[30760, 1173, 995, 33, 70, "Text",
 CellID->3380],
Cell[31758, 1208, 217, 8, 70, "MathCaption",
 CellID->19695],
Cell[CellGroupData[{
Cell[32000, 1220, 281, 10, 70, "Input",
 CellID->7918],
Cell[32284, 1232, 327, 12, 36, "Output",
 CellID->98198351]
}, Open  ]],
Cell[32626, 1247, 70, 1, 70, "MathCaption",
 CellID->1363],
Cell[CellGroupData[{
Cell[32721, 1252, 141, 5, 70, "Input",
 CellID->12628],
Cell[32865, 1259, 249, 9, 51, "Output",
 CellID->791808058]
}, Open  ]],
Cell[33129, 1271, 266, 9, 70, "MathCaption",
 CellID->1088],
Cell[CellGroupData[{
Cell[33420, 1284, 331, 12, 70, "Input",
 CellID->28854],
Cell[33754, 1298, 248, 9, 51, "Output",
 CellID->15481553]
}, Open  ]],
Cell[34017, 1310, 140, 5, 70, "MathCaption",
 CellID->24979],
Cell[CellGroupData[{
Cell[34182, 1319, 161, 6, 70, "Input",
 CellID->18569],
Cell[34346, 1327, 270, 10, 51, "Output",
 CellID->165320319]
}, Open  ]],
Cell[34631, 1340, 663, 22, 70, "DefinitionBox",
 CellID->2096742444],
Cell[35297, 1364, 71, 1, 70, "Caption",
 CellID->1891092685],
Cell[35371, 1367, 79, 1, 70, "MathCaption",
 CellID->128712739],
Cell[CellGroupData[{
Cell[35475, 1372, 428, 15, 70, "Input",
 CellID->89698293],
Cell[35906, 1389, 433, 8, 70, "Message",
 CellID->887817749],
Cell[36342, 1399, 517, 18, 36, "Output",
 CellID->23084840]
}, Open  ]],
Cell[36874, 1420, 498, 18, 70, "MathCaption",
 CellID->169712276],
Cell[CellGroupData[{
Cell[37397, 1442, 430, 15, 70, "Input",
 CellID->373251134],
Cell[37830, 1459, 269, 10, 51, "Output",
 CellID->70911698]
}, Open  ]]
}, Open  ]],
Cell[38126, 1473, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

