(* :Title: Documentation.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 6.0 *)
		     
(* :Copyright: Mathematica source code (c) 1999-2007, Wolfram Research, Inc. All rights reserved. *)

(* :Discussion: 
	
   This file is a component of the PacletManager Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   PacletManager uses a special system wherein one package context (PacletManager`) has its
   implementation split among a number of .m files. Each component file has its own private
   context, and also potentially introduces public symbols (in the PacletManager` context) and
   so-called "package" symbols, where the term "package" comes from Java terminology,
   referring to symbols that are visible everywhere within the implementation of PacletManager,
   but not to clients.
*)


GetVirtualCellGroup::usage = "GetVirtualCellGroup is an internal symbol."


Begin["`Package`"]

(* Called outside of this package, in buttons on failed fagment-loading messages. *)
resolveURL

docLinkCallback

initSearchCache

(* Called outside of this package, in buttons on Installed AddOns doc page. *)
helpBrowserLookup

clearMessageLinkCache

End[]  (* `Package` *)


(* Current context will be PacletManager`. *)

Begin["`Documentation`Private`"]


(* TODO: I don't want HelpLookupPacletURI exported from Documentation.m. Change that
   code to call a PM function instead of overloading HelpLookupPacletURI here.
   
   These are the functions from StartUp/Documentation.m that we add defs for.
   They have legacy definitions to support pure FE-based lookup of docs.
*)
Unprotect[Documentation`HelpLookupPacletURI]
Unprotect[Documentation`HelpLookupMessage]
Unprotect[Documentation`CreateMessageLink]
Unprotect[Documentation`ResolveLink]


(* Tells what file the PM thinks a paclet: link resolves to.
   Does allow downloads. Not asynchronous.
   Note that it works without a FE present.
*)
Documentation`ResolveLink[link_] :=
	Module[{language = Quiet[System`CurrentValue[$FrontEnd, Language]]},
		If[StringQ[language],
			Documentation`ResolveLink[link, language],
		(* else *)
			Documentation`ResolveLink[link, $Language]
		]
	]

Documentation`ResolveLink[link_, language_String] :=
	Module[{result},
		result = resolveURL[link, language, {Null, Null} (*ignored*), Null&, False, False, False, False];
		If[ListQ[result],
			First[result],
		(* else *)
			Null
		]
	]


(*  
  HelpLookupPacletURI
  
  Resolve link then open result notebook in FE or browser.
  For InstallFromDocRequest paclets, will result in download from server. Otherwise
  will open in browser window to web docs.
  Note that this is asynchronous; the NotebookLocate is called later, probably on ServiceLink.
  
  in:  String uri (e.g. "paclet:ref/Partition")
*)

Documentation`HelpLookupPacletURI::link = "Could not resolve paclet URL `1` to a document."

Documentation`HelpLookupPacletURI[link_String, feData_, language_, opts:OptionsPattern[]] := 
     Module[{feDataFixed, resolved, lang = language, completionFunc, nb},
     	If[!StringQ[lang], lang = "English"];
     	(* Resolve the FrontEnd`ButtonNotebook[] call right now in the kernel, because
     	   the FE gets the feData asynchronously and cannot resolve FrontEnd`ButtonNotebook[]
     	   at that time.
     	*)
     	feDataFixed = feData /. FrontEnd`ButtonNotebook[] :> ButtonNotebook[];
     	completionFunc = OptionValue[Documentation`CompletionFunction];
     	(* The AbortProtect is to prevent the FE's 6 second TimeConstrained for dynamic evals from
     	   aborting this operation. It doesn't do anything truly time-consuming, so it should
     	   be harmless to prevent aborts. [Note that it does need to wait for PM to initialize
     	   and for a DocumentationSearch Java query, and those operations could possibly hang in
     	   bug/error situations, but those operations are in Java and not abortable anyway.]
     	   No Mathematica code is time-consuming. One reason a 6-second TimeConstrained might kick
     	   in is that the search system takes several seconds to initialize the first time.
     	   Another reason is that prolonged swapping is always a possibility.
     	*)
     	resolved =
     		AbortProtect[
     			resolveURL[link, lang, feDataFixed, completionFunc, False, False, TrueQ[$allowDialogs], True]
     		];
     	(*  Note that the resolveURL call above is made asynchronously, so the ListQ[resolved]
     	    branch here will never be entered. Nevertheless, I preserve it in case we ever
	     	decide to handle some or all cases synchronously.
	    *)
     	Which[
     		resolved === $future,
	     		(* Was async call. Arbitrarily choose to return Null. *)
	     		Null,
	     	ListQ[resolved],
	     		(* Was successful synchronous call. *)
	     		nb = locateNotebook[resolved, feDataFixed];
	     		If[Head[nb] === NotebookObject,
	     			completionFunc[nb]
	     		];
	     		nb,
	     	True,
	     		(* Failure in async or sync call. Message should already hsve been issued. *)
	     		$Failed
     	]
     ]


(*  
  HelpLookupMessage
  
  LEGACY; remove eventually. Only existing MessageLink-style links will cal lthis function.
  No new message cells will.
  
  Resolve link to message page then open result notebook in FE or browser.
  For InstallFromDocRequest paclets, will result in download from server. This function is called from
  the button built for inclusion in message output, so it will rarely be called for non-system
  messages if the user doesn't already have the paclet installed. The exception is if a user with
  the paclet gives a notebook containing a message button to a user who doesn't have the paclet and
  that user clicks the button.
  Note that this is asynchronous; the NotebookLocate is called later, on ServiceLink.
*)
Documentation`HelpLookupMessage[{context_String, symbolName_String, msgTag_String},
									nb_, language_, opts:OptionsPattern[]] :=
	Module[{nbFixed},
     	(* Resolve the FrontEnd`ButtonNotebook[] call right now in the kernel, because
     	   the FE gets the feData asynchronously and cannot resolve FrontEnd`ButtonNotebook[]
     	   at that time.
     	*)
     	nbFixed = nb /. FrontEnd`ButtonNotebook[] :> ButtonNotebook[];
		getPacletManager[]@lookupMessageAsync[symbolName, context, msgTag, language, nbFixed];
		$future
	]
	

(* Messages pre-stuffed into the cache are ones that we don't want to enter Java when handling. *)
initMessageLinkCache[] := 
	(
		$messageLinkCache["PacletManager`", "PacletManager", _, _] = Null; (* PM failed to launch. *)
(* TODO: Create and cache a link for $messageLinkCache["JLink`", "Java", __] as a doc page for Java::excpn etc. *)
		$messageLinkCache["JLink`", "Java", _, _] = Null;
		$messageLinkCache["JLink`", "InstallJava", _, _] = Null;
		$messageLinkCache["JLink`", sym_, _, _] := "paclet:JLink/ref/" <> sym
	)
	
clearMessageLinkCache[] :=
	(
		Clear[$messageLinkCache];
		initMessageLinkCache[]
	)

Internal`SetValueNoTrack[$messageLinkCache, True]
initMessageLinkCache[]

(* This is the function that message-formatting code in the kernel should call to decide
   whether to create a link button for the message. This function returns either a "paclet:..."
   string suitable for the ButtonData of a link, or non-string to indicate that no link
   should be created. It will return a string iff such a link can be resolved by the
   PacletManager to a document. Will consider server paclets, but only if $AllowInternet and
   $AllowDocumentationUpdates are true, and if they are InstallFromDocRequest paclets.
*)
Documentation`CreateMessageLink[context_String, symbolName_String, msgTag_String, language_String] :=
	Module[{uri},
		(* Important to use a caching scheme, because it is good to avoid calls into
		   Java here. There are reentrancy issues (e.g., message during a failure in
		   InstallJava would not want to re-try a call into Java.
		*)
		uri = $messageLinkCache[context, symbolName, msgTag, language];
		If[Head[uri] === $messageLinkCache,
			uri = cml[context, symbolName, msgTag, language];
			If[!StringQ[uri],
				uri = Null
			];
			$messageLinkCache[context, symbolName, msgTag, language] = uri
		];
		uri
	]

cml[context_String, symbolName_String, msgTag_String, language_String] :=
	Block[{$inCML = True, $JavaExceptionHandler = Null&, oldExc, result},
		oldExc = GetJavaException[];
		result = getPacletManager[]@createMessageURI[context, symbolName, msgTag, language];
		JLink`Package`setJavaException[oldExc];
		result
	] /; !TrueQ[$inCML]


Protect[Documentation`HelpLookupPacletURI]
Protect[Documentation`HelpLookupMessage]
Protect[Documentation`CreateMessageLink]
Protect[Documentation`ResolveLink]



(* Returns {nbFramgentFileName, errorText}. errorText should be "" iff no error.
   errorText is put into cell that user sees.
   TODO: When I refactor resolveLink[] to Throw instead of issue Messages directly,
   I can Catch error types here and set errorText accordingly.
*)
GetVirtualCellGroup[uri_String] :=
	Module[{language, result, nbFragmentFile},
		language = System`CurrentValue[$FrontEnd, Language];
		If[!StringQ[language], language = $Language];
		(* Quiet messages here, as they interfere with clean presentation of
		   an error-describing cell in the newly-opened virtual cell group.
		*)
		result = Quiet[resolveURL[uri, language, {Null, Null} (*ignored*), Null&, False, False, False, False]];
		If[ListQ[result],
			{First[result], ""},
		(* else *)
			{"", "Unknown error." (* This text is not used. *)}
		]
	]




(* The first component of a paclet URI is the LinkBase. For convenience, though,
   we want to allow users to leave out the "WolframMathematica" LinkBase for built-in system
   docs. That is:
       paclet:ref/Partition
   instead of requiring
       paclet:WolframMathematica/ref/Partition
   Thus we look for one of the special categories of system docs as the first
   element of the URI and recognize that this is actually the beginning of the resourceName
   part and that there is an empty LinkBase component that should be defaulted to
   WolframMathematica. Add to this list as new categories (directories) are created for
   the root level of the docs hierarchy.
      
*) 

(* Including ReferencePages, Guides and Tutorials allows us to support non-shortened links. *)
$specialLinkBases = {"ref", "guide", "tutorial", "note", "ReferencePages", "Guides", "Tutorials", "Notes"}


(* 
    Accepts:
        paclet:Guides/Mathematica   (the typical use; default to WolframMathematica LinkBase))
        paclet://Guides/Mathematica (legacy support for incorrect paclet:// urls)
        
        paclet:LinkBase/Guides/SomeGuide   (the typical use)
        paclet://LinkBase/Guides/SomeGuide (legacy support for incorrect paclet:// urls)
        
        paclet:SingleWord   (must match a LinkBase provided by a paclet; give that paclet's root guide page)
        SingleWord          (if matches a LinkBase, treat like paclet:SingleWord. Otherwise, treat as
                             a poor-man's search: look for any paclet (start with WolframMathematica) that
                             can provide this doc via Symbols, Guides, etc.)
        
        Guides/Mathematica     (/ but no paclet:, simply prepend with paclet:)
        
        paclet:SearchResult?query   (special treatment as search)
        
   Does not accept:
        http://anything
        Symbols/Partition  (if it has a / char but no paclet:, it must be a full URL except missing paclet:)


   Returns list:
    {
     "full path string or http URL to notebook",
     "fragment part",
     "paclet URL" (possibly modified from original to make legal}
    }
    
    or $Failed or $future.
       
   The first element is the full path string or http URL to a notebook,
   or $Failed if there is an error. This will simply prevent the NotebookLocate
   from firing because the first part is not a string. Also will issue message on error.
   The second element is the fragment part, if it exists (as in
   the "abc" part of paclet:Guides/Foo#abc, typically a cell tag).
   If no fragment part exists, None is used, which is what the front end expects
   in NotebookLocate.
*)
resolveURL[url_String, language_String, feData_, completionFunc_,
			localFilesOnly:(True | False), layoutFilesOnly:(True | False), interactive:(True | False), async:(True | False)] :=
	Module[{resource, resourceWithoutFragment, wordsOnly, fallThroughToSearch, looksLikePacletURI,
				allowCaseInsensitiveLookup, fragPart, parts, linkBase, resourceName, context, pathOrURL,
					symbolNamePieces, directHitResult, searchTextTranslated, directHitURI, pmData},

		wordsOnly = False;
		context = Null;
		
		looksLikePacletURI = StringMatchQ[url, (WordCharacter.. ~~ "/").. ~~ (WordCharacter | "$")..];
		(* Some links should simply fail if not resolved, and not fall through to a search. In this category
		   we put links that have an explicit paclet: prefix, and links that have a / in them (like someone entered
		   ref/Foo).
		*)
		fallThroughToSearch = !StringMatchQ[url, "paclet:*"] && (StringMatchQ[url, "* *"] || !looksLikePacletURI);
		allowCaseInsensitiveLookup = !fallThroughToSearch;
		
		(* Drop paclet: at start. NOTE temporarily, handle either paclet: or paclet:// at beginning.
		   The paclet: form is the one we will use, but at this point some links still have paclet://.
		*)
		resource =
			Which[
				StringMatchQ[url, "paclet://*"],
					(* paclet:// is not a legal prefix, but we can easily handle it if users mistakenly use it. *)
					StringDrop[url, 9],
				StringMatchQ[url, "paclet:/*"],
					(* paclet:/ is not a legal prefix, but we can easily handle it if users mistakenly use it. *)
					StringDrop[url, 8],
				StringMatchQ[url, "paclet:*"],
					StringDrop[url, 7],
				StringMatchQ[url, "* *"],
					wordsOnly = True;
					url,
				looksLikePacletURI,
					(* Treat like a normal paclet: URL except user left out paclet: prefix. *)
					url,
				StringMatchQ[url, "installedaddons", IgnoreCase->True],
					(* Entering InstalledAddOns should bypass a search (don't set wordsOnly in this case). *)
					url,
				True,
					(* "wordsOnly" URLs do not start with paclet and either have spaces or do not have a / in them.
					   They are like weak searches, where a user types just the name of, say,
					   a symbol or a guide page. WordsOnly is not a very good name for this, because we also
					   get here on gibberish strings like %[+]#, but that's OK, and in fact the search system
					   wants first crack at these in case they are operators.
					*)
					wordsOnly = True;
					If[StringLength[url] < 100 && url != "`",
						(* Here we extract the context in case it's a symbol like Foo`Function.
						   For example, "a`b`c" --> "a`b`", or "abc" -> "". It doesn't matter
						   if it isn't a symbol name but happens to have a ` in it--it simply
						   won't be found by context-based lookup. 
						*)
						symbolNamePieces = StringSplit[url, "`"];
						context = StringJoin @@ Riffle[Append[Most[symbolNamePieces], ""], "`"];
						If[context == "", context = Null];
						Last[symbolNamePieces],
					(* else *)
						url
					]
			];
		
		(* Want to give the search system first crack at resolving "direct hits", which are 
		   cases where he user types in a word that is considered by us as a synonym for a
		   built-in symbol. 
		*)
		pmData = Null;
		If[wordsOnly,
			Needs["DocumentationSearch`"];
			directHitResult = Symbol["DocumentationSearch`DirectHitSearch"][resource];
			If[MatchQ[directHitResult, {_String, _String}],
				{searchTextTranslated, directHitURI} = directHitResult;
				wordsOnly = False;
				pmData = {resource, searchTextTranslated};
				context = Null;
				resource = directHitURI,
			(* else *)
				pmData = {resource, resource}
			]
		];
		
		(* Separate fragment part. *)
		parts = StringSplit[resource, "#"];
		resourceWithoutFragment = First[parts];
		If[Length[parts] > 1,
			fragPart = parts[[2]],
		(* else *)
			fragPart = ""
		];
		
		(* Split resource into linkbase and resourcename. *)
		Which[
			StringQ[context],
				{linkBase, resourceName} = {Null, resourceWithoutFragment},
			wordsOnly,
				(* The only time linkBase is empty is when we have a wordsOnly url (including no paclet:).
				   See header comment for this function on how such urls are treated.
				*)
				{linkBase, resourceName} = {"", resourceWithoutFragment},
			True,
				{linkBase, resourceName} = splitResource[resourceWithoutFragment]
		];

		If[async,
			getPacletManager[]@resolveDocResourceAsync[linkBase, resourceName, context, language, url, fragPart,
								allowCaseInsensitiveLookup, localFilesOnly, layoutFilesOnly, fallThroughToSearch, feData, completionFunc, pmData];
			$future,
		(* else *)
			Block[{$JavaExceptionHandler = If[interactive, JavaBlock[errorDialog[GetJavaException[]]]&, Null&]},
				pathOrURL = getPacletManager[]@resolveDocResource[linkBase, resourceName, context, language,
													allowCaseInsensitiveLookup, localFilesOnly, layoutFilesOnly]
			];
			If[StringQ[pathOrURL],
				{pathOrURL, fragPart, url},
			(* else *)
				(* Lookup failed. If desired, do a search; otherwise fail. *)
				Which[
					(* Bit of a hack: Because "note/" docs removed from product, map them to the web. *)
					StringMatchQ[resource, "note/*SourceInformation", IgnoreCase->True],
						{"http://reference.wolfram.com/mathematica/" <> resource <> ".html", "", ""},
					StringMatchQ[resource, "guide/installedaddons", IgnoreCase->True] || StringMatchQ[resource, "installedaddons", IgnoreCase->True],
						{generateNotebook["InstalledAddOns", "InstalledAddOns"], "", url},
					fallThroughToSearch,
						{generateNotebook[resource, "Search"], "", url},
					True,
						If[interactive,
							errorDialog[ToString[StringForm[Documentation`HelpLookupPacletURI::link, url]]],
						(* else *)
							Message[Documentation`HelpLookupPacletURI::link, url]
						];
						$Failed
				]
			]
		]
	]


(* Called for asynchronous resolveURL. *)
docLinkCallback[resourcePath:(_String | Null), fragPart_String, origURL_String, resourceName_String,
						fallThroughToSearch_, feData_, completionFunc_, pmData_] :=
	Module[{nb, searchText, searchTextTranslated},
		If[StringQ[resourcePath],
			nb = locateNotebook[{resourcePath, fragPart, origURL}, feData, pmData],
		(* else *)
			(* Resource could not be found. origURL is what user typed into the Go field.
			   Some links should simply fail if not resolved, and not fall through to a search. In this category
			   we put links that have an explicit paclet: prefix, and links that have a / in them (like someone entered
			   ref/Foo).
			*)
			Which[
				(* Bit of a hack: Because "note/" docs removed from product, map them to the web. *)
				StringMatchQ[resourceName, "note/*SourceInformation", IgnoreCase->True],
					nb = locateNotebook[{"http://reference.wolfram.com/mathematica/" <> resourceName <> ".html", "", ""}, feData],
				StringMatchQ[origURL, "installedaddons", IgnoreCase->True] ||
							StringMatchQ[origURL, "*/installedaddons", IgnoreCase->True],
					nb = locateNotebook[{generateNotebook["InstalledAddOns", "InstalledAddOns"], "", origURL}, feData],
				fallThroughToSearch,
					(* Note that we ignore fragment part if this is a search (# might be part of search text) *)
					nb = locateNotebook[{generateNotebook[origURL, "Search"], "", origURL}, feData],
				True,
					(* This just causes the "file you requested not found" FE dialog. *)
					nb = NotebookLocate[{origURL, None}, FrontEnd`HistoryData -> {feData, origURL}];
					$Failed
			]
		];
		(* Call completion func (supplied by FE). *)
		If[Head[nb] === NotebookObject,
			completionFunc[nb]
		];
	]
		
	
(* Generates and manages caching of two types of dynamically-generated notebooks:
   search results and the InstalledAddOns notebook.
*)
generateNotebook[query_String, type_String] :=
	Module[{encodedQuery, resultNbFile, result, nb},
		(* See if this result notebook exists in cache, but always regenerate if it's InstalledAddOns. *)
		LoadJavaClass["java.net.URLEncoder"];
		encodedQuery = Symbol["java`net`URLEncoder`encode"][query, "UTF-8"];
		resultNbFile = 
			If[type == "Search",
				lookupSearchCache[encodedQuery],
			(* else *)
				Null
			];
		If[resultNbFile === Null,
			If[type == "Search",
				Needs["DocumentationSearch`"];
				nb = Symbol["DocumentationSearch`ExportSearchResults"][
						Symbol["DocumentationSearch`SearchDocumentation"][query], "Notebook"],
			(* else *)
				(* type = "InstalledAddOns" is a request for the dynamically-generated page.
				   We store this notebook in the cache not so that _we_ can retrieve it again
				   (we always want to regenerate it), but so the FE can load it as a file.
				*)
				nb = Symbol["Documentation`MakeInstalledAddOnsNotebook"][getInstalledAddOns[]]
			];
			resultNbFile = cacheSearchResult[encodedQuery, nb]
		];
		resultNbFile
	]
	
	
(* Utility function to call NotebookLocate properly. *)
locateNotebook[{path_String, fragPart:(_String | None), fixedURL_String}, feData_, pmData:_:Null] :=
	Module[{nb, completionFunc = Null&, urlString},
		If[ListQ[pmData],
			(* If pmData is a list, then this page was found via direct-hit search (e.g., user
			   typed in Plot and got directed to ref/Plot). We want a link to appear in header
			   that lets user force a search on this term if a direct hit is not what they
			   wanted. This is done by setting special TaggingRules that are used in the
			   help viewer header cell to create a "search for all results containing ..." line.
			*)
			{searchText, searchTextTranslated} = pmData;
			(* Trim whitespace from ends of searchText. This is so that "integrate " doesn't
			   look like a different word than "Integrate".
			*)
			searchText = StringReplace[searchText, StartOfString ~~ WhitespaceCharacter... ~~ 
								ShortestMatch[chars___] ~~ WhitespaceCharacter... ~~ EndOfString :> chars];
			If[ToLowerCase[searchText] === ToLowerCase[searchTextTranslated],
				searchText = ""
			];
			With[{searchText = searchText, searchTextTranslated = searchTextTranslated},
				completionFunc = (
					FEPrivate`Set[CurrentValue[#, {TaggingRules, "SearchText"}], searchText];
					FEPrivate`Set[CurrentValue[#, {TaggingRules, "SearchTextTranslated"}], searchTextTranslated];
				)&
			];								
		];		
		Which[
			treatLikeURL[path],
				urlString = If[StringMatchQ[path, "http:*", IgnoreCase->True],
								path,
							(* else *)
								"file:" <> StringReplace[path, $PathnameSeparator->"/"]
							] <> If[StringQ[fragPart] && fragPart != "", "#" <> fragPart, ""];
				NotebookLocate[{URL[urlString], ""}, FrontEnd`HistoryData -> {feData, fixedURL}, FrontEnd`ReturnNotebookObject->True],
			StringQ[fragPart] && fragPart != "" && DigitQ[fragPart],
				(* NotebookLocate cannot handle CellIDs (we assume a CellID is intended when
			  	   frag is all digits). Therefore must also use NotebookFind to move to CellID.
				*)
				nb = NotebookLocate[{path, None}, FrontEnd`HistoryData -> {feData, fixedURL}, 
								FrontEnd`ReturnNotebookObject->True, FrontEnd`CompletionFunction -> completionFunc];
				If[Head[nb] === NotebookObject,
					NotebookFind[nb, fragPart, Next, CellID]
				];
				nb,
			True,
				(* No fragment specifier, or frag was a cell tag, not a cell id. *)
				NotebookLocate[{path, fragPart}, FrontEnd`HistoryData -> {feData, fixedURL},
							FrontEnd`ReturnNotebookObject->True, FrontEnd`CompletionFunction -> completionFunc]
		]
	]



(* Splits a resource into a LinkBase and resourceName components.
   Resource comes in as something like
		"ref/Partition"
		"tutorial/WorkingWithLists"
		"JLink/ref/InstallJava"
*)
splitResource[resource_String] :=
	Module[{linkBase, resourceName, slashPos},
		slashPos = Flatten[StringPosition[resource, "/"]];
		If[slashPos =!= {},
			linkBase = StringTake[resource, First[slashPos] - 1];
			If[MemberQ[$specialLinkBases, linkBase],
				(* Wasn't a LinkBase; rather, the LinkBase was missing and the first
				   part of the URI was a reserved word like ref, guide, etc.
				*)
				linkBase = "WolframMathematica";
				resourceName = resource,
			(* else *)
				resourceName = StringDrop[resource, First[slashPos]]
			],
		(* else *)
			linkBase = resource;
			resourceName = ""
		];
		{linkBase, resourceName}
	]
	

(* Treat everything that isn't a .m or .nb file as a URL. What this means in practice is that
   it will get wrapped in URL[]. The FE will launch these files with the viewer program that
   a web browser would.
*)
treatLikeURL[s_String] := 
	StringMatchQ[s, "http:*", IgnoreCase->True] ||
		!StringMatchQ[s, "*.nb", IgnoreCase->True] && !StringMatchQ[s, "*.m", IgnoreCase->True]


initSearchCache[dir_String] :=
	(
		$searchCacheDir = dir;
		(* Deactivating messages because at this point in time you often get a DeleteFile failure
		   because the FE still holds onto the file handle for some reason. When that changes,
		   remove the DeactivateMessages.
		*)
		Internal`DeactivateMessages[
			DeleteFile /@ FileNames["*.nb", $searchCacheDir]
		]
	)

lookupSearchCache[encodedQuery_String] := 
	Module[{nbFile = ToFileName[$searchCacheDir, nbNameFromSearchQuery[encodedQuery]]},
		If[FileType[nbFile] === File, nbFile, Null]
	]

cacheSearchResult[encodedQuery_String, nb_Notebook] :=
	Module[{nbFile = ToFileName[$searchCacheDir, nbNameFromSearchQuery[encodedQuery]], oldNbs},
		(* Keep max of 20 search results. Scan through cache and delete oldest files
		   down to the point where at most 20 remain.
		*)
		oldNbs = Developer`FileInformation /@ FileNames["*.nb", $searchCacheDir];
		oldNbs = Sort[oldNbs, ((Date /. #1) > (Date /. #2))&];
		oldNbs = Drop[oldNbs, Min[20, Length[oldNbs]]];
		(* Deactivating messages because at this point in time you often get a DeleteFile failure
		   because the FE still holds onto the file handle for some reason. When that changes,
		   remove the Quiet.
		*)
		Quiet[
			DeleteFile /@ (File /. oldNbs)
		];
		(* Pruning of cache finished. Save out the new result notebook. *)
		Export[nbFile, nb, "NB"]
	]
	

(* Query string is already URL-encoded. The only char that could still exist in it
   that isn't safe for filenames is *. Cap the filename at 80 chars in case the user had
   a large amount of text selected in the notebook and inadvertantly did a search.
   This means that if you type a legitimate 90 char search string and then search again
   after changing the last few chars it won't work--you'll get the same search result
   notebook from the cache.
*)
nbNameFromSearchQuery[encodedQuery_String] :=
	"SearchResults_" <>
		StringReplace[StringTake[encodedQuery, Min[StringLength[encodedQuery], 80]], "*" -> "%2A"] <>
			".nb"


(********************************  Installed AddOns List  ****************************)

(*
	Called from root guide page to get info on installed addons for dynamically-built guide page.
	Returns the list {pacletLinks, legacyAppLinks}, where:
	
	   pacletLinks is     {{"PacletName", "paclet:LinkBase"}, ...}
	   legacyAppLinks is  {{"AppName", "FirstIndexTag"}, ...}
*)

getInstalledAddOns[] :=
	Module[{appLocations, appDirs, pacletDirs, appLinks, pacletLinks, needsPacletRebuild}, 
		appLocations = CurrentValue[$FrontEnd, "AddOnHelpPath"];
		If[!ListQ[appLocations], Return[{{}, {}}]];
		appLocations = appLocations /. FrontEnd`FileName[a_, ___] :> ToFileName[a];
		appDirs = Select[Flatten[FileNames["*", #]& /@ appLocations], (FileType[#] === Directory) &];
		pacletDirs = Select[appDirs, FileNames["PacletInfo.m", #] =!= {} &];
		appDirs = Complement[appDirs, pacletDirs];
		(* Create {name, indexTagLink} pairs for apps that have a BrowserCategories.m file. Use Union to
		   avoid duplicates for apps present in more than one location.
		*)
		appLinks = Union[Cases[getAppNamesAndTags /@ appDirs, {_String, _String}], SameTest -> (First[#1] == First[#2]&)];

		(* Detect and record whether Help Browser index needs rebuilding (rebuilding is done when links are clicked). *)
		$needsHBRebuild = TrueQ[$needsHBRebuild] || ($lastAppLinks =!= appLinks);
		needsPacletRebuild = $lastPacletDirs =!= pacletDirs;
		$lastAppLinks = appLinks;
		$lastPacletDirs = pacletDirs;
	
		(* pacletLinks is a list of {"PacletName", "paclet:LinkBase"} pairs. It excludes
		   paclets that are inside the Mathematica layout or are newer versions of paclets within the layout.
		   The idea is that paclets in the layout are already woven into the system docs. We only want
		   "user" paclets here, meaning ones that have no official entry point already in the standard docs.
		   It is already verified that paclet:linkbase will resolve to a document for every one of these
		   LinkBase strings.
		*)
		pacletLinks = getPacletManager[]@getUserPacletLinks[$Language, needsPacletRebuild];
		{pacletLinks, appLinks}
	]
	

(* Private helper function. Reads BrowserCategories.m file and pulls out app name and IndexTag from
   first Item, which is the link we use as entry point. Can return various results (e.g., Null), but only
   string pairs {"appName", "indexTag"} should be kept.
*)
getAppNamesAndTags[appDir_String] :=
	Module[{browserCatsFile, cats, appName},
		browserCatsFile = ToFileName[{appDir, "Documentation", $Language}, "BrowserCategories.m"];
		(* If we couldn't find, say, Japanese, default to English. *)
		If[FileType[browserCatsFile] =!= File,
			browserCatsFile = ToFileName[{appDir, "Documentation", "English"}, "BrowserCategories.m"]
		];
		If[FileType[browserCatsFile] =!= File,
			Return[Null]
		];
		(* In case a package inadvertantly has symbols instead of strings for some values in
		   its BrowserCategories.m file, read it in a private context. This is a workaround
		   for bug 74474 (PCT had this problem).
		*)
		AbortProtect[
			Begin["PacletManager`Documentation`Private`"];
			cats = Get[browserCatsFile];
			End[]
		];
		appName = First[cats];
		{appName, appName}
	]
	
(* Called from buttons on Installed AddOns doc page to open legacy Help Browser. *)
helpBrowserLookup[appName_String] :=
	AbortProtect[ (* AbortProtect in case rebuilding hb index takes longer than FE's TimeConstrained that wraps this call. *)
		Module[{appsInBrowser, pos},
			If[$needsHBRebuild, FrontEndTokenExecute["RebuildHelpIndex"]; $needsHBRebuild = False];
			appsInBrowser = FrontEndExecute[FrontEnd`HelpBrowserGetListBoxList[1]];
			(* This brings Help Browser window to front. *)
			FrontEndExecute@FrontEnd`HelpBrowserLookup[FrontEnd`ButtonNotebook[]];
			(* Names end in ">" char, so drop if present. *)
			appsInBrowser = If[StringMatchQ[#, "*>"], StringDrop[#, -1], #]& /@ appsInBrowser;
			pos = Flatten[Position[appsInBrowser, appName]];
			If[Length[pos] > 0,
				FrontEndExecute[FrontEnd`HelpBrowserSetListBoxItem[1, First[pos]]],
			(* else *)
				(* Cause nothing to be selected in list boxes and beep. *)
				FrontEndExecute[FrontEnd`HelpBrowserSetListBoxItem[0, 0]];
				Beep[]
			]
		]
	]
	
Internal`SetValueNoTrack[$lastAppLinks, True];
Internal`SetValueNoTrack[$lastPacletDirs, True];
Internal`SetValueNoTrack[$needsHBRebuild, True];


End[]
