(* :Title: Manager.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 6.0 *)
		     
(* :Copyright: Mathematica source code (c) 1999-2007, Wolfram Research, Inc. All rights reserved. *)

(* :Discussion: 
	
   This file is a component of the PacletManager Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   PacletManager uses a special system wherein one package context (PacletManager`) has its
   implementation split among a number of .m files. Each component file has its own private
   context, and also potentially introduces public symbols (in the PacletManager` context) and
   so-called "package" symbols, where the term "package" comes from Java terminology,
   referring to symbols that are visible everywhere within the implementation of PacletManager,
   but not to clients.
*)

PacletManager::usage = "PacletManager is used only as a generic symbol for some messages."

RestartPacletManager::usage = "RestartPacletManager[] restarts the Paclet Manager."

PacletFind::usage = "PacletFind is an internal symbol."
PacletFindAll::usage = "PacletFindAll is an internal symbol."

PacletInstall::usage = "PacletInstall is an internal symbol."
PacletInstallQueued::usage = "PacletInstallQueued is an internal symbol."
PacletUninstall::usage = "PacletUninstall is an internal symbol."

PacletUpdate::usage = "PacletUpdate is an internal symbol."
PacletCheckUpdate::usage = "PacletCheckUpdate is an internal symbol."

PacletInformation::usage = "PacletInformation is an internal symbol."

PacletResource::usage = "PacletResource is an internal symbol."


PacletDirectoryAdd::usage = "PacletDirectoryAdd is an internal symbol."
PacletDirectoryRemove::usage = "PacletDirectoryRemove is an internal symbol."

Paclet::usage = "Paclet is an internal symbol."

PacletEnable::usage = "PacletEnable is an internal symbol."
PacletDisable::usage = "PacletDisable is an internal symbol."

UpdatePaclets::usage = "UpdatePaclets is an internal symbol."

RebuildPacletData::usage = "RebuildPacletData[] rebuilds cached paclet information by rescanning paclet directories."

(* Temporary. *)
PacletManagerEnabled::usage = "PacletManagerEnabled is an internal symbol."



Begin["`Package`"]

preparePacletManager
startPacletManager

makePaclet
makePacletInformation

(* Called from outside this package (e.g., WebServices) until we settle on PM API. *)
getPacletManager

$isRemoteKernel

(* Called by SystemInformation dialog (i.e., called from outside the PacletManager). *)
lastUpdatedPacletSite
lastUsedPacletSite
numPacletsDownloaded
$sharedPacletsBaseDir
$userPacletsBaseDir
$userPersistentDataDir


End[]  (* `Package` *)



(* Current context will be PacletManager`. *)

Begin["`Manager`Private`"]


(***********************  PacletManager  ************************)

(* Called in Startup/Documentation.m to decide whether to use old defs for HelpLookupPacletURI.
   Users can redefine this to stop using PM for doc resolution.
   If/when that code goes away, or there is no choice but to use the PM, get rid of this.
*)
PacletManagerEnabled[] = True


If[!valueQ[$pacletManagerPrepared], $pacletManagerPrepared = False]

(* Define the dirs the paclet manager uses for data and paclets. *)
If[!ValueQ[$sharedPacletsBaseDir], $sharedPacletsBaseDir = ToFileName[{$BaseDirectory, "Paclets"}, "Repository"]]
If[!ValueQ[$userPacletsBaseDir], $userPacletsBaseDir = ToFileName[{$UserBaseDirectory, "Paclets"}, "Repository"]]
If[!ValueQ[$userPersistentDataDir], $userPersistentDataDir = ToFileName[{$UserBaseDirectory, "Paclets"}, "Configuration"]]
If[!ValueQ[$userTemporaryDataDir], $userTemporaryDataDir = ToFileName[{$UserBaseDirectory, "Paclets"}, "Temporary"]]


PacletManager::fail = "The PacletManager could not start. Certain operations will not function correctly in this session unless the PacletManager can run. It will attempt to restart when needed, unless you set PacletManagerEnabled[] = False."


(* As an optimization, preparePacletManager can be called before InstallJava[] has executed.
   It does everything that can be done before Java is connected, including calling InstallJava[]
   (which is asynchronous now).
*)

preparePacletManager[] := preparePacletManager[$sharedPacletsBaseDir, $userPacletsBaseDir,
												$userPersistentDataDir, $userTemporaryDataDir]

preparePacletManager[sharedPacletsBaseDir_String, userPacletsBaseDir_String,
							userPersistentDataDir_String, userTemporaryDataDir_String] :=
	Module[{dataToPass, oldfiles, freshStart = False},
		If[!TrueQ[$pacletManagerPrepared],
			$sharedPacletsBaseDir = sharedPacletsBaseDir;
			$userPacletsBaseDir = userPacletsBaseDir;
			$userPersistentDataDir = userPersistentDataDir;
			$userTemporaryDataDir = userTemporaryDataDir;
			
			(* Temporary: Delete legacy paclet files and dirs if they exist. *)
			Quiet[
				oldfiles = FileNames["*", ToFileName[$BaseDirectory, "Paclets"]];
				oldfiles = Select[oldfiles, !StringMatchQ[#, "*Temporary"] && !StringMatchQ[#, "*Configuration"] && !StringMatchQ[#, "*Repository"]&];
				If[FileType[#] === Directory, DeleteDirectory[#, DeleteContents->True], DeleteFile[#]]& /@ oldfiles;
				oldfiles = FileNames["*", ToFileName[$UserBaseDirectory, "Paclets"]];
				oldfiles = Select[oldfiles, !StringMatchQ[#, "*Temporary"] && !StringMatchQ[#, "*Configuration"] && !StringMatchQ[#, "*Repository"]&];
				If[FileType[#] === Directory, DeleteDirectory[#, DeleteContents->True], DeleteFile[#]]& /@ oldfiles;
				If[FileType[ToFileName[$UserBaseDirectory, "PacletData"]] === Directory,
					DeleteDirectory[ToFileName[$UserBaseDirectory, "PacletData"], DeleteContents->True]
				]
			];	
			
			Quiet[
				(* Quiet permissions-related error messages. *)
				If[FileType[$sharedPacletsBaseDir] =!= Directory,
					CreateDirectory[$sharedPacletsBaseDir]
				]
			];
			If[FileType[$userPacletsBaseDir] =!= Directory,
				(* Set freshStart to force rebuild if user has blown away the Repository dir. *)
				freshStart = True;
				CreateDirectory[$userPacletsBaseDir]
			];
			If[FileType[$userPersistentDataDir] =!= Directory,
				CreateDirectory[$userPersistentDataDir]
			];
			If[FileType[$userTemporaryDataDir] =!= Directory,
				CreateDirectory[$userTemporaryDataDir]
			];
			
			initSearchCache[$userTemporaryDataDir];
			
			dataToPass = removeSpaces /@ Flatten[Replace[pmArgs[$sharedPacletsBaseDir, $userPacletsBaseDir,
								$userPersistentDataDir, $userTemporaryDataDir, freshStart], lis_List :> {Length[lis], lis}, 1]];
			RegisterJavaInitialization["run com.wolfram.paclet.client.PacletManager " <> (StringJoin @@ Riffle[dataToPass, " "])];
			$pacletManagerPrepared = True;
		];
		(* Put InstallJava outside the test. The parts inside the test are steps that need
		   to be taken only once in a kernel session. But we need InstallJava called every time we
		   enter this function (startPacletManager relies on preparePacletManager to call InstallJava). 
		*)
		InstallJava[];
		freshStart
	]



startPacletManager[] :=
	If[!JavaObjectQ[$pacletManager],
		AbortProtect[
			preemptProtect[
				Module[{freshStart},
					freshStart = preparePacletManager[];
					(* Use MathLink`NotebookFrontEndLinkQ to test for FE because we might be called very early
					   in kernel startup sequence, long before FE has sent its init code to the kernel (e.g.,
					   setting up $FrontEnd and such).
					*)
					(* NO, CANT DO THIS HERE. MATHLINK~CALLFRONTEND WON'T WORK AT INIT TIME.
					$isRemoteKernel = MathLink`NotebookFrontEndLinkQ[$ParentLink] &&
										$MachineID =!= MathLink`CallFrontEnd[FrontEnd`Value[$NotebookMachineID]];
					*)
					Quiet[
						Check[
							LoadJavaClass["com.wolfram.paclet.client.PacletManager", AllowShortContext->False]
							,
							Message[PacletManager::fail];
							Return[]
							,
							{LoadJavaClass::fail}
						],
						{LoadJavaClass::fail}
					];
					$pacletManager = com`wolfram`paclet`client`PacletManager`getPreparedInstance[];
					If[!JavaObjectQ[$pacletManager] || $pacletManager === Null,
						(* Only way that getPreparedInstance[] is Null is if we called InstallJava on a link,
						   in which case RegisterJavaInitialization code has not run.
						*)
						$pacletManager = JavaNew["com.wolfram.paclet.client.PacletManager", 
											Sequence @@ pmArgs[$sharedPacletsBaseDir, $userPacletsBaseDir,
															$userPersistentDataDir, $userTemporaryDataDir, freshStart]]
					];
					KeepJavaObject[$pacletManager, Manual]
				]
			]
		];
	]
	

RestartPacletManager[] :=
	(
		ReinstallJava[];
		startPacletManager[]
	)

getPacletManager[] :=
	(startPacletManager[]; $pacletManager)

	
	
pmArgs[sharedPacletsBaseDir_, userPacletsBaseDir_, userPersistentDataDir_, userTemporaryDataDir_, freshStart_] :=
	Module[{kernelVersion},
		kernelVersion =
			StringJoin[
				If[StringMatchQ[#, "*."], # <> "0", #]& [ToString[$VersionNumber]],
				".",
				ToString[$ReleaseNumber]
			];
		{kernelVersion, Round[AbsoluteTime[]], $LicenseID, $MachineID,
			SystemInformation["Kernel", "ReleaseID"], $SystemID,
				If[hasFrontEnd[], CurrentValue["Language"], $Language], $InstallationDirectory,
					sharedPacletsBaseDir, userPacletsBaseDir, userPersistentDataDir, userTemporaryDataDir,
						systemPacletDirs[], applicationDirs[], $extraPacletDirectories, ToString[freshStart],
							{{$PacletSite, $pacletSiteName}}}
	]
	

(* Dirs in the layout that can hold WRI paclets. Users will not be modifying the contents of these
   dirs, or if they do (such as to manually install a new J/Link version), they will have to call
   RebuildPacletData[] afterward.
*)
systemPacletDirs[] :=
	{
		ToFileName[{$InstallationDirectory, "Documentation"}],
		ToFileName[{$InstallationDirectory, "SystemFiles", "Links"}], 
		ToFileName[{$InstallationDirectory, "SystemFiles", "Autoload"}], 
		ToFileName[{$InstallationDirectory, "AddOns", "Packages"}]
	}

(* This is the set of dirs into which we want to support users hand-installing legacy-style apps
   that are at least partially pacletized (that is, they have a PacletInfo.m file). The PM will never
   install into these dirs, but we want to allow users to do this, especially with legacy apps that
   have PacletInfo.m files to support their new-style documentation. These paclets are rebuilt from
   their PacletInfo.m files every time, never serialized. This is because with the users doing manual
   installation/removal, we could never trust the serialized data anyway.
*)
applicationDirs[] =
	{
		ToFileName[{$InstallationDirectory, "AddOns", "Applications"}], 
		ToFileName[{$UserBaseDirectory, "Applications"}], 
		ToFileName[{$UserBaseDirectory, "AutoLoad"}], 
		ToFileName[{$BaseDirectory, "Applications"}], 
		ToFileName[{$BaseDirectory, "Autoload"}]
	}


removeSpaces[e_] := removeSpaces[ToString[e, FormatType->InputForm]]
removeSpaces[s_String] := StringReplace[s, " " -> "%20"]


(***********************  Paclet Expression Representation  ************************)

(* One issue with this expression representation is that I don't store enough info in M to
   correctly retrieve the Java object that corresponds to this expression. If I restricted
   to local paclets only, and I sent the name, version, and location to Java I could correctly
   do this, but that's a lot to send. Storing the JavaObject ref in the expression would 
   work great, but that is too inefficient for some uses (I don't want to create thousands
   of object refs in Mathematica). A possibility is to store the object's hashcode, and then
   do a quick lookup in Java, checking that the name and version match (not likely that hashcode
   would be the same for any two paclets, let alone ones with same name and version but different
   locations. I would have to write my own hashCode() method, of course. Because Paclet objects
   are immutable, I could cache and serialize the hashcode with each object.
*)

General::pclt = "`1` does not refer to a known paclet."
General::pcltn = "No appropriate paclet with name `1` was found."
General::pcltnv = "No appropriate paclet with name `1` and version `2` was found."
General::pcltni = "No appropriate paclet with name `1` is installed."
General::pcltnvi = "No appropriate paclet with name `1` and version `2` is installed."

Paclet::selector = "Unknown selector `1` for Paclet expression."

Attributes[Paclet] = {ReadProtected}

Format[Paclet[name_, version_, rest__], OutputForm] := "Paclet[" <> name <> ", " <> version <> ", <>]"
Format[Paclet[name_, version_, rest__], TextForm] := "Paclet[" <> name <> ", " <> version <> ", <>]"
Paclet /: MakeBoxes[Paclet[name_, version_, rest__], fmt_] :=
	With[{literalName = "\"\<" <> name <> "\>\""},
		InterpretationBox[RowBox[{"Paclet", "[", literalName, ",", "<>", "]"}], Paclet[name, version, rest]]
	]


(* Function called from Java to construct an expression representation of a Paclet object. *)
makePaclet[name_String, version_String, qualifiedName_String, location_String] :=
	Paclet[name, version, qualifiedName, location]

Paclet[name_, version_, qualifiedName_, location_, ___]["Name"]         := name
Paclet[name_, version_, qualifiedName_, location_, ___]["Version"]      := version
Paclet[name_, version_, qualifiedName_, location_, ___]["QualifiedName"] := qualifiedName
Paclet[name_, version_, qualifiedName_, location_, ___]["Location"]     := location
(* For other selectors, would have to send info back to Java to lookup paclet object and return info. *)
Paclet[name_, version_, qualifiedName_, location_, ___][s_]             := (Message[Paclet::selector, s]; Null)


(****************************  Looking up local paclets  *****************************)

(* PacletFind is a quick lookup of a locally-installed paclet given name and version.
   It only returns one result. Use "" for version to mean "find latest version". This
   is not a fancy search that looks for all available paclets, including server ones.
   It will mainly be used as a way to get a Paclet expression from a name and possibly
   also a version.

   Perhaps a PacletFindAll or something like that for a full search, including multiple
   results and server paclets included.
*)

LOCAL = 1    (* These constants must match ones in Java. *)
REMOTE = 2
LOCALANDREMOTE = 3

PacletFind[pacletName_String] := PacletFind[{pacletName, ""}]

PacletFind[{pacletName_String, pacletVersion_String}] := 
	Module[{exprs, info},
		exprs = getPacletManager[]@findPacletExpressions[pacletName, pacletVersion, LOCAL, False];
		If[exprs === {},
			Null,
		(* else *)
			First[exprs]
		]
	]


(* PacletFindAll returns multiple results, not just the latest version, and can also search
   for remote paclets. The list that is returnedis sorted, with latest version number first.
*)

PacletFindAll::loc = "The second argument is a location specification that must be either \"Local\", \"Remote\", or {\"Local\", \"Remote\"}."

(* TODO: Probably an option for IgnoreSystemRequirements. *)
Options[PacletFindAll] = {}

PacletFindAll[pacletName_String, location:(_String | {__String}), opts:OptionsPattern[]] := 
	PacletFindAll[{pacletName, ""}, location, opts]

PacletFindAll[{pacletName_String, pacletVersion_String}, location:(_String | {__String}), OptionsPattern[]] := 
	Module[{loc},
		loc = Sort[Flatten[{location}]];
		If[!MatchQ[loc, {"Local"} | {"Remote"} | {"Local", "Remote"}],
			Message[PacletFindAll::loc, location];
			Return[{}]
		];
		getPacletManager[]@findPacletExpressions[pacletName, pacletVersion,
							If[MemberQ[loc, "Local"], LOCAL, 0] + If[MemberQ[loc, "Remote"], REMOTE, 0], False]
	]
	
	
(********************************  Installing  ********************************)

PacletInstall::offline = "Mathematica cannot install paclet `1` because it is currently configured not to use the Internet for paclet downloads. To allow Internet use, set $AllowInternet = True."
PacletInstall::notfound = "No paclet named `1` is available for download from any currently enabled paclet sites."
PacletInstall::vnotfound = "No paclet named `1` with version number `2` is available for download from any currently enabled paclet sites."
PacletInstall::dwnld = "An error occurred downloading paclet `1` from site `2`: `3`. `4`"
PacletInstall::inst = "An error occurred installing paclet from file `1`: `2`. `3`"


(* Arg is either a file path to a .paclet file or a paclet name with no version spec. *)
PacletInstall[str_String] :=
	If[StringMatchQ[str, "*.paclet", IgnoreCase->True],
		installPacletFromFile[str],
	(* else *)
		PacletInstall[{str, ""}]
	]

PacletInstall[p_Paclet] := PacletInstall[{p["Name"], p["Version"]}]

PacletInstall[{pacletName_String, pacletVersion_String}] :=
	JavaBlock[
		Module[{q},
			q = PacletInstallQueued[{pacletName, pacletVersion}];
			(* q is either a Paclet expr (the paclet was already installed) or an
			   installer object.
			*)
			If[Head[q] === Paclet || q === $Failed,
				q,
			(* else *)
				(* q ia a FileDownloader object. *)
				PacletInstall[q]
			]
		]
	]

PacletInstall[installer_?JavaObjectQ] /; InstanceOf[installer, "com.wolfram.paclet.client.net.PacletInstaller"] :=
	Module[{installedPaclet, exc, msgLines},
		CheckAbort[
			(* Call ServiceJava to support Java dialogs (e.g., network password)
			   running in a standalone kernel.
			*)
			While[!installer@isFinished[],
				ServiceJava[];
				Pause[0.1]
			],
			(* Although we are already going to back out of waiting, propagate
			   the abort to Java so that the download is cancelled.
			*)
			installer@cancel[];
			Abort[]
		];
		JavaBlock[
			installedPaclet = installer@getPaclet[];
			If[installedPaclet =!= Null,
				installedPaclet@toExpressionForm[],
			(* else *)
				exc = installer@getException[];
				msgLines = userMessageFromException[exc] ~Append~ "";  (* Append to make sure it has at least 2 elements. *)
				If[installer@downloadOK[],
					Message[PacletInstall::dwnld, installer@getPacletFilename[], installer@getSite[], Sequence @@ msgLines],
				(* else *)
					Message[PacletInstall::inst, installer@getPacletFilename[], Sequence @@ msgLines]
				];
				$Failed
			]
		]
	]
		

PacletInstallQueued[pacletName_String] := PacletInstallQueued[{pacletName, ""}]

PacletInstallQueued[p_Paclet] := PacletInstallQueued[{p["Name"], p["Version"]}]

PacletInstallQueued[{pacletName_String, pacletVersion_String}] :=
	JavaBlock[
		Module[{result, pacletExpr, installer},
			pacletExpr = PacletFind[{pacletName, pacletVersion}];
			Which[
				Head[pacletExpr] === Paclet,
					result = pacletExpr,
				$AllowInternet === False,
					(* Deliberately call this message on PacletInstall, as that is the
					   function users are most likely to be directly calling when this is entered.
					*)
					Message[PacletInstall::offline, pacletName];
					result = $Failed,
				True,
					installer = getPacletManager[]@createPacletInstaller[pacletName, pacletVersion];
					If[JavaObjectQ[installer] && installer =!= Null,
						installer@start[];
						result = installer,
					(* else *)
						(* Either paclet not found on a server or server not enabled. *)
						If[pacletVersion == "",
							Message[PacletInstall::notfound, pacletName],
						(* else *)
							Message[PacletInstall::vnotfound, pacletName, pacletVersion]
						];
						result = $Failed
					]
			];
			result
		]
	]


installPacletFromFile[pacletFile_String] :=
	JavaBlock[
		Module[{pacletObj, result},
			If[FileType[pacletFile] === File,
				pacletObj = getPacletManager[]@installPaclet[pacletFile];
				result = pacletObj@toExpressionForm[];
				If[Head[result] =!= Paclet,
					(* Coerce all failures to the symbol $Failed. *)
					result = $Failed
				];
				result,
			(* else *)
				(* File doesn't exist. *)
				Message[PacletInstall::notfound, pacletFile];
				$Failed
			]
		]
	]
	
	
(*****************************  PacletUpdate  *******************************)

(* Returns a Paclet expression if a successful update occurred, Null if no update was available,
   or $Failed if there was a failure (this is just the return value of PacletInstall).
   More precisely:
       no local paclet with this name exists: result from PacletInstall
       local paclet exists, but no updates are available: Null
       local paclet exists and updates are available: result from PacletInstall
*)

(*
PacletUpdate[name_String] :=
	Module[{result, bestLocal, remote, bestRemote},
		result = Null;
		bestLocal = PacletFind[name];
		remote = PacletFindAll[name, "Remote"];
		If[Length[remote] > 0,
	    	bestRemote = First[remote];
	    	If[Head[bestLocal] =!= Paclet || PacletNewerQ[bestRemote, bestLocal],
		        result = PacletInstall[bestRemote]
			]
		];
		result
	]
*)


PacletUpdate[p_Paclet] := PacletUpdate[p["Name"]]

PacletUpdate[name_String] :=
	Module[{avail, bestLocal},
		avail = PacletCheckUpdate[name];
		If[Length[avail] > 0,
			(* TODO: fall back to second, third choice if there is a server problem. *)
			PacletInstall[First[avail]],
		(* else *)
			Null
		]
	]



PacletCheckUpdate[p_Paclet] := PacletCheckUpdate[p["Name"]]

PacletCheckUpdate[name_String] :=
	Module[{bestLocal, remote},
		bestLocal = PacletFind[name];
		remote = PacletFindAll[name, "Remote"];
		If[Head[bestLocal] === Paclet,
	    	Select[remote, PacletNewerQ[#, bestLocal]&],
	    (* else *)
	    	remote
		]
	]


(*  UNUSED. NOT PUBLIC.

	TODO: What is this needed for? Should it have an option to make it search
	only installed paclets, not server ones? Or should that be another function?
	Or is that the only useful function? Should this func return Fasle if $AllowInternet is False?
	Or if all servers are known to be unavailable?	
*)
PacletAvailable[pacletName_String] := PacletAvailable[{pacletName, ""}]
	
PacletAvailable[{pacletName_String, pacletVersion_String}] :=
	Module[{pacletExpr, downloader},
		(* Test local availability *)
		pacletExpr = PacletFind[{pacletName, pacletVersion}];
		If[Head[pacletExpr] === Paclet,
			True,
		(* else *)
			JavaBlock[
				downloader = getPacletManager[]@getPacletDownloader[pacletName, pacletVersion];
				(* TODO: chedk return value of getPacletDownloader. *)
				downloader@isAvailable[]
			]
		]
	]
	


PacletUninstall::fail = "Could not uninstall paclet named `1` at location `2`."
PacletUninstall::notfound = "`1` does not refer to a valid paclet in the current session."


PacletUninstall[paclet_Paclet] :=
	JavaBlock[
		Module[{pacletObj},
			pacletObj = getPacletManager[]@lookupPaclet[paclet["QualifiedName"], paclet["Location"]];
			If[JavaObjectQ[pacletObj] && pacletObj =!= Null,
				PacletUninstall[pacletObj],
			(* else *)
				Message[PacletUninstall::notfound, paclet];
				{paclet}
			]
		]
	]

PacletUninstall[pacletObj_?JavaObjectQ] := 
	JavaBlock[
		Module[{pm},
			If[pacletObj === Null,
				(* Because JavaObjectQ lets in Null, which isn't unlikely as an argument
				   (e.g., PacletUninstall[FindPaclet["DoesntExist"]]) handle it specially.
				*)
				Message[PacletUninstall::notfound, Null],
			(* else *)
				pm = getPacletManager[];
				If[TrueQ[pm@uninstallPaclet[pacletObj]],
					{},
				(* else *)
					Message[PacletUninstall::fail, pacletObj@getName[], pacletObj@getLocation[]];
					{pacletObj@toExpressionForm[]}
				]
			]
		]
	]

PacletUninstall[pacletName_String] := PacletUninstall[{pacletName, ""}]

PacletUninstall[{pacletName_String, pacletVersion_String}] :=
	JavaBlock[
		Module[{pm, pacletObjs},
			pm = getPacletManager[];
			pacletObjs = pm@findPaclets[pacletName, pacletVersion, LOCAL, False];
			If[Length[pacletObjs] == 0,
				If[pacletVersion == "",
					Message[PacletUninstall::pcltni, pacletName],
				(* else *)
					Message[PacletUninstall::pcltnvi, pacletName, pacletVersion]
				];
				$Failed,
			(* else *)
				Flatten[PacletUninstall /@ pacletObjs]
				(************
				results = pm@uninstallPaclet[#]& /@ pacletObjs;
				unsuccessfullyUninstalled = Cases[Thread[{results, pacletObjs}], {False, obj_} :> obj@toExpressionForm[]];
				If[Length[unsuccessfullyUninstalled] > 0,
					Message[PacletUninstall::fail, pacletName, #["Location"]]& /@ unsuccessfullyUninstalled
				];
				unsuccessfullyUninstalled
				***********)
			]
		]
	]



(***************************  PacletInformation  ****************************)

(* 
   Gives a list of rules describing attributes of a paclet. Modelled after FileInformation,
   except lhs of rules in result are strings.
   
   When given a name and version, it only looks in local paclets and only returns a result
   for one paclet (this will be the latest version). Paclets that do not match M version and 
   SystemID requirements will not be found. You can use this function to get info for server
   paclets or non-matching paclets if you use another function like PacletFind to acquire
   a Paclet[] expression and then pass that expression to PacletInformation.
   
   No version specified, or "", means to find the latest version.
   
   Args:    One of:
              - lone string that is either a paclet name or a path to a .paclet file.
              - {name, version} list.
              - Paclet[] expression
   
   Returns: A list of rules, the LHS of which are all strings. List is empty if paclet
            is not found (no messages are issued, in parallel to how FileInformation works)
            or arg was a .paclet file and there was a failure in creating the Paclet
            object from that file (in which case an exception message should have been issued).
*)

(* Arg is either a file path to a .paclet file or a paclet name with no version spec. *)
PacletInformation[str_String] :=
	Module[{pacletObj},
		If[StringMatchQ[str, "*.paclet", IgnoreCase->True],
			JavaBlock[
				LoadJavaClass["com.wolfram.paclet.PacletFactory", AllowShortContext->False];
				LoadJavaClass["com.wolfram.paclet.Utils", AllowShortContext->False];
				pacletObj = First[com`wolfram`paclet`PacletFactory`createPaclets[str]];
				If[JavaObjectQ[pacletObj],
					com`wolfram`paclet`Utils`makePacletInformation[pacletObj],
				(* else *)
					(* Will have gotten some exception message out of createPaclets(). *)
					{}
				]
			],
		(* else *)
			PacletInformation[{str, ""}]
		]
	]


PacletInformation[{name_String, version_String}] :=
	Module[{exprs, info},
		exprs = getPacletManager[]@findPacletExpressions[name, version, LOCAL, False];
		If[exprs === {},
			{},
		(* else *)
			PacletInformation[First[exprs]]
		]
	]

PacletInformation[paclet_Paclet] :=
	Module[{info},
		info = getPacletManager[]@getPacletInformation[paclet["QualifiedName"], paclet["Location"]];
		If[info === Null,
			{},
		(* else *)
			info
		]
	]
	

(* Called from Java to construct the result of a PacletInformation query. *)
makePacletInformation[name_String, version_String, qualifiedName_String, location_String,
						desc_String, mVersion_String, sysIDs:({___String} | Null), isInstalled_] :=
	{"Name" -> name, "Version" -> version, "QualifiedName" -> qualifiedName,
		"Description" -> desc, "MathematicaVersion" -> mVersion,
			"SystemIDs" -> (sysIDs /. Null -> All), "Location" -> location, "Installed" -> isInstalled}
	
	
(********************************  PacletResource  **********************************)

(* Returns either:
    - $Failed (after a message) if paclet not installed.
    - Null if paclet does not contain the resource
    - String that is the full path to the resource
*)

PacletResource[pacletName_String, resource_String] := PacletResource[{pacletName, ""}, resource]

PacletResource[p_Paclet, resource_String] := PacletResource[{p["Name"], p["Version"]}, resource]

PacletResource[{pacletName_String, pacletVersion_String}, resource_String] :=
	JavaBlock[
		Module[{p, pm, pacletObj},
			pm = getPacletManager[];
			p = PacletFind[{pacletName, pacletVersion}];
			If[p === Null,
				If[pacletVersion == "",
					Message[PacletResource::pcltni, pacletName],
				(* else *)
					Message[PacletResource::pcltnvi, pacletName, pacletVersion]
				];
				$Failed,
			(* else *)
				pacletObj = pm@lookupPaclet[p["QualifiedName"], p["Location"]];
				(* Quietly return Null if resource not found in this paclet. *)
				pm@getResource[pacletObj, resource]
			]
		]
	]



(******************************  RebuildPacletData  ********************************)

RebuildPacletData::err = "`1`"

RebuildPacletData[] := RebuildPacletData[True]

RebuildPacletData[fullRebuild:(True | False)] :=
	JavaBlock[
		Module[{errList},
			clearMessageLinkCache[];
			getPacletManager[]@restorePaclets[fullRebuild];
			errList = getPacletManager[]@getRestoreErrors[];
			Message[RebuildPacletData::err, #]& /@ errList;
		]
	]


(***********************  PacletManager Persistent Data  *************************)

(* These are Package`-level functions but called outside of this package (by the
   SystemInformation dialog).
*)

(* Returns information about the most-recently-updated PacletSite.
   This is a `Package-level function called by the SystemInformation dialog.
   Returns either:
   
      - A list consisting of two elements: The site URL and the date of last successful
        update in M Date[] format. Only considers wolfram.com sites.
        ex:  {"http://pacletserver.wolfram.com", {2006, 10, 2, 0, 49, 16.9325775}}
        
      - Null, if no wolfram.com sites exist, or none have yet been updated.

*)
lastUpdatedPacletSite[] :=
	JavaBlock[
		Module[{siteURL, date},
			siteURL = getPacletManager[]@getLastUpdatedSite[];
			date = JavaObjectToExpression[getPacletManager[]@getLastUpdatedSiteDate[]];
			If[StringQ[siteURL] && Head[date] === List,
				{siteURL, date},
			(* else *)
				(* No wolfram sites exist, or none have ever been updated. *)
				Null
			]
		]
	]
	

(* Returns information about the most-recently-used PacletSite, meaning the one from which
   a paclet was last downloaded.
   This is a `Package-level function called by the SystemInformation dialog.
   Returns either:
   
      - A list consisting of two elements: The site URL and the date of last successful
        use in M Date[] format. Only considers wolfram.com sites.
        ex:  {"http://pacletserver.wolfram.com", {2006, 10, 2, 0, 49, 16.9325775}}
        
      - Null, if no wolfram.com sites exist, or none have yet been used.

*)
lastUsedPacletSite[] :=
	JavaBlock[
		Module[{siteURL, date},
			siteURL = getPacletManager[]@getLastUsedSite[];
			date = JavaObjectToExpression[getPacletManager[]@getLastUsedSiteDate[]];
			If[StringQ[siteURL] && Head[date] === List,
				{siteURL, date},
			(* else *)
				(* No wolfram sites exist, or none have ever been updated. *)
				Null
			]
		]
	]


(* 
	Returns integer, or $Failed on any problem.
*)
numPacletsDownloaded[] := 
	Module[{num},
		num = getPacletManager[]@getNumPacletsDownloaded[];
		If[IntegerQ[num],
			num,
		(* else *)
			$Failed
		]
	]
	
	

(*********************  Paclet Directory Funcs  **********************)

(* When dirs are added, they are canonicalized by a call into Java. Whenever
   we test to see if a dir is among PacletDirectories[], we canonicalize it
   as well. This allows us to use ==, MemberQ, etc. to test if a dir is a member.
   Canonicalization handles expansion of . and .., conversions to correct case
   on Windows, etc.
*)

PacletDirectoryAdd[dirs___String] := PacletDirectoryAdd[{dirs}]

PacletDirectoryAdd[dirs:{___String}] :=
	Module[{pm, canonicalPath},
		pm = getPacletManager[];
		Function[{dir},
			canonicalPath = pm@addPacletDirectory[dir, Directory[]];
			If[!MemberQ[$extraPacletDirectories, canonicalPath],
				AppendTo[$extraPacletDirectories, canonicalPath]
			]
		] /@ dirs;
		$extraPacletDirectories
	]


PacletDirectoryRemove[dirs__String] := PacletDirectoryRemove[{dirs}]

PacletDirectoryRemove[dirs:{___String}] :=
	Module[{pm, canonicalPath},
		pm = getPacletManager[];
		Function[{dir},
			canonicalPath = pm@removePacletDirectory[dir, Directory[]];
			$extraPacletDirectories = DeleteCases[$extraPacletDirectories, canonicalPath]
		] /@ dirs;
		$extraPacletDirectories
	]


If[!ValueQ[$extraPacletDirectories], $extraPacletDirectories = {}]
	

End[]

