(* :Title: Services.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 6.0 *)
		     
(* :Copyright: Mathematica source code (c) 1999-2007, Wolfram Research, Inc. All rights reserved. *)

(* :Discussion: 
	
   This file is a component of the PacletManager Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   PacletManager uses a special system wherein one package context (PacletManager`) has its
   implementation split among a number of .m files. Each component file has its own private
   context, and also potentially introduces public symbols (in the PacletManager` context) and
   so-called "package" symbols, where the term "package" comes from Java terminology,
   referring to symbols that are visible everywhere within the implementation of PacletManager,
   but not to clients.
*)


$PacletSite::usage = "$PacletSite gives the location of the main Wolfram Research paclet server."

PacletSite::usage = "PacletSite is an internal symbol."

PacletSites::usage = "PacletSites[] gives the list of currently-enabled paclet sites."

PacletSiteAdd::usage = "PacletSiteAdd is an internal symbol."

PacletSiteRename::usage = "PacletSiteRename is an internal symbol."

PacletSiteUpdate::usage = "PacletSiteUpdate is an internal symbol."

PacletSiteRemove::usage = "PacletSiteRemove is an internal symbol."



$AllowInternet::usage = "$AllowInternet specifies whether Mathematica should attempt to use the Internet for certain operations. Set it to False to prevent Internet use."
$AllowDocumentationUpdates::usage = "$AllowDocumentationUpdates specifies whether Mathematica should automatically update documentation notebooks from the Internet as newer ones become available. Set it to False to prevent Mathematica from attempting to download newer notebooks as you browse the documentation."
$AllowDataUpdates::usage = "$AllowDataUpdates specifies whether Mathematica should automatically update data collections (like CountryData, ElementData, etc.) as newer data become available. Set it to False to prevent Mathematica from attempting to download updates."

$InternetProxyRules::usage = "$InternetProxyRules is a list of rules that control how Mathematica accesses the Internet. Unless the first rule is UseProxy->True, the other rules, which specify protocol-specific proxies, are not used. Instead of modifying $InternetProxyRules directly, use the UseInternetProxy and SetInternetProxy functions, or the Help > Internet Connectivity dialog in the front end."
PacletManager`SetInternetProxy::usage = "SetInternetProxy[\"protocol\" {\"proxyHost\", port}] causes Mathematica to use the specified proxy host and port when accessing URLs of the specified protocol (\"http\", for example). You can also use the Help > Internet Connectivity dialog in the front end to configure proxy settings."
UseInternetProxy::usage = "UseInternetProxy controls whether Mathematica should use a proxy server when accessing the Internet. UseInternetProxy[Automatic] causes Mathematica to attempt to use the proxy settings from your system or browser. UseInternetProxy[False] causes Mathematica to connect directly to the Internet, bypassing a proxy server. UseInternetProxy[True] causes Mathematica to use the proxy settings specified in $InternetProxyRules. You can also use the Help > Internet Connectivity dialog in the front end to configure proxy settings."
SetInternetProxyCredentials::usage = "SetInternetProxyCredentials[\"username\", \"password\"] sets the credentials to use if Mathematica needs to access the Internet via an authenticating proxy server. Alternatively, you can wait to be prompted for this information by Mathematica when it is needed."


Begin["`Package`"]

$pacletSiteName
makePacletSite
getPacletSiteInfo
errorDialog

passwordDialog

testConnectivity

(* A flag for users to prevent PM from putting up dialogs. *)
$allowDialogs


End[]  (* `Package` *)



(* Current context will be PacletManager`. *)

Begin["`Services`Private`"]


(***********************  Kernel-Wide Settings  **********************)

Unprotect[$AllowInternet, $AllowDocumentationUpdates, $AllowDataUpdates, $InternetProxyRules]

$AllowInternet::boolset = "Cannot set value of $AllowInternet to `1`; value must be True or False."
$AllowDocumentationUpdates::boolset = "Cannot set value of $AllowDocumentationUpdates to `1`; value must be True or False."
$AllowDataUpdates::boolset = "Cannot set value of $AllowDataUpdates to `1`; value must be True or False."


$AllowInternet /: Set[$AllowInternet, allow_] := 
	(
		If[allow === True || allow === False,
			getPacletManager[]@setAllowInternet[allow],
		(* else *)
			Message[$AllowInternet::boolset, allow]
		];
		allow
	)

$AllowInternet := TrueQ[getPacletManager[]@isAllowInternet[]]


$AllowDocumentationUpdates /: Set[$AllowDocumentationUpdates, allow_] := 
	(
		If[allow === True || allow === False,
			getPacletManager[]@setDocAutoUpdate[allow],
		(* else *)
			Message[$AllowDocumentationUpdates::boolset, allow]
		];
		allow
	)

$AllowDocumentationUpdates := TrueQ[getPacletManager[]@isDocAutoUpdate[]]

$AllowDataUpdates /: Set[$AllowDataUpdates, allow_] := 
	(
		If[allow === True || allow === False,
			getPacletManager[]@setDataAutoUpdate[allow],
		(* else *)
			Message[$AllowDataUpdates::boolset, allow]
		];
		allow
	)

$AllowDataUpdates := TrueQ[getPacletManager[]@isDataAutoUpdate[]]


$InternetProxyRules := 
	Module[{strs, useProxy, httpProxyHost, httpProxyPort, httpsProxyHost, httpsProxyPort,
			ftpProxyHost, ftpProxyPort, socksProxyHost, socksProxyPort},
		strs = getPacletManager[]@getProxyInfo[];
		If[Length[strs] === 9,
			useProxy = ToExpression[strs[[1]]];
			httpProxyHost = strs[[2]];
			httpProxyPort = ToExpression[strs[[3]]];
			httpsProxyHost = strs[[4]];
			httpsProxyPort = ToExpression[strs[[5]]];
			ftpProxyHost = strs[[6]];
			ftpProxyPort = ToExpression[strs[[7]]];
			socksProxyHost = strs[[8]];
			socksProxyPort = ToExpression[strs[[9]]];
			{"UseProxy"->useProxy,
				"HTTP"->If[httpProxyHost =!= Null && httpProxyPort =!= 0, {httpProxyHost, httpProxyPort}, {}],
				"HTTPS"->If[httpsProxyHost =!= Null && httpsProxyPort =!= 0, {httpsProxyHost, httpsProxyPort}, {}],
				"FTP"->If[ftpProxyHost =!= Null && ftpProxyPort =!= 0, {ftpProxyHost, ftpProxyPort}, {}],
				"Socks"->If[socksProxyHost =!= Null && socksProxyPort =!= 0, {socksProxyHost, socksProxyPort}, {}]
			},
		(* else *)
			proxyDefaults
		]
	]


(* These are only used if PacletManager cannot run for some reason. Needless to say,
   if the PM cannot run, then these values are not used in a meaningful way, but at
   least they can display.
*)
proxyDefaults = {"UseProxy"->Automatic, "HTTP"->{}, "FTP"->{}, "HTTPS"->{}, "Socks"->{}}


Protect[$AllowInternet, $AllowDocumentationUpdates, $AllowDataUpdates, $InternetProxyRules]


(***********************  SetInternetProxy  **********************)

UseInternetProxy::val = "The argument to UseInternetProxy must be either True, False, or Automatic."

UseInternetProxy[val:(True | False | Automatic)] := getPacletManager[]@setProxy[ToString[val], Null, Null, 0]

UseInternetProxy[_] := Null /; Message[UseInternetProxy::val]
UseInternetProxy[args___] := Null /; (ArgumentCountQ[UseInternetProxy, Length[{args}], 1, 1]; False)


(* J/Link's version of this function calls the one here in its implementation.
   Because JLink` might come before PacletManager` on $ContextPath, we also
   overload J/Link's version with defs that call us.
*)

PacletManager`SetInternetProxy::args = "Improper arguments to SetInternetProxy."
PacletManager`SetInternetProxy::prot = "Invalid protocol name `1`; must be one of \"HTTP\", \"HTTPS\", \"FTP\", or \"Socks\"."

JLink`SetInternetProxy[type_String, {}] := PacletManager`SetInternetProxy[type, {}]
PacletManager`SetInternetProxy[type_String, {}] :=
	getPacletManager[]@setProxy[Null, type, Null, 0] /; checkProtocol[type]

JLink`SetInternetProxy[type_String, {host_String, port:(_String | _Integer)}] :=
	PacletManager`SetInternetProxy[type, {host, port}]

PacletManager`SetInternetProxy[type_String, {host_String, port:(_String | _Integer)}] :=
	getPacletManager[]@setProxy[Null, type, host, ToExpression[port]] /; checkProtocol[type]

PacletManager`SetInternetProxy[args___] := Null /; Message[PacletManager`SetInternetProxy::args]
JLink`SetInternetProxy[args___] :=         Null /; Message[PacletManager`SetInternetProxy::args]


SetInternetProxyCredentials[username_String, password_String] := getPacletManager[]@setProxyCredentials[username, password]


checkProtocol[type_String] :=
	(If[Not[#], Message[PacletManager`SetInternetProxy::prot, type]]; #)& [
		StringMatchQ[type, "HTTP", IgnoreCase->True] || 
		StringMatchQ[type, "HTTPS", IgnoreCase->True] || 
		StringMatchQ[type, "FTP", IgnoreCase->True] || 
		StringMatchQ[type, "Socks", IgnoreCase->True]
	]


(***********************  Simple Server Interaction  **********************)

 (* $PacletSite = "http://127.0.0.1:8080/PacletServer" *)
$PacletSite = "http://pacletserver.wolfram.com"
$pacletSiteName = "Wolfram Research Paclet Server"


PacletSiteAdd::badurl = "URL `1` is not a properly formed http or file URL."
General::nosite = "Site `1` is not an existing Paclet site."
General::offline = "Mathematica is currently configured not to use the Internet. To allow Internet use, set $AllowInternet = True."


PacletSites[] :=
	JavaBlock[
		Module[{siteObjArray},
			siteObjArray = getPacletManager[]@getPacletSites[];
			#@toExpressionForm[]& /@ siteObjArray
		]
	]

(* Returns $Failed iff is obviously a bogus URL (e.g., htp:/foo). *)
PacletSiteAdd[siteURL_String] := PacletSiteAdd[siteURL, ""]

PacletSiteAdd[siteURL_String, name_String] :=
	JavaBlock[
		Module[{siteObj},
			siteObj = getPacletManager[]@addPacletSite[siteURL, name];
			If[JavaObjectQ[siteObj] && siteObj =!= Null,
				siteObj@toExpressionForm[],
			(* else *)
				Message[PacletSiteAdd::badurl, siteURL];
				$Failed
			]
		]
	]
	

PacletSiteRemove[site_PacletSite] := PacletSiteRemove[site["URL"]]

PacletSiteRemove[siteURL_String] :=
	JavaBlock[
		Module[{siteObj},
			siteObj = getPacletManager[]@removePacletSite[siteURL];
			If[JavaObjectQ[siteObj] && siteObj =!= Null,
				siteObj@toExpressionForm[],
			(* else *)
				Message[PacletSiteRemove::nosite, siteURL];
				$Failed
			]
		]
	]


PacletSiteRename[site_PacletSite, name_String] := PacletSiteRename[site["URL"], name]

PacletSiteRename[siteURL_String, name_String] :=
	JavaBlock[
		Module[{siteObj},
			siteObj = getPacletManager[]@renamePacletSite[siteURL, name];
			If[JavaObjectQ[siteObj] && siteObj =!= Null,
				siteObj@toExpressionForm[],
			(* else *)
				Message[PacletSiteRename::nosite, siteURL];
				$Failed
			]
		]
	]
	

PacletSiteUpdate::exc = "An error occurred attempting to update paclet information from site `1`. `2`"

Options[PacletSiteUpdate] = {"Asynchronous"->False, Interactive->False}

PacletSiteUpdate[site_PacletSite, opts:OptionsPattern[]] := PacletSiteUpdate[site["URL"], opts]

PacletSiteUpdate[siteURL_String, opts:OptionsPattern[]] :=
	JavaBlock[
		Module[{pm = getPacletManager[], site, siteObj, excptn, cause, result, interactive, async, msgLines},
			async = TrueQ[OptionValue["Asynchronous"]];
			interactive = TrueQ[OptionValue[Interactive]] && TrueQ[$allowDialogs];
			If[StringMatchQ[siteURL, "http:*", IgnoreCase->True] && !TrueQ[$AllowInternet],
				If[interactive,
					(* This message is actually defined for General, so need to use General to get the text as a string.
	                   The StringForm is to convert the special sequence `.` into a single `.
	                *)
					errorDialog[ToString[StringForm[General::offline]]],
				(* else *)
					Message[PacletSiteUpdate::offline, siteURL]
				];
				Return[$Failed]
			];
			(* This AbortProtect is only so that this func can be called from a FE button that will wrap the call in
			   TimeConstrained with a limit less than a possible network timeout. If that happens, the timeout
			   occurs normally but the rest of the function aborts as soon as it gets back from Java. Thus, user
			   sees no dialog boxes or message result informing of the problem. The AbortProtect doesn't
			   prevent a user from aborting, because the user cannot abort anyway while in Java. If I change
			   the Java side to respect aborts while waiting in network I/O then I will have to revisit this.
			*)
			AbortProtect[
				siteObj = pm@updatePacletSite[siteURL, async];
				If[JavaObjectQ[siteObj] && siteObj =!= Null,
					result = siteObj@toExpressionForm[];
					(* If not asynchronous, take the time to examine the site afterwards to see if
					   it succeeded, and issue a message if not.
					*)
					If[!async,
						excptn = siteObj@getLastException[];
						If[excptn === Null,
							If[interactive, errorDialog["Update from paclet site " <> siteURL <> " succeeded."]],
						(* else *)
							If[interactive,
								errorDialog[excptn],
							(* else *)
								msgLines = userMessageFromException[excptn];
								Message[PacletSiteUpdate::exc, siteURL, StringJoin @@ Riffle[msgLines, " "]]
							];
							result = $Failed;
						]
					],
				(* else *)
					(* Get here only if site was not already present. *)
					If[interactive,
						errorDialog["Update failed because site " <> siteURL <> " is not an existing paclet site."],
					(* else *)
						Message[PacletSiteUpdate::nosite, siteURL]
					];
					result = $Failed
				]
			];
			result
		]
	]


testConnectivity[] :=
	JavaBlock[
		Module[{exc, result, msgLines},
			result = False;
			If[TrueQ[$AllowInternet],
				exc = getPacletManager[]@testConnectivity[$PacletSite <> "/PacletSite.m"];
				If[exc === Null,
					result = True;
					errorDialog["Connectivity test succeeded using pacletserver.wolfram.com."],
				(* else *)
					(* Pass False for the "typeset" arg here because we don't want the Internet Connectivity
					   button added to the dialog (users invoked this function from that dialog).
					*)
					msgLines = {"Connectivity test failed using pacletserver.wolfram.com.",
					            "\n", "The server or network might be unavailable, or ",
								"you might need to configure proxy settings.",
					            Row[{"Click ", Button["here", Documentation`HelpLookup["paclet:tutorial/InternetConnectivity", Null], 
									BaseStyle->"Link", Appearance->Frameless, Evaluator->Automatic], " for more information."}]};
					errorDialog[msgLines]
				],
			(* else *)
				errorDialog[{"Connectivity test failed!", ToString[StringForm[General::offline]]}]
			];
			result
		]
	]

	
(***********  PacletSite Expression Representation  ************)

Attributes[PacletSite] = {ReadProtected}

(* Function called from Java to construct an expression representation of a PacletSite object.
   (It's too expensive to have to deal with Java objects for this; for example, we would need to
   load the PacletSite class at startup, which needs to be very fast.)
*)
makePacletSite[url_String, name_String, isEnabled:(True | False)] :=
	PacletSite[url, name, "Enabled"->isEnabled]

(* TODO: Might scrap these accessor methods and just extract data by position.
   If we don't give users a special hidden format for the PacletSite expression,
   and we don't document these accessors, then users will extract information by
   position, and we won't be able to change the represenation as a result.
*)
PacletSite[url_, name_, "Enabled"->enabled_]["URL"] := url
PacletSite[url_, name_, "Enabled"->enabled_]["Name"] := name
PacletSite[url_, name_, "Enabled"->enabled_]["Enabled"] := enabled


(*******************  PacletSite Dialogs  *******************)

pacletSiteDialogFromException[site_?JavaObjectQ, exc_?JavaObjectQ] :=
	Module[{msgText1, msgText2, msgText3, blankLine, radio = 2, result},
		msgText1 = "There was an error accessing the paclet site " <> site@getBaseURL[] <> ":";
		(* Many PacletSiteExceptions wrap other exceptions; unwrap to get the true cause. *)
		If[exc@getCause[] =!= Null,
			msgText2 = exc@getCause[]@toString[],
		(* else *)
			msgText2 = exc@toString[]
		];
		msgText3 = "What would you like to do with this site?";
		blankLine = {"", SpanFromLeft, SpanFromLeft};
		result = DialogInput[
	   				Grid[{{msgText1, SpanFromLeft, SpanFromLeft},
						  blankLine,
						  {msgText2, SpanFromLeft, SpanFromLeft},
						  blankLine,
						  {msgText3, SpanFromLeft, SpanFromLeft},
						  blankLine,
						  {Grid[{{RadioButton[Dynamic[radio], 1], "Continue to attempt to use this site in the current session"},
							     {RadioButton[Dynamic[radio], 2], "Disable this site for the current session"},
							     {RadioButton[Dynamic[radio], 3], "Remove this site"}}, 
								     Alignment -> {Left, Baseline}
						   ], "", ""},
						  {"  ", DefaultButton["OK", DialogReturn["OK"]], CancelButton["Cancel", DialogReturn["Cancel"]]}
						 }, Alignment -> {Left, Baseline}
					], WindowTitle -> "Mathematica Paclet Manager"
				 ];
		If[result == "OK",
			Switch[radio,
				1,
					Null,  (* Do nothing. *)
				2,
					site@setEnabled[False],
				3,
					PacletSiteRemove[site@getBaseURL[]]
			]
		]
	]


(* If I resurrect this, also consider java.net.UnknownHostException. Got that when WRI servers weren't
   available at all. That error is onlythrown after RetryHandler gives up.
*)

noRouteToHostDialog[] :=
	Module[{msgText1, msgText2, msgText3, blankLine, radio = 2, result},
		msgText1 = "There was an error accessing the paclet site " <> site@getBaseURL[] <> ".";
		msgText2 = "The remote host or network might be down.";
		blankLine = {"", SpanFromLeft, SpanFromLeft};
		result = DialogInput[
	   				Grid[{{msgText1, SpanFromLeft, SpanFromLeft},
						  blankLine,
						  {msgText2, SpanFromLeft, SpanFromLeft},
						  blankLine,
						  {msgText3, SpanFromLeft, SpanFromLeft},
						  blankLine,
						  {Grid[{{RadioButton[Dynamic[radio], 1], "Continue to attempt to use this site in the current session"},
							     {RadioButton[Dynamic[radio], 2], "Disable this site for the current session"},
							     {RadioButton[Dynamic[radio], 3], "Remove this site"}}, 
								     Alignment -> {Left, Baseline}
						   ], "", ""},
						  {"  ", DefaultButton["OK", DialogReturn["OK"]], CancelButton["Cancel", DialogReturn["Cancel"]]}
						 }, Alignment -> {Left, Baseline}
					], WindowTitle -> "Mathematica Paclet Manager"
				 ];
		If[result == "OK",
			Switch[radio,
				1,
					Null,  (* Do nothing. *)
				2,
					site@setEnabled[False],
				3,
					PacletSiteRemove[site@getBaseURL[]]
			]
		]
	]
	
		

(******************  Error/Warning Dialogs for Asynchronous Operations  ******************)

If[!ValueQ[$allowDialogs], $allowDialogs = True]


errorDialog[exc_?JavaObjectQ] :=
	Module[{msgLines},
		msgLines = userMessageFromException[exc, True];
		errorDialog[msgLines];
	]
	
errorDialog[line1_] := errorDialog[{line1, ""}]
	
errorDialog[lines:{__}] :=
	MessageDialog[Column[lines], WindowTitle -> "Mathematica Paclet Manager"]
	

(*******************  Password Dialog for Internet Operations  ********************)

(* passwordDialog is called from Java when either a proxy server or remote host needs authentication. *)

$pwdDlgResult  (* Read from Java and assigned to in Mathematica to indicate user has dismissed dialog. *)

passwordDialog[host_String, isProxy:(True | False)] :=
	Module[{prompt1, prompt2},
		If[isProxy, 
			prompt1 = "The proxy server " <> host <> " requires a username and password.";
			prompt2 = "These values will be stored in memory and used for the remainder of this session.",
		(* else *)
			prompt1 = "Mathematica is attempting to retrieve a URL from the host " <> host <> 
						", and this host requires a username and password.";
			prompt2 = "These values will be stored in memory and used for further requests to this host in this session."
		];
		Clear[$pwdDlgResult];
		Which[
			!TrueQ[$allowDialogs],
				Null,
			hasFrontEnd[],
				(* Use FE dialog box *)
				passwordDialogFE[prompt1, prompt2],
			(*
			StringMatchQ[$SystemID, "*Windows*"] ||
				(StringMatchQ[$SystemID, "*OSX*"] && Environment["SSH_CLIENT"] === $Failed) ||
					Environment["DISPLAY"] =!= $Failed,
				(* e.g., standalone kernel. Use Java dialog. *)
				passwordDialogJava[prompt1, prompt2],
			*)
			True,
				passwordDialogStandalone[prompt1, prompt2]
		]
	]

passwordDialogFE[prompt1_, prompt2_] :=
	Module[{prompt, cells, uname = "", pwd = ""},
		prompt = prompt1 <> "\n\n" <> prompt2;
		cells = {
			TextCell[prompt, NotebookDefault, "DialogStyle", "ControlStyle"],
			ExpressionCell[Row[{TextCell["Username:  ", NotebookDefault, "DialogStyle", "ControlStyle"], InputField[Dynamic[uname], String, ContinuousAction -> True]}]],
			ExpressionCell[Row[{TextCell["Password:  ", NotebookDefault, "DialogStyle", "ControlStyle"], InputField[Dynamic[pwd, {pwdFunc}], String, ContinuousAction -> True]}]],
			ExpressionCell[Row[{DefaultButton[$pwdDlgResult = {uname, If[StringQ[pwd], FromCharacterCode[ToCharacterCode[pwd] - 60000], ""]}; DialogReturn[], ImageSize->Dynamic[CurrentValue["DefaultButtonSize"]]], Spacer[{2.5`, 42, 16}],
									CancelButton[$pwdDlgResult = $Canceled; DialogReturn[], ImageSize->Dynamic[CurrentValue["DefaultButtonSize"]]]}], TextAlignment -> Right]
		};
		CreateDialog[DialogNotebook[cells],
					WindowTitle -> "Mathematica",
					WindowSize -> {400, FitAll},
					Evaluator -> CurrentValue["Evaluator"]
		];
		$pwdDlgResult
	]


passwordDialogStandalone[prompt1_, prompt2_] :=
	(
		Print[prompt1];
		Print[prompt2];
		{InputString["username: "], InputString["password (will echo as cleartext): "]}
	)


(* pwdFunc implements the masked password field. Chars typed are replaced by chars in the 60000
   range using the Symbol font, which are all black rectangles. To decode, we just subtract 60000.
*)

SetAttributes[pwdFunc, HoldRest];

pwdFunc[val_, expr_] :=  
	expr = FromCharacterCode[If[# >= 60000, # , # + 60000] & /@ ToCharacterCode[val], "Symbol"]


End[]
