(* ::Package:: *)

(* :Title: Graph Plot *)

(* :Context: DiscreteMath`GraphPlot` *)

(* :Author: Yifan Hu *)

(* :Summary: A collection of graph theory related function,
    including functions for aesthetic plotting of graphs *)

(* :Copyright: Copyright 2004-2007, Wolfram Research, Inc. *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 5.1, 6.0 *)

(* :History:
    Version 1.0, Oct. 2004, Yifan Hu.
*)

(* :Keywords:
Graph theory.
*)

(* :Sources:

*)

(* :Warnings: *)

(* :Limitations:  *)

(* :Discussion:

*)

Message[General::newpkg, "DiscreteMath`GraphPlot`", "Graph Utilities Package"]
Quiet[
BeginPackage["DiscreteMath`GraphPlot`","DiscreteMath`Combinatorica`"];
, {General::obspkg, General::newpkg}]

Coordinate::usage = "Coordinate is a GraphPlot package directive using inside EdgeStyleFunction and VertexStyleFunction options. Coordinate[{i1,i2,...,ik}, r] gives the coordinate of a point in the polyline {i1,i2,...,ik}, at a scaled distance of r from point i1. Coordinate[{i1,i2,...,ik}] is the same as  Coordinate[{i1,i2,...,ik}, .5], and Coordinate[i] gives the coordinate of vertex i. Coordinate[ind, r, {x,..,y}] gives Coordinate[ind, r] + {x,..,y}, Coordinate[ind, {x,y}] gives Coordinate[ind] + {x,..,y}.";

VertexName::usage = "VertexName is a GraphPlot package directive using inside EdgeStyleFunction and VertexStyleFunction options. VertexName[i] gives the label of vertex i.";

DiscreteMath`GraphPlot`GraphPlot::usage = "GraphPlot[g, options] calculates a visually appealing 2D layout of the vertices of a graph g and plots the graph using this layout.";

DiscreteMath`GraphPlot`GraphPlot3D::usage = "GraphPlot3D[g, options] calculates a visually appealing 3D layout of the vertices of a graph g and plots the graph using this layout.";

(* I Not quite happy with this function yet
GraphCoordinates1D::usage = "GraphCoordinates1D[g, options] calculates a 1D layout of the vertices of a graph g and returns the coordinates of the vertices.";
*)

GraphCoordinates::usage = "GraphCoordinates[g, options] calculates a visually appealing 2D layout of the vertices of a graph g and returns the coordinates of the vertices.";

GraphCoordinates3D::usage = "GraphCoordinates3D[g, options] calculates a visually appealing 3D layout of the vertices of a graph g and returns the coordinates of the vertices.";

(*GraphCoordinateRules1D::usage = "GraphCoordinateRules1D[g, options] calculates a 1D layout of the vertices of a graph g and returns the coordinates of the vertices as a list of rules.";

GraphCoordinateRules::usage = "GraphCoordinateRules[g, options] calculates a 2D layout of the vertices of a graph g and returns the coordinates of the vertices as a rule list.";

GraphCoordinateRules3D::usage = "GraphCoordinateRules3D[g, options] calculates a 3D layout of the vertices of a graph g and returns the coordinates of the vertices as a list of rules.";
*)

DiscreteMath`GraphPlot`TreePlot::usage = "TreePlot[g, options] generates a plot of graph g as a tree. TreePlot[g, root, options] generates a plot of the graph g as a tree, with vertex r as the root of the tree.";

GraphDistance::usage="GraphDistance[g, i, j] calculates the distance between vertices i and j in a graph g. If there is no path from i to j, the distance is Infinity";

MaximalBipartiteMatching::usage="MaximalBipartiteMatching[g] returns the maximal matching of the bipartite graph as represented by a matrix g. For a m \[Times] n matrix, the output is a list of index pairs, of form {{i_1, j_1}, ..., {i_k, j_k},...} where 1 <= i_k <= m and 1 <= j_k <= n";

MaximalIndependentVertexSet::usage="MaximalIndependentVertexSet[g] gives a maximal independent vertex set of an undirected graph g. If g is a directed graph, it is treated as if it is undirected.";

MaximalIndependentEdgeSet::usage="MaximalIndepndentEdgeSet[g] returns a maximal independent edge set, also known as maximal matching, of an undirected graph g.";

MinCut::usage="MinCut[g,k] gives a partition an undriected graph g into k parts. The partition groups the vertices into k groups, such that the number of edges between these groups are approximately minimized. A directed graph is treatd as if it is undirected.";

PseudoDiameter::usage="PseudoDiameter[g] gives the pseudo diameter of the undirected graph g, and the two vertices that achieve this diameter. If the graph is disconnected, then the diameter for each of the component, together with the indices of the two vertices that achieves this diameter, are returned. If g is a directed graph, it is treated as if it is undirected.";

StrongComponents::usage="StrongComponents[g] returns a list of all strongly connected components in the directed graph as represented by a matrix g.";

VertexList::usage="VertexList[g] returns a list of all vertices."

GraphToSparseArray::usage="GraphToSparseArray[g] returns the sparse array representing of the graph g. If the graph g has m vertices, GraphToSparseArray[g, n] returns the sparse array representing of the graph g, with the last n-m rows and columns padded by zeros. If n <= m, GraphToSparseArray[g, n] is the same as GraphToSparseArray[g].";

Bicomponents::usage="Bicomponents[g] returns the biconnected components of the undirected graph g."

MinimumBandwidthOrdering::usage="MinimumBandwidthOrdering[g] returns the vertex ordering r that attempts to minimize the bandwidth. MinimumBandwidthOrdering[m], for a matrix m, returns a pair of ordering {r, c} so that the bandwidth of the matrix m[[r,c]] is minimized."

PageRank::usage="PageRank[g] returns the PageRank of graph g."

LinkRank::usage="LinkRank[g] returns the LinkRank of graph g, in the form of a sparse matrix. The LinkRank of an edge u->v is defined as the PageRank of u, divided by the outer-degree of u."

PageRankRules::usage="PageRank[g] returns a list of rules giving the PageRank of each vertices of graph g."

LinkRankRules::usage="LinkRankRules[g] returns a list of rules giving the LinkRank of each edges of graph g."

CommunityStructurePartition::usage="CommunityStructurePartition[g] gives the partition of a graph g into communities. The partition groups the vertices into communities, such that there is a higher density of edges within communities than between them."

CommunityStructureAssignment::usage="CommunityStructureAssignment[g] gives the assignment of vertices of a graph g into communities. The assignment groups the vertices into communities, such that there is a higher density of edges within communities than between them."

CommunityModularity::usage="CommunityModularity[g, partition] gives community modularity of a partition. CommunityModularity[g, assignment] gives community modularity of an assignment."

AllGraphGeodesics::usage="AllGraphGeodesics[g] gives a matrix, where the (i, j)th entry is the length of a shortest path in g between vertices i and j. AllGraphGeodesics[g, Parent] returns a three-dimensional matrix, in which the (1, i, j)th entry is the length of a shortest path from i to j and the (2, i, j)th entry is the predecessor of j in a shortest path from i to j."

GraphGeodesic::usage = "GraphGeodesic[g, start, end] finds a shortest path between vertices start and end in graph g. GraphGeodesic[g, start] gives a list {d, p}, with d a vector containing the length of the shortest path from start to all vertices, and p a vector containing the predecessors in the shortest path from start to all vertices."

ClosenessCentrality::usage = "ClosenessCentrality[g] finds the closensss centrality."

ToCombinatoricaGraph::usage="Giving a graph g in any of the format acceptable by the DiscreteMath`GraphPlot`GraphPlot` package, ToCombinatoricaGraph[g] returns a DiscreteMath`Combinatoric representation of this graph. ToCombinatoricaGraph[g, n] returns a DiscreteMath`Combinatorica representation of g with at least n vertices, adding additional unconnected vertices when necessary."

(* usage messages for options *)
Aggressive::usage = "Aggressive is an option of PseudoDiameter, if option Aggressive->True is used, when the distance between a start and an end vertices u and v no longer increases, the algorithm will be carried out for one extra step by starting from each vertex w that is farthest away from u. The pseudo-diameter is the farthest distance possible from all such w. Possible values are False (the default) or True."

EdgeStyleFunction::usage = "EdgeStyleFunction is an option of GraphPlot, GraphPlot3D and TreePlot, it describes how edges are drawn. Possible values are Automatic (the default), None, or a user-supplied function. A user-supplied function should be of the form f[i_, j_, indlist_]:= ..., where i and j are the two vertex indices of an edge, and indlist is a list of point indices that represent this edge in the graphical output. The index list  indlist is of the form {i, ..., j}. Typically for straight edge,  indlist is the same as {i, j}. But for non-straight edges or loops, indlist can contain more than two indices of points."

Loops::usage = "Loops is an option of GraphPlot, GraphPlot3D and TreePlot. It specifies whether to draw loops for vertices that are linked to themselves. Possible values for Loops are Automatic (the default), None, True, False, or a positive real number. ."

Multiedges::usage = "Multiedges is an option of GraphPlot, GraphPlot3D and TreePlot. It specifies whether to draw multiedges for multiple edges between two vertices. Possible values for Multiedges are Automatic (the default), None, True, False, or a positive real number."

RecursionMethod::usage = "RecursionMethod is an option of MinimumBandwidthOrdering. This specifies whether to employ a multilevel process to find a minimal bandwidth ordering. Possible values for this option are None (the default) and Multilevel."

RefinementMethod::usage = "RefinementMethod is an option of MinimumBandwidthOrdering. This specifies the refinement method used to further improve the bandwidth following the application of one of the above methods. Possible values for this option are Automatic (the default), None, HillClimbing, and NodeCentroidHillClimbing."

RootPosition::usage = "RootPosition is an option of TreePlot. It specifies the position of the root of the tree. Possible values are Automatic, Top (the default), Bottom, Left, Right, and Center."

Rotation::usage = "Rotation is an option of GraphPlot, GraphPlot3D and TreePlot. It specifies the amount of clockwise rotation desired. Possible values are machine sized real numbers. The default value is 0."

TeleportProbability::usage = "TeleportProbability is an option of PageRank, PageRankRules, LikRank and LinkRankRules. It specified the probability that the Internet surfer may choose to visit nodes randomly, instead of following the out-links of a vertex. The default value is 0.15."

RemoveSinks::usage = "RemoveSinks is an option of PageRank, PageRankRules, LikRank and LinkRankRules. It specified whether sinks (a sink is a vertex with no outer links) are removed by link with all vertices. The default value is True."

TreeSizeFunction::usage = "TreeSizeFunction is an option of TreePlot. The value of this option is a function that specifies the height of each level of the tree plot. The default function specifies a height of one unit. The function takes the level in the tree as an argument. The first level is that between the root of the tree and its children."

VertexCoordinates::usage = "VertexStyleFunction is an option of GraphPlot, GraphPlot3D and TreePlot, it describes how vertices are drawn. Possible values for this option are Automatic (the default), None, or a user-supplied function. A user-supplied function should be of the form f[i_]:=..., where i is a vertex index.."


VertexStyleFunction::usage = "VertexStyleFunction is an option of GraphPlot, GraphPlot3D and TreePlot, it describes how vertices are drawn. Possible values for this option are Automatic (the default), None, or a user-supplied function. The user-supplied function should be of the form f[i_]:= ..., where i is a vertex index."

Weighted::usage = "Weighted is an option of MaximalIndependentEdgeSet, it specifies whether edges with higher weights are preferred during matching. Possible values are True or False (the default). Weighted is also an option of CommunityStructurePartition and CommunityModularity, it specifies whether vertices linked by edges with higher weights are preferrablly kept in the same community. Possible values are True or False (the default). Weighted is also an option of ShortestedPath, AllGraphGeodesics and ClosenessCentrality, it specifies edge weight is to be used in calculating distance."



(*================ Error messages =========== *)

LinkRankRules::sqma = LinkRank::sqma = PageRankRules::sqma = PageRank::sqma = Bicomponents::sqma = StrongComponents::sqma = GraphDistance::sqma = "`1` is not a square matrix";

MaximalIndependentVertexSet::vtxwgt = "`1` is not a list of length equal to the number of vertices.";

ClosenessCentrality::grp = AllGraphGeodesics::grp = GraphGeodesic::grp = CommunityStructureAssignment::grp = CommunityStructurePartition::grp = CommunityModularity::grp = "Argument `1` at position 1 does not represent a graph.";

ClosenessCentrality::negc = AllGraphGeodesics::negc = GraphGeodesic::negc = "Negative cycle found.";

ClosenessCentrality::negw = AllGraphGeodesics::negw = GraphGeodesic::negw = "Method->Dijkstra can not be applied to a graph containing negative edge weight.";

ClosenessCentrality::wgh1 = AllGraphGeodesics::wgh1 = GraphGeodesic::wgh1 = "Warning: some of the edge weights are not machine real numbers, it is assumed all edge weights are 1.";

ClosenessCentrality::zdis = "Close centrality can not be calculated because the distance of one vertex to others adds up to zero.";

MaximalIndependentVertexSet::rug = MinCut::rug = PseudoDiameter::rug = MaximalIndependentEdgeSet::rug = "Argument `1` at position 1 does not represent an undirected graph.";

CommunityStructureAssignment::wgt = CommunityStructurePartition::wgt = CommunityModularity::wgt = MaximalIndependentEdgeSet::wgt = "The value of option Weighted->`1` must be either True or False.";

CommunityStructureAssignment::wgtm = CommunityStructurePartition::wgtm = CommunityModularity::wgtm = "When Weighted->True is specified, the matrix, `1`, must have all nonnegative entries.";
CommunityModularity::part = "`1` is not a valid assignment or partition.";

MaximalIndependentEdgeSet::symat = "`1` is not a symmetric matrix.";

MinCut::kgtwo = "The value of the second argument, `1`, must be an integer >= 1";

GraphDistance::iind = DiscreteMath`GraphPlot`TreePlot::iind = "Argument `1` at position `3` is not a vertex index between 1 and `2`.";

GraphDistance::rind = DiscreteMath`GraphPlot`TreePlot::rind = "Argument `1` at position `2` is not a valid vertex.";

GraphCoordinates1D::gc1dm = GraphCoordinateRules1D::gc1dm = "The value of option Method -> `1` should be Automatic, \"Spectral\", \"SpectralOrdering\", or \"TwoNormApproximate\".";

LinkRankRules::tol = LinkRank::tol = PageRankRules::tol = PageRank::tol = "The value of option Tolerance -> `1` should be Automatic, or a positive machine sized number.";

LinkRankRules::tel = LinkRank::tel = PageRankRules::tel = PageRank::tel = "The value of option TeleportProbability -> `1` should be a positive machine real number less than 1.";

LinkRankRules::rms = LinkRank::rms = PageRankRules::rms = PageRank::rms = "The value of option RemoveSinks -> `1` should be True or False.";


Begin["`Private`"];

issueObsoleteFunMessage[fun_, context_] := Message[General::obspkgfn, fun, context];

 (*  option processing *)
StringName[sym_Symbol] := SymbolName[sym];
StringName[name_] := name;
SetAttributes[processOptionNames, Listable];
processOptionNames[(r : (Rule | RuleDelayed))[name_Symbol, val_]] := 
    r[SymbolName[name], val];
processOptionNames[opt___] := opt;
filterOptions[hiddenOpts_List, command_Symbol, options___?OptionQ] := 
  (filterOptions[
   First /@ processOptionNames[Flatten[{Options[command], hiddenOpts}]], 
   options]);
filterOptions[command_Symbol, options___?OptionQ] := 
  (filterOptions[{}, command, options]);
filterOptions[optnames_List, options___?OptionQ] := 
  (Select[Flatten[{options}], 
   MemberQ[optnames, StringName[First[#]]] &]);

DeleteOptions[optnames_List, opts___] := 
 Select[
   Flatten[{opts}], !MemberQ[optnames, StringName[First[#]]]&]
DeleteOptions[optnames_, opts___] := DeleteOptions[{optnames}, opts];

orderOptions[opts_]:= Module[{x}, x=Ordering[Map[StringName[#[[1]]]&, opts]]; opts[[x]]];

(* change to Loops ->automatic instead of Loops->None *)
gopt = Join[{"Loops"->Automatic,"Multiedges"->Automatic},
      DeleteOptions[{"Loops", "Multiedges"}, Options[SparseArray`GraphPlacement]]];

(* make these names a package symbols *)
gopt = gopt/.{"VertexStyleFunction"->DiscreteMath`GraphPlot`VertexStyleFunction, "EdgeStyleFunction"->DiscreteMath`GraphPlot`EdgeStyleFunction,"VertexCoordinates"->DiscreteMath`GraphPlot`VertexCoordinates, "Loops"->DiscreteMath`GraphPlot`Loops, "Multiedges"->DiscreteMath`GraphPlot`Multiedges, "Rotation"->DiscreteMath`GraphPlot`Rotation};

gopt = Flatten[{gopt, PlotRange->All, AspectRatio->Automatic, DeleteOptions[{"Method", "PlotRange", "AspectRatio"}, Options[Graphics]]}];

defaultOptions[DiscreteMath`GraphPlot`GraphPlot] = Options[DiscreteMath`GraphPlot`GraphPlot] = orderOptions[gopt];

(* change to Loops ->automatic instead of Loops->None *)
gopt3 = Join[{"Loops"->Automatic,"Multiedges"->Automatic},
      DeleteOptions[{"Loops", "Multiedges"}, Options[SparseArray`GraphPlacement3D]]];

(* make these names a package symbols *)
gopt3 = gopt3/.{"VertexStyleFunction"->DiscreteMath`GraphPlot`VertexStyleFunction, "EdgeStyleFunction"->DiscreteMath`GraphPlot`EdgeStyleFunction,"VertexCoordinates"->DiscreteMath`GraphPlot`VertexCoordinates, "Loops"->DiscreteMath`GraphPlot`Loops, "Multiedges"->DiscreteMath`GraphPlot`Multiedges};

gopt3 = Flatten[{gopt3, PlotRange->All, AspectRatio->Automatic, DeleteOptions[{"PlotRange", "AspectRatio", "Method"}, Options[Graphics3D]]}];

defaultOptions[DiscreteMath`GraphPlot`GraphPlot3D] = orderOptions[gopt3];
Options[DiscreteMath`GraphPlot`GraphPlot3D] = defaultOptions[DiscreteMath`GraphPlot`GraphPlot3D];

topt = Flatten[{DeleteOptions[{"Method","RandomSeed"}, Options[DiscreteMath`GraphPlot`GraphPlot]], {DiscreteMath`GraphPlot`RootPosition->Top, DiscreteMath`GraphPlot`TreeSizeFunction->(1&)}}];

defaultOptions[DiscreteMath`GraphPlot`TreePlot] = Options[DiscreteMath`GraphPlot`TreePlot] =  orderOptions[topt];

defaultOptions[GraphCoordinates1D] = Options[GraphCoordinates1D] = {Method->Automatic};
defaultOptions[GraphCoordinates] = Options[GraphCoordinates] = Options[DiscreteMath`GraphPlot`GraphPlot];
defaultOptions[GraphCoordinates3D] = Options[GraphCoordinates3D] = Options[DiscreteMath`GraphPlot`GraphPlot3D];
defaultOptions[GraphCoordinateRules1D] = Options[GraphCoordinateRules1D] = {Method->Automatic};
defaultOptions[GraphCoordinateRules] = Options[GraphCoordinateRules] = Options[DiscreteMath`GraphPlot`GraphPlot];
defaultOptions[GraphCoordinateRules3D] = Options[GraphCoordinateRules3D] = Options[DiscreteMath`GraphPlot`GraphPlot3D];
defaultOptions[ToCombinatoricaGraph] = Options[ToCombinatoricaGraph] = {Method->None};
defaultOptions[MaximalIndependentEdgeSet] = Options[MaximalIndependentEdgeSet] = {DiscreteMath`GraphPlot`Weighted->False};
defaultOptions[PseudoDiameter] = Options[PseudoDiameter] = {DiscreteMath`GraphPlot`Aggressive->False};
defaultOptions[MinimumBandwidthOrdering] = Options[MinimumBandwidthOrdering] = {Method->Automatic, DiscreteMath`GraphPlot`RefinementMethod->Automatic, DiscreteMath`GraphPlot`RecursionMethod->None};
defaultOptions[PageRank] = Options[PageRank] = Options[PageRankRules] = Options[LinkRankRules] = Options[LinkRank] = {Tolerance->Automatic, TeleportProbability->.15, RemoveSinks->True};
defaultOptions[CommunityModularity] = defaultOptions[CommunityStructurePartition] = defaultOptions[CommunityStructureAssignment] = Options[CommunityModularity] = Options[CommunityStructurePartition] = Options[CommunityStructureAssignment] = {DiscreteMath`GraphPlot`Weighted->False};
defaultOptions[AllGraphGeodesics] = defaultOptions[GraphGeodesic] = Options[AllGraphGeodesics] = Options[GraphGeodesic] = {Method->Automatic, DiscreteMath`GraphPlot`Weighted->True};
defaultOptions[ClosenessCentrality] = defaultOptions[ClosenessCentrality] = Options[ClosenessCentrality] = Options[ClosenessCentrality] = {Normalize->Automatic, DiscreteMath`GraphPlot`Weighted->True};

NondefaultOptions[symbol_]:= Module[
  {res={}, dfo, opt},
  (* get the nondefault options set by user
    with SetOptions[symbol] *)

  dfo = processOptionNames[defaultOptions[symbol]];
  opt = processOptionNames[Flatten[{Options[symbol]}]];
  ophash[x_]:= nothing;

  Map[(ophash[#[[1]]] = #[[2]])&,dfo];
  Map[If [ophash[#[[1]]] =!= #[[2]], 
         res = {res, #}]&, opt];
  ophash = .;
  Clear[ophash];
  Sequence@@Flatten[res]
] 

indhash = .;

GraphQ[A_]:= Module[{},
  res = (MatrixQ[A] || Head[A] ===  DiscreteMath`Combinatorica`Graph
              || RuleListQ[A]);
  res

];
GraphToSparseArray[G_?GraphQ]:= Module[
   {a, m, n},
   issueObsoleteFunMessage[GraphToSparseArray,"DiscreteMath`GraphPlot`"];
   a = G2S[G];
   {m, n} = Dimensions[a];
   n = Max[m, n];
   If [m == n && Head[a] == SparseArray, Return[a]];
   (* convert rectangular matrix to a square one *)
   SparseArray[a, {n, n}]
];

GraphToSparseArray[G_?GraphQ, Automatic]:= (GraphToSparseArray[G]);
GraphToSparseArray[G_?GraphQ, n_Integer]:= Module[
   {A},
   A = SparseArray[G2S[G]];
   If [n > Dimensions[A][[1]], SparseArray[A, {n,n}], A]
];
(e:GraphToSparseArray[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[GraphToSparseArray, argLength[e], 1, 2]},
  res/;(res != $Failed)
];


RuleListQ[x_] := (!MatrixQ[x] && ListQ[x] && Length[x]>0 &&
   Apply[And, 
    Map[(Head[#] === Rule)&, 
     x, {1}]]);

G2S[A_?MatrixQ] := A;

(* this generate the list of vertices from a rule list *)
VertexList[G_DiscreteMath`Combinatorica`Graph]:= Module[
   {l, n},
   issueObsoleteFunMessage[VertexList,"DiscreteMath`GraphPlot`"];
   l = DiscreteMath`Combinatorica`ToAdjacencyLists[G, EdgeWeight];
   n = Length[l];
   Table[i, {i, n}]
]
VertexList[G_?MatrixQ]:= Module[
   {n},
   issueObsoleteFunMessage[VertexList,"DiscreteMath`GraphPlot`"];
   n = Dimensions[G][[1]];
   Table[i, {i, n}]
]
VertexList[G_] := Module[{n, index, hash, res},
    (*rule list case*)
    issueObsoleteFunMessage[VertexList,"DiscreteMath`GraphPlot`"];
    Clear[hash];
    index[x_] := If[hash[x] =!= 1, hash[x] = 1; Sow[x], {}];
    res = Reap[Map[index, Flatten[Map[List @@ # &, G], 1]]];
    res[[2,1]]
];

(e:VertexList[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[VertexList, argLength[e], 1, 1]},
    issueObsoleteFunMessage[VertexList,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
];

setHash[G_]:= Module[{f, index, ind, nz = 1},
   (* set up hashing based on a rule list. 
      indhash is cached so other function can also use it. *)
   Clear[indhash];
   index[x_] := If[NumberQ[indhash[x]], indhash[x], indhash[x] = nz++];
   f[Rule[x_, y_]] := {index[x], index[y]};
   ind = Map[f[#] &, G]
];

G2S[G_] := 
  Module[{ind, n, a, b, A, s}, 
   ind = setHash[G];
   n = Max[ind];
   (* we want repeated entries to be summed *)
   s = "TreatRepeatedEntries"/.
       SystemOptions["SparseArrayOptions"][[1,2]];
   SetSystemOptions["SparseArrayOptions"->{"TreatRepeatedEntries"->1}];
   A = SparseArray[{ind -> Table[1, {Length[ind]}]}, {n, n}];
	SetSystemOptions["SparseArrayOptions"->{"TreatRepeatedEntries"->s}];
   A
];

RulesToSparseArrayWithLabels[G_] := 
  Module[{ind, n, hash, f, a, b, nz = 1,labels, s, A}, 
   Clear[hash];
   labels = {};
   index[x_] := If[NumberQ[hash[x]], hash[x], 
           labels = {labels, x}; hash[x] = nz++];
   f[Rule[x_, y_]] := {index[x], index[y]};
   ind = Map[f[#] &, G];
   n = Max[ind];
   (* we want repeated entries to be summed *)
   s = "TreatRepeatedEntries"/.
       SystemOptions["SparseArrayOptions"][[1,2]];
	SetSystemOptions["SparseArrayOptions"->{"TreatRepeatedEntries"->1}];
	A = SparseArray[Thread[Rule[ind,1]], {n, n}];
	SetSystemOptions["SparseArrayOptions"->{"TreatRepeatedEntries"->s}];
   {A,Flatten[labels]}
];


G2S[G_DiscreteMath`Combinatorica`Graph]:= Module[
   {l, n, ind,row, A, s},
   row[i_Integer,neighb_List]:= Map[({i,#[[1]]}->#[[2]])&, neighb];
   l = DiscreteMath`Combinatorica`ToAdjacencyLists[G,EdgeWeight];
   n = Length[l];
   (* we want repeated entries to be summed *)
   s = "TreatRepeatedEntries"/.
       SystemOptions["SparseArrayOptions"][[1,2]];
	SetSystemOptions["SparseArrayOptions"->{"TreatRepeatedEntries"->1}];

   ind = Thread[row[Table[i,{i,n}], l]];

	A = SparseArray[Flatten[ind],{n,n}];
	SetSystemOptions["SparseArrayOptions"->{"TreatRepeatedEntries"->s}];
   A

];

ToCombinatoricaGraph[g_, opts___?OptionQ]:= With[
 {res = ToCombinatoricaGraphInternal[g, Automatic, opts]},
    issueObsoleteFunMessage[ToCombinatoricaGraph,"DiscreteMath`GraphPlot`"];
 res/;(res =!= $Failed)

]

ToCombinatoricaGraph[g_, n_, opts___?OptionQ]:= With[
 {res = ToCombinatoricaGraphInternal[g, n, opts]},
    issueObsoleteFunMessage[ToCombinatoricaGraph,"DiscreteMath`GraphPlot`"];
 res/;(res =!= $Failed)

]

(e:ToCombinatoricaGraph[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[ToCombinatoricaGraph, argLength[e], 1, 2]},
  res/;(res != $Failed)
];

ToCombinatoricaGraphInternal[g_, n_, opts___?OptionQ] := Module[
  {res},
  res = Catch[ToCombinatoricaGraphInternalCore[g, n, opts]];
  res
];

NumericMatrixQ[B_?MatrixQ]:= Module[
  {res = True},
  If [SparsePatternArrayQ[B], Return[False]];
  If [Developer`PackedArrayQ[B], Return[True]];
  MatrixQ[B,NumericQ]
]

ToCombinatoricaGraphInternalCore[g_, n_, opts___?OptionQ] := 
  Module[
   {a, gr = g, m, coord, mthd, caller = ToCombinatoricaGraph, labels, ar},

	mthd = "Method"/.processOptionNames[Flatten[{opts}]]
         /.processOptionNames[Options[ToCombinatoricaGraph]];
   a = GraphToSparseArray[g, n];
   If [Head[a] =!= SparseArray, Throw[$Failed]];
   If [Head[g] =!= DiscreteMath`Combinatorica`Graph,
      gr = FromAdjacencyLists[a@AdjacencyLists[], 
         Type -> Directed];
      (* set labels *)
      labels = VertexList[g];
      m = Length[labels];
      If [n > m, labels = Join[labels, Range[m+1, n]]];
      gr = SetVertexLabels[gr, labels]
   ];

   (* add edge weight if the input os a matrix *)
	If [NumericMatrixQ[g],
      ar = Drop[ArrayRules[g], -1];
	   edges = Edges[gr];
      gr = SetEdgeWeights[gr, edges, edges/.ar]
   ];

   (* find coordinate *)
   Switch[StringName[mthd],
     "None",
      coord = None,
     _,
      coord = SparseArray`Private`GraphPlot[caller, a, Method->mthd];
      If [!ListQ[coord], Throw[$Failed]];
      coord = Take[coord[[1]],Dimensions[a][[1]]];
   ]; 

   (* set coord *)
   If [coord =!= None, gr = ChangeVertices[gr, coord]];
	
   gr
];

DiscreteMath`GraphPlot`GraphPlot[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[DiscreteMath`GraphPlot`GraphPlot, G, 2, "Plot", "Combinatorica", 
       opts, NondefaultOptions[DiscreteMath`GraphPlot`GraphPlot]]},
    issueObsoleteFunMessage[GraphPlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinates[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinates, G, 2, "Coordinates", "Combinatorica", opts]},
    issueObsoleteFunMessage[GraphCoordinates,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinateRules[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinateRules, G2S[G], 2, "Coordinates", "Combinatorica", opts], vtx = VertexList[G]},
    issueObsoleteFunMessage[GraphCoordinateRules,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`GraphPlot[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[G, 2, "Plot", opts, 
        NondefaultOptions[DiscreteMath`GraphPlot`GraphPlot]]},
    issueObsoleteFunMessage[GraphPlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinates[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[GraphCoordinates, G, 2, "Coordinates", opts]},
    issueObsoleteFunMessage[GraphCoordinates,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinateRules[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[GraphCoordinateRules, G, 2, "Coordinates", opts], vtx = VertexList[G]},
    issueObsoleteFunMessage[GraphCoordinateRules,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];


(* GraphPlot[{}] should give Graphics[{}] *)
DiscreteMath`GraphPlot`GraphPlot[{}, opts___?OptionQ] := Module[
  {},
    issueObsoleteFunMessage[GraphPlot,"DiscreteMath`GraphPlot`"];

  Graphics[{},Sequence@@filterOptions[Graphics, opts]]
];

SetAttributes[argLength, HoldFirst];
argLength[b:(f_[args___, opts:((_Rule|_RuleDelayed)...)])]:=
 Block[{},
   If [args==={}, Return[0]];
   Length[{args}]
];

IssueArgsMessage[caller_, n_, nmin_, nmax_]:= Module[
  {},
  (* do not issue message if args ok, this happens
     say for GraphDistance[{1 -> 2}, 1, 3] *)
  If [n >= nmin && n <= nmax, Return[$Failed]];
  If [nmin === nmax, 
     If [nmin == 1,
        Message[caller::argx, caller, n, nmax],
        Message[caller::argrx, caller, n, nmax]
     ],
     Message[caller::argb, caller, n, nmin, nmax]
  ];
  $Failed;
];

(e:DiscreteMath`GraphPlot`GraphPlot[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[DiscreteMath`GraphPlot`GraphPlot, argLength[e], 1, 1]},
    issueObsoleteFunMessage[GraphPlot,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
]


DiscreteMath`GraphPlot`GraphPlot[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphToSparseArray[A], 2, 
      "Plot", "Matrix", opts, NondefaultOptions[DiscreteMath`GraphPlot`GraphPlot]]},
    issueObsoleteFunMessage[GraphPlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinates[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinates, A, 2, "Coordinates", "Matrix", opts]},
    issueObsoleteFunMessage[GraphCoordinates,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinateRules[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinateRules, A, 2, "Coordinates", "Matrix", opts], vtx = VertexList[A]},
    issueObsoleteFunMessage[GraphCoordinateRules,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

GraphPlotInternalRuleList23D[G_, dim_Integer, task_String, opts___?OptionQ]:= Module[
   {},
   If [dim == 2,
     GraphPlotInternalRuleList23D[DiscreteMath`GraphPlot`GraphPlot, G, dim, task, opts],
     GraphPlotInternalRuleList23D[DiscreteMath`GraphPlot`GraphPlot3D, G, dim, task, opts]
   ]
];

GraphPlotInternalRuleList23D[caller_, G_, dim_Integer, task_String, opts___?OptionQ]:= Module[
   {A, labels, vtxfun, vlabelThreshold, n},
   {A, labels} = RulesToSparseArrayWithLabels[G];
   (* if VertexStyleFunction->Automatic/All, plot 
      the vertex labels *)
   n = Dimensions[A][[1]];
   vtxfun =  "VertexStyleFunction"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[DiscreteMath`GraphPlot`GraphPlot]];

   vlabelThreshold = "VertexLabelThreshold"/.
               processOptionNames[
                   SystemOptions["GraphPlotOptions"][[1,2]]];

    If [StringName[vtxfun] === "All",
	     vtxfun = (Text[labels[[#]], #]&)
    ];
 
    If [StringName[vtxfun] === "Automatic" && n <= vlabelThreshold,
       vtxfun = (Text[labels[[#]], #]&)
    ];
   GraphPlot23DInternalLabels[caller, A, dim, task, "RuleList", labels,
           Sequence@@Flatten[{"VertexStyleFunction"->vtxfun, opts}]]
];

(* we set the aspect ratio for tree by default to Sqrt[asp] *)
setDefaultAspectRatio[asp_, coord_]:= Module[
  {xmin,xmax,ymin,ymax,res},
  (* make sure this is 2D *)
  If [StringName[asp] === "Automatic" && Length[coord[[1]]] == 2,
     {{xmin,xmax},{ymin,ymax}} = Map[{Min[#],Max[#]}&,Transpose[coord]];
     If [xmax-xmin <= 0 || ymax-ymin <= 0,
        res = asp,
        res = Sqrt[(ymax-ymin)/(xmax-xmin)]
     ],
     res = asp
  ];
  res
];

GraphPlot23DInternal[A_?MatrixQ, dim_Integer, task_String, inputtype_String, 
             opts___?OptionQ] := Module[
   {},
   Switch[dim,
     2,
     GraphPlot23DInternal[DiscreteMath`GraphPlot`GraphPlot, A, dim, task, inputtype, opts],
     3,
     GraphPlot23DInternal[DiscreteMath`GraphPlot`GraphPlot3D, A, dim, task, inputtype, opts],
     1,
     GraphPlot23DInternal[GraphCoordinates1D, A, dim, task, inputtype, opts]
   ]
];


GraphPlot23DInternal[caller_, A_DiscreteMath`Combinatorica`Graph, dim_Integer, task_String, inputtype_String, 
             opts___?OptionQ] := Module[
   {method, opts2 = opts, coord},
   If [dim == 2,
     (* if Method->None, we use Combinatorica's coordinates for 2D plot *)
     method = "Method"/.processOptionNames[Flatten[{opts}]]
                 /.processOptionNames[Options[DiscreteMath`GraphPlot`GraphPlot]];
     coord = "VertexCoordinates"/.processOptionNames[Flatten[{opts}]]
                 /.processOptionNames[Options[DiscreteMath`GraphPlot`GraphPlot]];
     If [StringName[method] === "None" && StringName[coord] === "None",
       opts2 = Append[DeleteOptions[{"Method"}, Flatten[{opts}]], "VertexCoordinates"
                 ->Vertices[A]];
     ];
   ];
   GraphPlot23DInternal[caller, G2S[A], dim, task, inputtype, opts2]
];

GraphPlot23DInternal[caller_, A_?MatrixQ, dim_Integer, task_String, inputtype_String, opts___?OptionQ] := Module[
   {},
   GraphPlot23DInternalLabels[caller, A, dim, task, inputtype, "AutoLabels", opts]
]

GraphPlot23DInternalLabels[caller_, A_?MatrixQ, dim_Integer, task_String, inputtype_String, labels0_, 
                     opts0___?OptionQ] := Module[
   {res, range, spgplot, gphs, asp, mthd, out, multiedge, loop, opts = opts0,
    coord, labels = labels0, coords, noGCedge, noGCvtx, edges, vertices, n},

   n = Dimensions[A][[1]];

	If [dim == 1 && task == "Coordinates",
      res = Catch[
        getGraphCoordinates1D[caller, A, inputtype, opts]
      ];
      Return[res];
   ];

   If [dim == 2,
       spgplot = SparseArray`Private`GraphPlot;
       gphs = System`Graphics,
       spgplot = SparseArray`Private`GraphPlot3D;
       gphs = System`Graphics3D
   ];

   (* make loops and multiedges True if Loops->Automatic and Multiedges->Automatic,
      and task == Plot and inputtype is RuleList or Combinatorica *)
   If [task === "Plot" && (inputtype === "RuleList"), 
       loops = "Loops"/.processOptionNames[Flatten[{opts}]]
                 /.processOptionNames[Options[caller]]; 
       multiedges = "Multiedges"/.processOptionNames[Flatten[{opts}]]
                 /.processOptionNames[Options[caller]];
       If [StringName[loops] === "Automatic",
          opts = DeleteOptions["Loops", Flatten[{opts}]];
          opts = Flatten[{"Loops"->True,{opts}}];
       ];
       If [StringName[multiedges] === "Automatic",
          opts = DeleteOptions["Multiedges", Flatten[{opts}]];
          opts = Flatten[{"Multiedges"->True,{opts}}];
       ];
   ];

   (* prcess directive VertexName *)
   If [labels === "AutoLabels", labels = Range[Dimensions[A][[1]]]];
   Clear[VertexName];
   VertexName[ind_Integer,rr___]:= labels[[ind]];

   res = spgplot[caller,A,
               filterOptions[{Plot->something}, spgplot, opts]];
   
   Clear[VertexName];

   range = "PlotRange"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[caller]];
   asp = "AspectRatio"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[caller]];
   If [!ListQ[res], out = $Failed];
  	If [ListQ[res] && task === "Plot",
      mthd = "Method"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[caller]];
      If [ListQ[mthd], mthd = mthd[[1]]];
      (* we set the aspect ratio for tree by default to Sqrt[asp] *)
      If [StringName[mthd] === "LayeredDrawing" && dim == 2, 
           asp = setDefaultAspectRatio[asp, res[[1]]]
      ];

      (* process directive Coordinate *)
      res = processCoordinate[res];

	   {coords, {noGCedge, noGCvtx, edges, vertices}} = res;
      Switch[{noGCedge, noGCvtx},
       {False,False}, 
          graphicsStuff = Apply[System`GraphicsComplex, 
                             {coords, {edges, vertices}},
                             filterOptions[System`GraphicsComplex, opts]],
       {False,True},
          graphicsStuff = GraphicsGroup[
                            {Apply[System`GraphicsComplex, 
                             {coords, edges},
                             filterOptions[System`GraphicsComplex, opts]],
                            vertices}, ContentSelectable->True],
       {True,False},
          graphicsStuff = GraphicsGroup[
                            {edges,
                            Apply[System`GraphicsComplex, 
                             {coords, vertices},
                             filterOptions[System`GraphicsComplex, opts]]},
                             ContentSelectable->True],
       {True,True},
          graphicsStuff = {edges, vertices};
      ];

      out = gphs[
                Annotation[
                  graphicsStuff,
                  VertexCoordinateRules->Take[coords, n]
                ],
                PlotRange->range, 
                AspectRatio->asp,
                Sequence@@filterOptions[gphs, DeleteOptions[{"Method","PlotRange","AspectRatio", "FrameTicks"},opts]]
           ]
   ];
   If [ListQ[res] && task === "Coordinates",
       out = Take[res[[1]], Dimensions[A][[1]]];
   ]; 
   out
];

processCoordinate[res_]:= Module[
   {out, coord},

   coord = res[[1]];
   Clear[Coordinate];
   Coordinate[ind_List, rr_?(NumberQ), displacement_List]:= 
     Coordinate[ind, rr] + displacement;

   Coordinate[ind_List, displacement_List]:= 
     Coordinate[ind, .5] + displacement;

   Coordinate[ind_List] := Coordinate[ind, .5];

   Coordinate[ind_List, rr0_?(NumberQ)] :=
      Module[{dist, dist2, sta, sto, newpos, total, rr=rr0},
      If [ind === {}, Return[{}]];
      If [Length[ind] == 1, Return[coord[[ind[[1]]]]]];
      If [rr > 1, rr = 1];
      If [rr < 0, rr = 0];
      dist = Map[Norm, Drop[coord[[ind]] - coord[[RotateLeft[ind]]], -1]];
      total = Total[dist];
      If [total <= $MachineEpsilon, Return[coord[[ind[[1]]]]]];
      dist /= total;
      dist2 = FoldList[Plus, 0, dist];(* distance from start *)
      dist2[[Length[dist2]]] = 1.; (* make sure the last one is 1 *)
      sto = Flatten[Position[dist2, (_?(# >= rr &))]][[
        1]];
      If [sto == 1, Return[coord[[ind[[1]]]]]];
      sta = sto - 1;
      newpos = 
       coord[[ind[[
          sta]]]] + (rr - dist2[[sta]])/(dist2[[sto]] - dist2[[sta]])(coord[[
            ind[[sto]]]] - coord[[ind[[sta]]]]);
      newpos
    ];

    Coordinate[ind_Integer] :=
     Module[{}, 
       coord[[ind]]
    ];

    Coordinate[ind_Integer, displacement_] := 
      Coordinate[ind] + displacement;

    out = res;
    Clear[Coordinate];
    out
]





(* GraphPlot[{}] should give Graphics3D[{}] *)
DiscreteMath`GraphPlot`GraphPlot3D[{}, opts___?OptionQ] := Graphics3D[{},Sequence@@filterOptions[Graphics, opts]];

(e:DiscreteMath`GraphPlot`GraphPlot3D[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[DiscreteMath`GraphPlot`GraphPlot3D, argLength[e], 1, 1]},
    issueObsoleteFunMessage[GraphPlot3D,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
]


DiscreteMath`GraphPlot`GraphPlot3D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[G2S[G], 3, "Plot", "Combinatorica", opts, NondefaultOptions[DiscreteMath`GraphPlot`GraphPlot3D]]},
    issueObsoleteFunMessage[GraphPlot3D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinates3D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinates3D, G2S[G], 3, "Coordinates", "Combinatorica", opts]},
    issueObsoleteFunMessage[GraphCoordinates3D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinateRules3D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinateRules3D, G2S[G], 3, "Coordinates", "Combinatorica", opts], vtx = VertexList[G]},
    issueObsoleteFunMessage[GraphCoordinateRules3D,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`GraphPlot3D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[G, 3, "Plot", opts, NondefaultOptions[DiscreteMath`GraphPlot`GraphPlot3D]]},
    issueObsoleteFunMessage[GraphPlot3D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinates3D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[GraphCoordinates3D, G, 3, "Coordinates", opts]},
    issueObsoleteFunMessage[GraphCoordinates3D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinateRules3D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[GraphCoordinateRules3D, G, 3, "Coordinates", opts], vtx = VertexList[G]},
    issueObsoleteFunMessage[GraphCoordinateRules3D,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`GraphPlot3D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphToSparseArray[A], 3, "Plot", 
            "Matrix", opts, NondefaultOptions[DiscreteMath`GraphPlot`GraphPlot3D]]},
    issueObsoleteFunMessage[GraphPlot3D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinates3D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinates3D, A, 3, "Coordinates", "Matrix", opts]},
    issueObsoleteFunMessage[GraphCoordinates3D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

GraphCoordinateRules3D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinateRules3D, A, 3, "Coordinates", "Matrix", opts], vtx = VertexList[A]},
    issueObsoleteFunMessage[GraphCoordinateRules3D,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

(* 1D stuff *)
getGraphCoordinates1D[caller_, A_?MatrixQ, inputtype_String, opts___?OptionQ]:= 
 Module[{comps, x=Table[1,{Dimensions[A][[1]]}], offset = 0, max = 0, method},


   If [!SquareMatrixQ[A],
       Message[caller::sqma, A];
       Throw[$Failed]
   ];
	method = "Method"/.processOptionNames[Flatten[{opts}]]/.
                      processOptionNames[Options[caller]];
   method = 
    Switch[StringName[method],
     "Automatic","SpectralOrdering",
     "SpectralOrdering","SpectralOrdering",
     "Spectral","Spectral",
     (*"OneNormApproximate","OneNormApproximate",*)
     "TwoNormApproximate","TwoNormApproximate",
     "TwoNormApproximateOrdering","TwoNormApproximateOrdering",
     _,Message[caller::gc1dm, method]; Return[$Failed]
   ];
   comps = StrongComponents[A+Transpose[A]];
	Do[
      comp = comps[[i]];
      x[[comp]] = getGraphCoordinates1Dinternal[caller, A[[comp, comp]], 
            inputtype, method];
	   min = Min[x[[comp]]];
      x[[comp]] = x[[comp]] - min + max;
      max = Max[x[[comp]]] + 1;
   ,{i, Length[comps]}];
   x
];

getLaplacian[A_]:= Module[
   {ind, L, Aincident, n = Dimensions[A][[1]], ne = 1},
   ind = A@NonzeroPositions[];
   ind = Flatten[Map[({{#[[1]], ne}->1., {#[[2]], ne++}->-1.})&, ind]];
   Aincident = SparseArray[ind,{n, Length[ind]}];
   L = Aincident.Transpose[Aincident];
   L
];

getFiedler[A_]:= Module[
   {L, n = Dimensions[A][[1]]},

   (* incident matrix of dimension |v|x|E| *)
   L = getLaplacian[A];

   (* get the second small eigenvectors of the Laplacian *)
  	If [n <= 100,
      v = Eigenvectors[Normal[L], -2],
      v = Eigenvectors[L, 2, Method->{"Arnoldi", "Shift" -> -0.00001}]
   ];
   v[[1]]
];

getTwoNormApproximate[A_]:= Module[
   {L, rhs, i, j, v, diam},

   L = getLaplacian[A];
   (* find the diameter *)
   {diam, {i, j}} = PseudoDiameter[L][[1]];
   rhs = Table[0.,{n}];
   rhs[[i]] = 1;
   rhs[[j]] = -1;
   (* UMFPACK right now does not like singular system *)
   v = LinearSolve[L, rhs, Method->Krylov];
   (* scale the solution to that i and j are of distance diam apart.
      we can be sure that i and j can not be on the same point, 
      since L v = rhs, and v[[i]]-v[[j]] = rhs^T v, thus
      v[[i]]-v[[j]] = v^T L v, L is positive semidifinite,
      and assume connected graph, then v^T L v as long as v is not {1,...,1}
    *)
   v = v diam/(v[[i]] - v[[j]]);
   v
]

getGraphCoordinates1Dinternal[caller_, A0_?MatrixQ, inputtype_String, method_]:= Module[
   {n = Dimensions[A0][[1]], ind, ne = 1, Aincident, nE, B, v, res, c, b, 
    lb, ub, L, ind1, ind2, rhs, i, j, diam, A = A0},
   If [n == 1, Return[{1.}]];

   If [Head[A] =!= SparseArray, A = SparseArray[A]];

   ind = A@NonzeroPositions[];

   res = Switch[StringName[method],
      "SpectralOrdering",
      (* Spectral method using Fiedler vector *)
      v = getFiedler[A];
      Ordering[v],

      "Spectral",
      (* Spectral method using Fiedler vector *)
      v = getFiedler[A];
      v,
 
      "OneNormApproximate",
      (* one norm method: 
          Min Sum[Abs[x_i-x_j], {i,j}\in E]
  		    s.t. Abs[x_i-x_j] >= 1. 
	    This is approximated as 
          Min Sum[z_ij]
          s.t. x_i1 = 0; x_i2 = DIAMETER(G), {i1,i2} form the diameter of G
               -z_ij <= x_i - x_j <= z_ij
       *)
      (*
      nE = Length[ind];
      c = Join[Table[0.,{n}], Table[1.,{2*nE}]];
      ne = 1;
      ind1 = Map[{{ne, #[[1]]}->1., {ne, #[[2]]}->-1., 
                  {ne, n+ne}->1, {ne, n+(ne++)+nE}->-1}&, ind];
      ind2 = Table[{{i + nE, i + nE}->1., {i + nE, i + 2*nE}->1.},{i, nE}];

      B = SparseArray[Flatten[{ind1, ind2}],{2*nE, n + 2*nE}];
      b = Join[Table[{0.,0},{nE}],Table[{1.,1},{nE}]];
      lb = Table[0, {n + 2*nE}];
      v = LinearProgramming[c, B, b, lb];
      Take[v,n]*)
      Throw[$Failed],
 
      "TwoNormApproximate",
      v=getTwoNormApproximate[A];
      v,

      "TwoNormApproximateOrdering",
      v=getTwoNormApproximate[A];
      Ordering[v],

      (* this should not happen *)
      _, Throw[$Failed]
    ];
    res

];

GraphCoordinates1D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinates1D, G2S[G], 1, 
          "Coordinates", "Combinatorica", opts]},
    issueObsoleteFunMessage[GraphCoordinates1D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

(e:GraphCoordinates1D[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[GraphCoordinates1D, argLength[e], 1, 1]},
    issueObsoleteFunMessage[GraphCoordinates1D,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
];


GraphCoordinateRules1D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinateRules1D, G2S[G], 1, 
          "Coordinates", "Combinatorica", opts], vtx = VertexList[G]},
    issueObsoleteFunMessage[GraphCoordinateRules1D,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

GraphCoordinates1D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[GraphCoordinates1D, G, 1, 
           "Coordinates", opts]},
   res/;(res =!= $Failed)
];
GraphCoordinateRules1D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[GraphCoordinateRules1D, G, 1, 
           "Coordinates", opts], vtx = VertexList[G]},
    issueObsoleteFunMessage[GraphCoordinates1D,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];

GraphCoordinates1D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinates1D, A, 1, 
            "Coordinates", "Matrix", opts]},
    issueObsoleteFunMessage[GraphCoordinates1D,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];
GraphCoordinateRules1D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[GraphCoordinateRules1D, A, 1, 
            "Coordinates", "Matrix", opts], vtx = VertexList[A]},
    issueObsoleteFunMessage[GraphCoordinateRules1D,"DiscreteMath`GraphPlot`"];
   Thread[Rule[vtx, res]]/;(res =!= $Failed)
];
(e:GraphCoordinateRules1D[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[GraphCoordinateRules1D, argLength[e], 1, 1]},
    issueObsoleteFunMessage[GraphCoordinateRules1D,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
]

(* end 1D stuff *)

StructurallySymmetricMatrixQ[A_?MatrixQ]:= Module[
  {B, ind, val},
  If [Head[A] =!= SparseArray, B = SparseArray[A], B = A];
  SparseArray`SymmetricQ[B, "Test"->"Structural"]
];

SymmetricMatrixQ[A_?MatrixQ]:= Module[
  {B=A, res = False},
  If [SquareMatrixQ[B],
     res = (B == Transpose[B]);
     If [res =!= True && res =!= False, res = False]
  ];
  res
];

SquareMatrixQ[A_?MatrixQ]:= Module[
  {},
  {m,n}=Dimensions[A];
  If [m =!= n, False, True]
];

GraphDistance[G_?RuleListQ, i_, j_]:= With[
      {res = Catch[GraphDistanceInternal[G2S[G], checkVertex[GraphDistance, G, i, 2], checkVertex[GraphDistance, G, j, 3]]]},
    issueObsoleteFunMessage[GraphDistance,"DiscreteMath`GraphPlot`"];
      res /; (res =!= $Failed)
];
GraphDistance[G_DiscreteMath`Combinatorica`Graph, i_Integer, j_Integer]:= With[
      {res = Catch[GraphDistanceInternal[G2S[G], checkVertex[GraphDistance, G, i, 2], checkVertex[GraphDistance, G, j, 3]]]},
    issueObsoleteFunMessage[GraphDistance,"DiscreteMath`GraphPlot`"];
      res /; (res =!= $Failed)
];
GraphDistance[A_?MatrixQ, i_Integer, j_Integer]:= With[
      {res = Catch[GraphDistanceInternal[A, checkVertex[GraphDistance, A, i, 2], checkVertex[GraphDistance, A, j, 3]]]},
    issueObsoleteFunMessage[GraphDistance,"DiscreteMath`GraphPlot`"];
      res /; (res =!= $Failed)
];

(e:GraphDistance[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[GraphDistance, argLength[e], 3, 3]},
    issueObsoleteFunMessage[GraphDistance,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
]

GraphDistanceInternal[A_?MatrixQ, i_Integer, j_Integer]:= Module[
   {res, n = Dimensions[A][[1]]},
   If [!SquareMatrixQ[A],
       Message[GraphDistance::sqma, A];
       Return[$Failed]
   ];
   res = SparseArray`GraphDistance[A, i, j];
   If [!IntegerQ[res], Return[$Failed]];
   If [res <= 0, Infinity, res]
];

MaximalBipartiteMatching[G_?RuleListQ]:= With[
      {res = MaximalBipartiteMatchingInternal[G2S[G]]},
      res /; (res =!= $Failed)
];

MaximalBipartiteMatching[G_DiscreteMath`Combinatorica`Graph]:= With[
      {res = MaximalBipartiteMatchingInternal[G2S[G]]},
      res /; (res =!= $Failed)
];

MaximalBipartiteMatching[A_?MatrixQ]:= With[
      {res = MaximalBipartiteMatchingInternal[A]},
      res /; (res =!= $Failed)
];

(e:MaximalBipartiteMatching[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[MaximalBipartiteMatching, argLength[e], 1, 1]},
  res/;(res != $Failed)
]

MaximalBipartiteMatchingInternal[A_?MatrixQ]:= Module[
   {res, n = Dimensions[A][[1]]},
    issueObsoleteFunMessage[MaximalBipartiteMatching,"DiscreteMath`GraphPlot`"];
   res = SparseArray`MaximalBipartiteMatching[A];
   If [!List[res], Return[$Failed]];
   res
];

getUndirectedGraph[caller_, G_?RuleListQ]:= Module[{},
   getUndirectedGraphInternal[caller, G]
];

getUndirectedGraph[caller_, G_DiscreteMath`Combinatorica`Graph]:= 
  Module[{},
   getUndirectedGraphInternal[caller, G]
];

getUndirectedGraph[caller_, A_?MatrixQ]:= 
  Module[{},
   getUndirectedGraphInternal[caller, A]
];

getUndirectedGraphInternal[caller_, G_]:= Module[{A, lst, m, n},
   (* check that the matrix from data is symmetric,
      if not, convert to symmetric by A+Transpose[A] *)
	A = G2S[G];
   {m, n} = Dimensions[A];
	If [m != n,
       Message[caller::rug, G];
       Throw[$Failed]
   ];
   If [!StructurallySymmetricMatrixQ[A],
      A = A + Transpose[A];
   ];
   A
];


checkGraph[caller_, G_?RuleListQ]:= Module[{},
   checkGraphInternal[caller, G]
];

checkGraph[caller_, G_DiscreteMath`Combinatorica`Graph]:= 
  Module[{},
   checkGraphInternal[caller, G]
];

checkGraph[caller_, A_?MatrixQ]:= 
  Module[{},
   checkGraphInternal[caller, A]
];

checkGraphInternal[caller_, G_]:= Module[{A, lst, m, n},
   (* check that the matrix from data is square
       *)

	A = G2S[G];
   {m, n} = Dimensions[A];
	If [m != n,
       Message[caller::grp, G];
       Throw[$Failed]
   ];
   A
];



checkUndirectedGraph[caller_, G_?RuleListQ]:= Module[{A},
   (* check that the matrix from data is symmetric
      and is thus undirected *)
	A = G2S[G];
   If [!StructurallySymmetricMatrixQ[A],
       Message[caller::rug, G];
       Throw[$Failed]
   ];
   A
];
checkUndirectedGraph[caller_, G_DiscreteMath`Combinatorica`Graph]:= Module[{A},
   (* check that the matrix from data is symmetric
      and is thus undirected *)
	A = G2S[G];
   If [!StructurallySymmetricMatrixQ[A],
       Message[caller::rug, G];
       Throw[$Failed]
   ];
   A
];
checkUndirectedGraph[caller_, A_?MatrixQ]:= Module[{},
   (* check that the matrix from data is symmetric
      and is thus undirected *)
   If [!StructurallySymmetricMatrixQ[A],
       Message[caller::rug, A];
       Throw[$Failed]
   ];
   A
];

MaximalIndependentVertexSet[G_?RuleListQ]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], Automatic]], vtx = VertexList[G]},
   vtx[[res]] /; (res =!= $Failed)
];
MaximalIndependentVertexSet[G_DiscreteMath`Combinatorica`Graph]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], Automatic]]},
   res /; (res =!= $Failed)
];
MaximalIndependentVertexSet[G_?RuleListQ, vtxwgt_]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], vtxwgt]], vtx = VertexList[G]},
   vtx[[res]] /; (res =!= $Failed)
];
MaximalIndependentVertexSet[G_DiscreteMath`Combinatorica`Graph, vtxwgt_]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], vtxwgt]]},
   res /; (res =!= $Failed)
];

MaximalIndependentVertexSet[A_?MatrixQ]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, A], Automatic]]},
   res /; (res =!= $Failed)
];
MaximalIndependentVertexSet[A_?MatrixQ, vtxwgt_]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, A], vtxwgt]]},
   res /; (res =!= $Failed)
];

(e:MaximalIndependentVertexSet[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[MaximalIndependentVertexSet, argLength[e], 1, 2]},
  res/;(res != $Failed)
]

MaximalIndependentVertexSetInternal[A_?MatrixQ, vtxwgt_]:= Module[
   {n=Dimensions[A][[1]], dims = Dimensions[vtxwgt]},

    issueObsoleteFunMessage[MaximalIndependentVertexSet,"DiscreteMath`GraphPlot`"];

   (* this should already be checked *)
   If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];
   If [vtxwgt =!= Automatic && (Length[dims] =!= 1 ||
         dims[[1]] =!= n),
      Message[MaximalIndependentVertexSet::vtxwgt, vtxwgt];
      Return[$Failed]
   ];
   SparseArray`MaximalIndependentVertexSet[A, vtxwgt]
];

MaximalIndependentEdgeSet[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[MaximalIndependentEdgeSetInternal[checkUndirectedGraph[MaximalIndependentEdgeSet, G], opts, NondefaultOptions[MaximalIndependentEdgeSet]]], vtx = VertexList[G]},
   Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
MaximalIndependentEdgeSet[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[MaximalIndependentEdgeSetInternal[checkUndirectedGraph[MaximalIndependentEdgeSet, G], opts, NondefaultOptions[MaximalIndependentEdgeSet]]]},
   res /; (res =!= $Failed)
];
MaximalIndependentEdgeSet[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[MaximalIndependentEdgeSetInternal[checkUndirectedGraph[MaximalIndependentEdgeSet, A], opts, NondefaultOptions[MaximalIndependentEdgeSet]]]},
   res /; (res =!= $Failed)
];

(e:MaximalIndependentEdgeSet[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[MaximalIndependentEdgeSet, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

MaximalIndependentEdgeSetInternal[A_?MatrixQ, opts___?OptionQ]:= Module[
   {weighted},
   (* this should be already checked with checkUndirectedGraph*)
    issueObsoleteFunMessage[MaximalIndependentEdgeSet,"DiscreteMath`GraphPlot`"];
	If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];

   weighted = 
        "Weighted" /. processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[MaximalIndependentEdgeSet]];

   If [StringName[weighted] =!= "True" && StringName[weighted] =!= "False",
      Message[MaximalIndependentEdgeSet::wgt, weighted];
      Return[$Failed]
   ];
   If [StringName[weighted] === "True",
  	   If [!SymmetricMatrixQ[A],
         Message[MaximalIndependentEdgeSet::symat, A];
         Return[$Failed]
      ]
   ];

   SparseArray`MaximalMatching[A, "Weighted"->weighted]
];


MinCut[G_?RuleListQ, k_Integer]:= With[
   {res = Catch[MinCutInternal[getUndirectedGraph[MinCut, G], k]], vtx = VertexList[G]},
   Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
MinCut[G_DiscreteMath`Combinatorica`Graph, k_Integer]:= With[
   {res = Catch[MinCutInternal[getUndirectedGraph[MinCut, G], k]]},
   res /; (res =!= $Failed)
];
MinCut[A_?MatrixQ, k_Integer]:= With[
   {res = Catch[MinCutInternal[getUndirectedGraph[MinCut, A], k]]},
   res /; (res =!= $Failed)
];

(e:MinCut[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[MinCut, argLength[e], 2, 2]},
  res/;(res != $Failed)
];

MinCutInternal[A_?MatrixQ, k_Integer]:= Module[
   {},
    issueObsoleteFunMessage[MinCut,"DiscreteMath`GraphPlot`"];

   (* this should already be checked by getUndirectedGraph*)
	If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];
   If [k < 1,
      Message[MinCut::kgtwo, k];
      Return[$Failed]
   ];
   If [k == 1, Return[{Table[i,{i,Dimensions[A][[1]]}]}]];
   SparseArray`MinCut[A, k]
];


CommunityStructurePartition[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[CommunityStructurePartitionInternal[CommunityStructurePartition, checkGraph[CommunityStructurePartition, G], opts, NondefaultOptions[CommunityStructurePartition]]], vtx = VertexList[G]},
   Map[vtx[[#]]&, res] /; (res =!= $Failed)
];


CommunityStructurePartition[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[CommunityStructurePartitionInternal[CommunityStructurePartition, checkGraph[CommunityStructurePartition, G], opts, NondefaultOptions[CommunityStructurePartition]]]},
   res /; (res =!= $Failed)
];

CommunityStructurePartition[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[CommunityStructurePartitionInternal[CommunityStructurePartition, checkGraph[CommunityStructurePartition, A], opts, NondefaultOptions[CommunityStructurePartition]]]},
   res /; (res =!= $Failed)
];

(e:CommunityStructurePartition[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[CommunityStructurePartition, argLength[e], 1, 1]},
  res/;(res != $Failed)
];


CommunityStructureAssignment[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[CommunityStructurePartitionInternal[CommunityStructureAssignment, checkGraph[CommunityStructureAssignment, G], opts, NondefaultOptions[CommunityStructureAssignment]]]},
   res /; (res =!= $Failed)
];


CommunityStructureAssignment[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[CommunityStructurePartitionInternal[CommunityStructureAssignment, checkGraph[CommunityStructureAssignment, G], opts, NondefaultOptions[CommunityStructureAssignment]]]},
   res /; (res =!= $Failed)
];

CommunityStructureAssignment[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[CommunityStructurePartitionInternal[CommunityStructureAssignment, checkGraph[CommunityStructureAssignment, A], opts, NondefaultOptions[CommunityStructureAssignment]]]},
   res /; (res =!= $Failed)
];

(e:CommunityStructureAssignment[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[CommunityStructureAssignment, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

NumericNonnegativeMatrixQ[A_]:= MatrixQ[A, (NumericQ[#] && # >= 0)&];

CommunityStructurePartitionInternal[caller_, A_?MatrixQ, opts___?OptionQ]:= Module[
   {weighted},
    issueObsoleteFunMessage[CommunityStructurePartition,"DiscreteMath`GraphPlot`"];
   weighted = 
        "Weighted" /. processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[CommunityStructurePartition]];

   If [StringName[weighted] =!= "True" && StringName[weighted] =!= "False",
      Message[caller::wgt, weighted];
      Return[$Failed]
   ];

   If [StringName[weighted] === "True",
      If [!NumericNonnegativeMatrixQ[A],
         Message[caller::wgtm, A];
         Return[$Failed];
      ];
   ];
   If [caller === CommunityStructurePartition,
     SparseArray`CommunityStructurePartition[CommunityStructurePartition, A, opts],
     SparseArray`CommunityStructureAssignment[CommunityStructureAssignment, A, opts]
   ]
];

GraphGeodesic[G_?RuleListQ, i_, j_:All, opts___?OptionQ]:= With[
   {vtx = VertexList[G], res = Catch[GraphGeodesicInternal[checkGraph[GraphGeodesic, G], i, j, VertexList[G], opts, NondefaultOptions[GraphGeodesic]]]},
   If [j =!= All,
     Map[vtx[[#]]&, res],
     {res[[1]], Map[vtx[[#]]&, res[[2]]]}
   ] /; (res =!= $Failed)
];

GraphGeodesic[G_DiscreteMath`Combinatorica`Graph, i_, j_:All, opts___?OptionQ]:= With[
   {res = Catch[GraphGeodesicInternal[checkGraph[GraphGeodesic, G], i, j, None, opts, NondefaultOptions[GraphGeodesic]]]},
   res /; (res =!= $Failed)
];

GraphGeodesic[A_?MatrixQ, i_, j_:All, opts___?OptionQ]:= With[
   {res = Catch[GraphGeodesicInternal[checkGraph[GraphGeodesic, A], i, j, None, opts, NondefaultOptions[GraphGeodesic]]]},
   res /; (res =!= $Failed)
];

(e:GraphGeodesic[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[GraphGeodesic, argLength[e], 2, 3]},
  res/;(res != $Failed)
];


GraphGeodesicInternal[A_?MatrixQ, i0_, j0_, vtxlst_, opts___?OptionQ]:= Module[
   {mthd, vtxrule, n = Dimensions[A][[1]], i = i0, j = j0, 
    res, dist, predecessors, success},

    issueObsoleteFunMessage[GraphGeodesic,"DiscreteMath`GraphPlot`"];
   (* rule list specified graph, convert vertex name to indices *)
   If [vtxlst =!= None,
      vtxrule = Thread[Rule[vtxlst, Range[n]]];
      i = i/.vtxrule;
      j = j/.vtxrule;
   ];

   If [i > n || i <= 0, Return[$Failed]];
   If [j =!= All && (j > n || j <= 0), Return[$Failed]];

   (* if people use Combinatorica, they can do
     Algorithm->.., which is the same as our Method->.. *)
   mthd = "Algorithm"/.processOptionNames[Flatten[{opts}]]
         /.{"Algorithm"->Automatic};

   If [j =!= All,
      {success, res} = 
         SparseArray`ShortestPath[A, i, j, opts, Method->mthd],
      {success, res} = 
         SparseArray`ShortestPath[A, i, opts, Method->mthd]
   ];

   Switch[success,
    "NegativeCycle",
       Message[GraphGeodesic::negc];
       res = $Failed,
    "NegativeWeight",
       Message[GraphGeodesic::negw];
       res = $Failed,
    "AssumeWeightOne",
      Message[GraphGeodesic::wgh1],
     _,
       nothing
   ];
  (* replace $MaxMachineNumber by Infinity *)
  If [ListQ[res], res = res/.$MaxMachineNumber->Infinity];
   res

];

AllGraphGeodesics[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[AllGraphGeodesicsInternal[checkGraph[AllGraphGeodesics, G], False, opts, NondefaultOptions[AllGraphGeodesics]]]},
   res /; (res =!= $Failed)
];

AllGraphGeodesics[G_?RuleListQ, Parent, opts___?OptionQ]:= With[
   {res = Catch[AllGraphGeodesicsInternal[checkGraph[AllGraphGeodesics, G], True, opts, NondefaultOptions[AllGraphGeodesics]]], vtx = VertexList[G]},
   {res[[1]], Map[vtx[[#]]&, res[[2]],{2}]} /; (res =!= $Failed)
];



AllGraphGeodesics[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[AllGraphGeodesicsInternal[checkGraph[AllGraphGeodesics, G], False, opts, NondefaultOptions[AllGraphGeodesics]]]},
   res /; (res =!= $Failed)
];
AllGraphGeodesics[G_DiscreteMath`Combinatorica`Graph, Parent, opts___?OptionQ]:= With[
   {res = Catch[AllGraphGeodesicsInternal[checkGraph[AllGraphGeodesics, G], True, opts, NondefaultOptions[AllGraphGeodesics]]]},
   res /; (res =!= $Failed)
];

AllGraphGeodesics[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[AllGraphGeodesicsInternal[checkGraph[AllGraphGeodesics, A], False, opts, NondefaultOptions[AllGraphGeodesics]]]},
   res /; (res =!= $Failed)
];
AllGraphGeodesics[A_?MatrixQ, Parent, opts___?OptionQ]:= With[
   {res = Catch[AllGraphGeodesicsInternal[checkGraph[AllGraphGeodesics, A], True, opts, NondefaultOptions[AllGraphGeodesics]]]},
   res /; (res =!= $Failed)
];

(e:AllGraphGeodesics[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[AllGraphGeodesics, argLength[e], 1, 2]},
  res/;(res != $Failed)
];


AllGraphGeodesicsInternal[A_?MatrixQ, parent_, opts___?OptionQ]:= Module[
   {success, res},

    issueObsoleteFunMessage[AllGraphGeodesics,"DiscreteMath`GraphPlot`"];
  res = SparseArray`AllPairsShortestPath[A, opts];
  If [!ListQ[res], Throw[$Failed]];
  {success, res} = res;

  Switch[success,
    "NegativeCycle",
       Message[AllGraphGeodesics::negc];
       res = $Failed,
    "NegativeWeight",
       Message[AllGraphGeodesics::negw];
       res = $Failed,
    "AssumeWeightOne",
      (* need just distance matrix*)
      If [!parent, res = res[[1]]]; 
      Message[AllGraphGeodesics::wgh1],
    _,
      (* need just distance matrix*)
      If [!parent, res = res[[1]]]; 
  ];

  (* replace $MaxMachineNumber by Infinity *)
  If [ListQ[res], res = res/.$MaxMachineNumber->Infinity];
  res
];



ClosenessCentrality[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[ClosenessCentralityInternal[checkGraph[ClosenessCentrality, G], opts]]},
   res /; (res =!= $Failed)
];

ClosenessCentrality[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[ClosenessCentralityInternal[checkGraph[ClosenessCentrality, G], opts]]},
   res /; (res =!= $Failed)
];

ClosenessCentrality[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[ClosenessCentralityInternal[checkGraph[ClosenessCentrality, A], opts]]},
   res /; (res =!= $Failed)
];

(e:ClosenessCentrality[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[ClosenessCentrality, argLength[e], 1, 2]},
  res/;(res != $Failed)
];


ClosenessCentralityInternal[A_?MatrixQ, opts___?OptionQ]:= Module[
   {success, res},

    issueObsoleteFunMessage[ClosenessCentrality,"DiscreteMath`GraphPlot`"];
  res = SparseArray`Centrality[A, opts];
  If [!ListQ[res], Throw[$Failed]];

  {success, res} = res;

  Switch[success,
    "NegativeCycle",
       Message[ClosenessCentrality::negc];
       res = $Failed, 
    "NegativeWeight",
       Message[ClosenessCentrality::negw];
       res = $Failed,
    "ZeroDistance",
       (*Centrality can not be calculated because the distance of one vertex to others sum to zero*)
       Message[ClosenessCentrality::zdis];
       res = $Failed,
   "AssumeWeightOne",
      (* need just distance matrix*)
      If [!parent, res = res[[1]]]; 
      Message[ClosenessCentrality::wgh1],
    _,
      (* need just distance matrix*)
      If [!parent, res = res[[1]]]; 
  ];

  (* replace $MaxMachineNumber by Infinity *)
  If [ListQ[res], res = res/.$MaxMachineNumber->Infinity];
  res
];

CommunityModularity[G_?RuleListQ, part_, opts___?OptionQ]:= With[
   {res = Catch[CommunityModularityInternal[checkGraph[CommunityModularity, G], part, VertexList[G], opts, NondefaultOptions[CommunityModularity]]], vtx = VertexList[G]},
   Map[vtx[[#]]&, res] /; (res =!= $Failed)
];


CommunityModularity[G_DiscreteMath`Combinatorica`Graph, part_, opts___?OptionQ]:= With[
   {res = Catch[CommunityModularityInternal[checkGraph[CommunityModularity, G], part, VertexList[G], opts, NondefaultOptions[CommunityModularity]]]},
   res /; (res =!= $Failed)
];

CommunityModularity[A_?MatrixQ, part_, opts___?OptionQ]:= With[
   {res = Catch[CommunityModularityInternal[checkGraph[CommunityModularity, A], part, VertexList[A], opts, NondefaultOptions[CommunityModularity]]]},
   res /; (res =!= $Failed)
];

(e:CommunityModularity[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[CommunityModularity, argLength[e], 2, 2]},
  res/;(res != $Failed)
];

(* assignment is a vector of length n of the form
 {0,1,0,0,1,1}*)
isAssignment[v_, n_] := (VectorQ[v, IntegerQ] && Length[v] == n);

(* partition is a list of vectors, after
   flattening and sorting it should equal to
   Range[n] *)
isPartition[v_, n_] := Module[{},
  If[! VectorQ[v, VectorQ] || Length[v] > n, Return[False]];
  (Union[Map[VectorQ[#, IntegerQ] &, v]] === {True}) && (Sort[
      Union[Flatten[v]]] === Range[n])
  ]




CommunityModularityInternal[A_?MatrixQ, part0_, vtxlst_,
         opts___?OptionQ]:= Module[
   {weighted, n = Dimensions[A][[1]], part = part0},
    issueObsoleteFunMessage[CommunityModularity,"DiscreteMath`GraphPlot`"];
   weighted = 
        "Weighted" /. processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[CommunityModularity]];

   If [StringName[weighted] =!= "True" && StringName[weighted] =!= "False",
      Message[CommunityModularity::wgt, weighted];
      Return[$Failed]
   ];

   If [StringName[weighted] === "True",
      If [Min[A] < 0,
         Message[CommunityModularity::wgtm, A];
         Return[$Failed];
      ];
   ];

   (* is this an assignment?*)
   If [!isAssignment[part,n],
      (* partition could be of the form {{a,b},{c,d}}, so
         we need to convert to {{1,2},{3,4}} etc *)
      part = part/.Thread[Rule[vtxlst, Range[n]]];
      If [!isPartition[part, n],
          Message[CommunityModularity::part, part0];
          Return[$Failed]
      ];
   ];
   SparseArray`CommunityModularity[A, part, opts]
];

PseudoDiameterToVertex[res_, vtxlist_]:= Module[
  (* convert results from a rule list, {{diam,{i,j}},...}
     to {{diam,{v_i, v_j}}, ...} *)
  {},
  Map[{#[[1]], vtxlist[[#[[2]]]]} &, res]
];

PseudoDiameter[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[PseudoDiameterInternal[getUndirectedGraph[PseudoDiameter, A], opts, NondefaultOptions[PseudoDiameter]]]},
   res /; (res =!= $Failed)
];

PseudoDiameter[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[PseudoDiameterInternal[getUndirectedGraph[PseudoDiameter, G], opts, NondefaultOptions[PseudoDiameter]]], vtx = VertexList[G]},
   PseudoDiameterToVertex[res, vtx] /; (res =!= $Failed)
];

PseudoDiameter[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[PseudoDiameterInternal[getUndirectedGraph[PseudoDiameter, G], opts, NondefaultOptions[PseudoDiameter]]]},
   res /; (res =!= $Failed)
];


(e:PseudoDiameter[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[PseudoDiameter, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

PseudoDiameterInternal[A_?MatrixQ, opts___?OptionQ]:= Module[
   {agg, res},

    issueObsoleteFunMessage[PseudoDiameter,"DiscreteMath`GraphPlot`"];
   (* this should already be checked by getUndirectedGraph *)
	If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];
   agg = 
        "Aggressive" /. processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[PseudoDiameter]];

   If [StringName[agg] =!= "True" && StringName[agg] =!= "False",
      Message[PseudoDiameter::agg, agg];
      Return[$Failed]
   ];
 
   res = SparseArray`PseudoDiameter[A, Aggressive->agg];

   (* converting from {{d, i, j},...} to {{d,{i,j}},...} *)
	If [!ListQ[res], Return[$Failed]];
   Map[{#[[1]], {#[[2]], #[[3]]}} &, res]
];
 


StrongComponents[G_?RuleListQ]:= With[
      {res = StrongComponentsInternal[G2S[G]], vtx = VertexList[G]},
      Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
StrongComponents[G_DiscreteMath`Combinatorica`Graph]:= With[
      {res = StrongComponentsInternal[G2S[G]]},
      res /; (res =!= $Failed)
];
StrongComponents[A_?MatrixQ]:= With[
      {res = StrongComponentsInternal[A]},
      res /; (res =!= $Failed)
];


(e:StrongComponents[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[StrongComponents, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

StrongComponentsInternal[A_?MatrixQ]:= Module[
   {res},
    issueObsoleteFunMessage[StrongComponents,"DiscreteMath`GraphPlot`"];
   If [!SquareMatrixQ[A],
       Message[StrongComponents::sqma, A];
       Return[$Failed]
   ];
   SparseArray`StronglyConnectedComponents[A]
];


Bicomponents[G_?RuleListQ]:= With[
      {res = BicomponentsInternal[G2S[G]], vtx = VertexList[G]},
      Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
Bicomponents[G_DiscreteMath`Combinatorica`Graph]:= With[
      {res = BicomponentsInternal[G2S[G]]},
      res /; (res =!= $Failed)
];
Bicomponents[A_?MatrixQ]:= With[
      {res = BicomponentsInternal[A]},
      res /; (res =!= $Failed)
];
(e:Bicomponents[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[Bicomponents, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

BicomponentsInternal[A_?MatrixQ]:= Module[
   {res},
    issueObsoleteFunMessage[Bicomponents,"DiscreteMath`GraphPlot`"];
   If [!SquareMatrixQ[A],
       Message[Bicomponents::sqma, A];
       Return[$Failed]
   ];
   SparseArray`BiconnectedComponents[A]
];



PageRankRules[G_?RuleListQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[PageRankRules, G2S[G], "PageRank", opts], vtx = VertexList[G]},
      Thread[Rule[vtx, res]] /; (res =!= $Failed)
];
PageRankRules[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
      {res = PageRankInternal[PageRankRules, G2S[G], "PageRank", opts], vtx = VertexList[G]},
      Thread[Rule[vtx, res]] /; (res =!= $Failed)
];
PageRankRules[A_?MatrixQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[PageRankRules, A, "PageRank", opts], vtx = VertexList[A]},
      Thread[Rule[vtx, res]] /; (res =!= $Failed)
];

(e:PageRankRules[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[PageRankRules, argLength[e], 1, 1]},
  res/;(res != $Failed)
];


PageRank[G_?RuleListQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[PageRank, G2S[G], "PageRank", opts, NondefaultOptions[PageRank]]},
      res /; (res =!= $Failed)
];
PageRank[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
      {res = PageRankInternal[PageRank, G2S[G], "PageRank", opts, NondefaultOptions[PageRank]]},
      res /; (res =!= $Failed)
];
PageRank[A_?MatrixQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[PageRank, A, "PageRank", opts, NondefaultOptions[PageRank]]},
      res /; (res =!= $Failed)
];

(e:PageRank[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[PageRank, argLength[e], 1, 1]},
  res/;(res != $Failed)
];



LinkRankRules[G_?RuleListQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[LinkRankRules, G2S[G], "LinkRank", opts], vtx = VertexList[G]},
      Map[vtx[[#[[1]]]]->#[[2]]&, Drop[ArrayRules[res], -1]] /; (res =!= $Failed)
];
LinkRankRules[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
      {res = PageRankInternal[LinkRankRules, G2S[G], "LinkRank", opts], vtx = VertexList[G]},
      Map[vtx[[#[[1]]]]->#[[2]]&, Drop[ArrayRules[res], -1]] /; (res =!= $Failed)
];
LinkRankRules[A_?MatrixQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[LinkRankRules, A, "LinkRank", opts], vtx = VertexList[A]},
      Map[vtx[[#[[1]]]]->#[[2]]&, Drop[ArrayRules[res], -1]] /; (res =!= $Failed)
];

(e:LinkRankRules[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[LinkRankRules, argLength[e], 1, 1]},
  res/;(res != $Failed)
];


LinkRank[G_?RuleListQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[LinkRank, G2S[G], "LinkRank", opts]},
      res /; (res =!= $Failed)
];
LinkRank[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
      {res = PageRankInternal[LinkRank, G2S[G], "LinkRank", opts]},
      res /; (res =!= $Failed)
];
LinkRank[A_?MatrixQ, opts___?OptionQ]:= With[
      {res = PageRankInternal[LinkRank, A, "LinkRank", opts]},
      res /; (res =!= $Failed)
];
(e:LinkRank[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[LinkRank, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

PositiveMachineReal[x_]:= (NumericQ[x] && Head[x] =!= Complex && x > 0);

PageRankInternal[caller_, A_?MatrixQ, task_, opts___?OptionQ]:= Module[
   {res, tol, tel, rmsink},
   If [!SquareMatrixQ[A],
       Message[caller::sqma, A];
       Return[$Failed]
   ];

   If [task === "PageRank",
    issueObsoleteFunMessage[PageRank,"DiscreteMath`GraphPlot`"],
    issueObsoleteFunMessage[LinkRank,"DiscreteMath`GraphPlot`"];
   ];

	tol = "Tolerance"/.processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[PageRank]];
   tol = N[tol];

	If [StringName[tol] =!= "Automatic" && !PositiveMachineReal[tol],
       Message[caller::tol, tol];
       Return[$Failed]
   ];

	tel = "TeleportProbability"/.processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[PageRank]];

	If [!PositiveMachineReal[tel] || tel >= 1,
       Message[caller::tel, tel];
       Return[$Failed]
   ];

	rmsink = "RemoveSinks"/.processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[PageRank]];

	If [StringName[rmsink] =!= "True" && StringName[rmsink] =!= "False",
       Message[caller::rms, rmsink];
       Return[$Failed]
   ];

	Switch[task,
     "PageRank",
      SparseArray`PageRank[A, Tolerance->tol, TeleportProbability->tel,
          RemoveSinks->rmsink],
     "LinkRank",
      SparseArray`LinkRank[A, Tolerance->tol, TeleportProbability->tel,
          RemoveSinks->rmsink],
     _,
      Throw[$Failed] (* should not happen *)
   ]
   
];



MinimumBandwidthOrdering[G_?RuleListQ, opts___?OptionQ]:= With[
      {res = MinimumBandwidthOrderingInternal["GraphInput", G2S[G],
         opts, NondefaultOptions[MinimumBandwidthOrdering]], 
         vtx = VertexList[G]},
      Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
MinimumBandwidthOrdering[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
      {res = MinimumBandwidthOrderingInternal["GraphInput", G2S[G], opts, NondefaultOptions[MinimumBandwidthOrdering]]},
      res /; (res =!= $Failed)
];
MinimumBandwidthOrdering[A_?MatrixQ, opts___?OptionQ]:= With[
      {res = MinimumBandwidthOrderingInternal["MatrixInput", A, opts, NondefaultOptions[MinimumBandwidthOrdering]]},
      res /; (res =!= $Failed)
];
(e:MinimumBandwidthOrdering[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[MinimumBandwidthOrdering, argLength[e], 1, 1]},
  res/;(res != $Failed)
];

MinimumBandwidthOrderingInternal[inputform_String, A0_?MatrixQ, opts___?OptionQ]:= Module[
   {res, refinement, mlv, A = A0},

    issueObsoleteFunMessage[MinimumBandwidth,"DiscreteMath`GraphPlot`"];

   If [inputform === "GraphInput",
      (* if the input is a graph, we work with undirected graph
         (symmetric matrix) *)
       A = A + Transpose[A];
   ];

   refinement = "RefinementMethod"/.processOptionNames[Flatten[{opts}]] /. 
            processOptionNames[Options[MinimumBandwidthOrdering]];
   mlv = "RecursionMethod"/.processOptionNames[Flatten[{opts}]] /. 
            processOptionNames[Options[MinimumBandwidthOrdering]];
   res = SparseArray`MinimumBandwidthOrdering[A, "RefinementMethod"->refinement, "RecursionMethod"->mlv,
      Sequence@@filterOptions[SparseArray`MinimumBandwidthOrdering, Flatten[{opts}]]
   ];
   If [ListQ[res], 
        Switch [inputform,
          "MatrixInput", res[[2]], 
          "GraphInput", res[[2,1]],
          _, res[[2]] (* this should not happen *)
        ],
   $Failed]
];


DiscreteMath`GraphPlot`TreePlot[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[DiscreteMath`GraphPlot`TreePlot, G2S[G], 2, "Plot", "Combinatorica", 
                TreeOptions[opts]]]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`TreePlot[G_DiscreteMath`Combinatorica`Graph, root_Integer, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[DiscreteMath`GraphPlot`TreePlot, G2S[G], 2, "Plot", "Combinatorica", 
                TreeOptions[checkVertex[DiscreteMath`GraphPlot`TreePlot, G, root, 2], opts]]]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`TreePlot[G_?RuleListQ, opts___?OptionQ] := With[
   {res = Catch[GraphPlotInternalRuleList23D[DiscreteMath`GraphPlot`TreePlot, G, 2, "Plot",
                TreeOptions[opts]]]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`TreePlot[G_?RuleListQ, root_, opts___?OptionQ] := With[
   {res = Catch[GraphPlotInternalRuleList23D[DiscreteMath`GraphPlot`TreePlot, G, 2, "Plot",
                TreeOptions[checkVertex[DiscreteMath`GraphPlot`TreePlot, G, root, 2], opts]]]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`TreePlot[A_?MatrixQ, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[DiscreteMath`GraphPlot`TreePlot, A, 2, "Plot", "Matrix", TreeOptions[opts]]]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

DiscreteMath`GraphPlot`TreePlot[A_?MatrixQ, root_Integer, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[DiscreteMath`GraphPlot`TreePlot, A, 2, "Plot", "Matrix", 
                  TreeOptions[checkVertex[DiscreteMath`GraphPlot`TreePlot, A, root, 2], opts]]]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
   res/;(res =!= $Failed)
];

(e:DiscreteMath`GraphPlot`TreePlot[___, opts___?OptionQ]):= With[
  {res = IssueArgsMessage[DiscreteMath`GraphPlot`TreePlot, argLength[e], 1, 2]},
    issueObsoleteFunMessage[TreePlot,"DiscreteMath`GraphPlot`"];
  res/;(res != $Failed)
];

checkVertex[caller_, A_?MatrixQ, vertex_Integer, argnum_Integer]:= Module[{n},
   n = Dimensions[A][[1]];
   (* vertex mush be positive integer <= n *)
   If [vertex > n || vertex <= 0, Message[caller::iind, vertex, n, argnum]; Throw[$Failed]];
   vertex
];

checkVertex[caller_, G_?RuleListQ, vertex_, argnum_Integer]:= Module[{ind, n, vertex1},
   ind = setHash[G];
   n = Max[ind];
	vertex1 = indhash[vertex];
   (* vertex must be a member of the rule list entries *)
   If [!IntegerQ[vertex1] || vertex1 <= 0 || vertex1 > n, Message[caller::rind, vertex, argnum]; 
        Throw[$Failed]];
   vertex1
];

checkVertex[caller_, A_DiscreteMath`Combinatorica`Graph, vertex_Integer, argnum_Integer]:= Module[{l, n},

   l = DiscreteMath`Combinatorica`ToAdjacencyLists[A,EdgeWeight];
   n = Length[l];

   (* vertex mush be positive integer <= n *)
   If [vertex > n || vertex <= 0, Message[caller::iind, vertex, n, argnum]; Throw[$Failed]];
   vertex
];




TreeOptions[root_Integer, opts___?OptionQ]:= Module[
   {},
   TreeOptionsInternal[root, opts]
];
TreeOptions[opts___?OptionQ]:= Module[
   {},
   TreeOptionsInternal[-1, opts]
];


TreeOptionsInternal[root0_Integer, opts0___?OptionQ]:= Module[
   {root = root0, opts, opts2, res, rp, tsfun, rotation = 0},
   
   opts = Flatten[{opts0, NondefaultOptions[DiscreteMath`GraphPlot`TreePlot]}];

	rp = "RootPosition"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[DiscreteMath`GraphPlot`TreePlot]];
   tsfun = "TreeSizeFunction"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[DiscreteMath`GraphPlot`TreePlot]];
	opts2 = DeleteOptions[{"Method"},opts];
	Switch[StringName[rp],
      "Left", rotation = 1.5 Pi,
      "Right", rotation = .5 Pi,
      "Top", rotation = 0,
      "Bottom", rotation = Pi,
      "Center", rotation = 0,
      _, Message[DiscreteMath`GraphPlot`TreePlot::rp, root]; Throw[$Failed]
   ];

   If [root < 0, root = Automatic];

   If [StringName[rp] === "Center",       
      res = Sequence@@Flatten[{Method->{"RadialDrawing", "Root"->root, "TreeSizeFunction"->tsfun}, "Rotation"->rotation, opts2}],
      res = Sequence@@Flatten[{Method->{"LayeredDrawing", "Root"->root, "TreeSizeFunction"->tsfun}, "Rotation"->rotation, opts2}]
   ];
   res
];



(*==============================================
 utilities routine that layout balls of different
 radiuses so that they do not clash with each other and
 occupies the space evenly. Used for layout 
 disconnected graphs
*)
ComponentsLayout[radiuses_List] := ComponentsLayout[2, radiuses, 0.];
ComponentsLayout[radiuses_List, gap_] := ComponentsLayout[2, radiuses, gap];
ComponentsLayout[dim_Integer, radiuses_List] := 
 ComponentsLayout[dim, radiuses, 0.];
ComponentsLayout[dim_Integer, radiuses_List, gap_] := Module[
   {diams , scale, max, band, bin, order, revord, width, areas, arealist, nz, 
    lens, unit, centers, area, i, j, min, totalSquares, upperDiams, quad, res,
     vol = 4},
   (* lay out a list of circles with radius gives in the list radius, 
   return the center of the circle,
   WHen plotting reasonable the circles at their centers, 
   the layout looks. only dim = 2 work now.
   *)
   diams = 2*radiuses;
   diams += gap/2;
   order = Reverse[Ordering[diams]];
   diams = diams[[order]];
   scale = Max[diams];
   diams /= scale;(* scale*)
   max = Max[diams]; min = Min[diams];
   band = Ceiling[Log[vol, max]];
   bin = Map[Ceiling[Log[vol, #]] &, diams];
   areas = Map[(vol^#) &, bin]/vol^Max[bin];
   totalSquares = Ceiling[Total[areas]];
   width = Ceiling[Sqrt[totalSquares]];
   area  = 0; 
   lens = {};
   nz = 0;
   Do[
    If [(area = area + areas[[i]]) <= 1,
      nz++,
      area =  areas[[i]];
     lens = {lens, nz}; nz = 1],
    {i, Length[areas]}
    ];
   lens = {lens, nz};
   lens = Flatten[lens];
   arealist = areas; 
   arealist = 
    Map[(res = Take[arealist, #]; arealist = Drop[arealist, #]; res) &, 
     lens];
   unit = (vol^band) ;
   centers = 
    Partition[
     unit Flatten[
       Table[{j, i}, {i, width - 1, 0, -1}, {j, 0, width - 1, 1}]], {2}];
   SetAttributes[{squareAdd, OctreeSetFull, OctreeSetSta}, 
    HoldAll];
   OctreeFullQ[octree_] := (octree =!= None && octree[[4]]);
   OctreeEmptyQ[octree_] := (octree === None);
   OctreeSetFull[octree_] := (octree[[4]] = True);
   OctreeInit[width_, x_] := {width, x, Table[None, {vol}], False, 1};
   OctreeSetSta[octree_, i_] := (octree[[5]] = i);
   stuffSquare[areas_, x_] := Module[
     {width2, octree, coord},
     (* stuff a list of areas in a square at origin x.
       Return a list of ceters each area will reside
      *)
     coord = Table[1, {Length[areas]}];
     width2 = 1;
     octree = OctreeInit[width2, x];
     Do[
      
      coord[[i]] = squareAdd[octree, areas[[i]]],
      {i, Length[areas]}];
     
     coord
     ];
   quad[i_] := (Switch[i, 1, {-1, 1}, 2, {1, 1}, 3, {-1, -1}, 4, {1, -1}, _, 
      Throw[$Failed]]);
   squareAdd[octree_, area_] := Module[
     {areaWidth, width2, x, i2, leaf, coord = "Full", full, leafs, sta},
     areaWidth = Sqrt[area];
     If [OctreeFullQ[octree], Return["Full"]];
     {width2, x, leafs, full, sta} = octree;

     If [areaWidth >= width2, OctreeSetFull[octree]; Return[x]];
     Do[
      leaf = leafs[[i2]];
      If [OctreeFullQ[leaf], Continue[]];
      If [OctreeEmptyQ[leaf], 
       leafs[[i2]] = OctreeInit[width2/2, x + width2/4*quad[i2]]];
      leaf = leafs[[i2]];
      coord = squareAdd[leaf, area];
      leafs[[i2]] = leaf;
      octree[[3]] = leafs;
      If [coord =!= "Full", Break[], OctreeSetSta[octree, i2 + 1]];
      
      
      
      , {i2, sta, 4}];
     coord
     ];
   upperDiams = unit*Map[Sqrt[#] &, areas];
   coord = unit * Map[stuffSquare[#, {0, 0}] &, arealist];
   Do[
    coord[[i]] = Map[(# + centers[[i]]) &, coord[[i]]];
    , {i, Length[coord]}];
   coord = Partition[Flatten[coord], {2}];
   
   (* this put an object at the top left corner of the square instead of the default
	   position which is the center
   Do[
    coord[[i]] = coord[[i]] + {-0.5, .5}(upperDiams[[i]] - diams[[i]]);
    , {i, Length[coord]}];
   *)
   
   revord = Table[1, {Length[order]}];
   revord[[order]] = Range[Length[order]];
   (* in case dim > 2, pad zeros to coordinates *)
   
   res = Map[PadRight[#, dim] &, (coord*scale)[[revord]]];
   res = Developer`ToPackedArray[N[res]];
   res
   
   ];

(* ============ end ComponentsLayout =========*)



End[];(*end private*)
EndPackage[];
