(* ::Package:: *)

(* :Copyright: Copyright 1988-2007, Wolfram Research, Inc. *)

(* :Title: Colors *)

(* :Context: Graphics`Colors` *)

(* :Author: Wolfram Research, Inc. *)

(* :Mathematica Version: 2.0 *)

(* :Package Version: 1.2 *)

(* :Keywords: CMYColor, YIQColor, HLSColor, RGBColor *)

(* :History: 
    Version 1.1,  Jeff Adams, August 1992.
       - additional colors added
    Version 1.2,  Jeff Adams, June 1993.
       - standardized naming conventions 
       - automated the building of color names and values so 
         that it is easier to add colors to the package
	   - fixed problem of warning messages when package is
	     loaded again using Get.
*)

(* :Sources: None. *)

(* :Requirements: None. *)

(* :Limitations: *)

(* :Warnings: None. *)

(* :Summary:
This package provides functions that convert a color 
expressed in one of several color systems to an RGBColor.  
It also allows you to find the appropriate RGBColor for a 
particular color name (such as Blue).
*)

(* :Discussion: 
	   
   If you would like to add colors to this package, 
   you simply need to append the element:
    {"color name", RGBColor[r_,g_,b_]} to the symbol ColorListing,
   where the first element is a String of the color name 
   and the second argument is a RGBColor primitive that 
   specifies the color.  
 
   The package code automatically creates a symbol for each 
   color in ColorListing, builds a default usage message for 
   each color, and evaluates the required Set call so that 
   the color symbol evaluates to the RGBColor primitive 
   when used.  Make sure there are no spaces in the color
   name that you use.

*)

(* :Examples:  The following example code will display a vertical 
    listing of the name and a swatch of each color in the Colors package.

   (* Use this initial value for colors if you want an alphabetical 
      listing of the colors. *)
	  
   colors = AllColors;
   
   (* Or, use these two lines to create a listing of the colors 
      sorted by their CMYKColor value. *)
   
   colors = {#, ToColor[ToExpression[#],CMYKColor]}& /@ AllColors;
   colors = First[ Transpose[Sort[ colors, OrderedQ[{#1[[2]], #2[[2]]}]&]]];
   
   
   (* The rest of this code can be used by either value of 
      colors above to display the color names and swatches. *)
   
   colors = Table[{ToExpression[colors[[i]]],
   Rectangle[{0,-i*10},{15,-i*10-10}],
   RGBColor[0,0,0],
   Text[colors[[i]],{20,-i*10-5},{-1,0}]},{i,Length[colors]}];

   Show[Graphics[colors], AspectRatio->Automatic, 
   PlotRange->{{0,80},{0,-(Length[colors]*10+10)}}]
	
*)


Message[General::obspkg, "Graphics`Colors`"]
BeginPackage["Graphics`Colors`"]


CMYColor::usage =
"CMYColor[c,m,y] represents a color in the CMY \
(cyan-magenta-yellow) system."

YIQColor::usage =
"YIQColor[y,i,q] represents a color in the YIQ \
(NTSC video form) system."

HSBColor::usage =
"HSBColor[h,s,b] is an obsolete function, superseded \
by Hue." 
 
HLSColor::usage =
"HLSColor[h,l,s] represents a color in the HLS \
(hue-lightness-saturation) system." 
 
AllColors::usage =
"AllColors is a list of the names of colors \
defined symbolically, as strings."


Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] := Message[General::obspkgfn, fun, context];

CMYColor[c_, m_, y_] := (issueObsoleteFunMessage[CMYColor,"Graphics`Colors`"];
	RGBColor[1 - c, 1 - m, 1 - y])

YIQColor[y_, i_, q_] := 
	(issueObsoleteFunMessage[YIQColor,"Graphics`Colors`"];
	Apply[RGBColor,
	      {{1, .95, .625}, {1, -.28, -.64}, {1, -1.11, 1.73}} . {y, i, q}
	     ])

HSBColor[h_, s_, b_] := (issueObsoleteFunMessage[HSBColor,"Graphics`Colors`"];
	RGBColor[b, b, b] /; s == 0)

HSBColor[h_?NumberQ, s_, b_] := 
	(issueObsoleteFunMessage[HSBColor,"Graphics`Colors`"];
	Block[{mh = 6 Mod[h, 1.], i, f, p, q, t},
		i = Floor[mh] ;
		f = mh - i ;
		p = b (1. - s) ;
		q = b (1. - s f) ;
		t = b (1. - s (1. - f)) ;
		{RGBColor[b, t, p] ,
		 RGBColor[q, b, p] ,
		 RGBColor[p, b, t] ,
		 RGBColor[p, q, b] ,
		 RGBColor[t, p, b] ,
		 RGBColor[b, p, q]
		} [[i + 1]]
	]) 

HLSColor[h_, l_, s_] := (issueObsoleteFunMessage[HLSColor,"Graphics`Colors`"];
	RGBColor[l, l, l] /; s == 0)

HLSColor[h_?NumberQ, l_?NumberQ, s_] :=
    (issueObsoleteFunMessage[HLSColor,"Graphics`Colors`"];
	Block[{m1, m2},
	m2 = If[l <= 0.5 , l (1 + s) ,l + s - l s] ;
	m1 = 2 l - m2 ;
	RGBColor[HLSValue[m1, m2, h+1/3] ,
	         HLSValue[m1, m2, h] ,
	         HLSValue[m1, m2, h-1/3] ]
    ])

HLSValue[n1_, n2_, hue_] :=
    Block[{ hv = 6 Mod[hue, 1] } ,
	{n1 + (n2 - n1) hv,
	 n2,
	 n2,
	 n1 + (n2 - n1)(4 - hv),
	 n1,
	 n1
	} [[ 1 + Floor[hv] ]]
    ]

 
ColorListing =  {
  
{"AliceBlue" , RGBColor[0.941206,0.972503,1.] },
{"AlizarinCrimson" , RGBColor[0.889996,0.149998,0.209998] },
{"Antique" , RGBColor[0.980575,0.92233,0.844661] },
{"Aquamarine" , RGBColor[0.498001,1.,0.831401] },
{"MediumAquamarine" , RGBColor[0.400006,0.803903,0.666707] },
{"AureolineYellow" , RGBColor[1.,0.659993,0.140004] },
{"Azure" , RGBColor[0.94174,1.,1.] },
{"Banana" , RGBColor[0.889996,0.810007,0.340007]  },
{"Beige" , RGBColor[0.640004,0.580004,0.5] },
{"Bisque" , RGBColor[1.,0.898053,0.771844] },
{"BlanchedAlmond" , RGBColor[1.,0.921598,0.803903] },
{"LightBlue" , RGBColor[0.678396,0.847102,0.902005] },
{"MediumBlue" , RGBColor[0.,0.,0.803903] },
{"BlueViolet" , RGBColor[0.5412,0.1686,0.886303] },
{"Brick" , RGBColor[0.610004,0.400006,0.119999] },
{"BrownMadder" , RGBColor[0.859996,0.159993,0.159993] },
{"BrownOchre" , RGBColor[0.53,0.260002,0.119999] },
{"Burlywood" , RGBColor[0.870602,0.721595,0.529405] },
{"BurntSienna" , RGBColor[0.539994,0.209998,0.059999] },
{"BurntUmber" , RGBColor[0.539994,0.200003,0.140004] },
{"CadetBlue" , RGBColor[0.372494,0.619602,0.627506] },
{"CadmiumLemon" , RGBColor[1.,0.889996,0.009995] },
{"CadmiumOrange" , RGBColor[1.,0.380001,0.009995] },
{"DeepCadmiumRed" , RGBColor[0.889996,0.089999,0.050005] },
{"LightCadmiumRed" , RGBColor[1.,0.009995,0.050005] },
{"CadmiumYellow" , RGBColor[1.,0.599994,0.069994] },
{"LightCadmiumYellow" , RGBColor[1.,0.689993,0.059999] },
{"Carrot" , RGBColor[0.930006,0.569994,0.129994] },
{"Cerulean" , RGBColor[0.020005,0.719993,0.799997] },
{"Chartreuse" , RGBColor[0.498001,1.,0.] },
{"Chocolate" , RGBColor[0.823496,0.411802,0.117603] },
{"ChromeOxideGreen" , RGBColor[0.400006,0.5,0.080004] },
{"CinnabarGreen" , RGBColor[0.380001,0.700003,0.159993] },
{"Cobalt" , RGBColor[0.239998,0.350002,0.670003] },
{"CobaltGreen" , RGBColor[0.239998,0.569994,0.250008] },
{"DeepCobaltViolet" , RGBColor[0.569994,0.129994,0.619999] },
{"ColdGray" , RGBColor[0.5,0.539994,0.53] },
{"Coral" , RGBColor[1.,0.498001,0.3137] },
{"LightCoral" , RGBColor[0.941206,0.501999,0.501999] },
{"CornflowerBlue" , RGBColor[0.392193,0.584307,0.929395] },
{"Cornsilk" , RGBColor[1.,0.972503,0.862697] },
{"CyanWhite" , RGBColor[0.878399,1.,1.] },
{"DarkOrange" , RGBColor[1.,0.548997,0.] },
{"DeepPink" , RGBColor[1.,0.078402,0.576495] },
{"DeepOchre" , RGBColor[0.449995,0.239998,0.099994] },
{"DimGray" , RGBColor[0.333333,0.333333,0.333333] },
{"DodgerBlue" , RGBColor[0.117603,0.564699,1.] },
{"Eggshell" , RGBColor[0.990005,0.900006,0.790002] },
{"EmeraldGreen" , RGBColor[0.,0.790002,0.340007] },
{"EnglishRed" , RGBColor[0.829997,0.239998,0.099994] },
{"Firebrick" , RGBColor[0.698004,0.133305,0.133305] },
{"Oak" , RGBColor[1.,0.490005,0.250008] },
{"Apricot" , RGBColor[1.,0.340007,0.129994] },
{"Floral" , RGBColor[1.,0.980407,0.941206] },
{"ForestGreen" , RGBColor[0.133305,0.545106,0.133305] },
{"Gainsboro" , RGBColor[0.862697,0.862697,0.862697] },
{"GeraniumLake" , RGBColor[0.889996,0.069994,0.189993] },
{"Ghost" , RGBColor[0.972503,0.972503,1.] },
{"Gold" , RGBColor[1.,0.843104,0.] },
{"GoldOchre" , RGBColor[0.780007,0.47,0.149998] },
{"Goldenrod" , RGBColor[0.8549,0.647099,0.125507] },
{"DarkGoldenrod" , RGBColor[0.721595,0.525498,0.043107] },
(* {"LightGoldenrod" , RGBColor[0.980407,0.980407,0.823496] }
   The definition for this color later on is better, use
   it instead. --JMN *)
{"PaleGoldenrod" , RGBColor[0.933302,0.909803,0.666707] },
{"DarkGreen" , RGBColor[0.,0.392193,0.] },
{"PaleGreen" , RGBColor[0.596103,0.984298,0.596103] },
{"GreenYellow" , RGBColor[0.678396,1.,0.184301] },
{"GreenishUmber" , RGBColor[1.,0.239998,0.050005] },
{"Honeydew" , RGBColor[0.941206,1.,0.941206] },
{"HotPink" , RGBColor[1.,0.411802,0.705893] },
{"IndianRed" , RGBColor[0.689993,0.089999,0.119999] },
{"Indigo" , RGBColor[0.03,0.179998,0.329997] },
{"Ivory" , RGBColor[1.,1.,0.941206] },
{"IvoryBlack" , RGBColor[0.159993,0.140004,0.129994] },
{"Khaki" , RGBColor[0.941206,0.902005,0.548997] },
{"DarkKhaki" , RGBColor[0.741203,0.717597,0.419599] },
{"LampBlack" , RGBColor[0.179998,0.280007,0.230003] },
{"Lavender" , RGBColor[0.902005,0.902005,0.980407] },
{"LavenderBlush" , RGBColor[1.,0.941206,0.960799] },
{"LawnGreen" , RGBColor[0.486297,0.988205,0.] },
{"LemonChiffon" , RGBColor[1.,0.980407,0.803903] },
{"LightBeige" , RGBColor[0.960799,0.960799,0.862697] },
{"LightGoldenrod" , RGBColor[0.933302,0.866695,0.509796] },
{"LightGray" , RGBColor[0.666666,0.666666,0.666666] },
{"LightSalmon" , RGBColor[1.,0.627506,0.478393] },
{"LimeGreen" , RGBColor[0.196097,0.803903,0.196097] },
{"Linen" , RGBColor[0.980407,0.941206,0.902005] },
{"DeepMadderLake" , RGBColor[0.889996,0.179998,0.189993] },
{"ManganeseBlue" , RGBColor[0.009995,0.659993,0.619999] },
{"Maroon" , RGBColor[0.690207,0.188192,0.376507] },
{"MarsOrange" , RGBColor[0.589999,0.269997,0.080004] },
{"MarsYellow" , RGBColor[0.889996,0.440001,0.099994] },
{"Melon" , RGBColor[0.889996,0.659993,0.410001] },
{"MidnightBlue" , RGBColor[0.097995,0.097995,0.439207] },
{"Mint" , RGBColor[0.739998,0.990005,0.790002] },
{"MintCream" , RGBColor[0.960799,1.,0.980407] },
{"MistyRose" , RGBColor[1.,0.894101,0.882397] },
{"Moccasin" , RGBColor[1.,0.894101,0.709799] },
{"DeepNaplesYellow" , RGBColor[1.,0.659993,0.069994] },
{"Navajo" , RGBColor[1.,0.870602,0.678396] },
{"Navy" , RGBColor[0.,0.,0.501999] },
{"NavyBlue" , RGBColor[0.,0.,0.501999] },
{"OldLace" , RGBColor[0.992203,0.960799,0.902005] },
{"Olive" , RGBColor[0.230003,0.370006,0.170003] },
{"OliveDrab" , RGBColor[0.419599,0.556902,0.137303] },
{"DarkOliveGreen" , RGBColor[0.333293,0.419599,0.184301] },
{"OrangeRed" , RGBColor[1.,0.270608,0.] },
{"Orchid" , RGBColor[0.8549,0.439207,0.839198] },
{"DarkOrchid" , RGBColor[0.599994,0.196097,0.799997] },
{"MediumOrchid" , RGBColor[0.729408,0.333293,0.827494] },
{"PapayaWhip" , RGBColor[1.,0.9373,0.835307] },
{"Peach" , RGBColor[0.44, 0.26, 0.26] },
{"PeachPuff" , RGBColor[1.,0.8549,0.725501] },
{"Peacock" , RGBColor[0.200003,0.629994,0.790002] },
{"PermanentGreen" , RGBColor[0.039995,0.790002,0.170003] },
{"PermanentRedViolet" , RGBColor[0.859996,0.149998,0.269997] },
{"Peru" , RGBColor[0.803903,0.521607,0.247093] },
{"LightPink" , RGBColor[1.,0.713706,0.756905] },
{"Plum" , RGBColor[0.866695,0.627506,0.866695] },
{"PowderBlue" , RGBColor[0.690207,0.878399,0.902005] },
{"PrussianBlue" , RGBColor[0.18, 0.18, 0.31] },
{"MediumPurple" , RGBColor[0.576495,0.439207,0.858806] },
{"Raspberry" , RGBColor[0.53,0.149998,0.340007] },
{"RawSienna" , RGBColor[0.780007,0.380001,0.080004] },
{"RawUmber" , RGBColor[0.449995,0.290002,0.069994] },
{"RoseMadder" , RGBColor[0.889996,0.209998,0.219993] },
{"RosyBrown" , RGBColor[0.737297,0.560793,0.560793] },
{"RoyalBlue" , RGBColor[0.254906,0.411802,0.882397] },
{"SaddleBrown" , RGBColor[0.545106,0.270608,0.074496] },
{"Salmon" , RGBColor[0.980407,0.501999,0.447096] },
{"SandyBrown" , RGBColor[0.956893,0.643101,0.376507] },
{"SapGreen" , RGBColor[0.189993,0.5,0.080004] },
{"SeaGreen" , RGBColor[0.180395,0.545106,0.341197] },
{"DarkSeaGreen" , RGBColor[0.560793,0.737297,0.560793] },
{"LightSeaGreen" , RGBColor[0.125507,0.698004,0.666707] },
{"MediumSeaGreen" , RGBColor[0.235298,0.702002,0.443098] },
{"Seashell" , RGBColor[1.,0.960799,0.933302] },
{"Sepia" , RGBColor[0.370006,0.149998,0.069994] },
{"Sienna" , RGBColor[0.627506,0.321604,0.176504] },
{"SkyBlue" , RGBColor[0.529405,0.807794,0.921598] },
{"DeepSkyBlue" , RGBColor[0.,0.749,1.] },
{"LightSkyBlue" , RGBColor[0.529405,0.807794,0.980407] },
{"SlateBlue" , RGBColor[0.415693,0.352901,0.803903] },
{"DarkSlateBlue" , RGBColor[0.282403,0.239204,0.545106] },
{"LightSlateBlue" , RGBColor[0.517594,0.439207,1.] },
{"MediumSlateBlue" , RGBColor[0.482406,0.407804,0.933302] },
{"SlateGray" , RGBColor[0.439207,0.501999,0.564699] },
{"DarkSlateGray" , RGBColor[0.184301,0.309793,0.309793] },
{"LightSlateGray" , RGBColor[0.466704,0.533296,0.599994] },
{"Smoke" , RGBColor[0.960799,0.960799,0.960799] },
{"Snow" , RGBColor[1.,0.980407,0.980407] },
{"SpringGreen" , RGBColor[0.,1.,0.498001] },
{"MediumSpringGreen" , RGBColor[0.,0.980407,0.6039] },
{"SteelBlue" , RGBColor[0.274499,0.509796,0.705893] },
{"LightSteelBlue" , RGBColor[0.690207,0.768593,0.870602] },
{"TerreVerte" , RGBColor[0.219993,0.370006,0.059999] },
{"Thistle" , RGBColor[0.847102,0.749,0.847102] },
{"Titanium" , RGBColor[0.990005,1.,0.940001] },
{"Tomato" , RGBColor[1.,0.388195,0.278405] },
{"Turquoise" , RGBColor[0.250999,0.878399,0.815699] },
{"TurquoiseBlue" , RGBColor[0.,0.780007,0.550005] },
{"DarkTurquoise" , RGBColor[0.,0.807794,0.819605] },
{"MediumTurquoise" , RGBColor[0.282403,0.819605,0.799997] },
{"PaleTurquoise" , RGBColor[0.6863,0.933302,0.933302] },
{"Ultramarine" , RGBColor[0.069994,0.039995,0.559999] },
{"UltramarineViolet" , RGBColor[0.359996,0.140004,0.430006] },
{"VanDykeBrown" , RGBColor[0.370006,0.149998,0.020005] },
{"VenetianRed" , RGBColor[0.829997,0.099994,0.119999] },
{"Violet" , RGBColor[0.559999,0.370006,0.599994] },
{"DarkViolet" , RGBColor[0.580401,0.,0.827494] },
{"VioletRed" , RGBColor[0.815699,0.125507,0.564699] },
{"MediumVioletRed" , RGBColor[0.780404,0.0824,0.521607] },
{"PaleVioletRed" , RGBColor[0.858806,0.439207,0.576495] },
{"LightViridian" , RGBColor[0.430006,1.,0.440001] },
{"WarmGray" , RGBColor[0.5,0.5,0.410001] },
{"Wheat" , RGBColor[0.960799,0.870602,0.702002] },
{"YellowBrown" , RGBColor[0.86, 0.58, 0.44] },
{"YellowGreen" , RGBColor[0.6039,0.803903,0.196097] },
{"LightYellow" , RGBColor[1.,1.,0.878399] },
{"YellowOchre" , RGBColor[0.889996,0.509995,0.089999] },
{"Zinc" , RGBColor[0.990005,0.97,1.]}

}


BuildColor[{name_, rgb_}] :=
  ( Clear[Evaluate["Graphics`Colors`"<>name]];
    Evaluate[ToExpression["Graphics`Colors`"<>name]]::"usage" = 
	     name<>" is a color given in the RGBColor system.";
	Evaluate[ToExpression[name]] = rgb; )
	
Scan[BuildColor, ColorListing]
	
AllColors = Sort[{"Black", "Blue", "Brown", "Cyan", "Gray", "Green", "Magenta",
             "Orange", "Pink", "Purple", "Red", "White", "Yellow"} ~Join~
             First[ Transpose[ColorListing] ]
   ]



(***************************************************
       Add support for obsolete color names 
 ***************************************************)

BuildObsoleteColor[{old_, new_}] :=
  ( Clear[Evaluate["Graphics`Colors`"<>old]];
    Evaluate[ToExpression["Graphics`Colors`"<>old]]::"usage" = 
	     old<>" is an obsolete color name, superseded by "<>new<>".";
	ToExpression["Graphics`Colors`"<>old<>" = "<>new]; )
	
ObsoleteColorListing = { 
	     {"Cadet", "CadetBlue"}, 
		 {"GreenDark", "DarkGreen"},
		 {"OliveGreenDark", "DarkOliveGreen"},
		 {"OrchidDark", "DarkOrchid"},
         {"SlateBlueDark", "DarkSlateBlue"},
		 {"SlateGrayDark", "DarkSlateGray"},
		 {"TurquoiseDark", "DarkTurquoise"},
		 {"BlueLight", "LightBlue"},
	     {"SteelBlueLight", "LightSteelBlue"},
		 {"GreenPale", "PaleGreen"}, 
		 {"AquamarineMedium", "MediumAquamarine"},
		 {"BlueMedium", "MediumBlue"},
		 {"CadmiumYellowLight", "LightCadmiumYellow"},
		 {"CobaltVioletDeep", "DeepCobaltViolet"},
		 {"CoralLight", "LightCoral"},
		 {"GoldenrodDark", "DarkGoldenrod"},
		 {"GoldenrodLight", "LightGoldenrod"},
		 {"GoldenrodPale", "PaleGoldenrod"},
		 {"KhakiDark", "DarkKhaki"},
		 {"MadderLakeDeep", "DeepMadderLake"},
		 {"NaplesYellowDeep", "DeepNaplesYellow"},
		 {"OrchidMedium", "MediumOrchid"},
		 {"PinkLight", "LightPink"},
		 {"PurpleMedium", "MediumPurple"},
		 {"SeaGreenDark", "DarkSeaGreen"},
		 {"SeaGreenLight", "LightSeaGreen"},
		 {"SeaGreenMedium", "MediumSeaGreen"},
		 {"SkyBlueDeep", "DeepSkyBlue"},
		 {"SkyBlueLight", "LightSkyBlue"},
		 {"SlateBlueLight", "LightSlateBlue"},
		 {"SlateBlueMedium", "MediumSlateBlue"},
		 {"SlateGrayLight", "LightSlateGray"},
		 {"SpringGreenMedium", "MediumSpringGreen"},
		 {"TurquoiseMedium", "MediumTurquoise"},
		 {"TurquoisePale", "PaleTurquoise"},
		 {"VioletDark", "DarkViolet"},
		 {"VioletRedMedium", "MediumVioletRed"},
		 {"VioletRedPale", "PaleVioletRed"},
		 {"ViridianLight", "LightViridian"},
		 {"YellowLight", "LightYellow"},
		 {"CadmiumRedDeep", "DeepCadmiumRed"},
		 {"CadmiumRedLight", "LightCadmiumRed"}
	   }

Scan[ BuildObsoleteColor, ObsoleteColorListing]


(* Assign the Obsolete predicate to each obsolete color or function 
   so that they do not show up in the Function Browser. 
 *)
   
Attributes[System`Private`Obsolete] = {HoldAll}

Scan[(ToExpression[
    "System`Private`Obsolete /: System`Private`Obsolete["<>First[#]<>"] = True"
    ])&, 
    ObsoleteColorListing]

System`Private`Obsolete /: System`Private`Obsolete[HSBColor] = True
	   
End[]

EndPackage[]
