(* ::Package:: *)

(*:Name: Miscellaneous`Dictionary` *)

(*:Context: Miscellaneous`Dictionary` *)

(*:Title: Simple dictionary functions
*)

(*:Summary:
This package provides functions for basic dictionary operations,
such as verifying that a word is in the dictionary or finding words
matching a pattern
*)

(*:Author: Oyvind Tafjord *)

(*:Mathematica Version: 5.1 *)

(*:Package Version: 1.0 *)

(*:History: V1.0, Oyvind Tafjord
*)

(*:Keywords: Dictionary, Spelling.
*)

(*:Requirements: None. *)

(* :Copyright: Copyright 1992-2007, Wolfram Research, Inc.*)


Message[General::obspkg, "Miscellaneous`Dictionary`"]

BeginPackage["Miscellaneous`Dictionary`"]

DictionaryWordQ::usage = "DictionaryWordQ[\"word\"] returns True if \"word\" is in \
the dictionary $Dictionary. DictionaryWordQ[patt] returns True if \
a word matching the string pattern patt is in the dictionary."

GetDictionary::usage = "GetDictionary[\"file\"] loads the list of words \
stored in \"file\" and stores it in $Dictionary. GetDictionary[] loads the  \
default dictionary."

FindWords::usage = "FindWords[patt] finds all words in the dictionary $Dictionary \
matching the string pattern patt. FindWords[patt, n] returns at most n results. \
FindWords[patt, IgnoreCase -> True] treats upper and lower case as equivalent. \
FindWords[\"word\", SpellingCorrection -> True] finds words matching up to a small \
spelling correction."

SpellCheck::usage = "SpellCheck[\"string\"] finds all the words in string not in the \
$Dictionary and returns the string with these words highlighted in red, together with a list of \
these words."

Anagrams::usage = "Anagrams[\"word\"] lists all combinations of words in $Dictionary which are \
anagrams of \"word\". Anagrams[\"word\", n] only lists combinations with at most n words. \
Only the English letters a-z and A-Z are taken into account when forming the anagrams."

$Dictionary::usage = "$Dictionary contains the loaded dictionary, stored as a list of strings, \
used by DictionaryWordQ and FindWord."

$DefaultDictionaryFile::usage = "$DefaultDictionaryFile is the file name specification for the \
default dictionary file."

Unprotect[DictionaryWordQ, GetDictionary, FindWords, Anagrams, SpellCheck]

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=
        (Message[fun::obspkgfn, fun, context];
         )

Options[DictionaryWordQ] = Options[FindWords] = {IgnoreCase -> True, SpellingCorrection -> False};

$DefaultDictionaryFile = ToFileName[{$InstallationDirectory, "SystemFiles", "Dictionaries", "English"}, 
		"dictionary.txt"];
  
GetDictionary[file_String:""] := 
	(issueObsoleteFunMessage[GetDictionary,"Miscellaneous`Dictionary`"];
	With[{f = If[file === "", $DefaultDictionaryFile, file]},
		$Dictionary = ReadList[f, String]; f]);
	
getDictionaryIfNeeded[] := 
	If[Head[$Dictionary] =!= List, GetDictionary[]];
	
DictionaryWordQ[patt_, opts___] := 
	(issueObsoleteFunMessage[DictionaryWordQ,"Miscellaneous`Dictionary`"];
	Module[{ic = IgnoreCase/.{opts}/.Options[DictionaryWordQ],
	        sc = SpellingCorrection/.{opts}/.Options[DictionaryWordQ]},
	        getDictionaryIfNeeded[]; 
	        MemberQ[StringMatchQ[$Dictionary, patt, IgnoreCase->ic, SpellingCorrection->sc], True]])

FindWords[patt_, n_Integer:-1, opts___] := 
	(issueObsoleteFunMessage[FindWords,"Miscellaneous`Dictionary`"];
	Module[{res,ic = IgnoreCase/.{opts}/.Options[FindWords],
	        sc = SpellingCorrection/.{opts}/.Options[FindWords]}, 
	        getDictionaryIfNeeded[];
		res = Pick[$Dictionary, StringMatchQ[$Dictionary, patt, IgnoreCase->ic, SpellingCorrection->sc]];
		If[ n >=0 && n < Length[res], Take[res, n], res]
		]) 
		
SpellCheck[s_String] := 
	(issueObsoleteFunMessage[SpellCheck,"Miscellaneous`Dictionary`"];
	Module[{res, words = {}}, 
		getDictionaryIfNeeded[];

		res = StringReplace[s, word:(WordCharacter|"'"|"-")..:>
		If[DictionaryWordQ[word, IgnoreCase->True], word, AppendTo[words, word];
		"\!\(\*StyleBox[\"" <> word <> "\",FontColor->RGBColor[1,0,0]]\)"]];
		{res, words}])
		


lfreq[word_]:=
  Length/@Split[
        Sort[Join[Range[26],
            First/@tc[StringCases[ToLowerCase[word],RegularExpression["[a-z]"]]]-96]]]-1

tc[{}]={{-1}};

tc[x_]:=ToCharacterCode[x]

CheckDictionaryWordFreqCache[]:=
  (getDictionaryIfNeeded[]; 
  If[$WFCacheSavedDict=!=$Dictionary,
    With[{r=Range[26]},
    	$WordFreqCache=(Length/@Split[Sort[
            Join[r,First/@tc[StringCases[ToLowerCase[#], RegularExpression["[a-z]"]]]-96]]]-1)&/@$Dictionary];
        $WFCacheSavedDict=$Dictionary])

Anagrams[word_String,1]:=
	(issueObsoleteFunMessage[Anagrams,"Miscellaneous`Dictionary`"];
	(CheckDictionaryWordFreqCache[];
	Sort[List/@Pick[$Dictionary,$WordFreqCache,lfreq[word]]]))

Anagrams[word_String,n_:Infinity]:=(issueObsoleteFunMessage[Anagrams,"Miscellaneous`Dictionary`"];
	Module[{wfreq=lfreq[word]},
    CheckDictionaryWordFreqCache[];
    $Res={};
    anagrec[wfreq,Take[$WordFreqCache,All],Take[$Dictionary,All],{},n];
    Sort[$Res]])

anagrec[wfreq_,list_,wlist_,root_,n_]:=
  Module[{comp,newpos,newlist,newwlist},
    If[Total[wfreq]==0,AppendTo[$Res, root]];
    If[n<1, Return[]];
    comp=(wfreq-#)&/@list;
    newpos=Position[Min/@comp,_?(#>=0&),{1}];
    If[newpos=={},Return[]];
    newpos=First/@newpos;
    newlist=list[[newpos]];
    newwlist=wlist[[newpos]];
    MapIndexed[
      anagrec[wfreq-list[[#]],Drop[newlist,#2[[1]]-1],
          Drop[newwlist,#2[[1]]-1],Append[root,wlist[[#]]],n-1]&,newpos];]


End[]   (* Miscellaneous`Dictionary`Private` *)


Protect[DictionaryWordQ, GetDictionary, FindWords, Anagrams, SpellCheck]


EndPackage[]   (* Miscellaneous`Dictionary` *)



