(* ::Package:: *)

(* :Copyright: Copyright 1988-2007, Wolfram Research, Inc. *)

(* :Mathematica Version: 4.0 *)

(* :Package Version: 2.1 *)

(* :Name: NumberTheory`ContinuedFractions` *)

(* :Author: Mark Sofroniou *)

(* :Summary:
This package provides some typesetting functions for use in conjunction with
ContinuedFraction and RealDigits.
*)

(* :History:
V2.1 May 1998 by John M. Novak.
    Corrected some bugs in formatting; completely revised PeriodicForm;
    added formatting of periodic continued fractions; fixed bugs in
    Convergents.
Modified April 1998 by Mark Sofroniou.
    Moved functionality for periodic base expansions into RealDigits. 
Considerably extended January, September 1997.
    Moved ContinuedFractions into the kernel and added the
    Convergents function (suggested by Stan Wagon).
Totally rewritten March 1996 by Mark Sofroniou.
	Added PeriodicForm and ToPeriodicForm.
	Rewrote ContinuedFraction to unravel recursion and added
	unrestricted continued fractions for rationals.
Modified May 1992 by John M. Novak:
	Allow nonrational nonfloating-point numbers (estimates precision)
	Force machine numbers to bignums (machine precision) to reduce
	or eliminate roundoff error (with suggestions from J. Keiper).
Modified March 1992 by John M. Novak:
	Allow rational numbers. Fix behavior when partial fraction
	is finite and less than the order requested.
*)

(* :Keywords:
number theory, continued fractions, convergents, rational numbers,
decimal expansions.
*)

(* :Requirements: None. *)

(* :Discussion:
	The base expansion of any rational number can be written in terms
    of pre-periodic and periodic parts, either of which may be zero in
    length. For example, the decimal expansion of 1/8 terminates, but
    the expansion of 1/7 does not. 1/6 is a simple example of a rational
    number with non-zero pre-periodic and periodic parts
    (1 and 6 respectively).
*)

(* :Source:
    R. P. Brent, A. J. van der Porten and H. J. J. te Riele,
    A Comparative Study of Algorithms for Computing Continued
    Fractions of Algebraic Numbers. in Proceedings of the Second
    International Symposium on Algorithmic Number Theory (ANTS II),
    edited by H. Cohen, Springer Verlag, 1996, pp. 35--47.
	A. Ya. Khinchin, Continued Fractions,
	University of Chicago Press, Chicago, 1964.
	H. E. Rose, A Course in Number Theory,
	Second Edition, Oxford University Press, Oxford, 1994.
	K. H. Rosen, Elementary number theory and its applications,
	Third edition, Addison Wesley, Reading Mass., 1993.
	P. Shiu, Computation of continued fractions without input values,
	Math. Comp. 64 211, 1307--1317, 1995.
	S. Wagon, Mathematica in Action, W. H. Freeman, New York, 1991.
*)

(* :Limitations:
*)

Message[General::obspkg, "NumberTheory`ContinuedFractions`"]

BeginPackage["NumberTheory`ContinuedFractions`"]

Unprotect[ ContinuedFractionForm, PeriodicForm ]

ContinuedFractionForm::usage =
"ContinuedFractionForm[ cf ] represents a continued fraction as given \
by ContinuedFraction. The representation may be converted to a \
rational number using Normal."

PeriodicForm::usage = 
"PeriodicForm[{{d1, d2, ...,dn, {p1, p2, ..., pn}}, exp}] represents the \
decimal expansion of a rational in terms of the preperiodic and periodic \
parts as given by RealDigits. PeriodicForm[digits, b] represents the \
expansion in base b. The representation may be converted to a rational \
number using Normal."

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=
        (Message[General::obspkgfn, fun, context];
         )

(***** ContinuedFractionForm formatting wrapper *****)

(* Function to reconstruct a rational from a continued fraction expansion. *)
ContinuedFractionForm /:
  Normal[ ContinuedFractionForm[args__] ]:=
    (issueObsoleteFunMessage[ContinuedFractionForm,"NumberTheory`ContinuedFractions`"];
	Module[{res},
      res = FromContinuedFraction[args];
      res /; Head[res] =!= FromContinuedFraction
    ]);

ContinuedFractionForm /:
   FromContinuedFraction[ContinuedFractionForm[args__]] :=
       FromContinuedFraction[args]

(* One problem to avoid is that the normal typesetting rules
 reduce the font size of deeply nested fractions. *)

ContinuedFractionForm /:
  MakeBoxes[ cf:ContinuedFractionForm[a_?VectorQ], fmt_]:=
    (issueObsoleteFunMessage[ContinuedFractionForm,"NumberTheory`ContinuedFractions`"];
	InterpretationBox[#, cf]&[
      StyleBox[
        buildCFboxes[a],
        ScriptSizeMultipliers -> 1
      ]
    ])

(* formatting for cf with periodic part; notation due to Lou D'Andria *)

ContinuedFractionForm /:
  MakeBoxes[ cf:ContinuedFractionForm[{a___, per_?VectorQ}], fmt_]:=
    (issueObsoleteFunMessage[ContinuedFractionForm,"NumberTheory`ContinuedFractions`"];
	InterpretationBox[#, cf]&[
      StyleBox[
        If[Length[{a}] === 0,
            RowBox[{"(", buildCFboxes[Append[per, "\[Ellipsis]"]], ")"}],
            buildCFboxes[{a,
                RowBox[{"(", buildCFboxes[Append[per, "\[Ellipsis]"]], ")"}]}
            ]
        ],
        ScriptSizeMultipliers -> 1, SpanSymmetric -> False
      ]
    ])

buildCFboxes[{a___, b_}] :=
    Fold[RowBox[{#2, "+", FractionBox[1, #1]}]&,
          b,
          Reverse[{a}]
    ]/. (n_Integer :> ToString[n])

(***** PeriodicForm formatting wrapper *****)

(* Function to reconstruct a rational from a periodic base expansion. *)
PeriodicForm /:
  Normal[ PeriodicForm[
        digs:{{___Integer, {___Integer}} | {___Integer}, _Integer},
        rest___] ]:=
    (issueObsoleteFunMessage[PeriodicForm,"NumberTheory`ContinuedFractions`"];
	Module[{res},
      res = FromDigits[digs, rest];
      res /; Head[res] =!= FromDigits
    ]);

(* allow alternate form without exponent *)

PeriodicForm/:
  Normal[ PeriodicForm[digs:({___Integer, {___Integer}} | {___Integer}),
          rest___]] :=
      Normal[PeriodicForm[{digs, 0}, rest]]

(* Formatted version of periodic numbers (based on RealDigits output) *)
(* allow form without exponent specified; note that the Interpretation box
   needs to interpret as the original pf, not the one with the 0 exponent *)

PeriodicForm/:
  MakeBoxes[pf:PeriodicForm[digs:{___Integer, {___Integer}}, base_Integer:10],
            rest___] :=
       (issueObsoleteFunMessage[PeriodicForm,"NumberTheory`ContinuedFractions`"];
	MakeBoxes[PeriodicForm[{digs, 0}, base], rest]/.
           HoldPattern[InterpretationBox][f_, p_, r___] :>
               InterpretationBox[f, pf, r])

(* allow form without periodic part *)
PeriodicForm/:
  MakeBoxes[pf:PeriodicForm[{{digs___Integer},exp_}, base_Integer:10],
            rest___] :=
      (issueObsoleteFunMessage[PeriodicForm,"NumberTheory`ContinuedFractions`"];
	MakeBoxes[PeriodicForm[{{digs, {}}, exp}, base], StqndardForm]/.
           HoldPattern[InterpretationBox][f_, p_, r___] :>
               InterpretationBox[f, pf, r])

(* general form, with periodic part and exponent -
   note that 'ZeroWidthTimes' spacing is used in the row boxes if the
   base is <= 36 (which means a single character per digit); note that
   the product with the exponent is not added unless the exponent is
   between -6 and 6 (unless the nonperiodic part doesn't support moving
   the decimal appropriately). *)

PeriodicForm/:
  MakeBoxes[pf:PeriodicForm[
          digs:{{pre___Integer, per:{___Integer}}, exp_Integer},
          base_Integer:10],
      rest___] :=
      (issueObsoleteFunMessage[PeriodicForm,"NumberTheory`ContinuedFractions`"];
	InterpretationBox[#, pf]&[StyleBox[
         If[base <= 36, StyleBox[#, ZeroWidthTimes -> True], #]&[
             If[(-6 <= exp <=Length[{pre}]) && (exp <= 6),
                 RowBox[pform[{pre}, per, exp, base]],
                 RowBox[{RowBox[pform[{pre}, per, 0, base]], "\[Times]",
                         SuperscriptBox[ToString[base], ToString[exp]]}]
             ]
          ], AutoMultiplicationSymbol -> False]
      ])

(* formatting subutility, assembles list of digits with decimal point
    and base subscript (via fixbases) *)
(* empty number *)
pform[{}, {}, _, _] := {"0"}

(* following rule is redundant, but should be most called so slight
   performance improvement by including it, I think *)
pform[pre_, per_, 0, base_] :=
    fixbases[Flatten[{0, ".", pre, Overscribe[per]}], base]

(* not called in the following form unless n >= Length[pre]; shift
   decimal point right *)
pform[pre_, per_, n_?Positive, base_] :=
    fixbases[Flatten[{Take[pre, n], ".", Drop[pre, n], Overscribe[per]}],
             base]

(* shift decimal point left *)
pform[pre_, per_, n_, base_] :=
    fixbases[Flatten[{0, ".", Table[0, {Abs[n]}], pre, Overscribe[per]}],
             base]

(* fixbases also handles stringification of numbers, as well as
   formatting digits to the base and adding a subscript noting the base *)
fixbases[digs_, 10] := digs/. (i_Integer :> ToString[i])

(* for bases between 1 and 36 (other than 10), use letters for digits *)
$digitreplacements = Dispatch[
{0 -> "0", 1 -> "1", 2 -> "2", 3 -> "3", 4 -> "4", 5 -> "5", 6 -> "6",
 7 -> "7", 8 -> "8", 9 -> "9", 10 -> "A", 11 -> "B", 12 -> "C", 13 -> "D",
 14 -> "E", 15 -> "F", 16 -> "G", 17 -> "H", 18 -> "I", 19 -> "J", 20 -> "K",
 21 -> "L", 22 -> "M", 23 -> "N", 24 -> "O", 25 -> "P", 26 -> "Q", 27 -> "R",
 28 -> "S", 29 -> "T", 30 -> "U", 31 -> "V", 32 -> "W", 33 -> "X", 34 -> "Y",
 35 -> "Z"}
];

fixbases[digs_, b_?(# <= 36 &)] :=
    Append[Drop[#,-1], SubscriptBox[Last[#], ToString[b]]]&[
        digs/.$digitreplacements
    ]

fixbases[digs_, b_] :=
    Append[Drop[#,-1], SubscriptBox[Last[#], ToString[b]]]&[
        digs/.(d_Integer :> UnderscriptBox[ToString[d], "\[UnderBracket]"])
    ]

Overscribe[{}] = {};

Overscribe[per_]:=
  Map[OverscriptBox[#, "_"]&, per]

End[]  (* NumberTheory`ContinuedFractions`Private` *)

SetAttributes[{ContinuedFractionForm, PeriodicForm}, NHoldAll];

SetAttributes[ {ContinuedFractionForm, PeriodicForm}, ReadProtected ]

Protect[ ContinuedFractionForm, PeriodicForm]

EndPackage[]  (* NumberTheory`ContinuedFractions` *)

(*:Examples:

Normal[ ContinuedFractionForm[{1, 1, 2, 3}] ]
Normal[ ContinuedFractionForm[{1, 1, {2, 3}}] ]

PeriodicForm[ {{4, 2, {0, 3, 1, 7, 4}}, 0} ]
PeriodicForm[ {{4, 2, {0, 3, 1, 7, 4}}, 0}, 8 ]

Normal[ PeriodicForm[ {1, 2, {3, 4}} ] ]
Normal[ PeriodicForm[ {1, 2, {3, 4}}, 5 ] ]
*)
