(*:Title: Functions of Statistical Distributions *)

(*:Copyright: Copyright 1991-2007, Wolfram Research, Inc. *)

(*:Summary:
This package defines now obsolete functions used in statistical 
probability distribution packages.
*)

BeginPackage["Statistics`Common`DistributionsCommon`"]

(* Define usage messages for symbols shared among distribution packages only 
	if the messages are not already defined. 
*)

If[Head[Domain::usage] =!= String,

Domain::usage =
"Domain[distribution] is deprecated and will be removed in a future version.";

ParameterQ::usage =
"ParameterQ[distribution] is deprecated and will be removed in a future version.";

DomainQ::usage =
"DomainQ[distribution, x] is deprecated and will be removed in a future version."; 

RegionProbability::usage =
"RegionProbability[distribution, domain] is deprecated and will be removed in a future version. \
Use EllipsoidProbability in the MulltivariateStatistics` package instead.";

RandomArray::usage =
"RandomArray[distribution, n] is deprecated and will be removed in a future version. \
Use RandomReal or RandomInteger instead.";

]

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=Message[General::obspkgfn, fun, context]

ParameterQ[args___]:=Block[{res = DistributionParameterQ[args]}, 
 	issueObsoleteFunMessage[ParameterQ, "Statistics`Common`DistributionsCommon`"];
 	res /; FreeQ[res, DistributionParameterQ]]

DomainQ[args___]:=Block[{res = DistributionDomainQ[args]}, 
 	issueObsoleteFunMessage[DomainQ, "Statistics`Common`DistributionsCommon`"];
 	res /; FreeQ[res, DistributionDomainQ]]
	
Domain[args___]:=Block[{res = DistributionDomain[args]}, 
 	issueObsoleteFunMessage[Domain, "Statistics`Common`DistributionsCommon`"];
 	res /; FreeQ[res, DistributionDomain]]


(* RandomArray definitions inlined from obsoleted distribution packages *)

(* ======================== Beta Distribution ========================= *)
(* Continuous Beta (Binomial) Distribution, J & K, Vol.2, Chap. 24 *)

Unprotect[BetaDistribution]


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

BetaDistribution/: RandomArray[BetaDistribution[p_, q_], dim_] :=
  With[{x1 = RandomArray[GammaDistribution[p, 1], dim]},
    x1/(x1 + RandomArray[GammaDistribution[q, 1], dim])
  ] /; DistributionParameterQ[BetaDistribution[p, q]]&&VectorQ[{p,q},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[BetaDistribution]

(* ========================= Cauchy Distribution ========================== *)
(* Cauchy Distribution, J & K, Vol. 1, Chap. 16 *)

Unprotect[CauchyDistribution]


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

cauchyArray=Compile[{{a, _Real}, {b, _Real}, {q, _Real, 1}}, a + b Tan[Pi (q-1/2)] ]

CauchyDistribution/: RandomArray[CauchyDistribution[a_:0, b_:1], dim_] := 
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = cauchyArray[a, b, Table[Random[], {n}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /;DistributionParameterQ[CauchyDistribution[a, b]]&&VectorQ[{a,b},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[CauchyDistribution]

(* ========================== Chi Distribution =========================== *)
(* Chi Distribution, J & K, Vol. 1, Chap. 17, p. 196 *)

Unprotect[ChiDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)


ChiDistribution/: RandomArray[ChiDistribution[n_], dim_] :=
   (
	Sqrt[RandomArray[ChiSquareDistribution[n], dim]]
   ) /; DistributionParameterQ[ChiDistribution[n]]&&NumericQ[n]&&
   	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]
   	
Protect[ChiDistribution]
 
(* =================== Noncentral Chi-Square Distribution ================ *)
(* Noncentral Chi-Square Distribution, K & J, Vol. 2, Chap. 28 *)

Unprotect[NoncentralChiSquareDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

qRandomNoncentralChiSquare[n_, lambda_]:=Quantile[NoncentralChiSquareDistribution[n, lambda], Random[]]
 
NoncentralChiSquareDistribution/:
	RandomArray[NoncentralChiSquareDistribution[n_, lambda_], dim_] :=
  Module[{m, delta = Sqrt[lambda/n], array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    If[IntegerQ[n],
    array = RandomArray[NormalDistribution[0, 1], {m, n}];
    array = Map[Apply[Plus, (# + delta)^2]&, array],
 	(* non-integral degrees of freedom *)
 	array = Table[qRandomNoncentralChiSquare[n, lambda], {m}]];
 	(* Now array is a list of m NoncentralChiSquareDistribution[n, lambda]
 	variates. *)
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[NoncentralChiSquareDistribution[n, lambda]]&& n > 0 &&
  	NumericQ[lambda]&&(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[NoncentralChiSquareDistribution]

(* ========================= Exponential Distribution ===================== *)
(* Exponential Distribution, J & K, Vol. 1, Chap. 18 *)

Unprotect[ExponentialDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)


exponentialArray = Compile[{{lambda, _Real}, {q, _Real, 1}}, -Log[q]/lambda]

ExponentialDistribution/: RandomArray[ExponentialDistribution[lambda_:1], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = exponentialArray[lambda, Table[Random[], {n}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /;DistributionParameterQ[ExponentialDistribution[lambda]]&&NumericQ[lambda]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[ExponentialDistribution]
  
(* ======================= ExtremeValue Distribution ====================== *)
(* ExtremeValue (FisherTippett) Distribution, J & K, Vol. 1, Chap. 21 *)

Unprotect[ExtremeValueDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

extremevalueArray = Compile[{{alpha, _Real}, {beta, _Real}, {q, _Real, 1}},
			alpha - beta Log[Log[1/q]]   ]

ExtremeValueDistribution/: RandomArray[ExtremeValueDistribution[alpha_, beta_],
	 dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = extremevalueArray[alpha, beta, Table[Random[], {n}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /;DistributionParameterQ[ExtremeValueDistribution[alpha, beta]]&&VectorQ[{alpha,beta},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[ExtremeValueDistribution]

(* ============================ F-ratio Distribution ======================== *)
(* F-ratio distribution is defined in Statistics`NormalDistribution` *)


(* ==================== Noncentral F-ratio Distribution ===================== *)
(* Noncentral F-ratio Distribution, J & K, Vol. 2, Chap. 30 *)

Unprotect[NoncentralFRatioDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)


(* I am assuming that this is always more efficient than
	Quantile[NoncentralFRatioDistribution[n1,n2,lambda], Random[]]
*)

NoncentralFRatioDistribution/:
        RandomArray[NoncentralFRatioDistribution[n1_, n2_, lambda_], dim_] :=
   n2/n1 RandomArray[NoncentralChiSquareDistribution[n1, lambda], dim]/
	 RandomArray[ChiSquareDistribution[n2], dim]/;
   	DistributionParameterQ[NoncentralFRatioDistribution[n1, n2, lambda]]&&
   	VectorQ[{n1,n2,lambda},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[NoncentralFRatioDistribution]

(* =========================== Gamma Distribution ======================== *)
(* Gamma Distribution, J & K, Vol. 1, Chap. 17 *)

Unprotect[GammaDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)


(* additional efficiency is obtained by Compile[]ing Table[]s in the *GammaArray functions,
   rather than directly computing Table[iRandomGamma[alpha,beta],{n}] *)
  
iRandomGammaArray[alpha_, beta_, n_]:=Block[{r, k, a, b, c},
	Which[alpha < 1, r = alpha/E;
 	      beta gsGammaArray[alpha, r, n]
 	      ,
 	      alpha==1, exponentialArray[beta,Table[Random[], {n}]]
 	      ,
 	      alpha < 2.5, a = alpha-1;  b = (alpha - 1/(6 alpha))/a;  c = 2/a;
	      beta a gkm1GammaArray[b, c, n]
	      ,
	      True, k = 1/Sqrt[alpha]; a = alpha-1;  
	      b = (alpha - 1/(6 alpha))/a;  c = 2/a;
	      beta a gkm2GammaArray[k, b, c, n]
	      ]]

GammaDistribution/: RandomArray[GammaDistribution[alpha_, beta_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = iRandomGammaArray[alpha, beta, n]; 
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[GammaDistribution[alpha, beta]]&&VectorQ[{alpha, beta},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* acceptance-rejection method for Gamma variates;
   GS algorithm of Ahrens and Dieter (1974),
	pp. 24-25, 39, "Multivariate Statistical Simulation", M.E.Johnson *)

(* NOTE: The constant "r" is passed to gsGamma in an attempt to
	 minimize overhead. *)
(* NOTE: The initialization x = 1.0, t = 1.0 is done just so that the
	V4.0 Compile is happy. *)


gsGammaArray = Compile[{{alpha, _Real}, {r, _Real}, {n, _Integer}}, 
  	Module[{x = 1.0, t = 1.0, q},
  	  Table[q = (1 + r) Random[];
    	    If[q < 1, 
    	    	x = q^(1/alpha); t = Exp[-x], 
    	    	x = 1 - Log[1 + (1 - q)/r];t = x^(alpha - 1)];
    	    While[Random[] > t, 
    	    	q = (1 + r) Random[];
     	       	If[q < 1, 
     	       	   x = q^(1/alpha); t = Exp[-x], 
     	       	   x = 1 - Log[1 + (1 - q)/r]; t = x^(alpha - 1)]];
    	    x, {n}]]]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* ratio of uniforms technique for Gamma variates;
   GBH algorithm of Cheng and Feast (1979),
  pp. 28-29, 40-41, "Multivariate Statistical Simulation", M.E.Johnson;
  "Some Simple Gamma Variate Generators", R.C.H.Cheng & G.M.Feast,
   Applied Statistics (1979), 28, No. 3, pp. 290-295. *)

(* NOTE: The constants "b" and "c" are passed to gkm1Gamma in an attempt to
	minimize overhead. *)
(* NOTE: The initialization w = 1.0 is done just so that the
	V4.0 Compile is happy. *)
  
   
gkm1GammaArray = Compile[{{b, _Real}, {c, _Real}, {n, _Integer}},
   Module[{u1, u2, w = 1.0},
    Table[
    	u1 = Random[]; u2 = Random[]; w = b u1/u2;
     	While[! (c u2 - c - 2 + w + 1/w <= 0) && ! (c Log[u2] - Log[w] + w - 1 < 0), 
     	   u1 = Random[]; u2 = Random[]; w = b u1/u2];
     	w, {n}]]]
	
	
(* NOTE: The constants "k", "b", and "c" are passed to gkm2Gamma in an attempt 
	to minimize overhead. *)
(* NOTE: The initialization u2 = 1.0, w = 1.0 is done just so that the
        V4.0 Compile is happy. *)
  
  
gkm2GammaArray = Compile[{{k, _Real}, {b, _Real}, {c, _Real}, {n, _Integer}}, 
   Module[{u1, u2, w = 1.0},
     Table[u1 = Random[];
     	u2 = u1 + k (1 - 1.86 Random[]);
     	While[! (0 < u2 < 1),
     	   u1 = Random[]; u2 = u1 + k (1 - 1.86 Random[])];
     	w = b u1/u2;
     	While[! (c u2 - c - 2 + w + 1/w <= 0) && ! (c Log[u2] - Log[w] + w - 1 < 0),
      	   u1 = Random[]; u2 = u1 + k (1 - 1.86 Random[]);
      	While[! (0 < u2 < 1),
           u1 = Random[]; u2 = u1 + k (1 - 1.86 Random[])];
      	w = b u1/u2];
     	w, {n}]]]
     	
Protect[GammaDistribution] 
 
(* ========================== Half-Normal Distribution ===================== *)
(* Half-Normal Distribution, J & K, Vol. 1, Chap. 13, p. 93 *)

Unprotect[HalfNormalDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

halfnormal = Compile[{{theta, _Real}, {q1, _Real}, {q2, _Real}},
	Sqrt[Pi/2]/theta Sqrt[-2 Log[q1]] Abs[Cos[2Pi q2]]
	]

(*
generation of array using listability is as fast as an equivalent compilation
*)

HalfNormalDistribution/: RandomArray[HalfNormalDistribution[theta_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Sqrt[Pi/2]/theta*Flatten[
    	Block[{mat = Table[Random[], {Quotient[n, 2]}, {2}], q1, q2}, 
    		{q1, q2} = {mat[[All, 1]], mat[[All, 2]]};
    		Sqrt[-2 Log[q1]]*Abs[Transpose[({Cos[2Pi#], Sin[2Pi#]} &)[q2]]]]];
    If[OddQ[n],
       AppendTo[array, halfnormal[theta, Random[], Random[]] ]  ];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[HalfNormalDistribution[theta]]&&NumericQ[theta]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[HalfNormalDistribution]

(* ================= Laplace (Double Exponential) Distribution ============== *)
(* Laplace (Double Exponential) Distribution, J & K, Vol. 2, Chap. 23 *)

Unprotect[LaplaceDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

laplaceArray = Compile[{{mu, _Real}, {beta, _Real}, {$15, _Real, 1},{$16, _Real,1}},
		mu - beta $16 Log[1 - $16 $15]  ]
		
LaplaceDistribution/: RandomArray[LaplaceDistribution[mu_, beta_], dim_] :=
  Module[{n, array, q},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    q=2Table[Random[], {n}]-1;
    array = laplaceArray[mu, beta, q, Sign[q]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[LaplaceDistribution[mu, beta]]&&VectorQ[{mu,beta},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[LaplaceDistribution]
 
(* ========================= Log-Normal Distribution ======================= *)
(* Log-Normal Distribution, J & K, Vol. 1, Chap. 14, p. 112 *)

Unprotect[LogNormalDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

lognormal = Compile[{{mu, _Real}, {sigma, _Real}, {q1, _Real}, {q2, _Real}},
		Exp[mu + sigma Sqrt[-2 Log[q1]] Cos[2Pi q2]]	]

(*
generation of array using listability is as fast as an equivalent compilation
*)


LogNormalDistribution/: RandomArray[LogNormalDistribution[mu_, sigma_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Flatten[Block[{mat = Table[Random[], {Quotient[n, 2]}, {2}], q1, q2},
  	{q1, q2} = {mat[[All, 1]], mat[[All, 2]]};
  	With[{k = sigma Sqrt[-2 Log[q1]]}, 
  		({Exp[mu + k Cos[2Pi #]], Exp[mu + k Sin[2Pi #]]} &)[q2]]]];
    If[OddQ[n],
       AppendTo[array, lognormal[mu, sigma, Random[], Random[]] ]  ];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /;DistributionParameterQ[LogNormalDistribution[mu, sigma]]&&VectorQ[{mu,sigma},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[LogNormalDistribution]
   
(* ========================== Logistic Distribution ======================== *)
(* Logistic Distribution, K & J, Vol. 2, Chap. 22, p. 1 *)

Unprotect[LogisticDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

logisticArray = Compile[{{mu, _Real}, {beta, _Real}, {q, _Real, 1},{signbeta,_Real}},
		mu - beta signbeta Log[1-q] + beta signbeta Log[q] ]
		
LogisticDistribution/: RandomArray[LogisticDistribution[mu_:0, beta_:1],
	 dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = logisticArray[mu, beta, Table[Random[], {n}],Sign[beta]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /;DistributionParameterQ[LogisticDistribution[mu, beta]]&&VectorQ[{mu,beta},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[LogisticDistribution]

(* ========================== Rayleigh Distribution ======================= *)
(* Rayleigh Distribution, K & J, Vol. 1, Chap. 17, p. 197 *)

Unprotect[RayleighDistribution]
 
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

rayleighArray = Compile[{{sigma, _Real}, {q, _Real, 1}},
			sigma Sqrt[Log[1/q^2]]  ]
			
RayleighDistribution/: RandomArray[RayleighDistribution[sigma_:1], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = rayleighArray[sigma, Table[Random[], {n}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[RayleighDistribution[sigma]]&&NumericQ[sigma]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[RayleighDistribution]

(* ==================== Noncentral Student's t Distribution ================= *)
(* Noncentral Student's t Distribution, K & J, Vol. 2, Chap. 31 *)

Unprotect[NoncentralStudentTDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

 
NoncentralStudentTDistribution/:
	RandomArray[NoncentralStudentTDistribution[n_, delta_], dim_] :=
  (
  Sqrt[n] (RandomArray[NormalDistribution[0, 1], dim] + delta) /
		RandomArray[ChiDistribution[n], dim]
  ) /;DistributionParameterQ[NoncentralStudentTDistribution[n, delta]]&&VectorQ[{n,delta},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[NoncentralStudentTDistribution]

(* =========================== Pareto Distribution ======================= *)
(* Pareto Distribution, K & J, Vol. 1, Chap. 19 *)

Unprotect[ParetoDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

paretoArray = Compile[{{k, _Real}, {alpha, _Real}, {q, _Real, 1}},
		k / (1-q)^(1/alpha)	]
		
ParetoDistribution/: RandomArray[ParetoDistribution[k_, alpha_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = paretoArray[k, alpha, Table[Random[], {n}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[ParetoDistribution[k, alpha]]&&VectorQ[{k,alpha},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[ParetoDistribution]

(* ========================== Uniform Distribution ======================= *)
(* Uniform Distribution, K & J, Vol. 2, Chap. 25 *)

Unprotect[UniformDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* NOTE: For UniformDistribution, RandomArray provides no speedup over Random. *)
UniformDistribution/: RandomArray[UniformDistribution[{min_, max_}], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Table[Random[Real, {min, max}], {n}];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /;DistributionParameterQ[UniformDistribution[{min, max}]]&&VectorQ[{min,max},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[UniformDistribution]

(* =========================== Weibull Distribution ====================== *)
(* Weibull Distribution, K & J, Vol. 1, Chap. 20 *)

Unprotect[WeibullDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)


weibullArray = Compile[{{alpha, _Real}, {beta, _Real}, {q, _Real, 1}},
		beta Log[1/q]^(1/alpha)		]
		
WeibullDistribution/: RandomArray[WeibullDistribution[alpha_, beta_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = weibullArray[alpha, beta, Table[Random[],{n}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[WeibullDistribution[alpha, beta]]&&VectorQ[{alpha,beta},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[WeibullDistribution]

(* =========================== Inverse Gaussian Distribution ====================== *)
(* Inverse Gaussian Distribution, K & J, Vol. 1, Chap. 15 *)

Unprotect[InverseGaussianDistribution]
  

InverseGaussianDistribution/: RandomArray[InverseGaussianDistribution[mu_, lambda_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Map[({x1, u} = #;
      If[u <= mu/(mu + x1),
       x1,
       mu^2/x1]) &, 
    Transpose[{(mu + mu^2*#/(2lambda) - 
          mu/(2lambda)Sqrt[4mu*lambda*# + (mu*#)^2]) &[
       RandomArray[NormalDistribution[0, 1], n]^2], 
      RandomArray[UniformDistribution[{0, 1}], n]}]];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[InverseGaussianDistribution[mu, lambda]]&&VectorQ[{mu,lambda},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[InverseGaussianDistribution]
   
(* =========================== Triangular Distribution ====================== *)

Unprotect[TriangularDistribution]
   

TriangularDistribution/: RandomArray[
	TriangularDistribution[{a_, b_}, c_], dim_] :=
  Module[{bnd = N[a/(a - b) - c/(a - b)],
   lower = N[a + Sqrt[(a - b)*(a - c)*#1]],
   upper = N[b - Sqrt[(a - b)*(b - c)*(-1 + #1)]], f,n,array},
  n = If[VectorQ[dim], Apply[Times, dim], dim]; 
  f = If[# <= Evaluate[bnd], Evaluate[lower], Evaluate[upper]];
  array=Map[Evaluate[f] &, Table[Random[], {n}]];
  If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]] /; DistributionParameterQ[TriangularDistribution[{a, b}, c]]&&
  			VectorQ[N[{a,b,c}],NumberQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]
    

TriangularDistribution/: RandomArray[
	TriangularDistribution[{a_?NumericQ, b_?NumericQ}]?DistributionParameterQ, dim_] :=
  RandomArray[TriangularDistribution[{a, b},(a+b)/2],dim]

Protect[TriangularDistribution]

Unprotect[MaxwellDistribution]  

MaxwellDistribution /: 
 RandomArray[MaxwellDistribution[sigma_], dim_] := 
 (RandomArray[ChiDistribution[3], dim]sigma) /; 
  DistributionParameterQ[MaxwellDistribution[sigma]] && 
    NumericQ[sigma] && (IntegerQ[dim] && dim > 0) || 
   VectorQ[dim, (IntegerQ[#] && # > 0) &]

Protect[MaxwellDistribution]

(* ======================= Gumbel Distribution ====================== *)
(* Fisher-Tippett, type 1 Extreme Value distribution and Gumbel 
   distribution are used interchangeably for the type 1 extreme value
   distribution, where the extreme value can be either a maximum or a 
   minimum.  ExtremeValueDistribution[a, b] defines the distribution of 
   the maximum and GumbelDistribution[a, b] defines the distribution of
   the minimum. *)

Unprotect[GumbelDistribution]  

GumbelDistribution /: RandomArray[GumbelDistribution[a_, b_], dim_] := 
 Block[{array}, 
   array = Compile[{{aval, _Real}, {bval, _Real}, {nval, _Integer}}, 
      aval + bval Log[-Log[1 - Table[Random[], {nval}]]]][a, b, 
     If[VectorQ[dim], Apply[Times, dim], dim]];
   If[VectorQ[dim] && Length[dim] > 1, 
    Fold[Partition[#1, #2] &, array, Reverse[Drop[dim, 1]]], array]] /; 
  DistributionParameterQ[GumbelDistribution[a, b]] && 
    VectorQ[{a, b}, NumericQ] && (IntegerQ[dim] && dim > 0) || 
   VectorQ[dim, (IntegerQ[#] && # > 0) &]

Protect[GumbelDistribution] 

(* ========================= Chi-square Distribution ======================= *)
(* J & K, CUD Vol. 1, Chap. 17 *)

Unprotect[ChiSquareDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* additional efficiency is obtained by Compile[]ing Table[]s in the *GammaArray functions,
   rather than directly computing Table[iRandomGamma[alpha,beta],{n}] *)

ChiSquareDistribution/: RandomArray[ChiSquareDistribution[n_], dim_] :=
  Module[{m, array, r, k, a, b, c},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Which[n < 2,
                      r = n/2/E;
                      2 gsGammaArray[n/2, r, m],
		  		n == 2,
		      	2 exponentialArray[n,Table[Random[], {m}]],
                n < 5,
                      a = n/2-1;  b = (n/2 - 1/(6 n/2))/a;  c = 2/a;
                      2 a gkm1GammaArray[b, c, m],
                True,
                      k = 1/Sqrt[n/2];
                      a = n/2-1;  b = (n/2 - 1/(6 n/2))/a;  c = 2/a;
                      2 a gkm2GammaArray[k, b, c, m]             
                  ];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
       array  ]
  ] /; DistributionParameterQ[ChiSquareDistribution[n]]&&NumericQ[n]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[ChiSquareDistribution]

(* ========================= F-ratio Distribution ========================= *)
(* J & K, CUD Vol. 2, Chap. 26  *)

Unprotect[FRatioDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

FRatioDistribution/: RandomArray[FRatioDistribution[n1_, n2_], dim_] :=
   (
    n2/n1 RandomArray[ChiSquareDistribution[n1], dim] /
	  RandomArray[ChiSquareDistribution[n2], dim]
   ) /; DistributionParameterQ[FRatioDistribution[n1, n2]]&&VectorQ[{n1,n2},NumericQ]&&
   	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[FRatioDistribution]

(* ========================= Normal Distribution ========================= *)
(* K & J, CUD Vol. 1, Chap. 13 *)

Unprotect[NormalDistribution]
 
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

 
normal = Compile[{{mu, _Real}, {sigma, _Real}, {q1, _Real}, {q2, _Real}},
		mu + sigma Sqrt[-2 Log[q1]] Cos[2Pi q2]	]

(*
generation of array using listability is as fast as an equivalent compilation
*)

NormalDistribution/: RandomArray[NormalDistribution[mu_, sigma_], dim_] :=
  Module[{n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = mu + sigma Flatten[
    	Block[{mat = Table[Random[], {Quotient[n, 2]}, {2}], q1, q2},
 		{q1, q2} = {mat[[All, 1]], mat[[All, 2]]};
 		Sqrt[-2 Log[q1]]*Transpose[({Cos[2Pi#], Sin[2Pi#]} &)[q2]]]];
    If[OddQ[n],
       AppendTo[array, normal[mu, sigma, Random[], Random[]] ]  ];
    If[VectorQ[dim] && Length[dim] > 1,
       Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],	
       array  ]
  ] /; DistributionParameterQ[NormalDistribution[mu, sigma]]&&VectorQ[{mu,sigma},NumericQ]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[NormalDistribution]

(* ====================== Student's t Distribution ======================== *)
(* K & J, CUD Vol. 2, Chap. 27 *)

Unprotect[StudentTDistribution]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

StudentTDistribution/: RandomArray[StudentTDistribution[n_], dim_] :=
  (
    RandomArray[NormalDistribution[0, 1], dim] /
		Sqrt[ RandomArray[ChiSquareDistribution[n], dim]/n ]  
  ) /; DistributionParameterQ[StudentTDistribution[n]]&&NumericQ[n]&&
  	(IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[StudentTDistribution]

(* ===================================================================== *)


(* ========================================================================= *)
(* Bernoulli Distribution *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

      
bernoulliarray = Compile[{{p, _Real}, {len, _Integer}}, 
          Table[If[Random[] > p, 0, 1], {len}]]

Unprotect[BernoulliDistribution]

BernoulliDistribution/:
  RandomArray[BernoulliDistribution[p_?NumericQ]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = bernoulliarray[p, m];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[BernoulliDistribution]
    		
(* ========================================================================= *)
(* Binomial Distribution, K & J, DD, Chap. 3 *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

binomial = Compile[{{n, _Real}, {p, _Real}},
    Apply[Plus, Table[If[Random[] > p, 0, 1], {n}]]	]

(* Next few routines written by Darren Glosemeyer and Daniel Lichtblau
for faster generation of arrays of Binomial randomly distributed variables.
2003-09 DARRENG *)

compiledBinom = 
  Compile[{{n, _Integer}, {p, _Real}, {tabmin, _Integer}, 
    {tabmax, _Integer}, {pdf, _Real}, {len, _Integer}},
      Module[{tabl, high, low, mid, res, q, cdf, minval, comppdf},
        minval = Max[0, tabmin - 1];
        If[tabmin > 1,
           cdf = N[BetaRegularized[1. - p, n - tabmin - 1, 2 + tabmin]],
           cdf = 0.
          ];
        comppdf = pdf;
        tabl = Join[Table[cdf = cdf + comppdf; 
              		comppdf = (n - j)/(j + 1)*p/(1 - p)*comppdf; 
              		cdf,
              		{j, tabmin, tabmax}]
              	   , 
              	   Table[1, {tabmax - tabmin}]
              	  ];
        res = Table[q = Random[];
            	If[tabl[[1]] < q, 
            	  high = 1;
              	  While[tabl[[high]] < q, high *= 2]; 
              	  low = Round[high/2];
              	  While[high - low > 1, mid = Round[(high + low)/2];
                  If[tabl[[mid]] < q, low = mid, high = mid]]; 
          	  high + tabmin - 1
          	  , minval], {len}];
        res]]  

Binom[n_Integer, p_?NumericQ, len_Integer] := 
  Module[{tabmin, tabmax, pdf},
    Which[p == 0, Table[0, {len}],
      	  p == 1, Table[n, {len}],
      	  True, 
      	  {tabmin, tabmax} = {Quantile[BinomialDistribution[n, p], 10.^-10], 
      		Quantile[BinomialDistribution[n, p], 1. - 10^-10]}; 
      	  pdf = Binomial[N[n], tabmin]*p^(tabmin)*(1 - p)^(n - tabmin); 
      	  compiledBinom[n, p, tabmin, tabmax, pdf, len]
      	]]

Unprotect[BinomialDistribution]

BinomialDistribution/:
  RandomArray[BinomialDistribution[n_?NumericQ, p_?NumericQ]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = If[(n > 32 && m >= 150) || p == 0 || p == 1
    		, 
    		Binom[n, p, m]
    		, (* ELSE *) 
    		Table[binomial[n, p], {m}]
    		];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[BinomialDistribution]

(* ========================================================================= *)
(* Discrete Uniform Distribution, K & J, DD, Chap. 10, p. 239 *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

Unprotect[DiscreteUniformDistribution]

DiscreteUniformDistribution/: RandomArray[
	DiscreteUniformDistribution[{min_?NumericQ,max_?NumericQ}]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Table[Random[Integer, {min,max}], {m}];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[DiscreteUniformDistribution]

(* ========================================================================= *)
(* Geometric Distribution, K & J, DD, Chap. 5, p. 122 *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* The following code provided by Johannes Ludsteck is many times faster than
the previous code.  - DTERR, 6-02 *)

Unprotect[GeometricDistribution]

GeometricDistribution/:
  RandomArray[GeometricDistribution[p_?NumericQ]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Floor[Log[Table[Random[],{m}]]/Log[1-p]];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

Protect[GeometricDistribution]

(* ========================================================================= *)
(* Hypergeometric Distribution, K & J, DD, Chap. 6 *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* Next few routines written by Darren Glosemeyer for faster generation of 
arrays of Hypergeometric randomly distributed variables. 2003-09 DARRENG *)


compiledHyperGeom = Compile[
  {{n, _Real}, {nsucc, _Real}, {ntot, _Real}, {pdf, _Real}, {cdf, _Real}, 
    {tabmin, _Integer}, {tabmax, _Integer}, {len, _Real}},
      Module[{tabl, high, low, mid, res, q, comppdf, compcdf, minusx},
        comppdf = pdf;
        compcdf = cdf;
	tabl = Join[Table[compcdf = compcdf + comppdf; 
		  minusx=-x;
      	  comppdf = ((n + minusx)*(nsucc + minusx))/((1 + x)*
      	  	(1 - n - nsucc + ntot + x))*comppdf; 
      	  compcdf, {x, tabmin, tabmax}], Table[1, {tabmax - tabmin}]];
        res = Table[q = Random[];
            If[tabl[[1]] < q, high = 1;
              While[tabl[[high]] < q, high *= 2]; low = Round[high/2];
              While[high - low > 1, mid = Round[(high + low)/2];
              If[tabl[[mid]] < q, low = mid, high = mid]]; 
            high + tabmin - 1, tabmin], {len}];
        res]];
        
       
hyperGeom[n_, nsucc_, ntot_, m_] := Module[
  {pdf = Binomial[N[nsucc], Max[0, n - ntot + nsucc]] Binomial[N[ntot - nsucc], 
  	n - Max[0, n - ntot + nsucc]]/Binomial[N[ntot], n], cdf, tabmin, tabmax}, 
     If[MachineNumberQ[pdf],
      	{tabmin, tabmax} = {Max[0, n - ntot + nsucc], Min[n, nsucc]},
      	{tabmin, tabmax} = {Quantile[HypergeometricDistribution[n, nsucc, ntot], 
      		10^-10], Quantile[HypergeometricDistribution[n, nsucc, ntot], 1 - 10^-10]}];
     cdf = 0.; 
     Do[cdf = cdf + pdf; 
     	pdf = ((n - x)*(nsucc - x))/((1 + x)*(1 - n - nsucc + ntot + x))*pdf; 
     	cdf, 
     	{x, Max[0, n - ntot + nsucc], tabmin - 1}];
    compiledHyperGeom[n, nsucc, ntot, pdf, cdf, tabmin, tabmax, m]]

Unprotect[HypergeometricDistribution]    
   
HypergeometricDistribution/:
	RandomArray[HypergeometricDistribution[
          n_?NumericQ, nsucc_?NumericQ, ntot_?NumericQ]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    (* check special cases first *)
    array = Which[
    		n === ntot, (* sample is entire population *)
    		Table[nsucc, {m}]
    		, 
    		n===0 || nsucc===0, (* no chance for success *) 
    		Table[0, {m}]
    		, 
    		True,  (* general case *)
        	hyperGeom[n, nsucc, ntot, m]];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[HypergeometricDistribution]

(* ========================================================================= *)
(* Logarithmic Series Distribution, K & J, DD, Chap. 7 *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

internalRandomLogSeries[theta_] :=
    Quantile[LogSeriesDistribution[theta],Random[]]

(* Next few routines written by Darren Glosemeyer for faster generation of 
arrays of LogSeries randomly distributed variables. 2003-09 DARRENG *)


compiledLogSeries = Compile[
  {{theta, _Real}, {len, _Integer}}, 
    Module[{tabllen,  tabl, cdf, high, low, mid, res, q}, 
    	tabllen = 32;
      	cdf = 0.;
      	tabl = Table[cdf = cdf + theta^x/(-x Log[1 - theta]), {x, tabllen}];
      	res = Table[q = Random[];
          If[N[theta^2/(-2 Log[1 - theta])] < q, high = 1;
            While[tabl[[high]] < q, high *= 2;
              If[high > tabllen, tabllen *= 2;
                tabl = Join[tabl, 
                	Table[cdf = cdf + theta^x/(-x Log[1 - theta]), 
                		{x, tabllen/2 + 1, tabllen}]];]]; 
                low = Round[high/2];
            	While[high - low > 1, mid = Round[(high + low)/2];
             If[tabl[[mid]] < q, low = mid, high = mid]]; high, 1], {len}];
      res]]

logSeries[theta_?NumericQ, len_Integer] := 
  If[theta == 0, Table[1, {len}], 
    compiledLogSeries[theta, len]]

Unprotect[LogSeriesDistribution]

LogSeriesDistribution/:
  RandomArray[LogSeriesDistribution[theta_?NumericQ]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = If[theta == 0 || m > 35, 
    		logSeries[theta, m]
    		, (* ELSE *)
    		Table[internalRandomLogSeries[theta], {m}]
    	];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[LogSeriesDistribution]

(* ========================================================================= *)
(* Negative Binomial Distribution, K & J, DD, Chap. 5 *)


iNegativeBinomialQuantile[n_, p_, q_] := Module[{high, low, mid},
    If[q == 1, Return[Infinity]];
    If[N[p^n] < q,
        high = 1;
        While[N[CDF[NegativeBinomialDistribution[n, p], high]] < q, high *= 2];
        low = high/2;
        While[high - low > 1,
            mid = (high+low)/2;
            If[N[CDF[NegativeBinomialDistribution[n, p], mid]] < q,
                low = mid,
                high = mid,
                low = high
            ]
        ]; high,
    (* else *)
        0,
    (* indeterminate *)
        Fail
    ]
]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* Next few routines written by Darren Glosemeyer for faster generation of 
arrays of NegativeBinomial randomly distributed variables. 2003-09 DARRENG *)


compiledNegBinom = 
  Compile[{{r, _Real}, {p, _Real}, {tabmin, _Integer}, 
    {tabmax, _Integer}, {pdf, _Real}, {len, _Integer}},
      Module[{tabl, high, low, mid, res, q, cdf, minval, comppdf},
        minval = Max[0, tabmin - 1];
        If[tabmin > 1,
          cdf = N[BetaRegularized[p, r, 2 + tabmin]],
          cdf = 0.
          ];
        comppdf = pdf;
        tabl = Join[Table[cdf = cdf + comppdf; 
        	   comppdf = ((j + r)*(1 - p))/(1 + j)*comppdf; cdf,
              	   {j, tabmin, tabmax}]
              	   , 
              	   Table[1, {tabmax - tabmin}]];
        res = Table[q = Random[];
            If[tabl[[1]] < q, high = 1;
              While[tabl[[high]] < q, high *= 2]; low = Round[high/2];
              While[high - low > 1, mid = Round[(high + low)/2];
                If[tabl[[mid]] <q, low = mid, high = mid]]; high + tabmin - 1, 
                  minval], {len}];
        res]]

NegBinom[r_, p_, len_] := 
   Module[{tabmin, tabmax, pdf},
     If[p == 1, 
        Table[0, {len}]
        , (* ELSE*)
       	{tabmin, tabmax} = {Quantile[NegativeBinomialDistribution[r, p], 10.^-10], 
       		Quantile[NegativeBinomialDistribution[r, p], 1. - 10^-10]};
       	pdf = Binomial[N[tabmin + r - 1], r - 1]*p^r*(1 - p)^(tabmin); 
       	compiledNegBinom[r, p, tabmin, tabmax, pdf, len]]]

Unprotect[NegativeBinomialDistribution]

NegativeBinomialDistribution/: RandomArray[
	NegativeBinomialDistribution[r_?NumericQ, p_?NumericQ]?DistributionParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = If[!IntegerQ[r]||(r > 32 && m >= 50) || (r > 5 && p < r*m/10000) || p == 1
    		, 
    		NegBinom[r, p, m]
    		, (* ELSE *) 
    		Table[negativebinomial[r, p], {m}]];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[NegativeBinomialDistribution]

(* ========================================================================= *)
(* Poisson Distribution, K & J, DD, Chap. 4 *)


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

internalRandomPoisson[mu_] :=
    Quantile[PoissonDistribution[mu],Random[]]


(* Next few routines written by Darren Glosemeyer and Daniel Lichtblau
for faster generation of arrays of Poisson randomly distributed variables.
2003-08 DANL *)

compiledPoissonArray = 
    Compile[{{mu, _Real}, {tabmin, _Integer}, {tabmax, _Integer}, {len, _Integer, 1}}, 
      Module[ {tabl, high, low, mid, res, q, allrand, tabllen, minval}, 
      	allrand = Apply[Times, len]; 
        If[mu == 0, Return[Table[0, {allrand}]]]; 
        minval = Max[0, tabmin - 1];
        tabllen=tabmax-tabmin;
        tabl = Join[Table[GammaRegularized[1. + j, mu], {j, tabmin, tabmax}],Table[1,{tabllen}]]; 
        res = Table[ q = Random[]; 
            If [Exp[-mu] < q, high = 1; 
              While[tabl[[high]] < q, high *= 2]; 
              low = Round[high/2]; 
              While[high - low > 1, mid = Round[(high + low)/2]; 
                If[tabl[[mid]] < q, low = mid, high = mid ] ]; high+ tabmin - 1, 
              minval ], {allrand}];
	res ]];


uncompiledPoissonArray[mu_, tabmin_, tabmax_, {len__Integer}] := 
  Module[ {tabl, high, low, mid, res, q, allrand, tabllen, minval}, 
    allrand = Apply[Times, len]; 
        If[mu == 0, Return[Table[0, {allrand}]]]; 
        minval = Max[0, tabmin - 1];
        tabllen=tabmax-tabmin;
        tabl = Join[Table[GammaRegularized[1. + j, mu], {j, tabmin, tabmax}],Table[1,{tabllen}]]; 
        res = Table[ q = Random[]; 
            If [Exp[-mu] < q, high = 1; 
              While[tabl[[high]] < q, high *= 2]; 
              low = Round[high/2]; 
              While[high - low > 1, mid = Round[(high + low)/2]; 
                If[tabl[[mid]] < q, low = mid, high = mid ] ]; high+ tabmin - 1, 
              minval ], {allrand}];
	res ] 


PoissonArray[mu_, tabmin_, tabmax_, len_] := 
  If[MachineNumberQ[Exp[N[mu]]], 
      compiledPoissonArray[mu, tabmin, tabmax, {len}], 
      uncompiledPoissonArray[mu, tabmin, tabmax, {len}]]

Unprotect[PoissonDistribution]

PoissonDistribution /:
  RandomArray[PoissonDistribution[mu_?NumericQ]?DistributionParameterQ, dim_] :=
	Module[{m, array, tabmin, tabmax},
    m = If [VectorQ[dim], Apply[Times, dim], dim];
    {tabmin, tabmax} = {Quantile[PoissonDistribution[mu], 10.^-10], 
    	Quantile[PoissonDistribution[mu], 1. - 10^-10]};
    If [m<5||(mu>1000&&m<(tabmax-tabmin)/3),
	array = Table[internalRandomPoisson[mu], {m}]
	, (* else *)
	array=PoissonArray[mu, tabmin, tabmax, m]
	];
    If [VectorQ[dim] && Length[dim] > 1,
		Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
  		array
		]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[PoissonDistribution]

(* ========================================================================= *)
(* BetaBinomialDistribution *)

Unprotect[BetaBinomialDistribution]		
	
BetaBinomialDistribution /:RandomArray[
  BetaBinomialDistribution[alpha_?NumericQ, beta_?NumericQ, nn_?NumericQ]?DistributionParameterQ, dim_] :=
	RandomInteger[BetaBinomialDistribution[alpha, beta, nn],dim] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[BetaBinomialDistribution]	

(* ========================================================================= *)
(* BetaNegativeBinomialDistribution *)

Unprotect[BetaNegativeBinomialDistribution]	
	
BetaNegativeBinomialDistribution /:RandomArray[
  BetaNegativeBinomialDistribution[alpha_?NumericQ, beta_?NumericQ, nn_?NumericQ]?DistributionParameterQ, dim_] :=
	RandomInteger[BetaNegativeBinomialDistribution[alpha, beta, nn],dim] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[BetaNegativeBinomialDistribution]	  		

(* ========================================================================= *)
(* ZipfDistribution Johnson, Kotz, and Kemp, Chapter 11, section 20 *)

Unprotect[ZipfDistribution]	
  
ZipfDistribution /: 
 RandomArray[ZipfDistribution[rho_?(NumberQ[N[#]]&)]?DistributionParameterQ, dim_] := 
 RandomInteger[ZipfDistribution[rho],dim]/; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

Protect[ZipfDistribution]	



End[]

EndPackage[]

