(*

   IF YOU WANT TO CUSTOMIZE YOUR SECURITY SYSTEM DO *NOT* MODIFY 
   THIS FILE. THE WAY TO CUSTOMIZE SECURITY IS DESCRIBED IN THE 
   SECURITY SECTION OF THE USER GUIDE.

*)


(* :Name: Security` *)

(* :Title: Mathematica Server Pages Security *)

(* :Author: Tom Wickham-Jones *)

(* :Copyright: 
       webMathematica source code (c) 1999-2003,
       Wolfram Research, Inc. All rights reserved.
*)

(* :Mathematica Version: 4.2 *)

(* :Package Version: 2.0 *)

(* :History:
   Original Version by Tom Wickham-Jones as part of MSP tools.
   Developed January - August 2000.
*)

(*:Summary:
   This package provides security features to be used by 
   Mathematica Web Tools.
   
   
   InsecureExprQ[ expr] returns True if expr contains 'insecure symbols' 
   and False otherwise.
   
   Symbols are tested to see if they are allowed or not. The 
   following tests are performed that look at the names and contexts 
   of the symbols.  They steadily remove symbols from the expr,  
   if any remain then the expr is declared insecure.
   
   
   1) If AllowedContexts is set to a list of contexts then all symbols 
      with contexts on this list are allowed.
      
   2) If AllowedContexts is not a list of contexts then all symbols with 
      contexts not in DisallowedContexts are allowed.
   
   The remaining symbols are then tested as follows:
   
   3) If AllowedSymbols is set to a list of symbols then all symbols 
      that appear in the list are allowed.
      
   4) If AllowedSymbols is not set to a list of symbols then all symbols 
      that appear in DisallowedSymbols are not allowed.
      
   
   If any symbols are found that are disallowed then the expr is deemed to 
   be insecure.
      
   
   Using DisallowedSymbols and DisallowedContexts gives most flexibility 
   but with a higher risk.
   
   Using AllowedSymbols and AllowedContexts gives less flexibity but with 
   lower risk.
   
   See examples at bottom.
   
*)

(* :Context: Security` *)



BeginPackage[ "Security`"]

(* :Exports: *)

InsecureExprQ

SetSecurity

$AllowedContexts;
$DisallowedContexts;
$AllowedSymbols;
$DisallowedSymbols;

ToExpressionSecure

MakeSecurityFunction

FindInsecureSymbols

Begin[ "`Private`"]

ToExpressionSecure::security = "Input expression `1` is not secure."


ToExpressionSecure[ expr_, fmt_:InputForm, head_:Null] :=
	Module[ {ef},
		ef = ToExpression[ expr, fmt, HoldComplete];
		If[ InsecureExprQ[ ef],  
			Message[ ToExpressionSecure::security, expr];$Failed,
			If[ head === Null, ReleaseHold[ ef], head @@ ef]
			]
	]


defaultAllowedContexts = 
    {"Global`"}

defaultAllowedSymbols = 
    HoldComplete[ 
    Plus, Times, Power, Sqrt, Log, Exp,
    HoldComplete,
    Infinity,  Pi, E, Degree, GoldenRatio, Catalan, EulerGamma,
    OutputForm, StandardForm, List, 
    Sin, Cos, Tan, Sec, Csc, Cot,
    Sinh, Cosh, Tanh, Sech, Csch, Coth,
    ArcSin, ArcCos, ArcTan, ArcSec, ArcCsc, ArcCot,
    ArcSinh, ArcCosh, ArcTanh, ArcSech, ArcCsch, ArcCoth,
    True, False, Derivative, D, Dt, I,
    Greater, Less, GreaterEqual, LessEqual, Inequality, Equal,
    Re, Im, Abs, Sign, Conjugate, Arg, 
    Round, Floor, Ceiling, Max, Min, 
    Mod, Quotient,
    Not, And, Or, Xor,Union, Intersection, Complement,
    AiryAi, AiryAiPrime, AiryBi, AiryBiPrime,
    BesselJ, BesselK, BesselI, BesselY,
    Factorial, Binomial, Multinomial, 
    Gamma, Beta, LogGamma, PolyGamma,
    LegendreP, SphericalHarmonicY,
    HermiteH, LaguerreL,
    Erf,  Erfc,  Erfi, InverseErf, InverseErfc,
    ClebschGordan, ThreeJSymbol, SixJSymbol,
    Zeta, FresnelS,
    FresnelC, CosIntegral, SinIntegral, ExpIntegralE, 
    ExpIntegralEi, SinhIntegral, CoshIntegral,
    HypergeometricPFQ, Hypergeometric0F1, Hypergeometric1F1,
    Hypergeometric2F1, HypergeometricPFQRegularized,
    MeijerG,AppellF1,
    EllipticK, EllipticF, EllipticE, EllipticPi,
    JacobiZeta, EllipticNomeQ, EllipticLog,
    InverseEllipticNomeQ, JacobiAmplitude, 
    EllipticExp,
    DiracDelta, UnitStep, DiscreteDelta, KroneckerDelta,
    Identity, Function, Slot,
    GrayLevel, Hue, RGBColor, CMYKColor,
    Automatic, None, All, Null, O, C]


SetAttributes[ SymbolSecureByContext, HoldFirst]

(*
    If AllowedContexts is a list return True if context of symbol is 
    not in list.   Else return True if context of symbol is in 
    DisallowedContexts.
*)

SymbolSecureByContext[ x_Symbol] := 
    If[ ListQ[ $AllowedContexts],
            !MemberQ[ $AllowedContexts, Context[x]],
            MemberQ[ $DisallowedContexts, Context[ x]]]

SymbolSecureByContext[ x_] := False

FindInsecureSymbols[ e_HoldComplete] :=
    Module[ {work},
        work = Level[ e, {-1}, HoldComplete, Heads->True] ;
        If[Head[ $DisallowedSymbols] =!= HoldComplete || 
           Intersection[ work, $DisallowedSymbols] === HoldComplete[], 
          work = Select[ work, SymbolSecureByContext[#, $AllowedContexts, $DisallowedContexts]&];
          If[Head[ $AllowedSymbols] === HoldComplete,
            work = Complement[ work, $AllowedSymbols]
          ],
          work = Intersection[ work, $DisallowedSymbols]
        ];
        work
    ]

InsecureExprQ[ e_HoldComplete] :=
    Module[ {work = FindInsecureSymbols[e]},
        work =!= HoldComplete[] && work =!= HoldComplete[ HoldComplete]
    ]

InsecureExprQ[ a_] := True

LoadSecurityConfiguration[ dir_, file_] :=
    Module[ {arg},
        arg = ToFileName[ dir, file] ;
        If[ FileType[ arg] === File,
            Get[ arg]; 
            True,
            False]
        ]

SecurityOpenQ = True;

SetSecurity[ ] :=
    SetSecurity[ None, None]

SetSecurity[ dir_, file_] :=
    Module[ {found = True},
        If[ SecurityOpenQ,
            If[ StringQ[ dir] && StringQ[ file],
                found = LoadSecurityConfiguration[ dir, file]] ;
            Which[ 
                MatchQ[ $AllowedContexts, {___String}],
                    1,
                MatchQ[ $DisallowedContexts, {___String}],
                    Clear[ $AllowedContexts],
                True,
                    $AllowedContexts = defaultAllowedContexts];

            Which[ 
                MatchQ[ $AllowedSymbols, HoldComplete[___Symbol]],
                    1,
                MatchQ[ $DisallowedSymbols, HoldComplete[___Symbol]],
                    Clear[ $AllowedSymbols],
                True,
                    $AllowedSymbols = defaultAllowedSymbols];

            LockProtectSymbol[ $AllowedContexts];
            LockProtectSymbol[ $DisallowedContexts];
            LockProtectSymbol[ $AllowedSymbols];
            LockProtectSymbol[ $DisallowedSymbols];
            LockProtectSymbol[ InsecureExprQ];
            SecurityOpenQ = False;
            ];
        found
    ]


SetAttributes[ LockProtectSymbol, HoldAllComplete]

LockProtectSymbol[sym_] :=
    (
    Protect[ sym] ;
    SetAttributes[ sym, Locked];
    )


SymbolSecureByContext[ x_Symbol, allowedContexts_, disallowedContexts_] := 
    If[ ListQ[ allowedContexts],
            !MemberQ[ allowedContexts, Context[x]],
            MemberQ[ disallowedContexts, Context[ x]]]

SymbolSecureByContext[ x_, _, _] := False

    
MakeSecurityFunction[ allowedSymbols_, allowedContexts_, disallowedSymbols_, disallowedContexts_] :=
    Module[ {newFun},
        SetAttributes[ newFun, HoldAllComplete];
        newFun[ InsecureExprQ[ e_HoldComplete]] :=
            Module[ {work = FindInsecureSymbols[ e]},                
                work =!= HoldComplete[] && work =!= HoldComplete[ HoldComplete]
            ];
        newFun[ FindInsecureSymbols[ e_HoldComplete]] :=
            Module[ {work},
                work = Level[ e, {-1}, HoldComplete, Heads->True];
                If[Head[ disallowedSymbols] =!= HoldComplete || 
                   Intersection[ work, disallowedSymbols] === HoldComplete[], 
                  work = Select[ work, SymbolSecureByContext[#, allowedContexts, disallowedContexts]&];
                  If[Head[ allowedSymbols] === HoldComplete,
                    work = Complement[ work, allowedSymbols]
                  ],
                  work = Intersection[ work, disallowedSymbols]
                ];
                work
            ];            
        newFun[ ToExpressionSecure[e_, fmt_:InputForm, head_:Null]] :=
			Module[ {ef},
				ef = ToExpression[ e, fmt, HoldComplete];
				With[ {ef = ef},
					If[ newFun[InsecureExprQ[ ef]],  
						$Failed,
						If[ head === Null, ReleaseHold[ ef], head @@ ef]
					]
				]
			];
        LockProtectSymbol[ newFun];
        newFun
    ]

    
End[]

EndPackage[]
