(* ::Package:: *)

BeginPackage["DocumentationSearch`", {"JLink`", "ResourceLocator`"}];

DocumentationNotebookIndexer::usage="DocumentationNotebookIndexer[...] contains the notebook indexer object.";
NewDocumentationNotebookIndexer::usage="NewDocumentationNotebookIndexer[directory] creates a new NotebookIndexer.";
CloseDocumentationNotebookIndexer::usage="CloseDocumentationNotebookIndexer[indexer] closes the notebook indexer object."

CreateSpellIndex::usage="CreateSpellIndex[indexDir, spellIndexDir] takes the words from indexDir and creates a spelling index at spellIndexDir.";

AddDocumentationNotebook::usage="AddDocumentationNotebook[indexer, notebook] adds a notebook to the index.";
AddDocumentationDirectory::usage="AddDocumentationDirectory[indexer, directory] adds a directory of notebooks to the index.";

SearchDocumentation::usage="SearchDocumentation[criteria] searches the documentation and returns results based on the search criteria.";
SearchDocumentationMetaData::usage="SearchDocumentationMetaData[] returns valid meta data that can be returned from a search."

DirectHitSearch::usage="DirectHitSearch[criteria] searches the documentation and returns a URI if there is a one specific hit for the given query.";

ExportSearchResults::usage="ExportSearchResults[results, format] exports the search results to the specified format.";

$NumberOfExtraPages::usage="$NumberOfExtraPages sets the number of pages on either side of the current page in the searh results navigation";
$NumberOfExtraPages = 3;

DocumentationIndexes::usage="DocumentationIndexes[] returns the index directories of documentation.";
DocumentationSpellIndexes::usage="DocumentationSpellIndexes[] returns the spell index directories of documentation.";

`Information`$Version = "DocumentationSearch Version 1.0.0 (January 2007)";

`Information`$VersionNumber = 1.0;

`Information`$ReleaseNumber = 0;

`Information`$CVSVersion = "$Id: DocumentationSearch.m,v 1.62 2007/04/16 18:55:40 andy Exp $"

Begin["`Private`"];

ExportSearchResults::format="`1` is not a recognized ExportSearchResults format.";
$SearchLanguage = If[$Language === "Japanese", "Japanese", "English"];

DocumentationIndexes[] := 
  First /@ ResourcesLocate[ToFileName[{"Documentation", ToString[$SearchLanguage]}, "Index"]]

DocumentationSpellIndexes[] := 
  First /@ ResourcesLocate[ToFileName[{"Documentation", "English"}, "SpellIndex"]]


If[!MemberQ[DocumentationIndexes[], 
     ToFileName[{$InstallationDirectory, "Documentation", ToString[$SearchLanguage]}, "Index"]], 
  ResourceAdd[ToFileName[{$InstallationDirectory, "Documentation", ToString[$SearchLanguage]}, "Index"], ToFileName[{"Documentation", ToString[$SearchLanguage]}, "Index"]]
];

If[!MemberQ[DocumentationSpellIndexes[], 
     ToFileName[{$InstallationDirectory, "Documentation", "English"}, "SpellIndex"]], 
  ResourceAdd[ToFileName[{$InstallationDirectory, "Documentation", "English"}, "SpellIndex"], ToFileName[{"Documentation", "English"}, "SpellIndex"]]
];

Options[NewDocumentationNotebookIndexer] := {
  "Language"->$SearchLanguage
};

NewDocumentationNotebookIndexer[directory_String, opts___Rule] :=
  JavaBlock[
    Module[{indexer, useOpts, lang},
    
      useOpts  = canonicalOptions[Flatten[{opts}]];
      lang = "Language" /. useOpts /. Options[ NewDocumentationNotebookIndexer ];
      
      InstallJava[];
      Switch[lang, 
			"Japanese", 
				indexer = JavaNew["com.wolfram.documentationsearch.JapaneseDocumentationIndexer", directory],
      		_, 
				indexer = JavaNew["com.wolfram.documentationsearch.DocumentationIndexer", directory]];
      KeepJavaObject[indexer];
      DocumentationNotebookIndexer[indexer]
    ]
  ]
  
CloseDocumentationNotebookIndexer[DocumentationNotebookIndexer[indexer_?JavaObjectQ]] := 
  indexer@close[];

AddDocumentationNotebook[jo:DocumentationNotebookIndexer[indexer_?JavaObjectQ], notebook_String /; FileType[notebook] === File] := 
  JavaBlock[
    Module[{nb = quietGet[notebook], text, doc, taggingRules, metaData},
      doc = JavaNew["com.wolfram.documentationsearch.DocumentationNotebook"];
      (* plain text of notebook *)
      Developer`UseFrontEnd[text = Import[notebook, "Plaintext"]];
      (* gather meta data *)
      taggingRules = TaggingRules /. Options[nb] /. {TaggingRules -> {}};
      metaData = "Metadata" /. taggingRules /. {"Metadata" -> {}};      
      (* add notebook to index *)
      AddDocumentationNotebook[jo, text, metaData]
]];

AddDocumentationNotebook[DocumentationNotebookIndexer[indexer_?JavaObjectQ], text_String, metaData_List] := 
  JavaBlock[
    Module[{ doc, type, context, keywords, name, summary, title, uri, synonyms, status, label},
      
      doc = JavaNew["com.wolfram.documentationsearch.DocumentationNotebook"];

      {type, context, keywords, name, label, summary, title, uri, synonyms, status} = 
        {"type", "context", "keywords", "paclet", "label", "summary", "title", "uri", "synonyms", "status"} 
          /. metaData /. 
            {"type" -> "", "context" -> "", "keywords" -> {}, "paclet" -> "", "label"->"",
             "summary" -> "", "title" -> "", "uri"->"", "synonyms" -> {}, "status" -> "None"};

      doc@setType[type];
      doc@setContext[context];
      doc@setKeywords[keywords];
      doc@setSynonyms[synonyms];
      doc@setPacletName[name];
      doc@setTitle[title];
      doc@setLabel[label];
      doc@setSummary[summary];
      doc@setContent[ text ];
      doc@setURI[uri];
      doc@setStatus[status];
      doc@setLanguage["en"];
      
      indexer@addNotebook[doc];      
    ]
  ]

AddDocumentationDirectory[DocumentationNotebookIndexer[indexer_?JavaObjectQ], directory_String] := 
  Module[{files},
    If[FileType[directory] =!= Directory, 
      Return[];
    ];
    Block[{$rootDirectory = $rootDirectory},     
      If[$rootDirectory === Null, 
        $rootDirectory = directory;
      ];
      
      files = FileNames[ToFileName[directory, "*"]];
      (
        Switch[FileType[#], 
          Directory, 
            Which[
               StringMatchQ[#, "*ExampleData" | "*Examples" | "*RawGuides" | "*Messages" | "*Compatibility"],
                  Print["Skipping ", #];
                  Null, 
                True, 
                  AddDocumentationDirectory[DocumentationNotebookIndexer[indexer], #]
            ],          
          File, 
            AddDocumentationNotebook[DocumentationNotebookIndexer[indexer], #] 
        ]
      ) & /@ files;
    ];
  ]; 
  
CreateSpellIndex[indexDir_String, spellIndexDir_String] := 
  JavaBlock[
    Module[{},
      InstallJava[];
      LoadJavaClass["com.wolfram.documentationsearch.spelling.DidYouMeanIndexer"];
      DidYouMeanIndexer`createSpellIndex["text", indexDir, spellIndexDir];
    ]
  ]

quietGet[name_String] :=
  Module[{expr},
    Off[Syntax::"newl"];
    expr = Get[name];
    On[Syntax::"newl"];
    expr
  ]; 
    
DirectHitSearch[criteria_String] := 
  DirectHitSearch[DocumentationIndexes[], criteria]

DirectHitSearch[indexDir_String criteria_String] := 
  DirectHitSearch[{indexDir}, criteria]

DirectHitSearch[indexDir:{__String}, criteria_String] := 
  Module[{startCriteria, limitCriteria, results, matches, hit = Null, match, title, type, uri, englishWord},
  
    startCriteria = StringCases[criteria, RegularExpression[" start:(\\d+)"] -> "$1"];
    If[Length[startCriteria] > 0, Return[Null]];
    
    limitCriteria = StringCases[criteria, RegularExpression[" limit:(\\d+)"] -> "$1"];
    If[Length[limitCriteria] > 0, Return[Null]];    
    
    results = 
      SearchDocumentation[indexDir, "(exacttitle:\"" <> criteria
        <> "\" OR exactsynonyms:\"" <> criteria <> "\") AND type:Symbol", 
        "Limit"->2, 
        "MetaData"->{"Title", "Type", "URI"}];
    {matches} = {"Matches"} /. results  /. {"Matches" -> {}};
    (* If there are more than one match or less than one match, Null is returned *)
    If[Length[matches] == 1,
      match = First[matches];      
      {title, type, uri} = match;
      
      englishWord = DictionaryLookup[ToLowerCase[criteria]];
      If[Length[englishWord] > 0 && criteria =!= title, 
        Return[Null]
      ];
      
      hit = {title, uri}; 
    ];
    hit
  ]

Options[SearchDocumentation] := {
  "MetaData" -> {"Title", "Type", "ShortenedSummary", "URI", "Description", "Context"},
  "Start" -> 1, 
  "Limit" -> 10
};
  
SearchDocumentation[criteria_String, opts___Rule] := 
  SearchDocumentation[DocumentationIndexes[], 
    DocumentationSpellIndexes[], criteria, opts]

SearchDocumentation[indexDir_String, criteria_String, opts___Rule] := 
  SearchDocumentation[{indexDir}, DocumentationSpellIndexes[], criteria, opts]

SearchDocumentation[indexDir:{__String}, criteria_String, opts___Rule] := 
  SearchDocumentation[indexDir, DocumentationSpellIndexes[], criteria, opts]

SearchDocumentation[indexDir_String, spellIndexDir_String, criteria_String, opts___Rule] := 
  SearchDocumentation[{indexDir}, {spellIndexDir}, criteria, opts]

SearchDocumentation[indexDir:{__String}, spellIndexDir:{___String}, criteria_String, opts___Rule] := 
  Module[{useOpts, metaData, start, limit, result, newCriteria = criteria},
    useOpts  = canonicalOptions[Flatten[{opts}]];
    metaData = "MetaData" /. useOpts /. Options[ SearchDocumentation ];
    start    = "Start" /. useOpts /. Options[ SearchDocumentation ];
    limit    = "Limit" /. useOpts /. Options[ SearchDocumentation ];
    
    (* Process start in query *)
    startCriteria = StringCases[criteria, RegularExpression[" start:(\\d+)"] -> "$1"];
    If[Length[startCriteria] > 0, start = ToExpression[First[startCriteria]]];
    newCriteria = StringReplace[criteria, RegularExpression[" start:\\d+"] -> ""];
    
    (* Process limit in query *)
    limitCriteria = StringCases[newCriteria, RegularExpression[" limit:(\\d+)"] -> "$1"];
    If[Length[limitCriteria] > 0, limit = ToExpression[First[limitCriteria]]];
    newCriteria = StringReplace[newCriteria, RegularExpression[" limit:\\d+"] -> ""];

    JavaBlock[        
        InstallJava[];
        Switch[$SearchLanguage, 
				"Japanese", 
					LoadJavaClass["com.wolfram.documentationsearch.JapaneseDocumentationSearcher"];
        			result = JapaneseDocumentationSearcher`search[indexDir, newCriteria, start, limit];,
        		_, 
					LoadJavaClass["com.wolfram.documentationsearch.DocumentationSearcher"];
        			result = DocumentationSearcher`search[indexDir, spellIndexDir, newCriteria, start, limit]];

        {"Query"->result@getQuery[],
         "ParsedQuery"->result@getParsedQuery[],
         "Start"->result@getStart[], 
         "Limit"->result@getLimit[],
         "SearchTime"->result@getSearchTime[], 
         "TotalMatches"->result@getTotalMatches[],  
         "Suggestions"->result@getSuggestion[],       
         "Matches"->result@getMatchValues[metaData]}
    ]
  ];

SearchDocumentationMetaData[] := {
  "Title", "Summary", "URI", "URL", "Type", "Score", "Explanation", "Synonyms", "Boost",  
  "Keywords", "Context", "PacletName", "ShortenedSummary", "Language", "Description"
};

ExportSearchResults[results_List, "Notebook"] := 
  notebookSearchResult[results]

ExportSearchResults[results_List, format_] :=
  Message[ExportSearchResults::format, format] 


(* string to boxes *)
ItalizeWordsInString[str_String]:=
Module[{res},
  res = StringReplace[str, RegularExpression["(J/Link|\\w+)"] :> italizeWord@"$1"];
  Which[
    Head@res === StringExpression, 
      Flatten@{ List@@res },
    Head@res === String,
      res,
    True,
      $Failed
  ]
];

italizeWord[w_String, o___?OptionQ] :=
Module[{lst},
  wordList = wordList /. {o} /. Options[italizeWord];
  If[! FreeQ[wordList, w],
    StyleBox[w, FontSlant -> "Italic"], w]
];
Options[italizeWord] = {
  wordList -> {"Mathematica", "J/Link", "MathLink", "DatabaseLink", "NETLink", "MathLM", "MonitorLM", "Combinatorica"}
  };


(* Return notebook expr containing search results *)
notebookSearchResult[s_List]:= 
Module[{header, cells, nbExpr, query, start, limit, suggestions, 
        searchTime, totalMatches, matches, resultInfo={}, resultSearchCountCell=" ",
        suggestionCell=" ", allResults },
  query        = "Query"        /. s /. {"Query"->""};
  start        = "Start"        /. s /. {"Start"->1};
  limit        = "Limit"        /. s /. {"Limit"->"10"};
  searchTime   = "SearchTime"   /. s /. {"SearchTime"->0};
  totalMatches = "TotalMatches" /. s /. {"TotalMatches"->"Unknown"};
  suggestions  = "Suggestions"  /. s /. {"Suggestions"->Null};
  matches      = "Matches"      /. s /. {"Matches"->{}};
  
  (* allResults *)
  allResults = urlImport["http://search.wolfram.com/lucene/numberofhits.jsp?query="<>URLEscape[query, "UTF-8"]<>"&collection=tryonall"];
  If[StringQ[allResults], 
    allResults = Cell[TextData[{
            magnifyingGlass[], " ",
            ButtonBox[ StringJoin[localization["Try your search"], " ", localization["on all Wolfram sites"]], 
              BaseStyle->"Hyperlink", 
              ButtonData->{URL["http://search.wolfram.com/?query="<>URLEscape[query, "UTF-8"]<>"&collection=all"], None}],
            " (", 
            ButtonBox[allResults <> " " <> localization["matches"], 
              BaseStyle->"Hyperlink", 
              ButtonData->{URL["http://search.wolfram.com/?query="<>URLEscape[query, "UTF-8"]<>"&collection=all"], None}], 
            ")"
        }], "SearchAllSites"],
     allResults = {}
  ];
  
  Which[
    (* results found *)
    Length[matches] > 0, 
      resultSearchCountCell = 
        Cell[TextData[{
          ToString[start],
          " - ", 
          ToString[start + Length[matches] - 1],  
          " ", localization["of"], " ", 
          ToString[totalMatches], 
          " ", localization["for"], " ", 
          ButtonBox[ToString[query], 
             ButtonStyle -> "Link",
             ButtonData->query]
        }], "SearchCountCell"];
        suggestionCell = 
        If[suggestions =!= Null, 
          Cell[TextData[{
            localization["Did you mean"], ": ", 
            ButtonBox[suggestions, 
             ButtonStyle -> "Link",
             ButtonData->suggestions]
          }], "DidYouMean"], 
          " "
        ];
     ,
    (* no results found *)
    True, 
        resultInfo = {
          allResults,
          If[suggestions =!= Null, 
            Cell[TextData[{
              localization["Did you mean"], ": ", 
              ButtonBox[suggestions, 
                ButtonStyle -> "Link",
                ButtonData->suggestions]
            }], "DidYouMean"], 
            {}
          ],
          Cell[TextData[{
            warningSign[],
            localization[" Your search"], " - ", query, " - in ", 
            ButtonBox["Documentation Center", 
              BaseStyle->"Hyperlink",
              ButtonData->{URL["http://reference.wolfram.com"], None}
              ],
            localization[" did not match any documents"]
          }], "SearchCountCell", FontWeight->"Bold"],
          Cell[TextData[{
            localization["Suggestions"], "\n", 
            localization["\[FilledSmallSquare] Make sure all words are spelled correctly"],"\n", 
            localization["\[FilledSmallSquare] Try different keywords"], "\n", 
            localization["\[FilledSmallSquare] Try more general keywords"]        
          }], "SearchSuggestionsCell"]
        }
  ];
  cells = cellSearchResult/@matches;
  header = 
    Cell[BoxData[GridBox[{
      {
        Cell[TextData[{localization["Search Results"]}], "SearchPageHeading"],
        resultSearchCountCell
      },
      {
        suggestionCell,
        If[(allResults === {} || Length[matches] < 1), " ", allResults]
      }
      }]], "SearchPageHeadingGrid"];
  nbExpr = 
    Notebook[Flatten @ {
      header,
      resultInfo,
      cells,
      searchPageLinkCell[query, totalMatches, start, limit, $NumberOfExtraPages]
    }, 
    StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb"],
    Saveable->False, 
    WindowTitle->"Search Results: " <> StringTake[query, Min[StringLength[query], 60]]];
  nbExpr
];

notebookSearchResult[f___]:= ($Failed; Message[notebookSearchResult::args, f];)
notebookSearchResult::args = "Incorrect arguments: `1`";   

(* Return a cell expr from a list : *)
(* {"Title", "Type", "Summary", "URI"} *)
cellSearchResult[{title_String, type_String, summary_String, uri_String, description_String, context_String}]:= 
Module[{styledTitle, url=uri},
styledTitle = ItalizeWordsInString @ title;
url = If[StringMatchQ[url, "note/*"], 
         dothtml @ StringJoin["http://reference.wolfram.com/mathematica/", url, ".html" ],
         "paclet:" <> url];

Cell[TextData[Flatten@{
 Cell[TextData[ButtonBox[If[type === "Character Name", StringReplace[styledTitle, {"\\"->"\\[Backslash]"}], styledTitle], 
  BaseStyle->"Link",
  ButtonData->url]], "SearchResultTitle"],
 " ", StyleBox["(", "SearchResultType"],
 StyleBox[ ItalizeWordsInString @ description , "SearchResultType"],
 StyleBox[")", "SearchResultType"],
 If[StringLength[summary] > 0, 
   "\n",
   ""
 ],
 StyleBox[ ItalizeWordsInString @ summary, "SearchResultSummary"]
}], "SearchResultCell", 
  CellDingbat-> StyleBox["\[FilledSquare]",
    Which[
      type === "Symbol", "SymbolColor",
      type === "Guide", "GuideColor", 
      True, "OtherColor"] ] ] ];

cellSearchResult[f___]:= ($Failed; Message[cellSearchResult::args, f];)
cellSearchResult::args = "Incorrect arguments: `1`";   

searchPageLinkCell[query_String, totalResults_Integer, start_Integer, 
   limit_Integer, numPages_Integer] := 
  Module[{startPage, out = {}, currentPage = Ceiling[(start)/limit], 
    totalPages = Ceiling[totalResults/limit], prevnext={} },
   
   (*Check boundaries*)
   If[start > totalResults || totalResults < 0 || limit < 1 || numPages < 1 || totalResults <= limit, 
     Return[{}]
   ];
   
   (*determine starting page number*)
   If[Quotient[currentPage-1, numPages] === 0,    
     startPage = 1, 
     startPage = currentPage - numPages
   ];
   
   (*create range of pages*)
   For[i = startPage, i < currentPage + numPages + 1 && i <= totalPages, i++,
    If[i == currentPage,
      AppendTo[out, StyleBox[" "<>ToString[i]<>" ", FontColor -> RGBColor[0, 0, 1], FontWeight->"Bold"]];, 
      AppendTo[out, 
        ButtonBox[" "<>ToString[i]<>" ", BaseStyle -> "Link", 
         ButtonData -> query <> " start:" <> ToString[(i - 1)*limit + 1] <> " limit:" <> ToString[limit]]];
      ];
    ];
    
   (*add separator*)
   out = Riffle[out, "|"];
   
   (* Add first and last page *)
   If[startPage =!= 1, 
    out = Flatten@{ButtonBox["1", BaseStyle -> "Link", 
        ButtonData -> query <> " start:1 limit:" <> ToString[limit]], " ... ", out}];
   If[currentPage + numPages + 1 <= totalPages, 
    out = Flatten@{out, " ... ", 
       ButtonBox[ToString[totalPages], BaseStyle -> "Link", 
        ButtonData -> query <> " start:" <> ToString[(totalPages-1)*limit + 1] <> " limit:" <> ToString[limit]]}];
   
   (*add prev and next *)
   If[currentPage =!= 1, 
    prevnext = Flatten@{ButtonBox[localization["\[LeftGuillemet] PREVIOUS"], BaseStyle -> "Link", 
        ButtonData -> query <> " start:" <> ToString[(currentPage - 2) * limit + 1] <> " limit:" <> ToString[limit]], prevnext}];
   If[currentPage < totalPages, 
    prevnext = Flatten@{prevnext,  
       ButtonBox[localization["NEXT \[RightGuillemet]"], BaseStyle -> "Link", 
        ButtonData -> query <> " start:" <> ToString[currentPage * limit + 1] <> " limit:" <> ToString[limit]]}];
   (*add separator*)
   prevnext = If[Length@prevnext > 1, Riffle[prevnext, " | "], prevnext];
   
   
   (*return cell expr*)
    Cell[BoxData[GridBox[{
      {
        Cell[TextData[Flatten@{prevnext}], "SearchResultPageLinks"],
        Cell[TextData[Flatten@{out}], "SearchResultPageLinks"]
      }
      }]], "SearchResultPageLinksGrid"]
 ];
 
(* html anchors need to be file.html#xxx, this handles old bug where they could be file#xxx.html *)
dothtml[s_String] :=
  If[StringMatchQ[s, "*#*"],
    (StringTake[s, #] <> ".html" <> StringDrop[s, #])&[StringPosition[s, "#"][[-1, -1]] - 1],
    s];


(* The reason we don't just call Import for this is to set a low socket timeout. 
   Using the PacletManager guarantees it will fail gracefully and not hang *)
urlImport[url_String] := 
 Quiet[(*No reason not to be absolutely quiet in this function*)
  JavaBlock[
    Module[{outfile, result, uf, pm, strm},
      pm = PacletManager`Package`getPacletManager[];
      outfile = Close[OpenTemporary[]];
      uf = pm@createURLFetcher[url, outfile, False];
      uf@start[];
      uf@waitFor[1500];
      If[!uf@isFinished[], 
        uf@cancel[];
        ,
        If[uf@getException[] === Null, 
          strm = OpenRead[outfile];
          result = Read[strm, String];
          Close[strm];          
        ]
      ];
      DeleteFile[outfile];
      If[StringQ[result],
        result,
        (*else*)
        $Failed
        ]
      ]
    ]
  ] /; TrueQ[PacletManager`$AllowInternet]

SetAttributes[ canonicalOptions, {Listable}];
canonicalOptions[name_Symbol -> val_] := SymbolName[name] -> val;
canonicalOptions[expr___] := expr;

URLEscape[str_String, encoding_String] := 
  (
    LoadJavaClass["java.net.URLEncoder"];
    URLEncoder`encode[str, encoding]
  )


(* Localization of strings *)
localization[c___]:= c;

localization["Search Results"]:= localization["Search Results", ToString[$SearchLanguage] ]
localization["Search Results", language_String] :=
  Switch[language, 
    "Japanese", "\:691c\:7d22\:7d50\:679c",
     _, "Search Results" ];
localization["Try your search"]:= localization["Try your search", ToString[$SearchLanguage] ]
localization["Try your search", language_String] :=
  Switch[language, 
    "Japanese", "\:691c\:7d22\:5bfe\:8c61\:ff1a",
     _, "Try your search" ];
localization["on all Wolfram sites"]:= localization["on all Wolfram sites", ToString[$SearchLanguage] ]
localization["on all Wolfram sites", language_String] :=
  Switch[language, 
    "Japanese", "\:3059\:3079\:3066\:306eWolfram\:30b5\:30a4\:30c8",
     _, "on all Wolfram sites" ];
localization["matches"]:= localization["matches", ToString[$SearchLanguage] ]
localization["matches", language_String] :=
  Switch[language, 
    "Japanese", "\:4ef6",
     _, "matches" ];
localization["Results"]:= localization["Results", ToString[$SearchLanguage] ]
localization["Results", language_String] :=
  Switch[language, 
    "Japanese", "\:7d50\:679c",
     _, "Results" ];
localization["of"]:= localization["of", ToString[$SearchLanguage] ]
localization["of", language_String] :=
  Switch[language, 
    "Japanese", "/",
     _, "of" ];
localization["for"]:= localization["for", ToString[$SearchLanguage] ]
localization["for", language_String] :=
  Switch[language, 
    "Japanese", "\:691c\:7d22\:ff1a",
     _, "for" ];
localization["Did you mean"]:= localization["Did you mean", ToString[$SearchLanguage] ]
localization["Did you mean", language_String] :=
  Switch[language, 
    "Japanese", "\:3053\:306e\:5358\:8a9e\:3067\:3059\:304b\:ff1a",
     _, "Did you mean" ];
localization["Your search"]:= localization["Your search", ToString[$SearchLanguage] ]
localization["Your search", language_String] :=
  Switch[language, 
    "Japanese", "\:691c\:7d22\:3055\:308c\:305f\:5358\:8a9e",
     _, "Your search" ];
localization["did not match any documents"]:= localization["did not match any documents", ToString[$SearchLanguage] ]
localization["did not match any documents", language_String] :=
  Switch[language, 
    "Japanese", "\:30de\:30c3\:30c1\:3057\:307e\:305b\:3093\:3067\:3057\:305f\:ff0e",
     _, "did not match any documents" ];
localization["Suggestions"]:= localization["Suggestions", ToString[$SearchLanguage] ]
localization["Suggestions", language_String] :=
  Switch[language, 
    "Japanese", "\:63d0\:6848",
     _, "SUGGESTIONS" ];
localization["Make sure all words are spelled correctly"]:= localization["Make sure all words are spelled correctly", ToString[$SearchLanguage] ]
localization["Make sure all words are spelled correctly", language_String] :=
  Switch[language, 
    "Japanese", "\:30b9\:30da\:30eb\:306e\:9593\:9055\:3044\:306f\:3042\:308a\:307e\:305b\:3093\:304b\:ff1f",
     _, "Make sure all words are spelled correctly" ];
localization["Try different keywords"]:= localization["Try different keywords", ToString[$SearchLanguage] ]
localization["Try different keywords", language_String] :=
  Switch[language, 
    "Japanese", "\:4ed6\:306e\:30ad\:30fc\:30ef\:30fc\:30c9\:3092\:304a\:8a66\:3057\:304f\:3060\:3055\:3044\:ff0e",
     _, "Try different keywords" ];
localization["Try more general keywords"]:= localization["Try more general keywords", ToString[$SearchLanguage] ]
localization["Try more general keywords", language_String] :=
  Switch[language, 
    "Japanese", "\:4e00\:822c\:7684\:306a\:30ad\:30fc\:30ef\:30fc\:30c9\:3092\:304a\:8a66\:3057\:304f\:3060\:3055\:3044\:ff0e",
     _, "Try more general keywords" ];
localization["NEXT \[RightGuillemet]"]:= localization["NEXT \[RightGuillemet]", ToString[$SearchLanguage] ]
localization["NEXT \[RightGuillemet]", language_String] :=
  Switch[language, 
    "Japanese", "\:6b21\:3078",
     _, "NEXT \[RightGuillemet]" ];
localization["\[LeftGuillemet] PREVIOUS"]:= localization["\[LeftGuillemet] PREVIOUS", ToString[$SearchLanguage] ]
localization["\[LeftGuillemet] PREVIOUS", language_String] :=
  Switch[language, 
    "Japanese", "\:623b\:308b",
     _, "\[LeftGuillemet] PREVIOUS" ];


magnifyingGlass[]:=
Cell[GraphicsData["CompressedBitmap", "\<\
eJxVUk1vEkEY3rWlB436D8gSk8af0JignjQk8AdIOJDGpE1qlLaQoMA/4KZI
Qhq0bWI0XEwP6k2Kl2Js+IZtwZYFdoMW6IciSWUf3xm2tJ1kJrMz+7zPxzsO
9/Lco8fu5flZt/Rw0f10bn52SXrwZJGOJkRBEG7SvCEJbA/aGsskaAgTtAxh
DP4FYRK93ilarQ5kuYVarS2i3f4jCFeMW13/h/HQMQKZ0O2cQmkdoCQryOWq
CARCgohwOExsU8bfIypdP0OfjmpOodsdcEYGzmZrCISCMJvNIiySmfCmMY6x
kBpoWo/katoJgRXlAJWKgkymimCQA2Gx3BJ5bfLOiAyGI+zsqCiX6yiVftBU
UKCZz+8hl6mNwZJkZmLPCzBWBmbA7e0KXq++gd1ux/T0bdy9f4+5ZH4JLGFm
5g53SjGdFbhGkolZ1jg4GHwOl8uFVy8jeP/uLYKBZ8QocWaHwyFMjlGadsT1
JRIJ+LxLKBaL+PY9TdV1bG1tIRaLkVEL4vH4RZT6G/nSHnw+H76mkmg0GtTJ
XeSzORKlQ1VV7tRqtY5R1zkXs7ewsIDDbo/yOkSzqaC6s4tCoUD7JlZWVjjf
qBm68SJYQKwlstxANBqloFqcp9/v41f7J53LSKfTiEQisNls/M6ocB4QS1hV
j/Hx8ydsbn7BcDg0igxQr9dJWhlOpxNra2sjWvEi/ipPnHepeczjSiaT2N9X
MBj0kUql4PF4EAqFTGPC0eMFOp2/pP4EGxsfSOEL+P1+eL1erK+vX/Z5iVD8
Dwm2cTA=\
\>"], "Graphics",
 ImageSize->{15, 14},
 ImageMargins->3];

warningSign[]:=
Cell[GraphicsData["CompressedBitmap", "\<\
eJxVUstqIkEU7ba1e2ZgZn7Br5lZzReoCwkDCTiomex8g67cuApuRHcSCIK4
9IGIiogvFHwgMS4ERQJBRI2a8UzdshVT0HVvVZ9zz7lV9ct8d/37j/nu5sqs
/3lrtl3fXP3V/7Desi1JFAThG/u+6wXKwdLTJGG322GxWODl5YWtZrOZoGA6
nVJ2/ubzOV5fX7FerwUR7+/vjPsVy+WSIweDAVqtFprNJhqNxjnW63W+3+v1
MB6PqQJj7/d7xtah2+0iHA7D6/XC4/HA7XYLGrhcLkGG0+mkjPb4/0AggFwu
J2K73TKyhGg0ilQqhXK5zFbFYpHNpVKJMnXFZhE+n08kacbRIpPJIBKJ4Onp
Cc/PzxqMRiMG1Ol0MJlMfMX3ROTzeRF2u13EarViXBlvb2+wWq2oVCqcz2AU
tWqN4XCIYDCIWq3GOjwcDlyQRRQKBYRCIQIfBRXIsgyj0XiqgXQ6jfv7e04T
wQYn09hsNrDZbORHQ1Bm5JJMqn6/H+12W3PkHcMXVKvVkypHXrLo2GKxGMl9
YH2me4XFYkE2mz2zDAYD+v0+nSR3wLFqi5KqRndMN0R+FEXhSo+Pj7wWnQEf
B6i9HVkK/0M+yA/57HQ6cDgcmEwmJxXgn0rSqiQCUstk6eHhAYlE4lid+ZEu
ofK5m3g8zt8SvVQG06on/OnE+xDpkSeTSXr00rlT8eSGeOJ/ERVL1Q==\
\>"], "Graphics",
 ImageSize->{14, 15},
 ImageMargins->0];


End[];
 
EndPackage[];
