(* :Title: Benchmark *)

(* :Authors: Schoeller Porter *)

(* :Context: Benchmarking` *)

(* :Package Version: 6.0 *)

(* :Copyright: Copyright 1990-2007, Wolfram Research, Inc.  *)

(* :History: 
	Version 5.1 by Schoeller Porter, September 2004.
	Version 5.1.1 by Schoeller Porter, February 2005.
	Version 5.2 by Schoeller Porter, June 2005.
	Version 6.0 by Schoeller Porter, Feb 2006.
*)

(* :Keywords: *)

(* :Source: *)

(* :Warning: *)

(* :Mathematica Version: 6.0 *)

(* :Limitation: *)


BeginPackage["Benchmarking`"];

(* === Usage & Options =================================================== *)

$Benchmarks = {"MathematicaMark6"};
$Benchmarks::usage = "$Benchmarks is list of benchmarks that can be run by \
this package.";

If[!ValueQ[Benchmark::usage], Benchmark::usage = "\!\(\*RowBox[{\"Benchmark\", \"[\", \"]\"}]\) runs the \!\(\*StyleBox[\"MathematicaMark6\", FontSlant -> \"Italic\"]\) benchmark."];
If[!ValueQ[BenchmarkReport::usage], BenchmarkReport::usage = "\!\(\*RowBox[{\"BenchmarkReport\", \"[\", \"]\"}]\) runs the \!\(\*StyleBox[\"MathematicaMark6\", FontSlant -> \"Italic\"]\) benchmark and produces a report in a separate notebook comparing this system to a selection of reference systems. \n\!\(\*RowBox[{\"BenchmarkReport\", \"[\", RowBox[{SubscriptBox[StyleBox[\"system\", \"TI\"], StyleBox[\"1\", \"TR\"]], \",\", SubscriptBox[StyleBox[\"system\", \"TI\"], StyleBox[\"2\", \"TR\"]], \",\", StyleBox[\"\[Ellipsis]\", \"TR\"], \",\", SubscriptBox[StyleBox[\"data\", \"TI\"], StyleBox[\"1\", \"TR\"]], \",\", SubscriptBox[StyleBox[\"data\", \"TI\"], StyleBox[\"2\", \"TR\"]], \",\", StyleBox[\"\[Ellipsis]\", \"TR\"]}], \"]\"}]\) produces a custom report comparing the specified systems from $BenchmarkSystems and the specified data returned from Benchmark."];
If[!ValueQ[$BenchmarkSystems::usage], $BenchmarkSystems::usage = "$BenchmarkSystems gives the names of systems for which the \!\(\*StyleBox[\"MathematicaMark6\", FontSlant -> \"Italic\"]\) benchmark data is known."];


BenchmarkReport::unknown = "None of the requested systems could be found \
in $BenchmarkSystems: `1`.";
BenchmarkReport::oneknown = "Only one of the requested systems was found \
in $BenchmarkSystems: `1`";
BenchmarkReport::onedata = "Only one benchmark result was supplied. Two or \
more sets of results are needed to make a comparison report.";

(* === Function Definitions ============================================== *)

Begin["`Private`"]; (* Benchmarking`Private` *)

Needs["BarCharts`"];
$CheckpointFile = "MBench-" <> $MachineName <> ".m";

$ReportFont = "Times New Roman";
$GraphicsFont = "Times New Roman";
$MachineNameGraphicsFontSize = 12;
$SystemGraphicsFontSize = 10;
$FixedReportFont = "Courier";
$FixedFontSize = 10;
$TitleFontSize = 18;

$HighlightColor = RGBColor[0.180392, 0.54902, 0.729412];
$GrayTextColor = GrayLevel[0.6];
$ColorGradient = {
	{0.882353, 0.882353, 0.882353}, {0.878431, 0.878431, 0.878431}, 
	{0.87451, 0.87451, 0.87451}, {0.870588, 0.862745, 0.862745}, 
	{0.862745, 0.858824, 0.854902}, {0.854902, 0.847059, 0.839216}, 
	{0.847059, 0.835294, 0.831373}, {0.843137, 0.831373, 0.815686}, 
	{0.835294, 0.815686, 0.807843}, {0.827451, 0.807843, 0.792157}, 
	{0.827451, 0.807843, 0.792157}, {0.815686, 0.792157, 0.780392}, 
	{0.803922, 0.784314, 0.768627}, {0.784314, 0.764706, 0.752941}, 
	{0.784314, 0.764706, 0.752941}, {0.772549, 0.756863, 0.741176}, 
	{0.760784, 0.741176, 0.733333}, {0.741176, 0.733333, 0.72549}, 
	{0.721569, 0.72549, 0.717647}, {0.709804, 0.713725, 0.717647}, 
	{0.690196, 0.709804, 0.717647}, {0.666667, 0.705882, 0.717647}, 
	{0.666667, 0.705882, 0.717647}, {0.635294, 0.701961, 0.717647}, 
	{0.603922, 0.698039, 0.717647}, {0.580392, 0.694118, 0.721569}, 
	{0.54902, 0.694118, 0.72549}, {0.54902, 0.694118, 0.72549}, 
	{0.501961, 0.690196, 0.729412}, {0.478431, 0.686275, 0.733333}, 
	{0.45098, 0.682353, 0.741176}, {0.45098, 0.682353, 0.741176},
	{0.423529, 0.67451, 0.745098}, {0.396078, 0.666667, 0.752941},
	{0.376471, 0.658824, 0.756863}, {0.360784, 0.647059, 0.760784}, 
	{0.34902, 0.635294, 0.764706}, {0.337255, 0.619608, 0.768627}, 
	{0.333333, 0.607843, 0.768627}, {0.329412, 0.584314, 0.772549},
	{0.329412, 0.584314, 0.772549}, {0.329412, 0.560784, 0.776471}, 
	{0.333333, 0.529412, 0.776471}, {0.341176, 0.498039, 0.780392},
	{0.341176, 0.498039, 0.780392}, {0.352941, 0.466667, 0.784314}, 
	{0.368627, 0.427451, 0.784314}, {0.384314, 0.388235, 0.788235}, 
	{0.403922, 0.345098, 0.792157}, {0.403922, 0.345098, 0.792157}, 
	{0.423529, 0.298039, 0.796078}, {0.447059, 0.254902, 0.8}, 
	{0.447059, 0.254902, 0.8}, {0.466667, 0.211765, 0.803922}, 
	{0.490196, 0.172549, 0.807843}, {0.490196, 0.172549, 0.807843}, 
	{0.509804, 0.133333, 0.807843}, {0.533333, 0.0901961, 0.811765}, 
	{0.545098, 0.0470588, 0.811765}, {0.568627, 0., 0.815686}, 
	{0.568627, 0., 0.815686}, {0.580392, 0., 0.815686}, 
	{0.6, 0., 0.819608}, {0.6, 0., 0.819608}
	};

$CharacterLookupTable = FromCharacterCode /@ 
	Join[Table[i, {i, 48, 57}], Table[i, {i, 65, 90}]];

(* 
unsortedunion[]:
  Union[] that does not sort. Taken from the Help Browser.
*)
unsortedunion[x_] := Module[{f}, f[y_] := (f[y] = Sequence[]; y); f /@ x]

(*
mungeinput[]:
  mungeinput[] munges the given input into the proper input format for
  gettests[]. The definitions below are pretty self-explanitory.
*)
mungeinput[filter_] := Module[{mungedinput},
	mungedinput = Which[
		Depth[filter] == 1, {{{filter}}},
		Depth[filter] == 2, Table[{{filter[[i]]}},
			{i, Length[filter]}], 
		Depth[filter] == 3, {filter},
		Depth[filter] == 4, filter
        	]; (* Which *)
	If[MemberQ[Map[StringQ, Flatten[mungedinput]], False],
		Return[{}],
		Return[mungedinput]
		] (* If *)
] (* mungeinput *)


(*
formatresults[]:
  formatresults[] munges the numeric input so that it always
  returns a string with the specified number of decimal places.
*)
formatresults[a_Integer, b_] := formatresults[N[a], b]

formatresults[a_Real, b_] := ToString[NumberForm[a, {Infinity, b}, NumberPadding -> {"","0"}]]


(*printTemporaryOrPrint*)
printTemporaryOrPrint := If[$Notebooks,
	PrintTemporary, 
	Print]


(*
getbenchmarks[]:
  getbenchmarks[] gets a sublist from the given list of benchmark tests that
  match the given keyword in the Keywords or TestName tags.
*)
getbenchmarks[benchmarklist_List, keyword_?StringQ] := 
Module[{testidlist, keywordlist},
	testidlist = Cases[benchmarklist,
		{___, "TestName" -> keyword, ___}];
	keywordlist = Cases[benchmarklist,
		{___, "Keywords" -> {___, keyword, ___}, ___}];
	Return[Union[testidlist, keywordlist]]
] (* getbenchmarks *)


(*
gettests[]:
  gettests[] selects a subset of the entire list of available benchmark
  tests (stored in the variable benchmarktests). Its takes a structed list
  as input.
	
  This input list structure defines how to combine the keywords to select
  a particular subset of tests. The general structure is:
	
    {{{"a", "b", ...}, {"c", "d", ...}}, ...}
	
  So, this will find all tests with both keywords "a" and "b" but
  not both keywords "c" and "d" and so on.
*)
gettests[filter_] := 
Module[{testsgotten, sortedtestids, sortedtests}, 
	testsgotten = Union @@ Map[Complement @@ Map[Intersection @@ 
		Map[getbenchmarks[benchmarktests, #]&, #]&, #]&, filter];
	sortedtestids = Sort[Map[("TestName" /. #)&, testsgotten]];
	sortedtests = Flatten[Map[getbenchmarks[benchmarktests, #]&, 
		sortedtestids], 1];
	Return[sortedtests]
] (* gettests *)


(*
getlibraryfromsystem[]
*)
getlibraryfromsystem[systems__String] := 
Module[{benchmarksystems, truefalselist, pos},
	benchmarksystems = (("MachineName" /. #) <> " (" <> \
		("System" /. #) <> ")")& /@ $BenchmarkLibrary;
	truefalselist = Table[
		StringMatchQ[{systems}[[i]], #]& /@ benchmarksystems,
		{i, Length[{systems}]}];
	pos = Union[Flatten[Position[#, True] & /@ truefalselist]];
	Return[$BenchmarkLibrary[[pos]]] 
];


(*
makescore[]:
  makescore[] calculates the overall score of the benchmark as the 
  geometric mean of the run times of the individual tests.
*)
makescore[data_List, benchmark_String] := 
Module[{overallscore, relperf, reference},
	reference = "TotalTime" /. First[Cases[Cases[$BenchmarkLibrary,
		{___, "BenchmarkName" -> benchmark, ___}],
		{___, "BenchmarkReference" -> True, ___}]];
	overallscore = ToExpression[formatresults[ 
		(reference / (Plus @@ #))&[data[[All,2]]], 3]];
	Return[overallscore]
] (* makescore *)


(*
makegridboxtable[]:
*)
makegridboxtable[datatable_List] := Map[Which[
		NumericQ[#], "\<\"" <> formatresults[#, 2] <> "\"\>",
		StringQ[#], "\<\"" <> StringReplace[#, "\n" -> "\\n"] <> "\>\"",
		True, #
	]&, datatable, {2}] 


(*
makecomparisontable[]:
  makecomparison[] generates the comparison table displayed in the
  report notebook generated by BenchmarkReport[].
*)
makecomparisontable[ndata_, hdata_, testids_, benchmarkname_String] := 
Module[
	{results, nresults, hresults, gridboxdata, tabledata, orderlist,
		coloring, highlightedtable, orderedresults, title, caption},
	nresults = With[{mydata = #}, Flatten[{("BenchmarkResult"/.#),
		GridBox[{
			{StyleBox["\<\"" <> ("MachineName" /. #) <> "\>\"",
				FontFamily -> $GraphicsFont, 
				FontWeight -> "Bold",
				FontSize -> $MachineNameGraphicsFontSize]},
			{StyleBox["\<\"" <> ("System" /. #) <> "\>\"",
				FontFamily -> $GraphicsFont, 
				FontColor -> $GrayTextColor,
				FontSize -> $SystemGraphicsFontSize]}
		}, ColumnAlignments -> Right, RowSpacings -> 0.2]& [mydata],
		("TotalTime" /. mydata),
		("Results" /. mydata)[[All, 2]]
		}]] & /@ ndata;
	hresults = With[{mydata = #}, Flatten[{("BenchmarkResult"/.#),
		GridBox[{
			{StyleBox["\<\">> " <> ("MachineName" /. #) 
				<> " <<\"\>",
				FontFamily -> $GraphicsFont, 
				FontColor -> $HighlightColor,
				FontSize -> $MachineNameGraphicsFontSize,
				FontWeight -> "Bold"]},
			{StyleBox["\<\"" <> ("System" /. #) <> "\>\"",
				FontFamily -> $GraphicsFont, 
				FontColor -> GrayLevel[0.6], 
				FontSize -> $SystemGraphicsFontSize]}
		}, ColumnAlignments -> Right, RowSpacings -> 0.2]& [mydata],
		("TotalTime" /. mydata),
		("Results" /. mydata)[[All, 2]]
		}]] & /@ hdata;
	results = Drop[#,1]& /@ (With[{sortdata = #},
			sortdata[[Ordering[(#[[1]]) & /@ sortdata,
				All, Greater]]]]&[Join[nresults,hresults]]);
	orderlist = Ordering /@ Drop[Transpose[results], 1];
	coloring = Part[Transpose[Sort[Transpose[
		{#, Table[x, {x, 1.0, 0.01, -0.99/(Length[#] - 1)}]}
		]]], 2] & /@ orderlist;
	gridboxdata = Transpose[With[
		{transdata = Transpose[makegridboxtable[results]]},
		Join[{transdata[[1]]},
      		Map[StyleBox[GridBox[{{#[[1]]}}, ColumnWidths -> 4.5, 
      			RowMinHeight -> 3.5, 
			RowAlignments -> Center],
			FontColor -> If[#[[2]] > 0.5, GrayLevel[0.9], 
				GrayLevel[0]],
			FontFamily -> $GraphicsFont, 
			FontSize -> $SystemGraphicsFontSize, 
   			Background -> RGBColor @@ ($ColorGradient[[
   					Ceiling[Length[$ColorGradient]
					* #[[2]]]]])]&,
		MapThread[{#1, #2}&, {Drop[transdata,1], coloring}, 2], {2}]
		]]];

	tabledata = {GridBox[Join[{StyleBox[#,FontFamily -> $GraphicsFont,
            FontSize -> $MachineNameGraphicsFontSize,
            FontWeight -> "Bold"]&/@
       	 	Join[{"\<\" \"\>","\<\"Total\"\>"},#&/@(("\<\"Test " 
			<> ToString[#] <> "\"\>")& /@ 
			(Range[Length[testids]]))]}, gridboxdata],
            ColumnAlignments -> {"Right", "Center"}]};
    title = {GridBox[{{
		GridBox[{{" "}},
			ColumnWidths -> $MachineNameGraphicsFontSize * 1.2],
		GridBox[{{
			StyleBox[benchmarkname, 
				FontFamily -> $GraphicsFont, 
				FontSize -> $TitleFontSize,
				FontColor -> $HighlightColor, 
				FontSlant -> "Italic"],
			StyleBox["Detailed Timings",
				FontFamily -> $GraphicsFont, 
				FontSize -> $TitleFontSize, 
				FontColor -> $GrayTextColor]
			}}, ColumnAlignments -> {"Right","Left"},
				ColumnWidths -> $MachineNameGraphicsFontSize 
					* 2.03]
		}}, RowMinHeight -> 6]};
    caption = {GridBox[{{
		GridBox[{{" "}},
			ColumnWidths -> $MachineNameGraphicsFontSize * 1.2],
    	GridBox[{{
    		StyleBox["\<\"Timings are CPU time in seconds\"\>",
  			FontFamily -> $GraphicsFont, 
			FontSize -> $SystemGraphicsFontSize,
			FontColor -> $GrayTextColor]   		
    		}}, ColumnWidths -> $MachineNameGraphicsFontSize * 4.15]	
		}}]};
	Return[{title, tabledata, caption}]
] (* makecomparisontable *)


(*
makereportplot[]:
  makereportplot[] generates the comparison plot displayed in the 
  report notebook generated by BenchmarkReport[].
*)
makereportplot[data_List, label_String] :=
Module[{flipdata, mydata, rest, restplotdata, myplotdata, plotdata,
		highlightpos, chart, labels, maxnum},
	flipdata = Transpose[{Reverse[#1], #2, #3, #4} & @@ Transpose[data]];
	highlightpos = Flatten[Position[flipdata[[All,4]],True]];
	
	mydata = flipdata[[highlightpos]];
	If[Length[mydata] == 0,
		plotdata = {#[[1]], #[[3]], 0.5} & /@ flipdata,
		myplotdata = {#[[1]], #[[3]], 0.5} & /@ mydata;
		rest = flipdata[[Complement[Range[Length[flipdata]],
			highlightpos]]
		]; (* If *)
	restplotdata = If[Length[rest] === 0, 
		{{1, 0, 0}}, {#[[1]], #[[3]], 0.5} & /@ rest];
	plotdata = {myplotdata, restplotdata}];

	maxnum = Max[data[[All,3]]];
	
	chart = GeneralizedBarChart[plotdata,
		Axes -> False, 
		BarOrientation -> Horizontal, 
		BarStyle -> If[Length[highlightpos] == 0, GrayLevel[0.6], 
			{$HighlightColor, GrayLevel[0.6]}],
		DisplayFunction -> Identity, 
        Frame -> True, 
        BarEdges -> False, 
        PlotRange -> {{-0.02, maxnum + .2}, {0.3, Length[data] + 0.7}}, 
        FrameLabel -> {StyleForm["Faster systems give larger numbers", 
		FontFamily -> $ReportFont, FontColor -> GrayLevel[.6]], 
		None, 
		StyleForm[label <> " \!\(\* StyleBox[\"System Comparison\",\nFontSlant->\"Plain\",\nFontColor-> GrayLevel[.6]]\)",
			FontColor -> $HighlightColor, FontSlant -> "Italic", 
			FontFamily -> $ReportFont, FontSize -> 18.], None},
		Epilog -> {Text[
		StyleForm[formatresults[#[[3]], 2], FontSize -> 11, 
			FontFamily -> $ReportFont], {#[[3]] + .01, #[[1]]}, 
				{-1, 0}] & /@ flipdata},
		FrameTicks -> {None, {#[[1]], Grid[
			With[{compname = #[[1]], OSname = If[Length[#] > 1, 
				#[[2]], ""]}, 
			{If[#[[3]], 
			{StyleForm[">> " <> compname <> " <<", 
				FontFamily -> $ReportFont, 
				FontWeight -> "Bold", 
				FontSize -> 12., 
				FontColor -> $HighlightColor]}, 
			{StyleForm[compname, FontFamily -> $ReportFont, 
				FontWeight -> "Bold", FontSize -> 12.]}], 
			{StyleForm[OSname, FontFamily -> $ReportFont, 
				FontColor -> GrayLevel[.5], FontSize -> 11.]}
			}]&[Join[StringSplit[#[[2]],"\n"],{#[[4]]}]], 
				ColumnAlignments -> Right], {0, 0}, 
				{Opacity[0],AbsoluteThickness[0]}}& 
				/@ flipdata, 
			None, None}, 
		GridLines -> {{#, {GrayLevel[.6], AbsoluteDashing[{3, 6}]}}& 
			/@ Range[.25, maxnum + .1, .25], None},
		AspectRatio -> Length[data]/15,
		ImageSize -> 900];
	Return[chart]
] (* makereportplot *)


(*
sortresult[]:
*)
sortresult[listtosort_List, order_:Greater] := listtosort[[#]] & /@
	Ordering[listtosort[[All, 2]], All, order]

(*
mungecode[]:
*)
mungecode[testid_String] := First[ToBoxes[First[ToExpression[
	StringReplace[StringReplace[ToString[InputForm[
		HoldForm["Code"] /. gettests[mungeinput[testid]]
		]], "Benchmarking`Private`" -> ""],
		"benchmark" -> "Benchmarking`Private`benchmark"]
	]], StandardForm]] (* mungecode *)


(*
benchmarkcompare[]
  benchmarkcompare[] generates a comparison report of the given benchmark
  output, either in text or notebook form.
*)
Options[benchmarkcompare] = {"Report" -> "Notebook"};
benchmarkcompare[data__List, opts___Rule] := Module[
	{reportnb, benchmark, fullcomplist, textreport, nbreportopt,
		summarycomplist, highlightopt, highlightlist, 
		showdetailedtime},
	nbreportopt = "Report" /. {opts} /. Options[benchmarkcompare];
	highlightopt = "Highlight" /. {opts} /. {"Highlight" -> {{""}}};
	
	showdetailedtime = If[MemberQ["Results" /. {data}, "Results"], 
		False, True];
	
	fullcomplist = With[{sortdata = #},
		sortdata[[Ordering[("BenchmarkResult" /. #) & /@ sortdata,
			All, Greater]]]]&[List[data]];
	benchmark = Union["BenchmarkName" /. $BenchmarkLibrary];
	If[Length[benchmark] != 1, Return[]];
	benchmark = First[benchmark];
	
	testidlist = Sort["TestName" /.Cases[benchmarktests, {___,
		"Keywords" -> {___, benchmark, ___}, ___}]];
	
	summarycomplist = {"TestName" -> "Summary", "ComparisonTable" -> 
 		Transpose[Join[{Range[Length[fullcomplist]]}, 
			Transpose[sortresult[{("MachineName" /. #) <> "\n"
			<> ("System" /. #), "BenchmarkResult" /. #,
			With[{benchmarkresult=#}, Or @@ (If[StringMatchQ[
				("MachineName" /. benchmarkresult), #[[1]]]
				&& StringMatchQ[
				("System"/.benchmarkresult),#[[2]]],
			True,False]& /@ highlightopt)]}& /@ fullcomplist]]
		]]};

	highlightlist = fullcomplist[[Flatten[Position[
		("ComparisonTable" /. summarycomplist)[[All,4]], True]]]];

	textreport = StringJoin[
		"=== " <> benchmark <> " System Comparison ===\n\n",
		StringReplace[ToString[TableForm[Join[
			{("MachineName" /. #) <> "\n" <> ("System" /. #),
			ToString[NumberForm[("BenchmarkResult" /. #), {4,2},
			NumberPadding -> {"", "0"}]] <> "\n"}& 
		/@ fullcomplist], TableSpacing -> {2,3}]], "\n\n" -> "\n"],
		"\n\n(Faster systems give larger numbers)\n"];
	If[showdetailedtime,
		textreport = StringJoin[textreport,
		"\n\n",
		"=== " <> benchmark <> " Detailed Timings ===\n\n",
		StringReplace[ToString[TableForm[Join[
			{Join[{" ", "Total"}, Table["Test " <> ToString[i], 
				{i, Length[testidlist]}]]}, 
 			Join[{("MachineName" /. #) <> "\n" 
				<> ("System" /. #),
				NumberForm[("TotalTime" /. #), {4,1}, 
 					NumberPadding -> {"", "0"}]},
 				NumberForm[#, {3,2},
 					NumberPadding -> {"", "0"}]& 
					/@ (("Results" /. #)[[All,2]]) 
 				]& /@ fullcomplist
			], TableSpacing -> {2,2}]], "\n\n" -> "\n"],
		"\n\n(Timings are CPU time in seconds)\n"
		]];
	If[highlightlist != {},
		textreport = StringJoin[
			"=== System Information ===\n\n",
			StringJoin @@ (StringJoin[#, "\n\n"] & /@ (
			StringReplace[ToString[TableForm[#]], 
				"\n\n" -> "\n"]& /@ ({
				{"Machine Name:", "MachineName" /. #},
				{"System:", "System" /. #},
				{"Date:", "Date" /. #},
				{"Mathematica Version:", 
					"FullVersionNumber" /. #},
				{"Benchmark Result:", ToString[
           				NumberForm[("BenchmarkResult" /. #), 
					{3, 2}, NumberPadding -> {"", "0"}]]}
			} & /@ highlightlist)
			)), "\n", textreport], textreport
		]; (* If *)

	If[nbreportopt === "Notebook",
	reportnb = NotebookPut[Notebook[{}, 
		WindowTitle -> benchmark <> " Report", 
		WindowSize -> {1084, 700}]];

	NotebookWrite[reportnb, {
	Cell[TextData[{StyleBox[benchmark, FontSlant -> "Italic"], 
		" Benchmark Report"}], "Title",
		FontFamily -> $ReportFont]}];
	If[highlightlist != {},
		NotebookWrite[reportnb, {
		CellGroupData[
			Cell[BoxData[GridBox[makegridboxtable[{
				{"Machine Name:", "MachineName" /. #},
         			{"System:", "System" /. #},
				{"Date:", "Date" /. #},
				{"Mathematica Version:",
					"FullVersionNumber" /. #},
				{"Benchmark Result:", ToString[NumberForm[
					("BenchmarkResult" /. #), {3,2},
					NumberPadding -> {"","0"}]]}
         	}], ColumnAlignments -> Left, ColumnSpacings -> 2]],
			"Text", ShowStringCharacters -> False,
			CellMargins -> {{Inherited,Inherited},
			{Inherited, 30}}, FontFamily -> $ReportFont]& /@
			highlightlist, Open]}]
		]; (* If *)
	NotebookWrite[reportnb, {
	Cell[TextData[{StyleBox[benchmark, FontSlant -> "Italic"],
		" Results"}], "Section",
		FontFamily -> $ReportFont, CellFrame -> False],
	Cell[BoxData[ToBoxes[makereportplot["ComparisonTable" /. summarycomplist,
			benchmark]]], "Graphics", FontFamily -> $ReportFont,
			CellHorizontalScrolling -> True, 
			CellMargins -> {{64, Inherited},{Inherited,Inherited}}]
	}];
	If[showdetailedtime, NotebookWrite[reportnb, {
	Cell[BoxData[GridBox[
		makecomparisontable[Complement[fullcomplist, highlightlist],
			highlightlist, testidlist, benchmark],
		ColumnAlignments -> "Left", RowMinHeight -> 2]], 
		"Text", CellHorizontalScrolling -> True, ShowStringCharacters -> False]
	}]];
	NotebookWrite[reportnb, {
	CellGroupData[Join[
 		{Cell[TextData[{StyleBox[benchmark, FontSlant -> "Italic"],
			" Sources"}], "Section", FontFamily -> $ReportFont,
			CellFrame -> False]},
 		CellGroupData[{
		Cell["Test " <> ToString[Position[testidlist, #][[1,1]]] <> 
			": " <> #, "Subsection", FontFamily -> $ReportFont,
			CellDingbat -> None],
		Cell[BoxData[First[ToBoxes[First[ToExpression[
			StringReplace[ToString[InputForm[
				HoldForm["Code"] /. gettests[mungeinput[#]]
			]], {"Benchmarking`Private`" -> "",
			"BenchmarkTiming" -> "Timing"}]
			]], StandardForm]]], "Input"]
		}, Closed]& /@ testidlist
		], Open],
	Cell["", "Section", CellFrame -> False, ShowCellBracket -> False],
	CellGroupData[{
		Cell["[Plain Text Version]", "Subsubsection", 
			ShowGroupOpenCloseIcon->True,
			FontFamily -> $FixedReportFont, FontSize -> 14,
			CellDingbat -> None, ShowCellBracket -> False],
		Cell[textreport, "Text", FontFamily -> $FixedReportFont,
			FontSize -> $FixedFontSize, PageWidth -> Infinity,
			CellHorizontalScrolling -> True]
	}, Closed],
	Cell["", "Section", CellFrame -> False, ShowCellBracket -> False,
		FontSize -> 8, CellMargins -> {{0,0},{0,0}}],
	Cell[TextData[{
 		"This ", StyleBox[benchmark,
			FontSlant -> "Italic"],
		" report was created with Wolfram ",
 		StyleBox["Mathematica", FontSlant -> "Italic"], " ",
		ToString[NumberForm[$VersionNumber,{2,1}, 
			NumberPadding -> {"","0"}]], 
		". ", StyleBox["MathematicaMark", FontSlant -> "Italic"],
		" is a trademark of Wolfram Research, Inc."
		}], "SmallText", ShowCellBracket -> False]
	}];
	SelectionMove[reportnb, All, Notebook];
	SelectionMove[reportnb, Before, Cell];
	Return[reportnb],
	Return[textreport]
	] (* If nbreportopt *)
] (* benchmarkcompare *)


(* Random data *)

RandomArray[Integer, r_, d_]:= RandomInteger[r, d];
RandomArray[Real, r_, d_]:= RandomReal[r, d];
RandomArray[Complex, r_, d_]:= RandomComplex[r, d];

(*
BenchmarkTiming[]:
*)
SetAttributes[BenchmarkTiming, HoldAll];
BenchmarkTiming[func_] := 
Module[{runtime},
	ClearSystemCache[];
	runtime = AbsoluteTiming[func][[1]] /. {Second -> 1};
	SeedRandom[];
	Return[{ToExpression[formatresults[runtime, 3]]}]
] (* BenchmarkTiming *)


(*
BenchmarkReport[]:
  Benchmark[] is a wrapper for the Benchmark[] and Private`makereportnb[]
  functions.
*)
BenchmarkReport::benchnotfnd = "The requested benchmark `1` could not be \
found in $Benchmarks";
BenchmarkReport::rptype = "Value of option in ReportType -> `1` is not \
\"Notebook\" or \"Text\".";
BenchmarkReport::vrbs = "Value of option in Verbose -> `1` is not True \
or False.";

BenchmarkReport[opts___Rule] :=
Module[{results, reporttypeopt, verboseopt, compare},
	{benchmarktorun, verboseopt} = {"BenchmarkName", "Verbose"} /. {opts}
		/. {"BenchmarkName" -> "MathematicaMark6", 
		"Verbose" -> True}; 
	reporttypeopt = "ReportType" /. {opts} /. If[$FrontEnd =!= Null,
		{"ReportType" -> "Notebook"}, {"ReportType" -> "Text"}];
	If[!MemberQ[$Benchmarks, benchmarktorun], 
		Message[BenchmarkReport::benchnotfnd, benchmarktorun];
		Return[]];
	If[!MemberQ[{"Notebook", "Text"}, reporttypeopt],
		Message[BenchmarkReport::rptype, reporttypeopt];Return[]];
	If[!MemberQ[{True, False}, verboseopt],
		Message[BenchmarkReport::vrbs, verboseopt];Return[]];
	results = First[Benchmark["BenchmarkName" -> benchmarktorun,
		Verbose -> verboseopt]];
	compare = benchmarkcompare[Sequence @@ $BenchmarkLibrary, results,
		"Highlight" -> {{
			("MachineName" /. results),("System" /. results)
			}},
		"Report" -> reporttypeopt];
	Return[compare]
] (* BenchmarkReport["benchmark"]/Benchmark[]*)

BenchmarkReport[input:(_List | _InputForm).., opts___Rule]:= 
Module[{benchmarksystems, benchmarkdata, benchmarklibrarydata={}},
	benchmarksystems = Union@@Cases[{input}, {__String}];
	benchmarkdata = Join[
		Cases[First /@ Cases[{input}, _InputForm], {__Rule}],
		Cases[{input}, {__Rule}]
		];
	reporttypeopt = "ReportType" /. {opts} /. If[$FrontEnd =!= Null,
		{"ReportType" -> "Notebook"}, {"ReportType" -> "Text"}];
	If[!MemberQ[{"Notebook", "Text"}, reporttypeopt],
		Message[BenchmarkReport::rptype, reporttypeopt];Return[]];
	If[Length[benchmarksystems] > 0,
		benchmarklibrarydata = getlibraryfromsystem[Sequence@@ \
			benchmarksystems];
		If[Length[benchmarklibrarydata] === 0,
			Message[BenchmarkReport::unknown, StringReplace[
				ToString[benchmarksystems], 
				{"{"->"","}"->""}]];
			Return[$Failed]
		];
		If[Length[benchmarklibrarydata] === 1 && 
			Length[benchmarkdata] === 0,
			Message[BenchmarkReport::oneknown, (
				("MachineName" /. #) <> " (" <> 
				("System" /. #) <> ")"
				)&[benchmarklibrarydata]];Return[$Failed]
		],
		If[Length[benchmarkdata] === 1,
			Message[BenchmarkReport::onedata, 
				{benchmarksystems}];
			Return[$Failed]
		]		
	];
	compare = benchmarkcompare[Sequence@@benchmarklibrarydata, 
		Sequence@@benchmarkdata,
		"Highlight" -> If[Length[{benchmarkdata}] =!= 0,
			{("MachineName" /. #),("System" /. #)}& /@ 
			{benchmarkdata}, {{""},{""}}],
		"Report" -> reporttypeopt];
	Return[compare]
] (* BenchmarkReport[{data1, data2, ...}, {"system1", "system2"}, ...] *)

BenchmarkReport[input:(_String | _InputForm | _List).., opts___Rule]:=
Module[{benchmarklibrarydata, benchmarkdata, benchmarksystems,
		compare, reporttypeopt},
	benchmarksystems = Union[Cases[{input}, _String]];
	benchmarkdata = Join[
		Cases[First /@ Cases[{input}, _InputForm], {__Rule}],
		Cases[{input}, {__Rule}]
		];
	reporttypeopt = "ReportType" /. {opts} /. If[$FrontEnd =!= Null,
		{"ReportType" -> "Notebook"}, {"ReportType" -> "Text"}];
	If[!MemberQ[{"Notebook", "Text"}, reporttypeopt],
		Message[BenchmarkReport::rptype, reporttypeopt];Return[]];
	If[Length[benchmarksystems] > 0,
		benchmarklibrarydata = getlibraryfromsystem[Sequence@@ \
			benchmarksystems];
		If[Length[benchmarklibrarydata] === 0,
			Message[BenchmarkReport::unknown, StringReplace[
				ToString[benchmarksystems], 
				{"{"->"","}"->""}]];
			Return[$Failed]
		];
		If[Length[benchmarklibrarydata] === 1 &&
			Length[benchmarkdata] === 0,
			Message[BenchmarkReport::oneknown, (
				("MachineName" /. #) <> " (" <> 
				("System" /. #) <> ")"
				)&[benchmarklibrarydata]];Return[$Failed]
		]
	];
	compare = benchmarkcompare[Sequence@@benchmarklibrarydata, 
		Sequence@@benchmarkdata,
		"Highlight" -> If[Length[{benchmarkdata}] =!= 0,
			{("MachineName" /. #),("System" /. #)}& /@ 
			{benchmarkdata}, {{""},{""}}],
		"Report" -> reporttypeopt];
	Return[compare]
] (* BenchmarkReport["system1", "system2", ...]/
	BenchmarkReport["system1", "system2", ... , data1, data2, ...] *)


(*
Benchmark[]:
  Benchmark[] is the user interface to BenchmarkTiming[] function. There is
  no input. There are two options, BenchmarkName: a string or some variation
  of a list of strings that correspond to the Keywords and TestName tags for
  the benchmark tests (See the comments gettests[] for more details) and  
  Verbose (True/False) which toggles verbose reporting.
*)
Benchmark::benchnotfnd = "The requested benchmark `1` could not be found \
in $Benchmarks";
Benchmark::vrbs = "Value of option in Verbose -> `1` is not True \
or False.";
Benchmark[opts___Rule] := 
Module[{allbenchmarkstorun, benchmarkstorun, benchmarksran = {}, testresult,
		resultsran = {}, results, finaloutput, testid, testidlist,
		months, verboseopt, checkpointopt, runinparallel,
		systemnames, machinename, totalruntime, pfactor},
	{benchmarktorun, verboseopt} = {"BenchmarkName", "Verbose"} /. {opts}
		/. {"BenchmarkName" -> "MathematicaMark6", 
			"Verbose" -> True}; 
	If[!MemberQ[$Benchmarks, benchmarktorun], 
		Message[Benchmark::benchnotfnd, benchmarktorun];Return[]];
	If[!MemberQ[{True, False}, verboseopt],
		Message[BenchmarkReport::vrbs, verboseopt];Return[]];
	checkpointopt = "Checkpoint" /. {opts} /. {"Checkpoint" -> False};
	months = {"January", "February", "March", "April", "May",
		"June", "July", "August", "September", "October", 
		"November", "December"};
		
	runinparallel = (Length[Select[Contexts[], 
		StringMatchQ[#, "Parallel`*"]&]] > 0 &&
		Length[Parallel`Parallel`$Slaves] > 0);
	
	If[runinparallel,
		If[checkpointopt, 
			If[verboseopt, printTemporaryOrPrint["Disabling Checkpoint"]];
			checkpointopt = False
			]; (* If - checkpointopt *)
		Parallel`Parallel`RemoteEvaluate[
			Needs["Benchmarking`"]];
		pfactor = 3;
		systemnames = Union[Flatten[{$SystemID, 
			Parallel`Parallel`RemoteEvaluate[$SystemID]}]];
		If[Length[systemnames] > 1,
			machinename = ToString[
				Length[Parallel`Parallel`$Slaves] + 1] 
				<> "-node mixed cluster",
			machinename = ToString[
				Length[Parallel`Parallel`$Slaves] + 1] 
				<> "-node homogeneous cluster"
			]; (* If *)
		systemnames = StringReplace[ToString[systemnames], 
			{"{" -> "", "}" -> ""}],
		
		systemnames = $System;
		machinename = ToLowerCase[$MachineName]
		]; (* If - runinparallel *)
	
	allbenchmarkstorun = gettests[mungeinput[benchmarktorun]];
	If[allbenchmarkstorun == {}, 
		Message[Benchmark::nobnchmrks];Return[]];
	If[MemberQ[FileNames[#],#]&[$CheckpointFile],
		resultsran = ReadList[$CheckpointFile];
		benchmarksran = gettests[
			mungeinput[resultsran[[All,1]]]];
		benchmarkstorun = Complement[allbenchmarkstorun,
			benchmarksran],
		benchmarkstorun = allbenchmarkstorun;
		]; (* If *)
	If[runinparallel,
		benchmarkstorun = Flatten[Table[benchmarkstorun, 
			{pfactor * Length[Parallel`Parallel`$Slaves]}], 1];
		benchmarkstorun = benchmarkstorun[[unsortedunion[
			Flatten[Append[RandomArray[Integer, {1, #}, {3, #}], Range[#]]]
			]]]&[Length[benchmarkstorun]]
		]; (* If *)

	If[verboseopt, If[resultsran != {},
		printTemporaryOrPrint["Cached results found for: "<>
			StringReplace[ToString[resultsran[[All,1]]],
			{"{"->"","}"->""}]]
		]]; (* If - If *)
	If[runinparallel,
	Parallel`Parallel`ExportEnvironment[benchmarkstorun];
	Parallel`Parallel`ExportEnvironment[BenchmarkTiming]; 
	totalruntime = ToExpression[formatresults[(AbsoluteTiming[
		results = Parallel`Evaluate`ParallelMap[(
			testid = "TestName" /. #;
			Prepend["Code" /. #, testid]
			) &, benchmarkstorun]
		][[1]] /. {Second -> 1}), 3]], (* ParallelMap *)
	results = Map[
		(testid = "TestName" /. #;
		If[verboseopt,printTemporaryOrPrint[
			"Test " <> ToString[Position[allbenchmarkstorun,
				#][[1,1]]] <>
			" of " <> ToString[Length[allbenchmarkstorun]] <> 
			": " <> testid <> " ..."]
			]; (* If *)
   		testresult = Prepend["Code" /. #, testid];
		If[checkpointopt, Write[$CheckpointFile, testresult]];
			testresult) &, 
		benchmarkstorun
  		]; (* Map *)
	totalruntime = ToExpression[formatresults[
			(Plus @@ results[[All,2]]), 3]]
	]; (* If *)
	results = Join[resultsran, results];
	If[checkpointopt,
		Close[$CheckpointFile];
		DeleteFile[$CheckpointFile];
		];
	finaloutput = {
		"MachineName" -> ("MachineName" /. {opts} /. 
			("MachineName" -> machinename)),
		"System" -> ("System" /. {opts} /. ("System" -> 
			ToString[systemnames])),
		"BenchmarkName" -> benchmarktorun,
		"FullVersionNumber" -> ToString[NumberForm[$VersionNumber,
			{2,1}, NumberPadding -> {"","0"}]]
			<> "." <> ToString[$ReleaseNumber],
		"Date" -> (months[[#2]] <> " " <> ToString[#3] <> ", "
			<> ToString[#1]) & @@ Take[Date[], 3],
		If[runinparallel,
		"BenchmarkResult" -> ToExpression[formatresults[(
			(pfactor * Length[Parallel`Parallel`$Slaves] * 
				First["TotalTime" /. Cases[$BenchmarkLibrary,
				{__, "BenchmarkReference" -> True, ___}]]) 
				/ totalruntime
			), 3]],
		"BenchmarkResult" -> makescore[results, benchmarktorun]],
		"TotalTime" -> totalruntime
		};
		
	If[Not[runinparallel],
		AppendTo[finaloutput, "Results" -> results]
	];
	Return[InputForm[finaloutput]]
] (* Benchmark *)



(* === Benchmark Tests =================================================== *)

(* 
benchmarktests:
  The set of test programs used in the benchmark. The format is a list of
  rules with the lhs always a string:
	
    {"TestName" -> "foo", "Keywords" -> {"keyword1", "keyword2", ...}, ...}

  The valid rules are:

    TestName -> A unique string that identifies the test.

    Keywords -> A list of strings that describe the test. The keywords, along
        with the TestName are searched by the Benchmark[] command to 
        determine what tests to run. "All" is automatically added.

    Code -> The benchmark test itself. This must be wrapped in HoldForm[]
        to delay evaluation until called by Benchmark[]. It's also a good
        idea to use Modules for scoping.

  Some of the rules are not explicitly required, eg if the test doesn't use
  any variables, then the LocalVariables and Initialization rules don't need
  to be there. It's a good idea to put all of them in for good measure,
  though.
*)
benchmarktests = {

{
	"TestName" -> "Polynomial Expansion",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		BenchmarkTiming[
			Expand[Times @@ Table[(c + x)^3, {c, #}]];
		]&[405]
},
{
	"TestName" -> "Matrix Arithmetic",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{m}, BenchmarkTiming[
			SeedRandom[1];
			m = RandomArray[Real, {}, {#, #}];
			(* Exponent must small enough to avoid machine overflow *)
			Do[(1. + 0.5*m)^127,{50}];
		]]&[1000]
},
{
	"TestName" -> "Matrix Multiplication",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{m1,m2}, BenchmarkTiming[
			SeedRandom[1];
			m1 = RandomArray[Real, {}, {#, #}];
			m2 = RandomArray[Real, {}, {#, #}];
			Do[Dot[m1, m2], {12}]
		]]&[1100]
},
{
	"TestName" -> "Eigenvalues of a Matrix",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{a,b,m}, BenchmarkTiming[
			SeedRandom[1];
			a = RandomArray[Real, {}, {#, #}];
			b = DiagonalMatrix[RandomArray[Real, {}, {#}]];
			m = a.b.Inverse[a];
			Do[Eigenvalues[m],{6}]
		]]&[450]
},
{
	"TestName" -> "Digits of Pi",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :> 
		BenchmarkTiming[N[Pi, #];]&[380000]
},
{
	"TestName" -> "Large Integer Multiplication",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{a}, BenchmarkTiming[
			SeedRandom[1];
			a = RandomArray[Integer, {10^#, 10^(#+1)}, {}];
			Do[a*(a+1), {17}]
		]]&[500000]
},
{
	"TestName" -> "Gamma Function",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{a}, BenchmarkTiming[
			SeedRandom[1];
			a = RandomArray[Integer, {80000, 90000}, {#}];
			Gamma[a]
		]]&[10]
},
{
	"TestName" -> "Discrete Fourier Transform",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{data}, BenchmarkTiming[
			SeedRandom[1];
			data = RandomArray[Real, {}, {#}];
			Do[Fourier[data], {11}]
		]]&[600000]
},
{
	"TestName" -> "Numerical Integration",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		BenchmarkTiming[NIntegrate[Sin[x^2+y^2],
			{x, -#*Pi, #*Pi}, {y, -#*Pi, #*Pi}];
			]&[1.75]
},
{
	"TestName" -> "Matrix Transpose",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{m}, BenchmarkTiming[
			SeedRandom[1];
			m = RandomArray[Real, {}, {#, #}];
			Do[Transpose[m],{40}]
		]]&[2000]
},
{
	"TestName" -> "Random Number Sort",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{a}, BenchmarkTiming[
			SeedRandom[1];
			a = RandomArray[Integer, {1, 50000}, {#}];
			Do[Sort[a], {15}]
		]]&[460000]
},
{
	"TestName" -> "Data Fitting",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :>
		Module[{data}, BenchmarkTiming[
			data = Flatten[Table[{x, y, z, Log[120*x] - 
			Abs[Cos[1/300*z]/(140*y)]}, {x, .2, 10, #},
				{y, .2, 10, #}, {z, .2, 10, #}], 2];
			FindFit[data, Log[a*x] - Abs[Cos[b*z]/(c*y)],
				{a, b, c}, {x, y, z}, AccuracyGoal -> 6];
		]]&[0.22]
},
{
	"TestName" -> "Singular Value Decomposition",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :> 
		Module[{m}, BenchmarkTiming[
			SeedRandom[1];
			m = RandomArray[Real, {}, {#, #}];
			Do[SingularValueDecomposition[m],{2}]
		]] &[850]
},
{
	"TestName" -> "Solving a Linear System",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :> 
		Module[{m, v}, BenchmarkTiming[
			SeedRandom[1];
			m = RandomArray[Real, {}, {#, #}];
			v = RandomArray[Real, {}, {#}];
			Do[LinearSolve[m, v], {16}]
		]] &[1200]
},
{
	"TestName" -> "Elementary Functions",
	"Keywords" -> {"MathematicaMark6"},
	"Code" :> 
		Module[{m1, m2}, BenchmarkTiming[
			SeedRandom[1];
			m1 = RandomArray[Real, {}, {#}];
			m2 = RandomArray[Real, {}, {#}];
			Do[Exp[m1]; Sin[m1]; ArcTan[m1, m2], {30}]
		]] &[2*10^6]
}
}; (* benchmarktests *)

(* This adds "All" to the Keywords rule for each test in benchmarktests. *)
benchmarktests = Map[# /. ("Keywords" -> _) -> ("Keywords" -> (Join[{"All"}, 
	"Keywords" /. #])) &, benchmarktests];

(*
$BenchmarkLibrary:
  A collection of the benchmark run on various machines.
*)
$BenchmarkLibrary = {{
	"MachineName" -> "2p dual-core 2.5 GHz G5",
	"System" -> "Apple Mac OS 10.4.8 (32-bit)",
	"BenchmarkName" -> "MathematicaMark6", 
	"FullVersionNumber" -> "6.0.0", 
 	"Date" -> "March 29, 2007", 
 	"BenchmarkResult" -> 1.221, 
 	"TotalTime" -> 70.591, 
 	"Results" -> {
 		{"Data Fitting", 3.918}, 
   		{"Digits of Pi", 1.248}, 
   		{"Discrete Fourier Transform", 3.875}, 
   		{"Eigenvalues of a Matrix", 4.525}, 
   		{"Elementary Functions", 18.055}, 
   		{"Gamma Function", 0.813}, 
   		{"Large Integer Multiplication", 1.429}, 
   		{"Matrix Arithmetic", 6.388}, 
   		{"Matrix Multiplication", 2.287}, 
   		{"Matrix Transpose", 4.961}, 
   		{"Numerical Integration", 5.831}, 
   		{"Polynomial Expansion", 4.64}, 
   		{"Random Number Sort", 2.715}, 
   		{"Singular Value Decomposition", 6.018}, 
   		{"Solving a Linear System", 3.888}}
	},{
	"MachineName" -> "2p 552 MHz PA-8600",
	"System" -> "HP HP-UX 11.11 (64-bit)",
	"BenchmarkName" -> "MathematicaMark6", 
	"FullVersionNumber" -> "6.0.0",
	"Date" -> "March 27, 2007", 
	"BenchmarkResult" -> 0.213,
	"TotalTime" -> 405.527, 
	"Results" -> {
		{"Data Fitting", 25.269},
   		{"Digits of Pi", 3.918}, 
   		{"Discrete Fourier Transform", 30.481},
   		{"Eigenvalues of a Matrix", 9.762}, 
   		{"Elementary Functions", 64.918},
   		{"Gamma Function", 33.081}, 
   		{"Large Integer Multiplication", 3.406},
   		{"Matrix Arithmetic", 49.543}, 
   		{"Matrix Multiplication", 22.648},
   		{"Matrix Transpose", 28.228}, 
   		{"Numerical Integration", 41.798},
   		{"Polynomial Expansion", 24.047}, 
   		{"Random Number Sort", 21.098},
   		{"Singular Value Decomposition", 23.987}, 
   		{"Solving a Linear System", 23.343}}
    },{
	"MachineName" -> "4p 1 GHz Power 4", 
	"System" -> "IBM AIX 5.3 (64-bit)", 
 	"BenchmarkName" -> "MathematicaMark6", 
 	"FullVersionNumber" -> "6.0.0", 
 	"Date" -> "March 26, 2007", 
 	"BenchmarkResult" -> 0.526, 
 	"TotalTime" -> 163.745, 
 	"Results" -> {
 		{"Data Fitting", 7.343}, 
   		{"Digits of Pi", 1.564}, 
   		{"Discrete Fourier Transform", 5.904}, 
   		{"Eigenvalues of a Matrix", 9.201}, 
   		{"Elementary Functions", 60.129}, 
   		{"Gamma Function", 0.94}, 
   		{"Large Integer Multiplication", 1.638}, 
   		{"Matrix Arithmetic", 10.146}, 
   		{"Matrix Multiplication", 14.401}, 
   		{"Matrix Transpose", 3.304}, 
   		{"Numerical Integration", 13.183}, 
   		{"Polynomial Expansion", 6.989}, 
   		{"Random Number Sort", 6.039}, 
   		{"Singular Value Decomposition", 10.461}, 
   		{"Solving a Linear System", 12.503}}
   	},{
	"MachineName" -> "2p 1.4 GHz Itanium 2",
	"System" -> "Red Hat Enterprise Linux AS 3.0 (64-bit)",
	"BenchmarkName" -> "MathematicaMark6", 
	"FullVersionNumber" -> "6.0.0", 
 	"Date" -> "March 29, 2007", 
 	"BenchmarkResult" -> 1.775, 
 	"TotalTime" -> 48.552, 
 	"Results" -> {
 		{"Data Fitting", 3.819}, 
   		{"Digits of Pi", 0.83}, 
   		{"Discrete Fourier Transform", 1.388}, 
   		{"Eigenvalues of a Matrix", 1.921}, 
   		{"Elementary Functions", 2.804}, 
   		{"Gamma Function", 0.327}, 
   		{"Large Integer Multiplication", 0.762}, 
   		{"Matrix Arithmetic", 7.442}, 
   		{"Matrix Multiplication", 3.292}, 
   		{"Matrix Transpose", 4.836}, 
   		{"Numerical Integration", 7.111}, 
   		{"Polynomial Expansion", 4.898}, 
   		{"Random Number Sort", 2.902}, 
   		{"Singular Value Decomposition", 2.881}, 
   		{"Solving a Linear System", 3.339}}
   	},{
	"MachineName" -> "2.4 GHz Pentium 4",
	"System" -> "Microsoft Windows XP (32-bit)", 
	"BenchmarkName" -> "MathematicaMark6",
	"FullVersionNumber" -> "6.0.0", 
	"Date" -> "February 6, 2007", 
	"BenchmarkResult" -> 1.0, 
	"TotalTime" -> 86.190, 
	"Results" -> {
		{"Data Fitting", 5.046}, 
   		{"Digits of Pi", 1.234}, 
   		{"Discrete Fourier Transform", 3.}, 
   		{"Eigenvalues of a Matrix", 8.843}, 
   		{"Elementary Functions", 7.499}, 
   		{"Gamma Function", 0.922}, 
   		{"Large Integer Multiplication", 1.625}, 
   		{"Matrix Arithmetic", 9.171}, 
   		{"Matrix Multiplication", 8.781}, 
   		{"Matrix Transpose", 5.39}, 
   		{"Numerical Integration", 6.171}, 
   		{"Polynomial Expansion", 5.671}, 
   		{"Random Number Sort", 4.514}, 
   		{"Singular Value Decomposition", 9.419}, 
   		{"Solving a Linear System", 8.904}},
	"BenchmarkReference" -> True
	},{
	 "MachineName" -> "2p 1.28 GHz UltraSPARC III", 
	 "System" -> "Sun Solaris SPARC (64-bit)", 
	 "BenchmarkName" -> "MathematicaMark6", 
	 "FullVersionNumber" -> "6.0.0", 
	 "Date" -> "March 26, 2007", 
	 "BenchmarkResult" -> 0.366, 
	 "TotalTime" -> 235.183, 
	 "Results" -> {
	 	{"Data Fitting", 12.011}, 
	 	{"Digits of Pi", 1.974}, 
	 	{"Discrete Fourier Transform", 9.661}, 
	 	{"Eigenvalues of a Matrix", 12.686}, 
	 	{"Elementary Functions", 45.344}, 
	 	{"Gamma Function", 1.126}, 
	 	{"Large Integer Multiplication", 2.428}, 
	 	{"Matrix Arithmetic", 26.308}, 
	 	{"Matrix Multiplication", 32.336}, 
	 	{"Matrix Transpose", 7.401}, 
	 	{"Numerical Integration", 19.713}, 
	 	{"Polynomial Expansion", 11.921}, 
	 	{"Random Number Sort", 5.905}, 
	 	{"Singular Value Decomposition", 18.903}, 
	 	{"Solving a Linear System", 27.466}}
	},{
	"MachineName" -> "2p 2.4 GHz Opteron 250", 
	"System" -> "Sun Solaris 10 (64-bit)",
	"BenchmarkName" -> "MathematicaMark6", 
	"FullVersionNumber" -> "6.0.0",
	"Date" -> "March 26, 2007", 
	"BenchmarkResult" -> 1.243, 
	"TotalTime" -> 69.344,
	"Results" -> {
		{"Data Fitting", 3.92}, 
		{"Digits of Pi", 0.598},
		{"Discrete Fourier Transform", 3.678}, 
		{"Eigenvalues of a Matrix", 5.793},
		{"Elementary Functions", 15.347}, 
		{"Gamma Function", 0.413},
		{"Large Integer Multiplication", 0.901}, 
		{"Matrix Arithmetic", 7.62},
		{"Matrix Multiplication", 4.322}, 
		{"Matrix Transpose", 3.308},
		{"Numerical Integration", 4.267}, 
		{"Polynomial Expansion", 3.519},
		{"Random Number Sort", 5.677}, 
		{"Singular Value Decomposition", 4.53},
		{"Solving a Linear System", 5.451}
		}
	},{
	"MachineName" -> "2p quad-core 1.6 GHz Intel Xeon 5310", 
	"System" -> "CentOS Linux 4.4 (32-bit)",
 	"BenchmarkName" -> "MathematicaMark6", 
 	"FullVersionNumber" -> "6.0.0",
 	"Date" -> "February 5, 2007", 
 	"BenchmarkResult" -> 1.883,
 	"TotalTime" -> 45.781, 
 	"Results" -> {
	 	{"Data Fitting", 2.005},
	   	{"Digits of Pi", 1.2}, 
	   	{"Discrete Fourier Transform", 0.928},
	   	{"Eigenvalues of a Matrix", 4.42}, 
	   	{"Elementary Functions", 8.321},
	   	{"Gamma Function", 0.881}, 
	   	{"Large Integer Multiplication", 1.536},
	   	{"Matrix Arithmetic", 5.12}, 
	   	{"Matrix Multiplication", 1.611},
	   	{"Matrix Transpose", 3.142}, 
	   	{"Numerical Integration", 3.909},
	   	{"Polynomial Expansion", 2.731}, 
	   	{"Random Number Sort", 3.372},
	   	{"Singular Value Decomposition", 3.486}, 
	   	{"Solving a Linear System", 3.119}
    	}
	},{
	"MachineName" -> "2p dual-core 2.66 GHz Intel Xeon", 
	"System" -> "Apple Mac OS 10.4.8 (64-bit)", 
 	"BenchmarkName" -> "MathematicaMark6", 
 	"FullVersionNumber" -> "6.0.0", 
 	"Date" -> "February 5, 2007", 
 	"BenchmarkResult" -> 2.138, 
 	"TotalTime" -> 40.321, 
 	"Results" -> {
 		{"Data Fitting", 1.751}, 
 		{"Digits of Pi", 0.441}, 
 		{"Discrete Fourier Transform", 1.853}, 
   		{"Eigenvalues of a Matrix", 3.378}, 
   		{"Elementary Functions", 11.749}, 
   		{"Gamma Function", 0.297}, 
   		{"Large Integer Multiplication", 0.52}, 
   		{"Matrix Arithmetic", 3.624}, 
   		{"Matrix Multiplication", 2.073}, 
   		{"Matrix Transpose", 2.387}, 
   		{"Numerical Integration", 2.413}, 
   		{"Polynomial Expansion", 1.56}, 
   		{"Random Number Sort", 1.709}, 
   		{"Singular Value Decomposition", 3.661}, 
   		{"Solving a Linear System", 2.905}}
   	},{
   	"MachineName" -> "2p dual-core 3.0 GHz Intel Xeon 5160", 
   	"System" -> "Microsoft Windows (32-bit)", 
	"BenchmarkName" -> "MathematicaMark6", 
	"FullVersionNumber" -> "6.0.0", 
	"Date" -> "February 7, 2007", 
	"BenchmarkResult" -> 2.836, 
 	"TotalTime" -> 30.391, 
 	"Results" -> {{"Data Fitting", 2.078}, 
   		{"Digits of Pi", 0.672}, 
   		{"Discrete Fourier Transform", 1.094}, 
   		{"Eigenvalues of a Matrix", 2.703}, 
   		{"Elementary Functions", 4.031}, 
   		{"Gamma Function", 0.5}, 
   		{"Large Integer Multiplication", 0.891}, 
   		{"Matrix Arithmetic", 3.172}, 
   		{"Matrix Multiplication", 2.}, 
   		{"Matrix Transpose", 2.156}, 
   		{"Numerical Integration", 2.406}, 
   		{"Polynomial Expansion", 2.078}, 
   		{"Random Number Sort", 1.5}, 
   		{"Singular Value Decomposition", 2.594}, 
   		{"Solving a Linear System", 2.516}}}
   		
};

$BenchmarkSystems = (("MachineName" /. #) <> " (" <> ("System" /. #) 
	<> ")")& /@ $BenchmarkLibrary

End[]; (* Benchmarking`Private` *)

(******* Experimental ******************************************************)

Begin["`Experimental`"]; (* Benchmarking`Experimental` *)

ConvertBenchmarkToXML::usage = "ConvertBenchmarkToXML[benchmark data]";
ConvertBenchmarkToXML[benchmarkdata_List] := Module[{xmlout},
	xmlout = XMLObject["Document"][{},
	XMLElement["BenchmarkMachineReport", {}, {
	
	XMLElement["BenchmarkMachineID", {}, {StringJoin @@ Table[
		Benchmarking`Private`$CharacterLookupTable[[
		RandomInteger[{1, Length[
			Benchmarking`Private`$CharacterLookupTable]
		}]]], {8}]}],
	XMLElement["MathematicaSystem", {}, {"System" /. benchmarkdata}],
	XMLElement["MathematicaFullVersionNumber", {}, 
		{"FullVersionNumber" /. benchmarkdata}],
	XMLElement["MachineClass", {}, {""}],
	XMLElement["MachineIDString", {}, {""}],
	XMLElement["MachineInterconnect", {}, {""}],
	XMLElement["MachineComponentSystems", {}, {
	
	XMLElement["MachineComponentSystem", {}, {
			XMLElement["SystemCount", {}, {""}],
			XMLElement["SystemMakeModel", {}, {""}],
			XMLElement["SystemProductID", {}, {""}],
			XMLElement["SystemOS", {}, {""}],
			XMLElement["SystemCPUCount", {}, {""}],
			XMLElement["SystemCPUName", {}, {""}],
			XMLElement["SystemCPUInfo", {}, {""}],
			XMLElement["SystemCPUSpeed", {}, {""}],
			XMLElement["SystemCPUCoreCount", {}, {""}],
			XMLElement["SystemCPUFeatures", {}, {""}],
			XMLElement["SystemMemorySize", {}, {""}],
			XMLElement["SystemMemoryBus", {}, {""}]
		}]}],
		
	XMLElement["BenchmarkName", {}, {"BenchmarkName" /. benchmarkdata}],
	XMLElement["BenchmarkVersion", {}, 
		{"BenchmarkVersion" /. benchmarkdata}],
	XMLElement["BenchmarkOverallResult", {}, 
		{ToString["BenchmarkResult" /. benchmarkdata]}],
	XMLElement["BenchmarkTotalTime", {}, 
		{ToString["TotalTime" /. benchmarkdata]}],
	XMLElement["BenchmarkTestResults", {},
		Block[{data = ("TestResults" /. benchmarkdata)}, 
		If[0 < Length[data],
			XMLElement["BenchmarkTestResult", {}, {
			
			XMLElement["BenchmarkTestName", {}, {#[[1]]}],
			XMLElement["BenchmarkTestTime", {}, 
				{ToString[#[[2]]]}]}] & /@ data,
			{""}]]
		]
	}], {}];
	Return[xmlout]
] (* ConvertBenchmarkToXML *)
End[]; (* Benchmarking`Experimental` *)

EndPackage[]; (* Benchmarking` *)
