(*:Mathematica Version: 6.0 *)

(*:Package Version: 1.2 *)

(*:Name: Statistics`MultiDiscreteDistributions` *)

(*:Context: Statistics`MultiDiscreteDistributions` *)

(*:Title: Multivariate Discrete Statistical Distributions *)

(*:Author: E. C. Martin *)

(*:Copyright: Copyright 1997-2007, Wolfram Research, Inc. *)

(* :History:
Original version by ECM (Wolfram Research), September 1997.
1.1:improved parameter checking throughout and removed 
    undocumented hidden arguments from distributions,
    Darren Glosemeyer (Wolfram Research), 2005.
1.2:added RandomInteger for all distributions, 
	Darren Glosemeyer (Wolfram Research), August 2005.
*)

(*:Summary:
This package provides properties and functionals of discrete
multivariate probability distributions.
*)

(*:Keywords: discrete distribution, multivariate distribution *)

(*:Requirements: No special system requirements. *)

(*:Warning:
This package extends the definition of several descriptive statistics
functions and the definition of Random.  If the original usage messages
are reloaded, this change will not be reflected in the usage message,
although the extended functionality will remain.
This package also adds rules to Series.
*)

(*:Discussion:
Mean[dist], where dist is the distribution of a random vector, yields a
vector of means, where each element is the mean of the corresponding
element in the random vector.
The same goes for StandardDeviation, Variance, Skewness, Kurtosis,
and KurtosisExcess.
Multivariate functionals that take into account the dependence between
components of a random vector include: Covariance,
Correlation, MultivariateSkewness, and MultivariateKurtosis.

In this package, k is used to denote the dimensionality of the
multidimensional distribution, as in the Johnson, Kotz, Balakrishnan
reference.
*)

(*:Reference:
  Norman L. Johnson, Samuel Kotz, N. Balakrishnan, Discrete Multivariate
	Distributions, Wiley, 1997.
  Norman L. Johnson, Samuel Kotz, Discrete Distributions, Wiley, 1969.
  Multivariate Analysis, Mardia, Kent, Bibby, Academic Press, 1979.
	(p. 31: definitions of multivariate skewness and kurtosis)
  Gentle, James E., Random number generation and Monte Carlo methods, 
    2nd ed. New York: Springer-Verlag, 2003. 
*)

(*:Limitations:
	Quantile, EllipsoidQuantile, and RegionProbability are not
		implemented for MultinomialDistribution,
		NegativeMultinomialDistribution, or MultiPoissonDistribution.

	MultivariateSkewness, MultivariateKurtosis, and ExpectedValue for 
		NegativeMultinomialDistribution are implemented using NSum,
		which does not appear to work very well for anything except
		single sums (sums over one index).   ExpectedValue for
		MultiPoissonDistribution is also implemented using NSum.

	For dimension greater than two, CharacteristicFunction is not
		implemented for MultiPoissonDistribution.

        The vector generalization of BernoulliDistribution,
		MultiBernoulliDistribution, is not implemented.
		See p. 105, Section 36.7, of the Johnson reference.
        The vector generalization of HypergeometricDistribution,
		MultiHypergeometricDistribution, is not implemented.
		See p. 171, Chapter 39, of the Johnson reference.
        The vector generalization of DiscreteUniformDistribution,
		MultiDiscreteUniformDistribution, is not implemented.
		(Note that DiscreteUniformDistribution is a generalized
		HypergeometricDistribution ("Discrete Distributions",
		Johnson & Kotz), so perhaps MultiDiscreteUniformDistribution
		can be obtained as a generalization of
		MultiHypergeometricDistribution.)
	The vector generalization of GeometricDistribution,
		MultiGeometricDistribution, is not implemented.
		See p. 96-97, Section 36.2, of the Johnson reference.
	The vector generalization of LogSeriesDistribution,
		MultiLogSeriesDistribution, is not implemented.
		See p. 157, Section 38.4, of the Johnson reference.

*)


(* vector generalization of BinomialDistribution *)
If[ Not@ValueQ[MultinomialDistribution::usage],
MultinomialDistribution::usage =
"MultinomialDistribution[n, p] represents the multinomial distribution \
for n trials and probability vector p. A random k-vector {n1, ..., nk} \
generated by MultinomialDistribution[n, p], gives the number nj of \
occurrences of the jth event for the j = 1, ..., k mutually exclusive events \
possible at each trial. The jth element of p is the probability of the \
jth event occurring in a given trial. The index n must be a positive integer, \
and the cell probabilities in vector p must sum to unity."]

(* vector generalization of NegativeBinomialDistribution *)
If[ Not@ValueQ[NegativeMultinomialDistribution::usage],
NegativeMultinomialDistribution::usage =
"NegativeMultinomialDistribution[n, p] represents the negative multinomial \
distribution for success count n and failure probability vector p. A random \
k-vector {n1, ..., nk} generated by NegativeMultinomialDistribution[n, p] gives \
the number nj of failures of type j that occur before n successes occur. \
The jth element of p is the probability of a type j failure in each trial, and \
the probability of success in each trial is 1 - Apply[Plus, p]."]
 
(* vector generalization of PoissonDistribution (this generalization
	is not unique but is popular)  *)
If[ Not@ValueQ[MultiPoissonDistribution::usage],
MultiPoissonDistribution::usage =
"MultiPoissonDistribution[mu0, mu] represents the multiple Poisson \
distribution with mean vector mu0 + mu. A random k-vector {n1, ..., nk} \
generated by MultiPoissonDistribution[mu0, mu] gives the number nj of \
occurrences of the jth event for the j = 1, ..., k mutually exclusive events \
possible at each trial. The mean of nj is mu0 + muj, where muj is the \
jth element of mu."]


(* Unprotect MultiDiscreteDistribution.m symbols. *)
Unprotect[MultinomialDistribution, NegativeMultinomialDistribution,
	MultiPoissonDistribution]

(* Unprotect descriptive statistics symbols. *)
Unprotect[Mean, Variance, StandardDeviation, Skewness, Kurtosis,
         PDF, CDF, ExpectedValue, DistributionParameterQ,
         DistributionDomain,DistributionDomainQ,Covariance,
         Correlation,MultivariateSkewness,MultivariateKurtosis,
         CharacteristicFunction,Quantile,EllipsoidQuantile,
         EllipsoidProbability]

(* ======================================================================= *)
Begin["`Private`"]

$maxNumOfPart = 5000; (* Controls warning messages about lower bound on
			 domain size. *)
$maxDomainSize = 5000; (* Controls warning messages about domain size. *) 

(* ======================== Multinomial Distribution ====================== *)
(* p. 32, Chap. 35 "Multinomial Distributions",
	"Multivariate Discrete Distributions", Johnson et al *)

Options[MultinomialDistribution] = {}

(* - - - - - - - - -  DistributionParameterQ[MultinomialDistribution[]] - - - - - - - - - *)
MultinomialDistribution/: DistributionParameterQ[
	MultinomialDistribution[n_, p_]] := And[
    If[Positive[n]&&If[NumericQ[n],IntegerQ[n], True],True,
        Message[MultinomialDistribution::posint, n]; False, True],
    If[Apply[And,Map[(NonNegative[#]&&# <= 1)&, p]] && Apply[Plus, p] == 1, True,
        Message[MultinomialDistribution::probvect, p]; False, True]
]

MultinomialDistribution::posint =
"The parameter `1` describing the number of trials is expected to be a \
positive integer."

MultinomialDistribution::probvect =
"The parameter `1` is expected to be a vector of cell probabilities, \
each entry between 0 and 1, and all entries summing to unity."


(* - - - - - - - - -  DistributionDomainQ[MultinomialDistribution[]] - - - - - - - - - - *)

MultinomialDistribution/: DistributionDomainQ[MultinomialDistribution[n_Integer,
	 p_?VectorQ],
	 list_?((MatrixQ[#, IntegerQ])&)] :=
  Module[{v},
    Scan[(v = #;
	  If[!TrueQ[
		Apply[And, Map[(0 <= # <= n)&, v]] && Apply[Plus, v] == n
	     ],
	     Return[False]
	  ])&, list] =!= False
  ] /; DistributionParameterQ[MultinomialDistribution[n, p]]&&Dimensions[list][[2]] == Length[p]

MultinomialDistribution/: DistributionDomainQ[MultinomialDistribution[n_Integer,
	 p_?VectorQ],
	  x_?((VectorQ[#, IntegerQ])&)] :=
	Length[x] == Length[p] &&
  TrueQ[Apply[And, Map[(0 <= # <= n)&, x]]] && TrueQ[Apply[Plus, x] == n]/; 
  	DistributionParameterQ[MultinomialDistribution[n, p]]


(* shorten display of p using Format. *)
MultinomialDistribution /: Format[
	MultinomialDistribution[n_, p_?VectorQ]] :=
    (
        MultinomialDistribution[n, Short[p]]
    )
    

(* - - - - - - - - - -  DistributionDomain[MultinomialDistribution[]] - - - - - - - - - - *)

MultinomialDistribution/: DistributionDomain[MultinomialDistribution[n_Integer,
	 p_?VectorQ]] :=
	Module[{k = Length[p], (* num of possible events *)
	    numOfPart, (* number of partitions of n into at most k integers *)
		part,	(* partitions of n into at most k integers *)
	        estimatedDomainSize, (* size of domain *)
		domain, pos, zerovec},  
	  (* NOTE: Need a way of generating partitions of
		n exclusively of length k or less; lexicographic order
		is not equivalent to ordering according to partition length.
	  *)
	  If[(numOfPart = myPartitionsP[n, k]) > $maxNumOfPart,
	     (* Issue a warning message so users know what they are
			waiting for as the partitions of n are generated. *)	
	     Message[MultinomialDistribution::domszbnd, n, p, numOfPart ]
	  ];
	  If[k/n < .1 && numOfPart < 1000,
	     (* myPartitions1 is based on NextPartition *)
	     part = myPartitions1[n, k, numOfPart],
	     (* myPartitions2 is based on Partitions *)
	     part = myPartitions2[n, k]
	  ];
	  (* --------------------------------------------- *)
	  (* Note: find final size of domain *)
	  part = Map[Join[#, Table[0, {k-Length[#]}]]&, part];
	  estimatedDomainSize = 0;
	  Scan[(estimatedDomainSize +=
			Apply[Multinomial, Map[Length, Split[#]]])&, part];
	  If[estimatedDomainSize > $maxDomainSize,
	     (* Issue a warning message so users know what they are
			waiting for as the domain is generated. *)	
	     Message[MultinomialDistribution::domsz, n, p, estimatedDomainSize ]
	  ];
	  (* --------------------------------------------- *)
	  domain = Flatten[Map[Permutations, part], 1];
	  If[(pos = Position[p, prob_ /; TrueQ[prob == 0]]) === {},
	     domain,
	     pos = Flatten[pos];
	     zerovec = Table[0, {Length[pos]}];	
	     Select[domain, (#[[pos]] === zerovec)&]
	  ]
	]/; 
  	DistributionParameterQ[MultinomialDistribution[n, p]]

MultinomialDistribution::domszbnd =
"Warning: the size of the domain of MultinomialDistribution[``, ``] is \
bounded below by ``."

MultinomialDistribution::domsz =
"Warning: the size of the domain of MultinomialDistribution[``, ``] is ``."

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
myPartitionsP::usage =
"myPartitionsP[n, k] gives the number of unrestricted partitions of the \
integer n into at most k integers. Equivalently, it gives the number of \
unrestricted partitions of the integer n into integers the largest of which is \
k."

myPartitionsP[n_, k_] :=
	Module[{j, z},
		Last[ Series[1/Product[(1-z^j), {j, k}], {z, 0, n}][[3]] ]
	]

(* CHECK:
myPartitionsP[10, 5]
	30
Length[Select[Partitions[10], Length[#] <= 5 &]]
	30
*)
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
myPartitions1::usage =
"myPartitions1[n, k] constructs all partitions of integer n of length less \
than or equal to k, in reverse lexicographic order."

myPartitions1[n_, k_, numOfPartitions_] :=
 Module[{part = Table[1, {n}], result = {}},
	While[Length[result] < numOfPartitions,
	  part = NextPartition[part];
	  If[Length[part] <= k, AppendTo[result, part]]
	];
	result
 ]

myPartitions2::usage =
"myPartitions2[n, k] constructs all partitions of integer n of length less \
than or equal to k, in reverse lexicographic order."

myPartitions2[n_, k_] :=
	Select[Partitions[n], Length[#] <= k &]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* Partitions is from DiscreteMath`Combinatorica` by Steven Skiena *)
Partitions::usage = "Partitions[n] constructs all partitions of integer n in \
reverse lexicographic order."
Partitions[n_Integer] := Partitions[n,n]
Partitions[n_Integer,_] := {} /; (n<0)
Partitions[n_Integer,1] := { Table[1,{n}] }
Partitions[_,0] := {}
Partitions[n_Integer,maxpart_Integer] :=
        Join[
                Map[(Prepend[#,maxpart])&, Partitions[n-maxpart,maxpart]],
                Partitions[n,maxpart-1]
        ]

(* NextPartition is from DiscreteMath`Combinatorica` by Steven Skiena *)
NextPartition::usage = "NextPartition[p] gives the integer partition following p \
in reverse lexicographic order."
NextPartition[p_List] := Join[Drop[p,-1],{Last[p]-1,1}]  /; (Last[p] > 1)
NextPartition[p_List] := {Apply[Plus,p]}  /; (Max[p] == 1)
NextPartition[p_List] :=
        Module[{index,k,m},
                {index} = First[ Position[p,1] ];
                k = p[[index-1]] - 1;
                m = Apply[Plus,Drop[p,index-1]] + k + 1;
                Join[
                        Take[p,index-2],
                        Table[k,{Quotient[m,k]}],
                        If [Mod[m,k] == 0, {}, {Mod[m,k]}]
                ]
        ]


(* - - - - - - - - - -  PDF[MultinomialDistribution[], ] - - - - - - - - - - *)

(* one or more integers out of range, or integers fail to sum to n *)
MultinomialDistribution/: PDF[MultinomialDistribution[n_Integer, p_?VectorQ], 
	x_?(VectorQ[#, IntegerQ]&)] :=
		 0 /; DistributionParameterQ[MultinomialDistribution[n, p]]&& 
		      (!Apply[And, Map[(0 <= # <= n)&, x]] ||!(Apply[Plus, x] == n))

(* general case of PDF *)
MultinomialDistribution/: PDF[MultinomialDistribution[n_, p_?VectorQ], x_?VectorQ] :=
    Module[{nzpos},
    	nzpos=Complement[Range[Length[p]], Flatten[Position[p, p1_ /; TrueQ[p1 == 0]]]];
	If[Apply[Plus, x] == n,
	   Evaluate[
	    Apply[Multinomial, x] Apply[Times, (p[[nzpos]])^(x[[nzpos]])]
	   ],
	   0]]/; DistributionParameterQ[MultinomialDistribution[n, p]]&&TrueQ[Length[x] == Length[p]]


(* - - - - - - - - - -  CDF[MultinomialDistribution[], ] - - - - - - - - - - *)


(* one or more integers out of range *)
MultinomialDistribution/: CDF[MultinomialDistribution[n_Integer, p_?VectorQ], 
	x_?(VectorQ[#, IntegerQ]&), opts___] := 0 /; 
	DistributionParameterQ[MultinomialDistribution[n, p]]&&!Apply[And, Map[(0 <= # <= n)&, x]]


(* general case of CDF *)
MultinomialDistribution/: CDF[MultinomialDistribution[n_, p_?VectorQ], 
	x_?VectorQ, opts___] :=
  Module[{k = Length[p], y, yvec, prob, arglist, result},
   (
	yvec = Table[Unique[y], {k}];
        prob = PDF[MultinomialDistribution[n, p], yvec];
        arglist = Transpose[{yvec, Table[0, {k}], x}];
        PrependTo[arglist, prob];
        result = Apply[Sum, arglist];
	If[!FreeQ[result, Sum],
		result /. Thread[Rule[yvec, Table[Unique[System`K], {k}]]],
		result]
   )
  ] /; DistributionParameterQ[MultinomialDistribution[n, p]]&&
  	(Apply[And, Map[(!NumberQ[#] || (IntegerQ[#] && 0 <= # <= n))&, x]])
        (* NOTE: you don't want the condition "Apply[Plus, x] <= n" here. *)



(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* The marginal distributions of the MultinomialDistribution[n, {p1, ..., pk}]
	are BinomialDistribution[n, pi]. *)

MultinomialDistribution/: Mean[MultinomialDistribution[n_,
	 p_?VectorQ]] :=
	n p /; DistributionParameterQ[MultinomialDistribution[n, p]]


(* StandardDeviation yields a vector in multivar case *)
MultinomialDistribution/: StandardDeviation[MultinomialDistribution[
	n_, p_?VectorQ]] :=
   	  Sqrt[n p (1-p)] /; DistributionParameterQ[MultinomialDistribution[n, p]]

(* Variance yields a vector in multivar case *)
MultinomialDistribution/: Variance[MultinomialDistribution[
	n_, p_?VectorQ]] := 
		n p (1-p) /; DistributionParameterQ[MultinomialDistribution[n, p]]

(* The covariance matrix of the multinomial distribution is singular
	with rank = (Length[p]-1). *)
MultinomialDistribution/: Covariance[MultinomialDistribution[
	n_, p_?VectorQ]] :=
		n (DiagonalMatrix[p]-Map[List, p].{p}) /;
				DistributionParameterQ[MultinomialDistribution[n, p]]

(* The correlation matrix of the multinomial distribution is singular
	with rank = (Length[p]-1). *)
MultinomialDistribution/: Correlation[MultinomialDistribution[
	n_, p_?VectorQ]] :=
   Module[{sdvector, sigma},
	  (
	  DiagonalMatrix[1/sdvector].sigma.DiagonalMatrix[1/sdvector]
	  ) /; FreeQ[sdvector = StandardDeviation[MultinomialDistribution[
		n, p]], StandardDeviation]  &&
	       FreeQ[sigma = Covariance[MultinomialDistribution[
		n, p]], Covariance]	
   ] /; DistributionParameterQ[MultinomialDistribution[n, p]]

MultinomialDistribution/: Skewness[MultinomialDistribution[
	n_, p_?VectorQ]] := (1 - 2 p)/Sqrt[n p (1-p)] /;
				DistributionParameterQ[MultinomialDistribution[n, p]]

(* A value for the multivariate coefficient of skewness close to 0
	indicates elliptical symmetry. *)
(* From Mardia: A "suitable population counterpart of the measure of
	multivariate skewness for a random vector x distributed
	MultinormalDistribution[mu, sigma] is the expected value of	
	({x - mu}.Inverse[sigma].Transpose[{y-mu}])^3",  where x and
	y are independent and y is also MultinormalDistribution[mu, sigma]).
*)
MultinomialDistribution/: MultivariateSkewness[MultinomialDistribution[
	n_Integer, p_?VectorQ]] :=
   Module[{inverse,
	   domain,
	   m = Mean[MultinomialDistribution[n, p]],
	   ldom, mskewness, x, y, i, j },
    (
	(* Note that DistributionDomain will warn user of a large domain size. *)
	domain = DistributionDomain[MultinomialDistribution[n, p]];
	ldom = Length[domain];
	mskewness = 0;
	Do[({x, y} = domain[[{i, j}]];
	    mskewness += ( If[i==j, 1, 2] *
		           PDF[MultinomialDistribution[n, p], x] *
	                   PDF[MultinomialDistribution[n, p], y] *
	    		   (({x-m}.inverse.Transpose[{y-m}])^3)[[1, 1]] )
	   ),	{i, ldom}, {j, i}];
	mskewness
    ) /; FreeQ[inverse = PseudoInverse[Covariance[
		MultinomialDistribution[n, p]]], PseudoInverse]
   ] /; DistributionParameterQ[MultinomialDistribution[n, p]]

MultinomialDistribution/: Kurtosis[MultinomialDistribution[
	n_, p_?VectorQ]] :=
		 3 + (1 - 6 p (1-p))/(n p (1-p)) /;
				DistributionParameterQ[MultinomialDistribution[n, p]]

(* A value of multivariate kurtosis coefficient close to k*(k+2), where k is
the dimensionality of the data, indicates multinormality. *)
(* From Mardia: A "suitable population counterpart of the measure of
	multivariate kurtosis for a random vector x distributed
	MultinormalDistribution[mu, sigma] is the expected value of	
	({x - mu}.Inverse[sigma].Transpose[{x-mu}])^2
*)
MultinomialDistribution/: MultivariateKurtosis[MultinomialDistribution[
	n_Integer, p_?VectorQ]] := 
   Module[{inverse,
	   domain,
	   m = Mean[MultinomialDistribution[n, p]] },
    (
	(* Note that DistributionDomain will warn user of a large domain size. *)
        domain = DistributionDomain[MultinomialDistribution[n, p]];
	Apply[Plus, Map[ (PDF[MultinomialDistribution[n, p], #] *
		 (({#-m}.inverse.Transpose[{#-m}])^2)[[1, 1]] )&, domain]]
    ) /; FreeQ[inverse = PseudoInverse[Covariance[
		MultinomialDistribution[n, p]]], PseudoInverse]
   ] /; DistributionParameterQ[MultinomialDistribution[n, p]]

 
(* NOTE:
  CharacteristicFunction[dist, t] for a random vector x is defined as the
	expected value of Exp[I t.x], where Length[t] == Length[x].
*)
(* See p. 37 of "Discrete Multivariate Distributions". *)
MultinomialDistribution/: CharacteristicFunction[
      MultinomialDistribution[n_, p_?VectorQ], t_?VectorQ] :=
	(
		(p.Exp[I t])^n	
 	) /; TrueQ[Length[t] == Length[p]] &&
		DistributionParameterQ[MultinomialDistribution[n, p]]
 

(*  - - - - - - -  Quantile[MultinomialDistribution[], q] - - - - -  - *)
(* NOTE:
Quantile is not implemented for MultinomialDistribution[n, p].
Actually since there is no unique vector x such that
q = CDF[MultinomialDistribution[n, p], x], it is difficult to even
define Quantile[MultinomialDistribution[n, p], q].  One possibility is
the so-called "equicoordinate one-sided percentage points", i.e., the
quantile is {c, c, ..., c} such that CDF[MultinomialDistribution[n, p],
{c, c, ..., c}] = q. *)

MultinomialDistribution/: Quantile[MultinomialDistribution[n_, p_],
   q_] := NotImplemented /; False

(* - - - - -  EllipsoidQuantile[MultinomialDistribution[], ellipsoid] - - - - *)
MultinomialDistribution/: EllipsoidQuantile[MultinomialDistribution[
	n_, p_], q_] := NotImplemented /; False

(* - - - - -  EllipsoidProbability[MultinomialDistribution[], ellipsoid] - - - - *)
MultinomialDistribution/: EllipsoidProbability[MultinomialDistribution[
	n_, p_], EllipsoidQuantile[mu1_, radii_, dir___?MatrixQ]] :=
			NotImplemented /; False

(* - - - - - - -  ExpectedValue[f, MultinomialDistribution[], x] - - - - - - *)

(* 2 arg syntax *)
MultinomialDistribution/: ExpectedValue[f_Function,
  MultinomialDistribution[n_Integer, p_?VectorQ], opts___?OptionQ] :=
  Module[{nn, (* number of arguments of function f *)
	  k = Length[p], (* number of possible events in a trial; dimension
				of MultinomialDistribution[n, p] *)
	  domain},
   (
    Apply[Plus, Map[
		    (Apply[f, #] *
                        PDF[MultinomialDistribution[n, p], #]
		    )&, domain]]
   ) /; (If[Length[f]==1,
	    (* Function with only a body *)
	    nn = Max[Cases[{f}, Slot[z_]->z, Infinity]],
	    (* Function with a list of formal parameters *)
	    nn = Length[f[[1]]]	];
	 nn <= k) &&
	(
	 (* Note that DistributionDomain will warn users of a large domain size. *)
	 FreeQ[domain = DistributionDomain[MultinomialDistribution[n, p]], DistributionDomain]
	)
  ] /; DistributionParameterQ[MultinomialDistribution[n, p]]


(* 3 arg syntax *)
MultinomialDistribution/: ExpectedValue[f_,
	MultinomialDistribution[n_Integer, p_?VectorQ], xvec_?VectorQ,
	opts___?OptionQ] :=			
  Module[{domain},
    (
    Apply[Plus, Map[
                    ( (f /. Thread[xvec :> #]) *
                        PDF[MultinomialDistribution[n, p], #]
                    )&, domain]]
    ) /; (
	  (* Note that DistributionDomain will warn users of a large domain size. *)
	  FreeQ[domain = DistributionDomain[MultinomialDistribution[n, p]], DistributionDomain]
	 )
  ] /; DistributionParameterQ[MultinomialDistribution[n, p]]&&Length[xvec] <= Length[p]
	
	
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* single random vector *)
(* Kemp and Kemp (1987) method, p. 68 of "Discrete Multivariate
	Distributions" *)
	
iMultinomialInteger[n_,p_,m_]:=Module[
{array, newarray, ntot, pnew, porder, pval, ptot = 0, i = 2, res},
  {pnew, porder} = 
   Transpose[Sort[Transpose[{p, Range[Length[p]]}], (#1[[1]] < #2[[1]] &)]];
  ntot = RandomInteger[BinomialDistribution[n, pnew[[1]]], m];
  res = {ntot};
  While[i < Length[p],
   pval = pnew[[i]]/(1 - (ptot += pnew[[i - 1]]));
   array = 
    Map[If[# == 0, 0, Total[RandomChoice[{1 - pval, pval} -> {0, 1}, #]]] &, 
     n - ntot];
   ntot = ntot + array;
   i++;
   res = Join[res, {array}]];
  Transpose[Join[res, {n - ntot}]][[All, Ordering[porder]]]]


MultinomialDistribution /: 
 Random`DistributionVector[
  MultinomialDistribution[n_Integer, p_?(VectorQ[#, NumericQ] &)]?DistributionParameterQ, 
  m_Integer, Infinity] := iMultinomialInteger[n, p, m]
 
  
(MultinomialDistribution[n_, p_])@"RandomType"=RandomInteger; 

(* single random vector *)
(* Kemp and Kemp (1987) method, p. 68 of "Discrete Multivariate
	Distributions" *)
	
iRandomMultinomial[n_, p_] :=
  Module[{ntot, ptot, result, bvar},
	ntot = 0; ptot = 0; result = {};
	Scan[(bvar = If[ptot >= 1, 0,
               Random[BinomialDistribution[n-ntot, #/(1-ptot)]]];
	      ntot += bvar;
	      ptot += #;
	      AppendTo[result, bvar])&, Drop[p, -1]];
	Append[result, n-ntot]
  ]
  
MultinomialDistribution/: Random[MultinomialDistribution[n_Integer, 
	p_?(VectorQ[#,NumericQ]&)]] :=iRandomMultinomial[n, p]/; 
			DistributionParameterQ[MultinomialDistribution[n, p]]  

(* ====================================================================== *)
(* ====================================================================== *)
(* ====================================================================== *)
(* ====================================================================== *)
(* ====================================================================== *)

(* =================== Negative  Multinomial Distribution ================= *)
(* p. 93, Chapter 36 "Negative Multinomial and Other Multinomial-Related
	Distributions", "Multivariate Discrete Distributions", Johnson et al *)

Options[NegativeMultinomialDistribution] = {}

(* - - - - - -  DistributionParameterQ[NegativeMultinormalDistribution[]] - - - - - - *)
NegativeMultinomialDistribution/: DistributionParameterQ[
	NegativeMultinomialDistribution[n_, p_]] := And[
    If[Positive[n], (* NOTE: it is ok for n to be fractional, p. 94 Johnson *)
        True,Message[NegativeMultinomialDistribution::pos, n]; False, True],
    If[Apply[And,Map[(NonNegative[#]&&#< 1)&, p]] && Apply[Plus, p] < 1, True,
        Message[NegativeMultinomialDistribution::probvect, p]; False, True]
]

NegativeMultinomialDistribution::pos =
"The parameter `1` describing the number of successes is expected to be a \
positive, although it need not be an integer." 

NegativeMultinomialDistribution::probvect =
"The parameter `1` is expected to be a vector of failure probabilities, \
each entry between 0 and 1, and all entries summing to a quantity \
less than unity."


(* - - - - - -  DistributionDomainQ[NegativeMultinomialDistribution[]] - - - - - - - *)

NegativeMultinomialDistribution/: DistributionDomainQ[
	NegativeMultinomialDistribution[n_Integer, p_?VectorQ],
	 list_?((MatrixQ[#, IntegerQ])&)] :=
   Module[{v},
	Scan[(v = #;
	      If[!TrueQ[Apply[And, Map[(0 <= #)&, v]]],
                 Return[False]]
             )&, list] =!= False
   ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]&&
   	Dimensions[list][[2]] == Length[p]

NegativeMultinomialDistribution/: DistributionDomainQ[
	NegativeMultinomialDistribution[n_Integer, p_?VectorQ],
	 x_?((VectorQ[#, IntegerQ])&)] :=
		Length[x] == Length[p] && 
		TrueQ[Apply[And, Map[(0 <= #)&, x]]]/;
			DistributionParameterQ[NegativeMultinomialDistribution[n, p]]




(* shorten using Format. *)
NegativeMultinomialDistribution /: Format[
	NegativeMultinomialDistribution[n_, p_?VectorQ]] :=
    (
        NegativeMultinomialDistribution[n, Short[p]]
    )


(*  - - - - - - -  DistributionDomain[NegativeMultinomialDistribution[]] - - - - - - - *)

NegativeMultinomialDistribution/: DistributionDomain[
	NegativeMultinomialDistribution[n_Integer,
	 p_?VectorQ]] := 
        Module[{k = Length[p], domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Table[Range[0, Infinity], {k}];
          If[!range, On[Range::range]];
          domain
        ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]


(* - - - - - - -  PDF[NegativeMultinomialDistribution[], ] - - - - - - - *)

(* out of range or nonintegral *)
NegativeMultinomialDistribution/:
	 PDF[NegativeMultinomialDistribution[n_Integer, p_?VectorQ], x_] := 0 /;
        DistributionParameterQ[NegativeMultinomialDistribution[n, p]] &&
	Apply[And, Map[
		 ( NumberQ[N[#]] && (!IntegerQ[#] || Negative[#]) )&, x]]

(* general case of PDF *)
NegativeMultinomialDistribution/:PDF[NegativeMultinomialDistribution[n_, p_?VectorQ], 
	x_?VectorQ] :=
	  Module[{nzpos},
	    nzpos=Complement[Range[Length[p]], Flatten[Position[p, p1_ /; TrueQ[p1 == 0]]]];
	    Apply[Multinomial, Join[x, {n-1}]] (1-Total[p])^n *
			Apply[Times, (p[[nzpos]])^(x[[nzpos]])] 	
	  ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]&&
	  	TrueQ[Length[x] == Length[p]]


(* - - - - - - -  CDF[NegativeMultinomialDistribution[], ] - - - - - - - *)

(* one or more integers out of range *)
NegativeMultinomialDistribution/: CDF[NegativeMultinomialDistribution[
	n_Integer, p_?VectorQ],
          x_?(VectorQ[#, IntegerQ]&), opts___] := 
	0 /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]&&
		!Apply[And, Map[(0 <= #)&, x]]


(* general case of CDF *)
NegativeMultinomialDistribution/:
	 CDF[NegativeMultinomialDistribution[n_, p_?VectorQ],
	  x_?VectorQ, opts___] := 
  Module[{k = Length[p], y, yvec, prob, arglist, result},
   (
	yvec = Table[Unique[y], {k}];
        prob = PDF[NegativeMultinomialDistribution[n, p], yvec];
        arglist = Transpose[{yvec, Table[0, {k}], x}];
        PrependTo[arglist, prob];
	result = Apply[Sum, arglist];
	If[!FreeQ[result, Sum],
                result /. Thread[Rule[yvec, Table[Unique[System`K], {k}]]],
                result]
   )
  ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]&&
  	Apply[And, Map[(!NumberQ[#] || (IntegerQ[#] && 0 <= #))&, x]]



(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* The marginal distributions of the
	 NegativeMultinomialDistribution[n, {p1, ..., pk}]
	are NegativeBinomialDistribution[n, pi/q] 
	where q = 1-(p1+...+pk) is the probability of success in a given trial.
	(p. 100, Johnson)
*)

NegativeMultinomialDistribution/: Mean[
	NegativeMultinomialDistribution[n_,
	 p_?VectorQ]] := n p/(1-Total[p]) /; 
		DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

(* StandardDeviation yields a vector in multivar case *)
NegativeMultinomialDistribution/: StandardDeviation[
	NegativeMultinomialDistribution[n_, p_?VectorQ]] :=
   		With[{q=(1-Total[p])},
   			Sqrt[n p (q + p)]/q] /;
		DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

(* Variance yields a vector in multivar case *)
(* Using Eq. (36.27) for factorial moments and relationship between
	central and factorial moments. *)
NegativeMultinomialDistribution/: Variance[NegativeMultinomialDistribution[
	n_, p_?VectorQ]] := 
		With[{q=(1-Total[p])}, n p (q + p)/q^2] /;
		DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

NegativeMultinomialDistribution/: Covariance[
	NegativeMultinomialDistribution[
	n_, p_?VectorQ]] :=
		With[{q=(1-Total[p])},
			n/q^2 (q DiagonalMatrix[p]+Map[List, p].{p})] /;
		DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

NegativeMultinomialDistribution/:
	 Correlation[NegativeMultinomialDistribution[
	n_, p_?VectorQ]] :=
   Module[{sdvector, sigma},
	  (
	  DiagonalMatrix[1/sdvector].sigma.DiagonalMatrix[1/sdvector]
	  ) /; FreeQ[sdvector = StandardDeviation[
			NegativeMultinomialDistribution[n, p]], StandardDeviation]  
			&&
	       FreeQ[sigma = Covariance[
			NegativeMultinomialDistribution[n, p]], Covariance]	
   ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

(* Using Eq. (36.27) for factorial moments, relationship between
        central and factorial moments, and definition of skewness in
	terms of central moments. *)
NegativeMultinomialDistribution/: Skewness[NegativeMultinomialDistribution[
	n_, p_?VectorQ]] :=
		With[{q=(1-Total[p])}, (2*p + q)/Sqrt[n*p*(p + q)]] /;
		DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

(* - - - - - - - - - - - - -  MultivariateSkewness - - - - - - - - - - - - - *)
If[FreeQ[Options[MultivariateSkewness], NSumTerms],
	(* add Options[NSum] *)
   Options[MultivariateSkewness] = Join[Options[MultivariateSkewness],
					Options[NSum]]
	]

(* A value for the multivariate coefficient of skewness close to 0
	indicates elliptical symmetry. *)
(* From Mardia: A "suitable population counterpart of the measure of
	multivariate skewness for a random vector x distributed
	MultinormalDistribution[mu, sigma] is the expected value of	
	({x - mu}.Inverse[sigma].Transpose[{y-mu}])^3",  where x and
	y are independent and y is also MultinormalDistribution[mu, sigma]).
*)
NegativeMultinomialDistribution/: MultivariateSkewness[
	NegativeMultinomialDistribution[
	n_Integer, p_?VectorQ], opts___] :=
   Module[{inverse, k = Length[p], y, yvec, x, xvec, summand,
	   zerovec, infinityvec, arglist,
	   m = Mean[NegativeMultinomialDistribution[n, p]],
	   accgoal, compiled, method, extraterms, terms, precgoal,
           verify, wprec, wynn},
    (
	xvec = Table[Unique[x], {k}];
	yvec = Table[Unique[y], {k}];
        summand = PDF[NegativeMultinomialDistribution[n, p], xvec] *
		  PDF[NegativeMultinomialDistribution[n, p], yvec] *
		  (({xvec-m}.inverse.Transpose[{yvec-m}])^3)[[1, 1]];
	zerovec = Table[0, {k}]; infinityvec = Table[Infinity, {k}];
	arglist = Join[Transpose[{xvec, zerovec, infinityvec}],
			Transpose[{yvec, zerovec, infinityvec}]];
	PrependTo[arglist, summand];
	{accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn} = {AccuracyGoal, Compiled, Method,
                NSumExtraTerms, NSumTerms, PrecisionGoal,
                VerifyConvergence, WorkingPrecision, WynnDegree} /. {opts} /.
                        Options[MultivariateSkewness];
    (* NSumExtraTerms and WynnDegree are hidden options as of version 6.0.
       it is better to specify these as suboptions to Method->{WynnEpsilon},
       but they are allowed as hidden options to NSum *)
    If[extraterms===NSumExtraTerms, extraterms=12];
    If[wynn===WynnDegree, wynn=1];
        Apply[NSum, Join[arglist,
                {AccuracyGoal->accgoal, Compiled->compiled,
                Method -> method, NSumExtraTerms -> extraterms,
                NSumTerms -> terms, PrecisionGoal -> precgoal,
                VerifyConvergence -> verify, WorkingPrecision -> wprec,
                WynnDegree -> wynn}]
	]
    ) /; FreeQ[inverse = Inverse[Covariance[
		NegativeMultinomialDistribution[n, p]]], Inverse]
   ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]] 
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)


(* Using Eq. (36.27) for factorial moments, relationship between
        central and factorial moments, and definition of skewness in
	terms of central moments. *)
NegativeMultinomialDistribution/: Kurtosis[NegativeMultinomialDistribution[
	n_, p_?VectorQ]] := 
		With[{q=(1-Total[p])}, 3 + (6 p^2 + 6 p q + q^2)/(n p (p+q))] /;
		DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

(* - - - - - - - - - - - - - MultivariateKurtosis - - - - - - - - - - - - - - *)
If[FreeQ[Options[MultivariateKurtosis], NSumTerms],
	(* add Options[NSum] *)
   Options[MultivariateKurtosis] = Join[Options[MultivariateKurtosis],
					Options[NSum]]
   ]

(* A value of multivariate kurtosis coefficient close to k*(k+2), where k is
the dimensionality of the data, indicates multinormality. *)
(* From Mardia: A "suitable population counterpart of the measure of
	multivariate kurtosis for a random vector x distributed
	MultinormalDistribution[mu, sigma] is the expected value of	
	({x - mu}.Inverse[sigma].Transpose[{x-mu}])^2
*)
NegativeMultinomialDistribution/: MultivariateKurtosis[
	NegativeMultinomialDistribution[
	n_Integer, p_?VectorQ], opts___?OptionQ] := 
   Module[{inverse, k = Length[p], y, yvec, summand, arglist,
	   m = Mean[NegativeMultinomialDistribution[n, p]],
	   accgoal, compiled, method, extraterms, terms, precgoal,
	   verify, wprec, wynn },
    (
	yvec = Table[Unique[y], {k}];
	summand = PDF[NegativeMultinomialDistribution[n, p], yvec] *
		(({yvec-m}.inverse.Transpose[{yvec-m}])^2)[[1, 1]];
	arglist = Map[{#, 0, Infinity}&, yvec];
	PrependTo[arglist, summand];
	{accgoal, compiled, method, extraterms, terms, precgoal,
	 verify, wprec, wynn} = {AccuracyGoal, Compiled, Method,
		NSumExtraTerms, NSumTerms, PrecisionGoal,
		VerifyConvergence, WorkingPrecision, WynnDegree} /. {opts} /.
			Options[MultivariateKurtosis];
	(* NSumExtraTerms and WynnDegree are hidden options as of version 6.0.
       it is better to specify these as suboptions to Method->{WynnEpsilon},
       but they are allowed as hidden options to NSum *)
    If[extraterms===NSumExtraTerms, extraterms=12];
    If[wynn===WynnDegree, wynn=1];
	Apply[NSum, Join[arglist,
		{AccuracyGoal->accgoal, Compiled->compiled, 
		Method -> method, NSumExtraTerms -> extraterms,
		NSumTerms -> terms, PrecisionGoal -> precgoal,
		VerifyConvergence -> verify, WorkingPrecision -> wprec,
		WynnDegree -> wynn}] 
	]
    ) /; FreeQ[inverse = Inverse[Covariance[
		NegativeMultinomialDistribution[n, p]]], Inverse]
   ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]] 


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE:
  CharacteristicFunction[dist, t] for a random vector x is defined as the
	expected value of Exp[I t.x], where Length[t] == Length[x].
*)
(* See p. 101 of "Discrete Multivariate Distributions". *)
NegativeMultinomialDistribution/: CharacteristicFunction[
      NegativeMultinomialDistribution[n_, p_?VectorQ], t_?VectorQ] :=
	(
		((1 - p.Exp[I t])/(1-Total[p]))^(-n)
 	) /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]&&
 		TrueQ[Length[t] == Length[p]]


(*  - - - -  Quantile[NegativeMultinomialDistribution[], q] - - - - - - *)
(* NOTE:
Quantile is not implemented for NegativeMultinomialDistribution[n, p].
Actually since there is no unique vector x such that
q = CDF[NegativeMultinomialDistribution[n, p], x], it is difficult to even
define Quantile[NegativeMultinomialDistribution[n, p], q].  One possibility is
the so-called "equicoordinate one-sided percentage points", i.e., the
quantile is {c, c, ..., c} such that CDF[NegativeMultinomialDistribution[n, p],
{c, c, ..., c}] = q. *)

NegativeMultinomialDistribution/: Quantile[
	NegativeMultinomialDistribution[n_, p_], q_] :=
   		NotImplemented /; False

(* - -  EllipsoidQuantile[NegativeMultinomialDistribution[], ellipsoid] - - *)
NegativeMultinomialDistribution/: EllipsoidQuantile[
	NegativeMultinomialDistribution[
	n_, p_], q_] := NotImplemented /; False

(* - -  EllipsoidProbability[NegativeMultinomialDistribution[], ellipsoid] - - *)
NegativeMultinomialDistribution/: EllipsoidProbability[
	NegativeMultinomialDistribution[
	n_, p_], EllipsoidQuantile[mu1_, radii_, dir___?MatrixQ]] :=
			NotImplemented /; False

(* - - - -  ExpectedValue[f, NegativeMultinomialDistribution[], x] - - - - - *)
If[FreeQ[Options[ExpectedValue], NSumTerms],
	(* add Options[NSum] *)
	Unprotect[ExpectedValue];
   	Options[ExpectedValue] = Join[Options[ExpectedValue], Options[NSum]];
   	Protect[ExpectedValue];
]

(* 2 arg syntax *)
NegativeMultinomialDistribution/: ExpectedValue[f_Function,
  NegativeMultinomialDistribution[n_Integer,
	 p_?VectorQ], opts___?OptionQ] :=
  Module[{nn, (* number of arguments of function f *)
	  k = Length[p], (* number of possible events in a trial; dimension
				of NegativeMultinomialDistribution[n, p] *)
	  y, yvec, summand, arglist,
	  accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn
	 },
   (
    yvec = Table[Unique[y], {k}];
    summand = Apply[f, yvec] * PDF[NegativeMultinomialDistribution[n, p], yvec];
    arglist = Map[{#, 0, Infinity}&, yvec];
    PrependTo[arglist, summand];
    {accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn} = {AccuracyGoal, Compiled, Method,
                NSumExtraTerms, NSumTerms, PrecisionGoal,
                VerifyConvergence, WorkingPrecision, WynnDegree} /. {opts} /.
                        Options[ExpectedValue];
    (* NSumExtraTerms and WynnDegree are hidden options as of version 6.0.
       it is better to specify these as suboptions to Method->{WynnEpsilon},
       but they are allowed as hidden options to NSum *)
    If[extraterms===NSumExtraTerms, extraterms=12];
    If[wynn===WynnDegree, wynn=1];
    Apply[NSum, Join[arglist,
                {AccuracyGoal->accgoal, Compiled->compiled,
                Method -> method, NSumExtraTerms -> extraterms,
                NSumTerms -> terms, PrecisionGoal -> precgoal,
                VerifyConvergence -> verify, WorkingPrecision -> wprec,
                WynnDegree -> wynn}]
	]
   ) /; (If[Length[f]==1,
	    (* Function with only a body *)
	    nn = Max[Cases[{f}, Slot[z_]->z, Infinity]],
	    (* Function with a list of formal parameters *)
	    nn = Length[f[[1]]]	];
	 nn <= k)
  ] /; DistributionParameterQ[NegativeMultinomialDistribution[n, p]]


(* 3 arg syntax *)
NegativeMultinomialDistribution/: ExpectedValue[f_,
	NegativeMultinomialDistribution[n_Integer, p_?VectorQ], 
		xvec_?VectorQ, opts___?OptionQ] :=			
  Module[{summand, arglist,
	  accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn},
    summand = f * PDF[NegativeMultinomialDistribution[n, p], xvec];
    arglist = Map[{#, 0, Infinity}&, xvec];
    PrependTo[arglist, summand];
    {accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn} = {AccuracyGoal, Compiled, Method,
                NSumExtraTerms, NSumTerms, PrecisionGoal,
                VerifyConvergence, WorkingPrecision, WynnDegree} /. {opts} /.
                        Options[ExpectedValue];
    (* NSumExtraTerms and WynnDegree are hidden options as of version 6.0.
       it is better to specify these as suboptions to Method->{WynnEpsilon},
       but they are allowed as hidden options to NSum *)
    If[extraterms===NSumExtraTerms, extraterms=12];
    If[wynn===WynnDegree, wynn=1];
    Apply[NSum, Join[arglist,
                {AccuracyGoal->accgoal, Compiled->compiled,
                Method -> method, NSumExtraTerms -> extraterms,
                NSumTerms -> terms, PrecisionGoal -> precgoal,
                VerifyConvergence -> verify, WorkingPrecision -> wprec,
                WynnDegree -> wynn}]
	]
  ] /; Length[xvec] <= Length[p] &&
	DistributionParameterQ[NegativeMultinomialDistribution[n, p]]
	
	
(* the counting method used by negativemultinomialInteger is valid 
   only if r is an integer *)
negativemultinomialInteger = 
  Compile[{{r, _Real}, {p, _Real, 1}, {cumprob, _Real, 1}, 
  	{psum, _Real}, {nn, _Integer}}, 
   Module[{nsucc = 0, nfail = Table[0, {Length[p]}], p1}, 
    Map[(p1 = #;
       nsucc = 0; 
       nfail = Table[0, {Length[p]}];
       If[p1 > psum, nsucc++, 
        nfail[[ Position[Sort[Prepend[cumprob, p1]], p1][[1, 1]] - 
           1 ]] ++ ]; 
       While[nsucc < r, 
        If[(p1 = RandomReal[]) > psum, nsucc++, 
         nfail[[ Position[Sort[Prepend[cumprob, p1]], p1][[1, 1]] - 
            1 ]] ++ ] ];
        nfail) &, RandomReal[{0, 1}, nn] ]] ];

        
(* negativemultinomialInteger2 is valid for both integer and non-integer r, 
   but will be slower than negativemultinomialInteger in some cases
   the method uses the fact that if Z is distributed 
   NegativeBinomialDistribution[r, 1-Totat[pvec]], the distribution 
   MultinomialDistribution[Z, pvec/Total[pvec]] is equivalent to 
   NegativeMultinomialDistribution[r, pvec] *)
negativemultinomialInteger2[r_, pvec_, ptot_,n_] := 
 Block[{multipvec = pvec/ptot, zerovec={Table[0,{Length[pvec]}]}},
   Flatten[Map[
   If[#===0, zerovec, iMultinomialInteger[#, multipvec, 1]]&, 
    RandomInteger[NegativeBinomialDistribution[r, 1 - ptot], n]],1]]

NegativeMultinomialDistribution /: 
 Random`DistributionVector[  
  NegativeMultinomialDistribution[r_, p_?(VectorQ[#, NumericQ] &)],
	m_,Infinity] := Module[
	{cumprob, psum=Total[p]}, 
	Which[psum==0,
	Table[0, {m}, {Length[p]}],
	IntegerQ[r],
	cumprob = FoldList[Plus, 0, p];
   	negativemultinomialInteger[r, p, cumprob, psum, m],
   	True,
   	negativemultinomialInteger2[r, p, psum, m]] ] /; 
  		DistributionParameterQ[NegativeMultinomialDistribution[r, p]]
  		
  		
(NegativeMultinomialDistribution[r_, p_])@"RandomType"=RandomInteger; 

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

negativemultinomial = Compile[{{r, _Real}, {p, _Real, 1}},
        Module[{nsucc = 0, nfail = Table[0, {Length[p]}], p1,
		cumprob = FoldList[Plus, 0, p], psum},
	  psum = Last[cumprob];
	  p1 = Random[];
	  If[p1 > psum,
	     nsucc++,
	     nfail[[ Position[Sort[Prepend[cumprob, p1]], p1][[1, 1]]-1 ]] ++
	  ];
	  While[nsucc < r, 
	    p1 = Random[];
            If[p1 > psum,
               nsucc++,
               nfail[[ Position[Sort[Prepend[cumprob, p1]], p1][[1, 1]]-1 ]] ++
            ]
	  ];
	  nfail
	]	]
	

(* single random vector *)
NegativeMultinomialDistribution/: Random[
	NegativeMultinomialDistribution[n_Integer, p_?(VectorQ[#,NumericQ]&)]] :=
		negativemultinomial[n, p]/;
			DistributionParameterQ[NegativeMultinomialDistribution[n, p]]

(* ====================================================================== *)
(* ====================================================================== *)
(* ====================================================================== *)
(* ====================================================================== *)

(* ====================== Multiple Poisson Distribution ================= *)
(* p. 124, Chap. 37 "Multivariate Poisson Distributions",
	"Multivariate Discrete Distributions", Johnson et al *)

Options[MultiPoissonDistribution] = {}

(* - - - - - - - - -  DistributionParameterQ[MultiPoissonDistribution[]] - - - - - - - - *)
MultiPoissonDistribution/: DistributionParameterQ[
	MultiPoissonDistribution[mu0_, mu_]] := And[
    If[Positive[mu0],True, 
	Message[MultiPoissonDistribution::posparm, mu0]; False,
	True],
    If[Apply[And, Map[Positive, mu]],  True,
        Message[MultiPoissonDistribution::meanvect, mu]; False, True]
]

MultiPoissonDistribution::posparm =
"Parameter `1` is expected to be positive."

MultiPoissonDistribution::meanvect =
"The parameter `1` is expected to be a vector of positive values."



(* - - - - - - - - -  DistributionDomainQ[MultiPoissonDistribution[]] - - - - - - - - - - *)

MultiPoissonDistribution/: DistributionDomainQ[
        MultiPoissonDistribution[mu0_, mu_?VectorQ],
         list_?((MatrixQ[#, IntegerQ])&)] :=
   Module[{v},
	Scan[(v = #;
              If[!TrueQ[Apply[And, Map[(0 <= #)&, v]]],
                 Return[False]]
             )&, list] =!= False
   ] /; Dimensions[list][[2]] == Length[mu]&&
   	DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]


MultiPoissonDistribution/: DistributionDomainQ[
        MultiPoissonDistribution[mu0_, mu_?VectorQ],
	x_?((VectorQ[#, IntegerQ])&)] :=
                Length[x] == Length[mu] &&
                TrueQ[Apply[And, Map[(0 <= #)&, x]]]/;
                	DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]
                	
                	
(* shorten display of mu using Format. *)
MultiPoissonDistribution /: Format[
	MultiPoissonDistribution[mu0_, mu_?VectorQ]] :=
    (
        MultiPoissonDistribution[mu0, Short[mu]]
    )


(* - - - - - - - - - -  DistributionDomain[MultiPoissonDistribution[]] - - - - - - - - - *)

MultiPoissonDistribution/: DistributionDomain[
        MultiPoissonDistribution[mu0_, mu_?VectorQ]] :=
        Module[{k = Length[mu], domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Table[Range[0, Infinity], {k}];
          If[!range, On[Range::range]];
          domain
        ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]



(* - - - - - - - - - -  PDF[MultiPoissonDistribution[], ] - - - - - - - - - - *)

(* one or more negative integers *)
MultiPoissonDistribution/: PDF[MultiPoissonDistribution[mu0_, mu_?VectorQ], 
	x_?((VectorQ[#, IntegerQ])&)] := 0 /;
		Apply[Or, Map[(# < 0)&, x]]&&DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

(* univariate Poisson *)
MultiPoissonDistribution/: PDF[MultiPoissonDistribution[mu0_, mu_?VectorQ], x_?VectorQ] :=
	 Module[{mu1, x1},  
		mu1 = mu[[1]];
		x1 = x[[1]];
		PDF[PoissonDistribution[mu0 + mu1], x1]
	 ] /; TrueQ[Length[x] == Length[mu] == 1]&&DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

(* Eq. (37.3), p. 125,  "Discrete Multivariate Distributions" *)
(* bivariate Poisson *)
MultiPoissonDistribution/: PDF[MultiPoissonDistribution[mu0_, mu_?VectorQ], x_?VectorQ] :=
	 Module[{mu1, mu2, x1, x2, i, sum, result},  
		{mu1, mu2} = mu;
		{x1, x2} = x;
		result = Exp[-mu0-mu1-mu2] * sum[(mu1^(x1-i) mu2^(x2-i) mu0^i)/
			( (x1-i)! (x2-i)! i! ), {i, 0, Min[x1, x2]}];
	        result = result /. {sum :> Sum};
		If[!FreeQ[result, Sum],
		    result = result /. {Sum :> sum};
                    result = result /. {i -> Unique[System`K]};
		    result /. {sum :> Sum},
                    result]
	 ] /; TrueQ[Length[x] == Length[mu] == 2]&&DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]


(* for dimension > 2, use recurrence relations which are useful
	only for numerical x *)
MultiPoissonDistribution/: PDF[MultiPoissonDistribution[mu0_, mu_?VectorQ], x_?VectorQ] :=
	  (
		pP[mu0, mu][x]
	  ) /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]&&
	  	TrueQ[Length[x] == Length[mu]]  &&
		If[Apply[And, Map[NumberQ, x]],
		   True,
		   Message[MultiPoissonDistribution::pdf];  False]
		
MultiPoissonDistribution::pdf =
"For dimension greater than two, PDF[MultiPoissonDistribution[mu0, mu], x] \
evaluates only for numerical x."

pP[mu0_, mu_][x_ /;
	 !Apply[And, Map[(# == 0)&, x]] && !Apply[Or, Map[(# < 0)&, x]]
	     ] := pP[mu0, mu][x] =
  Module[{i, k = Length[x]},
	 i = First[Position[x, Max[x]]][[1]];
	 mu[[i]]/x[[i]] pP[mu0, mu][x-Table[If[j == i, 1, 0], {j, k}]] +
	 	mu0/x[[i]] pP[mu0, mu][x-Table[1, {k}]]
  ] (* Apply[And, Map[(# >= 1)&, x]] *)

pP[mu0_, mu_][x_] := 0 /; Apply[Or, Map[(# < 0)&, x]]

pP[mu0_, mu_][x_] := (
	Exp[-mu0 -Apply[Plus, mu]]
			) /; Apply[And, Map[(# == 0)&, x]]


(* - - - - - - - - - -  CDF[MultiPoissonDistribution[], ] - - - - - - - - - - *)

(* one or more integers out of range *)
MultiPoissonDistribution/: CDF[MultiPoissonDistribution[
	mu0_, mu_?VectorQ],
          x_?(VectorQ[#, IntegerQ]&), opts___] := 
	0 /; !Apply[And, Map[(0 <= #)&, x]]&&
		DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]


(* general case of CDF *)
MultiPoissonDistribution/:
         CDF[MultiPoissonDistribution[mu0_, mu_?VectorQ],
          x_?VectorQ, opts___] :=
  Module[{k = Length[mu], y, yvec, prob, arglist, result, flag},
   (
        yvec = Table[Unique[y], {k}];
      (* temporarily turn off the MultiPoissonDistribution::pdf
          message, since we are about to deliberately set up a call
          to the PDF with symbolic args *)
        flag = (Head[MultiPoissonDistribution::pdf] === String);
        Off[MultiPoissonDistribution::pdf];
        prob = PDF[MultiPoissonDistribution[mu0, mu], yvec];
        arglist = Transpose[{yvec, Table[0, {k}], x}];
        PrependTo[arglist, prob];
        result = Apply[Sum, arglist];
        If[flag, On[MultiPoissonDistribution::pdf]];
        If[!FreeQ[result, Sum],
                result /. Thread[Rule[yvec, Table[Unique[System`K], {k}]]],
                result]
   )
  ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]&&
  	Apply[And, Map[(!NumberQ[#] || (IntegerQ[#] && 0 <= #))&, x]]


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* The marginal distributions of the
	 MultiPoissonDistribution[mu0, {mu1, ..., muk}]
	are PoissonDistribution[mu0 + mui]. *)

MultiPoissonDistribution/: Mean[MultiPoissonDistribution[mu0_,
	 mu_?VectorQ]] :=
	mu0 + mu /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]


(* StandardDeviation yields a vector in multivar case *)
MultiPoissonDistribution/: StandardDeviation[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] :=
   	  Sqrt[mu0 + mu] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

(* Variance yields a vector in multivar case *)
MultiPoissonDistribution/: Variance[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] := 
	(mu0 + mu) /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

MultiPoissonDistribution/: Covariance[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] :=
	  With[{k = Length[mu]},
		Table[mu0, {k}, {k}] + DiagonalMatrix[mu]
	  ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

MultiPoissonDistribution/: Correlation[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] :=
   Module[{sdvector, sigma},
	  (
		DiagonalMatrix[1/sdvector].sigma.DiagonalMatrix[1/sdvector]
	  ) /; FreeQ[sdvector = StandardDeviation[MultiPoissonDistribution[
		mu0, mu]], StandardDeviation]  &&
	       FreeQ[sigma = Covariance[MultiPoissonDistribution[
		mu0, mu]], Covariance]	
   ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

MultiPoissonDistribution/: Skewness[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] := 1/Sqrt[mu0 + mu] /;
			DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

(* A value for the multivariate coefficient of skewness close to 0
	indicates elliptical symmetry. *)
(* From Mardia: A "suitable population counterpart of the measure of
	multivariate skewness for a random vector x distributed
	MultinormalDistribution[mu, sigma] is the expected value of	
	({x - mu}.Inverse[sigma].Transpose[{y-mu}])^3",  where x and
	y are independent and y is also MultinormalDistribution[mu, sigma]).
*)
MultiPoissonDistribution/: MultivariateSkewness[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] :=
   Module[{inverse, k = Length[mu], y, yvec, y0, x, xvec, x0,
	   m = Mean[MultiPoissonDistribution[mu0, mu]],
	   mskewness, table0, rule, table},
    (
	xvec = Table[Unique[x], {k}];
        yvec = Table[Unique[y], {k}];
	(* xvec[[j]] and yvec[[j]] are PoissonDistribution[mu[[j]]],
		j = 1, ..., k,
           and x0 and y0 are each PoissonDistribution[mu0] *)
	mskewness = Expand[
		(({xvec+x0-m}.inverse.Transpose[{yvec+y0-m}])^3)[[1, 1]] ];
	(* Now mskewness is in terms of powers of 2k+2 independent Poisson
		random variables
		x0, y0, xvec[[j]], j = 1, ..., k, and yvec[[j]], j = 1, ..., k.
		Next replace those powers of
		independent random variables with their expectations, i.e.,
		noncentral moments.  The associated Poisson parameter for
		x0 and y0 is mu0.  The associated Poisson parameters for
		xvec and yvec are the elements of mu. *)
	table0 = Table[poissonNoncentralMoment[mu0, j], {j, 3}];
	rule = Join[ Thread[Rule[x0^Range[3], table0]],
		     Thread[Rule[y0^Range[3], table0]] ];
	table = Map[Table[poissonNoncentralMoment[#, j], {j, 3}]&, mu];
	Do[
	   AppendTo[rule, Thread[Rule[ xvec[[i]]^Range[3], table[[i]] ]] ];
	   AppendTo[rule, Thread[Rule[ yvec[[i]]^Range[3], table[[i]] ]] ],
		{i, k}];
	rule = Flatten[rule];
	mskewness /. rule
    ) /; FreeQ[inverse = Inverse[Covariance[
		MultiPoissonDistribution[mu0, mu]]], Inverse]
   ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

(* p. 90 of "Discrete Distributions" by Johnson and Kotz *)
poissonNoncentralMoment[theta_, r_] := 
	Sum[StirlingS2[r, j] theta^j, {j, 0, r}]

MultiPoissonDistribution/: Kurtosis[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] :=
		3 + 1/(mu0 + mu) /;
			DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]

(* A value of multivariate kurtosis coefficient close to k*(k+2), where k is
the dimensionality of the data, indicates multinormality. *)
(* From Mardia: A "suitable population counterpart of the measure of
	multivariate kurtosis for a random vector x distributed
	MultinormalDistribution[mu, sigma] is the expected value of	
	({x - mu}.Inverse[sigma].Transpose[{x-mu}])^2
*)
MultiPoissonDistribution/: MultivariateKurtosis[MultiPoissonDistribution[
	mu0_, mu_?VectorQ]] := 
   Module[{inverse, k = Length[mu], y, yvec, y0,
	   m = Mean[MultiPoissonDistribution[mu0, mu]],
	   mkurtosis, table0, rule, table },
    (
	yvec = Table[Unique[System`K], {k}];
	(* yvec[[j]] is PoissonDistribution[mu[[j]]], j = 1, ..., k, and
	   y0 is PoissonDistribution[mu0] *)
	mkurtosis = Expand[
	   (({yvec+y0-m}.inverse.Transpose[{yvec+y0-m}])^2)[[1, 1]] ];
	(* Now mkurtosis is in terms of powers of 2k+2 independent Poisson
                random variables
                x0, y0, xvec[[j]], j = 1, ..., k, and yvec[[j]], j = 1, ..., k.
                Next replace those powers of
                independent random variables with their expectations, i.e.,
                noncentral moments.  The associated Poisson parameter for
                x0 and y0 is mu0.  The associated Poisson parameters for
                xvec and yvec are the elements of mu. *)
	table0 = Table[poissonNoncentralMoment[mu0, j], {j, 4}];
        rule = Thread[Rule[y0^Range[4], table0]];
        table = Map[Table[poissonNoncentralMoment[#, j], {j, 4}]&, mu];
        Do[
           AppendTo[rule, Thread[Rule[ yvec[[i]]^Range[4], table[[i]] ]] ],
                {i, k}];
        rule = Flatten[rule];
        mkurtosis /. rule
    ) /; FreeQ[inverse = Inverse[Covariance[
		MultiPoissonDistribution[mu0, mu]]], Inverse]
   ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]


 
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE:
  CharacteristicFunction[dist, t] for a random vector x is defined as the
	expected value of Exp[I t.x], where Length[t] == Length[x].
*)

(* univariate Poisson *)
MultiPoissonDistribution/: CharacteristicFunction[
      MultiPoissonDistribution[mu0_, mu_?VectorQ], t_?VectorQ] :=
	Module[{mu1, t1},
		mu1 = mu[[1]];
		t1 = t[[1]];	
		CharacteristicFunction[
			PoissonDistribution[mu0 + mu1], t1] 
 	] /; TrueQ[Length[t] == Length[mu] == 1] &&
 		DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]
 
(* bivariate Poisson:
	See p. 126 of "Discrete Multivariate Distributions". *)
MultiPoissonDistribution/: CharacteristicFunction[
      MultiPoissonDistribution[mu0_, mu_?VectorQ], t_?VectorQ] :=
	Module[{mu1, mu2, t1, t2},	
		{mu1, mu2} = mu;
		{t1, t2} = t;
		Exp[-mu1 (1 - Exp[I t1]) - mu2 (1 - Exp[I t2]) -
			mu0 (1 - Exp[I(t1 + t2)]) ]
 	] /; TrueQ[Length[t] == Length[mu] == 2] &&
 		DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]
 
(* for dimension > 2, characteristic function is not implemented *)
MultiPoissonDistribution/: CharacteristicFunction[
	MultiPoissonDistribution[mu0_, mu_?VectorQ], x_?VectorQ] :=
          (
		NotImplemented
          ) /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]&&
          	If[TrueQ[Length[x] == Length[mu] > 2],
		   Message[MultiPoissonDistribution::charfcn];  False,
		   False]

MultiPoissonDistribution::charfcn =
"For dimension greater than two, CharacteristicFunction[\
MultiPoissonDistribution[mu0, mu], t] is not implemented."


(*  - - - - - - -  Quantile[MultiPoissonDistribution[], q] - - - - -  - *)
(* NOTE:
Quantile is not implemented for MultiPoissonDistribution[mu0, mu].
Actually since there is no unique vector x such that
q = CDF[MultiPoissonDistribution[mu0, mu], x], it is difficult to even
define Quantile[MultiPoissonDistribution[mu0, mu], q].  One possibility is
the so-called "equicoordinate one-sided percentage points", i.e., the
quantile is {c, c, ..., c} such that CDF[MultiPoissonDistribution[mu0, mu],
{c, c, ..., c}] = q. *)

MultiPoissonDistribution/: Quantile[MultiPoissonDistribution[mu0_, mu_],
   q_] := NotImplemented /; False

(* - - - -  EllipsoidQuantile[MultiPoissonDistribution[], ellipsoid] - - - - *)
MultiPoissonDistribution/: EllipsoidQuantile[MultiPoissonDistribution[
	mu0_, mu_], q_] := NotImplemented /; False

(* - - - -  EllipsoidProbability[MultiPoissonDistribution[], ellipsoid] - - - - *)
MultiPoissonDistribution/: EllipsoidProbability[MultiPoissonDistribution[
	mu0_, mu_], EllipsoidQuantile[mu1_, radii_, dir___?MatrixQ]] :=
			NotImplemented /; False

(* - - - - - - -  ExpectedValue[f, MultiPoissonDistribution[], x] - - - - - - *)
If[FreeQ[Options[ExpectedValue], NSumTerms],
        (* add Options[NSum] *)
   Options[ExpectedValue] = Join[Options[ExpectedValue], Options[NSum]]
]


(* 2 arg syntax *)
MultiPoissonDistribution/: ExpectedValue[f_Function,
  MultiPoissonDistribution[mu0_, mu_?VectorQ], opts___?OptionQ] :=
  Module[{nn, (* number of arguments of function f *)
	  k = Length[mu], (* number of possible events in a trial; dimension
				of MultiPoissonDistribution[mu0, mu] *)
	  y, yvec, summand, arglist,
          accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn
         },
   (
    yvec = Table[Unique[y], {k}];
    summand = Apply[f, yvec] * PDF[MultiPoissonDistribution[
        mu0, mu], yvec];
    arglist = Map[{#, 0, Infinity}&, yvec];
    PrependTo[arglist, summand];
    {accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn} = {AccuracyGoal, Compiled, Method,
                NSumExtraTerms, NSumTerms, PrecisionGoal,
                VerifyConvergence, WorkingPrecision, WynnDegree} /. {opts} /.
                        Options[ExpectedValue];
    (* NSumExtraTerms and WynnDegree are hidden options as of version 6.0.
       it is better to specify these as suboptions to Method->{WynnEpsilon},
       but they are allowed as hidden options to NSum *)
    If[extraterms===NSumExtraTerms, extraterms=12];
    If[wynn===WynnDegree, wynn=1];
    Apply[NSum, Join[arglist,
                {AccuracyGoal->accgoal, Compiled->compiled,
                Method -> method, NSumExtraTerms -> extraterms,
                NSumTerms -> terms, PrecisionGoal -> precgoal,
                VerifyConvergence -> verify, WorkingPrecision -> wprec,
                WynnDegree -> wynn}]
	]
   ) /; (If[Length[f]==1,
	    (* Function with only a body *)
	    nn = Max[Cases[{f}, Slot[z_]->z, Infinity]],
	    (* Function with a list of formal parameters *)
	    nn = Length[f[[1]]]	];
	 nn <= k)
  ] /; DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]


(* 3 arg syntax *)
MultiPoissonDistribution/: ExpectedValue[f_,
	MultiPoissonDistribution[mu0_, mu_?VectorQ],
		 xvec_?VectorQ, opts___?OptionQ] :=
  Module[{summand, arglist,
          accgoal, compiled, method, extraterms, terms, precgoal,
         verify, wprec, wynn},
    (
	summand = f * PDF[MultiPoissonDistribution[
        	mu0, mu], xvec];
        arglist = Map[{#, 0, Infinity}&, xvec];
        PrependTo[arglist, summand];
        {accgoal, compiled, method, extraterms, terms, precgoal,
         	verify, wprec, wynn} = {AccuracyGoal, Compiled, Method,
                NSumExtraTerms, NSumTerms, PrecisionGoal,
                VerifyConvergence, WorkingPrecision, WynnDegree} /. {opts} /.
                        Options[ExpectedValue];
        (* NSumExtraTerms and WynnDegree are hidden options as of version 6.0.
       it is better to specify these as suboptions to Method->{WynnEpsilon},
       but they are allowed as hidden options to NSum *)
    If[extraterms===NSumExtraTerms, extraterms=12];
    If[wynn===WynnDegree, wynn=1];
        Apply[NSum, Join[arglist,
                {AccuracyGoal->accgoal, Compiled->compiled,
                Method -> method, NSumExtraTerms -> extraterms,
                NSumTerms -> terms, PrecisionGoal -> precgoal,
                VerifyConvergence -> verify, WorkingPrecision -> wprec,
                WynnDegree -> wynn}]
	]
    )
  ] /; Length[xvec] <= Length[mu] &&
	DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]
	
	

MultiPoissonDistribution /: 
 Random`DistributionVector[ 
  MultiPoissonDistribution[mu0_?NumericQ, mu_?(VectorQ[#, NumericQ] &)]?DistributionParameterQ, 
  	m_Integer, Infinity] :=Module[
  	{margins, muvals=Join[{mu0},mu]},
	margins=Map[RandomInteger[PoissonDistribution[muvals[[#]]],m]&,
		Range[Length[mu]+1]];
 	Transpose[Map[(margins[[1]]+#)&,Rest[margins]]]
 	]


(MultiPoissonDistribution[mu0_, mu_])@"RandomType"=RandomInteger; 

iRandomMultiPoisson[mu0_,mu_]:=Module[{ntot, ptot, result},
	Map[Random[PoissonDistribution[#]]&, mu] +
		Random[PoissonDistribution[mu0]]	
  ]

(* single random vector *)
MultiPoissonDistribution/: Random[
 MultiPoissonDistribution[mu0_?NumericQ, mu_?(VectorQ[#,NumericQ]&)]] :=
 	iRandomMultiPoisson[mu0,mu]/;
 		DistributionParameterQ[MultiPoissonDistribution[mu0, mu]]
 		
(* ====================================================================== *)
End[]

SetAttributes[ MultinomialDistribution, ReadProtected];
SetAttributes[ NegativeMultinomialDistribution, ReadProtected];
SetAttributes[ MultiPoissonDistribution, ReadProtected];

(* Protect descriptive statistics symbols *)
Protect[Mean, Variance, StandardDeviation, Skewness, Kurtosis,
         PDF, CDF, ExpectedValue, DistributionParameterQ,
         DistributionDomain,DistributionDomainQ,Covariance,
         Correlation,MultivariateSkewness,MultivariateKurtosis,
         CharacteristicFunction,Quantile,EllipsoidQuantile,
         EllipsoidProbability]

(* Protect MultiDiscreteDistributions symbols. *)
Protect[MultinomialDistribution, NegativeMultinomialDistribution,
	MultiPoissonDistribution]



 
