
(* :Title: Graphics Legends *)

(* :Author: John M. Novak *)

(* :Summary: 
This package provides functions for placing a legend box on a graphic.
It includes numerous options for specifying the legend and its
placement.
*)

(* :Context: Graphics`Legends` *)

(* :Package Version: 1.1 *)

(* :History:
    Version 1.0 by John M. Novak, February 1991
    Version 1.1 by John M. Novak, May 1999 -- Improved option handling
         a bit, allowing certain graphics options to be properly applied to
         the sub- or super- graphic as necessary.
*)

(* :Keywords: graphics, legends, key *)

(* :Mathematica Version: 4.0 *)

(* :Copyright: Copyright 1991-2007, Wolfram Research, Inc.*)

(* :Warning: Adds rules to Plot[]. *)

(* :Limitation: This does not yet deal with scaled coordinates. *)

(* :Limitation: Automatic placing of legend boxes is not very
    good; tweaking by hand is likely to be required (and is definitely
    needed if there is more than one box being placed). *)

(* :Limitation: Graphics options affect the entire graphic (with
    legend box emplaced). Because the boxes generally contain
    graphics in rectangles, if AspectRatio is changed, unexpected
    changes may occur.  This applies even more so to Legend (as
    opposed to ShowLegend). *)

(* :Limitation: Error checking is somewhat limited at this time. *)

BeginPackage["PlotLegends`"]

Legend::badsize =
"The LegendSize option must be a number or a pair of numbers greater than zero, or it must be Automatic. Using the default value instead.";

Begin["`Private`"]

Options[ShadowBox] =
    {ShadowBorder->{Thickness[.001],GrayLevel[0]},
    ShadowForeground->GrayLevel[1],
    ShadowBackground->GrayLevel[0],
    ShadowOffset->{.1,-.1}};

ShadowBox[pos:{_,_},size:{_,_},opts___] :=
    Module[{bordsty,foresty,backsty,offset,forebox,
                backbox,border},
        {bordsty,foresty,backsty,offset} =
            {ShadowBorder,ShadowForeground,ShadowBackground,ShadowOffset}/.
            Flatten[{{opts},Options[ShadowBox]}];
        If[foresty === Automatic, foresty = GrayLevel[1]];
        If[bordsty === Automatic, bordsty = {Thickness[.001],
                    GrayLevel[0]}];
        If[backsty === Automatic, backsty = GrayLevel[0]];
        forebox = Rectangle[pos,pos + size];
        backbox = Rectangle[pos + offset,pos + size + offset];
        Flatten[{backsty,backbox,foresty,EdgeForm[bordsty],forebox}]]

Options[Legend] =
    {LegendPosition->{-1,-1},LegendSize->Automatic,
    LegendShadow->Automatic,LegendTextSpace->Automatic,
    LegendTextDirection->Automatic,LegendTextOffset->Automatic,
    LegendLabel->None,LegendLabelSpace->Automatic,
    LegendOrientation->Vertical, LegendSpacing->Automatic,
    LegendBorder->Automatic,
    LegendBorderSpace->Automatic,LegendBackground->Automatic,
    ShadowBackground -> GrayLevel[0]};



Legend[fn:(_Function | _Symbol),boxes_?NumberQ,
        minstr_String:"",maxstr_String:"",opts___] :=
    Module[{its,strs},
        its = Map[fn,Range[0,1,1/(boxes - 1)]];
        strs = Table["",{Length[its]}];
        strs[[1]] = minstr;strs[[Length[strs]]] = maxstr;
        Legend[Transpose[{its,strs}],opts,
            LegendSpacing->0]]

Legend[items:{{_,_}..},opts___] :=
    Module[{ln = Length[items],boxes,lb,n,inc,rn,as,gr,sbox,
            pos,size,shadow,tspace,lspace,bspace,tdir,toff,
            label,orient,space,back,bord, pt, tmp},
        {pos,size,shadow,tspace,tdir,label,lspace,
            orient,space,bspace,toff,back,bord} =
            {LegendPosition,LegendSize,LegendShadow,
            LegendTextSpace,LegendTextDirection,LegendLabel,
            LegendLabelSpace,LegendOrientation,
            LegendSpacing,LegendBorderSpace,
            LegendTextOffset,LegendBackground,
            LegendBorder}/.Flatten[{opts, Options[Legend]}];
        If[Not[NumberQ[space]], inc = .08,inc = space];
        If[tspace === Automatic,
            If[Count[Transpose[items][[2]],""] == ln,
                tspace = 0,
                If[orient === Vertical,
                    tspace = 2,
                    tspace = 1]]];
        If[lspace === Automatic,
            If[(label =!= None) && (label =!= ""),
                lspace = 1,
                lspace = 0]];
        If[bspace === Automatic,
            bspace = .1];
        If[toff === Automatic,
            If[orient === Vertical,toff = {-1,0},
                toff = {0,-1}]];
        If[tdir === Automatic,
            tdir = {1,0}];
        boxes =
            If[orient === Vertical,
            Table[pt = {inc,inc (2 n - 1) + (n - 1)};
                {rec[pt,{1,1},items[[ln - n + 1,1]]],
                Text[items[[ln - n + 1,2]],
                    pt + {1 + inc + .05,1/2},toff,tdir]},
                {n,ln}],
            Table[pt = {inc (2 n - 1) + (n - 1),inc};
                {rec[pt,{1,1},items[[n,1]]],
                Text[items[[n,2]],
                    pt + {1/2, 1 + inc},toff,tdir]},
                {n,ln}]];
        lb = If[lspace != 0,
            Text[label,
                If[orient === Vertical,
                    {(2 inc + 1 + tspace)/2,
                        (2 inc + 1) ln + lspace/2},
                    {(2 inc + 1) ln /2,
                        2 inc + 1 + tspace + lspace/2}],
                {0,0}],
            {}];
        rn = If[orient === Vertical,
            {{-bspace,2 inc + 1 + tspace + bspace},
                {-bspace,(2 inc + 1) ln + lspace + bspace}},
            {{-bspace,(2 inc + 1) ln + bspace},
                {-bspace,2 inc + 1 + tspace + lspace + bspace}}];
        If[Min[size] <= 0 || !MatchQ[size, {_?NumericQ, _?NumericQ}],
            If[Not[NumberQ[size]],
                If[size =!= Automatic, Message[Legend::badsize]];
                size = .8
            ];
            tmp = Map[#[[2]] - #[[1]] &,rn];
            size = tmp (size/Max[tmp])];
        as = size[[2]]/size[[1]];
        gr = Graphics[{boxes,lb},AspectRatio->as,PlotRange->rn,
              Sequence@FilterRules[Flatten[{opts}], Options[Graphics]]];
        If[shadow =!= None,
            If[shadow === Automatic, shadow = {.05,-.05}];
            sbox = ShadowBox[pos,size,ShadowForeground->back,
                ShadowBorder->bord,ShadowOffset->shadow,opts],
            sbox = {}];
        Flatten[{sbox, Inset[gr, pos, {Left, Bottom}, size]}]]
            

rec[start:{_,_},size:{_,_},style_] :=
    Module[{},
		If[MemberQ[{RGBColor,Hue,CMYKColor,GrayLevel,Directive}, Head[style]],
            {style, Rectangle[start, start + size]},
            Inset[style, start, {Left, Bottom}, size]]]


ShowLegend[agr_, largs:({__}..), opts___?OptionQ] :=
    Module[{as, ls={largs}, rec, ap, bubbleupopts, bubbledownopts},
      (* options that 'bubble up' from the central graphic to the
         containing graphic *)
        bubbleupopts =
            AbsoluteOptions[agr, {ImageSize, Background, ColorOutput}];
        as = AspectRatio/.AbsoluteOptions[agr, AspectRatio];
      (* options that 'bubble down' from ShowLegend to the subgraphics --
         doesn't override if they are explicitly set in the subgraphics,
         but does override defaults for the subgraphics *)
        bubbledownopts = {Sequence@@FilterRules[Flatten[{opts}], {FormatType}]};
      (* aspect ratio of input graphic is used to compute the default position
         of the legend keys and the size of the containing rectangle *)         
        If[!NumberQ[as], as = 1];
        If[as > 1,
            rec = Inset[Append[agr, bubbledownopts], {-1/as, -1}, {Left, Bottom}, {2/as,2}];
                ap = {-1/as - .2,-1.2},
            rec = Inset[Append[agr, bubbledownopts], {-1, -as}, {Left, Bottom}, {2, 2as}];
                ap = {-1.2,-as - .2}
        ];
        ls = Apply[
            Legend[##, LegendPosition -> ap, Sequence @@ bubbledownopts]&,
            ls, {1}
        ];
        Show[Graphics[{rec,ls}, Sequence@@FilterRules[Flatten[{opts}], Graphics],
            Sequence @@ bubbleupopts, AspectRatio->Automatic,
            PlotRange->All]
        ]
    ]



Unprotect[Plot];

Plot/: Plot[fn_,r_,o1___,PlotLegend->None,o2___] :=
    Plot[fn,r,Evaluate[FilterOptions[Plot,o1,o2]]]

Plot/: Plot[fn_,r_,o1___,PlotLegend->lg_,o2___] :=
    Module[{txt = lg, sopts, gopts, lopts, ps, disp, ln, gr, tb, bubbledownopts,
            defaultcolors, opts},
		opts = Flatten[{o1,o2}];
        gopts = FilterRules[opts, Options[Plot]];
        sopts = FilterRules[opts, Options[ShadowBox]];
        lopts = FilterRules[opts, Options[Legend]];
        {ps} = {PlotStyle} /. Join[Flatten[{gopts}], Options[Plot]];
        disp = First[FilterRules[Flatten[{gopts, Options[Plot]}], DisplayFunction]];
        ln = If[Head[Unevaluated[fn]] === List,
                  Total[Dimensions[Unevaluated[fn]]], 1];

        If[Head[txt] =!= List, txt = {txt}];
        txt = PadRight[txt, ln, ""];
        
        defaultcolors =
            Table[Hue[FractionalPart[0.67 + 2.0 (i - 1)/GoldenRatio], 0.6, 0.6],
                {i, 1, ln}];
        If[ps === Automatic,
            ps = defaultcolors
        ];
        If[Head[ps] =!= List, ps = {ps},
        If[Length[ps] == 0, ps = {{}}]];
        While[Length[ps] < ln, ps = Join[ps,ps]];
        ps = Take[ps, ln];
        ps = Transpose[{defaultcolors, ps}];
        ps = ps/.Dashing[x_] -> Dashing[2/0.3 x]; (* scale dashes *)

        tb = Table[{Graphics[Flatten[{ps[[n]],
            Line[{{0,0},{1,0}}]}]],txt[[n]]},{n,ln}];

        gr = Insert[
            Plot[fn, r, DisplayFunction->Identity, Evaluate[gopts]],
            disp, 2];

        bubbledownopts = AbsoluteOptions[gr,
            {FormatType}
        ];

        ShowLegend[gr, {tb, sopts, lopts},
                   disp,
                   Sequence @@ bubbledownopts
        ]
    ]

    
(* update syntax coloring templates to allow the PlotLegend options *)
If[FreeQ[SyntaxInformation[Plot], "OptionNames"],
    SyntaxInformation[Plot] = Join[SyntaxInformation[Plot],
            {"OptionNames" -> ToString /@ Union[First /@ Options[Plot], {PlotLegend},
                First /@ Options[ShadowBox], First /@ Options[Legend]]}
        ]
]

Protect[Plot];

End[]

EndPackage[]

