#!/bin/ksh
#
# Copyright (c) 1999, by Sun Microsystems, Inc.
# All rights reserved.
#
# create_site.sh - private script to generate an empty site directory
# from the configuration templates
#
# create_site.sh [site_name] [site_root] [sws install path]

#-------------------------------------------------------------------------------
#
#	PutFile template-file target-file
#
# Routine to 'situate' a template by replacing macro-strings and
# by either removing or unadorning variant lines 
# (currently, lines marked #SERVLET)

PutFile() {
    INPUT=$1
    OUTPUT=$2

    #
    # Except for the default_site, url block must begin with
    # url //website {
    #     ^^^^^^^^^^
    #
    TMPSTR=""
    if [ "$SITE_NAME" != "default_site" ]; then
	TMPSTR="//$SITE_NAME"
    fi

    M4="/usr/ccs/bin/m4 -DSITE_ROOT=$SITE_ROOT -DSITE_NAME=$TMPSTR" 

    if [ -d ${SWSPATH}/etc/opt/SUNWut/http/site_template/servlets ]; then
	HAVESERVLETS=1
    else
	HAVESERVLETS=0
    fi

    if [ $HAVESERVLETS -eq 1 ]; then
	$M4 $INPUT | sed 's/#SERVLET//g' > $OUTPUT;
    else
	$M4 $INPUT | grep -v '#SERVLET' > $OUTPUT;
    fi;
}

# ConservativeCopy - copy a file from the template area to the
#		     target area, if it can be done without clobbering.
#
#	Args 1 - SourceRoot - top of the template area
#	     2 - SourcePath  - identifies the file or directory
#	     3 - TargetRoot - top of the destination
#		 	      (so the target is TargetRoot/SourcePath)
#	

ConservativeCopy () {
    SourceRoot=$1
    SourcePath=$2
    TargetRoot=$3

    if [ -d $TargetRoot/$SourcePath ]; then

    	if [ -d $SourceRoot/$SourcePath ]; then
		
		# We want, and have, a directory.
		return 0;
	fi;

	MSG=`gettext SUNWhttp 'Warning: %s is a directory'` 
	printf "$MSG" "$TargetRoot/$SourcePath"
	echo
	return 1
    fi;

    # The driver works top down.
    # If we don't have a directory to put the target in,
    # then we failed to create it earlier and made noise then.

    TargetPlace=`dirname $TargetRoot/$SourcePath`
    if [ ! -d $TargetPlace ]; then
 	return -1
    fi;

    if [ -d $SourceRoot/$SourcePath ]; then

	if [ -f $TargetRoot/$SourcePath ]; then
	    MSG=`gettext SUNWhttp 'Error: Cannot create directory %s. File exists.'` 
	    printf "$MSG" "$TargetRoot/$SourcePath"
	    echo
	    return 0
        fi;

	mkdir -p $TargetRoot/$SourcePath
	chown -f adm:adm $TargetRoot/$SourcePath
	return $?

    fi;

    # Dealing with a plain file

    if [ -r $TargetRoot/$SourcePath ]; then

	# Don't clobber.
        MSG=`gettext SUNWhttp 'File exists: %s.'`
	printf "$MSG" "$TargetRoot/$SourcePath"
	echo
	return 0
    fi;

    # Plain file, no obstacle.

    cp -p $SourceRoot/$SourcePath $TargetRoot/$SourcePath

}

# ConservativeCopyTree
# 	This is a scripted 'cp -r' that uses ConservativeCopy
#	to make the right noises when things are already present in
#	the target.  Since this script is  often run as root,
# 	cp -r can clobber things we'd rather not clobber.

ConservativeCopyTree () {

	SourceRoot=$1
	TargetRoot=$2

	Paths=`(cd $SourceRoot; find . -print)`

	for Path in $Paths; do
		ConservativeCopy "$SourceRoot" "$Path" "$TargetRoot"
	done;
}

#-------------------------------------------------------------------------------
#
# Argument processing

if [ $# -lt 2 ]; then
	echo `gettext SUNWhttp "usage: create_site.sh site_name site_root [sws_install_path] [site_config_file]"`
	echo `gettext SUNWhttp "provide \"-\" for optional parameters."`
	exit 1
fi

SITE_NAME=$1
SITE_ROOT=$2
SWSPATH=$3
SITE_CONFIG=$4

if [ x"-" = x"$SWSPATH" ]; then
	SWSPATH="/"
fi

if [ "/" = "$SWSPATH"  ]; then
	SWSPATH=""
fi

if [ ! -d ${SWSPATH}/etc/opt/SUNWut/http ]; then
	echo
	echo `gettext SRAS "SRAS does not appear to be installed in"` $SWSPATH
	echo
	echo `gettext SRAS "usage: create_site.sh site_name site_root [sws_install_path]"`
	exit 2
fi

if [ "/" = "$SWSPATH"  ]; then
	SWSPATH=""
fi

if [ x"-" = x"$SITE_CONFIG" ]; then
	SITE_CONFIG=${SITE_ROOT}/conf/${SITE_NAME}.site.conf;
fi

#-------------------------------------------------------------------------------
# Body - Build the site root area from the template tree.
#        We do nothing if the site configuration file already exists.
#        (This script is called from 'hthost add', which can be used
#        whether or not the host configuration files exist.)

if [ ! -f ${SITE_CONFIG} ]; then

	if [ ! -d $SITE_ROOT/conf ]; then

	    if [ ! -d ${SWSPATH}/etc/opt/SUNWut/http/site_template ]; then
		echo `gettext SUNWhttp "The site template directory is not installed."`
		echo `gettext SUNWhttp "No site definition can be created."`
		exit 1
	    fi;

	    echo `gettext SUNWhttp "Creating site directory:"`
		echo "$SITE_ROOT"

	    mkdir -p $SITE_ROOT
	    chown -f adm:adm ${SITE_ROOT}

	    ConservativeCopyTree ${SWSPATH}/etc/opt/SUNWut/http/site_template ${SITE_ROOT}

	    # This is a template leftover, not the real site config file.
	    /bin/rm -f ${SITE_ROOT}/conf/site.conf

	    PutFile ${SITE_ROOT}/conf/map.conf ${SITE_ROOT}/conf/ins.map.$$
	    mv -f ${SITE_ROOT}/conf/ins.map.$$ ${SITE_ROOT}/conf/map.conf
	    chown -f adm ${SITE_ROOT}/conf/map.conf
	    chgrp -f adm ${SITE_ROOT}/conf/map.conf
	fi

	# Generate the site configuration file.
	# BUG - need to account for ownership - is adm the only choice?
	#

	if [ -f ${SWSPATH}/etc/opt/SUNWut/http/site_template/conf/site.conf ]; then

		echo `gettext SUNWhttp "Creating site configuration:"`
		echo "${SITE_CONFIG}"

		PutFile ${SWSPATH}/etc/opt/SUNWut/http/site_template/conf/site.conf \
	        	${SITE_CONFIG};

		chown -f adm ${SITE_CONFIG};
		chgrp -f adm ${SITE_CONFIG};
	else
		echo `gettext SUNWhttp "The site template is not installed."`
		echo `gettext SUNWhttp "No site definition can be created."`
		exit 1
	fi;
fi

