package com.wolfram.databaselink;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Calendar;

import com.wolfram.jlink.Expr;

public class SQLStatementProcessor
{

  private static final Expr SYM_SQLBINARY = new Expr(Expr.SYMBOL, "SQLBinary");
  private static final Expr SYM_SQLDATETIME = new Expr(Expr.SYMBOL, "SQLDateTime");
  private static final Expr SYM_SQLEXPR = new Expr(Expr.SYMBOL, "SQLExpr");
  private static final Expr SYM_NULL = new Expr(Expr.SYMBOL, "Null");

  public static Object[] processSQLStatement(
    Connection connection,
    String sql,
    Expr params,
    int maxrows,
    int timeout,
    boolean getAsStrings,
    boolean showColumnHeadings,
    boolean returnResultSet,
    boolean returnGeneratedKeys,
    int resultSetType,
    int resultSetConcurrency, 
    boolean escapeProcessing,
    int fetchDirection, 
    int fetchSize,
    int maxFieldSize) throws Exception
  {
    PreparedStatement ps = null;
    if(returnGeneratedKeys)
      ps = connection.prepareStatement(sql, 1);
    else
      ps = connection.prepareStatement(sql, resultSetType, resultSetConcurrency);

    boolean batch = true;
    boolean k = false;
    int [] intArray = null;

    if(maxrows > 0)
      ps.setMaxRows(maxrows);

    if(timeout > 0)
      ps.setQueryTimeout(timeout);

    if(fetchDirection > 0)
        ps.setFetchDirection(fetchDirection);

    if(fetchSize > 0)
        ps.setFetchSize(fetchSize);

    if(maxFieldSize > 0)
        ps.setMaxFieldSize(maxFieldSize);
    
    ps.setEscapeProcessing(escapeProcessing);

    for(int h = 1; h <= params.length(); h++)
    {
      Expr list = params.part(h);
      for(int i = 1; i <= list.length(); i++)
      {
        Expr e = list.part(i);
        if(e.realQ())
          ps.setDouble(i, e.asDouble());
        else if(e.integerQ())
          ps.setLong(i, e.asLong());
        else if(e.stringQ())
          ps.setString(i, e.asString());
        else if(e.equals(Expr.SYM_TRUE) || e.equals(Expr.SYM_FALSE))
          ps.setBoolean(i, e.trueQ());
        else if(e.equals(SYM_NULL))
          ps.setNull(i, Types.NULL);

        else if(e.head().equals(SYM_SQLBINARY))
        {
          if(e.part(1).vectorQ(Expr.INTEGER))
          {
            int[] a = (int[])e.part(1).asArray(Expr.INTEGER, 1);
            byte[] bytes = new byte[a.length];
            for(int j = 0; j < a.length; j++)
            {
              if( a[j] > 127)
                bytes[j] = (byte)(a[j] - 256);
              else
                bytes[j] = (byte)a[j];
            }
            ps.setBytes(i, bytes);
          }
          else
          {
            byte[] bytes = new byte[e.length()];
            for(int j = 1; j <= e.length(); j++)
            { 
              Expr a = e.part(j);
              if(a.integerQ())
              {
                int b = a.asInt();
                if( b > 127)
                  bytes[j-1] = (byte)(b - 256);
                else
                  bytes[j-1] = (byte)b;
              }
              else
              {
                throw new Exception("SQLBinary may only contain integers from 0 to 255.");
              }
            }
            ps.setBytes(i, bytes);
          }
        }
        else if(e.head().equals(SYM_SQLDATETIME))
        {
          if(e.part(1).listQ())
            e = e.part(1);
            Calendar cal = Calendar.getInstance();
          if(e.length() == 6)
          {
            int nanval = 0;
            if(e.part(1).integerQ())
              cal.set(Calendar.YEAR, e.part(1).asInt());
            else
              throw new Exception("Illegal value for year in SQLDateTime: " + e.part(1).toString());
            if(e.part(2).integerQ())
              cal.set(Calendar.MONTH, e.part(2).asInt()-1);
            else
              throw new Exception("Illegal value for month in SQLDateTime: " + e.part(2).toString());
            if(e.part(3).integerQ())
              cal.set(Calendar.DATE, e.part(3).asInt());
            else
              throw new Exception("Illegal value for date in SQLDateTime: " + e.part(3).toString());
            if(e.part(4).integerQ())
              cal.set(Calendar.HOUR_OF_DAY, e.part(4).asInt());
            else
              throw new Exception("Illegal value for hour in SQLDateTime: " + e.part(4).toString());
            if(e.part(5).integerQ())
              cal.set(Calendar.MINUTE, e.part(5).asInt());
            else
              throw new Exception("Illegal value for minute in SQLDateTime: " + e.part(5).toString());
            if(e.part(6).realQ())
            {
              double dbval = e.part(6).asDouble();
              int secval = new Double(dbval).intValue();
              nanval = new Double((dbval - secval) * 1000000000).intValue();
              cal.set(Calendar.SECOND, secval);
            }
            else if(e.part(6).integerQ())
              cal.set(Calendar.SECOND, e.part(6).asInt());
            else
              throw new Exception("Illegal value for second in SQLDateTime: " + e.part(6).toString());
            Timestamp ts = new Timestamp(cal.getTime().getTime());
            ts.setNanos(nanval);
            ps.setTimestamp(i, ts);
          }
          else if(e.length() == 3)
          {
            if(e.part(1).integerQ() && e.part(1).asInt() > 24)
            {
              cal.set(Calendar.YEAR, e.part(1).asInt());
              if(e.part(2).integerQ())
                cal.set(Calendar.MONTH, e.part(2).asInt()-1);
              else
                throw new Exception("Illegal value for month in SQLDateTime: " + e.part(2).toString());
              if(e.part(3).integerQ())
                cal.set(Calendar.DATE, e.part(3).asInt());
              else
                throw new Exception("Illegal value for date in SQLDateTime: " + e.part(3).toString());
              
              Timestamp ts = new Timestamp(cal.getTime().getTime());
              ps.setTimestamp(i, ts);
            }
            else if(e.part(1).integerQ())
            {
              cal.set(Calendar.HOUR_OF_DAY, e.part(1).asInt());
              if(e.part(2).integerQ())
                cal.set(Calendar.MINUTE, e.part(2).asInt());
              else
                throw new Exception("Illegal value for minute in SQLDateTime: " + e.part(2).toString());
              if(e.part(3).integerQ())
                cal.set(Calendar.SECOND, e.part(3).asInt());
              else
                throw new Exception("Illegal value for second in SQLDateTime: " + e.part(3).toString());
              Time time = new Time(cal.getTime().getTime());
              ps.setTime(i, time);
            }
            else
              throw new Exception("Illegal value: " + e.toString());
          }
          else
            throw new Exception("Illegal value: " + e.toString());
        }
        else if(e.head().equals(SYM_SQLEXPR))
        {
          ps.setString(i, e.part(1).asString());
        }
        else
          throw new Exception("Illegal value: " + e.toString());
      }
      if(batch && params.length() > 1)
      {
        try
        {
          ps.addBatch();
        } 
        catch(Exception e)
        {
          if(intArray == null)
            intArray = new int[params.length()];
          batch = false;
          ps.execute();
          intArray[h-1] = ps.getUpdateCount();
        }
      }
      else
      {
        if(intArray == null)
          intArray = new int[params.length()];
        k = ps.execute();
        if(!k)
          intArray[h-1] = ps.getUpdateCount();
      } 
    }

    if(batch && params.length() > 1)
      intArray = ps.executeBatch();

    ResultSet rs = null;
    /* Generated Keys */
    if(returnGeneratedKeys) 
    {
        rs =  ps.getGeneratedKeys();
        if(returnResultSet)
          return new ResultSet[] { rs };
        Object[] results = getAllResultData(rs, getAsStrings, showColumnHeadings);
        ps.close();
        return results;
    }
    /* select statements */
    else if(k)
    {
      rs =  ps.getResultSet();
      if(returnResultSet)
        return new ResultSet[] { rs };
      Object[] results = getAllResultData(rs, getAsStrings, showColumnHeadings);
      ps.close();
      return results;
    }
    /* batch statements */
    else if(intArray != null)
    {
      Integer[] integerArray = new Integer[intArray.length];
      for(int l = 0; l < intArray.length; l++)
      {
        integerArray[l] = new Integer(intArray[l]);
      }
      ps.close();
      return integerArray;
    }
    /* insert, update, remove statements */
    int updateCount = ps.getUpdateCount();
    ps.close();
    return new Integer[] { new Integer(updateCount) };
  }

  public static Object[] getHeadings(ResultSet rs, boolean tables) throws Exception
  {
      ResultSetMetaData meta = rs.getMetaData();
      Object[] headings = new Object[meta.getColumnCount()];
      for(int i = 0; i < meta.getColumnCount();i++)
      {
        if(tables)
        {
            String[] col = new String[2];
            col[0] = meta.getTableName(i+1);
            col[1] = meta.getColumnName(i+1);
            headings[i] = col;
        }
        else
          headings[i] = meta.getColumnName(i+1);
      }      
      return headings;
  }
  
  public static Object[] getLimitedResultData(
          int limit, 
          ResultSet rs, 
          boolean getAsStrings) throws Exception
  {
      
    ArrayList data = new ArrayList();
    int[] columnTypes = getColumnTypes(rs);
    
    boolean valid = false;    
    if(limit == 0)
    {
      Object[] row = getRow(rs, columnTypes, getAsStrings);
      data.add(row);        
    }
    if(limit > 0)
    {
      Object[] row;
      for(int j = 0; j < limit; j++)
      {
        valid = rs.next();
        if(!valid)
          break;
        row = getRow(rs, columnTypes, getAsStrings);
        data.add(row);
      }
    }
    if(limit < 0)
    {    
      Object[] row;
      for(int k = 0; k > limit; k--)
      {
        valid = rs.previous();
        if(!valid)
          break;
        row = getRow(rs, columnTypes, getAsStrings);
        data.add(row);            
      }        
    }        
    if(data.size() == 0 && !valid)
      return null;
      
    return data.toArray(new Object[data.size()]);
  }

  public static Object[] getAllResultData(
          ResultSet rs, 
          boolean getAsStrings, 
          boolean showColumnHeadings) throws Exception
  {
    boolean valid = rs.next();
    
    ArrayList data = new ArrayList();
    if(showColumnHeadings)
        data.add(getHeadings(rs, false));
    int[] columnTypes = getColumnTypes(rs);
    
    while(valid)
    {
      Object[] row = getRow(rs, columnTypes, getAsStrings);
      data.add(row);
      valid = rs.next();            
    }
    
    return data.toArray(new Object[data.size()]);
  }
  
  private static Object[] getRow(ResultSet rs, int[] columnTypes, boolean getAsStrings) throws Exception
  {
    int cc = columnTypes.length;
    Object[] row = new Object[cc];
    if(getAsStrings)
    {
      for(int j = 0; j < cc; j++)
      {
        row[j] = rs.getString(j+1);
      }
    }
    else
    {
      for(int j = 0; j < cc; j++)
      {
        int ct = columnTypes[j];
        if (ct == 4 || ct == -7 || ct == 16 || ct == 6 || ct == 8 || ct == -5 || ct == 7 ||
            ct == 5 || ct == -6 || ct == 2 || ct == 3)
          row[j] = rs.getObject(j+1);  
        else if (ct == -2 || ct == -3 || ct == -4)
        {
          byte[] bytes = rs.getBytes(j+1);
          if(bytes != null)
          {
            int[] a = new int[bytes.length];
            for(int k = 0; k < bytes.length; k ++)
            {
              if(bytes[k] < 0)
                a[k] = bytes[k] + 256;
              else
                a[k] = bytes[k];
            }
            row[j] = new Expr(new Expr(Expr.SYMBOL, "SQLBinary"), new Expr[] {new Expr(a)});
          }
          else
          {
            row[j] = SYM_NULL;
          }
        }
        else if (ct == 91)
        {
          Date d = rs.getDate(j+1);
          if(d != null)
          {
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date(d.getTime()));
            row[j] = new Expr(
               new Expr(Expr.SYMBOL, "SQLDateTime"),
               new Expr[]
               {
                 new Expr(
                   new Expr(Expr.SYMBOL, "List"),
                   new Expr[] {
                     new Expr(cal.get(Calendar.YEAR)),
                     new Expr(cal.get(Calendar.MONTH)+1),
                     new Expr(cal.get(Calendar.DATE))
                   })
               });
          }
          else
            row[j] = SYM_NULL;
        }
        else if (ct == 92)
        {
          Time t = rs.getTime(j+1);
          if(t != null)
          {
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date(t.getTime()));
            row[j] = new Expr(
               new Expr(Expr.SYMBOL, "SQLDateTime"),
               new Expr[] {
                 new Expr(
                   new Expr(Expr.SYMBOL, "List"),
                   new Expr[]
                     {
                       new Expr(cal.get(Calendar.HOUR_OF_DAY )),
                       new Expr(cal.get(Calendar.MINUTE)),
                       new Expr(cal.get(Calendar.SECOND))
                     })
               });
          }
          else
            row[j] = SYM_NULL;
        }
        else if (ct == 93)
        {
          Timestamp ts = rs.getTimestamp(j+1);
          if(ts != null)
          {
            Calendar cal = Calendar.getInstance();
            cal.setTime(new Date(ts.getTime()));
            row[j] = new Expr(
               new Expr(Expr.SYMBOL, "SQLDateTime"),
               new Expr[]
               {
                 new Expr(
                   new Expr(Expr.SYMBOL, "List"),
                   new Expr[] {
                     new Expr(cal.get(Calendar.YEAR)),
                     new Expr(cal.get(Calendar.MONTH)+1),
                     new Expr(cal.get(Calendar.DATE)),
                     new Expr(cal.get(Calendar.HOUR_OF_DAY )),
                     new Expr(cal.get(Calendar.MINUTE)),
                     new Expr(cal.get(Calendar.SECOND)+(new Integer(ts.getNanos()).doubleValue()/1000000000))
                   })
               });
          } else
            row[j] = SYM_NULL;
        }
        else
        {
          String val = rs.getString(j+1);
          if(val != null && val.startsWith("SQLExpr["))
              row[j] = new Expr(
                         new Expr(Expr.SYMBOL, "ToExpression"),
                         new Expr[]{ new Expr(val) });
          else
            row[j] = val;
        }
      }
    }
    return row;
  }
  
  private static int[] getColumnTypes(ResultSet rs) throws Exception
  {
      ResultSetMetaData meta = rs.getMetaData();
      int cc = meta.getColumnCount();
      int[] columnTypes = new int[cc];
      for(int j = 0; j < cc; j++)
      {
        columnTypes[j] = meta.getColumnType(j+1);
      }
      return columnTypes;
  }
  
  public static Object[] getConnectionMetaData(Connection conn) throws Exception
  {
    Object[] data = new Object[2];
    DatabaseMetaData metaData = conn.getMetaData();
    String defaultTIL = "Undefined";    
    switch (metaData.getDefaultTransactionIsolation()) {
    case Connection.TRANSACTION_READ_COMMITTED: defaultTIL = "ReadCommitted"; break;
    case Connection.TRANSACTION_READ_UNCOMMITTED: defaultTIL = "ReadUncommitted"; break;
    case Connection.TRANSACTION_REPEATABLE_READ: defaultTIL = "RepeatableRead"; break;
    case Connection.TRANSACTION_SERIALIZABLE: defaultTIL = "Serializable"; break;
    }
    String sqlStateType = "Undefined";
    switch (metaData.getDefaultTransactionIsolation()) {
        case DatabaseMetaData.sqlStateSQL99: sqlStateType = "SQL99"; break;
        case DatabaseMetaData.sqlStateXOpen: sqlStateType = "XOpen"; break;
    }
        
    data[0] = new Object[] {
            "AllProceduresAreCallable", 
            "AllTablesAreSelectable", 
            "CatalogSeparator",
            "CatalogTerm",
            "DatabaseMajorVersion",
            "DatabaseMinorVersion",
            "DatabaseProductName",
            "DatabaseProductVersion",
            "DataDefinitionCausesTransactionCommit", 
            "DataDefinitionIgnoredInTransactions", 
            "DefaultTransactionIsolationLevel",
            "DeletesAreDetectedForForwardOnly", 
            "DeletesAreDetectedForScrollInsensitive", 
            "DeletesAreDetectedForScrollSensitive",
            "DoesMaxRowSizeIncludeBlobs",
            "DriverMajorVersion", 
            "DriverMinorVersion",
            "DriverName",
            "DriverVersion",
            "ExtraNameCharacters",
            "IdentifierQuoteString",
            "InsertsAreDetectedForForwardOnly", 
            "InsertsAreDetectedForScrollInsensitive", 
            "InsertsAreDetectedForScrollSensitive",
            "IsCatalogAtStartOfTableName",
            "JDBCMajorVersion", 
            "JDBCMinorVersion",
            "LocatorsUpdateCopy", 
            "MaxBinaryLiteralLength",
            "MaxCatalogNameLength", 
            "MaxCharLiteralLength", 
            "MaxColumnNameLength", 
            "MaxColumnsInGroupBy", 
            "MaxColumnsInIndex", 
            "MaxColumnsInOrderBy", 
            "MaxColumnsInSelect", 
            "MaxColumnsInTable", 
            "MaxConnections", 
            "MaxCursorNameLength", 
            "MaxIndexLength", 
            "MaxProcedureNameLength", 
            "MaxRowSize", 
            "MaxSchemaNameLength", 
            "MaxStatementLength", 
            "MaxStatements", 
            "MaxTableNameLength", 
            "MaxTablesInSelect", 
            "MaxUserNameLength", 
            "NullPlusNonNullIsNull", 
            "NullsAreSortedAtEnd", 
            "NullsAreSortedAtStart", 
            "NullsAreSortedHight", 
            "NullsAreSortedLow",
            "NumericFunctions",
            "OthersDeletesAreVisibleForForwardOnly", 
            "OthersDeletesAreVisibleForScrollInsensitive", 
            "OthersDeletesAreVisibleForScrollSensitive",
            "OthersInsertsAreVisibleForForwardOnly", 
            "OthersInsertsAreVisibleForScrollInsensitive", 
            "OthersInsertsAreVisibleForScrollSensitive",
            "OthersUpdatesAreVisibleForForwardOnly", 
            "OthersUpdatesAreVisibleForScrollInsensitive", 
            "OthersUpdatesAreVisibleForScrollSensitive", 
            "OwnDeletesAreVisibleForForwardOnly", 
            "OwnDeletesAreVisibleForScrollInsensitive", 
            "OwnDeletesAreVisibleForScrollSensitive",
            "OwnInsertsAreVisibleForForwardOnly", 
            "OwnInsertsAreVisibleForScrollInsensitive", 
            "OwnInsertsAreVisibleForScrollSensitive",
            "OwnUpdatesAreVisibleForForwardOnly", 
            "OwnUpdatesAreVisibleForScrollInsensitive", 
            "OwnUpdatesAreVisibleForScrollSensitive", 
            "ProcedureTerm",
            "ReadOnly", 
            "SchemaTerm",
            "SearchStringEscape",
            "SQLKeywords", 
            "SQLStateType",
            "StoresLowerCaseIdentifiers",
            "StoresLowerCaseQuotedIdentifiers",
            "StoresMixedCaseIdentifiers",
            "StoresMixedCaseQuotedIdentifiers",
            "StoresUpperCaseIdentifiers",
            "StoresUpperCaseQuotedIdentifiers",
            "StringFunctions", 
            "SupportsAlterTableWithAddColumn",
            "SupportsAlterTableWithDropColumn", 
            "SupportsANSI92EntryLevelSQL", 
            "SupportsANSI92FullSQL", 
            "SupportsANSI92IntermediateSQL", 
            "SupportsBatchUpdates", 
            "SupportsCatalogsInDataManipulation", 
            "SupportsCatalogsInIndexDefinitions", 
            "SupportsCatalogsInPrivilegeDefinitions", 
            "SupportsCatalogsInProcedureCalls", 
            "SupportsCatalogsInTableDefinitions", 
            "SupportsColumnAliasing", 
            "SupportsConvert", 
            "SupportsCoreSQLGrammar", 
            "SupportsCorrelatedSubqueries", 
            "SupportsDataDefinitionAndDataManipulationTransactions", 
            "SupportsDataManipulationTransactionsOnly", 
            "SupportsDifferentTableCorrelationNames", 
            "SupportsExpressionsInOrderBy", 
            "SupportsExtendedSQLGrammar", 
            "SupportsForwardOnlyResultSetReadOnlyConcurrency", 
            "SupportsForwardOnlyResultSetType", 
            "SupportsForwardOnlyResultSetUpdatableConcurrency", 
            "SupportsFullOuterJoins", 
            "SupportsGetGeneratedKeys", 
            "SupportsGroupBy", 
            "SupportsGroupByBeyondSelect", 
            "SupportsGroupByUnrelated", 
            "SupportsIntegrityEnhancementFacility", 
            "SupportsLikeEscapeClause", 
            "SupportsLimitedOuterJoins", 
            "SupportsMinimumSQLGrammar", 
            "SupportsMixedCaseIdentifiers", 
            "SupportsMixedCaseQuotedIdentifiers", 
            "SupportsMultipleOpenResults", 
            "SupportsMultipleResultSets", 
            "SupportsMultipleTransactions", 
            "SupportsNamedParameters", 
            "SupportsNonNullableColumns", 
            "SupportsOpenCursorsAcrossCommit", 
            "SupportsOpenCursorsAcrossRollback", 
            "SupportsOpenStatementsAcrossCommit", 
            "SupportsOpenStatementsAcrossRollback", 
            "SupportsOrderByUnrelated", 
            "SupportsOuterJoins", 
            "SupportsPositionedDelete", 
            "SupportsPositionedUpdate", 
            "SupportsResultSetHoldCursorsOverCommitHoldability", 
            "SupportsResultSetCloseCursorsAtCommitHoldability", 
            "SupportsSavepoints", 
            "SupportsSchemasInDataManipulation", 
            "SupportsSchemasInIndexDefinitions", 
            "SupportsSchemasInPrivilegeDefinitions", 
            "SupportsSchemasInProcedureCalls", 
            "SupportsSchemasInTableDefinitions", 
            "SupportsScrollInsensitiveResultSetReadOnlyConcurrency", 
            "SupportsScrollInsensitiveResultSetType", 
            "SupportsScrollInsensitiveResultSetUpdatableConcurrency", 
            "SupportsScrollSensitiveResultSetReadOnlyConcurrency", 
            "SupportsScrollSensitiveResultSetType", 
            "SupportsScrollSensitiveResultSetUpdatableConcurrency", 
            "SupportsSelectForUpdate", 
            "SupportsStatementPooling", 
            "SupportsStoredProcedures", 
            "SupportsSubqueriesInComparisons", 
            "SupportsSubqueriesInExists", 
            "SupportsSubqueriesInIns", 
            "SupportsSubqueriesInQuantifieds", 
            "SupportsTableCorrelationNames", 
            "SupportsReadCommitedTransactionIsolationLevel",
            "SupportsReadUncommitedTransactionIsolationLevel",
            "SupportsRepeatableReadTransactionIsolationLevel",
            "SupportsSerializableTransactionIsolationLevel",
            "SupportsTransactions", 
            "SupportsUnion", 
            "SupportsUnionAll", 
            "SystemFunctions", 
            "TimeDateFunctions",
            "UpdatesAreDetectedForForwardOnly",
            "UpdatesAreDetectedForScrollInsensitive", 
            "UpdatesAreDetectedForScrollSensitive",
            "URL", 
            "UserName", 
            "UsesLocalFilePerTable", 
            "UsesLocalFiles"
             
    };
    data[1] = new Object[] {
            new Boolean(metaData.allProceduresAreCallable()), 
            new Boolean(metaData.allTablesAreSelectable()),
            metaData.getCatalogSeparator(),
            metaData.getCatalogTerm(),
            new Integer(metaData.getDatabaseMajorVersion()),
            new Integer(metaData.getDatabaseMinorVersion()),
            metaData.getDatabaseProductName(),
            metaData.getDatabaseProductVersion(),
            new Boolean(metaData.dataDefinitionCausesTransactionCommit()), 
            new Boolean(metaData.dataDefinitionIgnoredInTransactions()),
            defaultTIL,
            new Boolean(metaData.deletesAreDetected(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.deletesAreDetected(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.deletesAreDetected(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.doesMaxRowSizeIncludeBlobs()),
            new Integer(metaData.getDriverMajorVersion()),
            new Integer(metaData.getDriverMinorVersion()),
            metaData.getDriverName(), 
            metaData.getDriverVersion(), 
            metaData.getExtraNameCharacters(),
            metaData.getIdentifierQuoteString(),
            new Boolean(metaData.insertsAreDetected(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.insertsAreDetected(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.insertsAreDetected(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.isCatalogAtStart()), 
            new Integer(metaData.getJDBCMajorVersion()), 
            new Integer(metaData.getJDBCMinorVersion()), 
            new Boolean(metaData.locatorsUpdateCopy()), 
            new Integer(metaData.getMaxBinaryLiteralLength()),
            new Integer(metaData.getMaxCatalogNameLength()), 
            new Integer(metaData.getMaxCharLiteralLength()), 
            new Integer(metaData.getMaxColumnNameLength()), 
            new Integer(metaData.getMaxColumnsInGroupBy()), 
            new Integer(metaData.getMaxColumnsInIndex()), 
            new Integer(metaData.getMaxColumnsInOrderBy()), 
            new Integer(metaData.getMaxColumnsInSelect()), 
            new Integer(metaData.getMaxColumnsInTable()), 
            new Integer(metaData.getMaxConnections()), 
            new Integer(metaData.getMaxCursorNameLength()), 
            new Integer(metaData.getMaxIndexLength()), 
            new Integer(metaData.getMaxProcedureNameLength()), 
            new Integer(metaData.getMaxRowSize()), 
            new Integer(metaData.getMaxSchemaNameLength()), 
            new Integer(metaData.getMaxStatementLength()), 
            new Integer(metaData.getMaxStatements()), 
            new Integer(metaData.getMaxTableNameLength()), 
            new Integer(metaData.getMaxTablesInSelect()), 
            new Integer(metaData.getMaxUserNameLength()), 
            new Boolean(metaData.nullPlusNonNullIsNull()), 
            new Boolean(metaData.nullsAreSortedAtEnd()), 
            new Boolean(metaData.nullsAreSortedAtStart()), 
            new Boolean(metaData.nullsAreSortedHigh()) ,
            new Boolean(metaData.nullsAreSortedLow()), 
            metaData.getNumericFunctions(), 
            new Boolean(metaData.othersDeletesAreVisible(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.othersDeletesAreVisible(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.othersDeletesAreVisible(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.othersInsertsAreVisible(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.othersInsertsAreVisible(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.othersInsertsAreVisible(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.othersUpdatesAreVisible(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.othersUpdatesAreVisible(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.othersUpdatesAreVisible(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.ownDeletesAreVisible(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.ownDeletesAreVisible(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.ownDeletesAreVisible(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.ownInsertsAreVisible(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.ownInsertsAreVisible(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.ownInsertsAreVisible(ResultSet.TYPE_SCROLL_SENSITIVE)),
            new Boolean(metaData.ownUpdatesAreVisible(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.ownUpdatesAreVisible(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.ownUpdatesAreVisible(ResultSet.TYPE_SCROLL_SENSITIVE)),
            metaData.getProcedureTerm(), 
            new Boolean(metaData.isReadOnly()), 
            metaData.getSchemaTerm(),
            metaData.getSearchStringEscape(),
            metaData.getSQLKeywords(),
            sqlStateType,
            new Boolean(metaData.storesLowerCaseIdentifiers()), 
            new Boolean(metaData.storesLowerCaseQuotedIdentifiers()), 
            new Boolean(metaData.storesMixedCaseIdentifiers()), 
            new Boolean(metaData.storesMixedCaseQuotedIdentifiers()), 
            new Boolean(metaData.storesUpperCaseIdentifiers()), 
            new Boolean(metaData.storesUpperCaseQuotedIdentifiers()), 
            metaData.getStringFunctions(), 
            new Boolean(metaData.supportsAlterTableWithAddColumn()),
            new Boolean(metaData.supportsAlterTableWithDropColumn()), 
            new Boolean(metaData.supportsANSI92EntryLevelSQL()), 
            new Boolean(metaData.supportsANSI92FullSQL()), 
            new Boolean(metaData.supportsANSI92IntermediateSQL()), 
            new Boolean(metaData.supportsBatchUpdates()), 
            new Boolean(metaData.supportsCatalogsInDataManipulation()), 
            new Boolean(metaData.supportsCatalogsInIndexDefinitions()), 
            new Boolean(metaData.supportsCatalogsInPrivilegeDefinitions()), 
            new Boolean(metaData.supportsCatalogsInProcedureCalls()), 
            new Boolean(metaData.supportsCatalogsInTableDefinitions()), 
            new Boolean(metaData.supportsColumnAliasing()), 
            new Boolean(metaData.supportsConvert()), 
            new Boolean(metaData.supportsCoreSQLGrammar()), 
            new Boolean(metaData.supportsCorrelatedSubqueries()), 
            new Boolean(metaData.supportsDataDefinitionAndDataManipulationTransactions()), 
            new Boolean(metaData.supportsDataManipulationTransactionsOnly()), 
            new Boolean(metaData.supportsDifferentTableCorrelationNames()), 
            new Boolean(metaData.supportsExpressionsInOrderBy()), 
            new Boolean(metaData.supportsExtendedSQLGrammar()), 
            new Boolean(metaData.supportsResultSetConcurrency(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY)), 
            new Boolean(metaData.supportsResultSetType(ResultSet.TYPE_FORWARD_ONLY)), 
            new Boolean(metaData.supportsResultSetConcurrency(ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_UPDATABLE)), 
            new Boolean(metaData.supportsFullOuterJoins()), 
            new Boolean(metaData.supportsGetGeneratedKeys()), 
            new Boolean(metaData.supportsGroupBy()), 
            new Boolean(metaData.supportsGroupByBeyondSelect()), 
            new Boolean(metaData.supportsGroupByUnrelated()), 
            new Boolean(metaData.supportsIntegrityEnhancementFacility()), 
            new Boolean(metaData.supportsLikeEscapeClause()), 
            new Boolean(metaData.supportsLimitedOuterJoins()), 
            new Boolean(metaData.supportsMinimumSQLGrammar()), 
            new Boolean(metaData.supportsMixedCaseIdentifiers()), 
            new Boolean(metaData.supportsMixedCaseQuotedIdentifiers()), 
            new Boolean(metaData.supportsMultipleOpenResults()), 
            new Boolean(metaData.supportsMultipleResultSets()), 
            new Boolean(metaData.supportsMultipleTransactions()), 
            new Boolean(metaData.supportsNamedParameters()), 
            new Boolean(metaData.supportsNonNullableColumns()), 
            new Boolean(metaData.supportsOpenCursorsAcrossCommit()), 
            new Boolean(metaData.supportsOpenCursorsAcrossRollback()), 
            new Boolean(metaData.supportsOpenStatementsAcrossCommit()), 
            new Boolean(metaData.supportsOpenStatementsAcrossRollback()), 
            new Boolean(metaData.supportsOrderByUnrelated()), 
            new Boolean(metaData.supportsOuterJoins()), 
            new Boolean(metaData.supportsPositionedDelete()), 
            new Boolean(metaData.supportsPositionedUpdate()), 
            new Boolean(metaData.supportsResultSetHoldability(ResultSet.HOLD_CURSORS_OVER_COMMIT)), 
            new Boolean(metaData.supportsResultSetHoldability(ResultSet.CLOSE_CURSORS_AT_COMMIT)), 
            new Boolean(metaData.supportsSavepoints()), 
            new Boolean(metaData.supportsSchemasInDataManipulation()), 
            new Boolean(metaData.supportsSchemasInIndexDefinitions()), 
            new Boolean(metaData.supportsSchemasInPrivilegeDefinitions()), 
            new Boolean(metaData.supportsSchemasInProcedureCalls()), 
            new Boolean(metaData.supportsSchemasInTableDefinitions()), 
            new Boolean(metaData.supportsResultSetConcurrency(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY)), 
            new Boolean(metaData.supportsResultSetType(ResultSet.TYPE_SCROLL_INSENSITIVE)), 
            new Boolean(metaData.supportsResultSetConcurrency(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_UPDATABLE)), 
            new Boolean(metaData.supportsResultSetConcurrency(ResultSet.TYPE_SCROLL_SENSITIVE, ResultSet.CONCUR_READ_ONLY)), 
            new Boolean(metaData.supportsResultSetType(ResultSet.TYPE_SCROLL_SENSITIVE)), 
            new Boolean(metaData.supportsResultSetConcurrency(ResultSet.TYPE_SCROLL_SENSITIVE, ResultSet.CONCUR_UPDATABLE)), 
            new Boolean(metaData.supportsSelectForUpdate()), 
            new Boolean(metaData.supportsStatementPooling()), 
            new Boolean(metaData.supportsStoredProcedures()), 
            new Boolean(metaData.supportsSubqueriesInComparisons()), 
            new Boolean(metaData.supportsSubqueriesInExists()), 
            new Boolean(metaData.supportsSubqueriesInIns()), 
            new Boolean(metaData.supportsSubqueriesInQuantifieds()), 
            new Boolean(metaData.supportsTableCorrelationNames()), 
            new Boolean(metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_READ_COMMITTED)),
            new Boolean(metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_READ_UNCOMMITTED)),
            new Boolean(metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_REPEATABLE_READ)),
            new Boolean(metaData.supportsTransactionIsolationLevel(Connection.TRANSACTION_SERIALIZABLE)),
            new Boolean(metaData.supportsTransactions()), 
            new Boolean(metaData.supportsUnion()), 
            new Boolean(metaData.supportsUnionAll()), 
            metaData.getSystemFunctions(),
            metaData.getTimeDateFunctions(),
            new Boolean(metaData.updatesAreDetected(ResultSet.TYPE_FORWARD_ONLY)),
            new Boolean(metaData.updatesAreDetected(ResultSet.TYPE_SCROLL_INSENSITIVE)),
            new Boolean(metaData.updatesAreDetected(ResultSet.TYPE_SCROLL_SENSITIVE)),
            metaData.getURL(),
            metaData.getUserName(),
            new Boolean(metaData.usesLocalFilePerTable()), 
            new Boolean(metaData.usesLocalFiles())
            
            };
    return data;
  }
}