(* :Name: NDSolveProblems` *)

(* :Title: Examples problems for NDSolve. *)

(* :Author: Mark Sofroniou *)

(* :Summary:
 This package adds a number of numerical examples for use in NDSolve.
*)

(* :Context: DifferentialEquations`NDSolveProblems` *)

(* :Package Version: 1.0 *)

(* :Copyright: Copyright 2003, Wolfram Research, Inc. *)

(* :History:
 Version 1.0 by Mark Sofroniou, April 2003.
*)

(* :Keywords:
 Numerical differential equation problems, Test problems,
 Initial Value Problems.
*)

(* :Source:
 DETEST, Bari Test set
*)

(* :Mathematica Version: 5.0 *)

(* :Limitation:
 Should allow the integration range specification through GetNDSolveProblem,
 rather than a second rule for NDSolve.
 Should allow differently named variables for each problem.
 Should allow the specification of parameters for example in
 the Van der Pol problem.
 Should allow the specification of different initial conditions
 and vector initial conditions.
*)

(* :Discussion:
 A number of common differential equation test problems are useful for
 testing and demonstrating the functionality and implementation of NDSolve.
*)

(*
 Still need to add more problems from:
 DETEST
 Bari Test set
 Hairer, Lubich and Wanner
 Hairer, Norsett and Wanner
 Hairer and Wanner
 Lambert

 Specific problems to add:
 Spherical pendulum
 *)

BeginPackage["DifferentialEquations`NDSolveProblems`"];

GetNDSolveProblem::usage =
"GetNDSolveProblem[name] returns an NDSolveProblem data structure for the problem name.";

NDSolveProblem::usage =
"NDSolveProblem is a data structure containing an example problem for use in NDSolve.";

T::usage = "T is the default symbol used for independent variables in NDSolveProblem objects.";
X::usage = "X is the default symbol used for spatial variables in NDSolveProblem objects.";
Y::usage = "Y is the default symbol used for dependent variables in NDSolveProblem objects.";

$NDSolveProblems = {
"ArnoldBeltramiChildress",
"Arenstorf",
"BrusselatorODE",
"BrusselatorPDE",
"CartesianPendulum",
"CurtissHirschfelder",
"DuffingOscillator",
"ElectricalEngineering",
"HarmonicOscillator",
"HenonHeiles",
"Kepler",
"LinearTest",
"Lorenz",
"LotkaVolterra",
"Pendulum",
"PerturbedKepler",
"Pleiades",
"RigidBody",
"Robertson",
"VanderPol"
};

Unprotect[ GetNDSolveProblem, NDSolveProblem];

Begin["`Private`"];

(* Arnold, Beltrami and Childress flow *)

GetNDSolveProblem["ArnoldBeltramiChildress"] :=
NDSolveProblem[{
{Derivative[1][Subscript[Y, 1]][T] ==
 c*Cos[Subscript[Y, 2][T]] + a*Sin[Subscript[Y, 3][T]], 
 Derivative[1][Subscript[Y, 2]][T] ==
 a*Cos[Subscript[Y, 3][T]] + b*Sin[Subscript[Y, 1][T]], 
 Derivative[1][Subscript[Y, 3]][T] ==
 b*Cos[Subscript[Y, 1][T]] + c*Sin[Subscript[Y, 2][T]]} /.
   {a -> 1, b -> 1, c -> 3/4},
{Subscript[Y, 1][0] == 1/4, 
 Subscript[Y, 2][0] == 1/3, 
 Subscript[Y, 3][0] == 1/2},
{Subscript[Y, 1][T], 
 Subscript[Y, 2][T], 
 Subscript[Y, 3][T]},
{T, 0, 100},
{},
{},
{}
}];

(* Restricted three body problem *)

GetNDSolveProblem["Arenstorf"] :=
NDSolveProblem[{
{Derivative[2][Subscript[Y, 1]][T] == Subscript[Y, 1][T] - 
   (muprime*(mu + Subscript[Y, 1][T]))/D1 - 
   (mu*(-muprime + Subscript[Y, 1][T]))/D2 + 
   2*Derivative[1][Subscript[Y, 2]][T], 
 Derivative[2][Subscript[Y, 2]][T] == Subscript[Y, 2][T] - 
   (mu*Subscript[Y, 2][T])/D2 - (muprime*Subscript[Y, 2][T])/D1 - 
   2*Derivative[1][Subscript[Y, 1]][T]} /.
   {D1 -> ((mu + Subscript[Y, 1][T])^2 + Subscript[Y, 2][T]^2)^(3/2), 
    D2 -> ((-muprime + Subscript[Y, 1][T])^2 + Subscript[Y, 2][T]^2)^(3/2)} /.
   {mu -> Rationalize[0.012277471, 0], muprime -> 1 - Rationalize[0.012277471, 0]},
{Subscript[Y, 1][0] ==Rationalize[0.994, 0], Derivative[1][Subscript[Y, 1]][0] == 
  0, Subscript[Y, 2][0] == 0, Derivative[1][Subscript[Y, 2]][0] == 
  Rationalize[-2.00158510637908252240537862224, 0]},
{Subscript[Y, 1][T], Subscript[Y, 2][T]},
{T, 0, Rationalize[17.0652165601579625588917206249, 0]},
{},
{},
{}
}];

GetNDSolveProblem["BrusselatorODE"] :=
NDSolveProblem[{
{Subscript[Y, 1]'[T] == 1 - 4*Subscript[Y, 1][T] + Subscript[Y, 1][T]^2*Subscript[Y, 2][T],
  Subscript[Y, 2]'[T] == 3*Subscript[Y, 1][T] - Subscript[Y, 1][T]^2*Subscript[Y, 2][T]},
{Subscript[Y, 1][0] == 3/2, Subscript[Y, 2][0] == 3},
{Subscript[Y, 1][T], Subscript[Y, 2][T]},
{T, 0, 20},
{},
{},
{}
}];

GetNDSolveProblem["BrusselatorPDE"] :=
NDSolveProblem[{
{Derivative[0, 1][Subscript[Y, 1]][X, T] == 1 - 4*Subscript[Y, 1][X, T] + Subscript[Y, 1][X, T]^2*Subscript[Y, 2][X, T] + \[Alpha]*Derivative[2, 0][Subscript[Y, 1]][X, T], 
 Derivative[0, 1][Subscript[Y, 2]][X, T] == 3*Subscript[Y, 1][X, T] - Subscript[Y, 1][X, T]^2*Subscript[Y, 2][X, T] + \[Alpha]*Derivative[2, 0][Subscript[Y, 2]][X, T]} /.
 \[Alpha] -> 1/50,
{Subscript[Y, 1][0, T] == 1, Subscript[Y, 1][1, T] == 1, Subscript[Y, 2][0, T] == 3, Subscript[Y, 2][1, T] == 3, Subscript[Y, 1][X, 0] == 1 + Sin[2*Pi*X], Subscript[Y, 2][X, 0] == 3},
{Subscript[Y, 1][X, T], Subscript[Y, 2][X, T]},
{T, 0, 10},
{X, 0, 1},
{},
{}
}];

(* Stiff scalar example of Curtiss and Hirschfelder *)

GetNDSolveProblem["CurtissHirschfelder"] :=
NDSolveProblem[{
{Y'[T] == -\[Alpha]*(Y[T] - Cos[T])} /. \[Alpha]->2000,
{Y[0] == 0},
{Y[T]},
{T, 0, 3/2},
{},
{},
{}
}];
(* Forced planar non-autonomous differential system *)

GetNDSolveProblem["DuffingOscillator"] :=
NDSolveProblem[{
{Derivative[1][Subscript[Y, 1]][T] == Subscript[Y, 2][T], 
 Derivative[1][Subscript[Y, 2]][T] == \[Gamma]*Cos[T] + 
   Subscript[Y, 1][T] - Subscript[Y, 1][T]^3 - 
   \[Delta]*Subscript[Y, 2][T]} /. {\[Delta] -> -1/4, \[Gamma] -> 3/10},
{Subscript[Y, 1][0] == 0, Subscript[Y, 2][0] == 1},
{Subscript[Y, 1][T], Subscript[Y, 2][T]},
{T, 0, 10},
{},
{},
{}
}];

(* Tests dynamic type changes from real to complex arithmetic *)

GetNDSolveProblem["ElectricalEngineering"] :=
NDSolveProblem[{
{Subscript[Y, 1]'[T] == (2 + 4I)*Subscript[Y, 1][T] + (-2 - 2I)*Subscript[Y, 2][T] + (4 + 2 I)*Subscript[Y, 3][T], 
 Subscript[Y, 2]'[T] == (3/4 + 11/4 I)*Subscript[Y, 1][T] + (-1 - I)*Subscript[Y, 2][T] + (7/2 + 2I)*Subscript[Y, 3][T], 
 Subscript[Y, 3]'[T] == (-1/2 + 3/2I)*Subscript[Y, 1][T] - 2I Subscript[Y, 2][T] + (2 + 3I)*Subscript[Y, 3][T]},
{Subscript[Y, 1][0] == 1, Subscript[Y, 2][0] == 1 + I, Subscript[Y, 3][0] == 1 + 2I},
{Subscript[Y, 1][T], Subscript[Y, 2][T], Subscript[Y, 3][T]},
{T, 0, 10},
{},
{},
{}
}];

GetNDSolveProblem["LinearTest"] :=
NDSolveProblem[{
{Y'[T] == -Y[T]},
{Y[0] == 1},
{Y[T]},
{T, 0, 10},
{},
{},
{}
}];

GetNDSolveProblem["Lorenz"] :=
NDSolveProblem[{
{Derivative[1][Subscript[Y, 1]][T] == 
  10*(-Subscript[Y, 1][T] + Subscript[Y, 2][T]), 
 Derivative[1][Subscript[Y, 2]][T] == 28*Subscript[Y, 1][T] - 
   Subscript[Y, 2][T] - Subscript[Y, 1][T]*Subscript[Y, 3][T], 
 Derivative[1][Subscript[Y, 3]][T] == 
  Subscript[Y, 1][T]*Subscript[Y, 2][T] - (8*Subscript[Y, 3][T])/3} /.
 {\[Sigma] -> 10, r -> 28, b -> 8/3},
{Subscript[Y, 1][0] == -8, Subscript[Y, 2][0] == 8, 
 Subscript[Y, 3][0] == 27},
{Subscript[Y, 1][T], Subscript[Y, 2][T], Subscript[Y, 3][T]},
{T, 0, 16},
{},
{},
{}
}];

(* Celestial mechanics problem - seven stars in the plane *)

GetNDSolveProblem["Pleiades"] :=
NDSolveProblem[{
Table[
  {Subscript[Y, 1, i]''[T] == 
      Sum[m[j](Subscript[Y, 1, j][T] - Subscript[Y, 1, i][T])/r[i, j], {j, i - 1}] + 
        Sum[m[j](Subscript[Y, 1, j][T] - Subscript[Y, 1, i][T])/r[i, j], {j, i + 1, 7}],
    Subscript[Y, 2, i]''[T] == 
      Sum[m[j](Subscript[Y, 2, j][T] - Subscript[Y, 2, i][T])/r[i, j], {j, i - 1}] + 
        Sum[m[j](Subscript[Y, 2, j][T] - Subscript[Y, 2, i][T])/r[i, j], {j, i + 1, 7}]},
  {i, 7}] /.
{r[i_, j_]:> ((Subscript[Y, 1, i][T] - Subscript[Y, 1, j][T])^2 + (Subscript[Y, 2, i][T] - Subscript[Y, 2, j][T])^2)^(3/2), m[i_]:> i},
{Subscript[Y, 1, 1][0] == 3, Subscript[Y, 1, 2][0] == 3, 
 Subscript[Y, 1, 3][0] == -1, Subscript[Y, 1, 4][0] == -3, 
 Subscript[Y, 1, 5][0] == 2, Subscript[Y, 1, 6][0] == -2, 
 Subscript[Y, 1, 7][0] == 2, Derivative[1][Subscript[Y, 1, 1]][0] == 
  0, Derivative[1][Subscript[Y, 1, 2]][0] == 0, 
 Derivative[1][Subscript[Y, 1, 3]][0] == 0, 
 Derivative[1][Subscript[Y, 1, 4]][0] == 0, 
 Derivative[1][Subscript[Y, 1, 5]][0] == 0, 
 Derivative[1][Subscript[Y, 1, 6]][0] == 7/4, 
 Derivative[1][Subscript[Y, 1, 7]][0] == -3/2, 
 Subscript[Y, 2, 1][0] == 3, Subscript[Y, 2, 2][0] == -3, 
 Subscript[Y, 2, 3][0] == 2, Subscript[Y, 2, 4][0] == 0, 
 Subscript[Y, 2, 5][0] == 0, Subscript[Y, 2, 6][0] == -4, 
 Subscript[Y, 2, 7][0] == 4, Derivative[1][Subscript[Y, 2, 1]][0] == 
  0, Derivative[1][Subscript[Y, 2, 2]][0] == 0, 
 Derivative[1][Subscript[Y, 2, 3]][0] == 0, 
 Derivative[1][Subscript[Y, 2, 4]][0] == -5/4, 
 Derivative[1][Subscript[Y, 2, 5]][0] == 1, 
 Derivative[1][Subscript[Y, 2, 6]][0] == 0, 
 Derivative[1][Subscript[Y, 2, 7]][0] == 0},
Join[Table[Subscript[Y, 1, i][T], {i, 7}], Table[Subscript[Y, 2, i][T], {i, 7}]],
{T, 0, 3},
{},
{},
{}
}];

(* Stiff ODE modelling an electrical circuit *)

GetNDSolveProblem["VanderPol"] :=
NDSolveProblem[{
{Subscript[Y, 1]'[T] == Subscript[Y, 2][T],
 \[Epsilon]*Subscript[Y, 2]'[T] ==  -Subscript[Y, 1][T] + (1 - Subscript[Y, 1][T]^2)*Subscript[Y, 2][T]} /.
   \[Epsilon] -> 3/1000,
{Subscript[Y, 1][0] == 2, Subscript[Y, 2][0] == 0},
{Subscript[Y, 1][T], Subscript[Y, 2][T]},
{T, 0, 5/2},
{},
{},
{}
}];

(**** Problems with invariants ****)

(* Henon Heiles Hamiltonian *)

GetNDSolveProblem["HenonHeiles"] :=
Module[{ics, dvars, idata, ivar, ivar0, sdata},
idata = {T, 0, 100};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar], Subscript[Y, 3][ivar], Subscript[Y, 4][ivar]};
ics = {3/25, 3/25, 3/25, 3/25};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{Part[dvars, 3], Part[dvars, 4], -Part[dvars, 1]*(1 + 2*Part[dvars, 2]),
-Part[dvars, 1]^2 + Part[dvars, 2]*(Part[dvars, 2] - 1)}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{(Part[dvars, 3]^2 + Part[dvars, 4]^2)/2 + (Part[dvars, 1]^2 + Part[dvars, 2]^2)/2 +
Part[dvars, 1]^2 * Part[dvars, 2] - Part[dvars, 2]^3/3},
{}
}]
];

GetNDSolveProblem["CartesianPendulum"] :=
Module[{ics, dvars, idata, ivar, ivar0, sdata},
idata = {T, 0, 50};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar], Subscript[Y, 3][ivar], Subscript[Y, 4][ivar]};
ics = {1, 0, 0, 0};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{Part[dvars, 3], Part[dvars, 4],
-Part[dvars, 1]*(Part[dvars, 3]^2 + Part[dvars, 4]^2 - Part[dvars, 2])/(Part[dvars, 1]^2 + Part[dvars, 2]^2),
-1 - Part[dvars, 2]*(Part[dvars, 3]^2 + Part[dvars, 4]^2 - Part[dvars, 2])/(Part[dvars, 1]^2 + Part[dvars, 2]^2)}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{Part[dvars, 1]*Part[dvars, 3] + Part[dvars, 2]*Part[dvars, 4],
Part[dvars, 1]^2 + Part[dvars, 2]^2},
{}
}]
];

GetNDSolveProblem["HarmonicOscillator"] :=
Module[{ics, dvars, idata, ivar, ivar0, sdata},
idata = {T, 0, 10};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar]};
ics = {1, 0};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{Part[dvars, 2], -Part[dvars, 1]}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{dvars.dvars/2},
{}
}]
];

(* Two body problem *)

GetNDSolveProblem["Kepler"] :=
Module[{ics, dvars, idata, ivar, ivar0, params, sdata},
idata = {T, 0, 100};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar], Subscript[Y, 3][ivar], Subscript[Y, 4][ivar]};
(* Eccentricity e *)
params = {3/5};
ics = {1 - params[[1]], 0, 0, Sqrt[(1 + params[[1]])/(1 - params[[1]])]};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{Part[dvars, 3], Part[dvars, 4],
-(Part[dvars, 1]/(Part[dvars, 1]^2 + Part[dvars, 2]^2)^(3/2)), 
 -(Part[dvars, 2]/(Part[dvars, 1]^2 + Part[dvars, 2]^2)^(3/2))}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{(dvars[[3]]^2 + dvars[[4]]^2)/2 -1/Sqrt[dvars[[1]]^2 + dvars[[2]]^2],
-(dvars[[2]]*dvars[[3]]) + dvars[[1]]*dvars[[4]]},
{}
}]
];

(* Predator-prey model *)

GetNDSolveProblem["LotkaVolterra"] :=
Module[{ics, dvars, idata, ivar, ivar0, sdata},
idata = {T, 0, 10};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar]};
ics = {109/40, 1};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{Part[dvars, 1]*(Part[dvars, 2] - 1), Part[dvars, 2]*(2 - Part[dvars, 1])}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{Log[Part[dvars, 2]] - Part[dvars, 2] + 2*Log[Part[dvars, 1]] - Part[dvars, 1]},
{}
}]
];

(* Frictionless nonlinear pendulum *)

GetNDSolveProblem["Pendulum"] :=
Module[{ics, dvars, idata, ivar, ivar0, sdata},
idata = {T, 0, 100};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar]};
ics = {1/2, 0};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{Part[dvars, 2], -Sin[Part[dvars, 1]]}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{Part[dvars, 2]^2/2 - Cos[Part[dvars, 1]]},
{}
}]
];

(* Restricted two body problem *)

GetNDSolveProblem["PerturbedKepler"] :=
Module[{ics, dvars, idata, ivar, ivar0, params, sdata},
idata = {T, 0, 100};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar], Subscript[Y, 3][ivar], Subscript[Y, 4][ivar]};
(* Eccentricity e *)
params = {3/5};
ics = {1 - params[[1]], 0, 0, Sqrt[(1 + params[[1]])/(1 - params[[1]])]};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{dvars[[3]], dvars[[4]],
(-3*dvars[[1]])/(400*(dvars[[1]]^2 + dvars[[2]]^2)^(5/2)) - dvars[[1]]/(dvars[[1]]^2 + dvars[[2]]^2)^(3/2), 
 (-3*dvars[[2]])/(400*(dvars[[1]]^2 + dvars[[2]]^2)^(5/2)) - dvars[[2]]/(dvars[[1]]^2 + dvars[[2]]^2)^(3/2)}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{-1/(400*(dvars[[1]]^2 + dvars[[2]]^2)^(3/2)) - 1/Sqrt[dvars[[1]]^2 + dvars[[2]]^2] + (dvars[[3]]^2 + dvars[[4]]^2)/2, 
-(dvars[[2]]*dvars[[3]]) + dvars[[1]]*dvars[[4]]},
{}
}]
];

(* Euler's equations for rigid body motion *)

GetNDSolveProblem["RigidBody"] :=
Module[{ics, dvars, idata, ivar, ivar0, params, sdata},
idata = {T, 0, 32};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar], Subscript[Y, 3][ivar]};
ics = {Cos[11/10], 0,Sin[11/10]};
(* Principal moments of inertia {I[1], I[2], I[3]} *)
params = {2, 1, 2/3};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{(dvars[[2]]*dvars[[3]]*(params[[2]] - params[[3]]))/(params[[2]]*params[[3]]), 
 (dvars[[1]]*dvars[[3]]*(-params[[1]] + params[[3]]))/(params[[1]]*params[[3]]), 
 (dvars[[1]]*dvars[[2]]*(params[[1]] - params[[2]]))/(params[[1]]*params[[2]])}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{dvars.dvars, 1/2 dvars.(dvars/params)},
{}
}]
];

(* Stiff ODE modelling a chemical reaction *)

GetNDSolveProblem["Robertson"] :=
Module[{ics, dvars, idata, ivar, ivar0, params, sdata},
idata = {T, 0, 3/10};
sdata = {};
ivar = Part[idata, 1];
ivar0 = Part[idata, 2];
dvars = {Subscript[Y, 1][ivar], Subscript[Y, 2][ivar], Subscript[Y, 3][ivar]};
ics = {1, 0, 0};
params = {1/25, 1*^4, 3*^7};
NDSolveProblem[{
Thread[Equal[D[dvars, ivar],
{-Part[params, 1]*dvars[[1]] + Part[params, 2]*dvars[[2]]*dvars[[3]],
 Part[params, 1] dvars[[1]] - Part[params, 3]*dvars[[2]]^2 - Part[params, 2]*dvars[[2]]*dvars[[3]],
 Part[params, 3]*dvars[[2]]^2}
]],
Thread[Equal[dvars /. ivar->ivar0, ics]],
dvars,
idata,
sdata,
{Apply[Plus, dvars]},
{}
}]
];

(**** Data layout and communcation ****)

NDSolveProblem[_]["Methods"]:=
  {"DependentVariables", "ExactSolution", "InitialConditions", "Invariants", "Methods", "SpaceData",
   "System", "TimeData"}

NDSolveProblem[data_][("System" | "System"[])]:= Part[data, 1] /; (Length[data] == 7);
NDSolveProblem[data_][("InitialConditions" | "InitialConditions"[])]:= Part[data, 2] /; (Length[data] == 7);
NDSolveProblem[data_][("DependentVariables" | "DependentVariables"[])]:= Part[data, 3] /;(Length[data] == 7);
NDSolveProblem[data_][("TimeData" | "TimeData"[])]:= Part[data, 4] /; (Length[data] == 7);
NDSolveProblem[data_][("SpaceData" | "SpaceData"[])]:= Part[data, 5] /; (Length[data] == 7);
NDSolveProblem[data_][("Invariants" | "Invariants"[])]:= Part[data, 6] /; (Length[data] == 7);
NDSolveProblem[data_][("ExactSolution" | "ExactSolution"[])]:= Part[data, 7] /; (Length[data] == 7);

(* Overloaded definitions for NDSolve *)

IntegrationRangeQ[{_, _?NumberQ, _?NumberQ}]:= True;
IntegrationRangeQ[___]:= False;

NDSolveProblem /:
NDSolve[ndp_NDSolveProblem, opts___?OptionQ]:=
  NDSolve[{ndp["System"], ndp["InitialConditions"]}, ndp["DependentVariables"],
    ndp["TimeData"], ndp["SpaceData"], opts];

NDSolveProblem /:
NDSolve[ndp_NDSolveProblem, tdata_?IntegrationRangeQ, opts___?OptionQ]:=
  NDSolve[{ndp["System"], ndp["InitialConditions"]}, ndp["DependentVariables"],
    tdata, opts];

NDSolveProblem /:
NDSolve[ndp_NDSolveProblem, tdata_?IntegrationRangeQ, sdata_?IntegrationRangeQ, opts___?OptionQ]:=
  NDSolve[{ndp["System"], ndp["InitialConditions"]}, ndp["DependentVariables"],
    tdata, sdata, opts];

(* Not yet correct for PDEs *)

NDSolveProblem /:
NDSolve`ProcessEquations[ndp_NDSolveProblem, opts___?OptionQ]:=
  NDSolve`ProcessEquations[{ndp["System"], ndp["InitialConditions"]}, ndp["DependentVariables"],
    First[ndp["TimeData"]], opts];

(* Rule for the NDSolveProblem object and NDSolve syntax *)

wasProtected = Unprotect[NDSolve];

ndspsyntax = {"ArgumentsPattern" -> {_, Optional[_], 
     Optional[{_, _, _}], Optional[{_, _, _}], OptionsPattern[]}};

remsyntax = Rest[SyntaxInformation[NDSolve]];

SyntaxInformation[NDSolve] = Join[ndspsyntax, remsyntax];

Apply[Protect, wasProtected];

End[ ]; (* End `Private` Context. *)

SetAttributes[{GetNDSolveProblem, NDSolveProblem}, {Protected, ReadProtected}];

EndPackage[ ]; (* End package Context. *)

