(* ::Package:: *)

(* :Name: DiscreteMath`Tree` *)

(* :Context: DiscreteMath`Tree` *)

(* :Author: Wolfram Research, Inc. *)

(* :Mathematica Version: 2.0 *)

(* :Package Version: 1.1 *)

(* :Title: Basic Operations on Trees *)

(* :Copyright: Copyright 1990-2007,  Wolfram Research, Inc. *)

(* :Summary:
This package introduces functions for creating, searching and
displaying trees represented as nested lists.
*)

(* :Discussion:
Trees are represented as nested list structures, with each node having
the form {label, child1, child2, ...}.
*)

Message[General::obspkg, "DiscreteMath`Tree`"]
BeginPackage["DiscreteMath`Tree`"]


MakeTree::usage =
"MakeTree[list] creates a binary tree with each node labeled \
by an element in list."

TreeFind::usage =
"TreeFind[treelist, x] finds the largest element smaller than or equal to x in \
the list from which treelist was constructed."

ExprPlot::usage =
"ExprPlot[expr] generates a graphical representation of an \
expression, viewed as a tree."

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=
        (Message[fun::obspkgfn, fun, context];
         )

MakeTree[{}] := (issueObsoleteFunMessage[MakeTree,"DiscreteMath`Tree`"];
	{})

MakeTree[list_List] :=
   (issueObsoleteFunMessage[MakeTree,"DiscreteMath`Tree`"];
	Block[{n, t},
		n = Length[list];
		t = Transpose[{Sort[list], Range[n]}] ;
		MakeTree0[ 1, n ]
   ])

MakeTree0[i_,j_] := Block[{midpoint,diff},
	diff = j-i;
	Which[
	   diff==3, {t[[i+1]],{t[[i]],{},{}},{t[[i+2]],{},{t[[i+3]],{},{}}}},
	   diff==2, {t[[i+1]],{t[[i]],{},{}},{t[[j]],{},{}}},
	   diff==1, {t[[i]],{},{t[[j]],{},{}}},
	   diff==0, {t[[i]],{},{}},
	   True, (
			midpoint = i + Quotient[diff,2];
			{t[[midpoint]], 
				MakeTree0[i,midpoint-1],
				MakeTree0[midpoint+1,j]}
		 )
	   ]]

TreeFind[{}, e_] := (issueObsoleteFunMessage[TreeFind,"DiscreteMath`Tree`"];
	0)

TreeFind[tree_List, e_] :=
	 (issueObsoleteFunMessage[TreeFind,"DiscreteMath`Tree`"];
	Block[{found=0, bar=e},
        	TreeFind0[tree];
	        found
	 ])

TreeFind0[tree_] :=
        Block[{m, k},
        {m, k} = First[tree] ;
        Which[
              bar < m, TreeFind0[tree[[2]]],
              bar > m, found = k ;TreeFind0[tree[[3]]],
              True, found = k; Return[]
        ]]

TreeFind0[{}] = 1

p = Unprotect[TreePlot];

TreePlot[t:{{_,n_},{{_,_},__}..}, rest___?OptionQ] :=
    (issueObsoleteFunMessage[TreePlot,"DiscreteMath`Tree`"];
	TreePlot[treetorules[t /. {} :> (Sequence @@ {})], Top, n, rest])

TreePlot[{},rest___] := (issueObsoleteFunMessage[TreePlot,"DiscreteMath`Tree`"];
	Graphics[{}])

Protect @@ p;

treetorules[{{_, n_}, b___}] :=
    Flatten[{Map[(n -> #[[1,2]])&, {b}], treetorules /@ {b}}]

ExprPlot[expr_, opts___?OptionQ] :=
    (issueObsoleteFunMessage[ExprPlot,"DiscreteMath`Tree`"];
	Block[{$nodecount = 0}, TreePlot[Flatten[exprtorules[expr, 0]], Top, 0, opts]])

exprtorules[e_[n__], count_] :=
    MapThread[{count -> #2, exprtorules[#1, #2]}&,
              {{n}, Map[(++$nodecount)&, {n}]}]

exprtorules[any_, ___] := {}

End[]
EndPackage[]
