(* ::Package:: *)

(* :Title: Rotations *)

(* :Author: Stephen Wolfram *)

(* :Summary:
This package provides functions for rotating points
in 2 and 3 dimensions.
*)

(* :Context: Geometry`Rotations` *)

(* :Package Version: 1.1 *)

(* :Copyright: Copyright 1987-2007, Wolfram Research,Inc. *)

(* :History: *)

(* :Keywords: *)

(* :Source: *)

(* :Warning: *)

(* :Mathematica Version: 2.0 *)

(* :Limitation: *)

(* :Discussion: *)

Message[General::obspkg, "Geometry`Rotations`"]
BeginPackage["Geometry`Rotations`"]


(** Two dimensions **)

RotationMatrix2D::usage = 
"RotationMatrix2D[theta] gives the matrix for rotation by angle \
theta in two dimensions."

Rotate2D::usage =
"Rotate2D[vec, theta, (pt:{0,0})] rotates the vector vec by angle \
theta about point pt."

(** Three dimensions **)
 
RotationMatrix3D::usage =
"RotationMatrix3D[phi, theta, psi] gives the matrix for rotation by \
the specified Euler angles in three dimensions."

Rotate3D::usage =
"Rotate3D[vec, phi, theta, psi, (pt:{0,0,0})] rotates the vector vec \
by specified Euler angles about point pt."

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=
        (Message[fun::obspkgfn, fun, context];
         )

RotationMatrix2D[theta_] :=
	(issueObsoleteFunMessage[RotationMatrix2D,"Geometry`Rotations`"];
{{Cos[theta], Sin[theta]}, {-Sin[theta], Cos[theta]}})

Rotate2D[vec:{_,_}, theta_] :=
	(issueObsoleteFunMessage[Rotate2D,"Geometry`Rotations`"];
RotationMatrix2D[theta] . vec)

Rotate2D[vec:{_,_}, theta_, pt:{_,_}] :=
	(issueObsoleteFunMessage[Rotate2D,"Geometry`Rotations`"];
pt + RotationMatrix2D[theta] . (vec - pt))

RotationMatrix3D[phi_,theta_,psi_] := 
	(issueObsoleteFunMessage[RotationMatrix3D,"Geometry`Rotations`"];
{ { Cos[psi] Cos[phi] - Cos[theta] Sin[phi] Sin[psi],
	    Cos[psi] Sin[phi] + Cos[theta] Cos[phi] Sin[psi],
	    Sin[psi] Sin[theta] } , 
	  { -Sin[psi] Cos[phi] - Cos[theta] Sin[phi] Cos[psi],
	    -Sin[psi] Sin[phi] + Cos[theta] Cos[phi] Cos[psi],
	    Cos[psi] Sin[theta] } ,
	  { Sin[theta] Sin[phi],
	    -Sin[theta] Cos[phi],
	    Cos[theta] } })

Rotate3D[vec:{_,_,_}, theta_, phi_, psi_] :=
	(issueObsoleteFunMessage[Rotate3D,"Geometry`Rotations`"];
RotationMatrix3D[theta,phi,psi] . vec)

Rotate3D[vec:{_,_,_}, theta_, phi_, psi_, pt:{_,_,_}] :=
	(issueObsoleteFunMessage[Rotate3D,"Geometry`Rotations`"];
pt + RotationMatrix3D[theta,phi,psi] . (vec - pt))

End[]

EndPackage[ ]

Null
