(* ::Package:: *)

(* :Title: ImplicitPlot *)

(* :Copyright: Copyright 1991-2007, Wolfram Research, Inc.*)

(* :Authors: Jerry B. Keiper, Wolfram Research, Inc.,
		contour plot method: Theo Gray, Jerry Glynn, Dan Grayson *)

(* :Summary:
The built-in function Plot requires one to specify an explicit function.
Many simple graphs (e.g., circles, ellipses, etc.) are not functions.
ImplicitPlot allows one to plot figures defined by equations.
*)

(* :Context: Graphics`ImplicitPlot` *)

(* :Mathematica Version: 3.0 *)

(* :Package Version: 2.2 *)

(* :History:
	V2.0 by Jerry B. Keiper, April 1991.
	V2.1 Modifications by John M. Novak, April 1992.
    V2.2 by John M. Novak, May 1999 -- modified core rangeplot routine
        to be more robust as number of solutions changes over plotting
        area.
*)

(* :Keywords: solution set, graphics *)

(* :Sources: The contour plot alternate method is from:
	Gray, Theodore and Glynn, Jerry, Exploring Mathematics
		with Mathematica, (Addison-Wesley, 1991) *)

(* :Warning: *)

(* :Limitation:
	ImplicitPlot relies on Solve for much of the work.  If Solve
	fails, no plot can be made.

	Subscripted variables (e.g., x[1], x[2]) cannot be used.
*)

Message[General::obspkg, "Graphics`ImplicitPlot`"]

Quiet[
BeginPackage["Graphics`ImplicitPlot`","Utilities`FilterOptions`"]
, {General::obspkg, General::newpkg}]


Unprotect[ImplicitPlot];

ImplicitPlot::usage =
"ImplicitPlot[eqn, {x, a, b}] draws a graph of the set of points \
that satisfy the equation eqn.  The variable x is associated with \
the horizontal axis and ranges from a to b.  The remaining \
variable in the equation is associated with the vertical axis. \
ImplicitPlot[eqn, {x, a, x1, x2, ..., b}] allows the user to specify \
values of x where special care must be exercised. \
ImplicitPlot[{eqn1, eqn2, ...}, {x, a, b}] allows more than one equation \
to be plotted, with PlotStyles set as in the Plot function. \
ImplicitPlot[eqn, {x, a, b}, {y, a, b}] uses a contour plot method of \
generating the plot.  This form does not allow specification \
of intermediate points."

Options[ImplicitPlot] =
Sort[
    {PlotPoints -> 39,
     PlotStyle -> Automatic} ~Join~
    Developer`GraphicsOptions[]
];

SetOptions[ImplicitPlot, PlotRangeClipping -> True, Axes -> Automatic]

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] := Message[General::obspkgfn, fun, context];

ImplicitPlot[eqns:{__Equal},
             xr:{_,_?NumericQ,_?NumericQ},yr:{_,_?NumericQ,_?NumericQ},
             opts___?OptionQ] :=
	(issueObsoleteFunMessage[ImplicitPlot,"Graphics`ImplicitPlot`"];
	Module[{ps,df,gr},
	{ps} = {PlotStyle}/.{opts}/.Options[ImplicitPlot];
    df = FilterOptions[{DisplayFunction}, Options[ImplicitPlot]];
	ps = cyclestyles[ps,Length[eqns]];
	gr = MapThread[ImplicitPlot[#1,xr,yr,
		ContourStyle->#2,DisplayFunction->Identity,opts]&, {eqns,ps}];
	gr = Select[gr,Head[#] === Graphics &];
	Show[gr,FilterOptions[Graphics, opts,
		Sequence @@ Options[ImplicitPlot]],df]/;
			gr =!= {}
	])
		
ImplicitPlot[eqns:{__Equal}, {x_,a_?NumericQ,m___?NumericQ,b_?NumericQ},
             opts___?OptionQ] :=
    (issueObsoleteFunMessage[ImplicitPlot,"Graphics`ImplicitPlot`"];
	Module[{ps, df, gr, ln},
	{ps} = {PlotStyle}/.{opts}/.Options[ImplicitPlot];
    df = FilterOptions[{DisplayFunction}, Options[ImplicitPlot]];
	ps = cyclestyles[ps,Length[eqns]];
	gr = MapThread[makegr[#1, {x,a,m,b},
		PlotStyle->#2,DisplayFunction->Identity,opts]&, {eqns,ps}];
	gr = Select[gr, (# =!= $Failed)&];
	Show[Graphics[gr], FilterOptions[Graphics, opts,
		Sequence @@ Options[ImplicitPlot]], df]/;
			gr=!={}
	])

ImplicitPlot[lhs_ == rhs_,
             xr:{_,_?NumericQ,_?NumericQ},yr:{_,_?NumericQ,_?NumericQ},
             opts___?OptionQ] :=
    (* outer wrapper to handle special-case options *)
  (issueObsoleteFunMessage[ImplicitPlot,"Graphics`ImplicitPlot`"];
	Module[{ps, pr},
         {ps, pr} = {PlotStyle, PlotRange}/.Flatten[{opts, Options[ImplicitPlot]}];
         pr = fixplotrange[pr];
    (* nested structure to handle direct substitution *)
	With[{sps = ps, spr = pr,
          copts = FilterOptions[ContourPlot, opts, Options[ImplicitPlot]]},
	ContourPlot[lhs - rhs, xr, yr,
        PlotRange -> spr,
		copts,
		ContourStyle -> sps,
		Contours -> {0},
		ContourLines -> True,
		ContourShading -> False
		]
	]
  ])

fixplotrange[r:(All | Automatic)] := r
fixplotrange[r:{{_,_} | All | Automatic, {_, _} | All | Automatic}] :=
    Append[r, Automatic]
fixplotrange[r:{_?NumberQ | All | Automatic, _?NumberQ | All | Automatic}] :=
    {Automatic, r, Automatic}
fixplotrange[any_] := any

ImplicitPlot[eqn_Equal, {x_,a_?NumericQ,m___?NumericQ,b_?NumericQ},
             opts___?OptionQ] :=
    Module[{ps, df, gr},
	{ps} = {PlotStyle}/.{opts}/.Options[ImplicitPlot];
    df = FilterOptions[{DisplayFunction}, Options[ImplicitPlot]];
	gr = makegr[eqn, {x,a,m,b}, PlotStyle->ps, opts];
	Show[Graphics[gr], FilterOptions[Graphics, opts,
		Sequence @@ Options[ImplicitPlot]], df]/;
			gr=!=$Failed
	]

cyclestyles[Automatic, ln_] :=
    Table[{Hue[.65 - .5 x, 1, .55]}, {x, 0, 1, 1/Max[1, (ln - 1)]}]

cyclestyles[ps_,ln_] :=
	Module[{style = ps},
		If[Head[ps] =!= List,
			style = {ps},
			If[Length[ps] == 0, 
				style = {{}}]
		];
		While[Length[style] < ln, style = Join[style,style]];
		Take[style,ln]
	]

ImplicitPlot::var =
"Equation `1` does not have a single variable other than `2`."

findy[f_, x_] :=
	Module[{nf},
	nf = Select[Union[Cases[f,
			(_Symbol | _[(_?NumberQ)...]),
				Infinity]],
		(!(NumberQ[N[#]] || #===x))&];
	If[Length[nf] == 1,
		nf[[1]],
		(* else *)
		Message[ImplicitPlot::var, f, x];
		$Failed
		]
	]

ImplicitPlot::epfail = "Equation `1` could not be solved for points to plot."

makegr[eqn_Equal, {x_, a_, m___, b_}, opts___] :=
    Module[{f = eqn[[1]] - eqn[[2]], ranges, plots, ar, y},
	If[(y = findy[eqn, x]) === $Failed, Return[$Failed]];
	ranges = Solve[f == 0 && D[f, y] == 0, {x, y}];
    If[ListQ[ranges] && Length[ranges] > 0, ranges = N[x /. ranges]];
	If[!VectorQ[ranges, NumberQ],
	    Message[ImplicitPlot::epfail, eqn];
	    Return[$Failed]];
	ranges = Select[Chop[ranges], FreeQ[#, Complex]&];
	ranges = Sort[Select[ranges, (a < # < b)&]];
	ranges = Union[Sort[Join[ranges, N[{a, m, b}]]]];
	ar = N[b-a]/10^8;
	ranges = Transpose[{Drop[ranges+ar, -1], Drop[ranges-ar, 1]}];
	(* ranges is now a (sorted) list of disjoint intervals with small
	    gaps between them where singularities probably exist. *)
	plots = Map[rangeplot[f, x, y, #, opts]&, ranges]
	];

distx[{x_, y_List}] :=
    Transpose[{Table[x, {Length[y]}], y}]

rangeplot[f_, x_, y_, {a_, b_}, opts___] :=
  Module[{pp, ps, j, multipoints, mdpt, len},
	{pp, ps} = {PlotPoints - 1, PlotStyle} /. {opts} /.
						Options[ImplicitPlot];
	If[ps === Automatic,ps = {}];
	mdpt = (a+b)/2;
	len = (b-a)/2;
	multipoints = Split[
        Map[{#,  (* a little bit of kludginess here... *)
                If[# =!= y,
                    Sort[Select[Chop[N[#]], FreeQ[#, Complex]&]]/.{} -> y,
                    #
                 ]&[y /. Solve[f==0 /. x -> #, y]]
            }&,
			Table[N[mdpt + len Cos[j Pi/pp]], {j, pp, 0, -1}]],
       (Length[Last[#1]] === Length[Last[#2]]) &];
	multipoints = Map[
        distx, Select[multipoints, (Last[First[#]] =!= y)&], {2}];
	(* connect the dots to form the various curves *)
	If[Length[multipoints] > 0,
		Map[Flatten[{ps,Line[#]}]&,
            Map[Transpose[#, {2,1,3}]&, multipoints],
            {2}
        ],
	  (* else *)
		{}
    ]
  ];

Protect[ImplicitPlot];

End[] (* "`Private`" *)

EndPackage[]  (* "Graphics`ImplicitPlot`" *)

(* :Tests: *)
(* :Examples:

ImplicitPlot[x^2 + 2 y^2 == 3, {x, -2, 2}] (* ellipse *)
ImplicitPlot[(x^2 + y^2)^2 == (x^2 - y^2), {x, -2, 2}] (*lemniscate *)
ImplicitPlot[(x^2 + y^2)^2 == 2 x y, {x, -2, 2}] (* lemniscate *)
ImplicitPlot[x^3 + y^3 == 3 x y, {x, -3, 3}] (* folium of Descarte *)
ImplicitPlot[x^2 + y^2 == x y + 3, {x, -3, 3}] (* ellipse *)
ImplicitPlot[x^2 + y^2 == 3 x y + 3, {x, -10, 10},
	PlotRange -> {{-10,10},{-10,10}}] (* hyperbola *)
ImplicitPlot[(x^2)^(1/3) + (y^2)^(1/3) == 1, {x, -1, 1}]
ImplicitPlot[(x^2)^(1/3) + (y^2)^(1/3) == 1, {x, -1, 2}]
ImplicitPlot[{(x^2 + y^2)^2 == (x^2 - y^2),
	(x^2 + y^2)^2 == 2 x y}, {x,-2,2},
	PlotStyle->{GrayLevel[0],Hue[0]}] (* combined plots *)
ImplicitPlot[{(x^2 + y^2)^2 == (x^2 - y^2),
	(x^2 + z^2)^2 == 2 x z}, {x,-2,2},
	PlotStyle->{GrayLevel[0],Dashing[{.01}]}] (* combined plots *)
ImplicitPlot[{a == b, x^2 + 2 y^2 == 3}, {x, -1, 1}] (* one bad plot *)
ImplicitPlot[x^2 + y^2 == Pi, {x, -2, 2}] (* OK eqn with 3 symbols *)
ImplicitPlot[Sin[x] == Cos[y], {x, 1.5, Pi/2, 1.7}]
(* contour method *)
ImplicitPlot[Sin[2 x] + Cos[3 y] == 1,{x,-2 Pi,2 Pi},{y,-2 Pi,2 Pi}]
ImplicitPlot[x^2 + x y + y^2 == 1,{x,-2Pi,2Pi},{y,-2Pi,2Pi}]
ImplicitPlot[x^3 + x y + y^2 == 1,{x,-2Pi,2Pi},{y,-2Pi,2Pi}]
ImplicitPlot[x^3 - x^2 == y^2 - y,{x,-1,2},{y,-1,2}]
(* failure cases *)
ImplicitPlot[a == b, {x, -1, 1}] (* bad plot *)
ImplicitPlot[x^y == y^x, {x, -1, 1}] (* bad plot *)
ImplicitPlot[{a == b, c == d}, {x, -1, 1}] (* bad plots *)
ImplicitPlot[x^2 + y^2 == z, {x, -2, 2}] (* bad eqn with 3 vars *)
ImplicitPlot[Sin[x] == y, {x, -3, 3}] (* Solve fails... *)
ImplicitPlot[Sin[x] == Cos[y], {x, -5, 5}]
ImplicitPlot[x^y == y^x, {x, -3, 3}]
*)
