(* ::Package:: *)

(* :Title: City Data *)

(* :Author: John M. Novak *)

(* :Summary:
This package provides a database for information about cities.
It is also a useful model of a database in Mathematica.
*)

(* :Copyright: The functions in this package are Copyright 1991-2007,
	Wolfram Research, Inc.  The data is from public domain sources. *)

(* :Context: Miscellaneous`CityData` *)

(* :Package Version: 1.0 *)

(* :History:
	V 1.0, April 1991, by John M. Novak.
	V 1.1, October 1993, by John M. Novak. Added countries to locations
	   and gave more precise locations for U.S. cities based on USGS data.
*)

(* :Keywords:
	cities, geography, databases
*)

(* :Sources:
	Map, a Macintosh Control Panel Device V1.1 by Mark Davis.
	Esselte Map Service, Concise Earth Book World Atlas,
		Earthbooks, Inc., 1990.
    Oxford Concise Atlas of the World,Reed International Books, Ltd. 1993.
	USGS-Geographic Names Information System File for Populated Places,
	        USGS Reston, Virgina, 1987.
*)

(* :Mathematica Version: 2.0 *)

(* :Discussion:
  The city and country names are unfortunately the English equivalents.
  This is a result of being limited to the ASCII character set.
*)

Message[General::obspkg, "Miscellaneous`CityData`"]

Quiet[
BeginPackage["Miscellaneous`CityData`",
	"Miscellaneous`Geodesy`"]
, {General::obspkg, General::newpkg}]

Miscellaneous`CityData`CityData::usage =
"CityData[city,type] gives data of the requested type \
for the city specified. CityData[type] gives a list \
of cities in the database for which information of the specified \
type is available. CityData[city] gives any information \
on city of types specified in the global variable $CityFields."

$CityFields::usage =
"$CityFields contains a list of data types (fields) in the CityData \
database."

CityPosition::usage =
"CityPosition is a type of data for CityData. Position \
information is in the form of latitude and longitude, \
expressed as {lat,long}. Coordinates will be in the form \
{d,m} or {d,m,s}."

CityDistance::usage =
"CityDistance[city1, city2] gives the distance in kilometers between \
the two cities. CityDistance[city1, city2, \
CityDistanceMethod->method] finds the distance using method, \
a pure function of two arguments, where each argument is \
a city position and the result is the distance between the two \
cities."

CityDistanceMethod::usage =
"CityDistanceMethod is an option of CityDistance; it specifies \
a pure function of two arguments, where each argument is \
a city position and the result is the distance between the two \
cities."

Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=
        (Message[fun::obspkgfn, fun, context];
         )

$CityFields = {CityPosition};

Miscellaneous`CityData`CityData::alternates =
	"Requested city not in database.  Possible alternatives: `1`";

Miscellaneous`CityData`CityData[symbol_Symbol] :=
	(issueObsoleteFunMessage[CityData,"Miscellaneous`CityData`"];
	Map[#[[1,1,1]]&,DownValues[symbol]])

Miscellaneous`CityData`CityData[cit:(_String | {__String})] :=
	(issueObsoleteFunMessage[CityData,"Miscellaneous`CityData`"];
	Module[{dat,alts},
		dat = Map[{#,#[cit]}&,$CityFields];
		dat = Select[dat,Not[MemberQ[$CityFields,Head[Last[#]]]]&];
		If[dat === {} && ((alts =
				Union @@ Map[checkspell[cit,Miscellaneous`CityData`CityData[#]]&,$CityFields]) =!=
				{}),
	                If[Length[alts] === 1 && First[First[alts]] === cit,
	                   dat = Miscellaneous`CityData`CityData[First[alts]],
			   Message[Miscellaneous`CityData`CityData::alternates,alts]
			]
		];
		dat/;dat =!= {}
	])
			
Miscellaneous`CityData`CityData[cit:(_String | {__String}),symbol_Symbol] :=
	(issueObsoleteFunMessage[CityData,"Miscellaneous`CityData`"];
	Module[{return = symbol[cit],dv,alts},
		If[Head[return] === symbol,
			dv = Miscellaneous`CityData`CityData[symbol];
			If[Length[tmp = checkclose[cit,dv]] === 1,
				return = symbol[First[tmp]],
				If[(alts = checkspell[cit,dv]) =!= {},
					Message[Miscellaneous`CityData`CityData::alternates,alts]
				]
			]
		];
		return/;Head[return] =!= symbol])

checkclose[cit_String,list_List] :=
	Select[list,(Head[#] === List && First[#] === cit)&]

checkclose[cit_List, list_List] :=
	Select[list,(Head[#] === String && # === First[cit])&]

checkspell[cit_,list_List] :=
	Module[{city = If[Head[cit] === List, First[cit],cit]},
		Select[list,If[Head[#] === List,
				StringMatchQ[city,First[#],SpellingCorrection->True,
					IgnoreCase->True],
				StringMatchQ[city,#,SpellingCorrection->True,
					IgnoreCase->True]]&]
	]

Options[CityDistance] =
	{CityDistanceMethod->SphericalDistance}

CityDistance[city1_,city2_,opts___] :=
	(issueObsoleteFunMessage[CityDistance,"Miscellaneous`CityData`"];
	Module[{meth,pos},
		{meth} = {CityDistanceMethod}/.{opts}/.Options[CityDistance];
		pos = {Miscellaneous`CityData`CityData[city1,CityPosition],
			Miscellaneous`CityData`CityData[city2,CityPosition]};
		N[Apply[meth,pos]]/;VectorQ[Flatten[pos]]
	])

(* City Information *)

CityPosition[{"Abidjan", "Ivory Coast"}] = {{5, 26}, {-3, -58}}

CityPosition[{"Abu Dhabi", "UAE"}] = {{24, 28}, {54, 25}}

CityPosition[{"Accra", "Ghana"}] = {{5, 35}, {0, -6}}

CityPosition[{"Addis Ababa", "Ethiopia"}] = {{9, 3}, {38, 42}}

CityPosition[{"Aden", "Yemen"}] = {{12, 46}, {45, 1}}

CityPosition[{"Agana", "USA", "GU"}] = {{13, 28}, {144, 45}}

CityPosition[{"Algiers", "Algeria"}] = {{36, 50}, {3, 0}}

CityPosition[{"Alma Alta", "Kazakhstan"}] = {{43, 15}, {76, 57}}

CityPosition[{"Amman", "Jordan"}] = {{31, 57}, {35, 56}}

CityPosition[{"Amsterdam", "Netherlands"}] = {{52, 21}, {4, 54}}

CityPosition[{"Anchorage", "USA", "AK"}] = {{61, 13, 5}, {-149, -54, -01}}

CityPosition[{"Andorre-la-Vella", "Andorra"}] = {{42, 31}, {1, 32}}

CityPosition[{"Ankara", "Turkey"}] = {{39, 55}, {32, 50}}

CityPosition[{"Antananarivo", "Madagascar"}] = {{-18, -55}, {47, 31}}

CityPosition[{"Apia", "Western Samoa"}] = {{-13, -50}, {-171, -50}}

CityPosition[{"Ashkhabad", "Turkmenistan"}] = {{38, 0}, {57, 50}}

CityPosition[{"Asmera", "Eritrea"}] = {{15, 19}, {38, 55}}

CityPosition[{"Asuncion", "Paraguay"}] = {{-25, -16}, {-57, -40}}

CityPosition[{"Athens", "Greece"}] = {{38, 0}, {23, 44}}

CityPosition[{"Atlanta", "USA", "GA"}] = {{33, 44, 56}, {-84, -23, -17}}

CityPosition[{"Baghdad", "Iraq"}] = {{33, 20}, {44, 26}}

CityPosition[{"Baku", "Azerbaijan"}] = {{40, 25}, {49, 45}}

CityPosition[{"Bamako", "Mali"}] = {{12, 34}, {-7, -55}}

CityPosition[{"Bandar Seri Begawan", "Brunei"}] = {{4, 52}, {115, 0}}

CityPosition[{"Bangkok", "Thailand"}] = {{13, 44}, {100, 30}}

CityPosition[{"Bangui", "Central African Republic"}] = {{4, 23}, {18, 35}}

CityPosition[{"Banjul", "The Gambia"}] = {{13, 28}, {-16, -40}}

CityPosition[{"Basseterre", "St. Christopher-Nevis"}] = {{17, 17}, {-62, -43}}

CityPosition[{"Basse-Terre", "Guadeloupe"}] = {{16, 0}, {-61, -44}}

CityPosition[{"Beijing", "China"}] = {{39, 55}, {116, 25}}

CityPosition[{"Beirut", "Lebanon"}] = {{33, 52}, {35, 30}}

CityPosition[{"Belgrade", "Yugoslavia"}] = {{44, 50}, {20, 30}}

CityPosition[{"Belmopan", "Belize"}] = {{17, 18}, {-88, -30}}

CityPosition[{"Berlin", "Germany"}] = {{52, 31}, {13, 24}}

CityPosition[{"Bern", "Switzerland"}] = {{46, 57}, {7, 30}}

CityPosition[{"Bishkek", "Kirghizia"}] = {{42, 54}, {74, 46}}

CityPosition[{"Bissau", "Guinea-Bissau"}] = {{11, 45}, {-15, -45}}

CityPosition[{"Bogota", "Colombia"}] = {{4, 36}, {-74, -5}}

CityPosition[{"Bombay", "India"}] = {{18, 56}, {72, 51}}

CityPosition[{"Bonn", "Germany"}] = {{50, 44}, {7, 6}}

CityPosition[{"Boston", "USA", "MA"}] = {{42, 21, 30}, {-71, -3, -37}}

CityPosition[{"Brasilia", "Brazil"}] = {{-15, -47}, {-47, -55}}

CityPosition[{"Bratislava", "Slovak Republic"}] = {{48, 10}, {17, 7}}

CityPosition[{"Brazzaville", "Congo"}] = {{-4, -9}, {15, 12}}

CityPosition[{"Bridgetown", "Barbados"}] = {{13, 5}, {-59, -30}}

CityPosition[{"Brussels", "Belgium"}] = {{50, 50}, {4, 21}}

CityPosition[{"Bucharest", "Romania"}] = {{44, 27}, {26, 10}}

CityPosition[{"Budapest", "Hungary"}] = {{47, 30}, {19, 5}}

CityPosition[{"Buenos Aires", "Argentina"}] = {{-34, -37}, {-58, -24}}

CityPosition[{"Bujumbura", "Burundi"}] = {{-3, -18}, {29, 18}}

CityPosition[{"Cairo", "Egypt"}] = {{30, 3}, {31, 15}}

CityPosition[{"Calcutta", "India"}] = {{22, 35}, {88, 21}}

CityPosition[{"Canberra", "Australia"}] = {{-35, -15}, {149, 9}}

CityPosition[{"Cape Town", "South Africa"}] = {{-33, -56}, {18, 22}}

CityPosition[{"Caracas", "Venezuela"}] = {{10, 30}, {-66, -56}}

CityPosition[{"Cardiff", "United Kingdom"}] = {{51, 30}, {-3, -13}}

CityPosition[{"Castries", "St. Lucia"}] = {{14, 2}, {-60, -58}}

CityPosition[{"Cayenne", "French Guiana"}] = {{4, 55}, {-52, -18}}

CityPosition[{"Charlotte Amalie", "USA", "VI"}] = {{18, 21}, {-64, -56}}

CityPosition[{"Chicago", "USA", "IL"}] = {{41, 51, 00}, {-87, -39, 00}}

CityPosition[{"Colombo", "Sri Lanka"}] = {{6, 56}, {79, 58}}

CityPosition[{"Conakry", "Guinea"}] = {{9, 29}, {-13, -48}}

CityPosition[{"Copenhagen", "Denmark"}] = {{55, 43}, {12, 34}}

CityPosition[{"Cupertino", "USA", "CA"}] = {{37, 19, 23}, {-122, -1, -52}}

CityPosition[{"Dacca", "Bangladesh"}] = {{23, 42}, {90, 22}}

CityPosition[{"Dakar", "Sengal"}] = {{14, 34}, {-17, -29}}

CityPosition[{"Dallas", "USA", "TX"}] = {{32, 47, 00}, {-96, -48, 00}}

CityPosition[{"Damascus", "Syria"}] = {{33, 30}, {36, 15}}

CityPosition[{"Dar es Salaam", "Tanzania"}] = {{-6, -50}, {39, 12}}

CityPosition[{"Denver", "USA", "CO"}] = {{39, 44, 21}, {-104, -59, -3}}

CityPosition[{"Djakarta", "Indonesia"}] = {{-6, -8}, {106, 45}}

CityPosition[{"Djibouti", "Djibouti"}] = {{11, 30}, {43, 5}}

CityPosition[{"Doha", "Qatar"}] = {{25, 17}, {51, 32}}

CityPosition[{"Domaneab", "Nauru"}] = {{-1, 0}, {166, 0}}

CityPosition[{"Dublin", "Ireland"}] = {{53, 20}, {-6, -15}}

CityPosition[{"Dushanbe", "Tajikistan"}] = {{38, 33}, {64, 47}}

CityPosition[{"Edinburgh", "United Kingdom"}] = {{55, 57}, {-3, -13}}

CityPosition[{"El Aaiun", "Western Sahara"}] = {{27, 9}, {-13, -12}}

CityPosition[{"Fort-de-France", "Martinique"}] = {{14, 36}, {-61, -2}}

CityPosition[{"Freetown", "Sierra Leone"}] = {{8, 30}, {-13, -17}}

CityPosition[{"Funafuti", "Tuvalu"}] = {{-8, 0}, {179, 0}}

CityPosition[{"Gaborone", "Botswana"}] = {{-24, -45}, {25, 57}}

CityPosition[{"Geneva", "Switzerland"}] = {{46, 10}, {6, 9}}

CityPosition[{"Georgetown", "Cayman Islands"}] = {{19, 20}, {-81, -24}}

CityPosition[{"Georgetown", "Guyana"}] = {{6, 46}, {-58, -10}}

CityPosition["Gibraltar"] = {{36, 7}, {-5, -22}}

CityPosition[{"Godthab", "Greenland"}] = {{64, 15}, {-51, -40}}

CityPosition[{"Guatemala City", "Guatemala"}] = {{14, 38}, {-90, -21}}

CityPosition[{"Hamilton", "Bermuda"}] = {{32, 45}, {64, 30}}

CityPosition[{"Hanoi", "Vietnam"}] = {{21, 5}, {105, 55}}

CityPosition[{"Harare", "Zimbabwe"}] = {{-17, -43}, {31, 2}}

CityPosition[{"Havana", "Cuba"}] = {{22, 8}, {-82, -22}}

CityPosition[{"Helsinki", "Finland"}] = {{60, 10}, {24, 58}}

CityPosition["Hong Kong"] = {{22, 15}, {114, 10}}

CityPosition[{"Honiara", "Solomon Islands"}] = {{-9, -27}, {159, 56}}

CityPosition[{"Honolulu", "USA", "HI"}] = {{21, 18, 25}, {-157, -51, -30}}

CityPosition[{"Islamabad", "Pakistan"}] = {{33, 39}, {73, 11}}

CityPosition[{"Istanbul", "Turkey"}] = {{41, 2}, {28, 59}}

CityPosition[{"Jerusalem", "Israel"}] = {{31, 47}, {35, 14}}

CityPosition[{"Kabul", "Afghanistan"}] = {{34, 30}, {69, 12}}

CityPosition[{"Kampala", "Uganda"}] = {{0, 20}, {32, 30}}

CityPosition[{"Karachi", "Pakistan"}] = {{24, 51}, {67, 2}}

CityPosition[{"Katmandu", "Nepal"}] = {{27, 45}, {85, 21}}

CityPosition[{"Khartoum", "Sudan"}] = {{15, 36}, {32, 32}}

CityPosition[{"Kiev", "Ukraine"}] = {{50, 30}, {30, 27}}

CityPosition[{"Kigali", "Rwanda"}] = {{-1, -59}, {30, 4}}

CityPosition[{"Kingston", "Jamaica"}] = {{18, 1}, {-76, -50}}

CityPosition[{"Kingstown", "St. Vincent-Grenadines"}] = {{13, 10}, {-61, 10}}

CityPosition[{"Kinshasa", "Zaire"}] = {{-4, -20}, {15, 14}}

CityPosition[{"Kishinev", "Moldavia"}] = {{47, 0}, {28, 50}}

CityPosition[{"Kolonia", "USA", "FM"}] = {{7, 45}, {158, 5}}

CityPosition[{"Koror", "Belau"}] = {{7, 20}, {134, 28}}

CityPosition[{"Kuala Lumpur", "Malaysia"}] = {{3, 9}, {101, 42}}

CityPosition[{"Kuwait City", "Kuwait"}] = {{29, 20}, {48, 0}}

CityPosition[{"Lagos", "Nigeria"}] = {{6, 27}, {3, 23}}

CityPosition[{"La Paz", "Bolivia"}] = {{-16, -30}, {-68, -9}}

CityPosition[{"Las Palmas", "Canary Islands"}] = {{28, 7}, {-15, -26}}

CityPosition[{"Libreville", "Gabon"}] = {{0, 25}, {9, 26}}

CityPosition[{"Lilongwe", "Malawi"}] = {{14, -1}, {33, 47}}

CityPosition[{"Lima", "Peru"}] = {{-12, -3}, {-77, -3}}

CityPosition[{"Lisbon", "Portugal"}] = {{38, 44}, {-9, -8}}

CityPosition[{"Ljubljana", "Slovenia"}] = {{46, 4}, {14, 33}}

CityPosition[{"Lome", "Togo"}] = {{6, 9}, {1, 20}}

CityPosition[{"London", "United Kingdom"}] = {{51, 30}, {0, -10}}

CityPosition[{"Los Angeles", "USA", "CA"}] = {{34, 3, 8}, {-118, -14, -34}}

CityPosition[{"Luanda", "Angola"}] = {{-8, -50}, {13, 15}}

CityPosition[{"Lusaka", "Zambia"}] = {{-15, -29}, {28, 16}}

CityPosition[{"Luxembourg", "Luxembourg"}] = {{49, 36}, {6, 9}}

CityPosition[{"Madrid", "Spain"}] = {{40, 25}, {-3, -43}}

CityPosition[{"Malabo", "Equatorial Guinea"}] = {{3, 45}, {8, 50}}

CityPosition[{"Male", "Maldives"}] = {{5, 0}, {73, 0}}

CityPosition[{"Managua", "Nicaragua"}] = {{12, 6}, {-86, -20}}

CityPosition[{"Manama", "Bahrain"}] = {{26, 12}, {50, 35}}

CityPosition[{"Manila", "Philippines"}] = {{14, 36}, {120, 59}}

CityPosition[{"Maputo", "Mozambique"}] = {{-25, -57}, {32, 33}}

CityPosition[{"Maseru", "Lesotho"}] = {{-29, -18}, {27, 30}}

CityPosition[{"Mbabane", "Swaziland"}] = {{-26, -18}, {31, 6}}

CityPosition[{"Mecca", "Saudi Arabia"}] = {{21, 27}, {39, 49}}

CityPosition[{"Mexico City", "Mexico"}] = {{19, 24}, {-99, -9}}

CityPosition[{"Miami", "USA", "FL"}] = {{25, 46, 26}, {-80, -11, -38}}

CityPosition[{"Minneapolis", "USA", "MN"}] = {{44, 58, 48}, {-93, -15, -49}}

CityPosition[{"Minsk", "Belorussia"}] = {{53, 53}, {27, 30}}

CityPosition[{"Mogadisho", "Somalia"}] = {{2, 2}, {45, 21}}

CityPosition["Monaco"] = {{43, 46}, {7, 23}}

CityPosition[{"Monrovia", "Liberia"}] = {{6, 18}, {-10, -47}}

CityPosition[{"Montevideo", "Uruguay"}] = {{-34, -55}, {-56, -10}}

CityPosition[{"Montreal", "Canada"}] = {{45, 30}, {-73, -36}}

CityPosition[{"Moroni", "Comoros"}] = {{-12, 0}, {-44, 0}}

CityPosition[{"Moscow", "Russia"}] = {{55, 45}, {37, 35}}

CityPosition[{"Muscat", "Oman"}] = {{23, 29}, {58, 33}}

CityPosition[{"Nairobi", "Kenya"}] = {{-1, -17}, {36, 50}}

CityPosition[{"Nassau", "Bahamas"}] = {{25, 5}, {-77, -20}}

CityPosition[{"Ndjamena", "Chad"}] = {{12, 10}, {14, 58}}

CityPosition[{"New Delhi", "India"}] = {{28, 37}, {77, 13}}

CityPosition[{"New York", "USA", "NY"}] = {{40, 42, 51}, {-74, 00, -23}}

CityPosition[{"Niamey", "Niger"}] = {{13, 27}, {2, 6}}

CityPosition[{"Nicosia", "Cyprus"}] = {{35, 10}, {33, 26}}

CityPosition[{"Nouakchott", "Mauritania"}] = {{18, 9}, {-15, -58}}

CityPosition[{"Noumea", "New Caledonia"}] = {{-22, -17}, {166,30}}

CityPosition[{"Nuku'alofa", "Tonga"}] = {{-21, -10}, {-174, 0}}

CityPosition[{"Nukunonu", "Tokelau"}] = {{-9, 0}, {-171, -45}}

CityPosition[{"Oranjestad", "Aruba"}] = {{12, 30}, {-70, 0}}

CityPosition[{"Osaka", "Japan"}] = {{34, 40}, {135, 30}}

CityPosition[{"Oslo", "Norway"}] = {{59, 56}, {10, 45}}

CityPosition[{"Ottawa", "Canada"}] = {{45, 25}, {-75, -42}}

CityPosition[{"Ouagadougou", "Burkina Faso"}] = {{12, 25}, {-1, -30}}

CityPosition[{"Pago Pago", "American Samoa"}] = {{-14, -16}, {-170, -44}}

CityPosition[{"Panama City", "Panama"}] = {{9, 1}, {-75, -25}}

CityPosition[{"Paramaribo", "Surinam"}] = {{5, 52}, {-55, -10}}

CityPosition[{"Paris", "France"}] = {{48, 52}, {2, 20}}

CityPosition[{"Peking", "China"}] = {{39, 55}, {116, 25}}

CityPosition[{"Perth", "Australia"}] = {{-31, -56}, {115, 50}}

CityPosition[{"Philadelphia", "USA", "PA"}] = {{39, 57, 08}, {-75, -9, -51}}

CityPosition[{"Phnom Penh", "Cambodia"}] = {{11, 32}, {104, 55}}

CityPosition[{"Plymouth", "Montserrat"}] = {{16, 40}, {-62, -10}}

CityPosition[{"Ponta Delgada", "Azores"}] = {{38, 44}, {-29, 0}}

CityPosition[{"Port-au-Prince", "Haiti"}] = {{18, 40}, {-72, -20}}

CityPosition[{"Portland", "USA", "OR"}] = {{45, 31, 25}, {-122, -40, -30}}

CityPosition[{"Port Louis", "Mauritius"}] = {{-19, -58}, {57, 0}}

CityPosition[{"Port Moresby", "Papua New Guinea"}] = {{-9, -24}, {147, 8}}

CityPosition[{"Port of Spain", "Trinidad and Tobago"}] = {{10, 40}, {-61, -31}}

CityPosition[{"Porto-Novo", "Benin"}] = {{6, 24}, {2, 42}}

CityPosition[{"Port Vila", "Vanuatu"}] = {{-19, 0}, {168, 30}}

CityPosition[{"Prague", "Czech Republic"}] = {{50, 5}, {14, 26}}

CityPosition[{"Praia", "Cape Verde Islands"}] = {{17, 10}, {-25, -20}}

CityPosition[{"Pretoria", "South Africa"}] = {{-25, -45}, {28, 12}}

CityPosition[{"Pyongyang", "North Korea"}] = {{39, 0}, {125, 30}}

CityPosition[{"Quito", "Ecuador"}] = {{0, -14}, {-78, -30}}

CityPosition[{"Rabat", "Morocco"}] = {{34, 2}, {-6, -51}}

CityPosition[{"Rangoon", "Burma"}] = {{16, 44}, {96, 19}}

CityPosition[{"Reykjavik", "Iceland"}] = {{64, 9}, {-21, -57}}

CityPosition[{"Riga", "Latvia"}] = {{56, 53}, {24, 8}}

CityPosition[{"Rio de Janeiro", "Brazil"}] = {{-22, -53}, {-43, -17}}

CityPosition[{"Riyadh", "Saudi Arabia"}] = {{24, 39}, {46, 43}}

CityPosition[{"Road Town", "United Kingdom", "Virgin Islands"}] = {{18, 26}, {-64, -37}}

CityPosition[{"Rome", "Italy"}] = {{41, 53}, {12, 30}}

CityPosition[{"Roseau", "Dominica"}] = {{15, 20}, {-61, -24}}

CityPosition[{"Saint Louis", "USA", "MO"}] = {{38, 37, 38}, {-90, -11, -52}}

CityPosition[{"Salt Lake City", "USA", "UT"}] = {{40, 44, 39}, {-111, -53, -25}}

CityPosition[{"Sana", "Yemen"}] = {{15, 27}, {44, 12}}

CityPosition[{"San Diego", "USA", "CA"}] = {{32, 42, 55}, {-117, -9, -23}}

CityPosition[{"San Francisco", "USA", "CA"}] = {{37, 46, 30}, {-122, -25, -6}}

CityPosition[{"San Jose", "Costa Rica"}] = {{9, 59}, {-84, -4}}

CityPosition[{"San Juan", "USA", "PR"}] = {{18, 28}, {-66, -7}}

CityPosition[{"San Marino", "San Marino"}] = {{43, 56}, {12, 25}}

CityPosition[{"Santiago", "Chile"}] = {{-33, -24}, {-70, -39}}

CityPosition[{"Santo Domingo", "Dominican Republic"}] = {{18, 30}, {-69, -58}}

CityPosition[{"Sao Tome", "Sao Tome and Principe"}] = {{0, 10}, {6, 39}}

CityPosition[{"Seoul", "South Korea"}] = {{37, 33}, {126, 58}}

CityPosition[{"Shanghai", "China"}] = {{31, 14}, {121, 30}}

CityPosition[{"Singapore", "Singapore"}] = {{1, 17}, {103, 51}}

CityPosition[{"St. Denis", "Reunion"}] = {{-21, 0}, {56, 0}}

CityPosition[{"St. George's", "Grenada"}] = {{12, 5}, {-61, -43}}

CityPosition[{"St. John's", "Antigua and Barbuda"}] = {{17, 6}, {-61, -51}}

CityPosition[{"St. Peterburg", "Russia"}] = {{59, 55}, {30, 15}}

CityPosition[{"San Salvador", "El Salvador"}] = {{13, 39}, {-89, -11}}

CityPosition[{"Sao Paulo", "Brazil"}] = {{-23, -33}, {-46, -39}}

CityPosition[{"Sarajevo", "Bosnia-Herzogovina"}] = {{43, 52}, {18, 26}}

CityPosition[{"Skopje", "Macedonia"}] = {{42, 0}, {21, 32}}

CityPosition[{"Sofia", "Bulgaria"}] = {{42, 45}, {23, 20}}

CityPosition[{"Stanley", "Falkland Islands"}] = {{-51, -40}, {-59, -50}}

CityPosition[{"Stockholm", "Sweden"}] = {{59, 20}, {18, 3}}

CityPosition[{"Sucre", "Bolivia"}] = {{-19, 0}, {-65, -14}}

CityPosition[{"Suva", "Fiji"}] = {{-18, -6}, {178, 30}}

CityPosition[{"Sydney", "Australia"}] = {{-33, -52}, {151, 13}}

CityPosition[{"Taipei", "Taiwan"}] = {{25, 2}, {121, 31}}

CityPosition[{"Tallinn", "Estonia"}] = {{59, 22}, {24, 48}}

CityPosition[{"Tarawa", "Kiribati"}] = {{1, 30}, {172, 30}}

CityPosition[{"Tashkent", "Uzbekistan"}] = {{41, 20}, {69, 10}}

CityPosition[{"Tblisi", "Georgia"}] = {{41, 44}, {44, 50}}

CityPosition[{"Tegucigalpa", "Honduras"}] = {{14, 5}, {-87, -15}}

CityPosition[{"Tehran", "Iran"}] = {{35, 40}, {51, 26}}

CityPosition[{"Thimphu", "Bhutan"}] = {{27, 32}, {89, 45}}

CityPosition[{"Tianjin", "China"}] = {{39, 8}, {117, 12}}

CityPosition[{"Tirana", "Albania"}] = {{41, 18}, {19, 49}}

CityPosition[{"Tokyo", "Japan"}] = {{35, 42}, {139, 46}}

CityPosition[{"Torshavn", "Faroe Islands"}] = {{62, 30}, {-7, 0}}

CityPosition[{"Tripoli", "Libya"}] = {{32, 53}, {13, 12}}

CityPosition[{"Tunis", "Tunisia"}] = {{36, 48}, {10, 11}}

CityPosition[{"Ulan Bator", "Mongolia"}] = {{47, 56}, {106, 52}}

CityPosition[{"Vaduz", "Liechtenstein"}] = {{47, 8}, {9, 34}}

CityPosition[{"Valletta", "Malta"}] = {{35, 54}, {14, 31}}

CityPosition[{"The Valley", "Anguilla"}] = {{18, 14}, {-63, -5}}

CityPosition[{"Vancouver", "Canada"}] = {{49, 16}, {-123, -7}}

CityPosition["Vatican City"] = {{41, 53}, {12, 30}}

CityPosition[{"Victoria", "Seychelles"}] = {{-5, 0}, {56, 0}}

CityPosition[{"Vienna", "Austria"}] = {{48, 13}, {16, 22}}

CityPosition[{"Vientiane", "Laos"}] = {{17, 58}, {102, 36}}

CityPosition[{"Vilnius", "Lithuania"}] = {{54, 38}, {25, 19}}

CityPosition[{"Warsaw", "Poland"}] = {{52, 15}, {21, 0}}

CityPosition[{"Washington", "USA", "DC"}] = {{38, 53, 42}, {-77, -2, -12}}

CityPosition[{"Wellington", "New Zealand"}] = {{-41, -17}, {174, 47}}

CityPosition[{"Willemstad", "Netherlands Antilles"}] = {{12, 6}, {-69, 0}}

CityPosition[{"Windhoek", "Namibia"}] = {{-22, -35}, {17, 4}}

CityPosition[{"Yaounde", "Cameroon"}] = {{3, 50}, {11, 35}}

CityPosition[{"Yerevan", "Armenia"}] = {{40, 11}, {44, 31}}

CityPosition[{"Zagreb", "Croatia"}] = {{45, 50}, {16, 1}}

CityPosition[{"Zurich", "Switzerland"}] = {{47, 23}, {8, 33}}

End[]

EndPackage[]
