(*:Mathematica Version: 2.0 *)

(*:Package Version: 1.2 *)

(*:Name: Statistics`DescriptiveStatistics` *)

(*:Context: Statistics`DescriptiveStatistics` *)

(*:Title: Basic Descriptive Statistics *)

(*:Author: Stephen Wolfram, David Withoff *)

(*:History:
  Original version by Stephen Wolfram (Wolfram Research), April 1989.
  Revised by David Withoff (Wolfram Research), February 1990.
  Revised Skewness, Kurtosis, and KurtosisExcess measures to make use of the 
	MLE of variance rather than the unbiased estimate,
	added data transformations, ECM (Wolfram Research), September 1993.
  Added CoefficientOfVariation and ExpectedValue, ECM, September 1995.
  V1.2 -- Revisions for improved performance by Serguei Chebalov
     and John Novak, November 1999.
*)

(*:Copyright: Copyright 1990-2007, Wolfram Research Inc. *)

(*:Reference: Usage messages only. *)

(*:Summary:
This package computes descriptive statistics (location, dispersion,
and shape statistics) for samples represented by lists.
*)

(*:Keywords: Maximum likelihood estimate, unbiased estimate *)

(*:Requirements: No special system requirements. *)

(*:Warning: None. *)

(*:Sources: Basic statistics texts. *)

Message[General::obspkg,"Statistics`DescriptiveStatistics`"]

Quiet[
BeginPackage["Statistics`DescriptiveStatistics`",
             "Statistics`DataManipulation`"]
, {General::obspkg, General::newpkg}]

Unprotect[{LocationReport, InterpolatedQuantile, CoefficientOfVariation,
    Mode, DispersionReport, SampleRange, VarianceMLE,
	VarianceOfSampleMean, StandardDeviationMLE,
	StandardErrorOfSampleMean, QuantileQ,
	ShapeReport, PearsonSkewness1, PearsonSkewness2,
	KurtosisExcess, ZeroMean}];


(* Location statistics *)

LocationReport::usage =
"LocationReport[list] gives the Mean, HarmonicMean, and Median location \
statistics for list."

InterpolatedQuantile::usage =
"InterpolatedQuantile[list, q] gives the q-th quantile of the \
probability distribution inferred by linear interpolation of the \
entries in list."

If[StringQ[Mode::usage] && StringPosition[Mode::usage, "Mode[list] gives the mode"] === {},
Mode::usage = Mode::usage <> " " <>
"Mode[list] gives the mode of the entries in list. A list of modes \
is returned if the data is bimodal, trimodal, or multimodal."
]

(* Dispersion statistics *)

DispersionReport::usage =
"DispersionReport[list] gives the Variance, StandardDeviation, SampleRange, \
MeanDeviation, MedianDeviation, and QuartileDeviation dispersion statistics for \
list."

SampleRange::usage =
"SampleRange[list] gives the range of values in list."

VarianceMLE::usage =
"VarianceMLE[list] gives the variance of the entries in list. \
Division by n (rather than n-1) is used, giving a maximum \
likelihood estimate of the population variance (use Variance[list] \
for an unbiased estimate)."

VarianceOfSampleMean::usage =
"VarianceOfSampleMean[list] gives an unbiased estimate of the variance \
of the sample mean, using the entries in list as a sample from the \
population."

StandardDeviationMLE::usage =
"StandardDeviationMLE[list] gives the standard deviation of the entries \
in list. Division by n is used, giving a maximum likelihood estimate of \
the population standard deviation."

StandardErrorOfSampleMean::usage =
"StandardErrorOfSampleMean[list] gives an unbiased estimate of the \
standard error (standard deviation) of the sample mean, using the entries \
in list as a sample from the population."

CoefficientOfVariation::usage =
"CoefficientOfVariation[list] gives the coefficient of variation, defined \
as the ratio of the standard deviation to the mean of the entries in list."

(* Shape statistics *)

ShapeReport::usage =
"ShapeReport[list] gives the Skewness, QuartileSkewness, and KurtosisExcess \
shape statistics for list."

PearsonSkewness1::usage =
"PearsonSkewness1[list] gives Pearson's first coefficient of skewness of the \
entries in list. Positive skewness indicates that the mean is greater than \
the mode, negative that the mode is greater than the mean."

PearsonSkewness2::usage = 
"PearsonSkewness2[list] gives Pearson's second coefficient of skewness of the \
entries in list. Positive skewness indicates that the mean is greater than \
the median, negative that the mode is less than the median."

KurtosisExcess::usage =
"KurtosisExcess[list] gives the kurtosis excess for the entries in list. \
Positive kurtosis excess indicates a distribution that is more peaked and has \
heavier tails than a normal distribution with the same variance. Negative \
kurtosis excess indicates a distribution that is flatter or has heavier flanks \
than the normal."

(* Data transformations *)

ZeroMean::usage =
"ZeroMean[list] subtracts the mean from the data in list and returns the data \
with zero mean."

MLE::usage =
"MLE is an option of descriptive statistic functions specifying whether \
the maximum likelihood estimate (MLE) or unbiased estimate of a statistic \
should be used."

(* Utility *)

QuantileQ::usage =
"QuantileQ[q] returns True if q is a valid quantile specification \
(a number between 0 and 1)."


Begin["`Private`"]

issueObsoleteFunMessage[fun_, context_] :=Message[General::obspkgfn, fun, context]


(* sorting utility to sort lists of NumericQ objects,
   without messing with the efficiency of sorting packed arrays 
   used in  TrimmedMean, InterpolatedQuantile, etc. *)
sort[s_ /; VectorQ[s, NumberQ]] := Sort[s]

sort[s_] := Sort[s, OrderedQ[N[{#1, #2},
          Precision[s] /. Infinity -> MachinePrecision]] &]

(* Quantile::frac message generated by QuantileQ *)
Quantile::frac =
"Quantile specification `1` is expected to be between 0 and 1."

QuantileQ[q_] :=
 If[FreeQ[N[q], Complex] && !TrueQ[N[q] < 0] && !TrueQ[N[q] > 1], True,
    Message[Quantile::frac, q]; False]

InterpolatedQuantile[list_, q_] :=(
	issueObsoleteFunMessage[InterpolatedQuantile, "Statistics`DescriptiveStatistics`"];
	Block[{s, n},
		s = sort[list] ;
		n = Length[list] ;
		Which[
			q < 1/(2n),  First[s],
			q > 1-1/(2n),  Last[s],
			True,
			  Block[{f, i, u},
			    u = n q + 1/2 ;
			    i = Floor[u] ;
			    f = u - i ;
			    If[TrueQ[f == 0],
			       (1 - f) s[[i]],
			       (1 - f) s[[i]] + f s[[i+1]]
			    ]
			  ]
		]
	])  /; (Developer`PackedArrayQ[list] || VectorQ[list, NumericQ]) && Length[list] > 2 &&
	      NumberQ[N[q]] && QuantileQ[q]


Mode[list_]:=(
	issueObsoleteFunMessage[Mode, "Statistics`DescriptiveStatistics`"];
    Module[{c, mc, ms},
        c = Frequencies[list];
        If[Length[c] === 1, Return[c[[1,2]]]]; (* one data value *)
        mc = Max[First[Transpose[c]]];
        If[mc === 1, Return[{}]]; (* no mode *)
        ms = Cases[c, {mc ,val_} -> val];
        If[Length[ms] == 1, ms[[1]], ms] (* scalar for one mode, list for multi-mode *)
    ]) /; VectorQ[list] && (Length[list]>0)


LocationReport[list_] :=(
	issueObsoleteFunMessage[LocationReport, "Statistics`DescriptiveStatistics`"];
	{
	Mean -> Mean[list] ,
	HarmonicMean -> HarmonicMean[list] ,
	Median -> Median[list]
	} )  /;  VectorQ[list]

VarianceMLE[list_?VectorQ] := (
  issueObsoleteFunMessage[VarianceMLE, "Statistics`DescriptiveStatistics`"];
  With[{len=Length[list]},
	If[len===1,
		0,
		(len-1)/len*Variance[list]]
	]) /; Length[list] > 0

VarianceOfSampleMean[list_?VectorQ] :=(
    issueObsoleteFunMessage[VarianceOfSampleMean, "Statistics`DescriptiveStatistics`"];
    1/Length[list] Variance[list]) /; Length[list] > 1

StandardDeviationMLE[list_?VectorQ]:= (
	issueObsoleteFunMessage[StandardDeviationMLE, "Statistics`DescriptiveStatistics`"];
	Sqrt[VarianceMLE[list]]) /;
							Length[list] > 0

StandardErrorOfSampleMean[list_?VectorQ] := (
	issueObsoleteFunMessage[StansardErrorOfSampleMean, "Statistics`DescriptiveStatistics`"];
	Sqrt[VarianceOfSampleMean[list]])

SampleRange[list_] := (
	issueObsoleteFunMessage[SampleRange, "Statistics`DescriptiveStatistics`"];
	Max[list] - Min[list])  /; 
				VectorQ[list] && Length[list] > 0

CoefficientOfVariation::negdat =
"Warning: CoefficientOfVariation is not well defined for negative data \
points, and at least one negative value has been found in the input.";

CoefficientOfVariation[list_?VectorQ /; Length[list] > 1] := 
       (issueObsoleteFunMessage[CoefficientOfVariation, "Statistics`DescriptiveStatistics`"];
       If[covtestQ[list],
             Message[CoefficientOfVariation::negdat]
        ];
	    StandardDeviation[list]/Mean[list])

covtestQ[list_?Developer`PackedArrayQ] :=
    TrueQ[N[Total[list - Abs[list]]] =!= 0.]

covtestQ[list_] :=
    Catch[Scan[If[# <= 0, Throw[True]] &, list]; False]

DispersionReport[list_] :=(
	issueObsoleteFunMessage[DispersionReport, "Statistics`DescriptiveStatistics`"];
	{
	Variance -> Variance[list] ,
	StandardDeviation -> StandardDeviation[list] ,
	SampleRange -> SampleRange[list],
	MeanDeviation -> MeanDeviation[list] ,
	MedianDeviation -> MedianDeviation[list] ,
	QuartileDeviation -> QuartileDeviation[list]
	} )  /; VectorQ[list]

PearsonSkewness1[list_?VectorQ] :=(
	issueObsoleteFunMessage[PearsonSkewness1, "Statistics`DescriptiveStatistics`"];
	3 (Mean[list] - Mode[list]) / StandardDeviation[list])

PearsonSkewness2[list_?VectorQ] :=(
	issueObsoleteFunMessage[PearsonSkewness2, "Statistics`DescriptiveStatistics`"];
	3 (Mean[list] - Median[list]) / StandardDeviation[list])

		
KurtosisExcess[list_] := Block[{res=Kurtosis[list]},
	issueObsoleteFunMessage[KurtosisExcess, "Statistics`DescriptiveStatistics`"];
	If[Head[res] === Piecewise,
		(* special case processing for piecewise results from 
		   statistical distributions *)
 		res[[1, All, 1]] = res[[1, All, 1]] - 3;
 		res[[2]] -= 3,
		If[FreeQ[res,Kurtosis],
			res -= 3]];
	res/;FreeQ[res,Kurtosis]]


ShapeReport[list_?VectorQ] :=(
	issueObsoleteFunMessage[ShapeReport, "Statistics`DescriptiveStatistics`"];
	{
	Skewness -> Skewness[list] ,
	QuartileSkewness -> QuartileSkewness[list] ,
	KurtosisExcess -> KurtosisExcess[list]
	})

ZeroMean[a_?VectorQ] := (
	issueObsoleteFunMessage[ZeroMean, "Statistics`DescriptiveStatistics`"];
	a - Mean[a])


End[ ]

SetAttributes[{LocationReport, InterpolatedQuantile, CoefficientOfVariation,
    Mode, DispersionReport, SampleRange, VarianceMLE,
	VarianceOfSampleMean, StandardDeviationMLE, CoefficientOfVariation,
	StandardErrorOfSampleMean, QuantileQ,
	ShapeReport, PearsonSkewness1, PearsonSkewness2,
	KurtosisExcess, ZeroMean},
  {Protected, ReadProtected}];

EndPackage[ ]


