(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    112089,       3526]
NotebookOptionsPosition[     95685,       2955]
NotebookOutlinePosition[     98346,       3009]
CellTagsIndexPosition[     98303,       3006]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["COMPUTER ARITHMETIC PACKAGE TUTORIAL", "PacletNameCell"], Cell[
    TextData[Cell[BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Arithmetic" :> 
         Documentation`HelpLookup["paclet:ComputerArithmetic/ref/Arithmetic"],
          "ComputerNumber" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/ComputerNumber"], "Floor" :> 
         Documentation`HelpLookup["paclet:ref/Floor"], "MachineError" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/MachineError"], "MicroscopePlot" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/MicroscopePlot"], 
         "MicroscopicErrorPlot" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"], "NaN" :> 
         Documentation`HelpLookup["paclet:ComputerArithmetic/ref/NaN"], 
         "RoundToEven" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/RoundToEven"], "RoundToInfinity" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/RoundToInfinity"], "SetArithmetic" :> 
         Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/SetArithmetic"], "Truncation" :> 
         Documentation`HelpLookup["paclet:ComputerArithmetic/ref/Truncation"],
          "Ulp" :> Documentation`HelpLookup[
          "paclet:ComputerArithmetic/ref/Ulp"]}, #, 2]& , Evaluator -> 
       Automatic], {1->"\<\"Arithmetic\"\>", 2->"\<\"ComputerNumber\"\>", 
      3->"\<\"Floor\"\>", 4->"\<\"MachineError\"\>", 
      5->"\<\"MicroscopePlot\"\>", 6->"\<\"MicroscopicErrorPlot\"\>", 
      7->"\<\"NaN\"\>", 8->"\<\"RoundToEven\"\>", 
      9->"\<\"RoundToInfinity\"\>", 10->"\<\"SetArithmetic\"\>", 
      11->"\<\"Truncation\"\>", 12->"\<\"Ulp\"\>"}, "\<\"functions\"\>", 
      StyleBox["\<\"functions\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Computer Arithmetic Package", "Title",
 CellID->29267],

Cell[TextData[{
 "The arithmetic used by ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " is a mixture of variable precision software arithmetic and whatever is \
provided by the manufacturer of the floating\[Hyphen]point hardware (or the \
designer of the compiler, if there is no floating\[Hyphen]point hardware). If \
you want to learn about the basic ideas of computer floating\[Hyphen]point \
arithmetic in general or examine the machine arithmetic on your machine, you \
can use ",
 StyleBox["ComputerArithmetic`", "MR"],
 ". This allows you to examine arithmetic with various bases, precisions, and \
rounding rules. "
}], "Text",
 CellID->28540],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["ComputerNumber",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/ComputerNumber"], "[", 
     StyleBox["x", "TI"], "]"}], Cell[TextData[{
     "convert the ordinary number ",
     StyleBox["x", "TI"],
     " to a computer number in the arithmetic currently in effect"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["ComputerNumber",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/ComputerNumber"], "[", 
     StyleBox["sign", "TI"], ",", 
     StyleBox["mantissa", "TI"], ",", 
     StyleBox["exp", "TI"], "]"}], Cell[TextData[{
     "form the computer number with sign ",
     StyleBox["sign", "TI"],
     ", mantissa ",
     StyleBox["mantissa", "TI"],
     ", and exponent ",
     StyleBox["exp", "TI"]
    }], "TableText"]},
   {
    StyleBox[
     RowBox[{
      ButtonBox["ComputerNumber",
       BaseStyle->"Link",
       ButtonData->"paclet:ComputerArithmetic/ref/ComputerNumber"], "[", 
      StyleBox["sign", "TI"], ",", 
      StyleBox["mantissa", "TI"], ",", 
      StyleBox["exp", "TI"], ",", 
      StyleBox["value", "TI"], ",", 
      StyleBox["x", "TI"], "]"}],
     GridElementStyleOptions -> {ColumnAlignments -> Left}], 
    "\[SpanFromLeft]"},
   {"", Cell[
    "the complete data object that makes up a computer number", 
     "TableText"]},
   {
    ButtonBox["NaN",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/NaN"], Cell[
    "a nonrepresentable number in the current arithmetic ", "TableText"]}
  }]], "DefinitionBox",
 CellID->10325],

Cell[TextData[{
 "Computer numbers and nonnumbers in ",
 StyleBox["ComputerArithmetic", "MR"],
 ". "
}], "Caption",
 CellID->10878],

Cell[TextData[{
 "Much of the information carried around in the data object that makes up a \
computer number is redundant. In particular, the first three arguments \
contain exactly the same information as the fourth argument. The redundancy \
exists partly for the sake of efficiency and partly to allow the user access \
to the various fields. The fifth argument has nothing to do with the computer \
number itself. It instead represents what the value of the number would be \
without the cumulative effects of all the roundoff errors that went into the \
computer number. It is computed using ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 "\[CloseCurlyQuote]s high\[Hyphen]precision arithmetic and can generally be \
regarded as the correct value of the number. Comparing the computer number \
with this number gives the error in the computer number. "
}], "Text",
 CellID->9428],

Cell["This loads the package. ", "MathCaption",
 CellID->24207],

Cell[BoxData[
 RowBox[{"<<", "ComputerArithmetic`"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30103],

Cell["\<\
Here is the computer number representing the ordinary number 2. \
\>", "MathCaption",
 CellID->26754],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"a", "=", 
  RowBox[{"ComputerNumber", "[", "2", "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6567],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"2.000000000000000000\"\>",
   2.`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->394187889]
}, Open  ]],

Cell[TextData[{
 "This gives the computer number representation of ",
 Cell[BoxData[
  FormBox["\[Pi]", TraditionalForm]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->17258],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"ComputerNumber", "[", "\[Pi]", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22857],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"3.142000000000000000\"\>",
   3.142`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->69753290]
}, Open  ]],

Cell["\<\
Arithmetic works with computer numbers just as with ordinary numbers. \
\>", "MathCaption",
 CellID->3069],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"c", "=", 
  RowBox[{"a", "+", "b"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->21922],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"5.142000000000000000\"\>",
   5.142`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->66468185]
}, Open  ]],

Cell[TextData[{
 "Here is the complete structure of ",
 StyleBox["c", "MR"],
 ". "
}], "MathCaption",
 CellID->1381],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InputForm", "[", "c", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->11821],

Cell["\<\
ComputerNumber[1, 5142, -3, 2571/500, 
 5.1415926535897932384626433832795028842`24.]\
\>", "Output",
 ImageSize->{315, 31},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]//InputForm=",
 CellID->343846521]
}, Open  ]],

Cell["You can also enter just the sign, mantissa, and exponent. ", \
"MathCaption",
 CellID->21517],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", 
  RowBox[{
   RowBox[{"-", "1"}], ",", "1234", ",", 
   RowBox[{"-", "6"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->28073],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"-0.001234000000000000000\"\>",
   -0.001234`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->480032946]
}, Open  ]],

Cell[TextData[{
 "But if your input doesn\[CloseCurlyQuote]t make sense in the arithmetic \
currently in effect you get ",
 StyleBox[ButtonBox["NaN",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/NaN"], "MR"],
 ". Here the problem is that the mantissa is only three digits long. "
}], "MathCaption",
 CellID->11046],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", 
  RowBox[{
   RowBox[{"-", "1"}], ",", "123", ",", "7"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->10805],

Cell[BoxData["NaN"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->776906102]
}, Open  ]],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["SetArithmetic",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/SetArithmetic"], "[", 
     StyleBox["n", "TI"], "]"}], Cell[TextData[{
     "set the arithmetic to be ",
     StyleBox["n", "TI"],
     " digits, base 10"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["SetArithmetic",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/SetArithmetic"], "[", 
     StyleBox["n", "TI"], ",", 
     StyleBox["b", "TI"], "]"}], Cell[TextData[{
     "set the arithmetic to be ",
     StyleBox["n", "TI"],
     " digits, base ",
     StyleBox["b", "TI"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["Arithmetic",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/Arithmetic"], "[", "]"}], 
    Cell["give the parameters of the arithmetic currently in effect", 
     "TableText"]}
  }]], "DefinitionBox",
 CellID->15268],

Cell["Changing the type of arithmetic to be used. ", "Caption",
 CellID->7103],

Cell[TextData[{
 "The default arithmetic is four digits in base 10 with a rounding rule of ",
 StyleBox[ButtonBox["RoundToEven",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/RoundToEven"], "MR"],
 ". Only numbers between ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "50"}]], TraditionalForm]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{".9999", "\[Cross]", 
    SuperscriptBox["10", "50"]}], TraditionalForm]], "InlineFormula"],
 " are allowed. Mixed\[Hyphen]mode arithmetic is not allowed and division is \
not done with correct rounding. (It is performed as two operations: \
multiplication by the reciprocal. Each operation involves rounding errors.) "
}], "MathCaption",
 CellID->9247],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Arithmetic", "[", "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->10714],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4", ",", "10", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "RoundToEven"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "50"}], ",", "50"}], "}"}]}], ",", 
   RowBox[{"MixedMode", "\[Rule]", "False"}], ",", 
   RowBox[{"IdealDivide", "\[Rule]", "False"}]}], "}"}]], "Output",
 ImageSize->{451, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->24576422]
}, Open  ]],

Cell["Now the arithmetic is set to be six digits in base 8. ", "MathCaption",
 CellID->12700],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SetArithmetic", "[", 
  RowBox[{"6", ",", "8"}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->2944],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6", ",", "8", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "RoundToEven"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "50"}], ",", "50"}], "}"}]}], ",", 
   RowBox[{"MixedMode", "\[Rule]", "False"}], ",", 
   RowBox[{"IdealDivide", "\[Rule]", "False"}]}], "}"}]], "Output",
 ImageSize->{451, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->262343759]
}, Open  ]],

Cell["\<\
The result is displayed in octal (with trailing zeros suppressed). \
\>", "MathCaption",
 CellID->6899],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", "\[Pi]", "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->28942],

Cell[BoxData[
 TagBox[
  InterpretationBox[
   SubscriptBox["\<\"3.11040000000000000000\"\>", "8"],
   3.1416015625`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 8]& ]], "Output",
 ImageSize->{164, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->383173507]
}, Open  ]],

Cell["\<\
There are several options that can be used when setting the arithmetic. In \
addition to changing the precision and the base, you can control the type of \
rounding used, the magnitude of the numbers allowed, whether \
mixed\[Hyphen]mode arithmetic is to be allowed, and whether division is to be \
done with a single rounding operation. \
\>", "Text",
 CellID->4473],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{"option", " ", "name"}], "TableHeader"], 
    StyleBox[
     RowBox[{"default", " ", "value"}], "TableHeader"], Cell[""]},
   {
    ButtonBox["RoundingRule",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/RoundingRule"], 
    ButtonBox["RoundToEven",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/RoundToEven"], Cell[
    "the type of rounding to be used ", "TableText"]},
   {
    ButtonBox["ExponentRange",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/ExponentRange"], 
    RowBox[{"{", 
     RowBox[{"-", "50"}], ",", "50", "}"}], Cell[
    "the range of exponents that are to be allowed ", "TableText"]},
   {
    ButtonBox["MixedMode",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/MixedMode"], 
    ButtonBox["False",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/False"], Cell["\<\
whether mixed\[Hyphen]mode arithmetic is to be allowed \
\>", "TableText"]},
   {
    ButtonBox["IdealDivision",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/IdealDivision"], 
    ButtonBox["False",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/False"], Cell[
    "whether correctly rounded division is to be used", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->27241],

Cell[TextData[{
 "Options for ",
 StyleBox[ButtonBox["SetArithmetic",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/SetArithmetic"], "MR"],
 "."
}], "Caption",
 CellID->7856],

Cell[TextData[{
 "It should be noted that correctly rounded division is implemented by the \
function ",
 StyleBox[ButtonBox["IdealDivide",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/IdealDivide"], "MR"],
 ". This function can be used whether or not the arithmetic is set to \
automatically use correct division. It is difficult to get ",
 StyleBox["x/y", "MR"],
 " to give correctly rounded division since the ",
 StyleBox["x/y", "MR"],
 " is converted to ",
 StyleBox["x", "MR"],
 " ",
 StyleBox["y^(-1)", "MR"],
 " before the package ever sees it. The way we get around this is to define \
",
 Cell[BoxData[
  ButtonBox["$PreRead",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$PreRead"]], "InlineFormula"],
 " to convert ",
 StyleBox["x/y", "MR"],
 " into ",
 StyleBox["x", "MR"],
 " ",
 StyleBox["~", "MR"],
 StyleBox[ButtonBox["IdealDivide",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/IdealDivide"], "MR"],
 StyleBox["~", "MR"],
 " ",
 StyleBox["y", "MR"],
 " before the parser has a chance to change it to ",
 StyleBox["x", "MR"],
 " ",
 StyleBox["y^(-1)", "MR"],
 ". If you want to use ",
 Cell[BoxData[
  ButtonBox["$PreRead",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$PreRead"]], "InlineFormula"],
 " for your own purposes this will interfere with your definition. "
}], "Text",
 CellID->16709],

Cell[BoxData[GridBox[{
   {
    ButtonBox["RoundToEven",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/RoundToEven"], Cell["\<\
round to the nearest representable number and, in the case of a tie round, \
round to the one represented by an even mantissa\
\>", "TableText"]},
   {
    ButtonBox["RoundToInfinity",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/RoundToInfinity"], Cell["\<\
round to the nearest representable number and, in the case of a tie round, \
round away from 0\
\>", "TableText"]},
   {
    ButtonBox["Truncation",
     BaseStyle->"Link",
     ButtonData->"paclet:ComputerArithmetic/ref/Truncation"], Cell[TextData[{
     "simply discard excess digits, much as ",
     ButtonBox["Floor",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Floor"],
     " does for positive numbers"
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->25913],

Cell["Types of rounding available. ", "Caption",
 CellID->14452],

Cell["\<\
Mixed\[Hyphen]mode arithmetic (any arithmetic operation involving an integer \
and a computer number) is not allowed. \
\>", "MathCaption",
 CellID->6893],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"3", " ", 
  RowBox[{"ComputerNumber", "[", "\[Pi]", "]"}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->21175],

Cell[BoxData[
 RowBox[{"3", " ", 
  TagBox[
   InterpretationBox[
    SubscriptBox["\<\"3.11040000000000000000\"\>", "8"],
    3.1416015625`18.954589770191006,
    AutoDelete->True],
   BaseForm[#, 8]& ]}]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->72070398]
}, Open  ]],

Cell["\<\
This turns on mixed\[Hyphen]mode arithmetic (and sets the arithmetic to six \
digits in hexadecimal). \
\>", "MathCaption",
 CellID->19972],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SetArithmetic", "[", 
  RowBox[{"6", ",", "16", ",", 
   RowBox[{"MixedMode", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->5826],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6", ",", "16", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "RoundToEven"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "50"}], ",", "50"}], "}"}]}], ",", 
   RowBox[{"MixedMode", "\[Rule]", "True"}], ",", 
   RowBox[{"IdealDivide", "\[Rule]", "False"}]}], "}"}]], "Output",
 ImageSize->{444, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->116751536]
}, Open  ]],

Cell["Now the product is computed. ", "MathCaption",
 CellID->1886],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"3", " ", 
  RowBox[{"ComputerNumber", "[", "\[Pi]", "]"}]}]], "Input",
 CellLabel->"In[13]:=",
 CellID->17283],

Cell[BoxData[
 TagBox[
  InterpretationBox[
   SubscriptBox["\<\"9.6cbe5000000000\"\>", "16"],
   9.424778938293457031`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 16]& ]], "Output",
 ImageSize->{127, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->77348868]
}, Open  ]],

Cell["\<\
There are many things about computer arithmetic that are very different from \
ordinary arithmetic. Most of these differences are quite easy to demonstrate. \
\
\>", "Text",
 CellID->19612],

Cell["This sets the arithmetic back to the default. ", "MathCaption",
 CellID->14894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SetArithmetic", "[", 
  RowBox[{"4", ",", "10"}], "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->10853],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4", ",", "10", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "RoundToEven"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "50"}], ",", "50"}], "}"}]}], ",", 
   RowBox[{"MixedMode", "\[Rule]", "False"}], ",", 
   RowBox[{"IdealDivide", "\[Rule]", "False"}]}], "}"}]], "Output",
 ImageSize->{451, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->216443954]
}, Open  ]],

Cell["\<\
Expressions are evaluated numerically before they become computer numbers. \
\>", "MathCaption",
 CellID->19999],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", 
  RowBox[{"\[Pi]", "-", 
   FractionBox["22", "7"]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->30896],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"-0.001264000000000000000\"\>",
   -0.001264`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->86518910]
}, Open  ]],

Cell["\<\
If the separate parts are converted to computer numbers first, catastrophic \
cancellation of digits can occur. \
\>", "MathCaption",
 CellID->4470],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ComputerNumber", "[", "\[Pi]", "]"}], "-", 
  RowBox[{"ComputerNumber", "[", 
   FractionBox["22", "7"], "]"}]}]], "Input",
 CellLabel->"In[16]:=",
 CellID->14222],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"-0.001000000000000000000\"\>",
   -0.001`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{172, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->231552149]
}, Open  ]],

Cell["\<\
Summing the terms from smallest to largest gives one answer. \
\>", "MathCaption",
 CellID->4255],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"sum", "=", "0"}], ";", 
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"sum", "+=", 
     FractionBox["1", 
      SuperscriptBox[
       RowBox[{"ComputerNumber", "[", "i", "]"}], "2"]]}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "200"}], "}"}]}], "]"}], ";", 
  RowBox[{"FullForm", "[", "sum", "]"}]}]], "Input",
 CellLabel->"In[17]:=",
 CellID->19942],

Cell[BoxData[
 TagBox[
  StyleBox[
   RowBox[{"ComputerNumber", "[", 
    RowBox[{"1", ",", "1625", ",", 
     RowBox[{"-", "3"}], ",", 
     RowBox[{"Rational", "[", 
      RowBox[{"13", ",", "8"}], "]"}], ",", 
     "1.63994654601499726794569455282818613228`23.698970004336047"}], "]"}],
   ShowSpecialCharacters->False,
   ShowStringCharacters->True,
   NumberMarks->True],
  FullForm]], "Output",
 ImageSize->{431, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]//FullForm=",
 CellID->81969117]
}, Open  ]],

Cell["\<\
As a general rule, it is better to sum from smallest to largest. You can see \
what the error is by comparing the mantissa (the second argument) to the \
correct value (the last argument). \
\>", "MathCaption",
 CellID->8119],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"sum", "=", "0"}], ";", 
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"sum", "+=", 
     FractionBox["1", 
      SuperscriptBox[
       RowBox[{"ComputerNumber", "[", "i", "]"}], "2"]]}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "200", ",", "1", ",", 
      RowBox[{"-", "1"}]}], "}"}]}], "]"}], ";", 
  RowBox[{"FullForm", "[", "sum", "]"}]}]], "Input",
 CellLabel->"In[18]:=",
 CellID->26066],

Cell[BoxData[
 TagBox[
  StyleBox[
   RowBox[{"ComputerNumber", "[", 
    RowBox[{"1", ",", "1640", ",", 
     RowBox[{"-", "3"}], ",", 
     RowBox[{"Rational", "[", 
      RowBox[{"41", ",", "25"}], "]"}], ",", 
     "1.63994654601499726794569455282818613228`23.69897000433602"}], "]"}],
   ShowSpecialCharacters->False,
   ShowStringCharacters->True,
   NumberMarks->True],
  FullForm]], "Output",
 ImageSize->{424, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]//FullForm=",
 CellID->142925106]
}, Open  ]],

Cell["\<\
Here is an example where summing from largest to smallest gives the better \
result. \
\>", "MathCaption",
 CellID->21654],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"sum", "=", "0"}], ";", 
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"sum", "+=", 
     FractionBox["1", 
      RowBox[{"ComputerNumber", "[", "i", "]"}]]}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "300"}], "}"}]}], "]"}], ";", 
  RowBox[{"FullForm", "[", "sum", "]"}]}]], "Input",
 CellLabel->"In[19]:=",
 CellID->31474],

Cell[BoxData[
 TagBox[
  StyleBox[
   RowBox[{"ComputerNumber", "[", 
    RowBox[{"1", ",", "6281", ",", 
     RowBox[{"-", "3"}], ",", 
     RowBox[{"Rational", "[", 
      RowBox[{"6281", ",", "1000"}], "]"}], ",", 
     "6.28266388029950346191948554104728928323`23.99999999999999"}], "]"}],
   ShowSpecialCharacters->False,
   ShowStringCharacters->True,
   NumberMarks->True],
  FullForm]], "Output",
 ImageSize->{424, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]//FullForm=",
 CellID->95426693]
}, Open  ]],

Cell["The difference is slight, and such examples are rare. ", "MathCaption",
 CellID->30651],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"sum", "=", "0"}], ";", 
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"sum", "+=", 
     FractionBox["1", 
      RowBox[{"ComputerNumber", "[", "i", "]"}]]}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "300", ",", "1", ",", 
      RowBox[{"-", "1"}]}], "}"}]}], "]"}], ";", 
  RowBox[{"FullForm", "[", "sum", "]"}]}]], "Input",
 CellLabel->"In[20]:=",
 CellID->22385],

Cell[BoxData[
 TagBox[
  StyleBox[
   RowBox[{"ComputerNumber", "[", 
    RowBox[{"1", ",", "6280", ",", 
     RowBox[{"-", "3"}], ",", 
     RowBox[{"Rational", "[", 
      RowBox[{"157", ",", "25"}], "]"}], ",", 
     "6.28266388029950346191948554104728928323`24.000000000000007"}], "]"}],
   ShowSpecialCharacters->False,
   ShowStringCharacters->True,
   NumberMarks->True],
  FullForm]], "Output",
 ImageSize->{431, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]//FullForm=",
 CellID->188064292]
}, Open  ]],

Cell["\<\
Basic arithmetic is all that is implemented in the package. We could easily \
extend things to include elementary functions. \
\>", "MathCaption",
 CellID->30047],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sin", "[", 
  RowBox[{"ComputerNumber", "[", 
   FractionBox[
    RowBox[{"N", "[", "\[Pi]", "]"}], "7"], "]"}], "]"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->1819],

Cell[BoxData[
 RowBox[{"Sin", "[", 
  TagBox[
   InterpretationBox["\<\"0.4488000000000000000\"\>",
    0.4488`18.954589770191003,
    AutoDelete->True],
   BaseForm[#, 10]& ], "]"}]], "Output",
 ImageSize->{186, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->376894762]
}, Open  ]],

Cell["Here is the square root of 47. ", "MathCaption",
 CellID->2829],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sq", "=", 
  RowBox[{"ComputerNumber", "[", 
   SqrtBox["47"], "]"}]}]], "Input",
 CellLabel->"In[22]:=",
 CellID->21434],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"6.856000000000000000\"\>",
   6.856`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->392279888]
}, Open  ]],

Cell["\<\
It is a theorem that correctly rounded square roots of small integers will \
always square back to the original integer if the arithmetic is correct. \
\>", "MathCaption",
 CellID->29570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sq", " ", "sq"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->26286],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"47.00000000000000000\"\>",
   47.`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->10383949]
}, Open  ]],

Cell["But a similar theorem for cube roots does not exist. ", "MathCaption",
 CellID->25231],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"cr", "=", 
   RowBox[{"ComputerNumber", "[", 
    SuperscriptBox["3", 
     RowBox[{"1", "/", "3"}]], "]"}]}], ";", 
  RowBox[{"cr", " ", "cr", " ", "cr"}]}]], "Input",
 CellLabel->"In[24]:=",
 CellID->22723],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"2.998000000000000000\"\>",
   2.998`18.954589770191006,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->481899894]
}, Open  ]],

Cell[TextData[{
 "This changes the arithmetic to seven digits in base 10 with a rounding rule \
of ",
 StyleBox[ButtonBox["RoundToEven",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/RoundToEven"], "MR"],
 " and an exponent range of ",
 Cell[BoxData[
  FormBox["-", TraditionalForm]], "InlineFormula"],
 "50 to 50. "
}], "MathCaption",
 CellID->17173],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SetArithmetic", "[", "7", "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->18782],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7", ",", "10", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "RoundToEven"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "50"}], ",", "50"}], "}"}]}], ",", 
   RowBox[{"MixedMode", "\[Rule]", "False"}], ",", 
   RowBox[{"IdealDivide", "\[Rule]", "False"}]}], "}"}]], "Output",
 ImageSize->{451, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->309657029]
}, Open  ]],

Cell["This number rounds down. ", "MathCaption",
 CellID->27348],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", ".9999999499999999999999999", "]"}]], "Input",\

 CellLabel->"In[26]:=",
 CellID->9699],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"1.000000000000000000\"\>",
   1.`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->240133286]
}, Open  ]],

Cell["\<\
This number rounds up because it rounds toward the mantissa that is even \
(1000000) rather than the one that is odd (9999999). \
\>", "MathCaption",
 CellID->303],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", ".9999999500000000000000000", "]"}]], "Input",\

 CellLabel->"In[27]:=",
 CellID->14190],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"1.000000000000000000\"\>",
   1.`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[27]=",
 CellID->383518689]
}, Open  ]],

Cell["\<\
Again it rounds toward the even mantissa (1000000 rather than 1000001). \
\>", "MathCaption",
 CellID->19879],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ComputerNumber", "[", "1.000000500000000000000000", "]"}]], "Input",\

 CellLabel->"In[28]:=",
 CellID->12951],

Cell[BoxData[
 TagBox[
  InterpretationBox["\<\"1.000000000000000000\"\>",
   1.`18.954589770191003,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[28]=",
 CellID->219552471]
}, Open  ]],

Cell["\<\
The reciprocal of the reciprocal is not the original number; in fact it may \
be quite different. \
\>", "MathCaption",
 CellID->24768],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"x", "=", 
   RowBox[{"ComputerNumber", "[", "9010004", "]"}]}], ";", 
  RowBox[{"y", "=", 
   FractionBox["1", "x"]}], ";", 
  RowBox[{"z", "=", 
   FractionBox["1", "y"]}]}]], "Input",
 CellLabel->"In[29]:=",
 CellID->27812],

Cell[BoxData[
 TagBox[
  InterpretationBox[
   RowBox[{"\<\"9.01000700000000000\"\>", "\[Times]", 
    SuperscriptBox["10", "\<\"6\"\>"]}],
   9.010007*^6,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[29]=",
 CellID->581122782]
}, Open  ]],

Cell["\<\
This is multiplication by the reciprocal. It involves two rounding \
operations. \
\>", "MathCaption",
 CellID->32646],

Cell[CellGroupData[{

Cell[BoxData[
 FractionBox[
  RowBox[{"ComputerNumber", "[", "2", "]"}], "x"]], "Input",
 CellLabel->"In[30]:=",
 CellID->32147],

Cell[BoxData[
 TagBox[
  InterpretationBox[
   RowBox[{"\<\"2.219754000000000000\"\>", "\[Times]", 
    SuperscriptBox["10", "\<\"-7\"\>"]}],
   2.219754*^-7,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->284035651]
}, Open  ]],

Cell[TextData[{
 "This is true division. It uses a single rounding operation. You can get \
this better type of division to work with \[OpenCurlyDoubleQuote]",
 StyleBox["/", "MR"],
 "\[CloseCurlyDoubleQuote] by setting the option ",
 StyleBox[ButtonBox["IdealDivide",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/IdealDivide"], "MR"],
 " to ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " when you use ",
 StyleBox[ButtonBox["SetArithmetic",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/SetArithmetic"], "MR"],
 ". "
}], "MathCaption",
 CellID->22894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IdealDivide", "[", 
  RowBox[{
   RowBox[{"ComputerNumber", "[", "2", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[31]:=",
 CellID->12596],

Cell[BoxData[
 TagBox[
  InterpretationBox[
   RowBox[{"\<\"2.219755000000000000\"\>", "\[Times]", 
    SuperscriptBox["10", "\<\"-7\"\>"]}],
   2.219755*^-7,
   AutoDelete->True],
  BaseForm[#, 10]& ]], "Output",
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->195998596]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Normal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Normal"]], "InlineFormula"],
 " converts a computer number back to an ordinary rational number with \
exactly the same value. "
}], "MathCaption",
 CellID->764],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Normal", "[", 
  RowBox[{"ComputerNumber", "[", "\[Pi]", "]"}], "]"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->3635],

Cell[BoxData[
 FractionBox["3141593", "1000000"]], "Output",
 ImageSize->{62, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->12872243]
}, Open  ]],

Cell["Now we change the arithmetic again. ", "MathCaption",
 CellID->839],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SetArithmetic", "[", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "Truncation"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "3"}], ",", "3"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->20520],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3", ",", "2", ",", 
   RowBox[{"RoundingRule", "\[Rule]", "Truncation"}], ",", 
   RowBox[{"ExponentRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "3"}], ",", "3"}], "}"}]}], ",", 
   RowBox[{"MixedMode", "\[Rule]", "False"}], ",", 
   RowBox[{"IdealDivide", "\[Rule]", "False"}]}], "}"}]], "Output",
 ImageSize->{437, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->666299335]
}, Open  ]],

Cell["\<\
This suppresses error messages that will result from plotting nonnumeric \
values in the following examples. \
\>", "MathCaption",
 CellID->15644],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{"ComputerNumber", "::", "ovrflw"}], "]"}], ";", 
  RowBox[{"Off", "[", 
   RowBox[{"ComputerNumber", "::", "undflw"}], "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Off", "[", 
   RowBox[{"Plot", "::", "plnr"}], "]"}], ";"}]}], "Input",
 CellLabel->"In[34]:=",
 CellID->30346],

Cell["\<\
It is easy to plot the error in the computer number representation of each \
number. \
\>", "MathCaption",
 CellID->13294],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"error", "[", 
    RowBox[{"n_", "?", "NumberQ"}], "]"}], ":=", 
   RowBox[{
    RowBox[{"Normal", "[", 
     RowBox[{"ComputerNumber", "[", "n", "]"}], "]"}], "-", "n"}]}], ";", 
  RowBox[{"Plot", "[", 
   RowBox[{
    RowBox[{"error", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "10"}], ",", "10"}], "}"}], ",", 
    RowBox[{"PlotPoints", "\[Rule]", "193"}]}], "]"}]}]], "Input",
 CellLabel->"In[35]:=",
 CellID->8298],

Cell[GraphicsData["CompressedBitmap", "\<\
eJytmz+MVccVxi/vz+7yJ2ZZcIIhJBuDY0pKSpokKIkllMJGspSArEjrIgqy
XdBRUlJSbkkV0ViipKSkpKSkpKR8+WZ+5545M/e++2DNk3g7M+/e73xn5syZ
M2eGr+7+dPDv/9z96fvv7u7/5Ye79w6+/+7H/T//9wc1zY91Xfez/v14vlN5
pVL+9/Nqpb/x+2/6nuunWaqlwip/upmK/8pP/L35ZZ6+FqXedcvmia9z85/0
nR6bf/D7f8z1d+/evXr1KhW3cv3ly5f2+3KkPnv79u2nH8A3PYCcx48f37lz
p+C+fv2ah03u1atXnz9/Xn6/d+/eo0ePUvHcEfsHuffv36/kPn36tJK7s7NT
yb1165Ye+eVyhVPJlTIm1/tVdYk2Hvv7++qSIjdZTLKOrfB3KEVvVVIODg4q
KWibpFDX8O3u7qYio/htg/pFbxNqErBJvXnzZoX68OHDCvXFixfXr1+vUWc+
8mha0ESgQrt9+3aF9uTJEzVt4nh4eKihS6i89ebNm/Pnz3feU9euXZOggio7
EO1NqHpKhlhQ0aygSqTqBbVYy3rUmR7q5gyTW1CaDwwWSAtNQkm+ceOGwJf9
jLCZOcVYT2Hi1bwpuKIn+eAu+tGVGqn462q0Mt6CX9W7glz2o//s2bOCKHgJ
iYgaYXV3jRiZXsnv8VRE1oBp2AqyOkTdEpFhvwkZiwE5v7codt5ja9g0mBFb
5qCe24SN3UTsYu09tkRJYMQWHZHahM1TERurADu3LLFrFYGnNVk3jnl0DPk1
4hZv2uMKS4gtroZeBrCJN09FfGwk4rsmho/VY+Wp+JtR/Mv5XX8KESAuiofs
ZbhWlYxiM1My/CmTkd9d+txAjMlx7So5xX6m5DBbWjk+Yyo5zIdWTrGlIieN
dR6LJXOmxXcfavi5dctnhYuYN9Y0pQkzqJY0e/DgQY5ptvxpfdCOWAc5qThr
LGxKFjOq1cp7otIq6oq0WWNr5/Pz/YoaI6WEMra6fp6Rfe45EXh4f1U8fIVp
eBR7hMdfszXHCGK7kd7GcrDxGduw8R6t2PjK1LApVgubPn7ElQVSo6xS9Uyu
SCR2zbhuCddZyCDVdbN+dLAa2mWutKMB7YzUfGBHrIA873akj17U658dsTcz
/22iEpqId1JEiix5GpcluZJFu/sbH4F2hHN9GYE1OjYiUQPm0joNcqQ9YpWZ
+UkWY5rQAreIFslvwFamK5I8hwcb9jzWMK9ZM31HWDMrp1hHtm09sz9N7EiT
bMPZezyssoSoLJ5FE/xi0+/FMzb9XnkN18BCxAur6A227e8o2z1iU+8rZwtz
YDWQsIW5Omk+6Gc8RtPPOIyWJdPcWPa+fi+zIRTlZZjBRkDOBmawAWrIBo/R
sMFhtGwsTC1shn3k8Tx6OqsU9vasYAgrVkTkOCtcgK2DDocDaDlZiDvOKQd9
Zw0ut0pVUdTIQ0vtKoMkivoJWiykXcOLqW0RiOOpAxc1JwuSpzjpd99IwY+I
BjxHUrvGGU4xDiicmLQVJ/AaThZeT3FiLGhFVTjhAkCCHz63oDacmIoVJ/Bo
iUgX1/PZ892L2wl8WONBgRt8mG+LmgvOteJCl0UuoBQus35l2/P9jhsFHMDg
bXSDA/uBhgOusuJAF0UOaDXVH75DcrXgAhYo8IILiMiAC05vGXjgiyIPHNE4
j7yaXlSHKNLTo0KUrUmGvtVRcZ/lWkMRMQiALhTN60WOzIzIEQ8VOeKe4Jg4
9dm0PhIumaaWO50jmQJieyku4hj3cR7pwR3xCEYPuJuPnJm8Lu3qzUP6+mGr
zDJwx42daVjnGdTtBK3h0uYIrzT1y6NaYo5iIpGYqTSWNnFH6RqrXX1OdAxF
NEZL87o+QnqTeek6miujLgDVU5xzdlSXlCQ9lv7ShNpdH3kP82mt/mnMFyMr
9GXrx9yDF3AJN/JH3PQto5OecefrfaF2NRKzYRX0Bfr3skx3/IFLN5fpuqub
07r8Prp/pLHexR1rskk8SSNpL53iLtz1VvssEd5BVxXRWwhku6Ku+BvX1dZE
11X1FHd8HF0/X8UsYWnPUe+5ZIG57/VN3CP5UlzqeDjIT2wvUJ+t8CwFUHM5
LNTsZrYOzfKcZdFPr7uetmK4nqonH/Xx9Bzm7Uo9P3+KhA0fMWcspC/RFeWZ
Kvnp02ryLTI9Qi+Iu8oExdKc5GBZhZjc0s53FrZOueaqJw/3Ppr3kfM8jKPn
oFr9PiFe5SNVyNUSLqIiZXFzDX3vLQ1VRds0tF3H9juNbdeluKPfMeUMuMld
9DzAKTqqKVnXXsMfx/RtrM0h1fuq7jj2iC6UiXKzlWXj5QfnLdm5fRu5bJHh
nXY8cGLupcDHxwWMwtkytr91jmGnLCCtWSkOyhwJ6LAhykTfyaKdo3cT88Y5
qkq7OLImVBxTYOQcbV1xjpYDDhwtkhdBnFVu6I6DxU4l+Ze+3K+tmaHvtHHi
ngXw3hVDSdwK7NKoOjtcQWFnZwqBXXqKBT4L7E5IkoA4NuBcyIYpb2mcme+H
WV6dGXkVmMksnJmNnTNjqhZmlo+B2R9yK7Ze3ir15Vh96fUVeYBs0/0e71T8
lfLwO53sdSf1ytv8kYNVb6I7RiQt8k4oj6BKUswDJH2zvyrZji5HQt5P0tn7
ybMS6idhez+Z/Xg/mZP3frK9Pv2UWv/Xe1u8cLuGTOQivA6XckbUxZMpDKDU
2zyRfKnrwpTL7b/SxPC9NtsZOkv2TeRlZrWdPVbqs3gieQQtjvj8h76Ph8RC
OSQqT20479xmPY+Zqiq/w2K9yhvpPr9znGiWwyaPZs1b5BmbTpAA7v4xynU/
F3kkyfOz5vbUFL9H/SSnpjEv5dkd86yMs6T7OGvhEkuO2tSI0zKvS35VT0s5
FE0NtwNDJCaG0dLKmal7Zn4/PcxFwZBcFDJg6yvTcZIMTbxoqxkMfRBGGCKx
MERSYSjtIsMLrHExPwVDRhBJjrHKybkuRHpywoIjhyOymd/CE9or1uhvAj/W
q8KPXinnovRGiiQy3wtIgp96wM1ZoybZ8BNXQm2V2eIQqkCLmAzPOEvnHam5
8sR1H85yUrU4DCKMkRNnuhKeC2IleGaEcyBo7KQCPMWZ7c8q+xuCFiIsQieU
lZzWT34delC64daJptadWktwAs1sL8Y81ipv2zmIXeU8lsoQ1DsqA2Z7iDzK
HkVxFJ5EjjP8fe4LttZs0vyMPJx+533O3EMYOJ+IeS34gkA2Kp1gdadXOa8l
G/cMEvYOrxI/UZ7iyNYYCcTdoyfqZJXoxRNkcGAYM0t2jpYZkmnx/BJR/kj8
RHmnmiEwY+MamY2eyJNjMmY8DbP4PnkmmJFVAAWWrJtN+GTlqb5juxkZrjvZ
F55IZY47MS9WIWR27PlBgGkTsnj+KvncKXbsriO7dXcDArsTMSUGUxBIieUe
/CTiwNdvZbHR8e/jE/zYFkd+6+4XwC+3nGwyX9AFhMxXlznaeSJY8D0zyuV3
6XvpeUHLqCB93b2EyIf5s+qDRriBYvsR+JCkAgtuZyqbh4dnvCoe6+4vBB5n
/dyeqANOnjNRWSxUJrYHyvJmU/3iGSjjQxZ96r5D4LQY5HROrfrNBjkdcVKB
OBobE7+9CT6OWPHZcDcCSrDc9ZxTTMB46gFOLAXgpTVsipHjVYw23KKIjPY8
PRQTI54qgBHuHDwWrSlOnmqpOG24eWGccuvZePOCJRmKnglwWlAsGYE+S4FV
X0rf1T0OKCF73Q2NXIh8tnxerKosxUn4kKWAD9zgk9aX8X66NECNvDbc5qiY
+QyBmYIEZ8YccWawLDv/KWYRFWbsPDbe8Ki4xfP/mJXQa56VwNzJAZa9f5Vj
au6ejN/rvTSQGHnHno68/S5Ixdvn1KrOWbD/dt59HmhOmjAVc33ydkWb7Yzs
o9zIPo7GYTnP33I/6ArAP57/6wHPJehFvw+k6dzvv1Psb7mp9+OfqpG337xs
eK+5/7Hla5bzhkc5HeN+Qtwb9GeonqVZ+vWZ3jYapq2djPd8zo6EO57eU4eb
7oLgY2lHC9qxtXODnuxzruP2C49yM3TIg1Ed8sCvDnngWcd5RPmL1Xo+5W4x
I+J8GKpu3c2NIRtioE9X9S2SKdnl5lUj2y4Zr7unMZRNvNOeFUVZ7c0Yl2Wn
leO3L+YDScyEK6u4BiGh3BlsJNgR/vg9Clpdgt2Pak8M0CSf9c65vdHnNIMc
O6YfvxvRyDH9puQUS23kjN93qG6LuBzTckpOsUBOPu9svMfQyDANp2SUu+5R
xtQ9hUaG3eb4ohp3sMuN94iNWQ3vHSxqXNtRTHEvdhXx0Sjic7epwTdtpvDL
/dOIj1YRP2oEPhq1p5Vg5zs24cbpImAzhyI2UyRis4JMYZc7hvgYKKbZswzI
TIqIjFfvz+a+CYjFHzkiiaQKEfOPiBjHFFePjAsyW5RylsQhrEkCl5GfwvX4
tuDaCafjcrxZ4TLiU7genxbcrpwr9qdL5ZYH2YzkUb4cRc03JOceW9pb60/9
ImrKh31ZjRZoHmNUaGPncxEtbRKnOPq+qqCaBUfU5P8iavJ8SbH+NksVlayN
CJBY7mO4RLNrl2iRS5SY/OBnDerw5sj46st75W6Ey7VUvMs1/x/lJs1/mVzp
a7YFrvrO/lcPdVWqk0o9bLHr+8ktsetXR+R51PfS/4yL/0Ptn7l1+H8fu2P/
B6UJWZ0=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]=",
 CellID->466081890]
}, Open  ]],

Cell["You can zoom in to see the hole at zero. ", "MathCaption",
 CellID->23955],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"error", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "47"}]}], "]"}]], "Input",
 CellLabel->"In[36]:=",
 CellID->11955],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]=",
 CellID->102322869]
}, Open  ]],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses both the machine arithmetic that is provided by the machine and its \
own arbitrary\[Hyphen]precision arithmetic, which is the same on all machines \
(except for the variations in the size of the largest representable number \
and the amount of memory on the machine). This package also deals with the \
floating\[Hyphen]point arithmetic provided by the machine. "
}], "Text",
 CellID->10973],

Cell["\<\
Numbers on a computer comprise a discrete set. There are gaps between the \
numbers and when you do arithmetic the result often is not representable. \
When the result falls between two representable numbers the best that can be \
done is to use the closest representable number as the result instead of the \
correct result. The set of numbers that can be represented on a computer in \
floating\[Hyphen]point format are commonly referred to as the set of machine \
numbers. This package allows you to vary certain fundamental parameters that \
control the arithmetic. \
\>", "Text",
 CellID->22384],

Cell[TextData[{
 "The difference between two consecutive machine numbers is called an ulp \
(unit in the last place, ",
 StyleBox["i.e.",
  FontSlant->"Italic"],
 ", one digit in the least significant place). The size of an ulp varies \
depending on where you are in the set of machine numbers. Between ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  FormBox["2", TraditionalForm]], "InlineFormula"],
 " an ulp is equal to ",
 Cell[BoxData[
  ButtonBox["$MachineEpsilon",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MachineEpsilon"]], "InlineFormula"],
 ". Between ",
 Cell[BoxData[
  FormBox["2", TraditionalForm]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  FormBox["4", TraditionalForm]], "InlineFormula"],
 " it is equal to ",
 StyleBox["2", "MR"],
 " ",
 Cell[BoxData[
  ButtonBox["$MachineEpsilon",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/$MachineEpsilon"]], "InlineFormula"],
 ". Ideally no function should ever return a result with error exceeding half \
of an ulp since the distance from the true result to the nearest machine \
number is always less than half of an ulp, the worst case being when it is \
exactly halfway between two machine numbers. It is relatively easy to achieve \
this ideal for the four arithmetic operations and for the square root \
function. For the elementary transcendental functions it becomes more \
difficult due to the fact that they cannot be evaluated in a single operation \
and the combined effects of all the rounding errors would have to be made \
less than half an ulp. Nevertheless, it is quite possible to design \
algorithms for these functions such that the error is never more than one \
ulp. "
}], "Text",
 CellID->2306],

Cell[TextData[{
 "The package ",
 StyleBox["ComputerArithmetic`", "MR"],
 " provides four functions that are useful in examining the machine floating\
\[Hyphen]point arithmetic on your machine. "
}], "Text",
 CellID->22299],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Ulp",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/Ulp"], "[", 
     StyleBox["a", "TI"], "]"}], Cell[TextData[{
     "give the size of an ulp near ",
     StyleBox["a", "TI"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["MachineError",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/MachineError"], "[", 
     StyleBox["f", "TI"], ",", 
     StyleBox["x", "TI"], "->", 
     StyleBox["a", "TI"], "]"}], Cell[TextData[{
     "give the error in evaluating ",
     StyleBox["f", "TI"],
     " at ",
     StyleBox["x", "TI"],
     " ",
     "=",
     " ",
     StyleBox["a", "TI"],
     " using machine arithmetic"
    }], "TableText"]},
   {
    RowBox[{
     RowBox[{
      ButtonBox["MicroscopePlot",
       BaseStyle->"Link",
       ButtonData->"paclet:ComputerArithmetic/ref/MicroscopePlot"], "[", 
      RowBox[{
       StyleBox["f", "TI"], ",", 
       RowBox[{"{", 
        RowBox[{
         StyleBox["x", "TI"], ",", 
         StyleBox["a", "TI"]}], "}"}]}], "]"}], " "}], Cell[TextData[{
     "plot ",
     StyleBox["f", "TI"],
     " near ",
     StyleBox["x", "TI"],
     " ",
     "=",
     " ",
     StyleBox["a", "TI"],
     " using machine arithmetic"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["MicroscopicErrorPlot",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"], "[", 
     
     StyleBox["f", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["a", "TI"], "}", "]"}], Cell[TextData[{
     "plot the error in evaluating ",
     StyleBox["f", "TI"],
     " near ",
     StyleBox["x", "TI"],
     " ",
     StyleBox["=", "MR"],
     " ",
     StyleBox["a", "TI"],
     " using machine arithmetic"
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->27215],

Cell["\<\
Functions for examining machine floating\[Hyphen]point arithmetic. \
\>", "Caption",
 CellID->1643],

Cell[TextData[{
 "The default neighborhood of the point ",
 StyleBox["x", "TI"],
 " ",
 StyleBox["=", "MR"],
 " ",
 StyleBox["a", "TI"],
 " used in ",
 StyleBox[ButtonBox["MicroscopePlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopePlot"], "MR"],
 " and ",
 StyleBox[ButtonBox["MicroscopicErrorPlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"], "MR"],
 " is from 30 ulps to the left of ",
 StyleBox["a", "TI"],
 " to 30 ulps to the right of ",
 StyleBox["a", "TI"],
 ", where the ulp used is that defined at ",
 StyleBox["x", "TI"],
 " ",
 StyleBox["=", "MR"],
 " ",
 StyleBox["a", "TI"],
 ", that is, ",
 StyleBox[ButtonBox["Ulp",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/Ulp"], "MR"],
 StyleBox["[", "MR"],
 StyleBox["a", "TI"],
 StyleBox["]", "MR"],
 ". Near powers of 2 and near 0 it is not clear what should be done since the \
size of an ulp changes. Near powers of 2 it is not much of a problem in the \
abscissa since an ulp is chosen to be the smaller of the two values and the \
resulting machine numbers just get included multiple times due to rounding \
effects. You can change the default size of the neighborhood by including a \
third value in the neighborhood specification. "
}], "Text",
 CellID->1634],

Cell[TextData[{
 "A problem arises with ",
 StyleBox[ButtonBox["MicroscopePlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopePlot"], "MR"],
 " and ",
 StyleBox[ButtonBox["MicroscopicErrorPlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"], "MR"],
 " near places where the function value becomes infinite. Since the function \
is changing by orders of magnitude very rapidly it is difficult to choose a \
good scale that displays the information in which you are interested. In such \
cases you may have to examine the function several hundreds of ulps away from \
the point you really want. "
}], "Text",
 CellID->18761],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["MicroscopePlot",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/MicroscopePlot"], "[", 
     StyleBox["f", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["a", "TI"], ",", 
     StyleBox["n", "TI"], "}", "]"}], Cell[TextData[{
     "plot ",
     StyleBox["f", "TI"],
     " from ",
     StyleBox["x", "TI"],
     " ",
     StyleBox["=", "MR"],
     " ",
     StyleBox["a", "TI"],
     " ",
     StyleBox["-", "MR"],
     " ",
     StyleBox["n", "TI"],
     " ",
     ButtonBox["Ulp",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/Ulp"],
     "[",
     StyleBox["a", "TI"],
     "]",
     " to ",
     StyleBox["x", "TI"],
     " ",
     "=",
     " ",
     StyleBox["a", "TI"],
     " ",
     "+",
     " ",
     StyleBox["n", "TI"],
     " ",
     ButtonBox["Ulp",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/Ulp"],
     "[",
     StyleBox["a", "TI"],
     "]",
     " using machine arithmetic"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["MicroscopicErrorPlot",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"], "[", 
     
     StyleBox["f", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["a", "TI"], ",", 
     StyleBox["n", "TI"], "}", "]"}], Cell[TextData[{
     "plot the error in evaluating ",
     StyleBox["f", "TI"],
     " from ",
     StyleBox["x", "TI"],
     " ",
     "=",
     " ",
     StyleBox["a", "TI"],
     " ",
     "-",
     " ",
     StyleBox["n", "TI"],
     " ",
     ButtonBox["Ulp",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/Ulp"],
     "[",
     StyleBox["a", "TI"],
     "]",
     " to ",
     StyleBox["x", "TI"],
     " ",
     "=",
     " ",
     StyleBox["a", "TI"],
     " ",
     "+",
     " ",
     StyleBox["n", "TI"],
     " ",
     ButtonBox["Ulp",
      BaseStyle->"Link",
      ButtonData->"paclet:ComputerArithmetic/ref/Ulp"],
     "[",
     StyleBox["a", "TI"],
     "]",
     " using machine arithmetic"
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->75390218],

Cell["Controlling the size of the neighborhood.", "Caption",
 CellID->254074879],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  ButtonBox["Joined",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Joined"]], "InlineFormula"],
 " controls the way the plot is drawn. It cannot be overemphasized that the \
functions that we are examining map the machine numbers to the machine \
numbers. For our purposes the real numbers in between do not even exist. Thus \
the plots produced by ",
 StyleBox[ButtonBox["MicroscopePlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopePlot"], "MR"],
 " and ",
 StyleBox[ButtonBox["MicroscopicErrorPlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"], "MR"],
 " should be nothing but a set of plotted points. However, it is easier to \
see what is happening if the points are joined with straight line segments. \
Finally, if you want to see what a function does as a map from the real \
numbers to the real numbers, that is, rounding the real argument to the \
nearest machine number and applying the function to this result, you can do \
so, but when you see large errors, don\[CloseCurlyQuote]t conclude that the \
function is to blame; it can\[CloseCurlyQuote]t do any better than deal with \
the machine numbers that it gets as arguments. "
}], "Text",
 CellID->32516],

Cell[BoxData[GridBox[{
   {
    StyleBox[
     RowBox[{"option", " ", "name"}], "TableHeader"], 
    StyleBox[
     RowBox[{"default", " ", "value"}], "TableHeader"], Cell[""]},
   {
    ButtonBox["Joined",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Joined"], 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell[TextData[{
     "the way the points are to be connected; either ",
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"],
     ", ",
     ButtonBox["False",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/False"],
     ", or ",
     ButtonBox["Real",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Real"]
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, {True}, False}, "RowsIndexed" -> {}}},
 CellID->13940],

Cell[TextData[{
 "Options for ",
 ButtonBox["MicroscopePlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopePlot"],
 " and ",
 ButtonBox["MicroscopicErrorPlot",
  BaseStyle->"Link",
  ButtonData->"paclet:ComputerArithmetic/ref/MicroscopicErrorPlot"],
 "."
}], "Caption",
 CellID->9389],

Cell["Here is the size of an ulp at 1. ", "MathCaption",
 CellID->7994],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Ulp", "[", "1", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->16999],

Cell[BoxData["1.1102230246251565`*^-16"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->219130507]
}, Open  ]],

Cell["The size of an ulp changes at powers of 2. ", "MathCaption",
 CellID->5075],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Ulp", "[", "7.999999", "]"}], ",", 
   RowBox[{"Ulp", "[", "8", "]"}], ",", 
   RowBox[{"Ulp", "[", "8.000001", "]"}]}], "}"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->24504],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "8.881784197001252`*^-16", ",", "8.881784197001252`*^-16", ",", 
   "1.7763568394002505`*^-15"}], "}"}]], "Output",
 ImageSize->{310, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->18416162]
}, Open  ]],

Cell["There is a large hole near 0. ", "MathCaption",
 CellID->26096],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Ulp", "[", "0", "]"}], ",", 
   RowBox[{"Ulp", "[", "$MinMachineNumber", "]"}]}], "}"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1618],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.2250738585072014`*^-308", ",", 
   "4.94065645841246544176568792868221`15.954589770191005*^-324"}], 
  "}"}]], "Output",
 ImageSize->{289, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->49838940]
}, Open  ]],

Cell[TextData[{
 "Here are the combined errors of rounding ",
 Cell[BoxData[
  FormBox[
   RowBox[{"49", 
    RowBox[{"\[Pi]", "/", "50"}]}], TraditionalForm]], "InlineFormula"],
 " to the nearest machine number and then taking the square root. "
}], "MathCaption",
 CellID->26756],

Cell[BoxData[
 RowBox[{"Clear", "[", "x", "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->22869420],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachineError", "[", 
  RowBox[{
   SqrtBox["x"], ",", 
   RowBox[{"x", "\[Rule]", 
    FractionBox[
     RowBox[{"49", " ", "\[Pi]"}], "50"]}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->5771],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1.2221878982084162`"}], " ", "Ulps"}]], "Output",
 ImageSize->{92, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->271228651]
}, Open  ]],

Cell["Here is the error of just the square root. ", "MathCaption",
 CellID->15031],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachineError", "[", 
  RowBox[{
   SqrtBox["x"], ",", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{"N", "[", 
     FractionBox[
      RowBox[{"49", " ", "\[Pi]"}], "50"], "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->26176],

Cell[BoxData[
 RowBox[{"0.04581618252996657`", " ", "Ulps"}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->11074195]
}, Open  ]],

Cell[TextData[{
 "For another number close to ",
 StyleBox["N[49", "MR"],
 " ",
 StyleBox["Pi/50]", "MR"],
 ", the error in the square root function is less than half an ulp. "
}], "MathCaption",
 CellID->28743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachineError", "[", 
  RowBox[{
   SqrtBox["x"], ",", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{
     RowBox[{"N", "[", 
      FractionBox[
       RowBox[{"49", " ", "\[Pi]"}], "50"], "]"}], "-", 
     RowBox[{"6", " ", 
      RowBox[{"Ulp", "[", 
       FractionBox[
        RowBox[{"49", " ", "\[Pi]"}], "50"], "]"}]}]}]}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->30524],

Cell[BoxData[
 RowBox[{"0.46532144311367607`", " ", "Ulps"}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->120849159]
}, Open  ]],

Cell[TextData[{
 "The function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"tan", 
    RowBox[{"(", 
     SqrtBox[
      SuperscriptBox[
       StyleBox["e", "TI"], 
       RowBox[{"sin", 
        RowBox[{"(", 
         RowBox[{"log", 
          StyleBox["x", "TI"]}], ")"}]}]]], ")"}]}], TraditionalForm]], 
  "InlineFormula"],
 " is not provided by the computer manufacturer so it is evaluated as a \
nested sequence of operations that are provided. This shows how much error is \
involved in the sequence, but this information is rarely useful. For testing \
purposes you normally want to look at an individual function evaluated at a \
machine number to avoid the effect of an ill\[Hyphen]conditioned function \
magnifying prior error. "
}], "MathCaption",
 CellID->17027],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MachineError", "[", 
  RowBox[{
   RowBox[{"Tan", "[", 
    SqrtBox[
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"Sin", "[", 
       RowBox[{"Log", "[", "x", "]"}], "]"}]]], "]"}], ",", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{"N", "[", "\[Pi]", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->4430],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "26.89112964425968`"}], " ", "Ulps"}]], "Output",
 ImageSize->{92, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->501237352]
}, Open  ]],

Cell[TextData[{
 "Here is the ",
 Cell[BoxData[
  FormBox["log", TraditionalForm]], "InlineFormula"],
 " function evaluated at machine numbers near 7. "
}], "MathCaption",
 CellID->22123],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopePlot", "[", 
  RowBox[{
   RowBox[{"Log", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "7"}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->7912],

Cell[GraphicsData["CompressedBitmap", "\<\
eJylVU1Lw0AQTbOtVBQU9FARMUgFBYUe9KCoKH6UFhU9KCgohNIvpX7U6kER
0YOePPhf+v/iJtm027epMsxCZvMm82beTDfNgduqlRtuq15ynXzTvavVSw/O
3m1TukTCsqy2vKoZS9578i642p4nd91WfSv8Z8GybMC7sVHFwFsDb4Hsrfg2
2fVaKrNfT3SQrjcLeb7J3hlpf6QzERuzDThvRElrTGnjXxZVaxIqfhK4g4Df
CdwhqPvG7mQUMj4TuOMeno9Htp4JwPcE7qSh54atx4H5XBG4s8Ats9XMA3YJ
3AVQc8lWkzPmfUZgL4OeE7aeVcBHf3DDVzhattZJ6EkBS6goW1FFlEKLWYOO
DoMd/40wMz4XQVAHq1xYS8R2hbkipWG46jUuWxRCrhBEpfUzoFu9186cjd5S
SuUWRBf6V+w5ddHvItRuGxGM3vpWMvvo353xze7JuqOJDNeA1z1peFbsmI6x
BuWtKQLeJHD3jV7WCex477GG/VmsELinwF1iqzkHnCNwL0DNIluNa8x7jsCu
gJ4sW8814GkCtwFqpthqmpAxQ+A+GZMdY+t5ATxC4L4aeobZej5gPmkC9wu4
gq1GfYcTvw9v/oo=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->190401268]
}, Open  ]],

Cell[TextData[{
 "Here is the ",
 Cell[BoxData[
  FormBox["log", TraditionalForm]], "InlineFormula"],
 " function evaluated at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"10", "+", "1", "+", "10"}], TraditionalForm]], "InlineFormula"],
 " machine numbers near 7 with the points joined by straight lines. "
}], "MathCaption",
 CellID->29190],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopePlot", "[", 
  RowBox[{
   RowBox[{"Log", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "7", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->22634],

Cell[BoxData[
 GraphicsBox[{
   {PointSize[0.02], PointBox[{-10., -0.625}], PointBox[{-9., -0.625}], 
    PointBox[{-8., -0.5}], PointBox[{-7., -0.5}], PointBox[{-6., -0.375}], 
    PointBox[{-5., -0.375}], PointBox[{-4., -0.25}], PointBox[{-3., -0.125}], 
    PointBox[{-2., -0.125}], PointBox[{-1., 0.}], PointBox[{0., 0.}], 
    PointBox[{1., 0.125}], PointBox[{2., 0.125}], PointBox[{3., 0.25}], 
    PointBox[{4., 0.375}], PointBox[{5., 0.375}], PointBox[{6., 0.5}], 
    PointBox[{7., 0.5}], PointBox[{8., 0.625}], PointBox[{9., 0.625}], 
    PointBox[{10., 0.75}]}, 
   {Thickness[0.001], 
    LineBox[{{-10., -0.625}, {-9., -0.625}, {-8., -0.5}, {-7., -0.5}, {-6., \
-0.375}, {-5., -0.375}, {-4., -0.25}, {-3., -0.125}, {-2., -0.125}, {-1., 
     0.}, {0., 0.}, {1., 0.125}, {2., 0.125}, {3., 0.25}, {4., 0.375}, {5., 
     0.375}, {6., 0.5}, {7., 0.5}, {8., 0.625}, {9., 0.625}, {10., 0.75}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  AxesOrigin->{-10., -0.625},
  PlotRange->All,
  Ticks->{{{0, 
      FormBox["\"7.\"", TraditionalForm]}}, {{0, 
      FormBox["\"1.94591\"", TraditionalForm]}}}]], "Output",
 ImageSize->{184, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->52840514]
}, Open  ]],

Cell[TextData[{
 "And here are the combined effects of rounding real numbers to the nearest \
machine number and taking the ",
 Cell[BoxData[
  FormBox["log", TraditionalForm]], "InlineFormula"],
 " of the result. This is ",
 StyleBox["not",
  FontSlant->"Italic"],
 " a valid test of the ",
 Cell[BoxData[
  FormBox["log", TraditionalForm]], "InlineFormula"],
 " function. "
}], "MathCaption",
 CellID->3078],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopePlot", "[", 
  RowBox[{
   RowBox[{"Log", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "7", ",", "10"}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "Real"}]}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->20010],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWE9rE0EUn+xs1gQ8BFHocWnVg4IIYix6CYS2qcGDKF4EzVLSJqBo2qD0
FChKzEGCpxzzEXIp5JiPkI+Qj5Bjjuvbt7N/Mvtm3EZKspBJ3rz3+817b96b
nfaV027UPzvt5pFjH5w6XxvNozN7/8spTPEMY+wSPidbDH678As/l64L3/Hx
xBu5p8OHGZK8R1q9xNmGNHt45dljbzSjWSaYvfV4KMX9vSvx/NbM7nhjdjAY
gKLb7YI0m82Y0el0GK9Wq6ZplstlIU+nU4G9QOw2jH8Al4mY+GQygZnxeMxM
9BvkSqUCMkgey3PJiwPStxsrWf+4Ni0GA0nK5XKQJIyMhyI7V2KZ1e/3bdsu
lUqFQgF0gDBBYDwUmYHyaDQSuDbJlvM88ZLLgbFYLPZ6PUwuruDLINVqNfD2
G8lwR8SyeVpmgRpchwBarZZlWfV6HWPjocj82BeLhcAda9jAtITPcs6tUAQJ
NgW2rkGybAWV7FeuopJp7M7GajHLEDmMYWP7fQ0dL+w+qtF5sOrgA6kDHSQz
5GMOibtP5o22fbBmrb7D3ymxq3T4a5LtoZu+w9+QDLsilv/X+iUfPPi+E28g
fyYrobmwMgSUBxTKFeR3qMwo6zkahbLwRF6DX2nNwFMfJmKlWAOTFCvp66hC
+oFcufgbPj7GMxPuRiITWRHLLlpDrYl4bmJ8UEHCtKz2YOmOEewmF99GwuJf
GXnhpjlF91bwJxm9Oidx1v2lNeO7brlR7cpVaBDZ0DPLvXW9WupcN8n7W3TO
P9WwJc95ZLOSx/4zkuUw1b7T2Ldr1i537xOlLbNALZ36/ikPcQq7R2p0Hu4x
E3ySRzt7TOLeCw83U6s/9e4psbqTyiZRH1JV1zaJbW6sNv0NeD6fC9wtDRv8
pZbmBgyL3iZZPqXKMo39vmatvhbzSizV0/RNLupxU8OW7HHVbc4iWX6KeDZT
u5xlQ207HA7R1sscY5Bf2vaXVHG0le4/GSzzF+ntMQc=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->93407858]
}, Open  ]],

Cell[TextData[{
 "This shows the error in evaluating the sine function at ",
 Cell[BoxData[
  FormBox["31", TraditionalForm]], "InlineFormula"],
 " machine numbers near 16. Note how the size of an ulp changes at 16. Note \
also that rounding has moved some points to coincide so that there appear to \
be fewer than ",
 Cell[BoxData[
  FormBox["31", TraditionalForm]], "InlineFormula"],
 " points. The scale on the vertical axis represents ulps. "
}], "MathCaption",
 CellID->14845],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopicErrorPlot", "[", 
  RowBox[{
   RowBox[{"Sin", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "16", ",", "15"}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "False"}]}], "]"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->8581],

Cell[GraphicsData["CompressedBitmap", "\<\
eJylV89vEkEU3rJLMbEHSGw1agIJl6aJHrSJHpBINXLS4MWDXiRbEmqCNS2J
XBpiL7304sEm1APeOHAt/wgXjvAf8CesM7O7zMzHm1lISdjdmfne97735sfb
fV9vNxutevvIrxeqJ/UfzSP/tPDu+IR1uRuO49yw/+EDhz0H7En8b4KA3e/z
q8vb4udcJ/SmWGdGjE6n02azOZlMnN+kzaZo9/t91tXtdq2o+Xw+GAzY1YBa
RdVfCyYTYVzeSi9GxZgTaUAb0evJ3mWcR9qFTsIf9+mkw7b0x+28eECxL0Fe
vxh0qH5j/bxZgoyb7NNRNjCeEszFuva/RHs4HOZyuV6v5+xouc8kZC+0LpfL
rCufz0fWOJ8hajweV6tVdlVQ3MdTyOA3kuMJ5MmGktmgUauvQezNg9aOBSW1
2lBSa0dTdQie9kkOHzzta7PnwppzifXrg4p9y/pRuZAnC3ovgtX2VdL+zEKE
FwZ9uFK5Y97MQnzr2K9yvnjWTEnLeE42A3ofKusto551MqNX2vpIQV6uDPYy
cgWh+K1EeUpFvI43Go0WsXwgbQ74Ja3G7Pu+mN+PmsY3uhr2MJvNWq2WoyGX
mLVspgDxOtDX2ZaZR53NWI2SMYvlUrV5G+iriLatRHF70V3PxktQ/pXkeAEq
bSiph0bZKvA9dv0n9iWO74LOxzBeI1l3Qfc6VjKOYxLVEm1Zo/a0vH4Xo7IG
7Vk4ZA2iUQ8h9jMLSsZ6pul5BDHRHElvR8m9d0HrpaZiCzReWjik1hB1JzDX
Dzyfa6CiGsSnnb1+0DxSr4kHz008p2sQkYlH9Gj7nMUgiP5ARCkbg3bWwmms
RGPiWK4ebqyfPS9ObSWelDZDcTXBmlOBGD6Rs18BlRLFOQ/AM81x+3X8DLQ2
NBXPQWPDwiG10qif4KlgQUmPBU1PBzzRHHSkRfDf1uaS/s7BmS2CurZhZSXt
kyLEYeKJv3v4nZtfW3C290OTXfxdclv8NuT2XMtt0tvdNmT1XJvzHciVHFU5
Pou2rFKvLChZrWwoWa1oFPUd42z8B7YcmKY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->216566391]
}, Open  ]],

Cell["\<\
Here is the error in evaluating sine at 21 machine numbers near 31 with the \
points joined by straight lines. \
\>", "MathCaption",
 CellID->28704],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopicErrorPlot", "[", 
  RowBox[{
   RowBox[{"Sin", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "31", ",", "10"}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->31974],

Cell[BoxData[
 GraphicsBox[{
   {PointSize[0.02], PointBox[{-10., 0.183827441535495}], 
    PointBox[{-9., -0.359683457953635}], PointBox[{-8., 0.0968056425571445}], 
    PointBox[{-7., -0.446705256932168}], PointBox[{-6., 0.00978384357842694}],
     PointBox[{-5., 0.46627294408893}], PointBox[{-4., -0.0772379554006581}], 
    PointBox[{-3., 0.379251145109662}], PointBox[{-2., -0.164259754380111}], 
    PointBox[{-1., 0.292229346130025}], PointBox[{0., -0.251281553359931}], 
    PointBox[{1., 0.205207547150022}], PointBox[{2., -0.338303352340118}], 
    PointBox[{3., 0.11818574816965}], PointBox[{4., -0.425325151320673}], 
    PointBox[{5., 0.0311639491889118}], PointBox[{6., 0.487653049698405}], 
    PointBox[{7., -0.0558578497921943}], PointBox[{8., 0.400631250717115}], 
    PointBox[{9., -0.142879648773668}], PointBox[{10., 0.313609451735458}]}, 
   {Thickness[0.001], 
    LineBox[{{-10., 0.183827441535495}, {-9., -0.359683457953635}, {-8., 
     0.0968056425571445}, {-7., -0.446705256932168}, {-6., 
     0.00978384357842694}, {-5., 
     0.46627294408893}, {-4., -0.0772379554006581}, {-3., 
     0.379251145109662}, {-2., -0.164259754380111}, {-1., 
     0.292229346130025}, {0., -0.251281553359931}, {1., 0.205207547150022}, {
     2., -0.338303352340118}, {3., 0.11818574816965}, {
     4., -0.425325151320673}, {5., 0.0311639491889118}, {6., 
     0.487653049698405}, {7., -0.0558578497921943}, {8., 0.400631250717115}, {
     9., -0.142879648773668}, {10., 0.313609451735458}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  AxesOrigin->{-10., 0},
  PlotRange->All,
  Ticks->{{{0, 
      FormBox["\"31.\"", TraditionalForm]}}, Automatic}]], "Output",
 ImageSize->{184, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->290323864]
}, Open  ]],

Cell[TextData[{
 "And here are the combined effects of rounding real numbers to the nearest \
machine number and taking the sine of the result. This is ",
 StyleBox["not",
  FontSlant->"Italic"],
 " a valid test of the sine function. The error in evaluating the sine \
function at a machine number near 31 is much smaller than the combination of \
errors resulting from first rounding a real number to the nearest machine \
number and then taking the sine of that machine number. "
}], "MathCaption",
 CellID->30960],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopicErrorPlot", "[", 
  RowBox[{
   RowBox[{"Sin", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "31", ",", "10"}], "}"}], ",", 
   RowBox[{"Joined", "\[Rule]", "Real"}]}], "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->15583],

Cell[BoxData[
 GraphicsBox[{
   {PointSize[0.02], PointBox[{-10., 0.183827441535495}], 
    PointBox[{-9., -0.359683457953635}], PointBox[{-8., 0.0968056425571445}], 
    PointBox[{-7., -0.446705256932168}], PointBox[{-6., 0.00978384357842694}],
     PointBox[{-5., 0.46627294408893}], PointBox[{-4., -0.0772379554006581}], 
    PointBox[{-3., 0.379251145109662}], PointBox[{-2., -0.164259754380111}], 
    PointBox[{-1., 0.292229346130025}], PointBox[{0., -0.251281553359931}], 
    PointBox[{1., 0.205207547150022}], PointBox[{2., -0.338303352340118}], 
    PointBox[{3., 0.11818574816965}], PointBox[{4., -0.425325151320673}], 
    PointBox[{5., 0.0311639491889118}], PointBox[{6., 0.487653049698405}], 
    PointBox[{7., -0.0558578497921943}], PointBox[{8., 0.400631250717115}], 
    PointBox[{9., -0.142879648773668}], PointBox[{10., 0.313609451735458}]}, 
   {Thickness[0.001], 
    LineBox[{{-9.5, -29.0879280082091}, {-9.5, 
     28.9120719917909}, {-8.5, -29.6314389076982}, {-8.5, 
     29.3685610923018}, {-7.5, -29.1749498071875}, {-7.5, 
     28.8250501928125}, {-6.5, -29.7184607066769}, {-6.5, 
     29.2815392933231}, {-5.5, -29.2619716061663}, {-5.5, 
     29.7380283938337}, {-4.5, -28.8054825056559}, {-4.5, 
     29.1945174943441}, {-3.5, -29.3489934051455}, {-3.5, 
     29.6510065948545}, {-2.5, -28.8925043046352}, {-2.5, 
     29.1074956953648}, {-1.5, -29.436015204125}, {-1.5, 
     29.563984795875}, {-0.5, -28.9795261036149}, {-0.5, 29.0204738963851}, {
     0.5, -29.5230370031049}, {0.5, 29.4769629968951}, {
     1.5, -29.066547902595}, {1.5, 28.933452097405}, {
     2.5, -29.6100588020852}, {2.5, 29.3899411979148}, {
     3.5, -29.1535697015755}, {3.5, 28.8464302984245}, {
     4.5, -29.6970806010659}, {4.5, 29.3029193989341}, {
     5.5, -29.2405915005563}, {5.5, 29.7594084994437}, {
     6.5, -28.7841024000469}, {6.5, 29.2158975999531}, {
     7.5, -29.3276132995375}, {7.5, 29.6723867004625}, {
     8.5, -28.8711241990283}, {8.5, 29.1288758009717}, {
     9.5, -29.4146350985191}, {9.5, 29.5853649014809}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.618033988749895],
  Axes->True,
  AxesOrigin->{-10., 0},
  PlotRange->All,
  Ticks->{{{0, 
      FormBox["\"31.\"", TraditionalForm]}}, Automatic}]], "Output",
 ImageSize->{184, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->280016469]
}, Open  ]],

Cell[TextData[{
 "Here you see how bad the square root function really is. It appears that \
near ",
 Cell[BoxData[
  FormBox["3.3", TraditionalForm]], "InlineFormula"],
 " its error is bounded by about ",
 Cell[BoxData[
  FormBox["1.0", TraditionalForm]], "InlineFormula"],
 " ulps. "
}], "MathCaption",
 CellID->4122],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopicErrorPlot", "[", 
  RowBox[{
   SqrtBox["x"], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "3.3", ",", "100"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->12153],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 100},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->741984904]
}, Open  ]],

Cell[TextData[{
 "You cannot examine this function exactly at 1 since it is singular there. \
You can, however, examine it near 1. The label for ",
 StyleBox["1", "MR"],
 " ",
 StyleBox["+", "MR"],
 " ",
 StyleBox["300", "MR"],
 " ",
 StyleBox["Ulp[1]", "MR"],
 " is printed as ",
 StyleBox["1.", "MR"],
 " because the label is limited to 6 characters. "
}], "MathCaption",
 CellID->9371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MicroscopicErrorPlot", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", 
     RowBox[{"x", "-", "1"}]], "-", 
    FractionBox["1", 
     RowBox[{"x", "+", "1"}]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"1", "+", 
      RowBox[{"300", " ", 
       RowBox[{"Ulp", "[", "1", "]"}]}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->24008],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->25121472]
}, Open  ]],

Cell[TextData[{
 "One point needs to be made clear here since rounding errors are often \
misunderstood and blame is incorrectly placed on an algorithm that has \
nothing to do with the error. Algorithms are designed to map the set of \
machine numbers into the set of machine numbers. To consider other numbers \
does not make sense. As far as the computer is concerned there ",
 StyleBox["are",
  FontSlant->"Italic"],
 " no other numbers. So, for example, if you want to find the square root of \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"49", 
    RowBox[{"\[Pi]", "/", "50"}]}], TraditionalForm]], "InlineFormula"],
 ", you might do it as follows. "
}], "Text",
 CellID->1575],

Cell[TextData[{
 "Start with a good numerical approximation to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"49", 
    RowBox[{"\[Pi]", "/", "50"}]}], TraditionalForm]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->5058],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"a", "=", 
  RowBox[{"N", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"49", " ", "\[Pi]"}], "50"], ",", "30"}], "]"}]}]], "Input",
 CellLabel->"In[17]:=",
 CellID->32699],

Cell[BoxData["3.078760800517997373693390515613912826513226011386`30."], \
"Output",
 ImageSize->{221, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->634297192]
}, Open  ]],

Cell[TextData[{
 "This finds the nearest machine number to ",
 StyleBox["a", "MR"],
 ". "
}], "MathCaption",
 CellID->460],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"N", "[", "a", "]"}]}]], "Input",
 CellLabel->"In[18]:=",
 CellID->6737],

Cell[BoxData["3.078760800517997`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->7296221]
}, Open  ]],

Cell[TextData[{
 "This finds the square root of the machine number ",
 StyleBox["b", "MR"],
 " using the machine\[Hyphen]precision algorithm for square root. "
}], "MathCaption",
 CellID->31406],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"c", "=", 
  SqrtBox["b"]}]], "Input",
 CellLabel->"In[19]:=",
 CellID->12755],

Cell[BoxData["1.7546397922417003`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->614456328]
}, Open  ]],

Cell[TextData[{
 "Here the error in ",
 StyleBox["c", "MR"],
 " is measured in ulps. This result gives the error for the particular \
machine on which this example was run. Your machine may give a very different \
result. "
}], "MathCaption",
 CellID->24539],

Cell[CellGroupData[{

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"SetPrecision", "[", 
    RowBox[{"c", ",", "30"}], "]"}], "-", 
   SqrtBox["a"]}], "$MachineEpsilon"]], "Input",
 CellLabel->"In[20]:=",
 CellID->2012],

Cell[BoxData[
 RowBox[{"-", "0.22218789820841633`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->662297251]
}, Open  ]],

Cell[TextData[{
 "Here the error is less than half an ulp. However, this error is due to both \
the square root function and the rounding error incurred in converting the \
number ",
 StyleBox["a", "MR"],
 " to the machine number ",
 StyleBox["b", "MR"],
 ". In fact, the square root function has an error of less than 0.1 ulps. "
}], "Text",
 CellID->18897],

Cell[TextData[{
 "Here is the correct value for the square root of ",
 StyleBox["b", "MR"],
 ". "
}], "MathCaption",
 CellID->13672],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"d", "=", 
  SqrtBox[
   RowBox[{"SetPrecision", "[", 
    RowBox[{"b", ",", "30"}], "]"}]]}]], "Input",
 CellLabel->"In[21]:=",
 CellID->14417],

Cell[BoxData["1.754639792241700292182175473517310164292198791789`30.\
30102999566398"], "Output",
 ImageSize->{228, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->25450052]
}, Open  ]],

Cell["This shows the error in the square root function. ", "MathCaption",
 CellID->3982],

Cell[CellGroupData[{

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"SetPrecision", "[", 
    RowBox[{"c", ",", "30"}], "]"}], "-", "d"}], "$MachineEpsilon"]], "Input",\

 CellLabel->"In[22]:=",
 CellID->23828],

Cell[BoxData["0.04581618252996657`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->245422680]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Computer Arithmetic Package - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 56, 59.4414650}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Computer Arithmetic Package Tutorial", "language" -> "en", "paclet" -> 
    "ComputerArithmetic Package", "status" -> "None", "summary" -> 
    "The arithmetic used by Mathematica is a mixture of variable precision \
software arithmetic and whatever is provided by the manufacturer of the \
floating-point hardware (or the designer of the compiler, if there is no \
floating-point hardware). If you want to learn about the basic ideas of \
computer floating-point arithmetic in general or examine the machine \
arithmetic on your machine, you can use ComputerArithmetic`. This allows you \
to examine arithmetic with various bases, precisions, and rounding rules. \
Computer numbers and nonnumbers in ComputerArithmetic. Much of the \
information carried around in the data object that makes up a computer number \
is redundant. In particular, the first three arguments contain exactly the \
same information as the fourth argument. The redundancy exists partly for the \
sake of efficiency and partly to allow the user access to the various fields. \
The fifth argument has nothing to do with the computer number itself. It \
instead represents what the value of the number would be without the \
cumulative effects of all the roundoff errors that went into the computer \
number. It is computed using Mathematica's high-precision arithmetic and can \
generally be regarded as the correct value of the number. Comparing the \
computer number with this number gives the error in the computer number. ", 
    "synonyms" -> {}, "title" -> "Computer Arithmetic Package", "type" -> 
    "Tutorial", "uri" -> "ComputerArithmetic/tutorial/ComputerArithmetic"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 2092, 41, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2717, 68, 60, 1, 70, "Title",
 CellID->29267],
Cell[2780, 71, 659, 14, 70, "Text",
 CellID->28540],
Cell[3442, 87, 1606, 48, 70, "DefinitionBox",
 CellID->10325],
Cell[5051, 137, 131, 5, 70, "Caption",
 CellID->10878],
Cell[5185, 144, 891, 15, 70, "Text",
 CellID->9428],
Cell[6079, 161, 63, 1, 70, "MathCaption",
 CellID->24207],
Cell[6145, 164, 102, 3, 70, "Input",
 CellID->30103],
Cell[6250, 169, 111, 3, 70, "MathCaption",
 CellID->26754],
Cell[CellGroupData[{
Cell[6386, 176, 128, 4, 28, "Input",
 CellID->6567],
Cell[6517, 182, 286, 10, 36, "Output",
 CellID->394187889]
}, Open  ]],
Cell[6818, 195, 181, 6, 70, "MathCaption",
 CellID->17258],
Cell[CellGroupData[{
Cell[7024, 205, 133, 4, 70, "Input",
 CellID->22857],
Cell[7160, 211, 288, 10, 36, "Output",
 CellID->69753290]
}, Open  ]],
Cell[7463, 224, 116, 3, 70, "MathCaption",
 CellID->3069],
Cell[CellGroupData[{
Cell[7604, 231, 111, 4, 70, "Input",
 CellID->21922],
Cell[7718, 237, 288, 10, 36, "Output",
 CellID->66468185]
}, Open  ]],
Cell[8021, 250, 116, 5, 70, "MathCaption",
 CellID->1381],
Cell[CellGroupData[{
Cell[8162, 259, 101, 3, 70, "Input",
 CellID->11821],
Cell[8266, 264, 252, 8, 66, "Output",
 CellID->343846521]
}, Open  ]],
Cell[8533, 275, 99, 2, 70, "MathCaption",
 CellID->21517],
Cell[CellGroupData[{
Cell[8657, 281, 180, 6, 70, "Input",
 CellID->28073],
Cell[8840, 289, 297, 10, 36, "Output",
 CellID->480032946]
}, Open  ]],
Cell[9152, 302, 335, 8, 70, "MathCaption",
 CellID->11046],
Cell[CellGroupData[{
Cell[9512, 314, 160, 5, 70, "Input",
 CellID->10805],
Cell[9675, 321, 160, 5, 36, "Output",
 CellID->776906102]
}, Open  ]],
Cell[9850, 329, 962, 31, 70, "DefinitionBox",
 CellID->15268],
Cell[10815, 362, 78, 1, 70, "Caption",
 CellID->7103],
Cell[10896, 365, 759, 19, 70, "MathCaption",
 CellID->9247],
Cell[CellGroupData[{
Cell[11680, 388, 97, 3, 70, "Input",
 CellID->10714],
Cell[11780, 393, 498, 14, 54, "Output",
 CellID->24576422]
}, Open  ]],
Cell[12293, 410, 93, 1, 70, "MathCaption",
 CellID->12700],
Cell[CellGroupData[{
Cell[12411, 415, 127, 4, 70, "Input",
 CellID->2944],
Cell[12541, 421, 498, 14, 54, "Output",
 CellID->262343759]
}, Open  ]],
Cell[13054, 438, 113, 3, 70, "MathCaption",
 CellID->6899],
Cell[CellGroupData[{
Cell[13192, 445, 111, 3, 70, "Input",
 CellID->28942],
Cell[13306, 450, 321, 11, 36, "Output",
 CellID->383173507]
}, Open  ]],
Cell[13642, 464, 377, 7, 70, "Text",
 CellID->4473],
Cell[14022, 473, 1519, 43, 70, "DefinitionBox3Col",
 CellID->27241],
Cell[15544, 518, 194, 7, 70, "Caption",
 CellID->7856],
Cell[15741, 527, 1362, 44, 70, "Text",
 CellID->16709],
Cell[17106, 573, 921, 26, 70, "DefinitionBox",
 CellID->25913],
Cell[18030, 601, 64, 1, 70, "Caption",
 CellID->14452],
Cell[18097, 604, 164, 4, 70, "MathCaption",
 CellID->6893],
Cell[CellGroupData[{
Cell[18286, 612, 134, 4, 70, "Input",
 CellID->21175],
Cell[18423, 618, 348, 12, 36, "Output",
 CellID->72070398]
}, Open  ]],
Cell[18786, 633, 149, 4, 70, "MathCaption",
 CellID->19972],
Cell[CellGroupData[{
Cell[18960, 641, 180, 5, 70, "Input",
 CellID->5826],
Cell[19143, 648, 499, 14, 54, "Output",
 CellID->116751536]
}, Open  ]],
Cell[19657, 665, 67, 1, 70, "MathCaption",
 CellID->1886],
Cell[CellGroupData[{
Cell[19749, 670, 134, 4, 70, "Input",
 CellID->17283],
Cell[19886, 676, 324, 11, 36, "Output",
 CellID->77348868]
}, Open  ]],
Cell[20225, 690, 199, 5, 70, "Text",
 CellID->19612],
Cell[20427, 697, 85, 1, 70, "MathCaption",
 CellID->14894],
Cell[CellGroupData[{
Cell[20537, 702, 130, 4, 70, "Input",
 CellID->10853],
Cell[20670, 708, 500, 14, 54, "Output",
 CellID->216443954]
}, Open  ]],
Cell[21185, 725, 122, 3, 70, "MathCaption",
 CellID->19999],
Cell[CellGroupData[{
Cell[21332, 732, 157, 5, 70, "Input",
 CellID->30896],
Cell[21492, 739, 297, 10, 36, "Output",
 CellID->86518910]
}, Open  ]],
Cell[21804, 752, 158, 4, 70, "MathCaption",
 CellID->4470],
Cell[CellGroupData[{
Cell[21987, 760, 198, 6, 70, "Input",
 CellID->14222],
Cell[22188, 768, 295, 10, 36, "Output",
 CellID->231552149]
}, Open  ]],
Cell[22498, 781, 107, 3, 70, "MathCaption",
 CellID->4255],
Cell[CellGroupData[{
Cell[22630, 788, 393, 13, 70, "Input",
 CellID->19942],
Cell[23026, 803, 541, 17, 68, "Output",
 CellID->81969117]
}, Open  ]],
Cell[23582, 823, 235, 5, 70, "MathCaption",
 CellID->8119],
Cell[CellGroupData[{
Cell[23842, 832, 435, 14, 70, "Input",
 CellID->26066],
Cell[24280, 848, 542, 17, 68, "Output",
 CellID->142925106]
}, Open  ]],
Cell[24837, 868, 132, 4, 70, "MathCaption",
 CellID->21654],
Cell[CellGroupData[{
Cell[24994, 876, 364, 12, 70, "Input",
 CellID->31474],
Cell[25361, 890, 545, 17, 68, "Output",
 CellID->95426693]
}, Open  ]],
Cell[25921, 910, 93, 1, 70, "MathCaption",
 CellID->30651],
Cell[CellGroupData[{
Cell[26039, 915, 406, 13, 70, "Input",
 CellID->22385],
Cell[26448, 930, 544, 17, 68, "Output",
 CellID->188064292]
}, Open  ]],
Cell[27007, 950, 172, 4, 70, "MathCaption",
 CellID->30047],
Cell[CellGroupData[{
Cell[27204, 958, 192, 6, 70, "Input",
 CellID->1819],
Cell[27399, 966, 326, 11, 36, "Output",
 CellID->376894762]
}, Open  ]],
Cell[27740, 980, 69, 1, 70, "MathCaption",
 CellID->2829],
Cell[CellGroupData[{
Cell[27834, 985, 145, 5, 70, "Input",
 CellID->21434],
Cell[27982, 992, 290, 10, 36, "Output",
 CellID->392279888]
}, Open  ]],
Cell[28287, 1005, 197, 4, 70, "MathCaption",
 CellID->29570],
Cell[CellGroupData[{
Cell[28509, 1013, 91, 3, 70, "Input",
 CellID->26286],
Cell[28603, 1018, 287, 10, 36, "Output",
 CellID->10383949]
}, Open  ]],
Cell[28905, 1031, 92, 1, 70, "MathCaption",
 CellID->25231],
Cell[CellGroupData[{
Cell[29022, 1036, 243, 8, 70, "Input",
 CellID->22723],
Cell[29268, 1046, 290, 10, 36, "Output",
 CellID->481899894]
}, Open  ]],
Cell[29573, 1059, 371, 11, 70, "MathCaption",
 CellID->17173],
Cell[CellGroupData[{
Cell[29969, 1074, 106, 3, 70, "Input",
 CellID->18782],
Cell[30078, 1079, 500, 14, 54, "Output",
 CellID->309657029]
}, Open  ]],
Cell[30593, 1096, 64, 1, 70, "MathCaption",
 CellID->27348],
Cell[CellGroupData[{
Cell[30682, 1101, 133, 4, 70, "Input",
 CellID->9699],
Cell[30818, 1107, 287, 10, 36, "Output",
 CellID->240133286]
}, Open  ]],
Cell[31120, 1120, 173, 4, 70, "MathCaption",
 CellID->303],
Cell[CellGroupData[{
Cell[31318, 1128, 134, 4, 70, "Input",
 CellID->14190],
Cell[31455, 1134, 287, 10, 36, "Output",
 CellID->383518689]
}, Open  ]],
Cell[31757, 1147, 119, 3, 70, "MathCaption",
 CellID->19879],
Cell[CellGroupData[{
Cell[31901, 1154, 134, 4, 70, "Input",
 CellID->12951],
Cell[32038, 1160, 287, 10, 36, "Output",
 CellID->219552471]
}, Open  ]],
Cell[32340, 1173, 145, 4, 70, "MathCaption",
 CellID->24768],
Cell[CellGroupData[{
Cell[32510, 1181, 260, 9, 70, "Input",
 CellID->27812],
Cell[32773, 1192, 342, 12, 36, "Output",
 CellID->581122782]
}, Open  ]],
Cell[33130, 1207, 128, 4, 70, "MathCaption",
 CellID->32646],
Cell[CellGroupData[{
Cell[33283, 1215, 128, 4, 70, "Input",
 CellID->32147],
Cell[33414, 1221, 345, 12, 36, "Output",
 CellID->284035651]
}, Open  ]],
Cell[33774, 1236, 663, 19, 70, "MathCaption",
 CellID->22894],
Cell[CellGroupData[{
Cell[34462, 1259, 169, 5, 70, "Input",
 CellID->12596],
Cell[34634, 1266, 345, 12, 36, "Output",
 CellID->195998596]
}, Open  ]],
Cell[34994, 1281, 262, 8, 70, "MathCaption",
 CellID->764],
Cell[CellGroupData[{
Cell[35281, 1293, 143, 4, 70, "Input",
 CellID->3635],
Cell[35427, 1299, 190, 6, 51, "Output",
 CellID->12872243]
}, Open  ]],
Cell[35632, 1308, 73, 1, 70, "MathCaption",
 CellID->839],
Cell[CellGroupData[{
Cell[35730, 1313, 314, 9, 70, "Input",
 CellID->20520],
Cell[36047, 1324, 496, 14, 54, "Output",
 CellID->666299335]
}, Open  ]],
Cell[36558, 1341, 156, 4, 70, "MathCaption",
 CellID->15644],
Cell[36717, 1347, 331, 10, 70, "Input",
 CellID->30346],
Cell[37051, 1359, 132, 4, 70, "MathCaption",
 CellID->13294],
Cell[CellGroupData[{
Cell[37208, 1367, 506, 16, 47, "Input",
 CellID->8298],
Cell[37717, 1385, 4980, 86, 136, "Output",
 Evaluatable->False,
 CellID->466081890]
}, Open  ]],
Cell[42712, 1474, 80, 1, 70, "MathCaption",
 CellID->23955],
Cell[CellGroupData[{
Cell[42817, 1479, 283, 9, 28, "Input",
 CellID->11955],
Cell[43103, 1490, 5057, 87, 127, "Output",
 Evaluatable->False,
 CellID->102322869]
}, Open  ]],
Cell[48175, 1580, 465, 9, 70, "Text",
 CellID->10973],
Cell[48643, 1591, 610, 10, 70, "Text",
 CellID->22384],
Cell[49256, 1603, 1740, 42, 70, "Text",
 CellID->2306],
Cell[50999, 1647, 223, 6, 70, "Text",
 CellID->22299],
Cell[51225, 1655, 1873, 70, 70, "DefinitionBox",
 CellID->27215],
Cell[53101, 1727, 109, 3, 70, "Caption",
 CellID->1643],
Cell[53213, 1732, 1323, 39, 70, "Text",
 CellID->1634],
Cell[54539, 1773, 696, 15, 70, "Text",
 CellID->18761],
Cell[55238, 1790, 2182, 97, 70, "DefinitionBox",
 CellID->75390218],
Cell[57423, 1889, 80, 1, 70, "Caption",
 CellID->254074879],
Cell[57506, 1892, 1297, 26, 70, "Text",
 CellID->32516],
Cell[58806, 1920, 925, 31, 70, "DefinitionBox3Col",
 CellID->13940],
Cell[59734, 1953, 315, 11, 70, "Caption",
 CellID->9389],
Cell[60052, 1966, 71, 1, 70, "MathCaption",
 CellID->7994],
Cell[CellGroupData[{
Cell[60148, 1971, 95, 3, 70, "Input",
 CellID->16999],
Cell[60246, 1976, 181, 5, 36, "Output",
 CellID->219130507]
}, Open  ]],
Cell[60442, 1984, 81, 1, 70, "MathCaption",
 CellID->5075],
Cell[CellGroupData[{
Cell[60548, 1989, 228, 7, 70, "Input",
 CellID->24504],
Cell[60779, 1998, 287, 9, 39, "Output",
 CellID->18416162]
}, Open  ]],
Cell[61081, 2010, 69, 1, 70, "MathCaption",
 CellID->26096],
Cell[CellGroupData[{
Cell[61175, 2015, 188, 6, 70, "Input",
 CellID->1618],
Cell[61366, 2023, 295, 10, 39, "Output",
 CellID->49838940]
}, Open  ]],
Cell[61676, 2036, 281, 8, 70, "MathCaption",
 CellID->26756],
Cell[61960, 2046, 100, 3, 70, "Input",
 CellID->22869420],
Cell[CellGroupData[{
Cell[62085, 2053, 224, 8, 70, "Input",
 CellID->5771],
Cell[62312, 2063, 219, 7, 36, "Output",
 CellID->271228651]
}, Open  ]],
Cell[62546, 2073, 82, 1, 70, "MathCaption",
 CellID->15031],
Cell[CellGroupData[{
Cell[62653, 2078, 257, 9, 70, "Input",
 CellID->26176],
Cell[62913, 2089, 201, 6, 36, "Output",
 CellID->11074195]
}, Open  ]],
Cell[63129, 2098, 211, 7, 70, "MathCaption",
 CellID->28743],
Cell[CellGroupData[{
Cell[63365, 2109, 406, 14, 70, "Input",
 CellID->30524],
Cell[63774, 2125, 202, 6, 36, "Output",
 CellID->120849159]
}, Open  ]],
Cell[63991, 2134, 772, 21, 70, "MathCaption",
 CellID->17027],
Cell[CellGroupData[{
Cell[64788, 2159, 345, 11, 70, "Input",
 CellID->4430],
Cell[65136, 2172, 218, 7, 36, "Output",
 CellID->501237352]
}, Open  ]],
Cell[65369, 2182, 187, 6, 70, "MathCaption",
 CellID->22123],
Cell[CellGroupData[{
Cell[65581, 2192, 257, 8, 28, "Input",
 CellID->7912],
Cell[65841, 2202, 839, 18, 124, "Output",
 Evaluatable->False,
 CellID->190401268]
}, Open  ]],
Cell[66695, 2223, 337, 10, 70, "MathCaption",
 CellID->29190],
Cell[CellGroupData[{
Cell[67057, 2237, 221, 7, 70, "Input",
 CellID->22634],
Cell[67281, 2246, 1286, 26, 124, "Output",
 CellID->52840514]
}, Open  ]],
Cell[68582, 2275, 409, 13, 70, "MathCaption",
 CellID->3078],
Cell[CellGroupData[{
Cell[69016, 2292, 269, 8, 28, "Input",
 CellID->20010],
Cell[69288, 2302, 1233, 24, 124, "Output",
 Evaluatable->False,
 CellID->93407858]
}, Open  ]],
Cell[70536, 2329, 482, 11, 70, "MathCaption",
 CellID->14845],
Cell[CellGroupData[{
Cell[71043, 2344, 276, 8, 28, "Input",
 CellID->8581],
Cell[71322, 2354, 1287, 25, 121, "Output",
 Evaluatable->False,
 CellID->216566391]
}, Open  ]],
Cell[72624, 2382, 158, 4, 70, "MathCaption",
 CellID->28704],
Cell[CellGroupData[{
Cell[72807, 2390, 276, 8, 70, "Input",
 CellID->31974],
Cell[73086, 2400, 1840, 34, 121, "Output",
 CellID->290323864]
}, Open  ]],
Cell[74941, 2437, 516, 10, 70, "MathCaption",
 CellID->30960],
Cell[CellGroupData[{
Cell[75482, 2451, 276, 8, 70, "Input",
 CellID->15583],
Cell[75761, 2461, 2386, 44, 129, "Output",
 CellID->280016469]
}, Open  ]],
Cell[78162, 2508, 319, 10, 70, "MathCaption",
 CellID->4122],
Cell[CellGroupData[{
Cell[78506, 2522, 212, 7, 36, "Input",
 CellID->12153],
Cell[78721, 2531, 8896, 150, 121, "Output",
 Evaluatable->False,
 CellID->741984904]
}, Open  ]],
Cell[87632, 2684, 388, 14, 70, "MathCaption",
 CellID->9371],
Cell[CellGroupData[{
Cell[88045, 2702, 397, 14, 42, "Input",
 CellID->24008],
Cell[88445, 2718, 2726, 49, 124, "Output",
 Evaluatable->False,
 CellID->25121472]
}, Open  ]],
Cell[91186, 2770, 680, 16, 70, "Text",
 CellID->1575],
Cell[91869, 2788, 222, 8, 70, "MathCaption",
 CellID->5058],
Cell[CellGroupData[{
Cell[92116, 2800, 197, 7, 70, "Input",
 CellID->32699],
Cell[92316, 2809, 215, 6, 36, "Output",
 CellID->634297192]
}, Open  ]],
Cell[92546, 2818, 122, 5, 70, "MathCaption",
 CellID->460],
Cell[CellGroupData[{
Cell[92693, 2827, 116, 4, 70, "Input",
 CellID->6737],
Cell[92812, 2833, 174, 5, 36, "Output",
 CellID->7296221]
}, Open  ]],
Cell[93001, 2841, 194, 5, 70, "MathCaption",
 CellID->31406],
Cell[CellGroupData[{
Cell[93220, 2850, 101, 4, 70, "Input",
 CellID->12755],
Cell[93324, 2856, 177, 5, 36, "Output",
 CellID->614456328]
}, Open  ]],
Cell[93516, 2864, 258, 7, 70, "MathCaption",
 CellID->24539],
Cell[CellGroupData[{
Cell[93799, 2875, 202, 7, 70, "Input",
 CellID->2012],
Cell[94004, 2884, 195, 6, 36, "Output",
 CellID->662297251]
}, Open  ]],
Cell[94214, 2893, 358, 9, 70, "Text",
 CellID->18897],
Cell[94575, 2904, 132, 5, 70, "MathCaption",
 CellID->13672],
Cell[CellGroupData[{
Cell[94732, 2913, 167, 6, 70, "Input",
 CellID->14417],
Cell[94902, 2921, 228, 6, 36, "Output",
 CellID->25450052]
}, Open  ]],
Cell[95145, 2930, 88, 1, 70, "MathCaption",
 CellID->3982],
Cell[CellGroupData[{
Cell[95258, 2935, 192, 7, 70, "Input",
 CellID->23828],
Cell[95453, 2944, 178, 5, 36, "Output",
 CellID->245422680]
}, Open  ]]
}, Open  ]],
Cell[95658, 2953, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

