(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     34825,       1315]
NotebookOptionsPosition[     26699,       1028]
NotebookOutlinePosition[     27988,       1065]
CellTagsIndexPosition[     27902,       1060]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Fields" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicNumberFields"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Algebraic Number Fields\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"AlgebraicNumberDenominator" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumberDenominator"], 
          "AlgebraicUnitQ" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicUnitQ"], 
          "AlgebraicNumber" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumber"], 
          "MinimalPolynomial" :> 
          Documentation`HelpLookup["paclet:ref/MinimalPolynomial"], 
          "RootReduce" :> Documentation`HelpLookup["paclet:ref/RootReduce"], 
          "Algebraics" :> 
          Documentation`HelpLookup["paclet:ref/Algebraics"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"AlgebraicNumberDenominator\"\>", 
       2->"\<\"AlgebraicUnitQ\"\>", 3->"\<\"AlgebraicNumber\"\>", 
       4->"\<\"MinimalPolynomial\"\>", 5->"\<\"RootReduce\"\>", 
       6->"\<\"Algebraics\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumberTheory"], 
          "Number Recognition" :> 
          Documentation`HelpLookup["paclet:guide/NumberRecognition"], 
          "Testing Expressions" :> 
          Documentation`HelpLookup["paclet:guide/TestingExpressions"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Algebraic Number Theory\"\>",
        2->"\<\"Number Recognition\"\>", 
       3->"\<\"Testing Expressions\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["AlgebraicIntegerQ", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["AlgebraicIntegerQ",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/AlgebraicIntegerQ"], "[", 
       StyleBox["a", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]yields ",
     Cell[BoxData[
      ButtonBox["True",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/True"]], "InlineFormula"],
     " if ",
     Cell[BoxData[
      StyleBox["a", "TI"]], "InlineFormula"],
     " is an algebraic integer, and yields ",
     Cell[BoxData[
      ButtonBox["False",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/False"]], "InlineFormula"],
     " otherwise."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
A number is considered to be an algebraic integer if the leading coefficient \
of its minimal polynomial is 1.\
\>", "Notes",
 CellID->1067943069]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->711098073],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(3)", "ExampleCount"]
}], "ExampleSection",
 CellID->148544992],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"Sqrt", "[", "2", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->94706934],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->149705113]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->45467489],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"1", "/", 
   RowBox[{"Sqrt", "[", "2", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->128095263],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1390483554]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->554564049],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"1", "+", 
     RowBox[{"Sqrt", "[", "5", "]"}]}], ")"}], "/", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8626734],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->940045424]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Radical expressions:", "ExampleText",
 CellID->59173675],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{
   RowBox[{"2", "^", 
    RowBox[{"(", 
     RowBox[{"1", "/", "3"}], ")"}]}], "+", 
   RowBox[{"4", "^", 
    RowBox[{"(", 
     RowBox[{"1", "/", "5"}], ")"}]}], "+", "17"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->766805639],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->306250189]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"1", "+", "I"}], ")"}], "/", 
   RowBox[{"Sqrt", "[", "2", "]"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->182625958],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1776108311]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1483870734],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->501764796],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"#1", "^", "2"}], "-", 
       RowBox[{"2", " ", "#1"}], "+", "7"}], "&"}], ",", "1"}], "]"}], "+", 
   "17"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1918551892],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1029689677]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"Root", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"2", 
       RowBox[{"#1", "^", "2"}]}], "-", 
      RowBox[{"2", " ", "#1"}], "+", "7"}], "&"}], ",", "1"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->171690775],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1571173813]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"2", ",", " ", "1"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2099317225],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1259351579]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->415671942],

Cell["Transcendental expressions: ", "ExampleText",
 CellID->90034276],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", "Pi", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1295560843],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->323163421]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"Exp", "[", "Pi", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18795415],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->421473484]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"Sin", "[", "1", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->33200644],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1303777782]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->158341992],

Cell["Approximate numeric expressions: ", "ExampleText",
 CellID->333892030],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", "1.2", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->697163619],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1817531733]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1075857913],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicIntegerQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicIntegerQ"]], "InlineFormula"],
 " automatically threads over lists:"
}], "ExampleText",
 CellID->1229914907],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"{", 
   RowBox[{"E", ",", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"-", "2"}], "]"}], ",", 
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "2", "]"}]}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2120704657],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"False", ",", "True", ",", "False"}], "}"}]], "Output",
 ImageSize->{139, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1448441603]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 "Recognize a Pisot number [",
 ButtonBox["more info",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://mathworld.wolfram.com/PisotNumber.html"], None}],
 "]:"
}], "ExampleText",
 CellID->72513110],

Cell[BoxData[
 RowBox[{
  RowBox[{"pisotNumberQ", "[", "a_", "]"}], ":=", " ", 
  RowBox[{
   RowBox[{"AlgebraicIntegerQ", "[", "a", "]"}], " ", "&&", " ", 
   RowBox[{"Element", "[", 
    RowBox[{"a", ",", "Reals"}], "]"}], " ", "&&", " ", 
   RowBox[{"(", 
    RowBox[{"a", ">", "1"}], ")"}], " ", "&&", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Count", "[", 
      RowBox[{
       RowBox[{"List", " ", "@@", " ", 
        RowBox[{"(", 
         RowBox[{"Last", " ", "/@", " ", 
          RowBox[{"Roots", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"MinimalPolynomial", "[", 
              RowBox[{"a", ",", "x"}], "]"}], "\[Equal]", "0"}], ",", "x"}], 
           "]"}]}], ")"}]}], ",", 
       RowBox[{"_", "?", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"Abs", "[", "#", "]"}], ">", "1"}], " ", "&"}], ")"}]}]}], 
      "]"}], " ", "\[Equal]", "1"}], ")"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1874209288],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pisotNumberQ", "[", "GoldenRatio", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1503214152],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->83399229]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"pisotNumberQ", "[", 
  RowBox[{"Root", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "-", "#1", "+", 
      RowBox[{"#1", "^", "3"}]}], " ", "&"}], ",", "1"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->918528068],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1531503959]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell["\<\
The sum and product of algebraic integers are algebraic integers:\
\>", "ExampleText",
 CellID->420667128],

Cell[BoxData[{
 RowBox[{
  RowBox[{"a1", "=", 
   RowBox[{"Sqrt", "[", "2", "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"a2", "=", 
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{"Pi", "*", 
      RowBox[{"I", "/", "7"}]}], ")"}]}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->38132864],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"a1", "+", "a2"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->69211719],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->248045971]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"a1", " ", "a2"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->100278137],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->642875770]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->353579879],

Cell[TextData[{
 "The leading coefficient of the ",
 Cell[BoxData[
  ButtonBox["MinimalPolynomial",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MinimalPolynomial"]], "InlineFormula"],
 " is one for algebraic integers:"
}], "ExampleText",
 CellID->141313866],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], " ", 
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{"Pi", "*", 
      RowBox[{"I", "/", "7"}]}], ")"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->349728433],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->822838333]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], " ", 
    RowBox[{"E", "^", 
     RowBox[{"(", 
      RowBox[{"Pi", "*", 
       RowBox[{"I", "/", "7"}]}], ")"}]}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->45723977],

Cell[BoxData[
 RowBox[{"64", "+", 
  RowBox[{"32", " ", 
   SuperscriptBox["x", "2"]}], "+", 
  RowBox[{"16", " ", 
   SuperscriptBox["x", "4"]}], "+", 
  RowBox[{"8", " ", 
   SuperscriptBox["x", "6"]}], "+", 
  RowBox[{"4", " ", 
   SuperscriptBox["x", "8"]}], "+", 
  RowBox[{"2", " ", 
   SuperscriptBox["x", "10"]}], "+", 
  SuperscriptBox["x", "12"]}]], "Output",
 ImageSize->{243, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1796659698]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->129506924],

Cell["\<\
If a number is an algebraic integer it is an algebraic number:\
\>", "ExampleText",
 CellID->93963083],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"Sqrt", "[", "2", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->166058389],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->691983819]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Element", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], ",", "Algebraics"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->78346027],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1548097428]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1462886412],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["NumberFieldIntegralBasis",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NumberFieldIntegralBasis"]], "InlineFormula"],
 " to get the integral basis for a number field:"
}], "ExampleText",
 CellID->191651981],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"a", " ", "=", " ", 
  RowBox[{"NumberFieldIntegralBasis", "[", 
   RowBox[{"3", "^", 
    RowBox[{"(", 
     RowBox[{"1", "/", "3"}], ")"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->321097749],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"-", "3"}], "+", 
         SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}], ",", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"-", "3"}], "+", 
         SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "}"}]], "Output",
 ImageSize->{342, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2109088991]
}, Open  ]],

Cell["Any integer linear combination will be an algebraic integer:", \
"ExampleText",
 CellID->107376240],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"a", ".", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1831419981],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1093594355]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1305812373],

Cell["\<\
In some cases it is not known whether the number is an algebraic integer:\
\>", "ExampleText",
 CellID->268466572],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"Pi", "+", "E"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->211849979],

Cell[BoxData[
 RowBox[{"AlgebraicIntegerQ", "[", 
  RowBox[{"\[ExponentialE]", "+", "\[Pi]"}], "]"}]], "Output",
 ImageSize->{165, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1875273459]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumberDenominator",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberDenominator"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicUnitQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicUnitQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumber",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumber"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MinimalPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MinimalPolynomial"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RootReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RootReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Algebraics",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Algebraics"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Algebraic Number Fields",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicNumberFields"]], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->777966367],

Cell[TextData[ButtonBox["Algebraic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumberTheory"]], "MoreAbout",
 CellID->662498452],

Cell[TextData[ButtonBox["Number Recognition",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberRecognition"]], "MoreAbout",
 CellID->706242981],

Cell[TextData[ButtonBox["Testing Expressions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/TestingExpressions"]], "MoreAbout",
 CellID->430197406]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"AlgebraicIntegerQ - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 23, 54.0484826}", "context" -> "System`", 
    "keywords" -> {"root of polynomial"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "AlgebraicIntegerQ[a] yields True if a is an algebraic integer, and \
yields False otherwise.", "synonyms" -> {"algebraic integer q"}, "title" -> 
    "AlgebraicIntegerQ", "type" -> "Symbol", "uri" -> 
    "ref/AlgebraicIntegerQ"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6517, 188, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->711098073]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 27759, 1053}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2729, 63, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3352, 90, 1582, 37, 70, "ObjectNameGrid"],
Cell[4937, 129, 872, 28, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5846, 162, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6324, 179, 156, 4, 70, "Notes",
 CellID->1067943069]
}, Closed]],
Cell[CellGroupData[{
Cell[6517, 188, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->711098073],
Cell[CellGroupData[{
Cell[6902, 202, 148, 5, 70, "ExampleSection",
 CellID->148544992],
Cell[CellGroupData[{
Cell[7075, 211, 143, 4, 28, "Input",
 CellID->94706934],
Cell[7221, 217, 161, 5, 36, "Output",
 CellID->149705113]
}, Open  ]],
Cell[7397, 225, 124, 3, 70, "ExampleDelimiter",
 CellID->45467489],
Cell[CellGroupData[{
Cell[7546, 232, 168, 5, 28, "Input",
 CellID->128095263],
Cell[7717, 239, 163, 5, 36, "Output",
 CellID->1390483554]
}, Open  ]],
Cell[7895, 247, 125, 3, 70, "ExampleDelimiter",
 CellID->554564049],
Cell[CellGroupData[{
Cell[8045, 254, 217, 7, 70, "Input",
 CellID->8626734],
Cell[8265, 263, 161, 5, 36, "Output",
 CellID->940045424]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8475, 274, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[8699, 283, 62, 1, 70, "ExampleText",
 CellID->59173675],
Cell[CellGroupData[{
Cell[8786, 288, 297, 10, 70, "Input",
 CellID->766805639],
Cell[9086, 300, 161, 5, 36, "Output",
 CellID->306250189]
}, Open  ]],
Cell[CellGroupData[{
Cell[9284, 310, 217, 7, 70, "Input",
 CellID->182625958],
Cell[9504, 319, 162, 5, 36, "Output",
 CellID->1776108311]
}, Open  ]],
Cell[9681, 327, 126, 3, 70, "ExampleDelimiter",
 CellID->1483870734],
Cell[9810, 332, 321, 12, 70, "ExampleText",
 CellID->501764796],
Cell[CellGroupData[{
Cell[10156, 348, 315, 11, 70, "Input",
 CellID->1918551892],
Cell[10474, 361, 162, 5, 36, "Output",
 CellID->1029689677]
}, Open  ]],
Cell[CellGroupData[{
Cell[10673, 371, 306, 11, 70, "Input",
 CellID->171690775],
Cell[10982, 384, 163, 5, 36, "Output",
 CellID->1571173813]
}, Open  ]],
Cell[CellGroupData[{
Cell[11182, 394, 269, 8, 70, "Input",
 CellID->2099317225],
Cell[11454, 404, 162, 5, 36, "Output",
 CellID->1259351579]
}, Open  ]],
Cell[11631, 412, 125, 3, 70, "ExampleDelimiter",
 CellID->415671942],
Cell[11759, 417, 70, 1, 70, "ExampleText",
 CellID->90034276],
Cell[CellGroupData[{
Cell[11854, 422, 115, 3, 70, "Input",
 CellID->1295560843],
Cell[11972, 427, 162, 5, 36, "Output",
 CellID->323163421]
}, Open  ]],
Cell[CellGroupData[{
Cell[12171, 437, 143, 4, 70, "Input",
 CellID->18795415],
Cell[12317, 443, 162, 5, 36, "Output",
 CellID->421473484]
}, Open  ]],
Cell[CellGroupData[{
Cell[12516, 453, 142, 4, 70, "Input",
 CellID->33200644],
Cell[12661, 459, 163, 5, 36, "Output",
 CellID->1303777782]
}, Open  ]],
Cell[12839, 467, 125, 3, 70, "ExampleDelimiter",
 CellID->158341992],
Cell[12967, 472, 76, 1, 70, "ExampleText",
 CellID->333892030],
Cell[CellGroupData[{
Cell[13068, 477, 115, 3, 70, "Input",
 CellID->697163619],
Cell[13186, 482, 163, 5, 36, "Output",
 CellID->1817531733]
}, Open  ]],
Cell[13364, 490, 126, 3, 70, "ExampleDelimiter",
 CellID->1075857913],
Cell[13493, 495, 230, 7, 70, "ExampleText",
 CellID->1229914907],
Cell[CellGroupData[{
Cell[13748, 506, 284, 9, 70, "Input",
 CellID->2120704657],
Cell[14035, 517, 226, 7, 36, "Output",
 CellID->1448441603]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14310, 530, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[14540, 539, 222, 8, 70, "ExampleText",
 CellID->72513110],
Cell[14765, 549, 984, 29, 70, "Input",
 CellID->1874209288],
Cell[CellGroupData[{
Cell[15774, 582, 119, 3, 70, "Input",
 CellID->1503214152],
Cell[15896, 587, 160, 5, 36, "Output",
 CellID->83399229]
}, Open  ]],
Cell[CellGroupData[{
Cell[16093, 597, 278, 10, 70, "Input",
 CellID->918528068],
Cell[16374, 609, 162, 5, 36, "Output",
 CellID->1531503959]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16585, 620, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[16826, 629, 116, 3, 70, "ExampleText",
 CellID->420667128],
Cell[16945, 634, 316, 11, 70, "Input",
 CellID->38132864],
Cell[CellGroupData[{
Cell[17286, 649, 137, 4, 70, "Input",
 CellID->69211719],
Cell[17426, 655, 161, 5, 36, "Output",
 CellID->248045971]
}, Open  ]],
Cell[CellGroupData[{
Cell[17624, 665, 138, 4, 70, "Input",
 CellID->100278137],
Cell[17765, 671, 161, 5, 36, "Output",
 CellID->642875770]
}, Open  ]],
Cell[17941, 679, 125, 3, 70, "ExampleDelimiter",
 CellID->353579879],
Cell[18069, 684, 262, 8, 70, "ExampleText",
 CellID->141313866],
Cell[CellGroupData[{
Cell[18356, 696, 271, 9, 70, "Input",
 CellID->349728433],
Cell[18630, 707, 161, 5, 36, "Output",
 CellID->822838333]
}, Open  ]],
Cell[CellGroupData[{
Cell[18828, 717, 299, 10, 70, "Input",
 CellID->45723977],
Cell[19130, 729, 501, 17, 36, "Output",
 CellID->1796659698]
}, Open  ]],
Cell[19646, 749, 125, 3, 70, "ExampleDelimiter",
 CellID->129506924],
Cell[19774, 754, 112, 3, 70, "ExampleText",
 CellID->93963083],
Cell[CellGroupData[{
Cell[19911, 761, 144, 4, 70, "Input",
 CellID->166058389],
Cell[20058, 767, 161, 5, 36, "Output",
 CellID->691983819]
}, Open  ]],
Cell[CellGroupData[{
Cell[20256, 777, 166, 5, 70, "Input",
 CellID->78346027],
Cell[20425, 784, 162, 5, 36, "Output",
 CellID->1548097428]
}, Open  ]],
Cell[20602, 792, 126, 3, 70, "ExampleDelimiter",
 CellID->1462886412],
Cell[20731, 797, 264, 8, 70, "ExampleText",
 CellID->191651981],
Cell[CellGroupData[{
Cell[21020, 809, 228, 7, 70, "Input",
 CellID->321097749],
Cell[21251, 818, 817, 27, 58, "Output",
 CellID->2109088991]
}, Open  ]],
Cell[22083, 848, 105, 2, 70, "ExampleText",
 CellID->107376240],
Cell[CellGroupData[{
Cell[22213, 854, 196, 6, 70, "Input",
 CellID->1831419981],
Cell[22412, 862, 162, 5, 36, "Output",
 CellID->1093594355]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22623, 873, 231, 7, 70, "ExampleSection",
 CellID->1305812373],
Cell[22857, 882, 124, 3, 70, "ExampleText",
 CellID->268466572],
Cell[CellGroupData[{
Cell[23006, 889, 137, 4, 70, "Input",
 CellID->211849979],
Cell[23146, 895, 244, 7, 36, "Output",
 CellID->1875273459]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[23451, 909, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[23770, 920, 1523, 53, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[25330, 978, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[25651, 989, 159, 3, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[25847, 997, 319, 9, 70, "MoreAboutSection",
 CellID->777966367],
Cell[26169, 1008, 156, 3, 70, "MoreAbout",
 CellID->662498452],
Cell[26328, 1013, 147, 3, 70, "MoreAbout",
 CellID->706242981],
Cell[26478, 1018, 149, 3, 70, "MoreAbout",
 CellID->430197406]
}, Open  ]],
Cell[26642, 1024, 27, 0, 70, "History"],
Cell[26672, 1026, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

