(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     65038,       2414]
NotebookOptionsPosition[     53274,       2004]
NotebookOutlinePosition[     54749,       2044]
CellTagsIndexPosition[     54663,       2039]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Fields" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicNumberFields"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Algebraic Number Fields\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Root" :> Documentation`HelpLookup["paclet:ref/Root"], 
          "AlgebraicNumberPolynomial" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumberPolynomial"], 
          "ToNumberField" :> 
          Documentation`HelpLookup["paclet:ref/ToNumberField"], "Algebraics" :> 
          Documentation`HelpLookup["paclet:ref/Algebraics"], "RootReduce" :> 
          Documentation`HelpLookup["paclet:ref/RootReduce"], 
          "MinimalPolynomial" :> 
          Documentation`HelpLookup["paclet:ref/MinimalPolynomial"], 
          "Extension" :> Documentation`HelpLookup["paclet:ref/Extension"], 
          "AlgebraicIntegerQ" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicIntegerQ"], 
          "AlgebraicUnitQ" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicUnitQ"], 
          "AlgebraicNumberTrace" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumberTrace"], 
          "AlgebraicNumberNorm" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumberNorm"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"Root\"\>", 
       2->"\<\"AlgebraicNumberPolynomial\"\>", 3->"\<\"ToNumberField\"\>", 
       4->"\<\"Algebraics\"\>", 5->"\<\"RootReduce\"\>", 
       6->"\<\"MinimalPolynomial\"\>", 7->"\<\"Extension\"\>", 
       8->"\<\"AlgebraicIntegerQ\"\>", 9->"\<\"AlgebraicUnitQ\"\>", 
       10->"\<\"AlgebraicNumberTrace\"\>", 
       11->"\<\"AlgebraicNumberNorm\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Numbers" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumbers"], 
          "Algebraic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumberTheory"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Algebraic Numbers\"\>", 
       2->"\<\"Algebraic Number Theory\"\>", 
       3->"\<\"New in 6.0: Symbolic Computation\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       5->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["AlgebraicNumber", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["AlgebraicNumber",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/AlgebraicNumber"], "[", 
       RowBox[{
        StyleBox["\[Theta]", "TR"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["c", "TI"], 
           StyleBox["0", "TR"]], ",", 
          SubscriptBox[
           StyleBox["c", "TI"], 
           StyleBox["1", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"], ",", 
          SubscriptBox[
           StyleBox["c", "TI"], 
           StyleBox["n", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the algebraic number in the field ",
     Cell[BoxData[
      FormBox[
       RowBox[{"\[DoubleStruckCapitalQ]", "[", "\[Theta]", "]"}], 
       TraditionalForm]], "InlineMath"],
     " given by ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["c", "0"], "+", 
        RowBox[{
         SubscriptBox["c", "1"], "\[Theta]"}], " ", "+", "\[Ellipsis]", "+", 
        RowBox[{
         SubscriptBox["c", "n"], " ", 
         SuperscriptBox["\[Theta]", "n"]}]}], TraditionalForm]], 
      "InlineMath"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->29430]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects in the same field are automatically combined by arithmetic \
operations."
}], "Notes",
 CellID->233291024],

Cell[TextData[{
 "The generator ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " can be any algebraic number, represented in terms of radicals or ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects. The coefficients ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["c", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " must be integers or rational numbers."
}], "Notes",
 CellID->3082117],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " is automatically reduced so that ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " is an algebraic integer, and the list of ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["c", "TI"], 
   StyleBox["i", "TI"]]], "InlineFormula"],
 " is of length equal to the degree of the minimal polynomial of ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->638031029],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects are always treated as numeric quantities."
}], "Notes",
 CellID->426868723],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["N",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/N"]], "InlineFormula"],
 " finds the approximate numerical value of an ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " object."
}], "Notes",
 CellID->12711],

Cell[TextData[{
 "Operations such as ",
 Cell[BoxData[
  ButtonBox["Abs",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Abs"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Re",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Re"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Round",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Round"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Less",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Less"]], "InlineFormula"],
 " can be used on ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects. "
}], "Notes",
 CellID->418],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["RootReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootReduce"]], "InlineFormula"],
 " can be used to transform ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects into ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects."
}], "Notes",
 CellID->32512],

Cell[TextData[{
 "A particular algebraic number can have many different representations as an \
",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " object. Each representation is characterized by the generator ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " specified for the field."
}], "Notes",
 CellID->38578760],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects representing integers or rational numbers are automatically \
reduced to explicit integer or rational form."
}], "Notes",
 CellID->40028476]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->354412043],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->153874062],

Cell["Represent an algebraic number:", "ExampleText",
 CellID->41531504],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"#", "^", "3"}], "+", "#", "+", "1"}], "&"}], ",", "3"}], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->185286594],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"1", "+", "#1", "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}]], "Output",
 ImageSize->{328, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1194236908]
}, Open  ]],

Cell["Do arithmetic:", "ExampleText",
 CellID->434020516],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"%", "^", "2"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->183036309],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"1", "+", "#1", "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", "2"}], ",", 
     RowBox[{"-", "1"}], ",", "5"}], "}"}]}], "]"}]], "Output",
 ImageSize->{344, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1157547894]
}, Open  ]],

Cell["Get a numerical approximation:", "ExampleText",
 CellID->46293880],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->236533123],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "8.50509198160593`"}], "+", 
  RowBox[{"2.801218562579987`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{134, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->853096285]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11000979],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects can be evaluated to any precision:"
}], "ExampleText",
 CellID->491609672],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Sqrt", "[", "2", "]"}], " ", "I"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{"-", "1"}]}], "}"}]}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->634278950],

Cell[BoxData[
 RowBox[{"1.`50.", "-", 
  RowBox[{
  "1.4142135623730950488016887242096980785696718753769480731766797379907`50.",
    " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{379, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1388434623]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->910589047],

Cell["\<\
Objects representing integers or rational numbers are automatically \
simplified:\
\>", "ExampleText",
 CellID->1301644407],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{"3", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->321636460],

Cell[BoxData["7"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->863400459]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"5", "/", "2"}], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2121649240],

Cell[BoxData["8"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->624511634]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->566959903],

Cell[TextData[{
 "The generator ",
 Cell[BoxData[
  StyleBox["\[Theta]", "TR"]], "InlineFormula"],
 " in ",
 Cell[BoxData[
  RowBox[{"AlgbraicNumber", "[", 
   RowBox[{
    StyleBox["\[Theta]", "TR"], ",", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["c", "TI"], 
       StyleBox["0", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"], ",", 
      SubscriptBox[
       StyleBox["c", "TI"], 
       StyleBox["n", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
 " will automatically reduce to an algebraic integer:"
}], "ExampleText",
 CellID->510147982],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "I"}], ")"}], "/", "2"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->71053135],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"1", "+", "\[ImaginaryI]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     FractionBox["3", "2"]}], "}"}]}], "]"}]], "Output",
 ImageSize->{206, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1414449749]
}, Open  ]],

Cell["Radical expressions:", "ExampleText",
 CellID->477247699],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1404345220],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "-", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "4"]}], "&"}], ",", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Output",
 ImageSize->{370, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->12959153]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->806182358],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"5", 
        RowBox[{"#", "^", "5"}]}], "+", 
       RowBox[{"11", "#"}], "+", "1"}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->66961133],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"625", "+", 
       RowBox[{"1375", " ", "#1"}], "+", 
       SuperscriptBox["#1", "5"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     FractionBox["1", "5"], ",", 
     FractionBox["2", "25"], ",", "0", ",", "0"}], "}"}]}], "]"}]], "Output",
 ImageSize->{425, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->908636124]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["AlgbraicNumber"], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->361449785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"-", "3"}], "+", 
         SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1984832010],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "16"}], "-", 
       RowBox[{"15", " ", "#1"}], "-", 
       RowBox[{"3", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Output",
 ImageSize->{402, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->691122251]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18950247],

Cell[TextData[{
 "Coefficients of ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects are integers or rational numbers: "
}], "ExampleText",
 CellID->188195129],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], "+", 
    RowBox[{"Sqrt", "[", "5", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     RowBox[{"1", "/", "2"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1899604719],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"9", "-", 
       RowBox[{"14", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", 
     FractionBox["1", "2"], ",", "0", ",", "0"}], "}"}]}], "]"}]], "Output",
 ImageSize->{373, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->65395062]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->353373074],

Cell["\<\
The number of coefficients is adjusted to match the degree of the algebraic \
number:\
\>", "ExampleText",
 CellID->1348970763],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"3", "^", 
    RowBox[{"(", 
     RowBox[{"1", "/", "5"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1166752362],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "3"}], "+", 
       SuperscriptBox["#1", "5"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1", ",", "0", ",", "0"}], "}"}]}], 
  "]"}]], "Output",
 ImageSize->{346, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->307115126]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"3", "^", 
    RowBox[{"(", 
     RowBox[{"1", "/", "5"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1", ",", "3", ",", "3", ",", "1"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2071553333],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "3"}], "+", 
       SuperscriptBox["#1", "5"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "2", ",", "1", ",", "3", ",", "3"}], "}"}]}], 
  "]"}]], "Output",
 ImageSize->{346, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1280718941]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->882242208],

Cell["Arithmetic in a number field:", "ExampleText",
 CellID->1617690968],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"1", "/", "2"}]}], "}"}]}], "]"}], "+", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1389128156],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   SqrtBox["2"], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", 
     FractionBox["5", "2"]}], "}"}]}], "]"}]], "Output",
 ImageSize->{202, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->814647506]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"1", "/", "2"}]}], "}"}]}], "]"}], "*", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2071423558],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   SqrtBox["2"], ",", 
   RowBox[{"{", 
    RowBox[{"3", ",", 
     FractionBox["5", "2"]}], "}"}]}], "]"}]], "Output",
 ImageSize->{202, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1572383717]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "/", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"1", "/", "2"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->272660911],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   SqrtBox["2"], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "]"}]], "Output",
 ImageSize->{205, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->119371637]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      RowBox[{"1", "/", "2"}]}], "}"}]}], "]"}], "^", "3"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->615212829],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   SqrtBox["2"], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["5", "2"], ",", 
     FractionBox["7", "4"]}], "}"}]}], "]"}]], "Output",
 ImageSize->{206, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->529376761]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1135506603],

Cell[TextData[{
 "Operations on ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->1919789379],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"a", "=", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"-", "1"}], "+", "#1", "+", 
        SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", 
      RowBox[{"-", "1"}], ",", "4"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1300848108],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "+", "#1", "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", "1"}], ",", "4"}], "}"}]}], "]"}]], "Output",
 ImageSize->{344, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->210472606]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Re", "[", "a", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->132329142],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "+", "#1", "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", "1"}], ",", "4"}], "}"}]}], "]"}]], "Output",
 ImageSize->{344, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->507522433]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Abs", "[", "a", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->528713277],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "+", "#1", "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"-", "1"}], ",", "4"}], "}"}]}], "]"}]], "Output",
 ImageSize->{344, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->124485547]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Round", "[", "a", "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->559407800],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->579516251]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "<", "a"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->992665634],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1091291412]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 "Computations with ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects in the same number field are fast:"
}], "ExampleText",
 CellID->1430223835],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"I", ",", 
     RowBox[{"Sqrt", "[", "2", "]"}], ",", 
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         SuperscriptBox["#1", "3"], "-", 
         RowBox[{"2", " ", "#1"}], "+", "3"}], "&"}], ",", "1"}], "]"}]}], 
    "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->166112142],

Cell["Make them part of the same number field:", "ExampleText",
 CellID->412749071],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], "=", 
   RowBox[{"ToNumberField", "[", 
    RowBox[{"{", 
     RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11754526],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RootReduce", "[", 
   FractionBox[
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", "y", " ", "z", " ", 
      RowBox[{"(", 
       RowBox[{"7", "+", "x", "-", "y", "+", 
        SuperscriptBox["z", "2"]}], ")"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"6", "+", 
        SuperscriptBox["x", "2"], "+", 
        RowBox[{"2", " ", "y"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "11"}], "+", 
        RowBox[{"x", " ", "y"}], "+", 
        SuperscriptBox["z", "2"]}], ")"}]}]}], 
    RowBox[{
     RowBox[{"2", " ", "y", " ", "z", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "4"}], "-", "x", "+", 
        RowBox[{"3", " ", "y", " ", "z"}]}], ")"}]}], "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"6", "+", 
        SuperscriptBox["x", "2"], "+", 
        RowBox[{"2", " ", "y"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{"2", "-", 
        RowBox[{"2", " ", "x"}], "+", 
        SuperscriptBox["z", "3"]}], ")"}]}]}]], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->543571549],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.06947699999999912`", ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"127463137729603858692", "+", 
       RowBox[{"15069520316552576640", " ", "#1"}], "+", 
       RowBox[{"3151085417830482145156", " ", 
        SuperscriptBox["#1", "2"]}], "-", 
       RowBox[{"10938243534840099267928", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"14492589303525156688533", " ", 
        SuperscriptBox["#1", "4"]}], "-", 
       RowBox[{"7171605298335082808820", " ", 
        SuperscriptBox["#1", "5"]}], "-", 
       RowBox[{"947445370794828405814", " ", 
        SuperscriptBox["#1", "6"]}], "+", 
       RowBox[{"2510661531113587622448", " ", 
        SuperscriptBox["#1", "7"]}], "-", 
       RowBox[{"606316032776880635517", " ", 
        SuperscriptBox["#1", "8"]}], "-", 
       RowBox[{"100899537810316084288", " ", 
        SuperscriptBox["#1", "9"]}], "+", 
       RowBox[{"74049398920051042942", " ", 
        SuperscriptBox["#1", "10"]}], "-", 
       RowBox[{"12985018306589245140", " ", 
        SuperscriptBox["#1", "11"]}], "+", 
       RowBox[{"879298673075259913", " ", 
        SuperscriptBox["#1", "12"]}]}], "&"}], ",", "4"}], "]"}]}], 
  "}"}]], "Output",
 ImageSize->{574, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1824416104]
}, Open  ]],

Cell[TextData[{
 "Compare to direct computations with ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->1079389238],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", ",", "y", ",", "z"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->488275586],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"RootReduce", "[", 
   FractionBox[
    RowBox[{
     RowBox[{
      RowBox[{"-", "2"}], " ", "y", " ", "z", " ", 
      RowBox[{"(", 
       RowBox[{"7", "+", "x", "-", "y", "+", 
        SuperscriptBox["z", "2"]}], ")"}]}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"6", "+", 
        SuperscriptBox["x", "2"], "+", 
        RowBox[{"2", " ", "y"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "11"}], "+", 
        RowBox[{"x", " ", "y"}], "+", 
        SuperscriptBox["z", "2"]}], ")"}]}]}], 
    RowBox[{
     RowBox[{"2", " ", "y", " ", "z", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "4"}], "-", "x", "+", 
        RowBox[{"3", " ", "y", " ", "z"}]}], ")"}]}], "-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"6", "+", 
        SuperscriptBox["x", "2"], "+", 
        RowBox[{"2", " ", "y"}]}], ")"}], " ", 
      RowBox[{"(", 
       RowBox[{"2", "-", 
        RowBox[{"2", " ", "x"}], "+", 
        SuperscriptBox["z", "3"]}], ")"}]}]}]], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->609407453],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"36.398896`", ",", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"127463137729603858692", "+", 
       RowBox[{"15069520316552576640", " ", "#1"}], "+", 
       RowBox[{"3151085417830482145156", " ", 
        SuperscriptBox["#1", "2"]}], "-", 
       RowBox[{"10938243534840099267928", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       RowBox[{"14492589303525156688533", " ", 
        SuperscriptBox["#1", "4"]}], "-", 
       RowBox[{"7171605298335082808820", " ", 
        SuperscriptBox["#1", "5"]}], "-", 
       RowBox[{"947445370794828405814", " ", 
        SuperscriptBox["#1", "6"]}], "+", 
       RowBox[{"2510661531113587622448", " ", 
        SuperscriptBox["#1", "7"]}], "-", 
       RowBox[{"606316032776880635517", " ", 
        SuperscriptBox["#1", "8"]}], "-", 
       RowBox[{"100899537810316084288", " ", 
        SuperscriptBox["#1", "9"]}], "+", 
       RowBox[{"74049398920051042942", " ", 
        SuperscriptBox["#1", "10"]}], "-", 
       RowBox[{"12985018306589245140", " ", 
        SuperscriptBox["#1", "11"]}], "+", 
       RowBox[{"879298673075259913", " ", 
        SuperscriptBox["#1", "12"]}]}], "&"}], ",", "4"}], "]"}]}], 
  "}"}]], "Output",
 ImageSize->{574, 122},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1189948062]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["RootReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootReduce"]], "InlineFormula"],
 " to transform an algebraic number to a ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " object:"
}], "ExampleText",
 CellID->602214316],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootReduce", "[", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"#", "^", "3"}], "+", "#", "+", "1"}], "&"}], ",", "3"}], 
     "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->555833108],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     RowBox[{"10", " ", "#1"}], "-", 
     SuperscriptBox["#1", "2"], "+", 
     SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}]], "Output",
 ImageSize->{198, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2089357170]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->620227656],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["ToNumberField",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ToNumberField"]], "InlineFormula"],
 " to get representations of ",
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " objects as ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->280591854],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ToNumberField", "[", 
  RowBox[{"Root", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"1", "+", 
      RowBox[{"2", "#1"}], "+", 
      RowBox[{"2", 
       RowBox[{"#1", "^", "2"}]}], "+", 
      RowBox[{"2", 
       RowBox[{"#1", "^", "3"}]}], "+", 
      RowBox[{"2", 
       RowBox[{"#1", "^", "4"}]}]}], "&"}], ",", "1"}], "]"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->33201366],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"8", "+", 
       RowBox[{"8", " ", "#1"}], "+", 
       RowBox[{"4", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "3"]}], "+", 
       SuperscriptBox["#1", "4"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     FractionBox["1", "2"], ",", "0", ",", "0"}], "}"}]}], "]"}]], "Output",
 ImageSize->{445, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->173360446]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->192506935],

Cell["Get the generator polynomial:", "ExampleText",
 CellID->51221769],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"#", "^", "3"}], "+", "#", "+", "1"}], "&"}], ",", "3"}], 
      "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->158783571],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberPolynomial", "[", 
  RowBox[{"a", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->56661088],

Cell[BoxData[
 RowBox[{"1", "+", 
  RowBox[{"2", " ", "x"}], "+", 
  SuperscriptBox["x", "2"]}]], "Output",
 ImageSize->{66, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1681190590]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->65911881],

Cell["Algebraic number theory operations:", "ExampleText",
 CellID->373544793],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "=", " ", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"#", "^", "3"}], "+", "#", "+", "1"}], "&"}], ",", "3"}], 
      "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->85236590],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", "a", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->678462690],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1327902551]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberTrace", "[", "a", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->73534234],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1408115033]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->499066392],

Cell["Minimal polynomial:", "ExampleText",
 CellID->849947766],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "=", " ", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"#", "^", "3"}], "+", "#", "+", "1"}], "&"}], ",", "3"}], 
      "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->787302585],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MinimalPolynomial", "[", 
  RowBox[{"a", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->175469999],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "3"}], "+", 
  RowBox[{"4", " ", "x"}], "-", 
  SuperscriptBox["x", "2"], "+", 
  SuperscriptBox["x", "3"]}]], "Output",
 ImageSize->{99, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->919569691]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "/.", 
  RowBox[{"x", "\[Rule]", "a"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->508831843],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1915472825]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->804342146],

Cell[TextData[{
 "Operations such as ",
 Cell[BoxData[
  ButtonBox["Sqrt",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sqrt"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Re",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Re"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Im",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Im"]], "InlineFormula"],
 " do not automatically reduce:"
}], "ExampleText",
 CellID->1369751610],

Cell[BoxData[
 RowBox[{
  RowBox[{"a", "=", 
   RowBox[{"AlgebraicNumber", "[", 
    RowBox[{
     RowBox[{"Root", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"#", "^", "3"}], "+", "#", "+", "1"}], "&"}], ",", "3"}], 
      "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1938262994],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Re", "[", "a", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->879499146],

Cell[BoxData[
 RowBox[{"Re", "[", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"1", "+", "#1", "+", 
        SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}], "]"}]], "Output",
 ImageSize->{356, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->424417598]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Sqrt", "[", "a", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->390613430],

Cell[BoxData[
 SqrtBox[
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{"Root", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"1", "+", "#1", "+", 
        SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "1"}], "}"}]}], "]"}]]], "Output",
 ImageSize->{347, 24},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1393377899]
}, Open  ]],

Cell[TextData[{
 "Convert to ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " using ",
 Cell[BoxData[
  ButtonBox["RootReduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootReduce"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->70700842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ToNumberField", "[", 
  RowBox[{"RootReduce", "[", 
   RowBox[{"Re", "[", "a", "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->178325316],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "9"}], "+", 
       RowBox[{"11", " ", "#1"}], "-", 
       RowBox[{"2", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "1"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     FractionBox["1", "2"], ",", "0"}], "}"}]}], "]"}]], "Output",
 ImageSize->{399, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->193177958]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ToNumberField", "[", 
  RowBox[{"RootReduce", "[", 
   RowBox[{"Sqrt", "[", "a", "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->488397029],

Cell[BoxData[
 RowBox[{"AlgebraicNumber", "[", 
  RowBox[{
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"4", " ", "#1"}], "-", 
       RowBox[{"3", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}]], "Output",
 ImageSize->{388, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1894149489]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["Root",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Root"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumberPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberPolynomial"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ToNumberField",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ToNumberField"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Algebraics",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Algebraics"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RootReduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RootReduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MinimalPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MinimalPolynomial"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Extension",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Extension"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicIntegerQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicIntegerQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicUnitQ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicUnitQ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumberTrace",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberTrace"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumberNorm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberNorm"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Algebraic Number Fields",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicNumberFields"]], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->81710559],

Cell[TextData[ButtonBox["Algebraic Numbers",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumbers"]], "MoreAbout",
 CellID->22927618],

Cell[TextData[ButtonBox["Algebraic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumberTheory"]], "MoreAbout",
 CellID->828038672],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->339624835],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->14446556],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->75309897]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"AlgebraicNumber - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 23, 55.0797590}", "context" -> "System`", 
    "keywords" -> {
     "algebraic field", "algebraic integer", "algebraic number field", 
      "algebraic rational", "ring of integers"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "AlgebraicNumber[\\[Theta], {c_0, c_1, ..., c_n}] represents the \
algebraic number in the field \\[DoubleStruckCapitalQ][\\[Theta]] given by \
c_0 + c_1 \\[Theta] + ... + c_n \\[Theta]^n.", 
    "synonyms" -> {"algebraic number"}, "title" -> "AlgebraicNumber", "type" -> 
    "Symbol", "uri" -> "ref/AlgebraicNumber"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[11800, 361, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->354412043]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 54519, 2032}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3815, 82, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4438, 109, 1580, 37, 70, "ObjectNameGrid"],
Cell[6021, 148, 1408, 44, 70, "Usage",
 CellID->29430]
}, Open  ]],
Cell[CellGroupData[{
Cell[7466, 197, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7944, 214, 266, 8, 70, "Notes",
 CellID->233291024],
Cell[8213, 224, 485, 16, 70, "Notes",
 CellID->3082117],
Cell[8701, 242, 569, 18, 70, "Notes",
 CellID->638031029],
Cell[9273, 262, 235, 7, 70, "Notes",
 CellID->426868723],
Cell[9511, 271, 344, 12, 70, "Notes",
 CellID->12711],
Cell[9858, 285, 698, 28, 70, "Notes",
 CellID->418],
Cell[10559, 315, 474, 17, 70, "Notes",
 CellID->32512],
Cell[11036, 334, 424, 12, 70, "Notes",
 CellID->38578760],
Cell[11463, 348, 300, 8, 70, "Notes",
 CellID->40028476]
}, Closed]],
Cell[CellGroupData[{
Cell[11800, 361, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->354412043],
Cell[CellGroupData[{
Cell[12185, 375, 148, 5, 70, "ExampleSection",
 CellID->153874062],
Cell[12336, 382, 72, 1, 70, "ExampleText",
 CellID->41531504],
Cell[CellGroupData[{
Cell[12433, 387, 341, 12, 28, "Input",
 CellID->185286594],
Cell[12777, 401, 429, 14, 39, "Output",
 CellID->1194236908]
}, Open  ]],
Cell[13221, 418, 57, 1, 70, "ExampleText",
 CellID->434020516],
Cell[CellGroupData[{
Cell[13303, 423, 115, 4, 28, "Input",
 CellID->183036309],
Cell[13421, 429, 471, 16, 39, "Output",
 CellID->1157547894]
}, Open  ]],
Cell[13907, 448, 72, 1, 70, "ExampleText",
 CellID->46293880],
Cell[CellGroupData[{
Cell[14004, 453, 97, 3, 70, "Input",
 CellID->236533123],
Cell[14104, 458, 267, 8, 36, "Output",
 CellID->853096285]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14420, 472, 219, 7, 70, "ExampleSection",
 CellID->11000979],
Cell[14642, 481, 234, 7, 70, "ExampleText",
 CellID->491609672],
Cell[CellGroupData[{
Cell[14901, 492, 326, 11, 70, "Input",
 CellID->634278950],
Cell[15230, 505, 301, 9, 54, "Output",
 CellID->1388434623]
}, Open  ]],
Cell[15546, 517, 125, 3, 70, "ExampleDelimiter",
 CellID->910589047],
Cell[15674, 522, 133, 4, 70, "ExampleText",
 CellID->1301644407],
Cell[CellGroupData[{
Cell[15832, 530, 183, 6, 70, "Input",
 CellID->321636460],
Cell[16018, 538, 158, 5, 36, "Output",
 CellID->863400459]
}, Open  ]],
Cell[CellGroupData[{
Cell[16213, 548, 208, 7, 70, "Input",
 CellID->2121649240],
Cell[16424, 557, 158, 5, 36, "Output",
 CellID->624511634]
}, Open  ]],
Cell[16597, 565, 125, 3, 70, "ExampleDelimiter",
 CellID->566959903],
Cell[16725, 570, 574, 20, 70, "ExampleText",
 CellID->510147982],
Cell[CellGroupData[{
Cell[17324, 594, 257, 9, 70, "Input",
 CellID->71053135],
Cell[17584, 605, 333, 11, 51, "Output",
 CellID->1414449749]
}, Open  ]],
Cell[17932, 619, 63, 1, 70, "ExampleText",
 CellID->477247699],
Cell[CellGroupData[{
Cell[18020, 624, 295, 9, 70, "Input",
 CellID->1404345220],
Cell[18318, 635, 518, 17, 39, "Output",
 CellID->12959153]
}, Open  ]],
Cell[18851, 655, 178, 7, 70, "ExampleText",
 CellID->806182358],
Cell[CellGroupData[{
Cell[19054, 666, 383, 13, 70, "Input",
 CellID->66961133],
Cell[19440, 681, 530, 17, 51, "Output",
 CellID->908636124]
}, Open  ]],
Cell[19985, 701, 118, 4, 70, "ExampleText",
 CellID->361449785],
Cell[CellGroupData[{
Cell[20128, 709, 495, 16, 70, "Input",
 CellID->1984832010],
Cell[20626, 727, 550, 18, 39, "Output",
 CellID->691122251]
}, Open  ]],
Cell[21191, 748, 124, 3, 70, "ExampleDelimiter",
 CellID->18950247],
Cell[21318, 753, 255, 8, 70, "ExampleText",
 CellID->188195129],
Cell[CellGroupData[{
Cell[21598, 765, 300, 10, 70, "Input",
 CellID->1899604719],
Cell[21901, 777, 520, 17, 51, "Output",
 CellID->65395062]
}, Open  ]],
Cell[22436, 797, 125, 3, 70, "ExampleDelimiter",
 CellID->353373074],
Cell[22564, 802, 137, 4, 70, "ExampleText",
 CellID->1348970763],
Cell[CellGroupData[{
Cell[22726, 810, 269, 9, 70, "Input",
 CellID->1166752362],
Cell[22998, 821, 463, 16, 39, "Output",
 CellID->307115126]
}, Open  ]],
Cell[CellGroupData[{
Cell[23498, 842, 302, 10, 70, "Input",
 CellID->2071553333],
Cell[23803, 854, 464, 16, 39, "Output",
 CellID->1280718941]
}, Open  ]],
Cell[24282, 873, 125, 3, 70, "ExampleDelimiter",
 CellID->882242208],
Cell[24410, 878, 73, 1, 70, "ExampleText",
 CellID->1617690968],
Cell[CellGroupData[{
Cell[24508, 883, 420, 14, 70, "Input",
 CellID->1389128156],
Cell[24931, 899, 309, 11, 51, "Output",
 CellID->814647506]
}, Open  ]],
Cell[CellGroupData[{
Cell[25277, 915, 420, 14, 70, "Input",
 CellID->2071423558],
Cell[25700, 931, 310, 11, 51, "Output",
 CellID->1572383717]
}, Open  ]],
Cell[CellGroupData[{
Cell[26047, 947, 269, 9, 70, "Input",
 CellID->272660911],
Cell[26319, 958, 306, 11, 43, "Output",
 CellID->119371637]
}, Open  ]],
Cell[CellGroupData[{
Cell[26662, 974, 269, 9, 70, "Input",
 CellID->615212829],
Cell[26934, 985, 333, 12, 51, "Output",
 CellID->529376761]
}, Open  ]],
Cell[27282, 1000, 126, 3, 70, "ExampleDelimiter",
 CellID->1135506603],
Cell[27411, 1005, 220, 8, 70, "ExampleText",
 CellID->1919789379],
Cell[CellGroupData[{
Cell[27656, 1017, 417, 14, 70, "Input",
 CellID->1300848108],
Cell[28076, 1033, 472, 16, 39, "Output",
 CellID->210472606]
}, Open  ]],
Cell[CellGroupData[{
Cell[28585, 1054, 98, 3, 70, "Input",
 CellID->132329142],
Cell[28686, 1059, 472, 16, 39, "Output",
 CellID->507522433]
}, Open  ]],
Cell[CellGroupData[{
Cell[29195, 1080, 99, 3, 70, "Input",
 CellID->528713277],
Cell[29297, 1085, 472, 16, 39, "Output",
 CellID->124485547]
}, Open  ]],
Cell[CellGroupData[{
Cell[29806, 1106, 101, 3, 70, "Input",
 CellID->559407800],
Cell[29910, 1111, 158, 5, 36, "Output",
 CellID->579516251]
}, Open  ]],
Cell[CellGroupData[{
Cell[30105, 1121, 92, 3, 70, "Input",
 CellID->992665634],
Cell[30200, 1126, 162, 5, 36, "Output",
 CellID->1091291412]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30411, 1137, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[30641, 1146, 258, 8, 70, "ExampleText",
 CellID->1430223835],
Cell[30902, 1156, 453, 16, 70, "Input",
 CellID->166112142],
Cell[31358, 1174, 83, 1, 70, "ExampleText",
 CellID->412749071],
Cell[31444, 1177, 270, 9, 70, "Input",
 CellID->11754526],
Cell[CellGroupData[{
Cell[31739, 1190, 1130, 37, 70, "Input",
 CellID->543571549],
Cell[32872, 1229, 1385, 35, 143, "Output",
 CellID->1824416104]
}, Open  ]],
Cell[34272, 1267, 220, 8, 70, "ExampleText",
 CellID->1079389238],
Cell[34495, 1277, 228, 8, 70, "Input",
 CellID->488275586],
Cell[CellGroupData[{
Cell[34748, 1289, 1130, 37, 70, "Input",
 CellID->609407453],
Cell[35881, 1328, 1375, 35, 143, "Output",
 CellID->1189948062]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[37305, 1369, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[37546, 1378, 353, 13, 70, "ExampleText",
 CellID->602214316],
Cell[CellGroupData[{
Cell[37924, 1395, 387, 13, 70, "Input",
 CellID->555833108],
Cell[38314, 1410, 382, 13, 39, "Output",
 CellID->2089357170]
}, Open  ]],
Cell[38711, 1426, 125, 3, 70, "ExampleDelimiter",
 CellID->620227656],
Cell[38839, 1431, 498, 18, 70, "ExampleText",
 CellID->280591854],
Cell[CellGroupData[{
Cell[39362, 1453, 417, 15, 70, "Input",
 CellID->33201366],
Cell[39782, 1470, 628, 20, 51, "Output",
 CellID->173360446]
}, Open  ]],
Cell[40425, 1493, 125, 3, 70, "ExampleDelimiter",
 CellID->192506935],
Cell[40553, 1498, 71, 1, 70, "ExampleText",
 CellID->51221769],
Cell[40627, 1501, 401, 14, 70, "Input",
 CellID->158783571],
Cell[CellGroupData[{
Cell[41053, 1519, 143, 4, 70, "Input",
 CellID->56661088],
Cell[41199, 1525, 238, 8, 36, "Output",
 CellID->1681190590]
}, Open  ]],
Cell[41452, 1536, 124, 3, 70, "ExampleDelimiter",
 CellID->65911881],
Cell[41579, 1541, 78, 1, 70, "ExampleText",
 CellID->373544793],
Cell[41660, 1544, 405, 14, 70, "Input",
 CellID->85236590],
Cell[CellGroupData[{
Cell[42090, 1562, 115, 3, 70, "Input",
 CellID->678462690],
Cell[42208, 1567, 159, 5, 36, "Output",
 CellID->1327902551]
}, Open  ]],
Cell[CellGroupData[{
Cell[42404, 1577, 115, 3, 70, "Input",
 CellID->73534234],
Cell[42522, 1582, 159, 5, 36, "Output",
 CellID->1408115033]
}, Open  ]],
Cell[42696, 1590, 125, 3, 70, "ExampleDelimiter",
 CellID->499066392],
Cell[42824, 1595, 62, 1, 70, "ExampleText",
 CellID->849947766],
Cell[42889, 1598, 406, 14, 70, "Input",
 CellID->787302585],
Cell[CellGroupData[{
Cell[43320, 1616, 136, 4, 70, "Input",
 CellID->175469999],
Cell[43459, 1622, 289, 10, 36, "Output",
 CellID->919569691]
}, Open  ]],
Cell[CellGroupData[{
Cell[43785, 1637, 122, 4, 70, "Input",
 CellID->508831843],
Cell[43910, 1643, 159, 5, 36, "Output",
 CellID->1915472825]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[44118, 1654, 230, 7, 70, "ExampleSection",
 CellID->804342146],
Cell[44351, 1663, 454, 18, 70, "ExampleText",
 CellID->1369751610],
Cell[44808, 1683, 402, 14, 70, "Input",
 CellID->1938262994],
Cell[CellGroupData[{
Cell[45235, 1701, 98, 3, 70, "Input",
 CellID->879499146],
Cell[45336, 1706, 465, 15, 39, "Output",
 CellID->424417598]
}, Open  ]],
Cell[CellGroupData[{
Cell[45838, 1726, 100, 3, 70, "Input",
 CellID->390613430],
Cell[45941, 1731, 449, 15, 45, "Output",
 CellID->1393377899]
}, Open  ]],
Cell[46405, 1749, 342, 13, 70, "ExampleText",
 CellID->70700842],
Cell[CellGroupData[{
Cell[46772, 1766, 176, 5, 70, "Input",
 CellID->178325316],
Cell[46951, 1773, 573, 19, 51, "Output",
 CellID->193177958]
}, Open  ]],
Cell[CellGroupData[{
Cell[47561, 1797, 178, 5, 70, "Input",
 CellID->488397029],
Cell[47742, 1804, 549, 18, 39, "Output",
 CellID->1894149489]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[48352, 1829, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[48671, 1840, 2800, 98, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[51508, 1943, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[51829, 1954, 159, 3, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[52025, 1962, 318, 9, 70, "MoreAboutSection",
 CellID->81710559],
Cell[52346, 1973, 144, 3, 70, "MoreAbout",
 CellID->22927618],
Cell[52493, 1978, 156, 3, 70, "MoreAbout",
 CellID->828038672],
Cell[52652, 1983, 169, 3, 70, "MoreAbout",
 CellID->339624835],
Cell[52824, 1988, 178, 3, 70, "MoreAbout",
 CellID->14446556],
Cell[53005, 1993, 197, 4, 70, "MoreAbout",
 CellID->75309897]
}, Open  ]],
Cell[53217, 2000, 27, 0, 70, "History"],
Cell[53247, 2002, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

