(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    244332,       6918]
NotebookOptionsPosition[    214654,       6023]
NotebookOutlinePosition[    221568,       6131]
CellTagsIndexPosition[    221525,       6128]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Mathematica Internet Connectivity" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/InternetConnectivity"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Mathematica Internet Connectivity\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integrated Data Sources" :> 
          Documentation`HelpLookup["paclet:guide/DataCollections"], 
          "New in 6.0: Data Handling & Data Sources" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60DataHandlingAndDataSources"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Integrated Data Sources\"\>", 
       2->"\<\"New in 6.0: Data Handling & Data Sources\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["AstronomicalData", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["AstronomicalData",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/AstronomicalData"], "[", 
       RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", " ", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the value of the specified property of the \
astronomical object with the specified name."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436],

Cell[TextData[{
 Cell[BoxData[
  TooltipBox[
   DynamicModuleBox[{$CellContext`exSum = False}, 
    TagBox[
     CheckboxBox[Dynamic[$CellContext`exSum = If[
         DocuTools`ExampleStatusSummary[
          EvaluationNotebook[]], True, False, $CellContext`exSum]], {False, 
      True, Blank[]}],
     EventHandlerTag[{
      "MouseDown" -> Null, PassEventsDown -> Automatic, PassEventsUp -> 
       True}]],
    DynamicModuleValues:>{}],
   "Automatic (non-interactive) summary of example status."]]],
 " Releasable"
}], "ExampleStatus",
 CellLabel->"Releasable"],

Cell[TextData[{
 Cell[BoxData[
  DynamicModuleBox[{$CellContext`exStat = False}, 
   CheckboxBox[Dynamic[DocuTools`ExampleStatus[
       EvaluationNotebook[], 
       "DQA"] = $CellContext`exStat, ($CellContext`exStat = #; 
     DocuTools`ExampleStatusStamp[$CellContext`exStat])& ]],
   DynamicModuleValues:>{}]]],
 " unsigned:undated"
}], "ExampleStatus",
 CellLabel->"DQA Completed"],

Cell[TextData[{
 Cell[BoxData[
  DynamicModuleBox[{$CellContext`exStat = False}, 
   CheckboxBox[Dynamic[DocuTools`ExampleStatus[
       EvaluationNotebook[], 
       "Authoring"] = $CellContext`exStat, ($CellContext`exStat = #; 
     DocuTools`ExampleStatusStamp[$CellContext`exStat])& ]],
   DynamicModuleValues:>{}]]],
 " unsigned:undated"
}], "ExampleStatus",
 CellLabel->"Authoring Completed"],

Cell["", "ExampleStatus",
 CellLabel->"Example Authors"],

Cell["", "ExampleStatus",
 CellLabel->"Comments"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Integrated Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DataCollections"]], "MoreAbout",
 CellID->1],

Cell[TextData[ButtonBox["New in 6.0: Data Handling & Data Sources",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60DataHandlingAndDataSources"]], "MoreAbout",
 CellID->236061720],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", " ", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " \[LineSeparator]gives the value of the specified property of the \
astronomical object with the specified name."
}], "Usage",
 CellID->1000335954],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AstronomicalData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AstronomicalData"]], "InlineFormula"],
 " includes data on planets, moons, stars, galaxies as well as other \
astronomical objects."
}], "Notes",
 CellID->297388943],

Cell[TextData[{
 "Astronomical objects can be specified by their standard names such as ",
 Cell[BoxData["\"\<Jupiter\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Sun\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Sirius\>\""], "InlineFormula"],
 ", or by catalog designations as ",
 Cell[BoxData["\"\<M42\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<NGC1704\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->111831564],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", "\"\<\!\(\*
StyleBox[\"tag\", \"TI\"]\)\>\"", "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " gives the standardized name for the astronomical object specified by ",
 Cell[BoxData["\"\<\!\(\*
StyleBox[\"tag\", \"TI\"]\)\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->295330312],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", "]"}]], 
  "InlineFormula"],
 " gives a list of all available objects."
}], "Notes",
 CellID->911156272],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", "\"\<Classes\>\"", 
   "]"}]], "InlineFormula"],
 " gives a list of available classes of objects."
}], "Notes",
 CellID->3895384],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", "\"\<Properties\>\"", 
   "]"}]], "InlineFormula"],
 " gives a list of all properties available for any class of object."
}], "Notes",
 CellID->42962864],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", "\"\<\!\(\*
StyleBox[\"class\", \"TI\"]\)\>\"", "]"}]], "InlineFormula"],
 " gives a list of objects in the specified class. "
}], "Notes",
 CellID->118722301],

Cell["Possible classes for solar system objects include:", "Notes",
 CellID->155343494],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<PlanetaryMoon\>\"", Cell[
    "natural planetary satellite", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Planet\>\"", Cell[
    "solar system planet (including Pluto)", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True}}}},
 CellID->595272836],

Cell["Possible classes for stars include:", "Notes",
 CellID->489250657],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<NakedEyeStar\>\"", Cell[
    "stars with apparent magnitude below 6.5", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Star\>\"", Cell[
    "all available stars", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StarBrightest10\>\"", Cell[
    "star that is among the 10 visually brightest from Earth", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StarBrightest100\>\"", Cell[
    "star that is among the 100 visually brightest from Earth", 
     "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StarNearest10\>\"", Cell[
    "star that is among the 10 nearest to Earth", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StarNearest100\>\"", Cell[
    "star that is among the 100 nearest to Earth", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True, True}}}},
 CellID->232238307],

Cell[TextData[{
 "Additional classes for stars include: ",
 Cell[BoxData["\"\<MainSequenceStar\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<ClassOStar\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<ClassAStar\>\""], "InlineFormula"],
 ", etc., ",
 Cell[BoxData["\"\<WhiteDwarf\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Subdwarf\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<NormalGiant\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<BrightGiant\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Supergiant\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<WolfRayetStar\>\""], "InlineFormula"],
 ". "
}], "Notes",
 CellID->339939503],

Cell[TextData[{
 "Classes based on catalogs include: ",
 Cell[BoxData["\"\<BayerObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<FlamsteedObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<GlieseObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<HDObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<HipparcosObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<HRObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<SAOObject\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->140000587],

Cell["Possible classes for deep-sky objects include:", "Notes",
 CellID->145743071],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<BrightHIIRegion\>\"", Cell[
    "bright HII region", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DeepSkyObject\>\"", Cell[
    "deep-sky object", "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<Galaxy\>\"", Cell[
    "galaxy", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GlobularCluster\>\"", Cell[
    "globular star cluster", "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<Nebula\>\"", Cell[
    "nebula", "TableText"]},
   {Cell[" ", "TableRowIcon"], "\"\<OpenCluster\>\"", Cell[
    "open star cluster", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PlanetaryNebula\>\"", Cell[
    "planetary nebula", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->403572293],

Cell[TextData[{
 "Additional classes for galaxies include: ",
 Cell[BoxData["\"\<SpiralGalaxy\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<EllipticalGalaxy\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<BarredSpiralGalaxy\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<IrregularGalaxy\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->630565069],

Cell[TextData[{
 "Classes based on catalogs include: ",
 Cell[BoxData["\"\<MessierObject\>\""], "InlineFormula"],
 ",",
 " ",
 Cell[BoxData["\"\<ICObject\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<NGCObject\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->744135164],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", 
   RowBox[{
    StyleBox["name", "TI"], ",", "\"\<\!\(\*
StyleBox[\"class\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula"],
 " gives ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " depending on whether the object corresponding to ",
 Cell[BoxData[
  StyleBox["name", "TI"]], "InlineFormula"],
 " is in the specified class."
}], "Notes",
 CellID->482514198],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", 
   RowBox[{
    StyleBox["name", "TI"], ",", "\"\<Classes\>\""}], "]"}]], 
  "InlineFormula"],
 " gives a list of the classes in which the object corresponding to ",
 Cell[BoxData[
  StyleBox["name", "TI"]], "InlineFormula"],
 " appears."
}], "Notes",
 CellID->282953651],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", 
   RowBox[{
    StyleBox["name", "TI"], ",", "\"\<ObjectType\>\""}], "]"}]], 
  "InlineFormula"],
 " gives the basic type of object corresponding to name, including ",
 Cell[BoxData["\"\<Planet\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<PlanetaryMoon\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Star\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<Galaxy\>\""], "InlineFormula"],
 "."
}], "Notes",
 CellID->364699022],

Cell["\<\
Location and coordinate properties supported for all objects include:\
\>", "Notes",
 CellID->101627253],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Declination\>\"", Cell[TextData[{
     "current declination ",
     Cell[BoxData[
      StyleBox["\[Delta]", "TR"]], "InlineFormula"],
     " in degrees"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Distance\>\"", Cell[
    "current distance from Earth in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DistanceLightYears\>\"", Cell[
    "current distance from Earth in light-years", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RightAscension\>\"", Cell[TextData[{
     "current right ascension ",
     Cell[BoxData[
      StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
     " in hours"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->135414431],

Cell["\<\
All quantities are specified in SI units, except as noted otherwise.\
\>", "Notes",
 CellID->200237957],

Cell["\<\
Some properties are available only for solar system objects. \
\>", "Notes",
 CellID->299816302],

Cell["Intrinsic mechanical properties include:", "Notes",
 CellID->155438026],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Density\>\"", Cell[TextData[{
     "average density in ",
     Cell[BoxData[
      FormBox[
       RowBox[{"kg", "/", 
        SuperscriptBox["m", "3"]}], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Diameter\>\"", Cell[
    "maximum diameter in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<EquatorialDiameter\>\"", Cell[
    "equatorial diameter in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<EquatorialRadius\>\"", Cell[
    "equatorial radius in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<EscapeVelocity\>\"", Cell[
    "escape velocity in meters per second", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Gravity\>\"", Cell[TextData[{
     "equatorial acceleration due to gravity in ",
     Cell[BoxData[
      FormBox[
       RowBox[{"m", "/", 
        SuperscriptBox["s", "2"]}], TraditionalForm]], "InlineMath"],
     " "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Mass\>\"", Cell[
    "mass in kilograms", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PolarDiameter\>\"", Cell[
    "polar diameter in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PolarRadius\>\"", Cell[
    "polar radius in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Radius\>\"", Cell[
    "maximum radius in meters", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True}}}},
 CellID->476849875],

Cell["Shape and rotational properties include:", "Notes",
 CellID->325949067],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<RotationPeriod\>\"", Cell[
    "intrinsic rotation period in seconds", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Oblateness\>\"", Cell[
    "relative difference between equatorial and polar radius", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Obliquity\>\"", Cell["\<\
tilt of rotation axis with respect to orbital plane in degrees \
\>", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True}}}},
 CellID->548947657],

Cell["Optical and other electromagnetic properties include:", "Notes",
 CellID->768863709],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<AbsoluteMagnitudeH\>\"", Cell[
    TextData[{
     "absolute visual magnitude ",
     Cell[BoxData[
      FormBox["H", TraditionalForm]], "InlineMath"],
     " for solar system bodies"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AbsoluteMagnitude\>\"", Cell[
    TextData[{
     "absolute visual magnitude ",
     Cell[BoxData[
      FormBox["M", TraditionalForm]], "InlineMath"],
     " for extrasolar objects"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Albedo\>\"", Cell[
    "ratio of returned to incident intensity", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ApparentMagnitude\>\"", Cell[
    "apparent magnitude as viewed from Earth", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True}}}},
 CellID->252134634],

Cell["\<\
Positional and orbital properties for solar system objects include:\
\>", "Notes",
 CellID->46333823],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Apoapsis\>\"", Cell[
    "furthest distance from orbital center in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AscendingNodeLongitude\>\"", Cell[
    TextData[{
     "longitude angle ",
     Cell[BoxData[
      StyleBox["\[CapitalOmega]", "TR"]], "InlineFormula"],
     " of the ascending orbit node in degrees"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Eccentricity\>\"", Cell[TextData[{
     "orbital eccentricity ",
     Cell[BoxData[
      FormBox["e", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Inclination\>\"", Cell[TextData[{
     "inclination ",
     Cell[BoxData[
      FormBox["i", TraditionalForm]], "InlineMath"],
     " with respect to the ecliptic, in degrees"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<OrbitCenter\>\"", Cell[
    "object around which the object orbits", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<OrbitPath\>\"", Cell[
    "graphics primitives representing a typical orbit", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<OrbitPeriod\>\"", Cell[
    "time in seconds for a complete orbit", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<OrbitPeriodYears\>\"", Cell[
    "time in years for a complete orbit", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<OrbitRules\>\"", Cell[
    "list of rules giving orbital elements", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Periapsis\>\"", Cell[
    "closest approach to orbital center in meters", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PeriapsisArgument\>\"", Cell[
    TextData[{
     "angle ",
     Cell[BoxData[
      StyleBox["\[Omega]", "TR"]], "InlineFormula"],
     " in degrees from ascending node to pericenter"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PeriapsisLongitude\>\"", Cell[
    "longitude angle of periapsis for zero inclination", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Satellites\>\"", Cell[
    "known natural satellites of the object ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SemimajorAxis\>\"", Cell[
    "orbital semimajor axis length in meters", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, None, None, 
     None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, True, 
      True, True, True}}}},
 CellID->45857499],

Cell["\<\
Orbital elements are intended as averages for the current epoch, but may be \
based only on particular observations. \
\>", "Notes",
 CellID->28659941],

Cell["Properties of relevance to stars include:", "Notes",
 CellID->588660618],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<BVColorIndex\>\"", Cell[
    "B\[Hyphen]V color index", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Constellation\>\"", Cell[
    "official constellation in which the star is found", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Parallax\>\"", Cell[
    "parallax in seconds of arc", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SpectralClass\>\"", Cell[TextData[{
     "spectral class as a string (",
     Cell[BoxData["\"\<G2V\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<O\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<M2Ib\>\""], "InlineFormula"],
     ", ...)"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SpectralClassList\>\"", Cell[
    TextData[{
     "spectral class list (",
     Cell[BoxData[
      RowBox[{" ", 
       RowBox[{"{", 
        RowBox[{"\"\<G\>\"", ",", " ", "2", ",", " ", "5"}], "}"}]}]], 
      "InlineFormula"],
     ")"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->665737471],

Cell["Properties of relevance to deep-sky objects include:", "Notes",
 CellID->87606451],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<BlueBandMagnitude\>\"", Cell[
    TextData[{
     "brightness in the ",
     Cell[BoxData[
      FormBox["B", TraditionalForm]], "InlineMath"],
     " band"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Constellation\>\"", Cell[
    "official constellation in which the object is found", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GalaxyType\>\"", Cell[TextData[{
     "basic galaxy type (",
     Cell[BoxData["\"\<Spiral\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<Irregular\>\""], "InlineFormula"],
     ", ...)"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GlobularClusterType\>\"", Cell["\<\
Shapley\[Dash]Sawyer class for globular clusters\
\>", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HubbleType\>\"", Cell[TextData[{
     "Hubble classification (",
     Cell[BoxData["\"\<E0\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<Sb\>\""], "InlineFormula"],
     ", ...)  "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<RadialVelocity\>\"", Cell[TextData[{
     "radial velocity in ",
     Cell[BoxData[
      FormBox[
       RowBox[{"m", "/", "s"}], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Redshift\>\"", Cell[TextData[{
     "redshift ",
     Cell[BoxData[
      FormBox["z", TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SurfaceBrightness\>\"", Cell[
    "surface brightness in magnitudes per square arcsecond", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<VisualBandMagnitude\>\"", Cell[
    TextData[{
     "brightness in the ",
     Cell[BoxData[
      FormBox["V", TraditionalForm]], "InlineMath"],
     " band"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True, True}}}},
 CellID->162684920],

Cell["Name and type properties include:", "Notes",
 CellID->454439520],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<AlphanumericName\>\"", Cell[
    "name in alphanumeric form", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AlternateNames\>\"", Cell[
    "alternate English names, as strings", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<AlternateStandardNames\>\"", Cell[
    TextData[{
     "alternate standard ",
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " names"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Name\>\"", Cell[
    "common name (including Greek letters if necessary)", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ObjectType\>\"", Cell[TextData[{
     Cell[BoxData["\"\<Star\>\""], "InlineFormula"],
     ", ",
     Cell[BoxData["\"\<Planet\>\""], "InlineFormula"],
     ", etc."
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<StandardName\>\"", Cell[TextData[{
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " standard name"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True, True}}}},
 CellID->707496748],

Cell["Catalog identifier properties include:", "Notes",
 CellID->374541617],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<BayerName\>\"", Cell[
    "Bayer star designation", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<FlamsteedName\>\"", Cell[
    "Flamsteed catalog designation", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<GlieseName\>\"", Cell[
    "Gliese catalog designation", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HDName\>\"", Cell[
    "Henry Draper Memorial catalog name", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HDNumber\>\"", Cell[
    "Henry Draper Memorial catalog number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HipparcosName\>\"", Cell[
    "Hipparcos catalog name", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HipparcosNumber\>\"", Cell[
    "Hipparcos catalog number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HRName\>\"", Cell[
    "Harvard Revised catalog name", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<HRNumber\>\"", Cell[
    "Harvard Revised catalog number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<ICNumber\>\"", Cell[
    "IC catalog number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<NGCNumber\>\"", Cell[
    "NGC catalog number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<PGCNumber\>\"", Cell[
    "Principal Galaxy Catalog number", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SAOName\>\"", Cell[
    "Smithsonian catalog name", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<SAONumber\>\"", Cell[
    "Smithsonian catalog number", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, None, None, 
     None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, True, 
      True, True, True}}}},
 CellID->109938254],

Cell["Additional properties include:", "Notes",
 CellID->106210209],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Image\>\"", Cell[
    "picture of the astronomical object ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<DiscoveryYear\>\"", Cell[
    "year in which the object was discovered ", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True}}}},
 CellID->496402173],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AstronomicalData",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AstronomicalData"], "[", 
   RowBox[{"\"\<\!\(\*
StyleBox[\"name\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"property\", \"TI\"]\)\>\"", ",", "\"\<\!\(\*
StyleBox[\"ann\", \"TI\"]\)\>\""}], "]"}]], "InlineFormula", 
  "TemplateInclusion"],
 " gives various annotations associated with a property. Typical annotations \
include:"
}], "Notes",
 CellID->276644756],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], "\"\<Description\>\"", Cell[TextData[{
     Cell[BoxData[Cell["", "TableText"]]],
     "short textual description of the property"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<LongDescription\>\"", Cell[
    "longer textual description of the property ", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Note\>\"", Cell[
    "additional information about the property", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<Units\>\"", Cell[
    "units in which the value is given", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsName\>\"", Cell[
    "English name for the units used", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsNotation\>\"", Cell[
    "notation for the units used", "TableText"]},
   {Cell["      ", "TableRowIcon"], "\"\<UnitsStandardName\>\"", Cell[
    TextData[{
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " standard name for the units used"
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->524183967],

Cell[TextData[{
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/InternetConnectivity"],
  FontSlant->"Italic"],
 ButtonBox[" Internet Connectivity",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/InternetConnectivity"]
}], "Tutorials",
 CellID->341631938],

Cell[TextData[ButtonBox["AstronomicalData Source Information",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
AstronomicalDataSourceInformation.html"], None}]], "RelatedLinks",
 CellID->1038487239],

Cell["A list of solar system planets:", "ExampleText",
 CellID->677248009],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->248531990],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Mercury\"\>", ",", "\<\"Venus\"\>", ",", "\<\"Earth\"\>", 
   ",", "\<\"Mars\"\>", ",", "\<\"Jupiter\"\>", ",", "\<\"Saturn\"\>", 
   ",", "\<\"Uranus\"\>", ",", "\<\"Neptune\"\>", ",", "\<\"Pluto\"\>"}], 
  "}"}]], "Output",
 ImageSize->{472, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1025003446]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->83773203],

Cell["Make a graphic of solar system orbit paths:", "ExampleText",
 CellID->6724361],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"AstronomicalData", "[", 
     RowBox[{"#", ",", "\"\<OrbitPath\>\""}], "]"}], "&"}], "/@", 
   RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->355551284],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->574005652]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->467797123],

Cell["The known satellites of Neptune:", "ExampleText",
 CellID->496580757],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Neptune\>\"", ",", "\"\<Satellites\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->326637340],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Naiad\"\>", ",", "\<\"Thalassa\"\>", ",", "\<\"Despina\"\>", 
   ",", "\<\"Galatea\"\>", ",", "\<\"Larissa\"\>", ",", "\<\"Proteus\"\>", 
   ",", "\<\"Triton\"\>", ",", "\<\"Nereid\"\>", ",", "\<\"S2002N1\"\>", 
   ",", "\<\"S2002N2\"\>", ",", "\<\"S2002N3\"\>", ",", "\<\"Psamathe\"\>"}], 
  "}"}]], "Output",
 ImageSize->{422, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->202270463]
}, Open  ]],

Cell["The escape velocity of Neptune:", "ExampleText",
 CellID->125102566],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Neptune\>\"", ",", "\"\<EscapeVelocity\>\""}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->56775408],

Cell[BoxData["23710.`4.000000000000001"], "Output",
 ImageSize->{70, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->238403244]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{
  "\"\<Neptune\>\"", ",", "\"\<EscapeVelocity\>\"", ",", "\"\<Units\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->229837987],

Cell[BoxData["\<\"MetersPerSecond\"\>"], "Output",
 ImageSize->{109, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->169146414]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->123085960],

Cell["The 10 nearest stars:", "ExampleText",
 CellID->657588733],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", "\"\<StarNearest10\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->97706448],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Sun\"\>", ",", "\<\"ProximaCentauri\"\>", 
   ",", "\<\"RigelKentaurusA\"\>", ",", "\<\"RigelKentaurusB\"\>", 
   ",", "\<\"BarnardsStar\"\>", ",", "\<\"Wolf359\"\>", 
   ",", "\<\"Lalande21185\"\>", ",", "\<\"Luyten726-8A\"\>", 
   ",", "\<\"Luyten726-8B\"\>", ",", "\<\"Sirius\"\>"}], "}"}]], "Output",
 ImageSize->{501, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->366015654]
}, Open  ]],

Cell["Distances in light-years:", "ExampleText",
 CellID->250695742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<DistanceLightYears\>\""}], "]"}], "&"}], "/@", 
  "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->233909923],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.0000158137284029598`2.999999999565706", ",", 
   "4.2209828`2.9995659225206794", ",", "4.3928174`2.9995659225206808", ",", 
   "4.3928174`2.9995659225206808", ",", 
   "5.9379596000000000001`2.9995659225206808", ",", 
   "7.7934538`2.9995659225206817", ",", 
   "8.3078491999999999999`2.9995659225206808", ",", 
   "8.5631724`2.9995659225206825", ",", "8.5631724`2.9995659225206825", ",", 
   "8.5968156`2.9995659225206794"}], "}"}]], "Output",
 ImageSize->{434, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->819679723]
}, Open  ]],

Cell["Spectral class information:", "ExampleText",
 CellID->264471952],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<SpectralClass\>\""}], "]"}], "&"}], "/@", 
  "%%"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->320241788],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"G2V\"\>", ",", "\<\"M5Ve\"\>", ",", "\<\"G2V\"\>", 
   ",", "\<\"K1V\"\>", ",", "\<\"sdM4\"\>", ",", "\<\"M6\"\>", 
   ",", "\<\"M2V\"\>", ",", "\<\"dM5.5e\"\>", ",", "\<\"dM5.5e\"\>", 
   ",", "\<\"A0m...\"\>"}], "}"}]], "Output",
 ImageSize->{399, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->373707133]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->35475128],

Cell["\<\
The first 15 Messier objects, in order of decreasing brightness:\
\>", "ExampleText",
 CellID->209857436],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<MessierObject\>\"", "]"}], ",", 
   "15"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->953265633],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"M45\"\>", ",", "\<\"M44\"\>", ",", "\<\"M7\"\>", 
   ",", "\<\"M31\"\>", ",", "\<\"M6\"\>", ",", "\<\"M47\"\>", 
   ",", "\<\"M41\"\>", ",", "\<\"M25\"\>", ",", "\<\"M39\"\>", 
   ",", "\<\"M35\"\>", ",", "\<\"M22\"\>", ",", "\<\"M34\"\>", 
   ",", "\<\"M4\"\>", ",", "\<\"M23\"\>", ",", "\<\"M33\"\>"}], 
  "}"}]], "Output",
 ImageSize->{469, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]="]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->556251086],

Cell["Distance to the Andromeda galaxy in light-years:", "ExampleText",
 CellID->223366887],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<AndromedaGalaxy\>\"", ",", "\"\<DistanceLightYears\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->80677821],

Cell[BoxData["2.572959395348838`4.*^6"], "Output",
 ImageSize->{70, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->258568371]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->399445155],

Cell["Find astronomical objects whose names begin with Alb:", "ExampleText",
 CellID->39714291],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", "\"\<Alb*\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->474826791],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Albireo\"\>", ",", "\<\"Albali\"\>", ",", "\<\"Albiorix\"\>"}],
   "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6893128]
}, Open  ]],

Cell["Find the types of these objects:", "ExampleText",
 CellID->572355833],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<ObjectType\>\""}], "]"}], "&"}], "/@", 
  "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->37495760],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Star\"\>", ",", "\<\"Star\"\>", ",", "\<\"PlanetaryMoon\"\>"}],
   "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->221476779]
}, Open  ]],

Cell["The solar system planets, including Pluto:", "ExampleText",
 CellID->114341380],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->670692101],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Mercury\"\>", ",", "\<\"Venus\"\>", ",", "\<\"Earth\"\>", 
   ",", "\<\"Mars\"\>", ",", "\<\"Jupiter\"\>", ",", "\<\"Saturn\"\>", 
   ",", "\<\"Uranus\"\>", ",", "\<\"Neptune\"\>", ",", "\<\"Pluto\"\>"}], 
  "}"}]], "Output",
 ImageSize->{472, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->132466339]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->121895515],

Cell["Orbital eccentricity for each planet:", "ExampleText",
 CellID->508605741],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<Eccentricity\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->393612353],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.20563069`8.", ",", "0.00677323`6.0000000000000036", ",", 
   "0.01671022`7.999999999999999", ",", "0.09341233`7.999999999999999", ",", 
   "0.04839266`8.000000000000002", ",", "0.0541506`5.999999999999999", ",", 
   "0.04716771`8.", ",", "0.00858587`6.0000000000000036", ",", 
   "0.24880766`7.999999999999999"}], "}"}]], "Output",
 ImageSize->{420, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]="]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->89516562],

Cell["Orbital periods in sidereal years:", "ExampleText",
 CellID->3614946],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<OrbitPeriodYears\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->204278449],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.240847`5.999999999999993", ",", "0.615197`6.0000000000000036", ",", 
   "1.00002`5.999999999999996", ",", "1.88085`6.0000000000000036", ",", 
   "11.8626`5.999999999999995", ",", "29.4475`5.999999999999995", ",", 
   "84.0168`5.999999999999994", ",", "164.791`5.999999999999995", ",", 
   "247.921`6.000000000000003"}], "}"}]], "Output",
 ImageSize->{563, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->487259273]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->446339696],

Cell["Rules for the orbital elements of Earth:", "ExampleText",
 CellID->288556773],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Earth\>\"", ",", "\"\<OrbitRules\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17049110],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"SemimajorAxis\"\>", "\[Rule]", 
    "1.4959788714676577601`8.999565922520677*^11"}], ",", 
   RowBox[{"\<\"Eccentricity\"\>", "\[Rule]", 
    "0.01671022`7.999999999999999"}], ",", 
   RowBox[{"\<\"Inclination\"\>", "\[Rule]", "0.00005`1.0000000000000016"}], 
   ",", 
   RowBox[{"\<\"PeriapsisArgument\"\>", "\[Rule]", 
    "114.20783`7.724140619488459"}], ",", 
   RowBox[{"\<\"AscendingNodeLongitude\"\>", "\[Rule]", 
    RowBox[{"-", "11.26064`6.999999999999996"}]}], ",", 
   RowBox[{"\<\"PeriapsisLongitude\"\>", "\[Rule]", "102.94719`8."}], ",", 
   RowBox[{"\<\"Periapsis\"\>", "\[Rule]", 
    "1.47098073541008147604`8.931464628195396*^11"}], ",", 
   RowBox[{"\<\"Apoapsis\"\>", "\[Rule]", 
    "1.52097700752523404416`8.933541465906872*^11"}]}], "}"}]], "Output",
 ImageSize->{468, 71},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->681784613]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->478329810],

Cell["Obliquity of each planet:", "ExampleText",
 CellID->417229875],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<Obliquity\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->416112851],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.01`2.0000000000000036", ",", "177.36`5.000000000000002", ",", 
   "23.45`3.9999999999999982", ",", "25.19`3.9999999999999996", ",", 
   "3.13`2.9999999999999973", ",", "26.73`4.000000000000001", ",", 
   "97.77`3.9999999999999996", ",", "28.32`4.000000000000001", ",", 
   "122.53`5.000000000000003"}], "}"}]], "Output",
 ImageSize->{430, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->227238577]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->669453872],

Cell["Albedo of each planet:", "ExampleText",
 CellID->591475599],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<Albedo\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14559333],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.106`2.999999999999995", ",", "0.65`2.", ",", "0.367`3.0000000000000053", 
   ",", "0.15`2.0000000000000044", ",", "0.52`2.0000000000000084", ",", 
   "0.47`2.000000000000004", ",", "0.51`2.000000000000003", ",", 
   "0.41`2.000000000000003", ",", "0.3`1.0000000000000002"}], "}"}]], "Output",\

 ImageSize->{367, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->790543063]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->605287824],

Cell["An image of Jupiter:", "ExampleText",
 CellID->316398652],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Jupiter\>\"", ",", "\"\<Image\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10741422],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnQVYW9nW91uk7i5TndrU3Wgp0NIWWlxa3N3dXUKQQLAQLBCSQAQJkKAJ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   "], {{0, 0}, {120, 120}}, {0, 255},
   ColorFunction->RGBColor],
  ImageSize->{120, 120},
  PlotRange->{{0, 120}, {0, 120}}]], "Output",
 ImageSize->{124, 120},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->349946031]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->405493821],

Cell["The known satellites of Pluto:", "ExampleText",
 CellID->525227218],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Pluto\>\"", ",", "\"\<Satellites\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9278646],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Charon\"\>", ",", "\<\"Nix\"\>", ",", "\<\"Hydra\"\>"}], 
  "}"}]], "Output",
 ImageSize->{139, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->105924377]
}, Open  ]],

Cell["Diameter of each satellite in meters:", "ExampleText",
 CellID->133182419],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<Diameter\>\""}], "]"}], "&"}], "/@", 
  "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->358890307],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.186`3.0000000000000036*^6", ",", "88000.`2.000000000000003", ",", 
   "88000.`2.000000000000003"}], "}"}]], "Output",
 ImageSize->{203, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->328989165]
}, Open  ]],

Cell["Orbit semimajor axis in meters:", "ExampleText",
 CellID->770440698],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<SemimajorAxis\>\""}], "]"}], "&"}], "/@", 
  "%%"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->5648616],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1.9599`5.000000000000003*^7", ",", "4.8675`5.000000000000002*^7", ",", 
   "6.478`5.000000000000002*^7"}], "}"}]], "Output",
 ImageSize->{259, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->263279802]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->314540477],

Cell["Number of known satellites for each planet:", "ExampleText",
 CellID->5759648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Length", "[", 
    RowBox[{"AstronomicalData", "[", 
     RowBox[{"#", ",", "\"\<Satellites\>\""}], "]"}], "]"}], "&"}], "/@", 
  RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->265543398],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "1", ",", "2", ",", "62", ",", "56", ",", "26", ",", 
   "12", ",", "3"}], "}"}]], "Output",
 ImageSize->{199, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1031151]
}, Open  ]],

Cell["The 15 brightest stars as seen from Earth:", "ExampleText",
 CellID->52084224],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<Star\>\"", "]"}], ",", "15"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17426213],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Sun\"\>", ",", "\<\"Sirius\"\>", ",", "\<\"Canopus\"\>", 
   ",", "\<\"Arcturus\"\>", ",", "\<\"RigelKentaurusA\"\>", 
   ",", "\<\"Vega\"\>", ",", "\<\"Capella\"\>", ",", "\<\"Rigel\"\>", 
   ",", "\<\"Procyon\"\>", ",", "\<\"Achernar\"\>", ",", "\<\"Betelgeuse\"\>",
    ",", "\<\"Hadar\"\>", ",", "\<\"Altair\"\>", ",", "\<\"Acrux\"\>", 
   ",", "\<\"Aldebaran\"\>"}], "}"}]], "Output",
 ImageSize->{482, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->117440584]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4804123],

Cell["The constellations for the 10 brightest stars:", "ExampleText",
 CellID->16164372],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<Constellation\>\""}], "]"}], "&"}], "/@", 
  RowBox[{
  "AstronomicalData", "[", "\"\<StarBrightest10\>\"", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->527578803],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Missing", "[", "\<\"Variable\"\>", "]"}], 
   ",", "\<\"CanisMajor\"\>", ",", "\<\"Carina\"\>", ",", "\<\"Bootes\"\>", 
   ",", "\<\"Centaurus\"\>", ",", "\<\"Lyra\"\>", ",", "\<\"Auriga\"\>", 
   ",", "\<\"Orion\"\>", ",", "\<\"CanisMinor\"\>", ",", "\<\"Orion\"\>"}], 
  "}"}]], "Output",
 ImageSize->{401, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->506680743]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->29459054],

Cell["Distance from Earth to Sirius in light-years:", "ExampleText",
 CellID->111196055],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Sirius\>\"", ",", "\"\<DistanceLightYears\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->81860064],

Cell[BoxData["8.5968156`2.9995659225206794"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->550676796]
}, Open  ]],

Cell["Distance in meters:", "ExampleText",
 CellID->117533139],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Sirius\>\"", ",", "\"\<Distance\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->600949417],

Cell[BoxData["8.1380199012`5.698970004336013*^16"], "Output",
 ImageSize->{82, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->93944194]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->79964759],

Cell["Position of Sirius:", "ExampleText",
 CellID->3743066],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"\"\<Sirius\>\"", ",", "\"\<RightAscension\>\""}], "]"}], ",", 
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"\"\<Sirius\>\"", ",", "\"\<Declination\>\""}], "]"}]}], 
  "}"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->301777630],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"6.75257`5.999999999999995", ",", 
   RowBox[{"-", "16.7131`5.999999999999994"}]}], "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->50202262]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->163343868],

Cell["Spectral class for the Sun:", "ExampleText",
 CellID->336797544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Sun\>\"", ",", "\"\<SpectralClass\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->431792553],

Cell[BoxData["\<\"G2V\"\>"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->830257140]
}, Open  ]],

Cell["Spectral class information in list form:", "ExampleText",
 CellID->43945354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Sun\>\"", ",", "\"\<SpectralClassList\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->284175261],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"G\"\>", ",", "2", ",", "5"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14326516]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->577251838],

Cell["Alternate names for Sirius:", "ExampleText",
 CellID->245821],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Sirius\>\"", ",", "\"\<AlternateNames\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->741704346],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Alpha Canis Majoris\"\>", ",", "\<\"Alp CMa\"\>", 
   ",", "\<\"\[Alpha] Canis Majoris\"\>", ",", "\<\"Alpha CMa\"\>", 
   ",", "\<\"\[Alpha] CMa\"\>", ",", "\<\"HIP 32349\"\>", 
   ",", "\<\"Gl 244 A\"\>", ",", "\<\"9 Canis Majoris\"\>", 
   ",", "\<\"9 CMa\"\>", ",", "\<\"HD 48915\"\>", ",", "\<\"HR 2491\"\>", 
   ",", "\<\"SAO 151881\"\>"}], "}"}]], "Output",
 ImageSize->{520, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15152276]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->45066016],

Cell["Henry Draper Memorial catalog number for Sirius:", "ExampleText",
 CellID->196045199],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Sirius\>\"", ",", "\"\<HDNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->819995344],

Cell[BoxData["48915"], "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->826663926]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->664978631],

Cell["Give the individual stars that form Alpha Centauri:", "ExampleText",
 CellID->157697162],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", "\"\<AlphaCentauri\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->188457272],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"RigelKentaurusA\"\>", ",", "\<\"RigelKentaurusB\"\>", 
   ",", "\<\"ProximaCentauri\"\>"}], "}"}]], "Output",
 ImageSize->{356, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->511166242]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->404283081],

Cell["Give the name of tau ceti, including Greek letters:", "ExampleText",
 CellID->494547118],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<TauCeti\>\"", ",", "\"\<Name\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->508142260],

Cell[BoxData["\<\"\[Tau] Ceti\"\>"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->258151589]
}, Open  ]],

Cell["The same thing without special characters:", "ExampleText",
 CellID->678291554],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<TauCeti\>\"", ",", "\"\<AlphanumericName\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->658834346],

Cell[BoxData["\<\"Tau Ceti\"\>"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->3891353]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->49714092],

Cell["Brightest 10 supergiants as seen from Earth:", "ExampleText",
 CellID->116384529],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<Supergiant\>\"", "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->401681999],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Canopus\"\>", ",", "\<\"Rigel\"\>", ",", "\<\"Betelgeuse\"\>", 
   ",", "\<\"Antares\"\>", ",", "\<\"Deneb\"\>", ",", "\<\"Alnilam\"\>", 
   ",", "\<\"Alnitak\"\>", ",", "\<\"Regor\"\>", ",", "\<\"Mirphak\"\>", 
   ",", "\<\"Wezen\"\>"}], "}"}]], "Output",
 ImageSize->{574, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->561389168]
}, Open  ]],

Cell["Number of available supergiants:", "ExampleText",
 CellID->49258605],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"AstronomicalData", "[", "\"\<Supergiant\>\"", "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11571315],

Cell[BoxData["988"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->139098614]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10718750],

Cell["Brightest 10 class O stars:", "ExampleText",
 CellID->647167067],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<ClassOStar\>\"", "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->119625805],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Alnitak\"\>", ",", "\<\"Regor\"\>", ",", "\<\"Naos\"\>", 
   ",", "\<\"Mintaka\"\>", ",", "\<\"ZetaOphiuchi\"\>", 
   ",", "\<\"IotaOrionis\"\>", ",", "\<\"Heka\"\>", 
   ",", "\<\"SigmaOrionis\"\>", ",", "\<\"Menkib\"\>", 
   ",", "\<\"AlphaCamelopardalis\"\>"}], "}"}]], "Output",
 ImageSize->{427, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->464276984]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->283013227],

Cell["Brightest 10 Wolf\[Dash]Rayet stars:", "ExampleText",
 CellID->609171371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<WolfRayetStar\>\"", "]"}], ",", 
   "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->84831946],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Regor\"\>", ",", "\<\"ThetaMuscae\"\>", 
   ",", "\<\"HIP52308\"\>", ",", "\<\"HIP82543\"\>", ",", "\<\"HIP52488\"\>", 
   ",", "\<\"HIP82706\"\>", ",", "\<\"HIP33165\"\>", ",", "\<\"HIP26380\"\>", 
   ",", "\<\"HIP99002\"\>", ",", "\<\"HIP100287\"\>"}], "}"}]], "Output",
 ImageSize->{417, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->508255469]
}, Open  ]],

Cell["Brightest 10 Messier objects:", "ExampleText",
 CellID->249443859],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<MessierObject\>\"", "]"}], ",", 
   "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->148780319],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"M45\"\>", ",", "\<\"M44\"\>", ",", "\<\"M7\"\>", 
   ",", "\<\"M31\"\>", ",", "\<\"M6\"\>", ",", "\<\"M47\"\>", 
   ",", "\<\"M41\"\>", ",", "\<\"M25\"\>", ",", "\<\"M39\"\>", 
   ",", "\<\"M35\"\>"}], "}"}]], "Output",
 ImageSize->{315, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->714709956]
}, Open  ]],

Cell["Object types:", "ExampleText",
 CellID->709914909],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<ObjectType\>\""}], "]"}], "&"}], "/@", 
  "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->295312528],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"OpenCluster\"\>", ",", "\<\"OpenCluster\"\>", 
   ",", "\<\"OpenCluster\"\>", ",", "\<\"SpiralGalaxy\"\>", 
   ",", "\<\"OpenCluster\"\>", ",", "\<\"OpenCluster\"\>", 
   ",", "\<\"OpenCluster\"\>", ",", "\<\"OpenCluster\"\>", 
   ",", "\<\"OpenCluster\"\>", ",", "\<\"OpenCluster\"\>"}], "}"}]], "Output",\

 ImageSize->{455, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->368713991]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->256339554],

Cell["The brightest 10 planetary nebulas:", "ExampleText",
 CellID->64125037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<PlanetaryNebula\>\"", "]"}], ",", 
   "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->34994936],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"NGC7293\"\>", ",", "\<\"M27\"\>", ",", "\<\"NGC3242\"\>", 
   ",", "\<\"NGC7009\"\>", ",", "\<\"NGC3918\"\>", ",", "\<\"NGC6543\"\>", 
   ",", "\<\"NGC6572\"\>", ",", "\<\"NGC7662\"\>", ",", "\<\"NGC7027\"\>", 
   ",", "\<\"M57\"\>"}], "}"}]], "Output",
 ImageSize->{553, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->552840004]
}, Open  ]],

Cell["Declinations:", "ExampleText",
 CellID->175933697],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<Declination\>\""}], "]"}], "&"}], "/@", 
  "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49098515],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "19.1636111111111111111`5.797387164707491"}], ",", 
   "22.7216666666666666667`5.657470179991214", ",", 
   RowBox[{"-", "17.3580555555555555556`5.304441880953846"}], ",", 
   RowBox[{"-", "10.6369444444444444444`4.911021533807849"}], ",", 
   RowBox[{"-", "56.8180555555555555557`5.570426178358969"}], ",", 
   "66.6336111111111111112`7.37999585062754", ",", 
   "6.8541666666666666666`5.216165902285993", ",", 
   "42.5355555555555555556`6.281964623259894", ",", 
   "42.2366666666666666667`6.102810890916912", ",", 
   "33.0297222222222222223`5.403109564150488"}], "}"}]], "Output",
 ImageSize->{362, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->632458544]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23696552],

Cell["Brightest 10 galaxies:", "ExampleText",
 CellID->366947188],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"AstronomicalData", "[", "\"\<Galaxy\>\"", "]"}], ",", "10"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->640742278],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"LargeMagellanicCloud\"\>", ",", "\<\"SmallMagellanicCloud\"\>",
    ",", "\<\"M31\"\>", ",", "\<\"M33\"\>", ",", "\<\"NGC5128\"\>", 
   ",", "\<\"M81\"\>", ",", "\<\"NGC253\"\>", ",", "\<\"M101\"\>", 
   ",", "\<\"M83\"\>", ",", "\<\"NGC55\"\>"}], "}"}]], "Output",
 ImageSize->{335, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->43478866]
}, Open  ]],

Cell["Distances in light-years:", "ExampleText",
 CellID->255319853],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<DistanceLightYears\>\""}], "]"}], "&"}], "/@", 
  "%"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->357775124],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"168000.`2.999999999999998", ",", 
   RowBox[{"Missing", "[", "\<\"NotAvailable\"\>", "]"}], ",", 
   "2.572959395348838`4.*^6", ",", "2.814753351351351`3.9999999999999996*^6", 
   ",", "1.134244`3.9999999999999982*^7", ",", 
   "1.2008452105263157`4.000000000000001*^7", ",", 
   "1.082984`3.9999999999999996*^7", ",", "2.4582432`3.9999999999999996*^7", 
   ",", "1.578808`4.000000000000001*^7", ",", "5.174619333333333`4.*^6"}], 
  "}"}]], "Output",
 ImageSize->{469, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->99743767]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->257205091],

Cell["Galaxy types of the brightest 10 galaxies:", "ExampleText",
 CellID->529776674],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<GalaxyType\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"Take", "[", 
   RowBox[{
    RowBox[{"AstronomicalData", "[", "\"\<Galaxy\>\"", "]"}], ",", "10"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13833634],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"BarredSpiral\"\>", ",", "\<\"BarredSpiral\"\>", 
   ",", "\<\"Spiral\"\>", ",", "\<\"Spiral\"\>", ",", "\<\"Spiral\"\>", 
   ",", "\<\"Spiral\"\>", ",", "\<\"BarredSpiral\"\>", ",", "\<\"Spiral\"\>", 
   ",", "\<\"Spiral\"\>", ",", "\<\"BarredSpiral\"\>"}], "}"}]], "Output",
 ImageSize->{410, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->529186187]
}, Open  ]],

Cell["Hubble types:", "ExampleText",
 CellID->21372812],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<HubbleType\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"Take", "[", 
   RowBox[{
    RowBox[{"AstronomicalData", "[", "\"\<Galaxy\>\"", "]"}], ",", "10"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->49139012],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"SB(s)m\"\>", ",", "\<\"SBm\"\>", ",", "\<\"Sb\"\>", 
   ",", "\<\"Sc\"\>", ",", "\<\"S0\"\>", ",", "\<\"Sb\"\>", 
   ",", "\<\"SBc\"\>", ",", "\<\"Sc\"\>", ",", "\<\"Sc\"\>", 
   ",", "\<\"SBm\"\>"}], "}"}]], "Output",
 ImageSize->{308, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->59411446]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32640975],

Cell["Surface brightness for the 10 brightest galaxies:", "ExampleText",
 CellID->297992517],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<SurfaceBrightness\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"Take", "[", 
   RowBox[{
    RowBox[{"AstronomicalData", "[", "\"\<Galaxy\>\"", "]"}], ",", "10"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->566279940],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Missing", "[", "\<\"NotAvailable\"\>", "]"}], ",", 
   "14.1`2.9999999999999964", ",", "13.5`2.9999999999999973", ",", "14", ",", 
   "13.3`2.9999999999999973", ",", "13", ",", "12.9`2.9999999999999987", ",", 
   "14.6`3.000000000000005", ",", "12.8`3.000000000000006", ",", 
   "13.3`2.9999999999999973"}], "}"}]], "Output",
 ImageSize->{490, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->307781515]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->152923281],

Cell["Alternate names for NGC 292:", "ExampleText",
 CellID->793447851],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<NGC292\>\"", ",", "\"\<AlternateNames\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->720248603],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"NGC 292\"\>", ",", "\<\"A 0051-73\"\>", 
   ",", "\<\"ESO 29-21\"\>", ",", "\<\"IRAS00510-7306\"\>", 
   ",", "\<\"Small Magellanic Cloud\"\>", ",", "\<\"SMC\"\>"}], 
  "}"}]], "Output",
 ImageSize->{522, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->514951502]
}, Open  ]],

Cell[TextData[{
 "Alternate names in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " form:"
}], "ExampleText",
 CellID->156881406],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<NGC292\>\"", ",", "\"\<AlternateStandardNames\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->202518144],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"NGC292\"\>", ",", "\<\"A0051-73\"\>", ",", "\<\"ESO29-21\"\>", 
   ",", "\<\"IRAS00510-7306\"\>", ",", "\<\"SMC\"\>"}], "}"}]], "Output",
 ImageSize->{336, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->423250161]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->212969740],

Cell["Principal Galaxy Catalog number for NGC 292:", "ExampleText",
 CellID->4597139],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<NGC292\>\"", ",", "\"\<PGCNumber\>\""}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->272378806],

Cell[BoxData["3085"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->71056133]
}, Open  ]],

Cell["Make a table of densities of planets:", "ExampleText",
 CellID->310272575],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Text", "[", 
  RowBox[{"Grid", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Name\>\""}], "]"}], ",", 
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Density\>\""}], "]"}]}], "}"}], "&"}], "/@", 
     RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}], ",", 
    RowBox[{"Frame", "\[Rule]", "All"}], ",", 
    RowBox[{"Background", "\[Rule]", "LightYellow"}]}], "]"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->382770234],

Cell[BoxData[
 InterpretationBox[Cell[BoxData[
   TagBox[GridBox[{
      {"\<\"Mercury\"\>", "5430.`3.000000000000005"},
      {"\<\"Venus\"\>", "5240.`2.9999999999999973"},
      {"\<\"Earth\"\>", "5515.0000000000000000001`3.9999999999999982"},
      {"\<\"Mars\"\>", "3940.`3.000000000000005"},
      {"\<\"Jupiter\"\>", "1330.`2.9999999999999973"},
      {"\<\"Saturn\"\>", "700.`1.000000000000002"},
      {"\<\"Uranus\"\>", "1300.`2.000000000000003"},
      {"\<\"Neptune\"\>", "1760.`3.000000000000006"},
      {"\<\"Pluto\"\>", "2000.`1.0000000000000009"}
     },
     ColumnsEqual->False,
     GridBoxBackground->{"Columns" -> {{
          RGBColor[1, 1, 0.85]}}},
     GridBoxFrame->{"Columns" -> {{True}}, "Rows" -> {{True}}},
     GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
     RowsEqual->False],
    "Grid"]], "Text", "TR"],
  Text[
   Grid[{{"Mercury", 5430.`3.000000000000005}, {
     "Venus", 5240.`2.9999999999999973}, {
     "Earth", 5515.`3.9999999999999982}, {"Mars", 3940.`3.000000000000005}, {
     "Jupiter", 1330.`2.9999999999999973}, {"Saturn", 700.}, {
     "Uranus", 1300.}, {"Neptune", 1760.`3.000000000000006}, {
     "Pluto", 2000.}}, Frame -> All, Background -> 
    RGBColor[1, 1, 0.85]]]]], "Output",
 ImageSize->{117, 191},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->4669868]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23684725],

Cell["\<\
Log-log plot of orbital period vs. distance for the planets, illustrating \
Kepler's Third Law:\
\>", "ExampleText",
 CellID->987398988],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogLogPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<OrbitPeriod\>\""}], "]"}], ",", 
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<SemimajorAxis\>\""}], "]"}]}], "}"}], "&"}], "/@",
    
   RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->416203837],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWb1uFDEQdm73Nro7DlEekKAUgZYHgEAkxE9DBRGiPAWkRIIkCqm4MmWO
AokGiRJ4ASJ4ABqoaKl5EWOv7fV4dtbr3fOhQFjpbnfs8TfjGX/27N398cHW
0+fjg+3N8crd/fHe1vbmi5U7u/uiKVlgjB2Lz7MRE89cPOWfY87Fvfx9TXwn
QiGVUldqmks8s0T3dsS9o+8sKbRyGfb3tJxUoqUYLYVaEKvvxUoJy4zdIPwz
shzVdbF0h1TIXF+IueYWFpEfABfHR7Z3jE0vct+DXO9XRs48QXJai1PvZ88T
XWODyGvQLPo12PXe0XE4PSvguvaAZnKK42IIqQ0VI72WOmCm1bw0ccA2DF8J
G6mdL2OLXgsV0XQQukVEoPdnvCNViz8GjIoJkgdwJWR4pVXgORZJ7tbPg845
W9P+GL8SDtYJyq3pK/tksg53ZepEgH3m9MgqPEgAFmMbCKmvtZaBZdenJTpy
JaReSe9iqaWsU4X+oNRe9mwW/Ic6KiZaZv5GvgTzp0/xy/mMP7z/IdpfTb+y
KWh9tPFRtN6+9U63rorvaTH+Cl4FSF4msgLlJSQPavqHpNVJkNZR41Y9SX1V
swRWEudIJB/rGQvZpYdBuJj79WdHGG47f6UI8MiKruS9F9VEOtF7Cuw7y/Uq
vrn2Vq7il+RMBrSN2hMPy8qaYIewJtmhrJnzSl5yn/XumEEWvn/7tbvz6cvn
n9rChYYoR3981Hk07wxpjVCe9jjFwBGKr9JKvfny8XCEoqnwDMNB8k4wK3Oc
1nWeB9NbPzRnpeW9O65R5eFZZ6vcXUFPgONxuKcs2NWnLDTlg0Kxa64dyjy5
2PxMnFdrUcm1PGHnxo2Z3irnxWPAMLDqG73xBJ6qrpViThFYbPfKoPeLILS/
/3S8l8t2b1vnFGeUlt2flFaMelUh2z1rvcjS6atfK3Aj1bGQueaXEVQxt6hb
cQ3j57O8Zj0raTaQMYk5yh9fJD/mLquu8hAuqlGWZWoU5qLSsoyhtZTn7etX
940UJPEEszDmORerTqVYR8UZ4oTWrf+5R43b4S73coRa7qlRlntqFGaV0rLc
o7X+zRp1guI6tDai1qwTlAlgJ/JcbB6VjVi1rf+kbVPbhjE9Vm17iDKt/p9o
yt9DlMdZUGymAIo3D685xfo3eWv5f2W28BvzgqK5\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->47458132]
}, Open  ]],

Cell["\<\
The corresponding plot for all known planetary moons, with a line for each \
planet:\
\>", "ExampleText",
 CellID->62958908],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogLogPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<OrbitPeriod\>\""}], "]"}], ",", 
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<SemimajorAxis\>\""}], "]"}]}], "}"}], "&"}], "/@",
    
   RowBox[{"AstronomicalData", "[", "\"\<PlanetaryMoon\>\"", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->574002368],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWVtvG0UUnnjXdpzUatqQNrRq46hchCJR4IGohcRxboQQIi6qoNCSpgXh
0EtQk4CEhPqAWgFq31qpsUnUVAWB+oAQVSvxgJCQEDwgXnnmjyxn5ox3dr8d
jx2HvBHVuz6z33zfOWfOXNZ9eX65/N75+eWFM/OFFy7Of1heOLNUmFy8SE1e
mxDiHn0+6hX0PaBv6nMvCOgevT5KV48epeieovs+sPc3sA8plsfk1ZcK6k8/
F55p6W2IePg/QBT+92ZL3ghp02dPQzaJ2mtHxVoaxyaZDjaZKbS7G0cDPR7a
ss821v1BbS7JaDx9TwNqNzxPKvcAwrd472Y4oG5P2/yL4QqAOKgZeSQORywJ
6gf0gTq6hy2sCTvWL9nCTE9pD/oidy9is5+8hvWCymq8tTi0eu3qr7pVZnid
GtvimMULPz64/7fYBUx9DWyMr9CMP9vc+oSyv77zl477vAP1xrFvCDUx9lUd
1C7g6lH2H7//o/Pl6mW4LztQhsuO2t7cqRUsTfFxmegon4eojxtslqIS6o+e
jQxXNOIoRBzpISMk+GixKjx6KvxSscJScyfvDkIOjoe+yhoXPjXfuP4bXamT
6u1RJx3BUfDydWucvFLd3vhzYnytOHRTpBTL1OQ6OyCOgOfMkgniu3wnsPo1
Vo5ERXYEYmGmbOBaS+XXdrDVqtlO3C9N3xovVYmZgg/DvNoiq+qRC70LAzb5
+6JF5g55ySSZm+WT+a3lOx/nCp270gSXbV9ie2X5PjXJ0VZzLpMM/rNw1N27
C4/76EiVJ4D2iPi5DmamNzJ1ssEKXUF0Z6Qxpu9ffv4L80heXU8pqPU01Baz
7Uz4hzuu8sYnHSqkkWJlZKgima812ZujNRWi8pHlyiSvycLqqcWXsTKZWcZ+
sc0EnaD1SWCbzYonG+Yh5kGqlnkSoKYxlUx6kANtO3M7oDzVyEqzMxtQLxkY
DxenQaXr1EYeYl+ysnUnYj/x1rdcdAJjtDPI9ViuVDnwqz5aeFcu/zxaql44
94O4ZEWp04pnPGmHWLrBftPKsgdQvvYUR6UbbBebQfkQCY6LB+p5eO5SMage
UC1bez0CWllgcfUy3Fgxpx29DHceWOy9+oG7C1i4V6Mdsh+0VoLmdgPcXfpB
vVWe1e3o55udhzGy4wDk76yLIR22aB5cOQcgjzlgnwtHI7pzedoXXNEHIJu4
bjBb451BndvqrMNvN8nxHGhvrp+JgfvV23kGrfnCmh+05sGOMsp21Had2Gch
kikHykTiQplIdgL3s9Zep4DbhTLcdtS7oPiMA2UUXSijaEcdA0XcmQa43Pyg
NqEjq1uXlQv2pSzbfPBYeP97zfu4IkyeJXHlyVs1TFTR/Z9PJHSc0/qtaqj3
1zRz8ouWVNoMmzTfgUzy6Y5tbnqyScbojoIzWamYMyMdNtLxFkrEDvCkz6Wb
WHuTOFx72AfKFyvSv/BMd/rU3ZMnvrv06U8s3QEVwp7wUYP/Yr9xubMC9pyy
8TxjtDqgclh7x6Z1FlRsnM3XXr0TLvT8dkJBahye+fa1qFeO1w2FkopEVnvP
jipz9sPn4aTbuxUPcswX/rpBbTy+Y6Wq8JUHVHy6h18b/4kx+ftCOCXZg84g
vi9jTZcjI0VzWrOTJjXJt8w0qCl/czBfg+juq1aJxNr3QU2H3420zscrD16c
XJNfWWd8dE3j+VQ6M31L/fpQrcN6tsaqRyf0nkqzNFQRGdto+a/M3g7Xhd1W
3nPKpvfMqcn10nCFR72DXOfnPBhTE2vF4QovtvSIz1dhhHS1cy+q8EpymCrF
4ZvRHxl0NZh5g+8JdsalSI3K2XfIgTJcdtRmzgPXg/isSzlQxjuDkvV4A7yy
c7B+8v/OolfR9i8STyAX\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->704475272]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->287381801],

Cell["\<\
Find the total number of known planetary moons in the solar system:\
\>", "ExampleText",
 CellID->235366638],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"AstronomicalData", "[", "\"\<PlanetaryMoon\>\"", "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->474882182],

Cell[BoxData["162"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20028348]
}, Open  ]],

Cell["\<\
Make a log rank plot of the masses of the moons, with names as tooltips:\
\>", "ExampleText",
 CellID->636489389],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{"Reverse", "[", 
   RowBox[{"Sort", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Tooltip", "[", 
       RowBox[{
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Mass\>\""}], "]"}], ",", 
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Name\>\""}], "]"}]}], "]"}], "&"}], "/@", 
     RowBox[{"AstronomicalData", "[", "\"\<PlanetaryMoon\>\"", "]"}]}], "]"}],
    "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->701856818],

Cell[BoxData[
 GraphicsBox[{{{}, 
    {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdzQ9MlHUcx/FnyGzIYhwN3BVDEBiDWDoWEhH6QUA8CjjuL3fA8XAQoMmR
wKFLjMdEL6k1cTbrWnqBpi0k5yAW0XnxR5Y16jDNVDZUDA2psaDRH1kU3+8a
z7M9e/Z6fs/7+URYazQv+giCULV4//v875JycCH0SNSas9lYepELd/vFkUEX
WcrFmQPJv0U383keApMarJodfJ6H8wmxO3VaPlfDErHdezmUDDVymu9crO5W
0fdqPD/Ve+WDka1L9qjx2oRZ6+glC/k4N+UTYnuHjHw4z1eH/rGbLOVjlVhb
Nbud+3y8kD0Z6a/jXgN3R9ItVxr3GvTV1lzyTeZeg0fe61K0/ZpFvQYnRFeT
FEgWtJgY6nxT5d1CvRY/qno3DHWSJS1W9F57pr6F7NGi69O436NfJgs62GKe
ffvqcCb1OtyfnP7edpYs6fC1sbJDPEr26HDjoF/P/sNkQY+bK70dnx/gXo+1
zT6pCVbu9TgY98R3kWnc6+EXM+P+czX3BqT2N6RFhnBvwIOS0TmlP/cGKKft
qaf+yqDeAJewKfn+A7JgxHDYPue602QY0b7n43hnK1kyIjunbIPBxr0RR+zm
K9HbuC9YbhSgNx9RP1VwXwDvU1kRu1K4L4Alypn+ZCL3Jgxum1I+t557E5KO
NsVOhnFvwtzDyS+Uq7g3YcC2J2NgMp16M+rnuwL0t8kwoyHleFLtTbJkhl2R
MXZoiOwx41rn7kxvN/eF6A7siz7+EfeFUP3devXh69wXorLfobBXc1+Ipom8
G5FG7ovQ0fbJqap07osQ4n5fHxTPfRFMPn1Pf6nkvgh3Nw4+Gj+/mfpinDwT
32a5Q0YxKhtfcTRfIksye2QWLCiY6T4ct4t7C/z2hd76sJ57C5L9DwVPa7i3
IKT6m7nPtnBfgqCekFwpjfsSNI76LsTGcV+CeytPmFoU3MssiIgbeXzz8L20
JYeL6HEMKU6OkSGirnx+7+wAWRQRHHPhpa/ayZKIY+X2uxENZJeIoIrMX2KK
yB4Rl8PyshYSyOMilBVvuB1ryUIpBlcrri+s4f1SrDfXODXBvF+KqURtSuZ1
0H4ptu6se+vdPrJUiln12LHZc2TX4v+8K9QBe8kemcdlFqzLHW7FtzbVTGsd
GTKLVryqCvh5nYX3rfAEtozuSOR9K/xPVwY4A3lf5nGZhbLFpyNXur2J9mVG
GTIaf/DtryGLMksyu2T2lOGxjvji/YaNtC+zUL7c4TLjf/8DSHvDkw==
      "]]}, {}}, {
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{59., 53.352512904495136`}}]},
      "\"Ganymede\""],
     Annotation[#, "Ganymede", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{60., 53.25600609492493}}]},
      "\"Titan\""],
     Annotation[#, "Titan", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{61., 53.0324081450338}}]},
      "\"Callisto\""],
     Annotation[#, "Callisto", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{62., 52.846264178964816`}}]},
      "\"Io\""],
     Annotation[#, "Io", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{63., 52.651014466396546`}}]},
      "\"Moon\""],
     Annotation[#, "Moon", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{64., 52.22521417314668}}]},
      "\"Europa\""],
     Annotation[#, "Europa", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{65., 51.41741262540072}}]},
      "\"Triton\""],
     Annotation[#, "Triton", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{66., 49.614313808697794`}}]},
      "\"Titania\""],
     Annotation[#, "Titania", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{67., 49.45725480974308}}]},
      "\"Oberon\""],
     Annotation[#, "Oberon", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{68., 49.190859557558156`}}]},
      "\"Rhea\""],
     Annotation[#, "Rhea", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{69., 48.94510226589519}}]},
      "\"Iapetus\""],
     Annotation[#, "Iapetus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{70., 48.835583740088694`}}]},
      "\"Charon\""],
     Annotation[#, "Charon", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{71., 48.65658997338741}}]},
      "\"Ariel\""],
     Annotation[#, "Ariel", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{72., 48.512722160515736`}}]},
      "\"Umbriel\""],
     Annotation[#, "Umbriel", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{73., 48.44545870248609}}]},
      "\"Dione\""],
     Annotation[#, "Dione", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{74., 47.8721334583152}}]},
      "\"Tethys\""],
     Annotation[#, "Tethys", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{75., 46.12892146426133}}]},
      "\"Enceladus\""],
     Annotation[#, "Enceladus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{76., 45.63505705388869}}]},
      "\"Miranda\""],
     Annotation[#, "Miranda", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{77., 45.36539348693959}}]},
      "\"Proteus\""],
     Annotation[#, "Proteus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{78., 45.081671815703906`}}]},
      "\"Mimas\""],
     Annotation[#, "Mimas", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{79., 44.876158495765964`}}]},
      "\"Nereid\""],
     Annotation[#, "Nereid", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{80., 43.56123605098297}}]},
      "\"Phoebe\""],
     Annotation[#, "Phoebe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{81., 43.3549448167467}}]},
      "\"Himalia\""],
     Annotation[#, "Himalia", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{82., 43.15920023962061}}]},
      "\"Hyperion\""],
     Annotation[#, "Hyperion", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{83., 43.129019622470615`}}]},
      "\"Sycorax\""],
     Annotation[#, "Sycorax", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{84., 43.044789888442864`}}]},
      "\"Larissa\""],
     Annotation[#, "Larissa", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{85., 42.76715815184458}}]},
      "\"Galatea\""],
     Annotation[#, "Galatea", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{86., 42.508905423416806`}}]},
      "\"Puck\""],
     Annotation[#, "Puck", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{87., 42.18733965659164}}]},
      "\"Despina\""],
     Annotation[#, "Despina", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{88., 42.172950919139545`}}]},
      "\"Amalthea\""],
     Annotation[#, "Amalthea", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{89., 42.086729743692416`}}]},
      "\"Janus\""],
     Annotation[#, "Janus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{90., 41.965980227070936`}}]},
      "\"Portia\""],
     Annotation[#, "Portia", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{91., 41.85086741997043}}]},
      "\"Thebe\""],
     Annotation[#, "Thebe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{92., 41.306140244528756`}}]},
      "\"Elara\""],
     Annotation[#, "Elara", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{93., 41.137517532092964`}}]},
      "\"Caliban\""],
     Annotation[#, "Caliban", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{94., 40.86200599526144}}]},
      "\"Juliet\""],
     Annotation[#, "Juliet", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{95., 40.821467084092674`}}]},
      "\"Nix\""],
     Annotation[#, "Nix", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{96., 40.821467084092674`}}]},
      "\"Hydra\""],
     Annotation[#, "Hydra", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{97., 40.803898364454156`}}]},
      "\"Epimetheus\""],
     Annotation[#, "Epimetheus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{98., 40.46457305885054}}]},
      "\"Thalassa\""],
     Annotation[#, "Thalassa", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{99., 40.415382814659765`}}]},
      "\"Belinda\""],
     Annotation[#, "Belinda", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{100., 40.37683414454253}}]},
      "\"Cressida\""],
     Annotation[#, "Cressida", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{101., 40.24142950753633}}]},
      "\"Pasiphae\""],
     Annotation[#, "Pasiphae", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{102., 40.07891057803855}}]},
      "\"Rosalind\""],
     Annotation[#, "Rosalind", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{103., 39.81064659144388}}]},
      "\"Naiad\""],
     Annotation[#, "Naiad", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{104., 39.768220747341644`}}]},
      "\"Prometheus\""],
     Annotation[#, "Prometheus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{105., 39.72223563409982}}]},
      "\"Desdemona\""],
     Annotation[#, "Desdemona", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{106., 39.54326978515284}}]},
      "\"Pandora\""],
     Annotation[#, "Pandora", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{107., 39.4204489554665}}]},
      "\"Carme\""],
     Annotation[#, "Carme", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{108., 39.325138775662175`}}]},
      "\"Metis\""],
     Annotation[#, "Metis", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{109., 39.07024652603338}}]},
      "\"Bianca\""],
     Annotation[#, "Bianca", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{110., 38.85513514641644}}]},
      "\"Sinope\""],
     Annotation[#, "Sinope", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{111., 38.68078175927166}}]},
      "\"Lysithea\""],
     Annotation[#, "Lysithea", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{112., 38.5266310794444}}]},
      "\"Ophelia\""],
     Annotation[#, "Ophelia", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{113., 38.34430952265045}}]},
      "\"Cordelia\""],
     Annotation[#, "Cordelia", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{114., 38.201208679009774`}}]},
      "\"Siarnaq\""],
     Annotation[#, "Siarnaq", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{115., 37.93884441454228}}]},
      "\"Ananke\""],
     Annotation[#, "Ananke", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{116., 37.77632548504451}}]},
      "\"Helene\""],
     Annotation[#, "Helene", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{117., 37.58216947060355}}]},
      "\"Setebos\""],
     Annotation[#, "Setebos", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{118., 37.58216947060355}}]},
      "\"Prospero\""],
     Annotation[#, "Prospero", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{119., 37.58216947060355}}]},
      "\"Albiorix\""],
     Annotation[#, "Albiorix", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{120., 36.93098648914264}}]},
      "\"Leda\""],
     Annotation[#, "Leda", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{121., 36.91134275333213}}]},
      "\"Phobos\""],
     Annotation[#, "Phobos", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{122., 36.647860233226716`}}]},
      "\"Paaliaq\""],
     Annotation[#, "Paaliaq", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{123., 36.552550053422394`}}]},
      "\"Adrastea\""],
     Annotation[#, "Adrastea", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{124., 36.511728058902136`}}]},
      "\"Telesto\""],
     Annotation[#, "Telesto", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{125., 36.329406502108185`}}]},
      "\"Stephano\""],
     Annotation[#, "Stephano", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{126., 36.13703460946073}}]},
      "\"Ymir\""],
     Annotation[#, "Ymir", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{127., 36.13703460946073}}]},
      "\"Pan\""],
     Annotation[#, "Pan", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{128., 35.81858087834219}}]},
      "\"Calypso\""],
     Annotation[#, "Calypso", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{129., 35.73156950135256}}]},
      "\"Kiviuq\""],
     Annotation[#, "Kiviuq", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{130., 35.53089880589041}}]},
      "\"Tarvos\""],
     Annotation[#, "Tarvos", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{131., 35.2795843776095}}]},
      "\"Atlas\""],
     Annotation[#, "Atlas", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{132., 34.922909433670775`}}]},
      "\"Deimos\""],
     Annotation[#, "Deimos", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{133., 34.71996858967408}}]},
      "\"Ijiraq\""],
     Annotation[#, "Ijiraq", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{134., 34.39838496554662}}]},
      "\"Callirrhoe\""],
     Annotation[#, "Callirrhoe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{135., 34.26976758772452}}]},
      "\"Erriapo\""],
     Annotation[#, "Erriapo", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{136., 34.24996496042834}}]},
      "\"Trinculo\""],
     Annotation[#, "Trinculo", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{137., 34.166583351489294`}}]},
      "\"Themisto\""],
     Annotation[#, "Themisto", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{138., 33.70523778498667}}]},
      "\"Praxidike\""],
     Annotation[#, "Praxidike", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{139., 33.47343617092935}}]},
      "\"Narvi\""],
     Annotation[#, "Narvi", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{140., 33.38246439272362}}]},
      "\"Skathi\""],
     Annotation[#, "Skathi", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{141., 32.97699928461546}}]},
      "\"Thrymr\""],
     Annotation[#, "Thrymr", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{142., 32.97699928461546}}]},
      "\"Suttungr\""],
     Annotation[#, "Suttungr", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{143., 32.97699928461546}}]},
      "\"Mundilfari\""],
     Annotation[#, "Mundilfari", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{144., 32.97699928461546}}]},
      "\"Megaclite\""],
     Annotation[#, "Megaclite", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{145., 32.90289131246174}}]},
      "\"Kalyke\""],
     Annotation[#, "Kalyke", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{146., 32.90289131246174}}]},
      "\"Iocaste\""],
     Annotation[#, "Iocaste", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{147., 32.73583722779857}}]},
      "\"Taygete\""],
     Annotation[#, "Taygete", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{148., 32.41738349668003}}]},
      "\"Harpalyke\""],
     Annotation[#, "Harpalyke", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{149., 32.129701424228266`}}]},
      "\"Thyone\""],
     Annotation[#, "Thyone", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{150., 32.129701424228266`}}]},
      "\"Hermippe\""],
     Annotation[#, "Hermippe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{151., 32.129701424228266`}}]},
      "\"Autonoe\""],
     Annotation[#, "Autonoe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{152., 31.88477030559443}}]},
      "\"Isonoe\""],
     Annotation[#, "Isonoe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{153., 31.88477030559443}}]},
      "\"Chaldene\""],
     Annotation[#, "Chaldene", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{154., 31.43655424366831}}]},
      "\"Eurydome\""],
     Annotation[#, "Eurydome", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{155., 31.43655424366831}}]},
      "\"Euanthe\""],
     Annotation[#, "Euanthe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{156., 31.43655424366831}}]},
      "\"Erinome\""],
     Annotation[#, "Erinome", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{157., 31.43655424366831}}]},
      "\"Aitne\""],
     Annotation[#, "Aitne", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{158., 30.3379419550002}}]},
      "\"Sponde\""],
     Annotation[#, "Sponde", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{159., 30.3379419550002}}]},
      "\"Pasithee\""],
     Annotation[#, "Pasithee", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{160., 30.3379419550002}}]},
      "\"Orthosie\""],
     Annotation[#, "Orthosie", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{161., 30.3379419550002}}]},
      "\"Kale\""],
     Annotation[#, "Kale", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{162., 30.3379419550002}}]},
      "\"Euporie\""],
     Annotation[#, "Euporie", "Tooltip"]& ]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{59., 30.3379419550002},
  FrameTicks->{{{{29.933606208922594`, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["13", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {34.538776394910684`, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["15", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {39.14394658089878, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["17", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {43.74911676688687, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["19", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {48.35428695287496, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["21", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {52.95945713886305, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["23", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {32.41851285871059, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.06910042485174, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.45996673353876, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.740268698692915`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.95895789965774, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.13829882831356, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.29031503561219, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.42224257865473, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.023683044698686`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.674270610839834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.06513691952685, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.345438884681, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.564128085645834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.74346901430165, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.895485221600275`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {39.027412764642825`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {41.62885323068678, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.27944079682793, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.67030710551494, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.950609070669096`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.16929827163393, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.348639200289746`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.50065540758837, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.63258295063092, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.234023416674866`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.88461098281602, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.27547729150303, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.55577925665719, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.774468457622014`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.95380938627783, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.10582559357646, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.237753136619006`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {50.83919360266296, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.48978116880411, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.880647477491124`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.160949442645276`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.37963864361011, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.558979572265926`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.71099577956455, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.8429233226071, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}, {{29.933606208922594`, 
       FormBox["\" \"", TraditionalForm]}, {34.538776394910684`, 
       FormBox["\" \"", TraditionalForm]}, {39.14394658089878, 
       FormBox["\" \"", TraditionalForm]}, {43.74911676688687, 
       FormBox["\" \"", TraditionalForm]}, {48.35428695287496, 
       FormBox["\" \"", TraditionalForm]}, {52.95945713886305, 
       FormBox["\" \"", TraditionalForm]}, {32.41851285871059, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.06910042485174, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.45996673353876, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.740268698692915`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.95895789965774, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.13829882831356, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.29031503561219, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.42224257865473, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.023683044698686`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.674270610839834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.06513691952685, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.345438884681, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.564128085645834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.74346901430165, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.895485221600275`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {39.027412764642825`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {41.62885323068678, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.27944079682793, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.67030710551494, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.950609070669096`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.16929827163393, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.348639200289746`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.50065540758837, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.63258295063092, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.234023416674866`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.88461098281602, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.27547729150303, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.55577925665719, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.774468457622014`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.95380938627783, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.10582559357646, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.237753136619006`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {50.83919360266296, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.48978116880411, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.880647477491124`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.160949442645276`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.37963864361011, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.558979572265926`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.71099577956455, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.8429233226071, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}}, {Automatic, Automatic}},
  GridLines->{None, None},
  Method->{},
  PlotRange->{{59., 162.}, {30.3379419550002, 53.352512904495136`}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  Ticks->FrontEndValueCache[{Automatic, {{29.933606208922594`, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["13", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {34.538776394910684`, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["15", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {39.14394658089878, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["17", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {43.74911676688687, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["19", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {48.35428695287496, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["21", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {52.95945713886305, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["23", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {32.41851285871059, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.06910042485174, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.45996673353876, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.740268698692915`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.95895789965774, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.13829882831356, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.29031503561219, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.42224257865473, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.023683044698686`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.674270610839834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.06513691952685, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.345438884681, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.564128085645834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.74346901430165, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.895485221600275`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {39.027412764642825`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {41.62885323068678, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.27944079682793, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.67030710551494, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.950609070669096`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.16929827163393, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.348639200289746`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.50065540758837, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.63258295063092, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.234023416674866`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.88461098281602, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.27547729150303, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.55577925665719, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.774468457622014`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.95380938627783, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.10582559357646, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.237753136619006`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {50.83919360266296, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.48978116880411, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.880647477491124`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.160949442645276`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.37963864361011, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.558979572265926`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.71099577956455, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.8429233226071, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}}, {Automatic, {{29.933606208922594`, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["13", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {34.538776394910684`, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["15", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {39.14394658089878, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["17", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {43.74911676688687, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["19", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {48.35428695287496, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["21", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {52.95945713886305, 
       FormBox[
        TagBox[
         SuperscriptBox[
          TagBox["10", Power, Editable -> True, Selectable -> True], 
          TagBox["23", Power, Editable -> True, Selectable -> True]], 
         InterpretTemplate[Superscript[#, #2]& ], Editable -> False, 
         Selectable -> False], TraditionalForm]}, {32.41851285871059, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.06910042485174, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.45996673353876, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.740268698692915`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {33.95895789965774, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.13829882831356, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.29031503561219, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {34.42224257865473, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.023683044698686`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {37.674270610839834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.06513691952685, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.345438884681, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.564128085645834`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.74346901430165, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {38.895485221600275`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {39.027412764642825`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {41.62885323068678, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.27944079682793, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.67030710551494, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {42.950609070669096`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.16929827163393, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.348639200289746`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.50065540758837, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {43.63258295063092, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.234023416674866`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {46.88461098281602, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.27547729150303, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.55577925665719, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.774468457622014`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {47.95380938627783, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.10582559357646, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {48.237753136619006`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {50.83919360266296, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.48978116880411, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {51.880647477491124`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.160949442645276`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.37963864361011, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.558979572265926`, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.71099577956455, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}, {52.8429233226071, 
       FormBox["\" \"", TraditionalForm], {0.00375, 0.}, {
        Thickness[0.001]}}}}]]], "Output",
 ImageSize->{184, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->168999918]
}, Open  ]],

Cell["\<\
Plot the RA and declination of the 100 brightest stars, with a tooltip for \
their name:\
\>", "ExampleText",
 CellID->559093282],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Tooltip", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<RightAscension\>\""}], "]"}], ",", 
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Declination\>\""}], "]"}]}], "}"}], ",", 
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<Name\>\""}], "]"}]}], "]"}], "&"}], "/@", 
   RowBox[{"AstronomicalData", "[", "\"\<StarBrightest100\>\"", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->210484892],

Cell[BoxData[
 GraphicsBox[{{{}, 
    {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJwllHk0VVscx6/ivZAhXUPqmrkPce/lIuP9yjxcInkoRMrQKpVKaUTKK1HJ
8EimkiEvmasbW5pU0vTQPCplpZCSennn8MdZZ62z9tn7u7+/z/erGR7ru3Ia
g8FYQT30e3/ntYbiaXMx69r55oONXJKfsdrhSu4cvHqfVTsz0J1w1paKiVLm
Y8P+9DfDHFOo6C2WKPU1wrP11IcYL9IfY/g1p8AEXuHyAZYRDojvXnvhzjxF
sIIirJ9POGO3a+sO1hgTiZpVS20jNUhHn+DE8+NqcJIpCxvrZ8LmZPLoLtlx
gfu2ZI7ncU8SpFBUll2tjKunGCpdRSxUJvm4FWjMh2X99c0tHC/yIO8t61Sv
KWqjrKzniDTBL6m3GHv1B4R7ZLzPpHmTkvBjrZIrFcDf3YSkJC4EV5807KH0
yKw/feAxwwVuxhOa/3Ub4IfVutupvnpE4koGL34HFxt1ciYOrbUiF81qliRJ
qCPp1bXcC7/ZgF7+ddQcaqf6XY0LbYl8gnmVXY4+Kpf7+bJ7hcTv9cn8yzV8
3E699+xFqTN8Hs3dHkb5M74zRrRxtReRk24o2+qig76w4m7+Jzaaz0x/l9Y/
d0rPmA0J8M87muSlhjZLbfdtrwVoUPxxc1BSHzfaF1ZHp3sS/x3/NIZn8MD1
LJNyPriQvG+iNjitiMhmsUL1VXNQIWK9rCeKmGGbv6Qw3wZ+FQ+nV6/Tgm+z
WsrAGh8yePZLsXSMEralOsfLW4+0PpsmuV9d2xx3HHs/f0lwIXLhkVIFTUoo
ucu6fJs50ao6zm4O8NFA3Ye4BsUqF5KpIu4v26mP+sDRc4/GPbDJZlWRiZ00
AsrNOq0HXTGSvnijTqQJ9D/KtHB59sS/sKq6V10PLtO3SOu88ILY7OZNFhnz
sLfDr608wopE0RtnGWLewCtz1XJXLBHKb1InGvh86qqHRZuQnJ5HgVPIw4bv
cu+bdjmR4dzEG2PflLGTtSDScaszWDRAXVyU/3nD5LK9DzFZ4B5SkKwKkYQC
U7CKi6l5akLye1FkXZAHsZXqKV23hg+PD+L8WSGehDJhcb+NOCZ9XOSP8bA9
n6OS54BJg/KFR+xl3XOjB7TwS9Yp0c5HkwhbHr69+oUBUfYhuYTNFmiwGLsU
UamDgt16l/b2mIJzot1BRfBOcPTrb6G5P3lkK23MgAmCfzmMf/SwIjOnGe5T
spyPRHFKGGch6fGeCBx53SaY5EjZFkPObw8LcocFp6t7Qzd32UNtkXmfFEcZ
hZ2VXf/6a5OJ2m3DNTVGuFns1Djyly+olJRJNZqDSuERq/UuhHuBLSedxcNG
Z0qBuv7Uf0qSaI+gCE1wxGz2N2tmCgMOtOEGTnj+QjqzooSNaEZsoriuEbKP
9OkctRsUpKYE3Rct9cLPiqF7wnB9KDfdZTxLcCXVey/bZqtpgMY1iuFIYjnR
ad2HtRDicf1321IhOVaife4nmwN52tAmK8TQBzpqYUT9XmxokxPJeiAZpXXL
AKOPexLjNnuAtuNhCH+Kfw1HmGSWK2/P6xNEx35O6XD0xEtKXtIQD9bPvQ/m
tLjBJzlaxU5LCXJ5wd/8de63cvY9jZ+Z0S4Y/zhALDOFoKex3M8QwWuu/Kq9
4E7SKJWxYlwc4sUbaRaak8m5d3NRHEmBp2tPLr5msjpWGuFTBxWwsy5k8loc
I2jTBcR0IpPcyekhnI4r1xNvqOV9u4zwYxl1gIL1VJ+IzJAecMBbjGhP5bKB
B8wOa7+f7EDM6sV6d894IrAWoxqJ7UTsh1bcCr7JBo3DQK87BGYh593m89BW
fsILcRwymQN5C6QM+DSrKdpA931eh0U+C5O9KrQlcT0Kw/beZljz7kfFkLs3
BFab+upEWsibMfd4SKIHoY/XvcIH1TaZKj8BigoxIdMCYSZLuAbRxjjz9/a6
TuEMZDVpH1NaJo/qaKq5j3OhMjkYXbLjTl3g6EPDqTyccCE0RqFb9FCkynmq
s5c/xa0yF1TazhrfNiX/A8+Vxvc=
      "]]}, {}}, {
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.75257, -16.7131}}]},
      "\"Sirius\""],
     Annotation[#, "Sirius", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.39919, -52.6957}}]},
      "\"Canopus\""],
     Annotation[#, "Canopus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.2612, 19.1873}}]},
      "\"Arcturus\""],
     Annotation[#, "Arcturus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.6614, -60.8351}}]},
      "\"Rigel Kentaurus A\""],
     Annotation[#, "Rigel Kentaurus A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.6156, 38.783}}]},
      "\"Vega\""],
     Annotation[#, "Vega", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.27814, 45.999}}]},
      "\"Capella\""],
     Annotation[#, "Capella", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.2423, -8.20164}}]},
      "\"Rigel\""],
     Annotation[#, "Rigel", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.65515, 5.22751}}]},
      "\"Procyon\""],
     Annotation[#, "Procyon", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{1.62854, -57.2367}}]},
      "\"Achernar\""],
     Annotation[#, "Achernar", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.91952, 7.40704}}]},
      "\"Betelgeuse\""],
     Annotation[#, "Betelgeuse", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.0637, -60.373}}]},
      "\"Hadar\""],
     Annotation[#, "Hadar", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.8463, 8.86738}}]},
      "\"Altair\""],
     Annotation[#, "Altair", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.4433, -63.0991}}]},
      "\"Acrux\""],
     Annotation[#, "Acrux", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{4.59867, 16.5098}}]},
      "\"Aldebaran\""],
     Annotation[#, "Aldebaran", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.27785, 46.0067}}]},
      "\"Capella Ab\""],
     Annotation[#, "Capella Ab", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{13.4199, -11.1612}}]},
      "\"Spica\""],
     Annotation[#, "Spica", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.4901, -26.4319}}]},
      "\"Antares\""],
     Annotation[#, "Antares", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.75538, 28.0263}}]},
      "\"Pollux\""],
     Annotation[#, "Pollux", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.9608, -29.6218}}]},
      "\"Fomalhaut\""],
     Annotation[#, "Fomalhaut", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.7954, -59.6887}}]},
      "\"Mimosa\""],
     Annotation[#, "Mimosa", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{20.6905, 45.2803}}]},
      "\"Deneb\""],
     Annotation[#, "Deneb", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.6609, -60.8395}}]},
      "\"Rigel Kentaurus B\""],
     Annotation[#, "Rigel Kentaurus B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.1396, 11.9672}}]},
      "\"Regulus\""],
     Annotation[#, "Regulus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.9771, -28.9721}}]},
      "\"Adhara\""],
     Annotation[#, "Adhara", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.57667, 31.8886}}]},
      "\"Castor\""],
     Annotation[#, "Castor", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.5194, -57.1126}}]},
      "\"Gacrux\""],
     Annotation[#, "Gacrux", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.5601, -37.1037}}]},
      "\"Shaula\""],
     Annotation[#, "Shaula", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.41885, 6.34973}}]},
      "\"Bellatrix\""],
     Annotation[#, "Bellatrix", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.43819, 28.6079}}]},
      "\"Alnath\""],
     Annotation[#, "Alnath", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{9.22007, -69.7175}}]},
      "\"Miaplacidus\""],
     Annotation[#, "Miaplacidus", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.60356, -1.20192}}]},
      "\"Alnilam\""],
     Annotation[#, "Alnilam", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.1372, -46.9606}}]},
      "\"Alnair\""],
     Annotation[#, "Alnair", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.67931, -1.94258}}]},
      "\"Alnitak\""],
     Annotation[#, "Alnitak", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.15888, -47.3366}}]},
      "\"Regor\""],
     Annotation[#, "Regor", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.9005, 55.9598}}]},
      "\"Alioth\""],
     Annotation[#, "Alioth", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{3.40537, 49.8612}}]},
      "\"Mirphak\""],
     Annotation[#, "Mirphak", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.4029, -34.3843}}]},
      "\"Kaus Australis\""],
     Annotation[#, "Kaus Australis", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.0622, 61.7511}}]},
      "\"Dubhe\""],
     Annotation[#, "Dubhe", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.13986, -26.3932}}]},
      "\"Wezen\""],
     Annotation[#, "Wezen", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{13.7924, 49.3133}}]},
      "\"Alkaid\""],
     Annotation[#, "Alkaid", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.37524, -59.5095}}]},
      "\"Avior\""],
     Annotation[#, "Avior", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.622, -42.9978}}]},
      "\"\[Theta] Scorpii\""],
     Annotation[#, "\[Theta] Scorpii", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.99216, 44.9474}}]},
      "\"Menkalinan\""],
     Annotation[#, "Menkalinan", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.8111, -69.0276}}]},
      "\"Atria\""],
     Annotation[#, "Atria", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.62853, 16.3994}}]},
      "\"Alhena\""],
     Annotation[#, "Alhena", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.74505, -54.7086}}]},
      "\"\[Delta] Velorum\""],
     Annotation[#, "\[Delta] Velorum", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{20.4275, -56.7349}}]},
      "\"Peacock\""],
     Annotation[#, "Peacock", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{2.52974, 89.2641}}]},
      "\"Polaris\""],
     Annotation[#, "Polaris", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.37833, -17.9559}}]},
      "\"Mirzam\""],
     Annotation[#, "Mirzam", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{9.45979, -8.65868}}]},
      "\"Alphard\""],
     Annotation[#, "Alphard", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{2.11952, 23.4628}}]},
      "\"Hamal\""],
     Annotation[#, "Hamal", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.3328, 19.8419}}]},
      "\"Algieba\""],
     Annotation[#, "Algieba", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.726452, -17.9867}}]},
      "\"Diphda\""],
     Annotation[#, "Diphda", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.9211, -26.2966}}]},
      "\"Nunki\""],
     Annotation[#, "Nunki", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.1115, -36.3687}}]},
      "\"Menkent\""],
     Annotation[#, "Menkent", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.139769, 29.0908}}]},
      "\"Alpheratz\""],
     Annotation[#, "Alpheratz", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{1.16217, 35.6208}}]},
      "\"Mirach\""],
     Annotation[#, "Mirach", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.79594, -9.6696}}]},
      "\"Saiph\""],
     Annotation[#, "Saiph", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.8451, 74.1555}}]},
      "\"Kochab\""],
     Annotation[#, "Kochab", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.7111, -46.8846}}]},
      "\"\[Beta] Gruis\""],
     Annotation[#, "\[Beta] Gruis", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.5822, 12.5606}}]},
      "\"Rasalhague\""],
     Annotation[#, "Rasalhague", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{3.13615, 40.9557}}]},
      "\"Algol\""],
     Annotation[#, "Algol", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{2.06498, 42.3298}}]},
      "\"Almach\""],
     Annotation[#, "Almach", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.8177, 14.5723}}]},
      "\"Denebola\""],
     Annotation[#, "Denebola", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.945139, 60.7167}}]},
      "\"Tsih\""],
     Annotation[#, "Tsih", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.692, -48.9599}}]},
      "\"\[Gamma] Centauri\""],
     Annotation[#, "\[Gamma] Centauri", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.05974, -40.0032}}]},
      "\"Naos\""],
     Annotation[#, "Naos", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{9.28484, -59.2753}}]},
      "\"Aspidiske\""],
     Annotation[#, "Aspidiske", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{15.5781, 26.7149}}]},
      "\"Alphekka\""],
     Annotation[#, "Alphekka", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{9.13327, -43.4326}}]},
      "\"Alsuhail\""],
     Annotation[#, "Alsuhail", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{13.3987, 54.9254}}]},
      "\"Mizar\""],
     Annotation[#, "Mizar", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{20.3705, 40.2567}}]},
      "\"Sadr\""],
     Annotation[#, "Sadr", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.675108, 56.5374}}]},
      "\"Shedir\""],
     Annotation[#, "Shedir", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.9434, 51.489}}]},
      "\"Etamin\""],
     Annotation[#, "Etamin", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.53344, -0.299093}}]},
      "\"Mintaka\""],
     Annotation[#, "Mintaka", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.152803, 59.1502}}]},
      "\"Caph\""],
     Annotation[#, "Caph", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{13.6648, -53.4664}}]},
      "\"\[Epsilon] Centauri\""],
     Annotation[#, "\[Epsilon] Centauri", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.0056, -22.6216}}]},
      "\"Dschubba\""],
     Annotation[#, "Dschubba", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.8362, -34.2926}}]},
      "\"\[Epsilon] Scorpii\""],
     Annotation[#, "\[Epsilon] Scorpii", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.6988, -47.3881}}]},
      "\"\[Alpha] Lupi\""],
     Annotation[#, "\[Alpha] Lupi", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.5918, -42.1577}}]},
      "\"\[Eta] Centauri\""],
     Annotation[#, "\[Eta] Centauri", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.0307, 56.3823}}]},
      "\"Merak\""],
     Annotation[#, "Merak", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.7498, 27.0742}}]},
      "\"Izar\""],
     Annotation[#, "Izar", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{21.7364, 9.87501}}]},
      "\"Enif\""],
     Annotation[#, "Enif", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.7081, -39.0299}}]},
      "\"\[Kappa] Scorpii\""],
     Annotation[#, "\[Kappa] Scorpii", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.438019, -42.3051}}]},
      "\"Ankaa\""],
     Annotation[#, "Ankaa", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.8972, 53.6947}}]},
      "\"Phad\""],
     Annotation[#, "Phad", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.173, -15.7251}}]},
      "\"Sabik\""],
     Annotation[#, "Sabik", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{23.0629, 28.0825}}]},
      "\"Scheat\""],
     Annotation[#, "Scheat", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.40158, -29.3031}}]},
      "\"Aludra\""],
     Annotation[#, "Aludra", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{21.3096, 62.5855}}]},
      "\"Alderamin\""],
     Annotation[#, "Alderamin", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{9.36856, -55.0107}}]},
      "\"\[Kappa] Velorum\""],
     Annotation[#, "\[Kappa] Velorum", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{20.7701, 33.9695}}]},
      "\"\[Epsilon] Cygni\""],
     Annotation[#, "\[Epsilon] Cygni", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{23.0793, 15.2054}}]},
      "\"Markab\""],
     Annotation[#, "Markab", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{3.03799, 4.08993}}]},
      "\"Menkar\""],
     Annotation[#, "Menkar", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.6193, -10.5672}}]},
      "\"\[Zeta] Ophiuchi\""],
     Annotation[#, "\[Zeta] Ophiuchi", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{13.9257, -47.2883}}]},
      "\"\[Zeta] Centauri\""],
     Annotation[#, "\[Zeta] Centauri", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.2351, 20.524}}]},
      "\"Zosma\""],
     Annotation[#, "Zosma", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.0906, -19.8054}}]},
      "\"Acrab\""],
     Annotation[#, "Acrab", "Tooltip"]& ]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 23.0793}, {-69.7175, 89.2641}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26389875]
}, Open  ]],

Cell["The corresponding picture for the 100 nearest stars:", "ExampleText",
 CellID->405468550],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Tooltip", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<RightAscension\>\""}], "]"}], ",", 
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Declination\>\""}], "]"}]}], "}"}], ",", 
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<Name\>\""}], "]"}]}], "]"}], "&"}], "/@", 
   RowBox[{"AstronomicalData", "[", "\"\<StarNearest100\>\"", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->68133414],

Cell[BoxData[
 GraphicsBox[{{{}, 
    {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxllHs01GkYx+e4RC67GSY0LjOWcR1mjBlSzOM2lMuElUvasuiIyqW1ZN06
pMx2YVU2NiHX6EKpjlRvuUUrKhGLWBROKkKLZN+ZOftP+8fv/M7nnOd93uf5
Pt/npf4Y6RUqQSAQgvEn/FOC3wXYfzEBl/tJmla+fKRG85a+6EWHoWjBybFw
D+TZT04MwryUHN7wU4QHmtu3PenKAhNqFm9zvq8mQkmUjVzvqB4ocTbf9eJo
ALdloC5dgwbENNRetGIH00Yln0sClrhuZXI8vXkm+poFHa11RRJkUGqtv33i
JgMtXdc21JEkQ6bd7vGuWgYKFxbSbQ6mq9SVhReWaPnxu7bGWkvYgHZQ7NR5
EB7eF7ArRAHkBurzy6d0UL7PharLapbg+kjGJn/aDpmEqv/tl6sPtAOkJuid
5AqPzVzkAG0sraDjkimyPeRXwdZkw9xKd/zkDw7QNs4teVWgBU6KZUH/TKjA
QdNmn9ALWrClfs9sx4IKBIRsesXH8f6qt54ScDy99vz78pvmIErX6Q57Po0F
FWEeLc1vonS5w0crhddXs55zRf1I8ECLk31OPvA592l24BcHzFnMODp1lQLv
rZycY9qsgfIscpfMNxwYrEtP7Ap2Qxt+Lbue82CZW1niAcojZshtuGdws89H
rig/gYn4a5zbqp5rQkvx5NuQDyrgql1feK9ZBVLwuHJSeUiY7skWNmR/m8Cp
OuCAepr3W6TGceBDeYurZZ8bzNp17nmBucn+8t4hzP1twwaH5tXh5KbStPmU
NUhZTilKFnNn/37VvIg1SKRTGR3scTs6ZGs0kqByxfkwAyp8282bPA2R/Zk8
m652BUhZto56Ajyks7IQO6xhCIpV8d99lNWBrxmN8AsaW25y9xIiD0tdtkfs
XwpJarNyILlWoB180gidS5c/JGU5xXVPV+RfPU2CO47zPWiBKvbXPAUYxP4Y
U2kC9Gx/a3ZH3wjiE0pNHBL1oPOTN5bIA205n0p7iOut2RaRF6dhja5PHawj
5etBBW4/s4iGdg7wJOPzmWBj0p33muQJorkkMAGrpu4Z44zOhB4L3qlhKdbv
Hh2JyjiiDyJd2dtQbrS3ZMebBi51ypZQqaiFcPQAb5QDplmF/zAzTJCrf4ZV
zzgLiKV3H6hyeFCYeW1kEnOsRJ4sGfPBXuKsnRlNrGejEwzThQdpoCBhfHT9
AydoPoUNEqQLnytnnrnXu4DxQu5gTrku4Oi/FodYIPL7BAPO1ipEV7P46MrL
XT/zk5nifS63gvGgoh6LU2ywFBow1QVJxsvrGhQwgayIbzLgIZHu0kpgbhOU
KXNaCxkRWlhhskqwKqVqHB71fxb5pZUjfh+MeNBrgZ2rag6MAjuHVSkpWIdl
ssWs6xTi10yWAqyqZvlLFtSGWW9Sb6CC/+DUxsmkLq7Ib1QGigzRWkyboQJW
NfRYIwv+Y9GeYq5cmIz/k6AG7/ROPLW+JC3er7P6EHjq4bxBtbdY1wxj+N2N
dHxrOx0I2cnL1sMWgLtMPr7IQCSvZK9pZSKECQef5A5ejhk+DUpEsb6H3cGY
ERJbVU4GB1xOdDEA7mpWu1YPcnzfqGXZkMHw8brWerX1MHqp9r1gtyK6fVXy
zXE/FgjtfewhD3F9d9Tcf8uAtV9u/aZZRRH7HfMcP6qmuIyCJvqqlwKnGUDK
DiJFNlGQ8Lm1rSDDUM7RIwEXAeR5erESAhYIsC3OhHvC/e7OfMENNZi8hS8a
YKPq5nKCWud68bv62QDW+W288aiFCS/Whunsq3BDf7CKb1i6sCDapy/O6ZoK
zMxt9c/AnKDeHONapQIy50LGTPZRwN4WVyJgI5FOU3RwevlhbuteNkqcvbYt
YkkLbIRG2q8Amc+GhuWVzaDLEQcUOyI5hknKPfoU90TuAWLahBv8C7As0jc=

      "]]}, {}}, {
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.4966, -62.6814}}]},
      "\"Proxima Centauri\""],
     Annotation[#, "Proxima Centauri", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.6614, -60.8351}}]},
      "\"Rigel Kentaurus A\""],
     Annotation[#, "Rigel Kentaurus A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.6609, -60.8395}}]},
      "\"Rigel Kentaurus B\""],
     Annotation[#, "Rigel Kentaurus B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.9636, 4.66829}}]},
      "\"Barnard's Star\""],
     Annotation[#, "Barnard's Star", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.945, 7.05305}}]},
      "\"Wolf 359\""],
     Annotation[#, "Wolf 359", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.0557, 35.9815}}]},
      "\"Lalande 21185\""],
     Annotation[#, "Lalande 21185", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{1.64713, -17.9576}}]},
      "\"Luyten 726-8 A\""],
     Annotation[#, "Luyten 726-8 A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{1.64713, -17.9576}}]},
      "\"Luyten 726-8 B\""],
     Annotation[#, "Luyten 726-8 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.75257, -16.7131}}]},
      "\"Sirius\""],
     Annotation[#, "Sirius", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.75306, -16.6985}}]},
      "\"Sirius B\""],
     Annotation[#, "Sirius B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.8303, -23.8358}}]},
      "\"HIP 92403\""],
     Annotation[#, "HIP 92403", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{23.6983, 44.1973}}]},
      "\"Gl 905\""],
     Annotation[#, "Gl 905", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{3.54901, -9.45831}}]},
      "\"\[Epsilon] Eridani\""],
     Annotation[#, "\[Epsilon] Eridani", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{23.0964, -35.8563}}]},
      "\"Lacaille 9352\""],
     Annotation[#, "Lacaille 9352", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.7956, 0.807526}}]},
      "\"HIP 57548\""],
     Annotation[#, "HIP 57548", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.6404, -15.3336}}]},
      "\"Gl 866 A\""],
     Annotation[#, "Gl 866 A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{21.1141, 38.7415}}]},
      "\"HIP 104214\""],
     Annotation[#, "HIP 104214", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.65515, 5.22751}}]},
      "\"Procyon\""],
     Annotation[#, "Procyon", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.65566, 5.24102}}]},
      "\"Gl 280 B\""],
     Annotation[#, "Gl 280 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{21.1145, 38.7344}}]},
      "\"HIP 104217\""],
     Annotation[#, "HIP 104217", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.7135, 59.6224}}]},
      "\"HIP 91772\""],
     Annotation[#, "HIP 91772", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.7134, 59.626}}]},
      "\"HIP 91768\""],
     Annotation[#, "HIP 91768", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.305706, 44.022}}]},
      "\"HIP 1475\""],
     Annotation[#, "HIP 1475", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.302493, 44.0177}}]},
      "\"Gl 15 B\""],
     Annotation[#, "Gl 15 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.4983, 26.7869}}]},
      "\"GJ 1111\""],
     Annotation[#, "GJ 1111", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.0548, -56.7798}}]},
      "\"\[Epsilon] Indi\""],
     Annotation[#, "\[Epsilon] Indi", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{1.73476, -15.9396}}]},
      "\"\[Tau] Ceti\""],
     Annotation[#, "\[Tau] Ceti", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{1.20831, -17.0005}}]},
      "\"HIP 5643\""],
     Annotation[#, "HIP 5643", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.45671, 5.23476}}]},
      "\"Luyten's Star\""],
     Annotation[#, "Luyten's Star", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.19311, -45.0045}}]},
      "\"Kapteyn's Star\""],
     Annotation[#, "Kapteyn's Star", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{21.2883, -38.8646}}]},
      "\"Lacaille 8760\""],
     Annotation[#, "Lacaille 8760", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.4668, 57.697}}]},
      "\"Kruger 60\""],
     Annotation[#, "Kruger 60", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.4681, 57.7021}}]},
      "\"Gl 860 B\""],
     Annotation[#, "Gl 860 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.48972, -2.81248}}]},
      "\"HIP 30920\""],
     Annotation[#, "HIP 30920", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.48929, -2.80254}}]},
      "\"Gl 234 B\""],
     Annotation[#, "Gl 234 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.826, -26.106}}]},
      "\"HIP 72511\""],
     Annotation[#, "HIP 72511", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.505, -12.6597}}]},
      "\"HIP 80824\""],
     Annotation[#, "HIP 80824", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{3.59903, -44.5063}}]},
      "\"GJ 1061\""],
     Annotation[#, "GJ 1061", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.5564, 9.01748}}]},
      "\"Gl 473 A\""],
     Annotation[#, "Gl 473 A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{12.5564, 9.01748}}]},
      "\"Gl 473 B\""],
     Annotation[#, "Gl 473 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.0889704, -37.3517}}]},
      "\"HIP 439\""],
     Annotation[#, "HIP 439", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{3.36821, -13.2781}}]},
      "\"HIP 15689\""],
     Annotation[#, "HIP 15689", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.819216, 5.3952}}]},
      "\"Van Maanen's Star\""],
     Annotation[#, "Van Maanen's Star", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.98194, 8.47975}}]},
      "\"NN 3522\""],
     Annotation[#, "NN 3522", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{2.00253, 13.0776}}]},
      "\"Gl 83.1\""],
     Annotation[#, "Gl 83.1", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.7427, -61.216}}]},
      "\"NN 3618\""],
     Annotation[#, "NN 3618", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.8257, -26.1112}}]},
      "\"HIP 72509\""],
     Annotation[#, "HIP 72509", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.8029, -11.3155}}]},
      "\"NN 3622\""],
     Annotation[#, "NN 3622", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.6073, 68.3422}}]},
      "\"HIP 86162\""],
     Annotation[#, "HIP 86162", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.4776, -46.8931}}]},
      "\"HIP 85523\""],
     Annotation[#, "HIP 85523", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{23.1108, -14.8724}}]},
      "\"HIP 114110\""],
     Annotation[#, "HIP 114110", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.7609, -64.8407}}]},
      "\"HIP 57367\""],
     Annotation[#, "HIP 57367", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.112912, -7.51334}}]},
      "\"GJ 1002\""],
     Annotation[#, "GJ 1002", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.8878, -14.2621}}]},
      "\"HIP 113020\""],
     Annotation[#, "HIP 113020", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.8978, 44.4226}}]},
      "\"GJ 1245 A\""],
     Annotation[#, "GJ 1245 A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.8981, 44.4227}}]},
      "\"GJ 1245 B\""],
     Annotation[#, "GJ 1245 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.0923, 43.5245}}]},
      "\"HIP 54211\""],
     Annotation[#, "HIP 54211", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.0976, 43.5085}}]},
      "\"Gl 412 B\""],
     Annotation[#, "Gl 412 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.1898, 49.4555}}]},
      "\"Groombridge 1618\""],
     Annotation[#, "Groombridge 1618", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.3273, 19.871}}]},
      "\"Gl 388\""],
     Annotation[#, "Gl 388", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.9089, -62.4037}}]},
      "\"HIP 82725\""],
     Annotation[#, "HIP 82725", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.4934, 24.6532}}]},
      "\"HIP 85605\""],
     Annotation[#, "HIP 85605", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{21.5594, -49.007}}]},
      "\"HIP 106440\""],
     Annotation[#, "HIP 106440", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.6178, -44.3169}}]},
      "\"HIP 86214\""],
     Annotation[#, "HIP 86214", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{4.2549, -7.64456}}]},
      "\"\[Omicron] 2 Eridani\""],
     Annotation[#, "\[Omicron] 2 Eridani", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{4.25821, -7.60978}}]},
      "\"Gl 166 B\""],
     Annotation[#, "Gl 166 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{4.25821, -7.60978}}]},
      "\"Gl 166 C\""],
     Annotation[#, "Gl 166 C", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{22.7806, 44.3351}}]},
      "\"HIP 112460\""],
     Annotation[#, "HIP 112460", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.0909, 2.50244}}]},
      "\"HIP 88601\""],
     Annotation[#, "HIP 88601", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{18.0908, 2.51356}}]},
      "\"Gl 702 B\""],
     Annotation[#, "Gl 702 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.8463, 8.86738}}]},
      "\"Altair\""],
     Annotation[#, "Altair", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.257685, -16.1323}}]},
      "\"HIP 1242\""],
     Annotation[#, "HIP 1242", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.97171, 19.7638}}]},
      "\"GJ 1116 A\""],
     Annotation[#, "GJ 1116 A", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.97171, 19.7654}}]},
      "\"GJ 1116 B\""],
     Annotation[#, "GJ 1116 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.00078, 2.70812}}]},
      "\"NN 3379\""],
     Annotation[#, "NN 3379", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{11.7942, 78.69}}]},
      "\"HIP 57544\""],
     Annotation[#, "HIP 57544", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{13.7619, 14.8951}}]},
      "\"HIP 67155\""],
     Annotation[#, "HIP 67155", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{20.8759, -16.9748}}]},
      "\"HIP 103039\""],
     Annotation[#, "HIP 103039", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{4.51946, 58.9821}}]},
      "\"HIP 21088\""],
     Annotation[#, "HIP 21088", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{4.51732, 59.0085}}]},
      "\"Gl 169.1B\""],
     Annotation[#, "Gl 169.1B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.91374, 33.2691}}]},
      "\"HIP 33226\""],
     Annotation[#, "HIP 33226", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{10.8479, 6.81012}}]},
      "\"HIP 53020\""],
     Annotation[#, "HIP 53020", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.52415, -3.67214}}]},
      "\"HIP 25878\""],
     Annotation[#, "HIP 25878", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.3459, -45.5198}}]},
      "\"Gl 754\""],
     Annotation[#, "Gl 754", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.9248, -8.33421}}]},
      "\"HIP 82817\""],
     Annotation[#, "HIP 82817", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.9253, -8.32546}}]},
      "\"Gl 644 B\""],
     Annotation[#, "Gl 644 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{16.9272, -8.38366}}]},
      "\"Gl 644 C\""],
     Annotation[#, "Gl 644 C", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.9143, 33.2762}}]},
      "\"Gl 251\""],
     Annotation[#, "Gl 251", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.5391, 69.6654}}]},
      "\"Alsaphi\""],
     Annotation[#, "Alsaphi", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{6.17631, -21.8629}}]},
      "\"HIP 29295\""],
     Annotation[#, "HIP 29295", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{5.70224, 12.4932}}]},
      "\"HIP 26857\""],
     Annotation[#, "HIP 26857", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{17.7765, -57.3158}}]},
      "\"HIP 86990\""],
     Annotation[#, "HIP 86990", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.2821, 5.17214}}]},
      "\"HIP 94761\""],
     Annotation[#, "HIP 94761", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{19.2833, 5.16728}}]},
      "\"Gl 752 B\""],
     Annotation[#, "Gl 752 B", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{8.21128, -21.5417}}]},
      "\"Gl 300\""],
     Annotation[#, "Gl 300", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{14.9576, -21.4113}}]},
      "\"HIP 73184\""],
     Annotation[#, "HIP 73184", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{7.74455, 3.55355}}]},
      "\"HIP 37766\""],
     Annotation[#, "HIP 37766", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{15.5372, -41.2731}}]},
      "\"HIP 76074\""],
     Annotation[#, "HIP 76074", "Tooltip"]& ], 
    TagBox[
     TooltipBox[
      {Opacity[0.], PointBox[{{0.818084, 57.8165}}]},
      "\"Achird\""],
     Annotation[#, "Achird", "Tooltip"]& ]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 23.6983}, {-64.8407, 78.69}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->730756979]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->510844526],

Cell["Rank plot of absolute magnitudes of the nearest 100 stars:", \
"ExampleText",
 CellID->155558654],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Sort", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"AstronomicalData", "[", 
      RowBox[{"#", ",", "\"\<AbsoluteMagnitude\>\""}], "]"}], "&"}], "/@", 
    RowBox[{"AstronomicalData", "[", "\"\<StarNearest100\>\"", "]"}]}], "]"}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->294669162],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWNtvDkEU33a/7+vXSoO2etMiJEIkLgmJl6Zt3B6o8EAjKfG5pG3Erfrg
EkEQIhVUEUrcWiIePBASrxLhyatn/8iambN75szZs+OjKTb5Zndmf+d3LnPm
zOzXUxoeOHi4NDy4v7Rg01Dp2MDg/hMLNh4dUkNhRRAE79SvvzlQz5F6Mr/a
KIqCUD+ZK5jQTWBGc2p0cuJbZ8f96yOfnrvjvTteqPEN6x6y8a9ffhw98vbD
++9kXGSvVINBiPTBM4vPAx5VCO9QDX1XhXrUZZmfZCAsfxbCarEIbXezumtn
K+N+a+I9SurRlvgeEinab1L3kLCE5J6OXDPxHlsjS1FNriUCYo5uChQh4xrK
xM3xakxs1MlXoe+iby1lWa0BQR5tidmb/kN/WgWJlA7+3tFZrleaqTKbYZr9
aPRqD8U3jdM8D8lKK0dHtv10teHgP80pkLBVrT5yq4Zcg1qJh1k1iNee7BoE
Nti6+TgblYfaee7sx9jeRyI2Z7DAuK5rPMaOe7DAu3XL09gGwNao9oWC5kTJ
/DT3c39Z39/2pyBGtSpKZyR938b67b/oc7xPK2ZMUKnuQaj0x+/veKQwd0Aq
BxkH5o+JctUmHEpu397Xu/teKVHQpzIvRoyinF43Rr6AeEz/UZF9JvFF57IP
Zc8iMmpCHJ1t7Ic5AktuOPbWmfdgARhxw8ODfikzZFxDSt+IB0f1+nBUr4xz
z5XNLDOviTJmh0zVnqseLK89MrYlsXlX70sgvSLiWtns+1B29i1Ke9rOPL0s
crS50VbASx6c0qXMAaMAF5fV+Er2El412omlKvFTK/OcIUnvtXy35P35qVy5
MCUmmk2/w6S7C0m8B/vfxD6qpzj+p//QsoUkF7o6HgBvQT2Bz6BAtTymQ1PR
NwPm6s7YZ4iHeoCJB71oB6l2oC8+HpgrOTHwbFiimxrUQPVgVRwSs3CFboo4
3+g7yBxGGa11ZRTXW2Y41HcQkiN3QNRt+IrMauRms4JFQ+Zao5vadIxxK3Hq
zj6RoyOS6o6M7UyyG+qO4HOfKNcl5V4Ilkk53uvMQLcbMeoczgNGTInvFG3Y
oJvqrHyU9vntIs/myNYg3OdJ/spSPUkW0fxM4mezSCnfLMpv869eDCWP5Pps
NmFtstDKNULRdjvzs8sbWTvfuMyCtNdrRTv3uDMvrcBiOgbAuFpkLLkrphz/
SUasEjkPJXNLrVTWcOnlovQRKZ8KeG5Ef5Y5MT+erET31JiHaNg1v1TUecLN
RggEnjytxYtF6ZOJbtzDTSXGf07gSoR41T5F/NWnDZDlu0fI+vz9GeMBngIx
SoumwIb7AQaRx2Ne2ey6e8YbZTtHhDX1dc2jd16ed7Li51q21L8LBcZ2MbGQ
riz023I2RnxPzsX28e84c5rL0/VlczpdR+rF/DJn3ioaN15J7IzMEhlGEr/o
ilRtHaK17TeJrfZfBJnxVmTrr94ffajs8+kMUeq2mVN6+qzNxglffpZVe3WP
WSrrvJvkEX5B1Ig4zuZD2TUto+Tvx/umb78vio43D5j+oofD6pdRsv5xpl/+
T2Cc2eFDWTssSvsyaXpBxU84y9S6\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->327665559]
}, Open  ]],

Cell["Rank plot of absolute magnitudes of the brightest 100 stars:", \
"ExampleText",
 CellID->81977173],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Sort", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"AstronomicalData", "[", 
      RowBox[{"#", ",", "\"\<AbsoluteMagnitude\>\""}], "]"}], "&"}], "/@", 
    RowBox[{"AstronomicalData", "[", "\"\<StarBrightest100\>\"", "]"}]}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21510336],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJw10XlQlHUcx/FHYQOUU2BZEEmBiGjjECRuPiw3ArvAwi6YJo5kiiMgGEco
61IYEx6jIIRIQxEdhgkUJtiwJpCNk1ooihByxFG0cigBgpL1fJ/fzDPPvJ73
8/nrt2FHWmzKSoZhdj5//nuzZ8r//exLH0xmbVDRB7SE6658VcuKrI184cFx
Zd5asiFmH9VZm3ZakE2Q4n7N1uaIOVmAgn5p7Y85ArIljp2UjQlumZHXIyk0
3jhoHZ9sDcH2F8+2NJqSbaHrM2fl6MnZDv5eB0aaLDnbw/jM8dRAHmcH5BRc
ltWXmJCFyNDw8y9x5uyIgj2Xs07NG5OdceDaL+ka1WvILvB+ID5aHs15I0xl
ZQdlmpxdYRYmWRI0GZHd0D7xhNc6aEjehEKnyLpVnZzdwd9UUTR4lfPrKLfM
jtBK5eyBoq/3zVY9NiB7wlJUVul7grMXKmKEF79N4uwNeaZi+7FqfbIP7JPr
1G+a65F9oT+/5Cya1yX7YYvIb7znb87+SDS7+CuzjTPQOpNYrTW3mrUCGE4f
lWUyZCYAUe/pib8p1aEegDL97mXPSDIjQvkX/Rb3vtKmLkJqZXbVJzZkJhBK
y1LnrndeoB6Ihs/XPasUkpkgzFyQpFbO8agHIbLYNs+8g8wEw8nt7u7aFE3q
wTgUq+6aFqygHoLzs8GiLTeW29gegtY7mb8tdj5lzYTigsZYib16iXoomLTD
mh0Gi9TDkPz046v5C0+ohyH9o64Ul4YF6uFwXnN/v2PwPPVwOHo+LNXvnqMe
gUaFTl/LWrIigr3fiVnqm1G4z2l3ScI09c04rdN2++aZKeqRSG/KaXBun6Qe
CTuL2oFXdqqpR0GpkdSiVE9Qj0J226qp30f/pB4Nh7RQj9dyxqhHI8PbTRiy
dYS6GFf23Wu2d/2DuhgdHTGXipOGqUtQdyv36FZrMiTg7dWy+UdniP6XwMa0
k6+sGmCtksCi+Ja2+/l+2segr9DwhFNNF+1jkP5GZ5SeG1kRgyGe0Ze7zciq
GHz2wxUzu7jrtI+FcNdkSbPDT7SPhUtlmTqwv4X2seBFNx/2lhyhfSwyhvmP
MgPv+LP7OPwcLzwUJ7/LGnGwrOizb+wmK+Lg4RSd/92nPaxVz7tt0IKSf5/2
Uty7nStdekCGFHKf2bDC3n7aS/Fyz1/7I1QDtJfi+q5cq76zQ7SPxymL+8xe
FRnxqBd72n347ijt43F8nXTIK2Sc9vEI0M1Srx4nMwmYNnJ56eTb07RPgChw
WdPs3AztEzDS5GPw8MZj2idgkdea3Gs9S3sZarrdJj2C52gvw0aNsulJ1wXa
y5Dnfs5P1rtIexlUN79PX1/7jPZyDO4I1qurWQF2L4fYir8nKpPHWiFHfVG7
7+ltRqxVcrz1/zFmzSTCqv2GiddNPv4F0lbWBA==
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 100.}, {-8.72839, 5.70237}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->241423021]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->198389329],

Cell["\<\
Distribution of basic spectral classes for the 100 brightest stars:\
\>", "ExampleText",
 CellID->713178108],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Tally", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"First", "[", 
     RowBox[{"AstronomicalData", "[", 
      RowBox[{"#", ",", "\"\<SpectralClassList\>\""}], "]"}], "]"}], "&"}], "/@",
    
   RowBox[{"AstronomicalData", "[", "\"\<StarBrightest100\>\"", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->876371901],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\<\"G\"\>", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"A\"\>", ",", "23"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"F\"\>", ",", "9"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"K\"\>", ",", "18"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"M\"\>", ",", "7"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"B\"\>", ",", "32"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Missing", "[", 
      RowBox[{"\<\"Uncertain\"\>", ",", 
       RowBox[{"{", 
        RowBox[{"\<\"B\"\>", ",", "\<\"M\"\>"}], "}"}]}], "]"}], ",", "1"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"O\"\>", ",", "5"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Missing", "[", 
      RowBox[{"\<\"Uncertain\"\>", ",", 
       RowBox[{"{", 
        RowBox[{"\<\"B\"\>", ",", "\<\"K\"\>"}], "}"}]}], "]"}], ",", "1"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Missing", "[", 
      RowBox[{"\<\"Uncertain\"\>", ",", 
       RowBox[{"{", 
        RowBox[{"\<\"A\"\>", ",", "\<\"B\"\>"}], "}"}]}], "]"}], ",", "1"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{550, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28604921]
}, Open  ]],

Cell["Distribution for the 100 nearest stars:", "ExampleText",
 CellID->305950212],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Tally", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"First", "[", 
     RowBox[{"AstronomicalData", "[", 
      RowBox[{"#", ",", "\"\<SpectralClassList\>\""}], "]"}], "]"}], "&"}], "/@",
    
   RowBox[{"AstronomicalData", "[", "\"\<StarNearest100\>\"", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->389325059],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\<\"G\"\>", ",", "4"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"M\"\>", ",", "59"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"K\"\>", ",", "15"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"A\"\>", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Missing", "[", "\<\"NotAvailable\"\>", "]"}], ",", "19"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"F\"\>", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{471, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->767134692]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->750604839],

Cell["Rank plot of the distances to the nearest 100 stars:", "ExampleText",
 CellID->273686367],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"AstronomicalData", "[", 
     RowBox[{"#", ",", "\"\<DistanceLightYears\>\""}], "]"}], "&"}], "/@", 
   RowBox[{"AstronomicalData", "[", "\"\<StarNearest100\>\"", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->59074871],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWFtLFVEUHp1zUxFL8gqBIXmjwkBIKPGAlQ+lUFQWKHWyULsdsVNRIV2g
F0m6YA9qEdXRiB56SAp6DaKeeu25PzLtvdecfVl7zZzjtSDB2e49a31r7W++
tWaPfanM6PnLqczYcKqhdyI1Pjo2fLXhYHqCLblFjuMssd9MrcP+9thf4nfJ
89ioX5vZNcJuFbOxmI31fJUveLkfvlrnj64/1qJ5DRtdDcXVRsdp4dcoYMqr
sHVcGUlgaJEJiyp+iekWtN2WAu2qQiPmcuRUFvFRrLUWlCU3cKIyto9W8w/k
30ZYWJj4vhGj0F1U+RoIQFjjvKtDo7nknep15Lm5QOzgvHmN5apGLm6oVlo8
s5rp3lCv7SCoN+CeoHqD3ZHKUeZZtcpjL2R/dnfNTj/8+kasl7LrIluOmFY/
vv9OX/n4+dOvCEKLrvF8vfE3eo73EyOfyVY035fn/nYSJZtnlatkYUWe/3MU
x5Ul4rxW9n4nGTi+yO71JOedYjY6LrPyUV5YtlBE/Ydf+bYHep4btlw7PA2h
oYTeF5jn2TPvhwbfsQp8KWwTKGfcn7KBVnwafNfGiPk8YGWru2F6B6u4j4Hv
wlzxqzhTEYFhxpXOqIUhG1SA1d9UWQztcj7ESu123sgijnZJY8RRpLkQK6Tc
yNjIBzaC40yIH1JxJNk1B34sadqvRDhLDT+b+SbF7MwYu1w+1/6Lyv/JvTOx
VisQL0+Fk31GwBWA5xXoGS0Hh08r0FNcaR6b0X4erwYnAvsBqEerQYqxnTFa
QFaANw3urnpKufMKfkaVlLq0fjplKKUqIJ6ysk5rUYltdGzQPWfxAelbQ+UV
k6qX7vdJb/6lZ2ofEmb+gBQFAJXFJIkjTgAJyEIHgatdwTdJlAYTBWEBH5JK
54bBeBOflUpf2I8kE1yukVHbLPZ57dC2O4TymO2pgbeAmSHtdgpM2cP8jCeC
bS1lKcbTxj7bTWUBx+zvNIndru0NGL0z+cXS7gXSt8OsPhbjUoid1BALgHcw
QvrtMfHlE1Z+54ydd/JZCanVgIo5Tcbda3ESVjeD0BqiqkPAJ5RL9IjuwjI0
KuGEJzuahuSiOb7fU8gepPCOrShGr4d7gywqi+kjBUfg0z5Kwyxbm51DCtf6
EsXs9wdzYpwCNEzrexufF4/yS5nNARO50RV7PfweiXj0mfKkmIPqZZfVanE/
qdmTOcb0nkgyljRqRqg39EyUJOMNWUpG+yaU3EkiDVP7TSCNSiI7gjHKUG9H
TZ7icjeJdjGMS03Tuwwm0xSTrmy/mrUea9yqIfnigL5veYj3SVx/c6iMWoM9
jPcXxUUT6Xu9QC4aDS5u5eGikYx1m18SOhfwOoIQ20ifu57xjmAJ0aeSuhBv
vc/omUqWlIpz+Bpvm0jkewGKREcWot4qDSanQpUdVm2lZF5PllVt9H97BEZ5
3mrznzWNMatpEmjh3xO0rf+lVPQHPdaH8Q==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->138569807]
}, Open  ]],

Cell["Differences between successive distances:", "ExampleText",
 CellID->597222975],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Differences", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"AstronomicalData", "[", 
      RowBox[{"#", ",", "\"\<DistanceLightYears\>\""}], "]"}], "&"}], "/@", 
    RowBox[{"AstronomicalData", "[", "\"\<StarNearest100\>\"", "]"}]}], "]"}],
   "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->206411732],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlH1MTXEYx09mIuQqJDVTYaEJKSrpW91y89LL7XbvpSZLVkwpVpkl10pb
5mVeZqZaroSWbmiRl6srIXqfab0s4woRK2+xKw3rOX/0nO3u7HM/z/d3nvOc
3zkOsbvk28YIgrD13+//eeQY8B2ck7Qo0CgB/YHds0tsy4Yf+Y7geAijDglW
3bQwf1/1i/w0vNDpymdd+0M8Eyebc1SfU18R2+Pw6n2zV6haiOew9RxhdE1P
13lqyc9lfj6cZhwpLkr/Qt4ZTvbbY1yzeogXIjolRZF0oZbYBaHHrbb8kTYR
L0Zhj2VeXcND4iU4avvV9txQHfFStJaXFQ5XtBIvQ1vWqcxJ8VeI3Vg/yxm7
Q3q9/pKVoojqPfBh6JlnvEmc3wpWv5KxJ94ab48vcBD78WLeGy1641izo9fJ
r4JH884b2sEGYh+o7ZuGdliK/a+Gvs/Re15bO7Evpu65eOxDXBUxENJ7Likz
4ekIa8Cu54dGx2lRfs/byPsx748LQ6d0of1d5P1xMOTngN30S7R+ACwnLMj1
mHqLfADi8lMzdpSK85SOXk8jhV327e76/Y/JB+LK5LOtHdb3KR+Ipf0NMRku
58kH4eRhSYLJg+alCWL9rUG+fPMi7d1r5NdgyQl708oeYkGG6rrpznt8SsnL
WD6Y9ReMPKecvQdsxPmuZX4turIq2+6Fivt7HfPr4Lbz9I+83BLy65lfj5po
v9xDEvF5bWB+A35oJ36MemcgHwLl6egpyfI71H8Iio2VZse+ifsjlOUZC2Ho
d7xsoX35YqQeYayesYGxED6aEQ5Te9VGG6tq6od5Qzha78Zbz6h/Qv3JUbgf
vz910f6FHJJ3dmnZe2k+GjnLy5HnbYrRBXZTPgJjEsdZdzZWUD4C+u9I1WfS
fDQRGGh5IG3U0v4wROBMae3rlEGqFxSsf8YaBeISC57KzDsor4CZPvl9jUzs
P5LlI5Hp7xJ+dQJ9fzTMGxgLSqQnuQcFxdLzgpJdn7FBiaK0aouBbvH9UiFx
sUzqXyzevwqaXblWb/rE91nF8owFNSIflVdkVIl5NZbv68wZVFZSXs3yapzf
5NbrFVDu+xdXjkiC
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 99.}, {0., 0.6061399499999948}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->408235679]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->702308210],

Cell[TextData[{
 "Find the constellations with the largest number of stars in ",
 Cell[BoxData[
  ButtonBox["AstronomicalData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AstronomicalData"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->55515327],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Take", "[", 
  RowBox[{
   RowBox[{"SortBy", "[", 
    RowBox[{
     RowBox[{"Tally", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"AstronomicalData", "[", 
         RowBox[{"#", ",", "\"\<Constellation\>\""}], "]"}], "&"}], "/@", 
       RowBox[{"AstronomicalData", "[", "\"\<Star\>\"", "]"}]}], "]"}], ",", 
     "Last"}], "]"}], ",", 
   RowBox[{"-", "10"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->94408],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\<\"Cetus\"\>", ",", "2000"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Eridanus\"\>", ",", "2024"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Draco\"\>", ",", "2117"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Pegasus\"\>", ",", "2163"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Virgo\"\>", ",", "2177"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"UrsaMajor\"\>", ",", "2308"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Hydra\"\>", ",", "2459"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Hercules\"\>", ",", "2562"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Cygnus\"\>", ",", "2617"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\<\"Centaurus\"\>", ",", "2971"}], "}"}]}], "}"}]], "Output",
 ImageSize->{588, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->208844345]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->316093768],

Cell["Find the number of overlaps of different star catalogs:", "ExampleText",
 CellID->515222443],

Cell[BoxData[
 RowBox[{
  RowBox[{"o", "=", 
   RowBox[{"AstronomicalData", "/@", 
    RowBox[{"{", 
     RowBox[{
     "\"\<BayerObject\>\"", ",", "\"\<FlamsteedObject\>\"", ",", 
      "\"\<GlieseObject\>\"", ",", "\"\<HDObject\>\"", ",", 
      "\"\<HipparcosObject\>\"", ",", "\"\<HRObject\>\"", ",", 
      "\"\<SAOObject\>\""}], "}"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->642429306],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Grid", "[", 
  RowBox[{
   RowBox[{"Outer", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Length", "[", 
       RowBox[{"Intersection", "[", 
        RowBox[{"#1", ",", "#2"}], "]"}], "]"}], "&"}], ",", "o", ",", "o", 
     ",", "1"}], "]"}], ",", 
   RowBox[{"Frame", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->363173261],

Cell[BoxData[
 TagBox[GridBox[{
    {"1486", "934", "252", "1486", "1486", "1486", "1486"},
    {"934", "2517", "278", "2517", "2517", "2517", "2517"},
    {"252", "278", "3802", "1384", "3802", "593", "1384"},
    {"1486", "2517", "1384", "81150", "81150", "9014", "81150"},
    {"1486", "2517", "3802", "81150", "87475", "9014", "81150"},
    {"1486", "2517", "593", "9014", "9014", "9014", "9014"},
    {"1486", "2517", "1384", "81150", "81150", "9014", "81150"}
   },
   ColumnsEqual->False,
   GridBoxFrame->{"Columns" -> {{True}}, "Rows" -> {{True}}},
   GridBoxItemSize->{"Columns" -> {{Automatic}}, "Rows" -> {{Automatic}}},
   RowsEqual->False],
  "Grid"]], "Output",
 ImageSize->{294, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4326025]
}, Open  ]],

Cell["Rank plot of the distances to the brightest 200 galaxies:", \
"ExampleText",
 CellID->284740032],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"Sort", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"AstronomicalData", "[", 
      RowBox[{"#", ",", "\"\<Distance\>\""}], "]"}], "&"}], "/@", 
    RowBox[{"Take", "[", 
     RowBox[{
      RowBox[{"AstronomicalData", "[", "\"\<Galaxy\>\"", "]"}], ",", "200"}], 
     "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29185774],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxd1HswlXkcx/FfRUvZsqXdRbZjSWqbLpJQ+LDSVbnL/YjMVnQhl+2iZyWO
XCKt0q4culgppAgVz8pgt1R2aVFJoSjTtpOm1NTs0/j+/tg9M2fOvM77+Ty/
f85z9NdvddkwmjGWJL0/fo68XtiYpvWbDCa529MXKDDZVGh6NYWshhxlY+uJ
5wfJmji9pTCoxOkQWQuetqlpVoWZ5C9xpb1Mf/JD7qk4ZRdzZfHxw2QZYh8V
pxYYZpG/xq6zDzbapnEb4rp5jPoxcyXZCNPebfhQdPkk2RhpWzTM7O6fIc/C
yqfRsZ1TysizsdfatqVs9wXyHIzeUBWqrXWRPA892T02Wbrl5Pl4Ha1eOt6K
2wR2dcfyVMO5F8BmVFqlURm3KcKDtxwZV1VBXoj0OUNNykZuM5Rd8Eyc+Y57
ESpkzmFGKZfI5nA/maDWNraSbIGIqZbTfp7AbYnmzduuHbXmXoy8YyqvBzWq
yEtQ2WfgGFDLbQXfWSEtg8Pc1tA6YV8fW1NNtoEeCnaodF4hA3pZehGZdldH
LABr/74+RiWvlrothNxcb8P8X6lL9n1mObGPzOzAxFPJY3ddoy557ycndb3q
qX8LNn7Spy1GDdQlT4//LD+ZzOzBhrutInY0UreHYHJDu/cNmS0FUy9SKHOb
qEuuL5x7b9Nv1B0gpMf1qUz+nbrk0wsNT0beoL4MzMnxQOoBsrAMQpf23W9G
3aS+HEJwVda2YrKwHKzONVvZfIv6CkBFvVRn3m3qKyCGXnQs9SKzlcA1OwcN
H94l68+YlB7M+yqgoESz+g3vkq3LkyrUW6ivhtC2ytNuKVlYDby9E1iQyLsj
WMry8IIk3h0hKoJOhVfyvgbYCbOay7yvgfhW73BQA+9rwc6bXVI84X0thPVN
GkO+f1B3gtjUK9OIIsMJQmRAXH79n3S95M81z9TNbBux6AQW8/H5IDNniDFH
tOKryXAGvu9ywDBZcAZ7mZvc4HqH9pJ3hhZVRpGZC0TDj78HMlwgTJ8UpaP+
F+1dALX03ouLyKLUjdKbR/eRmStQarDrxlsyXMEsnIoeqLbTXvKhpxaDr8ii
K0Rdt639sg7au0EY2FlerE+GG5hlbNz72WTBDehcNyXenixK11fk57nW8L07
xFYzVZ3bfO8OYWPObLN+vpe6i3tm4ahO2ruDCT/m1BqQmQeEopu2OuZkeECc
dW5MIbfgAdakzBry5nsPwDj5WVEP33tCOHhoyPgx33sC+5rn5mnfo70nxOdX
W1/tJouSjxv+siqFzNYB8fkNbW1kSGYWP1wY4HvJJfeX6o27T/t1EOMH3mRo
kZkXEDJNc/N2MiSrutmfDScLXmAJNQcbbvO91BNU6+ImdtHe+7+G5Pl6xqGG
ZMEbooaaoqyBLHqDKfSTI8Y9oL0PxDNJe6NCyPCRnpeY4Mx4siDZdIIi4y5Z
9AH2BLYn9vK9L7qPfwgL8uymveRzTbkOcrLgC9kL5d3ySLLoC6WuQWpVP5n5
QfzJ4fQXag9p7yf93+2JnmdOFvwgz3BS3b6CLEqeev6f+SFk5o/uiDGtIVF8
7w/x3tliq/187w/5nY7L1XV87w8hIfvW8JJHtA+ALKLczWI1GQGQv2wPO7KZ
LARAeD9hSLGnh/ZS98v+yiWHzOToPtBh/F0tWSaHXBbdn9xBxv8sl0N2tDHq
lKyX7i+HWJKRfiuRrJRD2eG84Mn4PjpPDnZuwT6F/eMRd8shzIjc3mjXT+cH
oltvSZiiYIDOD4RQXPOha/9z+38BIyOGjg==
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageSize->{228., Automatic},
  PlotRange->{{0., 137.}, {0., 1.1531238395045593`*^24}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{232, 126},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61473859]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->322191060],

Cell["\<\
Find how many objects are listed in both the NGC and Messier catalogs:\
\>", "ExampleText",
 CellID->54932935],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Length", "[", 
  RowBox[{"Intersection", "[", 
   RowBox[{
    RowBox[{"AstronomicalData", "[", "\"\<NGCObject\>\"", "]"}], ",", 
    RowBox[{"AstronomicalData", "[", "\"\<MessierObject\>\"", "]"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->168714980],

Cell[BoxData["106"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->244307830]
}, Open  ]],

Cell["Pluto is included in the list of Solar System planets:", "ExampleText",
 CellID->491774688],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->384592560],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"\<\"Mercury\"\>", ",", "\<\"Venus\"\>", ",", "\<\"Earth\"\>", 
   ",", "\<\"Mars\"\>", ",", "\<\"Jupiter\"\>", ",", "\<\"Saturn\"\>", 
   ",", "\<\"Uranus\"\>", ",", "\<\"Neptune\"\>", ",", "\<\"Pluto\"\>"}], 
  "}"}]], "Output",
 ImageSize->{472, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->365153391]
}, Open  ]],

Cell["Though it is considered a \"dwarf planet\":", "ExampleText",
 CellID->47977959],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AstronomicalData", "[", 
  RowBox[{"\"\<Pluto\>\"", ",", "\"\<ObjectType\>\""}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->656229],

Cell[BoxData["\<\"DwarfPlanet\"\>"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->506311957]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->390175453],

Cell["\<\
Some information is unknown, for a variety of different reasons:\
\>", "ExampleText",
 CellID->9042815],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"#", ",", "\"\<DiscoveryYear\>\""}], "]"}], "&"}], "/@", 
  RowBox[{"AstronomicalData", "[", "\"\<Planet\>\"", "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->204553130],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}], ",", 
   RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}], ",", 
   RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}], ",", 
   RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}], ",", 
   RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}], ",", 
   RowBox[{"Missing", "[", "\<\"Unknown\"\>", "]"}], ",", "1781", ",", "1846",
    ",", "1930"}], "}"}]], "Output",
 ImageSize->{494, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->361378564]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->154101200],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AstronomicalData",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AstronomicalData"]], "InlineFormula"],
 " combines star catalogs that select stars in different ways:"
}], "ExampleText",
 CellID->303067630],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{
   RowBox[{"BinCounts", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"AstronomicalData", "[", 
        RowBox[{"#", ",", "\"\<ApparentMagnitude\>\""}], "]"}], "&"}], "/@", 
      RowBox[{"AstronomicalData", "[", "\"\<Star\>\"", "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "30", ",", "1"}], "}"}]}], "]"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}], ",", 
   RowBox[{"Mesh", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->573837874],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->428608285]
}, Open  ]],

Cell[BoxData[""], "Input",
 CellID->282925789],

Cell["\<\
Plot diameter against discovery year for the known moons of Saturn:\
\>", "ExampleText",
 CellID->60751421],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<DiscoveryYear\>\""}], "]"}], ",", 
      RowBox[{"AstronomicalData", "[", 
       RowBox[{"#", ",", "\"\<Diameter\>\""}], "]"}]}], "}"}], "&"}], "/@", 
   RowBox[{"AstronomicalData", "[", 
    RowBox[{"\"\<Saturn\>\"", ",", "\"\<Satellites\>\""}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->185632713],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWUtPFEEQnt2ZBZEFTySoweBf0IMngrguJqBGY4yJMTGbVQMHgSB/QSQG
rkZBEiRo9OIFohdiiHqAqPEgRg968VdwG/s5PVNT0/PcYVc3MDNd89Wjq7qq
a3sv1GbGbt+tzYzXa/3npmtTY+P1e/3Dk9OEZBYMw9gj/496DfJskyf2v2fb
5O6/niBXkwCKdEQfbPZh4yIZmuLe5cZZDk5QJarsQrm5MSlFccf4GcHgHytA
hhtjInIsjR2c9SS9ltxzccS2KU+wB8RDnSh3mwtlIFwHvf4DCMiBSejUSpCx
C7UB4iPb5NgQoJE9RrGg5LyPMHM5psu7QO8qgvs/k2aIpy8njXbHuYFeRvQL
biuV1S0aPVtVpcgVwMddiqdzX1ZMOYbGXGOHx8Ijt5yH/5pMc3c28WqE7R6K
mZ/PmlQv4wqpHdEiRjsz2atgPVUXeC/pEle2g3sfyE/vVsb88D3eg7l70eMg
ArsodZteSjeuvxwceDJ8dplcF+Y/iM2uj1x/CFsM45h2DbDHlOOs5eU9bjX7
G20vk1+avf+OkGo3X01OrL998xPfXXNYXd0Zy/vX7IHjAxnL4/F8vvaVkGiN
sbD3yDo4AnBwfDTm+1NsfO3qC0KqVp4aHUwNWhfNne0/YtXiKE1NpTMV1VTx
UtC21E/e5q3fj2otO+7cek1QVy6vaVFnTi8RFPnTolaffRkdWXk4txWA2gTW
92hQ50dWCGpocFGLunRxlaAqQ4tcY6hnNjHfh3oK2t0RKYKbQLYJpByKZKEJ
pOBc7wEXjsLXFE7F5xzNv8CfEf0Lo5J1htCebge1R3an/tMyWlHdvS38Tojn
MpPVDmQJiez8Tbsrw11iV8nUdBqlNHJT2UqHXBY88fT7M8BK7SmlslXKp9eu
VJLc8XT3/JRuGd4zEsXV66JK2ZTrOxsfFiNco/4t3GEtj+RkGcvrpviqIT64
j8JWOczqHseHQSsy2UrneuYebBE9UxPrAXpgJsRf+fnOR1UHlx78VB/OUn8+
IGxS+ZfuHA7zrT+v4bdumClQwobtXZV9TBTsIHEuFSHOFZRDG8DPCu1bJ2jm
YfOIs1fGoeL+wFFw/jgKzlvudQtAUxWVMQ806VBKE47CZzwB7KhrUMoOHUrZ
gaOUHXFraBzqKJjXrAal5sVRyTuOUeABIY9e/5uOIzwKiTuRgPqKRzFCPxKj
1uKRDepUwuou9zVWZ+N0LPHqZ7Jsy6JDwfe/9Ds4LjdZB6K801A7U3YWzB5h
AwXxX+JUfn1EZxDv11bMhyx/nG/9S0KPrvPw/9oBVyiXKc8IpMzeiHwqExWf
mWG2+NHKy59tLF/4OuNViAM/aXDE/mpleXDgMSlYOK4NzBNHNaoTai5qIzsF
HTV5J9CI3bSVdv6UZw6g3jkZKGRR2jePBtsO2+WxOsfH6mxc7QBB5w9e+fxM
n+QoeaY5yvnDa5g6saDE32wEa9QvRpU1iY/4L51G4S/qYG/c\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{248, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->434454952]
}, Open  ]]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"AstronomicalData - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 25, 1.0501978}", "context" -> "System`", 
    "keywords" -> {
     "absolute magnitude", "Acamar", "Achernar", "Achird", "Acrab", "Acrux", 
      "Acubens", "Adhafera", "Adhara", "Adrastea", "Ain", "Aitne", "Alaraph", 
      "Albali", "albedo", "Albiorix", "Albireo", "Alchiba", "Alcyone", 
      "Aldebaran", "Alderamin", "Alfirk", "Algenib", "Algieba", "Al Giedi", 
      "Algol", "Algorab", "Alhena", "Alioth", "Alkaid", "Alkalurops", "Alkes",
       "Almach", "Alnair", "Alnath", "Alnilam", "Alnitak", "Alphard", 
      "Alphekka", "Alpheratz", "Alrai", "Alsaphi", "Alshain", "Alsuhail", 
      "Altair", "Altais", "Alterf", "Aludra", "Alula Borealis", "Alwaid", 
      "Alya", "Amalthea", "Ananke", "Ancha", "Angetenar", "Ankaa", "Antares", 
      "apoapsis", "apparent magnitude", "Arcturus", "Ariel", "Arkab", "Arneb",
       "Arrakis", "Ascella", "ascending node longitude", "Asellus Australis", 
      "Asellus Borealis", "astrology", "astronomy", "astrophysics", "Atik", 
      "Atlas", "Atria", "Autonoe", "Auva", "Avior", "Azha", "Baham", 
      "barred spiral galaxy", "Baten Kaitos", "Bayer name", "Bayer object", 
      "Beid", "Belinda", "Bellatrix", "Betelgeuse", "Bianca", 
      "blue band magnitude", "Botein", "bright giant", "bright region", 
      "Caliban", "Callirrhoe", "Callisto", "Calypso", "Canopus", "Capella", 
      "Caph", "Carme", "Castor", "Celbalrai", "Chaldene", "Chara", "Charon", 
      "Chort", "class star", "cluster type", "constellation", "Cor Caroli", 
      "Cordelia", "Cressida", "Cujam", "Cupid", "Cursa", "Dabih", "Daphnis", 
      "declination", "deep sky object", "Deimos", "Deneb", "Deneb Algedi", 
      "Denebola", "Desdemona", "Despina", "Dione", "Diphda", 
      "distance light years", "Dnoces", "Dschubba", "Dubhe", "Dziban", 
      "Earth", "Earth moon", "eccentricity", "Edasich", "Elara", 
      "elliptical galaxy", "Enceladus", "Enif", "Epimetheus", 
      "equatorial diameter", "equatorial radius", "Erinome", "Erriapo", 
      "escape velocity", "Etamin", "Euanthe", "Euporie", "Europa", "Eurydome",
       "Flamsteed name", "Flamsteed object", "Fomalhaut", "Furud", "Gacrux", 
      "Galatea", "galaxy", "galaxy type", "Ganymede", "Giauzar", "Gienah", 
      "Gliese name", "Gliese object", "globular cluster", "Gomeisa", 
      "gravity", "Grumium", "Hadar", "Hamal", "Harpalyke", "Heka", "Helene", 
      "Hermippe", "Himalia", "Hipparcos catalog name", 
      "Hipparcos catalog number", "Hipparcos catalogue", "Hipparcos object", 
      "Homam", "Hubble type", "Hydra", "hypergiant", "Hyperion", "Iapetus", 
      "Ijiraq", "inclination", "Io", "Iocaste", "irregular galaxy", "Isonoe", 
      "Izar", "Jabbah", "Janus", "Juliet", "Jupiter", "Jupiter moon", 
      "Kaffaljidhm", "Kale", "Kalyke", "Kaus Australis", "Kaus Borealis", 
      "Kaus Media", "Kitalpha", "Kiviuq", "Kochab", "Kornephoros", "Kraz", 
      "Kurah", "Larissa", "Leda", "Lysithea", "Maasym", "Maaz", "Mab", 
      "main sequence star", "Marfik", "Markab", "Mars", "Mars moon", "mass", 
      "Matar", "Mebsuta", "Megaclite", "Megrez", "Mekbuda", "Menkalinan", 
      "Menkar", "Menkent", "Menkib", "Merak", "Mercury", "Mesarthim", 
      "Messier object", "Methone", "Metis", "Miaplacidus", "Mimas", "Mimosa", 
      "Mintaka", "Mira", "Mirach", "Miranda", "Mirphak", "Mirzam", "Mizar", 
      "moon", "Mothallah", "multiple star", "Mundilfari", "Muphrid", 
      "Muscida", "Naiad", "naked eye star", "Naos", "Narvi", "Nashira", 
      "Nasl", "Navi", "nearest star", "nebula", "Nekkar", "Neptune", 
      "Neptune moon", "Nereid", "Nihal", "Nix", "normal giant", "Nunki", 
      "Nusakan", "Oberon", "object type", "oblateness", "obliquity", 
      "open cluster", "Ophelia", "orbit center", "orbit path", "orbit period",
       "orbit period years", "orbit rules", "Orthosie", "Paaliaq", "Pallene", 
      "Pan", "Pandora", "parallax", "Pasiphae", "Pasithee", "Peacock", 
      "Perdita", "periapsis", "periapsis argument", "periapsis longitude", 
      "Phact", "Phad", "Pherkad", "Phobos", "Phoebe", "planet", 
      "planetary moon", "planetary nebula", "planet radius", "Pluto", 
      "Pluto moon", "polar diameter", "Polaris", "polar radius", "Pollux", 
      "Polydeuces", "Porrima", "Portia", "Praxidike", "Procyon", "Prometheus",
       "Propus", "Prospero", "Proteus", "Puck", "radial velocity", 
      "radion astronomy", "Rana", "Rasalas", "Rasalgethi", "Rasalhague", 
      "red shift", "Regor", "Regulus", "Rhea", "Rigel", "Rigel Kentaurus A", 
      "Rigel Kentaurus B", "right ascension", "Rosalind", "Rotanev", 
      "rotation period", "rotation period years", "Rukbah", "Rukbat", "Sabik",
       "Sadachbia", "Sadalbari", "Sadalmelik", "Sadalsuud", "Sadr", "Saiph", 
      "Sarin", "satellites", "Saturn", "Saturn moon", "Scheat", 
      "semimajor axis", "Setebos", "Shaula", "Shedir", "Shelyak", "Sheratan", 
      "Siarnaq", "Sinope", "Sirius", "Skat", "Skathi", "space science", 
      "spectral class", "Spica", "spiral galaxy", "Sponde", "star", 
      "star brightest", "star gazing", "star multiplicity", "star watching", 
      "stargazing", "starwatching", "Stephano", "subdwarf", "subgiant", 
      "Sulaphat", "supergiant", "surface brightness", "Suttungr", "Sycorax", 
      "Syrma", "Talitha Australis", "Tania Australis", "Tania Borealis", 
      "Tarazed", "Tarf", "Tarvos", "Taygete", "Tegmine", "Tejat", "Telesto", 
      "Tethys", "Thalassa", "Thebe", "Themisto", "Thrymr", "Thuban", "Thyone",
       "Titan", "Titania", "Trinculo", "Triton", "Tsih", "Tyl", "Umbriel", 
      "Unukalhai", "Uranus", "Uranus moon", "Vega", "Venus", "Vindemiatrix", 
      "visual band magnitude", "Wasat", "Wazn", "Wezen", "white dwarf star", 
      "Wolf-Rayet star", "Yed Posterior", "Yed Prior", "Yildun", "Ymir", 
      "Zaurac", "Zosma", "Zubenelgenubi", "Zubenelhakrabi", "Zubeneshamali"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "AstronomicalData[\"name\", \" property\"] gives the value of the \
specified property of the astronomical object with the specified name.", 
    "synonyms" -> {"astronomical data"}, "title" -> "AstronomicalData", 
    "type" -> "Symbol", "uri" -> "ref/AstronomicalData"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1455, 36, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2078, 63, 1581, 37, 70, "ObjectNameGrid"],
Cell[3662, 102, 655, 18, 70, "Usage",
 CellID->982511436],
Cell[4320, 122, 566, 16, 70, "ExampleStatus"],
Cell[4889, 140, 386, 10, 70, "ExampleStatus"],
Cell[5278, 152, 398, 10, 70, "ExampleStatus"],
Cell[5679, 164, 56, 1, 70, "ExampleStatus"],
Cell[5738, 167, 49, 1, 70, "ExampleStatus"]
}, Open  ]],
Cell[CellGroupData[{
Cell[5824, 173, 299, 8, 70, "MoreAboutSection"],
Cell[6126, 183, 142, 3, 70, "MoreAbout",
 CellID->1],
Cell[6271, 188, 185, 3, 70, "MoreAbout",
 CellID->236061720],
Cell[6459, 193, 448, 12, 70, "Usage",
 CellID->1000335954],
Cell[6910, 207, 277, 8, 70, "Notes",
 CellID->297388943],
Cell[7190, 217, 440, 13, 70, "Notes",
 CellID->111831564],
Cell[7633, 232, 438, 13, 70, "Notes",
 CellID->295330312],
Cell[8074, 247, 255, 9, 70, "Notes",
 CellID->911156272],
Cell[8332, 258, 280, 9, 70, "Notes",
 CellID->3895384],
Cell[8615, 269, 304, 9, 70, "Notes",
 CellID->42962864],
Cell[8922, 280, 310, 9, 70, "Notes",
 CellID->118722301],
Cell[9235, 291, 87, 1, 70, "Notes",
 CellID->155343494],
Cell[9325, 294, 499, 11, 70, "2ColumnTableMod",
 CellID->595272836],
Cell[9827, 307, 72, 1, 70, "Notes",
 CellID->489250657],
Cell[9902, 310, 1106, 20, 70, "2ColumnTableMod",
 CellID->232238307],
Cell[11011, 332, 663, 21, 70, "Notes",
 CellID->339939503],
Cell[11677, 355, 528, 17, 70, "Notes",
 CellID->140000587],
Cell[12208, 374, 83, 1, 70, "Notes",
 CellID->145743071],
Cell[12294, 377, 1032, 23, 70, "2ColumnTableMod",
 CellID->403572293],
Cell[13329, 402, 361, 11, 70, "Notes",
 CellID->630565069],
Cell[13693, 415, 277, 10, 70, "Notes",
 CellID->744135164],
Cell[13973, 427, 696, 24, 70, "Notes",
 CellID->482514198],
Cell[14672, 453, 423, 14, 70, "Notes",
 CellID->282953651],
Cell[15098, 469, 587, 19, 70, "Notes",
 CellID->364699022],
Cell[15688, 490, 114, 3, 70, "Notes",
 CellID->101627253],
Cell[15805, 495, 986, 23, 70, "2ColumnTableMod",
 CellID->135414431],
Cell[16794, 520, 113, 3, 70, "Notes",
 CellID->200237957],
Cell[16910, 525, 106, 3, 70, "Notes",
 CellID->299816302],
Cell[17019, 530, 77, 1, 70, "Notes",
 CellID->155438026],
Cell[17099, 533, 1818, 41, 70, "2ColumnTableMod",
 CellID->476849875],
Cell[18920, 576, 77, 1, 70, "Notes",
 CellID->325949067],
Cell[19000, 579, 695, 14, 70, "2ColumnTableMod",
 CellID->548947657],
Cell[19698, 595, 90, 1, 70, "Notes",
 CellID->768863709],
Cell[19791, 598, 1037, 25, 70, "2ColumnTableMod",
 CellID->252134634],
Cell[20831, 625, 111, 3, 70, "Notes",
 CellID->46333823],
Cell[20945, 630, 2681, 57, 70, "2ColumnTableMod",
 CellID->45857499],
Cell[23629, 689, 161, 4, 70, "Notes",
 CellID->28659941],
Cell[23793, 695, 78, 1, 70, "Notes",
 CellID->588660618],
Cell[23874, 698, 1292, 32, 70, "2ColumnTableMod",
 CellID->665737471],
Cell[25169, 732, 88, 1, 70, "Notes",
 CellID->87606451],
Cell[25260, 735, 2165, 55, 70, "2ColumnTableMod",
 CellID->162684920],
Cell[27428, 792, 70, 1, 70, "Notes",
 CellID->454439520],
Cell[27501, 795, 1306, 31, 70, "2ColumnTableMod",
 CellID->707496748],
Cell[28810, 828, 75, 1, 70, "Notes",
 CellID->374541617],
Cell[28888, 831, 2008, 40, 70, "2ColumnTableMod",
 CellID->109938254],
Cell[30899, 873, 67, 1, 70, "Notes",
 CellID->106210209],
Cell[30969, 876, 509, 11, 70, "2ColumnTableMod",
 CellID->496402173],
Cell[31481, 889, 490, 14, 70, "Notes",
 CellID->276644756],
Cell[31974, 905, 1332, 29, 70, "2ColumnTableMod",
 CellID->524183967],
Cell[33309, 936, 299, 9, 70, "Tutorials",
 CellID->341631938],
Cell[33611, 947, 246, 5, 70, "RelatedLinks",
 CellID->1038487239],
Cell[33860, 954, 74, 1, 70, "ExampleText",
 CellID->677248009],
Cell[CellGroupData[{
Cell[33959, 959, 125, 3, 70, "Input",
 CellID->248531990],
Cell[34087, 964, 401, 10, 36, "Output",
 CellID->1025003446]
}, Open  ]],
Cell[34503, 977, 124, 3, 70, "ExampleDelimiter",
 CellID->83773203],
Cell[34630, 982, 84, 1, 70, "ExampleText",
 CellID->6724361],
Cell[CellGroupData[{
Cell[34739, 987, 295, 9, 28, "Input",
 CellID->355551284],
Cell[35037, 998, 3165, 56, 158, "Output",
 Evaluatable->False,
 CellID->574005652]
}, Open  ]],
Cell[38217, 1057, 125, 3, 70, "ExampleDelimiter",
 CellID->467797123],
Cell[38345, 1062, 75, 1, 70, "ExampleText",
 CellID->496580757],
Cell[CellGroupData[{
Cell[38445, 1067, 166, 4, 70, "Input",
 CellID->326637340],
Cell[38614, 1073, 485, 11, 54, "Output",
 CellID->202270463]
}, Open  ]],
Cell[39114, 1087, 74, 1, 70, "ExampleText",
 CellID->125102566],
Cell[CellGroupData[{
Cell[39213, 1092, 171, 5, 70, "Input",
 CellID->56775408],
Cell[39387, 1099, 181, 5, 36, "Output",
 CellID->238403244]
}, Open  ]],
Cell[CellGroupData[{
Cell[39605, 1109, 198, 6, 70, "Input",
 CellID->229837987],
Cell[39806, 1117, 181, 5, 36, "Output",
 CellID->169146414]
}, Open  ]],
Cell[40002, 1125, 125, 3, 70, "ExampleDelimiter",
 CellID->123085960],
Cell[40130, 1130, 64, 1, 70, "ExampleText",
 CellID->657588733],
Cell[CellGroupData[{
Cell[40219, 1135, 131, 3, 70, "Input",
 CellID->97706448],
Cell[40353, 1140, 479, 11, 54, "Output",
 CellID->366015654]
}, Open  ]],
Cell[40847, 1154, 68, 1, 70, "ExampleText",
 CellID->250695742],
Cell[CellGroupData[{
Cell[40940, 1159, 208, 7, 70, "Input",
 CellID->233909923],
Cell[41151, 1168, 620, 15, 36, "Output",
 CellID->819679723]
}, Open  ]],
Cell[41786, 1186, 70, 1, 70, "ExampleText",
 CellID->264471952],
Cell[CellGroupData[{
Cell[41881, 1191, 204, 7, 70, "Input",
 CellID->320241788],
Cell[42088, 1200, 406, 10, 36, "Output",
 CellID->373707133]
}, Open  ]],
Cell[42509, 1213, 124, 3, 70, "ExampleDelimiter",
 CellID->35475128],
Cell[42636, 1218, 115, 3, 70, "ExampleText",
 CellID->209857436],
Cell[CellGroupData[{
Cell[42776, 1225, 192, 6, 70, "Input",
 CellID->953265633],
Cell[42971, 1233, 480, 11, 36, "Output"]
}, Open  ]],
Cell[43466, 1247, 125, 3, 70, "ExampleDelimiter",
 CellID->556251086],
Cell[43594, 1252, 91, 1, 70, "ExampleText",
 CellID->223366887],
Cell[CellGroupData[{
Cell[43710, 1257, 184, 5, 70, "Input",
 CellID->80677821],
Cell[43897, 1264, 180, 5, 36, "Output",
 CellID->258568371]
}, Open  ]],
Cell[44092, 1272, 125, 3, 70, "ExampleDelimiter",
 CellID->399445155],
Cell[44220, 1277, 95, 1, 70, "ExampleText",
 CellID->39714291],
Cell[CellGroupData[{
Cell[44340, 1282, 123, 3, 70, "Input",
 CellID->474826791],
Cell[44466, 1287, 257, 8, 36, "Output",
 CellID->6893128]
}, Open  ]],
Cell[44738, 1298, 75, 1, 70, "ExampleText",
 CellID->572355833],
Cell[CellGroupData[{
Cell[44838, 1303, 199, 7, 70, "Input",
 CellID->37495760],
Cell[45040, 1312, 259, 8, 36, "Output",
 CellID->221476779]
}, Open  ]],
Cell[45314, 1323, 85, 1, 70, "ExampleText",
 CellID->114341380],
Cell[CellGroupData[{
Cell[45424, 1328, 125, 3, 70, "Input",
 CellID->670692101],
Cell[45552, 1333, 400, 10, 36, "Output",
 CellID->132466339]
}, Open  ]],
Cell[45967, 1346, 125, 3, 70, "ExampleDelimiter",
 CellID->121895515],
Cell[46095, 1351, 80, 1, 70, "ExampleText",
 CellID->508605741],
Cell[CellGroupData[{
Cell[46200, 1356, 255, 7, 70, "Input",
 CellID->393612353],
Cell[46458, 1365, 487, 11, 54, "Output"]
}, Open  ]],
Cell[46960, 1379, 124, 3, 70, "ExampleDelimiter",
 CellID->89516562],
Cell[47087, 1384, 75, 1, 70, "ExampleText",
 CellID->3614946],
Cell[CellGroupData[{
Cell[47187, 1389, 259, 7, 70, "Input",
 CellID->204278449],
Cell[47449, 1398, 513, 12, 36, "Output",
 CellID->487259273]
}, Open  ]],
Cell[47977, 1413, 125, 3, 70, "ExampleDelimiter",
 CellID->446339696],
Cell[48105, 1418, 83, 1, 70, "ExampleText",
 CellID->288556773],
Cell[CellGroupData[{
Cell[48213, 1423, 163, 4, 70, "Input",
 CellID->17049110],
Cell[48379, 1429, 961, 22, 92, "Output",
 CellID->681784613]
}, Open  ]],
Cell[49355, 1454, 125, 3, 70, "ExampleDelimiter",
 CellID->478329810],
Cell[49483, 1459, 68, 1, 70, "ExampleText",
 CellID->417229875],
Cell[CellGroupData[{
Cell[49576, 1464, 252, 7, 70, "Input",
 CellID->416112851],
Cell[49831, 1473, 496, 12, 36, "Output",
 CellID->227238577]
}, Open  ]],
Cell[50342, 1488, 125, 3, 70, "ExampleDelimiter",
 CellID->669453872],
Cell[50470, 1493, 65, 1, 70, "ExampleText",
 CellID->591475599],
Cell[CellGroupData[{
Cell[50560, 1498, 248, 7, 70, "Input",
 CellID->14559333],
Cell[50811, 1507, 470, 12, 36, "Output",
 CellID->790543063]
}, Open  ]],
Cell[51296, 1522, 125, 3, 70, "ExampleDelimiter",
 CellID->605287824],
Cell[51424, 1527, 63, 1, 70, "ExampleText",
 CellID->316398652],
Cell[CellGroupData[{
Cell[51512, 1532, 160, 4, 70, "Input",
 CellID->10741422],
Cell[51675, 1538, 23154, 385, 141, "Output",
 CellID->349946031]
}, Open  ]],
Cell[74844, 1926, 125, 3, 70, "ExampleDelimiter",
 CellID->405493821],
Cell[74972, 1931, 73, 1, 70, "ExampleText",
 CellID->525227218],
Cell[CellGroupData[{
Cell[75070, 1936, 162, 4, 70, "Input",
 CellID->9278646],
Cell[75235, 1942, 252, 8, 36, "Output",
 CellID->105924377]
}, Open  ]],
Cell[75502, 1953, 80, 1, 70, "ExampleText",
 CellID->133182419],
Cell[CellGroupData[{
Cell[75607, 1958, 198, 7, 70, "Input",
 CellID->358890307],
Cell[75808, 1967, 293, 9, 39, "Output",
 CellID->328989165]
}, Open  ]],
Cell[76116, 1979, 74, 1, 70, "ExampleText",
 CellID->770440698],
Cell[CellGroupData[{
Cell[76215, 1984, 202, 7, 70, "Input",
 CellID->5648616],
Cell[76420, 1993, 298, 9, 39, "Output",
 CellID->263279802]
}, Open  ]],
Cell[76733, 2005, 125, 3, 70, "ExampleDelimiter",
 CellID->314540477],
Cell[76861, 2010, 84, 1, 70, "ExampleText",
 CellID->5759648],
Cell[CellGroupData[{
Cell[76970, 2015, 289, 8, 70, "Input",
 CellID->265543398],
Cell[77262, 2025, 283, 9, 36, "Output",
 CellID->1031151]
}, Open  ]],
Cell[77560, 2037, 84, 1, 70, "ExampleText",
 CellID->52084224],
Cell[CellGroupData[{
Cell[77669, 2042, 181, 6, 70, "Input",
 CellID->17426213],
Cell[77853, 2050, 564, 12, 54, "Output",
 CellID->117440584]
}, Open  ]],
Cell[78432, 2065, 123, 3, 70, "ExampleDelimiter",
 CellID->4804123],
Cell[78558, 2070, 88, 1, 70, "ExampleText",
 CellID->16164372],
Cell[CellGroupData[{
Cell[78671, 2075, 268, 8, 70, "Input",
 CellID->527578803],
Cell[78942, 2085, 473, 12, 54, "Output",
 CellID->506680743]
}, Open  ]],
Cell[79430, 2100, 124, 3, 70, "ExampleDelimiter",
 CellID->29459054],
Cell[79557, 2105, 88, 1, 70, "ExampleText",
 CellID->111196055],
Cell[CellGroupData[{
Cell[79670, 2110, 175, 5, 70, "Input",
 CellID->81860064],
Cell[79848, 2117, 185, 5, 36, "Output",
 CellID->550676796]
}, Open  ]],
Cell[80048, 2125, 62, 1, 70, "ExampleText",
 CellID->117533139],
Cell[CellGroupData[{
Cell[80135, 2130, 163, 4, 70, "Input",
 CellID->600949417],
Cell[80301, 2136, 190, 5, 36, "Output",
 CellID->93944194]
}, Open  ]],
Cell[80506, 2144, 124, 3, 70, "ExampleDelimiter",
 CellID->79964759],
Cell[80633, 2149, 60, 1, 70, "ExampleText",
 CellID->3743066],
Cell[CellGroupData[{
Cell[80718, 2154, 321, 9, 70, "Input",
 CellID->301777630],
Cell[81042, 2165, 270, 8, 36, "Output",
 CellID->50202262]
}, Open  ]],
Cell[81327, 2176, 125, 3, 70, "ExampleDelimiter",
 CellID->163343868],
Cell[81455, 2181, 70, 1, 70, "ExampleText",
 CellID->336797544],
Cell[CellGroupData[{
Cell[81550, 2186, 165, 4, 70, "Input",
 CellID->431792553],
Cell[81718, 2192, 168, 5, 36, "Output",
 CellID->830257140]
}, Open  ]],
Cell[81901, 2200, 82, 1, 70, "ExampleText",
 CellID->43945354],
Cell[CellGroupData[{
Cell[82008, 2205, 169, 4, 70, "Input",
 CellID->284175261],
Cell[82180, 2211, 220, 7, 36, "Output",
 CellID->14326516]
}, Open  ]],
Cell[82415, 2221, 125, 3, 70, "ExampleDelimiter",
 CellID->577251838],
Cell[82543, 2226, 67, 1, 70, "ExampleText",
 CellID->245821],
Cell[CellGroupData[{
Cell[82635, 2231, 169, 4, 70, "Input",
 CellID->741704346],
Cell[82807, 2237, 538, 12, 54, "Output",
 CellID->15152276]
}, Open  ]],
Cell[83360, 2252, 124, 3, 70, "ExampleDelimiter",
 CellID->45066016],
Cell[83487, 2257, 91, 1, 70, "ExampleText",
 CellID->196045199],
Cell[CellGroupData[{
Cell[83603, 2262, 163, 4, 70, "Input",
 CellID->819995344],
Cell[83769, 2268, 162, 5, 36, "Output",
 CellID->826663926]
}, Open  ]],
Cell[83946, 2276, 125, 3, 70, "ExampleDelimiter",
 CellID->664978631],
Cell[84074, 2281, 94, 1, 70, "ExampleText",
 CellID->157697162],
Cell[CellGroupData[{
Cell[84193, 2286, 132, 3, 70, "Input",
 CellID->188457272],
Cell[84328, 2291, 284, 8, 36, "Output",
 CellID->511166242]
}, Open  ]],
Cell[84627, 2302, 125, 3, 70, "ExampleDelimiter",
 CellID->404283081],
Cell[84755, 2307, 94, 1, 70, "ExampleText",
 CellID->494547118],
Cell[CellGroupData[{
Cell[84874, 2312, 160, 4, 70, "Input",
 CellID->508142260],
Cell[85037, 2318, 176, 5, 36, "Output",
 CellID->258151589]
}, Open  ]],
Cell[85228, 2326, 85, 1, 70, "ExampleText",
 CellID->678291554],
Cell[CellGroupData[{
Cell[85338, 2331, 175, 5, 70, "Input",
 CellID->658834346],
Cell[85516, 2338, 171, 5, 36, "Output",
 CellID->3891353]
}, Open  ]],
Cell[85702, 2346, 124, 3, 70, "ExampleDelimiter",
 CellID->49714092],
Cell[85829, 2351, 87, 1, 70, "ExampleText",
 CellID->116384529],
Cell[CellGroupData[{
Cell[85941, 2356, 188, 6, 70, "Input",
 CellID->401681999],
Cell[86132, 2364, 431, 10, 36, "Output",
 CellID->561389168]
}, Open  ]],
Cell[86578, 2377, 74, 1, 70, "ExampleText",
 CellID->49258605],
Cell[CellGroupData[{
Cell[86677, 2382, 164, 5, 70, "Input",
 CellID->11571315],
Cell[86844, 2389, 160, 5, 36, "Output",
 CellID->139098614]
}, Open  ]],
Cell[87019, 2397, 124, 3, 70, "ExampleDelimiter",
 CellID->10718750],
Cell[87146, 2402, 70, 1, 70, "ExampleText",
 CellID->647167067],
Cell[CellGroupData[{
Cell[87241, 2407, 188, 6, 70, "Input",
 CellID->119625805],
Cell[87432, 2415, 457, 11, 54, "Output",
 CellID->464276984]
}, Open  ]],
Cell[87904, 2429, 125, 3, 70, "ExampleDelimiter",
 CellID->283013227],
Cell[88032, 2434, 79, 1, 70, "ExampleText",
 CellID->609171371],
Cell[CellGroupData[{
Cell[88136, 2439, 191, 6, 70, "Input",
 CellID->84831946],
Cell[88330, 2447, 447, 10, 54, "Output",
 CellID->508255469]
}, Open  ]],
Cell[88792, 2460, 72, 1, 70, "ExampleText",
 CellID->249443859],
Cell[CellGroupData[{
Cell[88889, 2465, 192, 6, 70, "Input",
 CellID->148780319],
Cell[89084, 2473, 394, 10, 36, "Output",
 CellID->714709956]
}, Open  ]],
Cell[89493, 2486, 56, 1, 70, "ExampleText",
 CellID->709914909],
Cell[CellGroupData[{
Cell[89574, 2491, 200, 7, 70, "Input",
 CellID->295312528],
Cell[89777, 2500, 483, 12, 54, "Output",
 CellID->368713991]
}, Open  ]],
Cell[90275, 2515, 125, 3, 70, "ExampleDelimiter",
 CellID->256339554],
Cell[90403, 2520, 77, 1, 70, "ExampleText",
 CellID->64125037],
Cell[CellGroupData[{
Cell[90505, 2525, 193, 6, 70, "Input",
 CellID->34994936],
Cell[90701, 2533, 428, 10, 36, "Output",
 CellID->552840004]
}, Open  ]],
Cell[91144, 2546, 56, 1, 70, "ExampleText",
 CellID->175933697],
Cell[CellGroupData[{
Cell[91225, 2551, 200, 7, 70, "Input",
 CellID->49098515],
Cell[91428, 2560, 772, 17, 54, "Output",
 CellID->632458544]
}, Open  ]],
Cell[92215, 2580, 124, 3, 70, "ExampleDelimiter",
 CellID->23696552],
Cell[92342, 2585, 65, 1, 70, "ExampleText",
 CellID->366947188],
Cell[CellGroupData[{
Cell[92432, 2590, 184, 6, 70, "Input",
 CellID->640742278],
Cell[92619, 2598, 439, 10, 54, "Output",
 CellID->43478866]
}, Open  ]],
Cell[93073, 2611, 68, 1, 70, "ExampleText",
 CellID->255319853],
Cell[CellGroupData[{
Cell[93166, 2616, 208, 7, 70, "Input",
 CellID->357775124],
Cell[93377, 2625, 620, 14, 58, "Output",
 CellID->99743767]
}, Open  ]],
Cell[94012, 2642, 125, 3, 70, "ExampleDelimiter",
 CellID->257205091],
Cell[94140, 2647, 85, 1, 70, "ExampleText",
 CellID->529776674],
Cell[CellGroupData[{
Cell[94250, 2652, 314, 10, 70, "Input",
 CellID->13833634],
Cell[94567, 2664, 450, 10, 54, "Output",
 CellID->529186187]
}, Open  ]],
Cell[95032, 2677, 55, 1, 70, "ExampleText",
 CellID->21372812],
Cell[CellGroupData[{
Cell[95112, 2682, 314, 10, 70, "Input",
 CellID->49139012],
Cell[95429, 2694, 392, 10, 36, "Output",
 CellID->59411446]
}, Open  ]],
Cell[95836, 2707, 124, 3, 70, "ExampleDelimiter",
 CellID->32640975],
Cell[95963, 2712, 92, 1, 70, "ExampleText",
 CellID->297992517],
Cell[CellGroupData[{
Cell[96080, 2717, 322, 10, 70, "Input",
 CellID->566279940],
Cell[96405, 2729, 508, 12, 36, "Output",
 CellID->307781515]
}, Open  ]],
Cell[96928, 2744, 125, 3, 70, "ExampleDelimiter",
 CellID->152923281],
Cell[97056, 2749, 71, 1, 70, "ExampleText",
 CellID->793447851],
Cell[CellGroupData[{
Cell[97152, 2754, 169, 4, 70, "Input",
 CellID->720248603],
Cell[97324, 2760, 361, 10, 36, "Output",
 CellID->514951502]
}, Open  ]],
Cell[97700, 2773, 137, 6, 70, "ExampleText",
 CellID->156881406],
Cell[CellGroupData[{
Cell[97862, 2783, 180, 5, 70, "Input",
 CellID->202518144],
Cell[98045, 2790, 312, 8, 36, "Output",
 CellID->423250161]
}, Open  ]],
Cell[98372, 2801, 125, 3, 70, "ExampleDelimiter",
 CellID->212969740],
Cell[98500, 2806, 85, 1, 70, "ExampleText",
 CellID->4597139],
Cell[CellGroupData[{
Cell[98610, 2811, 164, 4, 70, "Input",
 CellID->272378806],
Cell[98777, 2817, 160, 5, 36, "Output",
 CellID->71056133]
}, Open  ]],
Cell[98952, 2825, 80, 1, 70, "ExampleText",
 CellID->310272575],
Cell[CellGroupData[{
Cell[99057, 2830, 598, 17, 70, "Input",
 CellID->382770234],
Cell[99658, 2849, 1392, 32, 212, "Output",
 CellID->4669868]
}, Open  ]],
Cell[101065, 2884, 124, 3, 70, "ExampleDelimiter",
 CellID->23684725],
Cell[101192, 2889, 146, 4, 70, "ExampleText",
 CellID->987398988],
Cell[CellGroupData[{
Cell[101363, 2897, 455, 14, 65, "Input",
 CellID->416203837],
Cell[101821, 2913, 1411, 27, 129, "Output",
 Evaluatable->False,
 CellID->47458132]
}, Open  ]],
Cell[103247, 2943, 134, 4, 70, "ExampleText",
 CellID->62958908],
Cell[CellGroupData[{
Cell[103406, 2951, 463, 14, 65, "Input",
 CellID->574002368],
Cell[103872, 2967, 2124, 39, 139, "Output",
 Evaluatable->False,
 CellID->704475272]
}, Open  ]],
Cell[106011, 3009, 125, 3, 70, "ExampleDelimiter",
 CellID->287381801],
Cell[106139, 3014, 118, 3, 70, "ExampleText",
 CellID->235366638],
Cell[CellGroupData[{
Cell[106282, 3021, 168, 5, 70, "Input",
 CellID->474882182],
Cell[106453, 3028, 159, 5, 36, "Output",
 CellID->20028348]
}, Open  ]],
Cell[106627, 3036, 123, 3, 70, "ExampleText",
 CellID->636489389],
Cell[CellGroupData[{
Cell[106775, 3043, 536, 15, 70, "Input",
 CellID->701856818],
Cell[107314, 3060, 42090, 1013, 70, "Output",
 CellID->168999918]
}, Open  ]],
Cell[149419, 4076, 139, 4, 70, "ExampleText",
 CellID->559093282],
Cell[CellGroupData[{
Cell[149583, 4084, 617, 17, 70, "Input",
 CellID->210484892],
Cell[150203, 4103, 17603, 545, 70, "Output",
 CellID->26389875]
}, Open  ]],
Cell[167821, 4651, 95, 1, 70, "ExampleText",
 CellID->405468550],
Cell[CellGroupData[{
Cell[167941, 4656, 614, 17, 70, "Input",
 CellID->68133414],
Cell[168558, 4675, 17811, 544, 70, "Output",
 CellID->730756979]
}, Open  ]],
Cell[186384, 5222, 125, 3, 70, "ExampleDelimiter",
 CellID->510844526],
Cell[186512, 5227, 103, 2, 70, "ExampleText",
 CellID->155558654],
Cell[CellGroupData[{
Cell[186640, 5233, 346, 10, 47, "Input",
 CellID->294669162],
Cell[186989, 5245, 1827, 34, 124, "Output",
 Evaluatable->False,
 CellID->327665559]
}, Open  ]],
Cell[188831, 5282, 104, 2, 70, "ExampleText",
 CellID->81977173],
Cell[CellGroupData[{
Cell[188960, 5288, 348, 10, 70, "Input",
 CellID->21510336],
Cell[189311, 5300, 1921, 40, 129, "Output",
 CellID->241423021]
}, Open  ]],
Cell[191247, 5343, 125, 3, 70, "ExampleDelimiter",
 CellID->198389329],
Cell[191375, 5348, 118, 3, 70, "ExampleText",
 CellID->713178108],
Cell[CellGroupData[{
Cell[191518, 5355, 349, 11, 70, "Input",
 CellID->876371901],
Cell[191870, 5368, 1306, 42, 54, "Output",
 CellID->28604921]
}, Open  ]],
Cell[193191, 5413, 82, 1, 70, "ExampleText",
 CellID->305950212],
Cell[CellGroupData[{
Cell[193298, 5418, 347, 11, 70, "Input",
 CellID->389325059],
Cell[193648, 5431, 642, 21, 36, "Output",
 CellID->767134692]
}, Open  ]],
Cell[194305, 5455, 125, 3, 70, "ExampleDelimiter",
 CellID->750604839],
Cell[194433, 5460, 95, 1, 70, "ExampleText",
 CellID->273686367],
Cell[CellGroupData[{
Cell[194553, 5465, 310, 9, 47, "Input",
 CellID->59074871],
Cell[194866, 5476, 1713, 32, 137, "Output",
 Evaluatable->False,
 CellID->138569807]
}, Open  ]],
Cell[196594, 5511, 84, 1, 70, "ExampleText",
 CellID->597222975],
Cell[CellGroupData[{
Cell[196703, 5516, 354, 10, 70, "Input",
 CellID->206411732],
Cell[197060, 5528, 1596, 35, 137, "Output",
 CellID->408235679]
}, Open  ]],
Cell[198671, 5566, 125, 3, 70, "ExampleDelimiter",
 CellID->702308210],
Cell[198799, 5571, 258, 8, 70, "ExampleText",
 CellID->55515327],
Cell[CellGroupData[{
Cell[199082, 5583, 451, 14, 70, "Input",
 CellID->94408],
Cell[199536, 5599, 941, 27, 54, "Output",
 CellID->208844345]
}, Open  ]],
Cell[200492, 5629, 125, 3, 70, "ExampleDelimiter",
 CellID->316093768],
Cell[200620, 5634, 98, 1, 70, "ExampleText",
 CellID->515222443],
Cell[200721, 5637, 405, 11, 70, "Input",
 CellID->642429306],
Cell[CellGroupData[{
Cell[201151, 5652, 376, 12, 70, "Input",
 CellID->363173261],
Cell[201530, 5666, 806, 19, 136, "Output",
 CellID->4326025]
}, Open  ]],
Cell[202351, 5688, 102, 2, 70, "ExampleText",
 CellID->284740032],
Cell[CellGroupData[{
Cell[202478, 5694, 394, 12, 70, "Input",
 CellID->29185774],
Cell[202875, 5708, 2324, 47, 147, "Output",
 CellID->61473859]
}, Open  ]],
Cell[205214, 5758, 125, 3, 70, "ExampleDelimiter",
 CellID->322191060],
Cell[205342, 5763, 120, 3, 70, "ExampleText",
 CellID->54932935],
Cell[CellGroupData[{
Cell[205487, 5770, 294, 8, 70, "Input",
 CellID->168714980],
Cell[205784, 5780, 160, 5, 36, "Output",
 CellID->244307830]
}, Open  ]],
Cell[205959, 5788, 97, 1, 70, "ExampleText",
 CellID->491774688],
Cell[CellGroupData[{
Cell[206081, 5793, 125, 3, 70, "Input",
 CellID->384592560],
Cell[206209, 5798, 400, 10, 36, "Output",
 CellID->365153391]
}, Open  ]],
Cell[206624, 5811, 85, 1, 70, "ExampleText",
 CellID->47977959],
Cell[CellGroupData[{
Cell[206734, 5816, 161, 4, 70, "Input",
 CellID->656229],
Cell[206898, 5822, 176, 5, 36, "Output",
 CellID->506311957]
}, Open  ]],
Cell[207089, 5830, 125, 3, 70, "ExampleDelimiter",
 CellID->390175453],
Cell[207217, 5835, 113, 3, 70, "ExampleText",
 CellID->9042815],
Cell[CellGroupData[{
Cell[207355, 5842, 256, 7, 70, "Input",
 CellID->204553130],
Cell[207614, 5851, 581, 14, 54, "Output",
 CellID->361378564]
}, Open  ]],
Cell[208210, 5868, 125, 3, 70, "ExampleDelimiter",
 CellID->154101200],
Cell[208338, 5873, 253, 7, 70, "ExampleText",
 CellID->303067630],
Cell[CellGroupData[{
Cell[208616, 5884, 596, 16, 65, "Input",
 CellID->573837874],
Cell[209215, 5902, 2933, 52, 139, "Output",
 Evaluatable->False,
 CellID->428608285]
}, Open  ]],
Cell[212163, 5957, 46, 1, 70, "Input",
 CellID->282925789],
Cell[212212, 5960, 117, 3, 70, "ExampleText",
 CellID->60751421],
Cell[CellGroupData[{
Cell[212354, 5967, 488, 14, 65, "Input",
 CellID->185632713],
Cell[212845, 5983, 1725, 32, 171, "Output",
 Evaluatable->False,
 CellID->434454952]
}, Open  ]]
}, Open  ]],
Cell[214597, 6019, 27, 0, 70, "History"],
Cell[214627, 6021, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

