(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    144278,       3159]
NotebookOptionsPosition[    136873,       2896]
NotebookOutlinePosition[    138173,       2933]
CellTagsIndexPosition[    138086,       2928]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"GammaDistribution" :> 
          Documentation`HelpLookup["paclet:ref/GammaDistribution"], 
          "BinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BinomialDistribution"], "Beta" :> 
          Documentation`HelpLookup["paclet:ref/Beta"], "BetaRegularized" :> 
          Documentation`HelpLookup["paclet:ref/BetaRegularized"], 
          "InverseBetaRegularized" :> 
          Documentation`HelpLookup["paclet:ref/InverseBetaRegularized"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"GammaDistribution\"\>", 
       2->"\<\"BinomialDistribution\"\>", 3->"\<\"Beta\"\>", 
       4->"\<\"BetaRegularized\"\>", 
       5->"\<\"InverseBetaRegularized\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"], 
          "Functions Used in Statistics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInStatistics"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>", 
       2->"\<\"Functions Used in Statistics\"\>", 
       3->"\<\"Statistical Distributions\"\>", 
       4->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["BetaDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["BetaDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/BetaDistribution"], "[", 
       RowBox[{
        StyleBox["\[Alpha]", "TR"], ",", 
        StyleBox["\[Beta]", "TR"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a continuous beta distribution with shape \
parameters ",
     Cell[BoxData[
      StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["\[Beta]", "TR"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->11382]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability density for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in a gamma distribution is proportional to ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{
     SuperscriptBox["x", 
      RowBox[{"\[Alpha]", "-", "1"}]], "(", 
     RowBox[{"1", "-", "x"}], ")"}], 
    RowBox[{"\[Beta]", "-", "1"}]], TraditionalForm]], "InlineMath"],
 "for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"0", "<", "x", "<", "1"}], TraditionalForm]], "InlineMath"],
 ", and is zero for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "<", "0"}], TraditionalForm]], "InlineMath"],
 " or ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", ">", "1"}], TraditionalForm]], "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->25023,
  ButtonNote->"25023"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " to be any positive real numbers."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->7997],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->10542,
  ButtonNote->"10542"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->156092415],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->146239326],

Cell["The mean and variance of a beta distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10542],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13063],

Cell[BoxData[
 FractionBox["\[Alpha]", 
  RowBox[{"\[Alpha]", "+", "\[Beta]"}]]], "Output",
 ImageSize->{35, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->410769644]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->30502],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[Alpha]", " ", "\[Beta]"}], 
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], "2"], " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{118, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->5576855]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9099],

Cell["\<\
The probability density function depends on the beta function:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->25023],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17226],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "x"}], ")"}], 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Beta]"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}]]}], 
  RowBox[{"Beta", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}]]], "Output",
 ImageSize->{100, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->511341111]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"BetaDistribution", "[", 
      RowBox[{"5", ",", "2"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->30048],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 149},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->125483395]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->8636],

Cell["\<\
Generate a set of pseudorandom numbers that are beta distributed:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->9458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"10", ",", "4"}], "]"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25967],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.4923201789770412`", ",", "0.959721380228064`", ",", 
   "0.8199620968312817`", ",", "0.8310199827066586`", ",", 
   "0.7323568556296658`"}], "}"}]], "Output",
 ImageSize->{329, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->389953974]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9405],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11531],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "\[Alpha]"}], "+", "\[Beta]"}], ")"}], " ", 
   SqrtBox[
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}]]}], 
  RowBox[{
   SqrtBox["\[Alpha]"], " ", 
   SqrtBox["\[Beta]"], " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{133, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->388216223]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7975],

Cell[BoxData[
 FractionBox[
  RowBox[{"3", " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"\[Alpha]", " ", "\[Beta]", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "6"}], "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], "+", 
     RowBox[{"2", " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], "2"]}]}], ")"}]}], 
  RowBox[{"\[Alpha]", " ", "\[Beta]", " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"3", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{253, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->69994131]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7531],

Cell["Third moment of a beta distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->31495],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "3"}], ",", 
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8044],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[Alpha]", " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]"}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{176, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->587282675]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26803],

Cell[TextData[{
 "The 0.75 quantile of a beta distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Alpha]", "=", "5"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Beta]", "=", "2"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"5", ",", "2"}], "]"}], ",", "0.75"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18915],

Cell[BoxData["0.8388370832096734`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->332474747]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25791],

Cell["\<\
Plot the cumulative distribution function of the random variable:\
\>", "ExampleText",
 CellID->29591],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"BetaDistribution", "[", 
      RowBox[{"1", ",", "2"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25867],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 147},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->299675346]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->161355561],

Cell[TextData[{
 "A contour plot as both ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " are varied:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10722],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"BetaDistribution", "[", 
      RowBox[{"1", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Beta]", ",", "0.1", ",", "1.9"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24271],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxkXQVUlc/TRmzFxFbs/JmIXWMiJnYXdnd3d7diYKJiBwbdXZduLt2XbuS7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   "], {{{}, 
     {RGBColor[0.33397672886243995`, 0.1279285534584573, 0.5864250111696041], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlktM3VUQxg+Xe+Xey594eTVpXCANao0RSE21XQBJl6YudIeSJjYxJtZH
glYt0LLQKLho3FDaQtQGxYSINamlNFZaTbs2qfGxoBtrgK7UhQKhrX4/5iMm
DGfO/OfMmcc3c27rwdeeeTWXUvpdVC26KWJfL+oTTRVSek5rp+iU+A6t3eZ7
tL5knUNa79ZIN0up2bJPLd93T0pnzQ+I5sQf0fqZdbjn82JKr+vsV9p/WYgV
3RNax0VN4kd8dtT3n7YPK7K/LDopfqEQ/HiKlW9jVSntqktpKPv//gu286Hu
PSD5pO1N2ua6zkwUIu6z9gn/m5Wcj8T36vvzOncwi/2Ez6M/a33uOuI9/CHz
5Kxf56aLYbsi3+4VPSJfh7S+XErpaRVjzHEXsFNO6U3RN5If0zogWhA/I91e
2bmdi1xxhvxPys5fkq9UheyEbT1Ym9JuyX+Q/l7ZOCB+RHYu5FNqU/1qyKHO
dkp+WrLz4u+IX5Odj7W+iN+u27Rz0paLWlLH/dJ7yhgZke57osv6dlTrYBZ5
f0LrLtm6pnOfyP6/xcj335LXK54myc9J/ofky5J3SN5UjHv6xD+bRTyLVeHz
w14fMN+tuH7R+UXtG4xnsN2hbzu0NqbwY851QdZgOb532X/8XjBOWt0TDV7Z
3y/6sxi+4meDbfGNXOddu0bfn7cvWzaIk3z12FbF8s2Yq8LGZk7t15Rq971i
n8+HPljrMLXYBhg8ZRwua10S7ZStu8WoJXXstC5+7vG+1T422r8W220XCZab
GCq477r9DVt7nFPyRL7owVHvR9yDC5Z36f73s/gOdpgL4OdYFt+Qj4p/si70
B7QeFh5mcnH/krFdlY/afqG7L0nnW9Hx6sgLGADL3fYdPzm37LPfyV6DMdbu
OlNv+nXWeGYW3nQ9wPmUsV1x3lpsE5/IDTgkx/TOLd+107Il69Bvtyz/Vf5e
KUccXa7tfs+WXvdP3hiiHj3uK3x9QTq3FesZ7Vd14VHZ0V+a17+L5fiGLraw
w2y74fn2j/Tflk4JbJfDD3x4V9h4S/tL1XH/Zp+RK517qDZ66rC+LxUj1lVq
L7rKHMuFDWrRLJ13iqFPfagJcmYWtospZs2+QpybEOVtj3xvuD8qVWEf+ZTn
JW/QrN8N9vjZ59xRT2YHNWWGMouIoWK79BT3VHxXi/kfxW83z/f7RJlIf2k4
F7pXjQfk2NpmnS3ZNutjY7ttMov3FqJWj8qv9iz8JR/MVGwzl+uro3fAN/N+
1ffO+O4biqutGL3A2/hYXbyLl8V/XQo8L4q/UgqMkM9rzik9xPmTnifXczFT
mP3HS4FHZupPtZEr6ooMfJLH3zyb1lTHc6XAGP3+cy56njeV+KZTxAru2INL
ans+xRsyWBcx05f0J9hgpUfBA7i4aJ57wDR37VZcj2cR02A5sFtyn/QZD5yb
91nezP66yCP4ApP9zuuE89lTjnlBHd4oxXwh12CUvFNrfEXG/pVS+MH5zPjY
cN03jC1mG781mHU7PE+ZpeR6xbPxTk28x+zpJ/qEPCxQuyzmBe/5kN90fu8s
e/4zU6gJMyRfjPdy3XN33bN7K+/U4rqx1eE6EBu1mPM8pn/A3Zp9A0f8bmt2
b/bbFjka9vkmx9PoeU7tx435FvfqmO9HTq3bPId5/z/wbwDe/2H/BsAu+ca3
M/aR8+Cu2dijj4htzLjbihvbi563/wF6DUgm
         "],
         VertexColors->None], 
        PolygonBox[{{2106, 2045, 488, 722}, {2113, 2052, 505, 726}, {2980, 
         2979, 1208, 1209}, {3162, 2183, 1431, 1430}, {2616, 2617, 904, 
         417}, {2303, 2191, 1435, 505}, {2289, 2187, 785, 488}, {2301, 2113, 
         726, 504}, {2142, 2089, 1346, 904}, {3165, 2192, 1436, 1434}, {2684, 
         2683, 9, 958}, {2120, 2062, 533, 738}, {3185, 2215, 1456, 1454}, {
         2421, 3452, 849, 682}, {2062, 3169, 1438, 533}, {3175, 2343, 1602, 
         1446}, {2321, 2120, 738, 532}, {2446, 2279, 1513, 719}, {2187, 2302, 
         504, 785}, {2678, 2180, 956, 955}, {3118, 3119, 548, 345}, {3126, 
         3127, 709, 667}, {2375, 2474, 764, 405}, {2137, 2085, 1337, 764}, {
         3451, 2420, 682, 1178}, {2679, 2980, 1209, 956}, {3495, 2375, 405, 
         1768}, {2206, 3174, 1444, 1445}, {2084, 3495, 1768, 152}, {2085, 
         2744, 1020, 1337}, {2496, 2497, 532, 63}, {3264, 3263, 21, 1511}, {
         2183, 2288, 487, 1431}, {2214, 3185, 1454, 1455}, {3199, 3198, 137, 
         1463}, {3543, 2623, 910, 2012}, {3163, 3164, 1430, 21}, {3170, 3117, 
         345, 1438}, {2745, 2616, 417, 1020}, {2278, 2854, 719, 476}, {3210, 
         3209, 152, 1471}, {2287, 2106, 722, 487}, {2191, 3165, 1434, 1435}, {
         3266, 3265, 297, 1208}, {2343, 3186, 1455, 1602}, {3379, 3378, 122, 
         1456}, {2089, 2751, 1025, 1346}, {2752, 2622, 2012, 1025}, {2179, 
         2677, 476, 297}, {3343, 2206, 1445, 1579}, {2279, 3262, 1511, 
         1513}, {3442, 2413, 667, 849}, {3196, 3197, 1463, 122}, {3293, 3292, 
         63, 1436}, {2685, 2686, 958, 10}, {3517, 3518, 1178, 910}, {3207, 
         3208, 1471, 137}, {2680, 2681, 955, 9}, {3029, 3344, 1579, 548}, {
         3174, 2207, 1446, 1444}},
         VertexColors->None]}]}, 
     {RGBColor[0.4150981865873199, 0.26897686037537194`, 0.7004510335088123], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmFtwVlcVx0++JN/5LictCYUwoy+OFCG02gRbG5gpafugzmiYxnamoQaF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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlktsVGUUx7+Z6Uzn3rkjFKSYuDK+kKLyCAglUoGNcYGxrgQBBR9RKGAL
iLwVFGii0RVCKhSw0IDoAiktWGhFKIKvoEYXlEhqCA9RJBF00eLvnz+Lkznn
nsd33mfumjW/el46hLARKAGul4awLA7hSCaEg3y4Ab0vCqEIfQh6fzGEvnwI
41IhlMGbnYSwGXpbNoStwM+FEG6Hno6tq8hvAF+DvWPodwH/ID8Ae4NyyAHf
IH8KGIwTPcBAeBsj41N4Yz36X6NzFdvH0W/AVjPf6uCNQmYx/rwmHd5qxN4i
8KV82wneDNxEthH6NPgkdH4FP4KNFvy7hv9t4K3ALuh12D8Nfzv+HMV2E/xp
vN3H70V0P8SHZskCm5C/jPwq+MuAlfDb4D+K/srE+FTe7ES2Ax/2wq/D5uP4
twfoBX8Ln6ci+2xi28ORmYD+TN6vx/409OvhvQPUw+tE5lNsvQl9CLwW/RHY
Wp0YfwybregeAL6F/k72kb8PqMz6zZPYfwD+MOzNvVXbf/mdnnGNv8LWUaAH
fhX6o6WP/Djwcnw6RDyVwHb4vchvAx8b2dfzGfu6HujI2mflZgXQmnWOXkd+
LTaHgp9NuVYL0d+Vc80U+3MF51Y5WIPsIHReUT8CK2L3pHqxA3onsk3AOWz9
QHxLYvewercd/nuRY5Dv3SnX8nO+3ZZxTVchPwT98ch3ZtybncCdGffoX+ju
w8ciendjoz/y/YDZ8F8AlhPbSPhTco7xLLplsXnqkVp4c/m2Gd8+yllXPaDa
y0YreFvBtboX+T+hr2jewH9PWXd50b0sG+X40wTdAz0DmYd4/0FgFm/PBjqQ
PwOdyTmGt6E/QH5Lzj2tWdRMaZY0k5vynmHNbiPwJPRgoBd/riC/H/29mgfo
O3ivHVtfANXk42lghnZB7NnRTGn2H07si3ZAN/gMIJu2T+qd8UB9xj20HHwp
UJHyzrmI/ins15S6B/t4/1W+1WCvGv78yDOq2WzEZlvkHCp3l1Pu9YOxe1M9
XxZ552jXTEHmb/gN6J/IeYcplirkL5Q6pongk4DVac/sSWK/gX55zjvqHvS7
1VOqP/RM9TIwOeuY1FtHkL+QdY+NAR8NzIGeK38S7wTtAuksBn8jdq9qJgbg
T1nRtdVObYlcA+VePX6Nt05g40f0L0FfT1xD1U4+PoWtamAD/Bb4S/B/Hvpb
096R8qUmci/KJ+V+ZexZUg2U64qCZ185V+3ez3t3qoazoJ9PnEv1nGbzfuQr
SzyjHbF3rHbru9CjwOuQr0p7Rvph62O+nQOfiI0v4e0oeBdcwt7QgneUdpN6
II/8DugzyI9CfiH4WHS2pN3TO+HXQo+ArsVGBe8Pi70LdSN+K/hm6Faop/dA
7y64V5RD1fIJbM7Juab9wPsXfbte1P4CH84bx4J3lPDDRfe+vikW3SzdKsV0
El+6EtdGN0u742reu107RLM7BPmXUp7hFuhU5NgGlprXEHn2JPMMursj95pu
lG5ZCd9GBN80+b4h8W1QDNr9QTs96xug3EXQk1POYRe56Mq7N9Tj6o2bkWdJ
PaJa7co7l6rZy9oNeedCOf4PH5t5o73EO+p79Qf6lSm/eYCPceTalpe6VuOA
BVnXbCS2Hkk8Cz8h05v3fwr9l1BM6cg5US5kU7fus7xvi25eB/ThxLtcM3Ye
3qLYt0E3SrPde2tXa8Y/wde1sf97aIepVzsT95p6dkHk/xD676CYdAvG5H1L
dBPWJb7puuW6aXPy3knaRYrpuPZvwbv6D+hC5BwoduX8F/AJsWdBO6ZcuyX2
rdEN/h/xLUo7
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4962196443121998, 0.4100251672922865, 0.8144770558480204], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuQl2UVx9/9Xd/39/ut3JTkok1/xJpcIkPHhGWApNtUgE5mCwXdSGzA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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmuQjmUYx+99D+t5nvd9h3XYxulDM8WEDhQTllCa8aEsTSUUHQcNcmho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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.5737613661287864, 0.5435370519035628, 0.9097253540741593], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMVdUVx8/MPffec849o8YHT4FKbNEaIRADthVa5SFNA0FjbSuOQD+0
aLCmFrAmDi0kQIGpjfpFi4gSUIdqwXcjQxupoBREHiNO0yCiKNryEHxULGj/
P//L8GHfvc/ea6+99nr819r3vJ/ecvUvmpMkqemnov63LUkyO0+Sx2pJ0lZP
kjlqN6dJskLrD2nu601JslL9iprnHq17/Jb2bhODQnPDNX5ddGP13bPJPOA1
XGt3xPiW5NT3jaLZp/4GtRaN5+r8pZLjOu0/mpoG+qmam6K2UOc9rL3jtN7V
ZBkeqVu+r5VJ8o8sSRZr3xVqrZrfLJrbtTZC9PdqPE6tof1jxDMpkqRJ7Rqt
rWskyfM6+4TWJ2l9vvbO1TjV3A7xPanxpmbTcvfuZtNy32PqO3TG+erbKt7b
zrz2Ti4sa6H1N5qtoz1qWchwmdbXqy0RzXTJsF97Vic+Z2Xo+SrxzYO+W/fq
WfH+cyTX2WrjqknyjGhzndtLtD9XnyKDxveKvo/GH2i8ouLxI9q7QXoap3PX
aH6zaPrWT9H2DZoO9b2/Gtc85o6TJePL2rtY8/drbqnaPI2f1f4BaiubbZtV
zfaBNXWf/RVPznuuYnmxJTJj10m6xyKtzaxaZ6wNkU4eVBut+aOi+0h0w9WP
0N5Lw3/a5Be/UVuttQ2aO03tCL6qNls8/iK9HamETjS+qe6zdzSZ9vSgH5/6
+0LRzJReOzL7/2Ld9z7x3ypen6c+e6poeojm74XPOBrzyMPdO+NeF4ffnJvY
p7fG/Bep5Yffm3XbD72xD7pe4asnw8d6ab5fxb5A66vvqzXfH5uGb1yv+d4x
P0T8BlY8h79DB80BnddV8f7nRPNszXF3tu74Wen976Y+izNZfyZoNonmf6X5
7pHMh2q213Wyz26tzUlMyx50e7n2XyTaWRrvbzY/ZNsm+jMyn4O9XqjbrgvU
357aRkt0ztOZfZjYbdP8ZO1vqL9Tbbdojmv9FdF92GRfQe/oDR2vDD0PS41F
YBJ4wDyYsEB266M2Q2tPFo494m6k1l6pWOf3a26Z2mOp/RxceUnz6+q+N+PH
U9sLuxGH82r2O85BF8iwtm7fPJh4TDwQT+As8+AHuHhH4ORx6XJeZrzFZ4nn
iZp/UfwX1o1Pc+SPJ3Pz7JIOdoNRWl+Sez970efC0Ol5Wp/W4jgFO8AQ/Iv1
BfVTmLcgYuQ+7RlfMW5wzn/jrJ7i8YrG0xLH0E0Rw8THoty6v6w0toJhozS+
OXNMd5eWFTn/KtqdpeV4Tf3f9D1C26aJtqe+mzT/seY+yi0HemUO3Q4SzWS1
DvxZtO+WxtdurRWRF3pInl5qwzT/a/XTM+cpfPGSZucUcs6UyDv7xeNttUz0
t4n+9pbIczXnFnCYOF4RsQxW3V2xD4Chd1WMMyMLYzp4vlE6fDG1T4G1Vwbe
jtc5V6q9GtjbERjL98PadzIxLlxcc/xiMzCWvbfVnM/IZUNrvh+2GCTevSVz
q+aXV5wTuS+YDX/wdnDNPMGuJ+TXvyuMe4u0r73Fccu+R2PvvqrlQa6HYm1w
k+99V9z9sPjsyB3nYDy+jZ+wb1Xc5Ub8U3THUt9naMi8PGjg+apkuTFshM4f
jZyO392amSc5ndiFJ/cYEvppDZ5gC/yWx7m9497kLGKXuEXmy8XzB7rvDO0Z
W7VO0edB9Z11Y8tZqXEFW/9Zsq9p2J59WuxbX/pVzXvxw6Fan687XKDxIvXL
RLNFZz+e26fxXWJ9bcT+NNF/K3dcP6VxUTi/ob+1ocMBkm1KzXXId8Xv20FP
vsEX8Dd4dgSGtIp+QmpMOz3yIDkNrObuRWrMXl/3XcHSzpgnxl5rdrx/Knk/
yX1/1tcHzcdRQy3VuR9UrD/GQ8MO6B++1AXP6ftN8dibG79n6n7zM+dD6os+
UW/M1fyCmOe8Y6Fz8gH6IBf/UjRzM9uR+8+pGKNu1fzuFtt4pNafbpy6A/3w
yIf7I++AzxPrtvmWwDZovtDcZZltd5fk3V66viPnHK4472CLMaLrX3Uttidy
Lvm5O+pebPVJs+2FntiHfqgJR0c9yX7GbYn7MTEPXkyoG9+ukSwnGuaVBvZz
X/wFP+B7MVieO4evII5Ff6Bim6M79EbNBMYRN+RJ7IqOn9B4ben4PldzV2WW
Z1fp+3OfidrzWs3+ROuq2a/omR8lPrtq/h4VNTx1DDkWOtbQJ/i3K2i438S4
I3lmfuQd7MEacXRSMvTIXdOAMUsjZqmXRtdcX1Ebrop8X0TuZYxftIVvfKMw
JoIN50hXr0s/H6aOlTWBt9Q75+T2kRXqh+WuWT7X/Bdq/UV7Qv29uX3ouHh8
2rAv3Kn535fOyV/as2Id7tf6BS3O49wJu3ZEnQY/7kUMHo7aaX3UUcQXbwPe
CORl6N6J2p94Iq7AJWgZE1/wOBTj9wvnlS9zSpN7MJC6ZVnULhfqHoNarMc+
UZMTA2/L/ieLU+828ih1O/vuj9p+f9W6Zm1pzGOXJ3Xmnsz13qCI6S1hf2yP
D+DD1N/tkUfAe+qTP4LBmd8NW8TnI42PVV1/bYsa7NWKv6kNNlaNO/DqFP26
0nJhm54115cDpP/+Ddd9q3Wn9tIxvF5znWo7Nf9Abjtho7tFc09xqiamx2fe
07731Ro1y0KPfXoHfuM/vA32VlxLbI33D3Uf+Z67IzP3+1OL19apf77Ferpb
MrxeOi42i2ZSYTtSn8CTWrqraozGN8gf4D6x86n2HS99Z+KCWoF8+rL49CvN
Bz+GV1vk2TNS+8P88FUw50eS5fu58xi+SE2DTakTr81td3yQN0PfeLuOi/fI
Pp3zllpde8cXri/nxRuFOTASHKK2B2N7hS+Dw2AeMYQMvBsGZL4v7yHuQ4wn
VY+nRa7njtQt1MhHg4YcTO7BXpyzId5x8AZn4XlQdj7UsD7IFb8qnC9mBRaj
nyzuBZ5PbzL/W6K+fSbeMuDWhMgd6yKeisifjXiPfKx7bC0tA3HdGXn2Bc1v
0LmV1PVJv9QxgM2xGfbqVv9O4bUtGm8uzPdsydwvd+2G3kZHDvqJ1n9c2LZL
oi6mHkAu/AbZyEHkbN41rYHP+Of3xHNwYbykLqb2ou5ifV/UwzfEuDV08GbM
c/c98f55W3p9q2FbkasmRe7j/5Jq7twIPu2J+oGYoPYnLsCRrsAZ5lgbFe+m
AVHnFZJtk/j/K3IQ9Q15jbfRjsIYAl6yzhv3Cc09VfjOO9VvVzszdW16ScTD
aeEn+Ob44Em+ele0/8x87liNxxSuc9E1+/Cl1dLbe4Xl473SyHyXneq3la4Z
+U9oZMQIdQ71PVhxoPD+AfHuHxFvf/BuY7wRwFVyHrUdfkSNji/x5uAe1EX0
Z8WYmD5UdVw/3/B/WCcCezYG9o6IeCF2Ogu/nchHYDE4Sn2CDtAFuh0mHmsb
/kZnxAOxQH36SeaaDBnxZfytJj1c2nC9MLXmfIgf/Kxhe2AL9nXm3ossx0Ie
sKVv4EKraK8vnJOoLyaGP2C7GSEL9d/FUR/yFlkebwd0go7wB+qRXVGrvJT7
P4uBwbMr6hlqt7TmuN8ZvsXdX5TMeeb/2JZr/EDD73vkRW50wfuQ/XnI3hqY
1ifuAYZwHjUFNfHn8eYBx8qa6wB8FbtdEZh8Q8QZ37xH+aZ2O5j5/6/R8R/G
Janj5jzNDWz4m1xLziWPt+OzmXPmG5lzM9gFJuchH+9t4nJcxE4jsO8P0lN7
bn3yfwG5Fv/YG1iJDyxpMV94Yqf74lxq9vPDLg9qfXppXVHvU99jsxFRR4NJ
YCg9uLQ2c/1K7TpP/XdKv7OpNfjfhdpusWgWFs4j1GgHA8/h1x7vEf7HwPeQ
/aYy5GhyLfbN3LmeGp93GnHxLHm5cD3Bf64XZZaHO38QcdFRuIY4HP4+pmqf
59zDUc+cSdyXXpulfrbak/Gm5J4d8e5gDH986cPwf2rS7qhLyS/QrQz9H4k6
9g3x21vaX3nTHI03y3YwpzAGLCv8HxwYRU3C+XujFj4QdelLDfsHvsF/TTdH
jvuPeP+7tG+uyv32AMt/KPprG469y6XDzxqOb2oxsArcq+TGHTDnocx6R+f/
B1O8QF8=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllmeM1VUQxf/73u7b9/4lolSJsEJURAImhIARISJFREU0ikoJi7GAAY2A
IB9A2URwixBWY1BcECIkNEFAMBQNXcSFBSlrDF2aNKUJAuLv5HyY5M49M3Pb
zJnb7JW3n3srFQTB90g+cjgKgpa5IOiaCYJbTNTJBsEG9CbpIDiO/mEYBBOR
Yej3Y9+a8RJsTqCPQw7FQfAHksU/R+CnwOolQdAGvRj7vsSfhk87sGHEu4Zt
NXON0Btic57xXfiMCxzzCmv/g34hLwguIovBbxV67U349wWvIX4VcyXE6ACe
Ya432NPI46zVMzS2GP+XGNcQrw7YTOK3CX0G7X088gS+fYk3hr28h9wg3nWk
D/5XkJvYFqG3xX8f8bqhd0Wm4luJHOM8J5AQ3zBl3y7EK844xifY3ocMxHc3
9nWxLc76/rsVBMFHYHuwX8d5zmNzN/5rdKdgtegd2d+cnPcySHeM7w/oDYlV
q/Ni+y1yGPsd2EfgW4ixA7wDa4xAH8gas9HPoC8FG4G+En0r/h/j+x02Yxlv
JMZr4K8jl8GnY9+UtbYgzdEfZP3r2NdDvzPlufXEuiP2WdLEWBP5TfQWu5Aq
bGcg7bEfyv52Yr9LPmD3ov+L7eTYa+vOGnEXDZHBrN0WvT6+N3QfKa95kPEG
fLKM88GbgNfFvnXGOTeUs7wc+q2PEONq5DvU3RWxxib008SI0SPFxLdx4rX0
RssZX876LW7jTSqwLUcq0M8pZuSaUa0EBX7LTqxXnvabLmDchfWGEe8v5m4H
rwz99ifwaYPvGea6Mn4MqcC2POe7aIMMxrYYmYP93LRr8TDrF2Zck38y/gz/
pmmfIcj6TfWW05CR2F9GeoJ/jj5ctYTeA30U++1P7H6Kn/KcYv1HzKYZx5xB
rAnY92O8KN+1+AF3sjDjmuyP/mLiWpnH8muwXY20ZPw3+CrGFboTxjfyXCvr
ss4N1Uwe+ghsvsb/efTfwE6Fzt2tSEP20gAZzf7ywFuB10fvwd7ns8ZOfF+I
zCWtwE/iezy0bxFzJeCl7G8F8VciX2H7I/qgQt/5QfBXmRvHuAP+y1Uv4G8U
+o2Pstan2OxPmdP2g28FfxN8J/hx8O7MHUi55guw/ZW5WnEF+F5i18f+IvEv
IJfAO2G/J+U3mcReS5FtjNvneW+jwZM873ED2PqsuUt3sBZ9Tei7Vo4tY7w0
9F2oRqdie5L1I/ayXe8HVoWMBz+YNrf1Yg9jM+a4Box74XOM2I2R98HHJ+aG
+eCdiNU5dq7MTZkbpqBvKjBHVKKXE786bY4ZTbxxiX2VIyPB3kHuyTPH1MP3
GWxKMn7DU7E5U1ypPas3zCLmkDz3iFtgN2PvTTk5VrmWuJZno4+OfUadbWmB
a18cJW4SB4hLy7K+W3HqIPQifPpgOynj3BuTOJZy8MnEnCeuU83OUzzsSznr
uYxzZV7o2lfO/EzsWeC7iVeGPiB0zanWxLE666Kcc1dnPoh9M2K0A+/O3BB8
h8bODZ1ZaxUn3pvWXI0+Bp8w7R6wKnGNq7ZbFDpXBxOvd8o5e5bxuci10IW5
BeAnQtfCCuRZfGfnjKnHKrcPxOZO5bhyOUR+z3dOTwO/Grs3i3OriLc68drK
WXFTLfZN8s1R4uKNkXuROFlcsDYy94sTlCvKUeWmcqZKd60eh+1CbLaz1rWs
e33dfHPHNt1PgTnkdNacKa48y9xyxssic41qap9qTzkD1pm5Y6w3JWcu1R5r
wNI5c5Fq9GHGjybu/bpD1cahnLlANXKJ9b6J/NfQHvQWX0beq95kl+4u51pX
D64m1ijmhoOP5472Mt4Tu/dqT43ZT1v8e+b7D1Ub+Y+kv5E4QrW1IuteoRpT
rnYP/bdQzj6Q8xqKrR5ajX40cu9Rj2waOceUW/qjNI/8h9DfQTk3EayEeJvx
bS2OZTyAGJPBIt1/6DvW3eqPpL/Z9sh/P/3RisCuRe6d4hT9ve6Onfv6g4kb
xRniCnFkjfp71m+pHr8kck9VL9UdtEj8J9BfQH/O6apX9F9S7jnKjZmRe49y
pAK8PDH3itM3R+7h6t3KWXGZeqx6qzjtSOw/qf6i6pmq1W1Zn1U1O4/xu7G5
Q282AX1U6LsRZy2J3YPVe/WnKmVvXySuJe3xJ/UafFZiW4beEftHYvcy3fFD
7K0y51zUn7RJbE4Rl8hHXF+WuDeJ8/8HeMiTsQ==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6334044084039064, 0.6393668249866483, 0.911085031734951], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHuwVmUVxvf3ne+2v/19NE2gmBw4B50UcsZEjxHqoKJcjjbIocmhlClK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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnlslUUUxb/X91re977vYaIsRYW2QESqiWIBEcEEBcpiAhTUiGgiSjUR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         "],
         VertexColors->None]}]}, 
     {RGBColor[0.6930474506790265, 0.7351965980697337, 0.912444709395743], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmHmMldUZxr+7zv3udy+MVJTSYZOxKXUMgwjjDNQatlFRUYYJtIkzstnQ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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmmIV2UUxu9/nXv/916Xyn0dlw/lCLmMuabM6LgbGuLUh7FmNDB1goqs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         "],
         VertexColors->None], PolygonBox[{{2815, 2173, 1415, 3152, 2814}},
         VertexColors->None]}]}, 
     {RGBColor[0.7526904929541466, 0.8310263711528191, 0.9138043870565348], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmG2QlmUVx+/d5/V+nuempr7YTFO5Kr4A5oxguIiYtCRCTIK7QruDb5Ng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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtln1ollUYxs/er73v+zzPLPsj+yDTibrUNCItSy3dzEJHTd00Ta1IXdlm
hJTDz8jpHykpmpWWFVRQTcEZRqVWkGHljJphzgqDzAjdWgUqlf0urv1xwTnn
uj/Oc5/74+n3YGNNQyqE8DrIgF5xCLOSEFbkQhAxrRjCVLAtHcJ2sAt+J4hK
QliWDWFDFMJDhRCq4d7AQBmy4zmrZZ1Bf38eXeTbkF3D2X7ZQ35dxg7b8bWe
s7vYvleKvaLPtP4amyuRbWa/m/vsAUNZD+HsFmwPgz+D7bOgDG4SNvbh7xT8
Ivi3uOMI5K9nP5p9BfJNrO9Afin3aUHnAn6+w2cf7nIArEN+LSgtsc/efEsB
5MGlkb/lQN531zdVss6Bxdhu5WwY9oej35WyzynYngy2wr8AvmRdjo1adMuQ
r2VfBzYg+yE+V+FjJTiLi1cz9v0+8ntj36GA7TyoRfYMfH/WFeg3Yfs27ruX
u2xE9nPsN8G/GPsNFPvDnE1EvhJMxd985B/H5uXob2a/EZlf2T+FTiuyk+G3
8T1TOauHewCfx+E6QA370dwxxlYE6lgPQv5TZLN5c9OQKYXLgenolsMfgo/z
fqsfM36L8aAp6zfZw7oRvgG0st7F3XaCHch3840zuM/DcFnWa7jzvNg5o1x5
Gf0u9p3g7hLnxJ34HgG/Cn6B8gf9YZx1ppxDH3Of19gfy1lnCbarYue2cmAH
XIqzofg/gsxI9NsLzr069lcXLSNuHmcrkC3DZgKWs14IXw9m6u3xt5rzpyO/
bQv7m5TLig/6teA4Oi34/wn/S0BfuPLEbzsE+RS20uALFS3364N8GzYGYz+X
da4ohq09ObMWbmjs3JROG7aei50bz5L7f2PrAjgP/gL3YO8KUMg6h3egX4FO
Vc4xC5G/Ud92Efn58I9Frn/l0AnWHeAc9s+D05FzSrkkm3/guzv22ysnTqP/
W8Fv+RHmm1mvTlyb+8BsdGcpZnA/p53rg8HStHN+API18AsyztGx7N8E57C3
EP0nuOftsWOpmnqb/VLFiH0j8sOxtZuz0+o1oF/imCvWt6J/GH560bXZS/0N
3Zdi15J65AR8TUG+gXVjzrU0EH521vGYCTeKs2ZsL4P/N/IddbeBWb91B/7L
M37zSrgOcCPyi5GJ4CezX856TNa5fwOoy7oGlNv6Bt1dOd6N/WvgVwd/k3yP
BGvSvsOTsWOgb1eN6636Y39Mzm+m2G6HP5p2jMvhjhTcOxdmbOsD+PNp21Ts
JSNOb3Ad3/sLZzennDMHkf9M8QvugX2LnhGaDeqZXayzyGdAp87RzybuFbXo
1KiXJK4NfVNp0TNBs0A1tamnR6g3qOf9A7es4FmknDyKfhf7U7zfQXA/tmqU
b+g/k/Ms25Q41zTT1As+idy71BP0tnoTvYXeWLOwLe/erZm4GX4L2J9zjzgL
NyLy28un3uqronNHb3aS/Z8F94Kj8I8knrGarZuxUcp9+iPzSnCMt7CeG7u3
TVe/jHymtWbsf5FzSrk0CJTEasrwKdfAIXz9ELkW1aNH5T3zNOuUo+3sTxR9
F8XnG2R75x175YB6k3qeep161Fb2zyfuXfrmy5BtjzyLNMMHJJ4ZmhXKqXHK
d71t8Mz4veg3kS/l/ITYPU29TDOgHv1HE8diPT6OY6sq8axSTcSxa0K1MBbM
wFZ14tpTzNQLr01cu+qJqvV38u4dqvmT8ElsXc2UY+wvRo6deoxyUT1OvU05
Oa7gnqleqZgmrMsSzzrNYM3SSYlni2aqZnt10bNeM35O4n8S/Yso566K3HPV
a69kvShyD1Tv0z+RaudU5F6uGpqD3NyiZ4tiqF6tmuns6dn692qIPLuV799z
9m3R8dU/jWZR755/F82kS2L/4+nf7iK4t+CaUC2oRtVr3i2YU8+ZAXdf4t6l
f6r/AdVRREE=
         "],
         VertexColors->None], 
        PolygonBox[{{2954, 2948, 2947, 2427, 1188}, {2159, 2951, 1188, 2957, 
         2158}, {2948, 2961, 1189, 2428, 2947}},
         VertexColors->None]}]}, 
     {RGBColor[0.80726725, 0.861883, 0.894034], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMVdUVxw937uOce86hPtJ2wBRnAAkQhBL90IRGnClGGaCM0grGpowT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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnusznUcx3/P5Tx+t0dyVkJbEbqMU602Ni0Rcz2lyCWWEyLiHBplpNzO
YSnHOZvCllYdRcxRnFDSZd2xVnQzivojqrUM2WjV673388d7z/fzfC6/z/dz
/XaaWHNvdTYIgiaQB5OjIKhJgqAaTOIcpkEwgN/ZhSC4tSwIzvB/T+gx0O2h
J3CeEgZBIRcE9Rg6AT0VmUUY68V/F6CXF4NgB/KV0KvjIOgMRmeC4GvoK5H9
G/oc6Mj5BLbW8bsZ/YHwB3Nug41Lsd0CDiJ3AOzH1xfgZ/BvFDoN0Dfgzwpk
c/BHwWvBRkfoodiYwPlB8BSyl0C3Bk9yrkS/Fv+2499r8Juh+4OD+Hcce1Wc
J4Ch0F9hcx+230NvE/QC/NkI3QQa+f4p+N/zvXHYnoOtm6EHIlsEx6FPZH3X
i+BC6c6K9aNgdinm1XyrCfnpyG/h+7OgN0Ifw9ZW6J+Q+RH0xd9/kHkCXl7x
hP8maIRXD1aBBrCVu30IqlqRr7y/fRr8VfLhKvAy/CPYm4j+5/ifQF8H3QX6
Y+gh2HkA3U7QHfCnPRiGL5VgGLwvVA/cvwqZMZwnob+y4BydSR0D3b1twXe9
gm+2i33nAsiAAJSByejej416ZFeBCPoQ/1+NfjfoNZyPxr6L8vUY9DTkn4c/
HP/WFl0Dyv0HyL8Efyn0Pvg38t8MdGfGjtXr0OXwYxCBtuAIvMPgEPgh9l1H
ol9X8J3fx/680N9WTFVL5SVd1ZRqVz6Krxp+CPkpRd9lJzVTm7pmVCvvljlW
Y7H/TMEx2wO9CPpt/O2M/QroCtUwvMGgK/qj0d+A/lj0T0fuMfVWf3AT8q/g
X3XGOr04X8TOBdAzdOyvxUa31DlQ7Szg/7oy11Ar6AIoR383PuxBrgXsBLvB
Cu62EjwNlguRY65YV2HjYXTPRO7d5pxzq55TrynHqr1CKdeqwbvR3YR8Y8Yz
Rr5+x//fxva5JnZMFAvlsCuy3aD/4FwBZoX2Sb7UcJ4bOiaKxUL8qU3cE+qF
Zep51UriWtmVdW6bQT7nHM/nvBj7n8F7A1yO7M+ha6ci71ydArfnnLOziWei
ZmGHMtfub6BPzjWs2bIw8rc0Y97B1lrieADZxXnf9Uip1nTno9g6lrj3umc9
S7cppjnP1MnoV4CFWfus3E2Hvz3nHHZRrCLr/g6K0OuhT3K+nu/XKdZFzzbd
UbXyJf/1yLpmrsFWZeJZqRm9Dl/XgzvKPPOGo3sP2A89EZ3enG8De6GHQO/C
3mrkP4FukD3oKPXu0IzUbB2QOjeascMVW/SH4Nsc+DHn1kX72gl7g/DlbOrZ
oR3wDb6eSxzrucj35dwv8ezSjNasOAz+zHtmPI7uPBAiPz/j3lqfuPfVY12g
RySOlWKgWtwc+W6qyXaaV4lz/2nGvTEtdm2rR/rjf/fUtd8bf35Ft2Pq2dgc
uHdUo6pN9ZB25S1gas47U7upQ+TZqR2l2tirns25Rk6qtkLHRncYTGyC1Ltu
GjG5C95loXeBekK7QztDu0I7RLkPi56dqoEl8HaoT/LOmXZTfexdrB31XOoa
VW0qx+Whd4Bmv2Kq3d+kHGT9BtDs3FOaDZqh2hXPxt6V2hk9Is8wzS7leANy
H4Weharh8ci/iPygrHdmn8h31N1UQ2Phj0mca+1A+b4s9ezUHRo5N6SuNfWI
bG0JnWvZfAvdFnA+4x3RBtnx8LepdrBxXu+PxLWrOy3l20uKfmuop2bAGxF5
16vn06JrVLWpHalZWFuaLZqJK+DVFT079eZ5hP+mJ64F5XQU/t0X++2iGa5d
XpN612un38m5X+rZozeMdnMhdG1pR/+L/H+J3xYjs377vBr5rDeQdvH+yLNR
O1mzXjtAs18zf6beApHfappRrflvXNGxUI//ojkUutYVM8VqK/Izs45Zinwx
9dtO/+mtoTej3op6c/wPzStNcg==
         "],
         VertexColors->None], PolygonBox[{{2772, 2781, 1049, 2260, 2771}},
         VertexColors->None]}]}, 
     {RGBColor[0.8608307500000001, 0.879745, 0.8700376], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl2uQlmMYx589vIfnfd9nmDFEW0ZlNSM6UDJTwgeKDuiwbUZoo91C+65W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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmtMlmUYx2/eAzynd7mVaYJbyDDnirBo2jKjD7V0gUcStswgCALlBbVV
ZtYUUFdyUAOXZtls1VKwA2aus9oXN+3kkaylthb1pTJdgw/9/rveD/8993Vf
h/u6r/s6PPk16QXNMefcKyABlgWsQ+c+jDv3HtgPfdBzbluWc5UIjuY4d4a9
02CEdQuyK8EKkAY/g7HI12NrBIOtKeeeYu9adHdmO9fmO7cocq4z6dwh6Drf
bEi3lvU8vvNBDbqVnDmZvWOgEF9KkV+P7kxsVrHOw8YC1vfDfxJ6FSiH/z57
Ib5Nx8ZD0B6+FmCrC/mD8Jp882UcMps46yXwItgI/kH+MngY3Tnc4SK6g+wf
g84Fo9B/g790f1Ao37A5C3v3gEbfzpDt/cgPIXMS/ADOgQvwL+rOxOMN7F+P
7fGgGNlT+NgdWkwUiy7Wv/Ft5NsB/25i0I5vbWADssPQyznrBPH+FLqP8/ey
Xg1/kLu2orMN3e3gbdYD8A8j/yX4AnwFLsP7FxRhazY+TWPvdfmHvTLkH4G3
BLTj6yVkFmJ/j2+ybeyVcNbtoIHzngBl8JYg4+J2xynw7oSeAW5iHWLrEN+P
QZDJnRuIy/jAcqiA7ySwEv1VYBdyReh+hslvsP8H9v8EPdg+zl43/naBz6F7
4Bf7FgPdfRZ7dei/jL1T2HqMvaPwXmPvW2S/A1tDi7livQVMgJcbWazXoL8W
e20pi51i2BGajmTbQ4v1Ps9io5gvQrYiZbmommnh7AHoGPkwpBpCfnFkvm4B
U9DdCX1C78V5B1i3stcCBlnPB83Y+AD7G5OW25fgncuyHJ8Dfx17/aoN+LXw
BnzzXTF7AXoMfl4Dnmddw3dPYLnzLPepRrae/Zy4xfQB+GP0hgmrgQrfYqZY
rYX/EbqfQO/i/DR0nPNj4Gb4t4AHI8s5xasN+mpoOaNcmQj9E/r78HcvOM96
YcpqQrWgGm5GPx1Z70irHpDpCqwWlRM9rLd71hvUU/SWX0N/n7Q3LWD9I/ZK
kL0Dehn2H/WtNnrBWdaJyHxVTis216k/ZWLUhP1GMBb7/XHrHVWRvZV6SMD6
HWyMcPaMpOWy3mwwk9Me8CM7uzpmb7kp01v0puodOayzQ+shfXpjfFyBb9MV
P+Ug9ifFrOfdBe9KaLFTz1MurIc/kLCckO+qqSBzB91db6y3VQwmcOawZNBv
4L6bfevB6r3HOWMz9jvBjmyroQbW9SmrLcVUvUtvdj7Tw+pSVvOq9W7lXMpi
plgpB29knR+Y7XuJ1w7Wb3r2ljNjNjuGMr1QM6QP9Abmm2q0HF/z8G9qts2A
xejmgyLO+w/9X1kXhlZLqoFy6Mkp672aOeo9qinVknqQetlu33qBetpJ1pV6
i4T1vFzostB6jWK+1LecUa4MYaIqtDvId+mkA6tJ1aJmpGZZRWSzRTNtDWcv
D2x2Kod2o3/Bs96uN1OtlYCGuNXcOPinofMSNgP6OfsoqM6xvSOsV3s2y7Sn
WdWR6T2aWXF4vwQ2mzSjJrJ3FfoKyAut1z+DzNOe9fxX8bUYvJtlPUe9+4hn
vVU9XLNkmBhsiNlMqUXn8ZS9tfZU66WaSdlW89Ogb4usN8jm75x5q2+zQzmn
Wj+cmTWq+dmh9Sz1KvWYdZHNdM1yzbR0ymaoZqdyUrkwN7TerZyYC/8+32pL
NRgLrcept2WF1quf8633qAfNS1nNqlZVY4rNaGZ2K0a9kdWgak89Tb2607Pc
Vc9WreqMrEzNNoXWA9X7NJOTkf0j6d9IOXoA/lT2libtPfRvoH8Y/bvoH0Gx
e8u3XFYMSz17Q8lKR7NQM1OzUjNRs1P/XPrX0gz9H7M9ONQ=
         "],
         VertexColors->None], 
        PolygonBox[{{2966, 2408, 2407, 2967, 1192}, {2784, 2788, 1052, 2263, 
         2783}, {2590, 3115, 1375, 3114, 2589}, {2933, 2590, 2589, 2409, 
         1170}, {2786, 2784, 2783, 2262, 1051}, {2408, 2932, 1170, 2934, 
         2407}, {2169, 3147, 1412, 3149, 2168}},
         VertexColors->None]}]}, 
     {RGBColor[0.91439425, 0.8976069999999999, 0.8460412], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNV21M1WUUf/j/ue//u6S5tVg6sbUxXQtda7nl5qe+UFYm0ctsKy5psYVU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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllF1oznEUx4/n/fk//yfSSk1ksVqWbIuyonblZjY82zzkpcyGWm3zsmel
FKGMbbbMbLIaatxMuSAvKfOyKxfKBRKFTTE3SrkRPt/Oxalz/t9zvr/z+i9p
7si1R8ysDokh20OzHcjOqNlTgKa02a2s2VzC7O48sw70TuR83Gw04tjqlNkT
c58csaUZs3WQ9eBTKRyODfj+RKawnyF/8V0I3zj+0ynnPgvegH0B/BF4L/Gt
2GPgLeDXsc8FZv3yJ7+XyDj2JPZx8Fri12Dfx/4FdirqufYjDxKecyO5DePz
BqyNHMfgv4rcIf9W8Hr0Y8QXwXcGewv+pfgX8G/Bfxdc9dRzEr4TSAX6BN++
JjyHAfR86L1TDYvhupb22rt5YxNYbei1T8A3C/4YezbuPSvivQXIEvBy7BW8
PYLPa97vQr6BfUfCuPf0MvilwHujmv6g/0bm81YZ8Qez3mP1dpB89pJLO99G
0T/g8xz9BfIv4TWrFuWsXFXTNPZm3ltOfGvSLAlWAv9h3lpvPvsq+PfHfQcq
0StC76W+LQt8ZpqVYibh+4i9KOY7UEN8Of4dYKuIH+atkYz3Zpg3OwPfKe2S
Zp7H/yY15MGG8K/GLgNvA1uK3Qj/WmJOYxfgPwJXJ3Ix4j6a3RekOuozVC+a
sa8kvCdxuEpSzq0dGoSrm/ihmM9gQFzIjZhzNhC/kfh9Eed8h74bvABehR2F
Lxa6rpw/w/cp8FqL8dmT8R3W7ipGufxAaqKek3ZPO6DZaweLye0V+Mqo78Ah
3m9L+y0qx3r4tmZ8VzWzvrTPTLOawt5GLk2h67px7d5MynddO6je6YZ0O+ph
L1w9yJx5zX1Zv1Hdpm4qh92U9V6rB++JfRv4LDTj22AzgdeqHVPuukHdnmro
Qj+a8VlrB+rSfmO6Ld3kPd56GPotKIcIvsmM77ZuZA7/A1nfbfVM/wrtrHZV
/4z/s16dIw==
         "],
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2100, 2151, 2401, 3438, 2402, 2540, 2539, 3431, 3432, 3232, 
         2246, 3231, 2245, 2389, 3496, 2095, 3122, 3121, 3120, 3229, 3228, 
         2088, 2141, 2381, 3420, 2382, 2531, 2530, 3411, 2374, 3410, 3413, 
         3412, 2613, 2612, 2232, 2736, 3077, 2081, 2135, 2471, 2368, 3074, 
         3075, 3076, 2521, 2909, 2910, 2720, 2221, 3194, 2220, 2352, 3494, 
         2075, 2465, 2464, 3377, 2351, 3049, 3050, 2074, 2512, 2900, 2710, 
         2711, 2709, 3173, 2205, 3526, 2895, 2332, 2894, 2897, 2896, 2597, 
         2596, 2200, 2705, 2935, 3441, 2411, 3512, 2410, 3439, 3440, 3470, 
         3469, 3155, 3471, 3472, 3159, 3160, 3161, 2568, 3158, 3157, 3156, 
         2436, 2445, 2055, 2593, 2592, 2698, 2190, 3503, 2672, 3154, 2853, 
         2277, 3261, 2276, 3506, 3150, 2170, 3524, 3525, 3502, 2788, 2784, 
         2786, 2785, 2051, 2112, 3115, 2590, 2933, 2932, 2408, 2966, 2968, 
         2964, 3148, 3147, 2169, 3521, 3523, 3501, 2782, 2773, 2779, 2776, 
         2044, 2105, 3112, 2585, 2931, 2928, 2405, 2960, 2963, 2950, 2956, 
         2953, 2161, 3254, 3258, 2827, 2836, 2832, 2176, 2818, 2823, 2802, 
         2812, 2807, 2845}],
        "0.9`"],
       Annotation[#, 0.9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2099, 2150, 2399, 3436, 2400, 3437, 2255, 2538, 2394, 3428, 
         3430, 3429, 3124, 3123, 2244, 3230, 2094, 2148, 2479, 2388, 3097, 
         3098, 3099, 2535, 2922, 2923, 2743, 2236, 3224, 2235, 3222, 3223, 
         3409, 3408, 3082, 2473, 2472, 3407, 2373, 3080, 3081, 2083, 2524, 
         2914, 2724, 2725, 3205, 3204, 2227, 3202, 3203, 3388, 3387, 3062, 
         3063, 2131, 2468, 2359, 3059, 3060, 3061, 2517, 2904, 2905, 2715, 
         2213, 3184, 2212, 3182, 3183, 3038, 3039, 2069, 2126, 3366, 2341, 
         3365, 2342, 2507, 2506, 2892, 2330, 2891, 2331, 2893, 2595, 2594, 
         2704, 2703, 2061, 2119, 3315, 3316, 2888, 2320, 2889, 3146, 3458, 
         3459, 2667, 2666, 2171, 2564, 2172, 2669, 2668, 2970, 2432, 3513, 
         2671, 3153, 2670, 2435, 2977, 2274, 2838, 2840, 2839, 2263, 2783, 
         2262, 2787, 2997, 2998, 3114, 2589, 2409, 2934, 2407, 2967, 2430, 
         3460, 2429, 3461, 3149, 2168, 3520, 3522, 3500, 2781, 2772, 2778, 
         2775, 2043, 2104, 3111, 2584, 2930, 2927, 2404, 2959, 2962, 2949, 
         2955, 2952, 2160, 3253, 3257, 2826, 2835, 2831, 2175, 2817, 2822, 
         2801, 2811, 2806, 2844}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3109, 2647, 2646, 2645, 2765, 2764, 2097, 2149, 2392, 3427, 
         2393, 2537, 2536, 2924, 2387, 2750, 2239, 3227, 2238, 3225, 3226, 
         3086, 3419, 3087, 3088, 2140, 2476, 2380, 3083, 3084, 3085, 2529, 
         2919, 2734, 2735, 3218, 3217, 2231, 3216, 3534, 2367, 3401, 3072, 
         3073, 2134, 2470, 2366, 3069, 3070, 3071, 2520, 2907, 2908, 2719, 
         2219, 3193, 2218, 3191, 3192, 3047, 3048, 2073, 2128, 3375, 2349, 
         3374, 2350, 3376, 2211, 2511, 2339, 2898, 2340, 2899, 2459, 3364, 
         2338, 3035, 3036, 3037, 2505, 3339, 3340, 3338, 3342, 3341, 2458, 
         3336, 3337, 3027, 3028, 2067, 2501, 2890, 2701, 3540, 2702, 3168, 
         3167, 2197, 2881, 2882, 2574, 2591, 3490, 2054, 2116, 3116, 2439, 
         3480, 2440, 2563, 2562, 3139, 3138, 3137, 2426, 3457, 2438, 3474, 
         3246, 3247, 3140, 3141, 2165, 2545, 2554, 3463, 3464, 3462, 2431, 
         2567, 2566, 2665, 2664, 2547, 2273, 2969, 2965, 2780, 2260, 2771, 
         2777, 2774, 2042, 2103, 3110, 2583, 2929, 2926, 2403, 2958, 2961, 
         2948, 2954, 2951, 2159, 3252, 3256, 2825, 2834, 2830, 2174, 2816, 
         2821, 2800, 2810, 2805, 2843}],
        "0.7`"],
       Annotation[#, 0.7, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2398, 3433, 3435, 3434, 2644, 2643, 2254, 2763, 3107, 2638, 
         2637, 2636, 2758, 2757, 2093, 2147, 2385, 3421, 2386, 3422, 2237, 
         2534, 2379, 3416, 3418, 3417, 2615, 2614, 2234, 2742, 3079, 2082, 
         2136, 3406, 2371, 3405, 2372, 2523, 2522, 3397, 3398, 3396, 3400, 
         3399, 2608, 2607, 2226, 2723, 3058, 2078, 2130, 3386, 2357, 3385, 
         2358, 2516, 2515, 3372, 2347, 3371, 2348, 3373, 2463, 3369, 3370, 
         3045, 3046, 2072, 2510, 3360, 3361, 3359, 3363, 3362, 2601, 2600, 
         2204, 2708, 3026, 2066, 2123, 3332, 3333, 3331, 3335, 3334, 2199, 
         3313, 3314, 3016, 3017, 2060, 2118, 3309, 3310, 3308, 3312, 3311, 
         2196, 3008, 3489, 3009, 2880, 3010, 2053, 2115, 3290, 3291, 2877, 
         2879, 2878, 2164, 2654, 2655, 2572, 2588, 3487, 2050, 2111, 3476, 
         3477, 3475, 3479, 3478, 2423, 2945, 2946, 2942, 2944, 2943, 2163, 
         3136, 3135, 3528, 2266, 3245, 2267, 2795, 2261, 2544, 2285, 2658, 
         2286, 2864, 2406, 2925, 2555, 2556, 2428, 2947, 2427, 2957, 2158, 
         3251, 3255, 2824, 2833, 2829, 2173, 2815, 2820, 2799, 2809, 2804, 
         2842}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2397, 2762, 2253, 3235, 2252, 3234, 3537, 2391, 3426, 3106, 
         2635, 2634, 2633, 2243, 2756, 3096, 2092, 2146, 2478, 2384, 2578, 
         2579, 2580, 2533, 2921, 2740, 2741, 3221, 3220, 2233, 3219, 3535, 
         2370, 3404, 3078, 2611, 2610, 2609, 2733, 2732, 2080, 2133, 2364, 
         3394, 2365, 3395, 2225, 2519, 2355, 3383, 2356, 3384, 2467, 3381, 
         3382, 3056, 3057, 2077, 2514, 2903, 2713, 2714, 3181, 3180, 2210, 
         3179, 3532, 2337, 3358, 3033, 3034, 2125, 3356, 2335, 3355, 2336, 
         3357, 2203, 2504, 2328, 3329, 2329, 3330, 2457, 3328, 2327, 3023, 
         3024, 3025, 2500, 2886, 2318, 2885, 2319, 2887, 2453, 2454, 2316, 
         2883, 2317, 2884, 2195, 2876, 2312, 2573, 3006, 3007, 2114, 3289, 
         2311, 2873, 2875, 2874, 2189, 2867, 2868, 2571, 2587, 3486, 2049, 
         2110, 3113, 2937, 2938, 3237, 3236, 2259, 2482, 2481, 2487, 2940, 
         2424, 2939, 2425, 2941, 2298, 2552, 2299, 2869, 2300, 2870, 3145, 
         3504, 3505, 3239, 3240, 3241, 3238, 2796, 2797, 2268, 2660, 2659, 
         2794, 3244, 3243, 3242, 2661, 3142, 2543, 2257, 2565, 2480, 2828, 
         2272, 2662, 2663, 2851, 2852, 2837, 3151, 3152, 2814, 2819, 2798, 
         2808, 2803, 2841}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2275, 2813, 2847, 2846, 2271, 2849, 2850, 2848, 3260, 3259, 
         2167, 2972, 2973, 2971, 2975, 2974, 2433, 3483, 2442, 3481, 2441, 
         3482, 2156, 3499, 2791, 3498, 2792, 2790, 2265, 2793, 3144, 3143, 
         3249, 3248, 2657, 2656, 2264, 2789, 2447, 2444, 2443, 2553, 2178, 
         2769, 2770, 2768, 2258, 3278, 2297, 3277, 2296, 2550, 2295, 3456, 
         2422, 3453, 3455, 3454, 2557, 2551, 2162, 2653, 2652, 2419, 2549, 
         2548, 2109, 2996, 2995, 2570, 2294, 2492, 2491, 3005, 3004, 2871, 
         2310, 2872, 2450, 3287, 3288, 3284, 3286, 3285, 2495, 2059, 3015, 
         2700, 2194, 2699, 2193, 3304, 3305, 3301, 3303, 3302, 2117, 2058, 
         3014, 3013, 3307, 3306, 2198, 3326, 3327, 3325, 2326, 2122, 2065, 
         2706, 2707, 2598, 2599, 3353, 3354, 3350, 3352, 3351, 2509, 3043, 
         3042, 3041, 2345, 3367, 2461, 2462, 3044, 3368, 2346, 3533, 3188, 
         2217, 3189, 3190, 2718, 2717, 2906, 2518, 2079, 3065, 3064, 2363, 
         2469, 2132, 3068, 3067, 3393, 3066, 3214, 3213, 2229, 3215, 2230, 
         2731, 2369, 3403, 2917, 2918, 2737, 2739, 2738, 2920, 2532, 2091, 
         3093, 3092, 2383, 2477, 2145, 3095, 3094, 2755, 2242, 2630, 2631, 
         2632, 3105, 3425, 2390, 3536, 3104, 2250, 3233, 2251, 2761, 2396}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2395, 2760, 2249, 2640, 2248, 3101, 3497, 3102, 3424, 3423, 
         3103, 2629, 2628, 2627, 2241, 2754, 3090, 3091, 2144, 2621, 2620, 
         2749, 2748, 2087, 2139, 3415, 2377, 3414, 2378, 2528, 2527, 2916, 
         2729, 2730, 3212, 3211, 2228, 3527, 2913, 2361, 3391, 2362, 3392, 
         2606, 2605, 2224, 2722, 3055, 2076, 2129, 2466, 2354, 3052, 3053, 
         3054, 2513, 2901, 2902, 2712, 2209, 3178, 2208, 3176, 3177, 3030, 
         3349, 3031, 3032, 2124, 3348, 2333, 3347, 2334, 2503, 2502, 3323, 
         2324, 3322, 2325, 3324, 2456, 3320, 3321, 3021, 3022, 2064, 2499, 
         3297, 3298, 3296, 3300, 3299, 2452, 3294, 3295, 3011, 3012, 2057, 
         2494, 3282, 2308, 3281, 2309, 3283, 2449, 3280, 2307, 3002, 3003, 
         2490, 2188, 3275, 2292, 2990, 2991, 2047, 2108, 2865, 2693, 2694, 
         2586, 2866, 2992, 3508, 2293, 3276, 2993, 2696, 3541, 2186, 2695, 
         2697, 3485, 3515, 3514, 2048, 2994, 2674, 3542, 2675, 2673, 2177, 
         2676, 2181, 2689, 2182, 2690, 3131, 3132, 2560, 2561, 2418, 3447, 
         3449, 3448, 2152, 2767, 2153, 2649, 2648, 2256, 2766, 2283, 2862, 
         2284, 2863, 2976, 3519, 2166, 3466, 2434, 3465, 3468, 3467, 2270, 
         3250, 2269, 3529, 3133, 3134, 2157, 2546, 2651, 2650}],
        "0.3`"],
       Annotation[#, 0.3, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3108, 2642, 2641, 2639, 2247, 2759, 3100, 2626, 2625, 2624, 
         2240, 2753, 3089, 2090, 2143, 2619, 2618, 2747, 2746, 2086, 2138, 
         2475, 2376, 2575, 2576, 2577, 2526, 2915, 2727, 2728, 2726, 2912, 
         2911, 3390, 2360, 2721, 2223, 3201, 2222, 3200, 2353, 3380, 3051, 
         2604, 2603, 2602, 2216, 2716, 3040, 2071, 2127, 2460, 2344, 3493, 
         2070, 2508, 3345, 3346, 3172, 2202, 3171, 2201, 3318, 3319, 3018, 
         3019, 3020, 2121, 2455, 2323, 3492, 2063, 2498, 2314, 3511, 2315, 
         2451, 2313, 3491, 2056, 2493, 2305, 3510, 2306, 2448, 3279, 2304, 
         2999, 3000, 3001, 2488, 2489, 2291, 2985, 2986, 2987, 2107, 3272, 
         2290, 3271, 3274, 3273, 2184, 2691, 2185, 2692, 2988, 2989, 2485, 
         2486, 2280, 2569, 2981, 2982, 2102, 2856, 2687, 2688, 2582, 2857, 
         3530, 2858, 3531, 2281, 3267, 3268, 2860, 2859, 2154, 2558, 2155, 
         3129, 3128, 3444, 3443, 3473, 2437, 2978, 3538, 2415, 3445, 3130, 
         2559, 2414, 2936, 2417, 3446, 2416, 3539, 2983, 2984, 2041, 2541, 
         2542, 2282, 2861, 3270, 3269}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2040, 2483, 2039, 2685, 2686, 2682, 2684, 2683, 2038, 2680, 
         2681, 2678, 2180, 2679, 2980, 2979, 3266, 3265, 2855, 2581, 2179, 
         2677, 2278, 2854, 2101, 2446, 2279, 3262, 3264, 3263, 2484, 2046, 
         3163, 3164, 3162, 2183, 2288, 3507, 2287, 2106, 2045, 3484, 2289, 
         2187, 2302, 3509, 2301, 2113, 2052, 3488, 2303, 2191, 3165, 2192, 
         3166, 3293, 3292, 2496, 2497, 2322, 3317, 2321, 2120, 2062, 3169, 
         3170, 3117, 3118, 3119, 2068, 3029, 3344, 3343, 2206, 3174, 2207, 
         3175, 2343, 3186, 2214, 3185, 2215, 3187, 3379, 3378, 3196, 3197, 
         3195, 3199, 3198, 3389, 3207, 3208, 3206, 3210, 3209, 3402, 2525, 
         2084, 3495, 2375, 2474, 2137, 2085, 2744, 2745, 2616, 2617, 2142, 
         2089, 2751, 3544, 2752, 2622, 3543, 2623, 3516, 3517, 3518, 2096, 
         3451, 2420, 3450, 2421, 3452, 2412, 3442, 2413, 3125, 3126, 3127, 
         2098}],
        "0.1`"],
       Annotation[#, 0.1, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  FrameTicks->True,
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  Method->{},
  PlotRange->{{0, 1}, {0.1, 1.9}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 222},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32491078]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18990],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->2352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"BetaDistribution", "[", 
      RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"\[Alpha]", ">", "0"}], ",", 
      RowBox[{"\[Beta]", ">", "0"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9658],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32044712]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25302],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ev", "=", 
  RowBox[{"ExpectedValue", "[", 
   RowBox[{
    RowBox[{"x", "^", "4"}], ",", 
    RowBox[{"BetaDistribution", "[", 
     RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], ",", "x"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10002],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[Pi]", " ", 
   RowBox[{"Csc", "[", 
    RowBox[{"\[Pi]", " ", "\[Beta]"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"4", "+", "\[Alpha]"}], "]"}]}], 
  RowBox[{
   RowBox[{"Beta", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "-", "\[Beta]"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"4", "+", "\[Alpha]", "+", "\[Beta]"}], "]"}]}]]], "Output",
 ImageSize->{253, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->173368685]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"D", "[", 
   RowBox[{
    RowBox[{"CharacteristicFunction", "[", 
     RowBox[{
      RowBox[{"BetaDistribution", "[", 
       RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], ",", "t"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "4"}], "}"}]}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"t", "->", "0"}], "}"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4422],

Cell[BoxData[
 FractionBox[
  RowBox[{"\[Alpha]", " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"3", "+", "\[Alpha]"}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{"3", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{239, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->272046448]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{"ev", "\[Equal]", "%"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->27582622],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->109580977]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12361],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["BetaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaDistribution"], "[", 
   RowBox[{"1", ",", "1"}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["UniformDistribution",
    BaseStyle->"FunctionLink",
    ButtonData->"paclet:ref/UniformDistribution"], "[", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16742],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"1", ",", "1"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22194],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->301812525]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"UniformDistribution", "[", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23188],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNVM0OgjAMLozxp8ajHrnxJupF3mEhJnAwEuT9ZxkdYNlBDQdJ1m9f2/Tb
uo2r6qrbXXV1qbJzq5qqLp/Z6dGiS3gAUOA4HAHnGmdmFFojCrQC6WVk0FNt
vj/2hmgk0h2t33KJ6CP6hJL8Nu6bajaasqrb3siRm1yAhGUFxkx8WSdkfFKd
e3ldt3pEa40JkxmupWG7lhJazY1Tg59F/NU+okX35jtbS43XEvr9XvC4+8R+
084/qmX7sdfDDc6JB87sNd/R9N6HfwB4L365KMY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{78, 23},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->256063419]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->7469],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaDistribution"]], "InlineFormula"],
 " is not defined when either \[Alpha] or \[Beta] is not a positive real \
number:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16345],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"0", ",", "2"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4380],

Cell[BoxData[
 RowBox[{
  RowBox[{"BetaDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Parameter \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(0\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" is expected to be \
positive.\\\\\\\"\\\", \\\"MSG\\\"]\\)\"\>"}]], "Message", "MSG",
 CellID->31913],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaDistribution", "[", 
   RowBox[{"0", ",", "2"}], "]"}], "]"}]], "Output",
 ImageSize->{198, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14964]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4042],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"\[Alpha]", "\[Rule]", "I"}], ",", 
    RowBox[{"\[Beta]", "->", "3"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23941],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "10"], "+", 
  FractionBox[
   RowBox[{"3", " ", "\[ImaginaryI]"}], "10"]}]], "Output",
 ImageSize->{57, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->813910289]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->32195],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["GammaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GammaDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Beta",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Beta"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaRegularized",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaRegularized"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["InverseBetaRegularized",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InverseBetaRegularized"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->22566]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->26705],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->189188997],

Cell[TextData[ButtonBox["Functions Used in Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInStatistics"]], "MoreAbout",
 CellID->11484687],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->37090020],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->43891155]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"BetaDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 25, 54.5671928}", "context" -> "System`", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "BetaDistribution[\\[Alpha], \\[Beta]] represents a continuous beta \
distribution with shape parameters \\[Alpha] and \\[Beta].", 
    "synonyms" -> {"beta distribution"}, "title" -> "BetaDistribution", 
    "type" -> "Symbol", "uri" -> "ref/BetaDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8435, 260, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->156092415]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 137943, 2921}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2857, 65, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3480, 92, 1581, 37, 70, "ObjectNameGrid"],
Cell[5064, 131, 759, 24, 70, "Usage",
 CellID->11382]
}, Open  ]],
Cell[CellGroupData[{
Cell[5860, 160, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6338, 177, 856, 31, 70, "Notes",
 CellID->429991538],
Cell[7197, 210, 504, 17, 70, "Notes",
 CellID->7997],
Cell[7704, 229, 694, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[8435, 260, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->156092415],
Cell[CellGroupData[{
Cell[8820, 274, 148, 5, 70, "ExampleSection",
 CellID->146239326],
Cell[8971, 281, 269, 6, 70, "ExampleText",
 CellID->10542],
Cell[CellGroupData[{
Cell[9265, 291, 176, 5, 28, "Input",
 CellID->13063],
Cell[9444, 298, 221, 7, 51, "Output",
 CellID->410769644]
}, Open  ]],
Cell[CellGroupData[{
Cell[9702, 310, 180, 5, 28, "Input",
 CellID->30502],
Cell[9885, 317, 397, 13, 54, "Output",
 CellID->5576855]
}, Open  ]],
Cell[10297, 333, 120, 3, 70, "ExampleDelimiter",
 CellID->9099],
Cell[10420, 338, 294, 8, 70, "ExampleText",
 CellID->25023],
Cell[CellGroupData[{
Cell[10739, 350, 200, 6, 70, "Input",
 CellID->17226],
Cell[10942, 358, 476, 17, 56, "Output",
 CellID->511341111]
}, Open  ]],
Cell[CellGroupData[{
Cell[11455, 380, 307, 10, 28, "Input",
 CellID->30048],
Cell[11765, 392, 3108, 55, 170, "Output",
 Evaluatable->False,
 CellID->125483395]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14922, 453, 215, 7, 70, "ExampleSection",
 CellID->8636],
Cell[15140, 462, 248, 7, 70, "ExampleText",
 CellID->9458],
Cell[CellGroupData[{
Cell[15413, 473, 195, 6, 70, "Input",
 CellID->25967],
Cell[15611, 481, 334, 10, 36, "Output",
 CellID->389953974]
}, Open  ]],
Cell[15960, 494, 120, 3, 70, "ExampleDelimiter",
 CellID->9405],
Cell[16083, 499, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[16207, 506, 180, 5, 70, "Input",
 CellID->11531],
Cell[16390, 513, 500, 17, 61, "Output",
 CellID->388216223]
}, Open  ]],
Cell[CellGroupData[{
Cell[16927, 535, 179, 5, 70, "Input",
 CellID->7975],
Cell[17109, 542, 807, 24, 58, "Output",
 CellID->69994131]
}, Open  ]],
Cell[17931, 569, 120, 3, 70, "ExampleDelimiter",
 CellID->7531],
Cell[18054, 574, 212, 5, 70, "ExampleText",
 CellID->31495],
Cell[CellGroupData[{
Cell[18291, 583, 243, 7, 70, "Input",
 CellID->8044],
Cell[18537, 592, 570, 18, 53, "Output",
 CellID->587282675]
}, Open  ]],
Cell[19122, 613, 121, 3, 70, "ExampleDelimiter",
 CellID->26803],
Cell[19246, 618, 441, 15, 70, "ExampleText",
 CellID->3388],
Cell[CellGroupData[{
Cell[19712, 637, 195, 6, 70, "Input",
 CellID->18915],
Cell[19910, 645, 176, 5, 36, "Output",
 CellID->332474747]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[20135, 656, 223, 7, 70, "ExampleSection",
 CellID->25791],
Cell[20361, 665, 112, 3, 70, "ExampleText",
 CellID->29591],
Cell[CellGroupData[{
Cell[20498, 672, 307, 10, 70, "Input",
 CellID->25867],
Cell[20808, 684, 2657, 48, 70, "Output",
 Evaluatable->False,
 CellID->299675346]
}, Open  ]],
Cell[23480, 735, 125, 3, 70, "ExampleDelimiter",
 CellID->161355561],
Cell[23608, 740, 361, 13, 70, "ExampleText",
 CellID->10722],
Cell[CellGroupData[{
Cell[23994, 757, 399, 12, 70, "Input",
 CellID->24271],
Cell[24396, 771, 100829, 1683, 70, "Output",
 CellID->32491078]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[125274, 2460, 233, 7, 70, "ExampleSection",
 CellID->18990],
Cell[125510, 2469, 228, 5, 70, "ExampleText",
 CellID->2352],
Cell[CellGroupData[{
Cell[125763, 2478, 487, 15, 70, "Input",
 CellID->9658],
Cell[126253, 2495, 157, 5, 36, "Output",
 CellID->32044712]
}, Open  ]],
Cell[126425, 2503, 121, 3, 70, "ExampleDelimiter",
 CellID->25302],
Cell[126549, 2508, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[126808, 2518, 276, 9, 70, "Input",
 CellID->10002],
Cell[127087, 2529, 595, 18, 53, "Output",
 CellID->173368685]
}, Open  ]],
Cell[CellGroupData[{
Cell[127719, 2552, 404, 13, 70, "Input",
 CellID->4422],
Cell[128126, 2567, 718, 22, 53, "Output",
 CellID->272046448]
}, Open  ]],
Cell[CellGroupData[{
Cell[128881, 2594, 138, 4, 70, "Input",
 CellID->27582622],
Cell[129022, 2600, 161, 5, 36, "Output",
 CellID->109580977]
}, Open  ]],
Cell[129198, 2608, 121, 3, 70, "ExampleDelimiter",
 CellID->12361],
Cell[129322, 2613, 634, 21, 70, "ExampleText",
 CellID->16742],
Cell[CellGroupData[{
Cell[129981, 2638, 187, 6, 70, "Input",
 CellID->22194],
Cell[130171, 2646, 158, 5, 36, "Output",
 CellID->301812525]
}, Open  ]],
Cell[CellGroupData[{
Cell[130366, 2656, 216, 7, 70, "Input",
 CellID->23188],
Cell[130585, 2665, 495, 12, 70, "Output",
 Evaluatable->False,
 CellID->256063419]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[131129, 2683, 326, 9, 70, "ExampleSection",
 CellID->7469],
Cell[131458, 2694, 406, 12, 70, "ExampleText",
 CellID->16345],
Cell[CellGroupData[{
Cell[131889, 2710, 162, 5, 70, "Input",
 CellID->4380],
Cell[132054, 2717, 312, 6, 70, "Message",
 CellID->31913],
Cell[132369, 2725, 252, 8, 36, "Output",
 CellID->14964]
}, Open  ]],
Cell[132636, 2736, 120, 3, 70, "ExampleDelimiter",
 CellID->4042],
Cell[132759, 2741, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[133065, 2753, 319, 10, 70, "Input",
 CellID->23941],
Cell[133387, 2765, 260, 9, 51, "Output",
 CellID->813910289]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[133708, 2781, 311, 9, 70, "SeeAlsoSection",
 CellID->32195],
Cell[134022, 2792, 1262, 44, 70, "SeeAlso",
 CellID->22566]
}, Open  ]],
Cell[CellGroupData[{
Cell[135321, 2841, 314, 9, 70, "TutorialsSection",
 CellID->26705],
Cell[135638, 2852, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[135832, 2860, 299, 8, 70, "MoreAboutSection"],
Cell[136134, 2870, 184, 4, 70, "MoreAbout",
 CellID->189188997],
Cell[136321, 2876, 164, 3, 70, "MoreAbout",
 CellID->11484687],
Cell[136488, 2881, 160, 3, 70, "MoreAbout",
 CellID->37090020],
Cell[136651, 2886, 150, 3, 70, "MoreAbout",
 CellID->43891155]
}, Open  ]],
Cell[136816, 2892, 27, 0, 70, "History"],
Cell[136846, 2894, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

