(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     42359,       1414]
NotebookOptionsPosition[     35279,       1165]
NotebookOutlinePosition[     36796,       1205]
CellTagsIndexPosition[     36710,       1200]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"NegativeBinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/NegativeBinomialDistribution"],
           "BetaBinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BetaBinomialDistribution"], 
          "BetaDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BetaDistribution"], 
          "GeometricDistribution" :> 
          Documentation`HelpLookup["paclet:ref/GeometricDistribution"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"NegativeBinomialDistribution\"\>", 
       2->"\<\"BetaBinomialDistribution\"\>", 3->"\<\"BetaDistribution\"\>", 
       4->"\<\"GeometricDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["BetaNegativeBinomialDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["BetaNegativeBinomialDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/BetaNegativeBinomialDistribution"], "[", 
       RowBox[{
        StyleBox["\[Alpha]", "TR"], ",", 
        StyleBox["\[Beta]", "TR"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a beta negative binomial mixture \
distribution with beta distribution parameters ",
     Cell[BoxData[
      FormBox["\[Alpha]", TraditionalForm]], "InlineMath"],
     " and ",
     Cell[BoxData[
      FormBox["\[Beta]", TraditionalForm]], "InlineMath"],
     ", and ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     " successful trials."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->15620]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The beta negative binomial distribution is a negative binomial distribution \
whose probability parameter ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " follows a beta distribution with shape parameters ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->311311635,
  ButtonNote->"311311635"]
}], "Notes",
 CellID->272931153],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaNegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaNegativeBinomialDistribution"]], 
  "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " to be any positive real numbers."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->10777],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaNegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaNegativeBinomialDistribution"]], 
  "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->15726,
  ButtonNote->"15726"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->112389470],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->71285658],

Cell["Mean and variance of a beta negative binomial distribution:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->15726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaNegativeBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2922],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", "\[Beta]"}], 
  RowBox[{
   RowBox[{"-", "1"}], "+", "\[Alpha]"}]]], "Output",
 ImageSize->{43, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25566859]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"BetaNegativeBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23529],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "n", "+", "\[Alpha]"}], ")"}], " ", "\[Beta]", 
   " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "2"}], "+", "\[Alpha]"}], ")"}], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "\[Alpha]"}], ")"}], "2"]}]]], "Output",
 ImageSize->{167, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->144756479]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23627],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->9425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaNegativeBinomialDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19900],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Beta", "[", 
    RowBox[{
     RowBox[{"n", "+", "\[Alpha]"}], ",", 
     RowBox[{"k", "+", "\[Beta]"}]}], "]"}], " ", 
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "+", "k", "+", "n"}], ",", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "n"}]}], "]"}]}], 
  RowBox[{"Beta", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}]]], "Output",
 ImageSize->{287, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->182808]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"PDF", "[", 
      RowBox[{
       RowBox[{"BetaNegativeBinomialDistribution", "[", 
        RowBox[{"15", ",", "25", ",", "10"}], "]"}], ",", "k"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "50"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7975],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXctrHVUYP7mT3La3NlgfWNs86xMrCgpSBRWUKi66EJNKRRehFhowaaxZ
KLoQrG21FQVXxSdq/QMUdGnVhU0VdVkXXaQuuqiCFQJikPGcOXd+55uZM887
kzuPE8jNzLmT8/u+3+8735zX3Lt7ZvHA/rmZxdl9MxOPHppZODC774WJRw4e
4kXWAGNsmf++vYXxY5sfOb/Lts3/Bl938leLX9Dif1v87xj/O0jOx7vvW93z
SfFf4gLb/dFdtc6p+17vtd1amaVKRsTLkDzHa+CqcXKue38sFieuhsGE9o72
jDTS5ZcNweXulZM52TgWU89oV6NWoCbtf3quWCsb4xUN92I89J12Quu2iZc2
vSKb1tE2unGPwhA2d3ZrcdvWuOY8voVOEltUC6WZ4Fof+lJE6Wb++qswWb6/
Ae8/Of35g/ef3PXw+/x4dfU/eNVeWfn39wt/yYsuX/7HteSPSystl21eLN/n
F/LLUc6PZfn583/yOlte7ngJL1coeEMiSu4lYpeNJJi81iyYvLqLF//OgDlE
MX87d8lRMyWiqJvjyHJeyN9KhymvlrJR2MyoUC4dqlROMDCgYmwT3l46c+Hg
/Jdff3WOH3/6yc+8/K0T3+PfGNjnxvjZl/YVGGXpMPOJsuSYaaOMttguoh6D
6i7xZLnEk9gcDygqL0jILFxKzSIxO8CkcfLxRz/xHMWPHe8DGc7Juu09UyqR
ZdeY2pVIYzCeHVPiZMHMHlcUcy3iivqVQ1xtDfrFD0599gsPgTeOfYvj429+
h2MZS/x+xyFlmCBkouKKJq/844pysVZxFY+Zd1ylQcwnrhRKxriisUT7SGHH
NExwrOJK9OA2pPbaRFCJI2gr9eu9k2dlipFxc/TI6WB8zM994YsPerwRsUJz
0HoTM3WKmeEPP/gRt6LpJ07J8NDmkesi4gEdIhMP1YiHq2g8UPUQA/z4zA/L
83NCesTAtogY2DOl/tPEQAViYCMf0kB3qh7kZtsj9KY3EKN3qfQepqNWIrEm
q98MhUWPcFMGbo2eheupZiG0Gt4GDWkrvcJo2U8th4Nc2t6RPxLoHUa/8unX
8c3iyJkb9IzvMpqVWTPtCMZo1kfNhoOa2d5ZLScfCp3ugU6mT7Jm+nS0+qAd
3QdNgm2HziEYbQrVhs7mIa9FaUPHdUabQu9FaCsPROhBp1qNHvnq4UxMaFuI
XpErjQb5ayAWKdESHgLv4k6+2fBdDN/oOSm+TZwXxzuzsLrGdjmMy0rkj+Vj
391pJ4oGjTIFK4P+jquMZ9ekw7+nxK+W/9wolLdCanXBKNRXhdZRRIzh2GOZ
VRGnRo081EAHqgc1TBvJWxWVuYwqfVXFwo4FtptQLX/EMxr06YYhDeusjR6c
Hb4CF2WdvHpVv6sm8/6osvOu8pLknY7sWJvuHTGcJuVUZRXFaTBe6byR4TbA
bTu44+FxLZvXRPrT6FyrOMQuA8NhZg7RYA2HKTm0gMimwF3LfYrLcKbL7BY2
lrBp28RbOu5Ur8Zwl5Y71Xsx3CXn7inDVTwmUtpesGXuAgFMJC/FkompUEys
lBm2QjEtYLKnDUuZWbraVv2yo0dOsyA7tRqvd4KYNtk7+ww4EjnciSDS72ok
N3BfcROMH9W/aiRHcF9yFLfOLmoa1Kyzb6k3dxrMGUlUzApFi/DEDG+JeYtb
2WkMb9i/nYw3f3yJ2sRpY/hCzs/OVyPjDPeBUN4SfNpaw9izgMmetf13z7hV
0qZx4/ZMr2+g383Q3UkTGI7tb6L3uPs00nvcQ5rm/Sz8bbl9LQsdN824stqz
Nspbqq7z6TWRMw719RqTnlX3ug1MGb6vH/5G7/dIXfxEwNbcT+Rmb64arbZ/
FjDZc3XUz/hXcf/Urv96+ocdqcS/OuQVtSpQN92ed/xJMs9vaWYtKu93j/OB
xv96+++MIrtxL9yuiL8hT5otJPI5bq3F+UxK/3i2jON1DwvqOdVkLMRFvmTB
N74tPQvqPrZAxPbOUIfleumxb328jB5bQGGLtu5uPVEpy/ENMZWxuLpcV8Ny
9TRe1SzHgy6ViW+V76rD9UvGUmOpXZqWpXlW8EUtm+POObYwsr71qjVP5kVZ
7O8F9tViJKwoi/FRuP2zmDx19rLW0htLZZloSzeVyqJychU3k+QfV/mvKdyT
TtATmzz78oo0v8cxcp+8QCIK9WKIlrhu+TXpk/VYNg21fr3fevfVnZ/pRzt4
NVHEOHdQzajaWJzCYs1sRd/zSai1/hx9g3M1/UB6aUwhtqATpWxpue2j7fvK
16JsAEwUH/RzNgq0BblFb8stztU594fJE0uvNRJVRNyt1fMx5smqw2uFiPYT
hUi/cKZnRDRGhag0DPl+1QwaHovwBw21IJQdgdpSZBcsehRQN9QuoG4M91Td
Qtfb09dJ9jMfj7CTfsNZ+jqzrkbviIrTmP3JJyRkj6tC0oKk3xjr3SOcjwV3
BjT1thbvztzsmOLUweoEv4vQJvsF30mEENbXlAhUUaxmgrfkCIHv9g0gUeWw
lwWFXaSQXqiuN3q3c67mkN+1dW1mCaWijrPO2bIEGfgflmKFLg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 143},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->77718924]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1905],

Cell["\<\
Generate a set of pseudorandom numbers that have a beta negative binomial \
distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->19096],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"BetaNegativeBinomialDistribution", "[", 
    RowBox[{"3", ",", "7.5", ",", "20"}], "]"}], ",", "10"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->17016],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "42", ",", "22", ",", "38", ",", "27", ",", "142", ",", "47", ",", "124", 
   ",", "77", ",", "103", ",", "28"}], "}"}]], "Output",
 ImageSize->{280, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->435817532]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26905],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"BetaNegativeBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7251],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     RowBox[{"2", " ", "n"}], "+", "\[Alpha]"}], ")"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]", "+", 
     RowBox[{"2", " ", "\[Beta]"}]}], ")"}]}], 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "3"}], "+", "\[Alpha]"}], ")"}], " ", 
   SqrtBox[
    FractionBox[
     RowBox[{"n", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "1"}], "+", "n", "+", "\[Alpha]"}], ")"}], " ", 
      "\[Beta]", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "1"}], "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], 
     RowBox[{
      RowBox[{"-", "2"}], "+", "\[Alpha]"}]]], " ", 
   RowBox[{"Sign", "[", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Alpha]"}], "]"}]}]]], "Output",
 ImageSize->{259, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->237774067]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"BetaNegativeBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27703],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVm8+PFUUQx2d39j12YZddBJUoKAFFo4ne1IheJajoBeNRnhsTOCgGOenf
49EYE/4342l83TO9zvtMTVd1T4+wJMBUT3f96Kru6fp2vburJw9++HH15OHx
6trtx6ufHzw8/uXaZ48er5vqrara+rWqqjuXq/Vzs37yf92f39Z/28f2ad/9
W4e3TXUe9HXQ5xLffwB6CZry3wO9AL0D+thbQam04g3Qe4nvP1K0ovz3Fatp
xffeirP+1Unr+o/r0nYOf6T5q4V2p+H2mtxGO+kzgiWOZHsuP2fTQuBXm6yz
Su3Pp+PjxuwaNdkZkRGLvCBDi87Vf36l9ou+nytBr4MEfTxXQR/JG0PJWnRa
OHMdS7Os22yRvDBx0vSpBUmMGFkSdSzpc9dr0cXispO1CDIHozWfBG5j7dO4
Sn6Jc7XLtPlY06QWuSz7o/5X7+6ZeuV5hV880Y5krgVXdSGfm9b1gMvcK9vy
/cDXvZjn9zv55bhpHjd/LW0+7+akqiRPFfT3hpyEL/fGiczZ3gqV9e2fBN37
w6Y753RjTnROWMv7GTw4bweC7kcGvsvNGTCsyPMC18MBF9ta0871/bXoRL6M
9zzHyzvwS8ooavGONPcb1sWiwj3eGIzgvMsZgpatufnYFiyi/L3OR5Z+jt+7
I+3cC0P7ze5/zao2e3OPzEutT/fB8U9FIulvPZfv0PpXIhfSXxfhyj3qK8/1
bbQyAz8Ezf6vKlKYDbwO+k1lvLaz2qx8C63M0Jkd0corilWU+hrom4njUyOk
tZJS3PqRTuzWPJj7w9jZhOvc8a+FfSuWM/f3XS0XDn3I33qaGNNj2em9SNTH
QnP8XdFjnMmcjNZ7bfBVCr4TT/aTz8m2E7DtTG7JpfP0yTunl/HsRpaM0fZM
VvYtZ+xslGdq3jvFt/HcmV7N02Qer3KH/rLJXq+FvMqYf05Xrurl52ntng4v
7+TOYjRH1bPlcv50YpYDeVO/qXnes+TQ0smHYy6iD/M88jjq2d9yEe3Z6CFl
v9qdF0/su6rcVJoe/sKz5Rmb2Ryz6XBnI+GajjxS5pd+Hc6viGps9GD2IvKM
cqCM1JiW55L5h/OhOwddVrhxbZK+0M2thGQ4/hcbOdMO769GxgVfSuNTZ4X6
fe5n5Rtl7jjq91Ms9V6vtRI8dgbvU6Uy0u+ItnKPc//4XNk/ET15hN5/g/a5
R/UTWv8BfZDQ6xO0Er3gHsS9kf1fAX1DmTX2595MREzrL+GsVXULrUQvuMpp
JftzFyEGw1gj+sHV/GJif6It3mqvhWsM6EH8Ln8qtmG93Qy5VG4tgXY6GsMi
cu0Y+6qWrH0Yk2nFYywytTO359FGtjFXkGpTYhJy6hZsN7eMiZyKCS2uTHet
RazL0d7mobzZlfAjm3x+LeSY49lMzpKiVRLQJL96grZaEZW0SpCxPSq/diNm
E+Mprb4kjkNpsuomhlflRVGQPSV6yiAQ2h6QtU/ZcI+pO9XkKMqrYMnbp7Tb
nZLRJe9R4cZ02dhuROaJL+0u8DTH19g5q1SlivWMlRBLG/Js5yvuVuEWyVqz
4h5zala0FWSpEqEM5mISbpaqJ99beAxRvguzcOVO0WZZ8on5BfTVsDvpnKEj
xYwMeiTEOLXhOK0uh7sd+w9RPEpM5xAQYjFST3LbS8I8SrieVBVYCz4N+Bv1
49dOwqoluec27Wpa5J75fRh9Kam3Vr0j4z5EeJ6CJteSXIhnfdyjXXw/BU0u
Ld71hyLb+8YTxNMYSx+CJhpLvM1XoQzmWeNCpO3qjFxooYbfUQqRLuJ1xNu0
3xfxPoDfIca21p/821nQ0HEN39NmgXgeVzn5c/8kf+rHXUzrzy/tFXEWJFx8
DmRMxhfpKQ0PLFVTJd0mzGG1VMNcCg+UsujSMrW8x3/BTTFV5ndDNmxKzuGk
2+XplRwlUMNnh4mWQw0l/GhO+bGbucYcmWG/kc715bA0OVtkVOSiimm4n4Yq
lkVKx5C+Eqgiec8hi1EWZNuiax58Jy+6UlHHCdVWU3e6yTMi1j0UQh21qCsp
u8weVwbvG/sN2RjWKeHxMS1Oc/zFbmJLoJLWk16uvDKnPNYXHDTTEUgJ+aSc
VNTSgutpv6kqgVrm/XKPtXFluIq1J2rG7ONvEG95qGX4faTtHKthnGNVj6mI
q4ZxMscfIpREmdM5BBSS6631kYbt7HajadkY1snVJXmvFvaMMWxTw0THsFgZ
E5V+5RkqIe29GdekbdgbLWHkfgqaCCQj4VlK1TDPfhRXwn5xC++J0VJqi5He
Q+ttRffrLbetfwECy2oZ\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{378, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->756543052]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->25316],

Cell[TextData[{
 "The probability of at least 50 failures before 10 successes assuming a beta \
distribution on ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"BetaNegativeBinomialDistribution", "[", 
     RowBox[{"4", ",", "5", ",", "10"}], "]"}], ",", "49"}], "]"}]}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->10642],

Cell[BoxData[
 FractionBox["5062", "126697"]], "Output",
 ImageSize->{53, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->356768428]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3834],

Cell[BoxData["0.03995359006132742`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->58696483]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86062964],

Cell["\<\
Plot the cumulative distribution function of a beta negative binomial \
distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"BetaNegativeBinomialDistribution", "[", 
      RowBox[{"2.5", ",", "3.5", ",", "5"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWE1oE0EU3mTzs4iCnhS8VFDBRtOePAhitFZR0ZMHvQhpEFKkVtqIoBSs
B3vwogfBohgP9uAPCFYqiqioBwvSgFQQPUgEqQeL0OIhIIxv3rxmZjezf/kp
cdnM7jfvfTPzZt68N5vD2UL+5EC20J/Lduwfyp7J9+eGO/YNDkGVGTEMowy/
q+sMeGfwhr8yY/BUywyUJojS9IzCMwrPrYSNGKrRlQIQU7RMFPbwkiszvJBr
xCQW+qpGpwObDryFF3GBqyW2wg2L8GeLeu7UttDqXlMK1lu6m2a8K1hfUoN4
6UA9qO2kg/UUV3syyHOMhMpzm8c9NLZtTHrTku9FyfWi5HWqXHid6sWrHSOb
CV27EspZGl2tTjIkthrkt1t/jbZvame124G7fOROrG81/NpXa4u3P8BtJHlh
Pp36TFrvqlrc+yyUPrj/sUbapFG0oJaPuxSCKdavNPPDNiMTd0uVyl8uet5E
1vJYFL6W3AsvNV4jpuikYdkiZtzBC8JJOHR43xyuQQBTCRMMt2FB8VjDd+49
Z9QW7Uy//z54ehJuw+KF+fXLL5I/qqNNnR1oa22N5zwnaGXjWs3/xXOctesR
wwzv2jkOt2Hxwnz75hvJJ9puxM2ew02I5+f/2HxORtibWtZmxJ9mf5LHJ3lh
Li5WPFkuu7fmnKL3Mf+9GMTvYwH3uqqzNCYO3WKJ+94IpavMjTZT+66nyNQQ
iy6cfybXU+bEa1pWt3Y95+YWPFnttw92IB67/Jq8OYnWAyb5mAcLooDNehl3
Jatd7NyLGPK3zc6zhSnK5yO2Efcy2uMu9o00PJ56sjV/Dbpr/DPWEcTg5xTJ
kxjJe3tugdlcNBy4VbddHjhralsMGhtUHf+snGTye8x/PY4hhsW3xXrAJM9r
WcehWAFKp3xWPuze6EMMMYbOpOL0LkdzwoMFbg7OL1kyUulZ9efSAZRCZjt0
oChPCNApaR9tYn9q7TnEub6HNn+GoE7ygw33MIq4XP7tEhUytnm4hNrjN6Zd
ImsmRM9XmHc82u7Bco95grU80ea6wwILLRAD4aJUA63K83/VQuHe4v8bj/NE
/ZGryLwjw4a6Wr3DROTYWLMybl8T95h3TFir9QzBco8JetYko51t60uefFdp
WU8QQ56xfUnLEUoWt+8Vao9efGFbUcCkrf+f5CXT7UzAnizPHBz5B8fLXgo=
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 144},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->162342285]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->654623609],

Cell[TextData[{
 "Density functions for beta negative binomial distributions depend heavily \
on the parameters ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->24052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"PDF", "[", 
     RowBox[{
      RowBox[{"BetaNegativeBinomialDistribution", "[", 
       RowBox[{"\[Alpha]", ",", "7.5", ",", "20"}], "]"}], ",", "k"}], "]"}], 
    ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "1", ",", "12"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"\[Alpha]", ",", "3", ",", "36", ",", "3"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28041],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzt289LImEYB3Bd957zZaEFL8F66N5RC4JYu+ylQ2JEMOVCoZuLdglEEjJQ
L4KBEBYUIUYXD4pQhyICEQ8q9EtULAjRIg9BZGq2876y/0EXmWfgfXl/zHzm
9vK887zzS1xZ/P1HXFlaEIcMdvHv4tKCY+inzS4NqZQKheKOle8Kqf0htXpF
9fHBGuwqS0XZ66rYHL8UZalS9m741NEvUoWXlxdcXl4ikUggEokgGo0K8Xgc
Z2dnyOfzKJfLqFareHp6wvPz81eSSCKJJDlLr6+vuLm54dLu7i62t7cFJv7X
crkcisUi7u/v8fj4SBJJJJEka6nZbKJUKiGZTDIFm5ubQigUwt7eHmKxGE5O
TpDJZHB1dYVKpUISSSSRJGvp7e0Nt7e3OD4+Rjgcht/vF7xeLwKBAO8fHBzw
t5yfnyObzZJEEkkkyVpqt9s8Gj09PeWr78bGxg+n04n19XWusbHDw0OmkUIK
KaT0rdLpdFCv15FKpXh06fF4BLvdDovFAqvVCofDAZfLBZ/PBykCJYkkkkiS
tfT+/o5Go8G/XLIVVooyheXlZczNzcFoNGJ6ehqzs7Mwm82w2WwkkUQSSbKW
ut0uy4ijUCjwXXgwGBTYijszM4PJyUmMjY1Bp9NBr9djfHycJJJIIkn2Esue
s10+i1r39/cFt9uN+fl5GAwGjIyMQKvVQqPRYHBwkCSSSCKpv6RvTGq1WvzM
5PX1NcvfCFtbW1hdXYXJZMLo6ChX1Go1pMdIIYUUUvpWYXtqdo68Vqvh4uJC
ODo6ws7ODtbW1iCKIiYmJjA8PIyBgQFSSCGFlL5VWITI8tRMenh4ENg573Q6
zc9+s/y0x+Ph+ZepqSlSSCGFlH5UPvPv6rvenPIfUVHxQA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->879957788]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->18294],

Cell[TextData[{
 "The probability of getting negative integers, integers beyond ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ", or non-integer numbers is zero:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->6782],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaNegativeBinomialDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], ",", "\[Pi]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17831],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1538086]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->61963951],

Cell[TextData[{
 "Beta negative binomial is a mixture of ",
 Cell[BoxData[
  ButtonBox["NegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NegativeBinomialDistribution"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["BetaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->311311635],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"nb", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"NegativeBinomialDistribution", "[", 
     RowBox[{"10", ",", "p"}], "]"}], ",", "k"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->221558863],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], "k"], " ", 
  SuperscriptBox["p", "10"], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{
    RowBox[{"9", "+", "k"}], ",", "9"}], "]"}]}]], "Output",
 ImageSize->{190, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->183785378]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"dist", "=", 
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"2", ",", "4"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32781365],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ev", "=", 
  RowBox[{"ExpectedValue", "[", 
   RowBox[{"nb", ",", "dist", ",", "p", ",", 
    RowBox[{"Assumptions", "\[Rule]", 
     RowBox[{"k", "\[GreaterEqual]", "0"}]}]}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->759718569],

Cell[BoxData[
 FractionBox[
  RowBox[{"798336000", " ", "\[Pi]", " ", 
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{"9", "+", "k"}], ",", "9"}], "]"}], " ", 
   RowBox[{"Csc", "[", 
    RowBox[{"k", " ", "\[Pi]"}], "]"}]}], 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{
     RowBox[{"-", "3"}], "-", "k"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"16", "+", "k"}], "]"}]}]]], "Output",
 ImageSize->{261, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->403247660]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaNegativeBinomialDistribution", "[", 
    RowBox[{"2", ",", "4", ",", "10"}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->104120125],

Cell[BoxData[
 RowBox[{"20", " ", 
  RowBox[{"Beta", "[", 
   RowBox[{"12", ",", 
    RowBox[{"4", "+", "k"}]}], "]"}], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{
    RowBox[{"9", "+", "k"}], ",", "9"}], "]"}]}]], "Output",
 ImageSize->{233, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->253303730]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ev", "\[Equal]", "%"}], "//", "FullSimplify"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->829262369],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->855936819]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1967],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaNegativeBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaNegativeBinomialDistribution"]], 
  "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is non-positive:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->28540],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaNegativeBinomialDistribution", "[", 
   RowBox[{
    RowBox[{"-", "4"}], ",", "6.2", ",", "40"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31597],

Cell[BoxData[
 RowBox[{
  RowBox[{"BetaNegativeBinomialDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-4\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->263000487],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaNegativeBinomialDistribution", "[", 
   RowBox[{
    RowBox[{"-", "4"}], ",", "6.2`", ",", "40"}], "]"}], "]"}]], "Output",
 ImageSize->{357, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->127128335]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19046],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"BetaNegativeBinomialDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"\[Alpha]", "\[Rule]", 
     RowBox[{"3", "+", "I"}]}], ",", 
    RowBox[{"\[Beta]", "\[Rule]", "7"}], ",", 
    RowBox[{"n", "\[Rule]", "23"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25838],

Cell[BoxData[
 RowBox[{
  FractionBox["322", "5"], "-", 
  FractionBox[
   RowBox[{"161", " ", "\[ImaginaryI]"}], "5"]}]], "Output",
 ImageSize->{78, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->193149816]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12706],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["NegativeBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NegativeBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->25561]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->21229],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->297422826]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"BetaNegativeBinomialDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 25, 59.9735812}", "context" -> "System`", 
    "keywords" -> {
     "mixture distribution", "Pascal distribution", "Po'lya distribution"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "BetaNegativeBinomialDistribution[\\[Alpha], \\[Beta], n] represents a \
beta negative binomial mixture distribution with beta distribution parameters \
\\[Alpha] and \\[Beta], and n successful trials.", 
    "synonyms" -> {"beta negative binomial distribution"}, "title" -> 
    "BetaNegativeBinomialDistribution", "type" -> "Symbol", "uri" -> 
    "ref/BetaNegativeBinomialDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7945, 247, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->112389470]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 36567, 1193}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2325, 57, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2948, 84, 1597, 37, 70, "ObjectNameGrid"],
Cell[4548, 123, 972, 28, 70, "Usage",
 CellID->15620]
}, Open  ]],
Cell[CellGroupData[{
Cell[5557, 156, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6035, 173, 528, 17, 70, "Notes",
 CellID->272931153],
Cell[6566, 192, 604, 21, 70, "Notes",
 CellID->10777],
Cell[7173, 215, 735, 27, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7945, 247, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->112389470],
Cell[CellGroupData[{
Cell[8330, 261, 147, 5, 70, "ExampleSection",
 CellID->71285658],
Cell[8480, 268, 285, 7, 70, "ExampleText",
 CellID->15726],
Cell[CellGroupData[{
Cell[8790, 279, 201, 5, 28, "Input",
 CellID->2922],
Cell[8994, 286, 255, 9, 51, "Output",
 CellID->25566859]
}, Open  ]],
Cell[CellGroupData[{
Cell[9286, 300, 206, 5, 28, "Input",
 CellID->23529],
Cell[9495, 307, 617, 22, 54, "Output",
 CellID->144756479]
}, Open  ]],
Cell[10127, 332, 121, 3, 70, "ExampleDelimiter",
 CellID->23627],
Cell[10251, 337, 204, 5, 70, "ExampleText",
 CellID->9425],
Cell[CellGroupData[{
Cell[10480, 346, 229, 7, 70, "Input",
 CellID->19900],
Cell[10712, 355, 556, 19, 53, "Output",
 CellID->182808]
}, Open  ]],
Cell[CellGroupData[{
Cell[11305, 379, 449, 13, 47, "Input",
 CellID->7975],
Cell[11757, 394, 2826, 50, 164, "Output",
 Evaluatable->False,
 CellID->77718924]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14632, 450, 215, 7, 70, "ExampleSection",
 CellID->1905],
Cell[14850, 459, 273, 8, 70, "ExampleText",
 CellID->19096],
Cell[CellGroupData[{
Cell[15148, 471, 229, 7, 70, "Input",
 CellID->17016],
Cell[15380, 480, 304, 9, 36, "Output",
 CellID->435817532]
}, Open  ]],
Cell[15699, 492, 121, 3, 70, "ExampleDelimiter",
 CellID->26905],
Cell[15823, 497, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[15947, 504, 205, 5, 70, "Input",
 CellID->7251],
Cell[16155, 511, 977, 34, 77, "Output",
 CellID->237774067]
}, Open  ]],
Cell[CellGroupData[{
Cell[17169, 550, 206, 5, 70, "Input",
 CellID->27703],
Cell[17378, 557, 2550, 46, 70, "Output",
 Evaluatable->False,
 CellID->756543052]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19977, 609, 223, 7, 70, "ExampleSection",
 CellID->25316],
Cell[20203, 618, 348, 11, 70, "ExampleText",
 CellID->32514],
Cell[CellGroupData[{
Cell[20576, 633, 243, 8, 70, "Input",
 CellID->10642],
Cell[20822, 643, 186, 6, 51, "Output",
 CellID->356768428]
}, Open  ]],
Cell[CellGroupData[{
Cell[21045, 654, 92, 3, 70, "Input",
 CellID->3834],
Cell[21140, 659, 176, 5, 36, "Output",
 CellID->58696483]
}, Open  ]],
Cell[21331, 667, 124, 3, 70, "ExampleDelimiter",
 CellID->86062964],
Cell[21458, 672, 269, 8, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[21752, 684, 338, 10, 70, "Input",
 CellID->26124],
Cell[22093, 696, 1433, 28, 70, "Output",
 Evaluatable->False,
 CellID->162342285]
}, Open  ]],
Cell[23541, 727, 125, 3, 70, "ExampleDelimiter",
 CellID->654623609],
Cell[23669, 732, 428, 14, 70, "ExampleText",
 CellID->24052],
Cell[CellGroupData[{
Cell[24122, 750, 485, 15, 70, "Input",
 CellID->28041],
Cell[24610, 767, 1115, 22, 70, "Output",
 Evaluatable->False,
 CellID->879957788]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25774, 795, 233, 7, 70, "ExampleSection",
 CellID->18294],
Cell[26010, 804, 347, 10, 70, "ExampleText",
 CellID->6782],
Cell[CellGroupData[{
Cell[26382, 818, 233, 7, 70, "Input",
 CellID->17831],
Cell[26618, 827, 156, 5, 36, "Output",
 CellID->1538086]
}, Open  ]],
Cell[26789, 835, 124, 3, 70, "ExampleDelimiter",
 CellID->61963951],
Cell[26916, 840, 455, 14, 70, "ExampleText",
 CellID->311311635],
Cell[CellGroupData[{
Cell[27396, 858, 231, 7, 70, "Input",
 CellID->221558863],
Cell[27630, 867, 373, 13, 36, "Output",
 CellID->183785378]
}, Open  ]],
Cell[28018, 883, 181, 6, 70, "Input",
 CellID->32781365],
Cell[CellGroupData[{
Cell[28224, 893, 262, 7, 70, "Input",
 CellID->759718569],
Cell[28489, 902, 542, 18, 53, "Output",
 CellID->403247660]
}, Open  ]],
Cell[CellGroupData[{
Cell[29068, 925, 218, 6, 70, "Input",
 CellID->104120125],
Cell[29289, 933, 357, 12, 36, "Output",
 CellID->253303730]
}, Open  ]],
Cell[CellGroupData[{
Cell[29683, 950, 135, 4, 70, "Input",
 CellID->829262369],
Cell[29821, 956, 161, 5, 36, "Output",
 CellID->855936819]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[30031, 967, 225, 7, 70, "ExampleSection",
 CellID->1967],
Cell[30259, 976, 604, 21, 70, "ExampleText",
 CellID->28540],
Cell[CellGroupData[{
Cell[30888, 1001, 212, 6, 70, "Input",
 CellID->31597],
Cell[31103, 1009, 213, 5, 70, "Message",
 CellID->263000487],
Cell[31319, 1016, 306, 9, 36, "Output",
 CellID->127128335]
}, Open  ]],
Cell[31640, 1028, 121, 3, 70, "ExampleDelimiter",
 CellID->19046],
Cell[31764, 1033, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[32070, 1045, 418, 12, 70, "Input",
 CellID->25838],
Cell[32491, 1059, 262, 9, 51, "Output",
 CellID->193149816]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32814, 1075, 311, 9, 70, "SeeAlsoSection",
 CellID->12706],
Cell[33128, 1086, 1055, 35, 70, "SeeAlso",
 CellID->25561]
}, Open  ]],
Cell[CellGroupData[{
Cell[34220, 1126, 314, 9, 70, "TutorialsSection",
 CellID->21229],
Cell[34537, 1137, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[34727, 1145, 299, 8, 70, "MoreAboutSection"],
Cell[35029, 1155, 178, 3, 70, "MoreAbout",
 CellID->297422826]
}, Open  ]],
Cell[35222, 1161, 27, 0, 70, "History"],
Cell[35252, 1163, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

