(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     53177,       1858]
NotebookOptionsPosition[     44103,       1534]
NotebookOutlinePosition[     45427,       1571]
CellTagsIndexPosition[     45341,       1566]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BernoulliDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BernoulliDistribution"], 
          "NegativeBinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/NegativeBinomialDistribution"],
           "GeometricDistribution" :> 
          Documentation`HelpLookup["paclet:ref/GeometricDistribution"], 
          "BetaBinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BetaBinomialDistribution"], 
          "HypergeometricDistribution" :> 
          Documentation`HelpLookup["paclet:ref/HypergeometricDistribution"], 
          "PoissonDistribution" :> 
          Documentation`HelpLookup["paclet:ref/PoissonDistribution"], 
          "BetaDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BetaDistribution"], 
          "NormalDistribution" :> 
          Documentation`HelpLookup["paclet:ref/NormalDistribution"], 
          "Binomial" :> Documentation`HelpLookup["paclet:ref/Binomial"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"BernoulliDistribution\"\>", 
       2->"\<\"NegativeBinomialDistribution\"\>", 
       3->"\<\"GeometricDistribution\"\>", 
       4->"\<\"BetaBinomialDistribution\"\>", 
       5->"\<\"HypergeometricDistribution\"\>", 
       6->"\<\"PoissonDistribution\"\>", 7->"\<\"BetaDistribution\"\>", 
       8->"\<\"NormalDistribution\"\>", 
       9->"\<\"Binomial\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"], 
          "Functions Used in Statistics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInStatistics"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>", 
       2->"\<\"Functions Used in Statistics\"\>", 
       3->"\<\"Statistical Distributions\"\>", 
       4->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["BinomialDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["BinomialDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/BinomialDistribution"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["p", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a binomial distribution with ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"],
     " trials and success probability ",
     Cell[BoxData[
      StyleBox["p", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->29696]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in a binomial distribution is ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     TagBox[
      RowBox[{"(", GridBox[{
         {
          TagBox["n",
           Identity,
           Editable->True,
           Selectable->True]},
         {
          TagBox["x",
           Identity,
           Editable->True,
           Selectable->True]}
        }], ")"}],
      InterpretTemplate[Binomial[#, #2]& ],
      Editable->False,
      Selectable->False],
     TraditionalForm], 
    SuperscriptBox[
     RowBox[{
      SuperscriptBox["p", "x"], "(", 
      RowBox[{"1", "-", "p"}], ")"}], 
     RowBox[{"n", "-", "x"}]]}], TraditionalForm]], "InlineMath"],
 " for integers from 0 to ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->10032,
  ButtonNote->"10032"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " to be any non-negative integer."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->16876],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->11493,
  ButtonNote->"11493"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->367127666],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->62198871],

Cell["Mean and variance of a binomial distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->11493],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->82445090],

Cell[BoxData[
 RowBox[{"n", " ", "p"}]], "Output",
 ImageSize->{21, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->619335465]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->695],

Cell[BoxData[
 RowBox[{"n", " ", 
  RowBox[{"(", 
   RowBox[{"1", "-", "p"}], ")"}], " ", "p"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->493334367]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17213],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->10032],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5525],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], 
   RowBox[{
    RowBox[{"-", "k"}], "+", "n"}]], " ", 
  SuperscriptBox["p", "k"], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{"n", ",", "k"}], "]"}]}]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->96135653]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"k", ",", 
       RowBox[{"PDF", "[", 
        RowBox[{
         RowBox[{"BinomialDistribution", "[", 
          RowBox[{"50", ",", "0.3"}], "]"}], ",", "k"}], "]"}]}], "}"}], ",", 
     
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "50"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "->", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32295],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXEuLFEkQzunqaWUOgqA4go7toKCg4KIyehAPy+6ePO2oN2EQwUGdERU9
CHrw/T7LPkHdH6CoR9314gs96sWDevCgggoD4iC5kZVV0dmZkfWYqarOmq6G
roqMyor4IjIyMjLnsWXk0O5d+0YOje4caf5yYGT/7tGdB5s/jx8AltfDGHsN
38v9DGgOlP99zTnc1esGuHrwqAb3GtybcK8r7SXBcy9oD4q3RAcefsK31F6e
L3tje99AKvNanAFx6ZVtvBq9liht6/NIPU1CgnBRj7gnxBqvJQ7n4sC3rBfN
DXoOZoCPslHXXgv5bZJI3G09isA3ECthwGpB0/qklhCd8A5rqD2mNsbRGMNY
b4286ckNvDWHxH2p1hYI4mfkoIKjNSPVmT9fQ/4ogjtPXBdcv/YMHlw4/x+0
gN686Wo0feP6c6AvXXyg0mxYyNpBPksoN3O6I0ptdN33+Fy4Pg/iIvD/wKOH
b4Bz5/YLaAG9d89NpCP4d++8lPTYvpuS/vuvp8CX/p6Y+Pb2zSc/WhtA48B/
/vz13bsvkg80xRcx9+H9RBBzQNXCOYRc6AovIB9okg+KJR+gAB3Ln5z8Lvmv
Xn0E2uePIn/71n/Akz/9+LukgS9pcMH42C10h+qahK6cPh+Yki5SaSyf1cm5
70ddQ3WnFi1AJ42WMD70O2vIWDDjQuWr4y/pAKIfIyZfiYs5kuZhmpGzTZ0O
GBdzNPuXae3lWtuL8JoabAm9BnQxXrPxFa810Gs4/HtIa/v93pBRMIOr1tJW
sbrKd8BGHKyWjWLNXMij4t9p2zy0jY1FjJstSp2wrc+0jSuzeJy0axF3cJbR
lmBgld4SDKMSWULMnIOIXsx+uSNQM5vrqE2fbxtu5a7yoVezk1PoMQfR6JsR
KDu2PmC2KRFmDID2mbnUHawe8tlh1/1qYA0KWP/jabjDswOxyauXyaa2Mxcf
ufXch80w2/Qx1Ntltk0fJyHFC0/VZpBN5nj9+ccTqgJxz7YEp6e0hZYqJR8L
+0wLebiinzv77xGuZ8YGbz9/7SVtsNQqRdqAK7y0QV+J/NOMvGsnGhk6p4VM
eHJ5EYiIiD7aOf9UaGYEmip6KzTlRdP10YurvUQTtw/S6zS9T8HocZ0P0E9z
VyDR01VmDuixFrCi72VaHUntQSVq9Ww+B9RK1XvMhna2WfXKq6gTBWCJlK4P
C0Vq7IOpanaF3zOXqrWcGImdQFw+KBKfnkmL1i38sbKL7K10VzFW6Z55ujOK
sT5TJ1fqzeN52UrrxfJI6o2rc20nXLnhw6IowJfJmWI+kXMic4Rqxe0CQtGU
yLb+egOr6lyRGZW+Z40+tX7OFVPkbwnrld+qbNFYak2q5sxWs56RspcubFhd
MsyV9Go0u0G686OJ9ZOZJ9PUMdNEgdVUlrXANDHhsthxTB7y2WkbFuvpjlCe
m1aXPNA5LCl+or3KH6vfrj621qfJ9VKZgNW3DdsrzHSyw9wVXSFOHe+aSlqq
sfjBMUzdI80170et0h6zn+6TWvtMrTysC86cvn+Gm/lWZLno3ZzeTqEZa4HC
Nft/uSTS3NQ1i2bSsb2Qn32uaEuxMqaTTM0nel2NlhHObPpkJq1+ap1MLmOd
k28JD613Ellub2HuS+QFzFepdGComDOFyujqc+rMqk2Lsl5c5pZZPws5khmI
1+Wu9dvSIadO3mOJ5UfkEN0+qQPdOCUdxl6Vy1wsZaOzM5OttocUH0HQsCs2
maky7ZDmlStGnMRl1KHQdrmoBRLs/5tAyHjst+R/MWA9/wPrrAJo\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->670966614]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->168],

Cell["\<\
Generate a set of pseudorandom numbers that have the binomial distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->28084],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"BinomialDistribution", "[", 
    RowBox[{"40", ",", "0.3"}], "]"}], ",", "10"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20620],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "16", ",", "9", ",", "10", ",", "11", ",", "13", ",", "10", ",", "7", ",", 
   "14", ",", "10", ",", "13"}], "}"}]], "Output",
 ImageSize->{245, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->57205544]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14838],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19232],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "-", 
   RowBox[{"2", " ", "p"}]}], 
  SqrtBox[
   RowBox[{"n", " ", 
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}], " ", "p"}]]]], "Output",
 ImageSize->{86, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->290900214]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"n", ",", "p"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23566],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox[
   RowBox[{"1", "-", 
    RowBox[{"6", " ", 
     RowBox[{"(", 
      RowBox[{"1", "-", "p"}], ")"}], " ", "p"}]}], 
   RowBox[{"n", " ", 
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}], " ", "p"}]]}]], "Output",
 ImageSize->{108, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->38819615]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17262],

Cell["Second moment of a binomial distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->18534],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"ExpectedValue", "[", 
    RowBox[{
     RowBox[{"k", "^", "2"}], ",", 
     RowBox[{"BinomialDistribution", "[", 
      RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"n", "\[Element]", "Integers"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29831],

Cell[BoxData[
 RowBox[{"n", " ", "p", " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "n"}], ")"}], " ", "p"}]}], ")"}]}]], "Output",\

 ImageSize->{116, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2428707]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox["0.7`", TraditionalForm]], "InlineMath"],
 " quantile of a binomial distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "50"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "=", 
    FractionBox["1", "3"]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->48521834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"BinomialDistribution", "[", 
    RowBox[{"50", ",", 
     RowBox[{"1", "/", "3"}]}], "]"}], ",", ".7"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4591],

Cell[BoxData["18"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->210128162]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17011],

Cell["\<\
The probability of getting more than 5100 heads from flipping a fair coin \
10,000 times:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"BinomialDistribution", "[", 
     RowBox[{"10000", ",", 
      RowBox[{"1", "/", "2"}]}], "]"}], ",", "5100"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10642],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"BetaRegularized", "[", 
   RowBox[{
    FractionBox["1", "2"], ",", "4900", ",", "5101"}], "]"}]}]], "Output",
 ImageSize->{233, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->387094431]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3834],

Cell[BoxData["0.022212899523025076`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->184924216]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86062964],

Cell["\<\
Plot the cumulative distribution function of a binomial distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"BinomialDistribution", "[", 
      RowBox[{"20", ",", 
       RowBox[{"1", "/", "4"}]}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwPATiyRIMQPZ/IAuM
H/7/D6SRSQcgyQyU0oPSTECaCUjrQPkMLGBlUKAN5LAgqWIGSzqDSJDi/2AA
1svAguBD1COr0ELjM6PxNUEEK4QPJ8GmgDzGCKJpZLMWVhNobas2Eh+7Tx2h
Ia5PnF0IFVB9ekTZgGyOHnE2sSLbxABNOQxsyPpwhaMT1G26/xGpCZb2mKBJ
jwma6pDlIakOORULoLnsHDGi5889BSIGdhDB/PDhe6iqI2BVPEDyCtS1mCax
k8jnoFD/YLOPUvOZ8YTq4UP35887A0QMHCCC+fPnn1B5aTT1+mh8AwJ8fLYC
EwHUVnawrUuXnP/58w9IahdWXYRSGCidnidLJ7migmD+9m037G1nAhEDO4hg
dnWeA0naDBsG0G20CRdohIIBcokK5sOqLkxdKGUaK5o+YvSwoalBL82x6UHP
AaDSEMQlVi0h88HuxhRB0gXJPcBEDS33OMDlHiJ3LcUIUzZofLFiNQmYrPLz
NgMRAweIAOUgFJMGf44ZfnZj5AiM2hk9LqmVvkGpjYXI/IOsBuYmXHmB9T/2
NKgK5gNTL0pdgUiDs0k2C5gzgIU+wqzbt96gmoUUtljrs0GVEoaS3dR0Ea46
AcTEFvfklaW01oUrN8DkiPEFuhp010BaT8BUHxu9EtRU4AC3F9rbDkDlJ2KE
J/t/RLt8aKe40dROSrq1AvOBlTxKyxLIh8q3k2UqodIeXx6gYe4kov0z9NIc
yG5XsOzMGSehrTZ2cKutumontJ9Ti9UMFzD/7duvOHrL2HXRx0f+aG4DtWhL
serzA/M3briK0mJ9/vwTVL4Ij22oI2DobXHc6QNdJThh4U2BEWA+0BM42tUZ
ZJobCSQ4wfG2d89tlPbShnVXUOIVGJZQPSEk2AVSkASWRrShIL1oYGrLy92E
SG1ATqD/IkRtExiwGKVM8fNZiMJ3d52DwndxnA3lQ/Q7w/kgeSZg5xdeiwGT
KLDvi1FWMZMZhuTGKamlCuYILQMjAK9mrvY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 147},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->314288475]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4015],

Cell["\<\
The density functions of binomial random variables are highly concentrated \
about their means:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13063],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"PDF", "[", 
     RowBox[{
      RowBox[{"BinomialDistribution", "[", 
       RowBox[{"n", ",", "0.5"}], "]"}], ",", "k"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "0", ",", "30"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"n", ",", "1", ",", "60"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8117],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 121},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->10668035]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11999],

Cell[TextData[{
 "The probability of getting negative integers, integers beyond ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ", or non-integer numbers is zero:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->6782],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], ",", 
   RowBox[{"-", "3"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23454],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->99852644]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->12012],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Simplify", "[", 
  RowBox[{
   RowBox[{"ExpectedValue", "[", 
    RowBox[{
     RowBox[{"k", "^", "3"}], ",", 
     RowBox[{"BinomialDistribution", "[", 
      RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"n", "\[Element]", "Integers"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8922],

Cell[BoxData[
 RowBox[{"n", " ", "p", " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"3", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "n"}], ")"}], " ", "p"}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"2", "-", 
       RowBox[{"3", " ", "n"}], "+", 
       SuperscriptBox["n", "2"]}], ")"}], " ", 
     SuperscriptBox["p", "2"]}]}], ")"}]}]], "Output",
 ImageSize->{230, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->332031560]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"BinomialDistribution", "[", 
     RowBox[{"n", ",", "p"}], "]"}], ",", "t"}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->464069605],

Cell[BoxData[
 SuperscriptBox[
  RowBox[{"(", 
   RowBox[{"1", "-", "p", "+", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"\[ImaginaryI]", " ", "t"}]], " ", "p"}]}], ")"}], 
  "n"]], "Output",
 ImageSize->{93, 20},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->67698260]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Limit", "[", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{"cf", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "3"}], "}"}]}], "]"}], ",", 
    RowBox[{"t", "\[Rule]", "0"}]}], "]"}], "/", 
  RowBox[{"I", "^", "3"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->22174],

Cell[BoxData[
 RowBox[{"n", " ", "p", " ", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"3", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "1"}], "+", "n"}], ")"}], " ", "p"}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"2", "-", 
       RowBox[{"3", " ", "n"}], "+", 
       SuperscriptBox["n", "2"]}], ")"}], " ", 
     SuperscriptBox["p", "2"]}]}], ")"}]}]], "Output",
 ImageSize->{230, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->221172999]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->45709179],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "1"}], TraditionalForm]], "InlineMath"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["BernoulliDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BernoulliDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->109559857],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"BinomialDistribution", "[", 
      RowBox[{"1", ",", "p"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3178766],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"1", "-", "p"}], ",", "p"}], "}"}]], "Output",
 ImageSize->{63, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->113645830]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BernoulliDistribution", "[", "p", "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->114635251],

Cell[GraphicsData["CompressedBitmap", "\<\
eJylVM1OwzAMzpqUDY3lso11QkLdFSR4BjgMiQ1pr1BNk7YDAo29f7BTt41c
txS1Uup+jv058U8/ssvx8JldTvssfTtn38fT/iddf51BpQdKqR2su0TBt4Mv
v3bOgRzDWwN89egG3widf9SL12qyuQJpQKYgI5DK+G16pgBikFHpEzKhr4pL
TI7IpExllcfQNe5rxvbvGGSFeNGTC42GPU8ZFXkK9Pe1XNx24ouLaujqLCt2
Z9yfd2J7ZLhbth7oDAlJI1q9e21ENtsShTabFq3Mx7UJ1eZZ3F0y/NRYIyy+
dXnHT5zU8TOKpEWOEcPWSR01qVVdnoCh6C1Xg0eO6Tam+QRBr8x7RGq+Y/PU
9LmZdfIsjWtZnf0RBZ05e1n1YLIsyxbuT1u4LeNMGW7LdhF/5arpqvot/3+r
wS+xS+CD\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{80, 27},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->7337350]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->307904394],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " is the infinite population limit of ",
 Cell[BoxData[
  ButtonBox["HypergeometricDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HypergeometricDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->100094394],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"bpdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"BinomialDistribution", "[", 
     RowBox[{"n", ",", "p"}], "]"}], ",", "k"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->308109750],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], 
   RowBox[{
    RowBox[{"-", "k"}], "+", "n"}]], " ", 
  SuperscriptBox["p", "k"], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{"n", ",", "k"}], "]"}]}]], "Output",
 ImageSize->{181, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->58693468]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"hpdf", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"HypergeometricDistribution", "[", 
     RowBox[{"n", ",", 
      RowBox[{"p", "*", "ntot"}], ",", "ntot"}], "]"}], ",", "k"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->343715506],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{"ntot", " ", "p"}], ",", "k"}], "]"}], " ", 
   RowBox[{"Binomial", "[", 
    RowBox[{
     RowBox[{"ntot", "-", 
      RowBox[{"ntot", " ", "p"}]}], ",", 
     RowBox[{
      RowBox[{"-", "k"}], "+", "n"}]}], "]"}]}], 
  RowBox[{"Binomial", "[", 
   RowBox[{"ntot", ",", "n"}], "]"}]]], "Output",
 ImageSize->{332, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->407752385]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Assuming", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"n", ">", "0"}], "&&", 
    RowBox[{"k", ">", "0"}], "&&", 
    RowBox[{"0", "<", "p", "<", "1"}]}], ",", 
   RowBox[{"Limit", "[", 
    RowBox[{"hpdf", ",", 
     RowBox[{"ntot", "\[Rule]", "Infinity"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->662360908],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "-", "p"}], ")"}], 
    RowBox[{
     RowBox[{"-", "k"}], "+", "n"}]], " ", 
   SuperscriptBox["p", "k"], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", "n"}], "]"}]}], 
  RowBox[{
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "+", "k"}], "]"}], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{"1", "-", "k", "+", "n"}], "]"}]}]]], "Output",
 ImageSize->{182, 35},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->442704902]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Assuming", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"n", ">", "0"}], "&&", 
    RowBox[{"k", ">", "0"}], "&&", 
    RowBox[{"0", "<", "p", "<", "1"}]}], ",", 
   RowBox[{"FullSimplify", "[", 
    RowBox[{"%", "\[Equal]", "bpdf"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->592089036],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->363382890]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(3)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->16589],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " is not between zero and one:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->8001],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"8", ",", "1.1"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22015],

Cell[BoxData[
 RowBox[{
  RowBox[{"BinomialDistribution", "::", "\<\"probparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(1.1`\\) is expected to be a probability between 0 \
and 1.\"\>"}]], "Message", "MSG",
 CellID->991842184],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{"8", ",", "1.1`"}], "]"}], "]"}]], "Output",
 ImageSize->{240, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->121063633]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3029],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is not a positive integer:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->9921],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{
    RowBox[{"3", "/", "2"}], ",", "0.5"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5987],

Cell[BoxData[
 RowBox[{
  RowBox[{"BinomialDistribution", "::", "\<\"posint\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(3\\/2\\) is expected to be a positive \
integer.\"\>"}]], "Message", "MSG",
 CellID->486241439],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BinomialDistribution", "[", 
   RowBox[{
    FractionBox["3", "2"], ",", "0.5`"}], "]"}], "]"}]], "Output",
 ImageSize->{244, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->132413162]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9488],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"BinomialDistribution", "[", 
    RowBox[{"n", ",", "p"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"n", "->", 
     RowBox[{"-", "90"}]}], ",", 
    RowBox[{"p", "->", "3"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16714],

Cell[BoxData[
 RowBox[{"-", "270"}]], "Output",
 ImageSize->{33, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->64491663]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->32122],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BernoulliDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BernoulliDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NegativeBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NegativeBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GeometricDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GeometricDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HypergeometricDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HypergeometricDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["PoissonDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/PoissonDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NormalDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NormalDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Binomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Binomial"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->12698]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->19958],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection"],

Cell[TextData[{
 ButtonBox["Demonstrations with BinomialDistribution",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
BinomialDistribution"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->53879317],

Cell[TextData[ButtonBox["Functions Used in Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInStatistics"]], "MoreAbout",
 CellID->151638954],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->42404269],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->83074004]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"BinomialDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 26, 7.9737860}", "context" -> "System`", 
    "keywords" -> {"binomial trials", "coin tossing"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "BinomialDistribution[n, p] represents a binomial distribution with n \
trials and success probability p.", "synonyms" -> {"binomial distribution"}, 
    "title" -> "BinomialDistribution", "type" -> "Symbol", "uri" -> 
    "ref/BinomialDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9183, 278, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->367127666]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 45198, 1559}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3533, 78, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4156, 105, 1585, 37, 70, "ObjectNameGrid"],
Cell[5744, 144, 742, 23, 70, "Usage",
 CellID->29696]
}, Open  ]],
Cell[CellGroupData[{
Cell[6523, 172, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7001, 189, 999, 40, 70, "Notes",
 CellID->429991538],
Cell[8003, 231, 432, 14, 70, "Notes",
 CellID->16876],
Cell[8438, 247, 708, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[9183, 278, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->367127666],
Cell[CellGroupData[{
Cell[9568, 292, 147, 5, 70, "ExampleSection",
 CellID->62198871],
Cell[9718, 299, 269, 6, 70, "ExampleText",
 CellID->11493],
Cell[CellGroupData[{
Cell[10012, 309, 170, 5, 28, "Input",
 CellID->82445090],
Cell[10185, 316, 180, 6, 36, "Output",
 CellID->619335465]
}, Open  ]],
Cell[CellGroupData[{
Cell[10402, 327, 169, 5, 28, "Input",
 CellID->695],
Cell[10574, 334, 237, 8, 36, "Output",
 CellID->493334367]
}, Open  ]],
Cell[10826, 345, 121, 3, 70, "ExampleDelimiter",
 CellID->17213],
Cell[10950, 350, 253, 6, 70, "ExampleText",
 CellID->10032],
Cell[CellGroupData[{
Cell[11228, 360, 190, 6, 70, "Input",
 CellID->5525],
Cell[11421, 368, 390, 14, 36, "Output",
 CellID->96135653]
}, Open  ]],
Cell[CellGroupData[{
Cell[11848, 387, 490, 16, 47, "Input",
 CellID->32295],
Cell[12341, 405, 2034, 37, 163, "Output",
 Evaluatable->False,
 CellID->670966614]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14424, 448, 214, 7, 70, "ExampleSection",
 CellID->168],
Cell[14641, 457, 259, 7, 70, "ExampleText",
 CellID->28084],
Cell[CellGroupData[{
Cell[14925, 468, 205, 6, 70, "Input",
 CellID->20620],
Cell[15133, 476, 298, 9, 36, "Output",
 CellID->57205544]
}, Open  ]],
Cell[15446, 488, 121, 3, 70, "ExampleDelimiter",
 CellID->14838],
Cell[15570, 493, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[15694, 500, 171, 5, 70, "Input",
 CellID->19232],
Cell[15868, 507, 322, 12, 56, "Output",
 CellID->290900214]
}, Open  ]],
Cell[CellGroupData[{
Cell[16227, 524, 171, 5, 70, "Input",
 CellID->23566],
Cell[16401, 531, 398, 14, 53, "Output",
 CellID->38819615]
}, Open  ]],
Cell[16814, 548, 121, 3, 70, "ExampleDelimiter",
 CellID->17262],
Cell[16938, 553, 217, 5, 70, "ExampleText",
 CellID->18534],
Cell[CellGroupData[{
Cell[17180, 562, 342, 10, 70, "Input",
 CellID->29831],
Cell[17525, 574, 339, 13, 36, "Output",
 CellID->2428707]
}, Open  ]],
Cell[17879, 590, 119, 3, 70, "ExampleDelimiter",
 CellID->113],
Cell[18001, 595, 528, 19, 70, "ExampleText",
 CellID->48521834],
Cell[CellGroupData[{
Cell[18554, 618, 223, 7, 70, "Input",
 CellID->4591],
Cell[18780, 627, 159, 5, 36, "Output",
 CellID->210128162]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18988, 638, 223, 7, 70, "ExampleSection",
 CellID->17011],
Cell[19214, 647, 273, 8, 70, "ExampleText",
 CellID->32514],
Cell[CellGroupData[{
Cell[19512, 659, 251, 8, 70, "Input",
 CellID->10642],
Cell[19766, 669, 286, 9, 51, "Output",
 CellID->387094431]
}, Open  ]],
Cell[CellGroupData[{
Cell[20089, 683, 92, 3, 70, "Input",
 CellID->3834],
Cell[20184, 688, 178, 5, 36, "Output",
 CellID->184924216]
}, Open  ]],
Cell[20377, 696, 124, 3, 70, "ExampleDelimiter",
 CellID->86062964],
Cell[20504, 701, 253, 7, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[20782, 712, 341, 11, 70, "Input",
 CellID->26124],
Cell[21126, 725, 1286, 25, 70, "Output",
 Evaluatable->False,
 CellID->314288475]
}, Open  ]],
Cell[22427, 753, 120, 3, 70, "ExampleDelimiter",
 CellID->4015],
Cell[22550, 758, 279, 8, 70, "ExampleText",
 CellID->13063],
Cell[CellGroupData[{
Cell[22854, 770, 429, 13, 70, "Input",
 CellID->8117],
Cell[23286, 785, 2513, 45, 70, "Output",
 Evaluatable->False,
 CellID->10668035]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25848, 836, 233, 7, 70, "ExampleSection",
 CellID->11999],
Cell[26084, 845, 347, 10, 70, "ExampleText",
 CellID->6782],
Cell[CellGroupData[{
Cell[26456, 859, 210, 7, 70, "Input",
 CellID->23454],
Cell[26669, 868, 157, 5, 36, "Output",
 CellID->99852644]
}, Open  ]],
Cell[26841, 876, 121, 3, 70, "ExampleDelimiter",
 CellID->12012],
Cell[26965, 881, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[27224, 891, 341, 10, 70, "Input",
 CellID->8922],
Cell[27568, 903, 531, 18, 39, "Output",
 CellID->332031560]
}, Open  ]],
Cell[CellGroupData[{
Cell[28136, 926, 241, 7, 70, "Input",
 CellID->464069605],
Cell[28380, 935, 345, 12, 41, "Output",
 CellID->67698260]
}, Open  ]],
Cell[CellGroupData[{
Cell[28762, 952, 310, 11, 70, "Input",
 CellID->22174],
Cell[29075, 965, 531, 18, 39, "Output",
 CellID->221172999]
}, Open  ]],
Cell[29621, 986, 124, 3, 70, "ExampleDelimiter",
 CellID->45709179],
Cell[29748, 991, 606, 20, 70, "ExampleText",
 CellID->109559857],
Cell[CellGroupData[{
Cell[30379, 1015, 314, 10, 70, "Input",
 CellID->3178766],
Cell[30696, 1027, 227, 8, 36, "Output",
 CellID->113645830]
}, Open  ]],
Cell[CellGroupData[{
Cell[30960, 1040, 174, 6, 70, "Input",
 CellID->114635251],
Cell[31137, 1048, 644, 15, 70, "Output",
 Evaluatable->False,
 CellID->7337350]
}, Open  ]],
Cell[31796, 1066, 125, 3, 70, "ExampleDelimiter",
 CellID->307904394],
Cell[31924, 1071, 399, 12, 70, "ExampleText",
 CellID->100094394],
Cell[CellGroupData[{
Cell[32348, 1087, 224, 7, 70, "Input",
 CellID->308109750],
Cell[32575, 1096, 390, 14, 36, "Output",
 CellID->58693468]
}, Open  ]],
Cell[CellGroupData[{
Cell[33002, 1115, 277, 9, 70, "Input",
 CellID->343715506],
Cell[33282, 1126, 522, 18, 53, "Output",
 CellID->407752385]
}, Open  ]],
Cell[CellGroupData[{
Cell[33841, 1149, 351, 11, 70, "Input",
 CellID->662360908],
Cell[34195, 1162, 566, 20, 56, "Output",
 CellID->442704902]
}, Open  ]],
Cell[CellGroupData[{
Cell[34798, 1187, 323, 10, 70, "Input",
 CellID->592089036],
Cell[35124, 1199, 161, 5, 36, "Output",
 CellID->363382890]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[35334, 1210, 327, 9, 70, "ExampleSection",
 CellID->16589],
Cell[35664, 1221, 445, 14, 70, "ExampleText",
 CellID->8001],
Cell[CellGroupData[{
Cell[36134, 1239, 169, 5, 70, "Input",
 CellID->22015],
Cell[36306, 1246, 225, 5, 70, "Message",
 CellID->991842184],
Cell[36534, 1253, 263, 8, 36, "Output",
 CellID->121063633]
}, Open  ]],
Cell[36812, 1264, 120, 3, 70, "ExampleDelimiter",
 CellID->3029],
Cell[36935, 1269, 443, 14, 70, "ExampleText",
 CellID->9921],
Cell[CellGroupData[{
Cell[37403, 1287, 193, 6, 70, "Input",
 CellID->5987],
Cell[37599, 1295, 213, 5, 70, "Message",
 CellID->486241439],
Cell[37815, 1302, 286, 9, 51, "Output",
 CellID->132413162]
}, Open  ]],
Cell[38116, 1314, 120, 3, 70, "ExampleDelimiter",
 CellID->9488],
Cell[38239, 1319, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[38545, 1331, 314, 11, 70, "Input",
 CellID->16714],
Cell[38862, 1344, 176, 6, 36, "Output",
 CellID->64491663]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[39099, 1357, 311, 9, 70, "SeeAlsoSection",
 CellID->32122],
Cell[39413, 1368, 2384, 80, 70, "SeeAlso",
 CellID->12698]
}, Open  ]],
Cell[CellGroupData[{
Cell[41834, 1453, 314, 9, 70, "TutorialsSection",
 CellID->19958],
Cell[42151, 1464, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[42341, 1472, 305, 8, 70, "RelatedLinksSection"],
Cell[42649, 1482, 382, 12, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[43068, 1499, 299, 8, 70, "MoreAboutSection"],
Cell[43370, 1509, 177, 3, 70, "MoreAbout",
 CellID->53879317],
Cell[43550, 1514, 165, 3, 70, "MoreAbout",
 CellID->151638954],
Cell[43718, 1519, 160, 3, 70, "MoreAbout",
 CellID->42404269],
Cell[43881, 1524, 150, 3, 70, "MoreAbout",
 CellID->83074004]
}, Open  ]],
Cell[44046, 1530, 27, 0, 70, "History"],
Cell[44076, 1532, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

