(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    139095,       3271]
NotebookOptionsPosition[    130457,       2962]
NotebookOutlinePosition[    131711,       2998]
CellTagsIndexPosition[    131624,       2993]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ChiSquareDistribution" :> 
          Documentation`HelpLookup["paclet:ref/ChiSquareDistribution"], 
          "HalfNormalDistribution" :> 
          Documentation`HelpLookup["paclet:ref/HalfNormalDistribution"], 
          "RayleighDistribution" :> 
          Documentation`HelpLookup["paclet:ref/RayleighDistribution"], 
          "MaxwellDistribution" :> 
          Documentation`HelpLookup["paclet:ref/MaxwellDistribution"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"ChiSquareDistribution\"\>", 
       2->"\<\"HalfNormalDistribution\"\>", 3->"\<\"RayleighDistribution\"\>",
        4->"\<\"MaxwellDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ChiDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ChiDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ChiDistribution"], "[", 
       StyleBox["\[Nu]", "TR"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a ",
     Cell[BoxData[
      StyleBox["\[Chi]", "TR"]], "InlineFormula"],
     " distribution with ",
     Cell[BoxData[
      StyleBox["\[Nu]", "TR"]], "InlineFormula"],
     " degrees of freedom."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->21476]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["ChiDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiDistribution"], "[", 
   StyleBox["\[Nu]", "TR"], "]"}]], "InlineFormula"],
 " represents the distribution for the square root of a ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " random variable."
}], "Notes",
 CellID->21555],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " to be any positive real number."
}], "Notes",
 CellID->24917],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->12933,
  ButtonNote->"12933"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->55023376],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->504358159],

Cell[TextData[{
 "The mean and variance of a ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distribution are related to the ",
 Cell[BoxData[
  ButtonBox["Gamma",
   BaseStyle->"FunctionLink",
   ButtonData->"paclet:ref/Gamma"]], "InlineFormula"],
 " function:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->12933],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30137],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox["2"], " ", 
   RowBox[{"Gamma", "[", 
    FractionBox[
     RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{104, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->166364418]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3690],

Cell[BoxData[
 RowBox[{"2", " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox[
       RowBox[{"Gamma", "[", 
        FractionBox[
         RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], "2"], 
      SuperscriptBox[
       RowBox[{"Gamma", "[", 
        FractionBox["\[Nu]", "2"], "]"}], "2"]]}], "+", 
    FractionBox[
     RowBox[{"Gamma", "[", 
      FractionBox[
       RowBox[{"2", "+", "\[Nu]"}], "2"], "]"}], 
     RowBox[{"Gamma", "[", 
      FractionBox["\[Nu]", "2"], "]"}]]}], ")"}]}]], "Output",
 ImageSize->{199, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->217654494]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13694],

Cell["Probability density function:", "ExampleText",
 CellID->24668],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->32627],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{"1", "-", 
     FractionBox["\[Nu]", "2"]}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["x", "2"], "2"]}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Nu]"}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{94, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->102222496]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiDistribution", "[", "4", "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->989],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->97002615]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11647],

Cell[TextData[{
 "Generate a set of pseudorandom numbers that are ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distributed:"
}], "ExampleText",
 CellID->11149],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "10", "]"}], ",", "5"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30931],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.8326082094442593`", ",", "3.469610826619863`", ",", 
   "2.6538863273157887`", ",", "2.87395421903972`", ",", 
   "2.4423702314715507`"}], "}"}]], "Output",
 ImageSize->{308, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->509269069]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30875],

Cell["Properties based on higher-order moments:", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25598],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"2", " ", 
    SuperscriptBox[
     RowBox[{"Gamma", "[", 
      FractionBox[
       RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], "3"]}], "-", 
   RowBox[{"3", " ", 
    RowBox[{"Gamma", "[", 
     FractionBox["\[Nu]", "2"], "]"}], " ", 
    RowBox[{"Gamma", "[", 
     FractionBox[
      RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], " ", 
    RowBox[{"Gamma", "[", 
     FractionBox[
      RowBox[{"2", "+", "\[Nu]"}], "2"], "]"}]}], "+", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"Gamma", "[", 
      FractionBox["\[Nu]", "2"], "]"}], "2"], " ", 
    RowBox[{"Gamma", "[", 
     FractionBox[
      RowBox[{"3", "+", "\[Nu]"}], "2"], "]"}]}]}], 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      SuperscriptBox[
       RowBox[{"Gamma", "[", 
        FractionBox[
         RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], "2"]}], "+", 
     RowBox[{
      RowBox[{"Gamma", "[", 
       FractionBox["\[Nu]", "2"], "]"}], " ", 
      RowBox[{"Gamma", "[", 
       FractionBox[
        RowBox[{"2", "+", "\[Nu]"}], "2"], "]"}]}]}], ")"}], 
   RowBox[{"3", "/", "2"}]]]], "Output",
 ImageSize->{465, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66130545]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->24599],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "3"}], " ", 
     SuperscriptBox[
      RowBox[{"Gamma", "[", 
       FractionBox[
        RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], "4"]}], "+", 
    RowBox[{"6", " ", 
     RowBox[{"Gamma", "[", 
      FractionBox["\[Nu]", "2"], "]"}], " ", 
     SuperscriptBox[
      RowBox[{"Gamma", "[", 
       FractionBox[
        RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], "2"], " ", 
     RowBox[{"Gamma", "[", 
      FractionBox[
       RowBox[{"2", "+", "\[Nu]"}], "2"], "]"}]}], "-", 
    RowBox[{"4", " ", 
     SuperscriptBox[
      RowBox[{"Gamma", "[", 
       FractionBox["\[Nu]", "2"], "]"}], "2"], " ", 
     RowBox[{"Gamma", "[", 
      FractionBox[
       RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], " ", 
     RowBox[{"Gamma", "[", 
      FractionBox[
       RowBox[{"3", "+", "\[Nu]"}], "2"], "]"}]}], "+", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"Gamma", "[", 
       FractionBox["\[Nu]", "2"], "]"}], "3"], " ", 
     RowBox[{"Gamma", "[", 
      FractionBox[
       RowBox[{"4", "+", "\[Nu]"}], "2"], "]"}]}]}], ")"}], "/", 
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"Gamma", "[", 
       FractionBox[
        RowBox[{"1", "+", "\[Nu]"}], "2"], "]"}], "2"], "-", 
     RowBox[{
      RowBox[{"Gamma", "[", 
       FractionBox["\[Nu]", "2"], "]"}], " ", 
      RowBox[{"Gamma", "[", 
       FractionBox[
        RowBox[{"2", "+", "\[Nu]"}], "2"], "]"}]}]}], ")"}], "2"]}]], "Output",\

 ImageSize->{449, 93},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->236916505]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1443],

Cell[TextData[{
 "Third moment of a ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distribution:"
}], "ExampleText",
 CellID->1731],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "3"}], ",", 
   RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->18154],

Cell[BoxData[
 FractionBox[
  RowBox[{"2", " ", 
   SqrtBox["2"], " ", 
   RowBox[{"Gamma", "[", 
    FractionBox[
     RowBox[{"3", "+", "\[Nu]"}], "2"], "]"}]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{114, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->217944213]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4434],

Cell[TextData[{
 "The 0.75 quantile of a ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Nu]", "=", "10"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->9385],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "10", "]"}], ",", "0.75"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->23967],

Cell[BoxData["3.5424372114251197`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15945631]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->29866],

Cell["\<\
Plot the cumulative distribution function of the random variable:\
\>", "ExampleText",
 CellID->22667],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"ChiDistribution", "[", "15", "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10048],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWV1oXEUUvs3dTaykKiI0sVX6Xn3yQRBrlWraGtM0bWPa/LWbNW1ibTZN
07RNqya1pKIV/woaBAtqBX8eKkJFqCAWxIKkIAZEBYkg9cEiWHwICNfvnLl3
Z/bcs9PsFn/CZnZm5zvfnL85M3u3JTc60L8nNzqYz61YO5IbHhjM71vRVBjB
R+GiIAjm8H+yIUA/Qo//56II7+l2JdoQgEa8Z+J3Gi+nWepE/Ff8fBnea+J3
dz7DbHcoUsk4KMPqjuX8Uqa24yC4dUHrSpTUdmls7W1XXU9Khup6EtUQe8nV
g+y//R9aT/qxMV5/uaOHFlWdvSGllZRrcPh9fm9MMZl4rYxsFvn0W1YGZ1jc
PL5JSM54Pq1HO0vhSM2HVxnXVIivdFzkn5v7/eLML0FIzRKMPzv346k3v8br
1Ve+LAx9jE9Wr5ru6Xrv+NTnH7z/zQ/f/0bWZP8rbf+nfP+KvqkckntASt2i
zS82Y4T9k7PfIcb5vg8RXXRmv/01uEFILBbjSudvvEY+3e411GRhARJycOAM
Xue/+CmYUrFN1GRg2+HxT2EqsvyYilvHOKQ3cn7/6NnLl/88WsRRPWhm/IWv
foazaLVJlcWgrlyZx2rYQsGEimrhMeaxp6j7lIrawGMECerMz/8VPKmiWhO9
sCKhjqiojQkXUNQ9rKLaElSs13iJBzbz7PTrF06/c5G6h1QOg4IT4WnqHvSg
4GSEBG0Z1BYegwh01D2gotqFXmMeFDyEbKEiZlFkWwfP4nPoQ939KodBYasg
76g76kEhB5ApZJuO2spjFNvJiXPU3ceo+JjkP3c38zg5jRyWbTxGgiABqDti
sCXnYFbw6BzWdo2jVsiQbjTs5MELJ85/dGaWunsV2TohK2tjZ5ILHe3vUgZX
w9ElIjOscLDd6U9SLFaT4VRMauN8kedft1i/oMa8W/DrqB7BNVRE0cq9gmNI
5egVHHs8qEuX/gBXedR2HqNwoOxR9wkPymaRjtohuHaX2JYTHLtVDoPCNkZR
p+7jHhSWiSueDwU/wgO0V3VUH49RXlBkqDvoQdnIWBTZ9hjPorCivFJ3QOXI
Cw/oKMNla+wuD8p6wKBSFSZblIgnZG7387it9W2YRt2d0cLqg+SowYG3S5Gl
XZhRdmO/8Ke2riuT6E7DnSLP+hXZbKTvZSNrT51KZNm+EMFDCIsxKPG5rGTS
U4YB2xab1zLI2BoUzhiEhXyjozh7nOqcL6JId85PJ9fyKodB2d3W50HZm4MP
ZXebjuI96NSunAeFOyBO2VIU2cbVxzkXd6gcXF+cHeJDWa11lFnR3kK2e1Bg
QdzKo7gOO9nbW2IbV3y68cG80lmXw6BsJvZ4UDbbfCh4oHn9Kco2gyp3V6Gu
tjcKIq7dUZn97D2jCyLu1bEMCw93LZiFSIZF3pSTDhUv8O0mRCrh2KVupyIr
ZaT2e0XUOlPxqItzRZdEDJF/qB6xpIz3iIjUNg/K1meLopX51hvicosrLnW3
qhx863VqkI4yXHY3d3hQNio6im/2zr541INC2UTxLEWRbWOJF+OKoHOMiSi1
e1DggkrI6jKoA8K2LSrqoLBtswdluXTUIcG1qcQD44Jjk8oxLjjaPChbzwyq
0urC32idHb0xqqYuGBbEAUpTTlfHQt/B6bajSbs3BxI5IuLfWtWK/GzAubNs
qIqFn0M4e7YlFYly33+MpP0+36JG+mke2++8j6ioCR7bm3JzEUUrTwp/Nasc
kyKOD6uoo0Kf9SrqGR7bk32dijomfLfWg4JGSFRkvIMi26aSWUSRnsY1qRwG
heqNSxlxPKSijjsrQaXgQRX1bIKC4vS8ao2Keo7HmDcPxoIHVNTzUbyP4ydk
96uoE5yF5gkZzESI7ivxwYs8j0w2j5xxSt6r8rzEOKSIwaHC3KPiXqamFhYC
ah51IoxY9e7IboHkTz5V0DL9JDV1IEBCgAyU6MBo8stdTKY++/Xuu9eouR4m
QDW4GGczmOEceByhgy/ghTur5H6DmuvMQ37wIc4gxrm1etU02hp6wh+a4FP8
qWrxPR5o+kGgnn8aMFcVIrq5Ai0I8BbPQN7wwZoMr4CIYWVavJ41MArhFSyh
hnItiUGVds9ck9xCap5Bpn/nDBb9DfjFWbg=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->105770211]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13930],

Cell[TextData[{
 "A contour plot as both ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " are varied:"
}], "ExampleText",
 CellID->30811],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Nu]", ",", 
     RowBox[{"1", "/", "10"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19814],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJxsXQVUlM3XB5SwFQtQsRO7+4KB3YWifxAVsRWwG8FCULFFUREBixakl26k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   "], {{{}, 
     {RGBColor[0.33676591967233516`, 0.1327782030140431, 0.5903455562697186], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmFuMlOUZx7+Znfnmmx0IYrRKbCI0psWqSC9MvNilSVtbEFRkQcBdDmot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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllDtIlXEYxv/n6Pm+71zoRlBgUC5lF/UUFQ1l0FCWYnUKb5CDZYGSmkMN
mXSBtCA17UJlVkpBS6PV0igNbUE1BNUSNIlD5Njv4RleeJ//896+9/JVdvaV
etMhhEakHBlMQjheCOFzhKRCuJcPYRJZAXceg61w17Ih1MHvQ46B+/Dphf8E
bsC2ETmYCeEDb9exbYuJQZLBshCOwL0A12dscxffk8T4ju8f8l3NOodil8DT
uRCmkEr8+8td22V8hoJrrEAfI+YwsW4jPehfiV+H7WnwFfjt+HRE9tmJPsnb
IrG7edsNfgpeAg+Aa9FnyXcJ/2zKtY3zNh1cYwZcXvC3LRK/iD6QuFft8KP4
rgfXwXcToxb+HHgevhn+CXozbz8j1yD9MW9vgt8OUXuRHtTSqztp1z4E/yPl
byiBb4JHsf8Gfsf3vkdeUsthbF7DLdM34z+GfxP2LeAm7OdVH7ZrwavSnskI
+g5sOiPH3JD4G1R7j+YN3k89m7GtQlrxb8l7NifI+RfuH7JQ5h3Rt0/kPQv1
YBv4GXg84xhfYs9Is1GPWxPXqNpmiTmTc07lugi/BVxEUujQoT/xzmnX1FPV
fivx7ugbYmzLkEzKPkfhz8B34PsRvohezVtD5B5MJN4J7YJmrN7Xx+6dZqBe
LkeitHu6mtgrkd+6F+KvQa/IWVfOBfw3Em9X5J4ol3ZKu6Scbeg1vJUiz0S3
dCPr29BNdSWuWbXO4TOceOaatXbgPrke5nyLuok5am3Oexbq8YXEO6ld1E0q
V3viWMrZhf0B7PdgG/E2lfgGtPvawQ5iz8TeJd3QW2znkKXgHavG/jl4IuMY
ZxPvuHZbPY7Q44JvQzel21yHzUjGN1oFt6ng3OrRKXLtzXvXlEP/hleJd1//
iHG4X7F3QzGUuybxbqiGB+BHef8bNJP/NsqKag==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4234657590170055, 0.2835258090421294, 0.7122126688091559], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmXuMVdUVxs+9M/fcfe8ZX4iJTWF4CCggMlhLrWhQQDEoxQfxVSlahkdx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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlllsVVUUhk/v7e3d955boAwCUoUiM0hAREaDIgoixqHMMkSEgpGAIELC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         "],
         VertexColors->None], 
        PolygonBox[{{2705, 2086, 1419, 2964, 2704}, {2965, 2707, 2706, 2963, 
         1419}, {2707, 2708, 1018, 2549, 2706}},
         VertexColors->None]}]}, 
     {RGBColor[0.5101655983616759, 0.43427341507021555`, 0.8340797813485931], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHuwVmUVxr/b/r79ffsDzQOIoBYgGBxpYMxJDqLYZM4QCt7GLDSgxrJG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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmmMVFUQhV/3dNOv32sGHGQfQHaYAWVckEUWjUCCIyCCBBQENCoaGIYB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         "],
         VertexColors->None], PolygonBox[{{2715, 2087, 1137, 2719, 2714}},
         VertexColors->None]}]}, 
     {RGBColor[0.5881162706186073, 0.566601389124046, 0.9100526016828171], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1mHuQl1UZx9/d3+19f+/GBLrAKsZsOGkU/wSTMka0wDLjJRDQHZxJcFmn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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmlslVUQhr97by/3WxpEWcpSqmUX5Y80AgokZTERkB3UhK0tGkVFBQrK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         "],
         VertexColors->None], PolygonBox[{{2576, 1979, 1029, 2582, 2575}},
         VertexColors->None]}]}, 
     {RGBColor[0.6518607141765334, 0.6690209728415552, 0.9115057786603683], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmH+M13Udxz/f+3w/388vDrj6w1aEkn/gsqJUZjXnFh2shoKcAhIskKYW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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlllslWUQhr9z/p7+G8U2cQFEoHIhMQEUaFBjgpKWaJSlFZBCIwUXFAUv
2AtoMIgmGoNoDIYY5UKjMTGYGAWBG6UGQaBUdotRoYUaBWRTUBOf17cXk/PN
vDPzzT/fLKd69rMN84shhPVQGfR5EsIX0MxCCA9Cs9MQHoUeiUKYiOK0OIQv
wT/lvAzZWvjvwL/hvBMaz3kvsqE4ewD+9jyEpl4hVCA7if+3+T2GziGwI1Bf
sPYshFvQj8u5F/wA/lvx/zL4vtg+5WsYOhc4j8FnTSmEP5FdAj8FfwS+jXif
xNdv2I/irnHo70V/DzQyOMYdnF/CpgbdN6FX4V+BpnFuKjj2zfAfR/6G7/H1
QexcZMS0Hf/t8KvgN4B3odsJPY7/E9hsUm6gKnT7SwfdZeALwDqgVdgPQnYf
52bsV+TOiXKxgG/4EHwMOVlLLhYiuwwe8NeJr1Pg7yqPqe/qhO4AHw1NKvhN
dqP7Cfw76A7G9BT6i/H5EHdNgZ7LHZNiWagcoj8MfifYz9jXcD6O/2Pg7cjO
YTsKWS2+DiOrVX0gW1fmO3ZwPg6+nvs3IhuI/3vh16BfUfJbvYa/1ZHfbA76
HbHfpg56Cn5u5rPeRG/3OzSu6DdsQXdS5re/Fbwv/DD4oVAf1R7+90Bb8T0W
fBuypty10wy/Er0V0Bx8NeLjdfAlmXNxN/gcdCfCL5U+PpbDn0VnO9hZxQBW
hArQSOgn8BNQO7pzwfshewz90fDl+LwB7HqoDL4EDQTbiM4CsCpiaCYXM6EZ
3NcELeZbb4Tvxflp/A3Rt8PXRbap17fjr1r3IbuW83VQEb4AHc3dA6r9b+Hr
sD8NHcDXG9jvAp8Fv63kHJWlfmO97Rl0usCXgh8GOwRVwldB13DXvKJr+YXM
taqa7sR2UeZe/6jg2CbErjXFuIbz/NSxboGOwj+R+bwO/QFg9yBrjnzH++pv
fNyP7cXIs2IE/OXIM2Ma+o2pa1M1oVhvgu8dOeYiuhH0K/F1Q52clyTOZUXk
XmromT3qqenw3al7sxv+M7ArqXtbM0C1NarnrVVjg6DtsWupiP9fcvegek81
oN68hKyr5B6tRH9A5lrQDH0mcYyKrRWdkdw1InUtb+HOanxN7JlvNwf30l+J
c6ueeg9f/Yg5LfeM3J95ZmpWquen47sRmlxwj6kWyzPHqpqswldL6lkyA9mE
zDWvWl+BzWHOh6Dd6j3NAM6LEte2avI8tvW5Z63eRLVbn7jXVcNDuG8qsobI
M/a22DWm2tLOOILt/MS1ubPk2dea+Ns1A1uJpyHxbjkJXQQ/mDtXiqlNswl8
K9hwxZ9aNrxnv0zQ2yPrjlxDCXysmYDtOXQ2oftH6tpSjuYljkmxaCbsVq2D
Tw3uEeVaPateVc4Hg/XJ3cuaaR3E+0Pm3aEeW5l6J2oXaoaoN/unrj31qGbB
i4lnt2bCeGz3xM69vkmzVD2rXtVM7QK7mjgXk+FbsF+ZezdoJmlWt+F/V+SZ
PUW1k3gX6Q307c/nnu3KwY/gHalno2a6dlFL5lmonVSGbgWyKcE9NFNvlXv2
Pawdn3hGaDZoxx5MXFOqJe3o1YlnkmaReqBWvU+Ma7UroNNgLYlniXq2LrWO
MM24fzlvjr17ZXOe2C5AV9BvLPptKlOf9UbL0Z3XM1+0o7/m/FXq3pHsTtV/
7lrWTlZt/N/z5a6R/fD7Us9+5VD/RdbH3m36T6Jc9E797crJFXxdzb3LzmBT
BXZX7NpSjEX4WalnuXbkP+j+nVu3suT/Ct2JZ5/+M4yFr86NvRW8m/Xmemvt
6P8AY645Kw==
         "],
         VertexColors->None], PolygonBox[{{2562, 2907, 1026, 2570, 2561}},
         VertexColors->None]}]}, 
     {RGBColor[0.7156051577344593, 0.7714405565590643, 0.9129589556379195], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFV1lsVVUUve+9+3qnJlLETxWNYiKk0JZWopIaQUClKEOltBBlEEEZIkmh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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlm1Ml2UUxh/+/OF5cwu0PvShrFa2lcsXoFxlbWVqL1igiEquwt5FbG0l
oUD2otVaqzTdyvXBVTYtNdYHRSBXYQn5gmLGQKz1AbU186UoYa3ftasPZzvX
c53n3Pdz7nOu+7myeml5bSYIgvVYFlsQB0EVVpkbBDMgRsIguAxcg9+fEwQT
wYvAC+GnE98dBcHVo4JgUn4QnAefhM+HX0B8M/EHwA8kQdACbgDvwf8OW8T7
Ic92wveS4zG4hdgi/P1pEFzIOMcG+JXgZXlBMA1+P/w28Ifga1ivFr6NZ8XE
joWvIfcSbCYfcyP4Xfgj7Gcb/lRsDlwFNhl+PHgQfhm4i/18Bj4OvhDZ/wB7
AX+Q9fJzXYOH8R/BLub9Mazfjv8VtoX9zATvTl0DffvnPHuC3E9jM4hfx7MJ
+N/AjwYXYh34nazRSu5LwGPh23gW4YfYHPyd8Nvhn2MPL5L/L57dQb2fJ/9Z
4s9gPcQewV6De4bvXUzsLuwd/LexVfirsVvgfwYP4J9RjbQX8DD5+8AD+Mdi
n20vMRHr3UxMJfg0/Pv4/bG5Pmwo9J60F53RRfj17LcXf5j42eytm5iirGte
zbvl8Buz7qHF8APwd2d9ZpXw98Fv0LfDt4auiWoxj5w7VEv498BrVT/iXw99
1uuxWaxfit2Lf4r3z8JPBo/j3VKsIPUetbcR7ATv3kX/Nue7RzLkL058NpXE
r4GvI8ez5DqAfYvfB98K3wJuwF+O7efdBTw7qPMBV1HP3Xk+m2r2257nM9oE
PhZ5VlaAN4M7wKPBL2e9l1OJ19KetrP+KtZsBP+p8yP2I9ZYR75Xif8HPIGY
vfi/smZJ4pnSLKnHjsD3YDngCP4PuAOp96YZ60jc8+p1nclPmiWsAf/7jL+t
Hrszx99YAlfCforZS5POJPaZ6awqsH5iB6UJgWuksz2tfsj4jJX7E9Us6zVu
43vLQs/Om3xvF3w5/CS4tox7Uz2rXlWPngN/Aa5gL1VYPf5knhXlWmM0K7uw
5oxnpgb+ZOxeEF4CNwY8gv8o1gi/9P952Yq1R35HsV9i14WusWorTXwq8Yxp
tjTTbdRyduRZlwZ0k+sgVsL37NX8qd6xfWnMbPBG3llL7LVYL35t5L1IY/Tt
s4iZmHUN2sGlkbVYNbycXIWpe7eONbqkR/Dz4P7G9kTuKfWS9PWH1DOrWW3L
82z9jh3NeMbUW8PYv7nuMfVqQexc6lnNbkvs2miGt0eukWqzOeNZ/A37MeOZ
nMvaE1jzqjzXdCX4pcSx6nHNxlBirdGMfKyzjd3bmtlp5Nqa+uxvxwrhziXW
NmlQJ9y+1N+ib54qrU48C7oD1HsnIr+rHvya/OMSa5/umGbw+dhaOhfcSK6m
1NqmHn8cfi38G/BpjnutLLV2qOceAj+YWos0Uy3gqtRaoZnbwloFkWv/CjmX
w++Iffddj02PrGHSLt1hnYnvBOnNJqxM91fsWPWAtHx1aK1RPUfBp4n3Jk3v
gT8U+27bB56S+E7RXaI7qi7xjGm2pGFXJN6z9qo7Rr1YDm7MuielJYdj55Km
SGtujfwt0pxJqXtYvStNHZCeJdZe9cB4+HtC342649UL94e+q9UTv5D7eOy7
XHdQUeyeVb00w8Xgo5G1RxrzqfQntnZpj/rXOBT57PTPcQP8paFrpxqeSP3P
on8V3UFTePem2P8G0gT1+nxiKvLc89KioshaLE1aE1vjpG3qAWnFisTaLs3Y
KRz636FJmk+ut2JrkXI+SWx/6LteM9YDPpw4Vnf8ELnnx9ZKacR/vhtInw==

         "],
         VertexColors->None]}]}, 
     {RGBColor[0.7776454386698771, 0.8520049152225181, 0.9073045449308235], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJw1l3uQlWMcx9895+y+73nPDkZhRmEQZkzZUZ3dWKplXMq4hU1l3AqZ3P70
hyFMDBnjOmKSS2HMbq1VZsTWqVYuW1aJLkZyLbddNAYNZnw/vr/+eOZ53t/9
+d2e33v0tbdOuaWQJMkVdUlS1L6nnCRL0iSZrO+XK0kyrTFJFtQnSUdDkmwU
QYvgdwu+UDS79L1F9E06t4i3r5QkN4q2XftirdHCPyLhe7XeEd8KrSMkb1ee
JDXhHxJ+lmBXan0t2Ddak0S7Q7h1gk2WzI8l/5OydSN3g9Y4wScIv1XnSTo3
Sv+wBuO3i3+19mfqLAMa5NRk17aQy749eHeLfqPO3YKfLDk/5/6+XPs0rXE6
fyD9X+m8UDSdsmOFYEPFO1I652ndVrR+zofHAneKaKrCVaWjUrAvuzL7c7W+
75G8UXX223DRj+fOOq8Sz9N1lnl4wF+r8xmZ4yXndtkzVbSLStaFDSME3yz5
vZI9VvClgs0V31mCf59azwLhWrXGFGxHIt7T6u0HfLAIXfpeVm8fHyLeQxtN
hy+B40/8jI/vD7uwDzvBdwXNeeKbpLVK+GOk/1PZ/EbJd2fn/otT5xz5drD2
Zn1PE+967eNEPwef6DxSOXdkvWM9GPftLxhGjq0tGk5uvKh9omAnifdLyWwN
mXdpn102nNw7VbwHkiPSc7dw3UX7/ljBT0gce843CfZ8wb4bKvopBecyd8S/
+PkQnT+X7z/T6hO+TXomsshbwQZy80F7VL3pkY0OdN0l/FytK0vmP0m4w4hV
veMzpM45eFPFeUhttQace+OX9ogPcZoQ+QMcXz2hfWvZ8Jsl48uowdk6bxHu
XsF7hN+W+zxT8NN1HiaaLdo/yZ33PfhI+/GyeRd1obVTsJ3Cf5Eb91nJNYzP
F4T95Fyv/HBtxXdHD7rR1aW9X7hawXUwLHK+JvhqrU7xbxKus+B8wPc7oq53
hD7q+jrZcorkzpI9x2k/V7zN4vlHsi/I3FuQPTx6xbzQhX8+EO6GOufDHp0v
LzmnzxHfm7nz6U/tf+TW94JknyzcSMFfFmyszqN0/rDgfbZ41wh2m3z7hr6X
iOb7suFLxNuauQYnRh6Twx8KtjFzfWwW3yW58+sp8Y3P3EfoSTOiLw1qHy26
j0pRt3XuD8QMPvzMPabGXVZJxi2iXy45TbKhXfxV4c4V/CLpaCu4f9Nj8c/x
wq8T/QE6DxSdm42iaRDtD4L/WrBfv46+fXHq3oS+GfW+N/ECtyhsq4q3Kp4m
wfplw8zMvfmioKEnt0vXsyX3luXCt+SO/SU6r6qYNpWcH6XvN53Px2e5eWra
1+Tui9QRNcT70ivaptwx/jhzr6RPYsdMwXJ6UYN7Bzm5QfDrSua/T/tL0Tvx
I/cjP+m76OFeyO4L+b/Lttel7y/d4w7tK8t+N/Avd6Lem0v2PT2wU3KWNriH
nUHvj3phXxBnbFvS4B7FvjjsJC+xmXqkH/M+t0QPXBxw6Pim7/U0uB+Th+ik
B6CXPAaHH8jryY2GQQucM3F7vGyfURNnZ9a/IfyMH3sj9+nnozPXCbxztf8p
X2yO2qWG6Q+nZa6HMVE3PWHbj7nfLd4sbMTWhWFzZ5wHZMurvAdF+/Xp8O0O
yVyZuqZ/ylzD2ElePRP++UWwL1LzkHfdkXu7M9dDW8wRteD5Vvb8m/mdXCd7
L8v8TiJjTm459FTmG+zFPmYm4sHs0hFvU0fYz7kWMwBxuUFyluXu53siz7Dr
U93xMeH6i+5bfVGbg9HDkEO/3xbzDPeYEHehD5J3zCqt8ZbgZ/R3hA+B8ZYA
PzV3j6ZvkO8PF53z1PqaiuudnP5Z67vEvYD5htmGWuytuB5nyObpZfv7Ce3X
Cze/6D6wO+qc+Qt52EnMiT25xyy5L3OMuNPW6O2DoZe8Wis7/85cd8Rpe9QV
9VqMmm2W3nMqtg++gbCZnrQp+hLv4ZOp/fWeZL6be27l3svjrf8jde6St7xD
xIb+M0fwN2VDl84tkjOO2aXo2Qcf8DZ9hM9S28/9N0S/QXZz+I3ZhLmHWtgm
eWPL9gPvAXEZEvPG3pg5DhJNR24cduwJf04X7LvM+f982W8h8Z/NnJ5ZDm/U
9nin2oQ7M+aaMZnfMODUTTV37TwiGY+W3T+ZWZhdkMkbuKnid3CZeM8ou2/1
xvtH/TLX90XvvUz0l+aeSS8W7QjpuKDoOJ8Y/w7c76qi/xN2Cv5KblnUXnvU
90rBL00dx1kxw/FeUzfcc0j0w1rMfvgLedNFvy/sI3+Yu5jz9s912E59dIVf
iHt34OmfzIn4A13k0r7Iw07pPT3z+0CdPBy8B8S/DXefF/8CzMiD0eOZN4bH
HMz8fwxvVur+ybwyMmqBGbAx8vlmZr/c/wdN0XOhb4s7oBc68h+9+KA53pdj
xTcityxmA3SPCnpgyLmz3rVCTfNfx0yH3+hr+Iq86gz76TnEj7prixjV4t/w
PcX3/bK/X8/8fvN2vyvY/NQ0zCbDwgZqoBr2D4R+4luNeADHt9yRGJ1d8TxU
jXrATmrnGsGvrnjuWJq7PuFnBnwpt1+hXRH5QP1jF71xv9/xebNonysb/lbZ
cxj2Uv+7wif8Q8xNnWPMGuti3iBfeiJnuEcl7kI/mBI94bzUfZYe25f5Pwdb
Lkw9kzET3Cr5d6b2z/7cGhMzQHfIZ+bbG3IrxDZ37B8Q3/qy7SD21fDz8rAH
f/4/j8dccbtoH089z5wZ8we+/jX6HP2ZebA/4s6+Mc773zbqd75sf7DiPHo7
c61y9/8AErUO/A==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnto12UUxt/f77f5vfxGBGX/FEVmRqFG6i6WmnNlOKm8aw6snGnhNf8I
g/JSkKQSlhcSLANnFk1NTah0m5ulzrs5c1lzSmniLQvDogt9np79ceA87znv
7bznPOe9c+LMETOyIYSVSAFybxTCniSEhlwI9QwsTEPIIzn055Cu6F2Q6kwI
ReAx6DPx7wEuBl8D/460op9AHsuHMBgpZpNi1myOQ9iHTMX/Zvabiu90pBLf
VYxVFIXwLXgzvjWdQviCtY+BP8Ney9gk8B7wWvBS1ixCX8rYhIzPOAc8BzwA
PBZ8CXwCvBl8ALwXvBL8Fmutkx19K3f+E3wp57N3Yuxq1nd4SbFg7H30WdiH
cL4a8POsV8f51qP35j498D+IjMf/cfB7ig3+n2DvB+6NrRe4P3gv+92G/Vak
D/7FyP3YU9acjP5K1HFX5CHm1jLnjQKvMRv7m9h3YJuGvA3uhv2GAsegkViP
Ys6mwhA2Ihexf4T/5Y77VaHflfrtemI/D36ROS+g72OsRr7MuYJvJeuVoe9g
vTx7lzD2NbiN+XUFPkMj/j/oTbFPwr9/7BgrtjrzOvRejHXXfOQR9voO3Iz+
IHuOxD4a6YPvfOb3Yr2YPX7Dfpix49i2g1/Dtg18EXwcfIi994Mnsl513rHs
TD6tSLyH1l6Cz278b4wde+XMAt03ceyUY/Oxz1U82G+C8gPbbqQ04xpowtaI
9A3OgVHMv4b9D2xng2M7L/VcxbgdfQljjegNBX6LCP9fs36TXZxtDGMjsh4b
gL6BOZuyrpHRsd9Qb/cpUo/+L2M/Y6sFN6CfQg6BtyE7sY8ED8s6Z2ex9xnw
kQ57Cfg665/kLE9x3rHoT0au1RJkBnpp4rfdiv8F5l5EBqJ3C3777bHvqhwY
hO969lzFWQYxp3veNalavJ2xs/ieQ8rB9zCniv3Gp9bL2O9l9OGRa3NcgXNP
b6q3VA4OVu4y1pD1GcqwPQuuKjRnlIJLUnNHrWKGvov1pmScUz3wHRo5F+5g
zlfoO2PHQjFT7Cojn1W4KXLOKFdUQ9/H5jxx3X7s7eDTse93E3u0pOYw7S9O
OoreFrv2d2mP2DFWbHUHvcWW2GfVm5xKHWPFVjXUin4idu2LQ07GrlHVpmqq
Hv8W8MKMOUXccRCf+qw55EpszhZXtzL2D/iByFx3LuvcUcwVa+WQ3u5D5N1C
v6FqrRT/p3OuuU34d4bjQif79E0dU8VSfPQX9r+RIVnXxDRs0/PO9dWccRr+
i1ivLmdOvwy+krqWlfOqtcX4zy10zW2JzMHiJtVUdeweot7RgM8RfA8jTRnn
sLj6m9ixFmcfQO8SmRsUk53od6fmGvWoueAvE3OLelTXyDFTrJ7QGbnr0CJz
uTjqWOQepN7zesax/yV2b9Ib3AeuSsztpcFcMi51botTBibOOeXWQPCjnH1K
5Nzpx302sFYFPhXYynPWN8auJY0tAy9HFufcE9agf5A49sq5jyNzuuIljn4m
b05QfqtGhmEbnvhuqkH1vuXMeTjjHqjeVxe71tUDl2G7kJr71TPPYD+dmCva
sF/HvoA92tGPZlwL5YnvpppYjb09sa98binynXVX5dDnse+su4rjFmErSfwX
UA78yN4/pc5V5dR5bGsj957/e6zulzf3ieNWYGvKmxt1pxb9DxK/xQHWfEf5
kVhXj5ivXpn4LXtm3EvmRdbVU16N3HMV3w3IYnBz4lxWD7iqt8w7F5WDyu3Z
Hf1DOS6u1x9BfwNx/uTIHCLu0BuOSr2H1laP+w/0r1G7
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8348922753883199, 0.8710952098721363, 0.8816580133817192], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl2mIlWUUx9+73/u+VwjXagi0EhwjKj80jblLZk1JRkXYRouGUWoLon3I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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlWtolmUYx+9tz7v3fe5nQWDaMII8kdKHcoG2qXMzqK2paFDkqchyHtLN
UiLqg5ZFMhOD7DDLpmjWoLnNOQ9Q83WDLLSDSgcqyLStkdmH+lhBvz//Plxw
ne/rvo4TVrbe31IaQtgGJMDFfAhrshAezYVwDMbXMYSJ8MYh6y4L4ftCCD8C
5zD6C96yihC60hCakV0tD2E8uj3YtGHbic7r0N8g/wr5BaAP2yPALSUhrAMm
IR9A3ofsBDa1yN6E3gX9MvRP0GvQ2Qn9Zanxw/jfnph3N/JfgAvIXoV3L7HP
wH4svmv5Qyv6b6d+S28Oo7sU+2p0L2JzpeAY9LZ8zOE/i6Bv429t/Gcz+Grs
V/HWXuAp6Heh1+NrArAX/Mm8cb1xFN/z8FmDr7nodyCfAW8M/tuhLyM7CnQh
Pw+8iO1mdFqQHQOuoHsZGER/CKhDVp/672vxX0R2CrgLvAHekN6CvhHd49Ab
8VcsWFf/PQH+BLybwNuRf4qvM8BU/jfAe/OwncSfp5S7huPAxwJ7lAvyNx37
/eT0NfAPVA/0b0Z+Q7lj2IHvT1L7Ksof+OGC33oHqES3Hd5uZFux/x376djU
8P55eJPBFyNfUOaYzoLvhFeN7w5gU945U670vwbwkYJr9wb+DiL/FZsR7EeB
+yocs2IdIMZeYj8CdORc063q7//r2QN8Rzzb4PWDn8TmffCp8K4HvwrvW/CP
Uuseh1eF/A94X4A34W8Vsn2pa6seWML7hzL/fZT3F0A3Qr+iekKvBV+X+a3t
6N8J/hz2T+v/xNiL7271Z6lnSLmv4j/PJq7BYHQPqPYd8IrEMxE6A78uca27
omdHNe+EvpY6NwtL3CtNmufEPVOneYzuVb25B923UveOaqbYlBPlQjFqFmrV
A4lz+BuyPLzF2PZhUx2dE+VCbzyv+Y/eJbfD6wPvBGaWOieT1X/Ru0R/aOW/
G4HmnHtgDrLHoJdrlqGvQT9IzL0516CBt2bDq0w8k7eCP4L+UuTNJZ6lhegf
yHmmWpBdIqa6cr9RX3CNVBvNbDX0fOgidBV0I3hThXvpmcSys9G9LB3tqnsy
7xrtrEvINkA/Dt2EznuqFfTcnGuoWP4Bhksdk3pvEz42lLkHG5H9CfyAfL9m
DttOYBe2lepf/G2JruUK7WfsT6feHbMTz8IBzUSZZ0KzPT9vW834OXx/XnAu
tdOnRedMuVLPz4qeOc2acqrdvTvvWdUOr0dep30T3AP/av9G7yrN9Bl0D6Xe
leMT17qm4Fyq5i9oP6Xu/TuAEeTD0bOuPytXU6J7WTn7Gfyz1LulCO9k9I7W
btbOfADfg5lvi3rwb+jT0bOhHOuWaYY1u7ppmv0ted8y9e8p6I9Tx6IbtCj1
jtFu0U4ahT4YXXvldH3mmBSLajwz+qZp92vH9xecA/1dO34J8oeie/8loD/6
ZulWqabdBe8M7Qrt3Icz31DdTvXwh6lvrG6rcq5eWY6PWYl7ZmXmG6B6aUba
oHdk7g3dwGXRMeht3TztbvWoelM7fCjzDGt2NVM9mXtQvaeduSK6x9RbelO3
d3Xet0E3+D8H6hRh
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8921391121067627, 0.8901855045217544, 0.8560114818326151], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFlclvlWUUxr9Lvzt836Xtwg3FiVK81AQX1IWItSmtAjWQGLeKJkRTMaal
UBXjAltdtCpsDLTSqsWxE6WtHZTEASQxkThsTIwr47RiZfwDfH48x7g49573
fOc943PO23y4/5G+DUmSfCeqE03q50KaJEcLSTKTJ8knomGdVypJ8ptoq+Qt
5ST5XXyT9K/p+4GN0pXOPt3dKypJ3ir5x9I7KP3VLEleFD+jbxXp3qS7/xST
pKeq7zp3ij8n+lSBtIvGxJ+WvVOiIRmriepF10XTstFLjLL5qO6e5yx+UPZb
Co77Xcm26f+i5Iui93We1v9M5m+jstMqOiJ+RP/bRU+Ln9P3WdFUnWNFdrNy
OS7b9+XOfV3fn818t1ux/iJ6TfIPJfsgc/3apbs/cy0mlMucdDo2ONbborZX
xH8t2iH7V+t8vkv8Wsk88nHpjom6wgf/t0g+VbRf/F+iN9LfnFg2HvqD8v+q
4m7TvePiXxG/s2Dd5dA/Jdm1zHHUCpbT04cU/6x6dKv4OfVotuo8DlBj1fx2
+Rgo+J86z+j7dNU9JOem1LV6WPr3b3QOUwXnSY7L8vlC2T475WtP7rpsTV13
aj5UdM1OSGew4L4T80+6d0j6o5ItyM5zOt8hvk80UWcMdEe9qGGP8lisWgZ+
dqe2g903Co5nITVGyAksgrtN4h+QnwdFzanriE9ioz8n4i44wBd9+VG6P+SO
d1n/nfL9uu7Op46NGL9RvKsV83uls0/Uou9PKcYnRYeKnj9wCk6aoy74ZQ6a
g6fHYxHXlvT/nhwsu3/0Dt+jol3S/UM+uyueL/xtSx3zt9Lfn/vMfepAPQ8X
jS3q9YziOlK17Ky+j4nE3ugP2KaOc9Ibj9mfFX8mdAZk/+3MdxblZ3PZe4Y4
mHd8vceMZq4R2FnbYPywb5g9eOy9GTbxczZ8YWNLxH5R9pdy53xFdy9XbOty
ZtzQP/wdKxjntcAWeNhd8b5j1/0q3b6qMbtLTu4pWc6OGxKliTFyOmr7PXsy
s90u2VnJ/Y16r+bOlbtnir7/UWYc9AWuwSUxsBNfLrp3J4veUfDsIXZUW+wA
eHr3lewMlz3TTyje7fLVKPvrJdsBo8juDDm9vBT4r4+9CpbQHQr9edm5IJrU
+U/d+0t0NXWdB6J+4GckduZbimGTZjzVeVz8ucx96si9l5lHME2e1JqdPZ95
F7JLunJjl9rVoi/MFLkzV+djDtgz7YEXdF6Sjburfl94Q/ZUPLtfZH6fmmJO
2afkxVvEHfTBIhgFS9wZi5mfDF/ETE1ORi94DxYCY8SEnN0Ctpht+kgP87Jr
PBW7hJjZh8xGKXpKPNytK7lmYAl8gTN2DjjjDE7QXY/4eWPvrRhHE8qjv+yZ
YK//nHu3EzP6+HkHPJY9K0dVp35Rb9ExEy8z1RAx854uZcbxsdg5YAP5Sua9
dmNOpPt41fNL/1tih4BHcAk+eYP+e6/+rnj/snvxNRTvOP8NYf/5st8Dek/s
axF/Q+iS12O537CdEXNj7HHe4Z7ceOmIPUSteHM+y70fwfVkvCXk2hB3wddo
zBfvHxjfEbPWFnt+WbkuVf3etkZc5EUdecvYD58rhi8zv53Yvx55NUaOxFyL
nHvj7a3FzmEfDMdOYTfVxzzyz5m71HYk5v1fLjgO2w==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlMlvjWEUxt/be6/7vd8nsSBBULOQ2LAyNNRYTUjEGk0aQklapKYdYtMY
NkJLi1JTe1ta2hoScyOhxBZbwxrxB/g9HouTnPOe55z3zDPrm7Y0VoQQ6qAC
dKIUwtMYwv18CIPQcBpCOQuhvRjCCoArE3S8nQK8BLkG7BtsJiOvBz9lbAif
0S9DHjcmhJfgp6LvRR4G3w3+NjQnF8JV8KvAroT2IVeCWQf/Fvws+JlQB9j9
yK1gL0CP0Tfzthh8K/rVyA3//U1HHuK/QWgWciO0F916MLPR7cZ+CL6bfC6R
TxXxPEBeQMxLiPUEmCn8NSpMwfX4AH8FH23Ylnl4j9yOfB65B3kU/jtvI/Cv
eevH932oE/9r8H8Sfy/BtOQc8y9i+51YtxD/q+Bvgm8FX110ra6Db8+7Zl3w
16DZOb8tBd8F/nzRMU4Dfye6ln3Ie+DXpq7dLt7WwG8suTYzoB3YHpENum34
qEH/DnlOwTVS7Yfw0Z13DxrAfwHzBL4e/Db4rdCinHOqonabsJlGLi3UsB/b
u9BcdF34qER3L5q/i4938GdKrtUrqAG+NnPuk7AZSRyzYn2k/oA/BGYeugPQ
drB10Dj0mP/LRTkrV+XUl3lmNasd+NwAP4x+anCOPejnYlOu8Ez3oqtOPXvX
sB9A/5MYvlS4hz9S90y9Uo/PlpyDYh/RDETnqNw68+7NssSzpB4dLXkGNXuq
6YGSZ1qzrJnSX0/4o63gPydSz4voC9QyD/WgT3j7U3SNm7BvQ38u7xm8ET0D
6n0H8nJ8VaXeRe2AfG8ueVf0R230zkpfA31EvzPzLDSDvxQdg/7WH4Opd167
frbgWXgRfRs0E7WpZ0C9b+Bt0ljvkHZHOWi31GP1Vjv2HNtn0OTgG3MQ3UB0
bzUzA/jrT93bloJvyZ7Ms6ebUk0s91LrdIN0O1ak3g/dkNNgJ4A5Dv6Y+h+9
w+X/9+ly9A5rd3VTdLteJb5NumEL0M2HNuU8Y4fRD0bXRjPYlHkG1PtdRe/u
6sS10Q7rtpWjZ0E3SbH0Zp5FxTQ+cYyKTT39lPpm6lZqp5rxfSv6dikn3ba+
6NmUv0Zs92X+uxP5Tuad0a7opmnWH0b3QjP/Fd23xLdONdctuRk9K/L/F1SB
vXE=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{297, 374, 13}, {2465, 2467, 922}, {1912, 2916, 295}, {
         2742, 2989, 1169}, {2485, 1993, 373}, {375, 1061, 297}, {2242, 1990, 
         365}, {2987, 2742, 1169}, {258, 272, 233}, {573, 233, 30}, {2630, 
         2411, 296}, {1169, 573, 30}, {245, 258, 232}, {272, 258, 15}, {14, 
         297, 13}, {2410, 1912, 295}, {2478, 2480, 930}, {13, 296, 12}, {268, 
         258, 245}, {233, 573, 232}, {2471, 2473, 926}, {657, 1062, 375}, {
         2988, 2068, 525}, {1829, 2986, 30}, {245, 232, 14}, {2486, 2632, 
         938}, {525, 657, 232}, {2411, 1913, 296}, {1202, 935, 11}, {2956, 
         2067, 525}, {935, 933, 11}, {1199, 932, 10}, {573, 525, 232}, {938, 
         374, 297}, {657, 375, 232}, {1910, 2483, 934}, {3080, 2218, 657}, {
         1869, 3132, 1061}, {932, 930, 10}, {933, 934, 10}, {1201, 1200, 
         295}, {258, 268, 15}, {1990, 2375, 365}, {3132, 2633, 1061}, {934, 
         1199, 10}, {295, 1200, 12}, {2487, 2486, 938}, {2068, 2956, 525}, {
         2479, 1822, 9}, {2375, 2377, 365}, {2628, 2410, 295}, {573, 1169, 
         525}, {1993, 2629, 373}, {296, 373, 12}, {2472, 1821, 8}, {2469, 
         2465, 922}, {1201, 1202, 11}, {1994, 2631, 374}, {232, 375, 14}, {
         258, 233, 232}, {1062, 1061, 375}, {375, 297, 14}, {1200, 1201, 
         11}, {1822, 2474, 9}, {1821, 2468, 8}, {2475, 2471, 926}, {2484, 
         1911, 935}, {373, 295, 12}, {1820, 2241, 7}, {2481, 1909, 932}, {
         2917, 2767, 1201}, {374, 296, 13}, {2466, 1820, 7}},
         VertexColors->None], 
        PolygonBox[{{2634, 1869, 1061, 1062}, {2767, 2768, 1202, 1201}, {2218,
          2634, 1062, 657}, {2412, 2487, 938, 297}, {2633, 2412, 297, 1061}, {
         2241, 2242, 365, 7}, {2468, 2469, 922, 8}, {2483, 2766, 1199, 934}, {
         2986, 2987, 1169, 30}, {2632, 1994, 374, 938}, {2768, 2484, 935, 
         1202}, {2480, 2479, 9, 930}, {2916, 2917, 1201, 295}, {2473, 2472, 8,
          926}, {1913, 2485, 373, 296}, {2989, 2988, 525, 1169}, {2067, 3080, 
         657, 525}, {2474, 2475, 926, 9}, {2629, 2628, 295, 373}, {2631, 2630,
          296, 374}, {1909, 2478, 930, 932}, {1911, 2482, 933, 935}, {2467, 
         2466, 7, 922}, {2482, 1910, 934, 933}, {2766, 2481, 932, 1199}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1829, 2986, 2987, 2742, 2989, 2988, 2068, 2956, 2067, 3080, 
         2218, 2634, 1869, 3132, 2633, 2412, 2487, 2486, 2632, 1994, 2631, 
         2630, 2411, 1913, 2485, 1993, 2629, 2628, 2410, 1912, 2916, 2917, 
         2767, 2768, 2484, 1911, 2482, 1910, 2483, 2766, 2481, 1909, 2478, 
         2480, 2479, 1822, 2474, 2475, 2471, 2473, 2472, 1821, 2468, 2469, 
         2465, 2467, 2466, 1820, 2241, 2242, 1990, 2375, 2377}],
        "0.9`"],
       Annotation[#, 0.9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2016, 2926, 2928, 2927, 1934, 2506, 1933, 2507, 2789, 2790, 
         2289, 2290, 2121, 2992, 2994, 2993, 2288, 2746, 1872, 3133, 2745, 
         2420, 1924, 2497, 2005, 2653, 2652, 2419, 1923, 2776, 3099, 2777, 
         2922, 2921, 2778, 2496, 1922, 2493, 2495, 2494, 1827, 2275, 2276, 
         2115, 2741, 2985, 2984, 2244, 2627, 1868, 3107, 2626, 2409, 2477, 
         3163, 2476, 2624, 2625, 2623, 2622, 2407, 1908, 2470, 1991, 2229, 
         2228, 2274, 1907, 2619, 2620, 2376, 2765, 2462, 1906, 2460, 1905, 
         2461, 2764, 2459, 1903, 2456, 2458, 3095, 2457, 3096, 1819, 2763, 
         3148, 2105, 2979, 2106, 2729, 2739, 3113, 1897, 2889, 3149, 2113, 
         2981, 2114, 2740, 2111, 2738, 2112, 2735, 2737, 2736, 1818, 2910, 
         2911, 2599, 2601, 2600, 1959, 2427, 2867}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1847, 2312, 1842, 2798, 2797, 2134, 3011, 2026, 2311, 2132, 
         3009, 2133, 3010, 1941, 2518, 1940, 2519, 2796, 1841, 2301, 2788, 
         2787, 2786, 2126, 2300, 2299, 2753, 2754, 2752, 2751, 2422, 1932, 
         2783, 3101, 2784, 2925, 2924, 2785, 2505, 1931, 2502, 2504, 2503, 
         1833, 2286, 1828, 2385, 2384, 2120, 2278, 2277, 2650, 2651, 2649, 
         2648, 2417, 1921, 2774, 3139, 2003, 2920, 2775, 2492, 1920, 2408, 
         1919, 2491, 2773, 1826, 2252, 2253, 1992, 2621, 2915, 2914, 2243, 
         2618, 1867, 2861, 2617, 2405, 2464, 2463, 2614, 2615, 2227, 2226, 
         2273, 1904, 2240, 1988, 2368, 2369, 2272, 2271, 1896, 2734, 1895, 
         2885, 3094, 3093, 2223, 3161, 2982, 2983, 2370, 2222, 2371, 2221, 
         2232, 1899, 2733, 2110, 2455, 2886, 2887, 2358, 2954, 2955, 2686, 
         2688, 2687, 1960, 2539, 1961, 2541, 1956, 2535, 1955, 2536, 2724, 
         3145, 3147, 3146, 2097, 2977, 2098, 2725, 2888, 2597, 1984, 2594, 
         2596, 2595, 1883, 2266, 2268, 1985}],
        "0.7`"],
       Annotation[#, 0.7, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2038, 2944, 2039, 2945, 1948, 2527, 2529, 2528, 1851, 2322, 
         1846, 2807, 2806, 3024, 3023, 2032, 3022, 2147, 2804, 2805, 2310, 
         2025, 3008, 2131, 3005, 3007, 3006, 2302, 2681, 1873, 3134, 2680, 
         2424, 1939, 2794, 3141, 2024, 2929, 2795, 2517, 1938, 2514, 2516, 
         2515, 1839, 2297, 1834, 2395, 2394, 2125, 2750, 2004, 2287, 2014, 
         2667, 2669, 2668, 1930, 2418, 1929, 2501, 2781, 2782, 2282, 2283, 
         2119, 2744, 2118, 2991, 2254, 2647, 1871, 2863, 2646, 2415, 1918, 
         2490, 2001, 2643, 2645, 2644, 1917, 2406, 1916, 2489, 2771, 2772, 
         2249, 2250, 1989, 2616, 2913, 2912, 2239, 2270, 1866, 2860, 2404, 
         2759, 2758, 2064, 2731, 2108, 2980, 2107, 2732, 2109, 2760, 2454, 
         2237, 1902, 2359, 2360, 2236, 2235, 2233, 1876, 2548, 1970, 2540, 
         1971, 2258, 3108, 2257, 2269, 2727, 2099, 2726, 2100, 2264, 2263, 
         2356, 2357, 2101, 2451, 2452, 1894, 2267, 2695, 2571, 2572, 2448, 
         2449, 2598, 1986, 2453, 2884, 2265, 1977, 2613, 1987, 2607, 2762}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1856, 2334, 2833, 2832, 2831, 2170, 3048, 2044, 2333, 2168, 
         3046, 2169, 3047, 2037, 3033, 2158, 3032, 2157, 2323, 2684, 2685, 
         2683, 2682, 2425, 1947, 2942, 2943, 2823, 2824, 1850, 2321, 2803, 
         2802, 2801, 2146, 2320, 2319, 3019, 2145, 3018, 3021, 3020, 1946, 
         2524, 2526, 2525, 1845, 2308, 1840, 2403, 2402, 2130, 2757, 2015, 
         2298, 2022, 2678, 2023, 2679, 1937, 2511, 2513, 2512, 1838, 2295, 
         2393, 2392, 2391, 2124, 2285, 2284, 2664, 2012, 2663, 2666, 2665, 
         1928, 2416, 1927, 2500, 2779, 2780, 2279, 2280, 2117, 2743, 2116, 
         2990, 2251, 2642, 1870, 2862, 2641, 2413, 1915, 2488, 1998, 2381, 
         1999, 2640, 1914, 2354, 2355, 2246, 2730, 2245, 1824, 2689, 2690, 
         2225, 2224, 2238, 1900, 2537, 3162, 2538, 2894, 2893, 1957, 2545, 
         1968, 2438, 2555, 2905, 2906, 2446, 2879, 2547, 1969, 2897, 2546, 
         2866, 3164, 2865, 1875, 2978, 1874, 3135, 2728, 2426, 2065, 2261, 
         2260, 1887, 2908, 1980, 2591, 1981, 2592, 1978, 2907, 2562, 2612, 
         2606, 1817}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1859, 2343, 1858, 2849, 2848, 3068, 3067, 2052, 3065, 3066, 
         2846, 2847, 1857, 2339, 2840, 2839, 2838, 2181, 3062, 2048, 3061, 
         2180, 2836, 2837, 2332, 2043, 3045, 2167, 3044, 2166, 2331, 3041, 
         2165, 3040, 3043, 3042, 1951, 2532, 2534, 2533, 1854, 2327, 1849, 
         2822, 2821, 3031, 3030, 2036, 3029, 2156, 2818, 2819, 2820, 2317, 
         2318, 2144, 3016, 2143, 3017, 2309, 2939, 2031, 2938, 2941, 2940, 
         1945, 2522, 1944, 2523, 2800, 1844, 2306, 2401, 2400, 2399, 2129, 
         2756, 2013, 2296, 2020, 2676, 2021, 2677, 1936, 2508, 2510, 2509, 
         1836, 2294, 1832, 2390, 2389, 2123, 2749, 2002, 2281, 2008, 2661, 
         2009, 2662, 1926, 2414, 2499, 2498, 1831, 2255, 1825, 2383, 2382, 
         2000, 2248, 3100, 2247, 2637, 1997, 2638, 2919, 2639, 2430, 2429, 
         1880, 2234, 2872, 2871, 2428, 1964, 2769, 3097, 2770, 2635, 2720, 
         3098, 1823, 2721, 2088, 2374, 2373, 2372, 1898, 2557, 1976, 2556, 
         1975, 2444, 2443, 2440, 1972, 2553, 2066, 2259, 1958, 2366, 2230, 
         2231, 1979, 2576, 2590, 2586, 1892, 2566, 2570, 2561, 2611, 2605, 
         1816}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1862, 2349, 1861, 2859, 2858, 3074, 3073, 2058, 3071, 3072, 
         2856, 2857, 1860, 2346, 2855, 2854, 2853, 2201, 3070, 2055, 3069, 
         2200, 2850, 2851, 2852, 2342, 2845, 2844, 2843, 2191, 3064, 2051, 
         3063, 2190, 2841, 2842, 2338, 2047, 3060, 2179, 3059, 2178, 2337, 
         3057, 2176, 3056, 2177, 3058, 2042, 3039, 2164, 3038, 2163, 2328, 
         2947, 2948, 2946, 2950, 2949, 1950, 2530, 1949, 2531, 2830, 1853, 
         2326, 1848, 2817, 2816, 2155, 3028, 2035, 3027, 2154, 2813, 2814, 
         2815, 2315, 2316, 2142, 3014, 2141, 3015, 2307, 2935, 2030, 2934, 
         2937, 2936, 1943, 2520, 1942, 2521, 2799, 1843, 2304, 1837, 2398, 
         2397, 2128, 2755, 2011, 2675, 2018, 2396, 2793, 2292, 2010, 2748, 
         2122, 2747, 2996, 2995, 2256, 2659, 2660, 2658, 2657, 2421, 1925, 
         2655, 2656, 2386, 2691, 2692, 2074, 2380, 2379, 2378, 1901, 2544, 
         1966, 2761, 1965, 2696, 2697, 2439, 2698, 3109, 1884, 2699, 2077, 
         3084, 2220, 3083, 2219, 2362, 2069, 2361, 2070, 2554, 1973, 2445, 
         2877, 2878, 2262, 2442, 2441, 2552, 2901, 2900, 2898, 2447, 2899, 
         2880, 2881, 2364, 2365, 2087, 2715, 2971, 2968, 1879, 2579, 2582, 
         2575, 2589, 2585, 1891, 2565, 2569, 2560, 2610, 2604, 1815}],
        "0.3`"],
       Annotation[#, 0.3, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3130, 3131, 3090, 3092, 3091, 2102, 2104, 2103, 1888, 2082, 
         2081, 1882, 2061, 3116, 2060, 2353, 2215, 3129, 2216, 2059, 2213, 
         3128, 2212, 2351, 2210, 3160, 2211, 2057, 2209, 3127, 2208, 2348, 
         2206, 3159, 2207, 2054, 2199, 3125, 2198, 2345, 2196, 3157, 2197, 
         2050, 2189, 3123, 2188, 2341, 2186, 3155, 2187, 2046, 3054, 3055, 
         3053, 2175, 2336, 3051, 3052, 3050, 3049, 2864, 1952, 2951, 2952, 
         2834, 2835, 1855, 2330, 1852, 2829, 2828, 3037, 3036, 2041, 3034, 
         3035, 2825, 2826, 2827, 2325, 2812, 2811, 2810, 2153, 3026, 2034, 
         3025, 2152, 2808, 2809, 2314, 2029, 3013, 2140, 3012, 2139, 2305, 
         2931, 2028, 2930, 2933, 2932, 2019, 3003, 3004, 3002, 2127, 2293, 
         2673, 2674, 2672, 2671, 2423, 1935, 2791, 3102, 2792, 2670, 2693, 
         3103, 1835, 2694, 1830, 2388, 2387, 2437, 3136, 2436, 2701, 2700, 
         2961, 2078, 2873, 2874, 2875, 2291, 2973, 2089, 2972, 2090, 2974, 
         1886, 2636, 1996, 2918, 1995, 3138, 3143, 3142, 2083, 3088, 3089, 
         3085, 3087, 3086, 2363, 2071, 3081, 3082, 2868, 2870, 2869, 3137, 
         1982, 2909, 1983, 2593, 1963, 2896, 1962, 2895, 2550, 2708, 2707, 
         2965, 3114, 2086, 2705, 2712, 2710, 2085, 2717, 2719, 2714, 2970, 
         2967, 1878, 2578, 2581, 2574, 2588, 2584, 1890, 2564, 2568, 2559, 
         2609, 2603, 1814}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1813, 2602, 2608, 2558, 2567, 2563, 1889, 2583, 2587, 2573, 
         2580, 2577, 1877, 2966, 2969, 2713, 2718, 2716, 2084, 2709, 2711, 
         2704, 2964, 2963, 2706, 2549, 2551, 2542, 2543, 3110, 3112, 3111, 
         1893, 2882, 3144, 2093, 2976, 2094, 2722, 2091, 2367, 2883, 2450, 
         2092, 2975, 1974, 2902, 2904, 2903, 1967, 2431, 2432, 3115, 2433, 
         2435, 2434, 1881, 2958, 2072, 2957, 2073, 2096, 3117, 2095, 2723, 
         3140, 2006, 2923, 2007, 2654, 1954, 3000, 3001, 2997, 2999, 2998, 
         2303, 1885, 2876, 2703, 2080, 2702, 2079, 2962, 2075, 2959, 2076, 
         2960, 2017, 2136, 3150, 2135, 2313, 2137, 3118, 2138, 2027, 2149, 
         3151, 2148, 2324, 2150, 3119, 2151, 2033, 2160, 3152, 2159, 2329, 
         2161, 3120, 2162, 2040, 2172, 3153, 2171, 2335, 2173, 3121, 2174, 
         2045, 2183, 3154, 2182, 2340, 2184, 3122, 2185, 2049, 2193, 3156, 
         2192, 2344, 2194, 3124, 2195, 2053, 2203, 3158, 2202, 2347, 2204, 
         3126, 2205, 2056, 3078, 3079, 3075, 3077, 3076, 2350, 2953, 2891, 
         2892, 2890, 1953, 2214, 3104, 1863, 2352, 1864, 3105, 2217, 2062, 
         2063, 3106, 1865}],
        "0.1`"],
       Annotation[#, 0.1, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  PlotRange->NCache[{{0, 1}, {
      Rational[1, 10], 3}}, {{0, 1}, {0.1, 3}}],
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 224},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->173507074]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->5427],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellID->7195],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"{", 
     RowBox[{"\[Nu]", ">", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->506],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->176931390]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17589],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->27809],

Cell[BoxData[
 RowBox[{"\[Nu]", " ", 
  RowBox[{"(", 
   RowBox[{"2", "+", "\[Nu]"}], ")"}]}]], "Output",
 ImageSize->{55, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->415448397]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"cf", "=", 
  RowBox[{"CharacteristicFunction", "[", 
   RowBox[{
    RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "t"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->405502340],

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{"\[ImaginaryI]", " ", 
    SqrtBox["2"], " ", "t", " ", 
    RowBox[{"Gamma", "[", 
     RowBox[{
      FractionBox["1", "2"], "+", 
      FractionBox["\[Nu]", "2"]}], "]"}], " ", 
    RowBox[{"Hypergeometric1F1", "[", 
     RowBox[{
      RowBox[{
       FractionBox["1", "2"], "+", 
       FractionBox["\[Nu]", "2"]}], ",", 
      FractionBox["3", "2"], ",", 
      RowBox[{"-", 
       FractionBox[
        SuperscriptBox["t", "2"], "2"]}]}], "]"}]}], 
   RowBox[{"Gamma", "[", 
    FractionBox["\[Nu]", "2"], "]"}]], "+", 
  RowBox[{"Hypergeometric1F1", "[", 
   RowBox[{
    FractionBox["\[Nu]", "2"], ",", 
    FractionBox["1", "2"], ",", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["t", "2"], "2"]}]}], "]"}]}]], "Output",
 ImageSize->{577, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->430024272]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{"cf", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], "/.", 
   RowBox[{"{", 
    RowBox[{"t", "->", "0"}], "}"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->11447],

Cell[BoxData[
 RowBox[{"\[Nu]", " ", 
  RowBox[{"(", 
   RowBox[{"2", "+", "\[Nu]"}], ")"}]}]], "Output",
 ImageSize->{55, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->632959699]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->185796011],

Cell[TextData[{
 "The square of a ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " variable follows the ",
 Cell[BoxData[
  ButtonBox["ChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiSquareDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->475500449],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"chi2", "=", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", 
     RowBox[{"x", "^", "2"}]}], "]"}], 
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"x", "^", "2"}], ",", "x"}], "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->402962351],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{"1", "-", 
     FractionBox["\[Nu]", "2"]}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["x", "2"], "2"]}]], " ", "x", " ", 
   SuperscriptBox[
    RowBox[{"(", 
     SuperscriptBox["x", "2"], ")"}], 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["\[Nu]", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{126, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->482228164]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"chi", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], ",", "x"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19944730],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{"1", "-", 
     FractionBox["\[Nu]", "2"]}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["x", "2"], "2"]}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "\[Nu]"}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{94, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1610613]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FullSimplify", "[", 
  RowBox[{
   RowBox[{"chi2", "\[Equal]", "chi"}], ",", 
   RowBox[{
    RowBox[{"\[Nu]", ">", "0"}], "&&", 
    RowBox[{"x", ">", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->308050189],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->102154283]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3818],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Nu]", "=", "1"}], TraditionalForm]], "InlineMath"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["HalfNormalDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/HalfNormalDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Theta]", "=", 
    SqrtBox[
     RowBox[{"\[Pi]", "/", "2"}]]}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->9364],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "1", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17975],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", 
  SqrtBox[
   FractionBox["2", "\[Pi]"]]}]], "Output",
 ImageSize->{65, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->68961988]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"HalfNormalDistribution", "[", 
    SqrtBox[
     RowBox[{"\[Pi]", "/", "2"}]], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->25089],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", 
  SqrtBox[
   FractionBox["2", "\[Pi]"]]}]], "Output",
 ImageSize->{65, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->603392235]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30066],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Nu]", "=", "2"}], TraditionalForm]], "InlineMath"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["RayleighDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RayleighDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Sigma]", "=", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->26412],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "2", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6840],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", "x"}]], "Output",
 ImageSize->{40, 25},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->543260586]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"RayleighDistribution", "[", "1", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->14676],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", "x"}]], "Output",
 ImageSize->{40, 25},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->525385887]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->182150783],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  StyleBox["\[Chi]", "TR"]], "InlineFormula"],
 " distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Nu]", "=", "3"}], TraditionalForm]], "InlineMath"],
 " is equivalent to ",
 Cell[BoxData[
  ButtonBox["MaxwellDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxwellDistribution"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Sigma]", "=", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->10596059],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiDistribution", "[", "3", "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26427242],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", 
  SqrtBox[
   FractionBox["2", "\[Pi]"]], " ", 
  SuperscriptBox["x", "2"]}]], "Output",
 ImageSize->{81, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->133912449]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"MaxwellDistribution", "[", "1", "]"}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->35093816],

Cell[BoxData[
 RowBox[{
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["x", "2"], "2"]}]], " ", 
  SqrtBox[
   FractionBox["2", "\[Pi]"]], " ", 
  SuperscriptBox["x", "2"]}]], "Output",
 ImageSize->{81, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->46276145]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11381],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " is a not a positive real number:"
}], "ExampleText",
 CellID->28128],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"ChiDistribution", "[", 
   RowBox[{"-", "3"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25502],

Cell[BoxData[
 RowBox[{
  RowBox[{"ChiDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-3\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->508764354],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"ChiDistribution", "[", 
   RowBox[{"-", "3"}], "]"}], "]"}]], "Output",
 ImageSize->{180, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->229613276]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26424],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"ChiDistribution", "[", "\[Nu]", "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"\[Nu]", "->", "I"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3872],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SqrtBox["2"], " ", 
   RowBox[{"Gamma", "[", 
    RowBox[{
     FractionBox["1", "2"], "+", 
     FractionBox["\[ImaginaryI]", "2"]}], "]"}]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[ImaginaryI]", "2"], "]"}]]], "Output",
 ImageSize->{117, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->334066727]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->32255],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChiSquareDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiSquareDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["HalfNormalDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/HalfNormalDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["RayleighDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RayleighDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MaxwellDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MaxwellDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->29065]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->16760],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->745697306]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ChiDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 28, 24.8522900}", "context" -> "System`", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "ChiDistribution[\\[Nu]] represents a \\[Chi] distribution with \\[Nu] \
degrees of freedom.", "synonyms" -> {"chi distribution"}, "title" -> 
    "ChiDistribution", "type" -> "Symbol", "uri" -> "ref/ChiDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7121, 223, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->55023376]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 131482, 2986}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2302, 56, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2925, 83, 1580, 37, 70, "ObjectNameGrid"],
Cell[4508, 122, 670, 21, 70, "Usage",
 CellID->21476]
}, Open  ]],
Cell[CellGroupData[{
Cell[5215, 148, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5693, 165, 406, 13, 70, "Notes",
 CellID->21555],
Cell[6102, 180, 287, 10, 70, "Notes",
 CellID->24917],
Cell[6392, 192, 692, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7121, 223, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->55023376],
Cell[CellGroupData[{
Cell[7505, 237, 148, 5, 70, "ExampleSection",
 CellID->504358159],
Cell[7656, 244, 365, 12, 70, "ExampleText",
 CellID->12933],
Cell[CellGroupData[{
Cell[8046, 260, 142, 4, 28, "Input",
 CellID->30137],
Cell[8191, 266, 359, 13, 70, "Output",
 CellID->166364418]
}, Open  ]],
Cell[CellGroupData[{
Cell[8587, 284, 145, 4, 28, "Input",
 CellID->3690],
Cell[8735, 290, 673, 23, 70, "Output",
 CellID->217654494]
}, Open  ]],
Cell[9423, 316, 121, 3, 70, "ExampleDelimiter",
 CellID->13694],
Cell[9547, 321, 68, 1, 70, "ExampleText",
 CellID->24668],
Cell[CellGroupData[{
Cell[9640, 326, 167, 6, 70, "Input",
 CellID->32627],
Cell[9810, 334, 533, 19, 73, "Output",
 CellID->102222496]
}, Open  ]],
Cell[CellGroupData[{
Cell[10380, 358, 277, 9, 28, "Input",
 CellID->989],
Cell[10660, 369, 3058, 54, 166, "Output",
 Evaluatable->False,
 CellID->97002615]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[13767, 429, 216, 7, 70, "ExampleSection",
 CellID->11647],
Cell[13986, 438, 182, 6, 70, "ExampleText",
 CellID->11149],
Cell[CellGroupData[{
Cell[14193, 448, 169, 5, 70, "Input",
 CellID->30931],
Cell[14365, 455, 332, 10, 36, "Output",
 CellID->509269069]
}, Open  ]],
Cell[14712, 468, 121, 3, 70, "ExampleDelimiter",
 CellID->30875],
Cell[14836, 473, 83, 1, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[14944, 478, 146, 4, 70, "Input",
 CellID->25598],
Cell[15093, 484, 1271, 43, 71, "Output",
 CellID->66130545]
}, Open  ]],
Cell[CellGroupData[{
Cell[16401, 532, 146, 4, 70, "Input",
 CellID->24599],
Cell[16550, 538, 1655, 55, 114, "Output",
 CellID->236916505]
}, Open  ]],
Cell[18220, 596, 120, 3, 70, "ExampleDelimiter",
 CellID->1443],
Cell[18343, 601, 152, 6, 70, "ExampleText",
 CellID->1731],
Cell[CellGroupData[{
Cell[18520, 611, 211, 7, 70, "Input",
 CellID->18154],
Cell[18734, 620, 369, 13, 70, "Output",
 CellID->217944213]
}, Open  ]],
Cell[19118, 636, 120, 3, 70, "ExampleDelimiter",
 CellID->4434],
Cell[19241, 641, 261, 10, 70, "ExampleText",
 CellID->9385],
Cell[CellGroupData[{
Cell[19527, 655, 172, 6, 70, "Input",
 CellID->23967],
Cell[19702, 663, 175, 5, 36, "Output",
 CellID->15945631]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19926, 674, 223, 7, 70, "ExampleSection",
 CellID->29866],
Cell[20152, 683, 112, 3, 70, "ExampleText",
 CellID->22667],
Cell[CellGroupData[{
Cell[20289, 690, 280, 9, 70, "Input",
 CellID->10048],
Cell[20572, 701, 2384, 43, 70, "Output",
 Evaluatable->False,
 CellID->105770211]
}, Open  ]],
Cell[22971, 747, 121, 3, 70, "ExampleDelimiter",
 CellID->13930],
Cell[23095, 752, 222, 9, 70, "ExampleText",
 CellID->30811],
Cell[CellGroupData[{
Cell[23342, 765, 390, 12, 70, "Input",
 CellID->19814],
Cell[23735, 779, 90571, 1512, 70, "Output",
 CellID->173507074]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[114355, 2297, 232, 7, 70, "ExampleSection",
 CellID->5427],
Cell[114590, 2306, 91, 1, 70, "ExampleText",
 CellID->7195],
Cell[CellGroupData[{
Cell[114706, 2311, 396, 12, 70, "Input",
 CellID->506],
Cell[115105, 2325, 158, 5, 36, "Output",
 CellID->176931390]
}, Open  ]],
Cell[115278, 2333, 121, 3, 70, "ExampleDelimiter",
 CellID->17589],
Cell[115402, 2338, 234, 6, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[115661, 2348, 211, 7, 70, "Input",
 CellID->27809],
Cell[115875, 2357, 235, 8, 36, "Output",
 CellID->415448397]
}, Open  ]],
Cell[CellGroupData[{
Cell[116147, 2370, 218, 7, 70, "Input",
 CellID->405502340],
Cell[116368, 2379, 924, 31, 71, "Output",
 CellID->430024272]
}, Open  ]],
Cell[CellGroupData[{
Cell[117329, 2415, 278, 10, 70, "Input",
 CellID->11447],
Cell[117610, 2427, 235, 8, 36, "Output",
 CellID->632959699]
}, Open  ]],
Cell[117860, 2438, 125, 3, 70, "ExampleDelimiter",
 CellID->185796011],
Cell[117988, 2443, 314, 11, 70, "ExampleText",
 CellID->475500449],
Cell[CellGroupData[{
Cell[118327, 2458, 330, 11, 70, "Input",
 CellID->402962351],
Cell[118660, 2471, 620, 22, 75, "Output",
 CellID->482228164]
}, Open  ]],
Cell[CellGroupData[{
Cell[119317, 2498, 199, 7, 70, "Input",
 CellID->19944730],
Cell[119519, 2507, 531, 19, 73, "Output",
 CellID->1610613]
}, Open  ]],
Cell[CellGroupData[{
Cell[120087, 2531, 246, 8, 70, "Input",
 CellID->308050189],
Cell[120336, 2541, 161, 5, 36, "Output",
 CellID->102154283]
}, Open  ]],
Cell[120512, 2549, 120, 3, 70, "ExampleDelimiter",
 CellID->3818],
Cell[120635, 2554, 561, 21, 70, "ExampleText",
 CellID->9364],
Cell[CellGroupData[{
Cell[121221, 2579, 161, 5, 70, "Input",
 CellID->17975],
Cell[121385, 2586, 324, 12, 60, "Output",
 CellID->68961988]
}, Open  ]],
Cell[CellGroupData[{
Cell[121746, 2603, 212, 7, 70, "Input",
 CellID->25089],
Cell[121961, 2612, 325, 12, 60, "Output",
 CellID->603392235]
}, Open  ]],
Cell[122301, 2627, 121, 3, 70, "ExampleDelimiter",
 CellID->30066],
Cell[122425, 2632, 514, 19, 70, "ExampleText",
 CellID->26412],
Cell[CellGroupData[{
Cell[122964, 2655, 160, 5, 70, "Input",
 CellID->6840],
Cell[123127, 2662, 287, 10, 46, "Output",
 CellID->543260586]
}, Open  ]],
Cell[CellGroupData[{
Cell[123451, 2677, 169, 6, 70, "Input",
 CellID->14676],
Cell[123623, 2685, 287, 10, 46, "Output",
 CellID->525385887]
}, Open  ]],
Cell[123925, 2698, 125, 3, 70, "ExampleDelimiter",
 CellID->182150783],
Cell[124053, 2703, 515, 19, 70, "ExampleText",
 CellID->10596059],
Cell[CellGroupData[{
Cell[124593, 2726, 164, 5, 70, "Input",
 CellID->26427242],
Cell[124760, 2733, 359, 13, 60, "Output",
 CellID->133912449]
}, Open  ]],
Cell[CellGroupData[{
Cell[125156, 2751, 170, 6, 70, "Input",
 CellID->35093816],
Cell[125329, 2759, 358, 13, 60, "Output",
 CellID->46276145]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[125736, 2778, 327, 9, 70, "ExampleSection",
 CellID->11381],
Cell[126066, 2789, 307, 10, 70, "ExampleText",
 CellID->28128],
Cell[CellGroupData[{
Cell[126398, 2803, 157, 5, 70, "Input",
 CellID->25502],
Cell[126558, 2810, 196, 5, 70, "Message",
 CellID->508764354],
Cell[126757, 2817, 250, 8, 36, "Output",
 CellID->229613276]
}, Open  ]],
Cell[127022, 2828, 121, 3, 70, "ExampleDelimiter",
 CellID->26424],
Cell[127146, 2833, 144, 4, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[127315, 2841, 218, 7, 70, "Input",
 CellID->3872],
Cell[127536, 2850, 399, 14, 71, "Output",
 CellID->334066727]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[127996, 2871, 311, 9, 70, "SeeAlsoSection",
 CellID->32255],
Cell[128310, 2882, 1041, 35, 70, "SeeAlso",
 CellID->29065]
}, Open  ]],
Cell[CellGroupData[{
Cell[129388, 2922, 314, 9, 70, "TutorialsSection",
 CellID->16760],
Cell[129705, 2933, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[129899, 2941, 299, 8, 70, "MoreAboutSection"],
Cell[130201, 2951, 184, 4, 70, "MoreAbout",
 CellID->745697306]
}, Open  ]],
Cell[130400, 2958, 27, 0, 70, "History"],
Cell[130430, 2960, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

