(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    142808,       3213]
NotebookOptionsPosition[    134502,       2922]
NotebookOutlinePosition[    135872,       2960]
CellTagsIndexPosition[    135785,       2955]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/ContinuousDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Continuous Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"StudentTDistribution" :> 
          Documentation`HelpLookup["paclet:ref/StudentTDistribution"], 
          "FRatioDistribution" :> 
          Documentation`HelpLookup["paclet:ref/FRatioDistribution"], 
          "GammaDistribution" :> 
          Documentation`HelpLookup["paclet:ref/GammaDistribution"], 
          "ChiDistribution" :> 
          Documentation`HelpLookup["paclet:ref/ChiDistribution"], 
          "NoncentralChiSquareDistribution" :> 
          Documentation`HelpLookup[
           "paclet:ref/NoncentralChiSquareDistribution"], "FindFit" :> 
          Documentation`HelpLookup["paclet:ref/FindFit"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"StudentTDistribution\"\>", 
       2->"\<\"FRatioDistribution\"\>", 3->"\<\"GammaDistribution\"\>", 
       4->"\<\"ChiDistribution\"\>", 
       5->"\<\"NoncentralChiSquareDistribution\"\>", 
       6->"\<\"FindFit\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Continuous Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/ContinuousStatisticalDistributions"], 
          "Functions Used in Statistics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInStatistics"], 
          "Mathematical Functions" :> 
          Documentation`HelpLookup["paclet:guide/MathematicalFunctions"], 
          "Statistical Distributions" :> 
          Documentation`HelpLookup["paclet:guide/StatisticalDistributions"], 
          "Statistics" :> Documentation`HelpLookup["paclet:guide/Statistics"],
           "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Statistics" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60Statistics"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Continuous Statistical Distributions\"\>", 
       2->"\<\"Functions Used in Statistics\"\>", 
       3->"\<\"Mathematical Functions\"\>", 
       4->"\<\"Statistical Distributions\"\>", 5->"\<\"Statistics\"\>", 
       6->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       7->"\<\"New in 6.0: Statistics\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["ChiSquareDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["ChiSquareDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/ChiSquareDistribution"], "[", 
       StyleBox["\[Nu]", "TR"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a ",
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
     " distribution with ",
     Cell[BoxData[
      StyleBox["\[Nu]", "TR"]], "InlineFormula"],
     " degrees of freedom."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->31604]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The probability density for value ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " in a ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution is proportional to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["x", 
     RowBox[{
      RowBox[{"\[Nu]", "/", "2"}], "-", "1"}]], 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{
      RowBox[{"-", "x"}], "/", "2"}]]}], TraditionalForm]], "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", ">", "0"}], TraditionalForm]], "InlineMath"],
 ", and is zero for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "<", "0"}], TraditionalForm]], "InlineMath"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->21100,
  ButtonNote->"21100"]
}], "Notes",
 CellID->429991538],

Cell[TextData[{
 "For integer ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 ", the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution with ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " degrees of freedom gives the distribution of sums of squares of ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " values sampled from a normal distribution."
}], "Notes",
 CellID->73600657],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiSquareDistribution"]], "InlineFormula"],
 " allows ",
 "\[Nu]",
 " to be any positive real number."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->27322],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiSquareDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->27465,
  ButtonNote->"27465"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->157342261],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->150901550],

Cell[TextData[{
 "The mean and variance of a ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->27465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23005],

Cell[BoxData["\[Nu]"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->313588289]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6856],

Cell[BoxData[
 RowBox[{"2", " ", "\[Nu]"}]], "Output",
 ImageSize->{21, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->35238384]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3238],

Cell["Probability density function:", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->21100],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7194],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "\[Nu]"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["\[Nu]", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["\[Nu]", "2"], "]"}]]], "Output",
 ImageSize->{100, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->90574492]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiSquareDistribution", "[", "5", "]"}], ",", "x"}], "]"}], ",",
    
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2002],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->189219742]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->13933],

Cell[TextData[{
 "Generate a set of pseudorandom numbers that are ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distributed:"
}], "ExampleText",
 CellID->31953],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomReal", "[", 
  RowBox[{
   RowBox[{"ChiSquareDistribution", "[", "10", "]"}], ",", "5"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28942],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "11.431656852606826`", ",", "14.930411371316666`", ",", 
   "11.26411395910091`", ",", "8.300131413326277`", ",", 
   "4.208795365872743`"}], "}"}]], "Output",
 ImageSize->{301, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26988817]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24595],

Cell["Properties based on higher-order moments:", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2526],

Cell[BoxData[
 RowBox[{"2", " ", 
  SqrtBox["2"], " ", 
  SqrtBox[
   FractionBox["1", "\[Nu]"]]}]], "Output",
 ImageSize->{72, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->884612446]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->880],

Cell[BoxData[
 RowBox[{"3", "+", 
  FractionBox["12", "\[Nu]"]}]], "Output",
 ImageSize->{42, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->310959359]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1657],

Cell[TextData[{
 "Third moment of a ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution:"
}], "ExampleText",
 CellID->10330],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "3"}], ",", 
   RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25998],

Cell[BoxData[
 RowBox[{"\[Nu]", " ", 
  RowBox[{"(", 
   RowBox[{"2", "+", "\[Nu]"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{"4", "+", "\[Nu]"}], ")"}]}]], "Output",
 ImageSize->{99, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->39796058]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10737],

Cell[TextData[{
 "The 0.75 quantile of a ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Chi]", "2"], TraditionalForm]], "InlineMath"],
 " distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Nu]", "=", "8"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->10291],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"ChiSquareDistribution", "[", "8", "]"}], ",", "0.75"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16765],

Cell[BoxData["10.21885497024676`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->482753035]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->11928],

Cell[TextData[{
 "Compute the ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 "-value for a ",
 Cell[BoxData[
  SuperscriptBox[
   StyleBox["\[Chi]", "TR"], "2"]], "InlineFormula"],
 " test with ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " degrees of freedom:"
}], "ExampleText",
 CellID->813014373],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"ChiSquareDistribution", "[", "n", "]"}], ",", "x"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->944228315],

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"GammaRegularized", "[", 
   RowBox[{
    FractionBox["n", "2"], ",", "0", ",", 
    FractionBox["x", "2"]}], "]"}]}]], "Output",
 ImageSize->{202, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->312563814]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->44516724],

Cell["\<\
Plot the cumulative distribution function of the random variable:\
\>", "ExampleText",
 CellID->19469],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"ChiSquareDistribution", "[", "10", "]"}], ",", "x"}], "]"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "25"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6864],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6794571]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31467],

Cell[TextData[{
 "A contour plot as both ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " are varied:"
}], "ExampleText",
 CellID->18934],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ContourPlot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Nu]", ",", 
     RowBox[{"1", "/", "10"}], ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8106],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJx0XQVUVU20RhGxuzAQW1F/u2NfE7u7GxMTuzDADgxETFQUUUFElFIa6e7u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   "], {{{}, 
     {RGBColor[0.33676591967233516`, 0.1327782030140431, 0.5903455562697186], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFV11slFUQvd3ddvfb/TbygNGAUkkJDSgSoI2IkKAU9QGjYAIkRAH/BQMJ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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwllEtI1FEUxq/jODP/x4CLoLAoJHBRGYRKZVaaShslNVIh8BUVPdCNRCi5
yISgTWhmiwQXypgxLSzMgig06L1o08JILJAWWSRWJC3qd/oWB87jO897zs1t
66hrjzjn6qEo9NZz7kfcubso92Q6N59wrhrdHeRi5Jfw37GnkYuQn4XOJdEN
4byQ5dwH8BXI49gLsR/HnkY3nOFcNpgB5H6om1wz4MuSzrWBfw0+HXOuKnCu
FuoC24BuKa6clstizsJ/QZeCz0fnI3+GTiOfgurgP5KvNFM9NCB/spqQ7yNf
I/dgqNy/qOEg+V/5zuVZbdBjbI+gTdh6kQvhy7Cn8K/B5yvxWqGH2OrRBfj7
0HvwRdRfDb8Fe4R+3yC30MvRQL304LMXewmUgz0VU62Hwd+LqObn5HoBhWC3
E3N3QjOy2WzDJ/Q0c5t1BTQJdgqKgc8G/w7777h6LSPeE2wzkBdVT0fwHUjo
LbrxL4Jf8tTLDagLuQR5BP9W7PvJncJ/gvy3oCHsleimYurxAHJBqNmsIeYK
2G/QIrl2IO/D1gFmhdj1YHqZxUXoBPGvYs/FvjkUbzNpT8jHsDnozoLtCPQW
x/AZxTYRapaWY6unmdus+zL01o3opiN683FfPVjtNtP1+G6ELsOvy1BtJ8HP
R1Rjf0IzstnYTlwgdw90JaqaA7CXAu2K7dx5+HOBarOYq+T766v3tcRvBl9L
/vmYbmDYbMhjMb3ZYlw7bbtsM68l9yGoE2ynvTF8JdScpZv5Gdcb29vazizH
dbN2q3aj5Z5u0G7PbmSU2lrQPSDXdFS1NHnaJaspRE4mtbuW026xwNcu2E2O
kPtmqN2ymywkdl6o3dsAfpX8c/hUZmrmthvXwZxx2pEqT3+I/R32ZzzFdzbU
bttMS5OKabFsx0bJXZzUbdjOLiDP+ZplPvJOwyf1djZDu81SX7dsN3qbficD
YYH8v1WbgfVuN1seasdtt22mPdgHE9od24ExsDnIy04zs17SvnbbemrAvzHU
bhqmL9AfZH+P/YF/mEdNqLfbFdFsTGe8zegfllKukQ==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.4234657590170055, 0.2835258090421294, 0.7122126688091559], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmHuQ1lUZx3/77r7v7/29FxubaZxMwAVBRjdtU2TwbopmgihIUMPNjYsF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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl3uwVXMUx/c595x7995nn5IZ492720NKD24PRtyeVCqPQoloRDeJlEfi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         "],
         VertexColors->None], 
        PolygonBox[{{3390, 2099, 2098, 3388, 1704}, {3387, 2723, 1704, 3389, 
         3386}, {2099, 3106, 1346, 3105, 2098}},
         VertexColors->None]}]}, 
     {RGBColor[0.5101655983616759, 0.43427341507021555`, 0.8340797813485931], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNmH2Q1lUVx3/Ps/v8nuf37G9DxTHedAEXEwvYxWkNCzWdBF8gMJO0JrAM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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmeMFVUYhmfv3bt3Zu4sTaOAhg7SWSBSjBQh1CAYQAHFUEQjGhBQfuAP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         "],
         VertexColors->None], 
        PolygonBox[{{2097, 3101, 1344, 3100, 2096}, {2938, 2308, 2307, 2935, 
         1162}, {2308, 2926, 1157, 2925, 2307}},
         VertexColors->None]}]}, 
     {RGBColor[0.5881162706186073, 0.566601389124046, 0.9100526016828171], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMVVcVx8+dO+fee+45MwyQUkBHoUBtTKE8opa20IcRbVMFjYltDU0d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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlmlsVFUUx9/MdKbz3jyEQIKgopRFSQQsEI0gImgC2sqmRhEDIRAW0bBI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         "],
         VertexColors->None], PolygonBox[{{2716, 3156, 1031, 2719, 2715}},
         VertexColors->None]}]}, 
     {RGBColor[0.6518607141765334, 0.6690209728415552, 0.9115057786603683], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmH2MFtUVxmfffWfemfedjRoqC+VLbSgfqwJaE6WmCDGpBZVkWUxTtqUF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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlnuQTmUcx8973n13z3nPMWlWu8otSuQumkmaFtNMo5pqGhRtNYXsZqam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         "],
         VertexColors->None], PolygonBox[{{2300, 2896, 1148, 2895, 2299}},
         VertexColors->None]}]}, 
     {RGBColor[0.7156051577344593, 0.7714405565590643, 0.9129589556379195], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmH2QllUZxp/3+/l4cZRZoHIrmckmrBwFJmcChEqdhWTEZlxAnSUxlg/J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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJw1lllsVVUUhs/tudyee85pIqatxRCFRByYtSiJDEUNCUZ40MRS0BQpUmgZ
NFQFCq2oTKIUAX0otJDwYsAB+gAqIioxPuBA0dhaExGSIqQUlEQTIYB+f/74
sHLW2mvce/97rTO05rknlhYEQfAOlIV+KQyCjXEQ/BEGwZ8slOWDYBtUgLwX
+XgUBKvSIOhHPp8Jgmb4idBMfD8YEASn8X8Y+Sn4Bax9SqwjUAv2RdhPwH84
+mno7yHefvj5SRB8iO4Ea0fR1xQFQTlxjrH2DfoV5L9IkefQN8KvzDvWLNbW
w2+AGuHLyTeGWKOhSvRd0Fly/wadgnqhbuIVkqMK25uJ10WeXfgn1NdLPSXk
bkDuwHdqjhrFQ4PRX2XtNfzXQh341iLfmzinctXIH/4Q8R+Fnwa1wA8m7zr8
I+RS9MuINzSwzQ74EH0d+kdY+4l6GlUz8in0m9G3aM/s7z70N1HfFmI2k+9B
6ivANgMtDB2jGn1EfVOwH0asBch10BT454l3mvw9hc51AfuPkH/WuUCH4Afh
+2/ku5gELYJfH/uuq6FbU9tINxlaSOw+2WedowxdL/XOIPYE9NfRjcR/K3IT
NreQoyzx2fRRQw361eh7iD0TqkQehr4K/XTkA4lrVG37oZHEHwE9Ruzp0P2s
5fCpxX6B8Eo9J1l7Bv7xnLH0NfLJ0JgaAT88MVaexuYNah2VOtY4YQb+dtbW
cX7Foc+6SRjL+MyFpdGsjYqMqeV8K/CZNcCYv8FalvghdD02du+Cn501hoX9
ifi0h34DX8G/yNqZ0GdwHfsbUCY0pnvQDUW+gu0/0HhifRv5rS3MOtdF8vTH
zjmX/R5LjLWx7PcMfBMx3se+n/jLidXJeif6/gLf/dn/3wb0GbGvYDM39B1o
r9qT9qI9ay8FfGdlvac2+CcT3+0m4q8m1xLWLofO2UasdmgNuaZy5qfIPYga
K7mbWuQ2fHdAm9F3ZI0VYUjYEWb0VvQm9Rb1Zq4Rew72L2eNsU/QH4Za8V+B
/mP4S/iXYH8H+rfyflN6SzrT8bExLOzqDvbC1+fdS75nbYvON3Wv0ZvbrjeB
/j3890Hf4T+Gevbk3NNaqWc/a9szxkhnYgwKe+pBVeiL0L8UGoO3EWs3lIbG
2I+J37je9q/EL897D6pde3o78Rnq7FSDfIV5YV0xGqj1h8S+e6h5tnpp6r1o
TzqLhBqKcz6TZxPbSKczrEe3lvr6Qvf0ecgDkRtDY/Istr9DV7G9Ai1Gvyk2
lubofVLPzsR3px65FXls4l6omaHeqpmiWaIe+6rqJcYXOfdQYaWEeM2hMaNZ
Mil1b9ZMeUi9LDVW9MaEHfVk9WJh6FLimaNZE8qHeEtjz44ZUAu+F1hbRv4j
6P/C92+oOnSPF/ZaofkZYzDGvjRyb1LPXoP8Suq9y+YosXehfzPjHrsKeWXs
3qWerbu+O3Fv0Z1XwFdE7i2qYTf859hcLnAM9aoXUt+detYBdDux2ZgxprYg
N0SeNdegFfAbsD+IfX3GWBzCWi5jTKpX7INKQ/cMzQbNcM1uzYgdkWeEZoNy
6K1PTjwb9OaXYL8o9tmdQL4TXRq7F+vNn0c+l7gW5RA23s27Vwoj3ZHvRHeh
GnW3DyTuVbpjzQrdie5CM0PYq4s8i4RB9abFsXOrR70OvyE21vRPMo5YxbFn
kXq+sLot8dsUZofk3TPVKzXju9B1J57t6kHHCj1TNEu+hB8orKX+F9EZXoS/
lvhs9WYOE789cm/TnWn2zMt79moGHS90T1UvFaYORu7B6r36p/kP/Sc4Fw==

         "],
         VertexColors->None], 
        PolygonBox[{{2797, 2216, 1074, 2807, 2796}, {2069, 3241, 1486, 3240, 
         2068}, {2808, 2069, 2068, 2802, 1074}},
         VertexColors->None]}]}, 
     {RGBColor[0.7776454386698771, 0.8520049152225181, 0.9073045449308235], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFmGuMVdUVx8+de+69555zocFkaAFNtG2QaFLT1DJNEAaRACM6mjaNzAw0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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlntolmUYxp/t/b7tPXwNkma5DCyCPFARSQts9EcRYbnEErfhIM0h6Oam
tnnYPDDLqYHZnJBF0ElR/9GCMjIMzLKsJLN0ZYetZVpkiWU6zdnv4uqPm+++
n/u6D89zH97vxpnNU+YVhxC6oBzUn4QwKQ7hvqIQHopCuDaDkMflQ5gIYH4a
QjP0CPoj6Iejfxv9TuRfcdSB/S/IjWDPIG9D7i2E8A/25eAPloZwjrN6+JuI
twn+GGcnwOc4W4/vhdhfhB+EHrwqhEOc7SkJ4V1oJPFu5awLfgH6dWC7Uye/
mByG4G/j7F7i3clZI/6HI49F/pl8jhLrGD4G0D2AWUpuI7FZVmTMq+CPg5mD
77LIbzGfs11FfhPluhz8QOScL6P7F5qF3ACVw3dBF4m1Ff1Ucs3hYyKxzpPz
c9hWId+Bryvgn4BvJYfXiF3D2R70E8ivCdsbcn67UjDn8n7Dt+Bnc9YD/zH6
jeAfRY7gi6EPyO8N7HdB++D7+f0J+gvdWejJ2G+st70ErdDdodoi30m270C7
//fxFfQ11I9tn+Jxtx6og/tVcqcSYrdwxy3EGkA+CPbTzLUUph79Z6lr00nO
tWDLYmPz2DyGHMX2dZb3OYptBf5/z7tGqwrOWbm+yVkLuhOxe/Mk/v9G3i05
556aA/4APr5E3gz+FPx2MLX4v8TZcuUD/jT5TEV/mtxymXvlec7a0H+Ifiu6
LdBGdN2Za7OCeD3wIxLzKfEmxO5R9ebtYA5zp5bUs3EEnxvADiXujfHBb1Va
cK30ZrOxHwa+Hf0p6Fv8H8/ce53EGKF6gJkE/mHVD2wd8mRiHdDM4fs36Lti
99SGzG+qt1wUufbrsDkTuQc0y5oJzYJmWm81D/0Xkd9MvaweVu+qp1X7DvR9
kXugE74BH3XE+h4ag/xe6l59BnkNuincry3vmbwFuT3xLuiOrFsbu/eF0az+
qJnNeWbVe98g96buwXLw10CV6O6GJoNPUs/+XmhU4h2j3dJFjApsrs/8VveT
zzTwc8EfinzHJvjG1LtHZ6sz96R6cVXkXtmR2Jd6ppPYTxXce5fxtynzTGuW
VbNKfN0Tu/Z1xZ7NBjArc57RQfiqgnttBpir4UeBH41+PGfTsK+Onat2zMvE
fiVx7bUzxmE/BtqPrh6bteDXpN5dfxC/Eduu1Lxq/Cz8ycy9rh26N/ZMaBYO
czYkGf99+K9Gf6zgHJWbdszSxBjp/gSzP/YO0OxrR32euuaqte6wHf2Mgmu7
uci7eljBd9PO3gl/Jfbb6E0uoN8We9aUQwm+d8S2VQ9o9l9KPFvaAZrthYm/
LZrxGuSPMt+lGptG/DdBL+Q9E4vgFxc8u+rR9eD3Zd6V2vH6NlWkrrW+Ueqt
12PvJvVYLfyC1LOlb9xM5CXIP0SegU/ok+bEu2c68lzs3wfzYt45aFZrUvem
ZvbpxDtBu2BQ7wlfRT7NOc/sdPBlqWe1F2qHX5p6Vyimaj86813VA4rVmhqr
mPKlnJSLfK4m3tjMvaJv0CzwK1PPqna8dtuq1LtGO64VfJsI7M3B39YQm9c3
9jy6C4lzF6aD+y4r+FvyOP4qiXVXZl47Vd+uJYl7R9+wntQzrNnVN6oc/XWZ
/0voP8N/13Yskg==
         "],
         VertexColors->None], PolygonBox[{{2777, 3166, 1070, 2789, 2776}},
         VertexColors->None]}]}, 
     {RGBColor[0.8348922753883199, 0.8710952098721363, 0.8816580133817192], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFl2uMXVMUx8+998zcc8+5txFTKSWovjyTfiimL19ENYTqTBuP6ouWD6qD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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlVtslVUQhXc55/T8/39OC0FjpHgDxHrhwQcoBeTJiEYToC1E6elNaOFB
aUEKrfGCICJoTIrQFg0mgJeCwWsM8QIEA5FCy0tFEDVGUCpofBAM+CCJ33L5
MMnMrNmzZ+89s/a4hW3VrSNCCF1IGrk5F8ItSBXGA0UhbIhCeDUJoQi7A3sn
2HZkH4v24vsI/VAWH9gB7J/iEHbkjD2YCuEY+lHkJPYp5DL4aMWgN4In5B4H
Pg97BXY/+hP4TqCfwLcGvYUahrHPId3go/EV2G8PeHNJCAfBR2VCqKCOAtgC
ZBb4QeKfZq+nkGuwZ1LzC+h/I01gEznvGfRzrL8H7DySyodQj6+yOIRbybmC
XLXgJ4n/BpkGPhWpJl8N+GpiK8Bnof/F+jWxfdIvIx3ELgb/lbV9xCwDa9Md
BO+ZIf9s8H3grcgAZ6hNXPsxzneI8y6PXdtR7JXoN+Kbi/47vpewh1gzQD3/
YLfHzqG1v2EfzzpGWCeyi7V9yJ3Ef0HMDegx+92HPQd7PLXUcqfXsy6Pbwz4
WGQ22MfYUyO/sd72NWQD+sbYue/gTD2c9yIxRzjrV0h74jfVW+oOX0GvIf+f
3G879nrsssRvs5WYxdgF8DL27wIvRa8j/4ERfpMGcncSf1z3j1RjN2HvRb8f
ybD+XXxTyDUZ2YSdwu5D34VsJ1cPvvfQS1KupShy7apJvbkq9t2qR8cT+zn4
/LR7dC17fYtvOO2ePK37If4Pznoe35Gs30B3XwDfHLsG7Z3CPozeSI5PqHUA
303Yn5G/Ou2ZuxJ7RjQb6lHNznV6/4xn6AL6TOJ/Jn8N+CjsyYnfYlvavVCZ
d2+qJ2LWv0/8NLBK5OHEPaVeUs/pLtKRZ0l3spN83+O7gP4QdzIl8R7K/Trx
Hdj1ke9+EClEnhHNximkHPtR9l+f8cwtwG4DH0p5pudE7nn1+n5kFbGdec/G
j8TfC57+H9+DPAa2FNkG3g++BPz5xLHigCvUepUzX0utJcRsBetFzorPqLmO
+JWJ11YVmRvqkS9T5oh3wBvIv461PdiN2CPBO8F/SHnWZyDzMp75l3V27A7s
J7G3JPZJ31TkXtgRu7fUE4PU10b8Gxn3rN6+KnKvqgfmR+bA/7gPWYheE5l7
1dOaxSXkmJ72TOotu4i5mvKbqje2xO4t9YjOnmG/iRnfQVnOd6671gxPwG4m
5vbgnv4gco+oN7SHuCVE5gJxTC7nO9bd6k36yV0aeVbP4PtQ3Jf4rWbg6xb/
5Mz1laRaDrYsMdeqB+rQB7Pufb1Bs2Yldi/vF5+Rbywx61L+c1ajP5s411l8
lyJzjLilP+OztCTmEp1J3DOEPSltDlpK7lbk7SL3yMjIZ1Dt5Uhx3j7pt+mM
efeA3l5vtlt8kjNXioMOZ12zahWHd4GVJu4tcUwlaysSz4o4tkDsmznfjf5M
nW0QKU/7jBfBL8XmCtU4Bqw7ce+uTZkLJ0TmYnHiIuyWEv91DeI3YjdG/pvF
CX3ovXnfTS++dN5/mv4y9cRp4j/N+a9Wz4t7FiXuPXFQPbFvRe5l1fhi5JpU
i3pOf5n+ZP3F+tOasZ9J/Da/6AzYdyXmbr3BdPJ9nTVXaIaadL9513Y3+GZi
s3nfvWZKf+lwYkx/anHOM6rZFKfPRX8k8V+kP+u7nDlY3CvOejxvThIXiTN2
R/4T9BeIk8TdzyWuVRyuWZ+U811q5v8Ft+8oRg==
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.8921391121067627, 0.8901855045217544, 0.8560114818326151], 
      EdgeForm[None], GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxNl0mMVUUUhu/rd1+/2/d2I+CIGidiIjYIC0CFblT6AQqK2rQkMggq4JDI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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtlUlsllUUhu/3/1/7f733L8igLjAqEBMQEzcWsRQkHShQJdAWEplERofY
wsJq4kqUAE5x4xSiQhEKmDA4RFsgBQUqocVEVuAOE1i5UDYmxuF587o4yXm/
895z7xm/qRt6O3tKIYTtSI58XYSwMoZwqhxCGx9GUwhjyHX0a8ikZI5srUg9
+EvwAvTHkburIUwHv4beWBPCeeyjlRDuwvcP+Jxcj4+6EAYzzteGcAT7GnA3
tjOcOQM+jXyO/TjfNsHfjDTg4wrfhrBdQP8Y7tncvvYgf5XscxD77+CnODuM
/QrcrcRzGTwGvoB9Pfg8eDR3bK+Afyk7xg/Q30f68PUqshpf++B8g+1bZAL6
V8TXgt6M/Ij/N+D8yf0j4C3YZ8lH5hxcxL4IfAz/58Abot+guy/y7WXwVTiz
yM918Dhs45NzOR8f09CnIovRv0PGKs6hcvcgZ3Zw9+vIS9x/D7iFty3EZz/c
xsxvUw4Uu97YQH1mI+OpzRT4Z/F9Lvnt8rEb/rvgA7lzOo671vJtuOQaHcZ2
CLkMPg3nEr7+rfhu+WyNjlmxfgF/Pu95E/5n4E/BOfwaSW6fT8JvhdOUuea7
0N/j2/7MNViGvpBv89APIZN4zwTkKL6ayMEJbAeSa6OYu+AvKpy7AWQPtpnc
187bfkUa0ecinTXOeQ96b9W93Q2/g7MVfGzD/xDyBLgAby97JiL+tiI74K9U
TrA/o5piG0HuxXY/0qV+UQ8XPiOuYjoI/hn+0pJ7ZDK2TuL5rdZ3vECuB5Jz
ewNOJ/x18L8vu2fbwEujc6GcvQi/Cn8e/JvwV2DfGH23emYJuBbcy9lB5Dn0
nzRLmm3wDGJvhPNJ7hytxr4KuZPcDGNvx7Y8OpeqwTpwH/ha2TOzi7tPJc/i
Dc7Pwf42eB94b9m9cjR5ttUzmk31hHpBM/pWco1UG/XIH9hPqn7oq7Av5q52
ZG7mGUpw8+jeboKznvffAj+MvhMfLejN/9vVI7LdRB7IzFGu/q74rHK2OzkG
vX1v7lr8U/FsqCbTsQ0Vnr1L+GvA1+zo2LaVnIu1hWdXOXkWvAXpyLxz3uF8
Ai8B9+feHSuie0s7ZCP3PZTce9pxA9ifrjqWvty91VV4V6jHlJueOtdaOdLu
7Y6urXbwGr0nunbaucpFW/QsKie34X9YtW/14LTCO1u7Wj2tXHdE717l/KPo
nahdqJl9FP4j0bHP5NtjnJ1T9S7STKn26mH1rnpghJhu17mW6in1WgsxH691
z2nWJib/SzRz6tU7knerela9v7zwv0MzcF/yzGnWNGP6F0ys9y7QP0G7Kyuc
a+0w5X5zdO1Ug1a4zfWutd6gWj0fPQuqWT/n9yfHpp3/HzBi41I=
         "],
         VertexColors->None]}]}, 
     {RGBColor[0.941176, 0.906538, 0.834043], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxFU8tOFFEQLYZuprndPSMwW01kYXThxkQXmjgPBkRRI4h+gIkPXMDGL1AT
UJfsXLAWRaMziPhKfA1RF/oT8ti6YkPUcziVuDh9761bVffUqer9V2Ympgtm
dhXoBvpTs2Iwq3WZtWFYBoqwv4P9PfApMuvLzQaAEz1mK7h/BSQm38+R/Edh
uBh0R3yJ5DOCz4Ug3wTrWZzreKuD+zVgqUsrz3vhv2nal0w5XvpbB+BXAcrY
75j29GX8LRQEejaO/KcSxTyCfRVrr6m+sUQ19mB/Bvsq9qPYr4F8DJ/fcHyN
eo/A3kLcc9eB980gH9peAJBht+6W+zSR71zQ3QSIVJBnxLm3nT+5rDifFgx5
Ko4fsX5IpcF8rLgnsJ9MxIl8xmPZGsD9VHe0r8P+K1b9HRB5gzuE7/rRP3Od
Bl037qnjFvaz8J0DHkbizT6zrpJrSj0b+DSBZ7jYLuidJe8dtWe/7oLj16L2
N6HBNeB7t7ReBHC0Nvj2pTr35+JKnncQe7tXudkP6kF9OB+PAYTYeSQYTqT1
MtaBVHdj7DXwFPbrWH84h2Mo+mgmzfhe1Tns+GyxPsbXPD9rani95E2ffawd
QUPAImyTQT0jv+NYDyL/YZ+/Lffv+CwzP7Wvu/5vU2lNnYdyxTN2T6pZrHp/
573Hl4L0IvfhIB7kUIZ/Caj7f7MR672fqHsqiMOg/yOslX6r/h9sem3Uh/25
4T3ie5w3cr2H3A+AhUj9WPe5okY11/BPUfPNmEOZYhacy3bh/7/ImaSmf+H/
LdOZ8VXvb+Y9Zm72jTNEbpeDesz+sLenvb8Vn1nWkft8M8c/x+CNHQ==
         "],
         VertexColors->None], 
        PolygonBox[{{2664, 2663, 29, 970}, {2955, 2954, 306, 1182}, {3064, 
         3126, 1361, 322}, {3008, 2321, 1510, 1222}, {3396, 3395, 301, 
         1710}, {2643, 3007, 1220, 951}, {2123, 2645, 425, 320}, {2640, 3006, 
         1219, 948}, {2122, 2644, 424, 319}, {2627, 3345, 1595, 937}, {2434, 
         3274, 1509, 759}, {2124, 2646, 426, 321}, {2637, 3004, 1218, 945}, {
         3125, 3174, 1427, 1361}, {1992, 3009, 1222, 29}, {2835, 2834, 320, 
         426}, {3174, 2046, 1094, 1427}, {2665, 2666, 970, 30}, {3006, 2638, 
         946, 1219}, {3007, 2641, 949, 1220}, {2642, 2120, 951, 950}, {2633, 
         2632, 5, 941}, {3005, 2634, 6, 1218}, {2836, 2568, 321, 1094}, {3273,
          3064, 322, 1508}, {2634, 2635, 941, 6}, {2829, 2831, 1089, 424}, {
         2117, 2636, 944, 946}, {2321, 2435, 759, 1510}, {3002, 3003, 1178, 
         4}, {2830, 2121, 952, 1089}, {3347, 3346, 4, 1595}, {2121, 2642, 950,
          952}, {2833, 2832, 319, 425}, {3112, 3113, 1182, 301}, {2636, 2116, 
         945, 944}, {3274, 2047, 1508, 1509}, {2952, 3394, 1710, 1178}, {2639,
          2118, 948, 947}, {2628, 2629, 937, 5}, {2119, 2639, 947, 949}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1999, 2665, 2666, 2662, 2664, 2663, 1998, 2436, 1992, 3009, 
         3008, 2321, 2435, 2434, 3274, 2047, 3273, 3064, 3126, 3125, 3174, 
         2046, 3410, 2836, 2568, 2124, 2646, 2228, 2835, 2834, 2567, 2123, 
         2645, 2227, 2833, 2832, 2566, 2122, 2644, 2226, 2829, 2831, 2830, 
         2121, 2642, 2120, 2643, 3007, 2641, 2119, 2639, 2118, 2640, 3006, 
         2638, 2117, 2636, 2116, 2637, 3004, 3005, 2634, 2635, 2631, 2633, 
         2632, 1990, 2628, 2629, 2626, 2627, 3345, 3347, 3346, 1989, 3002, 
         3003, 2951, 2952, 3394, 3396, 3395, 2106, 3112, 3113, 2953, 2955, 
         2954, 2110, 2545, 3001}],
        "0.9`"],
       Annotation[#, 0.9, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2240, 3428, 3429, 3427, 2860, 2575, 2143, 3182, 3183, 3020, 
         3021, 2680, 2142, 2678, 2141, 2679, 2003, 3130, 3131, 3127, 3129, 
         3128, 2002, 2442, 1997, 3348, 2323, 2441, 2440, 2958, 2049, 3411, 
         2957, 2571, 2661, 2660, 2851, 2234, 2850, 2849, 2570, 2133, 2659, 
         2233, 2846, 2848, 2847, 2132, 2657, 2131, 2658, 3013, 2656, 2130, 
         2654, 2129, 2655, 3012, 1996, 2652, 2653, 2649, 2651, 2650, 1995, 
         2420, 1991, 2547, 2546, 2225, 2414, 2413, 2828, 2045, 3063, 2827, 
         2564, 2115, 2630, 2224, 2979, 2395, 2976, 2978, 2977, 2528, 2103, 
         2612, 2320, 2613, 2614, 2104, 2956, 2411, 2983, 2984, 2982, 2986, 
         2985, 2396, 3342, 2397, 3075, 3076, 3077, 2711, 2712, 2710, 2709, 
         2585, 2183, 2704, 2222, 2825, 2223, 2826, 2919, 3418, 2082, 3095, 
         2083, 3419, 2918, 2603, 2213, 2770, 2290, 3222, 2291, 3223, 2215, 
         3167, 3168, 2791, 2790, 2102, 2431, 2433}],
        "0.8`"],
       Annotation[#, 0.8, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2011, 2466, 3035, 3034, 3033, 2331, 2465, 2464, 2966, 2052, 
         3413, 2965, 2578, 2151, 2691, 2245, 3190, 3192, 3191, 2150, 2689, 
         2149, 2690, 3032, 2688, 2148, 2685, 2687, 2686, 2007, 2453, 3019, 
         3018, 3017, 2326, 2452, 2451, 2961, 2051, 3412, 2960, 2574, 2140, 
         2677, 2239, 3179, 3181, 3180, 2139, 2675, 2138, 2676, 3016, 2674, 
         2137, 2671, 2673, 2672, 2001, 2437, 2552, 2551, 2550, 2322, 2422, 
         2421, 2845, 2048, 3373, 2844, 2569, 2128, 2648, 2232, 2841, 2843, 
         2842, 2127, 2565, 2126, 2647, 3011, 2417, 2125, 2529, 2277, 2416, 
         2415, 1993, 2877, 1988, 2404, 2538, 2537, 2109, 2705, 2185, 3154, 
         2184, 2706, 2186, 2587, 2764, 3165, 2212, 3092, 3093, 2708, 2187, 
         3155, 2707, 3079, 3440, 3078, 2063, 3272, 2062, 3417, 2950, 2586, 
         2278, 2428, 2427, 2087, 3169, 2217, 2821, 2218, 2822, 2214, 3166, 
         2777, 3000, 2993, 1987}],
        "0.7`"],
       Annotation[#, 0.7, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2258, 3202, 3204, 3203, 2164, 2702, 2163, 2703, 3047, 3048, 
         2473, 2474, 2337, 3289, 2336, 2472, 3288, 2055, 3287, 3065, 3133, 
         3132, 3198, 2252, 2871, 2870, 2581, 2157, 3196, 3197, 3041, 3042, 
         2698, 2156, 2695, 2697, 2696, 2010, 2462, 2006, 3031, 3030, 2330, 
         2461, 2460, 2964, 2329, 2963, 2962, 2577, 2147, 3189, 2244, 3028, 
         3029, 2684, 2146, 2681, 2683, 2682, 2005, 2448, 2558, 2557, 2556, 
         2325, 2439, 2438, 2858, 2859, 2857, 2856, 2573, 2136, 3014, 3426, 
         2238, 3178, 3015, 2670, 2135, 2667, 2669, 2668, 2000, 2423, 1994, 
         2549, 2548, 2231, 2419, 3349, 2418, 2838, 2230, 2837, 2840, 2839, 
         2071, 2412, 2070, 3082, 2588, 2190, 3010, 3425, 2229, 3175, 2917, 
         3247, 2303, 3097, 3098, 2615, 2108, 3114, 2107, 3400, 2766, 3090, 
         3089, 2080, 3375, 2765, 2601, 2748, 2747, 2875, 2425, 2426, 2302, 
         2410, 3241, 2069, 2808, 2216, 2797, 2820, 2814, 2094, 2783, 2789, 
         2776, 2999, 2992, 1986}],
        "0.6`"],
       Annotation[#, 0.6, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2020, 2489, 2490, 2356, 3311, 2355, 2488, 3310, 2059, 3309, 
         3068, 2168, 3207, 2263, 3054, 2016, 3138, 2167, 3135, 3137, 3136, 
         2015, 2482, 2014, 3353, 2345, 2481, 2480, 2971, 2972, 2970, 2969, 
         2583, 2162, 3200, 3201, 3045, 3046, 2471, 2251, 3286, 2335, 3285, 
         2334, 2463, 2869, 2054, 3415, 2868, 2580, 2155, 2694, 2249, 3194, 
         2250, 3195, 2154, 2692, 2153, 2693, 3040, 2009, 2458, 2563, 2562, 
         2561, 2328, 2450, 2449, 2864, 2865, 2863, 2862, 2576, 2145, 3024, 
         3352, 3025, 3188, 3187, 3026, 3027, 2445, 2446, 2324, 2959, 3276, 
         3275, 2424, 2855, 2050, 3374, 2854, 2572, 2134, 2853, 2237, 2553, 
         2878, 3431, 3433, 3432, 2282, 3214, 2283, 2879, 2111, 2443, 3117, 
         3116, 3115, 2403, 3344, 2191, 2429, 2200, 2734, 2201, 2735, 3094, 
         2736, 2203, 3158, 2202, 3422, 3080, 3081, 2750, 2206, 2749, 2207, 
         2751, 2188, 2746, 2205, 3161, 2204, 2901, 2301, 2604, 3096, 2532, 
         2896, 2300, 2911, 2916, 2906, 3246, 3240, 2068, 2802, 2807, 2796, 
         2819, 2813, 2093, 2782, 2788, 2775, 2998, 2991, 1985}],
        "0.5`"],
       Annotation[#, 0.5, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2031, 2505, 2026, 3059, 3058, 2370, 3326, 2268, 2504, 2369, 
         3323, 3325, 3324, 2174, 3145, 2173, 3146, 2025, 2498, 2019, 3356, 
         2363, 2497, 2496, 3316, 3317, 3315, 3314, 3069, 2169, 3209, 3210, 
         3055, 3056, 3057, 2486, 2487, 2354, 3308, 2353, 2485, 3307, 2058, 
         3306, 3067, 2166, 3206, 2262, 3052, 3053, 2479, 2257, 3298, 2344, 
         3297, 2343, 2478, 2968, 2056, 3416, 2967, 2582, 2161, 3199, 2256, 
         3043, 3044, 2470, 2248, 3284, 2333, 3283, 2332, 2459, 2867, 2053, 
         3414, 2866, 2579, 2152, 3038, 3430, 2247, 3193, 3039, 2457, 2243, 
         3280, 2327, 3277, 3279, 3278, 2447, 2455, 2456, 2409, 2408, 2542, 
         2144, 3022, 3350, 3023, 2861, 2880, 3351, 2004, 2881, 2286, 2555, 
         2554, 2619, 3119, 3118, 2620, 2594, 2882, 3224, 3225, 3084, 3085, 
         2077, 2444, 3249, 3250, 3248, 3252, 3251, 2084, 2617, 2618, 2616, 
         2769, 2768, 3091, 2081, 3376, 2767, 2602, 2759, 2758, 2876, 2279, 
         2432, 2219, 2933, 2930, 2720, 2189, 3156, 2716, 2745, 2741, 2891, 
         3231, 3235, 2887, 2900, 2895, 2299, 2910, 2915, 2905, 3245, 3239, 
         2067, 2801, 2806, 2795, 2818, 2812, 2092, 2781, 2787, 2774, 2997, 
         2990, 1984}],
        "0.4`"],
       Annotation[#, 0.4, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{2039, 2517, 3062, 3061, 3060, 2385, 2516, 2515, 2383, 3438, 
         2384, 2179, 3149, 2178, 3150, 2035, 2512, 2030, 3362, 2379, 2511, 
         2510, 2377, 3437, 2378, 2176, 3147, 2175, 3148, 2029, 2503, 2024, 
         3359, 2368, 2502, 2501, 2366, 3407, 2367, 2172, 3142, 3144, 3143, 
         2023, 2495, 2018, 3355, 2362, 2265, 2494, 2360, 3312, 2361, 3313, 
         2261, 3304, 3305, 3303, 2352, 2484, 3302, 2057, 3301, 3066, 2165, 
         3205, 2260, 3049, 3050, 3051, 2476, 2477, 2342, 3296, 2341, 2475, 
         3293, 2340, 3292, 3295, 3294, 2160, 2699, 2701, 2700, 2013, 2468, 
         2008, 3037, 3036, 3282, 3281, 2246, 2530, 2531, 2544, 2543, 2405, 
         2406, 2287, 2560, 2559, 2621, 3121, 3120, 2622, 2600, 2883, 3226, 
         2292, 3086, 3087, 3088, 2454, 3254, 2304, 3253, 2305, 3255, 2085, 
         2852, 2236, 3177, 2235, 3176, 3434, 2293, 3403, 3402, 3213, 2281, 
         3211, 2280, 3212, 2072, 2592, 2593, 2591, 3170, 2590, 2589, 2192, 
         2756, 2757, 3163, 3162, 2208, 2929, 2309, 2605, 3104, 2533, 2926, 
         2308, 2938, 2311, 3266, 2310, 3265, 3101, 2097, 3382, 3385, 3379, 
         2719, 2715, 2744, 2740, 2890, 3230, 3234, 2886, 2899, 2894, 2298, 
         2909, 2914, 2904, 3244, 3238, 2066, 2800, 2805, 2794, 2817, 2811, 
         2091, 2780, 2786, 2773, 2996, 2989, 1983}],
        "0.3`"],
       Annotation[#, 0.3, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{3073, 3074, 2824, 3424, 2823, 2584, 2182, 2319, 3404, 2318, 
         2088, 2289, 3401, 2288, 2076, 2276, 3372, 2044, 2524, 2525, 2394, 
         3336, 2393, 2523, 3334, 3335, 3333, 3332, 3072, 2180, 2275, 3371, 
         2042, 2521, 2038, 3368, 2390, 2520, 2519, 2388, 3439, 2389, 2272, 
         3330, 3331, 3329, 2382, 2514, 3328, 2061, 3327, 3071, 2177, 2271, 
         3365, 2034, 2508, 2509, 2376, 3408, 2375, 2507, 2373, 3436, 2374, 
         2267, 3321, 3322, 3320, 2365, 2500, 3319, 2060, 3318, 3070, 2171, 
         2266, 3358, 2022, 2493, 2017, 3354, 2359, 2264, 2492, 2357, 3406, 
         2358, 2259, 3300, 2351, 3299, 2350, 2483, 2974, 2348, 2973, 2349, 
         2975, 2255, 3291, 2339, 3290, 2338, 2469, 2873, 2253, 2872, 2254, 
         2874, 3083, 2623, 2113, 3122, 2112, 3420, 2730, 2196, 2729, 2732, 
         2731, 2078, 2467, 3257, 3258, 3256, 3260, 3259, 2086, 3186, 2242, 
         3184, 2241, 3185, 2294, 3343, 2295, 3227, 2285, 3215, 2284, 3441, 
         3216, 3423, 2073, 2598, 2599, 2597, 3171, 2596, 2595, 2193, 2762, 
         2210, 3164, 2209, 2943, 2944, 2606, 3108, 3109, 2534, 2609, 2608, 
         2607, 2313, 3269, 3270, 3106, 2099, 3390, 2723, 3387, 2725, 2722, 
         2755, 2753, 2923, 3262, 3264, 2921, 2928, 2925, 2307, 2935, 2937, 
         2932, 3103, 3100, 2096, 3381, 3384, 3378, 2718, 2714, 2743, 2739, 
         2889, 3229, 3233, 2885, 2898, 2893, 2297, 2908, 2913, 2903, 3243, 
         3237, 2065, 2799, 2804, 2793, 2816, 2810, 2090, 2779, 2785, 2772, 
         2995, 2988, 1982}],
        "0.2`"],
       Annotation[#, 0.2, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{1981, 2987, 2994, 2771, 2784, 2778, 2089, 2809, 2815, 2792, 
         2803, 2798, 2064, 3236, 3242, 2902, 2912, 2907, 2296, 2892, 2897, 
         2884, 3232, 3228, 2888, 2738, 2742, 2713, 2717, 3377, 3383, 3380, 
         2095, 3099, 3102, 2931, 2936, 2934, 2306, 2924, 2927, 2920, 3263, 
         3261, 2922, 2752, 2754, 2721, 2724, 3386, 3389, 3388, 2098, 3105, 
         3107, 2945, 2947, 2946, 2312, 2941, 2942, 2939, 3268, 3267, 2940, 
         2760, 2761, 2726, 2728, 3391, 2727, 3392, 2100, 3110, 3435, 2316, 
         3271, 2317, 2949, 2314, 2535, 3111, 2610, 2315, 2948, 2211, 3399, 
         2763, 3398, 2194, 3157, 2195, 3173, 2221, 3172, 2220, 2399, 3409, 
         2398, 2539, 2105, 3393, 2402, 2611, 2101, 2540, 2541, 2401, 2980, 
         2400, 2981, 2074, 3159, 3160, 2737, 2158, 3134, 2159, 2407, 2012, 
         2430, 2197, 2347, 3405, 2346, 2536, 2079, 2199, 3397, 2198, 2733, 
         3421, 2114, 3123, 3124, 2625, 2624, 2075, 3220, 3221, 3217, 3219, 
         3218, 2491, 3208, 3140, 3141, 3139, 2170, 2364, 3357, 2021, 2499, 
         2027, 3360, 2371, 2269, 2372, 3361, 2028, 2506, 2032, 3363, 2380, 
         2270, 2381, 3364, 2033, 2513, 2036, 3366, 2386, 2273, 2387, 3367, 
         2037, 2518, 2040, 3369, 2391, 2274, 2392, 3370, 2041, 2522, 2043, 
         3152, 3153, 3151, 2181, 3340, 3341, 3337, 3339, 3338, 2526, 2527}],
        "0.1`"],
       Annotation[#, 0.1, "Tooltip"]& ]}, {}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  ImageMargins->0.,
  ImageSize->{225, Automatic},
  PlotRange->NCache[{{0, 1}, {
      Rational[1, 10], 3}}, {{0, 1}, {0.1, 3}}],
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{229, 224},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->501237726]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->20702],

Cell["The probability density function integrates to unity:", "ExampleText",
 CellID->7625],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "x"}], "]"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Assumptions", "->", 
    RowBox[{"{", 
     RowBox[{"\[Nu]", ">", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4004],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->93447042]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6695],

Cell["Moments can be obtained from the characteristic function:", \
"ExampleText",
 CellID->3912],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{
   RowBox[{"x", "^", "4"}], ",", 
   RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4577],

Cell[BoxData[
 RowBox[{"\[Nu]", " ", 
  RowBox[{"(", 
   RowBox[{"2", "+", "\[Nu]"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{"4", "+", "\[Nu]"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{"6", "+", "\[Nu]"}], ")"}]}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->460457410]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"CharacteristicFunction", "[", 
      RowBox[{
       RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], ",", "t"}], 
      "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "4"}], "}"}]}], "]"}], "/.", 
   RowBox[{"{", 
    RowBox[{"t", "->", "0"}], "}"}]}], "//", "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->26071],

Cell[BoxData[
 RowBox[{"\[Nu]", " ", 
  RowBox[{"(", 
   RowBox[{"2", "+", "\[Nu]"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{"4", "+", "\[Nu]"}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{"6", "+", "\[Nu]"}], ")"}]}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->415235928]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->194342027],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiSquareDistribution"]], "InlineFormula"],
 " is a special case of ",
 Cell[BoxData[
  ButtonBox["GammaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/GammaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->20709924],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiSquareDistribution", "[", "n", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->193204549],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "n"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["n", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["n", "2"], "]"}]]], "Output",
 ImageSize->{100, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3680834]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"GammaDistribution", "[", 
    RowBox[{
     RowBox[{"n", "/", "2"}], ",", "2"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->26051320],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "n"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["n", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["n", "2"], "]"}]]], "Output",
 ImageSize->{100, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->187923473]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->185796011],

Cell[TextData[{
 "The square root of a ",
 Cell[BoxData[
  SuperscriptBox[
   StyleBox["\[Chi]", "TR"], "2"]], "InlineFormula"],
 " variable follows the ",
 Cell[BoxData[
  ButtonBox["ChiDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->475500449],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"ChiSquareDistribution", "[", "n", "]"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->402962351],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "n"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["n", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["n", "2"], "]"}]]], "Output",
 ImageSize->{100, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180074521]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"ChiDistribution", "[", "n", "]"}], ",", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], "]"}], 
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "x", "]"}], ",", "x"}], "]"}]}], "//", 
  "Simplify"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19944730],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox["2", 
    RowBox[{
     RowBox[{"-", "n"}], "/", "2"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "x"}], "/", "2"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["n", "2"]}]]}], 
  RowBox[{"Gamma", "[", 
   FractionBox["n", "2"], "]"}]]], "Output",
 ImageSize->{100, 46},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->244182411]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[Cell[BoxData[
   InterpretationBox[Cell[TextData[{
     "Possible Issues",
     "\[NonBreakingSpace]\[NonBreakingSpace]",
     Cell["(2)", "ExampleCount"]
    }], "ExampleSection"],
    $Line = 0; Null]]]], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->12084],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["ChiSquareDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ChiSquareDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Nu]", "TR"]], "InlineFormula"],
 " is not a positive real number:"
}], "ExampleText",
 CellID->22580],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"ChiSquareDistribution", "[", 
   RowBox[{"-", "1"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31450],

Cell[BoxData[
 RowBox[{
  RowBox[{"ChiSquareDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-1\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->278492106],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"ChiSquareDistribution", "[", 
   RowBox[{"-", "1"}], "]"}], "]"}]], "Output",
 ImageSize->{222, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->137485302]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6730],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"ChiSquareDistribution", "[", "\[Nu]", "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{"\[Nu]", "->", "I"}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31619],

Cell[BoxData["\[ImaginaryI]"], "Output",
 ImageSize->{12, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->578511459]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->32535],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["StudentTDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/StudentTDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FRatioDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FRatioDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["GammaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/GammaDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ChiDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ChiDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NoncentralChiSquareDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NoncentralChiSquareDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindFit",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindFit"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->9376]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->12747],

Cell[TextData[ButtonBox["Continuous Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/ContinuousDistributions"]], "Tutorials",
 CellID->7471]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Continuous Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ContinuousStatisticalDistributions"]], "MoreAbout",\

 CellID->2643154],

Cell[TextData[ButtonBox["Functions Used in Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInStatistics"]], "MoreAbout",
 CellID->842945685],

Cell[TextData[ButtonBox["Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MathematicalFunctions"]], "MoreAbout",
 CellID->449134576],

Cell[TextData[ButtonBox["Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/StatisticalDistributions"]], "MoreAbout",
 CellID->269864663],

Cell[TextData[ButtonBox["Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Statistics"]], "MoreAbout",
 CellID->73977828],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->393629738],

Cell[TextData[ButtonBox["New in 6.0: Statistics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60Statistics"]], "MoreAbout",
 CellID->423247875]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ChiSquareDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 28, 26.5085824}", "context" -> "System`", 
    "keywords" -> {
     "chi squared distribution", "Pearson chi square", "Pearson chi squared"},
     "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "ChiSquareDistribution[\\[Nu]] represents a \\[Chi]^2 distribution with \
\\[Nu] degrees of freedom.", "synonyms" -> {"chi square distribution"}, 
    "title" -> "ChiSquareDistribution", "type" -> "Symbol", "uri" -> 
    "ref/ChiSquareDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[9470, 286, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->157342261]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 135642, 2948}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3534, 78, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[4157, 105, 1586, 37, 70, "ObjectNameGrid"],
Cell[5746, 144, 718, 22, 70, "Usage",
 CellID->31604]
}, Open  ]],
Cell[CellGroupData[{
Cell[6501, 171, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6979, 188, 861, 32, 70, "Notes",
 CellID->429991538],
Cell[7843, 222, 493, 16, 70, "Notes",
 CellID->73600657],
Cell[8339, 240, 387, 13, 70, "Notes",
 CellID->27322],
Cell[8729, 255, 704, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[9470, 286, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->157342261],
Cell[CellGroupData[{
Cell[9855, 300, 148, 5, 70, "ExampleSection",
 CellID->150901550],
Cell[10006, 307, 242, 8, 70, "ExampleText",
 CellID->27465],
Cell[CellGroupData[{
Cell[10273, 319, 148, 4, 28, "Input",
 CellID->23005],
Cell[10424, 325, 162, 5, 36, "Output",
 CellID->313588289]
}, Open  ]],
Cell[CellGroupData[{
Cell[10623, 335, 151, 4, 28, "Input",
 CellID->6856],
Cell[10777, 341, 183, 6, 36, "Output",
 CellID->35238384]
}, Open  ]],
Cell[10975, 350, 120, 3, 70, "ExampleDelimiter",
 CellID->3238],
Cell[11098, 355, 116, 2, 70, "ExampleText",
 CellID->21100],
Cell[CellGroupData[{
Cell[11239, 361, 173, 6, 70, "Input",
 CellID->7194],
Cell[11415, 369, 528, 19, 68, "Output",
 CellID->90574492]
}, Open  ]],
Cell[CellGroupData[{
Cell[11980, 393, 289, 10, 28, "Input",
 CellID->2002],
Cell[12272, 405, 2921, 52, 162, "Output",
 Evaluatable->False,
 CellID->189219742]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15242, 463, 216, 7, 70, "ExampleSection",
 CellID->13933],
Cell[15461, 472, 214, 7, 70, "ExampleText",
 CellID->31953],
Cell[CellGroupData[{
Cell[15700, 483, 178, 6, 70, "Input",
 CellID->28942],
Cell[15881, 491, 331, 10, 36, "Output",
 CellID->26988817]
}, Open  ]],
Cell[16227, 504, 121, 3, 70, "ExampleDelimiter",
 CellID->24595],
Cell[16351, 509, 83, 1, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[16459, 514, 151, 4, 70, "Input",
 CellID->2526],
Cell[16613, 520, 240, 9, 60, "Output",
 CellID->884612446]
}, Open  ]],
Cell[CellGroupData[{
Cell[16890, 534, 150, 4, 70, "Input",
 CellID->880],
Cell[17043, 540, 206, 7, 51, "Output",
 CellID->310959359]
}, Open  ]],
Cell[17264, 550, 120, 3, 70, "ExampleDelimiter",
 CellID->1657],
Cell[17387, 555, 185, 7, 70, "ExampleText",
 CellID->10330],
Cell[CellGroupData[{
Cell[17597, 566, 218, 7, 70, "Input",
 CellID->25998],
Cell[17818, 575, 295, 10, 36, "Output",
 CellID->39796058]
}, Open  ]],
Cell[18128, 588, 121, 3, 70, "ExampleDelimiter",
 CellID->10737],
Cell[18252, 593, 293, 11, 70, "ExampleText",
 CellID->10291],
Cell[CellGroupData[{
Cell[18570, 608, 178, 6, 70, "Input",
 CellID->16765],
Cell[18751, 616, 175, 5, 36, "Output",
 CellID->482753035]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[18975, 627, 223, 7, 70, "ExampleSection",
 CellID->11928],
Cell[19201, 636, 330, 13, 70, "ExampleText",
 CellID->813014373],
Cell[CellGroupData[{
Cell[19556, 653, 200, 7, 70, "Input",
 CellID->944228315],
Cell[19759, 662, 304, 10, 49, "Output",
 CellID->312563814]
}, Open  ]],
Cell[20078, 675, 124, 3, 70, "ExampleDelimiter",
 CellID->44516724],
Cell[20205, 680, 112, 3, 70, "ExampleText",
 CellID->19469],
Cell[CellGroupData[{
Cell[20342, 687, 290, 10, 70, "Input",
 CellID->6864],
Cell[20635, 699, 2638, 47, 70, "Output",
 Evaluatable->False,
 CellID->6794571]
}, Open  ]],
Cell[23288, 749, 121, 3, 70, "ExampleDelimiter",
 CellID->31467],
Cell[23412, 754, 222, 9, 70, "ExampleText",
 CellID->18934],
Cell[CellGroupData[{
Cell[23659, 767, 399, 13, 70, "Input",
 CellID->8106],
Cell[24061, 782, 97680, 1630, 70, "Output",
 CellID->501237726]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[121790, 2418, 233, 7, 70, "ExampleSection",
 CellID->20702],
Cell[122026, 2427, 91, 1, 70, "ExampleText",
 CellID->7625],
Cell[CellGroupData[{
Cell[122142, 2432, 407, 13, 70, "Input",
 CellID->4004],
Cell[122552, 2447, 157, 5, 36, "Output",
 CellID->93447042]
}, Open  ]],
Cell[122724, 2455, 120, 3, 70, "ExampleDelimiter",
 CellID->6695],
Cell[122847, 2460, 97, 2, 70, "ExampleText",
 CellID->3912],
Cell[CellGroupData[{
Cell[122969, 2466, 217, 7, 70, "Input",
 CellID->4577],
Cell[123189, 2475, 358, 12, 36, "Output",
 CellID->460457410]
}, Open  ]],
Cell[CellGroupData[{
Cell[123584, 2492, 420, 14, 70, "Input",
 CellID->26071],
Cell[124007, 2508, 358, 12, 36, "Output",
 CellID->415235928]
}, Open  ]],
Cell[124380, 2523, 125, 3, 70, "ExampleDelimiter",
 CellID->194342027],
Cell[124508, 2528, 367, 12, 70, "ExampleText",
 CellID->20709924],
Cell[CellGroupData[{
Cell[124900, 2544, 174, 6, 70, "Input",
 CellID->193204549],
Cell[125077, 2552, 515, 19, 67, "Output",
 CellID->3680834]
}, Open  ]],
Cell[CellGroupData[{
Cell[125629, 2576, 217, 7, 70, "Input",
 CellID->26051320],
Cell[125849, 2585, 517, 19, 67, "Output",
 CellID->187923473]
}, Open  ]],
Cell[126381, 2607, 125, 3, 70, "ExampleDelimiter",
 CellID->185796011],
Cell[126509, 2612, 332, 12, 70, "ExampleText",
 CellID->475500449],
Cell[CellGroupData[{
Cell[126866, 2628, 174, 6, 70, "Input",
 CellID->402962351],
Cell[127043, 2636, 517, 19, 67, "Output",
 CellID->180074521]
}, Open  ]],
Cell[CellGroupData[{
Cell[127597, 2660, 343, 12, 70, "Input",
 CellID->19944730],
Cell[127943, 2674, 517, 19, 67, "Output",
 CellID->244182411]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[128509, 2699, 327, 9, 70, "ExampleSection",
 CellID->12084],
Cell[128839, 2710, 317, 10, 70, "ExampleText",
 CellID->22580],
Cell[CellGroupData[{
Cell[129181, 2724, 163, 5, 70, "Input",
 CellID->31450],
Cell[129347, 2731, 202, 5, 70, "Message",
 CellID->278492106],
Cell[129552, 2738, 256, 8, 36, "Output",
 CellID->137485302]
}, Open  ]],
Cell[129823, 2749, 120, 3, 70, "ExampleDelimiter",
 CellID->6730],
Cell[129946, 2754, 144, 4, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[130115, 2762, 225, 7, 70, "Input",
 CellID->31619],
Cell[130343, 2771, 170, 5, 36, "Output",
 CellID->578511459]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[130574, 2783, 311, 9, 70, "SeeAlsoSection",
 CellID->32535],
Cell[130888, 2794, 1550, 53, 70, "SeeAlso",
 CellID->9376]
}, Open  ]],
Cell[CellGroupData[{
Cell[132475, 2852, 314, 9, 70, "TutorialsSection",
 CellID->12747],
Cell[132792, 2863, 157, 3, 70, "Tutorials",
 CellID->7471]
}, Open  ]],
Cell[CellGroupData[{
Cell[132986, 2871, 299, 8, 70, "MoreAboutSection"],
Cell[133288, 2881, 182, 4, 70, "MoreAbout",
 CellID->2643154],
Cell[133473, 2887, 165, 3, 70, "MoreAbout",
 CellID->842945685],
Cell[133641, 2892, 155, 3, 70, "MoreAbout",
 CellID->449134576],
Cell[133799, 2897, 161, 3, 70, "MoreAbout",
 CellID->269864663],
Cell[133963, 2902, 131, 3, 70, "MoreAbout",
 CellID->73977828],
Cell[134097, 2907, 179, 3, 70, "MoreAbout",
 CellID->393629738],
Cell[134279, 2912, 151, 3, 70, "MoreAbout",
 CellID->423247875]
}, Open  ]],
Cell[134445, 2918, 27, 0, 70, "History"],
Cell[134475, 2920, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

